return {
  modules = {
    {
      name = "lovr",
      summary = "In the beginning, there was nothing.",
      description = "`lovr` is the single global table that is exposed to every LÖVR app. It contains a set of **modules** and a set of **callbacks**.",
      key = "lovr",
      objects = {
        {
          name = "Object",
          summary = "The base object.",
          description = "The superclass of all LÖVR objects.\n\nIn addition to the methods here, all objects have a `__tostring` metamethod that returns the name of the object's type.  So `tostring(object) == 'Blob'` will check if a LÖVR object is a Blob.",
          key = "Object",
          module = "lovr",
          notes = "Note that the functions here don't apply to any vector objects, see `Vectors`.",
          methods = {
            {
              name = "release",
              summary = "Immediately release the Lua reference to an object.",
              description = "Immediately destroys Lua's reference to the object it's called on.  After calling this function on an object, it is an error to do anything with the object from Lua (call methods on it, pass it to other functions, etc.).  If nothing else is using the object, it will be destroyed immediately, which can be used to destroy something earlier than it would normally be garbage collected in order to reduce memory.",
              key = "Object:release",
              module = "lovr",
              notes = "The object may not be destroyed immediately if something else is referring to it (e.g. it is pushed to a Channel or exists in the payload of a pending event).",
              variants = {
                {
                  arguments = {},
                  returns = {}
                }
              }
            }
          }
        }
      },
      sections = {
        {
          name = "Modules",
          tag = "modules",
          description = "Modules are the **what** of your app; you can use the functions in modules to tell LÖVR to do things. For example, you can draw things on the screen, figure out what buttons on a controller are pressed, or load a 3D model from a file.  Each module does what it says on the tin, so the `lovr.graphics` module deals with rendering graphics and `lovr.headset` allows you to interact with VR hardware."
        },
        {
          name = "Callbacks",
          tag = "callbacks",
          description = "Callbacks are the **when** of the application; you write code inside callbacks which LÖVR then calls at certain points in time.  For example, the `lovr.load` callback is called once at startup, and `lovr.focus` is called when the VR application gains or loses input focus."
        },
        {
          name = "Version",
          tag = "version",
          description = "This function can be used to get the current version of LÖVR."
        }
      },
      enums = {},
      functions = {
        {
          name = "getVersion",
          tag = "version",
          summary = "Get the current version.",
          description = "Get the current major, minor, and patch version of LÖVR.",
          key = "lovr.getVersion",
          module = "lovr",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "major",
                  type = "number",
                  description = "The major version."
                },
                {
                  name = "minor",
                  type = "number",
                  description = "The minor version."
                },
                {
                  name = "patch",
                  type = "number",
                  description = "The patch number."
                }
              }
            }
          }
        }
      }
    },
    {
      name = "audio",
      tag = "modules",
      summary = "Plays sound.",
      description = "The `lovr.audio` module is responsible for playing sound effects and music.  To play a sound, create a `Source` object and call `Source:play` on it.  Currently ogg, wav, and mp3 audio formats are supported.",
      key = "lovr.audio",
      objects = {
        {
          name = "Source",
          summary = "A playable sound object.",
          description = "A Source is an object representing a single sound.  Currently ogg, wav, and mp3 formats are supported.\n\nWhen a Source is playing, it will send audio to the speakers.  Sources do not play automatically when they are created.  Instead, the `play`, `pause`, and `stop` functions can be used to control when they should play.\n\n`Source:seek` and `Source:tell` can be used to control the playback position of the Source.  A Source can be set to loop when it reaches the end using `Source:setLooping`.",
          key = "Source",
          module = "lovr.audio",
          constructors = {
            "lovr.audio.newSource",
            "Source:clone"
          },
          sections = {
            {
              name = "Playback",
              tag = "sourcePlayback"
            },
            {
              name = "Spatial Effects",
              tag = "sourceEffects"
            },
            {
              name = "Utility",
              tag = "sourceUtility"
            }
          },
          methods = {
            {
              name = "clone",
              tag = "sourceUtility",
              summary = "Create an identical copy of the Source.",
              description = "Creates a copy of the Source, referencing the same `Sound` object and inheriting all of the settings of this Source.  However, it will be created in the stopped state and will be rewound to the beginning.",
              key = "Source:clone",
              module = "lovr.audio",
              notes = "This is a good way to create multiple Sources that play the same sound, since the audio data won't be loaded multiple times and can just be reused.  You can also create multiple `Source` objects and pass in the same `Sound` object for each one, which will have the same effect.",
              related = {
                "lovr.audio.newSource"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "source",
                      type = "Source",
                      description = "A genetically identical copy of the Source."
                    }
                  }
                }
              }
            },
            {
              name = "getDirectivity",
              tag = "sourceEffects",
              summary = "Get the directivity of the Source.",
              description = "Returns the directivity settings for the Source.\n\nThe directivity is controlled by two parameters: the weight and the power.\n\nThe weight is a number between 0 and 1 controlling the general \"shape\" of the sound emitted. 0.0 results in a completely omnidirectional sound that can be heard from all directions.  1.0 results in a full dipole shape that can be heard only from the front and back.  0.5 results in a cardioid shape that can only be heard from one direction.  Numbers in between will smoothly transition between these.\n\nThe power is a number that controls how \"focused\" or sharp the shape is.  Lower power values can be heard from a wider set of angles.  It is an exponent, so it can get arbitrarily large.  Note that a power of zero will still result in an omnidirectional source, regardless of the weight.",
              key = "Source:getDirectivity",
              module = "lovr.audio",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "weight",
                      type = "number",
                      description = "The dipole weight.  0.0 is omnidirectional, 1.0 is a dipole, 0.5 is cardioid."
                    },
                    {
                      name = "power",
                      type = "number",
                      description = "The dipole power, controlling how focused the directivity shape is."
                    }
                  }
                }
              }
            },
            {
              name = "getDuration",
              tag = "sourcePlayback",
              summary = "Get the duration of the Source.",
              description = "Returns the duration of the Source.",
              key = "Source:getDuration",
              module = "lovr.audio",
              related = {
                "Sound:getDuration"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "unit",
                      type = "TimeUnit",
                      description = "The unit to return.",
                      default = "'seconds'"
                    }
                  },
                  returns = {
                    {
                      name = "duration",
                      type = "number",
                      description = "The duration of the Source."
                    }
                  }
                }
              }
            },
            {
              name = "getOrientation",
              tag = "sourceEffects",
              summary = "Get the orientation of the Source.",
              description = "Returns the orientation of the Source, in angle/axis representation.",
              key = "Source:getOrientation",
              module = "lovr.audio",
              related = {
                "Source:getPosition",
                "Source:getPose",
                "Source:getCone",
                "lovr.audio.getOrientation"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the Source is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getPose",
              tag = "sourceEffects",
              summary = "Get the pose of the Source.",
              description = "Returns the position and orientation of the Source.",
              key = "Source:getPose",
              module = "lovr.audio",
              related = {
                "Source:getPosition",
                "Source:getOrientation",
                "Source:getCone",
                "lovr.audio.getPose"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the Source, in meters."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the Source, in meters."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the Source, in meters."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the Source is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getPosition",
              tag = "sourceEffects",
              summary = "Get the position of the Source.",
              description = "Returns the position of the Source, in meters.  Setting the position will cause the Source to be distorted and attenuated based on its position relative to the listener.",
              key = "Source:getPosition",
              module = "lovr.audio",
              related = {
                "Source:getOrientation",
                "Source:getPose",
                "Source:getCone",
                "lovr.audio.getPosition"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    }
                  }
                }
              }
            },
            {
              name = "getRadius",
              tag = "sourceEffects",
              summary = "Get the radius of the Source.",
              description = "Returns the radius of the Source, in meters.\n\nThis does not control falloff or attenuation.  It is only used for smoothing out occlusion.  If a Source doesn't have a radius, then when it becomes occluded by a wall its volume will instantly drop.  Giving the Source a radius that approximates its emitter's size will result in a smooth transition between audible and occluded, improving realism.",
              key = "Source:getRadius",
              module = "lovr.audio",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the Source, in meters."
                    }
                  }
                }
              }
            },
            {
              name = "getSound",
              tag = "sourceUtility",
              summary = "Get the Sound object backing the Source.",
              description = "Returns the `Sound` object backing the Source.  Multiple Sources can share one Sound, allowing its data to only be loaded once.  An easy way to do this sharing is by using `Source:clone`.",
              key = "Source:getSound",
              module = "lovr.audio",
              related = {
                "Source:clone",
                "lovr.audio.newSource"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "sound",
                      type = "Sound",
                      description = "The Sound object."
                    }
                  }
                }
              }
            },
            {
              name = "getVolume",
              tag = "sourcePlayback",
              summary = "Get the volume of the Source.",
              description = "Returns the current volume factor for the Source.",
              key = "Source:getVolume",
              module = "lovr.audio",
              variants = {
                {
                  arguments = {
                    {
                      name = "units",
                      type = "VolumeUnit",
                      description = "The units to return (linear or db).",
                      default = "'linear'"
                    }
                  },
                  returns = {
                    {
                      name = "volume",
                      type = "number",
                      description = "The volume of the Source."
                    }
                  }
                }
              }
            },
            {
              name = "isEffectEnabled",
              tag = "sourceEffects",
              summary = "Check if an effect is enabled.",
              description = "Returns whether a given `Effect` is enabled for the Source.",
              key = "Source:isEffectEnabled",
              module = "lovr.audio",
              notes = "The active spatializer will determine which effects are supported.  If an unsupported effect is enabled on a Source, no error will be reported.  Instead, it will be silently ignored.  See `lovr.audio.getSpatializer` for a table showing the effects supported by each spatializer.\n\nCalling this function on a Source that was created with `{ effects = false }` will always return false.",
              variants = {
                {
                  arguments = {
                    {
                      name = "effect",
                      type = "Effect",
                      description = "The effect."
                    }
                  },
                  returns = {
                    {
                      name = "enabled",
                      type = "boolean",
                      description = "Whether the effect is enabled."
                    }
                  }
                }
              }
            },
            {
              name = "isLooping",
              tag = "sourcePlayback",
              summary = "Check if the Source is looping.",
              description = "Returns whether or not the Source will loop when it finishes.",
              key = "Source:isLooping",
              module = "lovr.audio",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "looping",
                      type = "boolean",
                      description = "Whether or not the Source is looping."
                    }
                  }
                }
              }
            },
            {
              name = "isPlaying",
              tag = "sourcePlayback",
              summary = "Check if the Source is playing.",
              description = "Returns whether or not the Source is playing.",
              key = "Source:isPlaying",
              module = "lovr.audio",
              related = {
                "Source:play",
                "Source:pause",
                "Source:stop"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "playing",
                      type = "boolean",
                      description = "Whether the Source is playing."
                    }
                  }
                }
              }
            },
            {
              name = "pause",
              tag = "sourcePlayback",
              summary = "Pause the Source.",
              description = "Pauses the source.  It can be resumed with `Source:resume` or `Source:play`. If a paused source is rewound, it will remain paused.",
              key = "Source:pause",
              module = "lovr.audio",
              variants = {
                {
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "play",
              tag = "sourcePlayback",
              summary = "Play the Source.",
              description = "Plays the Source.  This doesn't do anything if the Source is already playing.",
              key = "Source:play",
              module = "lovr.audio",
              notes = "There is a maximum of 64 Sources that can be playing at once.  If 64 Sources are already playing, this function will return `false`.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "success",
                      type = "boolean",
                      description = "Whether the Source successfully started playing."
                    }
                  }
                }
              }
            },
            {
              name = "seek",
              tag = "sourcePlayback",
              summary = "Set the playback position of the Source.",
              description = "Seeks the Source to the specified position.",
              key = "Source:seek",
              module = "lovr.audio",
              notes = "Seeking a Source backed by a stream `Sound` has no meaningful effect.",
              variants = {
                {
                  arguments = {
                    {
                      name = "position",
                      type = "number",
                      description = "The position to seek to."
                    },
                    {
                      name = "unit",
                      type = "TimeUnit",
                      description = "The units for the seek position.",
                      default = "'seconds'"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setDirectivity",
              tag = "sourceEffects",
              summary = "Set the directivity of the Source.",
              description = "Sets the directivity settings for the Source.\n\nThe directivity is controlled by two parameters: the weight and the power.\n\nThe weight is a number between 0 and 1 controlling the general \"shape\" of the sound emitted. 0.0 results in a completely omnidirectional sound that can be heard from all directions.  1.0 results in a full dipole shape that can be heard only from the front and back.  0.5 results in a cardioid shape that can only be heard from one direction.  Numbers in between will smoothly transition between these.\n\nThe power is a number that controls how \"focused\" or sharp the shape is.  Lower power values can be heard from a wider set of angles.  It is an exponent, so it can get arbitrarily large.  Note that a power of zero will still result in an omnidirectional source, regardless of the weight.",
              key = "Source:setDirectivity",
              module = "lovr.audio",
              variants = {
                {
                  arguments = {
                    {
                      name = "weight",
                      type = "number",
                      description = "The dipole weight.  0.0 is omnidirectional, 1.0 is a dipole, 0.5 is cardioid."
                    },
                    {
                      name = "power",
                      type = "number",
                      description = "The dipole power, controlling how focused the directivity shape is."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setEffectEnabled",
              tag = "sourceEffects",
              summary = "Enable or disable an effect.",
              description = "Enables or disables an effect on the Source.",
              key = "Source:setEffectEnabled",
              module = "lovr.audio",
              notes = "The active spatializer will determine which effects are supported.  If an unsupported effect is enabled on a Source, no error will be reported.  Instead, it will be silently ignored.  See `lovr.audio.getSpatializer` for a table showing the effects supported by each spatializer.\n\nCalling this function on a Source that was created with `{ effects = false }` will throw an error.",
              variants = {
                {
                  arguments = {
                    {
                      name = "effect",
                      type = "Effect",
                      description = "The effect."
                    },
                    {
                      name = "enable",
                      type = "boolean",
                      description = "Whether the effect should be enabled."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setLooping",
              tag = "sourcePlayback",
              summary = "Set whether or not the Source loops.",
              description = "Sets whether or not the Source loops.",
              key = "Source:setLooping",
              module = "lovr.audio",
              notes = "Attempting to loop a Source backed by a stream `Sound` will cause an error.",
              variants = {
                {
                  arguments = {
                    {
                      name = "loop",
                      type = "boolean",
                      description = "Whether or not the Source will loop."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setOrientation",
              tag = "sourceEffects",
              summary = "Set the orientation of the Source.",
              description = "Sets the orientation of the Source in angle/axis representation.",
              key = "Source:setOrientation",
              module = "lovr.audio",
              related = {
                "Source:setPosition",
                "Source:setPose",
                "Source:setCone",
                "lovr.audio.setOrientation"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the Source should be rotated around its rotation axis."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setPose",
              tag = "sourceEffects",
              summary = "Set the pose of the Source.",
              description = "Sets the position and orientation of the Source.",
              key = "Source:setPose",
              module = "lovr.audio",
              related = {
                "Source:setPosition",
                "Source:setOrientation",
                "lovr.audio.setPose"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the Source, in meters."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the Source, in meters."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the Source, in meters."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the Source is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setPosition",
              tag = "sourceEffects",
              summary = "Set the position of the Source.",
              description = "Sets the position of the Source, in meters.  Setting the position will cause the Source to be distorted and attenuated based on its position relative to the listener.\n\nOnly mono sources can be positioned.  Setting the position of a stereo Source will cause an error.",
              key = "Source:setPosition",
              module = "lovr.audio",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setRadius",
              tag = "sourceEffects",
              summary = "Set the radius of the Source.",
              description = "Sets the radius of the Source, in meters.\n\nThis does not control falloff or attenuation.  It is only used for smoothing out occlusion.  If a Source doesn't have a radius, then when it becomes occluded by a wall its volume will instantly drop.  Giving the Source a radius that approximates its emitter's size will result in a smooth transition between audible and occluded, improving realism.",
              key = "Source:setRadius",
              module = "lovr.audio",
              variants = {
                {
                  arguments = {
                    {
                      name = "radius",
                      type = "number",
                      description = "The new radius of the Source, in meters."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setVolume",
              tag = "sourcePlayback",
              summary = "Set the volume of the Source.",
              description = "Sets the current volume factor for the Source.",
              key = "Source:setVolume",
              module = "lovr.audio",
              notes = "The volume will be clamped to a 0-1 range (0 dB).",
              variants = {
                {
                  arguments = {
                    {
                      name = "volume",
                      type = "number",
                      description = "The new volume."
                    },
                    {
                      name = "units",
                      type = "VolumeUnit",
                      description = "The units of the value.",
                      default = "'linear'"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "stop",
              tag = "sourcePlayback",
              summary = "Stop the Source.",
              description = "Stops the source, also rewinding it to the beginning.",
              key = "Source:stop",
              module = "lovr.audio",
              related = {
                "Source:play",
                "Source:pause",
                "Source:isPlaying"
              },
              variants = {
                {
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "tell",
              tag = "sourcePlayback",
              summary = "Get the playback position of the Source.",
              description = "Returns the current playback position of the Source.",
              key = "Source:tell",
              module = "lovr.audio",
              notes = "The return value for Sources backed by a stream `Sound` has no meaning.",
              variants = {
                {
                  arguments = {
                    {
                      name = "unit",
                      type = "TimeUnit",
                      description = "The unit to return.",
                      default = "'seconds'"
                    }
                  },
                  returns = {
                    {
                      name = "position",
                      type = "number",
                      description = "The current playback position."
                    }
                  }
                }
              }
            }
          }
        }
      },
      functions = {
        {
          name = "getAbsorption",
          summary = "Get the absorption coefficients.",
          description = "Returns the global air absorption coefficients for the medium.  This affects Sources that have the `absorption` effect enabled, causing audio volume to drop off with distance as it is absorbed by the medium it's traveling through (air, water, etc.).  The difference between absorption and falloff is that absorption is more subtle and is frequency-dependent, so higher-frequency bands can get absorbed more quickly than lower ones.  This can be used to apply \"underwater\" effects and stuff.",
          key = "lovr.audio.getAbsorption",
          module = "lovr.audio",
          notes = "Absorption is currently only supported by the phonon spatializer.\n\nThe frequency bands correspond to `400Hz`, `2.5KHz`, and `15KHz`.\n\nThe default coefficients are `.0002`, `.0017`, and `.0182` for low, mid, and high.",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "low",
                  type = "number",
                  description = "The absorption coefficient for the low frequency band."
                },
                {
                  name = "mid",
                  type = "number",
                  description = "The absorption coefficient for the mid frequency band."
                },
                {
                  name = "high",
                  type = "number",
                  description = "The absorption coefficient for the high frequency band."
                }
              }
            }
          }
        },
        {
          name = "getDevices",
          tag = "devices",
          summary = "Get a list of audio devices.",
          description = "Returns a list of playback or capture devices.  Each device has an `id`, `name`, and a `default` flag indicating whether it's the default device.\n\nTo use a specific device id for playback or capture, pass it to `lovr.audio.setDevice`.",
          key = "lovr.audio.getDevices",
          module = "lovr.audio",
          related = {
            "lovr.audio.setDevice",
            "lovr.audio.start",
            "lovr.audio.stop"
          },
          variants = {
            {
              arguments = {
                {
                  name = "type",
                  type = "AudioType",
                  description = "The type of devices to query (playback or capture).",
                  default = "'playback'"
                }
              },
              returns = {
                {
                  name = "devices",
                  type = "table",
                  description = "The list of devices.",
                  table = {
                    {
                      name = "[].id",
                      type = "userdata",
                      description = "A unique, opaque id for the device."
                    },
                    {
                      name = "[].name",
                      type = "string",
                      description = "A human readable name for the device."
                    },
                    {
                      name = "[].default",
                      type = "boolean",
                      description = "Whether the device is the default audio device."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "getOrientation",
          tag = "listener",
          summary = "Get the orientation of the listener.",
          description = "Returns the orientation of the virtual audio listener in angle/axis representation.",
          key = "lovr.audio.getOrientation",
          module = "lovr.audio",
          related = {
            "lovr.audio.getPosition",
            "lovr.audio.getPose",
            "Source:getOrientation"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "angle",
                  type = "number",
                  description = "The number of radians the listener is rotated around its axis of rotation."
                },
                {
                  name = "ax",
                  type = "number",
                  description = "The x component of the axis of rotation."
                },
                {
                  name = "ay",
                  type = "number",
                  description = "The y component of the axis of rotation."
                },
                {
                  name = "az",
                  type = "number",
                  description = "The z component of the axis of rotation."
                }
              }
            }
          }
        },
        {
          name = "getPose",
          tag = "listener",
          summary = "Get the pose of the listener.",
          description = "Returns the position and orientation of the virtual audio listener.",
          key = "lovr.audio.getPose",
          module = "lovr.audio",
          related = {
            "lovr.audio.getPosition",
            "lovr.audio.getOrientation",
            "Source:getPose"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "x",
                  type = "number",
                  description = "The x position of the listener, in meters."
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y position of the listener, in meters."
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z position of the listener, in meters."
                },
                {
                  name = "angle",
                  type = "number",
                  description = "The number of radians the listener is rotated around its axis of rotation."
                },
                {
                  name = "ax",
                  type = "number",
                  description = "The x component of the axis of rotation."
                },
                {
                  name = "ay",
                  type = "number",
                  description = "The y component of the axis of rotation."
                },
                {
                  name = "az",
                  type = "number",
                  description = "The z component of the axis of rotation."
                }
              }
            }
          }
        },
        {
          name = "getPosition",
          tag = "listener",
          summary = "Get the position of the listener.",
          description = "Returns the position of the virtual audio listener, in meters.",
          key = "lovr.audio.getPosition",
          module = "lovr.audio",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "x",
                  type = "number",
                  description = "The x position of the listener."
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y position of the listener."
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z position of the listener."
                }
              }
            }
          }
        },
        {
          name = "getSampleRate",
          tag = "devices",
          summary = "Get the playback device sample rate.",
          description = "Returns the sample rate used by the playback device.  This can be changed using `lovr.conf`.",
          key = "lovr.audio.getSampleRate",
          module = "lovr.audio",
          related = {
            "lovr.conf"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "rate",
                  type = "number",
                  description = "The sample rate of the playback device, in Hz."
                }
              }
            }
          }
        },
        {
          name = "getSpatializer",
          tag = "listener",
          summary = "Get the name of the active spatializer",
          description = "Returns the name of the active spatializer (`simple`, `oculus`, or `phonon`).\n\nThe `t.audio.spatializer` setting in `lovr.conf` can be used to express a preference for a particular spatializer.  If it's `nil`, all spatializers will be tried in the following order: `phonon`, `oculus`, `simple`.",
          key = "lovr.audio.getSpatializer",
          module = "lovr.audio",
          notes = "Using a feature or effect that is not supported by the current spatializer will not error, it just won't do anything.\n\n<table>\n  <thead>\n    <tr>\n      <td>Feature</td>\n      <td>simple</td>\n      <td>phonon</td>\n      <td>oculus</td>\n    </tr>\n  </thead>\n  <tbody>\n    <tr>\n      <td>Effect: Spatialization</td>\n      <td>x</td>\n      <td>x</td>\n      <td>x</td>\n    </tr>\n    <tr>\n      <td>Effect: Attenuation</td>\n      <td>x</td>\n      <td>x</td>\n      <td>x</td>\n    </tr>\n    <tr>\n      <td>Effect: Absorption</td>\n      <td></td>\n      <td>x</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td>Effect: Occlusion</td>\n      <td></td>\n      <td>x</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td>Effect: Transmission</td>\n      <td></td>\n      <td>x</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td>Effect: Reverb</td>\n      <td></td>\n      <td>x</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td>lovr.audio.setGeometry</td>\n      <td></td>\n      <td>x</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td>Source:setDirectivity</td>\n      <td>x</td>\n      <td>x</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td>Source:setRadius</td>\n      <td></td>\n      <td>x</td>\n      <td></td>\n    </tr>\n  </tbody> </table>",
          related = {
            "lovr.conf"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "spatializer",
                  type = "string",
                  description = "The name of the active spatializer."
                }
              }
            }
          }
        },
        {
          name = "getVolume",
          tag = "listener",
          summary = "Get the master volume.",
          description = "Returns the master volume.  All audio sent to the playback device has its volume multiplied by this factor.",
          key = "lovr.audio.getVolume",
          module = "lovr.audio",
          notes = "The default volume is 1.0 (0 dB).",
          variants = {
            {
              arguments = {
                {
                  name = "units",
                  type = "VolumeUnit",
                  description = "The units to return (linear or db).",
                  default = "'linear'"
                }
              },
              returns = {
                {
                  name = "volume",
                  type = "number",
                  description = "The master volume."
                }
              }
            }
          }
        },
        {
          name = "isStarted",
          tag = "devices",
          summary = "Check if an audio device is started.",
          description = "Returns whether an audio device is started.",
          key = "lovr.audio.isStarted",
          module = "lovr.audio",
          related = {
            "lovr.audio.start",
            "lovr.audio.stop"
          },
          variants = {
            {
              arguments = {
                {
                  name = "type",
                  type = "AudioType",
                  description = "The type of device to check.",
                  default = "'playback'"
                }
              },
              returns = {
                {
                  name = "started",
                  type = "boolean",
                  description = "Whether the device is active."
                }
              }
            }
          }
        },
        {
          name = "newSource",
          tag = "sources",
          summary = "Create a new Source.",
          description = "Creates a new Source from an ogg, wav, or mp3 file.",
          key = "lovr.audio.newSource",
          module = "lovr.audio",
          examples = {
            {
              code = "function lovr.load()\n  sandstorm = lovr.audio.newSource('darude.ogg', {\n    decode = false,\n    effects = { 'spatialization', attenuation = false, reverb = true }\n  })\n\n  sandstorm:play()\nend"
            }
          },
          related = {
            "Source:clone"
          },
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The filename of the sound to load."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Optional options.",
                  table = {
                    {
                      name = "decode",
                      type = "boolean",
                      description = "Whether to immediately decode compressed sounds.",
                      default = "false"
                    },
                    {
                      name = "effects",
                      type = "table",
                      description = "A table of `Effect`s to enable.  Keys can be integers (list) or effect names (map), or a combination of both.  The special value `false` can be used to completely disable effects, bypassing the spatializer entirely and throwing an error when trying to enable effects.  `true` will enable all effects.",
                      default = "true"
                    }
                  }
                }
              },
              returns = {
                {
                  name = "source",
                  type = "Source",
                  description = "The new Source."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The Blob containing the Source data."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Optional options.",
                  table = {
                    {
                      name = "decode",
                      type = "boolean",
                      description = "Whether to immediately decode compressed sounds.",
                      default = "false"
                    },
                    {
                      name = "effects",
                      type = "table",
                      description = "A table of `Effect`s to enable.  Keys can be integers (list) or effect names (map), or a combination of both.  The special value `false` can be used to completely disable effects, bypassing the spatializer entirely and throwing an error when trying to enable effects.  `true` will enable all effects.",
                      default = "true"
                    }
                  }
                }
              },
              returns = {
                {
                  name = "source",
                  type = "Source",
                  description = "The new Source."
                }
              }
            },
            {
              arguments = {
                {
                  name = "sound",
                  type = "Sound",
                  description = "The Sound containing raw audio samples to play."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Optional options.",
                  table = {
                    {
                      name = "decode",
                      type = "boolean",
                      description = "Whether to immediately decode compressed sounds.",
                      default = "false"
                    },
                    {
                      name = "effects",
                      type = "table",
                      description = "A table of `Effect`s to enable.  Keys can be integers (list) or effect names (map), or a combination of both.  The special value `false` can be used to completely disable effects, bypassing the spatializer entirely and throwing an error when trying to enable effects.  `true` will enable all effects.",
                      default = "true"
                    }
                  }
                }
              },
              returns = {
                {
                  name = "source",
                  type = "Source",
                  description = "The new Source."
                }
              }
            }
          }
        },
        {
          name = "setAbsorption",
          summary = "Set the absorption coefficients.",
          description = "Sets the global air absorption coefficients for the medium.  This affects Sources that have the `absorption` effect enabled, causing audio volume to drop off with distance as it is absorbed by the medium it's traveling through (air, water, etc.).  The difference between absorption and falloff is that absorption is more subtle and is frequency-dependent, so higher-frequency bands can get absorbed more quickly than lower ones.  This can be used to apply \"underwater\" effects and stuff.",
          key = "lovr.audio.setAbsorption",
          module = "lovr.audio",
          notes = "Absorption is currently only supported by the phonon spatializer.\n\nThe frequency bands correspond to `400Hz`, `2.5KHz`, and `15KHz`.\n\nThe default coefficients are `.0002`, `.0017`, and `.0182` for low, mid, and high.",
          variants = {
            {
              arguments = {
                {
                  name = "low",
                  type = "number",
                  description = "The absorption coefficient for the low frequency band."
                },
                {
                  name = "mid",
                  type = "number",
                  description = "The absorption coefficient for the mid frequency band."
                },
                {
                  name = "high",
                  type = "number",
                  description = "The absorption coefficient for the high frequency band."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "setDevice",
          tag = "devices",
          summary = "Switch audio devices.",
          description = "Switches either the playback or capture device to a new one.\n\nIf a device for the given type is already active, it will be stopped and destroyed.  The new device will not be started automatically, use `lovr.audio.start` to start it.\n\nA device id (previously retrieved using `lovr.audio.getDevices`) can be given to use a specific audio device, or `nil` can be used for the id to use the default audio device.\n\nA sink can be also be provided when changing the device.  A sink is an audio stream (`Sound` object with a `stream` type) that will receive all audio samples played (for playback) or all audio samples captured (for capture).  When an audio device with a sink is started, be sure to periodically call `Sound:read` on the sink to read audio samples from it, otherwise it will overflow and discard old data.  The sink can have any format, data will be converted as needed. Using a sink for the playback device will reduce performance, but this isn't the case for capture devices.\n\nAudio devices can be started in `shared` or `exclusive` mode.  Exclusive devices may have lower latency than shared devices, but there's a higher chance that requesting exclusive access to an audio device will fail (either because it isn't supported or allowed).  One strategy is to first try the device in exclusive mode, switching to shared if it doesn't work.",
          key = "lovr.audio.setDevice",
          module = "lovr.audio",
          related = {
            "lovr.audio.getDevices",
            "lovr.audio.start",
            "lovr.audio.stop"
          },
          variants = {
            {
              arguments = {
                {
                  name = "type",
                  type = "AudioType",
                  description = "The device to switch.",
                  default = "'playback'"
                },
                {
                  name = "id",
                  type = "userdata",
                  description = "The id of the device to use, or `nil` to use the default device.",
                  default = "nil"
                },
                {
                  name = "sink",
                  type = "Sound",
                  description = "An optional audio stream to use as a sink for the device.",
                  default = "nil"
                },
                {
                  name = "mode",
                  type = "AudioShareMode",
                  description = "The sharing mode for the device.",
                  default = "shared"
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether creating the audio device succeeded."
                }
              }
            }
          }
        },
        {
          name = "setGeometry",
          tag = "listener",
          summary = "Set the geometry for audio effects.",
          description = "Sets a mesh of triangles to use for modeling audio effects, using a table of vertices or a Model.  When the appropriate effects are enabled, audio from `Source` objects will correctly be occluded by walls and bounce around to create realistic reverb.\n\nAn optional `AudioMaterial` may be provided to specify the acoustic properties of the geometry.",
          key = "lovr.audio.setGeometry",
          module = "lovr.audio",
          notes = "This is currently only supported/used by the `phonon` spatializer.\n\nThe `Effect`s that use geometry are:\n\n- `occlusion`\n- `reverb`\n- `transmission`\n\nIf an existing geometry has been set, this function will replace it.\n\nThe triangles must use counterclockwise winding.",
          related = {
            "lovr.audio.getSpatializer",
            "Source:setEffectEnabled"
          },
          variants = {
            {
              arguments = {
                {
                  name = "vertices",
                  type = "table",
                  description = "A flat table of vertices.  Each vertex is 3 numbers representing its x, y, and z position. The units used for audio coordinates are up to you, but meters are recommended."
                },
                {
                  name = "indices",
                  type = "table",
                  description = "A list of indices, indicating how the vertices are connected into triangles.  Indices are 1-indexed and are 32 bits (they can be bigger than 65535)."
                },
                {
                  name = "material",
                  type = "AudioMaterial",
                  description = "The acoustic material to use.",
                  default = "'generic'"
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether audio geometry is supported by the current spatializer and the geometry was loaded successfully."
                }
              }
            },
            {
              arguments = {
                {
                  name = "model",
                  type = "Model",
                  description = "A model to use for the audio geometry."
                },
                {
                  name = "material",
                  type = "AudioMaterial",
                  description = "The acoustic material to use.",
                  default = "'generic'"
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether audio geometry is supported by the current spatializer and the geometry was loaded successfully."
                }
              }
            }
          }
        },
        {
          name = "setOrientation",
          tag = "listener",
          summary = "Set the orientation of the listener.",
          description = "Sets the orientation of the virtual audio listener in angle/axis representation.",
          key = "lovr.audio.setOrientation",
          module = "lovr.audio",
          variants = {
            {
              arguments = {
                {
                  name = "angle",
                  type = "number",
                  description = "The number of radians the listener should be rotated around its rotation axis."
                },
                {
                  name = "ax",
                  type = "number",
                  description = "The x component of the axis of rotation."
                },
                {
                  name = "ay",
                  type = "number",
                  description = "The y component of the axis of rotation."
                },
                {
                  name = "az",
                  type = "number",
                  description = "The z component of the axis of rotation."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "setPose",
          tag = "listener",
          summary = "Set the pose of the listener.",
          description = "Sets the position and orientation of the virtual audio listener.",
          key = "lovr.audio.setPose",
          module = "lovr.audio",
          related = {
            "lovr.audio.setPosition",
            "lovr.audio.setOrientation",
            "Source:setPose"
          },
          variants = {
            {
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The x position of the listener, in meters."
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y position of the listener, in meters."
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z position of the listener, in meters."
                },
                {
                  name = "angle",
                  type = "number",
                  description = "The number of radians the listener is rotated around its axis of rotation."
                },
                {
                  name = "ax",
                  type = "number",
                  description = "The x component of the axis of rotation."
                },
                {
                  name = "ay",
                  type = "number",
                  description = "The y component of the axis of rotation."
                },
                {
                  name = "az",
                  type = "number",
                  description = "The z component of the axis of rotation."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "setPosition",
          tag = "listener",
          summary = "Set the position of the listener.",
          description = "Sets the position of the virtual audio listener, in meters.",
          key = "lovr.audio.setPosition",
          module = "lovr.audio",
          variants = {
            {
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The x position of the listener."
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y position of the listener."
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z position of the listener."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "setVolume",
          tag = "listener",
          summary = "Set the master volume.",
          description = "Sets the master volume.  All audio sent to the playback device has its volume multiplied by this factor.",
          key = "lovr.audio.setVolume",
          module = "lovr.audio",
          notes = "The volume will be clamped to a 0-1 range (0 dB).",
          variants = {
            {
              arguments = {
                {
                  name = "volume",
                  type = "number",
                  description = "The master volume."
                },
                {
                  name = "units",
                  type = "VolumeUnit",
                  description = "The units of the value.",
                  default = "'linear'"
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "start",
          tag = "devices",
          summary = "Start an audio device.",
          description = "Starts the active playback or capture device.  By default the playback device is initialized and started, but this can be controlled using the `t.audio.start` flag in `lovr.conf`.",
          key = "lovr.audio.start",
          module = "lovr.audio",
          notes = "Starting an audio device may fail if:\n\n- The device is already started\n- No device was initialized with `lovr.audio.setDevice`\n- Lack of `audiocapture` permission on Android (see `lovr.system.requestPermission`)\n- Some other problem accessing the audio device",
          related = {
            "lovr.audio.getDevices",
            "lovr.audio.setDevice",
            "lovr.audio.stop",
            "lovr.audio.isStarted",
            "lovr.system.requestPermission",
            "lovr.permission"
          },
          variants = {
            {
              arguments = {
                {
                  name = "type",
                  type = "AudioType",
                  description = "The type of device to start.",
                  default = "'playback'"
                }
              },
              returns = {
                {
                  name = "started",
                  type = "boolean",
                  description = "Whether the device was successfully started."
                }
              }
            }
          }
        },
        {
          name = "stop",
          tag = "devices",
          summary = "Stop an audio device.",
          description = "Stops the active playback or capture device.  This may fail if:\n\n- The device is not started\n- No device was initialized with `lovr.audio.setDevice`",
          key = "lovr.audio.stop",
          module = "lovr.audio",
          notes = "Switching devices with `lovr.audio.setDevice` will stop the existing one.",
          related = {
            "lovr.audio.getDevices",
            "lovr.audio.setDevice",
            "lovr.audio.start",
            "lovr.audio.isStarted"
          },
          variants = {
            {
              arguments = {
                {
                  name = "type",
                  type = "AudioType",
                  description = "The type of device to stop.",
                  default = "'playback'"
                }
              },
              returns = {
                {
                  name = "stopped",
                  type = "boolean",
                  description = "Whether the device was successfully stopped."
                }
              }
            }
          }
        }
      },
      enums = {
        {
          name = "AudioMaterial",
          summary = "Different types of audio materials.",
          description = "Different types of audio material presets, for use with `lovr.audio.setGeometry`.",
          key = "AudioMaterial",
          module = "lovr.audio",
          values = {
            {
              name = "generic",
              description = "Generic default audio material."
            },
            {
              name = "brick",
              description = "Brick."
            },
            {
              name = "carpet",
              description = "Carpet."
            },
            {
              name = "ceramic",
              description = "Ceramic."
            },
            {
              name = "concrete",
              description = "Concrete."
            },
            {
              name = "glass",
              descripion = "Glass."
            },
            {
              name = "gravel",
              descripion = "Gravel."
            },
            {
              name = "metal",
              descripion = "Metal."
            },
            {
              name = "plaster",
              descripion = "Plaster."
            },
            {
              name = "rock",
              descripion = "Rock."
            },
            {
              name = "wood",
              descripion = "Wood."
            }
          }
        },
        {
          name = "AudioShareMode",
          summary = "How audio devices are shared on the system.",
          description = "Audio devices can be created in shared mode or exclusive mode.  In exclusive mode, the audio device is the only one active on the system, which gives better performance and lower latency. However, exclusive devices aren't always supported and might not be allowed, so there is a higher chance that creating one will fail.",
          key = "AudioShareMode",
          module = "lovr.audio",
          related = {
            "lovr.audio.setDevice"
          },
          values = {
            {
              name = "shared",
              description = "Shared mode."
            },
            {
              name = "exclusive",
              description = "Exclusive mode."
            }
          }
        },
        {
          name = "AudioType",
          summary = "Different types of audio devices",
          description = "When referencing audio devices, this indicates whether it's the playback or capture device.",
          key = "AudioType",
          module = "lovr.audio",
          related = {
            "lovr.audio.getDevices",
            "lovr.audio.setDevice",
            "lovr.audio.start",
            "lovr.audio.stop",
            "lovr.audio.isStarted"
          },
          values = {
            {
              name = "playback",
              description = "The playback device (speakers, headphones)."
            },
            {
              name = "capture",
              description = "The capture device (microphone)."
            }
          }
        },
        {
          name = "Effect",
          summary = "Different types of Source effects.",
          description = "Different types of effects that can be applied with `Source:setEffectEnabled`.",
          key = "Effect",
          module = "lovr.audio",
          values = {
            {
              name = "absorption",
              description = "Models absorption as sound travels through the air, water, etc."
            },
            {
              name = "falloff",
              description = "Decreases audio volume with distance (1 / max(distance, 1))."
            },
            {
              name = "occlusion",
              description = "Causes audio to drop off when the Source is occluded by geometry."
            },
            {
              name = "reverb",
              description = "Models reverb caused by audio bouncing off of geometry."
            },
            {
              name = "spatialization",
              description = "Spatializes the Source using either simple panning or an HRTF."
            },
            {
              name = "transmission",
              description = "Causes audio to be heard through walls when occluded, based on audio materials."
            }
          },
          notes = "The active spatializer will determine which effects are supported.  If an unsupported effect is enabled on a Source, no error will be reported.  Instead, it will be silently ignored.\n\nSee `lovr.audio.getSpatializer` for a table of the supported effects for each spatializer."
        },
        {
          name = "TimeUnit",
          summary = "Time units for sound samples.",
          description = "When figuring out how long a Source is or seeking to a specific position in the sound file, units can be expressed in terms of seconds or in terms of frames.  A frame is one set of samples for each channel (one sample for mono, two samples for stereo).",
          key = "TimeUnit",
          module = "lovr.audio",
          values = {
            {
              name = "seconds",
              description = "Seconds."
            },
            {
              name = "frames",
              description = "Frames."
            }
          }
        },
        {
          name = "VolumeUnit",
          summary = "Different units of volume.",
          description = "When accessing the volume of Sources or the audio listener, this can be done in linear units with a 0 to 1 range, or in decibels with a range of -∞ to 0.",
          key = "VolumeUnit",
          module = "lovr.audio",
          values = {
            {
              name = "linear",
              description = "Linear volume range."
            },
            {
              name = "db",
              description = "Decibels."
            }
          }
        }
      },
      sections = {
        {
          name = "Sources",
          tag = "sources",
          description = "Sources are objects that represent a single sound instance."
        },
        {
          name = "Listener",
          tag = "listener",
          description = "The listener is a virtual object in 3D space that \"hears\" all the sounds that are playing. It can be positioned and oriented in 3D space, which controls how Sources in the world are heard.  Usually this would be locked to the headset pose."
        },
        {
          name = "Devices",
          tag = "devices",
          description = "It's possible to list the available audio devices on the system, and pick a specific device to use for either playback or capture.  Devices can also be manually started and stopped. Other useful features of `lovr.audio.setDevice` include the ability to stream all audio data to a custom sink and the option to create a device in exclusive mode for higher performance. By default, the default playback device is automatically initialized and started, but this can be configured using `lovr.conf`."
        }
      }
    },
    {
      name = "data",
      tag = "modules",
      summary = "Exposes low level functions for working with data.",
      description = "The `lovr.data` module provides functions for accessing underlying data representations for several LÖVR objects.",
      key = "lovr.data",
      objects = {
        {
          name = "Blob",
          summary = "A chunk of binary data.",
          description = "A Blob is an object that holds binary data.  It can be passed to most functions that take filename arguments, like `lovr.graphics.newModel` or `lovr.audio.newSource`.  Blobs aren't usually necessary for simple projects, but they can be really helpful if:\n\n- You need to work with low level binary data, potentially using the LuaJIT FFI for increased\n  performance.\n- You are working with data that isn't stored as a file, such as programmatically generated data\n  or a string from a network request.\n- You want to load data from a file once and then use it to create many different objects.\n\nA Blob's size cannot be changed once it is created.",
          key = "Blob",
          module = "lovr.data",
          constructors = {
            "lovr.data.newBlob",
            "lovr.filesystem.newBlob"
          },
          methods = {
            {
              name = "getName",
              summary = "Get the label of the Blob.",
              description = "Returns the filename the Blob was loaded from, or the custom name given to it when it was created.  This label is also used in error messages.",
              key = "Blob:getName",
              module = "lovr.data",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the Blob."
                    }
                  }
                }
              }
            },
            {
              name = "getPointer",
              summary = "Get a raw pointer to the Blob's data.",
              description = "Returns a raw pointer to the Blob's data.  This can be used to interface with other C libraries using the LuaJIT FFI.  Use this only if you know what you're doing!",
              key = "Blob:getPointer",
              module = "lovr.data",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "pointer",
                      type = "userdata",
                      description = "A pointer to the data."
                    }
                  }
                }
              }
            },
            {
              name = "getSize",
              summary = "Get the size of the Blob's data.",
              description = "Returns the size of the Blob's contents, in bytes.",
              key = "Blob:getSize",
              module = "lovr.data",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "bytes",
                      type = "number",
                      description = "The size of the Blob, in bytes."
                    }
                  }
                }
              }
            },
            {
              name = "getString",
              summary = "Get the Blob's contents as a string.",
              description = "Returns a binary string containing the Blob's data.",
              key = "Blob:getString",
              module = "lovr.data",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "data",
                      type = "string",
                      description = "The Blob's data."
                    }
                  }
                }
              },
              examples = {
                {
                  description = "Manually copy a file using Blobs:",
                  code = "blob = lovr.filesystem.newBlob('image.png')\nlovr.filesystem.write('copy.png', blob:getString())"
                }
              }
            }
          }
        },
        {
          name = "Image",
          summary = "An object that stores pixel data for Textures.",
          description = "An Image stores raw 2D pixel info for `Texture`s.  It has a width, height, and format.  The Image can be initialized with the contents of an image file or it can be created with uninitialized contents.  The supported image formats are `png`, `jpg`, `hdr`, `dds`, `ktx`, and `astc`.\n\nUsually you can just use Textures, but Image can be useful if you want to manipulate individual pixels, load Textures in a background thread, or use the FFI to efficiently access the raw image data.",
          key = "Image",
          module = "lovr.data",
          constructors = {
            "lovr.data.newImage"
          },
          methods = {
            {
              name = "encode",
              summary = "Encode the Image as png.",
              description = "Encodes the Image to an uncompressed png.  This intended mainly for debugging.",
              key = "Image:encode",
              module = "lovr.data",
              related = {
                "lovr.filesystem.write"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "A new Blob containing the PNG image data."
                    }
                  }
                }
              }
            },
            {
              name = "getBlob",
              summary = "Get the bytes backing this Image as a `Blob`.",
              description = "Returns a Blob containing the raw bytes of the Image.",
              key = "Image:getBlob",
              module = "lovr.data",
              related = {
                "Blob:getPointer",
                "Sound:getBlob"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "The Blob instance containing the bytes for the `Image`."
                    }
                  }
                }
              }
            },
            {
              name = "getDimensions",
              summary = "Get the dimensions of the Image.",
              description = "Returns the dimensions of the Image, in pixels.",
              key = "Image:getDimensions",
              module = "lovr.data",
              related = {
                "Image:getWidth",
                "Image:getHeight",
                "Texture:getDimensions"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the Image, in pixels."
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the Image, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getFormat",
              summary = "Get the pixel format of the Image.",
              description = "Returns the format of the Image.",
              key = "Image:getFormat",
              module = "lovr.data",
              related = {
                "TextureFormat",
                "Texture:getFormat"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "format",
                      type = "TextureFormat",
                      description = "The format of the pixels in the Image."
                    }
                  }
                }
              }
            },
            {
              name = "getHeight",
              summary = "Get the height of the Image.",
              description = "Returns the height of the Image, in pixels.",
              key = "Image:getHeight",
              module = "lovr.data",
              related = {
                "Image:getWidth",
                "Image:getDimensions",
                "Texture:getHeight"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the Image, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getPixel",
              summary = "Get the value of a pixel of the Image.",
              description = "Returns the value of a pixel of the Image.",
              key = "Image:getPixel",
              module = "lovr.data",
              notes = "The following texture formats are supported: `rgba`, `rgb`, `r32f`, `rg32f`, and `rgba32f`.",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the pixel to get (0-indexed)."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the pixel to get (0-indexed)."
                    }
                  },
                  returns = {
                    {
                      name = "r",
                      type = "number",
                      description = "The red component of the pixel, from 0.0 to 1.0."
                    },
                    {
                      name = "g",
                      type = "number",
                      description = "The green component of the pixel, from 0.0 to 1.0."
                    },
                    {
                      name = "b",
                      type = "number",
                      description = "The blue component of the pixel, from 0.0 to 1.0."
                    },
                    {
                      name = "a",
                      type = "number",
                      description = "The alpha component of the pixel, from 0.0 to 1.0."
                    }
                  }
                }
              },
              related = {
                "Image:setPixel",
                "Texture:replacePixels",
                "TextureFormat"
              }
            },
            {
              name = "getWidth",
              summary = "Get the width of the Image.",
              description = "Returns the width of the Image, in pixels.",
              key = "Image:getWidth",
              module = "lovr.data",
              related = {
                "Image:getHeight",
                "Image:getDimensions",
                "Texture:getWidth"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the Image, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "paste",
              summary = "Copy pixels from another Image to this one.",
              description = "Copies a rectangle of pixels from one Image to this one.",
              key = "Image:paste",
              module = "lovr.data",
              notes = "The two Images must have the same pixel format.\n\nCompressed images cannot be copied.\n\nThe rectangle cannot go outside the dimensions of the source or destination textures.",
              variants = {
                {
                  arguments = {
                    {
                      name = "source",
                      type = "Image",
                      description = "The Image to copy pixels from."
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate to paste to (0-indexed).",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate to paste to (0-indexed).",
                      default = "0"
                    },
                    {
                      name = "fromX",
                      type = "number",
                      description = "The x coordinate in the source to paste from (0-indexed).",
                      default = "0"
                    },
                    {
                      name = "fromY",
                      type = "number",
                      description = "The y coordinate in the source to paste from (0-indexed).",
                      default = "0"
                    },
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the region to copy.",
                      default = "source:getWidth()"
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the region to copy.",
                      default = "source:getHeight()"
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "Texture:replacePixels",
                "Image:getPixel",
                "Image:setPixel"
              }
            },
            {
              name = "setPixel",
              summary = "Set the value of a pixel of the Image.",
              description = "Sets the value of a pixel of the Image.",
              key = "Image:setPixel",
              module = "lovr.data",
              notes = "The following texture formats are supported: `rgba`, `rgb`, `r32f`, `rg32f`, and `rgba32f`.",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the pixel to set (0-indexed)."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the pixel to set (0-indexed)."
                    },
                    {
                      name = "r",
                      type = "number",
                      description = "The red component of the pixel, from 0.0 to 1.0."
                    },
                    {
                      name = "g",
                      type = "number",
                      description = "The green component of the pixel, from 0.0 to 1.0."
                    },
                    {
                      name = "b",
                      type = "number",
                      description = "The blue component of the pixel, from 0.0 to 1.0."
                    },
                    {
                      name = "a",
                      type = "number",
                      description = "The alpha component of the pixel, from 0.0 to 1.0.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "Image:getPixel",
                "Texture:replacePixels",
                "TextureFormat"
              }
            }
          }
        },
        {
          name = "ModelData",
          summary = "An object that loads and stores data for 3D models.",
          description = "A ModelData is a container object that loads and holds data contained in 3D model files.  This can include a variety of things like the node structure of the asset, the vertex data it contains, contains, the `Image` and `Material` properties, and any included animations.\n\nThe current supported formats are OBJ, glTF, and STL.\n\nUsually you can just load a `Model` directly, but using a `ModelData` can be helpful if you want to load models in a thread or access more low-level information about the Model.",
          key = "ModelData",
          module = "lovr.data",
          constructors = {
            "lovr.data.newModelData"
          },
          methods = {
            {
              name = "getAnimationChannelCount",
              summary = "Get the number of channels in an animation.",
              description = "Returns the number of channels in an animation.\n\nA channel is a set of keyframes for a single property of a node.",
              key = "ModelData:getAnimationChannelCount",
              module = "lovr.data",
              related = {
                "ModelData:getAnimationNode",
                "ModelData:getAnimationProperty"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of an animation."
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of channels in the animation."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of an animation."
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of channels in the animation."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationCount",
              summary = "Get the number of animations in the model.",
              description = "Returns the number of animations in the model.",
              key = "ModelData:getAnimationCount",
              module = "lovr.data",
              related = {
                "Model:getAnimationCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of animations in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationDuration",
              summary = "Get the duration of an animation.",
              description = "Returns the duration of an animation.",
              key = "ModelData:getAnimationDuration",
              module = "lovr.data",
              notes = "The duration of the animation is calculated as the latest timestamp of all of its channels.",
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the animation."
                    }
                  },
                  returns = {
                    {
                      name = "duration",
                      type = "number",
                      description = "The duration of the animation, in seconds."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the animation."
                    }
                  },
                  returns = {
                    {
                      name = "duration",
                      type = "number",
                      description = "The duration of the animation, in seconds."
                    }
                  }
                }
              },
              related = {
                "Model:getAnimationDuration"
              }
            },
            {
              name = "getAnimationKeyframe",
              summary = "Get a keyframe in a channel of an animation.",
              description = "Returns a single keyframe in a channel of an animation.",
              key = "ModelData:getAnimationKeyframe",
              module = "lovr.data",
              related = {
                "ModelData:getAnimationSmoothMode",
                "ModelData:getAnimationKeyframeCount"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    },
                    {
                      name = "keyframe",
                      type = "number",
                      description = "The index of a keyframe in the channel."
                    }
                  },
                  returns = {
                    {
                      name = "time",
                      type = "number",
                      description = "The timestamp of the keyframe."
                    },
                    {
                      name = "...",
                      type = "number",
                      description = "The data for the keyframe (either 3 or 4 numbers depending on the property)."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    },
                    {
                      name = "keyframe",
                      type = "number",
                      description = "The index of a keyframe in the channel."
                    }
                  },
                  returns = {
                    {
                      name = "time",
                      type = "number",
                      description = "The timestamp of the keyframe."
                    },
                    {
                      name = "...",
                      type = "number",
                      description = "The data for the keyframe (either 3 or 4 numbers depending on the property)."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationKeyframeCount",
              summary = "Get the number of keyframes in a channel of an animation.",
              description = "Returns the number of keyframes in a channel of an animation.",
              key = "ModelData:getAnimationKeyframeCount",
              module = "lovr.data",
              related = {
                "ModelData:getAnimationSmoothMode",
                "ModelData:getAnimationKeyframe"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of keyframes in the channel."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of keyframes in the channel."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationName",
              summary = "Get the name of an animation.",
              description = "Returns the name of an animation.",
              key = "ModelData:getAnimationName",
              module = "lovr.data",
              notes = "If the animation does not have a name, this function returns `nil`.",
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the animation."
                    }
                  },
                  returns = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the animation."
                    }
                  }
                }
              },
              related = {
                "Model:getAnimationName"
              }
            },
            {
              name = "getAnimationNode",
              summary = "Get the node targeted by the channel of an animation.",
              description = "Returns the index of a node targeted by an animation's channel.",
              key = "ModelData:getAnimationNode",
              module = "lovr.data",
              related = {
                "ModelData:getAnimationNode",
                "ModelData:getAnimationProperty"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    }
                  },
                  returns = {
                    {
                      name = "node",
                      type = "number",
                      description = "The index of the node targeted by the channel."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    }
                  },
                  returns = {
                    {
                      name = "node",
                      type = "number",
                      description = "The index of the node targeted by the channel."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationProperty",
              summary = "Get the property targeted by the channel of an animation.",
              description = "Returns the property targeted by an animation's channel.",
              key = "ModelData:getAnimationProperty",
              module = "lovr.data",
              related = {
                "ModelData:getAnimationNode",
                "ModelData:getAnimationProperty",
                "ModelData:getAnimationSmoothMode"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    }
                  },
                  returns = {
                    {
                      name = "property",
                      type = "AnimationProperty",
                      description = "The property (translation, rotation, scale) affected by the keyframes."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    }
                  },
                  returns = {
                    {
                      name = "property",
                      type = "AnimationProperty",
                      description = "The property (translation, rotation, scale) affected by the keyframes."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationSmoothMode",
              summary = "Get the smooth mode of a channel in an animation.",
              description = "Returns the smooth mode of a channel in an animation.",
              key = "ModelData:getAnimationSmoothMode",
              module = "lovr.data",
              related = {
                "ModelData:getAnimationNode",
                "ModelData:getAnimationProperty"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    }
                  },
                  returns = {
                    {
                      name = "smooth",
                      type = "SmoothMode",
                      description = "The smooth mode of the keyframes."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of an animation."
                    },
                    {
                      name = "channel",
                      type = "number",
                      description = "The index of a channel in the animation."
                    }
                  },
                  returns = {
                    {
                      name = "smooth",
                      type = "SmoothMode",
                      description = "The smooth mode of the keyframes."
                    }
                  }
                }
              }
            },
            {
              name = "getBlob",
              summary = "Get a Blob in the model.",
              description = "Returns one of the Blobs in the model, by index.",
              key = "ModelData:getBlob",
              module = "lovr.data",
              related = {
                "ModelData:getBlobCount",
                "ModelData:getImage"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the Blob to get."
                    }
                  },
                  returns = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "The Blob object."
                    }
                  }
                }
              }
            },
            {
              name = "getBlobCount",
              summary = "Get the number of Blobs stored in the model.",
              description = "Returns the number of Blobs in the model.",
              key = "ModelData:getBlobCount",
              module = "lovr.data",
              related = {
                "ModelData:getBlob",
                "ModelData:getImageCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of Blobs in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getBoundingBox",
              summary = "Get the bounding box of the model.",
              description = "Returns the 6 values of the model's axis-aligned bounding box.",
              key = "ModelData:getBoundingBox",
              module = "lovr.data",
              related = {
                "ModelData:getWidth",
                "ModelData:getHeight",
                "ModelData:getDepth",
                "ModelData:getDimensions",
                "ModelData:getCenter",
                "ModelData:getBoundingSphere",
                "Model:getBoundingBox"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "minx",
                      type = "number",
                      description = "The minimum x coordinate of the vertices in the model."
                    },
                    {
                      name = "maxx",
                      type = "number",
                      description = "The maximum x coordinate of the vertices in the model."
                    },
                    {
                      name = "miny",
                      type = "number",
                      description = "The minimum y coordinate of the vertices in the model."
                    },
                    {
                      name = "maxy",
                      type = "number",
                      description = "The maximum y coordinate of the vertices in the model."
                    },
                    {
                      name = "minz",
                      type = "number",
                      description = "The minimum z coordinate of the vertices in the model."
                    },
                    {
                      name = "maxz",
                      type = "number",
                      description = "The maximum z coordinate of the vertices in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getBoundingSphere",
              summary = "Get the bounding sphere of the model.",
              description = "Returns a sphere approximately enclosing the vertices in the model.",
              key = "ModelData:getBoundingSphere",
              module = "lovr.data",
              related = {
                "ModelData:getWidth",
                "ModelData:getHeight",
                "ModelData:getDepth",
                "ModelData:getDimensions",
                "ModelData:getCenter",
                "ModelData:getBoundingBox",
                "Model:getBoundingSphere"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the position of the sphere."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the position of the sphere."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the position of the sphere."
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the bounding sphere."
                    }
                  }
                }
              }
            },
            {
              name = "getCenter",
              summary = "Get the center of the model's bounding box.",
              description = "Returns the center of the model's axis-aligned bounding box, relative to the model's origin.",
              key = "ModelData:getCenter",
              module = "lovr.data",
              related = {
                "ModelData:getWidth",
                "ModelData:getHeight",
                "ModelData:getDepth",
                "ModelData:getDimensions",
                "ModelData:getBoundingBox",
                "Model:getCenter"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x offset of the center of the bounding box."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y offset of the center of the bounding box."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z offset of the center of the bounding box."
                    }
                  }
                }
              }
            },
            {
              name = "getDepth",
              summary = "Get the depth of the model.",
              description = "Returns the depth of the model, computed from its axis-aligned bounding box.",
              key = "ModelData:getDepth",
              module = "lovr.data",
              related = {
                "ModelData:getWidth",
                "ModelData:getHeight",
                "ModelData:getDimensions",
                "ModelData:getCenter",
                "ModelData:getBoundingBox",
                "Model:getDepth"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "depth",
                      type = "number",
                      description = "The depth of the model."
                    }
                  }
                }
              }
            },
            {
              name = "getDimensions",
              summary = "Get the dimensions of the model.",
              description = "Returns the width, height, and depth of the model, computed from its axis-aligned bounding box.",
              key = "ModelData:getDimensions",
              module = "lovr.data",
              related = {
                "ModelData:getWidth",
                "ModelData:getHeight",
                "ModelData:getDepth",
                "ModelData:getCenter",
                "ModelData:getBoundingBox",
                "Model:getDimensions"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the model."
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the model."
                    },
                    {
                      name = "depth",
                      type = "number",
                      description = "The depth of the model."
                    }
                  }
                }
              }
            },
            {
              name = "getHeight",
              summary = "Get the height of the model.",
              description = "Returns the height of the model, computed from its axis-aligned bounding box.",
              key = "ModelData:getHeight",
              module = "lovr.data",
              related = {
                "ModelData:getWidth",
                "ModelData:getDepth",
                "ModelData:getDimensions",
                "ModelData:getCenter",
                "ModelData:getBoundingBox",
                "Model:getHeight"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the model."
                    }
                  }
                }
              }
            },
            {
              name = "getImage",
              summary = "Get an Image in the model.",
              description = "Returns one of the Images in the model, by index.",
              key = "ModelData:getImage",
              module = "lovr.data",
              related = {
                "ModelData:getImageCount",
                "ModelData:getBlob"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the Image to get."
                    }
                  },
                  returns = {
                    {
                      name = "image",
                      type = "Image",
                      description = "The Image object."
                    }
                  }
                }
              }
            },
            {
              name = "getImageCount",
              summary = "Get the number of Images stored in the model.",
              description = "Returns the number of Images in the model.",
              key = "ModelData:getImageCount",
              module = "lovr.data",
              related = {
                "ModelData:getImage",
                "ModelData:getBlobCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of Images in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getMaterial",
              summary = "Get the material properties for a material in the model.",
              description = "Returns a table with all of the properties of a material.",
              key = "ModelData:getMaterial",
              module = "lovr.data",
              notes = "All images are optional and may be `nil`.",
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a material."
                    }
                  },
                  returns = {
                    {
                      name = "properties",
                      type = "table",
                      description = "The material properties.",
                      table = {
                        {
                          name = "color",
                          type = "table",
                          description = "The color of the material.  The table contains the `r`, `g`, `b`, and `a` components of the color, from 0 to 1."
                        },
                        {
                          name = "glow",
                          type = "table",
                          description = "The glow color of the material (sometimes called emissive).  The table contains the `r`, `g`, and `b` components of the color from 0 to 1, and a fourth number indicating the strength of the glow."
                        },
                        {
                          name = "uvShift",
                          type = "table",
                          description = "A table with 2 numbers indicating an offset to apply to UVs."
                        },
                        {
                          name = "uvScale",
                          type = "table",
                          description = "A table with 2 numbers indicating a scale to apply to UVs.  By default, shaders apply the UV scale before the UV offset."
                        },
                        {
                          name = "metalness",
                          type = "number",
                          description = "The metalness parameter of the material.  This is typically 0 or 1.  By default, shaders multiply this property with the value from the metalness texture (when present) to get the final metalness used for shading."
                        },
                        {
                          name = "roughness",
                          type = "number",
                          description = "The roughness parameter of the material.  By default, shaders multiply this property with the value from the roughness texture (when present) to get the final roughness used for shading."
                        },
                        {
                          name = "clearcoat",
                          type = "number",
                          description = "The clearcoat parameter of the material."
                        },
                        {
                          name = "clearcoatRoughness",
                          type = "number",
                          description = "The roughness of the clearcoat layer."
                        },
                        {
                          name = "occlusionStrength",
                          type = "number",
                          description = "A number multiplied by the value from the ambient occlusion texture to control how strong the occlusion effect is."
                        },
                        {
                          name = "normalScale",
                          type = "number",
                          description = "A number multiplied by the value from the normal texture to control how strong the normal mapping effect is."
                        },
                        {
                          name = "alphaCutoff",
                          type = "number",
                          description = "If a pixel has an alpha value less than the alpha cutoff, it will be discarded, which prevents it from occluding things behind it.  This is sometimes called \"holepunch\" or \"cutout\" alpha.  It's useful for textures with transparency."
                        },
                        {
                          name = "texture",
                          type = "number",
                          description = "The index of the Image used for the color texture."
                        },
                        {
                          name = "glowTexture",
                          type = "number",
                          description = "The index of the Image used for the glow texture."
                        },
                        {
                          name = "occlusionTexture",
                          type = "number",
                          description = "The index of the Image used for the ambient occlusion texture.  The red channel of the texture is used for ambient occlusion, allowing multiple parameters to use the same texture."
                        },
                        {
                          name = "metalnessTexture",
                          type = "number",
                          description = "The index of the Image used for the metalness texture.  The blue channel of the texture is used for metalness, allowing multiple parameters to use the same texture."
                        },
                        {
                          name = "roughnessTexture",
                          type = "number",
                          description = "The index of the Image to use for the roughness texture.  The green channel of the texture is used for roughness, allowing multiple parameters to use the same texture."
                        },
                        {
                          name = "clearcoatTexture",
                          type = "number",
                          description = "The index of the Image to use for the clearcoat texture.  The red channel of the texture is used for the clearcoat parameter, allowing multiple parameters to use the same texture."
                        },
                        {
                          name = "normalTexture",
                          type = "number",
                          description = "The index of the Image to use for the normal map."
                        }
                      }
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of a material."
                    }
                  },
                  returns = {
                    {
                      name = "properties",
                      type = "table",
                      description = "The material properties.",
                      table = {
                        {
                          name = "color",
                          type = "table",
                          description = "The color of the material.  The table contains the `r`, `g`, `b`, and `a` components of the color, from 0 to 1."
                        },
                        {
                          name = "glow",
                          type = "table",
                          description = "The glow color of the material (sometimes called emissive).  The table contains the `r`, `g`, and `b` components of the color from 0 to 1, and a fourth number indicating the strength of the glow."
                        },
                        {
                          name = "uvShift",
                          type = "table",
                          description = "A table with 2 numbers indicating an offset to apply to UVs."
                        },
                        {
                          name = "uvScale",
                          type = "table",
                          description = "A table with 2 numbers indicating a scale to apply to UVs.  By default, shaders apply the UV scale before the UV offset."
                        },
                        {
                          name = "metalness",
                          type = "number",
                          description = "The metalness parameter of the material.  This is typically 0 or 1.  By default, shaders multiply this property with the value from the metalness texture (when present) to get the final metalness used for shading."
                        },
                        {
                          name = "roughness",
                          type = "number",
                          description = "The roughness parameter of the material.  By default, shaders multiply this property with the value from the roughness texture (when present) to get the final roughness used for shading."
                        },
                        {
                          name = "clearcoat",
                          type = "number",
                          description = "The clearcoat parameter of the material."
                        },
                        {
                          name = "clearcoatRoughness",
                          type = "number",
                          description = "The roughness of the clearcoat layer."
                        },
                        {
                          name = "occlusionStrength",
                          type = "number",
                          description = "A number multiplied by the value from the ambient occlusion texture to control how strong the occlusion effect is."
                        },
                        {
                          name = "normalScale",
                          type = "number",
                          description = "A number multiplied by the value from the normal texture to control how strong the normal mapping effect is."
                        },
                        {
                          name = "alphaCutoff",
                          type = "number",
                          description = "If a pixel has an alpha value less than the alpha cutoff, it will be discarded, which prevents it from occluding things behind it.  This is sometimes called \"holepunch\" or \"cutout\" alpha.  It's useful for textures with transparency."
                        },
                        {
                          name = "texture",
                          type = "number",
                          description = "The index of the Image used for the color texture."
                        },
                        {
                          name = "glowTexture",
                          type = "number",
                          description = "The index of the Image used for the glow texture."
                        },
                        {
                          name = "occlusionTexture",
                          type = "number",
                          description = "The index of the Image used for the ambient occlusion texture.  The red channel of the texture is used for ambient occlusion, allowing multiple parameters to use the same texture."
                        },
                        {
                          name = "metalnessTexture",
                          type = "number",
                          description = "The index of the Image used for the metalness texture.  The blue channel of the texture is used for metalness, allowing multiple parameters to use the same texture."
                        },
                        {
                          name = "roughnessTexture",
                          type = "number",
                          description = "The index of the Image to use for the roughness texture.  The green channel of the texture is used for roughness, allowing multiple parameters to use the same texture."
                        },
                        {
                          name = "clearcoatTexture",
                          type = "number",
                          description = "The index of the Image to use for the clearcoat texture.  The red channel of the texture is used for the clearcoat parameter, allowing multiple parameters to use the same texture."
                        },
                        {
                          name = "normalTexture",
                          type = "number",
                          description = "The index of the Image to use for the normal map."
                        }
                      }
                    }
                  }
                }
              },
              related = {
                "ModelData:getMaterialCount",
                "ModelData:getMeshMaterial",
                "lovr.graphics.newMaterial",
                "Model:getMaterial"
              }
            },
            {
              name = "getMaterialCount",
              summary = "Get the number of materials in the model.",
              description = "Returns the number of materials in the model.",
              key = "ModelData:getMaterialCount",
              module = "lovr.data",
              related = {
                "ModelData:getMaterialName",
                "ModelData:getMeshMaterial",
                "ModelData:getMaterial",
                "Model:getMaterialCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of materials in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getMaterialName",
              summary = "Get the name of a material in the model.",
              description = "Returns the name of a material in the model.",
              key = "ModelData:getMaterialName",
              module = "lovr.data",
              related = {
                "ModelData:getMaterialCount",
                "ModelData:getMeshMaterial",
                "ModelData:getMaterial",
                "Model:getMaterialName"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a material."
                    }
                  },
                  returns = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the material, or nil if the material does not have a name."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshCount",
              summary = "Get the number of meshes in the model.",
              description = "Returns the number of meshes in the model.",
              key = "ModelData:getMeshCount",
              module = "lovr.data",
              related = {
                "ModelData:getNodeMeshes"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of meshes in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshDrawMode",
              summary = "Get the draw mode of a mesh.",
              description = "Returns the draw mode of a mesh.  This controls how its vertices are connected together (points, lines, or triangles).",
              key = "ModelData:getMeshDrawMode",
              module = "lovr.data",
              variants = {
                {
                  arguments = {
                    {
                      name = "mesh",
                      type = "number",
                      description = "The index of a mesh."
                    }
                  },
                  returns = {
                    {
                      name = "mode",
                      type = "DrawMode",
                      description = "The draw mode of the mesh."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshIndex",
              summary = "Get one of the vertex indices in a mesh.",
              description = "Returns one of the vertex indices in a mesh.  If a mesh has vertex indices, they define the order and connectivity of the vertices in the mesh, allowing a vertex to be reused multiple times without duplicating its data.",
              key = "ModelData:getMeshIndex",
              module = "lovr.data",
              related = {
                "ModelData:getMeshIndexFormat",
                "ModelData:getMeshIndexCount",
                "ModelData:getMeshVertex",
                "ModelData:getTriangles"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "mesh",
                      type = "number",
                      description = "The index of a mesh to get the vertex from."
                    },
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a vertex index in the mesh to retrieve."
                    }
                  },
                  returns = {
                    {
                      name = "vertexindex",
                      type = "number",
                      description = "The vertex index.  Like all indices in Lua, this is 1-indexed."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshIndexCount",
              summary = "Get the number of vertex indices in a mesh.",
              description = "Returns the number of vertex indices in a mesh.  Vertex indices allow for vertices to be reused when defining triangles.",
              key = "ModelData:getMeshIndexCount",
              module = "lovr.data",
              notes = "This may return zero if the mesh does not use indices.",
              variants = {
                {
                  arguments = {
                    {
                      name = "mesh",
                      type = "number",
                      description = "The index of a mesh."
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertex indices in the mesh."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshIndexFormat",
              summary = "Get the data format of vertex indices in a mesh.",
              description = "Returns the data format of vertex indices in a mesh.  If a mesh doesn't use vertex indices, this function returns nil.",
              key = "ModelData:getMeshIndexFormat",
              module = "lovr.data",
              related = {
                "ModelData:getMeshVertexFormat"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "mesh",
                      type = "number",
                      description = "The index of a mesh."
                    }
                  },
                  returns = {
                    {
                      name = "type",
                      type = "AttributeType",
                      description = "The data type of each vertex index (always u16 or u32)."
                    },
                    {
                      name = "blob",
                      type = "number",
                      description = "The index of a Blob in the mesh where the binary data is stored."
                    },
                    {
                      name = "offset",
                      type = "number",
                      description = "A byte offset into the Blob's data where the index data starts."
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The number of bytes between subsequent vertex indices.  Indices are always tightly packed, so this will always be 2 or 4 depending on the data type."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshMaterial",
              summary = "Get the index of the material applied to a mesh.",
              description = "Returns the index of the material applied to a mesh.",
              key = "ModelData:getMeshMaterial",
              module = "lovr.data",
              variants = {
                {
                  arguments = {
                    {
                      name = "mesh",
                      type = "number",
                      description = "The index of a mesh."
                    }
                  },
                  returns = {
                    {
                      name = "material",
                      type = "number",
                      description = "The index of the material applied to the mesh, or nil if the mesh does not have a material."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshVertex",
              summary = "Get the data for a single vertex in a mesh.",
              description = "Returns the data for a single vertex in a mesh.  The data returned depends on the vertex format of a mesh, which is given by `ModelData:getMeshVertexFormat`.",
              key = "ModelData:getMeshVertex",
              module = "lovr.data",
              related = {
                "ModelData:getMeshVertexFormat",
                "ModelData:getMeshVertexCount",
                "ModelData:getMeshIndex",
                "ModelData:getTriangles"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "mesh",
                      type = "number",
                      description = "The index of a mesh to get the vertex from."
                    },
                    {
                      name = "vertex",
                      type = "number",
                      description = "The index of a vertex in the mesh to retrieve."
                    }
                  },
                  returns = {
                    {
                      name = "...",
                      type = "number",
                      description = "The data for all of the attributes of the vertex."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshVertexCount",
              summary = "Get the number of vertices in a mesh.",
              description = "Returns the number of vertices in a mesh.",
              key = "ModelData:getMeshVertexCount",
              module = "lovr.data",
              related = {
                "ModelData:getMeshIndexCount"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "mesh",
                      type = "number",
                      description = "The index of a mesh."
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices in the mesh."
                    }
                  }
                }
              }
            },
            {
              name = "getMeshVertexFormat",
              summary = "Get the vertex format of a mesh.",
              description = "Returns the vertex format of a mesh.  The vertex format defines the properties associated with each vertex (position, color, etc.), including their types and binary data layout.",
              key = "ModelData:getMeshVertexFormat",
              module = "lovr.data",
              notes = "The format is given as a table of vertex attributes.  Each attribute is a table containing the following:\n\n    { name, type, components, blob, offset, stride }\n\n- The `name` will be a `DefaultAttribute`.\n- The `type` will be an `AttributeType`.\n- The `component` count will be 1-4.\n- The `blob` is an index of one of the Blobs in the model (see `ModelData:getBlob`).\n- The `offset` is a byte offset from the start of the Blob where the attribute's data starts.\n- The `stride` is the number of bytes between consecutive values.",
              variants = {
                {
                  arguments = {
                    {
                      name = "mesh",
                      type = "number",
                      description = "The index of a mesh."
                    }
                  },
                  returns = {
                    {
                      name = "format",
                      type = "table",
                      description = "The vertex format of the mesh."
                    }
                  }
                }
              },
              related = {
                "ModelData:getIndexFormat"
              }
            },
            {
              name = "getMetadata",
              summary = "Get extra information from the model file.",
              description = "Returns extra information stored in the model file.  Currently this is only implemented for glTF models and returns the JSON string from the glTF or glb file.  The metadata can be used to get application-specific data or add support for glTF extensions not supported by LÖVR.",
              key = "ModelData:getMetadata",
              module = "lovr.data",
              related = {
                "Model:getMetadata"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "metadata",
                      type = "string",
                      description = "The metadata from the model file."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeChildren",
              summary = "Get the children of a node.",
              description = "Given a parent node, this function returns a table with the indices of its children.",
              key = "ModelData:getNodeChildren",
              module = "lovr.data",
              notes = "If the node does not have any children, this function returns an empty table.",
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the parent node."
                    }
                  },
                  returns = {
                    {
                      name = "children",
                      type = "table",
                      description = "A table containing a node index for each child of the node."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the parent node."
                    }
                  },
                  returns = {
                    {
                      name = "children",
                      type = "table",
                      description = "A table containing a node index for each child of the node."
                    }
                  }
                }
              },
              related = {
                "ModelData:getNodeParent",
                "ModelData:getRootNode",
                "Model:getNodeChildren"
              }
            },
            {
              name = "getNodeCount",
              summary = "Get the number of nodes in the model.",
              description = "Returns the number of nodes in the model.",
              key = "ModelData:getNodeCount",
              module = "lovr.data",
              related = {
                "Model:getNodeCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of nodes in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeMeshes",
              summary = "Get the indices of meshes attached to a node.",
              description = "Returns a table of mesh indices attached to a node.  Meshes define the geometry and materials of a model, as opposed to the nodes which define the transforms and hierarchy.  A node can have multiple meshes, and meshes can be reused in multiple nodes.",
              key = "ModelData:getNodeMeshes",
              module = "lovr.data",
              related = {
                "ModelData:getMeshCount"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "meshes",
                      type = "table",
                      description = "A table with the node's mesh indices."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  },
                  returns = {
                    {
                      name = "meshes",
                      type = "table",
                      description = "A table with the node's mesh indices."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeName",
              summary = "Get the name of a node.",
              description = "Returns the name of a node.",
              key = "ModelData:getNodeName",
              module = "lovr.data",
              notes = "If the node does not have a name, this function returns `nil`.",
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  }
                }
              },
              related = {
                "Model:getNodeName"
              }
            },
            {
              name = "getNodeOrientation",
              summary = "Get the local orientation of a node.",
              description = "Returns local orientation of a node, relative to its parent.",
              key = "ModelData:getNodeOrientation",
              module = "lovr.data",
              related = {
                "ModelData:getNodePosition",
                "ModelData:getNodeScale",
                "ModelData:getNodePose",
                "ModelData:getNodeTransform"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  },
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeParent",
              summary = "Get the parent of a node.",
              description = "Given a child node, this function returns the index of its parent.",
              key = "ModelData:getNodeParent",
              module = "lovr.data",
              related = {
                "ModelData:getNodeChildren",
                "ModelData:getRootNode",
                "Model:getNodeParent"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the child node."
                    }
                  },
                  returns = {
                    {
                      name = "parent",
                      type = "number",
                      description = "The index of the parent."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the child node."
                    }
                  },
                  returns = {
                    {
                      name = "parent",
                      type = "number",
                      description = "The index of the parent."
                    }
                  }
                }
              }
            },
            {
              name = "getNodePose",
              summary = "Get the local pose of a node.",
              description = "Returns local pose (position and orientation) of a node, relative to its parent.",
              key = "ModelData:getNodePose",
              module = "lovr.data",
              related = {
                "ModelData:getNodePosition",
                "ModelData:getNodeOrientation",
                "ModelData:getNodeScale",
                "ModelData:getNodeTransform"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getNodePosition",
              summary = "Get the local position of a node.",
              description = "Returns local position of a node, relative to its parent.",
              key = "ModelData:getNodePosition",
              module = "lovr.data",
              related = {
                "ModelData:getNodeOrientation",
                "ModelData:getNodeScale",
                "ModelData:getNodePose",
                "ModelData:getNodeTransform"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeScale",
              summary = "Get the local scale of a node.",
              description = "Returns local scale of a node, relative to its parent.",
              key = "ModelData:getNodeScale",
              module = "lovr.data",
              related = {
                "ModelData:getNodePosition",
                "ModelData:getNodeOrientation",
                "ModelData:getNodePose",
                "ModelData:getNodeTransform"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "sx",
                      type = "number",
                      description = "The x scale."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y scale."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z scale."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  },
                  returns = {
                    {
                      name = "sx",
                      type = "number",
                      description = "The x scale."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y scale."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z scale."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeSkin",
              summary = "Get the index of the skin used by a node.",
              description = "Returns the index of the skin used by a node.  Skins are collections of joints used for skeletal animation.  A model can have multiple skins, and each node can use at most one skin to drive the animation of its meshes.",
              key = "ModelData:getNodeSkin",
              module = "lovr.data",
              related = {
                "ModelData:getSkinCount"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "skin",
                      type = "number",
                      description = "The index of the node's skin, or nil if the node isn't skeletally animated."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  },
                  returns = {
                    {
                      name = "skin",
                      type = "number",
                      description = "The index of the node's skin, or nil if the node isn't skeletally animated."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeTransform",
              summary = "Get the local transform of a node.",
              description = "Returns local transform (position, orientation, and scale) of a node, relative to its parent.",
              key = "ModelData:getNodeTransform",
              module = "lovr.data",
              notes = "For best results when animating, it's recommended to keep the 3 components of the scale the same.",
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    },
                    {
                      name = "sx",
                      type = "number",
                      description = "The x scale."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y scale."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z scale."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    },
                    {
                      name = "sx",
                      type = "number",
                      description = "The x scale."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y scale."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z scale."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              },
              related = {
                "ModelData:getNodePosition",
                "ModelData:getNodeOrientation",
                "ModelData:getNodeScale",
                "ModelData:getNodePose"
              }
            },
            {
              name = "getRootNode",
              summary = "Get the index of the root node.",
              description = "Returns the index of the model's root node.",
              key = "ModelData:getRootNode",
              module = "lovr.data",
              related = {
                "ModelData:getNodeCount",
                "ModelData:getNodeParent",
                "Model:getRootNode"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "root",
                      type = "number",
                      description = "The index of the root node."
                    }
                  }
                }
              }
            },
            {
              name = "getSkinCount",
              summary = "Get the number of skins in the model.",
              description = "Returns the number of skins in the model.  A skin is a collection of joints targeted by an animation.",
              key = "ModelData:getSkinCount",
              module = "lovr.data",
              notes = "There is currently a maximum of 256 skins.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of skins in the model."
                    }
                  }
                }
              },
              related = {
                "Model:hasJoints"
              }
            },
            {
              name = "getSkinInverseBindMatrix",
              summary = "Get the inverse bind matrix for a joint in the skin.",
              description = "Returns the inverse bind matrix for a joint in the skin.",
              key = "ModelData:getSkinInverseBindMatrix",
              module = "lovr.data",
              variants = {
                {
                  arguments = {
                    {
                      name = "skin",
                      type = "number",
                      description = "The index of a skin."
                    },
                    {
                      name = "joint",
                      type = "number",
                      description = "The index of a joint in the skin."
                    }
                  },
                  returns = {
                    {
                      name = "...",
                      type = "number",
                      description = "The 16 components of the 4x4 inverse bind matrix, in column-major order."
                    }
                  }
                }
              }
            },
            {
              name = "getSkinJoints",
              summary = "Get the joints in a skin.",
              description = "Returns a table with the node indices of the joints in a skin.",
              key = "ModelData:getSkinJoints",
              module = "lovr.data",
              variants = {
                {
                  arguments = {
                    {
                      name = "skin",
                      type = "number",
                      description = "The index of a skin."
                    }
                  },
                  returns = {
                    {
                      name = "joints",
                      type = "table",
                      description = "The joints in the skin."
                    }
                  }
                }
              }
            },
            {
              name = "getTriangleCount",
              summary = "Get the total number of triangles in the model.",
              description = "Returns the total number of triangles in the model.  This count includes meshes that are attached to multiple nodes, and the count corresponds to the triangles returned by `ModelData:getTriangles`.",
              key = "ModelData:getTriangleCount",
              module = "lovr.data",
              related = {
                "ModelData:getTriangles",
                "ModelData:getVertexCount",
                "Model:getTriangleCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The total number of triangles in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getTriangles",
              summary = "Get all the triangles in the model.",
              description = "Returns the data for all triangles in the model.  There are a few differences between this and the mesh-specific functions like `ModelData:getMeshVertex` and `ModelData:getMeshIndex`:\n\n- Only vertex positions are returned, not other vertex attributes.\n- Positions are relative to the origin of the whole model, instead of local to a node.\n- If a mesh is attached to more than one node, its vertices will be in the table multiple times.\n- Vertex indices will be relative to the whole triangle list instead of a mesh.",
              key = "ModelData:getTriangles",
              module = "lovr.data",
              notes = "After this function is called on a ModelData once, the result is cached.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "vertices",
                      type = "table",
                      description = "The triangle vertex positions, returned as a flat (non-nested) table of numbers.  The position of each vertex is given as an x, y, and z coordinate."
                    },
                    {
                      name = "indices",
                      type = "table",
                      description = "The vertex indices.  Every 3 indices describes a triangle."
                    }
                  }
                }
              },
              related = {
                "ModelData:getTriangleCount",
                "ModelData:getVertexCount",
                "Model:getTriangles"
              }
            },
            {
              name = "getVertexCount",
              summary = "Get the total vertex count of the model.",
              description = "Returns the total vertex count of a model.  This count includes meshes that are attached to multiple nodes, and the count corresponds to the vertices returned by `ModelData:getTriangles`.",
              key = "ModelData:getVertexCount",
              module = "lovr.data",
              related = {
                "ModelData:getTriangles",
                "ModelData:getTriangleCount",
                "Model:getVertexCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The total number of vertices in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getWidth",
              summary = "Get the width of the model.",
              description = "Returns the width of the model, computed from its axis-aligned bounding box.",
              key = "ModelData:getWidth",
              module = "lovr.data",
              related = {
                "ModelData:getHeight",
                "ModelData:getDepth",
                "ModelData:getDimensions",
                "ModelData:getCenter",
                "ModelData:getBoundingBox",
                "Model:getWidth"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the model."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Rasterizer",
          summary = "An object that rasterizes glyphs from font files.",
          description = "A Rasterizer is an object that parses a TTF file, decoding and rendering glyphs from it.\n\nUsually you can just use `Font` objects.",
          key = "Rasterizer",
          module = "lovr.data",
          constructors = {
            "lovr.data.newRasterizer"
          },
          methods = {
            {
              name = "getAdvance",
              summary = "Get the advance of the font.",
              description = "Returns the advance metric of the font, in pixels.  The advance is how many pixels the font advances horizontally after each glyph is rendered.  This does not include kerning.",
              key = "Rasterizer:getAdvance",
              module = "lovr.data",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "advance",
                      type = "number",
                      description = "The advance of the font, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getAscent",
              summary = "Get the ascent of the font.",
              description = "Returns the ascent metric of the font, in pixels.  The ascent represents how far any glyph of the font ascends above the baseline.",
              key = "Rasterizer:getAscent",
              module = "lovr.data",
              related = {
                "Rasterizer:getDescent",
                "Font:getAscent"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "ascent",
                      type = "number",
                      description = "The ascent of the font, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getDescent",
              summary = "Get the descent of the font.",
              description = "Returns the descent metric of the font, in pixels.  The descent represents how far any glyph of the font descends below the baseline.",
              key = "Rasterizer:getDescent",
              module = "lovr.data",
              related = {
                "Rasterzer:getAscent",
                "Font:getDescent"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "descent",
                      type = "number",
                      description = "The descent of the font, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getGlyphCount",
              summary = "Get the number of glyphs stored in the font file.",
              description = "Returns the number of glyphs stored in the font file.",
              key = "Rasterizer:getGlyphCount",
              module = "lovr.data",
              related = {
                "Rasterizer:hasGlyphs"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of glyphs stored in the font file."
                    }
                  }
                }
              }
            },
            {
              name = "getHeight",
              summary = "Get the height of the font.",
              description = "Returns the height metric of the font, in pixels.",
              key = "Rasterizer:getHeight",
              module = "lovr.data",
              related = {
                "Font:getHeight"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the font, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getLineHeight",
              summary = "Get the line height of the font.",
              description = "Returns the line height metric of the font, in pixels.  This is how far apart lines are.",
              key = "Rasterizer:getLineHeight",
              module = "lovr.data",
              related = {
                "Rasterizer:getHeight",
                "Font:getLineHeight",
                "Font:setLineHeight"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The line height of the font, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "hasGlyphs",
              summary = "Get whether the Rasterizer can rasterize a set of glyphs.",
              description = "Check if the Rasterizer can rasterize a set of glyphs.",
              key = "Rasterizer:hasGlyphs",
              module = "lovr.data",
              related = {
                "Rasterizer:getGlyphCount"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "...",
                      type = "*",
                      description = "Strings (sets of characters) or numbers (character codes) to check for."
                    }
                  },
                  returns = {
                    {
                      name = "hasGlyphs",
                      type = "boolean",
                      description = "true if the Rasterizer can rasterize all of the supplied characters, false otherwise."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Sound",
          summary = "An object that holds raw audio samples.",
          description = "A Sound stores the data for a sound.  The supported sound formats are OGG, WAV, and MP3.  Sounds cannot be played directly.  Instead, there are `Source` objects in `lovr.audio` that are used for audio playback.  All Source objects are backed by one of these Sounds, and multiple Sources can share a single Sound to reduce memory usage.\n\nMetadata\n---\n\nSounds hold a fixed number of frames.  Each frame contains one audio sample for each channel. The `SampleFormat` of the Sound is the data type used for each sample (floating point, integer, etc.).  The Sound has a `ChannelLayout`, representing the number of audio channels and how they map to speakers (mono, stereo, etc.).  The sample rate of the Sound indicates how many frames should be played per second.  The duration of the sound (in seconds) is the number of frames divided by the sample rate.\n\nCompression\n---\n\nSounds can be compressed.  Compressed sounds are stored compressed in memory and are decoded as they are played.  This uses a lot less memory but increases CPU usage during playback.  OGG and MP3 are compressed audio formats.  When creating a sound from a compressed format, there is an option to immediately decode it, storing it uncompressed in memory.  It can be a good idea to decode short sound effects, since they won't use very much memory even when uncompressed and it will improve CPU usage.  Compressed sounds can not be written to using `Sound:setFrames`.\n\nStreams\n---\n\nSounds can be created as a stream by passing `'stream'` as their contents when creating them. Audio frames can be written to the end of the stream, and read from the beginning.  This works well for situations where data is being generated in real time or streamed in from some other data source.\n\nSources can be backed by a stream and they'll just play whatever audio is pushed to the stream. The audio module also lets you use a stream as a \"sink\" for an audio device.  For playback devices, this works like loopback, so the mixed audio from all playing Sources will get written to the stream.  For capture devices, all the microphone input will get written to the stream. Conversion between sample formats, channel layouts, and sample rates will happen automatically.\n\nKeep in mind that streams can still only hold a fixed number of frames.  If too much data is written before it is read, older frames will start to get overwritten.  Similary, it's possible to read too much data without writing fast enough.\n\nAmbisonics\n---\n\nAmbisonic sounds can be imported from WAVs, but can not yet be played.  Sounds with a `ChannelLayout` of `ambisonic` are stored as first-order full-sphere ambisonics using the AmbiX format (ACN channel ordering and SN3D channel normalization).  The AMB format is supported for import and will automatically get converted to AmbiX.  See `lovr.data.newSound` for more info.",
          key = "Sound",
          module = "lovr.data",
          constructors = {
            "lovr.data.newSound"
          },
          methods = {
            {
              name = "getBlob",
              summary = "Get the bytes backing this Sound as a Blob.",
              description = "Returns a Blob containing the raw bytes of the Sound.",
              key = "Sound:getBlob",
              module = "lovr.data",
              notes = "Samples for each channel are stored interleaved.  The data type of each sample is given by `Sound:getFormat`.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "The Blob instance containing the bytes for the `Sound`."
                    }
                  }
                }
              },
              related = {
                "Blob:getPointer",
                "Image:getBlob"
              }
            },
            {
              name = "getCapacity",
              summary = "Get the number of frames that can be written to the Sound.",
              description = "Returns the number of frames that can be written to the Sound.  For stream sounds, this is the number of frames that can be written without overwriting existing data.  For normal sounds, this returns the same value as `Sound:getFrameCount`.",
              key = "Sound:getCapacity",
              module = "lovr.data",
              related = {
                "Sound:getFrameCount",
                "Sound:getSampleCount",
                "Source:getDuration"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "capacity",
                      type = "number",
                      description = "The number of frames that can be written to the Sound."
                    }
                  }
                }
              }
            },
            {
              name = "getChannelCount",
              summary = "Get the number of channels in the Sound.",
              description = "Returns the number of channels in the Sound.  Mono sounds have 1 channel, stereo sounds have 2 channels, and ambisonic sounds have 4 channels.",
              key = "Sound:getChannelCount",
              module = "lovr.data",
              related = {
                "Sound:getChannelLayout"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "channels",
                      type = "number",
                      description = "The number of channels in the sound."
                    }
                  }
                }
              }
            },
            {
              name = "getChannelLayout",
              summary = "Get the channel layout of the Sound.",
              description = "Returns the channel layout of the Sound.",
              key = "Sound:getChannelLayout",
              module = "lovr.data",
              related = {
                "Sound:getChannelCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "channels",
                      type = "ChannelLayout",
                      description = "The channel layout."
                    }
                  }
                }
              }
            },
            {
              name = "getDuration",
              summary = "Get the duration of the Sound.",
              description = "Returns the duration of the Sound, in seconds.",
              key = "Sound:getDuration",
              module = "lovr.data",
              notes = "This can be computed as `(frameCount / sampleRate)`.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "duration",
                      type = "number",
                      description = "The duration of the Sound, in seconds."
                    }
                  }
                }
              },
              related = {
                "Sound:getFrameCount",
                "Sound:getSampleCount",
                "Sound:getSampleRate",
                "Source:getDuration"
              }
            },
            {
              name = "getFormat",
              summary = "Get the sample format of the Sound.",
              description = "Returns the sample format of the Sound.",
              key = "Sound:getFormat",
              module = "lovr.data",
              related = {
                "Sound:getChannelLayout",
                "Sound:getSampleRate"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "format",
                      type = "SampleFormat",
                      description = "The data type of each sample."
                    }
                  }
                }
              }
            },
            {
              name = "getFrameCount",
              summary = "Get the number of frames in the Sound.",
              description = "Returns the number of frames in the Sound.  A frame stores one sample for each channel.",
              key = "Sound:getFrameCount",
              module = "lovr.data",
              notes = "For streams, this returns the number of frames in the stream's buffer.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "frames",
                      type = "number",
                      description = "The number of frames in the Sound."
                    }
                  }
                }
              },
              related = {
                "Sound:getDuration",
                "Sound:getSampleCount",
                "Sound:getChannelCount"
              }
            },
            {
              name = "getFrames",
              summary = "Read frames from the Sound.",
              description = "Reads frames from the Sound into a table, Blob, or another Sound.",
              key = "Sound:getFrames",
              module = "lovr.data",
              variants = {
                {
                  arguments = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames to read.  If nil, reads as many frames as possible.\n\nCompressed sounds will automatically be decoded.\n\nReading from a stream will ignore the source offset and read the oldest frames.",
                      default = "nil"
                    },
                    {
                      name = "srcOffset",
                      type = "number",
                      description = "A frame offset to apply to the sound when reading frames.",
                      default = "0"
                    }
                  },
                  returns = {
                    {
                      name = "t",
                      type = "table",
                      description = "A table containing audio frames."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames read."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "t",
                      type = "table",
                      description = "An existing table to read frames into."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames to read.  If nil, reads as many frames as possible.\n\nCompressed sounds will automatically be decoded.\n\nReading from a stream will ignore the source offset and read the oldest frames.",
                      default = "nil"
                    },
                    {
                      name = "srcOffset",
                      type = "number",
                      description = "A frame offset to apply to the sound when reading frames.",
                      default = "0"
                    },
                    {
                      name = "dstOffset",
                      type = "number",
                      description = "An offset to apply to the destination when writing frames (indices for tables, bytes for Blobs, frames for Sounds).",
                      default = "0"
                    }
                  },
                  returns = {
                    {
                      name = "t",
                      type = "table",
                      description = "A table containing audio frames."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames read."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "A Blob to read frames into."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames to read.  If nil, reads as many frames as possible.\n\nCompressed sounds will automatically be decoded.\n\nReading from a stream will ignore the source offset and read the oldest frames.",
                      default = "nil"
                    },
                    {
                      name = "srcOffset",
                      type = "number",
                      description = "A frame offset to apply to the sound when reading frames.",
                      default = "0"
                    },
                    {
                      name = "dstOffset",
                      type = "number",
                      description = "An offset to apply to the destination when writing frames (indices for tables, bytes for Blobs, frames for Sounds).",
                      default = "0"
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames read."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "sound",
                      type = "Sound",
                      description = "Another Sound to copy frames into."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames to read.  If nil, reads as many frames as possible.\n\nCompressed sounds will automatically be decoded.\n\nReading from a stream will ignore the source offset and read the oldest frames.",
                      default = "nil"
                    },
                    {
                      name = "srcOffset",
                      type = "number",
                      description = "A frame offset to apply to the sound when reading frames.",
                      default = "0"
                    },
                    {
                      name = "dstOffset",
                      type = "number",
                      description = "An offset to apply to the destination when writing frames (indices for tables, bytes for Blobs, frames for Sounds).",
                      default = "0"
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames read."
                    }
                  }
                }
              }
            },
            {
              name = "getSampleCount",
              summary = "Get the number of samples in the Sound.",
              description = "Returns the total number of samples in the Sound.",
              key = "Sound:getSampleCount",
              module = "lovr.data",
              notes = "For streams, this returns the number of samples in the stream's buffer.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "samples",
                      type = "number",
                      description = "The total number of samples in the Sound."
                    }
                  }
                }
              },
              related = {
                "Sound:getDuration",
                "Sound:getFrameCount",
                "Sound:getChannelCount"
              }
            },
            {
              name = "getSampleRate",
              summary = "Get the sample rate of the Sound.",
              description = "Returns the sample rate of the Sound, in Hz.  This is the number of frames that are played every second.  It's usually a high number like 48000.",
              key = "Sound:getSampleRate",
              module = "lovr.data",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "frequency",
                      type = "number",
                      description = "The number of frames per second in the Sound."
                    }
                  }
                }
              }
            },
            {
              name = "isCompressed",
              summary = "Check if the Sound is compressed.",
              description = "Returns whether the Sound is compressed.  Compressed sounds are loaded from compressed audio formats like MP3 and OGG.  They use a lot less memory but require some extra CPU work during playback.  Compressed sounds can not be modified using `Sound:setFrames`.",
              key = "Sound:isCompressed",
              module = "lovr.data",
              related = {
                "Sound:isStream",
                "lovr.data.newSound"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "compressed",
                      type = "boolean",
                      description = "Whether the Sound is compressed."
                    }
                  }
                }
              }
            },
            {
              name = "isStream",
              summary = "Check if the Sound is a stream.",
              description = "Returns whether the Sound is a stream.",
              key = "Sound:isStream",
              module = "lovr.data",
              related = {
                "Sound:isCompressed",
                "lovr.data.newSound"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "stream",
                      type = "boolean",
                      description = "Whether the Sound is a stream."
                    }
                  }
                }
              }
            },
            {
              name = "setFrames",
              summary = "Write frames to the Sound.",
              description = "Writes frames to the Sound.",
              key = "Sound:setFrames",
              module = "lovr.data",
              variants = {
                {
                  arguments = {
                    {
                      name = "t",
                      type = "table",
                      description = "A table containing frames to write."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "How many frames to write.  If nil, writes as many as possible.",
                      default = "nil"
                    },
                    {
                      name = "dstOffset",
                      type = "number",
                      description = "A frame offset to apply when writing the frames.",
                      default = "0"
                    },
                    {
                      name = "srcOffset",
                      type = "number",
                      description = "A frame, byte, or index offset to apply when reading frames from the source.",
                      default = "0"
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames written."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "A Blob containing frames to write."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "How many frames to write.  If nil, writes as many as possible.",
                      default = "nil"
                    },
                    {
                      name = "dstOffset",
                      type = "number",
                      description = "A frame offset to apply when writing the frames.",
                      default = "0"
                    },
                    {
                      name = "srcOffset",
                      type = "number",
                      description = "A frame, byte, or index offset to apply when reading frames from the source.",
                      default = "0"
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames written."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "sound",
                      type = "Sound",
                      description = "Another Sound to copy frames from."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "How many frames to write.  If nil, writes as many as possible.",
                      default = "nil"
                    },
                    {
                      name = "dstOffset",
                      type = "number",
                      description = "A frame offset to apply when writing the frames.",
                      default = "0"
                    },
                    {
                      name = "srcOffset",
                      type = "number",
                      description = "A frame, byte, or index offset to apply when reading frames from the source.",
                      default = "0"
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of frames written."
                    }
                  }
                }
              },
              examples = {
                {
                  description = "Generate a sine wave.",
                  code = "function lovr.load()\n  local length = 1\n  local rate = 48000\n  local frames = length * rate\n  local frequency = 440\n  local volume = 1.0\n\n  sound = lovr.data.newSound(frames, 'f32', 'stereo', rate)\n\n  local data = {}\n  for i = 1, frames do\n    local amplitude = math.sin((i - 1) * frequency / rate * (2 * math.pi)) * volume\n    data[2 * i - 1] = amplitude\n    data[2 * i - 0] = amplitude\n  end\n\n  sound:setFrames(data)\n\n  source = lovr.audio.newSource(sound)\n  source:setLooping(true)\n  source:play()\nend"
                }
              }
            }
          }
        }
      },
      functions = {
        {
          name = "newBlob",
          summary = "Create a new Blob.",
          description = "Creates a new Blob.",
          key = "lovr.data.newBlob",
          module = "lovr.data",
          related = {
            "lovr.filesystem.newBlob"
          },
          variants = {
            {
              arguments = {
                {
                  name = "size",
                  type = "number",
                  description = "The amount of data to allocate for the Blob, in bytes.  All of the bytes will be filled with zeroes."
                },
                {
                  name = "name",
                  type = "string",
                  description = "A name for the Blob (used in error messages)",
                  default = "''"
                }
              },
              returns = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The new Blob."
                }
              }
            },
            {
              arguments = {
                {
                  name = "contents",
                  type = "string",
                  description = "A string to use for the Blob's contents."
                },
                {
                  name = "name",
                  type = "string",
                  description = "A name for the Blob (used in error messages)",
                  default = "''"
                }
              },
              returns = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The new Blob."
                }
              }
            },
            {
              arguments = {
                {
                  name = "source",
                  type = "Blob",
                  description = "A Blob to copy the contents from."
                },
                {
                  name = "name",
                  type = "string",
                  description = "A name for the Blob (used in error messages)",
                  default = "''"
                }
              },
              returns = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The new Blob."
                }
              }
            }
          }
        },
        {
          name = "newImage",
          summary = "Create a new Image.",
          description = "Creates a new Image.  Image data can be loaded and decoded from an image file, or a raw block of pixels with a specified width, height, and format can be created.",
          key = "lovr.data.newImage",
          module = "lovr.data",
          notes = "The supported image file formats are png, jpg, hdr, dds (DXT1, DXT3, DXT5), ktx, and astc.\n\nOnly 2D textures are supported for DXT/ASTC.\n\nCurrently textures loaded as KTX need to be in DXT/ASTC formats.",
          variants = {
            {
              description = "Load image data from a file.",
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The filename of the image to load."
                },
                {
                  name = "flip",
                  type = "boolean",
                  description = "Whether to vertically flip the image on load.  This should be true for normal textures, and false for textures that are going to be used in a cubemap.",
                  default = "true"
                }
              },
              returns = {
                {
                  name = "image",
                  type = "Image",
                  description = "The new Image."
                }
              }
            },
            {
              description = "Create an Image with a given size and pixel format.",
              arguments = {
                {
                  name = "width",
                  type = "number",
                  description = "The width of the texture."
                },
                {
                  name = "height",
                  type = "number",
                  description = "The height of the texture."
                },
                {
                  name = "format",
                  type = "TextureFormat",
                  description = "The format of the texture's pixels.",
                  default = "rgba"
                },
                {
                  name = "data",
                  type = "Blob",
                  description = "Raw pixel values to use as the contents.  If `nil`, the data will all be zero.",
                  default = "nil"
                }
              },
              returns = {
                {
                  name = "image",
                  type = "Image",
                  description = "The new Image."
                }
              }
            },
            {
              description = "Clone an existing Image.",
              arguments = {
                {
                  name = "source",
                  type = "Image",
                  description = "The Image to clone."
                }
              },
              returns = {
                {
                  name = "image",
                  type = "Image",
                  description = "The new Image."
                }
              }
            },
            {
              description = "Decode image data from a Blob.",
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The Blob containing image data to decode."
                },
                {
                  name = "flip",
                  type = "boolean",
                  description = "Whether to vertically flip the image on load.  This should be true for normal textures, and false for textures that are going to be used in a cubemap.",
                  default = "true"
                }
              },
              returns = {
                {
                  name = "image",
                  type = "Image",
                  description = "The new Image."
                }
              }
            }
          }
        },
        {
          name = "newModelData",
          summary = "Create a new ModelData.",
          description = "Loads a 3D model from a file.  The supported 3D file formats are OBJ and glTF.",
          key = "lovr.data.newModelData",
          module = "lovr.data",
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The filename of the model to load."
                }
              },
              returns = {
                {
                  name = "modelData",
                  type = "ModelData",
                  description = "The new ModelData."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The Blob containing data for a model to decode."
                }
              },
              returns = {
                {
                  name = "modelData",
                  type = "ModelData",
                  description = "The new ModelData."
                }
              }
            }
          }
        },
        {
          name = "newRasterizer",
          summary = "Create a new Rasterizer.",
          description = "Creates a new Rasterizer from a TTF file.",
          key = "lovr.data.newRasterizer",
          module = "lovr.data",
          variants = {
            {
              description = "Create a Rasterizer for the default font included with LÖVR (Varela Round).",
              arguments = {
                {
                  name = "size",
                  type = "number",
                  description = "The resolution to render the fonts at, in pixels.  Higher resolutions use more memory and processing power but may provide better quality results for some fonts/situations.",
                  default = "32"
                }
              },
              returns = {
                {
                  name = "rasterizer",
                  type = "Rasterizer",
                  description = "The new Rasterizer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The filename of the font file to load."
                },
                {
                  name = "size",
                  type = "number",
                  description = "The resolution to render the fonts at, in pixels.  Higher resolutions use more memory and processing power but may provide better quality results for some fonts/situations.",
                  default = "32"
                }
              },
              returns = {
                {
                  name = "rasterizer",
                  type = "Rasterizer",
                  description = "The new Rasterizer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The Blob containing font data."
                },
                {
                  name = "size",
                  type = "number",
                  description = "The resolution to render the fonts at, in pixels.  Higher resolutions use more memory and processing power but may provide better quality results for some fonts/situations.",
                  default = "32"
                }
              },
              returns = {
                {
                  name = "rasterizer",
                  type = "Rasterizer",
                  description = "The new Rasterizer."
                }
              }
            }
          }
        },
        {
          name = "newSound",
          summary = "Create a new Sound.",
          description = "Creates a new Sound.  A sound can be loaded from an audio file, or it can be created empty with capacity for a certain number of audio frames.\n\nWhen loading audio from a file, use the `decode` option to control whether compressed audio should remain compressed or immediately get decoded to raw samples.\n\nWhen creating an empty sound, the `contents` parameter can be set to `'stream'` to create an audio stream.  On streams, `Sound:setFrames` will always write to the end of the stream, and `Sound:getFrames` will always read the oldest samples from the beginning.  The number of frames in the sound is the total capacity of the stream's buffer.",
          key = "lovr.data.newSound",
          module = "lovr.data",
          notes = "It is highly recommended to use an audio format that matches the format of the audio module: `f32` sample formats at a sample rate of 48000, with 1 channel for spatialized sources or 2 channels for unspatialized sources.  This will avoid the need to convert audio during playback, which boosts performance of the audio thread.\n\nThe WAV importer supports 16, 24, and 32 bit integer data and 32 bit floating point data.  The data must be mono, stereo, or 4-channel full-sphere ambisonic.  The `WAVE_FORMAT_EXTENSIBLE` extension is supported.\n\nAmbisonic channel layouts are supported for import (but not yet for playback).  Ambisonic data can be loaded from WAV files.  It must be first-order full-sphere ambisonic data with 4 channels.  If the WAV has a `WAVE_FORMAT_EXTENSIBLE` chunk with an `AMBISONIC_B_FORMAT` format GUID, then the data is understood as using the AMB format with Furse-Malham channel ordering and normalization.  *All other* 4-channel files are assumed to be using the AmbiX format with ACN channel ordering and SN3D normalization.  AMB files will get automatically converted to AmbiX on import, so ambisonic Sounds will always be in a consistent format.\n\nOGG and MP3 files will always have the `f32` format when loaded.",
          variants = {
            {
              description = "Create a raw or stream Sound from a frame count and format info:",
              arguments = {
                {
                  name = "frames",
                  type = "number",
                  description = "The number of frames the Sound can hold."
                },
                {
                  name = "format",
                  type = "SampleFormat",
                  description = "The sample data type.",
                  default = "'f32'"
                },
                {
                  name = "channels",
                  type = "ChannelLayout",
                  description = "The channel layout.",
                  default = "'stereo'"
                },
                {
                  name = "sampleRate",
                  type = "number",
                  description = "The sample rate, in Hz.",
                  default = "48000"
                },
                {
                  name = "contents",
                  type = "*",
                  description = "A Blob containing raw audio samples to use as the initial contents, 'stream' to create an audio stream, or `nil` to leave the data initialized to zero.",
                  default = "nil"
                }
              },
              returns = {
                {
                  name = "sound",
                  type = "Sound",
                  description = "Sounds good."
                }
              }
            },
            {
              description = "Load a sound from a file.  Compressed audio formats (OGG, MP3) can optionally be decoded into raw sounds.",
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The filename of a sound to load."
                },
                {
                  name = "decode",
                  type = "boolean",
                  description = "Whether compressed audio files should be immediately decoded."
                }
              },
              returns = {
                {
                  name = "sound",
                  type = "Sound",
                  description = "Sounds good."
                }
              }
            },
            {
              description = "Load a sound from a Blob containing the data of an audio file.  Compressed audio formats (OGG, MP3) can optionally be decoded into raw sounds.\n\nIf the Blob contains raw audio samples, use the first variant instead of this one.",
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The Blob containing audio file data to load."
                },
                {
                  name = "decode",
                  type = "boolean",
                  description = "Whether compressed audio files should be immediately decoded."
                }
              },
              returns = {
                {
                  name = "sound",
                  type = "Sound",
                  description = "Sounds good."
                }
              }
            }
          }
        }
      },
      enums = {
        {
          name = "AnimationProperty",
          summary = "Different animated properties.",
          description = "This indicates the different transform properties that can be animated.",
          key = "AnimationProperty",
          module = "lovr.data",
          values = {
            {
              name = "translation",
              description = "Node translation."
            },
            {
              name = "rotation",
              description = "Node rotation."
            },
            {
              name = "scale",
              description = "Node scale."
            }
          }
        },
        {
          name = "AttributeType",
          summary = "Data types for vertex attributes in meshes.",
          description = "These are the data types that can be used by vertex data in meshes.",
          key = "AttributeType",
          module = "lovr.data",
          values = {
            {
              name = "i8",
              description = "Signed 8 bit integers (-128 to 127)."
            },
            {
              name = "u8",
              description = "Unsigned 8 bit integers (0 to 255)."
            },
            {
              name = "i16",
              description = "Signed 16 bit integers (-32768 to 32767)."
            },
            {
              name = "u16",
              description = "Unsigned 16 bit integers (0 to 65535)."
            },
            {
              name = "i32",
              description = "Signed 32 bit integers (-2147483648 to 2147483647)."
            },
            {
              name = "u32",
              description = "Unsigned 32 bit integers (0 to 429467295)."
            },
            {
              name = "f32",
              description = "Floating point numbers."
            }
          }
        },
        {
          name = "ChannelLayout",
          summary = "Different channel layouts for Sounds.",
          description = "Sounds can have different numbers of channels, and those channels can map to various speaker layouts.",
          key = "ChannelLayout",
          module = "lovr.data",
          related = {
            "lovr.data.newSound",
            "Sound:getFormat"
          },
          values = {
            {
              name = "mono",
              description = "1 channel."
            },
            {
              name = "stereo",
              description = "2 channels.  The first channel is for the left speaker and the second is for the right."
            },
            {
              name = "ambisonic",
              description = "4 channels.  Ambisonic channels don't map directly to speakers but instead represent directions in 3D space, sort of like the images of a skybox.  Currently, ambisonic sounds can only be loaded, not played."
            }
          }
        },
        {
          name = "DefaultAttribute",
          summary = "Attributes that can be loaded from a model.",
          description = "These are the different types of attributes that may be present in meshes loaded from models.",
          key = "DefaultAttribute",
          module = "lovr.data",
          values = {
            {
              name = "position",
              description = "Vertex positions."
            },
            {
              name = "normal",
              description = "Vertex normal vectors."
            },
            {
              name = "uv",
              description = "Vertex texture coordinates."
            },
            {
              name = "color",
              description = "Vertex colors."
            },
            {
              name = "tangent",
              description = "Vertex tangent vectors."
            },
            {
              name = "joints",
              description = "Vertex joint indices."
            },
            {
              name = "weights",
              description = "Vertex joint weights."
            }
          }
        },
        {
          name = "DrawMode",
          summary = "Different draw modes for meshes in ModelDatas.",
          description = "The DrawMode of a mesh determines how its vertices are connected together.",
          key = "DrawMode",
          module = "lovr.data",
          related = {
            "ModelData:getMeshDrawMode"
          },
          values = {
            {
              name = "points",
              description = "Each vertex is draw as a single point."
            },
            {
              name = "lines",
              description = "Every pair of vertices is drawn as a line."
            },
            {
              name = "linestrip",
              description = "Draws a single line through all of the vertices."
            },
            {
              name = "lineloop",
              description = "Draws a single line through all of the vertices, then connects back to the first vertex."
            },
            {
              name = "strip",
              description = "Vertices are rendered as triangles.  After the first 3 vertices, each subsequent vertex connects to the previous two."
            },
            {
              name = "triangles",
              description = "Every 3 vertices forms a triangle."
            },
            {
              name = "fan",
              description = "Vertices are rendered as triangles.  After the first 3 vertices, each subsequent vertex is connected to the previous vertex and the first vertex."
            }
          }
        },
        {
          name = "SampleFormat",
          summary = "Different data types for samples in a Sound.",
          description = "Sounds can store audio samples as 16 bit integers or 32 bit floats.",
          key = "SampleFormat",
          module = "lovr.data",
          related = {
            "lovr.data.newSound",
            "Sound:getFormat"
          },
          values = {
            {
              name = "f32",
              description = "32 bit floating point samples (between -1.0 and 1.0)."
            },
            {
              name = "i16",
              description = "16 bit integer samples (between -32768 and 32767)."
            }
          }
        },
        {
          name = "SmoothMode",
          summary = "Different ways to interpolate between animation keyframes.",
          description = "Different ways to interpolate between animation keyframes.",
          key = "SmoothMode",
          module = "lovr.data",
          values = {
            {
              name = "step",
              description = "The animated property will snap to the nearest keyframe."
            },
            {
              name = "linear",
              description = "The animated property will linearly interpolate between keyframes."
            },
            {
              name = "cubic",
              description = "The animated property will follow a smooth curve between nearby keyframes."
            }
          }
        },
        {
          name = "TextureFormat",
          description = "Different data layouts for pixels in `Image` and `Texture` objects.\n\nFormats starting with `d` are depth formats, used for depth/stencil render targets.\n\nFormats starting with `bc` and `astc` are compressed formats.  Compressed formats have better performance since they stay compressed on the CPU and GPU, reducing the amount of memory bandwidth required to look up all the pixels needed for shading.\n\nFormats without the `f` suffix are unsigned normalized formats, which store values in the range `[0,1]`.  The `f` suffix indicates a floating point format which can store values outside this range, and is used for HDR rendering or storing data in a texture.",
          key = "TextureFormat",
          module = "lovr.data",
          values = {
            {
              name = "r8",
              description = "One 8-bit channel.  1 byte per pixel."
            },
            {
              name = "rg8",
              description = "Two 8-bit channels.  2 bytes per pixel."
            },
            {
              name = "rgba8",
              description = "Four 8-bit channels.  4 bytes per pixel."
            },
            {
              name = "r16",
              description = "One 16-bit channel.  2 bytes per pixel."
            },
            {
              name = "rg16",
              description = "Two 16-bit channels.  4 bytes per pixel."
            },
            {
              name = "rgba16",
              description = "Four 16-bit channels.  8 bytes per pixel."
            },
            {
              name = "r16f",
              description = "One 16-bit floating point channel.  2 bytes per pixel."
            },
            {
              name = "rg16f",
              description = "Two 16-bit floating point channels.  4 bytes per pixel."
            },
            {
              name = "rgba16f",
              description = "Four 16-bit floating point channels.  8 bytes per pixel."
            },
            {
              name = "r32f",
              description = "One 32-bit floating point channel.  4 bytes per pixel."
            },
            {
              name = "rg32f",
              description = "Two 32-bit floating point channels.  8 bytes per pixel."
            },
            {
              name = "rgba32f",
              description = "Four 32-bit floating point channels.  16 bytes per pixel."
            },
            {
              name = "rgb565",
              description = "Packs three channels into 16 bits.  2 bytes per pixel."
            },
            {
              name = "rgb5a1",
              description = "Packs four channels into 16 bits, with \"cutout\" alpha.  2 bytes per pixel."
            },
            {
              name = "rgb10a2",
              description = "Packs four channels into 32 bits.  4 bytes per pixel."
            },
            {
              name = "rg11b10f",
              description = "Packs three unsigned floating point channels into 32 bits.  4 bytes per pixel."
            },
            {
              name = "d16",
              description = "One 16-bit depth channel.  2 bytes per pixel."
            },
            {
              name = "d24s8",
              description = "One 24-bit depth channel and one 8-bit stencil channel.  4 bytes per pixel."
            },
            {
              name = "d32f",
              description = "One 32-bit floating point depth channel.  4 bytes per pixel."
            },
            {
              name = "bc1",
              description = "3 channels.  8 bytes per 4x4 block, or 0.5 bytes per pixel.  Good for opaque images."
            },
            {
              name = "bc2",
              description = "Four channels.  16 bytes per 4x4 block or 1 byte per pixel.  Not good for anything, because it only has 16 distinct levels of alpha."
            },
            {
              name = "bc3",
              description = "Four channels.  16 bytes per 4x4 block or 1 byte per pixel.  Good for color images with transparency."
            },
            {
              name = "bc4u",
              description = "One unsigned normalized channel.  8 bytes per 4x4 block or 0.5 bytes per pixel.  Good for grayscale images, like heightmaps."
            },
            {
              name = "bc4s",
              description = "One signed normalized channel.  8 bytes per 4x4 block or 0.5 bytes per pixel.  Similar to bc4u but has a range of -1 to 1."
            },
            {
              name = "bc5u",
              description = "Two unsigned normalized channels.  16 bytes per 4x4 block, or 1 byte per pixel.  Good for normal maps."
            },
            {
              name = "bc5s",
              description = "Two signed normalized channels.  16 bytes per 4x4 block or 1 byte per pixel.  Good for normal maps."
            },
            {
              name = "bc6uf",
              description = "Three unsigned floating point channels.  16 bytes per 4x4 block or 1 byte per pixel.  Good for HDR images."
            },
            {
              name = "bc6sf",
              description = "Three floating point channels.  16 bytes per 4x4 block or 1 byte per pixel.  Good for HDR images."
            },
            {
              name = "bc7",
              description = "Four channels.  16 bytes per 4x4 block or 1 byte per pixel.  High quality.  Good for most color images, including transparency."
            },
            {
              name = "astc4x4",
              description = "Four channels, 16 bytes per 4x4 block or 1 byte per pixel."
            },
            {
              name = "astc5x4",
              description = "Four channels, 16 bytes per 5x4 block or 0.80 bytes per pixel."
            },
            {
              name = "astc5x5",
              description = "Four channels, 16 bytes per 5x5 block or 0.64 bytes per pixel."
            },
            {
              name = "astc6x5",
              description = "Four channels, 16 bytes per 6x5 block or 0.53 bytes per pixel."
            },
            {
              name = "astc6x6",
              description = "Four channels, 16 bytes per 6x6 block or 0.44 bytes per pixel."
            },
            {
              name = "astc8x5",
              description = "Four channels, 16 bytes per 8x5 block or 0.40 bytes per pixel."
            },
            {
              name = "astc8x6",
              description = "Four channels, 16 bytes per 8x6 block or 0.33 bytes per pixel."
            },
            {
              name = "astc8x8",
              description = "Four channels, 16 bytes per 8x8 block or 0.25 bytes per pixel."
            },
            {
              name = "astc10x5",
              description = "Four channels, 16 bytes per 10x5 block or 0.32 bytes per pixel."
            },
            {
              name = "astc10x6",
              description = "Four channels, 16 bytes per 10x6 block or 0.27 bytes per pixel."
            },
            {
              name = "astc10x8",
              description = "Four channels, 16 bytes per 10x8 block or 0.20 bytes per pixel."
            },
            {
              name = "astc10x10",
              description = "Four channels, 16 bytes per 10x10 block or 0.16 bytes per pixel."
            },
            {
              name = "astc12x10",
              description = "Four channels, 16 bytes per 12x10 block or 0.13 bytes per pixel."
            },
            {
              name = "astc12x12",
              description = "Four channels, 16 bytes per 12x12 block or 0.11 bytes per pixel."
            }
          }
        }
      }
    },
    {
      name = "event",
      tag = "modules",
      summary = "Handles events from the operating system.",
      description = "The `lovr.event` module handles events from the operating system.\n\nDue to its low-level nature, it's rare to use `lovr.event` in simple projects.",
      key = "lovr.event",
      enums = {
        {
          name = "KeyCode",
          summary = "Keys that can be pressed.",
          description = "Keys that can be pressed on a keyboard.  Notably, numpad keys are missing right now.",
          key = "KeyCode",
          module = "lovr.event",
          related = {
            "lovr.keypressed",
            "lovr.keyreleased"
          },
          values = {
            {
              name = "a",
              description = "The A key."
            },
            {
              name = "b",
              description = "The B key."
            },
            {
              name = "c",
              description = "The C key."
            },
            {
              name = "d",
              description = "The D key."
            },
            {
              name = "e",
              description = "The E key."
            },
            {
              name = "f",
              description = "The F key."
            },
            {
              name = "g",
              description = "The G key."
            },
            {
              name = "h",
              description = "The H key."
            },
            {
              name = "i",
              description = "The I key."
            },
            {
              name = "j",
              description = "The J key."
            },
            {
              name = "k",
              description = "The K key."
            },
            {
              name = "l",
              description = "The L key."
            },
            {
              name = "m",
              description = "The M key."
            },
            {
              name = "n",
              description = "The N key."
            },
            {
              name = "o",
              description = "The O key."
            },
            {
              name = "p",
              description = "The P key."
            },
            {
              name = "q",
              description = "The Q key."
            },
            {
              name = "r",
              description = "The R key."
            },
            {
              name = "s",
              description = "The S key."
            },
            {
              name = "t",
              description = "The T key."
            },
            {
              name = "u",
              description = "The U key."
            },
            {
              name = "v",
              description = "The V key."
            },
            {
              name = "w",
              description = "The W key."
            },
            {
              name = "x",
              description = "The X key."
            },
            {
              name = "y",
              description = "The Y key."
            },
            {
              name = "z",
              description = "The Z key."
            },
            {
              name = "0",
              description = "The 0 key."
            },
            {
              name = "1",
              description = "The 1 key."
            },
            {
              name = "2",
              description = "The 2 key."
            },
            {
              name = "3",
              description = "The 3 key."
            },
            {
              name = "4",
              description = "The 4 key."
            },
            {
              name = "5",
              description = "The 5 key."
            },
            {
              name = "6",
              description = "The 6 key."
            },
            {
              name = "7",
              description = "The 7 key."
            },
            {
              name = "8",
              description = "The 8 key."
            },
            {
              name = "9",
              description = "The 9 key."
            },
            {
              name = "space",
              description = "The space bar."
            },
            {
              name = "return",
              description = "The enter key."
            },
            {
              name = "tab",
              description = "The tab key."
            },
            {
              name = "escape",
              description = "The escape key."
            },
            {
              name = "backspace",
              description = "The backspace key."
            },
            {
              name = "up",
              description = "The up arrow key."
            },
            {
              name = "down",
              description = "The down arrow key."
            },
            {
              name = "left",
              description = "The left arrow key."
            },
            {
              name = "right",
              description = "The right arrow key."
            },
            {
              name = "home",
              description = "The home key."
            },
            {
              name = "end",
              description = "The end key."
            },
            {
              name = "pageup",
              description = "The page up key."
            },
            {
              name = "pagedown",
              description = "The page down key."
            },
            {
              name = "insert",
              description = "The insert key."
            },
            {
              name = "delete",
              description = "The delete key."
            },
            {
              name = "f1",
              description = "The F1 key."
            },
            {
              name = "f2",
              description = "The F2 key."
            },
            {
              name = "f3",
              description = "The F3 key."
            },
            {
              name = "f4",
              description = "The F4 key."
            },
            {
              name = "f5",
              description = "The F5 key."
            },
            {
              name = "f6",
              description = "The F6 key."
            },
            {
              name = "f7",
              description = "The F7 key."
            },
            {
              name = "f8",
              description = "The F8 key."
            },
            {
              name = "f9",
              description = "The F9 key."
            },
            {
              name = "f10",
              description = "The F10 key."
            },
            {
              name = "f11",
              description = "The F11 key."
            },
            {
              name = "f12",
              description = "The F12 key."
            },
            {
              name = "`",
              description = "The backtick/backquote/grave accent key."
            },
            {
              name = "-",
              description = "The dash/hyphen/minus key."
            },
            {
              name = "=",
              description = "The equal sign key."
            },
            {
              name = "[",
              description = "The left bracket key."
            },
            {
              name = "]",
              description = "The right bracket key."
            },
            {
              name = "\\",
              description = "The backslash key."
            },
            {
              name = ";",
              description = "The semicolon key."
            },
            {
              name = "'",
              description = "The single quote key."
            },
            {
              name = ",",
              description = "The comma key."
            },
            {
              name = ".",
              description = "The period key."
            },
            {
              name = "/",
              description = "The slash key."
            },
            {
              name = "lctrl",
              description = "The left control key."
            },
            {
              name = "lshift",
              description = "The left shift key."
            },
            {
              name = "lalt",
              description = "The left alt key."
            },
            {
              name = "lgui",
              description = "The left OS key (windows, command, super)."
            },
            {
              name = "rctrl",
              description = "The right control key."
            },
            {
              name = "rshift",
              description = "The right shift key."
            },
            {
              name = "ralt",
              description = "The right alt key."
            },
            {
              name = "rgui",
              description = "The right OS key (windows, command, super)."
            },
            {
              name = "capslock",
              description = "The caps lock key."
            },
            {
              name = "scrolllock",
              description = "The scroll lock key."
            },
            {
              name = "numlock",
              description = "The numlock key."
            }
          }
        }
      },
      examples = {
        {
          description = "Adding a custom event.",
          code = "function lovr.load()\n  lovr.handlers['customevent'] = function(a, b, c)\n    print('custom event handled with args:', a, b, c)\n  end\n\n  lovr.event.push('customevent', 1, 2, 3)\nend"
        }
      },
      functions = {
        {
          name = "clear",
          summary = "Clear the event queue.",
          description = "Clears the event queue, removing any unprocessed events.",
          key = "lovr.event.clear",
          module = "lovr.event",
          variants = {
            {
              arguments = {},
              returns = {}
            }
          }
        },
        {
          name = "poll",
          summary = "Iterate over unprocessed events in the queue.",
          description = "This function returns a Lua iterator for all of the unprocessed items in the event queue.  Each event consists of a name as a string, followed by event-specific arguments.  This function is called in the default implementation of `lovr.run`, so it is normally not necessary to poll for events yourself.",
          key = "lovr.event.poll",
          module = "lovr.event",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "iterator",
                  type = "function",
                  description = "The iterator function, usable in a for loop.",
                  arguments = {},
                  returns = {}
                }
              }
            }
          }
        },
        {
          name = "pump",
          summary = "Pump new events into the queue for processing.",
          description = "Fills the event queue with unprocessed events from the operating system.  This function should be called often, otherwise the operating system will consider the application unresponsive. This function is called in the default implementation of `lovr.run`.",
          key = "lovr.event.pump",
          module = "lovr.event",
          related = {
            "lovr.event.poll"
          },
          variants = {
            {
              arguments = {},
              returns = {}
            }
          }
        },
        {
          name = "push",
          summary = "Manually push an event onto the queue.",
          description = "Pushes an event onto the event queue.  It will be processed the next time `lovr.event.poll` is called.  For an event to be processed properly, there needs to be a function in the `lovr.handlers` table with a key that's the same as the event name.",
          key = "lovr.event.push",
          module = "lovr.event",
          notes = "Only nil, booleans, numbers, strings, and LÖVR objects are supported types for event data.",
          variants = {
            {
              arguments = {
                {
                  name = "name",
                  type = "string",
                  description = "The name of the event."
                },
                {
                  name = "...",
                  type = "*",
                  description = "The arguments for the event.  Currently, up to 4 are supported."
                }
              },
              returns = {}
            }
          },
          related = {
            "lovr.event.poll",
            "lovr.event.quit"
          }
        },
        {
          name = "quit",
          summary = "Quit the application.",
          description = "Pushes an event to quit.  An optional number can be passed to set the exit code for the application.  An exit code of zero indicates normal termination, whereas a nonzero exit code indicates that an error occurred.",
          key = "lovr.event.quit",
          module = "lovr.event",
          notes = "This function is equivalent to calling `lovr.event.push('quit', <args>)`.\n\nThe event won't be processed until the next time `lovr.event.poll` is called.\n\nThe `lovr.quit` callback will be called when the event is processed, which can be used to do any cleanup work.  The callback can also return `false` to abort the quitting process.",
          variants = {
            {
              arguments = {
                {
                  name = "code",
                  type = "number",
                  description = "The exit code of the program.",
                  default = "0"
                }
              },
              returns = {}
            }
          },
          related = {
            "lovr.quit",
            "lovr.event.poll",
            "lovr.event.restart"
          }
        },
        {
          name = "restart",
          summary = "Restart the application.",
          description = "Pushes an event to restart the framework.",
          key = "lovr.event.restart",
          module = "lovr.event",
          notes = "The event won't be processed until the next time `lovr.event.poll` is called.\n\nThe `lovr.restart` callback can be used to persist a value between restarts.",
          variants = {
            {
              arguments = {},
              returns = {}
            }
          },
          related = {
            "lovr.restart",
            "lovr.event.poll",
            "lovr.event.quit"
          }
        }
      },
      objects = {},
      notes = "You can define your own custom events by adding a function to the `lovr.handlers` table with a key of the name of the event you want to add.  Then, push the event using `lovr.event.push`."
    },
    {
      name = "filesystem",
      tag = "modules",
      summary = "Provides access to the filesystem.",
      description = "The `lovr.filesystem` module provides access to the filesystem.",
      key = "lovr.filesystem",
      enums = {},
      objects = {},
      functions = {
        {
          name = "append",
          summary = "Append content to the end of a file.",
          description = "Appends content to the end of a file.",
          key = "lovr.filesystem.append",
          module = "lovr.filesystem",
          notes = "If the file does not exist, it is created.",
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The file to append to."
                },
                {
                  name = "content",
                  type = "string",
                  description = "A string to write to the end of the file."
                }
              },
              returns = {
                {
                  name = "bytes",
                  type = "number",
                  description = "The number of bytes actually appended to the file."
                }
              }
            },
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The file to append to."
                },
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob containing data to append to the file."
                }
              },
              returns = {
                {
                  name = "bytes",
                  type = "number",
                  description = "The number of bytes actually appended to the file."
                }
              }
            }
          }
        },
        {
          name = "createDirectory",
          summary = "Create a directory.",
          description = "Creates a directory in the save directory.  Any parent directories that don't exist will also be created.",
          key = "lovr.filesystem.createDirectory",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The directory to create, recursively."
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether the directory was created."
                }
              }
            }
          }
        },
        {
          name = "getAppdataDirectory",
          summary = "Get the application data directory.",
          description = "Returns the application data directory.  This will be something like:\n\n- `C:\\Users\\user\\AppData\\Roaming` on Windows.\n- `/home/user/.config` on Linux.\n- `/Users/user/Library/Application Support` on macOS.",
          key = "lovr.filesystem.getAppdataDirectory",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "path",
                  type = "string",
                  description = "The absolute path to the appdata directory."
                }
              }
            }
          }
        },
        {
          name = "getDirectoryItems",
          summary = "Get a list of files in a directory.",
          description = "Returns a sorted table containing all files and folders in a single directory.",
          key = "lovr.filesystem.getDirectoryItems",
          module = "lovr.filesystem",
          notes = "This function calls `table.sort` to sort the results, so if `table.sort` is not available in the global scope the results are not guaranteed to be sorted.",
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The directory."
                }
              },
              returns = {
                {
                  name = "table",
                  type = "items",
                  description = "A table with a string for each file and subfolder in the directory."
                }
              }
            }
          }
        },
        {
          name = "getExecutablePath",
          summary = "Get the path of the LÖVR executable.",
          description = "Returns the absolute path of the LÖVR executable.",
          key = "lovr.filesystem.getExecutablePath",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "path",
                  type = "string",
                  description = "The absolute path of the LÖVR executable, or `nil` if it is unknown."
                }
              }
            }
          }
        },
        {
          name = "getIdentity",
          summary = "Get the name of the save directory.",
          description = "Returns the identity of the game, which is used as the name of the save directory.  The default is `default`.  It can be changed using `t.identity` in `lovr.conf`.",
          key = "lovr.filesystem.getIdentity",
          module = "lovr.filesystem",
          notes = "On Android, this is always the package id (like `org.lovr.app`).",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "identity",
                  type = "string",
                  description = "The name of the save directory, or `nil` if it isn't set."
                }
              }
            }
          }
        },
        {
          name = "getLastModified",
          summary = "Get the modification time of a file.",
          description = "Returns when a file was last modified, since some arbitrary time in the past.",
          key = "lovr.filesystem.getLastModified",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The file to check."
                }
              },
              returns = {
                {
                  name = "time",
                  type = "number",
                  description = "The modification time of the file, in seconds, or `nil` if it's unknown."
                }
              }
            }
          }
        },
        {
          name = "getRealDirectory",
          summary = "Get the absolute path to a file.",
          description = "Get the absolute path of the mounted archive containing a path in the virtual filesystem.  This can be used to determine if a file is in the game's source directory or the save directory.",
          key = "lovr.filesystem.getRealDirectory",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The path to check."
                }
              },
              returns = {
                {
                  name = "realpath",
                  type = "string",
                  description = "The absolute path of the mounted archive containing `path`."
                }
              }
            }
          }
        },
        {
          name = "getRequirePath",
          summary = "Get the require path.",
          description = "Returns the require path.  The require path is a semicolon-separated list of patterns that LÖVR will use to search for files when they are `require`d.  Any question marks in the pattern will be replaced with the module that is being required.  It is similar to Lua\\'s `package.path` variable, but the main difference is that the patterns are relative to the virtual filesystem.",
          key = "lovr.filesystem.getRequirePath",
          module = "lovr.filesystem",
          notes = "The default reqiure path is '?.lua;?/init.lua'.",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "path",
                  type = "string",
                  description = "The semicolon separated list of search patterns."
                }
              }
            }
          }
        },
        {
          name = "getSaveDirectory",
          summary = "Get the location of the save directory.",
          description = "Returns the absolute path to the save directory.",
          key = "lovr.filesystem.getSaveDirectory",
          module = "lovr.filesystem",
          notes = "The save directory takes the following form:\n\n``` <appdata>/LOVR/<identity> ```\n\nWhere `<appdata>` is `lovr.filesystem.getAppdataDirectory` and `<identity>` is `lovr.filesystem.getIdentity` and can be customized using `lovr.conf`.",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "path",
                  type = "string",
                  description = "The absolute path to the save directory."
                }
              }
            }
          },
          related = {
            "lovr.filesystem.getIdentity",
            "lovr.filesystem.getAppdataDirectory"
          }
        },
        {
          name = "getSize",
          summary = "Get the size of a file.",
          description = "Returns the size of a file, in bytes.",
          key = "lovr.filesystem.getSize",
          module = "lovr.filesystem",
          notes = "If the file does not exist, an error is thrown.",
          variants = {
            {
              arguments = {
                {
                  name = "file",
                  type = "string",
                  description = "The file."
                }
              },
              returns = {
                {
                  name = "size",
                  type = "number",
                  description = "The size of the file, in bytes."
                }
              }
            }
          }
        },
        {
          name = "getSource",
          summary = "Get the location of the project source.",
          description = "Get the absolute path of the project's source directory or archive.",
          key = "lovr.filesystem.getSource",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "path",
                  type = "string",
                  description = "The absolute path of the project's source, or `nil` if it's unknown."
                }
              }
            }
          }
        },
        {
          name = "getUserDirectory",
          summary = "Get the location of the user's home directory.",
          description = "Returns the absolute path of the user's home directory.",
          key = "lovr.filesystem.getUserDirectory",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "path",
                  type = "string",
                  description = "The absolute path of the user's home directory."
                }
              }
            }
          }
        },
        {
          name = "getWorkingDirectory",
          summary = "Get the current working directory.",
          description = "Returns the absolute path of the working directory.  Usually this is where the executable was started from.",
          key = "lovr.filesystem.getWorkingDirectory",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "path",
                  type = "string",
                  description = "The current working directory, or `nil` if it's unknown."
                }
              }
            }
          }
        },
        {
          name = "isDirectory",
          summary = "Check whether a path is a directory.",
          description = "Check if a path exists and is a directory.",
          key = "lovr.filesystem.isDirectory",
          module = "lovr.filesystem",
          related = {
            "lovr.filesystem.isFile"
          },
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The path to check."
                }
              },
              returns = {
                {
                  name = "isDirectory",
                  type = "boolean",
                  description = "Whether or not the path is a directory."
                }
              }
            }
          }
        },
        {
          name = "isFile",
          summary = "Check whether a path is a file.",
          description = "Check if a path exists and is a file.",
          key = "lovr.filesystem.isFile",
          module = "lovr.filesystem",
          related = {
            "lovr.filesystem.isDirectory"
          },
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The path to check."
                }
              },
              returns = {
                {
                  name = "isFile",
                  type = "boolean",
                  description = "Whether or not the path is a file."
                }
              }
            }
          }
        },
        {
          name = "isFused",
          summary = "Check if the project is fused.",
          description = "Returns whether the current project source is fused to the executable.",
          key = "lovr.filesystem.isFused",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "fused",
                  type = "boolean",
                  description = "Whether or not the project is fused."
                }
              }
            }
          }
        },
        {
          name = "load",
          summary = "Load a file as Lua code.",
          description = "Load a file containing Lua code, returning a Lua chunk that can be run.",
          key = "lovr.filesystem.load",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The file to load."
                }
              },
              returns = {
                {
                  name = "chunk",
                  type = "function",
                  description = "The runnable chunk.",
                  arguments = {
                    {
                      name = "...",
                      type = "*"
                    }
                  },
                  returns = {
                    {
                      name = "...",
                      type = "*"
                    }
                  }
                }
              }
            }
          },
          notes = "An error is thrown if the file contains syntax errors.",
          examples = {
            {
              description = "Safely loading code:",
              code = "local success, chunk = pcall(lovr.filesystem.load, filename)\nif not success then\n  print('Oh no! There was an error: ' .. tostring(chunk))\nelse\n  local success, result = pcall(chunk)\n  print(success, result)\nend"
            }
          }
        },
        {
          name = "mount",
          summary = "Mount a directory or archive.",
          description = "Mounts a directory or `.zip` archive, adding it to the virtual filesystem.  This allows you to read files from it.",
          key = "lovr.filesystem.mount",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The path to mount."
                },
                {
                  name = "mountpoint",
                  type = "string",
                  description = "The path in the virtual filesystem to mount to.",
                  default = "'/'"
                },
                {
                  name = "append",
                  type = "boolean",
                  description = "Whether the archive will be added to the end or the beginning of the search path.",
                  default = "false"
                },
                {
                  name = "root",
                  type = "string",
                  description = "A subdirectory inside the archive to use as the root.  If `nil`, the actual root of the archive is used.",
                  default = "nil"
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether the archive was successfully mounted."
                }
              }
            }
          },
          notes = "The `append` option lets you control the priority of the archive's files in the event of naming collisions.\n\nThis function is not thread safe.  Mounting or unmounting an archive while other threads call lovr.filesystem functions is not supported.",
          related = {
            "lovr.filesystem.unmount"
          },
          examples = {
            {
              description = "Mount `data.zip` with a file `images/background.png`:",
              code = "lovr.filesystem.mount('data.zip', 'assets')\nprint(lovr.filesystem.isFile('assets/images/background.png')) -- true"
            }
          }
        },
        {
          name = "newBlob",
          summary = "Create a new Blob from a file.",
          description = "Creates a new Blob that contains the contents of a file.",
          key = "lovr.filesystem.newBlob",
          module = "lovr.filesystem",
          related = {
            "lovr.data.newBlob",
            "Blob"
          },
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The file to load."
                }
              },
              returns = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "The new Blob."
                }
              }
            }
          }
        },
        {
          name = "read",
          summary = "Read a file.",
          description = "Read the contents of a file.",
          key = "lovr.filesystem.read",
          module = "lovr.filesystem",
          notes = "If the file does not exist or cannot be read, nil is returned.",
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The name of the file to read."
                },
                {
                  name = "bytes",
                  type = "number",
                  description = "The number of bytes to read (if -1, all bytes will be read).",
                  default = "-1"
                }
              },
              returns = {
                {
                  name = "contents",
                  type = "string",
                  description = "The contents of the file."
                },
                {
                  name = "bytes",
                  type = "number",
                  description = "The number of bytes read from the file."
                }
              }
            }
          }
        },
        {
          name = "remove",
          summary = "Remove a file or directory.",
          description = "Remove a file or directory in the save directory.",
          key = "lovr.filesystem.remove",
          module = "lovr.filesystem",
          notes = "A directory can only be removed if it is empty.\n\nTo recursively remove a folder, use this function with `lovr.filesystem.getDirectoryItems`.",
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The file or directory to remove."
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether the path was removed."
                }
              }
            }
          }
        },
        {
          name = "setIdentity",
          summary = "Set the name of the save directory.",
          description = "Set the name of the save directory.",
          key = "lovr.filesystem.setIdentity",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {
                {
                  name = "identity",
                  type = "string",
                  description = "The new name of the save directory."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "setRequirePath",
          summary = "Set the require path.",
          description = "Sets the require path.  The require path is a semicolon-separated list of patterns that LÖVR will use to search for files when they are `require`d.  Any question marks in the pattern will be replaced with the module that is being required.  It is similar to Lua\\'s `package.path` variable, but the main difference is that the patterns are relative to the save directory and the project directory.",
          key = "lovr.filesystem.setRequirePath",
          module = "lovr.filesystem",
          notes = "The default reqiure path is '?.lua;?/init.lua'.",
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "An optional semicolon separated list of search patterns.",
                  default = "nil"
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "setSource",
          summary = "Set the project source.",
          description = "Sets the location of the project's source.  This can only be done once, and is usually done internally.",
          key = "lovr.filesystem.setSource",
          module = "lovr.filesystem",
          variants = {
            {
              arguments = {
                {
                  name = "identity",
                  type = "string",
                  description = "The path containing the project's source."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "unmount",
          summary = "Unmount a mounted archive.",
          description = "Unmounts a directory or archive previously mounted with `lovr.filesystem.mount`.",
          key = "lovr.filesystem.unmount",
          module = "lovr.filesystem",
          notes = "This function is not thread safe.  Mounting or unmounting an archive while other threads call lovr.filesystem functions is not supported.",
          variants = {
            {
              arguments = {
                {
                  name = "path",
                  type = "string",
                  description = "The path to unmount."
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether the archive was unmounted."
                }
              }
            }
          },
          related = {
            "lovr.filesystem.mount"
          }
        },
        {
          name = "write",
          summary = "Write to a file.",
          description = "Write to a file.",
          key = "lovr.filesystem.write",
          module = "lovr.filesystem",
          notes = "If the file does not exist, it is created.\n\nIf the file already has data in it, it will be replaced with the new content.",
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The file to write to."
                },
                {
                  name = "content",
                  type = "string",
                  description = "A string to write to the file."
                }
              },
              returns = {
                {
                  name = "bytes",
                  type = "number",
                  description = "The number of bytes written."
                }
              }
            },
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The file to write to."
                },
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob containing data to write to the file."
                }
              },
              returns = {
                {
                  name = "bytes",
                  type = "number",
                  description = "The number of bytes written."
                }
              }
            }
          },
          related = {
            "lovr.filesystem.append",
            "lovr.filesystem.read"
          }
        }
      },
      notes = "LÖVR programs can only write to a single directory, called the save directory.  The location of the save directory is platform-specific:\n\n<table>\n  <tr>\n    <td>Windows</td>\n    <td><code>C:\\Users\\&lt;user&gt;\\AppData\\Roaming\\LOVR\\&lt;identity&gt;</code></td>\n  </tr>\n  <tr>\n    <td>macOS</td>\n    <td><code>/Users/&lt;user&gt;/Library/Application Support/LOVR/&lt;identity&gt;</code></td>\n  </tr> </table>\n\n`<identity>` should be a unique identifier for your app.  It can be set either in `lovr.conf` or by using `lovr.filesystem.setIdentity`.\n\nAll filenames are relative to either the save directory or the directory containing the project source.  Files in the save directory take precedence over files in the project."
    },
    {
      name = "graphics",
      tag = "modules",
      summary = "Renders graphics using the GPU.",
      description = "The graphics module renders graphics and performs computation using the GPU.\n\nMost of the graphics functions are on the `Pass` object.",
      key = "lovr.graphics",
      objects = {
        {
          name = "Buffer",
          summary = "A block of memory on the GPU.",
          description = "A Buffer is a block of GPU memory.  Buffers are similar to Lua tables or arrays: they have a length and store a list of values.  The length of a Buffer and its format (the type of each value) are declared upfront and can't be changed.  Each value of a Buffer consists of one or more fields, and each field has a type.  For example, if a Buffer is used to store vertices, each value might store 3 fields for the position, normal vector, and UV coordinates of a vertex.\n\nBuffers are commonly used for:\n\n- Mesh data: Buffers hold the data that define the vertices in a mesh. Buffers also store the\n  vertex indices of a mesh, which define the order the vertices are connected together into\n  triangles. These are often called vertex buffers and index buffers.\n- Shader data: Buffers can be bound to a Shader, letting the Shader read arbitrary data. For\n  example, Lua code could create a Buffer with the positions and colors of all the lights in a\n  scene, which a Shader can use to do lighting calculations.\n- Compute: Compute shaders can write data to Buffers.  This GPU-generated data can be used in\n  later rendering work or sent back to Lua.\n- Indirect: Indirect rendering is an advanced technique where instructions for rendering work\n  are recorded to a Buffer (potentially by a compute shader) and later drawn.\n\nThere are two types of Buffers:\n\n- **Temporary** buffers are very inexpensive to create, and writing to them from Lua is fast.\n  However, they become invalid at the end of `lovr.draw` (i.e. when `lovr.graphics.submit` is\n  called).  The GPU is slightly slower at accessing data from temporary buffers, and compute\n  shaders can not write to them.  They are designed for storing data that changes every frame.\n- **Permanent** buffers are more expensive to create, and updating their contents requires a\n  transfer from CPU memory to VRAM.  They act like normal LÖVR objects and don't need to be\n  recreated every frame.  They often have faster performance when used by the GPU, and compute\n  shaders can write to them.  They are great for large pieces of data that are initialized once\n  at load time, or data that is updated infrequently.",
          key = "Buffer",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.getBuffer",
            "lovr.graphics.newBuffer"
          },
          methods = {
            {
              name = "clear",
              summary = "Clear the data in the Buffer.",
              description = "Clears some or all of the data in the **temporary** Buffer to zero.  Permanent Buffers can be cleared in a transfer pass using `Pass:clear`.",
              key = "Buffer:clear",
              module = "lovr.graphics",
              notes = "Clearing a permanent buffer requires the byte offset and byte count of the cleared range to be a multiple of 4.  This will usually be true for most data types.",
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the first item to clear.",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of items to clear.  If `nil`, clears to the end of the Buffer.",
                      default = "nil"
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "Pass:clear"
              }
            },
            {
              name = "getFormat",
              summary = "Get the format of the Buffer.",
              description = "Returns the format of the Buffer.  This is the list of fields that comprise each item in the buffer.  Each field has a type, byte offset, and vertex attribute location.",
              key = "Buffer:getFormat",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "format",
                      type = "table",
                      description = "The format of the Buffer."
                    }
                  }
                }
              },
              related = {
                "Buffer:getSize",
                "Buffer:getLength",
                "Buffer:getStride"
              },
              examples = {
                {
                  code = "function lovr.load()\n  buffer = lovr.graphics.newBuffer(1, { 'vec3', 'vec3', 'vec2' })\n\n  for i, field in ipairs(buffer:getFormat()) do\n    local type, offset, location = field.type, field.offset, field.location\n    local template = 'Field: %d: Type = %s, Offset = %d, Location = %d'\n    print(template:format(i, type, offset, location))\n  end\n\n  -- prints the following:\n  -- Field 1: Type = f32x3, Offset = 0, Location = 0\n  -- Field 2: Type = f32x3, Offset = 12, Location = 1\n  -- Field 3: Type = f32x2, Offset = 24, Location = 2\nend"
                }
              }
            },
            {
              name = "getLength",
              summary = "Get the length of the Buffer.",
              description = "Returns the length of the Buffer.",
              key = "Buffer:getLength",
              module = "lovr.graphics",
              related = {
                "Buffer:getSize",
                "Buffer:getStride"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "length",
                      type = "number",
                      description = "The length of the Buffer."
                    }
                  }
                }
              }
            },
            {
              name = "getPointer",
              summary = "Get a raw pointer to the Buffer's memory.",
              description = "Returns a raw pointer to the Buffer's memory as a lightuserdata, intended for use with the LuaJIT FFI or for passing to C libraries.  This is only available for temporary buffers, so the pointer is only valid until `lovr.graphics.submit` is called.",
              key = "Buffer:getPointer",
              module = "lovr.graphics",
              related = {
                "Blob:getPointer"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "pointer",
                      type = "lightuserdata",
                      description = "A pointer to the Buffer's memory."
                    }
                  }
                }
              }
            },
            {
              name = "getSize",
              summary = "Get the size of the Buffer, in bytes.",
              description = "Returns the size of the Buffer, in bytes.  This is the same as `length * stride`.",
              key = "Buffer:getSize",
              module = "lovr.graphics",
              related = {
                "Buffer:getLength",
                "Buffer:getStride"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "size",
                      type = "number",
                      description = "The size of the Buffer, in bytes."
                    }
                  }
                }
              }
            },
            {
              name = "getStride",
              summary = "Get the distance between each item in the Buffer, in bytes.",
              description = "Returns the distance between each item in the Buffer, in bytes.",
              key = "Buffer:getStride",
              module = "lovr.graphics",
              notes = "When a Buffer is created, the stride can be set explicitly, otherwise it will be automatically computed based on the fields in the Buffer.\n\nStrides can not be zero, and can not be smaller than the size of a single item.  To work around this, bind the Buffer as a storage buffer and fetch data from the buffer manually.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes."
                    }
                  }
                }
              },
              related = {
                "Buffer:getSize",
                "Buffer:getLength"
              }
            },
            {
              name = "isTemporary",
              summary = "Check if the Buffer is temporary.",
              description = "Returns whether the Buffer is temporary.",
              key = "Buffer:isTemporary",
              module = "lovr.graphics",
              related = {
                "lovr.graphics.getBuffer"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "temporary",
                      type = "boolean",
                      description = "Whether the Buffer is temporary."
                    }
                  }
                }
              }
            },
            {
              name = "setData",
              summary = "Change the data in the Buffer.",
              description = "Changes data in a temporary Buffer using a table or a Blob.  Permanent buffers can be changed using `Pass:copy`.",
              key = "Buffer:setData",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "data",
                      type = "table",
                      description = "A flat or nested table of items to copy to the Buffer (see notes for format)."
                    },
                    {
                      name = "sourceIndex",
                      type = "number",
                      description = "The index in the table to copy from.",
                      default = "1"
                    },
                    {
                      name = "destinationIndex",
                      type = "number",
                      description = "The index of the first value in the Buffer to update.",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of values to update.  `nil` will copy as many items as possible, based on the lengths of the source and destination.",
                      default = "nil"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "The Blob to copy data from."
                    },
                    {
                      name = "sourceOffset",
                      type = "number",
                      description = "A byte offset into the Blob to copy from.",
                      default = "0"
                    },
                    {
                      name = "destinationOffset",
                      type = "number",
                      description = "A byte offset in the Buffer to copy to.",
                      default = "0"
                    },
                    {
                      name = "size",
                      type = "number",
                      description = "The number of bytes to copy.  If nil, copies as many bytes as possible.",
                      default = "nil"
                    }
                  },
                  returns = {}
                }
              },
              notes = "When using a table, the table can contain a nested table for each value in the Buffer, or it can be a flat list of field component values.  It is not possible to mix both nested tables and flat values.\n\nFor each item updated, components of each field in the item (according to the Buffer's format) are read from either the nested subtable or the table itself.  A single number can be used to update a field with a scalar type.  Multiple numbers or a `lovr.math` vector can be used to update a field with a vector or mat4 type.  Multiple numbers can be used to update mat2 and mat3 fields.  When updating normalized field types, components read from the table will be clamped to the normalized range ([0,1] or [-1,1]).  In the Buffer, each field is written at its byte offset according to the Buffer's format, and subsequent items are separated by the byte stride of the Buffer.  Any missing components for an updated field will be set to zero.",
              related = {
                "Pass:copy"
              },
              examples = {
                {
                  code = "function lovr.draw(pass)\n  buffer = lovr.graphics.getBuffer(3, 'floats')\n  buffer:setData({ { 1.0 }, { 2.0 }, { 3.0 } })\n  buffer:setData({ 1.0, 2.0, 3.0 })\n\n  buffer = lovr.graphics.getBuffer(5, { 'vec3', 'vec3', 'vec2' })\n  buffer:setData({ vec3(1, 2, 3), vec3(4, 5, 6), vec2(7, 8) })\n  buffer:setData({ { 1, 2, 3, 4, 5, 6, 7, 8 } })\n  buffer:setData({ 1, 2, 3, 4, 5, 6, 7, 8 })\n  buffer:setData({\n    { x1, y1, z1, nx1, ny1, nz1, u1, v1 },\n    { x2, y2, z2, vec3(nx, ny, nz) }\n  }, 1, 3, 2)\nend"
                }
              }
            }
          }
        },
        {
          name = "Font",
          summary = "A Font used to render text.",
          description = "Font objects are used to render text with `Pass:text`.  The active font can be changed using `Pass:setFont`.  The default font is Varela Round, which is used when no font is active, and can be retrieved using `lovr.graphics.getDefaultFont`.  Custom fonts can be loaded from TTF files using `lovr.graphics.newFont`.\n\nEach Font uses a `Rasterizer` to load the TTF file and create images for each glyph. As text is drawn, the Font uploads images from the Rasterizer to a GPU texture atlas as needed.  The Font also performs text layout and mesh generation for strings of text.\n\nLÖVR uses a text rendering technique called \"multichannel signed distance fields\" (MSDF), which makes the font rendering remain crisp when text is viewed up close.",
          key = "Font",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.newFont",
            "lovr.graphics.getDefaultFont"
          },
          notes = "MSDF text requires a special shader to work.  LÖVR will automatically switch to this shader if no shader is active on the `Pass`.  This font shader is also available as a `DefaultShader`.",
          methods = {
            {
              name = "getAscent",
              summary = "Get the ascent of the Font.",
              description = "Returns the ascent of the font.  The ascent is the maximum amount glyphs ascend above the baseline.  The units depend on the font's pixel density.  With the default density, the units correspond to meters.",
              key = "Font:getAscent",
              module = "lovr.graphics",
              related = {
                "Rasterizer:getAscent",
                "Font:getDescent",
                "Font:getHeight",
                "Font:getKerning",
                "Font:getWidth"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "ascent",
                      type = "number",
                      description = "The ascent of the font."
                    }
                  }
                }
              }
            },
            {
              name = "getDescent",
              summary = "Get the descent of the Font.",
              description = "Returns the descent of the font.  The descent is the maximum amount glyphs descend below the baseline.  The units depend on the font's pixel density.  With the default density, the units correspond to meters.",
              key = "Font:getDescent",
              module = "lovr.graphics",
              related = {
                "Rasterizer:getDescent",
                "Font:getAscent",
                "Font:getHeight",
                "Font:getKerning",
                "Font:getWidth"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "descent",
                      type = "number",
                      description = "The descent of the font."
                    }
                  }
                }
              }
            },
            {
              name = "getHeight",
              summary = "Get the height of the Font.",
              description = "Returns the height of the font, sometimes also called the leading.  This is the full height of a line of text, including the space between lines.  Each line of a multiline string is separated on the y axis by this height, multiplied by the font's line spacing.  The units depend on the font's pixel density.  With the default density, the units correspond to meters.",
              key = "Font:getHeight",
              module = "lovr.graphics",
              related = {
                "Rasterizer:getLeading",
                "Font:getLineSpacing",
                "Font:setLineSpacing",
                "Font:getAscent",
                "Font:getDescent",
                "Font:getKerning",
                "Font:getWidth",
                "Font:getLines"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the font."
                    }
                  }
                }
              }
            },
            {
              name = "getKerning",
              summary = "Get the kerning between 2 glyphs.",
              description = "Returns the kerning between 2 glyphs.  Kerning is a slight horizontal adjustment between 2 glyphs to improve the visual appearance.  It will often be negative.  The units depend on the font's pixel density.  With the default density, the units correspond to meters.",
              key = "Font:getKerning",
              module = "lovr.graphics",
              related = {
                "Rasterizer:getKerning",
                "Font:getAscent",
                "Font:getDescent",
                "Font:getHeight",
                "Font:getWidth"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "first",
                      type = "string",
                      description = "The first character."
                    },
                    {
                      name = "second",
                      type = "string",
                      description = "The second letter."
                    }
                  },
                  returns = {
                    {
                      name = "keming",
                      type = "number",
                      description = "The kerning between the two glyphs."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "firstCodepoint",
                      type = "number",
                      description = "The first codepoint."
                    },
                    {
                      name = "second",
                      type = "string",
                      description = "The second letter."
                    }
                  },
                  returns = {
                    {
                      name = "keming",
                      type = "number",
                      description = "The kerning between the two glyphs."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "first",
                      type = "string",
                      description = "The first character."
                    },
                    {
                      name = "secondCodepoint",
                      type = "number",
                      description = "The second codepoint."
                    }
                  },
                  returns = {
                    {
                      name = "keming",
                      type = "number",
                      description = "The kerning between the two glyphs."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "firstCodepoint",
                      type = "number",
                      description = "The first codepoint."
                    },
                    {
                      name = "secondCodepoint",
                      type = "number",
                      description = "The second codepoint."
                    }
                  },
                  returns = {
                    {
                      name = "keming",
                      type = "number",
                      description = "The kerning between the two glyphs."
                    }
                  }
                }
              }
            },
            {
              name = "getLineSpacing",
              summary = "Get the line spacing of the Font.",
              description = "Returns the line spacing of the Font.  When spacing out lines, the height of the font is multiplied the line spacing to get the final spacing value.  The default is 1.0.",
              key = "Font:getLineSpacing",
              module = "lovr.graphics",
              related = {
                "Font:getHeight"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "spacing",
                      type = "number",
                      description = "The line spacing of the font."
                    }
                  }
                }
              }
            },
            {
              name = "getLines",
              summary = "Wrap a string into a sequence of lines.",
              description = "Returns a table of wrapped lines for a piece of text, given a line length limit.  Newlines are handled correctly.  The wrap limit units depend on the pixel density of the font.  With the default pixel density, the units correspond to meters when the font is drawn at 1.0 scale.",
              key = "Font:getLines",
              module = "lovr.graphics",
              related = {
                "Font:getWidth",
                "Font:getHeight",
                "Pass:text"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "string",
                      type = "string",
                      description = "The text to wrap."
                    },
                    {
                      name = "wrap",
                      type = "number",
                      description = "The line length to wrap at."
                    }
                  },
                  returns = {
                    {
                      name = "lines",
                      type = "table",
                      description = "A table strings, one for each wrapped line (without any color information)."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "strings",
                      type = "table",
                      description = "A table of colored strings, each given as a `{ color, string }` pair.  The color can be a `Vec3`, `Vec4`, table, or hexcode."
                    },
                    {
                      name = "wrap",
                      type = "number",
                      description = "The line length to wrap at."
                    }
                  },
                  returns = {
                    {
                      name = "lines",
                      type = "table",
                      description = "A table strings, one for each wrapped line (without any color information)."
                    }
                  }
                }
              }
            },
            {
              name = "getPixelDensity",
              summary = "Get the pixel density of the Font.",
              description = "Returns the pixel density of the font.  The density is a \"pixels per world unit\" factor that controls how the pixels in the font's texture are mapped to units in the coordinate space.\n\nThe default pixel density is set to the height of the font.  This means that lines of text rendered with a scale of 1.0 come out to 1 unit (meter) tall.  However, if this font was drawn to a 2D texture where the units are in pixels, the font would still be drawn 1 unit (pixel) tall!  Scaling the coordinate space or the size of the text by the height of the font would fix this.  However, a more convenient option is to set the pixel density of the font to 1.0 when doing 2D rendering to make the font's size match up with the pixels of the canvas.",
              key = "Font:getPixelDensity",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "density",
                      type = "number",
                      description = "The pixel density of the font."
                    }
                  }
                }
              }
            },
            {
              name = "getRasterizer",
              summary = "Get the Font's Rasterizer.",
              description = "Returns the Rasterizer object backing the Font.",
              key = "Font:getRasterizer",
              module = "lovr.graphics",
              related = {
                "lovr.graphics.newFont",
                "lovr.data.newRasterizer"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "rasterizer",
                      type = "Rasterizer",
                      description = "The Rasterizer."
                    }
                  }
                }
              }
            },
            {
              name = "getVertices",
              summary = "Get the vertices for a piece of text.",
              description = "Returns a table of vertices for a piece of text, along with a Material to use when rendering it. The Material returned by this function may not be the same if the Font's texture atlas needs to be recreated with a bigger size to make room for more glyphs.",
              key = "Font:getVertices",
              module = "lovr.graphics",
              notes = "Each vertex is a table of 4 floating point numbers with the following data:\n\n    { x, y, u, v }\n\nThese could be placed in a vertex buffer using the following buffer format:\n\n    { 'vec2:VertexPosition', 'vec2:VertexUV' }",
              variants = {
                {
                  arguments = {
                    {
                      name = "halign",
                      type = "HorizontalAlign",
                      description = "The horizontal align."
                    },
                    wrap = {
                      type = "number",
                      description = "        The maximum line length.  The units depend on the pixel density of the font, but are in\n        meters by default.\n      ",
                      default = "0"
                    },
                    string = {
                      type = "string",
                      description = "The text to render."
                    },
                    [2] = {
                      name = "valign",
                      type = "VerticalAlign",
                      description = "The vertical align."
                    },
                    strings = {
                      type = "table",
                      description = "        A table of colored strings, each given as a `{ color, string }` pair.  The color can be a\n        `Vec3`, `Vec4`, table, or hexcode.\n      "
                    }
                  },
                  returns = {
                    {
                      name = "vertices",
                      type = "table",
                      description = "The table of vertices.  See below for the format of each vertex."
                    },
                    {
                      name = "material",
                      type = "Material",
                      description = "A Material to use when rendering the vertices."
                    }
                  }
                }
              }
            },
            {
              name = "getWidth",
              summary = "Get the width of rendered text.",
              description = "Returns the maximum width of a piece of text.  This function does not perform wrapping but does respect newlines in the text.",
              key = "Font:getWidth",
              module = "lovr.graphics",
              related = {
                "Font:getAscent",
                "Font:getDescent",
                "Font:getHeight",
                "Font:getKerning",
                "Font:getLines"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "string",
                      type = "string",
                      description = "The text to measure."
                    }
                  },
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The maximum width of the text."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "strings",
                      type = "table",
                      description = "A table of colored strings, each given as a `{ color, string }` pair.  The color can be a `Vec3`, `Vec4`, table, or hexcode."
                    }
                  },
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The maximum width of the text."
                    }
                  }
                }
              }
            },
            {
              name = "setLineSpacing",
              summary = "Set the line spacing of the Font.",
              description = "Sets the line spacing of the Font.  When spacing out lines, the height of the font is multiplied the line spacing to get the final spacing value.  The default is 1.0.",
              key = "Font:setLineSpacing",
              module = "lovr.graphics",
              related = {
                "Font:getHeight"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "spacing",
                      type = "number",
                      description = "The new line spacing."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setPixelDensity",
              summary = "Set the pixel density of the Font.",
              description = "Returns the pixel density of the font.  The density is a \"pixels per world unit\" factor that controls how the pixels in the font's texture are mapped to units in the coordinate space.\n\nThe default pixel density is set to the height of the font.  This means that lines of text rendered with a scale of 1.0 come out to 1 unit (meter) tall.  However, if this font was drawn to a 2D texture where the units are in pixels, the font would still be drawn 1 unit (pixel) tall!  Scaling the coordinate space or the size of the text by the height of the font would fix this.  However, a more convenient option is to set the pixel density of the font to 1.0 when doing 2D rendering to make the font's size match up with the pixels of the canvas.",
              key = "Font:setPixelDensity",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "density",
                      type = "number",
                      description = "The new pixel density of the font."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Resets the pixel density to the default, which is given by `Font:getHeight`.",
                  arguments = {},
                  returns = {}
                }
              }
            }
          }
        },
        {
          name = "Material",
          summary = "A set of properties and textures that define the properties of a surface.",
          description = "Materials are a set of properties and textures that define the properties of a surface, like what color it is, how bumpy or shiny it is, etc. `Shader` code can use the data from a material to compute lighting.\n\nMaterials are immutable, and can't be changed after they are created.  Instead, a new Material should be created with the updated properties.\n\n`Pass:setMaterial` changes the active material, causing it to affect rendering until the active material is changed again.\n\nUsing material objects is optional.  `Pass:setMaterial` can take a `Texture`, and `Pass:setColor` can change the color of objects, so basic tinting and texturing of surfaces does not require a full material to be created.  Also, a custom material system could be developed by sending textures and other data to shaders manually.\n\n`Model` objects will create materials for all of the materials defined in the model file.\n\nIn shader code, non-texture material properties can be accessed as `Material.<property>`, and material textures can be accessed as `<Type>Texture`, e.g. `RoughnessTexture`.",
          key = "Material",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.newMaterial"
          },
          methods = {
            {
              name = "getProperties",
              summary = "Get the properties of the Material.",
              description = "Returns the properties of the Material in a table.",
              key = "Material:getProperties",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "properties",
                      type = "table",
                      description = "The Material properties."
                    }
                  }
                }
              }
            },
            {
              name = "setTexture",
              summary = "Set a texture for the Material.",
              description = "Sets a texture for a Material.  Several predefined `MaterialTexture`s are supported.  Any texture that is `nil` will use a single white pixel as a fallback.",
              key = "Material:setTexture",
              module = "lovr.graphics",
              notes = "Textures must have a `TextureType` of `2d` to be used with Materials.",
              variants = {
                {
                  arguments = {
                    {
                      name = "textureType",
                      type = "MaterialTexture",
                      description = "The type of texture to set.",
                      default = "'diffuse'"
                    },
                    {
                      name = "texture",
                      type = "Texture",
                      description = "The texture to apply, or `nil` to use the default."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "texture",
                      type = "Texture",
                      description = "The texture to apply, or `nil` to use the default."
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "MaterialTexture",
                "lovr.graphics.newTexture"
              }
            }
          }
        },
        {
          name = "Model",
          summary = "A 3D model.",
          description = "Models are 3D model assets loaded from files.  Currently, OBJ, glTF, and binary STL files are supported.\n\nA model can be drawn using `Pass:draw`.\n\nThe raw CPU data for a model is held in a `ModelData` object, which can be loaded on threads or reused for multiple Model instances.\n\nModels have a hierarchy of nodes which can have their transforms modified.  Meshes are attached to these nodes.  The same mesh can be attached to multiple nodes, allowing it to be drawn multiple times while only storing a single copy of its data.\n\nModels can have animations.  Animations have keyframes which affect the transforms of nodes. Right now each model can only be drawn with a single animated pose per frame.\n\nModels can have materials, which are collections of properties and textures that define how its surface is affected by lighting.  Each mesh in the model can use a single material.",
          key = "Model",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.newModel",
            "lovr.headset.newModel"
          },
          methods = {
            {
              name = "animate",
              summary = "Animate the Model.",
              description = "Animates a Model by setting or blending the transforms of nodes using data stored in the keyframes of an animation.\n\nThe animation from the model file is evaluated at the timestamp, resulting in a set of node properties.  These properties are then applied to the nodes in the model, using an optional blend factor.  If the animation doesn't have keyframes that target a given node, the node will remain unchanged.",
              key = "Model:animate",
              module = "lovr.graphics",
              notes = "If the timestamp is larger than the duration of the animation, it will wrap back around to zero, so looping an animation doesn't require using the modulo operator.\n\nTo change the speed of the animation, multiply the timestamp by a speed factor.\n\nFor each animated property in the animation, if the timestamp used for the animation is less than the timestamp of the first keyframe, the data of the first keyframe will be used.\n\nThis function can be called multiple times to layer and blend animations.  The model joints will be drawn in the final resulting pose.\n\n`Model:resetNodeTransforms` can be used to reset the model nodes to their initial transforms, which is helpful to ensure animating starts from a clean slate.",
              variants = {
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of an animation in the model file."
                    },
                    {
                      name = "time",
                      type = "number",
                      description = "The timestamp to evaluate the keyframes at, in seconds."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "How much of the animation's pose to blend into the nodes, from 0 to 1.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of an animation in the model file."
                    },
                    {
                      name = "time",
                      type = "number",
                      description = "The timestamp to evaluate the keyframes at, in seconds."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "How much of the animation's pose to blend into the nodes, from 0 to 1.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "Model:resetNodeTransforms",
                "Model:getAnimationCount",
                "Model:getAnimationName",
                "Model:getAnimationDuration",
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodeTransform",
                "Model:setNodeTransform"
              }
            },
            {
              name = "getAnimationCount",
              summary = "Get the number of animations in the Model.",
              description = "Returns the number of animations in the Model.",
              key = "Model:getAnimationCount",
              module = "lovr.graphics",
              related = {
                "Model:getAnimationName",
                "Model:getAnimationDuration",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of animations in the Model."
                    }
                  }
                }
              }
            },
            {
              name = "getAnimationDuration",
              summary = "Get the duration of an animation in the Model.",
              description = "Returns the duration of an animation in the Model, in seconds.",
              key = "Model:getAnimationDuration",
              module = "lovr.graphics",
              notes = "The duration of an animation is calculated as the largest timestamp of all of its keyframes.",
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The animation index."
                    }
                  },
                  returns = {
                    {
                      name = "duration",
                      type = "number",
                      description = "The duration of the animation, in seconds."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the animation."
                    }
                  },
                  returns = {
                    {
                      name = "duration",
                      type = "number",
                      description = "The duration of the animation, in seconds."
                    }
                  }
                }
              },
              related = {
                "Model:getAnimationCount",
                "Model:getAnimationName",
                "Model:animate"
              }
            },
            {
              name = "getAnimationName",
              summary = "Get the name of an animation in the Model.",
              description = "Returns the name of an animation in the Model.",
              key = "Model:getAnimationName",
              module = "lovr.graphics",
              related = {
                "Model:getAnimationCount",
                "Model:getAnimationDuration"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of an animation."
                    }
                  },
                  returns = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the animation."
                    }
                  }
                }
              }
            },
            {
              name = "getBoundingBox",
              summary = "Get the bounding box of the Model.",
              description = "Returns the 6 values of the Model's axis-aligned bounding box.",
              key = "Model:getBoundingBox",
              module = "lovr.graphics",
              related = {
                "Model:getWidth",
                "Model:getHeight",
                "Model:getDepth",
                "Model:getDimensions",
                "Model:getCenter",
                "Model:getBoundingSphere",
                "ModelData:getBoundingBox",
                "Collider:getAABB"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "minx",
                      type = "number",
                      description = "The minimum x coordinate of the vertices in the Model."
                    },
                    {
                      name = "maxx",
                      type = "number",
                      description = "The maximum x coordinate of the vertices in the Model."
                    },
                    {
                      name = "miny",
                      type = "number",
                      description = "The minimum y coordinate of the vertices in the Model."
                    },
                    {
                      name = "maxy",
                      type = "number",
                      description = "The maximum y coordinate of the vertices in the Model."
                    },
                    {
                      name = "minz",
                      type = "number",
                      description = "The minimum z coordinate of the vertices in the Model."
                    },
                    {
                      name = "maxz",
                      type = "number",
                      description = "The maximum z coordinate of the vertices in the Model."
                    }
                  }
                }
              }
            },
            {
              name = "getBoundingSphere",
              summary = "Get the bounding sphere of the Model.",
              description = "Returns a sphere approximately enclosing the vertices in the Model.",
              key = "Model:getBoundingSphere",
              module = "lovr.graphics",
              related = {
                "Model:getWidth",
                "Model:getHeight",
                "Model:getDepth",
                "Model:getDimensions",
                "Model:getCenter",
                "Model:getBoundingBox",
                "ModelData:getBoundingSphere"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the position of the sphere."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the position of the sphere."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the position of the sphere."
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the bounding sphere."
                    }
                  }
                }
              }
            },
            {
              name = "getCenter",
              summary = "Get the center of the Model's bounding box.",
              description = "Returns the center of the Model's axis-aligned bounding box, relative to the Model's origin.",
              key = "Model:getCenter",
              module = "lovr.graphics",
              related = {
                "Model:getWidth",
                "Model:getHeight",
                "Model:getDepth",
                "Model:getDimensions",
                "Model:getBoundingBox",
                "ModelData:getCenter"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x offset of the center of the bounding box."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y offset of the center of the bounding box."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z offset of the center of the bounding box."
                    }
                  }
                }
              }
            },
            {
              name = "getData",
              summary = "Get the ModelData backing the Model.",
              description = "Returns the ModelData this Model was created from.",
              key = "Model:getData",
              module = "lovr.graphics",
              related = {
                "lovr.data.newModelData"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "data",
                      type = "ModelData",
                      description = "The ModelData."
                    }
                  }
                }
              }
            },
            {
              name = "getDepth",
              summary = "Get the depth of the Model.",
              description = "Returns the depth of the Model, computed from its axis-aligned bounding box.",
              key = "Model:getDepth",
              module = "lovr.graphics",
              related = {
                "Model:getWidth",
                "Model:getHeight",
                "Model:getDimensions",
                "Model:getCenter",
                "Model:getBoundingBox",
                "ModelData:getDepth"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "depth",
                      type = "number",
                      description = "The depth of the Model."
                    }
                  }
                }
              }
            },
            {
              name = "getDimensions",
              summary = "Get the dimensions of the Model.",
              description = "Returns the width, height, and depth of the Model, computed from its axis-aligned bounding box.",
              key = "Model:getDimensions",
              module = "lovr.graphics",
              related = {
                "Model:getWidth",
                "Model:getHeight",
                "Model:getDepth",
                "Model:getCenter",
                "Model:getBoundingBox",
                "ModelData:getDimensions"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the Model."
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the Model."
                    },
                    {
                      name = "depth",
                      type = "number",
                      description = "The depth of the Model."
                    }
                  }
                }
              }
            },
            {
              name = "getHeight",
              summary = "Get the height of the Model.",
              description = "Returns the height of the Model, computed from its axis-aligned bounding box.",
              key = "Model:getHeight",
              module = "lovr.graphics",
              related = {
                "Model:getWidth",
                "Model:getDepth",
                "Model:getDimensions",
                "Model:getCenter",
                "Model:getBoundingBox",
                "ModelData:getHeight"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the Model."
                    }
                  }
                }
              }
            },
            {
              name = "getIndexBuffer",
              summary = "Get a Buffer containing the triangle indices in the Model.",
              description = "Returns the index buffer used by the Model.  The index buffer describes the order used to draw the vertices in each mesh.",
              key = "Model:getIndexBuffer",
              module = "lovr.graphics",
              related = {
                "Model:getVertexBuffer"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "The index buffer."
                    }
                  }
                }
              }
            },
            {
              name = "getMaterial",
              summary = "Get a Material from the Model.",
              description = "Returns a `Material` loaded from the Model.",
              key = "Model:getMaterial",
              module = "lovr.graphics",
              related = {
                "Model:getMaterialCount",
                "Model:getMaterialName",
                "Model:getNodeDraw"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the Material to return."
                    }
                  },
                  returns = {
                    {
                      name = "material",
                      type = "Material",
                      description = "The material."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the Material to return."
                    }
                  },
                  returns = {
                    {
                      name = "material",
                      type = "Material",
                      description = "The material."
                    }
                  }
                }
              }
            },
            {
              name = "getMaterialCount",
              summary = "Get the number of materials in the Model.",
              description = "Returns the number of materials in the Model.",
              key = "Model:getMaterialCount",
              module = "lovr.graphics",
              related = {
                "Model:getMaterialName",
                "Model:getMaterial"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of materials in the Model."
                    }
                  }
                }
              }
            },
            {
              name = "getMaterialName",
              summary = "Get the name of a material in the Model.",
              description = "Returns the name of a material in the Model.",
              key = "Model:getMaterialName",
              module = "lovr.graphics",
              related = {
                "Model:getMaterialCount",
                "Model:getMaterial"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a material."
                    }
                  },
                  returns = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the material."
                    }
                  }
                }
              }
            },
            {
              name = "getMetadata",
              summary = "Get extra information from the model file.",
              description = "Returns extra information stored in the model file.  Currently this is only implemented for glTF models and returns the JSON string from the glTF or glb file.  The metadata can be used to get application-specific data or add support for glTF extensions not supported by LÖVR.",
              key = "Model:getMetadata",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "metadata",
                      type = "string",
                      description = "The metadata from the model file."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeChildren",
              summary = "Get the children of a node.",
              description = "Given a parent node, this function returns a table with the indices of its children.",
              key = "Model:getNodeChildren",
              module = "lovr.graphics",
              notes = "If the node does not have any children, this function returns an empty table.",
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the parent node."
                    }
                  },
                  returns = {
                    {
                      name = "children",
                      type = "table",
                      description = "A table containing a node index for each child of the node."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the parent node."
                    }
                  },
                  returns = {
                    {
                      name = "children",
                      type = "table",
                      description = "A table containing a node index for each child of the node."
                    }
                  }
                }
              },
              related = {
                "Model:getNodeParent",
                "Model:getRootNode"
              }
            },
            {
              name = "getNodeCount",
              summary = "Get the number of nodes in the model.",
              description = "Returns the number of nodes in the model.",
              key = "Model:getNodeCount",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of nodes in the model."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeDraw",
              summary = "Get the information needed to draw one mesh attached to a node.",
              description = "Returns the draw mode, material, and vertex range of a mesh in the model.",
              key = "Model:getNodeDraw",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "node",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the draw."
                    }
                  },
                  returns = {
                    {
                      name = "mode",
                      type = "MeshMode",
                      description = "Whether the vertices are points, lines, or triangles."
                    },
                    {
                      name = "material",
                      type = "Material",
                      description = "The Material used by the draw."
                    },
                    {
                      name = "start",
                      type = "number",
                      description = "The offset of the first vertex in the draw."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices in the draw."
                    },
                    {
                      name = "base",
                      type = "number",
                      description = "The base vertex of the draw (added to each instance value), or nil if the draw does not use an index buffer."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the draw."
                    }
                  },
                  returns = {
                    {
                      name = "mode",
                      type = "MeshMode",
                      description = "Whether the vertices are points, lines, or triangles."
                    },
                    {
                      name = "material",
                      type = "Material",
                      description = "The Material used by the draw."
                    },
                    {
                      name = "start",
                      type = "number",
                      description = "The offset of the first vertex in the draw."
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices in the draw."
                    },
                    {
                      name = "base",
                      type = "number",
                      description = "The base vertex of the draw (added to each instance value), or nil if the draw does not use an index buffer."
                    }
                  }
                }
              },
              related = {
                "Pass:setMeshMode",
                "Pass:setMaterial",
                "Pass:mesh",
                "Model:getVertexBuffer",
                "Model:getIndexBuffer"
              },
              examples = {
                {
                  code = "function lovr.load()\n  local m = lovr.graphics.newModel('enraged-gorilla.gltf')\n\n  model = {\n    object = m,\n    data = m:getData(),\n    vertices = m:getVertexBuffer(),\n    indices = m:getIndexBuffer()\n  }\nend\n\nlocal function drawNode(model, pass, i)\n  for j = 1, model.object:getNodeDrawCount(i) do\n    local mode, material, start, count, base = model.object:getNodeDraw(i, j)\n    local transform = mat4(model.object:getNodeTransform(i))\n\n    pass:setMeshMode(mode)\n    pass:setMaterial(material)\n\n    if base then\n      pass:mesh(model.vertices, model.indices, transform, start, count, 1, base)\n    else\n      pass:mesh(model.vertices, transform, start, count)\n    end\n  end\n\n  for _, index in ipairs(model.data:getNodeChildren(i)) do\n    drawNode(model, pass, index)\n  end\nend\n\nfunction lovr.draw(pass)\n  drawNode(model, pass, model.data:getRootNode())\nend"
                }
              }
            },
            {
              name = "getNodeDrawCount",
              summary = "Get the number of meshes attached to a node.",
              description = "Returns the number of meshes attached to a node.  Each mesh is drawn individually.",
              key = "Model:getNodeDrawCount",
              module = "lovr.graphics",
              related = {
                "ModelData:getNodeMeshCount",
                "Model:getNodeDraw"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a node."
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of draws in the node."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of a node."
                    }
                  },
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of draws in the node."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeName",
              summary = "Get the name of a node in the Model.",
              description = "Returns the name of a node.",
              key = "Model:getNodeName",
              module = "lovr.graphics",
              related = {
                "Model:getNodeCount",
                "Model:getAnimationName",
                "Model:getMaterialName"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    }
                  },
                  returns = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeOrientation",
              summary = "Get the orientation of a node.",
              description = "Returns the orientation of a node.",
              key = "Model:getNodeOrientation",
              module = "lovr.graphics",
              related = {
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodePose",
                "Model:setNodePose",
                "Model:getNodeTransform",
                "Model:setNodeTransform",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "origin",
                      type = "OriginType",
                      description = "Whether the orientation should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "origin",
                      type = "OriginType",
                      description = "Whether the orientation should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeParent",
              summary = "Get the parent of a node.",
              description = "Given a child node, this function returns the index of its parent.",
              key = "Model:getNodeParent",
              module = "lovr.graphics",
              related = {
                "Model:getNodeChildren",
                "Model:getRootNode"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the child node."
                    }
                  },
                  returns = {
                    {
                      name = "parent",
                      type = "number",
                      description = "The index of the parent."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the child node."
                    }
                  },
                  returns = {
                    {
                      name = "parent",
                      type = "number",
                      description = "The index of the parent."
                    }
                  }
                }
              }
            },
            {
              name = "getNodePose",
              summary = "Get the pose of a node.",
              description = "Returns the pose (position and orientation) of a node.",
              key = "Model:getNodePose",
              module = "lovr.graphics",
              related = {
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodeTransform",
                "Model:setNodeTransform",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a node."
                    },
                    {
                      name = "origin",
                      type = "OriginType",
                      description = "Whether the pose should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the node."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the node."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the node."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of a node."
                    },
                    {
                      name = "origin",
                      type = "OriginType",
                      description = "Whether the pose should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the node."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the node."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the node."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getNodePosition",
              summary = "Get the position of a node.",
              description = "Returns the position of a node.",
              key = "Model:getNodePosition",
              module = "lovr.graphics",
              related = {
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodePose",
                "Model:setNodePose",
                "Model:getNodeTransform",
                "Model:setNodeTransform",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "space",
                      type = "OriginType",
                      description = "Whether the position should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "space",
                      type = "OriginType",
                      description = "Whether the position should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    }
                  }
                }
              }
            },
            {
              name = "getNodeScale",
              summary = "Get the scale of a node.",
              description = "Returns the scale of a node.",
              key = "Model:getNodeScale",
              module = "lovr.graphics",
              notes = "For best results when animating, it's recommended to keep the 3 components of the scale the same.",
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "origin",
                      type = "OriginType",
                      description = "Whether the scale should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x scale."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y scale."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z scale."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "origin",
                      type = "OriginType",
                      description = "Whether the scale should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x scale."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y scale."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z scale."
                    }
                  }
                }
              },
              related = {
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodePose",
                "Model:setNodePose",
                "Model:getNodeTransform",
                "Model:setNodeTransform",
                "Model:animate"
              }
            },
            {
              name = "getNodeTransform",
              summary = "Get the transform of a node.",
              description = "Returns the transform (position, scale, and rotation) of a node.",
              key = "Model:getNodeTransform",
              module = "lovr.graphics",
              related = {
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodePose",
                "Model:setNodePose",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of a node."
                    },
                    {
                      name = "origin",
                      type = "OriginType",
                      description = "Whether the transform should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the node."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the node."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the node."
                    },
                    {
                      name = "sx",
                      type = "number",
                      description = "The x scale of the node."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y scale of the node."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z scale of the node."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of a node."
                    },
                    {
                      name = "origin",
                      type = "OriginType",
                      description = "Whether the transform should be returned relative to the root node or the node's parent.",
                      default = "'root'"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the node."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the node."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the node."
                    },
                    {
                      name = "sx",
                      type = "number",
                      description = "The x scale of the node."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y scale of the node."
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z scale of the node."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the node is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getRootNode",
              summary = "Get the index of the root node.",
              description = "Returns the index of the model's root node.",
              key = "Model:getRootNode",
              module = "lovr.graphics",
              related = {
                "Model:getNodeCount",
                "Model:getNodeParent"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "root",
                      type = "number",
                      description = "The index of the root node."
                    }
                  }
                }
              }
            },
            {
              name = "getTexture",
              summary = "Get one of the textures in the Model.",
              description = "Returns one of the textures in the Model.",
              key = "Model:getTexture",
              module = "lovr.graphics",
              related = {
                "Model:getTextureCount",
                "Model:getMaterial"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the texture to get."
                    }
                  },
                  returns = {
                    {
                      name = "texture",
                      type = "Texture",
                      description = "The texture."
                    }
                  }
                }
              }
            },
            {
              name = "getTextureCount",
              summary = "Get the number of textures in the Model.",
              description = "Returns the number of textures in the Model.",
              key = "Model:getTextureCount",
              module = "lovr.graphics",
              related = {
                "Model:getTexture"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of textures in the Model."
                    }
                  }
                }
              }
            },
            {
              name = "getTriangleCount",
              summary = "Get the total number of triangles in the Model.",
              description = "Returns the total number of triangles in the Model.",
              key = "Model:getTriangleCount",
              module = "lovr.graphics",
              notes = "This isn't always related to the length of the vertex buffer, since a mesh in the Model could be drawn by multiple nodes.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The total number of triangles in the Model."
                    }
                  }
                }
              },
              related = {
                "Model:getTriangles",
                "Model:getVertexCount",
                "ModelData:getTriangleCount"
              }
            },
            {
              name = "getTriangles",
              summary = "Get all the triangles in the Model.",
              description = "Returns 2 tables containing mesh data for the Model.\n\nThe first table is a list of vertex positions and contains 3 numbers for the x, y, and z coordinate of each vertex.  The second table is a list of triangles and contains 1-based indices into the first table representing the first, second, and third vertices that make up each triangle.\n\nThe vertex positions will be affected by node transforms.",
              key = "Model:getTriangles",
              module = "lovr.graphics",
              notes = "After this function is called on a Model once, the result is cached (in its ModelData).",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "vertices",
                      type = "table",
                      description = "The triangle vertex positions, returned as a flat (non-nested) table of numbers.  The position of each vertex is given as an x, y, and z coordinate."
                    },
                    {
                      name = "indices",
                      type = "table",
                      description = "The vertex indices.  Every 3 indices describes a triangle."
                    }
                  }
                }
              },
              related = {
                "Model:getTriangleCount",
                "Model:getVertexCount",
                "ModelData:getTriangles"
              }
            },
            {
              name = "getVertexBuffer",
              summary = "Get a Buffer containing the vertices in the Model.",
              description = "Returns a `Buffer` that holds the vertices of all of the meshes in the Model.",
              key = "Model:getVertexBuffer",
              module = "lovr.graphics",
              related = {
                "Model:getIndexBuffer"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "The vertex buffer."
                    }
                  }
                }
              }
            },
            {
              name = "getVertexCount",
              summary = "Get the total vertex count of the Model.",
              description = "Returns the total vertex count of the Model.",
              key = "Model:getVertexCount",
              module = "lovr.graphics",
              notes = "This isn't always the same as the length of the vertex buffer, since a mesh in the Model could be drawn by multiple nodes.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The total number of vertices."
                    }
                  }
                }
              },
              related = {
                "Model:getTriangles",
                "Model:getTriangleCount",
                "ModelData:getVertexCount"
              }
            },
            {
              name = "getWidth",
              summary = "Get the width of the Model.",
              description = "Returns the width of the Model, computed from its axis-aligned bounding box.",
              key = "Model:getWidth",
              module = "lovr.graphics",
              related = {
                "Model:getHeight",
                "Model:getDepth",
                "Model:getDimensions",
                "Model:getCenter",
                "Model:getBoundingBox",
                "ModelData:getWidth"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the Model."
                    }
                  }
                }
              }
            },
            {
              name = "hasJoints",
              summary = "Check if the Model uses joints for skeletal animation.",
              description = "Returns whether the Model has any skeletal animations.",
              key = "Model:hasJoints",
              module = "lovr.graphics",
              notes = "This will return when there's at least one skin in the model, as returned by `ModelData:getSkinCount`.\n\nEven if this function returns true, the model could still have non-skeletal animations.\n\nRight now a model can only be drawn with one skeletal pose per frame.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "jointed",
                      type = "boolean",
                      description = "Whether the animation uses joint nodes for skeletal animation."
                    }
                  }
                }
              }
            },
            {
              name = "setNodeOrientation",
              summary = "Set or blend the orientation of a node.",
              description = "Sets or blends the orientation of a node to a new orientation.",
              key = "Model:setNodeOrientation",
              module = "lovr.graphics",
              related = {
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodePose",
                "Model:setNodePose",
                "Model:getNodeTransform",
                "Model:setNodeTransform",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "orientation",
                      type = "rotation",
                      description = "The target orientation."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target orientation to blend in.  A value of 0 will not change the node's orientation at all, whereas 1 will fully blend to the target orientation.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "orientation",
                      type = "rotation",
                      description = "The target orientation."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target orientation to blend in.  A value of 0 will not change the node's orientation at all, whereas 1 will fully blend to the target orientation.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setNodePose",
              summary = "Set or blend the pose of a node.",
              description = "Sets or blends the pose (position and orientation) of a node to a new pose.",
              key = "Model:setNodePose",
              module = "lovr.graphics",
              related = {
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodeTransform",
                "Model:setNodeTransform",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The target position.  Can also be provided as 3 numbers."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The target orientation.  Can also be provided as 4 numbers in angle-axis form."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target pose to blend in.  A value of 0 will not change the node's pose at all, whereas 1 will fully blend to the target pose.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The target position.  Can also be provided as 3 numbers."
                    },
                    {
                      name = "orientation",
                      type = "Quat",
                      description = "The target orientation.  Can also be provided as 4 numbers in angle-axis form."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target pose to blend in.  A value of 0 will not change the node's pose at all, whereas 1 will fully blend to the target pose.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setNodePosition",
              summary = "Set or blend the position of a node.",
              description = "Sets or blends the position of a node to a new position.",
              key = "Model:setNodePosition",
              module = "lovr.graphics",
              related = {
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodePose",
                "Model:setNodePose",
                "Model:getNodeTransform",
                "Model:setNodeTransform",
                "Model:animate"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The target position.  Can also be provided as 3 numbers."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target position to blend in.  A value of 0 will not change the node's position at all, whereas 1 will fully blend to the target position.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The target position.  Can also be provided as 3 numbers."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target position to blend in.  A value of 0 will not change the node's position at all, whereas 1 will fully blend to the target position.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setNodeScale",
              summary = "Set or blend the scale of a node.",
              description = "Sets or blends the scale of a node to a new scale.",
              key = "Model:setNodeScale",
              module = "lovr.graphics",
              notes = "For best results when animating, it's recommended to keep the 3 components of the scale the same.",
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "scale",
                      type = "Vec3",
                      description = "The target scale.  Can also be provided as 3 numbers."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target scale to blend in.  A value of 0 will not change the node's scale at all, whereas 1 will fully blend to the target scale.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "scale",
                      type = "Vec3",
                      description = "The target scale.  Can also be provided as 3 numbers."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target scale to blend in.  A value of 0 will not change the node's scale at all, whereas 1 will fully blend to the target scale.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodePose",
                "Model:setNodePose",
                "Model:getNodeTransform",
                "Model:setNodeTransform",
                "Model:animate"
              }
            },
            {
              name = "setNodeTransform",
              summary = "Set or blend the transform of a node.",
              description = "Sets or blends the transform of a node to a new transform.",
              key = "Model:setNodeTransform",
              module = "lovr.graphics",
              notes = "For best results when animating, it's recommended to keep the 3 components of the scale the same.",
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the node."
                    },
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The target transform.  Can also be provided as position, scale, and rotation using a mix of `Vectors` or numbers, with 3 scale components."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target transform to blend in.  A value of 0 will not change the node's transform at all, whereas 1 will fully blend to the target transform.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the node."
                    },
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The target transform.  Can also be provided as position, scale, and rotation using a mix of `Vectors` or numbers, with 3 scale components."
                    },
                    {
                      name = "blend",
                      type = "number",
                      description = "A number from 0 to 1 indicating how much of the target transform to blend in.  A value of 0 will not change the node's transform at all, whereas 1 will fully blend to the target transform.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "Model:getNodePosition",
                "Model:setNodePosition",
                "Model:getNodeOrientation",
                "Model:setNodeOrientation",
                "Model:getNodeScale",
                "Model:setNodeScale",
                "Model:getNodePose",
                "Model:setNodePose",
                "Model:animate"
              }
            }
          }
        },
        {
          name = "Pass",
          summary = "A stream of graphics commands.",
          description = "Pass objects are used to record commands for the GPU.  Commands can be recorded by calling functions on the Pass.  After recording a set of passes, they can be submitted for the GPU to process using `lovr.graphics.submit`.\n\nPass objects are **temporary** and only exist for a single frame.  Once `lovr.graphics.submit` is called to end the frame, any passes that were created during that frame become **invalid**. Each frame, a new set of passes must be created and recorded.  LÖVR tries to detect if you use a pass after it's invalid, but this error checking is not 100% accurate at the moment.\n\nThere are 3 types of passes.  Each type can record a specific type of command:\n\n- `render` passes render graphics to textures.  The `lovr.draw` callback receives a render pass\n  as an argument.\n- `compute` passes run compute shaders.\n- `transfer` passes can transfer data to/from GPU objects, like `Buffer` and `Texture`.",
          key = "Pass",
          module = "lovr.graphics",
          sections = {
            {
              name = "Drawing",
              tag = "drawing",
              description = "TODO"
            },
            {
              name = "Coordinate System",
              tag = "transform",
              description = "TODO"
            },
            {
              name = "Render States",
              tag = "pipeline",
              description = "TODO"
            },
            {
              name = "Shader Inputs",
              tag = "shader-inputs",
              description = "TODO"
            },
            {
              name = "Camera",
              tag = "camera",
              description = "TODO"
            },
            {
              name = "Compute",
              tag = "compute",
              description = "TODO"
            }
          },
          methods = {
            {
              name = "blit",
              tag = "transfer",
              summary = "Copy data between textures with scaling.",
              description = "Copies data between textures.  Similar to `Pass:copy`, except the source and destination sizes can be different.  The pixels from the source texture will be scaled to the destination size.",
              key = "Pass:blit",
              module = "lovr.graphics",
              notes = "When blitting between 3D textures, the layer counts do not need to match, and the layers will be treated as a continuous axis (i.e. pixels will be smoothed between layers).\n\nWhen blitting between array textures, the layer counts must match, and the blit occurs as a sequence of distinct 2D blits layer-by-layer.",
              variants = {
                {
                  arguments = {
                    {
                      name = "src",
                      type = "Texture",
                      description = "The texture to copy from."
                    },
                    {
                      name = "dst",
                      type = "Texture",
                      description = "The texture to copy to."
                    },
                    {
                      name = "srcx",
                      type = "number",
                      description = "The x offset from the left of the source texture to blit from, in pixels.",
                      default = "0"
                    },
                    {
                      name = "srcy",
                      type = "number",
                      description = "The y offset from the top of the source texture to blit from, in pixels.",
                      default = "0"
                    },
                    {
                      name = "srcz",
                      type = "number",
                      description = "The index of the first layer in the source texture to blit from.",
                      default = "1"
                    },
                    {
                      name = "dstx",
                      type = "number",
                      description = "The x offset from the left of the destination texture to blit to, in pixels.",
                      default = "0"
                    },
                    {
                      name = "dsty",
                      type = "number",
                      description = "The y offset from the top of the destination texture to blit to, in pixels.",
                      default = "0"
                    },
                    {
                      name = "dstz",
                      type = "number",
                      description = "The index of the first layer in the destination texture to blit to.",
                      default = "1"
                    },
                    {
                      name = "srcw",
                      type = "number",
                      description = "The width of the region in the source texture to blit.  If nil, the region will extend to the right side of the texture.",
                      default = "nil"
                    },
                    {
                      name = "srch",
                      type = "number",
                      description = "The height of the region in the source texture to blit.  If nil, the region will extend to the bottom of the texture.",
                      default = "nil"
                    },
                    {
                      name = "srcd",
                      type = "number",
                      description = "The number of layers in the source texture to blit.",
                      default = "nil"
                    },
                    {
                      name = "dstw",
                      type = "number",
                      description = "The width of the region in the destination texture to blit to.  If nil, the region will extend to the right side of the texture.",
                      default = "nil"
                    },
                    {
                      name = "dsth",
                      type = "number",
                      description = "The height of the region in the destination texture to blit to.  If nil, the region will extend to the bottom of the texture.",
                      default = "nil"
                    },
                    {
                      name = "dstd",
                      type = "number",
                      description = "The number of the layers in the destination texture to blit to.",
                      default = "nil"
                    },
                    {
                      name = "srclevel",
                      type = "number",
                      description = "The index of the mipmap level in the source texture to blit from.",
                      default = "1"
                    },
                    {
                      name = "dstlevel",
                      type = "number",
                      description = "The index of the mipmap level in the destination texture to blit to.",
                      default = "1"
                    },
                    {
                      name = "filter",
                      type = "FilterMode",
                      description = "The filtering algorithm used when rescaling.",
                      default = "linear"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "box",
              tag = "drawing",
              summary = "Draw a box.",
              description = "Draw a box.  This is like `Pass:cube`, except it takes 3 separate values for the scale.",
              key = "Pass:box",
              module = "lovr.graphics",
              related = {
                "Pass:cube"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the box.  Can also be provided as position, scale, and rotation using a mix of `Vectors` or numbers.  When using numbers for the scale, 3 numbers are used."
                    },
                    {
                      name = "style",
                      type = "DrawStyle",
                      description = "Whether the box should be drawn filled or outlined.",
                      default = "'fill'"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "capsule",
              tag = "drawing",
              summary = "Draw a capsule.",
              description = "Draws a capsule.  A capsule is shaped like a cylinder with a hemisphere on each end.",
              key = "Pass:capsule",
              module = "lovr.graphics",
              notes = "The length of the capsule does not include the end caps.  The local origin of the capsule is in the center, and the local z axis points towards the end caps.",
              variants = {
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the capsule.  Can also be provided as position, scale, and rotation using a mix of `Vectors` or numbers.  When using numbers for the scale, 2 should be provided: one for the radius and one for the length.  When using a matrix or a vector for the scale, the X and Y components are the radius and the Z component is the length."
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of circular segments to render.",
                      default = "32"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Draws a capsule between two points.",
                  arguments = {
                    {
                      name = "p1",
                      type = "Vec3",
                      description = "The starting point of the capsule."
                    },
                    {
                      name = "p2",
                      type = "Vec3",
                      description = "The ending point of the capsule."
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the capsule.",
                      default = "1.0"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of circular segments to render.",
                      default = "32"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "circle",
              tag = "drawing",
              summary = "Draw a circle.",
              description = "Draws a circle.",
              key = "Pass:circle",
              module = "lovr.graphics",
              notes = "The local origin of the circle is in its center, and the local z axis goes through the center.",
              variants = {
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the circle.  Can also be provided as position, radius, and rotation, using a mix of `Vectors` or numbers."
                    },
                    {
                      name = "style",
                      type = "DrawStyle",
                      description = "Whether the circle should be filled or outlined.",
                      default = "'fill'"
                    },
                    {
                      name = "angle1",
                      type = "number",
                      description = "The angle of the beginning of the arc.",
                      default = "0"
                    },
                    {
                      name = "angle2",
                      type = "number",
                      description = "angle of the end of the arc.",
                      default = "2 * math.pi"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of segments to render.",
                      default = "64"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "clear",
              tag = "transfer",
              summary = "Clear a Buffer or Texture.",
              description = "Clears a Buffer or Texture.",
              key = "Pass:clear",
              module = "lovr.graphics",
              related = {
                "Buffer:clear",
                "Pass:copy"
              },
              variants = {
                {
                  description = "Clears a range of a Buffer, setting the values to zero.",
                  arguments = {
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "The Buffer to clear."
                    },
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the first item to clear.",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of items to clear.  If `nil`, clears to the end of the Buffer.",
                      default = "nil"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Clears layers and mipmap levels in a Texture to a color.",
                  arguments = {
                    {
                      name = "texture",
                      type = "Texture",
                      description = "The Texture to clear."
                    },
                    {
                      name = "color",
                      type = "Color",
                      description = "The color to clear to."
                    },
                    {
                      name = "layer",
                      type = "number",
                      description = "The index of the first layer to clear.",
                      default = "1"
                    },
                    {
                      name = "layers",
                      type = "number",
                      description = "The number of layers to clear.  If `nil`, clears the remaining layers.",
                      default = "nil"
                    },
                    {
                      name = "level",
                      type = "number",
                      description = "The index of the first mipmap level to clear.",
                      default = "1"
                    },
                    {
                      name = "levels",
                      type = "number",
                      description = "The number of mipmap level to clear.  If `nil`, clears the remaining mipmaps.",
                      default = "nil"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "compute",
              tag = "compute",
              summary = "Dispatch a compute shader.",
              description = "TODO",
              key = "Pass:compute",
              module = "lovr.graphics",
              notes = "TODO",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "How many workgroups to dispatch in the x dimension.",
                      default = "1"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "How many workgroups to dispatch in the y dimension.",
                      default = "1"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "How many workgroups to dispatch in the z dimension.",
                      default = "1"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Perform an \"indirect\" dispatch, sourcing workgroup counts from a Buffer.",
                  arguments = {
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "A Buffer object containing the x, y, and z workgroup counts, stored as 4 byte unsigned integers."
                    },
                    {
                      name = "offset",
                      type = "number",
                      description = "The byte offset to read the workgroup counts from in the Buffer.",
                      default = "0"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "cone",
              tag = "drawing",
              summary = "Draw a cone.",
              description = "Draws a cone.",
              key = "Pass:cone",
              module = "lovr.graphics",
              notes = "The local origin is at the center of the base of the cone, and the negative z axis points towards the tip.",
              variants = {
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the cone.  Can also be provided as position, scale, and rotation using a mix of `Vectors` or numbers.  When using numbers for the scale, 2 should be provided: one for the radius and one for the length.  When using a matrix or a vector for the scale, the X and Y components are the radius and the Z component is the length."
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of segments in the cone.",
                      default = "64"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "copy",
              tag = "transfer",
              summary = "Copy data to or between GPU resources.",
              description = "TODO",
              key = "Pass:copy",
              module = "lovr.graphics",
              variants = {
                {
                  description = "Copy table data to a Buffer.",
                  arguments = {
                    {
                      name = "table",
                      type = "table",
                      description = "TODO"
                    },
                    {
                      name = "bufferdst",
                      type = "Buffer",
                      description = "TODO"
                    },
                    {
                      name = "srcindex",
                      type = "number",
                      description = "TODO",
                      default = "1"
                    },
                    {
                      name = "dstindex",
                      type = "number",
                      description = "TODO",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "TODO",
                      default = "nil"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Copy binary data to a Buffer.",
                  arguments = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "TODO"
                    },
                    {
                      name = "bufferdst",
                      type = "Buffer",
                      description = "TODO"
                    },
                    {
                      name = "srcoffset",
                      type = "number",
                      description = "TODO",
                      default = "0"
                    },
                    {
                      name = "dstoffset",
                      type = "number",
                      description = "TODO",
                      default = "0"
                    },
                    {
                      name = "size",
                      type = "number",
                      description = "TODO",
                      default = "nil"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Copy data between Buffers.",
                  arguments = {
                    {
                      name = "buffersrc",
                      type = "Buffer",
                      description = "TODO"
                    },
                    {
                      name = "bufferdst",
                      type = "Buffer",
                      description = "TODO"
                    },
                    {
                      name = "srcoffset",
                      type = "number",
                      description = "TODO",
                      default = "0"
                    },
                    {
                      name = "dstoffset",
                      type = "number",
                      description = "TODO",
                      default = "0"
                    },
                    {
                      name = "size",
                      type = "number",
                      description = "TODO",
                      default = "nil"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Copy Image data to a Texture.",
                  arguments = {
                    {
                      name = "image",
                      type = "Image",
                      description = "TODO"
                    },
                    {
                      name = "texturedst",
                      type = "Texture",
                      description = "TODO"
                    },
                    {
                      name = "srcx",
                      type = "number",
                      description = "TODO",
                      default = "0"
                    },
                    {
                      name = "srcy",
                      type = "number",
                      description = "TODO",
                      default = "0"
                    },
                    {
                      name = "dstx",
                      type = "number",
                      description = "TODO",
                      default = "0"
                    },
                    {
                      name = "dsty",
                      type = "number",
                      description = "TODO",
                      default = "0"
                    },
                    {
                      name = "width",
                      type = "number",
                      description = "TODO",
                      default = "nil"
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "TODO",
                      default = "nil"
                    },
                    {
                      name = "srclayer",
                      type = "number",
                      description = "TODO",
                      default = "1"
                    },
                    {
                      name = "dstlayer",
                      type = "number",
                      description = "TODO",
                      default = "1"
                    },
                    {
                      name = "layers",
                      type = "number",
                      description = "TODO",
                      default = "nil"
                    },
                    {
                      name = "srclevel",
                      type = "number",
                      description = "TODO",
                      default = "1"
                    },
                    {
                      name = "dstlevel",
                      type = "number",
                      description = "TODO",
                      default = "1"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Copy data between Textures.",
                  arguments = {
                    {
                      name = "texturesrc",
                      type = "Texture",
                      description = "TODO"
                    },
                    {
                      name = "texturedst",
                      type = "Texture",
                      description = "TODO"
                    },
                    {
                      name = "srcx",
                      type = "number",
                      description = "TODO",
                      default = "0"
                    },
                    {
                      name = "srcy",
                      type = "number",
                      description = "TODO",
                      default = "0"
                    },
                    {
                      name = "dstx",
                      type = "number",
                      description = "TODO",
                      default = "0"
                    },
                    {
                      name = "dsty",
                      type = "number",
                      description = "TODO",
                      default = "0"
                    },
                    {
                      name = "width",
                      type = "number",
                      description = "TODO",
                      default = "nil"
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "TODO",
                      default = "nil"
                    },
                    {
                      name = "srclayer",
                      type = "number",
                      description = "TODO",
                      default = "1"
                    },
                    {
                      name = "dstlayer",
                      type = "number",
                      description = "TODO",
                      default = "1"
                    },
                    {
                      name = "layers",
                      type = "number",
                      description = "TODO",
                      default = "nil"
                    },
                    {
                      name = "srclevel",
                      type = "number",
                      description = "TODO",
                      default = "1"
                    },
                    {
                      name = "dstlevel",
                      type = "number",
                      description = "TODO",
                      default = "1"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Copy tally data to a Buffer.",
                  arguments = {
                    {
                      name = "tally",
                      type = "Tally",
                      description = "TODO"
                    },
                    {
                      name = "srcindex",
                      type = "number",
                      description = "TODO",
                      default = "1"
                    },
                    {
                      name = "dstoffset",
                      type = "number",
                      description = "TODO",
                      default = "0"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "TODO",
                      default = "nil"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "cube",
              tag = "drawing",
              summary = "Draw a cube.",
              description = "Draws a cube.",
              key = "Pass:cube",
              module = "lovr.graphics",
              notes = "The local origin is in the center of the cube.",
              variants = {
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the cube.  Can also be provided as position, scale, and rotation using a mix of `Vectors` or numbers.  When using numbers for the scale, 1 number is used."
                    },
                    {
                      name = "style",
                      type = "DrawStyle",
                      description = "Whether the cube should be drawn filled or outlined.",
                      default = "'fill'"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "cylinder",
              tag = "drawing",
              summary = "Draw a cylinder.",
              description = "Draws a cylinder.",
              key = "Pass:cylinder",
              module = "lovr.graphics",
              notes = "The local origin is in the center of the cylinder, and the length of the cylinder is along the z axis.",
              variants = {
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The transform of the cylinder.  Can also be provided as position, scale, and rotation using a mix of `Vectors` or numbers.  When using numbers for the scale, 2 should be provided: one for the radius and one for the length.  When using a matrix or a vector for the scale, the X and Y components are the radius and the Z component is the length."
                    },
                    {
                      name = "capped",
                      type = "boolean",
                      description = "Whether the tops and bottoms of the cylinder should be rendered.",
                      default = "true"
                    },
                    {
                      name = "angle1",
                      type = "number",
                      description = "The angle of the beginning of the arc.",
                      default = "0"
                    },
                    {
                      name = "angle2",
                      type = "number",
                      description = "angle of the end of the arc.",
                      default = "2 * math.pi"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of circular segments to render.",
                      default = "64"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "p1",
                      type = "Vec3",
                      description = "The starting point of the cylinder."
                    },
                    {
                      name = "p2",
                      type = "Vec3",
                      description = "The ending point of the cylinder."
                    },
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the cylinder."
                    },
                    {
                      name = "capped",
                      type = "boolean",
                      description = "Whether the tops and bottoms of the cylinder should be rendered.",
                      default = "true"
                    },
                    {
                      name = "angle1",
                      type = "number",
                      description = "The angle of the beginning of the arc.",
                      default = "0"
                    },
                    {
                      name = "angle2",
                      type = "number",
                      description = "angle of the end of the arc.",
                      default = "2 * math.pi"
                    },
                    {
                      name = "segments",
                      type = "number",
                      description = "The number of circular segments to render.",
                      default = "64"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "draw",
              tag = "drawing",
              summary = "Draw a drawable object.",
              description = "TODO",
              key = "Pass:draw",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "model",
                      type = "Model",
                      description = "The model to draw."
                    },
                    {
                      name = "transform",
                      type = "Transform",
                      description = "The transform of the object."
                    },
                    {
                      name = "nodeindex",
                      type = "number",
                      description = "TODO"
                    },
                    {
                      name = "children",
                      type = "boolean",
                      description = "TODO"
                    },
                    {
                      name = "instances",
                      type = "number",
                      description = "TODO"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "model",
                      type = "Model",
                      description = "The model to draw."
                    },
                    {
                      name = "transform",
                      type = "Transform",
                      description = "The transform of the object."
                    },
                    {
                      name = "nodename",
                      type = "string",
                      description = "TODO"
                    },
                    {
                      name = "children",
                      type = "boolean",
                      description = "TODO"
                    },
                    {
                      name = "instances",
                      type = "number",
                      description = "TODO"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "fill",
              tag = "drawing",
              summary = "Draw a fullscreen triangle.",
              description = "Draws a fullscreen triangle.  The `fill` shader is used, which stretches the triangle across the screen.",
              key = "Pass:fill",
              module = "lovr.graphics",
              notes = "This function has some special behavior for array textures:\n\n- Filling a single-layer texture to a multi-layer canvas will mirror the texture to all layers,\n  just like regular drawing.\n- Filling a 2-layer texture to a mono canvas will render the 2 layers side-by-side.\n- Filling a multi-layer texture to a multi-layer canvas will do a layer-by-layer fill (the layer\n  counts must match).",
              variants = {
                {
                  arguments = {
                    {
                      name = "texture",
                      type = "Texture",
                      description = "The texture to fill.  If nil, the texture from the active material is used."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "getClear",
              summary = "TODO",
              description = "TODO",
              key = "Pass:getClear",
              module = "lovr.graphics",
              notes = "TODO",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "clears",
                      type = "table",
                      description = "TODO"
                    }
                  }
                }
              }
            },
            {
              name = "getDimensions",
              summary = "Get the texture dimensions of a render pass.",
              description = "Returns the dimensions of the textures attached to the render pass.",
              key = "Pass:getDimensions",
              module = "lovr.graphics",
              notes = "If the pass is not a render pass, this function returns zeros.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The texture width."
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The texture height."
                    }
                  }
                }
              },
              related = {
                "Pass:getWidth",
                "Pass:getHeight",
                "Pass:getViewCount",
                "lovr.graphics.getPass",
                "lovr.system.getWindowDimensions",
                "lovr.headset.getDisplayDimensions"
              }
            },
            {
              name = "getHeight",
              summary = "Get the texture height of a render pass.",
              description = "Returns the height of the textures attached to the render pass.",
              key = "Pass:getHeight",
              module = "lovr.graphics",
              notes = "If the pass is not a render pass, this function returns zero.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The texture height."
                    }
                  }
                }
              },
              related = {
                "Pass:getWidth",
                "Pass:getDimensions",
                "Pass:getViewCount",
                "lovr.graphics.getPass",
                "lovr.system.getWindowHeight",
                "lovr.headset.getDisplayHeight"
              }
            },
            {
              name = "getProjection",
              tag = "camera",
              summary = "Get the field of view.",
              description = "Returns the projection for a single view.",
              key = "Pass:getProjection",
              module = "lovr.graphics",
              related = {
                "lovr.headset.getViewAngles",
                "lovr.headset.getViewCount",
                "lovr.graphics.getViewPose",
                "lovr.graphics.setViewPose"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The view index."
                    }
                  },
                  returns = {
                    {
                      name = "left",
                      type = "number",
                      description = "The left field of view angle, in radians."
                    },
                    {
                      name = "right",
                      type = "number",
                      description = "The right field of view angle, in radians."
                    },
                    {
                      name = "up",
                      type = "number",
                      description = "The top field of view angle, in radians."
                    },
                    {
                      name = "down",
                      type = "number",
                      description = "The bottom field of view angle, in radians."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The view index."
                    },
                    {
                      name = "matrix",
                      type = "Mat4",
                      description = "The matrix to fill with the projection."
                    }
                  },
                  returns = {
                    {
                      name = "matrix",
                      type = "Mat4",
                      description = "The matrix containing the projection."
                    }
                  }
                }
              }
            },
            {
              name = "getSampleCount",
              summary = "Get the antialiasing setting of a render pass.",
              description = "Returns the antialiasing setting of a render pass.",
              key = "Pass:getSampleCount",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of samples used for rendering.  Currently, will be 1 or 4."
                    }
                  }
                }
              }
            },
            {
              name = "getTarget",
              summary = "TODO",
              description = "TODO",
              key = "Pass:getTarget",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "target",
                      type = "table",
                      description = "TODO"
                    }
                  }
                }
              }
            },
            {
              name = "getType",
              summary = "Get the type of the Pass.",
              description = "Returns the type of the pass (render, compute, or transfer).  The type restricts what kinds of functions can be called on the pass.",
              key = "Pass:getType",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = " type",
                      type = "PassType",
                      description = "The type of the Pass."
                    }
                  }
                }
              }
            },
            {
              name = "getViewCount",
              summary = "Returns the view count of a render pass.",
              description = "Returns the view count of a render pass.  This is the layer count of the textures it is rendering to.",
              key = "Pass:getViewCount",
              module = "lovr.graphics",
              notes = "A render pass has one \"camera\" for each view.  Whenever something is drawn, it is broadcast to each view (layer) of each texture, using the corresponding camera.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "views",
                      type = "number",
                      description = "The view count."
                    }
                  }
                }
              },
              related = {
                "Pass:getViewPose",
                "Pass:setViewPose",
                "Pass:getProjection",
                "Pass:setProjection",
                "lovr.headset.getViewCount"
              }
            },
            {
              name = "getViewPose",
              tag = "camera",
              summary = "Get the camera pose.",
              description = "Get the pose of a single view.",
              key = "Pass:getViewPose",
              module = "lovr.graphics",
              related = {
                "lovr.headset.getViewPose",
                "lovr.headset.getViewCount",
                "lovr.graphics.getProjection",
                "lovr.graphics.setProjection"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The view index."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the viewer, in meters."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the viewer, in meters."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the viewer, in meters."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the viewer is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The view index."
                    },
                    {
                      name = "matrix",
                      type = "Mat4",
                      description = "The matrix to fill with the view pose."
                    },
                    {
                      name = "invert",
                      type = "boolean",
                      description = "Whether the matrix should be inverted."
                    }
                  },
                  returns = {
                    {
                      name = "matrix",
                      type = "Mat4",
                      description = "The matrix containing the view pose."
                    }
                  }
                }
              }
            },
            {
              name = "getWidth",
              summary = "Get the texture width of a render pass.",
              description = "Returns the width of the textures attached to the render pass.",
              key = "Pass:getWidth",
              module = "lovr.graphics",
              notes = "If the pass is not a render pass, this function returns zero.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The texture width."
                    }
                  }
                }
              },
              related = {
                "Pass:getHeight",
                "Pass:getDimensions",
                "Pass:getViewCount",
                "lovr.graphics.getPass",
                "lovr.system.getWindowWidth",
                "lovr.headset.getDisplayWidth"
              }
            },
            {
              name = "line",
              tag = "drawing",
              summary = "Draw a line.",
              description = "TODO",
              key = "Pass:line",
              module = "lovr.graphics",
              notes = "TODO",
              variants = {
                {
                  arguments = {
                    {
                      name = "x1",
                      type = "number",
                      description = "The x coordinate of the first point."
                    },
                    {
                      name = "y1",
                      type = "number",
                      description = "The y coordinate of the first point."
                    },
                    {
                      name = "z1",
                      type = "number",
                      description = "The z coordinate of the first point."
                    },
                    {
                      name = "x2",
                      type = "number",
                      description = "The x coordinate of the next point."
                    },
                    {
                      name = "y2",
                      type = "number",
                      description = "The y coordinate of the next point."
                    },
                    {
                      name = "z2",
                      type = "number",
                      description = "The z coordinate of the next point."
                    },
                    {
                      name = "...",
                      type = "*",
                      description = "More points to add to the line."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "t",
                      type = "table",
                      description = "A table of numbers or Vec3 objects (not both) representing points of the line."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "v1",
                      type = "Vec3",
                      description = "A vector containing the position of the first point of the line."
                    },
                    {
                      name = "v2",
                      type = "Vec3",
                      description = "A vector containing the position of the next point on the line."
                    },
                    {
                      name = "...",
                      type = "*",
                      description = "More points to add to the line."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "mesh",
              tag = "drawing",
              summary = "Draw a mesh.",
              description = "TODO",
              key = "Pass:mesh",
              module = "lovr.graphics",
              notes = "TODO",
              variants = {
                {
                  arguments = {
                    {
                      name = "vertices",
                      type = "Buffer",
                      description = "TODO",
                      default = "nil"
                    },
                    {
                      name = "transform",
                      type = "transform",
                      description = "The transform to apply to the mesh."
                    },
                    {
                      name = "start",
                      type = "number",
                      description = "The 1-based index of the first vertex to render from the vertex buffer (or the first index, when using an index buffer).",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices to render (or the number of indices, when using an index buffer). When `nil`, as many vertices or indices as possible will be drawn (based on the length of the Buffers and `start`).",
                      default = "nil"
                    },
                    {
                      name = "instances",
                      type = "number",
                      description = "The number of copies of the mesh to render.",
                      default = "1"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "vertices",
                      type = "Buffer",
                      description = "TODO",
                      default = "nil"
                    },
                    {
                      name = "indices",
                      type = "Buffer",
                      description = "TODO"
                    },
                    {
                      name = "transform",
                      type = "transform",
                      description = "The transform to apply to the mesh."
                    },
                    {
                      name = "start",
                      type = "number",
                      description = "The 1-based index of the first vertex to render from the vertex buffer (or the first index, when using an index buffer).",
                      default = "1"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "The number of vertices to render (or the number of indices, when using an index buffer). When `nil`, as many vertices or indices as possible will be drawn (based on the length of the Buffers and `start`).",
                      default = "nil"
                    },
                    {
                      name = "instances",
                      type = "number",
                      description = "The number of copies of the mesh to render.",
                      default = "1"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "vertices",
                      type = "Buffer",
                      description = "TODO",
                      default = "nil"
                    },
                    {
                      name = "indices",
                      type = "Buffer",
                      description = "TODO"
                    },
                    {
                      name = "draws",
                      type = "Buffer",
                      description = "TODO"
                    },
                    {
                      name = "drawcount",
                      type = "number",
                      description = "TODO"
                    },
                    {
                      name = "offset",
                      type = "number",
                      description = "TODO"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "TODO"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "mipmap",
              tag = "transfer",
              summary = "Generate mipmaps for a texture.",
              description = "TODO",
              key = "Pass:mipmap",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "texture",
                      type = "Texture",
                      description = "TODO"
                    },
                    {
                      name = "base",
                      type = "number",
                      description = "TODO",
                      default = "0"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "TODO",
                      default = "nil"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "origin",
              tag = "transform",
              summary = "Reset the transform to the origin.",
              description = "TODO",
              key = "Pass:origin",
              module = "lovr.graphics",
              related = {
                "lovr.graphics.translate",
                "lovr.graphics.rotate",
                "lovr.graphics.scale",
                "lovr.graphics.transform"
              },
              variants = {
                {
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "plane",
              tag = "drawing",
              summary = "Draw a flat plane.",
              description = "TODO",
              key = "Pass:plane",
              module = "lovr.graphics",
              notes = "TODO",
              variants = {
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Transform2",
                      description = "The transform to apply to the plane."
                    },
                    {
                      name = "style",
                      type = "DrawStyle",
                      description = "Whether the plane should be drawn filled or outlined.",
                      default = "'fill'"
                    },
                    {
                      name = "columns",
                      type = "number",
                      description = "The number of horizontal segments in the plane.",
                      default = "1"
                    },
                    {
                      name = "rows",
                      type = "number",
                      description = "The number of vertical segments in the plane.",
                      default = "cols"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "points",
              tag = "drawing",
              summary = "Draw points.",
              description = "TODO",
              key = "Pass:points",
              module = "lovr.graphics",
              notes = "TODO",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the first point."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the first point."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the first point."
                    },
                    {
                      name = "...",
                      type = "*",
                      description = "More points."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "t",
                      type = "table",
                      description = "A table of numbers or Vec3 objects (not both) representing point positions."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "A vector containing the position of the first point to draw."
                    },
                    {
                      name = "...",
                      type = "*",
                      description = "More points."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "pop",
              tag = "transform",
              summary = "Restore original state from a stack.",
              description = "TODO",
              key = "Pass:pop",
              module = "lovr.graphics",
              notes = "TODO stack balancing/error",
              related = {
                "lovr.graphics.push",
                "StackType"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "stack",
                      type = "StackType",
                      description = "The type of stack to pop.",
                      default = "'transform'"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "push",
              tag = "transform",
              summary = "Push state onto a stack.",
              description = "TODO",
              key = "Pass:push",
              module = "lovr.graphics",
              notes = "TODO stack balancing/error",
              related = {
                "lovr.graphics.pop",
                "StackType"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "stack",
                      type = "StackType",
                      description = "The type of stack to push.",
                      default = "'transform'"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "read",
              tag = "transfer",
              summary = "Download data from a GPU resource.",
              description = "TODO",
              key = "Pass:read",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "TODO"
                    },
                    {
                      name = "index",
                      type = "number",
                      description = "TODO"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "TODO"
                    }
                  },
                  returns = {
                    {
                      name = "readback",
                      type = "Readback",
                      description = "TODO"
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "texture",
                      type = "Texture",
                      description = "TODO"
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "TODO",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "TODO",
                      default = "0"
                    },
                    {
                      name = "layer",
                      type = "number",
                      description = "TODO",
                      default = "1"
                    },
                    {
                      name = "level",
                      type = "number",
                      description = "TODO",
                      default = "1"
                    },
                    {
                      name = "width",
                      type = "number",
                      description = "TODO",
                      default = "nil"
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "TODO",
                      default = "nil"
                    }
                  },
                  returns = {
                    {
                      name = "readback",
                      type = "Readback",
                      description = "TODO"
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "tally",
                      type = "Tally",
                      description = "TODO"
                    },
                    {
                      name = "index",
                      type = "number",
                      description = "TODO"
                    },
                    {
                      name = "count",
                      type = "number",
                      description = "TODO"
                    }
                  },
                  returns = {
                    {
                      name = "readback",
                      type = "Readback",
                      description = "TODO"
                    }
                  }
                }
              }
            },
            {
              name = "rotate",
              tag = "transform",
              summary = "Rotate the coordinate system.",
              description = "TODO",
              key = "Pass:rotate",
              module = "lovr.graphics",
              notes = "TODO axis does not need to be normalized TODO order matters",
              related = {
                "lovr.graphics.translate",
                "lovr.graphics.scale",
                "lovr.graphics.transform"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians to rotate around the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "q",
                      type = "Quat",
                      description = "A quaternion containing the rotation to apply."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "scale",
              tag = "transform",
              summary = "Scale the coordinate system.",
              description = "TODO",
              key = "Pass:scale",
              module = "lovr.graphics",
              related = {
                "lovr.graphics.translate",
                "lovr.graphics.rotate",
                "lovr.graphics.transform"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The amount to scale the x axis.",
                      default = "1"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The amount to scale the y axis.",
                      default = "1"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The amount to scale the z axis.",
                      default = "1"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "A vector to translate by."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "send",
              summary = "Set the value of a shader variable.",
              description = "TODO",
              key = "Pass:send",
              module = "lovr.graphics",
              notes = "TODO",
              variants = {
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the Shader variable."
                    },
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "The Buffer to assign."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the Shader variable."
                    },
                    {
                      name = "texture",
                      type = "Texture",
                      description = "The Texture to assign."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the Shader variable."
                    },
                    {
                      name = "sampler",
                      type = "Sampler",
                      description = "The Sampler to assign."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the Shader variable."
                    },
                    {
                      name = "constant",
                      type = "*",
                      description = "Numbers or vectors to assign to the push constant."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "binding",
                      type = "number",
                      description = "The binding number of the Shader variable."
                    },
                    {
                      name = "buffer",
                      type = "Buffer",
                      description = "The Buffer to assign."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "binding",
                      type = "number",
                      description = "The binding number of the Shader variable."
                    },
                    {
                      name = "texture",
                      type = "Texture",
                      description = "The Texture to assign."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "binding",
                      type = "number",
                      description = "The binding number of the Shader variable."
                    },
                    {
                      name = "sampler",
                      type = "Sampler",
                      description = "The Sampler to assign."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setAlphaToCoverage",
              tag = "pipeline",
              summary = "Enable or disable alpha to coverage.",
              description = "TODO",
              key = "Pass:setAlphaToCoverage",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "enable",
                      type = "boolean",
                      description = "Whether alpha to coverage should be enabled."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setBlendMode",
              tag = "pipeline",
              summary = "Set the blend mode.",
              description = "TODO",
              key = "Pass:setBlendMode",
              module = "lovr.graphics",
              notes = "TODO",
              variants = {
                {
                  arguments = {
                    {
                      name = "blend",
                      type = "BlendMode",
                      description = "The blend mode."
                    },
                    {
                      name = "alphaBlend",
                      type = "BlendAlphaMode",
                      description = "The alpha blend mode."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setColor",
              tag = "pipeline",
              summary = "Set the color.",
              description = "TODO",
              key = "Pass:setColor",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "color",
                      type = "Color",
                      description = "The new color."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setColorWrite",
              tag = "pipeline",
              summary = "Change the color components affected by drawing.",
              description = "TODO",
              key = "Pass:setColorWrite",
              module = "lovr.graphics",
              notes = "TODO",
              variants = {
                {
                  arguments = {
                    {
                      name = "r",
                      type = "boolean",
                      description = "Whether the red component should be affected by drawing."
                    },
                    {
                      name = "g",
                      type = "boolean",
                      description = "Whether the green component should be affected by drawing."
                    },
                    {
                      name = "b",
                      type = "boolean",
                      description = "Whether the blue component should be affected by drawing."
                    },
                    {
                      name = "a",
                      type = "boolean",
                      description = "Whether the alpha component should be affected by drawing."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setCullMode",
              tag = "pipeline",
              summary = "Control triangle face culling.",
              description = "TODO",
              key = "Pass:setCullMode",
              module = "lovr.graphics",
              notes = "TODO",
              related = {
                "lovr.graphics.setWinding"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "mode",
                      type = "CullMode",
                      description = "Whether `front` faces, `back` faces, or `none` of the faces should be culled.",
                      default = "'none'"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setDepthClamp",
              tag = "pipeline",
              summary = "Enable or disable depth clamp.",
              description = "TODO",
              key = "Pass:setDepthClamp",
              module = "lovr.graphics",
              notes = "TODO depthClamp feature!",
              related = {
                "lovr.graphics.setDepthTest",
                "lovr.graphics.setDepthWrite",
                "lovr.graphics.setDepthOffset"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "enable",
                      type = "boolean",
                      description = "Whether depth clamp should be enabled."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setDepthOffset",
              tag = "pipeline",
              summary = "Configure the depth offset.",
              description = "TODO",
              key = "Pass:setDepthOffset",
              module = "lovr.graphics",
              notes = "TODO",
              related = {
                "lovr.graphics.setDepthTest",
                "lovr.graphics.setDepthWrite"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "offset",
                      type = "number",
                      description = "The depth offset.",
                      default = "0.0"
                    },
                    {
                      name = "sloped",
                      type = "number",
                      description = "The sloped depth offset.",
                      default = "0.0"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setDepthTest",
              tag = "pipeline",
              summary = "Configure the depth test.",
              description = "TODO",
              key = "Pass:setDepthTest",
              module = "lovr.graphics",
              notes = "TODO",
              related = {
                "lovr.graphics.setDepthWrite",
                "lovr.graphics.setDepthOffset",
                "lovr.graphics.setDepthClamp",
                "lovr.graphics.setStencilTest"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "test",
                      type = "CompareMode",
                      description = "The new depth test to use."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Disable the depth test.",
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "setDepthWrite",
              tag = "pipeline",
              summary = "Set whether draws write to the depth buffer.",
              description = "TODO",
              key = "Pass:setDepthWrite",
              module = "lovr.graphics",
              notes = "TODO",
              related = {
                "lovr.graphics.setDepthTest",
                "lovr.graphics.setStencilWrite",
                "lovr.graphics.setColorWrite"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "write",
                      type = "boolean",
                      description = "The new depth write setting."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setFont",
              tag = "pipeline",
              summary = "Enable or disable depth clamp.",
              description = "TODO",
              key = "Pass:setFont",
              module = "lovr.graphics",
              related = {
                "Pass:text"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "font",
                      type = "Font",
                      description = "The Font to use when rendering text."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setMaterial",
              tag = "pipeline",
              summary = "Set the material.",
              description = "TODO",
              key = "Pass:setMaterial",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "material",
                      type = "Material",
                      description = "TODO"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setMeshMode",
              tag = "pipeline",
              summary = "Change whether mesh vertices are drawn as points, lines, or triangles.",
              description = "TODO",
              key = "Pass:setMeshMode",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "mode",
                      type = "MeshMode",
                      description = "TODO"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setProjection",
              tag = "camera",
              summary = "Set the field of view.",
              description = "Sets the projection for a single view.  4 field of view angles can be used, similar to the field of view returned by `lovr.headset.getViewAngles`.  Alternatively, a projection matrix can be used for other types of projections like orthographic, oblique, etc.  There is also a shorthand string \"orthographic\" that can be used to configure an orthographic projection.\n\nUp to 6 views are supported.  When rendering to the headset, both projections are changed to match the ones used by the headset.  This is also available by calling `lovr.headset.getViewAngles`.",
              key = "Pass:setProjection",
              module = "lovr.graphics",
              notes = "A far clipping plane of 0.0 can be used for an infinite far plane with reversed Z range.  This is the default.",
              related = {
                "lovr.headset.getViewAngles",
                "lovr.headset.getViewCount",
                "lovr.graphics.getViewPose",
                "lovr.graphics.setViewPose"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The index of the view to update."
                    },
                    {
                      name = "left",
                      type = "number",
                      description = "The left field of view angle, in radians."
                    },
                    {
                      name = "right",
                      type = "number",
                      description = "The right field of view angle, in radians."
                    },
                    {
                      name = "up",
                      type = "number",
                      description = "The top field of view angle, in radians."
                    },
                    {
                      name = "down",
                      type = "number",
                      description = "The bottom field of view angle, in radians."
                    },
                    {
                      name = "near",
                      type = "number",
                      description = "The near clipping plane distance, in meters.",
                      default = ".01"
                    },
                    {
                      name = "far",
                      type = "number",
                      description = "The far clipping plane distance, in meters.",
                      default = "100.0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The index of the view to update."
                    },
                    {
                      name = "matrix",
                      type = "Mat4",
                      description = "The projection matrix for the view."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setSampler",
              tag = "pipeline",
              summary = "Set the sampler.",
              description = "TODO",
              key = "Pass:setSampler",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "sampler",
                      type = "Sampler",
                      description = "TODO"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setScissor",
              tag = "pipeline",
              summary = "Set the scissor.",
              description = "TODO",
              key = "Pass:setScissor",
              module = "lovr.graphics",
              notes = "TODO not floating point, negative, limits, not pipeline, initial pass state",
              related = {
                "lovr.graphics.setViewport"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the upper-left corner of the scissor rectangle."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the upper-left corner of the scissor rectangle."
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "The width of the scissor rectangle."
                    },
                    {
                      name = "h",
                      type = "number",
                      description = "The height of the scissor rectangle."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setShader",
              tag = "pipeline",
              summary = "Set the active Shader.",
              description = "TODO",
              key = "Pass:setShader",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "shader",
                      type = "Shader",
                      description = "A custom Shader object to use for rendering."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Use one of the default shaders for drawing.",
                  arguments = {
                    {
                      name = "default",
                      type = "DefaultShader",
                      description = "One of the default shaders to use."
                    }
                  },
                  returns = {}
                },
                {
                  description = "Switch back to using an automatic shader for drawing.",
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "setStencilTest",
              tag = "pipeline",
              summary = "Configure the stencil test.",
              description = "TODO",
              key = "Pass:setStencilTest",
              module = "lovr.graphics",
              notes = "TODO",
              related = {
                "lovr.graphics.setStencilWrite",
                "lovr.graphics.setDepthTest"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "test",
                      type = "CompareMode",
                      description = "The new stencil test to use."
                    },
                    {
                      name = "value",
                      type = "number",
                      description = "The stencil value to compare against."
                    },
                    {
                      name = "mask",
                      type = "number",
                      description = "An optional mask to apply to stencil values before the comparison.",
                      default = "0xff"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Disable the stencil test.",
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "setStencilWrite",
              tag = "pipeline",
              summary = "Set whether draws write to the stencil buffer.",
              description = "TODO",
              key = "Pass:setStencilWrite",
              module = "lovr.graphics",
              notes = "TODO",
              related = {
                "lovr.graphics.setStencilTest",
                "lovr.graphics.setDepthTest"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "action",
                      type = "StencilAction",
                      description = "How pixels drawn will update the stencil buffer."
                    },
                    {
                      name = "value",
                      type = "number",
                      description = "When using the 'replace' action, this is the value to replace with.",
                      default = "1"
                    },
                    {
                      name = "mask",
                      type = "number",
                      description = "An optional mask to apply to stencil values before writing.",
                      default = "0xff"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "actions",
                      type = "table",
                      description = "A list of 3 stencil actions, used when a pixel fails the stencil test, fails the depth test, or passes the stencil test, respectively."
                    },
                    {
                      name = "value",
                      type = "number",
                      description = "When using the 'replace' action, this is the value to replace with.",
                      default = "1"
                    },
                    {
                      name = "mask",
                      type = "number",
                      description = "An optional mask to apply to stencil values before writing.",
                      default = "0xff"
                    }
                  },
                  returns = {}
                },
                {
                  description = "Disables stencil writing.",
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "setViewPose",
              tag = "camera",
              summary = "Set the camera pose.",
              description = "Sets the pose for a single view.  Objects rendered in this view will appear as though the camera is positioned using the given pose.\n\nUp to 6 views are supported.  When rendering to the headset, views are changed to match the eye positions.  These view poses are also available using `lovr.headset.getViewPose`.",
              key = "Pass:setViewPose",
              module = "lovr.graphics",
              related = {
                "lovr.headset.getViewPose",
                "lovr.headset.getViewCount",
                "lovr.graphics.getProjection",
                "lovr.graphics.setProjection"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The index of the view to update."
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the viewer, in meters."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the viewer, in meters."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the viewer, in meters."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the viewer is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "view",
                      type = "number",
                      description = "The index of the view to update."
                    },
                    {
                      name = "matrix",
                      type = "Mat4",
                      description = "A matrix containing the viewer pose."
                    },
                    {
                      name = "inverted",
                      type = "boolean",
                      description = "Whether the matrix is an inverted pose (a view matrix)."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setViewport",
              tag = "pipeline",
              summary = "Set the viewport.",
              description = "TODO",
              key = "Pass:setViewport",
              module = "lovr.graphics",
              notes = "TODO floating point, negative, flipped depth range, limits, not pipeline, initial pass state, what the hell is depth range",
              related = {
                "lovr.graphics.setScissor"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the upper-left corner of the viewport."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the upper-left corner of the viewport."
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "The width of the viewport."
                    },
                    {
                      name = "h",
                      type = "number",
                      description = "The height of the viewport."
                    },
                    {
                      name = "minDepth",
                      type = "number",
                      description = "The min component of the depth range.",
                      default = "0.0"
                    },
                    {
                      name = "maxDepth",
                      type = "number",
                      description = "The max component of the depth range.",
                      default = "1.0"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setWinding",
              tag = "pipeline",
              summary = "Set the winding direction of triangle vertices.",
              description = "TODO",
              key = "Pass:setWinding",
              module = "lovr.graphics",
              notes = "TODO",
              related = {
                "lovr.graphics.setCullMode"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "winding",
                      type = "Winding",
                      description = "Whether triangle vertices are ordered `clockwise` or `counterclockwise`."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setWireframe",
              tag = "pipeline",
              summary = "Enable or disable wireframe rendering.",
              description = "TODO",
              key = "Pass:setWireframe",
              module = "lovr.graphics",
              notes = "TODO",
              variants = {
                {
                  arguments = {
                    {
                      name = "enable",
                      type = "boolean",
                      description = "Whether wireframe rendering should be enabled."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "skybox",
              tag = "drawing",
              summary = "Draw a skybox.",
              description = "TODO",
              key = "Pass:skybox",
              module = "lovr.graphics",
              notes = "TODO",
              variants = {
                {
                  arguments = {
                    {
                      name = "skybox",
                      type = "Texture",
                      description = "The skybox to render.  Its `TextureType` can be `cube` to render as a cubemap, or `2d` to render as an equirectangular (spherical) 2D image."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {},
                  returns = {}
                }
              }
            },
            {
              name = "sphere",
              tag = "drawing",
              summary = "Draw a sphere.",
              description = "TODO",
              key = "Pass:sphere",
              module = "lovr.graphics",
              notes = "TODO",
              variants = {
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "transform",
                      description = "The transform to apply to the sphere."
                    },
                    {
                      name = "longitudes",
                      type = "number",
                      description = "The number of \"horizontal\" segments.",
                      default = "48"
                    },
                    {
                      name = "latitudes",
                      type = "number",
                      description = "The number of \"vertical\" segments.",
                      default = "longitudes / 2"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "text",
              tag = "drawing",
              summary = "Draw text.",
              description = "TODO",
              key = "Pass:text",
              module = "lovr.graphics",
              notes = "TODO",
              related = {
                "Pass:setFont"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "text",
                      type = "string",
                      description = "The text to render."
                    },
                    {
                      name = "transform",
                      type = "transform",
                      description = "The transform of the text."
                    },
                    {
                      name = "wrap",
                      type = "number",
                      description = "The maximum width of each line in meters (before scale is applied).  When zero, the text will not wrap.",
                      default = "0"
                    },
                    {
                      name = "halign",
                      type = "HorizontalAlign",
                      description = "The horizontal alignment.",
                      default = "'center'"
                    },
                    {
                      name = "valign",
                      type = "VerticalAlign",
                      description = "The vertical alignment.",
                      default = "'middle'"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "colortext",
                      type = "table",
                      description = "A table of strings with colors to render, in the form `{ color1, string1, color2, string2\n      }`, where color is a `Color`."
                    },
                    {
                      name = "transform",
                      type = "transform",
                      description = "The transform of the text."
                    },
                    {
                      name = "wrap",
                      type = "number",
                      description = "The maximum width of each line in meters (before scale is applied).  When zero, the text will not wrap.",
                      default = "0"
                    },
                    {
                      name = "halign",
                      type = "HorizontalAlign",
                      description = "The horizontal alignment.",
                      default = "'center'"
                    },
                    {
                      name = "valign",
                      type = "VerticalAlign",
                      description = "The vertical alignment.",
                      default = "'middle'"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "tick",
              tag = "transfer",
              summary = "Begin measuring GPU counters.",
              description = "TODO",
              key = "Pass:tick",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "tally",
                      type = "Tally",
                      description = "TODO"
                    },
                    {
                      name = "index",
                      type = "number",
                      description = "TODO"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "tock",
              tag = "transfer",
              summary = "Stop measuring GPU counters.",
              description = "TODO",
              key = "Pass:tock",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "tally",
                      type = "Tally",
                      description = "TODO"
                    },
                    {
                      name = "index",
                      type = "number",
                      description = "TODO"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "torus",
              tag = "drawing",
              summary = "Draw a donut.",
              description = "TODO",
              key = "Pass:torus",
              module = "lovr.graphics",
              notes = "TODO",
              variants = {
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "TransformXY2",
                      description = "The transform to apply to the torus.  The x scale is the radius, the z scale is the thickness."
                    },
                    {
                      name = "tsegments",
                      type = "number",
                      description = "The number of toroidal (circular) segments to render.",
                      default = "64"
                    },
                    {
                      name = "psegments",
                      type = "number",
                      description = "The number of poloidal (tubular) segments to render.",
                      default = "32"
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "transform",
              tag = "transform",
              summary = "Apply a general transform to the coordinate system.",
              description = "TODO",
              key = "Pass:transform",
              module = "lovr.graphics",
              notes = "TODO you can use combos of numbers/vectors/quats too (or use meta Transform type to explain)",
              related = {
                "lovr.graphics.translate",
                "lovr.graphics.rotate",
                "lovr.graphics.scale"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the translation.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the translation.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z component of the translation.",
                      default = "0"
                    },
                    {
                      name = "sx",
                      type = "number",
                      description = "The x scale factor.",
                      default = "1"
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y scale factor.",
                      default = "1"
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z scale factor.",
                      default = "1"
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians to rotate around the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "transform",
                      type = "Mat4",
                      description = "The matrix holding the transform to apply."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "translate",
              tag = "transform",
              summary = "Translate the coordinate system.",
              description = "TODO",
              key = "Pass:translate",
              module = "lovr.graphics",
              notes = "Order matters when scaling, translating, and rotating the coordinate system.",
              related = {
                "lovr.graphics.rotate",
                "lovr.graphics.scale",
                "lovr.graphics.transform"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The amount to translate on the x axis.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The amount to translate on the y axis.",
                      default = "0"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The amount to translate on the z axis.",
                      default = "0"
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "A vector to translate by."
                    }
                  },
                  returns = {}
                }
              }
            }
          }
        },
        {
          name = "Readback",
          summary = "TODO",
          description = "TODO",
          key = "Readback",
          module = "lovr.graphics",
          constructors = {
            "Pass:read"
          },
          methods = {
            {
              name = "getBlob",
              summary = "Get Readback's data as a Blob.",
              description = "Returns the Readback's data as a Blob.",
              key = "Readback:getBlob",
              module = "lovr.graphics",
              notes = "TODO what if it's an image?!",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "blob",
                      type = "Blob",
                      description = "The Blob."
                    }
                  }
                }
              },
              related = {
                "Readback:getData",
                "Readback:getImage"
              }
            },
            {
              name = "getData",
              summary = "Get Readback's data as a table.",
              description = "Returns the data from the Readback, as a table.",
              key = "Readback:getData",
              module = "lovr.graphics",
              notes = "TODO what if the readback is a buffer/texture?!",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "data",
                      type = "table",
                      description = "A table containing the values that were read back."
                    }
                  }
                }
              },
              related = {
                "Readback:getBlob",
                "Readback:getImage"
              }
            },
            {
              name = "getImage",
              summary = "Get Readback's data as an Image.",
              description = "Returns the Readback's data as an Image.",
              key = "Readback:getImage",
              module = "lovr.graphics",
              notes = "TODO what if it's a buffer or tally?!",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "image",
                      type = "Image",
                      description = "The Image."
                    }
                  }
                }
              },
              related = {
                "Readback:getData",
                "Readback:getBlob"
              }
            },
            {
              name = "isComplete",
              summary = "Check if a Readback is complete.",
              description = "Returns whether the Readback has completed on the GPU and its data is available.",
              key = "Readback:isComplete",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "complete",
                      type = "boolean",
                      description = "Whether the readback is complete."
                    }
                  }
                }
              }
            },
            {
              name = "wait",
              summary = "Wait for the Readback to finish.",
              description = "Blocks the CPU until the Readback is finished on the GPU.",
              key = "Readback:wait",
              module = "lovr.graphics",
              notes = "TODO what if the readback will never complete?!",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "waited",
                      type = "boolean",
                      description = "Whether the CPU had to be blocked for waiting."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Sampler",
          summary = "An object that controls how texture pixels are read.",
          description = "Samplers are objects that control how pixels are read from a texture.  They can control whether the pixels are smoothed, whether the texture wraps at the edge of its UVs, and more.\n\nEach has a default sampler that will be used by default, which can be changed using `Pass:setSampler`.  Also, samplers can be declared in shaders using the following syntax:\n\n    layout(set = 2, binding = X) uniform sampler mySampler;\n\nA Sampler can be sent to the variable using `Pass:send('mySampler', sampler)`.\n\nThe properties of a Sampler are immutable, and can't be changed after it's created.",
          key = "Sampler",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.newSampler"
          },
          methods = {
            {
              name = "getAnisotropy",
              summary = "Get the anisotropy level of the Sampler.",
              description = "Returns the anisotropy level of the Sampler.  Anisotropy smooths out a texture's appearance when viewed at grazing angles.",
              key = "Sampler:getAnisotropy",
              module = "lovr.graphics",
              notes = "Not all GPUs support anisotropy.  The maximum anisotropy level is given by the `anisotropy` limit of `lovr.graphics.getLimits`, which may be zero.  It's very common for the maximum to be 16, however.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "anisotropy",
                      type = "number",
                      description = "The anisotropy level of the sampler."
                    }
                  }
                }
              },
              related = {
                "Sampler:getFilter",
                "Sampler:getWrap",
                "Sampler:getCompareMode",
                "Sampler:getMipmapRange"
              }
            },
            {
              name = "getCompareMode",
              summary = "Get the compare mode of the Sampler.",
              description = "Returns the compare mode of the Sampler.  This is a feature typically only used for shadow mapping.  Using a sampler with a compare mode requires it to be declared in a shader as a `samplerShadow` instead of a `sampler` variable, and used with a texture that has a depth format.  The result of sampling a depth texture with a shadow sampler is a number between 0 and 1, indicating the percentage of sampled pixels that passed the comparison.",
              key = "Sampler:getCompareMode",
              module = "lovr.graphics",
              related = {
                "Sampler:getFilter",
                "Sampler:getWrap",
                "Sampler:getAnisotropy",
                "Sampler:getMipmapRange"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "compare",
                      type = "CompareMode",
                      description = "The compare mode of the sampler."
                    }
                  }
                }
              }
            },
            {
              name = "getFilter",
              summary = "Get the filter mode of the Sampler.",
              description = "Returns the filter mode of the Sampler.",
              key = "Sampler:getFilter",
              module = "lovr.graphics",
              related = {
                "Sampler:getWrap",
                "Sampler:getCompareMode",
                "Sampler:getAnisotropy",
                "Sampler:getMipmapRange"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "min",
                      type = "FilterMode",
                      description = "The filter mode used when the texture is minified."
                    },
                    {
                      name = "mag",
                      type = "FilterMode",
                      description = "The filter mode used when the texture is magnified."
                    },
                    {
                      name = "mip",
                      type = "FilterMode",
                      description = "The filter mode used to select a mipmap level."
                    }
                  }
                }
              }
            },
            {
              name = "getMipmapRange",
              summary = "Get the mipmap range of the Sampler.",
              description = "Returns the mipmap range of the Sampler.  This is used to clamp the range of mipmap levels that can be accessed from a texture.",
              key = "Sampler:getMipmapRange",
              module = "lovr.graphics",
              related = {
                "Sampler:getFilter",
                "Sampler:getWrap",
                "Sampler:getCompareMode",
                "Sampler:getAnisotropy"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "min",
                      type = "number",
                      description = "The minimum mipmap level that will be sampled (0 is the largest image)."
                    },
                    {
                      name = "max",
                      type = "number",
                      description = "The maximum mipmap level that will be sampled."
                    }
                  }
                }
              }
            },
            {
              name = "getWrap",
              summary = "Get the wrap mode of the Sampler.",
              description = "Returns the wrap mode of the sampler, used to wrap or clamp texture coordinates when the extend outside of the 0-1 range.",
              key = "Sampler:getWrap",
              module = "lovr.graphics",
              related = {
                "Sampler:getFilter",
                "Sampler:getCompareMode",
                "Sampler:getAnisotropy",
                "Sampler:getMipmapRange"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "WrapMode",
                      description = "The wrap mode used in the horizontal direction."
                    },
                    {
                      name = "y",
                      type = "WrapMode",
                      description = "The wrap mode used in the vertical direction."
                    },
                    {
                      name = "z",
                      type = "WrapMode",
                      description = "The wrap mode used in the \"z\" direction, for 3D textures only."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Shader",
          summary = "TODO",
          description = "TODO",
          key = "Shader",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.newShader",
            "Shader:clone"
          },
          methods = {
            {
              name = "clone",
              summary = "Clone a Shader.",
              description = "TODO",
              key = "Shader:clone",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "source",
                      type = "Shader",
                      description = "The Shader to clone."
                    },
                    {
                      name = "flags",
                      type = "table",
                      description = "TODO"
                    }
                  },
                  returns = {
                    {
                      name = "shader",
                      type = "Shader",
                      description = "The new Shader."
                    }
                  }
                }
              }
            },
            {
              name = "getType",
              summary = "Get the type of the Shader.",
              description = "TODO",
              key = "Shader:getType",
              module = "lovr.graphics",
              related = {
                "Shader:hasStage"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "type",
                      type = "ShaderType",
                      description = "The type of the Shader."
                    }
                  }
                }
              }
            },
            {
              name = "getWorkgroupSize",
              summary = "Get the workgroup size of a compute shader.",
              description = "Returns the workgroup size of a compute shader.  TODO what is it.",
              key = "Shader:getWorkgroupSize",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x size of a workgroup."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y size of a workgroup."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z size of a workgroup."
                    }
                  }
                }
              }
            },
            {
              name = "hasAttribute",
              summary = "Check if the Shader has a given vertex attribute.",
              description = "Returns whether the Shader has a vertex attribute, by name or location.",
              key = "Shader:hasAttribute",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {
                    {
                      name = "name",
                      type = "string",
                      description = "The name of an attribute."
                    }
                  },
                  returns = {
                    {
                      name = "exists",
                      type = "boolean",
                      description = "Whether the Shader has the attribute."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "location",
                      type = "number",
                      description = "The location of an attribute."
                    }
                  },
                  returns = {
                    {
                      name = "exists",
                      type = "boolean",
                      description = "Whether the Shader has the attribute."
                    }
                  }
                }
              },
              examples = {
                {
                  code = "function lovr.load()\n  shader = lovr.graphics.newShader([[\n    layout(location = 7) in uint coolAttribute;\n\n    vec4 lovrmain() {\n      return DefaultPosition;\n    }\n  ]], [[\n    vec4 lovrmain() {\n      return DefaultColor;\n    }\n  ]])\n\n  print(shader:hasAttribute('coolAttribute')) --> true\n  print(shader:hasAttribute(7)) --> true\n  print(shader:hasAttribute(8)) --> false\nend"
                }
              }
            },
            {
              name = "hasStage",
              summary = "Check if the Shader has a given stage.",
              description = "Returns whether the Shader has a given stage.",
              key = "Shader:hasStage",
              module = "lovr.graphics",
              related = {
                "Shader:getType"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "stage",
                      type = "ShaderStage",
                      description = "The stage."
                    }
                  },
                  returns = {
                    {
                      name = "exists",
                      type = "boolean",
                      description = "Whether the Shader has the stage."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Tally",
          summary = "TODO",
          description = "TODO",
          key = "Tally",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.newTally"
          },
          methods = {
            {
              name = "getCount",
              summary = "Get the number of slots in the Tally.",
              description = "Returns the number of slots in the Tally.",
              key = "Tally:getCount",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of slots in the Tally."
                    }
                  }
                }
              }
            },
            {
              name = "getType",
              summary = "TODO",
              description = "TODO",
              key = "Tally:getType",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "type",
                      type = "TallyType",
                      description = "TODO"
                    }
                  }
                }
              }
            },
            {
              name = "getViewCount",
              summary = "Get the number of render Pass views the Tally is configured for.",
              description = "Tally objects with the `time` type can only be used in render passes with a certain number of views.  This returns that number.",
              key = "Tally:getViewCount",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "views",
                      type = "number",
                      description = "The number of views the Tally is compatible with."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Texture",
          summary = "A multidimensional block of memory on the GPU.",
          description = "Textures are multidimensional blocks of memory on the GPU, contrasted with `Buffer` objects which are one-dimensional.  Textures are used as the destination for rendering operations, and textures loaded from images provide surface data to `Material` objects.",
          key = "Texture",
          module = "lovr.graphics",
          constructors = {
            "lovr.graphics.newTexture",
            "Texture:newView"
          },
          methods = {
            {
              name = "getDimensions",
              summary = "Get the dimensions of the Texture.",
              description = "Returns the width, height, and depth of the Texture.",
              key = "Texture:getDimensions",
              module = "lovr.graphics",
              related = {
                "Texture:getWidth",
                "Texture:getHeight",
                "Texture:getLayerCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the Texture."
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the Texture."
                    },
                    {
                      name = "layers",
                      type = "number",
                      description = "The number of layers in the Texture."
                    }
                  }
                }
              }
            },
            {
              name = "getFormat",
              summary = "Get the format of the Texture.",
              description = "Returns the format of the texture.",
              key = "Texture:getFormat",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "format",
                      type = "TextureFormat",
                      description = "The format of the Texture."
                    }
                  }
                }
              }
            },
            {
              name = "getHeight",
              summary = "Get the height of the Texture, in pixels.",
              description = "Returns the height of the Texture, in pixels.",
              key = "Texture:getHeight",
              module = "lovr.graphics",
              related = {
                "Texture:getWidth",
                "Texture:getLayerCount",
                "Texture:getDimensions"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the Texture, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "getLayerCount",
              summary = "Get the layer count of the Texture.",
              description = "Returns the layer count of the Texture.  2D textures always have 1 layer and cubemaps always have 6 layers.  3D and array textures have a variable number of layers.",
              key = "Texture:getLayerCount",
              module = "lovr.graphics",
              related = {
                "Texture:getWidth",
                "Texture:getHeight",
                "Texture:getDimensions"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "layers",
                      type = "number",
                      description = "The layer count of the Texture."
                    }
                  }
                }
              }
            },
            {
              name = "getMipmapCount",
              summary = "Get the number of mipmap levels in the Texture.",
              description = "Returns the number of mipmap levels in the Texture.",
              key = "Texture:getMipmapCount",
              module = "lovr.graphics",
              related = {
                "lovr.graphics.newTexture",
                "Sampler:getMipmapRange",
                "Pass:mipmap"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "mipmaps",
                      type = "number",
                      description = "The number of mipmap levels in the Texture."
                    }
                  }
                }
              }
            },
            {
              name = "getParent",
              summary = "Get the parent of a texture view.",
              description = "Returns the parent of a Texture view, which is the Texture that it references.  Returns `nil` if the Texture is not a view.",
              key = "Texture:getParent",
              module = "lovr.graphics",
              related = {
                "Texture:isView",
                "Texture:newView"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "parent",
                      type = "Texture",
                      description = "The parent of the texture, or `nil` if the texture is not a view."
                    }
                  }
                }
              }
            },
            {
              name = "getSampleCount",
              summary = "Get the number of MSAA samples in the Texture.",
              description = "Returns the number of samples in the texture.  Multiple samples are used for multisample antialiasing when rendering to the texture.  Currently, the sample count is either 1 (not antialiased) or 4 (antialiased).",
              key = "Texture:getSampleCount",
              module = "lovr.graphics",
              related = {
                "lovr.graphics.newTexture",
                "Pass:getSampleCount"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of samples in the Texture."
                    }
                  }
                }
              }
            },
            {
              name = "getType",
              summary = "Get the type of the Texture.",
              description = "Returns the type of the texture.",
              key = "Texture:getType",
              module = "lovr.graphics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "type",
                      type = "TextureType",
                      description = "The type of the Texture."
                    }
                  }
                }
              }
            },
            {
              name = "getWidth",
              summary = "Get the width of the Texture, in pixels.",
              description = "Returns the width of the Texture, in pixels.",
              key = "Texture:getWidth",
              module = "lovr.graphics",
              related = {
                "Texture:getHeight",
                "Texture:getLayerCount",
                "Texture:getDimensions"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the Texture, in pixels."
                    }
                  }
                }
              }
            },
            {
              name = "hasUsage",
              summary = "Check if a Texture was created with a set of usage flags.",
              description = "Returns whether a Texture was created with a set of `TextureUsage` flags.  Usage flags are specified when the Texture is created, and restrict what you can do with a Texture object.  By default, only the `sample` usage is enabled.  Applying a smaller set of usage flags helps LÖVR optimize things better.",
              key = "Texture:hasUsage",
              module = "lovr.graphics",
              related = {
                "lovr.graphics.newTexture"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "...",
                      type = "TextureUsage",
                      description = "One or more usage flags."
                    }
                  },
                  returns = {
                    {
                      name = "supported",
                      type = "boolean",
                      description = "Whether the Texture has all the provided usage flags."
                    }
                  }
                }
              }
            },
            {
              name = "isView",
              summary = "Check if a Texture is a texture view.",
              description = "Returns whether a Texture is a texture view, created with `Texture:newView`.",
              key = "Texture:isView",
              module = "lovr.graphics",
              related = {
                "Texture:getParent",
                "Texture:newView"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "view",
                      type = "boolean",
                      description = "Whether the Texture is a texture view."
                    }
                  }
                }
              }
            },
            {
              name = "newView",
              summary = "Create a texture view referencing a parent Texture.",
              description = "Creates a new Texture view.  A texture view does not store any pixels on its own, but instead uses the pixel data of a \"parent\" Texture object.  The width, height, format, sample count, and usage flags all match the parent.  The view may have a different `TextureType` from the parent, and it may reference a subset of the parent texture's layers and mipmap levels.\n\nTexture views can be used as render targets in a render pass and they can be bound to Shaders. They can not currently be used for transfer operations.  They are used for:\n\n- Reinterpretation of texture contents.  For example, a cubemap can be treated as\n  an array texture.\n- Rendering to a particular image or mipmap level of a texture.\n- Binding a particular image or mipmap level to a shader.",
              key = "Texture:newView",
              module = "lovr.graphics",
              related = {
                "Texture:isView",
                "Texture:getParent",
                "lovr.graphics.newTexture"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "parent",
                      type = "Texture",
                      description = "The parent Texture to create the view of."
                    },
                    {
                      name = "type",
                      type = "TextureType",
                      description = "The texture type of the view."
                    },
                    {
                      name = "layer",
                      type = "number",
                      description = "The index of the first layer in the view.",
                      default = "1"
                    },
                    {
                      name = "layerCount",
                      type = "number",
                      description = "The number of layers in the view, or `nil` to use all remaining layers.",
                      default = "nil"
                    },
                    {
                      name = "mipmap",
                      type = "number",
                      description = "The index of the first mipmap in the view.",
                      default = "1"
                    },
                    {
                      name = "mipmapCount",
                      type = "number",
                      description = "The number of mipmaps in the view, or `nil` to use all remaining mipmaps.",
                      default = "nil"
                    }
                  },
                  returns = {
                    {
                      name = "view",
                      type = "Texture",
                      description = "The new texture view."
                    }
                  }
                }
              }
            }
          }
        }
      },
      functions = {
        {
          name = "compileShader",
          tag = "graphics-objects",
          summary = "Compile a Shader to bytecode.",
          description = "Compiles shader code to SPIR-V bytecode.  The bytecode can be passed to `lovr.graphics.newShader` to create shaders, which will be faster than creating it from GLSL. The bytecode is portable, so bytecode compiled on one platform will work on other platforms. This allows shaders to be precompiled in a build step.",
          key = "lovr.graphics.compileShader",
          module = "lovr.graphics",
          notes = "The input can be GLSL or SPIR-V.  If it's SPIR-V, it will be returned unchanged as a Blob.\n\nIf the shader fails to compile, an error will be thrown with the error message.",
          related = {
            "lovr.graphics.newShader",
            "Shader"
          },
          variants = {
            {
              arguments = {
                {
                  name = "stage",
                  type = "ShaderStage",
                  description = "The type of shader to compile."
                },
                {
                  name = "source",
                  type = "string",
                  description = "A string or filename with shader code."
                }
              },
              returns = {
                {
                  name = "bytecode",
                  type = "Blob",
                  description = "A Blob containing compiled SPIR-V code."
                }
              }
            },
            {
              arguments = {
                {
                  name = "stage",
                  type = "ShaderStage",
                  description = "The type of shader to compile."
                },
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob containing shader code."
                }
              },
              returns = {
                {
                  name = "bytecode",
                  type = "Blob",
                  description = "A Blob containing compiled SPIR-V code."
                }
              }
            }
          }
        },
        {
          name = "getBackgroundColor",
          tag = "graphics-global",
          summary = "Get the background color.",
          description = "Returns the global background color.  The textures in a render pass will be cleared to this color at the beginning of the pass if no other clear option is specified.  Additionally, the headset and window will be cleared to this color before rendering.",
          key = "lovr.graphics.getBackgroundColor",
          module = "lovr.graphics",
          notes = "Setting the background color in `lovr.draw` will apply on the following frame, since the default pass is cleared before `lovr.draw` is called.\n\nInternally, this color is applied to the default pass objects when retrieving one of them using `lovr.headset.getPass` or `lovr.graphics.getPass`.  Both are called automatically by the default `lovr.run` implementation.\n\nUsing the background color to clear the display is expected to be more efficient than manually clearing after a render pass begins, especially on mobile GPUs.",
          related = {
            "lovr.graphics.getPass",
            "Pass:clear",
            "Pass:fill"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "r",
                  type = "number",
                  description = "The red component of the background color."
                },
                {
                  name = "g",
                  type = "number",
                  description = "The green component of the background color."
                },
                {
                  name = "b",
                  type = "number",
                  description = "The blue component of the background color."
                },
                {
                  name = "a",
                  type = "number",
                  description = "The alpha component of the background color."
                }
              }
            }
          }
        },
        {
          name = "getBuffer",
          tag = "graphics-objects",
          summary = "Get a temporary Buffer.",
          description = "Creates a temporary Buffer.",
          key = "lovr.graphics.getBuffer",
          module = "lovr.graphics",
          notes = "The format table can contain a list of `FieldType`s or a list of tables to provide extra information about each field.  Each inner table has the following keys:\n\n- `type` is the `FieldType` of the field and is required.\n- `offset` is the byte offset of the field.  Any fields with a `nil` offset will be placed next\n  to each other sequentially in memory, subject to any padding required by the Buffer's layout.\n  In practice this means that an `offset` should be set for either all of the fields or none of\n  them.\n- `location` is the vertex attribute location of each field.  This is used to match up each\n  field with an attribute declared in a shader, and doesn't have any purpose when binding the\n  buffer as a uniform or storage buffer.  Any fields with a `nil` location will use an\n  autoincrementing location starting at zero.  Named locations are not currently supported, but\n  may be added in the future.\n\nIf no table or Blob is used to define the initial Buffer contents, its data will be undefined.\n\nThere is currently a max of 16 fields.",
          related = {
            "lovr.graphics.newBuffer"
          },
          variants = {
            {
              arguments = {
                {
                  name = "length",
                  type = "number",
                  description = "The length of the Buffer."
                },
                {
                  name = "type",
                  type = "FieldType",
                  description = "The type of each item in the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "data",
                  type = "table",
                  description = "The initial data to put into the Buffer.  The length of the Buffer will be determined by the contents of the table.  The contents can be a mix of tables, numbers, and vectors, but the length calculation requires each field to consistently use one type of data."
                },
                {
                  name = "type",
                  type = "FieldType",
                  description = "The type of each item in the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "length",
                  type = "number",
                  description = "The length of the Buffer."
                },
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer (see notes).  `nil` is a valid format, but means only `Blob`s can be written to the Buffer from Lua.",
                  default = "nil",
                  table = {
                    {
                      name = "layout",
                      type = "BufferLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "data",
                  type = "table",
                  description = "The initial data to put into the Buffer.  The length of the Buffer will be determined by the contents of the table.  The contents can be a mix of tables, numbers, and vectors, but the length calculation requires each field to consistently use one type of data."
                },
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer (see notes).  `nil` is a valid format, but means only `Blob`s can be written to the Buffer from Lua.",
                  default = "nil",
                  table = {
                    {
                      name = "layout",
                      type = "BufferLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob with the initial contents of the Buffer.  The size of the Blob will be used to determine the length of the Buffer."
                },
                {
                  name = "type",
                  type = "FieldType",
                  description = "The type of each item in the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob with the initial contents of the Buffer.  The size of the Blob will be used to determine the length of the Buffer."
                },
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer (see notes).  `nil` is a valid format, but means only `Blob`s can be written to the Buffer from Lua.",
                  default = "nil",
                  table = {
                    {
                      name = "layout",
                      type = "BufferLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            }
          }
        },
        {
          name = "getDefaultFont",
          tag = "graphics-objects",
          summary = "Get the default Font.",
          description = "Returns the default Font.  The default font is Varela Round, created at 32px with a spread value of `4.0`.  It's used by `Pass:text` if no Font is provided.",
          key = "lovr.graphics.getDefaultFont",
          module = "lovr.graphics",
          related = {
            "Pass:text",
            "lovr.graphics.newFont"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "font",
                  type = "Font",
                  description = "The default Font object."
                }
              }
            }
          }
        },
        {
          name = "getDevice",
          tag = "graphics-misc",
          summary = "Get information about the graphics device and driver.",
          description = "Returns information about the graphics device and driver.",
          key = "lovr.graphics.getDevice",
          module = "lovr.graphics",
          notes = "The device and vendor ID numbers will usually be PCI IDs, which are standardized numbers consisting of 4 hex digits.  Various online databases and system utilities can be used to look up these numbers.  Here are some example vendor IDs for a few popular GPU manufacturers:\n\n<table>\n  <thead>\n    <tr>\n      <td>ID</td>\n      <td>Vendor</td>\n    </tr>\n  </thead>\n  <tbody>\n    <tr>\n      <td><code>0x1002</code></td>\n      <td>Advanced Micro Devices, Inc.</td>\n    </tr>\n    <tr>\n      <td><code>0x8086</code></td>\n      <td>Intel Corporation</td>\n    </tr>\n    <tr>\n      <td><code>0x10de</code></td>\n      <td>NVIDIA Corporation</td>\n    </tr>\n  </tbody> </table>\n\nIt is not currently possible to get the version of the driver, although this could be added.\n\nRegarding multiple GPUs: If OpenXR is enabled, the OpenXR runtime has control over which GPU is used, which ensures best compatibility with the VR headset.  Otherwise, the \"first\" GPU returned by the renderer will be used.  There is currently no other way to pick a GPU to use.",
          related = {
            "lovr.graphics.getFeatures",
            "lovr.graphics.getLimits"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "device",
                  type = "table",
                  table = {
                    {
                      name = "id",
                      type = "number",
                      description = "The vendor-unique number for this GPU."
                    },
                    {
                      name = "vendor",
                      type = "number",
                      description = "The identifier of the GPU vendor."
                    },
                    {
                      name = "name",
                      type = "string",
                      description = "The name of the GPU."
                    },
                    {
                      name = "renderer",
                      type = "string",
                      description = "The renderer in use, currently either \"Vulkan\" or \"WebGPU\"."
                    },
                    {
                      name = "subgroupSize",
                      type = "number",
                      description = "The number of threads that run in a single GPU compute unit.  This is usually 32 or 64, and is sometimes called the \"wave\" or \"warp\" size.  This can be used to optimize compute shaders for the current hardware."
                    },
                    {
                      name = "discrete",
                      type = "boolean",
                      description = "Whether the GPU is a discrete graphics card."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "getFeatures",
          tag = "graphics-misc",
          summary = "Get the supported GPU features.",
          description = "Returns a table indicating which features are supported by the GPU.",
          key = "lovr.graphics.getFeatures",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.isFormatSupported",
            "lovr.graphics.getDevice",
            "lovr.graphics.getLimits"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "features",
                  type = "table",
                  description = "",
                  table = {
                    {
                      name = "textureBC",
                      type = "boolean",
                      description = "Whether `TextureFormat`s starting with `bc` are supported. This will almost always be `true` on desktop GPUs and will almost always be `false` on mobile GPUs."
                    },
                    {
                      name = "textureASTC",
                      type = "boolean",
                      description = "Whether `TextureFormat`s beginning with `astc` are supported.  This will almost always be `true` on mobile GPUs and will almost always be `false` on desktop GPUs."
                    },
                    {
                      name = "wireframe",
                      type = "boolean",
                      description = "When supported, `Pass:setWireframe` will work, otherwise it will do nothing. This will always be `true` when using Vulkan, and will always be `false` when using WebGPU."
                    },
                    {
                      name = "depthClamp",
                      type = "boolean",
                      description = "When supported, `Pass:setDepthClamp` will work, otherwise it will do nothing."
                    },
                    {
                      name = "indirectDrawFirstInstance",
                      type = "boolean",
                      description = "Whether indirect draws can set the firstInstance property of buffer memory to something other than zero."
                    },
                    {
                      name = "float64",
                      type = "boolean",
                      description = "Whether shader code can use doubles."
                    },
                    {
                      name = "int64",
                      type = "boolean",
                      description = "Whether shader code can use signed and unsigned 64-bit integers."
                    },
                    {
                      name = "int16",
                      type = "boolean",
                      description = "Whether shader code can use signed and unsigned 16-bit integers."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "getLimits",
          tag = "graphics-misc",
          summary = "Get the limits of the current GPU.",
          description = "Returns limits of the current GPU.",
          key = "lovr.graphics.getLimits",
          module = "lovr.graphics",
          notes = "The limit ranges are as follows:\n\n<table>\n  <thead>\n    <tr>\n      <td>Limit</td>\n      <td>Minimum</td>\n      <td>Maximum</td>\n    </tr>\n  </thead>\n  <tbody>\n    <tr>\n      <td><code>textureSize2D</code></td>\n      <td>4096</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>textureSize3D</code></td>\n      <td>256</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>textureSizeCube</code></td>\n      <td>4096</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>textureLayers</code></td>\n      <td>256</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>renderSize</code></td>\n      <td>{ 4096, 4096, 6 }</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>uniformBuffersPerStage</code></td>\n      <td>9</td>\n      <td>32*</td>\n    </tr>\n    <tr>\n      <td><code>storageBuffersPerStage</code></td>\n      <td>4</td>\n      <td>32*</td>\n    </tr>\n    <tr>\n      <td><code>sampledTexturesPerStage</code></td>\n      <td>32</td>\n      <td>32*</td>\n    </tr>\n    <tr>\n      <td><code>storageTexturesPerStage</code></td>\n      <td>4</td>\n      <td>32*</td>\n    </tr>\n    <tr>\n      <td><code>samplersPerStage</code></td>\n      <td>15</td>\n      <td>32*</td>\n    </tr>\n    <tr>\n      <td><code>resourcesPerShader</code></td>\n      <td>32</td>\n      <td>32*</td>\n    </tr>\n    <tr>\n      <td><code>uniformBufferRange</code></td>\n      <td>65536</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>storageBufferRange</code></td>\n      <td>134217728 (128MB)</td>\n      <td>1073741824 (1GB)*</td>\n    </tr>\n    <tr>\n      <td><code>uniformBufferAlign</code></td>\n      <td></td>\n      <td>256</td>\n    </tr>\n    <tr>\n      <td><code>storageBufferAlign</code></td>\n      <td></td>\n      <td>64</td>\n    </tr>\n    <tr>\n      <td><code>vertexAttributes</code></td>\n      <td>16</td>\n      <td>16*</td>\n    </tr>\n    <tr>\n      <td><code>vertexBufferStride</code></td>\n      <td>2048</td>\n      <td>65535*</td>\n    </tr>\n    <tr>\n      <td><code>vertexShaderOutputs</code></td>\n      <td>64</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>clipDistances</code></td>\n      <td>0</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>cullDistances</code></td>\n      <td>0</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>clipAndCullDistances</code></td>\n      <td>0</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>computeDispatchCount</code></td>\n      <td>{ 65536, 65536, 65536 }</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>computeWorkgroupSize</code></td>\n      <td>{ 128, 128, 64 }</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>computeWorkgroupVolume</code></td>\n      <td>128</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>computeSharedMemory</code></td>\n      <td>16384 (16KB)</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>pushConstantSize</code></td>\n      <td>128</td>\n      <td>256*</td>\n    </tr>\n    <tr>\n      <td><code>indirectDrawCount</code></td>\n      <td>1</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>instances</code></td>\n      <td>134217727</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>anisotropy</code></td>\n      <td>0.0</td>\n      <td></td>\n    </tr>\n    <tr>\n      <td><code>pointSize</code></td>\n      <td>1.0</td>\n      <td></td>\n    </tr>\n  </tbody> </table>\n\nNote: in the table above, `*` means that LÖVR itself is imposing a cap on the limit, instead of the GPU.",
          related = {
            "lovr.graphics.isFormatSupported",
            "lovr.graphics.getDevice",
            "lovr.graphics.getFeatures"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "limits",
                  type = "table",
                  description = "",
                  table = {
                    {
                      name = "textureSize2D",
                      type = "number",
                      description = "The maximum width/height of `2d` and `array` textures."
                    },
                    {
                      name = "textureSize3D",
                      type = "number",
                      description = "The maximum width/height/depth of `3d` textures."
                    },
                    {
                      name = "textureSizeCube",
                      type = "number",
                      description = "The maximum width/height of `cube` textures."
                    },
                    {
                      name = "textureLayers",
                      type = "number",
                      description = "The maximum depth of `array` textures."
                    },
                    {
                      name = "renderSize",
                      type = "table",
                      description = "The maximum width, height, and layer count of a texture (or texture view) used as a render target."
                    },
                    {
                      name = "uniformBuffersPerStage",
                      type = "number",
                      description = "The maximum number of uniform buffers in a shader stage."
                    },
                    {
                      name = "storageBuffersPerStage",
                      type = "number",
                      description = "The maximum number of storage buffers in a shader stage."
                    },
                    {
                      name = "sampledTexturesPerStage",
                      type = "number",
                      description = "The maximum number of sampled textures in a shader stage."
                    },
                    {
                      name = "storageTexturesPerStage",
                      type = "number",
                      description = "The maximum number of storage textures in a shader stage."
                    },
                    {
                      name = "samplersPerStage",
                      type = "number",
                      description = "The maximum number of samplers in a shader stage."
                    },
                    {
                      name = "resourcesPerShader",
                      type = "number",
                      description = "The maximum combined number of buffers, textures, and sampler variables in a Shader."
                    },
                    {
                      name = "uniformBufferRange",
                      type = "number",
                      description = "The maximum range of bytes that can be bound to a uniform buffer in a shader."
                    },
                    {
                      name = "storageBufferRange",
                      type = "number",
                      description = "The maximum range of bytes that can be bound to a storage buffer in a shader."
                    },
                    {
                      name = "uniformBufferAlign",
                      type = "number",
                      description = "When binding a range of bytes to a uniform buffer binding in a shader, the byte offset of the range must be a multiple of this limit's value."
                    },
                    {
                      name = "storageBufferAlign",
                      type = "number",
                      description = "When binding a range of bytes to a storage buffer binding in a shader, the byte offset of the range must be a multiple of this limit's value."
                    },
                    {
                      name = "vertexAttributes",
                      type = "number",
                      description = "The maximum number of input attributes in a vertex shader."
                    },
                    {
                      name = "vertexBufferStride",
                      type = "number",
                      description = "The maximum stride of a buffer used as a vertex buffer, in bytes."
                    },
                    {
                      name = "vertexShaderOutputs",
                      type = "number",
                      description = "The maximum number of components output from a vertex shader."
                    },
                    {
                      name = "clipDistances",
                      type = "number",
                      description = "The maximum number of clipping planes declared by a shader."
                    },
                    {
                      name = "cullDistances",
                      type = "number",
                      description = "The maximum number of cull distances declared by a shader."
                    },
                    {
                      name = "clipAndCullDistances",
                      type = "number",
                      description = "The maximum number of clipping planes and cull distances declared by a shader."
                    },
                    {
                      name = "workgroupCount",
                      type = "table",
                      description = "The maximum values of `x`, `y`, and `z` in `Pass:compute`."
                    },
                    {
                      name = "workgroupSize",
                      type = "table",
                      description = "The maximum values of `local_size_x`, `local_size_y`, and `local_size_z` declared in a compute shader."
                    },
                    {
                      name = "totalWorkgroupSize",
                      type = "number",
                      description = "The maximum product of `local_size_x`, `local_size_y`, and `local_size_z` in a compute shader."
                    },
                    {
                      name = "computeSharedMemory",
                      type = "number",
                      description = "The maximum number of bytes used by `shared` variables in compute shaders."
                    },
                    {
                      name = "shaderConstantSize",
                      type = "number",
                      description = "The maximum number of bytes of push constants that can be in a Shader.  Push constants are shared between stages, so the stage with the largest amount of push constant data will count towards this limit."
                    },
                    {
                      name = "indirectDrawCount",
                      type = "number",
                      description = "The maximum number of draws that can be issued by an indirect draw call."
                    },
                    {
                      name = "instances",
                      type = "number",
                      description = "The maximum number of instances that can be rendered in a draw call."
                    },
                    {
                      name = "anisotropy",
                      type = "number",
                      description = "The maximum value of the `anisotropy` parameter in `lovr.graphics.newSampler`."
                    },
                    {
                      name = "pointSize",
                      type = "number",
                      description = "The maximum point size."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "getPass",
          tag = "graphics-objects",
          summary = "Get a temporary Pass.",
          description = "Creates and returns a temporary Pass object.",
          key = "lovr.graphics.getPass",
          module = "lovr.graphics",
          notes = "Fun facts about render passes:\n\n- Textures must have the same dimensions, layer counts, and sample counts.\n- Textures must have been created with the `render` `TextureUsage`.\n- If `mipmap` is true, then any textures with mipmaps must have the `transfer` `TextureUsage`.\n- It's okay to have zero color textures, but in this case there must be a depth texture.\n- Setting `clear` to `false` for textures is usually very slow on mobile GPUs.\n\nFor `compute` and `transfer` passes, all of the commands in the pass act as though they run in parallel.  This means that writing to the same element of a buffer twice, or writing to it and reading from it again is not guaranteed to work properly on all GPUs.  LÖVR is not currently able to check for this.  If compute or transfers need to be sequenced, multiple passes should be used.  It is, however, completely fine to read and write to non-overlapping regions of the same buffer or texture.",
          related = {
            "lovr.graphics.submit",
            "lovr.graphics.getWindowPass",
            "lovr.headset.getPass"
          },
          variants = {
            {
              description = "Create a compute or transfer pass.",
              arguments = {
                {
                  name = "type",
                  type = "PassType",
                  description = "The type of pass to create."
                }
              },
              returns = {
                {
                  name = "pass",
                  type = "Pass",
                  description = "The new Pass."
                }
              }
            },
            {
              description = "Create a render pass.",
              arguments = {
                {
                  name = "type",
                  type = "PassType",
                  description = "The type of pass to create."
                },
                {
                  name = "texture",
                  type = "Texture",
                  description = "The texture the render pass will render to.  Ignored for non-render passes."
                }
              },
              returns = {
                {
                  name = "pass",
                  type = "Pass",
                  description = "The new Pass."
                }
              }
            },
            {
              description = "Create a render pass, with options.",
              arguments = {
                {
                  name = "type",
                  type = "PassType",
                  description = "The type of pass to create."
                },
                {
                  name = "canvas",
                  type = "table",
                  description = "Render pass configuration.  Up to 4 textures can be provided in table keys 1 through 4. Ignored for non-render passes.",
                  table = {
                    {
                      name = "depth",
                      type = "table",
                      description = "Depth/stencil buffer configuration.  In addition to a table, it can be a `Texture`, a `TextureFormat`, or `false` to disable the depth buffer.",
                      table = {
                        {
                          name = "format",
                          type = "TextureFormat",
                          description = "The format of the depth buffer texture, which must be a depth format (the ones that start with `d`).  LÖVR will create or reuse an internal depth buffer with this format.",
                          default = "'d32f'"
                        },
                        {
                          name = "texture",
                          type = "Texture",
                          description = "A Texture to use as the depth buffer.  Takes precedence over `format`."
                        },
                        {
                          name = "clear",
                          type = "number",
                          description = "How to clear the depth buffer at the beginning of the pass.  Can be a floating point number to clear each pixel to, `true` to do a \"fast clear\" that clears to random data, or `false` to not clear at all and instead load the depth texture's pixels.",
                          default = "0"
                        }
                      }
                    },
                    {
                      name = "clear",
                      type = "*",
                      description = "How to clear the color textures at the beginning of the pass.  If this is a boolean or a color, that value will be used for all color textures.  It can also be a table of colors or booleans, one for each color texture.  Colors may be provided as `Vec3`, `Vec4`, hexcodes, or tables of numbers.  Note that tables of hexcode colors are ambiguous and therefore unsupported.  When using a boolean, `true` means to do a \"fast clear\" that clears the texture to random data, and `false` means to not clear at all and instead load the texture's existing pixels."
                    },
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of multisamples to use.  Can be 4 for antialiasing, or 1 to disable antialiasing.",
                      default = "4"
                    },
                    {
                      name = "mipmap",
                      type = "boolean",
                      description = "Whether mipmaps for the color and depth textures should be regenerated after the pass is finished.",
                      default = "false"
                    }
                  }
                }
              },
              returns = {
                {
                  name = "pass",
                  type = "Pass",
                  description = "The new Pass."
                }
              }
            }
          }
        },
        {
          name = "getWindowPass",
          tag = "graphics-objects",
          summary = "Get the window pass.",
          description = "Returns the window pass.  This is a builtin render `Pass` object that renders to the desktop window texture.  If the desktop window was not open when the graphics module was initialized, this function will return `nil`.",
          key = "lovr.graphics.getWindowPass",
          module = "lovr.graphics",
          notes = "`lovr.conf` may be used to change the settings for the pass:  `t.graphics.antialias` enables antialiasing, and `t.graphics.stencil` enables the stencil buffer.\n\nThis pass clears the window texture to the background color, which can be changed using `lovr.graphics.setBackgroundColor`.",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "pass",
                  type = "Pass",
                  description = "The window pass, or `nil` if there is no window."
                }
              }
            }
          }
        },
        {
          name = "isFormatSupported",
          tag = "graphics-misc",
          summary = "Check if a Texture format is supported.",
          description = "Returns the type of operations the GPU supports for a texture format, if any.",
          key = "lovr.graphics.isFormatSupported",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.getDevice",
            "lovr.graphics.getFeatures",
            "lovr.graphics.getLimits"
          },
          variants = {
            {
              arguments = {
                {
                  name = "format",
                  type = "TextureFormat",
                  description = "The texture format to query."
                },
                {
                  name = "...features",
                  type = "TextureFeature",
                  description = "Zero or more features to check.  If no features are given, this function will return whether the GPU supports *any* feature for this format.  Otherwise, this function will only return true if *all* of the input features are supported."
                }
              },
              returns = {
                {
                  name = "supported",
                  type = "boolean",
                  description = "Whether the GPU supports these operations for textures with this format."
                }
              }
            }
          }
        },
        {
          name = "newBuffer",
          tag = "graphics-objects",
          summary = "Create a new Buffer.",
          description = "Creates a Buffer.",
          key = "lovr.graphics.newBuffer",
          module = "lovr.graphics",
          notes = "The format table can contain a list of `FieldType`s or a list of tables to provide extra information about each field.  Each inner table has the following keys:\n\n- `type` is the `FieldType` of the field and is required.\n- `offset` is the byte offset of the field.  Any fields with a `nil` offset will be placed next\n  to each other sequentially in memory, subject to any padding required by the Buffer's layout.\n  In practice this means that you probably want to provide an `offset` for either all of the\n  fields or none of them.\n- `location` is the vertex attribute location of each field.  This is used to match up each\n  field with an attribute declared in a shader, and doesn't have any purpose when binding the\n  buffer as a uniform or storage buffer.  Any fields with a `nil` location will use an\n  autoincrementing location starting at zero.  Named locations are not currently supported, but\n  may be added in the future.\n\nIf no table or Blob is used to define the initial Buffer contents, its data will be undefined.\n\nThere is currently a max of 16 fields.",
          related = {
            "lovr.graphics.getBuffer"
          },
          variants = {
            {
              arguments = {
                {
                  name = "length",
                  type = "number",
                  description = "The length of the Buffer."
                },
                {
                  name = "type",
                  type = "FieldType",
                  description = "The type of each item in the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "data",
                  type = "table",
                  description = "The initial data to put into the Buffer.  The length of the Buffer will be determined by the contents of the table.  The contents can be a mix of tables, numbers, and vectors, but the length calculation requires each field to consistently use one type of data."
                },
                {
                  name = "type",
                  type = "FieldType",
                  description = "The type of each item in the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "length",
                  type = "number",
                  description = "The length of the Buffer."
                },
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer (see notes).  `nil` is a valid format, but means only `Blob`s can be written to the Buffer from Lua.",
                  default = "nil",
                  table = {
                    {
                      name = "layout",
                      type = "BufferLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "data",
                  type = "table",
                  description = "The initial data to put into the Buffer.  The length of the Buffer will be determined by the contents of the table.  The contents can be a mix of tables, numbers, and vectors, but the length calculation requires each field to consistently use one type of data."
                },
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer (see notes).  `nil` is a valid format, but means only `Blob`s can be written to the Buffer from Lua.",
                  default = "nil",
                  table = {
                    {
                      name = "layout",
                      type = "BufferLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob with the initial contents of the Buffer.  The size of the Blob will be used to determine the length of the Buffer."
                },
                {
                  name = "type",
                  type = "FieldType",
                  description = "The type of each item in the Buffer."
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob with the initial contents of the Buffer.  The size of the Blob will be used to determine the length of the Buffer."
                },
                {
                  name = "format",
                  type = "table",
                  description = "A list of fields in the Buffer (see notes).  `nil` is a valid format, but means only `Blob`s can be written to the Buffer from Lua.",
                  default = "nil",
                  table = {
                    {
                      name = "layout",
                      type = "BufferLayout",
                      description = "How to lay out the Buffer fields in memory.",
                      default = "packed"
                    },
                    {
                      name = "stride",
                      type = "number",
                      description = "The stride of the Buffer, in bytes.  When `nil`, the stride will be automatically computed based on the fields.  The stride can not be zero or smaller than the max byte occupied by one of the fields.  The layout of the Buffer may adjust the stride."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "buffer",
                  type = "Buffer",
                  description = "The new Buffer."
                }
              }
            }
          }
        },
        {
          name = "newFont",
          tag = "graphics-objects",
          summary = "Create a new Font.",
          description = "Creates a new Font.",
          key = "lovr.graphics.newFont",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.getDefaultFont",
            "lovr.data.newRasterizer",
            "Pass:text"
          },
          variants = {
            {
              description = "Creates a new Font from a TTF file.",
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "A path to a TTF file."
                },
                {
                  name = "size",
                  type = "number",
                  description = "The size of the Font in pixels.  Larger sizes are slower to initialize and use more memory, but have better quality.",
                  default = "32"
                },
                {
                  name = "spread",
                  type = "number",
                  description = "For signed distance field fonts (currently all fonts), the width of the SDF, in pixels.  The greater the distance the font is viewed from, the larger this value needs to be for the font to remain properly antialiased.  Increasing this will have a performance penalty similar to increasing the size of the font.",
                  default = "4"
                }
              },
              returns = {
                {
                  name = "font",
                  type = "Font",
                  description = "The new Font."
                }
              }
            },
            {
              description = "Creates a new Font from TTF data.",
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob containing TTF file data."
                },
                {
                  name = "size",
                  type = "number",
                  description = "The size of the Font in pixels.  Larger sizes are slower to initialize and use more memory, but have better quality.",
                  default = "32"
                },
                {
                  name = "spread",
                  type = "number",
                  description = "For signed distance field fonts (currently all fonts), the width of the SDF, in pixels.  The greater the distance the font is viewed from, the larger this value needs to be for the font to remain properly antialiased.  Increasing this will have a performance penalty similar to increasing the size of the font.",
                  default = "4"
                }
              },
              returns = {
                {
                  name = "font",
                  type = "Font",
                  description = "The new Font."
                }
              }
            },
            {
              description = "Creates a new Font using the default typeface (Varela Round).",
              arguments = {
                {
                  name = "size",
                  type = "number",
                  description = "The size of the Font in pixels.  Larger sizes are slower to initialize and use more memory, but have better quality.",
                  default = "32"
                },
                {
                  name = "spread",
                  type = "number",
                  description = "For signed distance field fonts (currently all fonts), the width of the SDF, in pixels.  The greater the distance the font is viewed from, the larger this value needs to be for the font to remain properly antialiased.  Increasing this will have a performance penalty similar to increasing the size of the font.",
                  default = "4"
                }
              },
              returns = {
                {
                  name = "font",
                  type = "Font",
                  description = "The new Font."
                }
              }
            },
            {
              description = "Creates a new Font from an existing Rasterizer.",
              arguments = {
                {
                  name = "rasterizer",
                  type = "Rasterizer",
                  description = "An existing Rasterizer to use to load glyph images."
                },
                {
                  name = "spread",
                  type = "number",
                  description = "For signed distance field fonts (currently all fonts), the width of the SDF, in pixels.  The greater the distance the font is viewed from, the larger this value needs to be for the font to remain properly antialiased.  Increasing this will have a performance penalty similar to increasing the size of the font.",
                  default = "4"
                }
              },
              returns = {
                {
                  name = "font",
                  type = "Font",
                  description = "The new Font."
                }
              }
            }
          }
        },
        {
          name = "newMaterial",
          tag = "graphics-objects",
          summary = "Create a new Material.",
          description = "Creates a new Material from a table of properties and textures.  All fields are optional.  Once a Material is created, its properties can not be changed.  Instead, a new Material should be created with the updated properties.",
          key = "lovr.graphics.newMaterial",
          module = "lovr.graphics",
          notes = "The non-texture material properties can be accessed in shaders using `Material.<property>`, where the property is the same as the Lua table key.  The textures use capitalized names in shader code, e.g. `ColorTexture`.",
          variants = {
            {
              arguments = {
                {
                  name = "properties",
                  type = "table",
                  description = "Material properties.",
                  table = {
                    {
                      name = "color",
                      type = "Vec4",
                      description = "The base color of the surface.  Can be a `Vec3`, `Vec4`, table of numbers, or hexcode. Can be toggled in shaders using the `materialColor` flag, which defaults to `true`.",
                      default = "{ 1, 1, 1, 1 }"
                    },
                    {
                      name = "glow",
                      type = "Vec4",
                      description = "The glow color of the surface, sometimes called \"emissive\".  The glow is not affected by lighting, so it's a good fit for e.g. headlights on a car or LED lights on a panel.  The alpha of the glow color is used as the glow strength.  Can be a `Vec3`, `Vec4`, table of numbers, or hexcode.  Can be toggled in shaders using the `glow` flag, which defaults to `false`.",
                      default = "{ 0, 0, 0, 0 }"
                    },
                    {
                      name = "uvShift",
                      type = "Vec2",
                      description = "An offset to apply to the UV coordinates used to sample textures.  The offset is not affected by `uvScale`.  This can be used to map UV coordinates to a sub-rectangle of a texture atlas.  Can be a `Vec2`, table of numbers, or a single number which gets assigned to both axes.  Can be toggled in shaders using the `uvTransform` flag, which defaults to `true`.",
                      default = "{ 0, 0 }"
                    },
                    {
                      name = "uvScale",
                      type = "Vec2",
                      description = "A scale factor to apply to the UV coordinates used to sample textures.  The scale is not affected by `uvOffset`.  This can be used to map UV coordinates to a sub-rectangle of a texture atlas, or repeat a texture multiple times across a surface.  Can be a `Vec2`, table of numbers, or a single number which gets assigned to both axes. Can be toggled in shaders using the `uvTransform` flag, which defaults to `true`.",
                      default = "{ 1, 1 }"
                    },
                    {
                      name = "metalness",
                      type = "number",
                      description = "The metalness the surface, used for physically-based rendering.  1.0 means the surface is metallic (conductor), and 0.0 means the surface is non-metallic (dielectric).  Values in between are seldom used and are only used in textures to transition between a metallic and non-metallic surface.  Metals reflect light differently than non-metals. Used by the lighting helper functions `initSurface` and `getLighting`.",
                      default = "0"
                    },
                    {
                      name = "roughness",
                      type = "number",
                      description = "The roughness of the surface, used for physically-based rendering.  1.0 means the surface is rough (blurry reflections), and 0.0 means the surface is smooth (sharp reflections).  Used by the lighting helper functions `initSurface` and `getLighting`.",
                      default = "0"
                    },
                    {
                      name = "clearcoat",
                      type = "number",
                      description = "The clearcoat factor.  Not currently used by LÖVR.",
                      default = "0"
                    },
                    {
                      name = "clearcoatRoughness",
                      type = "number",
                      description = "The roughness of the clearcoat layer.  Not currently used by LÖVR.",
                      default = "0"
                    },
                    {
                      name = "occlusionStrength",
                      type = "number",
                      description = "The strength of the ambient occlusion effect.  Ambient occlusion only affects indirect lighting.  Used by the lighting helper functions `initSurface` and `getIndirectLighting`.  Can be toggled in shaders using the `ambientOcclusion` flag, which defaults to `true`.",
                      default = "1"
                    },
                    {
                      name = "normalScale",
                      type = "number",
                      description = "The strength of the normal map.  Used by the `initSurface` function to bend the surface normal.  Can be toggled in shaders using the `normalMap` flag, which defaults to `false`.",
                      default = "1"
                    },
                    {
                      name = "alphaCutoff",
                      type = "number",
                      description = "The alpha cutoff.  At the end of the fragment shader, if the alpha of the final color is below the alpha cutoff, then the pixel will be \"discarded\" which means that it won't write a depth value.  Often used for transparent textures, especially with the \"alpha to coverage\" state set by `Pass:setAlphaToCoverage`.  Can be toggled in shaders using the `alphaCutoff` flag, which defaults to `false`.",
                      default = "0"
                    },
                    {
                      name = "texture",
                      type = "Texture",
                      description = "The base color texture.  In shaders this gets multiplied with the `color` property to get the base color of the pixel.  Can be toggled in shaders using the `colorTexture` flag, which defaults to `true`."
                    },
                    {
                      name = "glowTexture",
                      type = "Texture",
                      description = "The glow color texture.  In shaders, samples from this texture get multiplied with the `glow` property to get the glow color of the pixel.  Can be toggled in shaders using the `glowTexture` flag, which defaults to `true` (also requires the `glow` flag to be enabled)."
                    },
                    {
                      name = "metalnessTexture",
                      type = "Texture",
                      description = "The metalness texture.  In shaders, samples from the blue channel of this texture get multiplied with the `metalness` property to get the metalness value of the pixel.  Can be toggled in shaders using the `metalnessTexture` flag, which defaults to `true`."
                    },
                    {
                      name = "roughnessTexture",
                      type = "Texture",
                      description = "The roughness texture.  In shaders, samples from the green channel of this texture get multiplied with the `roughness` property to get the roughness value of the pixel.  Can be toggled in shaders using the `roughnessTexture` flag, which defaults to `true`."
                    },
                    {
                      name = "clearcoatTexture",
                      type = "Texture",
                      description = "Not currently used by LÖVR."
                    },
                    {
                      name = "occlusionTexture",
                      type = "Texture",
                      description = "The ambient occlusion texture.  In shaders, samples from the red channel of this texture get multiplied with the `occlusionStrength` property to get the ambient occlusion value of the pixel. Used by the lighting helper functions `initSurface` and `getIndirectLighting`.  Can be toggled in shaders using the `ambientOcclusion` flag, which defaults to `true`."
                    },
                    {
                      name = "normalTexture",
                      type = "Texture",
                      description = "The normal map, used to apply details to a surface without adding mesh geometry.  The `normalScale` property can be used to control how strong the effect is.  Can be toggled in shaders using the `normalMap` flag, which defaults to `false`."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "material",
                  type = "Material",
                  description = "The new material."
                }
              }
            }
          }
        },
        {
          name = "newModel",
          tag = "graphics-objects",
          summary = "Create a new Model.",
          description = "Loads a 3D model from a file.  Currently, OBJ, glTF, and binary STL files are supported.",
          key = "lovr.graphics.newModel",
          module = "lovr.graphics",
          related = {
            "lovr.data.newModelData",
            "Pass:draw"
          },
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The path to model file."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Model options.",
                  table = {
                    {
                      name = "mipmaps",
                      type = "boolean",
                      description = "Whether the textures created for the Model should have mipmaps generated.",
                      default = "true"
                    }
                  }
                }
              },
              returns = {
                {
                  name = "model",
                  type = "Model",
                  description = "The new Model."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob containing 3D model data."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Model options.",
                  table = {
                    {
                      name = "mipmaps",
                      type = "boolean",
                      description = "Whether the textures created for the Model should have mipmaps generated.",
                      default = "true"
                    }
                  }
                }
              },
              returns = {
                {
                  name = "model",
                  type = "Model",
                  description = "The new Model."
                }
              }
            },
            {
              arguments = {
                {
                  name = "modelData",
                  type = "ModelData",
                  description = "An existing ModelData object to use for the Model."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Model options.",
                  table = {
                    {
                      name = "mipmaps",
                      type = "boolean",
                      description = "Whether the textures created for the Model should have mipmaps generated.",
                      default = "true"
                    }
                  }
                }
              },
              returns = {
                {
                  name = "model",
                  type = "Model",
                  description = "The new Model."
                }
              }
            }
          }
        },
        {
          name = "newSampler",
          tag = "graphics-objects",
          summary = "Create a new Sampler.",
          description = "Creates a new Sampler.  Samplers are immutable, meaning their parameters can not be changed after the sampler is created.  Instead, a new sampler should be created with the updated properties.",
          key = "lovr.graphics.newSampler",
          module = "lovr.graphics",
          related = {
            "Pass:setSampler"
          },
          variants = {
            {
              arguments = {
                {
                  name = "parameters",
                  type = "table",
                  description = "Parameters for the sampler.",
                  table = {
                    {
                      name = "filter",
                      type = "table",
                      description = "How the sampler smooths texture pixels.  Can be a table of 3 FilterModes, or a single FilterMode to use for all three.",
                      default = "'linear'",
                      table = {
                        {
                          name = "[1]",
                          type = "FilterMode",
                          description = "The filter mode to use when minifying a texture (drawing it at a smaller size than its native pixel resolution)."
                        },
                        {
                          name = "[2]",
                          type = "FilterMode",
                          description = "The filter mode to use when magnifying a texture (drawing it at a larger size than its native pixel resolution)."
                        },
                        {
                          name = "[3]",
                          type = "FilterMode",
                          description = "The filter mode used to smooth between mipmap levels in a texture."
                        }
                      }
                    },
                    {
                      name = "wrap",
                      type = "table",
                      description = "How the sampler behaves when wrapping UVs outside the 0-1 range.  Can be a table of 3 WrapModes, or a single WrapMode to use for all three axes.",
                      default = "'repeat'",
                      table = {
                        {
                          name = "[1]",
                          type = "WrapMode",
                          description = "The horizontal wrap mode."
                        },
                        {
                          name = "[2]",
                          type = "WrapMode",
                          description = "The vertical wrap mode."
                        },
                        {
                          name = "[3]",
                          type = "FilterMode",
                          description = "The \"z\" wrap mode for 3D textures."
                        }
                      }
                    },
                    {
                      name = "compare",
                      type = "CompareMode",
                      description = "The compare mode of the sampler (for shadow samplers).",
                      default = "'none'"
                    },
                    {
                      name = "anisotropy",
                      type = "number",
                      description = "The maximum amount of anisotropic filtering to use.",
                      default = "1"
                    },
                    {
                      name = "mipmaprange",
                      type = "table",
                      description = "A table of 2 mipmap levels the sampler will clamp to."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "sampler",
                  type = "Sampler",
                  description = "The new sampler."
                }
              }
            }
          }
        },
        {
          name = "newShader",
          tag = "graphics-objects",
          summary = "Create a Shader.",
          description = "TODO",
          key = "lovr.graphics.newShader",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.compileShader"
          },
          variants = {
            {
              description = "Create a graphics shader.",
              arguments = {
                {
                  name = "vertex",
                  type = "ShaderSource",
                  description = "TODO"
                },
                {
                  name = "fragment",
                  type = "ShaderSource",
                  description = "TODO"
                },
                {
                  name = "options",
                  type = "table",
                  description = "Shader options.",
                  table = {
                    {
                      name = "flags",
                      type = "table",
                      description = "TODO"
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label to use for the shader in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "shader",
                  type = "Shader",
                  description = "The new shader."
                }
              }
            },
            {
              description = "Create a compute shader.",
              arguments = {
                {
                  name = "compute",
                  type = "ShaderSource",
                  description = "TODO"
                },
                {
                  name = "options",
                  type = "table",
                  description = "Shader options.",
                  table = {
                    {
                      name = "flags",
                      type = "table",
                      description = "TODO"
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label to use for the shader in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "shader",
                  type = "Shader",
                  description = "The new shader."
                }
              }
            }
          }
        },
        {
          name = "newTally",
          tag = "graphics-objects",
          summary = "Create a new Tally.",
          description = "TODO",
          key = "lovr.graphics.newTally",
          module = "lovr.graphics",
          related = {
            "Pass:tick",
            "Pass:tock"
          },
          variants = {
            {
              arguments = {
                {
                  name = "type",
                  type = "TallyType",
                  description = "The type of the Tally, which controls what \"thing\" it measures."
                },
                {
                  name = "count",
                  type = "number",
                  description = "The number of slots in the Tally.  Each slot performs one measurement."
                },
                {
                  name = "views",
                  type = "number",
                  description = "Tally objects with the `time` type can only be used in render passes with a certain number of views.  This is ignored for other types of tallies.",
                  default = "2"
                }
              },
              returns = {
                {
                  name = "tally",
                  type = "Tally",
                  description = "The new Tally."
                }
              }
            }
          }
        },
        {
          name = "newTexture",
          tag = "graphics-objects",
          summary = "Create a new Texture.",
          description = "Creates a new Texture.  Image filenames or `Image` objects can be used to provide the initial pixel data and the dimensions, format, and type.  Alternatively, dimensions can be provided, which will create an empty texture.",
          key = "lovr.graphics.newTexture",
          module = "lovr.graphics",
          notes = "If no `type` is provided in the options table, LÖVR will guess the `TextureType` of the Texture based on the number of layers:\n\n- If there's 1 layer, the type will be `2d`.\n- If there are 6 layers, the type will be `cube`.\n- Otherwise, the type will be `array`.\n\nNote that an Image can contain multiple layers and mipmaps.  When a single Image is provided, its layer count will be used as the Texture's layer count.\n\nIf multiple Images are used to initialize the Texture, they must all have a single layer, and their dimensions, format, and mipmap counts must match.\n\nWhen providing cubemap images in a table, they can be in one of the following forms:\n\n    { 'px.png', 'nx.png', 'py.png', 'ny.png', 'pz.png', 'nz.png' }\n    { right = 'px.png', left = 'nx.png', top = 'py.png', bottom = 'ny.png', back = 'pz.png', front = 'nz.png' }\n    { px = 'px.png', nx = 'nx.png', py = 'py.png', ny = 'ny.png', pz = 'pz.png', nz = 'nz.png' }\n\n(Where 'p' stands for positive and 'n' stands for negative).\n\nIf no `usage` is provided in the options table, LÖVR will guess the `TextureUsage` of the Texture.  The `sample` usage is always included, but if the texture was created without any images then the texture will have the `render` usage as well.\n\nThe supported image formats are png, jpg, hdr, dds, ktx1, ktx2, and astc.\n\nIf image data is provided, mipmaps will be generated for any missing mipmap levels.",
          related = {
            "Texture:newView"
          },
          variants = {
            {
              arguments = {
                {
                  name = "filename",
                  type = "string",
                  description = "The filename of an image to load."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Texture options.",
                  table = {
                    {
                      name = "type",
                      type = "TextureType",
                      description = "The type of the texture."
                    },
                    {
                      name = "format",
                      type = "TextureFormat",
                      description = "The format of the texture (ignored when images are provided).",
                      default = "'rgba8'"
                    },
                    {
                      name = "linear",
                      type = "boolean",
                      description = "Whether the texture is in linear color space instead of sRGB.  Linear textures should be used for non-color data, like normal maps.",
                      default = "false"
                    },
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of samples in the texture, used for multisample antialiasing.  Currently must be 1 or 4.  Ignored when images are provided.",
                      default = "1"
                    },
                    {
                      name = "mipmaps",
                      type = "*",
                      description = "The number of mipmap levels in the texture, or a boolean.  If true, a full mipmap chain will be created.  If false, the texture will only have a single mipmap.",
                      default = "true"
                    },
                    {
                      name = "usage",
                      type = "table",
                      description = "A list of `TextureUsage` indicating how the texture will be used."
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label for the Texture that will show up in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "texture",
                  type = "Texture",
                  description = "The new Texture."
                }
              }
            },
            {
              arguments = {
                {
                  name = "width",
                  type = "number",
                  description = "The width of the Texture, in pixels."
                },
                {
                  name = "height",
                  type = "number",
                  description = "The height of the Texture, in pixels."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Texture options.",
                  table = {
                    {
                      name = "type",
                      type = "TextureType",
                      description = "The type of the texture."
                    },
                    {
                      name = "format",
                      type = "TextureFormat",
                      description = "The format of the texture (ignored when images are provided).",
                      default = "'rgba8'"
                    },
                    {
                      name = "linear",
                      type = "boolean",
                      description = "Whether the texture is in linear color space instead of sRGB.  Linear textures should be used for non-color data, like normal maps.",
                      default = "false"
                    },
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of samples in the texture, used for multisample antialiasing.  Currently must be 1 or 4.  Ignored when images are provided.",
                      default = "1"
                    },
                    {
                      name = "mipmaps",
                      type = "*",
                      description = "The number of mipmap levels in the texture, or a boolean.  If true, a full mipmap chain will be created.  If false, the texture will only have a single mipmap.",
                      default = "true"
                    },
                    {
                      name = "usage",
                      type = "table",
                      description = "A list of `TextureUsage` indicating how the texture will be used."
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label for the Texture that will show up in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "texture",
                  type = "Texture",
                  description = "The new Texture."
                }
              }
            },
            {
              arguments = {
                {
                  name = "width",
                  type = "number",
                  description = "The width of the Texture, in pixels."
                },
                {
                  name = "height",
                  type = "number",
                  description = "The height of the Texture, in pixels."
                },
                {
                  name = "layers",
                  type = "number",
                  description = "The number of layers in the Texture."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Texture options.",
                  table = {
                    {
                      name = "type",
                      type = "TextureType",
                      description = "The type of the texture."
                    },
                    {
                      name = "format",
                      type = "TextureFormat",
                      description = "The format of the texture (ignored when images are provided).",
                      default = "'rgba8'"
                    },
                    {
                      name = "linear",
                      type = "boolean",
                      description = "Whether the texture is in linear color space instead of sRGB.  Linear textures should be used for non-color data, like normal maps.",
                      default = "false"
                    },
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of samples in the texture, used for multisample antialiasing.  Currently must be 1 or 4.  Ignored when images are provided.",
                      default = "1"
                    },
                    {
                      name = "mipmaps",
                      type = "*",
                      description = "The number of mipmap levels in the texture, or a boolean.  If true, a full mipmap chain will be created.  If false, the texture will only have a single mipmap.",
                      default = "true"
                    },
                    {
                      name = "usage",
                      type = "table",
                      description = "A list of `TextureUsage` indicating how the texture will be used."
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label for the Texture that will show up in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "texture",
                  type = "Texture",
                  description = "The new Texture."
                }
              }
            },
            {
              arguments = {
                {
                  name = "image",
                  type = "string",
                  description = "An Image object holding pixel data to load into the Texture."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Texture options.",
                  table = {
                    {
                      name = "type",
                      type = "TextureType",
                      description = "The type of the texture."
                    },
                    {
                      name = "format",
                      type = "TextureFormat",
                      description = "The format of the texture (ignored when images are provided).",
                      default = "'rgba8'"
                    },
                    {
                      name = "linear",
                      type = "boolean",
                      description = "Whether the texture is in linear color space instead of sRGB.  Linear textures should be used for non-color data, like normal maps.",
                      default = "false"
                    },
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of samples in the texture, used for multisample antialiasing.  Currently must be 1 or 4.  Ignored when images are provided.",
                      default = "1"
                    },
                    {
                      name = "mipmaps",
                      type = "*",
                      description = "The number of mipmap levels in the texture, or a boolean.  If true, a full mipmap chain will be created.  If false, the texture will only have a single mipmap.",
                      default = "true"
                    },
                    {
                      name = "usage",
                      type = "table",
                      description = "A list of `TextureUsage` indicating how the texture will be used."
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label for the Texture that will show up in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "texture",
                  type = "Texture",
                  description = "The new Texture."
                }
              }
            },
            {
              arguments = {
                {
                  name = "images",
                  type = "table",
                  description = "A table of filenames or Images to load into the Texture."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Texture options.",
                  table = {
                    {
                      name = "type",
                      type = "TextureType",
                      description = "The type of the texture."
                    },
                    {
                      name = "format",
                      type = "TextureFormat",
                      description = "The format of the texture (ignored when images are provided).",
                      default = "'rgba8'"
                    },
                    {
                      name = "linear",
                      type = "boolean",
                      description = "Whether the texture is in linear color space instead of sRGB.  Linear textures should be used for non-color data, like normal maps.",
                      default = "false"
                    },
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of samples in the texture, used for multisample antialiasing.  Currently must be 1 or 4.  Ignored when images are provided.",
                      default = "1"
                    },
                    {
                      name = "mipmaps",
                      type = "*",
                      description = "The number of mipmap levels in the texture, or a boolean.  If true, a full mipmap chain will be created.  If false, the texture will only have a single mipmap.",
                      default = "true"
                    },
                    {
                      name = "usage",
                      type = "table",
                      description = "A list of `TextureUsage` indicating how the texture will be used."
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label for the Texture that will show up in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "texture",
                  type = "Texture",
                  description = "The new Texture."
                }
              }
            },
            {
              arguments = {
                {
                  name = "blob",
                  type = "Blob",
                  description = "A Blob object holding pixel data to load into the Texture."
                },
                {
                  name = "options",
                  type = "table",
                  description = "Texture options.",
                  table = {
                    {
                      name = "type",
                      type = "TextureType",
                      description = "The type of the texture."
                    },
                    {
                      name = "format",
                      type = "TextureFormat",
                      description = "The format of the texture (ignored when images are provided).",
                      default = "'rgba8'"
                    },
                    {
                      name = "linear",
                      type = "boolean",
                      description = "Whether the texture is in linear color space instead of sRGB.  Linear textures should be used for non-color data, like normal maps.",
                      default = "false"
                    },
                    {
                      name = "samples",
                      type = "number",
                      description = "The number of samples in the texture, used for multisample antialiasing.  Currently must be 1 or 4.  Ignored when images are provided.",
                      default = "1"
                    },
                    {
                      name = "mipmaps",
                      type = "*",
                      description = "The number of mipmap levels in the texture, or a boolean.  If true, a full mipmap chain will be created.  If false, the texture will only have a single mipmap.",
                      default = "true"
                    },
                    {
                      name = "usage",
                      type = "table",
                      description = "A list of `TextureUsage` indicating how the texture will be used."
                    },
                    {
                      name = "label",
                      type = "string",
                      description = "A label for the Texture that will show up in debugging tools."
                    }
                  }
                }
              },
              returns = {
                {
                  name = "texture",
                  type = "Texture",
                  description = "The new Texture."
                }
              }
            }
          }
        },
        {
          name = "present",
          tag = "work-submission",
          summary = "Update the desktop window contents.",
          description = "Presents the window texture to the desktop window.  This function is called automatically by the default implementation of `lovr.run`, so it normally does not need to be called.",
          key = "lovr.graphics.present",
          module = "lovr.graphics",
          notes = "This should be called after submitting the window pass (`lovr.graphics.getWindowPass`).  If the window texture has not been rendered to since the last present, this function does nothing.",
          related = {
            "lovr.graphics.submit",
            "lovr.graphics.getWindowPass"
          },
          variants = {
            {
              arguments = {},
              returns = {}
            }
          }
        },
        {
          name = "setBackgroundColor",
          tag = "graphics-global",
          summary = "Set the background color.",
          description = "Changes the global background color.  The textures in a render pass will be cleared to this color at the beginning of the pass if no other clear option is specified.  Additionally, the headset and window will be cleared to this color before rendering.",
          key = "lovr.graphics.setBackgroundColor",
          module = "lovr.graphics",
          notes = "Setting the background color in `lovr.draw` will apply on the following frame, since the default pass is cleared before `lovr.draw` is called.\n\nInternally, this color is applied to the default pass objects when retrieving one of them using `lovr.headset.getPass` or `lovr.graphics.getPass`.  Both are called automatically by the default `lovr.run` implementation.\n\nUsing the background color to clear the display is expected to be more efficient than manually clearing after a render pass begins, especially on mobile GPUs.",
          related = {
            "lovr.graphics.getPass",
            "Pass:clear",
            "Pass:fill"
          },
          variants = {
            {
              arguments = {
                {
                  name = "r",
                  type = "number",
                  description = "The red component of the background color."
                },
                {
                  name = "g",
                  type = "number",
                  description = "The green component of the background color."
                },
                {
                  name = "b",
                  type = "number",
                  description = "The blue component of the background color."
                },
                {
                  name = "a",
                  type = "number",
                  description = "The alpha component of the background color.",
                  default = "1.0"
                }
              },
              returns = {}
            },
            {
              arguments = {
                {
                  name = "hex",
                  type = "number",
                  description = "A hexcode (like `0xffffff`) to use for the background color (does not support alpha)."
                },
                {
                  name = "a",
                  type = "number",
                  description = "The alpha component of the background color.",
                  default = "1.0"
                }
              },
              returns = {}
            },
            {
              arguments = {
                {
                  name = "table",
                  type = "table",
                  description = "A table containing 3 or 4 color components."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "submit",
          tag = "work-submission",
          summary = "Submit recorded graphics work to the GPU.",
          description = "TODO",
          key = "lovr.graphics.submit",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.wait"
          },
          variants = {
            {
              arguments = {
                {
                  name = "...",
                  type = "Pass",
                  description = "The pass objects to submit.  Falsy values will be skipped."
                }
              },
              returns = {
                {
                  name = "true",
                  type = "boolean",
                  description = "Always returns true, for convenience when returning from `lovr.draw`."
                }
              }
            },
            {
              arguments = {
                {
                  name = "t",
                  type = "table",
                  description = "A table of passes to submit.  Falsy values will be skipped."
                }
              },
              returns = {
                {
                  name = "true",
                  type = "boolean",
                  description = "Always returns true, for convenience when returning from `lovr.draw`."
                }
              }
            }
          }
        },
        {
          name = "wait",
          tag = "work-submission",
          summary = "Stall the CPU until all submitted GPU work is finished.",
          description = "Waits for all submitted GPU work to finish.  A normal application that is trying to render graphics at a high framerate should never use this function, since waiting like this prevents the CPU from doing other useful work.  Otherwise, reasons to use this function might be for debugging or to force a `Readback` to finish immediately.",
          key = "lovr.graphics.wait",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.submit"
          },
          variants = {
            {
              arguments = {},
              returns = {}
            }
          }
        }
      },
      enums = {
        {
          name = "BufferLayout",
          description = "The different ways to pack Buffer fields into memory.\n\nThe default is `packed`, which is suitable for vertex buffers and index buffers.  It doesn't add any padding between elements, and so it doesn't waste any space.  However, this layout won't necessarily work for uniform buffers and storage buffers.\n\nThe `std140` layout corresponds to the std140 layout used for uniform buffers in GLSL.  It adds the most padding between fields, and requires the stride to be a multiple of 16.  Example:\n\n``` layout(std140) uniform ObjectScales { float scales[64]; }; ```\n\nThe `std430` layout corresponds to the std430 layout used for storage buffers in GLSL.  It adds some padding between certain types, and may round up the stride.  Example:\n\n``` layout(std430) buffer TileSizes { vec2 sizes[]; } ```",
          key = "BufferLayout",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.newBuffer",
            "lovr.graphics.getBuffer",
            "Buffer:getFormat",
            "Buffer:getStride",
            "FieldType"
          },
          values = {
            {
              name = "packed",
              description = "The packed layout, without any padding."
            },
            {
              name = "std140",
              description = "The std140 layout."
            },
            {
              name = "std430",
              description = "The std430 layout."
            }
          }
        },
        {
          name = "FieldType",
          description = "Different types for `Buffer` fields.  These are scalar, vector, or matrix types, usually packed into small amounts of space to reduce the amount of memory they occupy.\n\nThe names are encoded as follows:\n\n- The data type:\n  - `i` for signed integer\n  - `u` for unsigned integer\n  - `sn` for signed normalized (-1 to 1)\n  - `un` for unsigned normalized (0 to 1)\n  - `f` for floating point\n- The bit depth of each component\n- The letter `x` followed by the component count (for vectors)",
          key = "FieldType",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.newBuffer",
            "lovr.graphics.getBuffer",
            "Buffer:getFormat"
          },
          notes = "In addition to these values, the following aliases can be used:\n\n<table>\n  <thead>\n    <tr>\n      <td>Alias</td>\n      <td>Maps to</td>\n    </tr>\n  </thead>\n  <tbody>\n    <tr>\n      <td><code>vec2</code></td>\n      <td><code>f32x2</code></td>\n    </tr>\n    <tr>\n      <td><code>vec3</code></td>\n      <td><code>f32x3</code></td>\n    </tr>\n    <tr>\n      <td><code>vec4</code></td>\n      <td><code>f32x4</code></td>\n    </tr>\n    <tr>\n      <td><code>int</code></td>\n      <td><code>i32</code></td>\n    </tr>\n    <tr>\n      <td><code>uint</code></td>\n      <td><code>u32</code></td>\n    </tr>\n    <tr>\n      <td><code>float</code></td>\n      <td><code>f32</code></td>\n    </tr>\n    <tr>\n      <td><code>color</code></td>\n      <td><code>un8x4</code></td>\n    </tr>\n  </tbody> </table>\n\nAdditionally, the following convenience rules apply:\n\n- Field types can end in an `s`, which will be stripped off.\n- Field types can end in `x1`, which will be stripped off.\n\nSo you can write, e.g. `lovr.graphics.newBuffer(4, 'floats')`, which is cute!",
          values = {
            {
              name = "i8x4",
              description = "Four 8-bit signed integers."
            },
            {
              name = "u8x4",
              description = "Four 8-bit unsigned integers."
            },
            {
              name = "sn8x4",
              description = "Four 8-bit signed normalized values."
            },
            {
              name = "un8x4",
              description = "Four 8-bit unsigned normalized values (aka `color`)."
            },
            {
              name = "un10x3",
              description = "Three 10-bit unsigned normalized values, and 2 padding bits (aka `normal`)."
            },
            {
              name = "i16",
              description = "One 16-bit signed integer."
            },
            {
              name = "i16x2",
              description = "Two 16-bit signed integers."
            },
            {
              name = "i16x4",
              description = "Four 16-bit signed integers."
            },
            {
              name = "u16",
              description = "One 16-bit unsigned integer."
            },
            {
              name = "u16x2",
              description = "Two 16-bit unsigned integers."
            },
            {
              name = "u16x4",
              description = "Four 16-bit unsigned integers."
            },
            {
              name = "sn16x2",
              description = "Two 16-bit signed normalized values."
            },
            {
              name = "sn16x4",
              description = "Four 16-bit signed normalized values."
            },
            {
              name = "un16x2",
              description = "Two 16-bit unsigned normalized values."
            },
            {
              name = "un16x4",
              description = "Four 16-bit unsigned normalized values."
            },
            {
              name = "i32",
              description = "One 32-bit signed integer (aka `int`)."
            },
            {
              name = "i32x2",
              description = "Two 32-bit signed integers."
            },
            {
              name = "i32x2",
              description = "Two 32-bit signed integers."
            },
            {
              name = "i32x3",
              description = "Three 32-bit signed integers."
            },
            {
              name = "i32x4",
              description = "Four 32-bit signed integers."
            },
            {
              name = "u32",
              description = "One 32-bit unsigned integer (aka `uint`)."
            },
            {
              name = "u32x2",
              description = "Two 32-bit unsigned integers."
            },
            {
              name = "u32x3",
              description = "Three 32-bit unsigned integers."
            },
            {
              name = "u32x4",
              description = "Four 32-bit unsigned integers."
            },
            {
              name = "f16x2",
              description = "Two 16-bit floating point numbers."
            },
            {
              name = "f16x4",
              description = "Four 16-bit floating point numbers."
            },
            {
              name = "f32",
              description = "One 32-bit floating point number (aka `float`)."
            },
            {
              name = "f32x2",
              description = "Two 32-bit floating point numbers (aka `vec2`)."
            },
            {
              name = "f32x3",
              description = "Three 32-bit floating point numbers (aka `vec3`)."
            },
            {
              name = "f32x4",
              description = "Four 32-bit floating point numbers (aka `vec4`)."
            },
            {
              name = "mat2",
              description = "A 2x2 matrix containing four 32-bit floats."
            },
            {
              name = "mat3",
              description = "A 3x3 matrix containing nine 32-bit floats."
            },
            {
              name = "mat4",
              description = "A 4x4 matrix containing sixteen 32-bit floats."
            }
          }
        },
        {
          name = "FilterMode",
          summary = "Different ways to smooth textures.",
          description = "Controls how `Sampler` objects smooth pixels in textures.",
          key = "FilterMode",
          module = "lovr.graphics",
          related = {
            "Pass:blit"
          },
          values = {
            {
              {
                name = "nearest",
                description = "A pixelated appearance where the \"nearest neighbor\" pixel is used."
              },
              {
                name = "linear",
                description = "A smooth appearance where neighboring pixels are averaged."
              }
            }
          }
        },
        {
          name = "MeshMode",
          summary = "Different ways to draw mesh vertices.",
          description = "Different ways vertices in a mesh can be connected together and filled in with pixels.",
          key = "MeshMode",
          module = "lovr.graphics",
          values = {
            {
              name = "points",
              description = "Each vertex is rendered as a single point.  The size of the point can be controlled using the `pointSize` shader flag, or by writing to the `PointSize` variable in shaders.  The maximum point size is given by the `pointSize` limit from `lovr.graphics.getLimits`."
            },
            {
              name = "lines",
              description = "Pairs of vertices are connected with line segments.  To draw a single line through all of the vertices, an index buffer can be used to repeat vertices.  It is not currently possible to change the width of the lines, although cylinders or capsules can be used as an alternative."
            },
            {
              name = "triangles",
              description = "Every 3 vertices form a triangle, which is filled in with pixels (unless `Pass:setWireframe` is used).  This mode is the most commonly used."
            }
          }
        },
        {
          name = "PassType",
          summary = "Different types of Passes.",
          description = "The three different types of `Pass` objects.  Each Pass has a single type, which determines the type of work it does and which functions can be called on it.",
          key = "PassType",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.getPass",
            "lovr.graphics.submit",
            "Pass:getType"
          },
          values = {
            {
              name = "render",
              description = "A render pass renders graphics to a set of up to four color textures and an optional depth texture.  The textures all need to have the same dimensions and sample counts.  The textures can have multiple layers, and all rendering work will be broadcast to each layer.  Each layer can use a different camera pose, which is used for stereo rendering."
            },
            {
              name = "compute",
              description = "A compute pass runs compute shaders.  Compute passes usually only call `Pass:setShader`, `Pass:send`, and `Pass:compute`.  All of the compute work in a single compute pass is run in parallel, so multiple compute passes should be used if one compute pass needs to happen after a different one."
            },
            {
              name = "transfer",
              description = "A transfer pass copies data to and from GPU memory in `Buffer` and `Texture` objects. Transfer passes use `Pass:copy`, `Pass:clear`, `Pass:blit`, `Pass:mipmap`, and `Pass:read`. Similar to compute passes, all the work in a transfer pass happens in parallel, so multiple passes should be used if the transfers need to be ordered."
            }
          }
        },
        {
          name = "ShaderStage",
          description = "TODO",
          key = "ShaderStage",
          module = "lovr.graphics",
          values = {
            {
              name = "vertex",
              description = "TODO"
            },
            {
              name = "fragment",
              description = "TODO"
            },
            {
              name = "compute",
              description = "TODO"
            }
          }
        },
        {
          name = "ShaderType",
          summary = "Different types of Shaders.",
          description = "The two types of shaders that can be created.",
          key = "ShaderType",
          module = "lovr.graphics",
          related = {
            "lovr.graphics.newShader",
            "Shader:getType",
            "ShaderStage"
          },
          values = {
            {
              name = "graphics",
              description = "A graphics shader with a vertex and pixel stage."
            },
            {
              name = "compute",
              description = "A compute shader with a single compute stage."
            }
          }
        },
        {
          name = "StackType",
          summary = "Types of stacks that can be pushed and popped.",
          description = "TODO",
          key = "StackType",
          module = "lovr.graphics",
          values = {
            {
              name = "transform",
              description = "TODO"
            },
            {
              name = "state",
              descriptioin = "TODO"
            }
          }
        },
        {
          name = "TallyType",
          summary = "Different values a Tally can measure.",
          description = "TODO",
          key = "TallyType",
          module = "lovr.graphics",
          values = {
            {
              name = "time",
              description = "Each slot measures elapsed time in nanoseconds."
            },
            {
              name = "pixel",
              description = "Each slot measures the approximate number of pixels affected by rendering."
            },
            {
              name = "shader",
              description = "Each slot measures the number of times different shader stages are invoked."
            }
          }
        },
        {
          name = "TextureFeature",
          summary = "Different ways Textures can be used.",
          description = "These are the different ways `Texture` objects can be used.  These are passed in to `lovr.graphics.isFormatSupported` to see which texture operations are supported by the GPU for a given format.",
          key = "TextureFeature",
          module = "lovr.graphics",
          values = {
            {
              name = "sample",
              description = "The Texture can be sampled (e.g. a `texture2D` or `sampler2D` variable in shaders)."
            },
            {
              name = "filter",
              description = "The Texture can be used with a `Sampler` using a `FilterMode` of `linear`."
            },
            {
              name = "render",
              description = "The Texture can be rendered to by using it as a target in a render `Pass`."
            },
            {
              name = "blend",
              description = "Blending can be enabled when rendering to this format in a render pass."
            },
            {
              name = "storage",
              description = "The Texture can be sent to an image variable in shaders (e.g. `image2D`)."
            },
            {
              name = "atomic",
              description = "Atomic operations can be used on storage textures with this format."
            },
            {
              name = "blitsrc",
              description = "Source textures in `Pass:blit` can use this format."
            },
            {
              name = "blitdst",
              description = "Destination textures in `Pass:blit` can use this format."
            }
          }
        },
        {
          name = "TextureType",
          description = "Different types of textures.  Textures are multidimensional blocks of GPU memory, and the texture's type determines how many dimensions there are, and adds some semantics about what the 3rd dimension means.",
          key = "TextureType",
          module = "lovr.graphics",
          values = {
            {
              name = "2d",
              description = "A single 2D image, the most common type."
            },
            {
              name = "3d",
              description = "A 3D image, where a sequence of 2D images defines a 3D volume.  Each mipmap level of a 3D texture gets smaller in the x, y, and z axes, unlike cubemap and array textures."
            },
            {
              name = "cube",
              description = "Six square 2D images with the same dimensions that define the faces of a cubemap, used for skyboxes or other \"directional\" images."
            },
            {
              name = "array",
              description = "Array textures are sequences of distinct 2D images that all have the same dimensions."
            }
          }
        },
        {
          name = "TextureUsage",
          description = "These are the different things `Texture`s can be used for.  When creating a Texture, a set of these flags can be provided, restricting what operations are allowed on the texture.  Using a smaller set of flags may improve performance.  If none are provided, the only usage flag applied is `sample`.",
          key = "TextureUsage",
          module = "lovr.graphics",
          values = {
            {
              name = "sample",
              description = "Whether the texture can be sampled from in Shaders (i.e. used in a material, or bound to a variable with a `texture` type, like `texture2D`)."
            },
            {
              name = "render",
              description = "Whether the texture can be rendered to (i.e. by using it as a render target in `lovr.graphics.pass`)."
            },
            {
              name = "storage",
              description = "Whether the texture can be used as a storage texture for compute operations (i.e. bound to a variable with an `image` type, like `image2D`)."
            },
            {
              name = "transfer",
              description = "Whether the texture can be used in a transfer pass."
            }
          }
        },
        {
          name = "WrapMode",
          summary = "Different ways to wrap textures.",
          description = "Controls how `Sampler` objects wrap textures.",
          key = "WrapMode",
          module = "lovr.graphics",
          values = {
            {
              {
                name = "clamp",
                description = "          Pixels will be clamped to the edge, with pixels outside the 0-1 uv range using colors from\n          the nearest edge.\n        "
              },
              {
                name = "repeat",
                description = "Tiles the texture."
              }
            }
          }
        }
      },
      sections = {
        {
          name = "Objects",
          tag = "graphics-objects"
        },
        {
          name = "Global State",
          tag = "graphics-global"
        },
        {
          name = "Work Submission",
          tag = "work-submission",
          description = "The only way to get the GPU to do anything is to submit `Pass` objects to it.  LÖVR submits the default pass automatically at the end of `lovr.draw`, but work can also be submitted manually."
        },
        {
          name = "System Info",
          tag = "graphics-misc",
          description = "Information about the GPU hardware and the features it supports."
        }
      }
    },
    {
      name = "headset",
      tag = "modules",
      summary = "Connects to VR hardware.",
      description = "The `lovr.headset` module is where all the magical VR functionality is.  With it, you can access connected VR hardware and get information about the available space the player has.  Note that all units are reported in meters.  Position `(0, 0, 0)` is on the floor in the center of the play area.",
      key = "lovr.headset",
      objects = {},
      functions = {
        {
          name = "animate",
          tag = "input",
          summary = "Animate a model to match its Device input state.",
          description = "Animates a device model to match its current input state.  The buttons and joysticks on a controller will move as they're pressed/moved and hand models will move to match skeletal input.\n\nThe model should have been created using `lovr.headset.newModel` with the `animated` flag set to `true`.",
          key = "lovr.headset.animate",
          module = "lovr.headset",
          notes = "Currently this function is only supported for hand models on the Oculus Quest.\n\nIt's possible to use models that weren't created with `lovr.headset.newModel` but they need to be set up carefully to have the same structure as the models provided by the headset SDK.",
          related = {
            "lovr.headset.newModel"
          },
          variants = {
            {
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The device to use for the animation data.",
                  default = "'head'"
                },
                {
                  name = "model",
                  type = "Model",
                  description = "The model to animate."
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether the animation was applied successfully to the Model.  If the Model was not compatible or animation data for the device was not available, this will be `false`."
                }
              }
            }
          }
        },
        {
          name = "getAngularVelocity",
          tag = "input",
          summary = "Get the angular velocity of a device.",
          description = "Returns the current angular velocity of a device.",
          key = "lovr.headset.getAngularVelocity",
          module = "lovr.headset",
          related = {
            "lovr.headset.getVelocity",
            "lovr.headset.getPosition",
            "lovr.headset.getOrientation"
          },
          variants = {
            {
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The device to get the velocity of.",
                  default = "'head'"
                }
              },
              returns = {
                {
                  name = "x",
                  type = "number",
                  description = "The x component of the angular velocity."
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y component of the angular velocity."
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z component of the angular velocity."
                }
              }
            }
          }
        },
        {
          name = "getAxis",
          tag = "input",
          summary = "Get the state of an analog axis on a device.",
          description = "Get the current state of an analog axis on a device.  Some axes are multidimensional, for example a 2D touchpad or thumbstick with x/y axes.  For multidimensional axes, this function will return multiple values, one number for each axis.  In these cases, it can be useful to use the `select` function built in to Lua to select a particular axis component.",
          key = "lovr.headset.getAxis",
          module = "lovr.headset",
          notes = "The axis values will be between 0 and 1 for 1D axes, and between -1 and 1 for each component of a multidimensional axis.\n\nWhen hand tracking is active, pinch strength will be mapped to the `trigger` axis.",
          related = {
            "DeviceAxis",
            "lovr.headset.isDown"
          },
          variants = {
            {
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The device."
                },
                {
                  name = "axis",
                  type = "DeviceAxis",
                  description = "The axis."
                }
              },
              returns = {
                {
                  name = "...",
                  type = "number",
                  description = "The current state of the components of the axis, or `nil` if the device does not have any information about the axis."
                }
              }
            }
          }
        },
        {
          name = "getBoundsDepth",
          tag = "playArea",
          summary = "Get the depth of the play area.",
          description = "Returns the depth of the play area, in meters.",
          key = "lovr.headset.getBoundsDepth",
          module = "lovr.headset",
          notes = "This currently returns 0 on the Quest.",
          related = {
            "lovr.headset.getBoundsWidth",
            "lovr.headset.getBoundsDimensions"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "depth",
                  type = "number",
                  description = "The depth of the play area, in meters."
                }
              }
            }
          }
        },
        {
          name = "getBoundsDimensions",
          tag = "playArea",
          summary = "Get the size of the play area.",
          description = "Returns the size of the play area, in meters.",
          key = "lovr.headset.getBoundsDimensions",
          module = "lovr.headset",
          notes = "This currently returns 0 on the Quest.",
          related = {
            "lovr.headset.getBoundsWidth",
            "lovr.headset.getBoundsDepth",
            "lovr.headset.getBoundsGeometry"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "width",
                  type = "number",
                  description = "The width of the play area, in meters."
                },
                {
                  name = "depth",
                  type = "number",
                  description = "The depth of the play area, in meters."
                }
              }
            }
          }
        },
        {
          name = "getBoundsGeometry",
          tag = "playArea",
          summary = "Get a list of points that make up the play area boundary.",
          description = "Returns a list of points representing the boundaries of the play area, or `nil` if the current headset driver does not expose this information.",
          key = "lovr.headset.getBoundsGeometry",
          module = "lovr.headset",
          related = {
            "lovr.headset.getBoundsDimensions"
          },
          variants = {
            {
              arguments = {
                {
                  name = "t",
                  type = "table",
                  description = "A table to fill with the points.  If `nil`, a new table will be created.",
                  default = "nil"
                }
              },
              returns = {
                {
                  name = "points",
                  type = "table",
                  description = "A flat table of 3D points representing the play area boundaries."
                }
              }
            }
          }
        },
        {
          name = "getBoundsWidth",
          tag = "playArea",
          summary = "Get the width of the play area.",
          description = "Returns the width of the play area, in meters.",
          key = "lovr.headset.getBoundsWidth",
          module = "lovr.headset",
          notes = "This currently returns 0 on the Quest.",
          related = {
            "lovr.headset.getBoundsDepth",
            "lovr.headset.getBoundsDimensions"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "width",
                  type = "number",
                  description = "The width of the play area, in meters."
                }
              }
            }
          }
        },
        {
          name = "getClipDistance",
          tag = "headset",
          summary = "Get the near and far clipping planes of the headset.",
          description = "Returns the near and far clipping planes used to render to the headset.  Objects closer than the near clipping plane or further than the far clipping plane will be clipped out of view.",
          key = "lovr.headset.getClipDistance",
          module = "lovr.headset",
          notes = "The default near and far clipping planes are 0.1 meters and 100.0 meters.\n\nThis is not currently supported by the `vrapi` headset driver.",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "near",
                  type = "number",
                  description = "The distance to the near clipping plane, in meters."
                },
                {
                  name = "far",
                  type = "number",
                  description = "The distance to the far clipping plane, in meters."
                }
              }
            }
          }
        },
        {
          name = "getDisplayDimensions",
          tag = "headset",
          summary = "Get the dimensions of the headset display.",
          description = "Returns the texture dimensions of the headset display (for one eye), in pixels.",
          key = "lovr.headset.getDisplayDimensions",
          module = "lovr.headset",
          related = {
            "lovr.headset.getDisplayWidth",
            "lovr.headset.getDisplayHeight"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "width",
                  type = "number",
                  description = "The width of the display."
                },
                {
                  name = "height",
                  type = "number",
                  description = "The height of the display."
                }
              }
            }
          }
        },
        {
          name = "getDisplayFrequencies",
          tag = "headset",
          summary = "Get the list of refresh rates supported by the headset display.",
          description = "Returns a table with all the refresh rates supported by the headset display, in Hz.",
          key = "lovr.headset.getDisplayFrequencies",
          module = "lovr.headset",
          related = {
            "lovr.headset.setDisplayFrequency"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "frequencies",
                  type = "table",
                  description = "A flat table of the refresh rates supported by the headset display, nil if not supported."
                }
              }
            }
          }
        },
        {
          name = "getDisplayFrequency",
          tag = "headset",
          summary = "Get the refresh rate of the headset display.",
          description = "Returns the refresh rate of the headset display, in Hz.",
          key = "lovr.headset.getDisplayFrequency",
          module = "lovr.headset",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "frequency",
                  type = "number",
                  description = "The frequency of the display, or `nil` if I have no idea what it is."
                }
              }
            }
          }
        },
        {
          name = "getDisplayHeight",
          tag = "headset",
          summary = "Get the height of the headset display.",
          description = "Returns the height of the headset display (for one eye), in pixels.",
          key = "lovr.headset.getDisplayHeight",
          module = "lovr.headset",
          related = {
            "lovr.headset.getDisplayWidth",
            "lovr.headset.getDisplayDimensions"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "height",
                  type = "number",
                  description = "The height of the display."
                }
              }
            }
          }
        },
        {
          name = "getDisplayMask",
          tag = "headset",
          summary = "Get a mesh that masks out the visible display area.",
          description = "Returns 2D triangle vertices that represent areas of the headset display that will never be seen by the user (due to the circular lenses).  This area can be masked out by rendering it to the depth buffer or stencil buffer.  Then, further drawing operations can skip rendering those pixels using the depth test (`lovr.graphics.setDepthTest`) or stencil test (`lovr.graphics.setStencilTest`), which improves performance.",
          key = "lovr.headset.getDisplayMask",
          module = "lovr.headset",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "points",
                  type = "table",
                  description = "A table of points.  Each point is a table with two numbers between 0 and 1."
                }
              }
            }
          },
          examples = {
            {
              code = "function lovr.load()\n  lovr.graphics.setBackgroundColor(1, 1, 1)\n\n  shader = lovr.graphics.newShader([[\n    vec4 position(mat4 projection, mat4 transform, vec4 vertex) {\n\n      // Rescale mesh coordinates from (0,1) to (-1,1)\n      vertex.xy *= 2.;\n      vertex.xy -= 1.;\n\n      // Flip the mesh if it's being drawn in the right eye\n      if (lovrViewID == 1) {\n        vertex.x = -vertex.x;\n      }\n\n      return vertex;\n    }\n  ]], [[\n    // The fragment shader returns solid black for illustration purposes.  It could be transparent.\n    vec4 color(vec4 graphicsColor, sampler2D image, vec2 uv) {\n      return vec4(0., 0., 0., 1.);\n    }\n  ]])\n\n  mask = lovr.headset.getDisplayMask()\n\n  if mask then\n    mesh = lovr.graphics.newMesh({ { 'lovrPosition', 'float', 2 } }, mask, 'triangles')\n  end\nend\n\nfunction lovr.draw()\n  if mask then\n    -- Mask out parts of the display that aren't visible to skip rendering those pixels later\n    lovr.graphics.setShader(shader)\n    mesh:draw()\n    lovr.graphics.setShader()\n\n    -- Draw a red cube\n    lovr.graphics.setColor(0xff0000)\n    lovr.graphics.cube('fill', 0, 1.7, -1, .5, lovr.timer.getTime())\n    lovr.graphics.setColor(0xffffff)\n  else\n    lovr.graphics.setColor(0x000000)\n    lovr.graphics.print('No mask found.', 0, 1.7, -3, .2)\n    lovr.graphics.setColor(0xffffff)\n  end\nend"
            }
          }
        },
        {
          name = "getDisplayWidth",
          tag = "headset",
          summary = "Get the width of the headset display.",
          description = "Returns the width of the headset display (for one eye), in pixels.",
          key = "lovr.headset.getDisplayWidth",
          module = "lovr.headset",
          related = {
            "lovr.headset.getDisplayHeight",
            "lovr.headset.getDisplayDimensions"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "width",
                  type = "number",
                  description = "The width of the display."
                }
              }
            }
          }
        },
        {
          name = "getDriver",
          tag = "headset",
          summary = "Get the VR API currently in use for a device.",
          description = "Returns the `HeadsetDriver` that is currently in use, optionally for a specific device.  The order of headset drivers can be changed using `lovr.conf` to prefer or exclude specific VR APIs.",
          key = "lovr.headset.getDriver",
          module = "lovr.headset",
          variants = {
            {
              description = "Get the current headset driver that LÖVR is submitting frames to.",
              arguments = {},
              returns = {
                {
                  name = "driver",
                  type = "HeadsetDriver",
                  description = "The driver of the headset in use, e.g. \"OpenVR\"."
                }
              }
            },
            {
              description = "Get the current input driver for a device.",
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The device to get the active driver of.  This will be the first driver that is currently returning a pose for the device."
                }
              },
              returns = {
                {
                  name = "driver",
                  type = "HeadsetDriver",
                  description = "The driver of the headset in use, e.g. \"OpenVR\"."
                }
              }
            }
          }
        },
        {
          name = "getHands",
          tag = "input",
          summary = "Get a list of currently tracked hand devices.",
          description = "Returns a table with all of the currently tracked hand devices.",
          key = "lovr.headset.getHands",
          module = "lovr.headset",
          notes = "The hand paths will *always* be either `hand/left` or `hand/right`.",
          examples = {
            {
              code = "function lovr.update(dt)\n  for i, hand in ipairs(lovr.headset.getHands()) do\n    print(hand, lovr.headset.getPose(hand))\n  end\nend"
            }
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "hands",
                  type = "table",
                  description = "The currently tracked hand devices.",
                  arguments = {},
                  returns = {}
                }
              }
            }
          }
        },
        {
          name = "getMirrorTexture",
          tag = "headset",
          summary = "Get the Texture containing a view of what's in the headset.",
          description = "Returns a Texture that contains whatever is currently rendered to the headset.\n\nSometimes this can be `nil` if the current headset driver doesn't have a mirror texture, which can happen if the driver renders directly to the display.  Currently the `desktop`, `webxr`, and `vrapi` drivers do not have a mirror texture.\n\nIt also isn't guaranteed that the same Texture will be returned by subsequent calls to this function.  Currently, the `oculus` driver exhibits this behavior.",
          key = "lovr.headset.getMirrorTexture",
          module = "lovr.headset",
          related = {
            "lovr.mirror"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "mirror",
                  type = "Texture",
                  description = "The mirror texture."
                }
              }
            }
          }
        },
        {
          name = "getName",
          tag = "headset",
          summary = "Get the name of the connected headset display.",
          description = "Returns the name of the headset as a string.  The exact string that is returned depends on the hardware and VR SDK that is currently in use.",
          key = "lovr.headset.getName",
          module = "lovr.headset",
          notes = "<table>\n  <thead>\n    <tr>\n      <td>driver</td>\n      <td>name</td>\n    </tr>\n  </thead>\n  <tbody>\n    <tr>\n      <td>desktop</td>\n      <td><code>Simulator</code></td>\n    </tr>\n    <tr>\n      <td>openvr</td>\n      <td>varies</td>\n    </tr>\n    <tr>\n      <td>openxr</td>\n      <td>varies</td>\n    </tr>\n    <tr>\n      <td>vrapi</td>\n      <td><code>Oculus Quest</code> or <code>Oculus Quest 2</code></td>\n    </tr>\n    <tr>\n      <td>webxr</td>\n      <td>always nil</td>\n    </tr>\n    <tr>\n      <td>oculus</td>\n      <td>varies</td>\n    </tr>\n    <tr>\n      <td>pico</td>\n      <td><code>Pico</code></td>\n    </tr>\n  </tbody> </table>",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "name",
                  type = "string",
                  description = "The name of the headset as a string."
                }
              }
            }
          }
        },
        {
          name = "getOrientation",
          tag = "input",
          summary = "Get the orientation of a device.",
          description = "Returns the current orientation of a device, in angle/axis form.",
          key = "lovr.headset.getOrientation",
          module = "lovr.headset",
          notes = "If the device isn't tracked, all zeroes will be returned.",
          related = {
            "lovr.headset.getPose",
            "lovr.headset.getPosition",
            "lovr.headset.getVelocity",
            "lovr.headset.getAngularVelocity",
            "lovr.headset.isTracked",
            "lovr.headset.getDriver"
          },
          variants = {
            {
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The device to get the orientation of.",
                  default = "'head'"
                }
              },
              returns = {
                {
                  name = "angle",
                  type = "number",
                  description = "The amount of rotation around the axis of rotation, in radians."
                },
                {
                  name = "ax",
                  type = "number",
                  description = "The x component of the axis of rotation."
                },
                {
                  name = "ay",
                  type = "number",
                  description = "The y component of the axis of rotation."
                },
                {
                  name = "az",
                  type = "number",
                  description = "The z component of the axis of rotation."
                }
              }
            }
          }
        },
        {
          name = "getOriginType",
          tag = "headset",
          summary = "Get the type of tracking origin of the headset.",
          description = "Returns the type of origin used for the tracking volume.  The different types of origins are explained on the `HeadsetOrigin` page.",
          key = "lovr.headset.getOriginType",
          module = "lovr.headset",
          related = {
            "HeadsetOrigin"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "origin",
                  type = "HeadsetOrigin",
                  description = "The type of origin."
                }
              }
            }
          }
        },
        {
          name = "getPass",
          summary = "Get a Pass that renders to the headset.",
          description = "Returns a `Pass` that renders to the headset display.",
          key = "lovr.headset.getPass",
          module = "lovr.headset",
          notes = "The same Pass will be returned until `lovr.headset.submit` is called.\n\nThe first time this function is called during a frame, the views of the Pass will be initialized with the headset view poses and view angles.\n\nThe pass will be cleared to the background color, which can be changed using `lovr.graphics.setBackgroundColor`.\n\nThe pass will have a depth buffer.  If `t.headset.stencil` was set to a truthy value in `lovr.conf`, the depth buffer will use the `d32fs8` format, otherwise `d32f` will be used.\n\nIf `t.headset.antialias` was set to a truthy value in `lovr.conf`, the pass will be multisampled.",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "pass",
                  type = "Pass",
                  description = "The pass."
                }
              }
            }
          },
          related = {
            "lovr.graphics.getPass",
            "lovr.graphics.getWindowPass",
            "lovr.conf"
          }
        },
        {
          name = "getPose",
          tag = "input",
          summary = "Get the pose of a device.",
          description = "Returns the current position and orientation of a device.",
          key = "lovr.headset.getPose",
          module = "lovr.headset",
          notes = "Units are in meters.\n\nIf the device isn't tracked, all zeroes will be returned.",
          related = {
            "lovr.headset.getPosition",
            "lovr.headset.getOrientation",
            "lovr.headset.getVelocity",
            "lovr.headset.getAngularVelocity",
            "lovr.headset.getSkeleton",
            "lovr.headset.isTracked",
            "lovr.headset.getDriver"
          },
          variants = {
            {
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The device to get the pose of.",
                  default = "'head'"
                }
              },
              returns = {
                {
                  name = "x",
                  type = "number",
                  description = "The x position."
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y position."
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z position."
                },
                {
                  name = "angle",
                  type = "number",
                  description = "The amount of rotation around the axis of rotation, in radians."
                },
                {
                  name = "ax",
                  type = "number",
                  description = "The x component of the axis of rotation."
                },
                {
                  name = "ay",
                  type = "number",
                  description = "The y component of the axis of rotation."
                },
                {
                  name = "az",
                  type = "number",
                  description = "The z component of the axis of rotation."
                }
              }
            }
          }
        },
        {
          name = "getPosition",
          tag = "input",
          summary = "Get the position of a device.",
          description = "Returns the current position of a device, in meters, relative to the play area.",
          key = "lovr.headset.getPosition",
          module = "lovr.headset",
          notes = "If the device isn't tracked, all zeroes will be returned.",
          related = {
            "lovr.headset.getPose",
            "lovr.headset.getOrientation",
            "lovr.headset.getVelocity",
            "lovr.headset.getAngularVelocity",
            "lovr.headset.isTracked",
            "lovr.headset.getDriver"
          },
          variants = {
            {
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The device to get the position of.",
                  default = "'head'"
                }
              },
              returns = {
                {
                  name = "x",
                  type = "number",
                  description = "The x position of the device."
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y position of the device."
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z position of the device."
                }
              }
            }
          }
        },
        {
          name = "getSkeleton",
          tag = "input",
          summary = "Get skeletal joint poses tracked by a device.",
          description = "Returns a list of joint poses tracked by a device.  Currently, only hand devices are able to track joints.",
          key = "lovr.headset.getSkeleton",
          module = "lovr.headset",
          related = {
            "lovr.headset.getPose",
            "lovr.headset.animate"
          },
          examples = {
            {
              code = "function lovr.draw()\n  for _, hand in ipairs({ 'left', 'right' }) do\n    for _, joint in ipairs(lovr.headset.getSkeleton(hand) or {}) do\n      lovr.graphics.points(unpack(joint, 1, 3))\n    end\n  end\nend"
            }
          },
          notes = "If the Device does not support tracking joints or the poses are unavailable, `nil` is returned.\n\nThe joint orientation is similar to the graphics coordinate system: -Z is the forwards direction, pointing towards the fingertips.  The +Y direction is \"up\", pointing out of the back of the hand.  The +X direction is to the right, perpendicular to X and Z.\n\nHand joints are returned in the following order:\n\n<table>\n  <thead>\n    <tr>\n      <td colspan=\"2\">Joint</td>\n      <td>Index</td>\n    </tr>\n  </thead>\n  <tbody>\n    <tr>\n      <td colspan=\"2\">Palm</td>\n      <td>1</td>\n    </tr>\n    <tr>\n      <td colspan=\"2\">Wrist</td>\n      <td>2</td>\n    </tr>\n    <tr>\n      <td rowspan=\"4\">Thumb</td>\n      <td>Metacarpal</td>\n      <td>3</td>\n    </tr>\n    <tr>\n      <td>Proximal</td>\n      <td>4</td>\n    </tr>\n    <tr>\n      <td>Distal</td>\n      <td>5</td>\n    </tr>\n    <tr>\n      <td>Tip</td>\n      <td>6</td>\n    </tr>\n    <tr>\n      <td rowspan=\"5\">Index</td>\n      <td>Metacarpal</td>\n      <td>7</td>\n    </tr>\n    <tr>\n      <td>Proximal</td>\n      <td>8</td>\n    </tr>\n    <tr>\n      <td>Intermediate</td>\n      <td>9</td>\n    </tr>\n    <tr>\n      <td>Distal</td>\n      <td>10</td>\n    </tr>\n    <tr>\n      <td>Tip</td>\n      <td>11</td>\n    </tr>\n    <tr>\n      <td rowspan=\"5\">Middle</td>\n      <td>Metacarpal</td>\n      <td>12</td>\n    </tr>\n    <tr>\n      <td>Proximal</td>\n      <td>13</td>\n    </tr>\n    <tr>\n      <td>Intermediate</td>\n      <td>14</td>\n    </tr>\n    <tr>\n      <td>Distal</td>\n      <td>15</td>\n    </tr>\n    <tr>\n      <td>Tip</td>\n      <td>16</td>\n    </tr>\n    <tr>\n      <td rowspan=\"5\">Ring</td>\n      <td>Metacarpal</td>\n      <td>17</td>\n    </tr>\n    <tr>\n      <td>Proximal</td>\n      <td>18</td>\n    </tr>\n    <tr>\n      <td>Intermediate</td>\n      <td>19</td>\n    </tr>\n    <tr>\n      <td>Distal</td>\n      <td>20</td>\n    </tr>\n    <tr>\n      <td>Tip</td>\n      <td>21</td>\n    </tr>\n    <tr>\n      <td rowspan=\"5\">Pinky</td>\n      <td>Metacarpal</td>\n      <td>22</td>\n    </tr>\n    <tr>\n      <td>Proximal</td>\n      <td>23</td>\n    </tr>\n    <tr>\n      <td>Intermediate</td>\n      <td>24</td>\n    </tr>\n    <tr>\n      <td>Distal</td>\n      <td>25</td>\n    </tr>\n    <tr>\n      <td>Tip</td>\n      <td>26</td>\n    </tr>\n  </tbody> </table>",
          variants = {
            {
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The Device to query."
                }
              },
              returns = {
                {
                  name = "poses",
                  type = "table",
                  description = "A list of joint poses for the device.  Each pose is a table with 3 numbers for the position of the joint followed by 4 numbers for the angle/axis orientation of the joint."
                }
              }
            },
            {
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The Device to query."
                },
                {
                  name = "t",
                  type = "table",
                  description = "A table to fill with the joint poses, instead of allocating a new one."
                }
              },
              returns = {
                {
                  name = "poses",
                  type = "table",
                  description = "A list of joint poses for the device.  Each pose is a table with 3 numbers for the position of the joint followed by 4 numbers for the angle/axis orientation of the joint."
                }
              }
            }
          }
        },
        {
          name = "getTime",
          summary = "Get the predicted display time.",
          description = "Returns the estimated time in the future at which the light from the pixels of the current frame will hit the eyes of the user.\n\nThis can be used as a replacement for `lovr.timer.getTime` for timestamps that are used for rendering to get a smoother result that is synchronized with the display of the headset.",
          key = "lovr.headset.getTime",
          module = "lovr.headset",
          notes = "This has a different epoch than `lovr.timer.getTime`, so it is not guaranteed to be close to that value.",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "time",
                  type = "number",
                  description = "The predicted display time, in seconds."
                }
              }
            }
          },
          related = {
            "lovr.timer.getTime"
          }
        },
        {
          name = "getVelocity",
          tag = "input",
          summary = "Get the linear velocity of a device.",
          description = "Returns the current linear velocity of a device, in meters per second.",
          key = "lovr.headset.getVelocity",
          module = "lovr.headset",
          related = {
            "lovr.headset.getAngularVelocity",
            "lovr.headset.getPose",
            "lovr.headset.getPosition",
            "lovr.headset.getOrientation"
          },
          variants = {
            {
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The device to get the velocity of.",
                  default = "'head'"
                }
              },
              returns = {
                {
                  name = "vx",
                  type = "number",
                  description = "The x component of the linear velocity."
                },
                {
                  name = "vy",
                  type = "number",
                  description = "The y component of the linear velocity."
                },
                {
                  name = "vz",
                  type = "number",
                  description = "The z component of the linear velocity."
                }
              }
            }
          }
        },
        {
          name = "getViewAngles",
          tag = "headset",
          summary = "Get the field of view angles of a view.",
          description = "Returns the view angles of one of the headset views.\n\nThese can be used with `Mat4:fov` to create a projection matrix.\n\nIf tracking data is unavailable for the view or the index is invalid, `nil` is returned.",
          key = "lovr.headset.getViewAngles",
          module = "lovr.headset",
          related = {
            "lovr.headset.getViewCount",
            "lovr.headset.getViewPose"
          },
          variants = {
            {
              arguments = {
                {
                  name = "view",
                  type = "number",
                  description = "The view index."
                }
              },
              returns = {
                {
                  name = "left",
                  type = "number",
                  description = "The left view angle, in radians."
                },
                {
                  name = "right",
                  type = "number",
                  description = "The right view angle, in radians."
                },
                {
                  name = "top",
                  type = "number",
                  description = "The top view angle, in radians."
                },
                {
                  name = "bottom",
                  type = "number",
                  description = "The bottom view angle, in radians."
                }
              }
            }
          }
        },
        {
          name = "getViewCount",
          tag = "headset",
          summary = "Get the number of views used for rendering.",
          description = "Returns the number of views used for rendering.  Each view consists of a pose in space and a set of angle values that determine the field of view.\n\nThis is usually 2 for stereo rendering configurations, but it can also be different.  For example, one way of doing foveated rendering uses 2 views for each eye -- one low quality view with a wider field of view, and a high quality view with a narrower field of view.",
          key = "lovr.headset.getViewCount",
          module = "lovr.headset",
          related = {
            "lovr.headset.getViewPose",
            "lovr.headset.getViewAngles"
          },
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "count",
                  type = "number",
                  description = "The number of views."
                }
              }
            }
          }
        },
        {
          name = "getViewPose",
          tag = "headset",
          summary = "Get the pose of one of the views.",
          description = "Returns the pose of one of the headset views.  This info can be used to create view matrices or do other eye-dependent calculations.\n\nIf tracking data is unavailable for the view or the index is invalid, `nil` is returned.",
          key = "lovr.headset.getViewPose",
          module = "lovr.headset",
          related = {
            "lovr.headset.getViewCount",
            "lovr.headset.getViewAngles"
          },
          variants = {
            {
              arguments = {
                {
                  name = "view",
                  type = "number",
                  description = "The view index."
                }
              },
              returns = {
                {
                  name = "x",
                  type = "number",
                  description = "The x coordinate of the view position, in meters."
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y coordinate of the view position, in meters."
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z coordinate of the view position, in meters."
                },
                {
                  name = "angle",
                  type = "number",
                  description = "The amount of rotation around the rotation axis, in radians."
                },
                {
                  name = "ax",
                  type = "number",
                  description = "The x component of the axis of rotation."
                },
                {
                  name = "ay",
                  type = "number",
                  description = "The y component of the axis of rotation."
                },
                {
                  name = "az",
                  type = "number",
                  description = "The z component of the axis of rotation."
                }
              }
            }
          }
        },
        {
          name = "isDown",
          tag = "input",
          summary = "Get the state of a button on a device.",
          description = "Returns whether a button on a device is pressed.",
          key = "lovr.headset.isDown",
          module = "lovr.headset",
          notes = "When hand tracking is active, pinching will be mapped to the `trigger` button.",
          related = {
            "DeviceButton",
            "lovr.headset.wasPressed",
            "lovr.headset.wasReleased",
            "lovr.headset.isTouched",
            "lovr.headset.getAxis"
          },
          variants = {
            {
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The device."
                },
                {
                  name = "button",
                  type = "DeviceButton",
                  description = "The button."
                }
              },
              returns = {
                {
                  name = "down",
                  type = "boolean",
                  description = "Whether the button on the device is currently pressed, or `nil` if the device does not have the specified button."
                }
              }
            }
          }
        },
        {
          name = "isTouched",
          tag = "input",
          summary = "Check if a button on a device is touched.",
          description = "Returns whether a button on a device is currently touched.",
          key = "lovr.headset.isTouched",
          module = "lovr.headset",
          related = {
            "DeviceButton",
            "lovr.headset.isDown",
            "lovr.headset.getAxis"
          },
          variants = {
            {
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The device."
                },
                {
                  name = "button",
                  type = "DeviceButton",
                  description = "The button."
                }
              },
              returns = {
                {
                  name = "touched",
                  type = "boolean",
                  description = "Whether the button on the device is currently touched, or `nil` if the device does not have the button or it isn't touch-sensitive."
                }
              }
            }
          }
        },
        {
          name = "isTracked",
          tag = "input",
          summary = "Check if a device is currently tracked.",
          description = "Returns whether any active headset driver is currently returning pose information for a device.",
          key = "lovr.headset.isTracked",
          module = "lovr.headset",
          notes = "If a device is tracked, it is guaranteed to return a valid pose until the next call to `lovr.headset.update`.",
          variants = {
            {
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The device to get the pose of.",
                  default = "'head'"
                }
              },
              returns = {
                {
                  name = "tracked",
                  type = "boolean",
                  description = "Whether the device is currently tracked."
                }
              }
            }
          }
        },
        {
          name = "newModel",
          tag = "input",
          summary = "Get a Model for a device.",
          description = "Returns a new Model for the specified device.",
          key = "lovr.headset.newModel",
          module = "lovr.headset",
          related = {
            "lovr.headset.animate"
          },
          examples = {
            {
              code = "local models = {}\n\nfunction lovr.draw()\n  for i, hand in ipairs(lovr.headset.getHands()) do\n    models[hand] = models[hand] or lovr.headset.newModel(hand)\n\n    if models[hand] then\n      local x, y, z, angle, ax, ay, az = lovr.headset.getPose(hand)\n      models[hand]:draw(x, y, z, 1, angle, ax, ay, az)\n    end\n  end\nend"
            }
          },
          notes = "Currently this is only implemented for hand models on the Oculus Quest.",
          variants = {
            {
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The device to load a model for.",
                  default = "'head'"
                },
                {
                  name = "options",
                  type = "table",
                  description = "Options for loading the model.",
                  default = "{}",
                  table = {
                    {
                      name = "animated",
                      type = "boolean",
                      description = "Whether an animatable model should be loaded, for use with `lovr.headset.animate`.'",
                      default = "false"
                    }
                  }
                }
              },
              returns = {
                {
                  name = "model",
                  type = "Model",
                  description = "The new Model, or `nil` if a model could not be loaded."
                }
              }
            }
          }
        },
        {
          name = "renderTo",
          tag = "headset",
          summary = "Render to the headset using a function.",
          description = "Renders to each eye of the headset using a function.\n\nThis function takes care of setting the appropriate graphics transformations to ensure that the scene is rendered as though it is being viewed through each eye of the player.  It also takes care of setting the correct projection for the headset lenses.\n\nIf the headset module is enabled, this function is called automatically by `lovr.run` with `lovr.draw` as the callback.",
          key = "lovr.headset.renderTo",
          module = "lovr.headset",
          notes = "When using the `pico` headset driver, headset rendering is asynchronous and the callback passed to `lovr.headset.renderTo` will not be called immediately.\n\nAt the beginning of the callback, the display is cleared to the background color.  The background color can be changed using `lovr.graphics.setBackgroundColor`.\n\nIf the callback is `nil`, an empty frame cleared to current graphics background color will be submitted to the headset.",
          variants = {
            {
              arguments = {
                {
                  name = "callback",
                  type = "function",
                  description = "The function used to render.  Any functions called will render to the headset instead of to the window.",
                  arguments = {},
                  returns = {},
                  variants = {
                    {
                      arguments = {
                        "callback"
                      },
                      returns = {}
                    }
                  }
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "setClipDistance",
          tag = "headset",
          summary = "Set the near and far clipping planes of the headset.",
          description = "Sets the near and far clipping planes used to render to the headset.  Objects closer than the near clipping plane or further than the far clipping plane will be clipped out of view.",
          key = "lovr.headset.setClipDistance",
          module = "lovr.headset",
          notes = "The default clip distances are 0.1 and 100.0.",
          variants = {
            {
              arguments = {
                {
                  name = "near",
                  type = "number",
                  description = "The distance to the near clipping plane, in meters."
                },
                {
                  name = "far",
                  type = "number",
                  description = "The distance to the far clipping plane, in meters."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "setDisplayFrequency",
          tag = "headset",
          summary = "Set the display refresh rate.",
          description = "Sets the display refresh rate, in Hz.",
          key = "lovr.headset.setDisplayFrequency",
          module = "lovr.headset",
          notes = "Changing the display refresh-rate also changes the frequency of lovr.update() and lovr.draw() as they depend on the display frequency.",
          variants = {
            {
              arguments = {
                {
                  name = "frequency",
                  type = "number",
                  description = "The new refresh rate, in Hz."
                }
              },
              returns = {
                {
                  name = "success",
                  type = "boolean",
                  description = "Whether the display refresh rate was successfully set."
                }
              }
            }
          }
        },
        {
          name = "vibrate",
          tag = "input",
          summary = "Make a device go BZZZ!",
          description = "Causes the device to vibrate with a custom strength, duration, and frequency, if possible.",
          key = "lovr.headset.vibrate",
          module = "lovr.headset",
          notes = "When using the `openvr` headset driver on an HTC Vive, the value for the `duration` currently must be less than .004 seconds.  Call this function several frames in a row for stronger or prolonged vibration patterns.\n\nOn the Oculus Quest, devices can only be vibrated once per frame.  Any attempts after the first will return `false`.",
          variants = {
            {
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The device to vibrate.",
                  default = "'head'"
                },
                {
                  name = "strength",
                  type = "number",
                  description = "The strength of the vibration (amplitude), between 0 and 1.",
                  default = "1"
                },
                {
                  name = "duration",
                  type = "number",
                  description = "The duration of the vibration, in seconds.",
                  default = ".5"
                },
                {
                  name = "frequency",
                  type = "number",
                  description = "The frequency of the vibration, in hertz.  0 will use a default frequency.",
                  default = "0"
                }
              },
              returns = {
                {
                  name = "vibrated",
                  type = "boolean",
                  description = "Whether the vibration was successfully triggered by an active headset driver."
                }
              }
            }
          }
        },
        {
          name = "wasPressed",
          tag = "input",
          summary = "Check if a button was just pressed.",
          description = "Returns whether a button on a device was pressed this frame.",
          key = "lovr.headset.wasPressed",
          module = "lovr.headset",
          notes = "Some headset backends are not able to return pressed/released information.  These drivers will always return false for `lovr.headset.wasPressed` and `lovr.headset.wasReleased`.\n\nTypically the internal `lovr.headset.update` function will update pressed/released status.",
          related = {
            "DeviceButton",
            "lovr.headset.isDown",
            "lovr.headset.wasReleased",
            "lovr.headset.isTouched",
            "lovr.headset.getAxis"
          },
          variants = {
            {
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The device."
                },
                {
                  name = "button",
                  type = "DeviceButton",
                  description = "The button to check."
                }
              },
              returns = {
                {
                  name = "pressed",
                  type = "boolean",
                  description = "Whether the button on the device was pressed this frame."
                }
              }
            }
          }
        },
        {
          name = "wasReleased",
          tag = "input",
          summary = "Check if a button was just released.",
          description = "Returns whether a button on a device was released this frame.",
          key = "lovr.headset.wasReleased",
          module = "lovr.headset",
          notes = "Some headset backends are not able to return pressed/released information.  These drivers will always return false for `lovr.headset.wasPressed` and `lovr.headset.wasReleased`.\n\nTypically the internal `lovr.headset.update` function will update pressed/released status.",
          related = {
            "DeviceButton",
            "lovr.headset.isDown",
            "lovr.headset.wasPressed",
            "lovr.headset.isTouched",
            "lovr.headset.getAxis"
          },
          variants = {
            {
              arguments = {
                {
                  name = "device",
                  type = "Device",
                  description = "The device."
                },
                {
                  name = "button",
                  type = "DeviceButton",
                  description = "The button to check."
                }
              },
              returns = {
                {
                  name = "released",
                  type = "boolean",
                  description = "Whether the button on the device was released this frame."
                }
              }
            }
          }
        }
      },
      enums = {
        {
          name = "Device",
          description = "Different types of input devices supported by the `lovr.headset` module.",
          key = "Device",
          module = "lovr.headset",
          related = {
            "DeviceAxis",
            "DeviceButton",
            "lovr.headset.getPose",
            "lovr.headset.getPosition",
            "lovr.headset.getOrientation",
            "lovr.headset.getVelocity",
            "lovr.headset.getAngularVelocity",
            "lovr.headset.getSkeleton",
            "lovr.headset.isTracked",
            "lovr.headset.isDown",
            "lovr.headset.isTouched",
            "lovr.headset.wasPressed",
            "lovr.headset.wasReleased",
            "lovr.headset.getAxis",
            "lovr.headset.vibrate",
            "lovr.headset.animate"
          },
          values = {
            {
              name = "head",
              description = "The headset."
            },
            {
              name = "hand/left",
              description = "The left controller."
            },
            {
              name = "hand/right",
              description = "The right controller."
            },
            {
              name = "left",
              description = "A shorthand for hand/left."
            },
            {
              name = "right",
              description = "A shorthand for hand/right."
            },
            {
              name = "elbow/left",
              description = "A device tracking the left elbow."
            },
            {
              name = "elbow/right",
              description = "A device tracking the right elbow."
            },
            {
              name = "shoulder/left",
              description = "A device tracking the left shoulder."
            },
            {
              name = "shoulder/right",
              description = "A device tracking the right shoulder."
            },
            {
              name = "chest",
              description = "A device tracking the chest."
            },
            {
              name = "waist",
              description = "A device tracking the waist."
            },
            {
              name = "knee/left",
              description = "A device tracking the left knee."
            },
            {
              name = "knee/right",
              description = "A device tracking the right knee."
            },
            {
              name = "foot/left",
              description = "A device tracking the left foot or ankle."
            },
            {
              name = "foot/right",
              description = "A device tracking the right foot or ankle."
            },
            {
              name = "camera",
              description = "A camera device, often used for recording \"mixed reality\" footage."
            },
            {
              name = "keyboard",
              description = "A tracked keyboard."
            },
            {
              name = "eye/left",
              description = "The left eye."
            },
            {
              name = "eye/right",
              description = "The right eye."
            }
          }
        },
        {
          name = "DeviceAxis",
          description = "Axes on an input device.",
          key = "DeviceAxis",
          module = "lovr.headset",
          related = {
            "lovr.headset.getAxis",
            "DeviceButton"
          },
          values = {
            {
              name = "trigger",
              description = "A trigger (1D)."
            },
            {
              name = "thumbstick",
              description = "A thumbstick (2D)."
            },
            {
              name = "touchpad",
              description = "A touchpad (2D)."
            },
            {
              name = "grip",
              description = "A grip button or grab gesture (1D)."
            }
          }
        },
        {
          name = "DeviceButton",
          description = "Buttons on an input device.",
          key = "DeviceButton",
          module = "lovr.headset",
          values = {
            {
              name = "trigger",
              description = "The trigger button."
            },
            {
              name = "thumbstick",
              description = "The thumbstick."
            },
            {
              name = "touchpad",
              description = "The touchpad."
            },
            {
              name = "grip",
              description = "The grip button."
            },
            {
              name = "menu",
              description = "The menu button."
            },
            {
              name = "a",
              description = "The A button."
            },
            {
              name = "b",
              description = "The B button."
            },
            {
              name = "x",
              description = "The X button."
            },
            {
              name = "y",
              description = "The Y button."
            },
            {
              name = "proximity",
              description = "The proximity sensor on a headset."
            }
          }
        },
        {
          name = "HeadsetDriver",
          summary = "VR APIs.",
          description = "These are all of the supported VR APIs that LÖVR can use to power the lovr.headset module.  You can change the order of headset drivers using `lovr.conf` to prefer or exclude specific VR APIs.\n\nAt startup, LÖVR searches through the list of drivers in order.  One headset driver will be used for rendering to the VR display, and all supported headset drivers will be used for device input.  The way this works is that when poses or button input is requested, the input drivers are queried (in the order they appear in `conf.lua`) to see if any of them currently have data for the specified device.  The first one that returns data will be used to provide the result. This allows projects to support multiple types of hardware devices.",
          key = "HeadsetDriver",
          module = "lovr.headset",
          values = {
            {
              name = "desktop",
              description = "A VR simulator using keyboard/mouse."
            },
            {
              name = "oculus",
              description = "Oculus Desktop SDK."
            },
            {
              name = "openvr",
              description = "OpenVR."
            },
            {
              name = "openxr",
              description = "OpenXR."
            },
            {
              name = "vrapi",
              description = "Oculus Mobile SDK."
            },
            {
              name = "pico",
              description = "Pico."
            },
            {
              name = "webxr",
              description = "WebXR."
            }
          }
        },
        {
          name = "HeadsetOrigin",
          summary = "Different types of coordinate space origins.",
          description = "Represents the different types of origins for coordinate spaces.  An origin of \"floor\" means that the origin is on the floor in the middle of a room-scale play area.  An origin of \"head\" means that no positional tracking is available, and consequently the origin is always at the position of the headset.",
          key = "HeadsetOrigin",
          module = "lovr.headset",
          values = {
            {
              name = "head",
              description = "The origin is at the head."
            },
            {
              name = "floor",
              description = "The origin is on the floor."
            }
          }
        }
      },
      sections = {
        {
          name = "Headset",
          tag = "headset",
          description = "Functions that return information about the active head mounted display (HMD)."
        },
        {
          name = "Input",
          tag = "input",
          description = "Functions for accessing input devices, like controllers, hands, trackers, or gamepads."
        },
        {
          name = "Play area",
          tag = "playArea",
          description = "Retrieve information about the size and shape of the room the player is in, and provides information about the \"chaperone\", a visual indicator that appears whenever a player is about to run into a wall."
        }
      }
    },
    {
      name = "math",
      tag = "modules",
      summary = "Contains useful math helpers.",
      description = "The `lovr.math` module provides math helpers commonly used for 3D applications.",
      key = "lovr.math",
      objects = {
        {
          name = "Curve",
          summary = "A Bézier curve.",
          description = "A Curve is an object that represents a Bézier curve in three dimensions.  Curves are defined by an arbitrary number of control points (note that the curve only passes through the first and last control point).\n\nOnce a Curve is created with `lovr.math.newCurve`, you can use `Curve:evaluate` to get a point on the curve or `Curve:render` to get a list of all of the points on the curve.  These points can be passed directly to `lovr.graphics.points` or `lovr.graphics.line` to render the curve.\n\nNote that for longer or more complicated curves (like in a drawing application) it can be easier to store the path as several Curve objects.",
          key = "Curve",
          module = "lovr.math",
          constructors = {
            "lovr.math.newCurve",
            "Curve:slice"
          },
          methods = {
            {
              name = "addPoint",
              summary = "Add a new control point to the Curve.",
              description = "Inserts a new control point into the Curve at the specified index.",
              key = "Curve:addPoint",
              module = "lovr.math",
              notes = "An error will be thrown if the index is less than one or more than the number of control points.",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the control point."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the control point."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the control point."
                    },
                    {
                      name = "index",
                      type = "number",
                      description = "The index to insert the control point at.  If nil, the control point is added to the end of the list of control points.",
                      default = "nil"
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "Curve:getPointCount",
                "Curve:getPoint",
                "Curve:setPoint",
                "Curve:removePoint"
              }
            },
            {
              name = "evaluate",
              summary = "Turn a number from 0 to 1 into a point on the Curve.",
              description = "Returns a point on the Curve given a parameter `t` from 0 to 1.  0 will return the first control point, 1 will return the last point, .5 will return a point in the \"middle\" of the Curve, etc.",
              key = "Curve:evaluate",
              module = "lovr.math",
              notes = "An error will be thrown if `t` is not between 0 and 1, or if the Curve has less than two points.",
              variants = {
                {
                  arguments = {
                    {
                      name = "t",
                      type = "number",
                      description = "The parameter to evaluate the Curve at."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the point."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the point."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the point."
                    }
                  }
                }
              },
              related = {
                "Curve:getTangent",
                "Curve:render",
                "Curve:slice"
              }
            },
            {
              name = "getPoint",
              summary = "Get a control point of the Curve.",
              description = "Returns a control point of the Curve.",
              key = "Curve:getPoint",
              module = "lovr.math",
              notes = "An error will be thrown if the index is less than one or more than the number of control points.",
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index to retrieve."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the control point."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the control point."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the control point."
                    }
                  }
                }
              },
              related = {
                "Curve:getPointCount",
                "Curve:setPoint",
                "Curve:addPoint",
                "Curve:removePoint"
              }
            },
            {
              name = "getPointCount",
              summary = "Get the number of control points in the Curve.",
              description = "Returns the number of control points in the Curve.",
              key = "Curve:getPointCount",
              module = "lovr.math",
              related = {
                "Curve:getPoint",
                "Curve:setPoint",
                "Curve:addPoint",
                "Curve:removePoint"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "count",
                      type = "number",
                      description = "The number of control points."
                    }
                  }
                }
              }
            },
            {
              name = "getTangent",
              summary = "Get the direction of the Curve at a point.",
              description = "Returns a direction vector for the Curve given a parameter `t` from 0 to 1.  0 will return the direction at the first control point, 1 will return the direction at the last point, .5 will return the direction at the \"middle\" of the Curve, etc.",
              key = "Curve:getTangent",
              module = "lovr.math",
              notes = "The direction vector returned by this function will have a length of one.",
              variants = {
                {
                  arguments = {
                    {
                      name = "t",
                      type = "number",
                      description = "Where on the Curve to compute the direction."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the point."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the point."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the point."
                    }
                  }
                }
              },
              related = {
                "Curve:evaluate",
                "Curve:render",
                "Curve:slice"
              }
            },
            {
              name = "removePoint",
              summary = "Remove a control point from the Curve.",
              description = "Removes a control point from the Curve.",
              key = "Curve:removePoint",
              module = "lovr.math",
              notes = "An error will be thrown if the index is less than one or more than the number of control points.",
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index of the control point to remove."
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "Curve:getPointCount",
                "Curve:getPoint",
                "Curve:setPoint",
                "Curve:addPoint"
              }
            },
            {
              name = "render",
              summary = "Get a list of points on the Curve.",
              description = "Returns a list of points on the Curve.  The number of points can be specified to get a more or less detailed representation, and it is also possible to render a subsection of the Curve.",
              key = "Curve:render",
              module = "lovr.math",
              notes = "This function will always return 2 points if the Curve is a line with only 2 control points.",
              variants = {
                {
                  arguments = {
                    {
                      name = "n",
                      type = "number",
                      description = "The number of points to use.",
                      default = "32"
                    },
                    {
                      name = "t1",
                      type = "number",
                      description = "How far along the curve to start rendering.",
                      default = "0"
                    },
                    {
                      name = "t2",
                      type = "number",
                      description = "How far along the curve to stop rendering.",
                      default = "1"
                    }
                  },
                  returns = {
                    {
                      name = "t",
                      type = "table",
                      description = "A (flat) table of 3D points along the curve."
                    }
                  }
                }
              },
              related = {
                "Curve:evaluate",
                "Curve:slice",
                "lovr.graphics.points",
                "lovr.graphics.line"
              }
            },
            {
              name = "setPoint",
              summary = "Set a control point of the Curve.",
              description = "Changes the position of a control point on the Curve.",
              key = "Curve:setPoint",
              module = "lovr.math",
              notes = "An error will be thrown if the index is less than one or more than the number of control points.",
              variants = {
                {
                  arguments = {
                    {
                      name = "index",
                      type = "number",
                      description = "The index to modify."
                    },
                    {
                      name = "x",
                      type = "number",
                      description = "The new x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The new y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The new z coordinate."
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "Curve:getPointCount",
                "Curve:getPoint",
                "Curve:addPoint",
                "Curve:removePoint"
              }
            },
            {
              name = "slice",
              summary = "Get a new Curve from a slice of an existing one.",
              description = "Returns a new Curve created by slicing the Curve at the specified start and end points.",
              key = "Curve:slice",
              module = "lovr.math",
              notes = "The new Curve will have the same number of control points as the existing curve.\n\nAn error will be thrown if t1 or t2 are not between 0 and 1, or if the Curve has less than two points.",
              variants = {
                {
                  arguments = {
                    {
                      name = "t1",
                      type = "number",
                      description = "The starting point to slice at."
                    },
                    {
                      name = "t2",
                      type = "number",
                      description = "The ending point to slice at."
                    }
                  },
                  returns = {
                    {
                      name = "curve",
                      type = "Curve",
                      description = "A new Curve."
                    }
                  }
                }
              },
              related = {
                "Curve:evaluate",
                "Curve:render"
              }
            }
          }
        },
        {
          name = "Mat4",
          summary = "A 4x4 matrix.",
          description = "A `mat4` is a math type that holds 16 values in a 4x4 grid.",
          key = "Mat4",
          module = "lovr.math",
          constructors = {
            "lovr.math.newMat4",
            "lovr.math.mat4"
          },
          related = {
            "Vec3",
            "Quat"
          },
          methods = {
            {
              name = "equals",
              summary = "Check if a matrix equals another matrix.",
              description = "Returns whether a matrix is approximately equal to another matrix.",
              key = "Mat4:equals",
              module = "lovr.math",
              related = {
                "Vec2:equals",
                "Vec3:equals",
                "Vec4:equals",
                "Quat:equals"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "n",
                      type = "Mat4",
                      description = "The other matrix."
                    }
                  },
                  returns = {
                    {
                      name = "equal",
                      type = "boolean",
                      description = "Whether the 2 matrices approximately equal each other."
                    }
                  }
                }
              }
            },
            {
              name = "fov",
              summary = "Set a projection using raw FoV angles.",
              description = "Sets a projection matrix using raw projection angles and clipping planes.\n\nThis can be used for asymmetric or oblique projections.",
              key = "Mat4:fov",
              module = "lovr.math",
              related = {
                "Mat4:orthographic",
                "Mat4:perspective",
                "lovr.graphics.setProjection"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "left",
                      type = "number",
                      description = "The left half-angle of the projection, in radians."
                    },
                    {
                      name = "right",
                      type = "number",
                      description = "The right half-angle of the projection, in radians."
                    },
                    {
                      name = "up",
                      type = "number",
                      description = "The top half-angle of the projection, in radians."
                    },
                    {
                      name = "down",
                      type = "number",
                      description = "The bottom half-angle of the projection, in radians."
                    },
                    {
                      name = "near",
                      type = "number",
                      description = "The near plane of the projection."
                    },
                    {
                      name = "far",
                      type = "number",
                      description = "The far plane of the projection."
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The original matrix."
                    }
                  }
                }
              }
            },
            {
              name = "identity",
              summary = "Reset the matrix to the identity.",
              description = "Resets the matrix to the identity, effectively setting its translation to zero, its scale to 1, and clearing any rotation.",
              key = "Mat4:identity",
              module = "lovr.math",
              related = {
                "lovr.graphics.origin"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The original matrix."
                    }
                  }
                }
              }
            },
            {
              name = "invert",
              summary = "Invert the matrix.",
              description = "Inverts the matrix, causing it to represent the opposite of its old transform.",
              key = "Mat4:invert",
              module = "lovr.math",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The original matrix."
                    }
                  }
                }
              }
            },
            {
              name = "lookAt",
              summary = "Create a view transform that looks from a position to target position.",
              description = "Sets a view transform matrix that moves and orients camera to look at a target point.\n\nThis is useful for changing camera position and orientation. The resulting Mat4 matrix can be passed to `lovr.graphics.transform()` directly (without inverting) before rendering the scene.\n\nThe lookAt() function produces same result as target() after matrix inversion.",
              key = "Mat4:lookAt",
              module = "lovr.math",
              related = {
                "Mat4:target",
                "Quat:direction"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "from",
                      type = "Vec3",
                      description = "The position of the viewer."
                    },
                    {
                      name = "to",
                      type = "Vec3",
                      description = "The position of the target."
                    },
                    {
                      name = "up",
                      type = "Vec3",
                      description = "The up vector of the viewer.",
                      default = "Vec3(0, 1, 0)"
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The original matrix."
                    }
                  }
                }
              }
            },
            {
              name = "mul",
              summary = "Multiply a matrix with another matrix or a vector.",
              description = "Multiplies this matrix by another value.  Multiplying by a matrix combines their two transforms together.  Multiplying by a vector applies the transformation from the matrix to the vector and returns the vector.",
              key = "Mat4:mul",
              module = "lovr.math",
              notes = "When multiplying by a vec4, the vector is treated as either a point if its w component is 1, or a direction vector if the w is 0 (the matrix translation won't be applied).",
              variants = {
                {
                  arguments = {
                    {
                      name = "n",
                      type = "Mat4",
                      description = "The matrix."
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The original matrix, containing the result."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "v3",
                      type = "Vec3",
                      description = "A 3D vector, treated as a point."
                    }
                  },
                  returns = {
                    {
                      name = "v3",
                      type = "Vec3",
                      description = "The transformed vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "v4",
                      type = "Vec4",
                      description = "A 4D vector."
                    }
                  },
                  returns = {
                    {
                      name = "v4",
                      type = "Vec4",
                      description = "The transformed vector."
                    }
                  }
                }
              },
              related = {
                "Mat4:translate",
                "Mat4:rotate",
                "Mat4:scale"
              }
            },
            {
              name = "orthographic",
              summary = "Turn the matrix into an orthographic projection.",
              description = "Sets this matrix to represent an orthographic projection, useful for 2D/isometric rendering.\n\nThis can be used with `lovr.graphics.setProjection`, or it can be sent to a `Shader` for use in GLSL.",
              key = "Mat4:orthographic",
              module = "lovr.math",
              related = {
                "Mat4:perspective",
                "Mat4:fov",
                "lovr.graphics.setProjection"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "left",
                      type = "number",
                      description = "The left edge of the projection."
                    },
                    {
                      name = "right",
                      type = "number",
                      description = "The right edge of the projection."
                    },
                    {
                      name = "top",
                      type = "number",
                      description = "The top edge of the projection."
                    },
                    {
                      name = "bottom",
                      type = "number",
                      description = "The bottom edge of the projection."
                    },
                    {
                      name = "near",
                      type = "number",
                      description = "The position of the near clipping plane."
                    },
                    {
                      name = "far",
                      type = "number",
                      description = "The position of the far clipping plane."
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The original matrix."
                    }
                  }
                }
              }
            },
            {
              name = "perspective",
              summary = "Turn the matrix into a perspective projection.",
              description = "Sets this matrix to represent a perspective projection.\n\nThis can be used with `lovr.graphics.setProjection`, or it can be sent to a `Shader` for use in GLSL.",
              key = "Mat4:perspective",
              module = "lovr.math",
              related = {
                "Mat4:orthographic",
                "Mat4:fov",
                "lovr.graphics.setProjection"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "near",
                      type = "number",
                      description = "The near plane."
                    },
                    {
                      name = "far",
                      type = "number",
                      description = "The far plane."
                    },
                    {
                      name = "fov",
                      type = "number",
                      description = "The vertical field of view (in radians)."
                    },
                    {
                      name = "aspect",
                      type = "number",
                      description = "The horizontal aspect ratio of the projection (width / height)."
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The original matrix."
                    }
                  }
                }
              }
            },
            {
              name = "rotate",
              summary = "Rotate the matrix.",
              description = "Rotates the matrix using a quaternion or an angle/axis rotation.",
              key = "Mat4:rotate",
              module = "lovr.math",
              related = {
                "Mat4:translate",
                "Mat4:scale",
                "Mat4:identity"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "q",
                      type = "Quat",
                      description = "The rotation to apply to the matrix."
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The original matrix."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The angle component of the angle/axis rotation (radians)."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "1"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The original matrix."
                    }
                  }
                }
              }
            },
            {
              name = "scale",
              summary = "Scale the matrix.",
              description = "Scales the matrix.",
              key = "Mat4:scale",
              module = "lovr.math",
              related = {
                "Mat4:translate",
                "Mat4:rotate",
                "Mat4:identity"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "scale",
                      type = "Vec3",
                      description = "The 3D scale to apply."
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The original matrix."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "sx",
                      type = "number",
                      description = "The x component of the scale to apply."
                    },
                    {
                      name = "sy",
                      type = "number",
                      description = "The y component of the scale to apply.",
                      default = "sx"
                    },
                    {
                      name = "sz",
                      type = "number",
                      description = "The z component of the scale to apply.",
                      default = "sx"
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The original matrix."
                    }
                  }
                }
              }
            },
            {
              name = "set",
              summary = "Set the components of the matrix.",
              description = "Sets the components of the matrix from separate position, rotation, and scale arguments or an existing matrix.",
              key = "Mat4:set",
              module = "lovr.math",
              related = {
                "Mat4:unpack"
              },
              variants = {
                {
                  description = "Resets the matrix to the identity matrix.",
                  arguments = {},
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The input matrix."
                    }
                  }
                },
                {
                  description = "Copies the values from an existing matrix.",
                  arguments = {
                    {
                      name = "n",
                      type = "mat4",
                      description = "An existing matrix to copy the values from."
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The input matrix."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The translation of the matrix.",
                      default = "0, 0, 0"
                    },
                    {
                      name = "scale",
                      type = "Vec3",
                      description = "The scale of the matrix.",
                      default = "1, 1, 1"
                    },
                    {
                      name = "rotation",
                      type = "Quat",
                      description = "The rotation of the matrix.",
                      default = "0, 0, 0, 1"
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The input matrix."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "position",
                      type = "Vec3",
                      description = "The translation of the matrix.",
                      default = "0, 0, 0"
                    },
                    {
                      name = "rotation",
                      type = "Quat",
                      description = "The rotation of the matrix.",
                      default = "0, 0, 0, 1"
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The input matrix."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "...",
                      type = "number",
                      description = "16 numbers to use as the raw values of the matrix (column-major)."
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The input matrix."
                    }
                  }
                },
                {
                  description = "Sets the diagonal values to a number and everything else to 0.",
                  arguments = {
                    {
                      name = "d",
                      type = "number",
                      description = "A number to use for the diagonal elements."
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The input matrix."
                    }
                  }
                }
              }
            },
            {
              name = "target",
              summary = "Create a model transform that targets from a position to target position.",
              description = "Sets a model transform matrix that moves to `from` and orients model towards `to` point.\n\nThis is used when rendered model should always point towards a point of interest. The resulting Mat4 object can be used as model pose.\n\nThe target() function produces same result as lookAt() after matrix inversion.",
              key = "Mat4:target",
              module = "lovr.math",
              related = {
                "Mat4:lookAt",
                "Quat:direction"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "from",
                      type = "Vec3",
                      description = "The position of the viewer."
                    },
                    {
                      name = "to",
                      type = "Vec3",
                      description = "The position of the target."
                    },
                    {
                      name = "up",
                      type = "Vec3",
                      description = "The up vector of the viewer.",
                      default = "Vec3(0, 1, 0)"
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The original matrix."
                    }
                  }
                }
              }
            },
            {
              name = "translate",
              summary = "Translate the matrix.",
              description = "Translates the matrix.",
              key = "Mat4:translate",
              module = "lovr.math",
              related = {
                "Mat4:rotate",
                "Mat4:scale",
                "Mat4:identity"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The translation vector."
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The original matrix."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the translation."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the translation."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z component of the translation."
                    }
                  },
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The original matrix."
                    }
                  }
                }
              }
            },
            {
              name = "transpose",
              summary = "Transpose the matrix.",
              description = "Transposes the matrix, mirroring its values along the diagonal.",
              key = "Mat4:transpose",
              module = "lovr.math",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The original matrix."
                    }
                  }
                }
              }
            },
            {
              name = "unpack",
              summary = "Get the individual components of the matrix.",
              description = "Returns the components of matrix, either as 10 separated numbers representing the position, scale, and rotation, or as 16 raw numbers representing the individual components of the matrix in column-major order.",
              key = "Mat4:unpack",
              module = "lovr.math",
              related = {
                "Mat4:set"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "raw",
                      type = "boolean",
                      description = "Whether to return the 16 raw components."
                    }
                  },
                  returns = {
                    {
                      name = "...",
                      type = "number",
                      description = "The requested components of the matrix."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Quat",
          summary = "A quaternion.",
          description = "A `quat` is a math type that represents a 3D rotation, stored as four numbers.",
          key = "Quat",
          module = "lovr.math",
          constructors = {
            "lovr.math.newQuat",
            "lovr.math.quat"
          },
          related = {
            "Vec3",
            "Mat4"
          },
          methods = {
            {
              name = "conjugate",
              summary = "Conjugate (invert) the quaternion.",
              description = "Conjugates the input quaternion in place, returning the input.  If the quaternion is normalized, this is the same as inverting it.  It negates the (x, y, z) components of the quaternion.",
              key = "Quat:conjugate",
              module = "lovr.math",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "q",
                      type = "Quat",
                      description = "The original quaternion."
                    }
                  }
                }
              }
            },
            {
              name = "direction",
              summary = "Get the direction of the quaternion.",
              description = "Creates a new temporary vec3 facing the forward direction, rotates it by this quaternion, and returns the vector.",
              key = "Quat:direction",
              module = "lovr.math",
              related = {
                "Mat4:lookAt"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The direction vector."
                    }
                  }
                }
              }
            },
            {
              name = "equals",
              summary = "Check if a quaternion equals another quaternion.",
              description = "Returns whether a quaternion is approximately equal to another quaternion.",
              key = "Quat:equals",
              module = "lovr.math",
              related = {
                "Vec2:equals",
                "Vec3:equals",
                "Vec4:equals",
                "Mat4:equals"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "r",
                      type = "Quat",
                      description = "The other quaternion."
                    }
                  },
                  returns = {
                    {
                      name = "equal",
                      type = "boolean",
                      description = "Whether the 2 quaternions approximately equal each other."
                    }
                  }
                }
              }
            },
            {
              name = "length",
              summary = "Get the length of the quaternion.",
              description = "Returns the length of the quaternion.",
              key = "Quat:length",
              module = "lovr.math",
              related = {
                "Quat:normalize"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "length",
                      type = "number",
                      description = "The length of the quaternion."
                    }
                  }
                }
              }
            },
            {
              name = "mul",
              summary = "Multiply a quaternion by another quaternion or a vector.",
              description = "Multiplies this quaternion by another value.  If the value is a quaternion, the rotations in the two quaternions are applied sequentially and the result is stored in the first quaternion.  If the value is a vector, then the input vector is rotated by the quaternion and returned.",
              key = "Quat:mul",
              module = "lovr.math",
              variants = {
                {
                  arguments = {
                    {
                      name = "r",
                      type = "quat",
                      description = "A quaternion to combine with the original."
                    }
                  },
                  returns = {
                    {
                      name = "q",
                      type = "quat",
                      description = "The original quaternion."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "v3",
                      type = "vec3",
                      description = "A vector to rotate."
                    }
                  },
                  returns = {
                    {
                      name = "v3",
                      type = "vec3",
                      description = "Vector rotated by quaternion."
                    }
                  }
                }
              }
            },
            {
              name = "normalize",
              summary = "Normalize the length of the quaternion to 1.",
              description = "Adjusts the values in the quaternion so that its length becomes 1.",
              key = "Quat:normalize",
              module = "lovr.math",
              notes = "A common source of bugs with quaternions is to forget to normalize them after performing a series of operations on them.  Try normalizing a quaternion if some of the calculations aren't working quite right!",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "q",
                      type = "Quat",
                      description = "The original quaternion."
                    }
                  }
                }
              },
              related = {
                "Quat:length"
              }
            },
            {
              name = "set",
              summary = "Set the components of the quaternion.",
              description = "Sets the components of the quaternion.  There are lots of different ways to specify the new components, the summary is:\n\n- Four numbers can be used to specify an angle/axis rotation, similar to other LÖVR functions.\n- Four numbers plus the fifth `raw` flag can be used to set the raw values of the quaternion.\n- An existing quaternion can be passed in to copy its values.\n- A single direction vector can be specified to turn its direction (relative to the default\n  forward direction of \"negative z\") into a rotation.\n- Two direction vectors can be specified to set the quaternion equal to the rotation between the\n  two vectors.\n- A matrix can be passed in to extract the rotation of the matrix into a quaternion.",
              key = "Quat:set",
              module = "lovr.math",
              related = {
                "Quat:unpack"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "angle",
                      description = "The angle to use for the rotation, in radians.",
                      default = "0"
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation.",
                      default = "0"
                    },
                    {
                      name = "raw",
                      type = "boolean",
                      description = "Whether the components should be interpreted as raw `(x, y, z, w)` components.",
                      default = "false"
                    }
                  },
                  returns = {
                    {
                      name = "q",
                      type = "quat",
                      description = "The original quaternion."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "r",
                      type = "quat",
                      description = "An existing quaternion to copy the values from."
                    }
                  },
                  returns = {
                    {
                      name = "q",
                      type = "quat",
                      description = "The original quaternion."
                    }
                  }
                },
                {
                  description = "Sets the values from a direction vector.",
                  arguments = {
                    {
                      name = "v",
                      type = "vec3",
                      description = "A normalized direction vector."
                    }
                  },
                  returns = {
                    {
                      name = "q",
                      type = "quat",
                      description = "The original quaternion."
                    }
                  }
                },
                {
                  description = "Sets the values to represent the rotation between two vectors.",
                  arguments = {
                    {
                      name = "v",
                      type = "vec3",
                      description = "A normalized direction vector."
                    },
                    {
                      name = "u",
                      type = "vec3",
                      description = "Another normalized direction vector."
                    }
                  },
                  returns = {
                    {
                      name = "q",
                      type = "quat",
                      description = "The original quaternion."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "m",
                      type = "mat4",
                      description = "A matrix to use the rotation from."
                    }
                  },
                  returns = {
                    {
                      name = "q",
                      type = "quat",
                      description = "The original quaternion."
                    }
                  }
                },
                {
                  description = "Reset the quaternion to the identity (0, 0, 0, 1).",
                  arguments = {},
                  returns = {
                    {
                      name = "q",
                      type = "quat",
                      description = "The original quaternion."
                    }
                  }
                }
              }
            },
            {
              name = "slerp",
              summary = "Moves this quaternion some amount towards another one.",
              description = "Performs a spherical linear interpolation between this quaternion and another one, which can be used for smoothly animating between two rotations.\n\nThe amount of interpolation is controlled by a parameter `t`.  A `t` value of zero leaves the original quaternion unchanged, whereas a `t` of one sets the original quaternion exactly equal to the target.  A value between `0` and `1` returns a rotation between the two based on the value.",
              key = "Quat:slerp",
              module = "lovr.math",
              related = {
                "Vec3:lerp"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "r",
                      type = "Quat",
                      description = "The quaternion to slerp towards."
                    },
                    {
                      name = "t",
                      type = "number",
                      description = "The lerping parameter."
                    }
                  },
                  returns = {
                    {
                      name = "q",
                      type = "Quat",
                      description = "The original quaternion, containing the new lerped values."
                    }
                  }
                }
              }
            },
            {
              name = "unpack",
              summary = "Get the components of the quaternion.",
              description = "Returns the components of the quaternion as numbers, either in an angle/axis representation or as raw quaternion values.",
              key = "Quat:unpack",
              module = "lovr.math",
              related = {
                "Quat:set"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "raw",
                      type = "boolean",
                      description = "Whether the values should be returned as raw values instead of angle/axis.",
                      default = "false"
                    }
                  },
                  returns = {
                    {
                      name = "a",
                      type = "number",
                      description = "The angle in radians, or the x value."
                    },
                    {
                      name = "b",
                      type = "number",
                      description = "The x component of the rotation axis or the y value."
                    },
                    {
                      name = "c",
                      type = "number",
                      description = "The y component of the rotation axis or the z value."
                    },
                    {
                      name = "d",
                      type = "number",
                      description = "The z component of the rotation axis or the w value."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "RandomGenerator",
          summary = "A pseudo-random number generator.",
          description = "A RandomGenerator is a standalone object that can be used to independently generate pseudo-random numbers. If you just need basic randomness, you can use `lovr.math.random` without needing to create a random generator.",
          key = "RandomGenerator",
          module = "lovr.math",
          constructors = {
            "lovr.math.newRandomGenerator"
          },
          methods = {
            {
              name = "getSeed",
              summary = "Get the seed value of the RandomGenerator.",
              description = "Returns the seed used to initialize the RandomGenerator.",
              key = "RandomGenerator:getSeed",
              module = "lovr.math",
              notes = "Since the seed is a 64 bit integer, each 32 bits of the seed are returned separately to avoid precision issues.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "low",
                      type = "number",
                      description = "The lower 32 bits of the seed."
                    },
                    {
                      name = "high",
                      type = "number",
                      description = "The upper 32 bits of the seed."
                    }
                  }
                }
              },
              related = {
                "lovr.math.newRandomGenerator"
              }
            },
            {
              name = "getState",
              summary = "Get the current state of the RandomGenerator.",
              description = "Returns the current state of the RandomGenerator.  This can be used with `RandomGenerator:setState` to reliably restore a previous state of the generator.",
              key = "RandomGenerator:getState",
              module = "lovr.math",
              notes = "The seed represents the starting state of the RandomGenerator, whereas the state represents the current state of the generator.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "state",
                      type = "string",
                      description = "The serialized state."
                    }
                  }
                }
              }
            },
            {
              name = "random",
              summary = "Get a random number.",
              description = "Returns the next uniformly distributed pseudo-random number from the RandomGenerator's sequence.",
              key = "RandomGenerator:random",
              module = "lovr.math",
              related = {
                "lovr.math.random",
                "RandomGenerator:randomNormal"
              },
              variants = {
                {
                  description = "Generate a pseudo-random floating point number in the range `[0,1)`",
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "A pseudo-random number."
                    }
                  }
                },
                {
                  description = "Generate a pseudo-random integer in the range `[1,high]`",
                  arguments = {
                    {
                      name = "high",
                      type = "number",
                      description = "The maximum number to generate."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "A pseudo-random number."
                    }
                  }
                },
                {
                  description = "Generate a pseudo-random integer in the range `[low,high]`",
                  arguments = {
                    {
                      name = "low",
                      type = "number",
                      description = "The minimum number to generate."
                    },
                    {
                      name = "high",
                      type = "number",
                      description = "The maximum number to generate."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "A pseudo-random number."
                    }
                  }
                }
              }
            },
            {
              name = "randomNormal",
              summary = "Get a random number from a normal distribution.",
              description = "Returns a pseudo-random number from a normal distribution (a bell curve).  You can control the center of the bell curve (the mean value) and the overall width (sigma, or standard deviation).",
              key = "RandomGenerator:randomNormal",
              module = "lovr.math",
              related = {
                "lovr.math.randomNormal",
                "RandomGenerator:random"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "sigma",
                      type = "number",
                      description = "The standard deviation of the distribution.  This can be thought of how \"wide\" the range of numbers is or how much variability there is.",
                      default = "1"
                    },
                    {
                      name = "mu",
                      type = "number",
                      description = "The average value returned.",
                      default = "0"
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "A normally distributed pseudo-random number."
                    }
                  }
                }
              }
            },
            {
              name = "setSeed",
              summary = "Reinitialize the RandomGenerator with a new seed.",
              description = "Seed the RandomGenerator with a new seed.  Each seed will cause the RandomGenerator to produce a unique sequence of random numbers.",
              key = "RandomGenerator:setSeed",
              module = "lovr.math",
              notes = "For precise 64 bit seeds, you should specify the lower and upper 32 bits of the seed separately. Otherwise, seeds larger than 2^53 will start to lose precision.",
              variants = {
                {
                  arguments = {
                    {
                      name = "seed",
                      type = "number",
                      description = "The random seed."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "low",
                      type = "number",
                      description = "The lower 32 bits of the seed."
                    },
                    {
                      name = "high",
                      type = "number",
                      description = "The upper 32 bits of the seed."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setState",
              summary = "Set the state of the RandomGenerator.",
              description = "Sets the state of the RandomGenerator, as previously obtained using `RandomGenerator:getState`. This can be used to reliably restore a previous state of the generator.",
              key = "RandomGenerator:setState",
              module = "lovr.math",
              notes = "The seed represents the starting state of the RandomGenerator, whereas the state represents the current state of the generator.",
              variants = {
                {
                  arguments = {
                    {
                      name = "state",
                      type = "string",
                      description = "The serialized state."
                    }
                  },
                  returns = {}
                }
              }
            }
          }
        },
        {
          name = "Vec2",
          summary = "A 2D vector.",
          description = "A vector object that holds two numbers.",
          key = "Vec2",
          module = "lovr.math",
          constructors = {
            "lovr.math.newVec2",
            "lovr.math.vec2"
          },
          related = {
            "Vec3",
            "Vec4"
          },
          methods = {
            {
              name = "add",
              summary = "Add a vector or a number to the vector.",
              description = "Adds a vector or a number to the vector.",
              key = "Vec2:add",
              module = "lovr.math",
              related = {
                "Vec2:sub",
                "Vec2:mul",
                "Vec2:div"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec2",
                      description = "The other vector."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec2",
                      description = "The original vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value to add to x component."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value to add to y component.",
                      default = "x"
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec2",
                      description = "The original vector."
                    }
                  }
                }
              }
            },
            {
              name = "angle",
              summary = "Get the angle to another vector.",
              description = "Returns the angle between vectors.",
              key = "Vec2:angle",
              module = "lovr.math",
              notes = "If any of the two vectors have a length of zero, the angle between them is not well defined.  In this case the function returns `math.pi / 2`.",
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec2",
                      description = "The other vector."
                    }
                  },
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The angle to the other vector, in radians."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the other vector."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the other vector."
                    }
                  },
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The angle to the other vector, in radians."
                    }
                  }
                }
              },
              related = {
                "Vec2:distance",
                "Vec2:length"
              }
            },
            {
              name = "distance",
              summary = "Get the distance to another vector.",
              description = "Returns the distance to another vector.",
              key = "Vec2:distance",
              module = "lovr.math",
              related = {
                "Vec2:angle",
                "Vec2:length"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec2",
                      description = "The vector to measure the distance to."
                    }
                  },
                  returns = {
                    {
                      name = "distance",
                      type = "number",
                      description = "The distance to `u`."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value of x component to measure distance to."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value of y component to measure distance to."
                    }
                  },
                  returns = {
                    {
                      name = "distance",
                      type = "number",
                      description = "The distance to `u`."
                    }
                  }
                }
              }
            },
            {
              name = "div",
              summary = "Divides the vector by a vector or a number.",
              description = "Divides the vector by a vector or a number.",
              key = "Vec2:div",
              module = "lovr.math",
              related = {
                "Vec2:add",
                "Vec2:sub",
                "Vec2:mul"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec2",
                      description = "The other vector to divide the components by."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec2",
                      description = "The original vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value to divide x component by."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value to divide y component by.",
                      default = "x"
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec2",
                      description = "The original vector."
                    }
                  }
                }
              }
            },
            {
              name = "dot",
              summary = "Get the dot product with another vector.",
              description = "Returns the dot product between this vector and another one.",
              key = "Vec2:dot",
              module = "lovr.math",
              notes = "This is computed as:\n\n    dot = v.x * u.x + v.y * u.y\n\nThe vectors are not normalized before computing the dot product.",
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec2",
                      description = "The vector to compute the dot product with."
                    }
                  },
                  returns = {
                    {
                      name = "dot",
                      type = "number",
                      description = "The dot product between `v` and `u`."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value of x component to compute the dot product with."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value of y component to compute the dot product with."
                    }
                  },
                  returns = {
                    {
                      name = "dot",
                      type = "number",
                      description = "The dot product between `v` and `u`."
                    }
                  }
                }
              }
            },
            {
              name = "equals",
              summary = "Check if a vector equals another vector.",
              description = "Returns whether a vector is approximately equal to another vector.",
              key = "Vec2:equals",
              module = "lovr.math",
              notes = "To handle floating point precision issues, this function returns true as long as the squared distance between the vectors is below `1e-10`.",
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec2",
                      description = "The other vector."
                    }
                  },
                  returns = {
                    {
                      name = "equal",
                      type = "boolean",
                      description = "Whether the 2 vectors approximately equal each other."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the other vector."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the other vector."
                    }
                  },
                  returns = {
                    {
                      name = "equal",
                      type = "boolean",
                      description = "Whether the 2 vectors approximately equal each other."
                    }
                  }
                }
              },
              related = {
                "Vec3:equals",
                "Vec4:equals",
                "Quat:equals",
                "Mat4:equals"
              }
            },
            {
              name = "length",
              summary = "Get the length of the vector.",
              description = "Returns the length of the vector.",
              key = "Vec2:length",
              module = "lovr.math",
              notes = "The length is equivalent to this:\n\n    math.sqrt(v.x * v.x + v.y * v.y)",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "length",
                      type = "number",
                      description = "The length of the vector."
                    }
                  }
                }
              },
              related = {
                "Vec2:normalize",
                "Vec2:distance"
              }
            },
            {
              name = "lerp",
              summary = "Moves this vector some amount towards another one.",
              description = "Performs a linear interpolation between this vector and another one, which can be used to smoothly animate between two vectors, based on a parameter value.  A parameter value of `0` will leave the vector unchanged, a parameter value of `1` will set the vector to be equal to the input vector, and a value of `.5` will set the components to be halfway between the two vectors.",
              key = "Vec2:lerp",
              module = "lovr.math",
              related = {
                "Quat:slerp"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec2",
                      description = "The vector to lerp towards."
                    },
                    {
                      name = "t",
                      type = "number",
                      description = "The lerping parameter."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec2",
                      description = "The original vector, containing the new lerped values."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value of x component to lerp towards."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value of y component to lerp towards."
                    },
                    {
                      name = "t",
                      type = "number",
                      description = "The lerping parameter."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec2",
                      description = "The original vector, containing the new lerped values."
                    }
                  }
                }
              }
            },
            {
              name = "mul",
              summary = "Multiply the vector by a vector or a number.",
              description = "Multiplies the vector by a vector or a number.",
              key = "Vec2:mul",
              module = "lovr.math",
              related = {
                "Vec2:add",
                "Vec2:sub",
                "Vec2:div"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec2",
                      description = "The other vector to multiply the components by."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec2",
                      description = "The original vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value to multiply x component by."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value to multiply y component by.",
                      default = "x"
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec2",
                      description = "The original vector."
                    }
                  }
                }
              }
            },
            {
              name = "normalize",
              summary = "Normalize the length of the vector to 1.",
              description = "Adjusts the values in the vector so that its direction stays the same but its length becomes 1.",
              key = "Vec2:normalize",
              module = "lovr.math",
              related = {
                "Vec2:length"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "v",
                      type = "Vec2",
                      description = "The original vector."
                    }
                  }
                }
              }
            },
            {
              name = "set",
              summary = "Set the components of the vector.",
              description = "Sets the components of the vector, either from numbers or an existing vector.",
              key = "Vec2:set",
              module = "lovr.math",
              related = {
                "Vec2:unpack"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The new x value of the vector.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The new y value of the vector.",
                      default = "x"
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec2",
                      description = "The input vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec2",
                      description = "The vector to copy the values from."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec2",
                      description = "The input vector."
                    }
                  }
                }
              }
            },
            {
              name = "sub",
              summary = "Subtract a vector or a number from the vector.",
              description = "Subtracts a vector or a number from the vector.",
              key = "Vec2:sub",
              module = "lovr.math",
              related = {
                "Vec2:add",
                "Vec2:mul",
                "Vec2:div"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec2",
                      description = "The other vector."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec2",
                      description = "The original vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value to subtract from x component."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value to subtract from y component.",
                      default = "x"
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec2",
                      description = "The original vector."
                    }
                  }
                }
              }
            },
            {
              name = "unpack",
              summary = "Get the components of the vector.",
              description = "Returns the 2 components of the vector as numbers.",
              key = "Vec2:unpack",
              module = "lovr.math",
              related = {
                "Vec2:set"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x value."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y value."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Vec3",
          summary = "A 3D vector.",
          description = "A vector object that holds three numbers.",
          key = "Vec3",
          module = "lovr.math",
          constructors = {
            "lovr.math.newVec3",
            "lovr.math.vec3"
          },
          related = {
            "Vec2",
            "Vec4"
          },
          methods = {
            {
              name = "add",
              summary = "Add a vector or a number to the vector.",
              description = "Adds a vector or a number to the vector.",
              key = "Vec3:add",
              module = "lovr.math",
              related = {
                "Vec3:sub",
                "Vec3:mul",
                "Vec3:div"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec3",
                      description = "The other vector."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The original vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value to add to x component."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value to add to y component.",
                      default = "x"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "A value to add to z component.",
                      default = "x"
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The original vector."
                    }
                  }
                }
              }
            },
            {
              name = "angle",
              summary = "Get the angle to another vector.",
              description = "Returns the angle between vectors.",
              key = "Vec3:angle",
              module = "lovr.math",
              notes = "If any of the two vectors have a length of zero, the angle between them is not well defined.  In this case the function returns `math.pi / 2`.",
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec3",
                      description = "The other vector."
                    }
                  },
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The angle to the other vector, in radians."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the other vector."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the other vector."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z component of the other vector."
                    }
                  },
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The angle to the other vector, in radians."
                    }
                  }
                }
              },
              related = {
                "Vec3:distance",
                "Vec3:length"
              }
            },
            {
              name = "cross",
              summary = "Get the cross product with another vector.",
              description = "Sets this vector to be equal to the cross product between this vector and another one.  The new `v` will be perpendicular to both the old `v` and `u`.",
              key = "Vec3:cross",
              module = "lovr.math",
              notes = "The vectors are not normalized before or after computing the cross product.",
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec3",
                      description = "The vector to compute the cross product with."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The original vector, with the cross product as its values."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value of x component to compute cross product with."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value of y component to compute cross product with."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "A value of z component to compute cross product with."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The original vector, with the cross product as its values."
                    }
                  }
                }
              },
              related = {
                "Vec3:dot"
              }
            },
            {
              name = "distance",
              summary = "Get the distance to another vector.",
              description = "Returns the distance to another vector.",
              key = "Vec3:distance",
              module = "lovr.math",
              related = {
                "Vec3:angle",
                "Vec3:length"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec3",
                      description = "The vector to measure the distance to."
                    }
                  },
                  returns = {
                    {
                      name = "distance",
                      type = "number",
                      description = "The distance to `u`."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value of x component to measure distance to."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value of y component to measure distance to."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "A value of z component to measure distance to."
                    }
                  },
                  returns = {
                    {
                      name = "distance",
                      type = "number",
                      description = "The distance to `u`."
                    }
                  }
                }
              }
            },
            {
              name = "div",
              summary = "Divides the vector by a vector or a number.",
              description = "Divides the vector by a vector or a number.",
              key = "Vec3:div",
              module = "lovr.math",
              related = {
                "Vec3:add",
                "Vec3:sub",
                "Vec3:mul"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec3",
                      description = "The other vector to divide the components by."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The original vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value to divide x component by."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value to divide y component by.",
                      default = "x"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "A value to divide z component by.",
                      default = "x"
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The original vector."
                    }
                  }
                }
              }
            },
            {
              name = "dot",
              summary = "Get the dot product with another vector.",
              description = "Returns the dot product between this vector and another one.",
              key = "Vec3:dot",
              module = "lovr.math",
              notes = "This is computed as:\n\n    dot = v.x * u.x + v.y * u.y + v.z * u.z\n\nThe vectors are not normalized before computing the dot product.",
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec3",
                      description = "The vector to compute the dot product with."
                    }
                  },
                  returns = {
                    {
                      name = "dot",
                      type = "number",
                      description = "The dot product between `v` and `u`."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value of x component to compute the dot product with."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value of y component to compute the dot product with."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "A value of z component to compute the dot product with."
                    }
                  },
                  returns = {
                    {
                      name = "dot",
                      type = "number",
                      description = "The dot product between `v` and `u`."
                    }
                  }
                }
              },
              related = {
                "Vec3:cross"
              }
            },
            {
              name = "equals",
              summary = "Check if a vector equals another vector.",
              description = "Returns whether a vector is approximately equal to another vector.",
              key = "Vec3:equals",
              module = "lovr.math",
              notes = "To handle floating point precision issues, this function returns true as long as the squared distance between the vectors is below `1e-10`.",
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec3",
                      description = "The other vector."
                    }
                  },
                  returns = {
                    {
                      name = "equal",
                      type = "boolean",
                      description = "Whether the 2 vectors approximately equal each other."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the other vector."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the other vector."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z component of the other vector."
                    }
                  },
                  returns = {
                    {
                      name = "equal",
                      type = "boolean",
                      description = "Whether the 2 vectors approximately equal each other."
                    }
                  }
                }
              },
              related = {
                "Vec2:equals",
                "Vec4:equals",
                "Quat:equals",
                "Mat4:equals"
              }
            },
            {
              name = "length",
              summary = "Get the length of the vector.",
              description = "Returns the length of the vector.",
              key = "Vec3:length",
              module = "lovr.math",
              notes = "The length is equivalent to this:\n\n    math.sqrt(v.x * v.x + v.y * v.y + v.z * v.z)",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "length",
                      type = "number",
                      description = "The length of the vector."
                    }
                  }
                }
              },
              related = {
                "Vec3:normalize",
                "Vec3:distance"
              }
            },
            {
              name = "lerp",
              summary = "Moves this vector some amount towards another one.",
              description = "Performs a linear interpolation between this vector and another one, which can be used to smoothly animate between two vectors, based on a parameter value.  A parameter value of `0` will leave the vector unchanged, a parameter value of `1` will set the vector to be equal to the input vector, and a value of `.5` will set the components to be halfway between the two vectors.",
              key = "Vec3:lerp",
              module = "lovr.math",
              related = {
                "Quat:slerp"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec3",
                      description = "The vector to lerp towards."
                    },
                    {
                      name = "t",
                      type = "number",
                      description = "The lerping parameter."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The original vector, containing the new lerped values."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value of x component to lerp towards."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value of y component to lerp towards."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "A value of z component to lerp towards."
                    },
                    {
                      name = "t",
                      type = "number",
                      description = "The lerping parameter."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The original vector, containing the new lerped values."
                    }
                  }
                }
              }
            },
            {
              name = "mul",
              summary = "Multiply the vector by a vector or a number.",
              description = "Multiplies the vector by a vector or a number.",
              key = "Vec3:mul",
              module = "lovr.math",
              related = {
                "Vec3:add",
                "Vec3:sub",
                "Vec3:div"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec3",
                      description = "The other vector to multiply the components by."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The original vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value to multiply x component by."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value to multiply y component by.",
                      default = "x"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "A value to multiply z component by.",
                      default = "x"
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The original vector."
                    }
                  }
                }
              }
            },
            {
              name = "normalize",
              summary = "Normalize the length of the vector to 1.",
              description = "Adjusts the values in the vector so that its direction stays the same but its length becomes 1.",
              key = "Vec3:normalize",
              module = "lovr.math",
              related = {
                "Vec3:length"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The original vector."
                    }
                  }
                }
              }
            },
            {
              name = "set",
              summary = "Set the components of the vector.",
              description = "Sets the components of the vector, either from numbers or an existing vector.",
              key = "Vec3:set",
              module = "lovr.math",
              related = {
                "Vec3:unpack"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The new x value of the vector.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The new y value of the vector.",
                      default = "x"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The new z value of the vector.",
                      default = "x"
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The input vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec3",
                      description = "The vector to copy the values from."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The input vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "m",
                      type = "Mat4",
                      description = "The matrix to use the position of."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The input vector."
                    }
                  }
                }
              }
            },
            {
              name = "sub",
              summary = "Subtract a vector or a number from the vector.",
              description = "Subtracts a vector or a number from the vector.",
              key = "Vec3:sub",
              module = "lovr.math",
              related = {
                "Vec3:add",
                "Vec3:mul",
                "Vec3:div"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec3",
                      description = "The other vector."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The original vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value to subtract from x component."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value to subtract from y component.",
                      default = "x"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "A value to subtract from z component.",
                      default = "x"
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec3",
                      description = "The original vector."
                    }
                  }
                }
              }
            },
            {
              name = "unpack",
              summary = "Get the components of the vector.",
              description = "Returns the 3 components of the vector as numbers.",
              key = "Vec3:unpack",
              module = "lovr.math",
              related = {
                "Vec3:set"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x value."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y value."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z value."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Vec4",
          summary = "A 4D vector.",
          description = "A vector object that holds four numbers.",
          key = "Vec4",
          module = "lovr.math",
          constructors = {
            "lovr.math.newVec4",
            "lovr.math.vec4"
          },
          related = {
            "Vec2",
            "Vec3"
          },
          methods = {
            {
              name = "add",
              summary = "Add a vector or a number to the vector.",
              description = "Adds a vector or a number to the vector.",
              key = "Vec4:add",
              module = "lovr.math",
              related = {
                "Vec4:sub",
                "Vec4:mul",
                "Vec4:div"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec4",
                      description = "The other vector."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec4",
                      description = "The original vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value to add to x component."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value to add to y component.",
                      default = "x"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "A value to add to z component.",
                      default = "x"
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "A value to add to w component.",
                      default = "x"
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec4",
                      description = "The original vector."
                    }
                  }
                }
              }
            },
            {
              name = "angle",
              summary = "Get the angle to another vector.",
              description = "Returns the angle between vectors.",
              key = "Vec4:angle",
              module = "lovr.math",
              notes = "If any of the two vectors have a length of zero, the angle between them is not well defined.  In this case the function returns `math.pi / 2`.",
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec4",
                      description = "The other vector."
                    }
                  },
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The angle to other vector, in radians."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the other vector."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the other vector."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z component of the other vector."
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "The w component of the other vector."
                    }
                  },
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The angle to other vector, in radians."
                    }
                  }
                }
              },
              related = {
                "Vec4:distance",
                "Vec4:length"
              }
            },
            {
              name = "distance",
              summary = "Get the distance to another vector.",
              description = "Returns the distance to another vector.",
              key = "Vec4:distance",
              module = "lovr.math",
              related = {
                "Vec4:angle",
                "Vec4:length"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec4",
                      description = "The vector to measure the distance to."
                    }
                  },
                  returns = {
                    {
                      name = "distance",
                      type = "number",
                      description = "The distance to `u`."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value of x component to measure distance to."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value of y component to measure distance to."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "A value of z component to measure distance to."
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "A value of w component to measure distance to."
                    }
                  },
                  returns = {
                    {
                      name = "distance",
                      type = "number",
                      description = "The distance to `u`."
                    }
                  }
                }
              }
            },
            {
              name = "div",
              summary = "Divides the vector by a vector or a number.",
              description = "Divides the vector by a vector or a number.",
              key = "Vec4:div",
              module = "lovr.math",
              related = {
                "Vec4:add",
                "Vec4:sub",
                "Vec4:mul"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec4",
                      description = "The other vector to divide the components by."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec4",
                      description = "The original vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value to divide x component by."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value to divide y component by.",
                      default = "x"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "A value to divide z component by.",
                      default = "x"
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "A value to divide w component by.",
                      default = "x"
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec4",
                      description = "The original vector."
                    }
                  }
                }
              }
            },
            {
              name = "dot",
              summary = "Get the dot product with another vector.",
              description = "Returns the dot product between this vector and another one.",
              key = "Vec4:dot",
              module = "lovr.math",
              notes = "This is computed as:\n\n    dot = v.x * u.x + v.y * u.y + v.z * u.z + v.w * u.w\n\nThe vectors are not normalized before computing the dot product.",
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec4",
                      description = "The vector to compute the dot product with."
                    }
                  },
                  returns = {
                    {
                      name = "dot",
                      type = "number",
                      description = "The dot product between `v` and `u`."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value of x component to compute the dot product with."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value of y component to compute the dot product with."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "A value of z component to compute the dot product with."
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "A value of w component to compute the dot product with."
                    }
                  },
                  returns = {
                    {
                      name = "dot",
                      type = "number",
                      description = "The dot product between `v` and `u`."
                    }
                  }
                }
              }
            },
            {
              name = "equals",
              summary = "Check if a vector equals another vector.",
              description = "Returns whether a vector is approximately equal to another vector.",
              key = "Vec4:equals",
              module = "lovr.math",
              notes = "To handle floating point precision issues, this function returns true as long as the squared distance between the vectors is below `1e-10`.",
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec4",
                      description = "The other vector."
                    }
                  },
                  returns = {
                    {
                      name = "equal",
                      type = "boolean",
                      description = "Whether the 2 vectors approximately equal each other."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the other vector."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the other vector."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z component of the other vector."
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "The w component of the other vector."
                    }
                  },
                  returns = {
                    {
                      name = "equal",
                      type = "boolean",
                      description = "Whether the 2 vectors approximately equal each other."
                    }
                  }
                }
              },
              related = {
                "Vec2:equals",
                "Vec3:equals",
                "Quat:equals",
                "Mat4:equals"
              }
            },
            {
              name = "length",
              summary = "Get the length of the vector.",
              description = "Returns the length of the vector.",
              key = "Vec4:length",
              module = "lovr.math",
              notes = "The length is equivalent to this:\n\n    math.sqrt(v.x * v.x + v.y * v.y + v.z * v.z + v.w * v.w)",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "length",
                      type = "number",
                      description = "The length of the vector."
                    }
                  }
                }
              },
              related = {
                "Vec4:normalize",
                "Vec4:distance"
              }
            },
            {
              name = "lerp",
              summary = "Moves this vector some amount towards another one.",
              description = "Performs a linear interpolation between this vector and another one, which can be used to smoothly animate between two vectors, based on a parameter value.  A parameter value of `0` will leave the vector unchanged, a parameter value of `1` will set the vector to be equal to the input vector, and a value of `.5` will set the components to be halfway between the two vectors.",
              key = "Vec4:lerp",
              module = "lovr.math",
              related = {
                "Quat:slerp"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec4",
                      description = "The vector to lerp towards."
                    },
                    {
                      name = "t",
                      type = "number",
                      description = "The lerping parameter."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec4",
                      description = "The original vector, containing the new lerped values."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value of x component to lerp towards."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value of y component to lerp towards."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "A value of z component to lerp towards."
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "A value of w component to lerp towards."
                    },
                    {
                      name = "t",
                      type = "number",
                      description = "The lerping parameter."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec4",
                      description = "The original vector, containing the new lerped values."
                    }
                  }
                }
              }
            },
            {
              name = "mul",
              summary = "Multiply the vector by a vector or a number.",
              description = "Multiplies the vector by a vector or a number.",
              key = "Vec4:mul",
              module = "lovr.math",
              related = {
                "Vec4:add",
                "Vec4:sub",
                "Vec4:div"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec4",
                      description = "The other vector to multiply the components by."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec4",
                      description = "The original vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value to multiply x component by."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value to multiply y component by.",
                      default = "x"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "A value to multiply z component by.",
                      default = "x"
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "A value to multiply w component by.",
                      default = "x"
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec4",
                      description = "The original vector."
                    }
                  }
                }
              }
            },
            {
              name = "normalize",
              summary = "Normalize the length of the vector to 1.",
              description = "Adjusts the values in the vector so that its direction stays the same but its length becomes 1.",
              key = "Vec4:normalize",
              module = "lovr.math",
              related = {
                "Vec4:length"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "v",
                      type = "Vec4",
                      description = "The original vector."
                    }
                  }
                }
              }
            },
            {
              name = "set",
              summary = "Set the components of the vector.",
              description = "Sets the components of the vector, either from numbers or an existing vector.",
              key = "Vec4:set",
              module = "lovr.math",
              related = {
                "Vec4:unpack"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The new x value of the vector.",
                      default = "0"
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The new y value of the vector.",
                      default = "x"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The new z value of the vector.",
                      default = "x"
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "The new w value of the vector.",
                      default = "x"
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec4",
                      description = "The input vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec4",
                      description = "The vector to copy the values from."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec4",
                      description = "The input vector."
                    }
                  }
                }
              }
            },
            {
              name = "sub",
              summary = "Subtract a vector or a number from the vector.",
              description = "Subtracts a vector or a number from the vector.",
              key = "Vec4:sub",
              module = "lovr.math",
              related = {
                "Vec4:add",
                "Vec4:mul",
                "Vec4:div"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "u",
                      type = "Vec4",
                      description = "The other vector."
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec4",
                      description = "The original vector."
                    }
                  }
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "A value to subtract from x component."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "A value to subtract from y component.",
                      default = "x"
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "A value to subtract from z component.",
                      default = "x"
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "A value to subtract from w component.",
                      default = "x"
                    }
                  },
                  returns = {
                    {
                      name = "v",
                      type = "Vec4",
                      description = "The original vector."
                    }
                  }
                }
              }
            },
            {
              name = "unpack",
              summary = "Get the components of the vector.",
              description = "Returns the 4 components of the vector as numbers.",
              key = "Vec4:unpack",
              module = "lovr.math",
              related = {
                "Vec4:set"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x value."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y value."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z value."
                    },
                    {
                      name = "w",
                      type = "number",
                      description = "The w value."
                    }
                  }
                }
              }
            }
          }
        },
        {
          name = "Vectors",
          summary = "What is your vector victor.",
          description = "LÖVR has math objects for vectors, matrices, and quaternions, collectively called \"vector objects\".  Vectors are useful because they can represent a multidimensional quantity (like a 3D position) using just a single value.",
          key = "Vectors",
          module = "lovr.math",
          constructors = {
            "lovr.math.vec2",
            "lovr.math.vec3",
            "lovr.math.vec4",
            "lovr.math.quat",
            "lovr.math.mat4",
            "lovr.math.newVec2",
            "lovr.math.newVec3",
            "lovr.math.newVec4",
            "lovr.math.newQuat",
            "lovr.math.newMat4"
          },
          notes = "Most LÖVR functions that accept positions, orientations, transforms, velocities, etc. also accept vector objects, so they can be used interchangeably with numbers:\n\n    function lovr.draw()\n      -- position and size are vec3's, rotation is a quat\n      lovr.graphics.box('fill', position, size, rotation)\n    end\n\n### Temporary vs. Permanent\n\nVectors can be created in two different ways: **permanent** and **temporary**.\n\n**Permanent** vectors behave like normal Lua values.  They are individual objects that are garbage collected when no longer needed.  They're created using the usual `lovr.math.new<Type>` syntax:\n\n    self.position = lovr.math.newVec3(x, y, z)\n\n**Temporary** vectors are created from a shared pool of vector objects.  This makes them faster because they use temporary memory and do not need to be garbage collected.  To make a temporary vector, leave off the `new` prefix:\n\n    local position = lovr.math.vec3(x, y, z)\n\nAs a further shorthand, these vector constructors are placed on the global scope.  If you prefer to keep the global scope clean, this can be configured using the `t.math.globals` flag in `lovr.conf`.\n\n    local position = vec3(x1, y1, z1) + vec3(x2, y2, z2)\n\nTemporary vectors, with all their speed, come with an important restriction: they can only be used during the frame in which they were created.  Saving them into variables and using them later on will throw an error:\n\n    local position = vec3(1, 2, 3)\n\n    function lovr.update(dt)\n      -- Reusing a temporary vector across frames will error:\n      position:add(vec3(dt))\n    end\n\nIt's possible to overflow the temporary vector pool.  If that happens, `lovr.math.drain` can be used to periodically drain the pool, invalidating any existing temporary vectors.\n\n### Metamethods\n\nVectors have metamethods, allowing them to be used using the normal math operators like `+`, `-`, `*`, `/`, etc.\n\n    print(vec3(2, 4, 6) * .5 + vec3(10, 20, 30))\n\nThese metamethods will create new temporary vectors.\n\n### Components and Swizzles\n\nThe raw components of a vector can be accessed like normal fields:\n\n    print(vec3(1, 2, 3).z) --> 3\n    print(mat4()[16]) --> 1\n\nAlso, multiple fields can be accessed and combined into a new (temporary) vector, called swizzling:\n\n    local position = vec3(10, 5, 1)\n    print(position.xy) --> vec2(10, 5)\n    print(position.xyy) --> vec3(10, 5, 5)\n    print(position.zyxz) --> vec4(1, 5, 10, 1)\n\nThe following fields are supported for vectors:\n\n- `x`, `y`, `z`, `w`\n- `r`, `g`, `b`, `a`\n- `s`, `t`, `p`, `q`\n\nQuaternions support `x`, `y`, `z`, and `w`.\n\nMatrices use numbers for accessing individual components in \"column-major\" order.\n\nAll fields can also be assigned to.\n\n    -- Swap the components of a 2D vector\n    v.xy = v.yx\n\nThe `unpack` function can be used (on any vector type) to access all of the individual components of a vector object.  For quaternions you can choose whether you want to unpack the angle/axis representation or the raw quaternion components.  Similarly, matrices support raw unpacking as well as decomposition into translation/scale/rotation values.",
          methods = {}
        }
      },
      functions = {
        {
          name = "drain",
          tag = "vectors",
          summary = "Drain the temporary vector pool.",
          description = "Drains the temporary vector pool, invalidating existing temporary vectors.\n\nThis is called automatically at the end of each frame.",
          key = "lovr.math.drain",
          module = "lovr.math",
          variants = {
            {
              arguments = {},
              returns = {}
            }
          }
        },
        {
          name = "gammaToLinear",
          tag = "mathOther",
          summary = "	Convert a color from gamma space to linear space.",
          description = "Converts a color from gamma space to linear space.",
          key = "lovr.math.gammaToLinear",
          module = "lovr.math",
          related = {
            "lovr.math.linearToGamma"
          },
          variants = {
            {
              arguments = {
                {
                  name = "gr",
                  type = "number",
                  description = "The red component of the gamma-space color."
                },
                {
                  name = "gg",
                  type = "number",
                  description = "The green component of the gamma-space color."
                },
                {
                  name = "gb",
                  type = "number",
                  description = "The blue component of the gamma-space color."
                }
              },
              returns = {
                {
                  name = "lr",
                  type = "number",
                  description = "The red component of the resulting linear-space color."
                },
                {
                  name = "lg",
                  type = "number",
                  description = "The green component of the resulting linear-space color."
                },
                {
                  name = "lb",
                  type = "number",
                  description = "The blue component of the resulting linear-space color."
                }
              }
            },
            {
              description = "A table can also be used.",
              arguments = {
                {
                  name = "color",
                  type = "table",
                  description = "A table containing the components of a gamma-space color."
                }
              },
              returns = {
                {
                  name = "lr",
                  type = "number",
                  description = "The red component of the resulting linear-space color."
                },
                {
                  name = "lg",
                  type = "number",
                  description = "The green component of the resulting linear-space color."
                },
                {
                  name = "lb",
                  type = "number",
                  description = "The blue component of the resulting linear-space color."
                }
              }
            },
            {
              description = "Convert a single color channel.",
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The color channel to convert."
                }
              },
              returns = {
                {
                  name = "y",
                  type = "number",
                  description = "The converted color channel."
                }
              }
            }
          }
        },
        {
          name = "getRandomSeed",
          tag = "random",
          summary = "Get the random seed.",
          description = "Get the seed used to initialize the random generator.",
          key = "lovr.math.getRandomSeed",
          module = "lovr.math",
          variants = {
            {
              arguments = {},
              returns = {
                {
                  name = "seed",
                  type = "number",
                  description = "The new seed."
                }
              }
            }
          }
        },
        {
          name = "linearToGamma",
          tag = "mathOther",
          summary = "	Convert a color from linear space to gamma space.",
          description = "Converts a color from linear space to gamma space.",
          key = "lovr.math.linearToGamma",
          module = "lovr.math",
          related = {
            "lovr.math.gammaToLinear"
          },
          variants = {
            {
              arguments = {
                {
                  name = "lr",
                  type = "number",
                  description = "The red component of the linear-space color."
                },
                {
                  name = "lg",
                  type = "number",
                  description = "The green component of the linear-space color."
                },
                {
                  name = "lb",
                  type = "number",
                  description = "The blue component of the linear-space color."
                }
              },
              returns = {
                {
                  name = "gr",
                  type = "number",
                  description = "The red component of the resulting gamma-space color."
                },
                {
                  name = "gg",
                  type = "number",
                  description = "The green component of the resulting gamma-space color."
                },
                {
                  name = "gb",
                  type = "number",
                  description = "The blue component of the resulting gamma-space color."
                }
              }
            },
            {
              description = "A table can also be used.",
              arguments = {
                {
                  name = "color",
                  type = "table",
                  description = "A table containing the components of a linear-space color."
                }
              },
              returns = {
                {
                  name = "gr",
                  type = "number",
                  description = "The red component of the resulting gamma-space color."
                },
                {
                  name = "gg",
                  type = "number",
                  description = "The green component of the resulting gamma-space color."
                },
                {
                  name = "gb",
                  type = "number",
                  description = "The blue component of the resulting gamma-space color."
                }
              }
            },
            {
              description = "Convert a single color channel.",
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The color channel to convert."
                }
              },
              returns = {
                {
                  name = "y",
                  type = "number",
                  description = "The converted color channel."
                }
              }
            }
          }
        },
        {
          name = "mat4",
          tag = "vectors",
          summary = "Create a temporary Mat4.",
          description = "Creates a temporary 4D matrix.  This function takes the same arguments as `Mat4:set`.",
          key = "lovr.math.mat4",
          module = "lovr.math",
          related = {
            "lovr.math.newMat4",
            "Mat4"
          },
          variants = {
            {
              description = "Sets the matrix to the identity matrix.",
              arguments = {},
              returns = {
                {
                  name = "m",
                  type = "Mat4",
                  description = "The new matrix."
                }
              }
            },
            {
              description = "Copies the values from an existing matrix.",
              arguments = {
                {
                  name = "n",
                  type = "mat4",
                  description = "An existing matrix to copy the values from."
                }
              },
              returns = {
                {
                  name = "m",
                  type = "Mat4",
                  description = "The new matrix."
                }
              }
            },
            {
              arguments = {
                {
                  name = "position",
                  type = "Vec3",
                  description = "The translation of the matrix.",
                  default = "0, 0, 0"
                },
                {
                  name = "scale",
                  type = "Vec3",
                  description = "The scale of the matrix.",
                  default = "1, 1, 1"
                },
                {
                  name = "rotation",
                  type = "Quat",
                  description = "The rotation of the matrix.",
                  default = "0, 0, 0, 1"
                }
              },
              returns = {
                {
                  name = "m",
                  type = "Mat4",
                  description = "The new matrix."
                }
              }
            },
            {
              arguments = {
                {
                  name = "position",
                  type = "Vec3",
                  description = "The translation of the matrix.",
                  default = "0, 0, 0"
                },
                {
                  name = "rotation",
                  type = "Quat",
                  description = "The rotation of the matrix.",
                  default = "0, 0, 0, 1"
                }
              },
              returns = {
                {
                  name = "m",
                  type = "Mat4",
                  description = "The new matrix."
                }
              }
            },
            {
              arguments = {
                {
                  name = "...",
                  type = "number",
                  description = "16 numbers to use as the raw values of the matrix (column-major)."
                }
              },
              returns = {
                {
                  name = "m",
                  type = "Mat4",
                  description = "The new matrix."
                }
              }
            },
            {
              description = "Sets the diagonal values to a number and everything else to 0.",
              arguments = {
                {
                  name = "d",
                  type = "number",
                  description = "A number to use for the diagonal elements."
                }
              },
              returns = {
                {
                  name = "m",
                  type = "Mat4",
                  description = "The new matrix."
                }
              }
            }
          }
        },
        {
          name = "newCurve",
          tag = "mathOther",
          summary = "Create a new Curve.",
          description = "Creates a new `Curve` from a list of control points.",
          key = "lovr.math.newCurve",
          module = "lovr.math",
          variants = {
            {
              description = "Create a Curve from a set of initial control points.",
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The x coordinate of the first control point."
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y coordinate of the first control point."
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z coordinate of the first control point."
                },
                {
                  name = "...",
                  type = "*",
                  description = "Additional control points."
                }
              },
              returns = {
                {
                  name = "curve",
                  type = "Curve",
                  description = "The new Curve."
                }
              }
            },
            {
              description = "Create a Curve from a (flat) table of points.",
              arguments = {
                {
                  name = "points",
                  type = "table",
                  description = "A table of points, as above."
                }
              },
              returns = {
                {
                  name = "curve",
                  type = "Curve",
                  description = "The new Curve."
                }
              }
            },
            {
              description = "Create an empty Curve, reserving space ahead of time for a certain number of control points.",
              arguments = {
                {
                  name = "n",
                  type = "number",
                  description = "The number of points to reserve for the Curve."
                }
              },
              returns = {
                {
                  name = "curve",
                  type = "Curve",
                  description = "The new Curve."
                }
              }
            }
          }
        },
        {
          name = "newMat4",
          tag = "vectors",
          summary = "Create a new Mat4.",
          description = "Creates a new 4D matrix.  This function takes the same arguments as `Mat4:set`.",
          key = "lovr.math.newMat4",
          module = "lovr.math",
          related = {
            "lovr.math.mat4",
            "Mat4"
          },
          variants = {
            {
              description = "Sets the matrix to the identity matrix.",
              arguments = {},
              returns = {
                {
                  name = "m",
                  type = "Mat4",
                  description = "The new matrix."
                }
              }
            },
            {
              description = "Copies the values from an existing matrix.",
              arguments = {
                {
                  name = "n",
                  type = "mat4",
                  description = "An existing matrix to copy the values from."
                }
              },
              returns = {
                {
                  name = "m",
                  type = "Mat4",
                  description = "The new matrix."
                }
              }
            },
            {
              arguments = {
                {
                  name = "position",
                  type = "Vec3",
                  description = "The translation of the matrix.",
                  default = "0, 0, 0"
                },
                {
                  name = "scale",
                  type = "Vec3",
                  description = "The scale of the matrix.",
                  default = "1, 1, 1"
                },
                {
                  name = "rotation",
                  type = "Quat",
                  description = "The rotation of the matrix.",
                  default = "0, 0, 0, 1"
                }
              },
              returns = {
                {
                  name = "m",
                  type = "Mat4",
                  description = "The new matrix."
                }
              }
            },
            {
              arguments = {
                {
                  name = "position",
                  type = "Vec3",
                  description = "The translation of the matrix.",
                  default = "0, 0, 0"
                },
                {
                  name = "rotation",
                  type = "Quat",
                  description = "The rotation of the matrix.",
                  default = "0, 0, 0, 1"
                }
              },
              returns = {
                {
                  name = "m",
                  type = "Mat4",
                  description = "The new matrix."
                }
              }
            },
            {
              arguments = {
                {
                  name = "...",
                  type = "number",
                  description = "16 numbers to use as the raw values of the matrix (column-major)."
                }
              },
              returns = {
                {
                  name = "m",
                  type = "Mat4",
                  description = "The new matrix."
                }
              }
            },
            {
              description = "Sets the diagonal values to a number and everything else to 0.",
              arguments = {
                {
                  name = "d",
                  type = "number",
                  description = "A number to use for the diagonal elements."
                }
              },
              returns = {
                {
                  name = "m",
                  type = "Mat4",
                  description = "The new matrix."
                }
              }
            }
          }
        },
        {
          name = "newQuat",
          tag = "vectors",
          summary = "Create a new Quat.",
          description = "Creates a new quaternion.  This function takes the same arguments as `Quat:set`.",
          key = "lovr.math.newQuat",
          module = "lovr.math",
          related = {
            "lovr.math.quat",
            "Quat"
          },
          variants = {
            {
              arguments = {
                {
                  name = "angle",
                  description = "An angle to use for the rotation, in radians.",
                  default = "0"
                },
                {
                  name = "ax",
                  type = "number",
                  description = "The x component of the axis of rotation.",
                  default = "0"
                },
                {
                  name = "ay",
                  type = "number",
                  description = "The y component of the axis of rotation.",
                  default = "0"
                },
                {
                  name = "az",
                  type = "number",
                  description = "The z component of the axis of rotation.",
                  default = "0"
                },
                {
                  name = "raw",
                  type = "boolean",
                  description = "Whether the components should be interpreted as raw `(x, y, z, w)` components.",
                  default = "false"
                }
              },
              returns = {
                {
                  name = "q",
                  type = "quat",
                  description = "The new quaternion."
                }
              }
            },
            {
              arguments = {
                {
                  name = "r",
                  type = "quat",
                  description = "An existing quaternion to copy the values from."
                }
              },
              returns = {
                {
                  name = "q",
                  type = "quat",
                  description = "The new quaternion."
                }
              }
            },
            {
              description = "Uses the direction of a vector.",
              arguments = {
                {
                  name = "v",
                  type = "vec3",
                  description = "A normalized direction vector."
                }
              },
              returns = {
                {
                  name = "q",
                  type = "quat",
                  description = "The new quaternion."
                }
              }
            },
            {
              description = "Sets the rotation to represent the rotation between two vectors.",
              arguments = {
                {
                  name = "v",
                  type = "vec3",
                  description = "A normalized direction vector."
                },
                {
                  name = "u",
                  type = "vec3",
                  description = "Another normalized direction vector."
                }
              },
              returns = {
                {
                  name = "q",
                  type = "quat",
                  description = "The new quaternion."
                }
              }
            },
            {
              arguments = {
                {
                  name = "m",
                  type = "mat4",
                  description = "A matrix to use the rotation from."
                }
              },
              returns = {
                {
                  name = "q",
                  type = "quat",
                  description = "The new quaternion."
                }
              }
            },
            {
              description = "Set the quaternion to the identity (0, 0, 0, 1).",
              arguments = {},
              returns = {
                {
                  name = "q",
                  type = "quat",
                  description = "The new quaternion."
                }
              }
            }
          }
        },
        {
          name = "newRandomGenerator",
          tag = "random",
          summary = "Create a new RandomGenerator.",
          description = "Creates a new `RandomGenerator`, which can be used to generate random numbers. If you just want some random numbers, you can use `lovr.math.random`. Individual RandomGenerator objects are useful if you need more control over the random sequence used or need a random generator isolated from other instances.",
          key = "lovr.math.newRandomGenerator",
          module = "lovr.math",
          variants = {
            {
              description = "Create a RandomGenerator with a default seed.",
              arguments = {},
              returns = {
                {
                  name = "randomGenerator",
                  type = "RandomGenerator",
                  description = "The new RandomGenerator."
                }
              }
            },
            {
              arguments = {
                {
                  name = "seed",
                  type = "number",
                  description = "The initial seed for the RandomGenerator."
                }
              },
              returns = {
                {
                  name = "randomGenerator",
                  type = "RandomGenerator",
                  description = "The new RandomGenerator."
                }
              }
            },
            {
              description = "This variant allows creation of random generators with precise 64-bit seed values, since Lua's number format loses precision with really big numbers.",
              arguments = {
                {
                  name = "low",
                  type = "number",
                  description = "The lower 32 bits of the seed."
                },
                {
                  name = "high",
                  type = "number",
                  description = "The upper 32 bits of the seed."
                }
              },
              returns = {
                {
                  name = "randomGenerator",
                  type = "RandomGenerator",
                  description = "The new RandomGenerator."
                }
              }
            }
          }
        },
        {
          name = "newVec2",
          tag = "vectors",
          summary = "Create a new Vec2.",
          description = "Creates a new 2D vector.  This function takes the same arguments as `Vec2:set`.",
          key = "lovr.math.newVec2",
          module = "lovr.math",
          related = {
            "lovr.math.vec2",
            "Vec2"
          },
          variants = {
            {
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The x value of the vector.",
                  default = "0"
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y value of the vector.",
                  default = "x"
                }
              },
              returns = {
                {
                  name = "v",
                  type = "Vec2",
                  description = "The new vector."
                }
              }
            },
            {
              arguments = {
                {
                  name = "u",
                  type = "Vec2",
                  description = "A vector to copy the values from."
                }
              },
              returns = {
                {
                  name = "v",
                  type = "Vec2",
                  description = "The new vector."
                }
              }
            }
          }
        },
        {
          name = "newVec3",
          tag = "vectors",
          summary = "Create a new Vec3.",
          description = "Creates a new 3D vector.  This function takes the same arguments as `Vec3:set`.",
          key = "lovr.math.newVec3",
          module = "lovr.math",
          related = {
            "lovr.math.vec3",
            "Vec3"
          },
          variants = {
            {
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The x value of the vector.",
                  default = "0"
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y value of the vector.",
                  default = "x"
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z value of the vector.",
                  default = "x"
                }
              },
              returns = {
                {
                  name = "v",
                  type = "Vec3",
                  description = "The new vector."
                }
              }
            },
            {
              arguments = {
                {
                  name = "u",
                  type = "Vec3",
                  description = "A vector to copy the values from."
                }
              },
              returns = {
                {
                  name = "v",
                  type = "Vec3",
                  description = "The new vector."
                }
              }
            },
            {
              arguments = {
                {
                  name = "m",
                  type = "Mat4",
                  description = "A matrix to use the position of."
                }
              },
              returns = {
                {
                  name = "v",
                  type = "Vec3",
                  description = "The new vector."
                }
              }
            }
          }
        },
        {
          name = "newVec4",
          tag = "vectors",
          summary = "Create a new Vec4.",
          description = "Creates a new 4D vector.  This function takes the same arguments as `Vec4:set`.",
          key = "lovr.math.newVec4",
          module = "lovr.math",
          related = {
            "lovr.math.vec4",
            "Vec4"
          },
          variants = {
            {
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The x value of the vector.",
                  default = "0"
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y value of the vector.",
                  default = "x"
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z value of the vector.",
                  default = "x"
                },
                {
                  name = "w",
                  type = "number",
                  description = "The w value of the vector.",
                  default = "x"
                }
              },
              returns = {
                {
                  name = "v",
                  type = "Vec4",
                  description = "The new vector."
                }
              }
            },
            {
              arguments = {
                {
                  name = "u",
                  type = "Vec4",
                  description = "A vector to copy the values from."
                }
              },
              returns = {
                {
                  name = "v",
                  type = "Vec4",
                  description = "The new vector."
                }
              }
            }
          }
        },
        {
          name = "noise",
          tag = "random",
          summary = "Generate perlin noise.",
          description = "Returns a 1D, 2D, 3D, or 4D perlin noise value.  The number will be between 0 and 1, and it will always be 0.5 when the inputs are integers.",
          key = "lovr.math.noise",
          module = "lovr.math",
          related = {
            "lovr.math.random"
          },
          variants = {
            {
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The x coordinate of the input."
                }
              },
              returns = {
                {
                  name = "noise",
                  type = "number",
                  description = "The noise value, between 0 and 1."
                }
              }
            },
            {
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The x coordinate of the input."
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y coordinate of the input."
                }
              },
              returns = {
                {
                  name = "noise",
                  type = "number",
                  description = "The noise value, between 0 and 1."
                }
              }
            },
            {
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The x coordinate of the input."
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y coordinate of the input."
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z coordinate of the input."
                }
              },
              returns = {
                {
                  name = "noise",
                  type = "number",
                  description = "The noise value, between 0 and 1."
                }
              }
            },
            {
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The x coordinate of the input."
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y coordinate of the input."
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z coordinate of the input."
                },
                {
                  name = "w",
                  type = "number",
                  description = "The w coordinate of the input."
                }
              },
              returns = {
                {
                  name = "noise",
                  type = "number",
                  description = "The noise value, between 0 and 1."
                }
              }
            }
          }
        },
        {
          name = "quat",
          tag = "vectors",
          summary = "Create a temporary Quat.",
          description = "Creates a temporary quaternion.  This function takes the same arguments as `Quat:set`.",
          key = "lovr.math.quat",
          module = "lovr.math",
          related = {
            "lovr.math.newQuat",
            "Quat"
          },
          variants = {
            {
              arguments = {
                {
                  name = "angle",
                  description = "An angle to use for the rotation, in radians.",
                  default = "0"
                },
                {
                  name = "ax",
                  type = "number",
                  description = "The x component of the axis of rotation.",
                  default = "0"
                },
                {
                  name = "ay",
                  type = "number",
                  description = "The y component of the axis of rotation.",
                  default = "0"
                },
                {
                  name = "az",
                  type = "number",
                  description = "The z component of the axis of rotation.",
                  default = "0"
                },
                {
                  name = "raw",
                  type = "boolean",
                  description = "Whether the components should be interpreted as raw `(x, y, z, w)` components.",
                  default = "false"
                }
              },
              returns = {
                {
                  name = "q",
                  type = "quat",
                  description = "The new quaternion."
                }
              }
            },
            {
              arguments = {
                {
                  name = "r",
                  type = "quat",
                  description = "An existing quaternion to copy the values from."
                }
              },
              returns = {
                {
                  name = "q",
                  type = "quat",
                  description = "The new quaternion."
                }
              }
            },
            {
              description = "Uses the direction of a vector.",
              arguments = {
                {
                  name = "v",
                  type = "vec3",
                  description = "A normalized direction vector."
                }
              },
              returns = {
                {
                  name = "q",
                  type = "quat",
                  description = "The new quaternion."
                }
              }
            },
            {
              description = "Sets the rotation to represent the rotation between two vectors.",
              arguments = {
                {
                  name = "v",
                  type = "vec3",
                  description = "A normalized direction vector."
                },
                {
                  name = "u",
                  type = "vec3",
                  description = "Another normalized direction vector."
                }
              },
              returns = {
                {
                  name = "q",
                  type = "quat",
                  description = "The new quaternion."
                }
              }
            },
            {
              arguments = {
                {
                  name = "m",
                  type = "mat4",
                  description = "A matrix to use the rotation from."
                }
              },
              returns = {
                {
                  name = "q",
                  type = "quat",
                  description = "The new quaternion."
                }
              }
            },
            {
              description = "Set the quaternion to the identity (0, 0, 0, 1).",
              arguments = {},
              returns = {
                {
                  name = "q",
                  type = "quat",
                  description = "The new quaternion."
                }
              }
            }
          }
        },
        {
          name = "random",
          tag = "random",
          summary = "Get a random number.",
          description = "Returns a uniformly distributed pseudo-random number.  This function has improved randomness over Lua's `math.random` and also guarantees that the sequence of random numbers will be the same on all platforms (given the same seed).",
          key = "lovr.math.random",
          module = "lovr.math",
          notes = "You can set the random seed using `lovr.math.setRandomSeed`.",
          related = {
            "lovr.math.randomNormal",
            "RandomGenerator",
            "lovr.math.noise"
          },
          variants = {
            {
              description = "Generate a pseudo-random floating point number in the range `[0,1)`",
              arguments = {},
              returns = {
                {
                  name = "x",
                  type = "number",
                  description = "A pseudo-random number."
                }
              }
            },
            {
              description = "Generate a pseudo-random integer in the range `[1,high]`",
              arguments = {
                {
                  name = "high",
                  type = "number",
                  description = "The maximum number to generate."
                }
              },
              returns = {
                {
                  name = "x",
                  type = "number",
                  description = "A pseudo-random number."
                }
              }
            },
            {
              description = "Generate a pseudo-random integer in the range `[low,high]`",
              arguments = {
                {
                  name = "low",
                  type = "number",
                  description = "The minimum number to generate."
                },
                {
                  name = "high",
                  type = "number",
                  description = "The maximum number to generate."
                }
              },
              returns = {
                {
                  name = "x",
                  type = "number",
                  description = "A pseudo-random number."
                }
              }
            }
          }
        },
        {
          name = "randomNormal",
          tag = "random",
          summary = "Get a random number from a normal distribution.",
          description = "Returns a pseudo-random number from a normal distribution (a bell curve).  You can control the center of the bell curve (the mean value) and the overall width (sigma, or standard deviation).",
          key = "lovr.math.randomNormal",
          module = "lovr.math",
          related = {
            "lovr.math.random",
            "RandomGenerator"
          },
          variants = {
            {
              arguments = {
                {
                  name = "sigma",
                  type = "number",
                  description = "The standard deviation of the distribution.  This can be thought of how \"wide\" the range of numbers is or how much variability there is.",
                  default = "1"
                },
                {
                  name = "mu",
                  type = "number",
                  description = "The average value returned.",
                  default = "0"
                }
              },
              returns = {
                {
                  name = "x",
                  type = "number",
                  description = "A normally distributed pseudo-random number."
                }
              }
            }
          }
        },
        {
          name = "setRandomSeed",
          tag = "random",
          summary = "Set the random seed.",
          description = "Seed the random generator with a new seed.  Each seed will cause `lovr.math.random` and `lovr.math.randomNormal` to produce a unique sequence of random numbers.  This is done once automatically at startup by `lovr.run`.",
          key = "lovr.math.setRandomSeed",
          module = "lovr.math",
          variants = {
            {
              arguments = {
                {
                  name = "seed",
                  type = "number",
                  description = "The new seed."
                }
              },
              returns = {}
            }
          }
        },
        {
          name = "vec2",
          tag = "vectors",
          summary = "Create a temporary Vec2.",
          description = "Creates a temporary 2D vector.  This function takes the same arguments as `Vec2:set`.",
          key = "lovr.math.vec2",
          module = "lovr.math",
          related = {
            "lovr.math.newVec2",
            "Vec2"
          },
          variants = {
            {
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The x value of the vector.",
                  default = "0"
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y value of the vector.",
                  default = "x"
                }
              },
              returns = {
                {
                  name = "v",
                  type = "Vec2",
                  description = "The new vector."
                }
              }
            },
            {
              arguments = {
                {
                  name = "u",
                  type = "Vec2",
                  description = "A vector to copy the values from."
                }
              },
              returns = {
                {
                  name = "v",
                  type = "Vec2",
                  description = "The new vector."
                }
              }
            }
          }
        },
        {
          name = "vec3",
          tag = "vectors",
          summary = "Create a temporary Vec3.",
          description = "Creates a temporary 3D vector.  This function takes the same arguments as `Vec3:set`.",
          key = "lovr.math.vec3",
          module = "lovr.math",
          related = {
            "lovr.math.newVec3",
            "Vec3"
          },
          variants = {
            {
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The x value of the vector.",
                  default = "0"
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y value of the vector.",
                  default = "x"
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z value of the vector.",
                  default = "x"
                }
              },
              returns = {
                {
                  name = "v",
                  type = "Vec3",
                  description = "The new vector."
                }
              }
            },
            {
              arguments = {
                {
                  name = "u",
                  type = "Vec3",
                  description = "A vector to copy the values from."
                }
              },
              returns = {
                {
                  name = "v",
                  type = "Vec3",
                  description = "The new vector."
                }
              }
            },
            {
              arguments = {
                {
                  name = "m",
                  type = "Mat4",
                  description = "A matrix to use the position of."
                }
              },
              returns = {
                {
                  name = "v",
                  type = "Vec3",
                  description = "The new vector."
                }
              }
            }
          }
        },
        {
          name = "vec4",
          tag = "vectors",
          summary = "Create a temporary Vec4.",
          description = "Creates a temporary 4D vector.  This function takes the same arguments as `Vec4:set`.",
          key = "lovr.math.vec4",
          module = "lovr.math",
          related = {
            "lovr.math.newVec4",
            "Vec4"
          },
          variants = {
            {
              arguments = {
                {
                  name = "x",
                  type = "number",
                  description = "The x value of the vector.",
                  default = "0"
                },
                {
                  name = "y",
                  type = "number",
                  description = "The y value of the vector.",
                  default = "x"
                },
                {
                  name = "z",
                  type = "number",
                  description = "The z value of the vector.",
                  default = "x"
                },
                {
                  name = "w",
                  type = "number",
                  description = "The w value of the vector.",
                  default = "x"
                }
              },
              returns = {
                {
                  name = "v",
                  type = "Vec4",
                  description = "The new vector."
                }
              }
            },
            {
              arguments = {
                {
                  name = "u",
                  type = "Vec4",
                  description = "A vector to copy the values from."
                }
              },
              returns = {
                {
                  name = "v",
                  type = "Vec4",
                  description = "The new vector."
                }
              }
            }
          }
        }
      },
      enums = {},
      sections = {
        {
          name = "Randomization",
          tag = "random",
          description = "Functions for generating random numbers and noise."
        },
        {
          name = "Vectors",
          tag = "vectors",
          description = "A collection of vector objects.  Check out the `Vectors` guide for an introduction."
        },
        {
          name = "Other",
          tag = "mathOther",
          description = "Other miscellaneous math objects/helpers."
        }
      }
    },
    {
      name = "physics",
      tag = "modules",
      summary = "Simulates 3D physics.",
      description = "The `lovr.physics` module simulates 3D rigid body physics.",
      key = "lovr.physics",
      objects = {
        {
          name = "BallJoint",
          summary = "A ball and socket joint.",
          description = "A BallJoint is a type of `Joint` that acts like a ball and socket between two colliders.  It allows the colliders to rotate freely around an anchor point, but does not allow the colliders' distance from the anchor point to change.",
          key = "BallJoint",
          module = "lovr.physics",
          constructors = {
            "lovr.physics.newBallJoint"
          },
          related = {
            "Collider"
          },
          extends = "Joint",
          methods = {
            {
              name = "getAnchors",
              summary = "Get the anchor points of the BallJoint.",
              description = "Returns the anchor points of the BallJoint, in world coordinates.  The first point is the anchor on the first collider, and the second point is on the second collider.  The joint tries to keep these points the same, but they may be different if the joint is forced apart by some other means.",
              key = "BallJoint:getAnchors",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x1",
                      type = "number",
                      description = "The x coordinate of the first anchor point, in world coordinates."
                    },
                    {
                      name = "y1",
                      type = "number",
                      description = "The y coordinate of the first anchor point, in world coordinates."
                    },
                    {
                      name = "z1",
                      type = "number",
                      description = "The z coordinate of the first anchor point, in world coordinates."
                    },
                    {
                      name = "x2",
                      type = "number",
                      description = "The x coordinate of the second anchor point, in world coordinates."
                    },
                    {
                      name = "y2",
                      type = "number",
                      description = "The y coordinate of the second anchor point, in world coordinates."
                    },
                    {
                      name = "z2",
                      type = "number",
                      description = "The z coordinate of the second anchor point, in world coordinates."
                    }
                  }
                }
              }
            },
            {
              name = "getResponseTime",
              summary = "Get the response time of the joint.",
              description = "Returns the response time of the joint.  See `World:setResponseTime` for more info.",
              key = "BallJoint:getResponseTime",
              module = "lovr.physics",
              related = {
                "DistanceJoint:getResponseTime",
                "DistanceJoint:setResponseTime",
                "World:getResponseTime",
                "World:setResponseTime"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "responseTime",
                      type = "number",
                      description = "The response time setting for the joint."
                    }
                  }
                }
              }
            },
            {
              name = "getTightness",
              summary = "Get the joint tightness.",
              description = "Returns the tightness of the joint.  See `World:setTightness` for how this affects the joint.",
              key = "BallJoint:getTightness",
              module = "lovr.physics",
              related = {
                "DistanceJoint:getTightness",
                "DistanceJoint:setTightness",
                "World:getTightness",
                "World:setTightness"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "tightness",
                      type = "number",
                      description = "The tightness of the joint."
                    }
                  }
                }
              }
            },
            {
              name = "setAnchor",
              summary = "Set the anchor point of the BallJoint.",
              description = "Sets a new anchor point for the BallJoint.",
              key = "BallJoint:setAnchor",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the anchor point, in world coordinates."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the anchor point, in world coordinates."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the anchor point, in world coordinates."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setResponseTime",
              summary = "Set the response time of the joint.",
              description = "Sets the response time of the joint.  See `World:setResponseTime` for more info.",
              key = "BallJoint:setResponseTime",
              module = "lovr.physics",
              related = {
                "DistanceJoint:getResponseTime",
                "DistanceJoint:setResponseTime",
                "World:getResponseTime",
                "World:setResponseTime"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "responseTime",
                      type = "number",
                      description = "The new response time setting for the joint."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setTightness",
              summary = "Set the joint tightness.",
              description = "Sets the tightness of the joint.  See `World:setTightness` for how this affects the joint.",
              key = "BallJoint:setTightness",
              module = "lovr.physics",
              related = {
                "DistanceJoint:getTightness",
                "DistanceJoint:setTightness",
                "World:getTightness",
                "World:setTightness"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "tightness",
                      type = "number",
                      description = "The tightness of the joint."
                    }
                  },
                  returns = {}
                }
              }
            }
          }
        },
        {
          name = "BoxShape",
          summary = "A box Shape.",
          description = "A type of `Shape` that can be used for cubes or boxes.",
          key = "BoxShape",
          module = "lovr.physics",
          constructors = {
            "lovr.physics.newBoxShape",
            "World:newBoxCollider"
          },
          extends = "Shape",
          methods = {
            {
              name = "getDimensions",
              summary = "Get the dimensions of the BoxShape.",
              description = "Returns the width, height, and depth of the BoxShape.",
              key = "BoxShape:getDimensions",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the box, in meters."
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the box, in meters."
                    },
                    {
                      name = "depth",
                      type = "number",
                      description = "The depth of the box, in meters."
                    }
                  }
                }
              }
            },
            {
              name = "setDimensions",
              summary = "Set the dimensions of the BoxShape.",
              description = "Sets the width, height, and depth of the BoxShape.",
              key = "BoxShape:setDimensions",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {
                    {
                      name = "width",
                      type = "number",
                      description = "The width of the box, in meters."
                    },
                    {
                      name = "height",
                      type = "number",
                      description = "The height of the box, in meters."
                    },
                    {
                      name = "depth",
                      type = "number",
                      description = "The depth of the box, in meters."
                    }
                  },
                  returns = {}
                }
              }
            }
          }
        },
        {
          name = "CapsuleShape",
          summary = "A capsule Shape.",
          description = "A type of `Shape` that can be used for capsule-shaped things.",
          key = "CapsuleShape",
          module = "lovr.physics",
          constructors = {
            "lovr.physics.newCapsuleShape",
            "World:newCapsuleCollider"
          },
          extends = "Shape",
          methods = {
            {
              name = "getLength",
              summary = "Get the length of the CapsuleShape.",
              description = "Returns the length of the CapsuleShape, not including the caps.",
              key = "CapsuleShape:getLength",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "length",
                      type = "number",
                      description = "The length of the capsule, in meters."
                    }
                  }
                }
              }
            },
            {
              name = "getRadius",
              summary = "Get the radius of the CapsuleShape.",
              description = "Returns the radius of the CapsuleShape.",
              key = "CapsuleShape:getRadius",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the capsule, in meters."
                    }
                  }
                }
              }
            },
            {
              name = "setLength",
              summary = "Set the length of the CapsuleShape.",
              description = "Sets the length of the CapsuleShape.",
              key = "CapsuleShape:setLength",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {
                    {
                      name = "length",
                      type = "number",
                      description = "The new length, in meters, not including the caps."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setRadius",
              summary = "Set the radius of the CapsuleShape.",
              description = "Sets the radius of the CapsuleShape.",
              key = "CapsuleShape:setRadius",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {
                    {
                      name = "radius",
                      type = "number",
                      description = "The new radius, in meters."
                    }
                  },
                  returns = {}
                }
              }
            }
          }
        },
        {
          name = "Collider",
          summary = "A single entity in a physics simulation.",
          description = "Colliders are objects that represent a single rigid body in a physics simulation.  They can have forces applied to them and collide with other colliders.",
          key = "Collider",
          module = "lovr.physics",
          constructors = {
            "World:newCollider",
            "World:newBoxCollider",
            "World:newCapsuleCollider",
            "World:newCylinderCollider",
            "World:newSphereCollider",
            "World:newMeshCollider"
          },
          methods = {
            {
              name = "addShape",
              summary = "Add a Shape to the Collider.",
              description = "Attaches a Shape to the collider.  Attached shapes will collide with other shapes in the world.",
              key = "Collider:addShape",
              module = "lovr.physics",
              related = {
                "Collider:removeShape",
                "Collider:getShapes",
                "Shape"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "shape",
                      type = "Shape",
                      description = "The Shape to attach."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "applyForce",
              summary = "Apply a force to the Collider.",
              description = "Applies a force to the Collider.",
              key = "Collider:applyForce",
              module = "lovr.physics",
              notes = "If the Collider is asleep, it will need to be woken up with `Collider:setAwake` for this function to have any affect.",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the force to apply."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the force to apply."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z component of the force to apply."
                    }
                  },
                  returns = {}
                },
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the force to apply."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the force to apply."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z component of the force to apply."
                    },
                    {
                      name = "px",
                      type = "number",
                      description = "The x position to apply the force at, in world coordinates."
                    },
                    {
                      name = "py",
                      type = "number",
                      description = "The y position to apply the force at, in world coordinates."
                    },
                    {
                      name = "pz",
                      type = "number",
                      description = "The z position to apply the force at, in world coordinates."
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "Collider:applyTorque"
              }
            },
            {
              name = "applyTorque",
              summary = "Apply torque to the Collider.",
              description = "Applies torque to the Collider.",
              key = "Collider:applyTorque",
              module = "lovr.physics",
              notes = "If the Collider is asleep, it will need to be woken up with `Collider:setAwake` for this function to have any effect.",
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x component of the torque."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y component of the torque."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z component of the torque."
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "Collider:applyForce"
              }
            },
            {
              name = "destroy",
              summary = "Destroy the Collider.",
              description = "Destroy the Collider, removing it from the World.",
              key = "Collider:destroy",
              module = "lovr.physics",
              notes = "Calling functions on the collider after destroying it is a bad idea.",
              variants = {
                {
                  arguments = {},
                  returns = {}
                }
              },
              related = {
                "World:destroy",
                "Shape:destroy",
                "Joint:destroy"
              }
            },
            {
              name = "getAABB",
              summary = "Get the Collider's axis aligned bounding box.",
              description = "Returns the bounding box for the Collider, computed from attached shapes.",
              key = "Collider:getAABB",
              module = "lovr.physics",
              related = {
                "Shape:getAABB"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "minx",
                      type = "number",
                      description = "The minimum x coordinate of the box."
                    },
                    {
                      name = "maxx",
                      type = "number",
                      description = "The maximum x coordinate of the box."
                    },
                    {
                      name = "miny",
                      type = "number",
                      description = "The minimum y coordinate of the box."
                    },
                    {
                      name = "maxy",
                      type = "number",
                      description = "The maximum y coordinate of the box."
                    },
                    {
                      name = "minz",
                      type = "number",
                      description = "The minimum z coordinate of the box."
                    },
                    {
                      name = "maxz",
                      type = "number",
                      description = "The maximum z coordinate of the box."
                    }
                  }
                }
              }
            },
            {
              name = "getAngularDamping",
              summary = "Get the angular damping of the Collider.",
              description = "Returns the angular damping parameters of the Collider.  Angular damping makes things less \"spinny\", making them slow down their angular velocity over time.",
              key = "Collider:getAngularDamping",
              module = "lovr.physics",
              notes = "Angular damping can also be set on the World.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "damping",
                      type = "number",
                      description = "The angular damping."
                    },
                    {
                      name = "threshold",
                      type = "number",
                      description = "Velocity limit below which the damping is not applied."
                    }
                  }
                }
              },
              related = {
                "World:getAngularDamping",
                "World:setAngularDamping"
              }
            },
            {
              name = "getAngularVelocity",
              summary = "Get the angular velocity of the Collider.",
              description = "Returns the angular velocity of the Collider.",
              key = "Collider:getAngularVelocity",
              module = "lovr.physics",
              related = {
                "Collider:getLinearVelocity",
                "Collider:setLinearVelocity",
                "Collider:applyTorque",
                "Collider:getOrientation",
                "Collider:setOrientation"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "vx",
                      type = "number",
                      description = "The x component of the angular velocity."
                    },
                    {
                      name = "vy",
                      type = "number",
                      description = "The y component of the angular velocity."
                    },
                    {
                      name = "vz",
                      type = "number",
                      description = "The z component of the angular velocity."
                    }
                  }
                }
              }
            },
            {
              name = "getFriction",
              summary = "Get the friction of the Collider.",
              description = "Returns the friction of the Collider.  By default, the friction of two Colliders is combined (multiplied) when they collide to generate a friction force.  The initial friction is 0.",
              key = "Collider:getFriction",
              module = "lovr.physics",
              related = {
                "Collider:getRestitution",
                "Collider:setRestitution",
                "World:collide"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "friction",
                      type = "number",
                      description = "The friction of the Collider."
                    }
                  }
                }
              }
            },
            {
              name = "getJoints",
              summary = "Get a list of Joints attached to the Collider.",
              description = "Returns a list of Joints attached to the Collider.",
              key = "Collider:getJoints",
              module = "lovr.physics",
              related = {
                "Joint"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "joints",
                      type = "table",
                      description = "A list of Joints attached to the Collider."
                    }
                  }
                }
              }
            },
            {
              name = "getLinearDamping",
              summary = "Get the linear damping of the Collider.",
              description = "Returns the Collider's linear damping parameters.  Linear damping is similar to drag or air resistance, slowing the Collider down over time.",
              key = "Collider:getLinearDamping",
              module = "lovr.physics",
              notes = "A linear damping of 0 means the Collider won't slow down over time.  This is the default.\n\nLinear damping can also be set on the World using `World:setLinearDamping`, which will affect all new colliders.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "damping",
                      type = "number",
                      description = "The linear damping."
                    },
                    {
                      name = "threshold",
                      type = "number",
                      description = "Velocity limit below which the damping is not applied."
                    }
                  }
                }
              },
              related = {
                "World:getLinearDamping",
                "World:setLinearDamping"
              }
            },
            {
              name = "getLinearVelocity",
              summary = "Get the linear velocity of the Collider.",
              description = "Returns the linear velocity of the Collider.  This is how fast the Collider is moving.  There is also angular velocity, which is how fast the Collider is spinning.",
              key = "Collider:getLinearVelocity",
              module = "lovr.physics",
              related = {
                "Collider:getLinearVelocityFromLocalPoint",
                "Collider:getLinearVelocityFromWorldPoint",
                "Collider:getAngularVelocity",
                "Collider:setAngularVelocity",
                "Collider:applyForce",
                "Collider:getPosition",
                "Collider:setPosition"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "vx",
                      type = "number",
                      description = "The x velocity of the Collider, in meters per second."
                    },
                    {
                      name = "vy",
                      type = "number",
                      description = "The y velocity of the Collider, in meters per second."
                    },
                    {
                      name = "vz",
                      type = "number",
                      description = "The z velocity of the Collider, in meters per second."
                    }
                  }
                }
              }
            },
            {
              name = "getLinearVelocityFromLocalPoint",
              summary = "Get the linear velocity of the Collider at a point.",
              description = "Returns the linear velocity of a point relative to the Collider.",
              key = "Collider:getLinearVelocityFromLocalPoint",
              module = "lovr.physics",
              related = {
                "Collider:getLinearVelocity",
                "Collider:getLinearVelocityFromWorldPoint"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate."
                    }
                  },
                  returns = {
                    {
                      name = "vx",
                      type = "number",
                      description = "The x component of the velocity of the point."
                    },
                    {
                      name = "vy",
                      type = "number",
                      description = "The y component of the velocity of the point."
                    },
                    {
                      name = "vz",
                      type = "number",
                      description = "The z component of the velocity of the point."
                    }
                  }
                }
              }
            },
            {
              name = "getLinearVelocityFromWorldPoint",
              summary = "Get the linear velocity of the Collider at a world space point.",
              description = "Returns the linear velocity of a point on the Collider specified in world space.",
              key = "Collider:getLinearVelocityFromWorldPoint",
              module = "lovr.physics",
              related = {
                "Collider:getLinearVelocity",
                "Collider:getLinearVelocityFromLocalPoint"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate in world space."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate in world space."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate in world space."
                    }
                  },
                  returns = {
                    {
                      name = "vx",
                      type = "number",
                      description = "The x component of the velocity of the point."
                    },
                    {
                      name = "vy",
                      type = "number",
                      description = "The y component of the velocity of the point."
                    },
                    {
                      name = "vz",
                      type = "number",
                      description = "The z component of the velocity of the point."
                    }
                  }
                }
              }
            },
            {
              name = "getLocalCenter",
              summary = "Get the Collider's center of mass.",
              description = "Returns the Collider's center of mass.",
              key = "Collider:getLocalCenter",
              module = "lovr.physics",
              related = {
                "Collider:getLocalPoint",
                "Collider:getMassData",
                "Collider:setMassData"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "cx",
                      type = "number",
                      description = "The x position of the center of mass."
                    },
                    {
                      name = "cy",
                      type = "number",
                      description = "The y position of the center of mass."
                    },
                    {
                      name = "cz",
                      type = "number",
                      description = "The z position of the center of mass."
                    }
                  }
                }
              }
            },
            {
              name = "getLocalPoint",
              summary = "Convert a point from world space to collider space.",
              description = "Converts a point from world coordinates into local coordinates relative to the Collider.",
              key = "Collider:getLocalPoint",
              module = "lovr.physics",
              related = {
                "Collider:getWorldPoint",
                "Collider:getLocalVector",
                "Collider:getWorldVector"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "wx",
                      type = "number",
                      description = "The x coordinate of the world point."
                    },
                    {
                      name = "wy",
                      type = "number",
                      description = "The y coordinate of the world point."
                    },
                    {
                      name = "wz",
                      type = "number",
                      description = "The z coordinate of the world point."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the local-space point."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the local-space point."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the local-space point."
                    }
                  }
                }
              }
            },
            {
              name = "getLocalVector",
              summary = "Convert a vector from world space to local space.",
              description = "Converts a direction vector from world space to local space.",
              key = "Collider:getLocalVector",
              module = "lovr.physics",
              related = {
                "Collider:getWorldVector",
                "Collider:getLocalPoint",
                "Collider:getWorldPoint"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "wx",
                      type = "number",
                      description = "The x component of the world vector."
                    },
                    {
                      name = "wy",
                      type = "number",
                      description = "The y component of the world vector."
                    },
                    {
                      name = "wz",
                      type = "number",
                      description = "The z component of the world vector."
                    }
                  },
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the local vector."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the local vector."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the local vector."
                    }
                  }
                }
              }
            },
            {
              name = "getMass",
              summary = "Get the total mass of the Collider.",
              description = "Returns the total mass of the Collider.  The mass of a Collider depends on its attached shapes.",
              key = "Collider:getMass",
              module = "lovr.physics",
              related = {
                "Collider:getMassData",
                "Collider:setMassData",
                "Shape:getMass"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "mass",
                      type = "number",
                      description = "The mass of the Collider, in kilograms."
                    }
                  }
                }
              }
            },
            {
              name = "getMassData",
              summary = "Compute mass properties for the Collider.",
              description = "Computes mass properties for the Collider.",
              key = "Collider:getMassData",
              module = "lovr.physics",
              related = {
                "Collider:getMass",
                "Collider:setMass",
                "Shape:getMass"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "cx",
                      type = "number",
                      description = "The x position of the center of mass."
                    },
                    {
                      name = "cy",
                      type = "number",
                      description = "The y position of the center of mass."
                    },
                    {
                      name = "cz",
                      type = "number",
                      description = "The z position of the center of mass."
                    },
                    {
                      name = "mass",
                      type = "number",
                      description = "The computed mass of the Collider."
                    },
                    {
                      name = "inertia",
                      type = "table",
                      description = "A table containing 6 values of the rotational inertia tensor matrix.  The table contains the 3 diagonal elements of the matrix (upper left to bottom right), followed by the 3 elements of the upper right portion of the 3x3 matrix."
                    }
                  }
                }
              }
            },
            {
              name = "getOrientation",
              summary = "Get the orientation of the Collider.",
              description = "Returns the orientation of the Collider in angle/axis representation.",
              key = "Collider:getOrientation",
              module = "lovr.physics",
              related = {
                "Collider:applyTorque",
                "Collider:getAngularVelocity",
                "Collider:setAngularVelocity",
                "Collider:getPosition",
                "Collider:setPosition",
                "Collider:getPose",
                "Collider:setPose"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the Collider is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getPose",
              summary = "Get the pose of the Collider.",
              description = "Returns the position and orientation of the Collider.",
              key = "Collider:getPose",
              module = "lovr.physics",
              related = {
                "Collider:getPosition",
                "Collider:getOrientation"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the Collider, in meters."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the Collider, in meters."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the Collider, in meters."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the Collider is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  }
                }
              }
            },
            {
              name = "getPosition",
              summary = "Get the position of the Collider.",
              description = "Returns the position of the Collider.",
              key = "Collider:getPosition",
              module = "lovr.physics",
              related = {
                "Collider:applyForce",
                "Collider:getLinearVelocity",
                "Collider:setLinearVelocity",
                "Collider:getOrientation",
                "Collider:setOrientation",
                "Collider:getPose",
                "Collider:setPose"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the Collider, in meters."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the Collider, in meters."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the Collider, in meters."
                    }
                  }
                }
              }
            },
            {
              name = "getRestitution",
              summary = "Get the bounciness of the Collider.",
              description = "Returns the restitution (bounciness) of the Collider.  By default, the restitution of two Colliders is combined (the max is used) when they collide to cause them to bounce away from each other.  The initial restitution is 0.",
              key = "Collider:getRestitution",
              module = "lovr.physics",
              related = {
                "Collider:getFriction",
                "Collider:setFriction",
                "World:collide"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "restitution",
                      type = "number",
                      description = "The restitution of the Collider."
                    }
                  }
                }
              }
            },
            {
              name = "getShapes",
              summary = "Get a list of Shapes attached to the Collider.",
              description = "Returns a list of Shapes attached to the Collider.",
              key = "Collider:getShapes",
              module = "lovr.physics",
              related = {
                "Collider:addShape",
                "Collider:removeShape",
                "Shape"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "shapes",
                      type = "table",
                      description = "A list of Shapes attached to the Collider."
                    }
                  }
                }
              }
            },
            {
              name = "getTag",
              summary = "Get the Collider's tag.",
              description = "Returns the Collider's tag.",
              key = "Collider:getTag",
              module = "lovr.physics",
              notes = "Collision between tags can be enabled and disabled using `World:enableCollisionBetween` and `World:disableCollisionBetween`.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "tag",
                      type = "string",
                      description = "The Collider's collision tag."
                    }
                  }
                }
              },
              related = {
                "World:disableCollisionBetween",
                "World:enableCollisionBetween",
                "World:isCollisionEnabledBetween",
                "lovr.physics.newWorld"
              }
            },
            {
              name = "getUserData",
              summary = "Get the Collider's user data.",
              description = "Returns the user data associated with the Collider.",
              key = "Collider:getUserData",
              module = "lovr.physics",
              notes = "User data can be useful to identify the Collider in callbacks.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "data",
                      type = "*",
                      description = "The custom value associated with the Collider."
                    }
                  }
                }
              }
            },
            {
              name = "getWorld",
              summary = "Get the World the Collider is in.",
              description = "Returns the World the Collider is in.",
              key = "Collider:getWorld",
              module = "lovr.physics",
              notes = "Colliders can only be in one World at a time.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "world",
                      type = "World",
                      description = "The World the Collider is in."
                    }
                  }
                }
              },
              related = {
                "World"
              }
            },
            {
              name = "getWorldPoint",
              summary = "Convert a point from local space to world space.",
              description = "Convert a point relative to the collider to a point in world coordinates.",
              key = "Collider:getWorldPoint",
              module = "lovr.physics",
              related = {
                "Collider:getLocalPoint",
                "Collider:getLocalVector",
                "Collider:getWorldVector"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the point."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the point."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the point."
                    }
                  },
                  returns = {
                    {
                      name = "wx",
                      type = "number",
                      description = "The x coordinate of the world point."
                    },
                    {
                      name = "wy",
                      type = "number",
                      description = "The y coordinate of the world point."
                    },
                    {
                      name = "wz",
                      type = "number",
                      description = "The z coordinate of the world point."
                    }
                  }
                }
              }
            },
            {
              name = "getWorldVector",
              summary = "Convert a vector from local space to world space.",
              description = "Converts a direction vector from local space to world space.",
              key = "Collider:getWorldVector",
              module = "lovr.physics",
              related = {
                "Collider:getLocalVector",
                "Collider:getLocalPoint",
                "Collider:getWorldPoint"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x coordinate of the local vector."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y coordinate of the local vector."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z coordinate of the local vector."
                    }
                  },
                  returns = {
                    {
                      name = "wx",
                      type = "number",
                      description = "The x component of the world vector."
                    },
                    {
                      name = "wy",
                      type = "number",
                      description = "The y component of the world vector."
                    },
                    {
                      name = "wz",
                      type = "number",
                      description = "The z component of the world vector."
                    }
                  }
                }
              }
            },
            {
              name = "isAwake",
              summary = "Check if the Collider is awake.",
              description = "Returns whether the Collider is currently awake.",
              key = "Collider:isAwake",
              module = "lovr.physics",
              related = {
                "World:isSleepingAllowed",
                "World:setSleepingAllowed",
                "Collider:isSleepingAllowed",
                "Collider:setSleepingAllowed"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "awake",
                      type = "boolean",
                      description = "Whether the Collider is awake."
                    }
                  }
                }
              }
            },
            {
              name = "isGravityIgnored",
              summary = "Check if the Collider ignores gravity.",
              description = "Returns whether the Collider is currently ignoring gravity.",
              key = "Collider:isGravityIgnored",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "ignored",
                      type = "boolean",
                      description = "Whether gravity is ignored for this Collider."
                    }
                  }
                }
              }
            },
            {
              name = "isKinematic",
              summary = "Check if the Collider is kinematic.",
              description = "Returns whether the Collider is kinematic.",
              key = "Collider:isKinematic",
              module = "lovr.physics",
              notes = "Kinematic colliders behave as though they have infinite mass, ignoring external forces like gravity, joints, or collisions (though non-kinematic colliders will collide with them). They can be useful for static objects like floors or walls.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "kinematic",
                      type = "boolean",
                      description = "Whether the Collider is kinematic."
                    }
                  }
                }
              }
            },
            {
              name = "isSleepingAllowed",
              summary = "Check if the Collider is allowed to sleep.",
              description = "Returns whether the Collider is allowed to sleep.",
              key = "Collider:isSleepingAllowed",
              module = "lovr.physics",
              notes = "If sleeping is enabled, the simulation will put the Collider to sleep if it hasn't moved in a while. Sleeping colliders don't impact the physics simulation, which makes updates more efficient and improves physics performance.  However, the physics engine isn't perfect at waking up sleeping colliders and this can lead to bugs where colliders don't react to forces or collisions properly.\n\nIt is possible to set the default value for new colliders using `World:setSleepingAllowed`.\n\nColliders can be manually put to sleep or woken up using `Collider:setAwake`.",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "allowed",
                      type = "boolean",
                      description = "Whether the Collider can go to sleep."
                    }
                  }
                }
              },
              related = {
                "World:isSleepingAllowed",
                "World:setSleepingAllowed",
                "Collider:isAwake",
                "Collider:setAwake"
              }
            },
            {
              name = "removeShape",
              summary = "Remove a Shape from the Collider.",
              description = "Removes a Shape from the Collider.",
              key = "Collider:removeShape",
              module = "lovr.physics",
              notes = "Colliders without any shapes won't collide with anything.",
              variants = {
                {
                  arguments = {
                    {
                      name = "shape",
                      type = "Shape",
                      description = "The Shape to remove."
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "Collider:addShape",
                "Collider:getShapes",
                "Shape"
              }
            },
            {
              name = "setAngularDamping",
              summary = "Set the angular damping of the Collider.",
              description = "Sets the angular damping of the Collider.  Angular damping makes things less \"spinny\", causing them to slow down their angular velocity over time. Damping is only applied when angular velocity is over the threshold value.",
              key = "Collider:setAngularDamping",
              module = "lovr.physics",
              notes = "Angular damping can also be set on the World.",
              variants = {
                {
                  arguments = {
                    {
                      name = "damping",
                      type = "number",
                      description = "The angular damping."
                    },
                    {
                      name = "threshold",
                      type = "number",
                      description = "Velocity limit below which the damping is not applied.",
                      default = "0"
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "World:getAngularDamping",
                "World:setAngularDamping"
              }
            },
            {
              name = "setAngularVelocity",
              summary = "Set the angular velocity of the Collider.",
              description = "Sets the angular velocity of the Collider.",
              key = "Collider:setAngularVelocity",
              module = "lovr.physics",
              related = {
                "Collider:getLinearVelocity",
                "Collider:setLinearVelocity",
                "Collider:applyTorque",
                "Collider:getOrientation",
                "Collider:setOrientation"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "vx",
                      type = "number",
                      description = "The x component of the angular velocity."
                    },
                    {
                      name = "vy",
                      type = "number",
                      description = "The y component of the angular velocity."
                    },
                    {
                      name = "vz",
                      type = "number",
                      description = "The z component of the angular velocity."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setAwake",
              summary = "Put the Collider to sleep or wake it up.",
              description = "Manually puts the Collider to sleep or wakes it up.  You can do this if you know a Collider won't be touched for a while or if you need to it be active.",
              key = "Collider:setAwake",
              module = "lovr.physics",
              related = {
                "World:isSleepingAllowed",
                "World:setSleepingAllowed",
                "Collider:isSleepingAllowed",
                "Collider:setSleepingAllowed"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "awake",
                      type = "boolean",
                      description = "Whether the Collider should be awake."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setFriction",
              summary = "Set the friction of the Collider.",
              description = "Sets the friction of the Collider.  By default, the friction of two Colliders is combined (multiplied) when they collide to generate a friction force.  The initial friction is 0.",
              key = "Collider:setFriction",
              module = "lovr.physics",
              related = {
                "Collider:getRestitution",
                "Collider:setRestitution",
                "World:collide"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "friction",
                      type = "number",
                      description = "The new friction."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setGravityIgnored",
              summary = "Set whether the Collider ignores gravity.",
              description = "Sets whether the Collider should ignore gravity.",
              key = "Collider:setGravityIgnored",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {
                    {
                      name = "ignored",
                      type = "boolean",
                      description = "Whether gravity should be ignored."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setKinematic",
              summary = "Set whether the Collider is kinematic.",
              description = "Sets whether the Collider is kinematic.",
              key = "Collider:setKinematic",
              module = "lovr.physics",
              notes = "Kinematic colliders behave as though they have infinite mass, ignoring external forces like gravity, joints, or collisions (though non-kinematic colliders will collide with them). They can be useful for static objects like floors or walls.",
              variants = {
                {
                  arguments = {
                    {
                      name = "kinematic",
                      type = "boolean",
                      description = "Whether the Collider is kinematic."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setLinearDamping",
              summary = "Set the linear damping of the Collider.",
              description = "Sets the Collider's linear damping parameter.  Linear damping is similar to drag or air resistance, slowing the Collider down over time. Damping is only applied when linear velocity is over the threshold value.",
              key = "Collider:setLinearDamping",
              module = "lovr.physics",
              notes = "A linear damping of 0 means the Collider won't slow down over time.  This is the default.\n\nLinear damping can also be set on the World using `World:setLinearDamping`, which will affect all new colliders.",
              variants = {
                {
                  arguments = {
                    {
                      name = "damping",
                      type = "number",
                      description = "The linear damping."
                    },
                    {
                      name = "threshold",
                      type = "number",
                      description = "Velocity limit below which the damping is not applied.",
                      default = "0"
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "World:getLinearDamping",
                "World:setLinearDamping"
              }
            },
            {
              name = "setLinearVelocity",
              summary = "Set the linear velocity of the Collider.",
              description = "Sets the linear velocity of the Collider directly.  Usually it's preferred to use `Collider:applyForce` to change velocity since instantaneous velocity changes can lead to weird glitches.",
              key = "Collider:setLinearVelocity",
              module = "lovr.physics",
              related = {
                "Collider:getLinearVelocityFromLocalPoint",
                "Collider:getLinearVelocityFromWorldPoint",
                "Collider:getAngularVelocity",
                "Collider:setAngularVelocity",
                "Collider:applyForce",
                "Collider:getPosition",
                "Collider:setPosition"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "vx",
                      type = "number",
                      description = "The x velocity of the Collider, in meters per second."
                    },
                    {
                      name = "vy",
                      type = "number",
                      description = "The y velocity of the Collider, in meters per second."
                    },
                    {
                      name = "vz",
                      type = "number",
                      description = "The z velocity of the Collider, in meters per second."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setMass",
              summary = "Set the total mass of the Collider.",
              description = "Sets the total mass of the Collider.",
              key = "Collider:setMass",
              module = "lovr.physics",
              related = {
                "Collider:getMassData",
                "Collider:setMassData",
                "Shape:getMass"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "mass",
                      type = "number",
                      description = "The new mass for the Collider, in kilograms."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setMassData",
              summary = "Set mass properties for the Collider.",
              description = "Sets mass properties for the Collider.",
              key = "Collider:setMassData",
              module = "lovr.physics",
              related = {
                "Collider:getMass",
                "Collider:setMass",
                "Shape:getMass"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "cx",
                      type = "number",
                      description = "The x position of the center of mass."
                    },
                    {
                      name = "cy",
                      type = "number",
                      description = "The y position of the center of mass."
                    },
                    {
                      name = "cz",
                      type = "number",
                      description = "The z position of the center of mass."
                    },
                    {
                      name = "mass",
                      type = "number",
                      description = "The computed mass of the Collider."
                    },
                    {
                      name = "inertia",
                      type = "table",
                      description = "A table containing 6 values of the rotational inertia tensor matrix.  The table contains the 3 diagonal elements of the matrix (upper left to bottom right), followed by the 3 elements of the upper right portion of the 3x3 matrix."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setOrientation",
              summary = "Set the orientation of the Collider.",
              description = "Sets the orientation of the Collider in angle/axis representation.",
              key = "Collider:setOrientation",
              module = "lovr.physics",
              related = {
                "Collider:applyTorque",
                "Collider:getAngularVelocity",
                "Collider:setAngularVelocity",
                "Collider:getPosition",
                "Collider:setPosition",
                "Collider:getPose",
                "Collider:setPose"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the Collider is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setPose",
              summary = "Set the pose of the Collider.",
              description = "Sets the position and orientation of the Collider.",
              key = "Collider:setPose",
              module = "lovr.physics",
              related = {
                "Collider:setPosition",
                "Collider:setOrientation"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the Collider, in meters."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the Collider, in meters."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the Collider, in meters."
                    },
                    {
                      name = "angle",
                      type = "number",
                      description = "The number of radians the Collider is rotated around its axis of rotation."
                    },
                    {
                      name = "ax",
                      type = "number",
                      description = "The x component of the axis of rotation."
                    },
                    {
                      name = "ay",
                      type = "number",
                      description = "The y component of the axis of rotation."
                    },
                    {
                      name = "az",
                      type = "number",
                      description = "The z component of the axis of rotation."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setPosition",
              summary = "Set the position of the Collider.",
              description = "Sets the position of the Collider.",
              key = "Collider:setPosition",
              module = "lovr.physics",
              related = {
                "Collider:applyForce",
                "Collider:getLinearVelocity",
                "Collider:setLinearVelocity",
                "Collider:getOrientation",
                "Collider:setOrientation",
                "Collider:getPose",
                "Collider:setPose"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "x",
                      type = "number",
                      description = "The x position of the Collider, in meters."
                    },
                    {
                      name = "y",
                      type = "number",
                      description = "The y position of the Collider, in meters."
                    },
                    {
                      name = "z",
                      type = "number",
                      description = "The z position of the Collider, in meters."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setRestitution",
              summary = "Set the bounciness of the Collider.",
              description = "Sets the restitution (bounciness) of the Collider.  By default, the restitution of two Colliders is combined (the max is used) when they collide to cause them to bounce away from each other. The initial restitution is 0.",
              key = "Collider:setRestitution",
              module = "lovr.physics",
              related = {
                "Collider:getFriction",
                "Collider:setFriction",
                "World:collide"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "restitution",
                      type = "number",
                      description = "The new restitution."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setSleepingAllowed",
              summary = "Set whether the Collider is allowed to sleep.",
              description = "Sets whether the Collider is allowed to sleep.",
              key = "Collider:setSleepingAllowed",
              module = "lovr.physics",
              notes = "If sleeping is enabled, the simulation will put the Collider to sleep if it hasn't moved in a while. Sleeping colliders don't impact the physics simulation, which makes updates more efficient and improves physics performance.  However, the physics engine isn't perfect at waking up sleeping colliders and this can lead to bugs where colliders don't react to forces or collisions properly.\n\nIt is possible to set the default value for new colliders using `World:setSleepingAllowed`.\n\nColliders can be manually put to sleep or woken up using `Collider:setAwake`.",
              variants = {
                {
                  arguments = {
                    {
                      name = "allowed",
                      type = "boolean",
                      description = "Whether the Collider can go to sleep."
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "World:isSleepingAllowed",
                "World:setSleepingAllowed",
                "Collider:isAwake",
                "Collider:setAwake"
              }
            },
            {
              name = "setTag",
              summary = "Set the Collider's tag.",
              description = "Sets the Collider's tag.",
              key = "Collider:setTag",
              module = "lovr.physics",
              notes = "Collision between tags can be enabled and disabled using `World:enableCollisionBetween` and `World:disableCollisionBetween`.",
              variants = {
                {
                  arguments = {
                    {
                      name = "tag",
                      type = "string",
                      description = "The Collider's collision tag."
                    }
                  },
                  returns = {}
                }
              },
              related = {
                "World:disableCollisionBetween",
                "World:enableCollisionBetween",
                "World:isCollisionEnabledBetween",
                "lovr.physics.newWorld"
              }
            },
            {
              name = "setUserData",
              summary = "Set the Collider's user data.",
              description = "Associates a custom value with the Collider.",
              key = "Collider:setUserData",
              module = "lovr.physics",
              notes = "User data can be useful to identify the Collider in callbacks.",
              variants = {
                {
                  arguments = {
                    {
                      name = "data",
                      type = "*",
                      description = "The custom value to associate with the Collider."
                    }
                  },
                  returns = {}
                }
              }
            }
          }
        },
        {
          name = "CylinderShape",
          summary = "A cylinder Shape.",
          description = "A type of `Shape` that can be used for cylinder-shaped things.",
          key = "CylinderShape",
          module = "lovr.physics",
          constructors = {
            "lovr.physics.newCylinderShape",
            "World:newCylinderCollider"
          },
          extends = "Shape",
          methods = {
            {
              name = "getLength",
              summary = "Get the length of the CylinderShape.",
              description = "Returns the length of the CylinderShape.",
              key = "CylinderShape:getLength",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "length",
                      type = "number",
                      description = "The length of the cylinder, in meters."
                    }
                  }
                }
              }
            },
            {
              name = "getRadius",
              summary = "Get the radius of the CylinderShape.",
              description = "Returns the radius of the CylinderShape.",
              key = "CylinderShape:getRadius",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "radius",
                      type = "number",
                      description = "The radius of the cylinder, in meters."
                    }
                  }
                }
              }
            },
            {
              name = "setLength",
              summary = "Set the length of the CylinderShape.",
              description = "Sets the length of the CylinderShape.",
              key = "CylinderShape:setLength",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {
                    {
                      name = "length",
                      type = "number",
                      description = "The new length, in meters."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setRadius",
              summary = "Set the radius of the CylinderShape.",
              description = "Sets the radius of the CylinderShape.",
              key = "CylinderShape:setRadius",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {
                    {
                      name = "radius",
                      type = "number",
                      description = "The new radius, in meters."
                    }
                  },
                  returns = {}
                }
              }
            }
          }
        },
        {
          name = "DistanceJoint",
          summary = "A fixed distance joint.",
          description = "A DistanceJoint is a type of `Joint` that tries to keep two colliders a fixed distance apart. The distance is determined by the initial distance between the anchor points.  The joint allows for rotation on the anchor points.",
          key = "DistanceJoint",
          module = "lovr.physics",
          constructors = {
            "lovr.physics.newDistanceJoint"
          },
          related = {
            "Collider"
          },
          extends = "Joint",
          methods = {
            {
              name = "getAnchors",
              summary = "Get the anchor points of the DistanceJoint.",
              description = "Returns the anchor points of the DistanceJoint.",
              key = "DistanceJoint:getAnchors",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "x1",
                      type = "number",
                      description = "The x coordinate of the first anchor point, in world coordinates."
                    },
                    {
                      name = "y1",
                      type = "number",
                      description = "The y coordinate of the first anchor point, in world coordinates."
                    },
                    {
                      name = "z1",
                      type = "number",
                      description = "The z coordinate of the first anchor point, in world coordinates."
                    },
                    {
                      name = "x2",
                      type = "number",
                      description = "The x coordinate of the second anchor point, in world coordinates."
                    },
                    {
                      name = "y2",
                      type = "number",
                      description = "The y coordinate of the second anchor point, in world coordinates."
                    },
                    {
                      name = "z2",
                      type = "number",
                      description = "The z coordinate of the second anchor point, in world coordinates."
                    }
                  }
                }
              }
            },
            {
              name = "getDistance",
              summary = "Get the target distance of the DistanceJoint.",
              description = "Returns the target distance for the DistanceJoint.  The joint tries to keep the Colliders this far apart.",
              key = "DistanceJoint:getDistance",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "distance",
                      type = "number",
                      description = "The target distance."
                    }
                  }
                }
              }
            },
            {
              name = "getResponseTime",
              summary = "Get the response time of the joint.",
              description = "Returns the response time of the joint.  See `World:setResponseTime` for more info.",
              key = "DistanceJoint:getResponseTime",
              module = "lovr.physics",
              related = {
                "BallJoint:getResponseTime",
                "BallJoint:setResponseTime",
                "World:getResponseTime",
                "World:setResponseTime"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "responseTime",
                      type = "number",
                      description = "The response time setting for the joint."
                    }
                  }
                }
              }
            },
            {
              name = "getTightness",
              summary = "Get the joint tightness.",
              description = "Returns the tightness of the joint.  See `World:setTightness` for how this affects the joint.",
              key = "DistanceJoint:getTightness",
              module = "lovr.physics",
              related = {
                "BallJoint:getTightness",
                "BallJoint:setTightness",
                "World:getTightness",
                "World:setTightness"
              },
              variants = {
                {
                  arguments = {},
                  returns = {
                    {
                      name = "tightness",
                      type = "number",
                      description = "The tightness of the joint."
                    }
                  }
                }
              }
            },
            {
              name = "setAnchors",
              summary = "Set the anchor points of the DistanceJoint.",
              description = "Sets the anchor points of the DistanceJoint.",
              key = "DistanceJoint:setAnchors",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {
                    {
                      name = "x1",
                      type = "number",
                      description = "The x coordinate of the first anchor point, in world coordinates."
                    },
                    {
                      name = "y1",
                      type = "number",
                      description = "The y coordinate of the first anchor point, in world coordinates."
                    },
                    {
                      name = "z1",
                      type = "number",
                      description = "The z coordinate of the first anchor point, in world coordinates."
                    },
                    {
                      name = "x2",
                      type = "number",
                      description = "The x coordinate of the second anchor point, in world coordinates."
                    },
                    {
                      name = "y2",
                      type = "number",
                      description = "The y coordinate of the second anchor point, in world coordinates."
                    },
                    {
                      name = "z2",
                      type = "number",
                      description = "The z coordinate of the second anchor point, in world coordinates."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setDistance",
              summary = "Set the target distance of the DistanceJoint.",
              description = "Sets the target distance for the DistanceJoint.  The joint tries to keep the Colliders this far apart.",
              key = "DistanceJoint:setDistance",
              module = "lovr.physics",
              variants = {
                {
                  arguments = {
                    {
                      name = "distance",
                      type = "number",
                      description = "The new target distance."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setResponseTime",
              summary = "Set the response time of the joint.",
              description = "Sets the response time of the joint.  See `World:setResponseTime` for more info.",
              key = "DistanceJoint:setResponseTime",
              module = "lovr.physics",
              related = {
                "BallJoint:getResponseTime",
                "BallJoint:setResponseTime",
                "World:getResponseTime",
                "World:setResponseTime"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "responseTime",
                      type = "number",
                      description = "The new response time setting for the joint."
                    }
                  },
                  returns = {}
                }
              }
            },
            {
              name = "setTightness",
              summary = "Set the joint tightness.",
              description = "Sets the tightness of the joint.  See `World:setTightness` for how this affects the joint.",
              key = "DistanceJoint:setTightness",
              module = "lovr.physics",
              related = {
                "BallJoint:getTightness",
                "BallJoint:setTightness",
                "World:getTightness",
                "World:setTightness"
              },
              variants = {
                {
                  arguments = {
                    {
                      name = "tightness",
                      type = "number",
                      description = "The tightness of the joint."
                    }
                  },
                  returns = {}
                }
              }
            }
          }
        },
        {
          name = "HingeJoint",
          summary = "A hinge joint.",
          description = "A HingeJoint is a type of `Joint` that only allows colliders to rotate on a single axis.",
          key = "HingeJoint",
          module = "lovr.physics",
          constructors = {
            "lovr.physics.newHingeJoint"
          },
          related = {
            "Collider"
          },
          extends = "Joint",
          methods = {
            {
              name = "getAnchors",
              summary = "Get the anchor points of the