/* SPDX-License-Identifier: MIT */

use zram_generator::{config, generator};

use anyhow::Result;
use fs_extra::dir::{copy, CopyOptions};
use std::fs;
use std::path::Path;
use std::process::Command;
use tempfile::TempDir;

fn prepare_directory(srcroot: &Path) -> Result<TempDir> {
    let rootdir = TempDir::new()?;
    let root = rootdir.path();

    let opts = CopyOptions::new();
    for p in ["etc", "usr", "proc"]
        .iter()
        .map(|p| srcroot.join(p))
        .filter(|p| p.exists())
    {
        copy(p, root, &opts)?;
    }

    let output_directory = root.join("run/units");
    fs::create_dir_all(output_directory)?;

    Ok(rootdir)
}

fn test_generation(path: &str) -> Result<Vec<config::Device>> {
    let srcroot = Path::new(path);
    let rootdir = prepare_directory(&srcroot)?;
    let root = rootdir.path();

    let kernel_override = match config::kernel_zram_option(root) {
        Some(true) => true,
        Some(false) => {
            return Ok(vec![]);
        }
        _ => false,
    };
    let devices = config::read_all_devices(root, kernel_override)?;

    let output_directory = root.join("run/units");
    generator::run_generator(&devices, &output_directory, true)?;

    // Compare output directory to expected value.
    // ExecStart lines include the full path to the generating binary,
    // so exclude them from comparison.
    let diff = Command::new("diff")
        .arg("-u")
        .arg("--recursive")
        .arg("--exclude=.empty")
        .arg("--ignore-matching-lines=^# Automatically generated by .*")
        .arg("--ignore-matching-lines=^ExecStart=/.* --setup-device '%i'")
        .arg("--ignore-matching-lines=^ExecStop=/.* --reset-device '%i'")
        .arg(srcroot.join("run.expected"))
        .arg(root.join("run"))
        .output()?;
    println!("stdout:\n{}", String::from_utf8_lossy(&diff.stdout));
    println!("stderr:\n{}", String::from_utf8_lossy(&diff.stderr));
    assert!(diff.status.success());

    Ok(devices)
}

#[test]
fn test_01_basic() {
    let devices = test_generation("tests/01-basic").unwrap();
    assert_eq!(devices.len(), 1);
    let d = devices.iter().next().unwrap();
    assert!(d.is_swap());
    assert_eq!(d.host_memory_limit_mb, None);
    assert_eq!(d.zram_fraction, 0.5);
    assert_eq!(d.options, "discard");
}

#[test]
fn test_02_zstd() {
    let devices = test_generation("tests/02-zstd").unwrap();
    assert_eq!(devices.len(), 1);
    let d = devices.iter().next().unwrap();
    assert!(d.is_swap());
    assert_eq!(d.host_memory_limit_mb.unwrap(), 2050);
    assert_eq!(d.zram_fraction, 0.75);
    assert_eq!(d.compression_algorithm.as_ref().unwrap(), "zstd");
    assert_eq!(d.options, "discard");
}

#[test]
fn test_03_too_much_memory() {
    let devices = test_generation("tests/03-too-much-memory").unwrap();
    assert_eq!(devices.len(), 0);
}

#[test]
fn test_04_dropins() {
    let devices = test_generation("tests/04-dropins").unwrap();
    assert_eq!(devices.len(), 2);

    for d in &devices {
        assert!(d.is_swap());

        match d.name.as_str() {
            "zram0" => {
                assert_eq!(d.host_memory_limit_mb.unwrap(), 1235);
                assert_eq!(d.zram_fraction, 0.5);
                assert_eq!(d.options, "discard");
            }
            "zram2" => {
                assert!(d.host_memory_limit_mb.is_none());
                assert_eq!(d.zram_fraction, 0.8);
                assert_eq!(d.options, "");
            }
            _ => panic!("Unexpected device {}", d),
        }
    }
}

#[test]
fn test_05_kernel_disabled() {
    let devices = test_generation("tests/05-kernel-disabled").unwrap();
    assert_eq!(devices.len(), 0);
}

#[test]
fn test_06_kernel_enabled() {
    let devices = test_generation("tests/06-kernel-enabled").unwrap();
    assert_eq!(devices.len(), 1);
    let d = devices.iter().next().unwrap();
    assert!(d.is_swap());
    assert_eq!(d.host_memory_limit_mb, None);
    assert_eq!(d.zram_fraction, 0.5);
    assert_eq!(d.options, "discard");
}

#[test]
fn test_07_mount_point() {
    let devices = test_generation("tests/07-mount-point").unwrap();
    assert_eq!(devices.len(), 4);
    test_07_devices(devices);
}

/// cargo-package refuses to pack files with `\`s in them,
/// so we split them off to be able to push to crates.io
#[test]
fn test_07a_mount_point_excl() {
    if !Path::new("tests/07a-mount-point-excl").exists() {
        return;
    }

    let devices = test_generation("tests/07a-mount-point-excl").unwrap();
    assert_eq!(devices.len(), 1);
    test_07_devices(devices);
}

fn test_07_devices(devices: Vec<config::Device>) {
    for d in &devices {
        assert!(!d.is_swap());
        assert_eq!(d.host_memory_limit_mb, None);
        assert_eq!(d.zram_fraction, 0.5);
        assert_eq!(d.fs_type.as_ref().unwrap(), "ext4");
        assert_eq!(d.effective_fs_type(), "ext4");
        match &d.name[..] {
            "zram11" => {
                assert_eq!(
                    d.mount_point.as_ref().unwrap(),
                    Path::new("/var/compressed")
                );
                assert_eq!(d.options, "discard");
            }
            "zram12" => {
                assert_eq!(d.mount_point.as_ref().unwrap(), Path::new("/var/folded"));
                assert_eq!(d.options, "discard,casefold");
            }
            "zram13" => {
                assert_eq!(d.mount_point.as_ref().unwrap(), Path::new("/foo//bar/baz/"));
                assert_eq!(d.options, "discard");
            }
            "zram14" => {
                assert_eq!(d.mount_point.as_ref().unwrap(), Path::new("/.żupan-ci3pły"));
                assert_eq!(d.options, "discard");
            }
            "zram15" => {
                assert_eq!(d.mount_point.as_ref().unwrap(), Path::new("///"));
                assert_eq!(d.options, "discard");
            }
            _ => panic!("Unexpected device {}", d),
        }
    }
}

#[test]
fn test_08_plain_device() {
    let devices = test_generation("tests/08-plain-device").unwrap();
    assert_eq!(devices.len(), 1);
    let d = devices.iter().next().unwrap();
    assert!(!d.is_swap());
    assert_eq!(d.host_memory_limit_mb, None);
    assert_eq!(d.zram_fraction, 0.5);
    assert!(d.mount_point.is_none());
    assert_eq!(d.fs_type.as_ref().unwrap(), "ext2");
    assert_eq!(d.effective_fs_type(), "ext2");
    assert_eq!(d.options, "discard");
}
