/*
 * Copyright (c) 2017, 2021 Frank Fischer <frank-fischer@shadow-soft.de>
 *
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see  <http://www.gnu.org/licenses/>
 */

//! Simple crate that automatically open compressed files.
//!
//! The compressor used is determined by the file extension. If the
//! corresponding compression library is not available (i.e. the corresponding
//! feature is not activated), the crate tries to use an external compression
//! tool (gzip, bzip2 or xz).
use std::fs::File;

#[allow(dead_code)]
mod tool;

#[cfg(feature = "flate2")]
mod gz {
    use std::fs::File;
    use std::io::Result;

    type Read = flate2::read::GzDecoder<File>;
    type Write = flate2::write::GzEncoder<File>;

    pub fn read(fname: &str) -> Result<Read> {
        File::open(fname).map(Read::new)
    }

    pub fn write(fname: &str) -> Result<Write> {
        File::create(fname).map(|f| Write::new(f, flate2::Compression::default()))
    }
}

#[cfg(not(feature = "flate2"))]
mod gz {
    use std::io::Result;

    type Read = crate::tool::ToolRead;
    type Write = crate::tool::ToolWrite;

    pub fn read(fname: &str) -> Result<Read> {
        Read::new("/usr/bin/gzip", &["-d", "-c", fname])
    }

    pub fn write(fname: &str) -> Result<Write> {
        Write::new_with_file("/usr/bin/gzip", &["-"], fname)
    }
}

#[cfg(feature = "bzip2")]
mod bzip {
    use std::fs::File;
    use std::io::Result;

    type Read = bzip2::read::BzDecoder<File>;
    type Write = bzip2::write::BzEncoder<File>;

    pub fn read(fname: &str) -> Result<Read> {
        Ok(Read::new(File::open(fname)?))
    }

    pub fn write(fname: &str) -> Result<Write> {
        Ok(Write::new(File::create(fname)?, bzip2::Compression::default()))
    }
}

#[cfg(not(feature = "bzip2"))]
mod bzip {
    use std::io::Result;
    type Read = crate::tool::ToolRead;
    type Write = crate::tool::ToolWrite;

    pub fn read(fname: &str) -> Result<Read> {
        Read::new("/usr/bin/bzip2", &["-d", "-c", fname])
    }

    pub fn write(fname: &str) -> Result<Write> {
        Write::new_with_file("/usr/bin/bzip2", &["-"], fname)
    }
}

#[cfg(feature = "rust-lzma")]
mod xz {
    use std::fmt::Arguments;
    use std::fs::File;
    use std::io::{self, Error, ErrorKind, Result};
    use std::result;

    type Read = lzma::LzmaReader<File>;

    pub struct Write(Option<lzma::LzmaWriter<File>>);

    impl Write {
        fn new(fname: &str) -> result::Result<Self, lzma::error::LzmaError> {
            Ok(Write(Some(lzma::LzmaWriter::new_compressor(File::create(fname)?, 6)?)))
        }
    }

    impl io::Write for Write {
        fn write(&mut self, buf: &[u8]) -> io::Result<usize> {
            self.0.as_mut().unwrap().write(buf)
        }

        fn flush(&mut self) -> io::Result<()> {
            self.0.as_mut().unwrap().flush()
        }

        fn write_all(&mut self, buf: &[u8]) -> io::Result<()> {
            self.0.as_mut().unwrap().write_all(buf)
        }

        fn write_fmt(&mut self, fmt: Arguments) -> io::Result<()> {
            self.0.as_mut().unwrap().write_fmt(fmt)
        }
    }

    impl Drop for Write {
        fn drop(&mut self) {
            self.0.take().unwrap().finish().expect("Finish XZ stream");
        }
    }

    pub fn read(fname: &str) -> Result<Read> {
        match Read::new_decompressor(File::open(fname)?) {
            Ok(r) => Ok(r),
            Err(lzma::error::LzmaError::Io(e)) => Err(e),
            Err(e) => Err(Error::new(ErrorKind::Other, e)),
        }
    }

    pub fn write(fname: &str) -> Result<Write> {
        match Write::new(fname) {
            Ok(r) => Ok(r),
            Err(lzma::error::LzmaError::Io(e)) => Err(e),
            Err(e) => Err(Error::new(ErrorKind::Other, e)),
        }
    }
}

#[cfg(not(feature = "rust-lzma"))]
mod xz {
    use std::io::Result;
    type Read = crate::tool::ToolRead;
    type Write = crate::tool::ToolWrite;

    pub fn read(fname: &str) -> Result<Read> {
        Read::new("/usr/bin/xz", &["-d", "-c", fname])
    }

    pub fn write(fname: &str) -> Result<Write> {
        Write::new_with_file("/usr/bin/xz", &["-"], fname)
    }
}

/// Open a possibly compressed file for reading.
///
/// The file is specified by the given `filename`. The file is
/// decompressed by an external compression tool determined by the
/// file extension:
///
///  * .gz uses `gzip`
///  * .bzip2 uses `xz`
///  * .xz and .lzma uses `xz`
///  * everything else open the file directly without compression.
pub fn read(filename: &str) -> std::io::Result<Box<dyn std::io::Read>> {
    Ok(if filename.ends_with(".gz") {
        Box::new(gz::read(filename)?)
    } else if filename.ends_with(".bz2") {
        Box::new(bzip::read(filename)?)
    } else if filename.ends_with(".xz") || filename.ends_with(".lzma") {
        Box::new(xz::read(filename)?)
    } else {
        Box::new(File::open(filename)?)
    })
}

/// Open a possibly compressed file for writing.
///
/// The file is specified by the given `filename`. The file is
/// decompressed by an external compression tool determined by the
/// file extension:
///
///  * .gz uses `gzip`
///  * .bzip2 uses `xz`
///  * .xz and .lzma use `xz`
///  * everything else open the file directly without compression.
pub fn write(filename: &str) -> std::io::Result<Box<dyn std::io::Write>> {
    Ok(if filename.ends_with(".gz") {
        Box::new(gz::write(filename)?)
    } else if filename.ends_with(".bz2") {
        Box::new(bzip::write(filename)?)
    } else if filename.ends_with(".xz") || filename.ends_with(".lzma") {
        Box::new(xz::write(filename)?)
    } else {
        Box::new(File::create(filename)?)
    })
}

#[test]
fn test_write_and_read() {
    let test_str = "Hello World!\n";

    for &ext in &["", ".gz", ".bz2", ".xz"] {
        let mut dir = std::env::temp_dir();
        dir.push(format!("__zopen-rs-test__{}", ext));
        {
            let mut f = write(dir.to_str().unwrap()).unwrap();
            write!(f, "{}", test_str).unwrap();
            f.flush().unwrap();
        }

        assert!(dir.exists());

        {
            let mut f = read(dir.to_str().unwrap()).unwrap();
            let mut data = "".to_string();
            assert_eq!(f.read_to_string(&mut data).unwrap(), test_str.len());
            assert_eq!(data, test_str);
        }

        std::fs::remove_file(dir).unwrap();
    }
}
