/*
 * Copyright (c) 2017, 2021 Frank Fischer <frank-fischer@shadow-soft.de>
 *
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see  <http://www.gnu.org/licenses/>
 */

use std::ffi::OsStr;
use std::fs::File;
use std::io;
use std::process::{Child, Command, Stdio};

/// Read handle that waits for command on close.
pub struct ToolRead(Child);

impl io::Read for ToolRead {
    fn read(&mut self, buf: &mut [u8]) -> io::Result<usize> {
        self.0.stdout.as_mut().unwrap().read(buf)
    }

    fn read_to_end(&mut self, buf: &mut Vec<u8>) -> io::Result<usize> {
        self.0.stdout.as_mut().unwrap().read_to_end(buf)
    }
}

impl Drop for ToolRead {
    fn drop(&mut self) {
        self.0.wait().unwrap();
    }
}

impl ToolRead {
    /// Run a command and read from its standard output.
    ///
    /// The command's standard input will be closed.
    pub fn new<I, S>(cmd: &str, args: I) -> io::Result<ToolRead>
    where
        I: IntoIterator<Item = S>,
        S: AsRef<OsStr>,
    {
        Ok(ToolRead(
            Command::new(cmd)
                .env_clear()
                .args(args)
                .stdin(Stdio::null())
                .stdout(Stdio::piped())
                .spawn()?,
        ))
    }

    /// Run a command and read from its standard output.
    ///
    /// The command's standard input will read from the given file.
    pub fn new_with_file<I, S>(cmd: &str, args: I, filename: &str) -> io::Result<ToolRead>
    where
        I: IntoIterator<Item = S>,
        S: AsRef<OsStr>,
    {
        Ok(ToolRead(
            Command::new(cmd)
                .env_clear()
                .args(args)
                .stdin(File::open(filename)?)
                .stdout(Stdio::piped())
                .spawn()?,
        ))
    }
}

/// Write handle that waits for command on close.
pub struct ToolWrite(Child);

impl io::Write for ToolWrite {
    fn write(&mut self, buf: &[u8]) -> io::Result<usize> {
        self.0.stdin.as_mut().unwrap().write(buf)
    }

    fn flush(&mut self) -> io::Result<()> {
        self.0.stdin.as_mut().unwrap().flush()
    }
}

impl Drop for ToolWrite {
    fn drop(&mut self) {
        self.0.wait().unwrap();
    }
}

impl ToolWrite {
    /// Run a command and write to its standard input.
    ///
    /// The command's standard output will be closed.
    pub fn new<I, S>(cmd: &str, args: I) -> io::Result<ToolWrite>
    where
        I: IntoIterator<Item = S>,
        S: AsRef<OsStr>,
    {
        Ok(ToolWrite(
            Command::new(cmd)
                .env_clear()
                .args(args)
                .stdin(Stdio::piped())
                .stdout(Stdio::null())
                .spawn()?,
        ))
    }

    /// Run a command and write to its standard input.
    ///
    /// The command's standard output will written to the given file.
    pub fn new_with_file<I, S>(cmd: &str, args: I, filename: &str) -> io::Result<ToolWrite>
    where
        I: IntoIterator<Item = S>,
        S: AsRef<OsStr>,
    {
        Ok(ToolWrite(
            Command::new(cmd)
                .env_clear()
                .args(args)
                .stdin(Stdio::piped())
                .stdout(File::create(filename)?)
                .spawn()?,
        ))
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use std::error::Error;
    use std::io::{Read, Write};
    use tempfile::NamedTempFile;

    #[test]
    fn test_tool_read() -> Result<(), Box<dyn Error>> {
        let mut input = ToolRead::new("/usr/bin/echo", &["Hallo Welt"])?;
        let mut s = String::new();

        input.read_to_string(&mut s)?;
        assert_eq!(s, "Hallo Welt\n");

        Ok(())
    }

    #[test]
    fn test_tool_read_with_file() -> Result<(), Box<dyn Error>> {
        let mut f = NamedTempFile::new()?;
        writeln!(f, "Hello world")?;
        let mut input = ToolRead::new_with_file("/usr/bin/cat", &[] as &[&str], f.path().to_str().unwrap())?;
        let mut s = String::new();

        input.read_to_string(&mut s)?;
        assert_eq!(s, "Hello world\n");

        Ok(())
    }

    #[test]
    fn test_tool_write() -> Result<(), Box<dyn Error>> {
        let mut f = NamedTempFile::new()?;
        {
            let mut output = ToolWrite::new("/usr/bin/tee", &[f.path().to_str().unwrap()])?;
            writeln!(output, "Hello world!")?;
        }
        let mut s = String::new();

        f.read_to_string(&mut s)?;
        assert_eq!(s, "Hello world!\n");

        Ok(())
    }

    #[test]
    fn test_tool_write_file() -> Result<(), Box<dyn Error>> {
        let mut f = NamedTempFile::new()?;
        {
            let mut output = ToolWrite::new_with_file("/usr/bin/tac", &[] as &[&str], f.path().to_str().unwrap())?;
            writeln!(output, "Welt")?;
            writeln!(output, "Hallo")?;
        }

        let mut s = String::new();
        f.read_to_string(&mut s)?;
        assert_eq!(s, "Hallo\nWelt\n");

        Ok(())
    }
}
