//! The data types sent to and returned from the API client.
use schemars::JsonSchema;
use serde::{Deserialize, Serialize};

/**
 * Payee:<br>`master` - master account holder pays.<br>`sub` - Sub account holder pays.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum PayMode {
    #[serde(rename = "master")]
    Master,
    #[serde(rename = "sub")]
    Sub,
    FallthroughString(String),
}

impl std::fmt::Display for PayMode {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PayMode::Master => "master",
            PayMode::Sub => "sub",
            PayMode::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for PayMode {
    fn default() -> PayMode {
        PayMode::Master
    }
}

/// Account options object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Options {
    /**
     * Account options object.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub billing_auto_renew: bool,
    /**
     * Account options object.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub meeting_connector_list: Vec<String>,
    /**
     * Account options object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub pay_mode: Option<PayMode>,
    /**
     * Account options object.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub room_connector_list: Vec<String>,
    /**
     * Account options object.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub share_mc: bool,
    /**
     * Account options object.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub share_rc: bool,
}

/// The account object represents an account on Zoom. The account owner is the user who created the account or had an account created for them. You can read more about the Zoom account structure <a href='https://medium.com/zoom-developer-blog/a-brief-look-at-zoom-account-structures-1d19c745bf8a' target='_blank'>here</a>.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Account {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
     * The account object represents an account on Zoom. The account owner is the user who created the account or had an account created for them. You can read more about the Zoom account structure <a href='https://medium.com/zoom-developer-blog/a-brief-look-at-zoom-account-structures-1d19c745bf8a' target='_blank'>here</a>.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub options: Option<Options>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    /**
     * The account object represents an account on Zoom. The account owner is the user who created the account or had an account created for them. You can read more about the Zoom account structure <a href='https://medium.com/zoom-developer-blog/a-brief-look-at-zoom-account-structures-1d19c745bf8a' target='_blank'>here</a>.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub vanity_url: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Accounts {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub account_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub account_number: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub account_type: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub created_at: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub owner_email: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub seats: i64,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub subscription_end_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub subscription_start_time: Option<chrono::DateTime<chrono::Utc>>,
}

/// Account object in the account list.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountListItem {
    /**
     * Account object in the account list.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub accounts: Vec<Accounts>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Domains {}

/// Account plan object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountPlan {
    /**
     * Account plan object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub hosts: i64,
    /**
     * Account plan object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

/// Account base plan object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PlanBase {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub hosts: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

/// Additional audio conferencing <a href="https://marketplace.zoom.us/docs/api-reference/other-references/plans#audio-conferencing-plans">plan type</a>.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PlanAudio {
    /**
     * Additional audio conferencing <a href="https://marketplace.zoom.us/docs/api-reference/other-references/plans#audio-conferencing-plans">plan type</a>.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub callout_countries: String,
    /**
     * Additional audio conferencing <a href="https://marketplace.zoom.us/docs/api-reference/other-references/plans#audio-conferencing-plans">plan type</a>.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub ddi_numbers: i64,
    /**
     * Additional audio conferencing <a href="https://marketplace.zoom.us/docs/api-reference/other-references/plans#audio-conferencing-plans">plan type</a>.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub premium_countries: String,
    /**
     * Additional audio conferencing <a href="https://marketplace.zoom.us/docs/api-reference/other-references/plans#audio-conferencing-plans">plan type</a>.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub tollfree_countries: String,
    /**
     * Additional audio conferencing <a href="https://marketplace.zoom.us/docs/api-reference/other-references/plans#audio-conferencing-plans">plan type</a>.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

/// Additional phone base plans.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountPlansPhonePlanBase {
    /**
     * Additional phone base plans.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub callout_countries: String,
    /**
     * Additional phone base plans.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PlanNumber {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub hosts: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

/// Phone Plan Object
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PhonePlan {
    /**
     * Phone Plan Object
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub plan_base: Option<AccountPlansPhonePlanBase>,
    /**
     * Phone Plan Object
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub plan_calling: Vec<PlanNumber>,
    /**
     * Phone Plan Object
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub plan_number: Vec<PlanNumber>,
}

/// Account Plans object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountPlans {
    /**
     * Account Plans object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub plan_audio: Option<PlanAudio>,
    /**
     * Account base plan object.
     */
    #[serde()]
    pub plan_base: PlanBase,
    /**
     * Account Plans object.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub plan_large_meeting: Vec<AccountPlan>,
    /**
     * Account Plans object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub plan_phone: Option<PhonePlan>,
    /**
     * Account Plans object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub plan_recording: String,
    /**
     * Account Plans object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub plan_room_connector: Option<AccountPlan>,
    /**
     * Account Plans object.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub plan_webinar: Vec<AccountPlan>,
    /**
     * Account Plans object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub plan_zoom_rooms: Option<AccountPlan>,
}

/**
 * Determine how participants can join the audio portion of the meeting.<br>`both` - Telephony and VoIP.<br>`telephony` - Audio PSTN telephony only.<br>`voip` - VoIP only.<br>`thirdParty` - 3rd party audio conference.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AudioType {
    #[serde(rename = "both")]
    Both,
    #[serde(rename = "telephony")]
    Telephony,
    #[serde(rename = "thirdParty")]
    ThirdParty,
    #[serde(rename = "voip")]
    Voip,
    FallthroughString(String),
}

impl std::fmt::Display for AudioType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AudioType::Both => "both",
            AudioType::Telephony => "telephony",
            AudioType::ThirdParty => "thirdParty",
            AudioType::Voip => "voip",
            AudioType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for AudioType {
    fn default() -> AudioType {
        AudioType::Both
    }
}

/**
 * Require a password for a meeting held using Personal Meeting ID (PMI) This setting is always enabled for free accounts and Pro accounts with a single host and cannot be modified for these accounts.
 *   
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum RequirePasswordPmiMeetings {
    #[serde(rename = "all")]
    All,
    #[serde(rename = "jbh_only")]
    JbhOnly,
    #[serde(rename = "none")]
    None,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for RequirePasswordPmiMeetings {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            RequirePasswordPmiMeetings::All => "all",
            RequirePasswordPmiMeetings::JbhOnly => "jbh_only",
            RequirePasswordPmiMeetings::None => "none",
            RequirePasswordPmiMeetings::Noop => "",
            RequirePasswordPmiMeetings::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for RequirePasswordPmiMeetings {
    fn default() -> RequirePasswordPmiMeetings {
        RequirePasswordPmiMeetings::Noop
    }
}
impl RequirePasswordPmiMeetings {
    pub fn is_noop(&self) -> bool {
        matches!(self, RequirePasswordPmiMeetings::Noop)
    }
}

/// Account wide meeting/webinar [password requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingPasswordRequirement {
    /**
     * Account wide meeting/webinar [password requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub consecutive_characters_length: i64,
    /**
     * Account wide meeting/webinar [password requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_letter: bool,
    /**
     * Account wide meeting/webinar [password requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_number: bool,
    /**
     * Account wide meeting/webinar [password requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_special_character: bool,
    /**
     * Account wide meeting/webinar [password requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_upper_and_lower_characters: bool,
    /**
     * Account wide meeting/webinar [password requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub length: i64,
    /**
     * Account wide meeting/webinar [password requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub only_allow_numeric: bool,
    /**
     * Account wide meeting/webinar [password requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub weak_enhance_detection: bool,
}

/// Account Settings: Schedule Meeting.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ScheduleMeeting {
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub audio_type: Option<AudioType>,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enforce_login: bool,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub enforce_login_domains: String,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enforce_login_with_domains: bool,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub force_pmi_jbh_password: bool,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub host_video: bool,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub jbh_time: i64,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub join_before_host: bool,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub meeting_password_requirement: Option<MeetingPasswordRequirement>,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub not_store_meeting_topic: bool,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub participant_video: bool,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub personal_meeting: bool,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_instant_meetings: bool,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub require_password_for_pmi_meetings: Option<RequirePasswordPmiMeetings>,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_scheduled_meetings: bool,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_scheduling_new_meetings: bool,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub use_pmi_for_instant_meetings: bool,
    /**
     * Account Settings: Schedule Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub use_pmi_for_scheduled_meetings: bool,
}

/**
 * Play sound when participants join or leave.<br>`host` - Heard by host only.<br>`all` - Heard by host and all attendees.<br>`none` - Disable.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum EntryExitChime {
    #[serde(rename = "all")]
    All,
    #[serde(rename = "host")]
    Host,
    #[serde(rename = "none")]
    None,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for EntryExitChime {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            EntryExitChime::All => "all",
            EntryExitChime::Host => "host",
            EntryExitChime::None => "none",
            EntryExitChime::Noop => "",
            EntryExitChime::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for EntryExitChime {
    fn default() -> EntryExitChime {
        EntryExitChime::Noop
    }
}
impl EntryExitChime {
    pub fn is_noop(&self) -> bool {
        matches!(self, EntryExitChime::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Files {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub is_default: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub size: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

/// Settings to manage virtual background.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct VirtualBackgroundSettings {
    /**
     * Settings to manage virtual background.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_upload_custom: bool,
    /**
     * Settings to manage virtual background.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_videos: bool,
    /**
     * Settings to manage virtual background.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * Settings to manage virtual background.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub files: Vec<Files>,
}

/**
 * Indicates who can share their screen or content during meetings. The value can be one of the following: <br>
 *   `host`: Only host can share the screen.<br>
 *   `all`: Both hosts and attendees can share their screen during meetings. For Webinar, the hosts and panelists can start screen sharing, but not the attendees.
 *   
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum WhoCanShareScreen {
    #[serde(rename = "all")]
    All,
    #[serde(rename = "host")]
    Host,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for WhoCanShareScreen {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            WhoCanShareScreen::All => "all",
            WhoCanShareScreen::Host => "host",
            WhoCanShareScreen::Noop => "",
            WhoCanShareScreen::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for WhoCanShareScreen {
    fn default() -> WhoCanShareScreen {
        WhoCanShareScreen::Noop
    }
}
impl WhoCanShareScreen {
    pub fn is_noop(&self) -> bool {
        matches!(self, WhoCanShareScreen::Noop)
    }
}

/// Account Settings: In Meeting.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct InMeeting {
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub alert_guest_join: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_live_streaming: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_participants_to_rename: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_show_zoom_windows: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub annotation: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub anonymous_question_answer: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub attendee_on_hold: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub auto_answer: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub auto_saving_chat: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub breakout_room: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub chat: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub closed_caption: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub co_host: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub custom_data_center_regions: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub custom_live_streaming_service: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub custom_service_instructions: String,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub data_center_regions: Vec<String>,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub dscp_audio: i64,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub dscp_marking: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub dscp_video: i64,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub e_2e_encryption: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub entry_exit_chime: Option<EntryExitChime>,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub far_end_camera_control: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub feedback: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub file_transfer: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub group_hd: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub join_from_desktop: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub join_from_mobile: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub meeting_reactions: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub original_audio: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub p_2p_connetion: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub p_2p_ports: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub polling: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub ports_range: String,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub post_meeting_feedback: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub private_chat: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub record_play_own_voice: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub remote_control: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub request_permission_to_unmute: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub screen_sharing: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub sending_default_email_invites: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_a_join_from_your_browser_link: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_meeting_control_toolbar: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub stereo_audio: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub use_html_format_email: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub virtual_background: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub virtual_background_settings: Option<VirtualBackgroundSettings>,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub watermark: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub webinar_question_answer: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub whiteboard: bool,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub who_can_share_screen: Option<WhoCanShareScreen>,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub who_can_share_screen_when_someone_is_sharing: Option<WhoCanShareScreen>,
    /**
     * Account Settings: In Meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub workplace_by_facebook: bool,
}

/// Account Settings: Notification.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct EmailNotification {
    /**
     * Account Settings: Notification.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub alternative_host_reminder: bool,
    /**
     * Account Settings: Notification.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cancel_meeting_reminder: bool,
    /**
     * Account Settings: Notification.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cloud_recording_avaliable_reminder: bool,
    /**
     * Account Settings: Notification.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub jbh_reminder: bool,
    /**
     * Account Settings: Notification.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub low_host_count_reminder: bool,
    /**
     * Account Settings: Notification.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub schedule_for_reminder: bool,
}

/// Account Settings: Zoom Rooms.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ZoomRooms {
    /**
     * Account Settings: Zoom Rooms.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub auto_start_stop_scheduled_meetings: bool,
    /**
     * Account Settings: Zoom Rooms.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cmr_for_instant_meeting: bool,
    /**
     * Account Settings: Zoom Rooms.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub force_private_meeting: bool,
    /**
     * Account Settings: Zoom Rooms.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub hide_host_information: bool,
    /**
     * Account Settings: Zoom Rooms.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub list_meetings_with_calendar: bool,
    /**
     * Account Settings: Zoom Rooms.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub start_airplay_manually: bool,
    /**
     * Account Settings: Zoom Rooms.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub ultrasonic: bool,
    /**
     * Account Settings: Zoom Rooms.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub upcoming_meeting_alert: bool,
    /**
     * Account Settings: Zoom Rooms.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub weekly_system_restart: bool,
    /**
     * Account Settings: Zoom Rooms.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub zr_post_meeting_feedback: bool,
}

/// This object refers to the [enhanced password rules](https://support.zoom.us/hc/en-us/articles/360034675592-Advanced-security-settings#h_bf8a25f6-9a66-447a-befd-f02ed3404f89) that allows Zoom account admins and owners to apply extra requiremets to the users' Zoom login password.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PasswordRequirement {
    /**
     * This object refers to the [enhanced password rules](https://support.zoom.us/hc/en-us/articles/360034675592-Advanced-security-settings#h_bf8a25f6-9a66-447a-befd-f02ed3404f89) that allows Zoom account admins and owners to apply extra requiremets to the users' Zoom login password.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub consecutive_characters_length: i64,
    /**
     * This object refers to the [enhanced password rules](https://support.zoom.us/hc/en-us/articles/360034675592-Advanced-security-settings#h_bf8a25f6-9a66-447a-befd-f02ed3404f89) that allows Zoom account admins and owners to apply extra requiremets to the users' Zoom login password.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_special_character: bool,
    /**
     * This object refers to the [enhanced password rules](https://support.zoom.us/hc/en-us/articles/360034675592-Advanced-security-settings#h_bf8a25f6-9a66-447a-befd-f02ed3404f89) that allows Zoom account admins and owners to apply extra requiremets to the users' Zoom login password.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub minimum_password_length: i64,
    /**
     * This object refers to the [enhanced password rules](https://support.zoom.us/hc/en-us/articles/360034675592-Advanced-security-settings#h_bf8a25f6-9a66-447a-befd-f02ed3404f89) that allows Zoom account admins and owners to apply extra requiremets to the users' Zoom login password.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub weak_enhance_detection: bool,
}

/**
 * Settings for 2FA( [two factor authentication](https://support.zoom.us/hc/en-us/articles/360038247071) ). The value can be one of the following:
 *   `all`: Two factor authentication will be enabled for all users in the account.<br>
 *   `none`: Two factor authentication is disabled.<br>
 *   `group`: Two factor authentication will be enabled for users belonging to specific groups. If 2FA is enabled for certain groups, the group IDs of the group(s) will be provided in the `sign_in_with_two_factor_auth_groups` field.<br>
 *   `role`: Two factor authentication will be enabled only for users assigned with specific roles in the account. If 2FA is enabled for specific roles, the role IDs will be provided in the
 *   `sign_in_with_two_factor_auth_roles` field.
 *   
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum SignInWithTwoFactorAuth {
    #[serde(rename = "all")]
    All,
    #[serde(rename = "group")]
    Group,
    #[serde(rename = "none")]
    None,
    #[serde(rename = "role")]
    Role,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for SignInWithTwoFactorAuth {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            SignInWithTwoFactorAuth::All => "all",
            SignInWithTwoFactorAuth::Group => "group",
            SignInWithTwoFactorAuth::None => "none",
            SignInWithTwoFactorAuth::Role => "role",
            SignInWithTwoFactorAuth::Noop => "",
            SignInWithTwoFactorAuth::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for SignInWithTwoFactorAuth {
    fn default() -> SignInWithTwoFactorAuth {
        SignInWithTwoFactorAuth::Noop
    }
}
impl SignInWithTwoFactorAuth {
    pub fn is_noop(&self) -> bool {
        matches!(self, SignInWithTwoFactorAuth::Noop)
    }
}

/// [Security settings](https://support.zoom.us/hc/en-us/articles/360034675592-Advanced-security-settings#h_bf8a25f6-9a66-447a-befd-f02ed3404f89) of an Account.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Security {
    /**
     * [Security settings](https://support.zoom.us/hc/en-us/articles/360034675592-Advanced-security-settings#h_bf8a25f6-9a66-447a-befd-f02ed3404f89) of an Account.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub admin_change_name_pic: bool,
    /**
     * [Security settings](https://support.zoom.us/hc/en-us/articles/360034675592-Advanced-security-settings#h_bf8a25f6-9a66-447a-befd-f02ed3404f89) of an Account.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub hide_billing_info: bool,
    /**
     * [Security settings](https://support.zoom.us/hc/en-us/articles/360034675592-Advanced-security-settings#h_bf8a25f6-9a66-447a-befd-f02ed3404f89) of an Account.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub import_photos_from_devices: bool,
    /**
     * [Security settings](https://support.zoom.us/hc/en-us/articles/360034675592-Advanced-security-settings#h_bf8a25f6-9a66-447a-befd-f02ed3404f89) of an Account.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub password_requirement: Option<PasswordRequirement>,
    /**
     * [Security settings](https://support.zoom.us/hc/en-us/articles/360034675592-Advanced-security-settings#h_bf8a25f6-9a66-447a-befd-f02ed3404f89) of an Account.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub sign_again_period_for_inactivity_on_client: i64,
    /**
     * [Security settings](https://support.zoom.us/hc/en-us/articles/360034675592-Advanced-security-settings#h_bf8a25f6-9a66-447a-befd-f02ed3404f89) of an Account.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub sign_again_period_for_inactivity_on_web: i64,
    /**
     * [Security settings](https://support.zoom.us/hc/en-us/articles/360034675592-Advanced-security-settings#h_bf8a25f6-9a66-447a-befd-f02ed3404f89) of an Account.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sign_in_with_two_factor_auth: Option<SignInWithTwoFactorAuth>,
    /**
     * [Security settings](https://support.zoom.us/hc/en-us/articles/360034675592-Advanced-security-settings#h_bf8a25f6-9a66-447a-befd-f02ed3404f89) of an Account.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub sign_in_with_two_factor_auth_groups: Vec<String>,
    /**
     * [Security settings](https://support.zoom.us/hc/en-us/articles/360034675592-Advanced-security-settings#h_bf8a25f6-9a66-447a-befd-f02ed3404f89) of an Account.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub sign_in_with_two_factor_auth_roles: Vec<String>,
}

/**
 * Automatic recording:<br>`local` - Record on local.<br>`cloud` -  Record on cloud.<br>`none` - Disabled.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AutoRecording {
    #[serde(rename = "cloud")]
    Cloud,
    #[serde(rename = "local")]
    Local,
    #[serde(rename = "none")]
    None,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for AutoRecording {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AutoRecording::Cloud => "cloud",
            AutoRecording::Local => "local",
            AutoRecording::None => "none",
            AutoRecording::Noop => "",
            AutoRecording::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for AutoRecording {
    fn default() -> AutoRecording {
        AutoRecording::Noop
    }
}
impl AutoRecording {
    pub fn is_noop(&self) -> bool {
        matches!(self, AutoRecording::Noop)
    }
}

/// This object represents the minimum password requirements set for recordings via Account Recording Settings.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct RecordingPasswordRequirement {
    /**
     * This object represents the minimum password requirements set for recordings via Account Recording Settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_letter: bool,
    /**
     * This object represents the minimum password requirements set for recordings via Account Recording Settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_number: bool,
    /**
     * This object represents the minimum password requirements set for recordings via Account Recording Settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_special_character: bool,
    /**
     * This object represents the minimum password requirements set for recordings via Account Recording Settings.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub length: i64,
    /**
     * This object represents the minimum password requirements set for recordings via Account Recording Settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub only_allow_numeric: bool,
}

/// Setting to allow cloud recording access only from specific IP address ranges.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct IpAddressAccessControl {
    /**
     * Setting to allow cloud recording access only from specific IP address ranges.
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * Setting to allow cloud recording access only from specific IP address ranges.
     *
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub ip_addresses_or_ranges: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Settings {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub audio_file: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cc_transcript_file: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub chat_file: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub chat_with_sender_email: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub video_file: bool,
}

/// [Archiving solution](https://support.zoom.us/hc/en-us/articles/360050431572-Archiving-Meeting-and-Webinar-data) settings. This setting can only be used if you have been granted with archiving solution access by the Zoom support team.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Archive {
    /**
     * [Archiving solution](https://support.zoom.us/hc/en-us/articles/360050431572-Archiving-Meeting-and-Webinar-data) settings. This setting can only be used if you have been granted with archiving solution access by the Zoom support team.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * [Archiving solution](https://support.zoom.us/hc/en-us/articles/360050431572-Archiving-Meeting-and-Webinar-data) settings. This setting can only be used if you have been granted with archiving solution access by the Zoom support team.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub settings: Option<Settings>,
    /**
     * [Archiving solution](https://support.zoom.us/hc/en-us/articles/360050431572-Archiving-Meeting-and-Webinar-data) settings. This setting can only be used if you have been granted with archiving solution access by the Zoom support team.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

/// Account Settings: Recording.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Recording {
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub account_user_access_recording: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_recovery_deleted_cloud_recordings: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub archive: Option<Archive>,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub auto_delete_cmr: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub auto_delete_cmr_days: i64,
    /**
     * Account Settings: Recording.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub auto_recording: Option<AutoRecording>,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cloud_recording: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cloud_recording_download: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cloud_recording_download_host: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub host_delete_cloud_recording: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub ip_address_access_control: Option<IpAddressAccessControl>,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub local_recording: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub prevent_host_access_recording: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub record_audio_file: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub record_audio_file_each_participant: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub record_gallery_view: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub record_speaker_view: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub recording_audio_transcript: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub recording_disclaimer: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recording_password_requirement: Option<RecordingPasswordRequirement>,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub required_password_for_existing_cloud_recordings: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub save_chat_text: bool,
    /**
     * Account Settings: Recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_timestamp: bool,
}

/// Indicates where most of the participants call into or call from duriing a meeting.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct TelephonyRegions {
    /**
     * Indicates where most of the participants call into or call from duriing a meeting.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub allowed_values: Vec<String>,
    /**
     * Indicates where most of the participants call into or call from duriing a meeting.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub selection_values: Vec<String>,
}

/// Account Settings: Telephony.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Telephony {
    /**
     * Account Settings: Telephony.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub audio_conference_info: String,
    /**
     * Account Settings: Telephony.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub telephony_regions: Option<TelephonyRegions>,
    /**
     * Account Settings: Telephony.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub third_party_audio: bool,
}

/// Account Settings: Integration.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Integration {
    /**
     * Account Settings: Integration.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize",
        rename = "box"
    )]
    pub box_: bool,
    /**
     * Account Settings: Integration.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub dropbox: bool,
    /**
     * Account Settings: Integration.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub google_calendar: bool,
    /**
     * Account Settings: Integration.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub google_drive: bool,
    /**
     * Account Settings: Integration.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub kubi: bool,
    /**
     * Account Settings: Integration.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub microsoft_one_drive: bool,
}

/// Account Settings: Feature.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Feature {
    /**
     * Account Settings: Feature.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub meeting_capacity: i64,
}

/// Account Settings: TSP.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Tsp {
    /**
     * Account Settings: TSP.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub call_out: bool,
    /**
     * Account Settings: TSP.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub call_out_countries: Vec<String>,
    /**
     * Account Settings: TSP.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub display_toll_free_numbers: bool,
    /**
     * Account Settings: TSP.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_international_numbers_link: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct RecordingStorageLocation {
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub allowed_values: Vec<String>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub value: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Profile {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recording_storage_location: Option<RecordingStorageLocation>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountSettings {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub email_notification: Option<EmailNotification>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub feature: Option<Feature>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub in_meeting: Option<InMeeting>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub integration: Option<Integration>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub profile: Option<Profile>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recording: Option<Recording>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub schedule_meeting: Option<ScheduleMeeting>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub security: Option<Security>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub telephony: Option<Telephony>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tsp: Option<Tsp>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub zoom_rooms: Option<ZoomRooms>,
}

/// Indicates where most of the participants call into or call from duriing a meeting.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateAccountSettingsTelephonyRegions {
    /**
     * Indicates where most of the participants call into or call from duriing a meeting.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub selection_values: Vec<String>,
}

/// Account Settings Update: Telephony.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateAccountSettingsTelephony {
    /**
     * Account Settings Update: Telephony.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub audio_conference_info: String,
    /**
     * Account Settings Update: Telephony.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub telephony_regions: Option<UpdateAccountSettingsTelephonyRegions>,
    /**
     * Account Settings Update: Telephony.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub third_party_audio: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateAccountSettings {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub email_notification: Option<EmailNotification>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub feature: Option<Feature>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub in_meeting: Option<InMeeting>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub integration: Option<Integration>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub profile: Option<Profile>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recording: Option<Recording>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub schedule_meeting: Option<ScheduleMeeting>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub security: Option<Security>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub telephony: Option<UpdateAccountSettingsTelephony>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tsp: Option<Tsp>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub zoom_rooms: Option<ZoomRooms>,
}

/// List of managed domains.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DomainsList {
    /**
     * List of managed domains.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub domains: Vec<Domains>,
    /**
     * List of managed domains.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

/// Billing Contact object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct BillingContact {
    /**
     * Billing Contact object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address: String,
    /**
     * Billing Contact object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub apt: String,
    /**
     * Billing Contact object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub city: String,
    /**
     * Billing Contact object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
     * Billing Contact object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * Billing Contact object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
     * Billing Contact object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
     * Billing Contact object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number: String,
    /**
     * Billing Contact object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub state: String,
    /**
     * Billing Contact object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub zip: String,
}

/// Billing Contact object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Contact {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address: String,
    /**
     * Billing Contact object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub apt: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub city: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub state: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub zip: String,
}

/// Custom Question.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CustomQuestion {
    /**
     * Custom Question.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub title: String,
    /**
     * Custom Question.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub value: String,
}

/// DateTime Object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DateTime {
    /**
     * DateTime Object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub from: Option<chrono::NaiveDate>,
    /**
     * DateTime Object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub to: Option<chrono::NaiveDate>,
}

/**
 * Device protocol:<br>`H.323` - H.323.<br>`SIP` - SIP.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum Protocol {
    #[serde(rename = "H.323")]
    H323,
    #[serde(rename = "SIP")]
    Sip,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for Protocol {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Protocol::H323 => "H.323",
            Protocol::Sip => "SIP",
            Protocol::Noop => "",
            Protocol::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for Protocol {
    fn default() -> Protocol {
        Protocol::Noop
    }
}
impl Protocol {
    pub fn is_noop(&self) -> bool {
        matches!(self, Protocol::Noop)
    }
}

/**
 * Device encryption:<br>`auto` - auto.<br>`yes` - yes.<br>`no` - no.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum Encryption {
    #[serde(rename = "auto")]
    Auto,
    #[serde(rename = "no")]
    No,
    #[serde(rename = "yes")]
    Yes,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for Encryption {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Encryption::Auto => "auto",
            Encryption::No => "no",
            Encryption::Yes => "yes",
            Encryption::Noop => "",
            Encryption::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for Encryption {
    fn default() -> Encryption {
        Encryption::Noop
    }
}
impl Encryption {
    pub fn is_noop(&self) -> bool {
        matches!(self, Encryption::Noop)
    }
}

/// The H.323/SIP device object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Device {
    /**
     * Device encryption:<br>`auto` - auto.<br>`yes` - yes.<br>`no` - no.
     */
    #[serde(default, skip_serializing_if = "Encryption::is_noop")]
    pub encryption: Encryption,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub ip: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Device protocol:<br>`H.323` - H.323.<br>`SIP` - SIP.
     */
    #[serde(default, skip_serializing_if = "Protocol::is_noop")]
    pub protocol: Protocol,
}

/// Group object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Group {
    /**
     * Group object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Group object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_members: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Groups {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
}

/// All of the following types:
///
/// - `Group`
/// - `Groups`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum GroupsAllOf {
    /**
     * Group object.
     */
    Group(Group),
    Groups(Groups),
}

impl GroupsAllOf {
    pub fn group(&self) -> Option<&Group> {
        if let GroupsAllOf::Group(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn groups(&self) -> Option<&Groups> {
        if let GroupsAllOf::Groups(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// List of Groups.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GroupList {
    /**
     * List of Groups.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub groups: Vec<GroupsAllOf>,
    /**
     * List of Groups.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

/// Group member object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GroupMember {
    /**
     * Group member object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * Group member object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
     * Group member object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Group member object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
     * Group member object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct TrackingFields {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub field: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub value: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub visible: bool,
}

/// Occurence object. This object is only returned for Recurring Webinars.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Occurrence {
    /**
     * Occurence object. This object is only returned for Recurring Webinars.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Occurence object. This object is only returned for Recurring Webinars.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub occurrence_id: String,
    /**
     * Occurence object. This object is only returned for Recurring Webinars.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Occurence object. This object is only returned for Recurring Webinars.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub status: String,
}

/**
 * Determine how participants can join the audio portion of the meeting.<br>`both` - Both Telephony and VoIP.<br>`telephony` - Telephony only.<br>`voip` - VoIP only.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum Audio {
    #[serde(rename = "both")]
    Both,
    #[serde(rename = "telephony")]
    Telephony,
    #[serde(rename = "voip")]
    Voip,
    FallthroughString(String),
}

impl std::fmt::Display for Audio {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Audio::Both => "both",
            Audio::Telephony => "telephony",
            Audio::Voip => "voip",
            Audio::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for Audio {
    fn default() -> Audio {
        Audio::Both
    }
}

/**
 * Type of number.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum Type {
    #[serde(rename = "toll")]
    Toll,
    #[serde(rename = "tollfree")]
    Tollfree,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for Type {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Type::Toll => "toll",
            Type::Tollfree => "tollfree",
            Type::Noop => "",
            Type::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for Type {
    fn default() -> Type {
        Type::Noop
    }
}
impl Type {
    pub fn is_noop(&self) -> bool {
        matches!(self, Type::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GlobalDialInNumbers {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub city: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
    /**
     * Type of number.
     */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<Type>,
}

/**
 * Choose between enhanced encryption and [end-to-end encryption](https://support.zoom.us/hc/en-us/articles/360048660871) when starting or a meeting. When using end-to-end encryption, several features (e.g. cloud recording, phone/SIP/H.323 dial-in) will be **automatically disabled**. <br><br>The value of this field can be one of the following:<br>
 *   `enhanced_encryption`: Enhanced encryption. Encryption is stored in the cloud if you enable this option. <br>
 *   
 *   `e2ee`: [End-to-end encryption](https://support.zoom.us/hc/en-us/articles/360048660871). The encryption key is stored in your local device and can not be obtained by anyone else. Enabling this setting also **disables** the following features: join before host, cloud recording, streaming, live transcription, breakout rooms, polling, 1:1 private chat, and meeting reactions.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum EncryptionType {
    #[serde(rename = "e2ee")]
    E2Ee,
    #[serde(rename = "enhanced_encryption")]
    EnhancedEncryption,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for EncryptionType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            EncryptionType::E2Ee => "e2ee",
            EncryptionType::EnhancedEncryption => "enhanced_encryption",
            EncryptionType::Noop => "",
            EncryptionType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for EncryptionType {
    fn default() -> EncryptionType {
        EncryptionType::Noop
    }
}
impl EncryptionType {
    pub fn is_noop(&self) -> bool {
        matches!(self, EncryptionType::Noop)
    }
}

/**
 * Specify whether to allow users from specific regions to join this meeting; or block users from specific regions from joining this meeting. <br><br>
 *   `approve`: Allow users from specific regions/countries to join this meeting. If this setting is selected, the approved regions/countries must be included in the `approved_list`.<br><br>
 *   `deny`: Block users from specific regions/countries from joining this meeting. If this setting is selected, the approved regions/countries must be included in the `denied_list`
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum Method {
    #[serde(rename = "approve")]
    Approve,
    #[serde(rename = "deny")]
    Deny,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for Method {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Method::Approve => "approve",
            Method::Deny => "deny",
            Method::Noop => "",
            Method::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for Method {
    fn default() -> Method {
        Method::Noop
    }
}
impl Method {
    pub fn is_noop(&self) -> bool {
        matches!(self, Method::Noop)
    }
}

/// Approve or block users from specific regions/countries from joining this meeting.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ApprovedDeniedCountriesRegions {
    /**
     * Approve or block users from specific regions/countries from joining this meeting.
     *
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub approved_list: Vec<String>,
    /**
     * Approve or block users from specific regions/countries from joining this meeting.
     *
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub denied_list: Vec<String>,
    /**
     * Approve or block users from specific regions/countries from joining this meeting.
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * Approve or block users from specific regions/countries from joining this meeting.
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub method: Option<Method>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AuthenticationException {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Rooms {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub participants: Vec<String>,
}

/// Setting to [pre-assign breakout rooms](https://support.zoom.us/hc/en-us/articles/360032752671-Pre-assigning-participants-to-breakout-rooms#h_36f71353-4190-48a2-b999-ca129861c1f4).
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct BreakoutRoom {
    /**
     * Setting to [pre-assign breakout rooms](https://support.zoom.us/hc/en-us/articles/360032752671-Pre-assigning-participants-to-breakout-rooms#h_36f71353-4190-48a2-b999-ca129861c1f4).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * Setting to [pre-assign breakout rooms](https://support.zoom.us/hc/en-us/articles/360032752671-Pre-assigning-participants-to-breakout-rooms#h_36f71353-4190-48a2-b999-ca129861c1f4).
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub rooms: Vec<Rooms>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Interpreters {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub languages: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct LanguageInterpretation {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * Information associated with the interpreter.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub interpreters: Vec<Interpreters>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CustomKeys {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub key: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub value: String,
}

/// Meeting settings.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingSettings {
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_multiple_devices: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub alternative_hosts: String,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub alternative_hosts_email_notification: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub approval_type: i64,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub approved_or_denied_countries_or_regions: Option<ApprovedDeniedCountriesRegions>,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub audio: Option<Audio>,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authentication_domains: String,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub authentication_exception: Vec<AuthenticationException>,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authentication_name: String,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authentication_option: String,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub auto_recording: Option<AutoRecording>,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub breakout_room: Option<BreakoutRoom>,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub close_registration: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cn_meeting: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub contact_email: String,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub contact_name: String,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub custom_keys: Vec<CustomKeys>,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub encryption_type: Option<EncryptionType>,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enforce_login: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub enforce_login_domains: String,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub global_dial_in_countries: Vec<String>,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub global_dial_in_numbers: Vec<GlobalDialInNumbers>,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub host_video: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub in_meeting: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub jbh_time: i64,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub join_before_host: bool,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub language_interpretation: Option<LanguageInterpretation>,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub meeting_authentication: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub mute_upon_entry: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub participant_video: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub registrants_confirmation_email: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub registrants_email_notification: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub registration_type: i64,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_share_button: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub use_pmi: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub waiting_room: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub watermark: bool,
}

/**
 * This field is required **if you're scheduling a recurring meeting of type** `2` to state which day(s) of the week the meeting should repeat. <br> <br> The value for this field could be a number between `1` to `7` in string format. For instance, if the meeting should recur on Sunday, provide `"1"` as the value of this field.<br><br> **Note:** If you would like the meeting to occur on multiple days of a week, you should provide comma separated values for this field. For instance, if the meeting should recur on Sundays and Tuesdays provide `"1,3"` as the value of this field.
 *   
 *    <br>`1`  - Sunday. <br>`2` - Monday.<br>`3` - Tuesday.<br>`4` -  Wednesday.<br>`5` -  Thursday.<br>`6` - Friday.<br>`7` - Saturday.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum WeeklyDays {
    #[serde(rename = "1")]
    One,
    #[serde(rename = "2")]
    Two,
    #[serde(rename = "3")]
    Three,
    #[serde(rename = "4")]
    Four,
    #[serde(rename = "5")]
    Five,
    #[serde(rename = "6")]
    Six,
    #[serde(rename = "7")]
    Seven,
    FallthroughString(String),
}

impl std::fmt::Display for WeeklyDays {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            WeeklyDays::One => "1",
            WeeklyDays::Two => "2",
            WeeklyDays::Three => "3",
            WeeklyDays::Four => "4",
            WeeklyDays::Five => "5",
            WeeklyDays::Six => "6",
            WeeklyDays::Seven => "7",
            WeeklyDays::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for WeeklyDays {
    fn default() -> WeeklyDays {
        WeeklyDays::One
    }
}

/// Recurrence object. Use this object only for a meeting with type `8` i.e., a recurring meeting with fixed time.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Recurrence {
    /**
     * Recurrence object. Use this object only for a meeting with type `8` i.e., a recurring meeting with fixed time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub end_date_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Recurrence object. Use this object only for a meeting with type `8` i.e., a recurring meeting with fixed time.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub end_times: i64,
    /**
     * Recurrence object. Use this object only for a meeting with type `8` i.e., a recurring meeting with fixed time.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub monthly_day: i64,
    /**
     * Recurrence object. Use this object only for a meeting with type `8` i.e., a recurring meeting with fixed time.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub monthly_week: i64,
    /**
     * Recurrence object. Use this object only for a meeting with type `8` i.e., a recurring meeting with fixed time.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub monthly_week_day: i64,
    /**
     * Recurrence object. Use this object only for a meeting with type `8` i.e., a recurring meeting with fixed time.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub repeat_interval: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
    /**
     * Recurrence object. Use this object only for a meeting with type `8` i.e., a recurring meeting with fixed time.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub weekly_days: Option<WeeklyDays>,
}

/// Meeting object
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingInfo {
    /**
     * Meeting object
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub agenda: String,
    /**
     * Meeting object
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub created_at: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Meeting object
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Meeting object
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub h_323_password: String,
    /**
     * Meeting object
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub join_url: String,
    /**
     * Meeting object
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub occurrences: Vec<Occurrence>,
    /**
     * Meeting object
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    /**
     * Meeting object
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub pmi: i64,
    /**
     * Meeting object
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub pre_schedule: bool,
    /**
     * Meeting object
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recurrence: Option<Recurrence>,
    /**
     * Meeting object
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub settings: Option<MeetingSettings>,
    /**
     * Meeting object
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Meeting object
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub start_url: String,
    /**
     * Meeting object
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
    /**
     * Meeting object
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Meeting object
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub tracking_fields: Vec<TrackingFields>,
    /**
     * Meeting object
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

/**
 * Meeting status
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum Status {
    #[serde(rename = "started")]
    Started,
    #[serde(rename = "waiting")]
    Waiting,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for Status {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Status::Started => "started",
            Status::Waiting => "waiting",
            Status::Noop => "",
            Status::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for Status {
    fn default() -> Status {
        Status::Noop
    }
}
impl Status {
    pub fn is_noop(&self) -> bool {
        matches!(self, Status::Noop)
    }
}

/// Meeting object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingInfoGet {
    /**
     * Meeting object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub agenda: String,
    /**
     * Meeting object.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub created_at: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Meeting object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Meeting object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub encrypted_password: String,
    /**
     * Meeting object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub h_323_password: String,
    /**
     * Meeting object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub join_url: String,
    /**
     * Meeting object.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub occurrences: Vec<Occurrence>,
    /**
     * Meeting object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    /**
     * Meeting object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub pmi: i64,
    /**
     * Meeting object.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub pre_schedule: bool,
    /**
     * Meeting object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recurrence: Option<Recurrence>,
    /**
     * Meeting object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub settings: Option<MeetingSettings>,
    /**
     * Meeting object.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Meeting object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub start_url: String,
    /**
     * Meeting object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<Status>,
    /**
     * Meeting object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
    /**
     * Meeting object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Meeting object.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub tracking_fields: Vec<TrackingFields>,
    /**
     * Meeting object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

/// Meeting metric details.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingMetric {
    /**
     * Meeting metric details.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub custom_keys: Vec<CustomKeys>,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub dept: String,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub duration: String,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub end_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_3rd_party_audio: bool,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_pstn: bool,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_recording: bool,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_screen_share: bool,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_sip: bool,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_video: bool,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_voip: bool,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub host: String,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub in_room_participants: i64,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub participants: i64,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_type: String,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub uuid: String,
}

/// Meeting invitation details.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingInvitation {
    /**
     * Meeting invitation details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub invitation: String,
}

/// Meeting live stream.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingLiveStream {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub page_url: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub stream_key: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub stream_url: String,
}

/**
 * Update the status of a livestream.
 *   
 *   The value can be one of the following:<br>
 *   `start`: Start a live stream. <br>
 *   `stop`: Stop an ongoing live stream.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum Action {
    #[serde(rename = "start")]
    Start,
    #[serde(rename = "stop")]
    Stop,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for Action {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Action::Start => "start",
            Action::Stop => "stop",
            Action::Noop => "",
            Action::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for Action {
    fn default() -> Action {
        Action::Noop
    }
}
impl Action {
    pub fn is_noop(&self) -> bool {
        matches!(self, Action::Noop)
    }
}

/// Update the settings of a live streaming session. The settings can only be updated for a live stream that has been stopped. You can not update the settings of an ongoing live stream.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingLiveStreamStatusSettings {
    /**
     * Update the settings of a live streaming session. The settings can only be updated for a live stream that has been stopped. You can not update the settings of an ongoing live stream.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub active_speaker_name: bool,
    /**
     * Update the settings of a live streaming session. The settings can only be updated for a live stream that has been stopped. You can not update the settings of an ongoing live stream.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
}

/// Meeting live stream status.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingLiveStreamStatus {
    /**
     * Meeting live stream status.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub action: Option<Action>,
    /**
     * Meeting live stream status.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub settings: Option<MeetingLiveStreamStatusSettings>,
}

/// Update the live stream session's settings. You can **only** update settings for a stopped live stream.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct WebinarLiveStreamStatusSettings {
    /**
     * Update the live stream session's settings. You can \*\*only\*\* update settings for a stopped live stream.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub active_speaker_name: bool,
    /**
     * Update the live stream session's settings. You can \*\*only\*\* update settings for a stopped live stream.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
}

/// Webinar live stream status.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct WebinarLiveStreamStatus {
    /**
     * Webinar live stream status.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub action: Option<Action>,
    /**
     * Webinar live stream status.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub settings: Option<WebinarLiveStreamStatusSettings>,
}

/// Pagination Object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Pagination {
    /**
     * Pagination Object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Pagination Object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_count: i64,
    /**
     * Pagination Object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_number: i64,
    /**
     * Pagination Object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * Pagination Object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

/// Pagination object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PaginationToken {
    /**
     * Pagination object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Pagination object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_count: i64,
    /**
     * Pagination object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * Pagination object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

/// Pagination object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PaginationToken4ImChat {
    /**
     * Pagination object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Pagination object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
}

/// Panelist base object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Panelist {
    /**
     * Panelist base object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * Panelist base object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Panelists {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub join_url: String,
}

/// All of the following types:
///
/// - `Groups`
/// - `Panelist`
/// - `Panelists`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum PanelistsAllOf {
    Groups(Groups),
    /**
     * Panelist base object.
     */
    Panelist(Panelist),
    Panelists(Panelists),
}

impl PanelistsAllOf {
    pub fn groups(&self) -> Option<&Groups> {
        if let PanelistsAllOf::Groups(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn panelist(&self) -> Option<&Panelist> {
        if let PanelistsAllOf::Panelist(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn panelists(&self) -> Option<&Panelists> {
        if let PanelistsAllOf::Panelists(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// List of panelists.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PanelistList {
    /**
     * List of panelists.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub panelists: Vec<PanelistsAllOf>,
    /**
     * List of panelists.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

/// Quality of Service object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct QosAudio {
    /**
     * Quality of Service object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub avg_loss: String,
    /**
     * Quality of Service object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bitrate: String,
    /**
     * Quality of Service object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub jitter: String,
    /**
     * Quality of Service object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub latency: String,
    /**
     * Quality of Service object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub max_loss: String,
}

/**
 * The recording status.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum RecordingStatus {
    #[serde(rename = "completed")]
    Completed,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for RecordingStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            RecordingStatus::Completed => "completed",
            RecordingStatus::Noop => "",
            RecordingStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for RecordingStatus {
    fn default() -> RecordingStatus {
        RecordingStatus::Noop
    }
}
impl RecordingStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, RecordingStatus::Noop)
    }
}

/// Recording file object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct RecordingData {
    /**
     * Recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub deleted_time: String,
    /**
     * Recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub download_url: String,
    /**
     * Recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_f64",
        deserialize_with = "crate::utils::deserialize_null_f64::deserialize"
    )]
    pub file_size: f64,
    /**
     * Recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub file_type: String,
    /**
     * Recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub meeting_id: String,
    /**
     * Recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub play_url: String,
    /**
     * Recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub recording_end: String,
    /**
     * Recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub recording_start: String,
    /**
     * Recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub recording_type: String,
    /**
     * Recording file object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<RecordingStatus>,
}

/// The recording file object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct RecordingSeparateAudio {
    /**
     * The recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub download_url: String,
    /**
     * The recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub file_name: String,
    /**
     * The recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_f64",
        deserialize_with = "crate::utils::deserialize_null_f64::deserialize"
    )]
    pub file_size: f64,
    /**
     * The recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub file_type: String,
    /**
     * The recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * The recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub play_url: String,
    /**
     * The recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub recording_end: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * The recording file object.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub recording_start: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * The recording file object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<RecordingStatus>,
}

/**
 * Determine how the meeting recording is shared.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ShareRecording {
    #[serde(rename = "internally")]
    Internally,
    #[serde(rename = "none")]
    None,
    #[serde(rename = "publicly")]
    Publicly,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for ShareRecording {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ShareRecording::Internally => "internally",
            ShareRecording::None => "none",
            ShareRecording::Publicly => "publicly",
            ShareRecording::Noop => "",
            ShareRecording::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for ShareRecording {
    fn default() -> ShareRecording {
        ShareRecording::Noop
    }
}
impl ShareRecording {
    pub fn is_noop(&self) -> bool {
        matches!(self, ShareRecording::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct RecordingSettings {
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub approval_type: i64,
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authentication_domains: String,
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authentication_option: String,
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub on_demand: bool,
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub recording_authentication: bool,
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub send_email_to_host: bool,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub share_recording: Option<ShareRecording>,
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_social_share_buttons: bool,
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub viewer_download: bool,
}

/// Registrant Status
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct RecordingRegistrantStatus {
    /**
     * Specify whether to allow users from specific regions to join this meeting; or block users from specific regions from joining this meeting. <br><br>
     *  `approve`: Allow users from specific regions/countries to join this meeting. If this setting is selected, the approved regions/countries must be included in the `approved_list`.<br><br>
     *  `deny`: Block users from specific regions/countries from joining this meeting. If this setting is selected, the approved regions/countries must be included in the `denied_list`
     */
    #[serde(default, skip_serializing_if = "Method::is_noop")]
    pub action: Method,
    /**
     * Registrant Status
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub registrants: Vec<String>,
}

/// Registrant base object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Registrant {
    /**
     * Registrant base object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address: String,
    /**
     * Registrant base object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub city: String,
    /**
     * Registrant base object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub comments: String,
    /**
     * Registrant base object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
     * Registrant base object.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub custom_questions: Vec<CustomQuestion>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
     * Registrant base object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub industry: String,
    /**
     * Registrant base object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub job_title: String,
    /**
     * Registrant base object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
     * Registrant base object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub no_of_employees: String,
    /**
     * Registrant base object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub org: String,
    /**
     * Registrant base object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone: String,
    /**
     * Registrant base object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub purchasing_time_frame: String,
    /**
     * Registrant base object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub role_in_purchase_process: String,
    /**
     * Registrant base object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub state: String,
    /**
     * Registrant base object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub zip: String,
}

/**
 * Registrant Status:<br>`approve` - Approve registrant.<br>`cancel` - Cancel previously approved registrant's registration.<br>`deny` - Deny registrant.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum RegistrantStatusAction {
    #[serde(rename = "approve")]
    Approve,
    #[serde(rename = "cancel")]
    Cancel,
    #[serde(rename = "deny")]
    Deny,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for RegistrantStatusAction {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            RegistrantStatusAction::Approve => "approve",
            RegistrantStatusAction::Cancel => "cancel",
            RegistrantStatusAction::Deny => "deny",
            RegistrantStatusAction::Noop => "",
            RegistrantStatusAction::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for RegistrantStatusAction {
    fn default() -> RegistrantStatusAction {
        RegistrantStatusAction::Noop
    }
}
impl RegistrantStatusAction {
    pub fn is_noop(&self) -> bool {
        matches!(self, RegistrantStatusAction::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Assistants {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct RegistrantStatus {
    /**
     * Registrant Status:<br>`approve` - Approve registrant.<br>`cancel` - Cancel previously approved registrant's registration.<br>`deny` - Deny registrant.
     */
    #[serde(default, skip_serializing_if = "RegistrantStatusAction::is_noop")]
    pub action: RegistrantStatusAction,
    /**
     * List of registrants.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub registrants: Vec<Assistants>,
}

/// List of a Role Members
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct RoleMembersList {
    /**
     * List of a Role Members
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub members: Vec<Domains>,
    /**
     * List of a Role Members
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * List of a Role Members
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_count: i64,
    /**
     * List of a Role Members
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_number: i64,
    /**
     * List of a Role Members
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * List of a Role Members
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

/// Add Role Members
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddRoleMembers {
    /**
     * Add Role Members
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub members: Vec<String>,
}

/// Base object for sessions.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Session {
    /**
     * Base object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub agenda: String,
    /**
     * Base object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Base object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    /**
     * Base object for sessions.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recurrence: Option<Recurrence>,
    /**
     * Base object for sessions.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub settings: Option<MeetingSettings>,
    /**
     * Base object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Base object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
    /**
     * Base object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Base object for sessions.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub tracking_fields: Vec<String>,
    /**
     * Base object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SessionUpdateTrackingFields {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub field: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub value: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SettingsData {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub registrants_confirmation_email: bool,
}

/// All of the following types:
///
/// - `MeetingSettings`
/// - `SettingsData`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum SettingsAllOf {
    /**
     * Meeting settings.
     */
    MeetingSettings(MeetingSettings),
    SettingsData(SettingsData),
}

impl SettingsAllOf {
    pub fn meeting_settings(&self) -> Option<&MeetingSettings> {
        if let SettingsAllOf::MeetingSettings(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn settings_data(&self) -> Option<&SettingsData> {
        if let SettingsAllOf::SettingsData(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// Base object for sessions.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SessionUpdate {
    /**
     * Base object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub agenda: String,
    /**
     * Base object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Base object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    /**
     * Base object for sessions.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub pre_schedule: bool,
    /**
     * Base object for sessions.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recurrence: Option<Recurrence>,
    /**
     * Base object for sessions.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub settings: Option<SettingsAllOf>,
    /**
     * Base object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Base object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub template_id: String,
    /**
     * Base object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
    /**
     * Base object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Base object for sessions.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub tracking_fields: Vec<SessionUpdateTrackingFields>,
    /**
     * Base object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

/// Recurrence object. Use this object only for a webinar of type `9` i.e., a recurring webinar with fixed time.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct RecurrenceWebinar {
    /**
     * Recurrence object. Use this object only for a webinar of type `9` i.e., a recurring webinar with fixed time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub end_date_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Recurrence object. Use this object only for a webinar of type `9` i.e., a recurring webinar with fixed time.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub end_times: i64,
    /**
     * Recurrence object. Use this object only for a webinar of type `9` i.e., a recurring webinar with fixed time.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub monthly_day: i64,
    /**
     * Recurrence object. Use this object only for a webinar of type `9` i.e., a recurring webinar with fixed time.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub monthly_week: i64,
    /**
     * Recurrence object. Use this object only for a webinar of type `9` i.e., a recurring webinar with fixed time.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub monthly_week_day: i64,
    /**
     * Recurrence object. Use this object only for a webinar of type `9` i.e., a recurring webinar with fixed time.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub repeat_interval: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
    /**
     * Recurrence object. Use this object only for a webinar of type `9` i.e., a recurring webinar with fixed time.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub weekly_days: String,
}

/**
 * Indicate whether you want attendees to be able to view answered questions only or view all questions.
 *   
 *   * `only`: Attendees are able to view answered questions only.
 *   
 *   *  `all`: Attendees are able to view all questions submitted in the Q&A.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AnswerQuestions {
    #[serde(rename = "all")]
    All,
    #[serde(rename = "only")]
    Only,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for AnswerQuestions {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AnswerQuestions::All => "all",
            AnswerQuestions::Only => "only",
            AnswerQuestions::Noop => "",
            AnswerQuestions::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for AnswerQuestions {
    fn default() -> AnswerQuestions {
        AnswerQuestions::Noop
    }
}
impl AnswerQuestions {
    pub fn is_noop(&self) -> bool {
        matches!(self, AnswerQuestions::Noop)
    }
}

/// [Q&A](https://support.zoom.us/hc/en-us/articles/203686015-Using-Q-A-as-the-webinar-host#:~:text=Overview,and%20upvote%20each%20other's%20questions.) for webinar.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct QuestionAnswer {
    /**
     * [Q&A](https://support.zoom.us/hc/en-us/articles/203686015-Using-Q-A-as-the-webinar-host#:~:text=Overview,and%20upvote%20each%20other's%20questions.) for webinar.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_anonymous_questions: bool,
    /**
     * [Q&A](https://support.zoom.us/hc/en-us/articles/203686015-Using-Q-A-as-the-webinar-host#:~:text=Overview,and%20upvote%20each%20other's%20questions.) for webinar.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub answer_questions: Option<AnswerQuestions>,
    /**
     * [Q&A](https://support.zoom.us/hc/en-us/articles/203686015-Using-Q-A-as-the-webinar-host#:~:text=Overview,and%20upvote%20each%20other's%20questions.) for webinar.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub attendees_can_comment: bool,
    /**
     * [Q&A](https://support.zoom.us/hc/en-us/articles/203686015-Using-Q-A-as-the-webinar-host#:~:text=Overview,and%20upvote%20each%20other's%20questions.) for webinar.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub attendees_can_upvote: bool,
    /**
     * [Q&A](https://support.zoom.us/hc/en-us/articles/203686015-Using-Q-A-as-the-webinar-host#:~:text=Overview,and%20upvote%20each%20other's%20questions.) for webinar.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
}

/// Send reminder email to attendees and panelists.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AttendeesPanelistsReminderEmailNotification {
    /**
     * Send reminder email to attendees and panelists.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * Send reminder email to attendees and panelists.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

/// Send follow-up email to attendees.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct FollowUpAttendeesEmailNotification {
    /**
     * Send follow-up email to attendees.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * Send follow-up email to attendees.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

/// Send follow-up email to absentees.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct FollowUpAbsenteesEmailNotification {
    /**
     * Send follow-up email to absentees.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * Send follow-up email to absentees.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

/// Webinar settings.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct WebinarSettings {
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_multiple_devices: bool,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub alternative_hosts: String,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub approval_type: i64,
    /**
     * Webinar settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub attendees_and_panelists_reminder_email_notification:
        Option<AttendeesPanelistsReminderEmailNotification>,
    /**
     * Webinar settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub audio: Option<Audio>,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authentication_domains: String,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authentication_name: String,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authentication_option: String,
    /**
     * Webinar settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub auto_recording: Option<AutoRecording>,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub close_registration: bool,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub contact_email: String,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub contact_name: String,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email_language: String,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enforce_login: bool,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub enforce_login_domains: String,
    /**
     * Webinar settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub follow_up_absentees_email_notification: Option<FollowUpAbsenteesEmailNotification>,
    /**
     * Webinar settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub follow_up_attendees_email_notification: Option<FollowUpAttendeesEmailNotification>,
    /**
     * Webinar settings.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub global_dial_in_countries: Vec<String>,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub hd_video: bool,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub host_video: bool,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub meeting_authentication: bool,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub notify_registrants: bool,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub on_demand: bool,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub panelists_invitation_email_notification: bool,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub panelists_video: bool,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub post_webinar_survey: bool,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub practice_session: bool,
    /**
     * Webinar settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub question_and_answer: Option<QuestionAnswer>,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub registrants_confirmation_email: bool,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub registrants_email_notification: bool,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub registrants_restrict_number: i64,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub registration_type: i64,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_share_button: bool,
    /**
     * Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub survey_url: String,
}

/// Base webinar object for sessions.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SessionWebinar {
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub agenda: String,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    /**
     * Base webinar object for sessions.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recurrence: Option<RecurrenceWebinar>,
    /**
     * Base webinar object for sessions.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub settings: Option<WebinarSettings>,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Base webinar object for sessions.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub tracking_fields: Vec<SessionUpdateTrackingFields>,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

/// All of the following types:
///
/// - `SettingsData`
/// - `WebinarSettings`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum SessionWebinarUpdateSettingsAllOf {
    SettingsData(SettingsData),
    /**
     * Webinar settings.
     */
    WebinarSettings(WebinarSettings),
}

impl SessionWebinarUpdateSettingsAllOf {
    pub fn settings_data(&self) -> Option<&SettingsData> {
        if let SessionWebinarUpdateSettingsAllOf::SettingsData(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn webinar_settings(&self) -> Option<&WebinarSettings> {
        if let SessionWebinarUpdateSettingsAllOf::WebinarSettings(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// Base webinar object for sessions.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SessionWebinarUpdate {
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub agenda: String,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    /**
     * Base webinar object for sessions.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recurrence: Option<Recurrence>,
    /**
     * Base webinar object for sessions.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub settings: Option<SessionWebinarUpdateSettingsAllOf>,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Base webinar object for sessions.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub tracking_fields: Vec<SessionUpdateTrackingFields>,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

/**
 * Dial-in number types:<br>`toll` - Toll number.<br>`tollfree` -Toll free number.<br>
 *   `media_link` - Media link.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum TspAccountsListDialInNumbersType {
    #[serde(rename = "media_link")]
    MediaLink,
    #[serde(rename = "toll")]
    Toll,
    #[serde(rename = "tollfree")]
    Tollfree,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for TspAccountsListDialInNumbersType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            TspAccountsListDialInNumbersType::MediaLink => "media_link",
            TspAccountsListDialInNumbersType::Toll => "toll",
            TspAccountsListDialInNumbersType::Tollfree => "tollfree",
            TspAccountsListDialInNumbersType::Noop => "",
            TspAccountsListDialInNumbersType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for TspAccountsListDialInNumbersType {
    fn default() -> TspAccountsListDialInNumbersType {
        TspAccountsListDialInNumbersType::Noop
    }
}
impl TspAccountsListDialInNumbersType {
    pub fn is_noop(&self) -> bool {
        matches!(self, TspAccountsListDialInNumbersType::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DialInNumbers {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub code: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country_label: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
    /**
     * Dial-in number types:<br>`toll` - Toll number.<br>`tollfree` -Toll free number.<br>
     *  `media_link` - Media link.
     */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<TspAccountsListDialInNumbersType>,
}

/**
 * Telephony bridge
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum TspBridge {
    #[serde(rename = "EU_TSP_TB")]
    EuTspTb,
    #[serde(rename = "US_TSP_TB")]
    UsTspTb,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for TspBridge {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            TspBridge::EuTspTb => "EU_TSP_TB",
            TspBridge::UsTspTb => "US_TSP_TB",
            TspBridge::Noop => "",
            TspBridge::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for TspBridge {
    fn default() -> TspBridge {
        TspBridge::Noop
    }
}
impl TspBridge {
    pub fn is_noop(&self) -> bool {
        matches!(self, TspBridge::Noop)
    }
}

/// List of TSP accounts.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct TspAccountsList {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub conference_code: String,
    /**
     * List of TSP accounts.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub dial_in_numbers: Vec<DialInNumbers>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub leader_pin: String,
    /**
     * List of TSP accounts.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tsp_bridge: Option<TspBridge>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DedicatedDialInNumber {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
}

/// List of PAC accounts.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Pac {
    /**
     * List of PAC accounts.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub conference_id: i64,
    /**
     * List of PAC accounts.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub dedicated_dial_in_number: Vec<DedicatedDialInNumber>,
    /**
     * List of PAC accounts.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub global_dial_in_numbers: Vec<DedicatedDialInNumber>,
    /**
     * List of PAC accounts.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub listen_only_password: String,
    /**
     * List of PAC accounts.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub participant_password: String,
}

/// The user object represents a specific user on Zoom.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct User {
    /**
     * The user object represents a specific user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub created_at: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * The user object represents a specific user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub dept: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * The user object represents a specific user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
     * The user object represents a specific user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_client_version: String,
    /**
     * The user object represents a specific user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub last_login_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * The user object represents a specific user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
     * The user object represents a specific user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub pmi: i64,
    /**
     * The user object represents a specific user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub role_name: String,
    /**
     * The user object represents a specific user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
    /**
     * The user object represents a specific user on Zoom.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub use_pmi: bool,
}

/// Custom attribute(s) of the user.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CustomAttributes {
    /**
     * Custom attribute(s) of the user.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub key: String,
    /**
     * Custom attribute(s) of the user.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Custom attribute(s) of the user.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub value: String,
}

/**
 * The label to add to the user's phone number. You can only add one label to the user's phone number:
 *   
 *   * `Mobile`
 *   * `Office`
 *   * `Home`
 *   * `Fax`
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum Label {
    #[serde(rename = "Fax")]
    Fax,
    #[serde(rename = "Home")]
    Home,
    #[serde(rename = "Mobile")]
    Mobile,
    #[serde(rename = "Office")]
    Office,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for Label {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Label::Fax => "Fax",
            Label::Home => "Home",
            Label::Mobile => "Mobile",
            Label::Office => "Office",
            Label::Noop => "",
            Label::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for Label {
    fn default() -> Label {
        Label::Noop
    }
}
impl Label {
    pub fn is_noop(&self) -> bool {
        matches!(self, Label::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PhoneNumbers {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub code: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
     * The label to add to the user's phone number. You can only add one label to the user's phone number:
     *  
     *  \* `Mobile`
     *  \* `Office`
     *  \* `Home`
     *  \* `Fax`
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub label: Option<Label>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
}

/// The user update object represents a user on Zoom.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserUpdate {
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub cms_user_id: String,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub company: String,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub custom_attributes: Option<CustomAttributes>,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub dept: String,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub group_id: String,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub host_key: String,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub job_title: String,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub language: String,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location: String,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub manager: String,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_country: String,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number: String,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub phone_numbers: Option<PhoneNumbers>,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub pmi: i64,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub use_pmi: bool,
    /**
     * The user update object represents a user on Zoom.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub vanity_name: String,
}

/// List of user's assistants.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserAssistantsList {
    /**
     * List of user's assistants.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub assistants: Vec<Assistants>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Schedulers {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub pmi: i64,
}

/// List of user's schedulers.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserSchedulersList {
    /**
     * List of user's schedulers.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub schedulers: Vec<Schedulers>,
}

/// Account wide meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserSettingsMeetingPasswordRequirement {
    /**
     * Account wide meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub consecutive_characters_length: i64,
    /**
     * Account wide meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_letter: bool,
    /**
     * Account wide meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_number: bool,
    /**
     * Account wide meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_special_character: bool,
    /**
     * Account wide meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_upper_and_lower_characters: bool,
    /**
     * Account wide meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub length: i64,
    /**
     * Account wide meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub only_allow_numeric: bool,
    /**
     * Account wide meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub weak_enhance_detection: bool,
}

///
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserSettingsMeeting {
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub audio_type: Option<AudioType>,
    /**
     *
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub default_password_for_scheduled_meetings: String,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub embed_password_in_join_link: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub force_pmi_jbh_password: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub host_video: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub join_before_host: bool,
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub meeting_password_requirement: Option<UserSettingsMeetingPasswordRequirement>,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub participants_video: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub personal_meeting: bool,
    /**
     *
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub pmi_password: String,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub pstn_password_protected: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_instant_meetings: bool,
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub require_password_for_pmi_meetings: Option<RequirePasswordPmiMeetings>,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_scheduled_meetings: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_scheduling_new_meetings: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub use_pmi_for_instant_meetings: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub use_pmi_for_scheduled_meetings: bool,
}

///
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserSettingsInMeeting {
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_live_streaming: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub annotation: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub attendee_on_hold: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub auto_saving_chat: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub breakout_room: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub chat: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub closed_caption: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub co_host: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub custom_data_center_regions: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub custom_live_streaming_service: bool,
    /**
     *
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub custom_service_instructions: String,
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub data_center_regions: Vec<String>,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub e_2e_encryption: bool,
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub entry_exit_chime: Option<EntryExitChime>,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub far_end_camera_control: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub feedback: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub file_transfer: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub group_hd: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub join_from_desktop: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub join_from_mobile: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub non_verbal_feedback: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub polling: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub private_chat: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub record_play_voice: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub remote_control: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub remote_support: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub request_permission_to_unmute: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub screen_sharing: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub share_dual_camera: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_a_join_from_your_browser_link: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_meeting_control_toolbar: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub virtual_background: bool,
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub virtual_background_settings: Option<VirtualBackgroundSettings>,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub waiting_room: bool,
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub who_can_share_screen: Option<WhoCanShareScreen>,
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub who_can_share_screen_when_someone_is_sharing: Option<WhoCanShareScreen>,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub workplace_by_facebook: bool,
}

///
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserSettingsNotification {
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub alternative_host_reminder: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cancel_meeting_reminder: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub jbh_reminder: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub schedule_for_reminder: bool,
}

/// This object represents the minimum passcode requirements set for recordings via Account Recording Settings.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserSettingsRecordingPasswordRequirement {
    /**
     * This object represents the minimum passcode requirements set for recordings via Account Recording Settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_letter: bool,
    /**
     * This object represents the minimum passcode requirements set for recordings via Account Recording Settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_number: bool,
    /**
     * This object represents the minimum passcode requirements set for recordings via Account Recording Settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_special_character: bool,
    /**
     * This object represents the minimum passcode requirements set for recordings via Account Recording Settings.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub length: i64,
    /**
     * This object represents the minimum passcode requirements set for recordings via Account Recording Settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub only_allow_numeric: bool,
}

///
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserSettingsRecording {
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub ask_host_to_confirm_disclaimer: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub ask_participants_to_consent_disclaimer: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub auto_delete_cmr: bool,
    /**
     *
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub auto_delete_cmr_days: i64,
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub auto_recording: Option<AutoRecording>,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cloud_recording: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub host_pause_stop_recording: bool,
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub ip_address_access_control: Option<IpAddressAccessControl>,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub local_recording: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub record_audio_file: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub record_audio_file_each_participant: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub record_gallery_view: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub record_speaker_view: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub recording_audio_transcript: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub recording_disclaimer: bool,
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recording_password_requirement: Option<UserSettingsRecordingPasswordRequirement>,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub save_chat_text: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_timestamp: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserSettingsTelephony {
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub audio_conference_info: String,
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_international_numbers_link: bool,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub telephony_regions: Option<TelephonyRegions>,
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub third_party_audio: bool,
}

///
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserSettingsFeature {
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cn_meeting: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub in_meeting: bool,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub large_meeting: bool,
    /**
     *
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub large_meeting_capacity: i64,
    /**
     *
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub meeting_capacity: i64,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub webinar: bool,
    /**
     *
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub webinar_capacity: i64,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub zoom_events: bool,
    /**
     *
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub zoom_events_capacity: i64,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub zoom_phone: bool,
}

/// Account Settings: TSP.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserSettingsTsp {
    /**
     * Account Settings: TSP.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub call_out: bool,
    /**
     * Account Settings: TSP.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub call_out_countries: Vec<String>,
    /**
     * Account Settings: TSP.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_international_numbers_link: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserSettings {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub email_notification: Option<UserSettingsNotification>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub feature: Option<UserSettingsFeature>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub in_meeting: Option<UserSettingsInMeeting>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub profile: Option<Profile>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recording: Option<UserSettingsRecording>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub schedule_meeting: Option<UserSettingsMeeting>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub telephony: Option<UserSettingsTelephony>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tsp: Option<UserSettingsTsp>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SettingsUpdateTelephony {
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub audio_conference_info: String,
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_international_numbers_link: bool,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub telephony_regions: Option<UpdateAccountSettingsTelephonyRegions>,
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub third_party_audio: bool,
}

///
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserSettingsUpdateFeature {
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub large_meeting: bool,
    /**
     *
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub large_meeting_capacity: i64,
    /**
     *
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub meeting_capacity: i64,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub webinar: bool,
    /**
     *
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub webinar_capacity: i64,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub zoom_events: bool,
    /**
     *
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub zoom_events_capacity: i64,
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub zoom_phone: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserSettingsUpdate {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub email_notification: Option<UserSettingsNotification>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub feature: Option<UserSettingsUpdateFeature>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub in_meeting: Option<UserSettingsInMeeting>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub profile: Option<Profile>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recording: Option<UserSettingsRecording>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub schedule_meeting: Option<UserSettingsMeeting>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub telephony: Option<SettingsUpdateTelephony>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tsp: Option<UserSettingsTsp>,
}

///
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserPermissions {
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub permissions: Vec<String>,
}

/// Webinar object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct WebinarInfo {
    /**
     * Webinar object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub agenda: String,
    /**
     * Webinar object.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub created_at: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Webinar object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Webinar object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub join_url: String,
    /**
     * Webinar object.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub occurrences: Vec<Occurrence>,
    /**
     * Webinar object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    /**
     * Webinar object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recurrence: Option<RecurrenceWebinar>,
    /**
     * Webinar object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub settings: Option<WebinarSettings>,
    /**
     * Webinar object.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Webinar object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub start_url: String,
    /**
     * Webinar object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
    /**
     * Webinar object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Webinar object.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub tracking_fields: Vec<SessionUpdateTrackingFields>,
    /**
     * Webinar object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

/// Webinar metric details.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Webinars {
    /**
     * Webinar metric details.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub custom_keys: Vec<CustomKeys>,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub dept: String,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub duration: String,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub end_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_3rd_party_audio: bool,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_pstn: bool,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_recording: bool,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_screen_share: bool,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_sip: bool,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_video: bool,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_voip: bool,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub host: String,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub participants: i64,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_type: String,
    /**
     * Webinar metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub uuid: String,
}

/// All of the following types:
///
/// - `Panelist`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum WebinarPanelistPanelistsAllOf {
    /**
     * Panelist base object.
     */
    Panelist(Panelist),
}

impl WebinarPanelistPanelistsAllOf {
    pub fn panelist(&self) -> Option<&Panelist> {
        let WebinarPanelistPanelistsAllOf::Panelist(ref_) = self;
        Some(ref_)
    }
}

/// Webinar panelist.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct WebinarPanelist {
    /**
     * Webinar panelist.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub panelists: Vec<WebinarPanelistPanelistsAllOf>,
}

/// Zoom room.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ZoomRoom {
    /**
     * Zoom room.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub account_type: String,
    /**
     * Zoom room.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub calender_name: String,
    /**
     * Zoom room.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub camera: String,
    /**
     * Zoom room.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub device_ip: String,
    /**
     * Zoom room.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * Zoom room.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub health: String,
    /**
     * Zoom room.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Zoom room.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub issues: Vec<String>,
    /**
     * Zoom room.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_start_time: String,
    /**
     * Zoom room.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location: String,
    /**
     * Zoom room.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub microphone: String,
    /**
     * Zoom room.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub room_name: String,
    /**
     * Zoom room.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub speaker: String,
    /**
     * Zoom room.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub status: String,
}

/**
 * Poll Question & Answer type:<br>`single` - Single choice<br>`mutliple` - Multiple choice
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum PollQuestionsType {
    #[serde(rename = "multiple")]
    Multiple,
    #[serde(rename = "single")]
    Single,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for PollQuestionsType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PollQuestionsType::Multiple => "multiple",
            PollQuestionsType::Single => "single",
            PollQuestionsType::Noop => "",
            PollQuestionsType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for PollQuestionsType {
    fn default() -> PollQuestionsType {
        PollQuestionsType::Noop
    }
}
impl PollQuestionsType {
    pub fn is_noop(&self) -> bool {
        matches!(self, PollQuestionsType::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Questions {
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub answers: Vec<String>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Poll Question & Answer type:<br>`single` - Single choice<br>`mutliple` - Multiple choice
     */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<PollQuestionsType>,
}

/// Poll
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Poll {
    /**
     * Poll
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub questions: Vec<Questions>,
    /**
     * Poll
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub title: String,
}

/**
 * Field name of the question.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum FieldName {
    #[serde(rename = "address")]
    Address,
    #[serde(rename = "city")]
    City,
    #[serde(rename = "comments")]
    Comments,
    #[serde(rename = "country")]
    Country,
    #[serde(rename = "industry")]
    Industry,
    #[serde(rename = "job_title")]
    JobTitle,
    #[serde(rename = "no_of_employees")]
    NoOfEmployees,
    #[serde(rename = "org")]
    Org,
    #[serde(rename = "phone")]
    Phone,
    #[serde(rename = "purchasing_time_frame")]
    PurchasingTimeFrame,
    #[serde(rename = "role_in_purchase_process")]
    RoleInPurchaseProcess,
    #[serde(rename = "state")]
    State,
    #[serde(rename = "zip")]
    Zip,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for FieldName {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            FieldName::Address => "address",
            FieldName::City => "city",
            FieldName::Comments => "comments",
            FieldName::Country => "country",
            FieldName::Industry => "industry",
            FieldName::JobTitle => "job_title",
            FieldName::NoOfEmployees => "no_of_employees",
            FieldName::Org => "org",
            FieldName::Phone => "phone",
            FieldName::PurchasingTimeFrame => "purchasing_time_frame",
            FieldName::RoleInPurchaseProcess => "role_in_purchase_process",
            FieldName::State => "state",
            FieldName::Zip => "zip",
            FieldName::Noop => "",
            FieldName::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for FieldName {
    fn default() -> FieldName {
        FieldName::Noop
    }
}
impl FieldName {
    pub fn is_noop(&self) -> bool {
        matches!(self, FieldName::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingRegistrantQuestions {
    /**
     * Field name of the question.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub field_name: Option<FieldName>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub required: bool,
}

/**
 * Type of the question being asked.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum MeetingRegistrantQuestionsCustomType {
    #[serde(rename = "short")]
    Short,
    #[serde(rename = "single")]
    Single,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for MeetingRegistrantQuestionsCustomType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            MeetingRegistrantQuestionsCustomType::Short => "short",
            MeetingRegistrantQuestionsCustomType::Single => "single",
            MeetingRegistrantQuestionsCustomType::Noop => "",
            MeetingRegistrantQuestionsCustomType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for MeetingRegistrantQuestionsCustomType {
    fn default() -> MeetingRegistrantQuestionsCustomType {
        MeetingRegistrantQuestionsCustomType::Noop
    }
}
impl MeetingRegistrantQuestionsCustomType {
    pub fn is_noop(&self) -> bool {
        matches!(self, MeetingRegistrantQuestionsCustomType::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CustomQuestions {
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub answers: Vec<String>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub required: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub title: String,
    /**
     * Type of the question being asked.
     */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<MeetingRegistrantQuestionsCustomType>,
}

/// Meeting Registrant Questions
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingRegistrantQuestionsData {
    /**
     * Meeting Registrant Questions
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub custom_questions: Vec<CustomQuestions>,
    /**
     * Meeting Registrant Questions
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub questions: Vec<MeetingRegistrantQuestions>,
}

/// Webinar Registrant Questions
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct WebinarRegistrantQuestions {
    /**
     * Webinar Registrant Questions
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub custom_questions: Vec<String>,
    /**
     * Webinar Registrant Questions
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub questions: Vec<String>,
}

/**
 * Field name.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum RecordingRegistrantQuestionsFieldName {
    #[serde(rename = "address")]
    Address,
    #[serde(rename = "city")]
    City,
    #[serde(rename = "comments")]
    Comments,
    #[serde(rename = "country")]
    Country,
    #[serde(rename = "industry")]
    Industry,
    #[serde(rename = "job_title")]
    JobTitle,
    #[serde(rename = "last_name")]
    LastName,
    #[serde(rename = "no_of_employees")]
    NoOfEmployees,
    #[serde(rename = "org")]
    Org,
    #[serde(rename = "phone")]
    Phone,
    #[serde(rename = "purchasing_time_frame")]
    PurchasingTimeFrame,
    #[serde(rename = "role_in_purchase_process")]
    RoleInPurchaseProcess,
    #[serde(rename = "state")]
    State,
    #[serde(rename = "zip")]
    Zip,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for RecordingRegistrantQuestionsFieldName {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            RecordingRegistrantQuestionsFieldName::Address => "address",
            RecordingRegistrantQuestionsFieldName::City => "city",
            RecordingRegistrantQuestionsFieldName::Comments => "comments",
            RecordingRegistrantQuestionsFieldName::Country => "country",
            RecordingRegistrantQuestionsFieldName::Industry => "industry",
            RecordingRegistrantQuestionsFieldName::JobTitle => "job_title",
            RecordingRegistrantQuestionsFieldName::LastName => "last_name",
            RecordingRegistrantQuestionsFieldName::NoOfEmployees => "no_of_employees",
            RecordingRegistrantQuestionsFieldName::Org => "org",
            RecordingRegistrantQuestionsFieldName::Phone => "phone",
            RecordingRegistrantQuestionsFieldName::PurchasingTimeFrame => "purchasing_time_frame",
            RecordingRegistrantQuestionsFieldName::RoleInPurchaseProcess => {
                "role_in_purchase_process"
            }
            RecordingRegistrantQuestionsFieldName::State => "state",
            RecordingRegistrantQuestionsFieldName::Zip => "zip",
            RecordingRegistrantQuestionsFieldName::Noop => "",
            RecordingRegistrantQuestionsFieldName::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for RecordingRegistrantQuestionsFieldName {
    fn default() -> RecordingRegistrantQuestionsFieldName {
        RecordingRegistrantQuestionsFieldName::Noop
    }
}
impl RecordingRegistrantQuestionsFieldName {
    pub fn is_noop(&self) -> bool {
        matches!(self, RecordingRegistrantQuestionsFieldName::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct RecordingRegistrantQuestions {
    /**
     * Field name.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub field_name: Option<RecordingRegistrantQuestionsFieldName>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub required: bool,
}

/**
 * The type of registration question and answers.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum RecordingRegistrantQuestionsCustomType {
    #[serde(rename = "multiple")]
    Multiple,
    #[serde(rename = "short")]
    Short,
    #[serde(rename = "single")]
    Single,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for RecordingRegistrantQuestionsCustomType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            RecordingRegistrantQuestionsCustomType::Multiple => "multiple",
            RecordingRegistrantQuestionsCustomType::Short => "short",
            RecordingRegistrantQuestionsCustomType::Single => "single",
            RecordingRegistrantQuestionsCustomType::Noop => "",
            RecordingRegistrantQuestionsCustomType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for RecordingRegistrantQuestionsCustomType {
    fn default() -> RecordingRegistrantQuestionsCustomType {
        RecordingRegistrantQuestionsCustomType::Noop
    }
}
impl RecordingRegistrantQuestionsCustomType {
    pub fn is_noop(&self) -> bool {
        matches!(self, RecordingRegistrantQuestionsCustomType::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct RecordingRegistrantQuestionsCustom {
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub answers: Vec<String>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub required: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub title: String,
    /**
     * The type of registration question and answers.
     */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<RecordingRegistrantQuestionsCustomType>,
}

/// Recording Registrant Questions
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct RecordingRegistrantQuestionsData {
    /**
     * Recording Registrant Questions
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub custom_questions: Vec<RecordingRegistrantQuestionsCustom>,
    /**
     * Recording Registrant Questions
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub questions: Vec<RecordingRegistrantQuestions>,
}

/// Tracking Field
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct TrackingField {
    /**
     * Tracking Field
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub field: String,
    /**
     * Tracking Field
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub recommended_values: Vec<String>,
    /**
     * Tracking Field
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub required: bool,
    /**
     * Tracking Field
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub visible: bool,
}

/// Create Webinar settings.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CreateWebinarSettings {
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_multiple_devices: bool,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub alternative_hosts: String,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub approval_type: i64,
    /**
     * Create Webinar settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub attendees_and_panelists_reminder_email_notification:
        Option<AttendeesPanelistsReminderEmailNotification>,
    /**
     * Create Webinar settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub audio: Option<Audio>,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authentication_domains: String,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authentication_option: String,
    /**
     * Create Webinar settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub auto_recording: Option<AutoRecording>,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub close_registration: bool,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub contact_email: String,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub contact_name: String,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email_language: String,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enforce_login: bool,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub enforce_login_domains: String,
    /**
     * Create Webinar settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub follow_up_absentees_email_notification: Option<FollowUpAbsenteesEmailNotification>,
    /**
     * Create Webinar settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub follow_up_attendees_email_notification: Option<FollowUpAttendeesEmailNotification>,
    /**
     * Create Webinar settings.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub global_dial_in_countries: Vec<String>,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub hd_video: bool,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub host_video: bool,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub meeting_authentication: bool,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub on_demand: bool,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub panelists_invitation_email_notification: bool,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub panelists_video: bool,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub post_webinar_survey: bool,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub practice_session: bool,
    /**
     * Create Webinar settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub question_and_answer: Option<QuestionAnswer>,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub registrants_email_notification: bool,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub registrants_restrict_number: i64,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub registration_type: i64,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_share_button: bool,
    /**
     * Create Webinar settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub survey_url: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingCreateTrackingFields {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub field: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub value: String,
}

/// Language interpretation [settings](https://support.zoom.us/hc/en-us/articles/360034919791-Language-interpretation-in-meetings-and-webinars#h_01EGGQFD3Q4BST3378SA762MJ1) for meetings.
///
/// **Note:** This feature is only available on certain Webinar add-on, Education, Business and higher plans. If this feature is not enabled on the host's account, this setting will not be applied for the meeting.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingCreateSettingsLanguageInterpretation {
    /**
     * Language interpretation [settings](https://support.zoom.us/hc/en-us/articles/360034919791-Language-interpretation-in-meetings-and-webinars#h_01EGGQFD3Q4BST3378SA762MJ1) for meetings.
     *  
     *  \*\*Note:\*\* This feature is only available on certain Webinar add-on, Education, Business and higher plans. If this feature is not enabled on the host's account, this setting will not be applied for the meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * Language interpretation [settings](https://support.zoom.us/hc/en-us/articles/360034919791-Language-interpretation-in-meetings-and-webinars#h_01EGGQFD3Q4BST3378SA762MJ1) for meetings.
     *  
     *  \*\*Note:\*\* This feature is only available on certain Webinar add-on, Education, Business and higher plans. If this feature is not enabled on the host's account, this setting will not be applied for the meeting.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub interpreters: Vec<Interpreters>,
}

/// Meeting settings.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingCreateSettings {
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub additional_data_center_regions: Vec<String>,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_multiple_devices: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub alternative_hosts: String,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub alternative_hosts_email_notification: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub approval_type: i64,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub approved_or_denied_countries_or_regions: Option<ApprovedDeniedCountriesRegions>,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub audio: Option<Audio>,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authentication_domains: String,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub authentication_exception: Vec<AuthenticationException>,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authentication_option: String,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub auto_recording: Option<AutoRecording>,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub breakout_room: Option<BreakoutRoom>,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub close_registration: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cn_meeting: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub contact_email: String,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub contact_name: String,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub encryption_type: Option<EncryptionType>,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub global_dial_in_countries: Vec<String>,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub host_video: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub in_meeting: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub jbh_time: i64,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub join_before_host: bool,
    /**
     * Meeting settings.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub language_interpretation: Option<MeetingCreateSettingsLanguageInterpretation>,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub meeting_authentication: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub mute_upon_entry: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub participant_video: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub registrants_confirmation_email: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub registrants_email_notification: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub registration_type: i64,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_share_button: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub use_pmi: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub waiting_room: bool,
    /**
     * Meeting settings.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub watermark: bool,
}

/// Base object for meeting.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingCreate {
    /**
     * Base object for meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub agenda: String,
    /**
     * Base object for meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Base object for meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    /**
     * Base object for meeting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub pre_schedule: bool,
    /**
     * Base object for meeting.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recurrence: Option<Recurrence>,
    /**
     * Base object for meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub schedule_for: String,
    /**
     * Base object for meeting.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub settings: Option<MeetingCreateSettings>,
    /**
     * Base object for meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Base object for meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub template_id: String,
    /**
     * Base object for meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
    /**
     * Base object for meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Base object for meeting.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub tracking_fields: Vec<MeetingCreateTrackingFields>,
    /**
     * Base object for meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

/// Base webinar object for sessions.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CreateWebinar {
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub agenda: String,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    /**
     * Base webinar object for sessions.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recurrence: Option<RecurrenceWebinar>,
    /**
     * Base webinar object for sessions.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub settings: Option<CreateWebinarSettings>,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Base webinar object for sessions.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub tracking_fields: Vec<MeetingCreateTrackingFields>,
    /**
     * Base webinar object for sessions.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct TspGlobalDialIn {
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub audio_url: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ChannelSettings {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub allow_to_add_external_users: i64,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub new_members_can_see_previous_messages_files: bool,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub posting_permissions: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Properties {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub channel_settings: Option<ChannelSettings>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub jid: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

/// The channel object represents a Zoom chat [channel](https://support.zoom.us/hc/en-us/articles/200912909-Getting-Started-With-Channels-Group-Messaging-).
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Channel {
    /**
     * The channel object represents a Zoom chat [channel](https://support.zoom.us/hc/en-us/articles/200912909-Getting-Started-With-Channels-Group-Messaging-).
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub properties: Option<Properties>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CpuUsage {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub system_max_cpu_usage: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub zoom_avg_cpu_usage: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub zoom_max_cpu_usage: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub zoom_min_cpu_usage: String,
}

/// QoS metrics on audio being sent by a participant who joined the meeting via a Cloud Room Connector.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AudioDeviceFromCrc {
    /**
     * QoS metrics on audio being sent by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub avg_loss: String,
    /**
     * QoS metrics on audio being sent by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bitrate: String,
    /**
     * QoS metrics on audio being sent by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub jitter: String,
    /**
     * QoS metrics on audio being sent by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub latency: String,
    /**
     * QoS metrics on audio being sent by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub max_loss: String,
}

/// QoS metrics on audio received by a participant who joined the meeting via a Cloud Room Connector.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AudioDeviceCrc {
    /**
     * QoS metrics on audio received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub avg_loss: String,
    /**
     * QoS metrics on audio received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bitrate: String,
    /**
     * QoS metrics on audio received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub jitter: String,
    /**
     * QoS metrics on audio received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub latency: String,
    /**
     * QoS metrics on audio received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub max_loss: String,
}

/// QoS metrics on video input being sent from a Cloud Room Connector used by the participant to join the meeting.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct VideoDeviceFromCrc {
    /**
     * QoS metrics on video input being sent from a Cloud Room Connector used by the participant to join the meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub avg_loss: String,
    /**
     * QoS metrics on video input being sent from a Cloud Room Connector used by the participant to join the meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bitrate: String,
    /**
     * QoS metrics on video input being sent from a Cloud Room Connector used by the participant to join the meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub jitter: String,
    /**
     * QoS metrics on video input being sent from a Cloud Room Connector used by the participant to join the meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub latency: String,
    /**
     * QoS metrics on video input being sent from a Cloud Room Connector used by the participant to join the meeting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub max_loss: String,
}

/// QoS metrics on video output received by a participant who joined the meeting via a Cloud Room Connector.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct VideoDeviceCrc {
    /**
     * QoS metrics on video output received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub avg_loss: String,
    /**
     * QoS metrics on video output received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bitrate: String,
    /**
     * QoS metrics on video output received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub jitter: String,
    /**
     * QoS metrics on video output received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub latency: String,
    /**
     * QoS metrics on video output received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub max_loss: String,
}

/// QoS metrics on screen shares by a participant who joined the meeting via a Cloud Room Connector.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AsDeviceFromCrc {
    /**
     * QoS metrics on screen shares by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub avg_loss: String,
    /**
     * QoS metrics on screen shares by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bitrate: String,
    /**
     * QoS metrics on screen shares by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub jitter: String,
    /**
     * QoS metrics on screen shares by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub latency: String,
    /**
     * QoS metrics on screen shares by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub max_loss: String,
}

/// QoS metrics on screen shares output being received by a participant who joined the meeting via a Cloud Room Connector.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AsDeviceCrc {
    /**
     * QoS metrics on screen shares output being received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub avg_loss: String,
    /**
     * QoS metrics on screen shares output being received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bitrate: String,
    /**
     * QoS metrics on screen shares output being received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub jitter: String,
    /**
     * QoS metrics on screen shares output being received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub latency: String,
    /**
     * QoS metrics on screen shares output being received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub max_loss: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserQos {
    /**
     * QoS metrics on screen shares by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub as_device_from_crc: Option<AsDeviceFromCrc>,
    /**
     * QoS metrics on screen shares output being received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub as_device_to_crc: Option<AsDeviceCrc>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub as_input: Option<Domains>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub as_output: Option<Domains>,
    /**
     * QoS metrics on audio being sent by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub audio_device_from_crc: Option<AudioDeviceFromCrc>,
    /**
     * QoS metrics on audio received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub audio_device_to_crc: Option<AudioDeviceCrc>,
    /**
     * Quality of Service object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub audio_input: Option<QosAudio>,
    /**
     * Quality of Service object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub audio_output: Option<QosAudio>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub cpu_usage: Option<CpuUsage>,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub date_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * QoS metrics on video input being sent from a Cloud Room Connector used by the participant to join the meeting.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub video_device_from_crc: Option<VideoDeviceFromCrc>,
    /**
     * QoS metrics on video output received by a participant who joined the meeting via a Cloud Room Connector.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub video_device_to_crc: Option<VideoDeviceCrc>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub video_input: Option<Domains>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub video_output: Option<Domains>,
}

/// Participant QOS.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ParticipantQos {
    /**
     * Participant QOS.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub device: String,
    /**
     * Participant QOS.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub domain: String,
    /**
     * Participant QOS.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub harddisk_id: String,
    /**
     * Participant QOS.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub ip_address: String,
    /**
     * Participant QOS.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub join_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Participant QOS.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub leave_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Participant QOS.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location: String,
    /**
     * Participant QOS.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub mac_addr: String,
    /**
     * Participant QOS.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub pc_name: String,
    /**
     * Participant QOS.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_id: String,
    /**
     * Participant QOS.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_name: String,
    /**
     * Participant QOS.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub user_qos: Vec<UserQos>,
    /**
     * Participant QOS.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub version: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AccountSettingsRecordingAuthenticationUpdateOptionType {
    #[serde(rename = "enforce_login")]
    EnforceLogin,
    #[serde(rename = "enforce_login_with_domains")]
    EnforceLoginWithDomains,
    #[serde(rename = "internally")]
    Internally,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for AccountSettingsRecordingAuthenticationUpdateOptionType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AccountSettingsRecordingAuthenticationUpdateOptionType::EnforceLogin => "enforce_login",
            AccountSettingsRecordingAuthenticationUpdateOptionType::EnforceLoginWithDomains => {
                "enforce_login_with_domains"
            }
            AccountSettingsRecordingAuthenticationUpdateOptionType::Internally => "internally",
            AccountSettingsRecordingAuthenticationUpdateOptionType::Noop => "",
            AccountSettingsRecordingAuthenticationUpdateOptionType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for AccountSettingsRecordingAuthenticationUpdateOptionType {
    fn default() -> AccountSettingsRecordingAuthenticationUpdateOptionType {
        AccountSettingsRecordingAuthenticationUpdateOptionType::Noop
    }
}
impl AccountSettingsRecordingAuthenticationUpdateOptionType {
    pub fn is_noop(&self) -> bool {
        matches!(
            self,
            AccountSettingsRecordingAuthenticationUpdateOptionType::Noop
        )
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AccountSettingsRecordingAuthenticationUpdateOptionAction {
    #[serde(rename = "add")]
    Add,
    #[serde(rename = "delete")]
    Delete,
    #[serde(rename = "update")]
    Update,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for AccountSettingsRecordingAuthenticationUpdateOptionAction {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AccountSettingsRecordingAuthenticationUpdateOptionAction::Add => "add",
            AccountSettingsRecordingAuthenticationUpdateOptionAction::Delete => "delete",
            AccountSettingsRecordingAuthenticationUpdateOptionAction::Update => "update",
            AccountSettingsRecordingAuthenticationUpdateOptionAction::Noop => "",
            AccountSettingsRecordingAuthenticationUpdateOptionAction::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for AccountSettingsRecordingAuthenticationUpdateOptionAction {
    fn default() -> AccountSettingsRecordingAuthenticationUpdateOptionAction {
        AccountSettingsRecordingAuthenticationUpdateOptionAction::Noop
    }
}
impl AccountSettingsRecordingAuthenticationUpdateOptionAction {
    pub fn is_noop(&self) -> bool {
        matches!(
            self,
            AccountSettingsRecordingAuthenticationUpdateOptionAction::Noop
        )
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AuthenticationOption {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub action: Option<AccountSettingsRecordingAuthenticationUpdateOptionAction>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub default_option: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub domains: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<AccountSettingsRecordingAuthenticationUpdateOptionType>,
}

/// Only authenticated users can view cloud recordings
/// The viewers need to authenticate prior to viewing the cloud recordings, hosts can choose one of the authentication methods when sharing a cloud recording.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountSettingsRecordingAuthenticationUpdate {
    /**
     * Only authenticated users can view cloud recordings
     *  The viewers need to authenticate prior to viewing the cloud recordings, hosts can choose one of the authentication methods when sharing a cloud recording.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub authentication_option: Option<AuthenticationOption>,
    /**
     * Only authenticated users can view cloud recordings
     *  The viewers need to authenticate prior to viewing the cloud recordings, hosts can choose one of the authentication methods when sharing a cloud recording.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub recording_authentication: bool,
}

/// Quality of Service object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct QoSPhone {
    /**
     * Quality of Service object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub avg_loss: String,
    /**
     * Quality of Service object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bitrate: String,
    /**
     * Quality of Service object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub jitter: String,
    /**
     * Quality of Service object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub max_loss: String,
    /**
     * Quality of Service object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub mos: String,
    /**
     * Quality of Service object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub network_delay: String,
}

/// Meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingSecuritySettingsPasswordRequirement {
    /**
     * Meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub consecutive_characters_length: i64,
    /**
     * Meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_letter: bool,
    /**
     * Meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_number: bool,
    /**
     * Meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_special_character: bool,
    /**
     * Meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub have_upper_and_lower_characters: bool,
    /**
     * Meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub length: i64,
    /**
     * Meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub only_allow_numeric: bool,
    /**
     * Meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub weak_enhance_detection: bool,
}

/// Specify the settings to be applied if waiting room is enabled.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct WaitingRoomSettings {
    /**
     * Specify the settings to be applied if waiting room is enabled.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub participants_to_place_in_waiting_room: i64,
    /**
     * Specify the settings to be applied if waiting room is enabled.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub users_who_can_admit_participants_from_waiting_room: i64,
    /**
     * Specify the settings to be applied if waiting room is enabled.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub whitelisted_domains_for_waiting_room: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingSecurity {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub auto_security: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub block_user_domain: bool,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub block_user_domain_list: Vec<String>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub embed_password_in_join_link: bool,
    /**
     * Choose between enhanced encryption and [end-to-end encryption](https://support.zoom.us/hc/en-us/articles/360048660871) when starting or a meeting. When using end-to-end encryption, several features (e.g. cloud recording, phone/SIP/H.323 dial-in) will be \*\*automatically disabled\*\*. <br><br>The value of this field can be one of the following:<br>
     *  `enhanced_encryption`: Enhanced encryption. Encryption is stored in the cloud if you enable this option. <br>
     *  
     *  `e2ee`: [End-to-end encryption](https://support.zoom.us/hc/en-us/articles/360048660871). The encryption key is stored in your local device and can not be obtained by anyone else. Enabling this setting also \*\*disables\*\* the following features: join before host, cloud recording, streaming, live transcription, breakout rooms, polling, 1:1 private chat, and meeting reactions.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub encryption_type: Option<EncryptionType>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub end_to_end_encrypted_meetings: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub meeting_password: bool,
    /**
     * Meeting/webinar [passcode requirements](https://support.zoom.us/hc/en-us/articles/360033559832-Meeting-and-webinar-passwords#h_a427384b-e383-4f80-864d-794bf0a37604).
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub meeting_password_requirement: Option<MeetingSecuritySettingsPasswordRequirement>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub phone_password: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub pmi_password: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_scheduled_meeting: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_scheduled_webinar: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub waiting_room: bool,
    /**
     * Specify the settings to be applied if waiting room is enabled.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub waiting_room_settings: Option<WaitingRoomSettings>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub webinar_password: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingSecuritySettings {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub meeting_security: Option<MeetingSecurity>,
}

/// Meeting metric details.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Metrics {
    /**
     * Meeting metric details.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub custom_keys: Vec<CustomKeys>,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub dept: String,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub duration: String,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub end_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_3rd_party_audio: bool,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_pstn: bool,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_recording: bool,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_screen_share: bool,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_sip: bool,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_video: bool,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_voip: bool,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub host: String,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub in_room_participants: i64,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub participants: i64,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Meeting metric details.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub tracking_fields: Vec<SessionUpdateTrackingFields>,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_type: String,
    /**
     * Meeting metric details.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub uuid: String,
}

/**
 * The archived file's file type:
 *   * `MP4` — Video file.
 *   * `M4A` — Audio-only file.
 *   * `TRANSCRIPT` — A transcript file of the recording, in VTT file format.
 *   * `CHAT` — A TXT file containing chat messages that were sent during the meeting.
 *   * `CC` — A file containing the recording's closed captions, in VTT file format.
 *   * `CSV` — A file containing polling data in, in CSV format.
 *   
 *   For more information, read our [Managing and sharing cloud recordings](https://support.zoom.us/hc/en-us/articles/205347605-Managing-and-sharing-cloud-recordings#h_9898497b-e736-4980-a749-d55608f10773) documentation.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum FileType {
    #[serde(rename = "CC")]
    Cc,
    #[serde(rename = "CHAT")]
    Chat,
    #[serde(rename = "CSV")]
    Csv,
    #[serde(rename = "M4A")]
    M4A,
    #[serde(rename = "MP4")]
    Mp4,
    #[serde(rename = "TRANSCRIPT")]
    Transcript,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for FileType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            FileType::Cc => "CC",
            FileType::Chat => "CHAT",
            FileType::Csv => "CSV",
            FileType::M4A => "M4A",
            FileType::Mp4 => "MP4",
            FileType::Transcript => "TRANSCRIPT",
            FileType::Noop => "",
            FileType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for FileType {
    fn default() -> FileType {
        FileType::Noop
    }
}
impl FileType {
    pub fn is_noop(&self) -> bool {
        matches!(self, FileType::Noop)
    }
}

/**
 * The archived file's recording type:
 *   * `shared_screen_with_speaker_view(CC)`
 *   * `shared_screen_with_speaker_view`
 *   * `shared_screen_with_gallery_view`
 *   * `speaker_view`
 *   * `gallery_view`
 *   * `shared_screen`
 *   * `audio_only`
 *   * `audio_transcript`
 *   * `chat_file`
 *   * `active_speaker`
 *   * `host_video`
 *   
 *   For more information, read our [Managing and sharing cloud recordings](https://support.zoom.us/hc/en-us/articles/205347605-Managing-and-sharing-cloud-recordings#h_9898497b-e736-4980-a749-d55608f10773) documentation.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum RecordingType {
    #[serde(rename = "active_speaker")]
    ActiveSpeaker,
    #[serde(rename = "audio_only")]
    AudioOnly,
    #[serde(rename = "audio_transcript")]
    AudioTranscript,
    #[serde(rename = "chat_file")]
    ChatFile,
    #[serde(rename = "gallery_view")]
    GalleryView,
    #[serde(rename = "host_video")]
    HostVideo,
    #[serde(rename = "shared_screen")]
    SharedScreen,
    #[serde(rename = "shared_screen_with_gallery_view")]
    SharedScreenWithGalleryView,
    #[serde(rename = "shared_screen_with_speaker_view")]
    SharedScreenWithSpeakerView,
    #[serde(rename = "shared_screen_with_speaker_view(CC)")]
    SharedScreenWithSpeakerViewCc,
    #[serde(rename = "speaker_view")]
    SpeakerView,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for RecordingType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            RecordingType::ActiveSpeaker => "active_speaker",
            RecordingType::AudioOnly => "audio_only",
            RecordingType::AudioTranscript => "audio_transcript",
            RecordingType::ChatFile => "chat_file",
            RecordingType::GalleryView => "gallery_view",
            RecordingType::HostVideo => "host_video",
            RecordingType::SharedScreen => "shared_screen",
            RecordingType::SharedScreenWithGalleryView => "shared_screen_with_gallery_view",
            RecordingType::SharedScreenWithSpeakerView => "shared_screen_with_speaker_view",
            RecordingType::SharedScreenWithSpeakerViewCc => "shared_screen_with_speaker_view(CC)",
            RecordingType::SpeakerView => "speaker_view",
            RecordingType::Noop => "",
            RecordingType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for RecordingType {
    fn default() -> RecordingType {
        RecordingType::Noop
    }
}
impl RecordingType {
    pub fn is_noop(&self) -> bool {
        matches!(self, RecordingType::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ArchiveFiles {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub download_url: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub file_extension: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub file_size: i64,
    /**
     * The archived file's file type:
     *  \* `MP4` — Video file.
     *  \* `M4A` — Audio-only file.
     *  \* `TRANSCRIPT` — A transcript file of the recording, in VTT file format.
     *  \* `CHAT` — A TXT file containing chat messages that were sent during the meeting.
     *  \* `CC` — A file containing the recording's closed captions, in VTT file format.
     *  \* `CSV` — A file containing polling data in, in CSV format.
     *  
     *  For more information, read our [Managing and sharing cloud recordings](https://support.zoom.us/hc/en-us/articles/205347605-Managing-and-sharing-cloud-recordings#h_9898497b-e736-4980-a749-d55608f10773) documentation.
     */
    #[serde(default, skip_serializing_if = "FileType::is_noop")]
    pub file_type: FileType,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * The archived file's recording type:
     *  \* `shared_screen_with_speaker_view(CC)`
     *  \* `shared_screen_with_speaker_view`
     *  \* `shared_screen_with_gallery_view`
     *  \* `speaker_view`
     *  \* `gallery_view`
     *  \* `shared_screen`
     *  \* `audio_only`
     *  \* `audio_transcript`
     *  \* `chat_file`
     *  \* `active_speaker`
     *  \* `host_video`
     *  
     *  For more information, read our [Managing and sharing cloud recordings](https://support.zoom.us/hc/en-us/articles/205347605-Managing-and-sharing-cloud-recordings#h_9898497b-e736-4980-a749-d55608f10773) documentation.
     */
    #[serde(default, skip_serializing_if = "RecordingType::is_noop")]
    pub recording_type: RecordingType,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub status: String,
}

/// All of the following types:
///
/// - `ArchiveFiles`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ArchiveFilesAnyOf {
    ArchiveFiles(ArchiveFiles),
}

impl ArchiveFilesAnyOf {
    pub fn archive_files(&self) -> Option<&ArchiveFiles> {
        let ArchiveFilesAnyOf::ArchiveFiles(ref_) = self;
        Some(ref_)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CloudArchivedFiles {
    /**
     * Information about the archive files.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub archive_files: Vec<ArchiveFilesAnyOf>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub host_id: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub recording_count: i64,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_size: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub uuid: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Attendees {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

/// Invite Links
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct InviteLink {
    /**
     * Invite Links
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub attendees: Vec<Attendees>,
    /**
     * Invite Links
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub ttl: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct InviteLinksAttendees {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub join_url: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

/// Invite links response.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct InviteLinks {
    /**
     * Invite links response.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub attendees: Vec<InviteLinksAttendees>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum RecordingStatusUpdateBodyRequestAction {
    #[serde(rename = "recover")]
    Recover,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for RecordingStatusUpdateBodyRequestAction {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            RecordingStatusUpdateBodyRequestAction::Recover => "recover",
            RecordingStatusUpdateBodyRequestAction::Noop => "",
            RecordingStatusUpdateBodyRequestAction::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for RecordingStatusUpdateBodyRequestAction {
    fn default() -> RecordingStatusUpdateBodyRequestAction {
        RecordingStatusUpdateBodyRequestAction::Noop
    }
}
impl RecordingStatusUpdateBodyRequestAction {
    pub fn is_noop(&self) -> bool {
        matches!(self, RecordingStatusUpdateBodyRequestAction::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct RecordingStatusUpdateBodyRequest {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub action: Option<RecordingStatusUpdateBodyRequestAction>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UploadVbRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub file: String,
}

/// Specify emergency address for the account.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct EmergencyAddress {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address_line_1: String,
    /**
     * Specify emergency address for the account.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address_line_2: String,
    /**
     * Specify emergency address for the account.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub city: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
     * Specify emergency address for the account.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub state_code: String,
    /**
     * Specify emergency address for the account.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub zip: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SetUpAccountRequest {
    /**
     * Specify emergency address for the account.
     */
    #[serde()]
    pub emergency_address: EmergencyAddress,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub extension_number: String,
}

/**
 * Query response by number assignment. The value can be one of the following:
 *   <br>
 *   `assigned`: The number has been assigned to either a user, a call queue, an auto-receptionist or a common area phone in an account. <br>`unassigned`: The number is not assigned to anyone.<br>
 *   `all`: Include both assigned and unassigned numbers in the response.<br>
 *   `byoc`: Include Bring Your Own Carrier (BYOC) numbers only in the response.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ListAccountPhoneNumbersType {
    #[serde(rename = "all")]
    All,
    #[serde(rename = "assigned")]
    Assigned,
    #[serde(rename = "unassigned")]
    Unassigned,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for ListAccountPhoneNumbersType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ListAccountPhoneNumbersType::All => "all",
            ListAccountPhoneNumbersType::Assigned => "assigned",
            ListAccountPhoneNumbersType::Unassigned => "unassigned",
            ListAccountPhoneNumbersType::Noop => "",
            ListAccountPhoneNumbersType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for ListAccountPhoneNumbersType {
    fn default() -> ListAccountPhoneNumbersType {
        ListAccountPhoneNumbersType::Noop
    }
}
impl ListAccountPhoneNumbersType {
    pub fn is_noop(&self) -> bool {
        matches!(self, ListAccountPhoneNumbersType::Noop)
    }
}

/**
 * The type of assignee to whom the number is assigned. The value can be one of the following:<br>
 *   `user`<br> `callQueue`<br> `autoReceptionist`<br>
 *   `commonAreaPhone`
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ExtensionType {
    #[serde(rename = "autoReceptionist")]
    AutoReceptionist,
    #[serde(rename = "callQueue")]
    CallQueue,
    #[serde(rename = "commonAreaPhone")]
    CommonAreaPhone,
    #[serde(rename = "user")]
    User,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for ExtensionType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ExtensionType::AutoReceptionist => "autoReceptionist",
            ExtensionType::CallQueue => "callQueue",
            ExtensionType::CommonAreaPhone => "commonAreaPhone",
            ExtensionType::User => "user",
            ExtensionType::Noop => "",
            ExtensionType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for ExtensionType {
    fn default() -> ExtensionType {
        ExtensionType::Noop
    }
}
impl ExtensionType {
    pub fn is_noop(&self) -> bool {
        matches!(self, ExtensionType::Noop)
    }
}

/**
 * Source of phone number.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum Source {
    #[serde(rename = "external")]
    External,
    #[serde(rename = "internal")]
    Internal,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for Source {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Source::External => "external",
            Source::Internal => "internal",
            Source::Noop => "",
            Source::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for Source {
    fn default() -> Source {
        Source::Noop
    }
}
impl Source {
    pub fn is_noop(&self) -> bool {
        matches!(self, Source::Noop)
    }
}

/**
 * Status of the number.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ListAccountPhoneNumbersResponseStatus {
    #[serde(rename = "available")]
    Available,
    #[serde(rename = "pending")]
    Pending,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for ListAccountPhoneNumbersResponseStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ListAccountPhoneNumbersResponseStatus::Available => "available",
            ListAccountPhoneNumbersResponseStatus::Pending => "pending",
            ListAccountPhoneNumbersResponseStatus::Noop => "",
            ListAccountPhoneNumbersResponseStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for ListAccountPhoneNumbersResponseStatus {
    fn default() -> ListAccountPhoneNumbersResponseStatus {
        ListAccountPhoneNumbersResponseStatus::Noop
    }
}
impl ListAccountPhoneNumbersResponseStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, ListAccountPhoneNumbersResponseStatus::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Assignee {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * The type of assignee to whom the number is assigned. The value can be one of the following:<br>
     *  `user`<br> `callQueue`<br> `autoReceptionist`<br>
     *  `commonAreaPhone`
     */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<ExtensionType>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Site {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

/// Displayed when the `type` request parameter is `byoc`.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Carrier {
    /**
     * Displayed when the `type` request parameter is `byoc`.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub code: String,
    /**
     * Displayed when the `type` request parameter is `byoc`.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

/// Displayed when the `type` request parameter is `byoc`.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SipGroup {
    /**
     * Displayed when the `type` request parameter is `byoc`.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * Displayed when the `type` request parameter is `byoc`.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListAccountPhoneNumbersResponse {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub assignee: Option<Assignee>,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub capability: Vec<String>,
    /**
     * Displayed when the `type` request parameter is `byoc`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub carrier: Option<Carrier>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
    /**
     * Type of number.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub number_type: Option<Type>,
    /**
     * Displayed when the `type` request parameter is `byoc`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sip_group: Option<SipGroup>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub site: Option<Site>,
    /**
     * Source of phone number.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub source: Option<Source>,
    /**
     * Status of the number.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<ListAccountPhoneNumbersResponseStatus>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListAccountPhoneNumbersResponseData {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub phone_numbers: Vec<ListAccountPhoneNumbersResponse>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

/**
 * Protocols supported by the SIP provider.<br> The value must be either `UDP`, `TCP`, `TLS`, `AUTO`.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum TransportProtocol {
    #[serde(rename = "AUTO")]
    Auto,
    #[serde(rename = "TCP")]
    Tcp,
    #[serde(rename = "TLS")]
    Tls,
    #[serde(rename = "UDP")]
    Udp,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for TransportProtocol {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            TransportProtocol::Auto => "AUTO",
            TransportProtocol::Tcp => "TCP",
            TransportProtocol::Tls => "TLS",
            TransportProtocol::Udp => "UDP",
            TransportProtocol::Noop => "",
            TransportProtocol::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for TransportProtocol {
    fn default() -> TransportProtocol {
        TransportProtocol::Noop
    }
}
impl TransportProtocol {
    pub fn is_noop(&self) -> bool {
        matches!(self, TransportProtocol::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Phones {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authorization_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub domain: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub proxy_server: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub proxy_server_2: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub proxy_server_3: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub register_server: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub register_server_2: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub register_server_3: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub registration_expire_time: i64,
    /**
     * Protocols supported by the SIP provider.<br> The value must be either `UDP`, `TCP`, `TLS`, `AUTO`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transport_protocol: Option<TransportProtocol>,
    /**
     * Protocols supported by the SIP provider.<br> The value must be either `UDP`, `TCP`, `TLS`, `AUTO`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transport_protocol_2: Option<TransportProtocol>,
    /**
     * Protocols supported by the SIP provider.<br> The value must be either `UDP`, `TCP`, `TLS`, `AUTO`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transport_protocol_3: Option<TransportProtocol>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub voice_mail: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListSipPhonesResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_count: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_number: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * SIP phones object.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub phones: Vec<Phones>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CreateSipPhoneRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authorization_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub domain: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub proxy_server: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub proxy_server_2: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub proxy_server_3: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub register_server: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub register_server_2: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub register_server_3: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub registration_expire_time: i64,
    /**
     * Protocols supported by the SIP provider.<br> The value must be either `UDP`, `TCP`, `TLS`, `AUTO`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transport_protocol: Option<TransportProtocol>,
    /**
     * Protocols supported by the SIP provider.<br> The value must be either `UDP`, `TCP`, `TLS`, `AUTO`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transport_protocol_2: Option<TransportProtocol>,
    /**
     * Protocols supported by the SIP provider.<br> The value must be either `UDP`, `TCP`, `TLS`, `AUTO`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transport_protocol_3: Option<TransportProtocol>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub voice_mail: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateSipPhoneRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authorization_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub domain: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub proxy_server: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub proxy_server_2: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub proxy_server_3: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub register_server: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub register_server_2: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub register_server_3: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub registration_expire_time: i64,
    /**
     * Protocols supported by the SIP provider.<br> The value must be either `UDP`, `TCP`, `TLS`, `AUTO`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transport_protocol: Option<TransportProtocol>,
    /**
     * Protocols supported by the SIP provider.<br> The value must be either `UDP`, `TCP`, `TLS`, `AUTO`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transport_protocol_2: Option<TransportProtocol>,
    /**
     * Protocols supported by the SIP provider.<br> The value must be either `UDP`, `TCP`, `TLS`, `AUTO`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transport_protocol_3: Option<TransportProtocol>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub voice_mail: String,
}

/**
 * The status of the Zoom Room.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ListZoomRoomsStatus {
    #[serde(rename = "Available")]
    Available,
    #[serde(rename = "InMeeting")]
    InMeeting,
    #[serde(rename = "Offline")]
    Offline,
    #[serde(rename = "UnderConstruction")]
    UnderConstruction,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for ListZoomRoomsStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ListZoomRoomsStatus::Available => "Available",
            ListZoomRoomsStatus::InMeeting => "InMeeting",
            ListZoomRoomsStatus::Offline => "Offline",
            ListZoomRoomsStatus::UnderConstruction => "UnderConstruction",
            ListZoomRoomsStatus::Noop => "",
            ListZoomRoomsStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for ListZoomRoomsStatus {
    fn default() -> ListZoomRoomsStatus {
        ListZoomRoomsStatus::Noop
    }
}
impl ListZoomRoomsStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, ListZoomRoomsStatus::Noop)
    }
}

/**
 * Type of the Zoom Rooms.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ListZoomRoomsType {
    #[serde(rename = "DigitalSignageOnly")]
    DigitalSignageOnly,
    #[serde(rename = "SchedulingDisplayOnly")]
    SchedulingDisplayOnly,
    #[serde(rename = "ZoomRoom")]
    ZoomRoom,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for ListZoomRoomsType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ListZoomRoomsType::DigitalSignageOnly => "DigitalSignageOnly",
            ListZoomRoomsType::SchedulingDisplayOnly => "SchedulingDisplayOnly",
            ListZoomRoomsType::ZoomRoom => "ZoomRoom",
            ListZoomRoomsType::Noop => "",
            ListZoomRoomsType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for ListZoomRoomsType {
    fn default() -> ListZoomRoomsType {
        ListZoomRoomsType::Noop
    }
}
impl ListZoomRoomsType {
    pub fn is_noop(&self) -> bool {
        matches!(self, ListZoomRoomsType::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListZoomRoomsResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub activation_code: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub room_id: String,
    /**
     * The status of the Zoom Room.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<ListZoomRoomsStatus>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListZoomRoomsResponseData {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * List of existing Zoom Rooms.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub rooms: Vec<ListZoomRoomsResponse>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddRoomRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Type of the Zoom Rooms.
     */
    #[serde(
        default,
        skip_serializing_if = "ListZoomRoomsType::is_noop",
        rename = "type"
    )]
    pub type_: ListZoomRoomsType,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddRoomResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub room_id: String,
    /**
     * Type of the Zoom Rooms.
     */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<ListZoomRoomsType>,
}

/// Auto Receptionist for each site.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MainAutoReceptionist {
    /**
     * Auto Receptionist for each site.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub extension_id: String,
    /**
     * Auto Receptionist for each site.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub extension_number: String,
    /**
     * Auto Receptionist for each site.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Auto Receptionist for each site.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

/// Site country
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Country {
    /**
     * Site country
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub code: String,
    /**
     * Site country
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Sites {
    /**
     * Site country
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub country: Option<Country>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Auto Receptionist for each site.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub main_auto_receptionist: Option<MainAutoReceptionist>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub site_code: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListPhoneSitesResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub page_size: String,
    /**
     * List of site(s).
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub sites: Vec<Sites>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub total_records: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DefaultEmergencyAddress {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address_line_1: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address_line_2: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub city: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub state_code: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub zip: String,
}

/// Short extension of the phone site.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ShortExtension {
    /**
     * Short extension of the phone site.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub length: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CreatePhoneSiteRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub auto_receptionist_name: String,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub default_emergency_address: Option<DefaultEmergencyAddress>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Short extension of the phone site.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub short_extension: Option<ShortExtension>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub site_code: i64,
}

/// [Main Auto Receptionist](https://support.zoom.us/hc/en-us/articles/360021121312#h_bc7ff1d5-0e6c-40cd-b889-62010cb98c57) for each site.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetSiteResponseMainAutoReceptionist {
    /**
     * [Main Auto Receptionist](https://support.zoom.us/hc/en-us/articles/360021121312#h_bc7ff1d5-0e6c-40cd-b889-62010cb98c57) for each site.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub extension_id: String,
    /**
     * [Main Auto Receptionist](https://support.zoom.us/hc/en-us/articles/360021121312#h_bc7ff1d5-0e6c-40cd-b889-62010cb98c57) for each site.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * [Main Auto Receptionist](https://support.zoom.us/hc/en-us/articles/360021121312#h_bc7ff1d5-0e6c-40cd-b889-62010cb98c57) for each site.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * [Main Auto Receptionist](https://support.zoom.us/hc/en-us/articles/360021121312#h_bc7ff1d5-0e6c-40cd-b889-62010cb98c57) for each site.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetSiteResponse {
    /**
     * Site country
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub country: Option<Country>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * [Main Auto Receptionist](https://support.zoom.us/hc/en-us/articles/360021121312#h_bc7ff1d5-0e6c-40cd-b889-62010cb98c57) for each site.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub main_auto_receptionist: Option<GetSiteResponseMainAutoReceptionist>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Short extension of the phone site.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub short_extension: Option<ShortExtension>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub site_code: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateSiteDetailsRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub site_code: i64,
}

/**
 * Presence status of the contact in Zoom Client. The value of this field can be one of the following:
 *   `Do_Not_Disturb`<br> `away`<br> `Available`<br> `Offline`
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum PresenceStatus {
    #[serde(rename = "Available")]
    Available,
    #[serde(rename = "Away")]
    Away,
    #[serde(rename = "Do_Not_Disturb")]
    DoNotDisturb,
    #[serde(rename = "Offline")]
    Offline,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for PresenceStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PresenceStatus::Available => "Available",
            PresenceStatus::Away => "Away",
            PresenceStatus::DoNotDisturb => "Do_Not_Disturb",
            PresenceStatus::Offline => "Offline",
            PresenceStatus::Noop => "",
            PresenceStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for PresenceStatus {
    fn default() -> PresenceStatus {
        PresenceStatus::Noop
    }
}
impl PresenceStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, PresenceStatus::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Contacts {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub dept: String,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub direct_numbers: Vec<String>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub extension_number: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub im_group_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub im_group_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub job_title: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number: String,
    /**
     * Presence status of the contact in Zoom Client. The value of this field can be one of the following:
     *  `Do_Not_Disturb`<br> `away`<br> `Available`<br> `Offline`
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub presence_status: Option<PresenceStatus>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub sip_phone_number: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SearchCompanyContactsResponse {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub contacts: Vec<Contacts>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
}

/**
 * This field indicates whether a message is an original message(unedited), an edited message or a deleted message.
 *   
 *   This field is only returned if you set the value of `for include_deleted_and_edited_message` query parameter to `true`.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum GetChatMessagesResponseStatus {
    #[serde(rename = "Deleted")]
    Deleted,
    #[serde(rename = "Edited")]
    Edited,
    #[serde(rename = "Normal")]
    Normal,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for GetChatMessagesResponseStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GetChatMessagesResponseStatus::Deleted => "Deleted",
            GetChatMessagesResponseStatus::Edited => "Edited",
            GetChatMessagesResponseStatus::Normal => "Normal",
            GetChatMessagesResponseStatus::Noop => "",
            GetChatMessagesResponseStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for GetChatMessagesResponseStatus {
    fn default() -> GetChatMessagesResponseStatus {
        GetChatMessagesResponseStatus::Noop
    }
}
impl GetChatMessagesResponseStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, GetChatMessagesResponseStatus::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Messages {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub bot_message: Option<Domains>,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub date_time: Option<chrono::NaiveDate>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub download_url: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub file_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub file_name: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub file_size: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub message: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub reply_main_message_id: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub reply_main_message_timestamp: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub sender: String,
    /**
     * This field indicates whether a message is an original message(unedited), an edited message or a deleted message.
     *  
     *  This field is only returned if you set the value of `for include_deleted_and_edited_message` query parameter to `true`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<GetChatMessagesResponseStatus>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub timestamp: i64,
}

/// Chat message object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetChatMessagesResponse {
    /**
     * Chat message object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub date: Option<chrono::NaiveDate>,
    /**
     * Chat message object.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub messages: Vec<Messages>,
    /**
     * Chat message object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Chat message object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AtItems {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub at_contact: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub at_type: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub end_position: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub start_position: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SendaChatMessageRequest {
    /**
     * [Chat mentions](https://support.zoom.us/hc/en-us/articles/360037567431-Using-chat-mentions-and-slash-commands) object. Use this object to include mentions in the message that will be sent to  a channel.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub at_items: Vec<AtItems>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub message: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub to_channel: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub to_contact: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MarkMessageRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub action: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub timestamp: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub to_channel: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub to_contact: String,
}

/**
 * The action to perform with the message:
 *   * `add` — Add an emoji reaction.
 *   * `remove` — Remove an emoji reaction.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ReactMessageRequestAction {
    #[serde(rename = "add")]
    Add,
    #[serde(rename = "remove")]
    Remove,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for ReactMessageRequestAction {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ReactMessageRequestAction::Add => "add",
            ReactMessageRequestAction::Remove => "remove",
            ReactMessageRequestAction::Noop => "",
            ReactMessageRequestAction::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for ReactMessageRequestAction {
    fn default() -> ReactMessageRequestAction {
        ReactMessageRequestAction::Noop
    }
}
impl ReactMessageRequestAction {
    pub fn is_noop(&self) -> bool {
        matches!(self, ReactMessageRequestAction::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ReactMessageRequest {
    /**
     * The action to perform with the message:
     *  \* `add` — Add an emoji reaction.
     *  \* `remove` — Remove an emoji reaction.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub action: Option<ReactMessageRequestAction>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub emoji: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub to_channel: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub to_contact: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Reactions {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub emoji: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_count: i64,
}

/// A chat message object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetChatMessageResponse {
    /**
     * A chat message object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub bot_message: Option<Domains>,
    /**
     * A chat message object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub date_time: Option<chrono::NaiveDate>,
    /**
     * A chat message object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub download_url: String,
    /**
     * A chat message object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub file_id: String,
    /**
     * A chat message object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub file_name: String,
    /**
     * A chat message object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub file_size: i64,
    /**
     * A chat message object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * A chat message object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub message: String,
    /**
     * A chat message object.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub reactions: Vec<Reactions>,
    /**
     * A chat message object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub reply_main_message_id: String,
    /**
     * A chat message object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub reply_main_message_timestamp: i64,
    /**
     * A chat message object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub sender: String,
    /**
     * A chat message object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub timestamp: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct EditMessageRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub message: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub to_channel: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub to_contact: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Channels {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub channels_settings: Option<ChannelSettings>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetChannelsResponse {
    /**
     * Chat Channel object(s).
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub channels: Vec<Channels>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Members {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CreateChannelRequest {
    /**
     * Member(s) to include in the channel. A max of 5 members can be added to the channel at once with this API.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub members: Vec<Members>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CreateChannelResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub jid: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetUserContactsResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetUserContactsResponseData {
    /**
     * The contacts object.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub contacts: Vec<GetUserContactsResponse>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetUserContactResponse {
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub direct_numbers: Vec<String>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub extension_number: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number: String,
    /**
     * Presence status of the contact in Zoom Client. The value of this field can be one of the following:
     *  `Do_Not_Disturb`<br> `away`<br> `Available`<br> `Offline`
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub presence_status: Option<PresenceStatus>,
}

/**
 * The role of the member. The value can be one of these: `owner`, `admin`, `member`.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum Role {
    #[serde(rename = "admin")]
    Admin,
    #[serde(rename = "member")]
    Member,
    #[serde(rename = "owner")]
    Owner,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for Role {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Role::Admin => "admin",
            Role::Member => "member",
            Role::Owner => "owner",
            Role::Noop => "",
            Role::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for Role {
    fn default() -> Role {
        Role::Noop
    }
}
impl Role {
    pub fn is_noop(&self) -> bool {
        matches!(self, Role::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListChannelMembersResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * The role of the member. The value can be one of these: `owner`, `admin`, `member`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub role: Option<Role>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListChannelMembersResponseData {
    /**
     * Members in a channel.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub members: Vec<ListChannelMembersResponse>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct InviteChannelMembersRequest {
    /**
     * Member(s) to include in the channel. A max of 5 members can be added to the channel at once with this API.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub members: Vec<Members>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct InviteChannelMembersResponse {
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub added_at: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub ids: Vec<String>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct JoinChannelResponse {
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub added_at: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
}

/// The account object represents an account on Zoom. The account owner is the user who created the account or had an account created for them. You can read more about the Zoom account structure <a href='https://medium.com/zoom-developer-blog/a-brief-look-at-zoom-account-structures-1d19c745bf8a' target='_blank'>here</a>.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountCreateRequest {
    /**
     * The account object represents an account on Zoom. The account owner is the user who created the account or had an account created for them. You can read more about the Zoom account structure <a href='https://medium.com/zoom-developer-blog/a-brief-look-at-zoom-account-structures-1d19c745bf8a' target='_blank'>here</a>.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub account_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
     * The account object represents an account on Zoom. The account owner is the user who created the account or had an account created for them. You can read more about the Zoom account structure <a href='https://medium.com/zoom-developer-blog/a-brief-look-at-zoom-account-structures-1d19c745bf8a' target='_blank'>here</a>.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub options: Option<Options>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountCreateResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub created_at: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub owner_email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub owner_id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountResponse {
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub created_at: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Account options object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub options: Option<Options>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub owner_email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub owner_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub vanity_url: String,
}

/**
 * Use the following options to filter the results of the account's information:
 *   * `meeting_authentication` — View the account's [meeting authentication settings](https://support.zoom.us/hc/en-us/articles/360037117472-Authentication-Profiles-for-Meetings-and-Webinars).
 *   * `recording_authentication` — View the account's [recording authentication settings](https://support.zoom.us/hc/en-us/articles/360037756671-Authentication-Profiles-for-Cloud-Recordings).
 *   * `security` — View the account's security settings. For example, password requirements for user login or two-factor authentication.<br>
 *   * `meeting_security` — View the account's meeting security settings.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum OptionData {
    #[serde(rename = "meeting_authentication")]
    MeetingAuthentication,
    #[serde(rename = "recording_authentication")]
    RecordingAuthentication,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for OptionData {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            OptionData::MeetingAuthentication => "meeting_authentication",
            OptionData::RecordingAuthentication => "recording_authentication",
            OptionData::Noop => "",
            OptionData::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for OptionData {
    fn default() -> OptionData {
        OptionData::Noop
    }
}
impl OptionData {
    pub fn is_noop(&self) -> bool {
        matches!(self, OptionData::Noop)
    }
}

/// All of the following types:
///
/// - `Domains`
/// - `Security`
/// - `AccountSettings`
/// - `MeetingSecuritySettings`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AccountSettingsResponseOneOf {
    Domains(Domains),
    /**
     * [Security settings](https://support.zoom.us/hc/en-us/articles/360034675592-Advanced-security-settings#h_bf8a25f6-9a66-447a-befd-f02ed3404f89) of an Account.
     */
    Security(Security),
    AccountSettings(AccountSettings),
    MeetingSecuritySettings(MeetingSecuritySettings),
}

impl AccountSettingsResponseOneOf {
    pub fn account_settings(&self) -> Option<&AccountSettings> {
        if let AccountSettingsResponseOneOf::AccountSettings(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn domains(&self) -> Option<&Domains> {
        if let AccountSettingsResponseOneOf::Domains(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn meeting_security_settings(&self) -> Option<&MeetingSecuritySettings> {
        if let AccountSettingsResponseOneOf::MeetingSecuritySettings(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn security(&self) -> Option<&Security> {
        if let AccountSettingsResponseOneOf::Security(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `Domains`
/// - `UpdateAccountSettings`
/// - `MeetingSecuritySettings`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AccountSettingsUpdateRequestOneOf {
    Domains(Domains),
    UpdateAccountSettings(UpdateAccountSettings),
    MeetingSecuritySettings(MeetingSecuritySettings),
}

impl AccountSettingsUpdateRequestOneOf {
    pub fn domains(&self) -> Option<&Domains> {
        if let AccountSettingsUpdateRequestOneOf::Domains(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn meeting_security_settings(&self) -> Option<&MeetingSecuritySettings> {
        if let AccountSettingsUpdateRequestOneOf::MeetingSecuritySettings(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn update_account_settings(&self) -> Option<&UpdateAccountSettings> {
        if let AccountSettingsUpdateRequestOneOf::UpdateAccountSettings(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AccountSettingsUpdateOption {
    #[serde(rename = "meeting_authentication")]
    MeetingAuthentication,
    #[serde(rename = "meeting_security")]
    MeetingSecurity,
    #[serde(rename = "recording_authentication")]
    RecordingAuthentication,
    #[serde(rename = "security")]
    Security,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for AccountSettingsUpdateOption {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AccountSettingsUpdateOption::MeetingAuthentication => "meeting_authentication",
            AccountSettingsUpdateOption::MeetingSecurity => "meeting_security",
            AccountSettingsUpdateOption::RecordingAuthentication => "recording_authentication",
            AccountSettingsUpdateOption::Security => "security",
            AccountSettingsUpdateOption::Noop => "",
            AccountSettingsUpdateOption::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for AccountSettingsUpdateOption {
    fn default() -> AccountSettingsUpdateOption {
        AccountSettingsUpdateOption::Noop
    }
}
impl AccountSettingsUpdateOption {
    pub fn is_noop(&self) -> bool {
        matches!(self, AccountSettingsUpdateOption::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountTrustedDomainResponse {
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub trusted_domains: Vec<String>,
}

/**
 * Status of the plan.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AccountPlansResponsePlanBaseStatus {
    #[serde(rename = "active")]
    Active,
    #[serde(rename = "cancel")]
    Cancel,
    #[serde(rename = "expired")]
    Expired,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for AccountPlansResponsePlanBaseStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AccountPlansResponsePlanBaseStatus::Active => "active",
            AccountPlansResponsePlanBaseStatus::Cancel => "cancel",
            AccountPlansResponsePlanBaseStatus::Expired => "expired",
            AccountPlansResponsePlanBaseStatus::Noop => "",
            AccountPlansResponsePlanBaseStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for AccountPlansResponsePlanBaseStatus {
    fn default() -> AccountPlansResponsePlanBaseStatus {
        AccountPlansResponsePlanBaseStatus::Noop
    }
}
impl AccountPlansResponsePlanBaseStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, AccountPlansResponsePlanBaseStatus::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PlanZoomRooms {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub hosts: i64,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub next_invoice_date: Option<chrono::NaiveDate>,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub service_effective_date: Option<chrono::NaiveDate>,
    /**
     * Status of the plan.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<AccountPlansResponsePlanBaseStatus>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

/// Account plan object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PlanWebinar {
    /**
     * Account plan object.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub hosts: i64,
    /**
     * Account plan object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub next_invoice_date: Option<chrono::NaiveDate>,
    /**
     * Account plan object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub service_effective_date: Option<chrono::NaiveDate>,
    /**
     * Account plan object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<AccountPlansResponsePlanBaseStatus>,
    /**
     * Account plan object.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

/**
 * Status of additional Cloud Recording plan.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum PlanRecordingStatus {
    #[serde(rename = "Active")]
    Active,
    #[serde(rename = "Cancelled")]
    Cancelled,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for PlanRecordingStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PlanRecordingStatus::Active => "Active",
            PlanRecordingStatus::Cancelled => "Cancelled",
            PlanRecordingStatus::Noop => "",
            PlanRecordingStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for PlanRecordingStatus {
    fn default() -> PlanRecordingStatus {
        PlanRecordingStatus::Noop
    }
}
impl PlanRecordingStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, PlanRecordingStatus::Noop)
    }
}

/// Additional audio conferencing <a href="https://marketplace.zoom.us/docs/api-reference/other-references/plans#audio-conferencing-plans">plan type.</a>
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountPlansResponsePlanAudio {
    /**
     * Additional audio conferencing <a href="https://marketplace.zoom.us/docs/api-reference/other-references/plans#audio-conferencing-plans">plan type.</a>
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub callout_countries: String,
    /**
     * Additional audio conferencing <a href="https://marketplace.zoom.us/docs/api-reference/other-references/plans#audio-conferencing-plans">plan type.</a>
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub ddi_numbers: i64,
    /**
     * Additional audio conferencing <a href="https://marketplace.zoom.us/docs/api-reference/other-references/plans#audio-conferencing-plans">plan type.</a>
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub next_invoice_date: Option<chrono::NaiveDate>,
    /**
     * Additional audio conferencing <a href="https://marketplace.zoom.us/docs/api-reference/other-references/plans#audio-conferencing-plans">plan type.</a>
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub premium_countries: String,
    /**
     * Additional audio conferencing <a href="https://marketplace.zoom.us/docs/api-reference/other-references/plans#audio-conferencing-plans">plan type.</a>
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub service_effective_date: Option<chrono::NaiveDate>,
    /**
     * Additional audio conferencing <a href="https://marketplace.zoom.us/docs/api-reference/other-references/plans#audio-conferencing-plans">plan type.</a>
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<AccountPlansResponsePlanBaseStatus>,
    /**
     * Additional audio conferencing <a href="https://marketplace.zoom.us/docs/api-reference/other-references/plans#audio-conferencing-plans">plan type.</a>
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub tollfree_countries: String,
    /**
     * Additional audio conferencing <a href="https://marketplace.zoom.us/docs/api-reference/other-references/plans#audio-conferencing-plans">plan type.</a>
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

/// Additional phone base plans.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountPlansResponsePlanPhoneBase {
    /**
     * Additional phone base plans.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub callout_countries: String,
    /**
     * Additional phone base plans.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub next_invoice_date: Option<chrono::NaiveDate>,
    /**
     * Additional phone base plans.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub service_effective_date: Option<chrono::NaiveDate>,
    /**
     * Additional phone base plans.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<AccountPlansResponsePlanBaseStatus>,
    /**
     * Additional phone base plans.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

/// Phone Plan Object
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PlanPhone {
    /**
     * Phone Plan Object
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub plan_base: Option<AccountPlansResponsePlanPhoneBase>,
    /**
     * Phone Plan Object
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub plan_calling: Vec<PlanZoomRooms>,
    /**
     * Phone Plan Object
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub plan_number: Vec<PlanZoomRooms>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountPlansResponse {
    /**
     * Additional audio conferencing <a href="https://marketplace.zoom.us/docs/api-reference/other-references/plans#audio-conferencing-plans">plan type.</a>
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub plan_audio: Option<AccountPlansResponsePlanAudio>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub plan_base: Option<PlanZoomRooms>,
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub plan_large_meeting: Vec<PlanWebinar>,
    /**
     * Phone Plan Object
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub plan_phone: Option<PlanPhone>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub plan_recording: String,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub plan_recording_next_invoice_date: Option<chrono::NaiveDate>,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub plan_recording_service_effective_date: Option<chrono::NaiveDate>,
    /**
     * Status of additional Cloud Recording plan.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub plan_recording_status: Option<PlanRecordingStatus>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub plan_room_connector: Option<PlanZoomRooms>,
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub plan_webinar: Vec<PlanWebinar>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub plan_zoom_rooms: Option<PlanZoomRooms>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountPlanCreateRequest {
    /**
     * Billing Contact object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub contact: Option<Contact>,
}

/// All of the following types:
///
/// - `AccountPlans`
/// - `AccountPlanCreateRequest`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AccountPlanCreateRequestAllOf {
    /**
     * Account Plans object.
     */
    AccountPlans(AccountPlans),
    AccountPlanCreateRequest(AccountPlanCreateRequest),
}

impl AccountPlanCreateRequestAllOf {
    pub fn account_plan_create_request(&self) -> Option<&AccountPlanCreateRequest> {
        if let AccountPlanCreateRequestAllOf::AccountPlanCreateRequest(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn account_plans(&self) -> Option<&AccountPlans> {
        if let AccountPlanCreateRequestAllOf::AccountPlans(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// Zoom Phone Plan Object
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountPlanAddonCreateRequest {
    /**
     * Phone Plan Object
     */
    #[serde()]
    pub plan_details: PhonePlan,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

/// All of the following types:
///
/// - `PlanBase`
/// - `AccountPlanAddonCreateRequest`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AccountPlanAddonCreateRequestOneOf {
    /**
     * Account base plan object.
     */
    PlanBase(PlanBase),
    /**
     * Zoom Phone Plan Object
     */
    AccountPlanAddonCreateRequest(AccountPlanAddonCreateRequest),
}

impl AccountPlanAddonCreateRequestOneOf {
    pub fn account_plan_addon_create_request(&self) -> Option<&AccountPlanAddonCreateRequest> {
        if let AccountPlanAddonCreateRequestOneOf::AccountPlanAddonCreateRequest(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn plan_base(&self) -> Option<&PlanBase> {
        if let AccountPlanAddonCreateRequestOneOf::PlanBase(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GroupCreateRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GroupResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_members: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserCreateResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GroupMembersResponseData {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub members: Vec<UserCreateResponse>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_count: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_number: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddRoleMembersRequest {
    /**
     * List of registrants.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub members: Vec<Assistants>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GroupMembersCreateResponse {
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub added_at: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub ids: String,
}

/**
 * The action to perform:
 *   * `move` — Remove the group member from one group and move them to a different group.
 *   * `set_primary` — Set the user's primary group.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum UpdateGroupMemberRequestAction {
    #[serde(rename = "move")]
    Move,
    #[serde(rename = "set_primary")]
    SetPrimary,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for UpdateGroupMemberRequestAction {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            UpdateGroupMemberRequestAction::Move => "move",
            UpdateGroupMemberRequestAction::SetPrimary => "set_primary",
            UpdateGroupMemberRequestAction::Noop => "",
            UpdateGroupMemberRequestAction::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for UpdateGroupMemberRequestAction {
    fn default() -> UpdateGroupMemberRequestAction {
        UpdateGroupMemberRequestAction::Noop
    }
}
impl UpdateGroupMemberRequestAction {
    pub fn is_noop(&self) -> bool {
        matches!(self, UpdateGroupMemberRequestAction::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateGroupMemberRequest {
    /**
     * The action to perform:
     *  \* `move` — Remove the group member from one group and move them to a different group.
     *  \* `set_primary` — Set the user's primary group.
     */
    #[serde(
        default,
        skip_serializing_if = "UpdateGroupMemberRequestAction::is_noop"
    )]
    pub action: UpdateGroupMemberRequestAction,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub target_group_id: String,
}

/// All of the following types:
///
/// - `Device`
/// - `Groups`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DeviceCreateResponseAllOf {
    /**
     * The H.323/SIP device object.
     */
    Device(Device),
    Groups(Groups),
}

impl DeviceCreateResponseAllOf {
    pub fn device(&self) -> Option<&Device> {
        if let DeviceCreateResponseAllOf::Device(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn groups(&self) -> Option<&Groups> {
        if let DeviceCreateResponseAllOf::Groups(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `Groups`
/// - `TrackingField`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum TrackingfieldGetResponseAllOf {
    Groups(Groups),
    /**
     * Tracking Field
     */
    TrackingField(TrackingField),
}

impl TrackingfieldGetResponseAllOf {
    pub fn groups(&self) -> Option<&Groups> {
        if let TrackingfieldGetResponseAllOf::Groups(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn tracking_field(&self) -> Option<&TrackingField> {
        if let TrackingfieldGetResponseAllOf::TrackingField(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/**
 * IM Group types:<br>`normal` - Only members can see automatically see the other members of this group. Other people can search for members within this group. <br>`shared` - Everyone under an account can see the group members automatically.<br>`restricted` - Nobody can see the group or search for members except the members in the group.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ImGroupCreateRequestType {
    #[serde(rename = "normal")]
    Normal,
    #[serde(rename = "restricted")]
    Restricted,
    #[serde(rename = "shared")]
    Shared,
    FallthroughString(String),
}

impl std::fmt::Display for ImGroupCreateRequestType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ImGroupCreateRequestType::Normal => "normal",
            ImGroupCreateRequestType::Restricted => "restricted",
            ImGroupCreateRequestType::Shared => "shared",
            ImGroupCreateRequestType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for ImGroupCreateRequestType {
    fn default() -> ImGroupCreateRequestType {
        ImGroupCreateRequestType::Normal
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ImGroupCreateRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub search_by_account: Option<bool>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub search_by_domain: Option<bool>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub search_by_ma_account: Option<bool>,
    /**
     * IM Group types:<br>`normal` - Only members can see automatically see the other members of this group. Other people can search for members within this group. <br>`shared` - Everyone under an account can see the group members automatically.<br>`restricted` - Nobody can see the group or search for members except the members in the group.
     */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<ImGroupCreateRequestType>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ImGroupCreateResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub search_by_account: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub search_by_domain: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub search_by_ma_account: bool,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_members: i64,
}

/// All of the following types:
///
/// - `Domains`
/// - `Groups`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ImGroupResponseAllOf {
    Domains(Domains),
    Groups(Groups),
}

impl ImGroupResponseAllOf {
    pub fn domains(&self) -> Option<&Domains> {
        if let ImGroupResponseAllOf::Domains(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn groups(&self) -> Option<&Groups> {
        if let ImGroupResponseAllOf::Groups(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardImResponse {
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub from: Option<chrono::NaiveDate>,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub to: Option<chrono::NaiveDate>,
}

/**
 * IM chat session type.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ImChatSessionsResponseType {
    #[serde(rename = "1:1")]
    One,
    #[serde(rename = "Group")]
    Group,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for ImChatSessionsResponseType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ImChatSessionsResponseType::One => "1:1",
            ImChatSessionsResponseType::Group => "Group",
            ImChatSessionsResponseType::Noop => "",
            ImChatSessionsResponseType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for ImChatSessionsResponseType {
    fn default() -> ImChatSessionsResponseType {
        ImChatSessionsResponseType::Noop
    }
}
impl ImChatSessionsResponseType {
    pub fn is_noop(&self) -> bool {
        matches!(self, ImChatSessionsResponseType::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Sessions {
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub last_message_sent_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub session_id: String,
    /**
     * IM chat session type.
     */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<ImChatSessionsResponseType>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ImChatSessionsResponseData {
    /**
     * Array of session objects.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub sessions: Vec<Sessions>,
}

/// All of the following types:
///
/// - `PaginationToken4ImChat`
/// - `DashboardImResponse`
/// - `ImChatSessionsResponseData`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ImChatSessionsResponseAllOf {
    /**
     * Pagination object.
     */
    PaginationToken4ImChat(PaginationToken4ImChat),
    DashboardImResponse(DashboardImResponse),
    ImChatSessionsResponseData(ImChatSessionsResponseData),
}

impl ImChatSessionsResponseAllOf {
    pub fn dashboard_im_response(&self) -> Option<&DashboardImResponse> {
        if let ImChatSessionsResponseAllOf::DashboardImResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn im_chat_sessions_response_data(&self) -> Option<&ImChatSessionsResponseData> {
        if let ImChatSessionsResponseAllOf::ImChatSessionsResponseData(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn pagination_token_4_im_chat(&self) -> Option<&PaginationToken4ImChat> {
        if let ImChatSessionsResponseAllOf::PaginationToken4ImChat(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ImChatMessagesResponse {
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub from: Option<chrono::NaiveDate>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub session_id: String,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub to: Option<chrono::NaiveDate>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ImChatMessagesResponseData {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub action: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub action_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub date_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub message: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub sender: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ImChatMessagesResponseDataType {
    /**
     * Array of session objects.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub messages: Vec<ImChatMessagesResponseData>,
}

/// All of the following types:
///
/// - `PaginationToken4ImChat`
/// - `ImChatMessagesResponse`
/// - `ImChatMessagesResponseDataType`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ImChatMessagesResponseAllOf {
    /**
     * Pagination object.
     */
    PaginationToken4ImChat(PaginationToken4ImChat),
    ImChatMessagesResponse(ImChatMessagesResponse),
    ImChatMessagesResponseDataType(ImChatMessagesResponseDataType),
}

impl ImChatMessagesResponseAllOf {
    pub fn im_chat_messages_response(&self) -> Option<&ImChatMessagesResponse> {
        if let ImChatMessagesResponseAllOf::ImChatMessagesResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn im_chat_messages_response_data_type(&self) -> Option<&ImChatMessagesResponseDataType> {
        if let ImChatMessagesResponseAllOf::ImChatMessagesResponseDataType(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn pagination_token_4_im_chat(&self) -> Option<&PaginationToken4ImChat> {
        if let ImChatMessagesResponseAllOf::PaginationToken4ImChat(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/**
 * The meeting types: <br>`scheduled` - This includes all valid past meetings (unexpired), live meetings and upcoming scheduled meetings. It is equivalent to the combined list of "Previous Meetings" and "Upcoming Meetings" displayed in the user's [Meetings page](https://zoom.us/meeting) on the Zoom Web Portal.<br>`live` - All the ongoing meetings.<br>`upcoming` - All upcoming meetings including live meetings.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum MeetingsType {
    #[serde(rename = "live")]
    Live,
    #[serde(rename = "scheduled")]
    Scheduled,
    #[serde(rename = "upcoming")]
    Upcoming,
    FallthroughString(String),
}

impl std::fmt::Display for MeetingsType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            MeetingsType::Live => "live",
            MeetingsType::Scheduled => "scheduled",
            MeetingsType::Upcoming => "upcoming",
            MeetingsType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for MeetingsType {
    fn default() -> MeetingsType {
        MeetingsType::Live
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingCreateResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub assistant_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub host_email: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub registration_url: String,
}

/// All of the following types:
///
/// - `MeetingInfo`
/// - `MeetingCreateResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum MeetingCreateResponseAllOf {
    /**
     * Meeting object
     */
    MeetingInfo(MeetingInfo),
    MeetingCreateResponse(MeetingCreateResponse),
}

impl MeetingCreateResponseAllOf {
    pub fn meeting_create_response(&self) -> Option<&MeetingCreateResponse> {
        if let MeetingCreateResponseAllOf::MeetingCreateResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn meeting_info(&self) -> Option<&MeetingInfo> {
        if let MeetingCreateResponseAllOf::MeetingInfo(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub assistant_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub host_email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub host_id: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub uuid: String,
}

/// All of the following types:
///
/// - `MeetingInfoGet`
/// - `MeetingResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum MeetingResponseAllOf {
    /**
     * Meeting object.
     */
    MeetingInfoGet(MeetingInfoGet),
    MeetingResponse(MeetingResponse),
}

impl MeetingResponseAllOf {
    pub fn meeting_info_get(&self) -> Option<&MeetingInfoGet> {
        if let MeetingResponseAllOf::MeetingInfoGet(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn meeting_response(&self) -> Option<&MeetingResponse> {
        if let MeetingResponseAllOf::MeetingResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingUpdateRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub schedule_for: String,
}

/// All of the following types:
///
/// - `Domains`
/// - `MeetingUpdateRequest`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum MeetingUpdateRequestAllOf {
    Domains(Domains),
    MeetingUpdateRequest(MeetingUpdateRequest),
}

impl MeetingUpdateRequestAllOf {
    pub fn domains(&self) -> Option<&Domains> {
        if let MeetingUpdateRequestAllOf::Domains(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn meeting_update_request(&self) -> Option<&MeetingUpdateRequest> {
        if let MeetingUpdateRequestAllOf::MeetingUpdateRequest(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/**
 * `end` - End a meeting.<br>
 *   `recover` - [Recover](https://support.zoom.us/hc/en-us/articles/360038297111-Recover-a-deleted-meeting) a deleted meeting.
 *   
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum MeetingStatusRequestAction {
    #[serde(rename = "end")]
    End,
    #[serde(rename = "recover")]
    Recover,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for MeetingStatusRequestAction {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            MeetingStatusRequestAction::End => "end",
            MeetingStatusRequestAction::Recover => "recover",
            MeetingStatusRequestAction::Noop => "",
            MeetingStatusRequestAction::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for MeetingStatusRequestAction {
    fn default() -> MeetingStatusRequestAction {
        MeetingStatusRequestAction::Noop
    }
}
impl MeetingStatusRequestAction {
    pub fn is_noop(&self) -> bool {
        matches!(self, MeetingStatusRequestAction::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingStatusRequest {
    /**
     * `end` - End a meeting.<br>
     *  `recover` - [Recover](https://support.zoom.us/hc/en-us/articles/360038297111-Recover-a-deleted-meeting) a deleted meeting.
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub action: Option<MeetingStatusRequestAction>,
}

/**
 * The registrant status:<br>`pending` - Registrant's status is pending.<br>`approved` - Registrant's status is approved.<br>`denied` - Registrant's status is denied.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum MeetingRegistrantsStatus {
    #[serde(rename = "approved")]
    Approved,
    #[serde(rename = "denied")]
    Denied,
    #[serde(rename = "pending")]
    Pending,
    FallthroughString(String),
}

impl std::fmt::Display for MeetingRegistrantsStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            MeetingRegistrantsStatus::Approved => "approved",
            MeetingRegistrantsStatus::Denied => "denied",
            MeetingRegistrantsStatus::Pending => "pending",
            MeetingRegistrantsStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for MeetingRegistrantsStatus {
    fn default() -> MeetingRegistrantsStatus {
        MeetingRegistrantsStatus::Approved
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingRegistrantCreateResponse {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub join_url: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub registrant_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub start_time: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PastMeetingDetailsResponse {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub end_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub host_id: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub participants_count: i64,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_minutes: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub uuid: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Participants {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_email: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PastMeetingParticipantsResponse {
    /**
     * Array of meeting participant objects.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub participants: Vec<Participants>,
}

/// All of the following types:
///
/// - `PaginationToken`
/// - `PastMeetingParticipantsResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum PastMeetingParticipantsResponseAllOf {
    /**
     * Pagination object.
     */
    PaginationToken(PaginationToken),
    PastMeetingParticipantsResponse(PastMeetingParticipantsResponse),
}

impl PastMeetingParticipantsResponseAllOf {
    pub fn pagination_token(&self) -> Option<&PaginationToken> {
        if let PastMeetingParticipantsResponseAllOf::PaginationToken(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn past_meeting_participants_response(&self) -> Option<&PastMeetingParticipantsResponse> {
        if let PastMeetingParticipantsResponseAllOf::PastMeetingParticipantsResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `Poll`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum MeetingPollCreateRequestAllOf {
    /**
     * Poll
     */
    Poll(Poll),
}

impl MeetingPollCreateRequestAllOf {
    pub fn poll(&self) -> Option<&Poll> {
        let MeetingPollCreateRequestAllOf::Poll(ref_) = self;
        Some(ref_)
    }
}

/**
 * Status of the Meeting Poll:<br>`notstart` - Poll not started<br>`started` - Poll started<br>`ended` - Poll ended<br>`sharing` - Sharing poll results
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum MeetingPollCreateResponseStatus {
    #[serde(rename = "ended")]
    Ended,
    #[serde(rename = "notstart")]
    Notstart,
    #[serde(rename = "sharing")]
    Sharing,
    #[serde(rename = "started")]
    Started,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for MeetingPollCreateResponseStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            MeetingPollCreateResponseStatus::Ended => "ended",
            MeetingPollCreateResponseStatus::Notstart => "notstart",
            MeetingPollCreateResponseStatus::Sharing => "sharing",
            MeetingPollCreateResponseStatus::Started => "started",
            MeetingPollCreateResponseStatus::Noop => "",
            MeetingPollCreateResponseStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for MeetingPollCreateResponseStatus {
    fn default() -> MeetingPollCreateResponseStatus {
        MeetingPollCreateResponseStatus::Noop
    }
}
impl MeetingPollCreateResponseStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, MeetingPollCreateResponseStatus::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingPollGetResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Status of the Meeting Poll:<br>`notstart` - Poll not started<br>`started` - Poll started<br>`ended` - Poll ended<br>`sharing` - Sharing poll results
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<MeetingPollCreateResponseStatus>,
}

/// All of the following types:
///
/// - `Poll`
/// - `MeetingPollGetResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum MeetingPollGetResponseAllOf {
    /**
     * Poll
     */
    Poll(Poll),
    MeetingPollGetResponse(MeetingPollGetResponse),
}

impl MeetingPollGetResponseAllOf {
    pub fn meeting_poll_get_response(&self) -> Option<&MeetingPollGetResponse> {
        if let MeetingPollGetResponseAllOf::MeetingPollGetResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn poll(&self) -> Option<&Poll> {
        if let MeetingPollGetResponseAllOf::Poll(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `MeetingRegistrantQuestionsData`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum MeetingRegistrantsQuestionsGetResponseAllOf {
    /**
     * Meeting Registrant Questions
     */
    MeetingRegistrantQuestionsData(MeetingRegistrantQuestionsData),
}

impl MeetingRegistrantsQuestionsGetResponseAllOf {
    pub fn meeting_registrant_questions_data(&self) -> Option<&MeetingRegistrantQuestionsData> {
        let MeetingRegistrantsQuestionsGetResponseAllOf::MeetingRegistrantQuestionsData(ref_) =
            self;
        Some(ref_)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct RecordingGetResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub download_access_token: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
}

/// All of the following types:
///
/// - `Domains`
/// - `RecordingGetResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum RecordingGetResponseAllOf {
    Domains(Domains),
    RecordingGetResponse(RecordingGetResponse),
}

impl RecordingGetResponseAllOf {
    pub fn domains(&self) -> Option<&Domains> {
        if let RecordingGetResponseAllOf::Domains(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn recording_get_response(&self) -> Option<&RecordingGetResponse> {
        if let RecordingGetResponseAllOf::RecordingGetResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/**
 * The recording delete actions:<br>`trash` - Move recording to trash.<br>`delete` - Delete recording permanently.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum RecordingDeleteAction {
    #[serde(rename = "delete")]
    Delete,
    #[serde(rename = "trash")]
    Trash,
    FallthroughString(String),
}

impl std::fmt::Display for RecordingDeleteAction {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            RecordingDeleteAction::Delete => "delete",
            RecordingDeleteAction::Trash => "trash",
            RecordingDeleteAction::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for RecordingDeleteAction {
    fn default() -> RecordingDeleteAction {
        RecordingDeleteAction::Trash
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MeetingRecordingRegistrantCreateResponse {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub registrant_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub share_url: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
}

/// All of the following types:
///
/// - `RecordingRegistrantQuestionsData`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum RecordingRegistrantQuestionUpdateRequestAllOf {
    /**
     * Recording Registrant Questions
     */
    RecordingRegistrantQuestionsData(RecordingRegistrantQuestionsData),
}

impl RecordingRegistrantQuestionUpdateRequestAllOf {
    pub fn recording_registrant_questions_data(&self) -> Option<&RecordingRegistrantQuestionsData> {
        let RecordingRegistrantQuestionUpdateRequestAllOf::RecordingRegistrantQuestionsData(ref_) =
            self;
        Some(ref_)
    }
}

/**
 * Specify a value to get the response for the corresponding meeting type. The value of this field can be one of the following:<br> <br>`past` - Meeting that already occurred in the specified date range.<br>`pastOne` - Past meetings that were attended by only one user. <br>`live` - Live meetings.<br><br>
 *   
 *   If you do not provide this field, the default value will be `live` and thus, the API will only query responses for live meetings.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DashboardMeetingsType {
    #[serde(rename = "live")]
    Live,
    #[serde(rename = "past")]
    Past,
    #[serde(rename = "pastOne")]
    PastOne,
    FallthroughString(String),
}

impl std::fmt::Display for DashboardMeetingsType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DashboardMeetingsType::Live => "live",
            DashboardMeetingsType::Past => "past",
            DashboardMeetingsType::PastOne => "pastOne",
            DashboardMeetingsType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for DashboardMeetingsType {
    fn default() -> DashboardMeetingsType {
        DashboardMeetingsType::Live
    }
}

/**
 * Set the value of this field to "tracking_fields" if you would like to include tracking fields of each meeting in the response.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum IncludeFields {
    #[serde(rename = "tracking_fields")]
    TrackingFields,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for IncludeFields {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IncludeFields::TrackingFields => "tracking_fields",
            IncludeFields::Noop => "",
            IncludeFields::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for IncludeFields {
    fn default() -> IncludeFields {
        IncludeFields::Noop
    }
}
impl IncludeFields {
    pub fn is_noop(&self) -> bool {
        matches!(self, IncludeFields::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardMeetingsResponse {
    /**
     * Array of meeting objects.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub meetings: Vec<Metrics>,
}

/// All of the following types:
///
/// - `PaginationToken`
/// - `DashboardImResponse`
/// - `DashboardMeetingsResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DashboardMeetingsResponseAllOf {
    /**
     * Pagination object.
     */
    PaginationToken(PaginationToken),
    DashboardImResponse(DashboardImResponse),
    DashboardMeetingsResponse(DashboardMeetingsResponse),
}

impl DashboardMeetingsResponseAllOf {
    pub fn dashboard_im_response(&self) -> Option<&DashboardImResponse> {
        if let DashboardMeetingsResponseAllOf::DashboardImResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn dashboard_meetings_response(&self) -> Option<&DashboardMeetingsResponse> {
        if let DashboardMeetingsResponseAllOf::DashboardMeetingsResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn pagination_token(&self) -> Option<&PaginationToken> {
        if let DashboardMeetingsResponseAllOf::PaginationToken(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/**
 * Provide `registrant_id` as the value for this field if you would like to see the registrant ID attribute in the response of this API call. A registrant ID is a unique identifier of a [meeting registrant](https://marketplace.zoom.us/docs/api-reference/zoom-api/meetings/meetingregistrants). This is not supported for `live` meeting types.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DashboardMeetingParticipantsIncludeFields {
    #[serde(rename = "registrant_id")]
    RegistrantId,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for DashboardMeetingParticipantsIncludeFields {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DashboardMeetingParticipantsIncludeFields::RegistrantId => "registrant_id",
            DashboardMeetingParticipantsIncludeFields::Noop => "",
            DashboardMeetingParticipantsIncludeFields::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for DashboardMeetingParticipantsIncludeFields {
    fn default() -> DashboardMeetingParticipantsIncludeFields {
        DashboardMeetingParticipantsIncludeFields::Noop
    }
}
impl DashboardMeetingParticipantsIncludeFields {
    pub fn is_noop(&self) -> bool {
        matches!(self, DashboardMeetingParticipantsIncludeFields::Noop)
    }
}

/**
 * The type of device the participant used to join the meeting:
 *   * `Phone` — Participant joined via PSTN.
 *   * `H.323/SIP` — Participant joined via an H.323 or SIP device.
 *   * `Windows` — Participant joined via VoIP using a Windows device.
 *   * `Mac` — Participant joined via VoIP using a Mac device.
 *   * `iOS` — Participant joined via VoIP using an iOS device.
 *   * `Android` — Participant joined via VoIP using an Android device.
 *   
 *   **Note:** This response returns an empty string (`““`) value for any users who are **not** a part of the host’s account (external users).
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DashboardMeetingParticipantsResponseDevice {
    #[serde(rename = "Android")]
    Android,
    #[serde(rename = "H.323/SIP")]
    H323Sip,
    #[serde(rename = "Mac")]
    Mac,
    #[serde(rename = "Phone")]
    Phone,
    #[serde(rename = "Windows")]
    Windows,
    #[serde(rename = "iOS")]
    IOs,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for DashboardMeetingParticipantsResponseDevice {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DashboardMeetingParticipantsResponseDevice::Android => "Android",
            DashboardMeetingParticipantsResponseDevice::H323Sip => "H.323/SIP",
            DashboardMeetingParticipantsResponseDevice::Mac => "Mac",
            DashboardMeetingParticipantsResponseDevice::Phone => "Phone",
            DashboardMeetingParticipantsResponseDevice::Windows => "Windows",
            DashboardMeetingParticipantsResponseDevice::IOs => "iOS",
            DashboardMeetingParticipantsResponseDevice::Noop => "",
            DashboardMeetingParticipantsResponseDevice::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for DashboardMeetingParticipantsResponseDevice {
    fn default() -> DashboardMeetingParticipantsResponseDevice {
        DashboardMeetingParticipantsResponseDevice::Noop
    }
}
impl DashboardMeetingParticipantsResponseDevice {
    pub fn is_noop(&self) -> bool {
        matches!(self, DashboardMeetingParticipantsResponseDevice::Noop)
    }
}

/**
 * The participant's network type:
 *   
 *   * `Wired`
 *   * `Wifi`
 *   * `PPP` — Point-to-Point.
 *   * `Cellular` — 3G, 4G, and 5G cellular.
 *   * `Others` — An unknown device.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum NetworkType {
    #[serde(rename = "Cellular")]
    Cellular,
    #[serde(rename = "Others")]
    Others,
    #[serde(rename = "PPP")]
    Ppp,
    #[serde(rename = "Wifi")]
    Wifi,
    #[serde(rename = "Wired")]
    Wired,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for NetworkType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            NetworkType::Cellular => "Cellular",
            NetworkType::Others => "Others",
            NetworkType::Ppp => "PPP",
            NetworkType::Wifi => "Wifi",
            NetworkType::Wired => "Wired",
            NetworkType::Noop => "",
            NetworkType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for NetworkType {
    fn default() -> NetworkType {
        NetworkType::Noop
    }
}
impl NetworkType {
    pub fn is_noop(&self) -> bool {
        matches!(self, NetworkType::Noop)
    }
}

/**
 * Audio quality of the participant.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AudioQuality {
    #[serde(rename = "bad")]
    Bad,
    #[serde(rename = "fair")]
    Fair,
    #[serde(rename = "good")]
    Good,
    #[serde(rename = "poor")]
    Poor,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for AudioQuality {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AudioQuality::Bad => "bad",
            AudioQuality::Fair => "fair",
            AudioQuality::Good => "good",
            AudioQuality::Poor => "poor",
            AudioQuality::Noop => "",
            AudioQuality::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for AudioQuality {
    fn default() -> AudioQuality {
        AudioQuality::Noop
    }
}
impl AudioQuality {
    pub fn is_noop(&self) -> bool {
        matches!(self, AudioQuality::Noop)
    }
}

/**
 * Indicates whether the participant is in the waiting room or in the meeting.
 *   
 *   The value of this field can be `in_meeting` or `in_waiting_room`.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DashboardMeetingParticipantsResponseStatus {
    #[serde(rename = "in_meeting")]
    InMeeting,
    #[serde(rename = "in_waiting_room")]
    InWaitingRoom,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for DashboardMeetingParticipantsResponseStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DashboardMeetingParticipantsResponseStatus::InMeeting => "in_meeting",
            DashboardMeetingParticipantsResponseStatus::InWaitingRoom => "in_waiting_room",
            DashboardMeetingParticipantsResponseStatus::Noop => "",
            DashboardMeetingParticipantsResponseStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for DashboardMeetingParticipantsResponseStatus {
    fn default() -> DashboardMeetingParticipantsResponseStatus {
        DashboardMeetingParticipantsResponseStatus::Noop
    }
}
impl DashboardMeetingParticipantsResponseStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, DashboardMeetingParticipantsResponseStatus::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardMeetingParticipantsResponse {
    /**
     * Audio quality of the participant.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub audio_quality: Option<AudioQuality>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub camera: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub connection_type: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub customer_key: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub data_center: String,
    /**
     * The type of device the participant used to join the meeting:
     *  \* `Phone` — Participant joined via PSTN.
     *  \* `H.323/SIP` — Participant joined via an H.323 or SIP device.
     *  \* `Windows` — Participant joined via VoIP using a Windows device.
     *  \* `Mac` — Participant joined via VoIP using a Mac device.
     *  \* `iOS` — Participant joined via VoIP using an iOS device.
     *  \* `Android` — Participant joined via VoIP using an Android device.
     *  
     *  \*\*Note:\*\* This response returns an empty string (`““`) value for any users who are \*\*not\*\* a part of the host’s account (external users).
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub device: Option<DashboardMeetingParticipantsResponseDevice>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub domain: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub harddisk_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub in_room_participants: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub ip_address: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub join_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub leave_reason: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub leave_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub mac_addr: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub microphone: String,
    /**
     * The participant's network type:
     *  
     *  \* `Wired`
     *  \* `Wifi`
     *  \* `PPP` — Point-to-Point.
     *  \* `Cellular` — 3G, 4G, and 5G cellular.
     *  \* `Others` — An unknown device.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub network_type: Option<NetworkType>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub pc_name: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub recording: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub registrant_id: String,
    /**
     * Audio quality of the participant.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub screen_share_quality: Option<AudioQuality>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub share_application: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub share_desktop: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub share_whiteboard: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub speaker: String,
    /**
     * Indicates whether the participant is in the waiting room or in the meeting.
     *  
     *  The value of this field can be `in_meeting` or `in_waiting_room`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<DashboardMeetingParticipantsResponseStatus>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub version: String,
    /**
     * Audio quality of the participant.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub video_quality: Option<AudioQuality>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardMeetingParticipantsResponseData {
    /**
     * Array of participant session objects. If a participant left a meeting and rejoined the same meeting, their information will appear multiple times (as many times as they joined the meeting).
     *  
     *  If the account calling this API is a [business associate (BAA) under HIPAA](https://www.ecfr.gov/cgi-bin/retrieveECFR?gp=1&n=se45.1.160_1103&r=SECTION&ty=HTML), this array returns an empty string value for the `user_name`, `ip_address`, `location`, and `email` responses.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub participants: Vec<DashboardMeetingParticipantsResponse>,
}

/// All of the following types:
///
/// - `PaginationToken`
/// - `DashboardMeetingParticipantsResponseData`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DashboardMeetingParticipantsResponseAllOf {
    /**
     * Pagination object.
     */
    PaginationToken(PaginationToken),
    DashboardMeetingParticipantsResponseData(DashboardMeetingParticipantsResponseData),
}

impl DashboardMeetingParticipantsResponseAllOf {
    pub fn dashboard_meeting_participants_response_data(
        &self,
    ) -> Option<&DashboardMeetingParticipantsResponseData> {
        if let DashboardMeetingParticipantsResponseAllOf::DashboardMeetingParticipantsResponseData(
            ref_,
        ) = self
        {
            return Some(ref_);
        }
        None
    }

    pub fn pagination_token(&self) -> Option<&PaginationToken> {
        if let DashboardMeetingParticipantsResponseAllOf::PaginationToken(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Details {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub content: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub end_time: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub start_time: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardMeetingParticipantShareResponseParticipants {
    /**
     * Array of sharing and recording details.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub details: Vec<Details>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_name: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardMeetingParticipantShareResponse {
    /**
     * Array of participants.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub participants: Vec<DashboardMeetingParticipantShareResponseParticipants>,
}

/// All of the following types:
///
/// - `PaginationToken`
/// - `DashboardMeetingParticipantShareResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DashboardMeetingParticipantShareResponseAllOf {
    /**
     * Pagination object.
     */
    PaginationToken(PaginationToken),
    DashboardMeetingParticipantShareResponse(DashboardMeetingParticipantShareResponse),
}

impl DashboardMeetingParticipantShareResponseAllOf {
    pub fn dashboard_meeting_participant_share_response(
        &self,
    ) -> Option<&DashboardMeetingParticipantShareResponse> {
        if let DashboardMeetingParticipantShareResponseAllOf::DashboardMeetingParticipantShareResponse(ref_) = self {
                                return Some(ref_);
                            }
        None
    }

    pub fn pagination_token(&self) -> Option<&PaginationToken> {
        if let DashboardMeetingParticipantShareResponseAllOf::PaginationToken(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/**
 * The webinar type.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DashboardWebinarsType {
    #[serde(rename = "live")]
    Live,
    #[serde(rename = "past")]
    Past,
    FallthroughString(String),
}

impl std::fmt::Display for DashboardWebinarsType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DashboardWebinarsType::Live => "live",
            DashboardWebinarsType::Past => "past",
            DashboardWebinarsType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for DashboardWebinarsType {
    fn default() -> DashboardWebinarsType {
        DashboardWebinarsType::Live
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardWebinarsResponse {
    /**
     * Array of webinar objects.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub webinars: Vec<Webinars>,
}

/// All of the following types:
///
/// - `PaginationToken`
/// - `DashboardImResponse`
/// - `DashboardWebinarsResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DashboardWebinarsResponseAllOf {
    /**
     * Pagination object.
     */
    PaginationToken(PaginationToken),
    DashboardImResponse(DashboardImResponse),
    DashboardWebinarsResponse(DashboardWebinarsResponse),
}

impl DashboardWebinarsResponseAllOf {
    pub fn dashboard_im_response(&self) -> Option<&DashboardImResponse> {
        if let DashboardWebinarsResponseAllOf::DashboardImResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn dashboard_webinars_response(&self) -> Option<&DashboardWebinarsResponse> {
        if let DashboardWebinarsResponseAllOf::DashboardWebinarsResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn pagination_token(&self) -> Option<&PaginationToken> {
        if let DashboardWebinarsResponseAllOf::PaginationToken(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardWebinarParticipantsResponse {
    /**
     * Audio quality of the participant.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub audio_quality: Option<AudioQuality>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub connection_type: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub customer_key: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub data_center: String,
    /**
     * The type of device the participant used to join the meeting:
     *  \* `Phone` — Participant joined via PSTN.
     *  \* `H.323/SIP` — Participant joined via an H.323 or SIP device.
     *  \* `Windows` — Participant joined via VoIP using a Windows device.
     *  \* `Mac` — Participant joined via VoIP using a Mac device.
     *  \* `iOS` — Participant joined via VoIP using an iOS device.
     *  \* `Android` — Participant joined via VoIP using an Android device.
     *  
     *  \*\*Note:\*\* This response returns an empty string (`““`) value for any users who are \*\*not\*\* a part of the host’s account (external users).
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub device: Option<DashboardMeetingParticipantsResponseDevice>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub domain: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub harddisk_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub ip_address: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub join_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub leave_reason: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub leave_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub mac_addr: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub microphone: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub network_type: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub pc_name: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub recording: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub registrant_id: String,
    /**
     * Audio quality of the participant.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub screen_share_quality: Option<AudioQuality>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub share_application: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub share_desktop: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub share_whiteboard: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub speaker: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub version: String,
    /**
     * Audio quality of the participant.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub video_quality: Option<AudioQuality>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardWebinarParticipantsResponseData {
    /**
     * Array of user objects.
     *  
     *  If the account calling this API is a [business associate (BAA) under HIPAA](https://www.ecfr.gov/cgi-bin/retrieveECFR?gp=1&n=se45.1.160_1103&r=SECTION&ty=HTML), this array returns an empty string value for the `user_name`, `ip_address`, `location`, and `email` responses.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub participants: Vec<DashboardWebinarParticipantsResponse>,
}

/// All of the following types:
///
/// - `PaginationToken`
/// - `DashboardWebinarParticipantsResponseData`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DashboardWebinarParticipantsResponseAllOf {
    /**
     * Pagination object.
     */
    PaginationToken(PaginationToken),
    DashboardWebinarParticipantsResponseData(DashboardWebinarParticipantsResponseData),
}

impl DashboardWebinarParticipantsResponseAllOf {
    pub fn dashboard_webinar_participants_response_data(
        &self,
    ) -> Option<&DashboardWebinarParticipantsResponseData> {
        if let DashboardWebinarParticipantsResponseAllOf::DashboardWebinarParticipantsResponseData(
            ref_,
        ) = self
        {
            return Some(ref_);
        }
        None
    }

    pub fn pagination_token(&self) -> Option<&PaginationToken> {
        if let DashboardWebinarParticipantsResponseAllOf::PaginationToken(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Users {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub calls_receive: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub calls_send: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub emoji_receive: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub emoji_send: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub files_receive: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub files_send: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub group_receive: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub group_send: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub images_receive: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub images_send: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_receive: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_send: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_name: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub videos_receive: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub videos_send: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub voice_receive: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub voice_send: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardImResponseData {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub users: Vec<Users>,
}

/// All of the following types:
///
/// - `PaginationToken`
/// - `DashboardImResponse`
/// - `DashboardImResponseData`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DashboardImResponseAllOf {
    /**
     * Pagination object.
     */
    PaginationToken(PaginationToken),
    DashboardImResponse(DashboardImResponse),
    DashboardImResponseData(DashboardImResponseData),
}

impl DashboardImResponseAllOf {
    pub fn dashboard_im_response(&self) -> Option<&DashboardImResponse> {
        if let DashboardImResponseAllOf::DashboardImResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn dashboard_im_response_data(&self) -> Option<&DashboardImResponseData> {
        if let DashboardImResponseAllOf::DashboardImResponseData(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn pagination_token(&self) -> Option<&PaginationToken> {
        if let DashboardImResponseAllOf::PaginationToken(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardChatResponse {
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub from: Option<chrono::NaiveDate>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub to: Option<chrono::NaiveDate>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardChatResponseUsers {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub audio_sent: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub code_sippet_sent: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub files_sent: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub giphys_sent: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub group_sent: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub images_sent: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub p_2p_sent: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub text_sent: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_sent: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_name: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub video_sent: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardChatResponseData {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub users: Vec<DashboardChatResponseUsers>,
}

/// All of the following types:
///
/// - `DashboardChatResponse`
/// - `DashboardChatResponseData`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DashboardChatResponseAllOf {
    DashboardChatResponse(DashboardChatResponse),
    DashboardChatResponseData(DashboardChatResponseData),
}

impl DashboardChatResponseAllOf {
    pub fn dashboard_chat_response(&self) -> Option<&DashboardChatResponse> {
        if let DashboardChatResponseAllOf::DashboardChatResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn dashboard_chat_response_data(&self) -> Option<&DashboardChatResponseData> {
        if let DashboardChatResponseAllOf::DashboardChatResponseData(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ClientFeedbacks {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub feedback_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub feedback_name: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub participants_count: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardClientFeedbackResponse {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub client_feedbacks: Vec<ClientFeedbacks>,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub from: Option<chrono::NaiveDate>,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub to: Option<chrono::NaiveDate>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardIssueZoomRoomResponse {
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub from: Option<chrono::NaiveDate>,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub to: Option<chrono::NaiveDate>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardIssueZoomRoomResponseData {
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub zoom_rooms: Vec<String>,
}

/// All of the following types:
///
/// - `DashboardIssueZoomRoomResponse`
/// - `DashboardIssueZoomRoomResponseData`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DashboardIssueZoomRoomResponseAllOf {
    DashboardIssueZoomRoomResponse(DashboardIssueZoomRoomResponse),
    DashboardIssueZoomRoomResponseData(DashboardIssueZoomRoomResponseData),
}

impl DashboardIssueZoomRoomResponseAllOf {
    pub fn dashboard_issue_zoom_room_response(&self) -> Option<&DashboardIssueZoomRoomResponse> {
        if let DashboardIssueZoomRoomResponseAllOf::DashboardIssueZoomRoomResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn dashboard_issue_zoom_room_response_data(
        &self,
    ) -> Option<&DashboardIssueZoomRoomResponseData> {
        if let DashboardIssueZoomRoomResponseAllOf::DashboardIssueZoomRoomResponseData(ref_) = self
        {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct IssueDetails {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub issue: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub time: Option<chrono::DateTime<chrono::Utc>>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardIssueDetailZoomRoomResponse {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub issue_details: Vec<IssueDetails>,
}

/// All of the following types:
///
/// - `PaginationToken`
/// - `DashboardImResponse`
/// - `DashboardIssueDetailZoomRoomResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DashboardIssueDetailZoomRoomResponseAllOf {
    /**
     * Pagination object.
     */
    PaginationToken(PaginationToken),
    DashboardImResponse(DashboardImResponse),
    DashboardIssueDetailZoomRoomResponse(DashboardIssueDetailZoomRoomResponse),
}

impl DashboardIssueDetailZoomRoomResponseAllOf {
    pub fn dashboard_im_response(&self) -> Option<&DashboardImResponse> {
        if let DashboardIssueDetailZoomRoomResponseAllOf::DashboardImResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn dashboard_issue_detail_zoom_room_response(
        &self,
    ) -> Option<&DashboardIssueDetailZoomRoomResponse> {
        if let DashboardIssueDetailZoomRoomResponseAllOf::DashboardIssueDetailZoomRoomResponse(
            ref_,
        ) = self
        {
            return Some(ref_);
        }
        None
    }

    pub fn pagination_token(&self) -> Option<&PaginationToken> {
        if let DashboardIssueDetailZoomRoomResponseAllOf::PaginationToken(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Dates {
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub date: Option<chrono::NaiveDate>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub meeting_minutes: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub meetings: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub new_users: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub participants: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ReportDailyResponse {
    /**
     * Array of date objects.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub dates: Vec<Dates>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub month: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub year: i64,
}

/**
 * Active or inactive hosts.<br>`active` - Active hosts. <br>`inactive` - Inactive hosts.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ReportUsersType {
    #[serde(rename = "active")]
    Active,
    #[serde(rename = "inactive")]
    Inactive,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for ReportUsersType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ReportUsersType::Active => "active",
            ReportUsersType::Inactive => "inactive",
            ReportUsersType::Noop => "",
            ReportUsersType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for ReportUsersType {
    fn default() -> ReportUsersType {
        ReportUsersType::Noop
    }
}
impl ReportUsersType {
    pub fn is_noop(&self) -> bool {
        matches!(self, ReportUsersType::Noop)
    }
}

/**
 * The meeting types: <br>`past` - Past meetings.<br>`pastOne` - Past one user meetings.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ReportMeetingsType {
    #[serde(rename = "past")]
    Past,
    #[serde(rename = "pastOne")]
    PastOne,
    FallthroughString(String),
}

impl std::fmt::Display for ReportMeetingsType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ReportMeetingsType::Past => "past",
            ReportMeetingsType::PastOne => "pastOne",
            ReportMeetingsType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for ReportMeetingsType {
    fn default() -> ReportMeetingsType {
        ReportMeetingsType::Past
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Meetings {
    /**
     * Custom keys and values assigned to the meeting.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub custom_keys: Vec<CustomKeys>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub end_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub participants_count: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub source: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_minutes: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub uuid: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ReportMeetingsResponse {
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub from: Option<chrono::NaiveDate>,
    /**
     * Array of meeting objects.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub meetings: Vec<Meetings>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub to: Option<chrono::NaiveDate>,
}

/// All of the following types:
///
/// - `Pagination`
/// - `ReportMeetingsResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ReportMeetingsResponseAllOf {
    /**
     * Pagination Object.
     */
    Pagination(Pagination),
    ReportMeetingsResponse(ReportMeetingsResponse),
}

impl ReportMeetingsResponseAllOf {
    pub fn pagination(&self) -> Option<&Pagination> {
        if let ReportMeetingsResponseAllOf::Pagination(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn report_meetings_response(&self) -> Option<&ReportMeetingsResponse> {
        if let ReportMeetingsResponseAllOf::ReportMeetingsResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ReportMeetingDetailsResponse {
    /**
     * Custom keys and values assigned to the meeting.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub custom_keys: Vec<CustomKeys>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub dept: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub end_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub participants_count: i64,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_minutes: i64,
    /**
     * Tracking fields
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub tracking_fields: Vec<SessionUpdateTrackingFields>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub uuid: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ReportMeetingParticipantsResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub customer_key: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub failover: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub join_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub leave_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub registrant_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ReportMeetingParticipantsResponseData {
    /**
     * Array of meeting participant objects.
     *  
     *  If the account calling this API is a [business associate (BAA) under HIPAA](https://www.ecfr.gov/cgi-bin/retrieveECFR?gp=1&n=se45.1.160_1103&r=SECTION&ty=HTML), this array returns an empty string value for the `name` and `user_email` responses.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub participants: Vec<ReportMeetingParticipantsResponse>,
}

/// All of the following types:
///
/// - `PaginationToken`
/// - `ReportMeetingParticipantsResponseData`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ReportMeetingParticipantsResponseAllOf {
    /**
     * Pagination object.
     */
    PaginationToken(PaginationToken),
    ReportMeetingParticipantsResponseData(ReportMeetingParticipantsResponseData),
}

impl ReportMeetingParticipantsResponseAllOf {
    pub fn pagination_token(&self) -> Option<&PaginationToken> {
        if let ReportMeetingParticipantsResponseAllOf::PaginationToken(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn report_meeting_participants_response_data(
        &self,
    ) -> Option<&ReportMeetingParticipantsResponseData> {
        if let ReportMeetingParticipantsResponseAllOf::ReportMeetingParticipantsResponseData(ref_) =
            self
        {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct QuestionDetails {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub answer: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub date_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub polling_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub question: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ReportMeetingPollsResponseQuestions {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Array of questions from user.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub question_details: Vec<QuestionDetails>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ReportMeetingPollsResponse {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * Array of meeting question objects.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub questions: Vec<ReportMeetingPollsResponseQuestions>,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub uuid: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ReportWebinarParticipantsResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub customer_key: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub failover: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub join_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub leave_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ReportWebinarParticipantsResponseData {
    /**
     * Array of webinar participant objects.
     *  
     *  If the account calling this API is a [business associate (BAA) under HIPAA](https://www.ecfr.gov/cgi-bin/retrieveECFR?gp=1&n=se45.1.160_1103&r=SECTION&ty=HTML), this array returns an empty string value for the `name` and `user_email` responses.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub participants: Vec<ReportWebinarParticipantsResponse>,
}

/// All of the following types:
///
/// - `PaginationToken`
/// - `ReportWebinarParticipantsResponseData`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ReportWebinarParticipantsResponseAllOf {
    /**
     * Pagination object.
     */
    PaginationToken(PaginationToken),
    ReportWebinarParticipantsResponseData(ReportWebinarParticipantsResponseData),
}

impl ReportWebinarParticipantsResponseAllOf {
    pub fn pagination_token(&self) -> Option<&PaginationToken> {
        if let ReportWebinarParticipantsResponseAllOf::PaginationToken(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn report_webinar_participants_response_data(
        &self,
    ) -> Option<&ReportWebinarParticipantsResponseData> {
        if let ReportWebinarParticipantsResponseAllOf::ReportWebinarParticipantsResponseData(ref_) =
            self
        {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ReportWebinarQaResponseQuestionsQuestionDetails {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub answer: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub question: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ReportWebinarQaResponseQuestions {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Array of questions from user.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub question_details: Vec<ReportWebinarQaResponseQuestionsQuestionDetails>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ReportWebinarQaResponse {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * Array of webinar question objects.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub questions: Vec<ReportWebinarQaResponseQuestions>,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub uuid: String,
}

/**
 * Date types:<br>`start_time` - Query by call start time.<br>`end_time` - Query by call end time.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum QueryDateType {
    #[serde(rename = "end_time")]
    EndTime,
    #[serde(rename = "start_time")]
    StartTime,
    FallthroughString(String),
}

impl std::fmt::Display for QueryDateType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            QueryDateType::EndTime => "end_time",
            QueryDateType::StartTime => "start_time",
            QueryDateType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for QueryDateType {
    fn default() -> QueryDateType {
        QueryDateType::StartTime
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ReportCloudRecordingResponse {
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub cloud_recording_storage: Vec<String>,
}

/// All of the following types:
///
/// - `DashboardImResponse`
/// - `ReportCloudRecordingResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ReportCloudRecordingResponseAllOf {
    DashboardImResponse(DashboardImResponse),
    ReportCloudRecordingResponse(ReportCloudRecordingResponse),
}

impl ReportCloudRecordingResponseAllOf {
    pub fn dashboard_im_response(&self) -> Option<&DashboardImResponse> {
        if let ReportCloudRecordingResponseAllOf::DashboardImResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn report_cloud_recording_response(&self) -> Option<&ReportCloudRecordingResponse> {
        if let ReportCloudRecordingResponseAllOf::ReportCloudRecordingResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/**
 * **Optional**<br>
 *   Filter your response by a category type to see reports for a specific category.
 *   The value for this field can be one of the following:<br> `all`<br>`user`<br>`user_settings`<br>`account`<br>`billing`<br>`im`<br>`recording`<br>`phone_contacts`<br>`webinar`<br>`sub_account`<br>`role`<br>`zoom_rooms`
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum CategoryType {
    #[serde(rename = "account")]
    Account,
    #[serde(rename = "all")]
    All,
    #[serde(rename = "billing")]
    Billing,
    #[serde(rename = "im")]
    Im,
    #[serde(rename = "phone_contacts")]
    PhoneContacts,
    #[serde(rename = "recording")]
    Recording,
    #[serde(rename = "role")]
    Role,
    #[serde(rename = "sub_account")]
    SubAccount,
    #[serde(rename = "user")]
    User,
    #[serde(rename = "user_settings")]
    UserSettings,
    #[serde(rename = "webinar")]
    Webinar,
    #[serde(rename = "zoom_rooms")]
    ZoomRooms,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for CategoryType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CategoryType::Account => "account",
            CategoryType::All => "all",
            CategoryType::Billing => "billing",
            CategoryType::Im => "im",
            CategoryType::PhoneContacts => "phone_contacts",
            CategoryType::Recording => "recording",
            CategoryType::Role => "role",
            CategoryType::SubAccount => "sub_account",
            CategoryType::User => "user",
            CategoryType::UserSettings => "user_settings",
            CategoryType::Webinar => "webinar",
            CategoryType::ZoomRooms => "zoom_rooms",
            CategoryType::Noop => "",
            CategoryType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for CategoryType {
    fn default() -> CategoryType {
        CategoryType::Noop
    }
}
impl CategoryType {
    pub fn is_noop(&self) -> bool {
        matches!(self, CategoryType::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct OperationLogs {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub action: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub category_type: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub operation_detail: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub operator: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub time: Option<chrono::DateTime<chrono::Utc>>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ReportOperationLogsResponse {
    /**
     * Array of operation log objects
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub operation_logs: Vec<OperationLogs>,
}

/// All of the following types:
///
/// - `PaginationToken4ImChat`
/// - `ReportOperationLogsResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ReportOperationLogsResponseAllOf {
    /**
     * Pagination object.
     */
    PaginationToken4ImChat(PaginationToken4ImChat),
    ReportOperationLogsResponse(ReportOperationLogsResponse),
}

impl ReportOperationLogsResponseAllOf {
    pub fn pagination_token_4_im_chat(&self) -> Option<&PaginationToken4ImChat> {
        if let ReportOperationLogsResponseAllOf::PaginationToken4ImChat(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn report_operation_logs_response(&self) -> Option<&ReportOperationLogsResponse> {
        if let ReportOperationLogsResponseAllOf::ReportOperationLogsResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CreateRoleRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub privileges: Vec<String>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CreateRoleResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub privileges: Vec<String>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_members: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddRoleMembersResponse {
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub add_at: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub ids: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct TspResponseDialInNumbers {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub code: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct TspResponse {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub dial_in_number_unrestricted: bool,
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub dial_in_numbers: Vec<TspResponseDialInNumbers>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub master_account_setting_extended: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub modify_credential_forbidden: bool,
    /**
     * Telephony bridge
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tsp_bridge: Option<TspBridge>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub tsp_enabled: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub tsp_provider: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct TspUpdateRequest {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub dial_in_number_unrestricted: Option<bool>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub enable: Option<bool>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub master_account_setting_extended: Option<bool>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub modify_credential_forbidden: Option<bool>,
    /**
     * Telephony bridge
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tsp_bridge: Option<TspBridge>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tsp_enabled: Option<bool>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub tsp_provider: String,
}

/**
 * The user's status:
 *   * `active` — An active user.
 *   * `inactive` — A deactivated user.
 *   * `pending` — A pending user.
 *   
 *   This value defaults to `active`.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum UsersStatus {
    #[serde(rename = "active")]
    Active,
    #[serde(rename = "inactive")]
    Inactive,
    #[serde(rename = "pending")]
    Pending,
    FallthroughString(String),
}

impl std::fmt::Display for UsersStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            UsersStatus::Active => "active",
            UsersStatus::Inactive => "inactive",
            UsersStatus::Pending => "pending",
            UsersStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for UsersStatus {
    fn default() -> UsersStatus {
        UsersStatus::Active
    }
}

/**
 * Use this parameter to display one of the following attributes in the API call's response:
 *   * `custom_attributes` — Return the user's custom attributes.
 *   * `host_key` — Return the user's [host key](https://support.zoom.us/hc/en-us/articles/205172555-Using-your-host-key).
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum UsersIncludeFields {
    #[serde(rename = "custom_attributes")]
    CustomAttributes,
    #[serde(rename = "host_key")]
    HostKey,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for UsersIncludeFields {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            UsersIncludeFields::CustomAttributes => "custom_attributes",
            UsersIncludeFields::HostKey => "host_key",
            UsersIncludeFields::Noop => "",
            UsersIncludeFields::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for UsersIncludeFields {
    fn default() -> UsersIncludeFields {
        UsersIncludeFields::Noop
    }
}
impl UsersIncludeFields {
    pub fn is_noop(&self) -> bool {
        matches!(self, UsersIncludeFields::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UsersResponseCustomAttributes {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub key: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub value: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UsersResponse {
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub created_at: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Information about the user's custom attributes.
     *  
     *  This field is \*\*only\*\* returned if users are assigned custom attributes and you provided the `custom_attributes` value for the `include_fields` query parameter in the API request.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub custom_attributes: Vec<UsersResponseCustomAttributes>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub dept: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub group_ids: Vec<String>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub host_key: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub im_group_ids: Vec<String>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_client_version: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub last_login_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub plan_united_type: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub pmi: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub role_id: String,
    /**
     * The user's status:
     *  \* `active` — An active user.
     *  \* `inactive` — A deactivated user.
     *  \* `pending` — A pending user.
     *  
     *  This value defaults to `active`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<UsersStatus>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub verified: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UsersResponseData {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_count: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_number: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
    /**
     * Information about the users.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub users: Vec<UsersResponse>,
}

/**
 * Specify how to create the new user: <br>`create` - User will get an email sent from Zoom. There is a confirmation link in this email. The user will then need to use the link to activate their Zoom account. The user can then set or change their password.<br>`autoCreate` - This action is provided for the enterprise customer who has a managed domain. This feature is disabled by default because of the security risk involved in creating a user who does not belong to your domain.<br>`custCreate` - Users created via this option do not have passwords and will not have the ability to log into the Zoom Web Portal or the Zoom Client. These users can still host and join meetings using the `start_url` and `join_url` respectively. To use this option, you must contact the ISV Platform Sales team at isv@zoom.us.<br>`ssoCreate` - This action is provided for the enabled “Pre-provisioning SSO User” option. A user created in this way has no password. If not a basic user, a personal vanity URL using the user name (no domain) of the provisioning email will be generated. If the user name or PMI is invalid or occupied, it will use a random number or random personal vanity URL.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum UserCreateRequestAction {
    #[serde(rename = "autoCreate")]
    AutoCreate,
    #[serde(rename = "create")]
    Create,
    #[serde(rename = "custCreate")]
    CustCreate,
    #[serde(rename = "ssoCreate")]
    SsoCreate,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for UserCreateRequestAction {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            UserCreateRequestAction::AutoCreate => "autoCreate",
            UserCreateRequestAction::Create => "create",
            UserCreateRequestAction::CustCreate => "custCreate",
            UserCreateRequestAction::SsoCreate => "ssoCreate",
            UserCreateRequestAction::Noop => "",
            UserCreateRequestAction::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for UserCreateRequestAction {
    fn default() -> UserCreateRequestAction {
        UserCreateRequestAction::Noop
    }
}
impl UserCreateRequestAction {
    pub fn is_noop(&self) -> bool {
        matches!(self, UserCreateRequestAction::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserInfo {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserCreateRequest {
    /**
     * Specify how to create the new user: <br>`create` - User will get an email sent from Zoom. There is a confirmation link in this email. The user will then need to use the link to activate their Zoom account. The user can then set or change their password.<br>`autoCreate` - This action is provided for the enterprise customer who has a managed domain. This feature is disabled by default because of the security risk involved in creating a user who does not belong to your domain.<br>`custCreate` - Users created via this option do not have passwords and will not have the ability to log into the Zoom Web Portal or the Zoom Client. These users can still host and join meetings using the `start_url` and `join_url` respectively. To use this option, you must contact the ISV Platform Sales team at isv@zoom.us.<br>`ssoCreate` - This action is provided for the enabled “Pre-provisioning SSO User” option. A user created in this way has no password. If not a basic user, a personal vanity URL using the user name (no domain) of the provisioning email will be generated. If the user name or PMI is invalid or occupied, it will use a random number or random personal vanity URL.
     */
    #[serde(default, skip_serializing_if = "UserCreateRequestAction::is_noop")]
    pub action: UserCreateRequestAction,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub user_info: Option<UserInfo>,
}

/**
 * The user's login method:
 *   
 *   `0` — Facebook OAuth</br>`1` — Google OAuth</br>`24` — Apple OAuth</br>`27` — Microsoft OAuth</br>`97` — Mobile device</br>`98` — RingCentral OAuth</br>`99` — API user</br>`100` — Zoom Work email</br>`101` — Single Sign-On (SSO)
 *   
 *   The following login methods are only available in China:
 *   
 *   `11` — Phone number</br>`21` — WeChat</br>`23` — Alipay
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum LoginType {
    #[serde(rename = "0")]
    Zero,
    #[serde(rename = "1")]
    One,
    #[serde(rename = "100")]
    OneHundred,
    #[serde(rename = "101")]
    OneHundredAndOne,
    #[serde(rename = "11")]
    Eleven,
    #[serde(rename = "21")]
    TwentyOne,
    #[serde(rename = "23")]
    TwentyThree,
    #[serde(rename = "24")]
    TwentyFour,
    #[serde(rename = "27")]
    TwentySeven,
    #[serde(rename = "97")]
    NinetySeven,
    #[serde(rename = "98")]
    NinetyEight,
    #[serde(rename = "99")]
    NinetyNine,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for LoginType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            LoginType::Zero => "0",
            LoginType::One => "1",
            LoginType::OneHundred => "100",
            LoginType::OneHundredAndOne => "101",
            LoginType::Eleven => "11",
            LoginType::TwentyOne => "21",
            LoginType::TwentyThree => "23",
            LoginType::TwentyFour => "24",
            LoginType::TwentySeven => "27",
            LoginType::NinetySeven => "97",
            LoginType::NinetyEight => "98",
            LoginType::NinetyNine => "99",
            LoginType::Noop => "",
            LoginType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for LoginType {
    fn default() -> LoginType {
        LoginType::Noop
    }
}
impl LoginType {
    pub fn is_noop(&self) -> bool {
        matches!(self, LoginType::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserResponsePhoneNumbers {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub code: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
     * The label to add to the user's phone number. You can only add one label to the user's phone number:
     *  
     *  \* `Mobile`
     *  \* `Office`
     *  \* `Home`
     *  \* `Fax`
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub label: Option<Label>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub verified: bool,
}

/// Custom attribute(s) that have been assigned to the user.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserResponseCustomAttributes {
    /**
     * Custom attribute(s) that have been assigned to the user.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub key: String,
    /**
     * Custom attribute(s) that have been assigned to the user.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Custom attribute(s) that have been assigned to the user.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub value: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub account_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub cms_user_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub company: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub created_at: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Custom attribute(s) that have been assigned to the user.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub custom_attributes: Option<UserResponseCustomAttributes>,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub group_ids: Vec<String>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub host_key: String,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub im_group_ids: Vec<String>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub jid: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub job_title: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub language: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub login_type: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub manager: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub personal_meeting_url: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_country: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number: String,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub phone_numbers: Option<UserResponsePhoneNumbers>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub pic_url: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub plan_united_type: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub role_id: String,
    /**
     * The user's status:
     *  \* `active` — An active user.
     *  \* `inactive` — A deactivated user.
     *  \* `pending` — A pending user.
     *  
     *  This value defaults to `active`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<UsersStatus>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub use_pmi: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub vanity_url: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub verified: i64,
}

/// All of the following types:
///
/// - `Groups`
/// - `User`
/// - `UserResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum UserResponseAllOf {
    Groups(Groups),
    /**
     * The user object represents a specific user on Zoom.
     */
    User(User),
    UserResponse(UserResponse),
}

impl UserResponseAllOf {
    pub fn groups(&self) -> Option<&Groups> {
        if let UserResponseAllOf::Groups(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn user(&self) -> Option<&User> {
        if let UserResponseAllOf::User(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn user_response(&self) -> Option<&UserResponse> {
        if let UserResponseAllOf::UserResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/**
 * Delete action options:<br>`disassociate` - Disassociate a user.<br>`delete`-  Permanently delete a user.<br>Note: To delete pending user in the account, use `disassociate`
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum UserDeleteAction {
    #[serde(rename = "delete")]
    Delete,
    #[serde(rename = "disassociate")]
    Disassociate,
    FallthroughString(String),
}

impl std::fmt::Display for UserDeleteAction {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            UserDeleteAction::Delete => "delete",
            UserDeleteAction::Disassociate => "disassociate",
            UserDeleteAction::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for UserDeleteAction {
    fn default() -> UserDeleteAction {
        UserDeleteAction::Disassociate
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserZakResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub token: String,
}

/// All of the following types:
///
/// - `Domains`
/// - `UserSettings`
/// - `MeetingSecuritySettings`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum UserSettingsResponseOneOf {
    Domains(Domains),
    UserSettings(UserSettings),
    MeetingSecuritySettings(MeetingSecuritySettings),
}

impl UserSettingsResponseOneOf {
    pub fn domains(&self) -> Option<&Domains> {
        if let UserSettingsResponseOneOf::Domains(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn meeting_security_settings(&self) -> Option<&MeetingSecuritySettings> {
        if let UserSettingsResponseOneOf::MeetingSecuritySettings(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn user_settings(&self) -> Option<&UserSettings> {
        if let UserSettingsResponseOneOf::UserSettings(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `Domains`
/// - `UserSettingsUpdate`
/// - `MeetingSecuritySettings`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum UserSettingsUpdateRequestOneOf {
    Domains(Domains),
    UserSettingsUpdate(UserSettingsUpdate),
    MeetingSecuritySettings(MeetingSecuritySettings),
}

impl UserSettingsUpdateRequestOneOf {
    pub fn domains(&self) -> Option<&Domains> {
        if let UserSettingsUpdateRequestOneOf::Domains(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn meeting_security_settings(&self) -> Option<&MeetingSecuritySettings> {
        if let UserSettingsUpdateRequestOneOf::MeetingSecuritySettings(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn user_settings_update(&self) -> Option<&UserSettingsUpdate> {
        if let UserSettingsUpdateRequestOneOf::UserSettingsUpdate(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum UserSettingsUpdateOption {
    #[serde(rename = "meeting_authentication")]
    MeetingAuthentication,
    #[serde(rename = "meeting_secuirty")]
    MeetingSecuirty,
    #[serde(rename = "recording_authentication")]
    RecordingAuthentication,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for UserSettingsUpdateOption {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            UserSettingsUpdateOption::MeetingAuthentication => "meeting_authentication",
            UserSettingsUpdateOption::MeetingSecuirty => "meeting_secuirty",
            UserSettingsUpdateOption::RecordingAuthentication => "recording_authentication",
            UserSettingsUpdateOption::Noop => "",
            UserSettingsUpdateOption::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for UserSettingsUpdateOption {
    fn default() -> UserSettingsUpdateOption {
        UserSettingsUpdateOption::Noop
    }
}
impl UserSettingsUpdateOption {
    pub fn is_noop(&self) -> bool {
        matches!(self, UserSettingsUpdateOption::Noop)
    }
}

/**
 * The action types:<br>`activate` - Activate a deactivated user.<br>`deactivate` - Deactivate a user.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum UserStatusRequestAction {
    #[serde(rename = "activate")]
    Activate,
    #[serde(rename = "deactivate")]
    Deactivate,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for UserStatusRequestAction {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            UserStatusRequestAction::Activate => "activate",
            UserStatusRequestAction::Deactivate => "deactivate",
            UserStatusRequestAction::Noop => "",
            UserStatusRequestAction::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for UserStatusRequestAction {
    fn default() -> UserStatusRequestAction {
        UserStatusRequestAction::Noop
    }
}
impl UserStatusRequestAction {
    pub fn is_noop(&self) -> bool {
        matches!(self, UserStatusRequestAction::Noop)
    }
}

/// The action.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserStatusRequest {
    /**
     * The action types:<br>`activate` - Activate a deactivated user.<br>`deactivate` - Deactivate a user.
     */
    #[serde(default, skip_serializing_if = "UserStatusRequestAction::is_noop")]
    pub action: UserStatusRequestAction,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserPasswordRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PacAccounts {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub conference_id: i64,
    /**
     * List of dedicated dial-in numbers.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub dedicated_dial_in_number: Vec<DedicatedDialInNumber>,
    /**
     * List of dedicated dial-in numbers.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub global_dial_in_numbers: Vec<DedicatedDialInNumber>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub listen_only_password: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub participant_password: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserPaCsResponse {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub pac_accounts: Vec<PacAccounts>,
}

/// List of TSP accounts.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct TspAccounts {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub conference_code: String,
    /**
     * List of TSP accounts.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub dial_in_numbers: Vec<DialInNumbers>,
    /**
     * List of TSP accounts.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub leader_pin: String,
    /**
     * List of TSP accounts.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tsp_bridge: Option<TspBridge>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserTsPsResponse {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub tsp_accounts: Vec<TspAccounts>,
}

/// TSP account of the user.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct TspAccount {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub conference_code: String,
    /**
     * TSP account of the user.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub dial_in_numbers: Vec<DialInNumbers>,
    /**
     * TSP account of the user.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub leader_pin: String,
    /**
     * TSP account of the user.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tsp_bridge: Option<TspBridge>,
}

/// TSP account.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct TspAccountData {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub conference_code: String,
    /**
     * TSP account.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub dial_in_numbers: Vec<DialInNumbers>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub leader_pin: String,
    /**
     * TSP account.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tsp_bridge: Option<TspBridge>,
}

/**
 * User token types:<br>`token` - Used for starting meetings with the client SDK. This token expires in 14 days and a new token will be returned after the expiry.<br>`zak` - Used for generating the start meeting URL. The token expiration time is two hours. For API users, the expiration time is 90 days.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum UserTokenType {
    #[serde(rename = "token")]
    Token,
    #[serde(rename = "zak")]
    Zak,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for UserTokenType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            UserTokenType::Token => "token",
            UserTokenType::Zak => "zak",
            UserTokenType::Noop => "",
            UserTokenType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for UserTokenType {
    fn default() -> UserTokenType {
        UserTokenType::Noop
    }
}
impl UserTokenType {
    pub fn is_noop(&self) -> bool {
        matches!(self, UserTokenType::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserEmailResponse {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub existed_email: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UserVanityNameResponse {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub existed: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct WebinarCreateResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub host_email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub host_id: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub registrants_confirmation_email: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub template_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub uuid: String,
}

/// All of the following types:
///
/// - `WebinarInfo`
/// - `WebinarCreateResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum WebinarCreateResponseAllOf {
    /**
     * Webinar object.
     */
    WebinarInfo(WebinarInfo),
    WebinarCreateResponse(WebinarCreateResponse),
}

impl WebinarCreateResponseAllOf {
    pub fn webinar_create_response(&self) -> Option<&WebinarCreateResponse> {
        if let WebinarCreateResponseAllOf::WebinarCreateResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn webinar_info(&self) -> Option<&WebinarInfo> {
        if let WebinarCreateResponseAllOf::WebinarInfo(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct WebinarResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub host_email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub host_id: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub id: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub uuid: String,
}

/// All of the following types:
///
/// - `WebinarInfo`
/// - `WebinarResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum WebinarResponseAllOf {
    /**
     * Webinar object.
     */
    WebinarInfo(WebinarInfo),
    WebinarResponse(WebinarResponse),
}

impl WebinarResponseAllOf {
    pub fn webinar_info(&self) -> Option<&WebinarInfo> {
        if let WebinarResponseAllOf::WebinarInfo(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn webinar_response(&self) -> Option<&WebinarResponse> {
        if let WebinarResponseAllOf::WebinarResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListWebinarParticipantsResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_count: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * Array of meeting participant objects.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub participants: Vec<Participants>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum WebinarStatusRequestAction {
    #[serde(rename = "end")]
    End,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for WebinarStatusRequestAction {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            WebinarStatusRequestAction::End => "end",
            WebinarStatusRequestAction::Noop => "",
            WebinarStatusRequestAction::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for WebinarStatusRequestAction {
    fn default() -> WebinarStatusRequestAction {
        WebinarStatusRequestAction::Noop
    }
}
impl WebinarStatusRequestAction {
    pub fn is_noop(&self) -> bool {
        matches!(self, WebinarStatusRequestAction::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct WebinarStatusRequest {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub action: Option<WebinarStatusRequestAction>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct WebinarPanelistCreateResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub updated_at: Option<chrono::DateTime<chrono::Utc>>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct WebinarRegistrantCreateResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub join_url: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub registrant_id: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddBatchWebinarRegistrantsRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddBatchRegistrantsRequest {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub auto_approve: Option<bool>,
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub registrants: Vec<AddBatchWebinarRegistrantsRequest>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddBatchWebinarRegistrantsResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub join_url: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub registrant_id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddBatchRegistrantsResponse {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub registrants: Vec<AddBatchWebinarRegistrantsResponse>,
}

/// All of the following types:
///
/// - `WebinarRegistrantQuestions`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum WebinarRegistrantsQuestionsGetResponseAllOf {
    /**
     * Webinar Registrant Questions
     */
    WebinarRegistrantQuestions(WebinarRegistrantQuestions),
}

impl WebinarRegistrantsQuestionsGetResponseAllOf {
    pub fn webinar_registrant_questions(&self) -> Option<&WebinarRegistrantQuestions> {
        let WebinarRegistrantsQuestionsGetResponseAllOf::WebinarRegistrantQuestions(ref_) = self;
        Some(ref_)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SendchatbotRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub account_id: String,
    #[serde()]
    pub content: Domains,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub is_markdown_support: Option<bool>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub robot_jid: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub to_jid: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_jid: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub visible_to_user: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct EditChatbotMessageRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub account_id: String,
    #[serde()]
    pub content: Domains,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub is_markdown_support: Option<bool>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub robot_jid: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_jid: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct EditChatbotMessageResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub message_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub robot_jid: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub sent_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub to_jid: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_jid: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DeleteChatbotMessageRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub account_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub robot_jid: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_jid: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DeleteChatbotMessageResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub message_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub robot_jid: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub sent_time: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub to_jid: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_jid: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetLiveStreamDetailsResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub page_url: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub stream_key: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub stream_url: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ClientFeedbackDetails {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub meeting_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub participant_name: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub time: Option<chrono::DateTime<chrono::Utc>>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DashboardClientFeedbackDetailResponse {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub client_feedback_details: Vec<ClientFeedbackDetails>,
}

/// All of the following types:
///
/// - `PaginationToken4ImChat`
/// - `DashboardImResponse`
/// - `DashboardClientFeedbackDetailResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DashboardClientFeedbackDetailResponseAllOf {
    /**
     * Pagination object.
     */
    PaginationToken4ImChat(PaginationToken4ImChat),
    DashboardImResponse(DashboardImResponse),
    DashboardClientFeedbackDetailResponse(DashboardClientFeedbackDetailResponse),
}

impl DashboardClientFeedbackDetailResponseAllOf {
    pub fn dashboard_client_feedback_detail_response(
        &self,
    ) -> Option<&DashboardClientFeedbackDetailResponse> {
        if let DashboardClientFeedbackDetailResponseAllOf::DashboardClientFeedbackDetailResponse(
            ref_,
        ) = self
        {
            return Some(ref_);
        }
        None
    }

    pub fn dashboard_im_response(&self) -> Option<&DashboardImResponse> {
        if let DashboardClientFeedbackDetailResponseAllOf::DashboardImResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn pagination_token_4_im_chat(&self) -> Option<&PaginationToken4ImChat> {
        if let DashboardClientFeedbackDetailResponseAllOf::PaginationToken4ImChat(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListimmessagesResponseMessages {
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub date_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub message: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub sender: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub timstamp: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListimmessagesResponse {
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub date: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Array of im messages.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub messages: Vec<ListimmessagesResponseMessages>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SendimmessagesRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub message: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CallingPlan {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PhoneUserResponseNumbers {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
}

/// A list of the user's policies. Policies are exceptions to the user's calling plan restrictions.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Policy {
    /**
     * A list of the user's policies. Policies are exceptions to the user's calling plan restrictions.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub international_calling: bool,
}

/// Zoom User Profile
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PhoneUserResponse {
    /**
     * Zoom User Profile
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub calling_plan: Vec<CallingPlan>,
    /**
     * Zoom User Profile
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * Zoom User Profile
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * Zoom User Profile
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Zoom User Profile
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub phone_numbers: Vec<PhoneUserResponseNumbers>,
    /**
     * Zoom User Profile
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_user_id: String,
    /**
     * Zoom User Profile
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub policy: Option<Policy>,
    /**
     * Zoom User Profile
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub site_admin: bool,
    /**
     * Zoom User Profile
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub site_id: String,
    /**
     * Zoom User Profile
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<UserStatusRequestAction>,
}

/// A list of the user's policies.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateUserProfileRequestPolicy {
    /**
     * A list of the user's policies.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub international_calling: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateUserProfileRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub extension_number: String,
    /**
     * A list of the user's policies.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub policy: Option<UpdateUserProfileRequestPolicy>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub site_id: String,
}

///
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PhoneSettingResponseCountry {
    /**
     *
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub code: String,
    /**
     *
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

///
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct MultipleSites {
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enabled: bool,
    /**
     *
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub site_code: String,
}

/// BYOC setting
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Byoc {
    /**
     * BYOC setting
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ShowDeviceIpCallLog {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
}

/// Zoom Account Setting
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PhoneSettingResponse {
    /**
     * Zoom Account Setting
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub byoc: Option<Byoc>,
    /**
     * Zoom Account Setting
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub country: Option<PhoneSettingResponseCountry>,
    /**
     * Zoom Account Setting
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub multiple_sites: Option<MultipleSites>,
    /**
     * Zoom Account Setting
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub show_device_ip_for_call_log: Option<ShowDeviceIpCallLog>,
}

/// Only [master account owners](https://marketplace.zoom.us/docs/api-reference/master-account-apis) can use this MA API to enable BYOC(Bring your own carrier) option for a sub account.<br>
///
/// **Scope**:<br>
/// * `phone:master` <br>
///
/// **Prerequisites**: <br>
/// * Business or enterprise Account.<br>
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdatePhoneSettingsRequestByoc {
    /**
     * Only [master account owners](https://marketplace.zoom.us/docs/api-reference/master-account-apis) can use this MA API to enable BYOC(Bring your own carrier) option for a sub account.<br>
     *  
     *  \*\*Scope\*\*:<br>
     *  \* `phone:master` <br>
     *  
     *  \*\*Prerequisites\*\*: <br>
     *  \* Business or enterprise Account.<br>
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdatePhoneSettingsRequest {
    /**
     * Only [master account owners](https://marketplace.zoom.us/docs/api-reference/master-account-apis) can use this MA API to enable BYOC(Bring your own carrier) option for a sub account.<br>
     *  
     *  \*\*Scope\*\*:<br>
     *  \* `phone:master` <br>
     *  
     *  \*\*Prerequisites\*\*: <br>
     *  \* Business or enterprise Account.<br>
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub byoc: Option<UpdatePhoneSettingsRequestByoc>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub show_device_ip_for_call_log: Option<ShowDeviceIpCallLog>,
}

/// Outbound Caller Info
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct OutboundCaller {
    /**
     * Outbound Caller Info
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct KeysPositions {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub primary_number: String,
}

/// Contains information on phones or devices provisioned for the user.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct DeskPhone {
    /**
     * Contains information on phones or devices provisioned for the user.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub keys_positions: Option<KeysPositions>,
}

/// [Shared voicemail access](https://support.zoom.us/hc/en-us/articles/360033863991-Sharing-and-controlling-access-to-a-voicemail-inbox) assigned to user.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct VoiceMail {
    /**
     * [Shared voicemail access](https://support.zoom.us/hc/en-us/articles/360033863991-Sharing-and-controlling-access-to-a-voicemail-inbox) assigned to user.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub access_user_id: String,
    /**
     * [Shared voicemail access](https://support.zoom.us/hc/en-us/articles/360033863991-Sharing-and-controlling-access-to-a-voicemail-inbox) assigned to user.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub delete: bool,
    /**
     * [Shared voicemail access](https://support.zoom.us/hc/en-us/articles/360033863991-Sharing-and-controlling-access-to-a-voicemail-inbox) assigned to user.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub download: bool,
    /**
     * [Shared voicemail access](https://support.zoom.us/hc/en-us/articles/360033863991-Sharing-and-controlling-access-to-a-voicemail-inbox) assigned to user.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub shared_id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct OutboundCallerIds {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub is_default: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
}

/// The site's country.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PhoneUserSettingsResponseCountry {
    /**
     * The site's country.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub code: String,
    /**
     * The site's country.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country_code: String,
    /**
     * The site's country.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

/// Phone User Setting
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PhoneUserSettingsResponse {
    /**
     * Phone User Setting
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub area_code: String,
    /**
     * Phone User Setting
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub company_number: String,
    /**
     * Phone User Setting
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub country: Option<PhoneUserSettingsResponseCountry>,
    /**
     * Phone User Setting
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub desk_phone: Option<DeskPhone>,
    /**
     * Phone User Setting
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * Phone User Setting
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub outbound_caller: Option<OutboundCaller>,
    /**
     * Phone User Setting
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub outbound_caller_ids: Option<OutboundCallerIds>,
    /**
     * Phone User Setting
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub voice_mail: Option<VoiceMail>,
}

/**
 * Template type.<br>
 *   The value of this field can be one of the following:<br>
 *   * `user`<br>
 *   * `group`<br>
 *   * `autReceptionist`<br>
 *   * `commonArea`<br>
 *   * `zr`<br>
 *   * `interop`<br>
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ListSettingTemplatesResponseType {
    #[serde(rename = "autReceptionist")]
    AutReceptionist,
    #[serde(rename = "commonArea")]
    CommonArea,
    #[serde(rename = "group")]
    Group,
    #[serde(rename = "interop")]
    Interop,
    #[serde(rename = "user")]
    User,
    #[serde(rename = "zr")]
    Zr,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for ListSettingTemplatesResponseType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ListSettingTemplatesResponseType::AutReceptionist => "autReceptionist",
            ListSettingTemplatesResponseType::CommonArea => "commonArea",
            ListSettingTemplatesResponseType::Group => "group",
            ListSettingTemplatesResponseType::Interop => "interop",
            ListSettingTemplatesResponseType::User => "user",
            ListSettingTemplatesResponseType::Zr => "zr",
            ListSettingTemplatesResponseType::Noop => "",
            ListSettingTemplatesResponseType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for ListSettingTemplatesResponseType {
    fn default() -> ListSettingTemplatesResponseType {
        ListSettingTemplatesResponseType::Noop
    }
}
impl ListSettingTemplatesResponseType {
    pub fn is_noop(&self) -> bool {
        matches!(self, ListSettingTemplatesResponseType::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Templates {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Template type.<br>
     *  The value of this field can be one of the following:<br>
     *  \* `user`<br>
     *  \* `group`<br>
     *  \* `autReceptionist`<br>
     *  \* `commonArea`<br>
     *  \* `zr`<br>
     *  \* `interop`<br>
     */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<ListSettingTemplatesResponseType>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListSettingTemplatesResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub templates: Vec<Templates>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

/**
 * The type of template. Values include `user`, `group`, `autoReceptionist`, `commonarea`, or `interop`.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AddSettingTemplateRequestType {
    #[serde(rename = "autoReceptionist")]
    AutoReceptionist,
    #[serde(rename = "commonarea")]
    Commonarea,
    #[serde(rename = "group")]
    Group,
    #[serde(rename = "interop")]
    Interop,
    #[serde(rename = "user")]
    User,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for AddSettingTemplateRequestType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AddSettingTemplateRequestType::AutoReceptionist => "autoReceptionist",
            AddSettingTemplateRequestType::Commonarea => "commonarea",
            AddSettingTemplateRequestType::Group => "group",
            AddSettingTemplateRequestType::Interop => "interop",
            AddSettingTemplateRequestType::User => "user",
            AddSettingTemplateRequestType::Noop => "",
            AddSettingTemplateRequestType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for AddSettingTemplateRequestType {
    fn default() -> AddSettingTemplateRequestType {
        AddSettingTemplateRequestType::Noop
    }
}
impl AddSettingTemplateRequestType {
    pub fn is_noop(&self) -> bool {
        matches!(self, AddSettingTemplateRequestType::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddSettingTemplateRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub site_id: String,
    /**
     * The type of template. Values include `user`, `group`, `autoReceptionist`, `commonarea`, or `interop`.
     */
    #[serde(
        default,
        skip_serializing_if = "AddSettingTemplateRequestType::is_noop",
        rename = "type"
    )]
    pub type_: AddSettingTemplateRequestType,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddSettingTemplateResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct BatchAddLocationsRequestEmergencyAddress {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address_line_1: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address_line_2: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub city: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub state_code: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub vat_number: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub zip: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct NetworkSwitches {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub mac_address: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub port: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub port_prefix: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub port_range_from: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub port_range_to: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Locations {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bssid: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub elin: String,
    #[serde()]
    pub emergency_address: BatchAddLocationsRequestEmergencyAddress,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub identifier: String,
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub network_switches: Vec<NetworkSwitches>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub parent_identifier: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub private_ip: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub public_ip: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub sip_group_name: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct BatchAddLocationsRequest {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub locations: Vec<Locations>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub site_id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct BatchAddLocationsResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location_id: String,
}

/// The emergency service location's ELIN (Emergency Location Identification Number).
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Elin {
    /**
     * The emergency service location's ELIN (Emergency Location Identification Number).
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number: String,
    /**
     * The emergency service location's ELIN (Emergency Location Identification Number).
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number_id: String,
}

/// The emergency service location's SIP group information.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListLocationsResponseSipGroup {
    /**
     * The emergency service location's SIP group information.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * The emergency service location's SIP group information.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
}

/// The emergency service location's site information.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListLocationsResponseSite {
    /**
     * The emergency service location's site information.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * The emergency service location's site information.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListLocationsResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bssid: String,
    /**
     * The emergency service location's ELIN (Emergency Location Identification Number).
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub elin: Option<Elin>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub identifier: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub network_switches: Vec<NetworkSwitches>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub parent_location_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub private_ip: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub public_ip: String,
    /**
     * The emergency service location's SIP group information.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sip_group: Option<ListLocationsResponseSipGroup>,
    /**
     * The emergency service location's site information.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub site: Option<ListLocationsResponseSite>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListLocationsResponseData {
    /**
     * Information about emergency service locations.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub locations: Vec<ListLocationsResponse>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddLocationRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bssid: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub elin_phone_number_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub emergency_address_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub parent_location_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub private_ip: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub public_ip: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub sip_group_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub site_id: String,
}

/// The ELIN (Emergency Location Identification Number).
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetLocationResponseElin {
    /**
     * The ELIN (Emergency Location Identification Number).
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number: String,
    /**
     * The ELIN (Emergency Location Identification Number).
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number_id: String,
}

/// The emergency location's address information.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetLocationResponseEmergencyAddress {
    /**
     * The emergency location's address information.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address_line_1: String,
    /**
     * The emergency location's address information.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address_line_2: String,
    /**
     * The emergency location's address information.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub city: String,
    /**
     * The emergency location's address information.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
     * The emergency location's address information.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * The emergency location's address information.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub state_code: String,
    /**
     * The emergency location's address information.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub zip: String,
}

/// The emergency location's SIP group information.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetLocationResponseSipGroup {
    /**
     * The emergency location's SIP group information.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * The emergency location's SIP group information.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
}

/// The emergency location's site information.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetLocationResponseSite {
    /**
     * The emergency location's site information.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * The emergency location's site information.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetLocationResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bssid: String,
    /**
     * The ELIN (Emergency Location Identification Number).
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub elin: Option<GetLocationResponseElin>,
    /**
     * The emergency location's address information.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub emergency_address: Option<GetLocationResponseEmergencyAddress>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub network_switches: Vec<NetworkSwitches>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub parent_location_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub private_ip: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub public_ip: String,
    /**
     * The emergency location's SIP group information.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sip_group: Option<GetLocationResponseSipGroup>,
    /**
     * The emergency location's site information.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub site: Option<GetLocationResponseSite>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateLocationRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bssid: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub elin_phone_number_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub emergency_address_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub network_switches: Vec<NetworkSwitches>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub private_ip: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub public_ip: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub sip_group_id: String,
}

/// The SIP trunk group.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SipTrunk {
    /**
     * The SIP trunk group.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * The SIP trunk group.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SipGroups {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub send_sip_group_name: bool,
    /**
     * The SIP trunk group.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sip_trunk: Option<SipTrunk>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListSipGroupsResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * SIP group information.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub sip_groups: Vec<SipGroups>,
}

/**
 * The type of template being queried. Values: `user`, `group`, `auto receptionist` `common area`,`zr`, `interop`.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum GetSettingTemplateResponseType {
    #[serde(rename = "autoReceptionist")]
    AutoReceptionist,
    #[serde(rename = "commonArea")]
    CommonArea,
    #[serde(rename = "group")]
    Group,
    #[serde(rename = "interop")]
    Interop,
    #[serde(rename = "user")]
    User,
    #[serde(rename = "zr")]
    Zr,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for GetSettingTemplateResponseType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GetSettingTemplateResponseType::AutoReceptionist => "autoReceptionist",
            GetSettingTemplateResponseType::CommonArea => "commonArea",
            GetSettingTemplateResponseType::Group => "group",
            GetSettingTemplateResponseType::Interop => "interop",
            GetSettingTemplateResponseType::User => "user",
            GetSettingTemplateResponseType::Zr => "zr",
            GetSettingTemplateResponseType::Noop => "",
            GetSettingTemplateResponseType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for GetSettingTemplateResponseType {
    fn default() -> GetSettingTemplateResponseType {
        GetSettingTemplateResponseType::Noop
    }
}
impl GetSettingTemplateResponseType {
    pub fn is_noop(&self) -> bool {
        matches!(self, GetSettingTemplateResponseType::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetSettingTemplateResponseProfile {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub area_code: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AutoCallRecording {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub recording_calls: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub recording_start_prompt: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub recording_transcription: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AdHocCallRecording {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub recording_start_prompt: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub recording_transcription: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Sms {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub international_sms: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Voicemail {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_transcription: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetSettingTemplateResponsePolicy {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub ad_hoc_call_recording: Option<AdHocCallRecording>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub auto_call_recording: Option<AutoCallRecording>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sms: Option<Sms>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub voicemail: Option<Voicemail>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CustomHours {
    /**
     * Values:<br>
     *  hh:mm
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub from: Option<chrono::NaiveTime>,
    /**
     * Values:<br>
     *  hh:mm
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub to: Option<chrono::NaiveTime>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub weekday: i64,
}

/**
 * Values:<br>
 *   1-user,<br>
 *   2-callQueue, <br>
 *   3-autoReceptionist,<br>
 *   4-commonAreaPhone,<br>
 *   5-zoomRoom, <br>
 *   7-sharedLineGroup
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum GetSettingTemplateResponseUserSettingsCallHandlingBusinessHoursConnectOperatorType {
    #[serde(rename = "autoReceptionist")]
    AutoReceptionist,
    #[serde(rename = "callQueue")]
    CallQueue,
    #[serde(rename = "commonAreaPhone")]
    CommonAreaPhone,
    #[serde(rename = "sharedLineGroup")]
    SharedLineGroup,
    #[serde(rename = "user")]
    User,
    #[serde(rename = "zoomRoom")]
    ZoomRoom,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display
    for GetSettingTemplateResponseUserSettingsCallHandlingBusinessHoursConnectOperatorType
{
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
GetSettingTemplateResponseUserSettingsCallHandlingBusinessHoursConnectOperatorType::AutoReceptionist => "autoReceptionist",
GetSettingTemplateResponseUserSettingsCallHandlingBusinessHoursConnectOperatorType::CallQueue => "callQueue",
GetSettingTemplateResponseUserSettingsCallHandlingBusinessHoursConnectOperatorType::CommonAreaPhone => "commonAreaPhone",
GetSettingTemplateResponseUserSettingsCallHandlingBusinessHoursConnectOperatorType::SharedLineGroup => "sharedLineGroup",
GetSettingTemplateResponseUserSettingsCallHandlingBusinessHoursConnectOperatorType::User => "user",
GetSettingTemplateResponseUserSettingsCallHandlingBusinessHoursConnectOperatorType::ZoomRoom => "zoomRoom",
GetSettingTemplateResponseUserSettingsCallHandlingBusinessHoursConnectOperatorType::Noop => "",
GetSettingTemplateResponseUserSettingsCallHandlingBusinessHoursConnectOperatorType::FallthroughString(s) => s,
}
.fmt(f)
    }
}

impl Default
    for GetSettingTemplateResponseUserSettingsCallHandlingBusinessHoursConnectOperatorType
{
    fn default(
    ) -> GetSettingTemplateResponseUserSettingsCallHandlingBusinessHoursConnectOperatorType {
        GetSettingTemplateResponseUserSettingsCallHandlingBusinessHoursConnectOperatorType::Noop
    }
}
impl GetSettingTemplateResponseUserSettingsCallHandlingBusinessHoursConnectOperatorType {
    pub fn is_noop(&self) -> bool {
        matches!(self, GetSettingTemplateResponseUserSettingsCallHandlingBusinessHoursConnectOperatorType::Noop)
    }
}

/// Allow callers to press Zero to reach an operator or press One to leave a message, or allow neither of these options.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ConnectOperator {
    /**
     * Allow callers to press Zero to reach an operator or press One to leave a message, or allow neither of these options.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * Allow callers to press Zero to reach an operator or press One to leave a message, or allow neither of these options.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Allow callers to press Zero to reach an operator or press One to leave a message, or allow neither of these options.
     */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_:
        Option<GetSettingTemplateResponseUserSettingsCallHandlingBusinessHoursConnectOperatorType>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct BusinessHours {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub business_hour_action: i64,
    /**
     * Allow callers to press Zero to reach an operator or press One to leave a message, or allow neither of these options.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub connect_to_operator: Option<ConnectOperator>,
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub custom_hours: Vec<CustomHours>,
    /**
     * Audio types:<br>`1` - Toll-free Call-in & Call-out.<br>`2` - Toll <br>
     *  `3` - SIP Connected Audio
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub ring_type: Option<serde_json::Value>,
    /**
     * Audio types:<br>`1` - Toll-free Call-in & Call-out.<br>`2` - Toll <br>
     *  `3` - SIP Connected Audio
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub ringing_duration: Option<serde_json::Value>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CloseHours {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub close_hour_action: i64,
    /**
     * Allow callers to press Zero to reach an operator or press One to leave a message, or allow neither of these options.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub connect_to_operator: Option<ConnectOperator>,
    /**
     * Audio types:<br>`1` - Toll-free Call-in & Call-out.<br>`2` - Toll <br>
     *  `3` - SIP Connected Audio
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub max_wait_time: Option<serde_json::Value>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CallHandling {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub business_hours: Option<BusinessHours>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub close_hours: Option<CloseHours>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetSettingTemplateResponseUserSettingsDeskPhone {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub pin_code: String,
}

/**
 * The value of this field can be either `default` or `disable`.
 *   
 *   * `default`: This means that the hold music can be set using the [audio library](https://support.zoom.us/hc/en-us/articles/360028212652-Using-the-audio-library-to-customize-greetings-and-hold-music).
 *   
 *   * `disable`: This means that the hold music is disabled.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum HoldMusic {
    #[serde(rename = "default")]
    Default,
    #[serde(rename = "disable")]
    Disable,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for HoldMusic {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            HoldMusic::Default => "default",
            HoldMusic::Disable => "disable",
            HoldMusic::Noop => "",
            HoldMusic::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for HoldMusic {
    fn default() -> HoldMusic {
        HoldMusic::Noop
    }
}
impl HoldMusic {
    pub fn is_noop(&self) -> bool {
        matches!(self, HoldMusic::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetSettingTemplateResponseUserSettings {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub audio_prompt_language: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub block_calls_without_caller_id: bool,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub call_handling: Option<CallHandling>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub desk_phone: Option<GetSettingTemplateResponseUserSettingsDeskPhone>,
    /**
     * The value of this field can be either `default` or `disable`.
     *  
     *  \* `default`: This means that the hold music can be set using the [audio library](https://support.zoom.us/hc/en-us/articles/360028212652-Using-the-audio-library-to-customize-greetings-and-hold-music).
     *  
     *  \* `disable`: This means that the hold music is disabled.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub hold_music: Option<HoldMusic>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetSettingTemplateResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub policy: Option<GetSettingTemplateResponsePolicy>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub profile: Option<GetSettingTemplateResponseProfile>,
    /**
     * The type of template being queried. Values: `user`, `group`, `auto receptionist` `common area`,`zr`, `interop`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<GetSettingTemplateResponseType>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub user_settings: Option<GetSettingTemplateResponseUserSettings>,
}

/**
 * Values: inbound, outbound, both.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum RecordingCalls {
    #[serde(rename = "both")]
    Both,
    #[serde(rename = "inbound")]
    Inbound,
    #[serde(rename = "outbound")]
    Outbound,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for RecordingCalls {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            RecordingCalls::Both => "both",
            RecordingCalls::Inbound => "inbound",
            RecordingCalls::Outbound => "outbound",
            RecordingCalls::Noop => "",
            RecordingCalls::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for RecordingCalls {
    fn default() -> RecordingCalls {
        RecordingCalls::Noop
    }
}
impl RecordingCalls {
    pub fn is_noop(&self) -> bool {
        matches!(self, RecordingCalls::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateSettingTemplateRequestPolicyAutoCallRecording {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * Values: inbound, outbound, both.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recording_calls: Option<RecordingCalls>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub recording_start_prompt: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub recording_transcription: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateSettingTemplateRequestPolicy {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub ad_hoc_call_recording: Option<AdHocCallRecording>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub auto_call_recording: Option<UpdateSettingTemplateRequestPolicyAutoCallRecording>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sms: Option<Sms>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub voicemail: Option<Voicemail>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateSettingTemplateRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub policy: Option<UpdateSettingTemplateRequestPolicy>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub profile: Option<GetSettingTemplateResponseProfile>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub user_settings: Option<GetSettingTemplateResponseUserSettings>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum PhoneUserCallLogsType {
    #[serde(rename = "all")]
    All,
    #[serde(rename = "missed")]
    Missed,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for PhoneUserCallLogsType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PhoneUserCallLogsType::All => "all",
            PhoneUserCallLogsType::Missed => "missed",
            PhoneUserCallLogsType::Noop => "",
            PhoneUserCallLogsType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for PhoneUserCallLogsType {
    fn default() -> PhoneUserCallLogsType {
        PhoneUserCallLogsType::Noop
    }
}
impl PhoneUserCallLogsType {
    pub fn is_noop(&self) -> bool {
        matches!(self, PhoneUserCallLogsType::Noop)
    }
}

/**
 * Enables you to sort call logs by start or end time. Choose the sort time value. Values include `startTime` or `endTime`.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum TimeType {
    #[serde(rename = "endTime")]
    EndTime,
    #[serde(rename = "startTime")]
    StartTime,
    FallthroughString(String),
}

impl std::fmt::Display for TimeType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            TimeType::EndTime => "endTime",
            TimeType::StartTime => "startTime",
            TimeType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for TimeType {
    fn default() -> TimeType {
        TimeType::StartTime
    }
}

/// Indicates who the call was forwarded to.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ForwardedTo {
    /**
     * Indicates who the call was forwarded to.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub extension_number: String,
    /**
     * Indicates who the call was forwarded to.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location: String,
    /**
     * Indicates who the call was forwarded to.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Indicates who the call was forwarded to.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number_type: String,
    /**
     * Indicates who the call was forwarded to.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum PhoneUserCallLogsResponseForwardedByExtensionType {
    #[serde(rename = "autoReceptionist")]
    AutoReceptionist,
    #[serde(rename = "callQueue")]
    CallQueue,
    #[serde(rename = "commonAreaPhone")]
    CommonAreaPhone,
    #[serde(rename = "sharedLineGroup")]
    SharedLineGroup,
    #[serde(rename = "user")]
    User,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for PhoneUserCallLogsResponseForwardedByExtensionType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PhoneUserCallLogsResponseForwardedByExtensionType::AutoReceptionist => {
                "autoReceptionist"
            }
            PhoneUserCallLogsResponseForwardedByExtensionType::CallQueue => "callQueue",
            PhoneUserCallLogsResponseForwardedByExtensionType::CommonAreaPhone => "commonAreaPhone",
            PhoneUserCallLogsResponseForwardedByExtensionType::SharedLineGroup => "sharedLineGroup",
            PhoneUserCallLogsResponseForwardedByExtensionType::User => "user",
            PhoneUserCallLogsResponseForwardedByExtensionType::Noop => "",
            PhoneUserCallLogsResponseForwardedByExtensionType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for PhoneUserCallLogsResponseForwardedByExtensionType {
    fn default() -> PhoneUserCallLogsResponseForwardedByExtensionType {
        PhoneUserCallLogsResponseForwardedByExtensionType::Noop
    }
}
impl PhoneUserCallLogsResponseForwardedByExtensionType {
    pub fn is_noop(&self) -> bool {
        matches!(
            self,
            PhoneUserCallLogsResponseForwardedByExtensionType::Noop
        )
    }
}

/// Indicates where the call was forwarded from.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ForwardedBy {
    /**
     * Indicates where the call was forwarded from.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub extension_number: String,
    /**
     * Indicates where the call was forwarded from.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub extension_type: Option<PhoneUserCallLogsResponseForwardedByExtensionType>,
    /**
     * Indicates where the call was forwarded from.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location: String,
    /**
     * Indicates where the call was forwarded from.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Indicates where the call was forwarded from.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number_type: String,
    /**
     * Indicates where the call was forwarded from.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number: String,
}

/// Indicates who accepted the call.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AcceptedBy {
    /**
     * Indicates who accepted the call.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub extension_number: String,
    /**
     * Indicates who accepted the call.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location: String,
    /**
     * Indicates who accepted the call.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Indicates who accepted the call.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number_type: String,
    /**
     * Indicates who accepted the call.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct OutgoingBy {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub extension_number: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number_type: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CallLogs {
    /**
     * Indicates who accepted the call.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub accepted_by: Option<AcceptedBy>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub answer_start_time: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub call_end_time: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub call_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub callee_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub callee_number: String,
    /**
     * Audio types:<br>`1` - Toll-free Call-in & Call-out.<br>`2` - Toll <br>
     *  `3` - SIP Connected Audio
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub callee_number_type: Option<serde_json::Value>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub caller_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub caller_number: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub caller_number_type: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub charge: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub client_code: String,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub date_time: Option<chrono::NaiveDate>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub direction: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Indicates where the call was forwarded from.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub forwarded_by: Option<ForwardedBy>,
    /**
     * Indicates who the call was forwarded to.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub forwarded_to: Option<ForwardedTo>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_recording: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_voicemail: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub outgoing_by: Option<OutgoingBy>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub path: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub rate: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub recording_type: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub result: String,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub site: Option<Site>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_id: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub waiting_time: i64,
}

/// Phone User Call Logs
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PhoneUserCallLogsResponse {
    /**
     * Phone User Call Logs
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub call_logs: Vec<CallLogs>,
    /**
     * Phone User Call Logs
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub from: String,
    /**
     * Phone User Call Logs
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Phone User Call Logs
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_count: i64,
    /**
     * Phone User Call Logs
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * Phone User Call Logs
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub to: String,
    /**
     * Phone User Call Logs
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Recordings {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub call_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub call_log_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub callee_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub callee_number: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub callee_number_type: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub caller_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub caller_number: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub caller_number_type: String,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub date_time: Option<chrono::NaiveDate>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub direction: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub download_url: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub duration: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
}

/// Phone User Recordings
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PhoneUserRecordingsResponse {
    /**
     * Phone User Recordings
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub from: Option<chrono::NaiveDate>,
    /**
     * Phone User Recordings
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Phone User Recordings
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_count: i64,
    /**
     * Phone User Recordings
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * Phone User Recordings
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub recordings: Vec<Recordings>,
    /**
     * Phone User Recordings
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub to: Option<chrono::NaiveDate>,
    /**
     * Phone User Recordings
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

/**
 * Status of the voice mail
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum PhoneUserVoiceMailsStatus {
    #[serde(rename = "all")]
    All,
    #[serde(rename = "read")]
    Read,
    #[serde(rename = "unread")]
    Unread,
    FallthroughString(String),
}

impl std::fmt::Display for PhoneUserVoiceMailsStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PhoneUserVoiceMailsStatus::All => "all",
            PhoneUserVoiceMailsStatus::Read => "read",
            PhoneUserVoiceMailsStatus::Unread => "unread",
            PhoneUserVoiceMailsStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for PhoneUserVoiceMailsStatus {
    fn default() -> PhoneUserVoiceMailsStatus {
        PhoneUserVoiceMailsStatus::All
    }
}

/**
 * Status of the voice mail. Can be either 'read' or 'unread'
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum PhoneUserVoiceMailsResponseStatus {
    #[serde(rename = "read")]
    Read,
    #[serde(rename = "unread")]
    Unread,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for PhoneUserVoiceMailsResponseStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PhoneUserVoiceMailsResponseStatus::Read => "read",
            PhoneUserVoiceMailsResponseStatus::Unread => "unread",
            PhoneUserVoiceMailsResponseStatus::Noop => "",
            PhoneUserVoiceMailsResponseStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for PhoneUserVoiceMailsResponseStatus {
    fn default() -> PhoneUserVoiceMailsResponseStatus {
        PhoneUserVoiceMailsResponseStatus::Noop
    }
}
impl PhoneUserVoiceMailsResponseStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, PhoneUserVoiceMailsResponseStatus::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct VoiceMails {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub callee_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub callee_number: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub callee_number_type: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub caller_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub caller_number: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub caller_number_type: String,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub date_time: Option<chrono::NaiveDate>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub download_url: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub duration: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Status of the voice mail. Can be either 'read' or 'unread'
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<PhoneUserVoiceMailsResponseStatus>,
}

/// Phone User Voice Mails
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PhoneUserVoiceMailsResponse {
    /**
     * Phone User Voice Mails
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub from: Option<chrono::NaiveDate>,
    /**
     * Phone User Voice Mails
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Phone User Voice Mails
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_count: i64,
    /**
     * Phone User Voice Mails
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * Phone User Voice Mails
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub to: Option<chrono::NaiveDate>,
    /**
     * Phone User Voice Mails
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
    /**
     * Phone User Voice Mails
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub voice_mails: Vec<VoiceMails>,
}

/// Update the voicemail setting.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddUserSettingRequestVoiceMail {
    /**
     * Update the voicemail setting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub access_user_id: String,
    /**
     * Update the voicemail setting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub delete: bool,
    /**
     * Update the voicemail setting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub download: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddUserSettingRequest {
    /**
     * Update the voicemail setting.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub voice_mail: Option<AddUserSettingRequestVoiceMail>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddUserSettingResponseVoiceMail {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub access_user_id: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub delete: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub download: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub shared_id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddUserSettingResponse {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub voice_mail: Option<AddUserSettingResponseVoiceMail>,
}

/// Update the voicemail setting.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateUserSettingRequestVoiceMail {
    /**
     * Update the voicemail setting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub access_user_id: String,
    /**
     * Update the voicemail setting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub delete: bool,
    /**
     * Update the voicemail setting.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub download: bool,
    /**
     * Update the voicemail setting.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub shared_id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateUserSettingRequest {
    /**
     * Update the voicemail setting.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub voice_mail: Option<UpdateUserSettingRequestVoiceMail>,
}

/**
 * The type of call:
 *   *`voip` (Voice over IP)
 *   *`pstn` (Public Switched Telephone Network)
 *   *`tollfree`
 *   *`international`
 *   *`contactCenter`
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum CallType {
    #[serde(rename = "contactCenter")]
    ContactCenter,
    #[serde(rename = "international")]
    International,
    #[serde(rename = "pstn")]
    Pstn,
    #[serde(rename = "tollfree")]
    Tollfree,
    #[serde(rename = "voip")]
    Voip,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for CallType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CallType::ContactCenter => "contactCenter",
            CallType::International => "international",
            CallType::Pstn => "pstn",
            CallType::Tollfree => "tollfree",
            CallType::Voip => "voip",
            CallType::Noop => "",
            CallType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for CallType {
    fn default() -> CallType {
        CallType::Noop
    }
}
impl CallType {
    pub fn is_noop(&self) -> bool {
        matches!(self, CallType::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Owner {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<PhoneUserCallLogsResponseForwardedByExtensionType>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountCallLogsResponse {
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub answer_start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub call_id: String,
    /**
     * The type of call:
     *  \*`voip` (Voice over IP)
     *  \*`pstn` (Public Switched Telephone Network)
     *  \*`tollfree`
     *  \*`international`
     *  \*`contactCenter`
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub call_type: Option<CallType>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub callee_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub callee_number: String,
    /**
     * Audio types:<br>`1` - Toll-free Call-in & Call-out.<br>`2` - Toll <br>
     *  `3` - SIP Connected Audio
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub callee_number_type: Option<serde_json::Value>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub caller_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub caller_number: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub caller_number_type: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub charge: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub client_code: String,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub date_time: Option<chrono::NaiveDate>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub device_private_ip: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub device_public_ip: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub direction: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub owner: Option<Owner>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub path: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub rate: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub recording_id: String,
    /**
     * Audio types:<br>`1` - Toll-free Call-in & Call-out.<br>`2` - Toll <br>
     *  `3` - SIP Connected Audio
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recording_type: Option<serde_json::Value>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub result: String,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub site: Option<Site>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub waiting_time: String,
}

/// Account Call Logs
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountCallLogsResponseData {
    /**
     * Account Call Logs
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub call_logs: Vec<AccountCallLogsResponse>,
    /**
     * Account Call Logs
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub from: String,
    /**
     * Account Call Logs
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account Call Logs
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_count: i64,
    /**
     * Account Call Logs
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * Account Call Logs
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub to: String,
    /**
     * Account Call Logs
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddByocNumberResponse {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub phone_numbers: Vec<PhoneUserResponseNumbers>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CallingPlans {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AssignCallingPlanRequest {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub calling_plans: Vec<CallingPlans>,
}

/**
 * The owner type: `user` or `call queue`.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum GetPhoneRecordingsResponseOwnerType {
    #[serde(rename = "call queue")]
    CallQueue,
    #[serde(rename = "user")]
    User,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for GetPhoneRecordingsResponseOwnerType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GetPhoneRecordingsResponseOwnerType::CallQueue => "call queue",
            GetPhoneRecordingsResponseOwnerType::User => "user",
            GetPhoneRecordingsResponseOwnerType::Noop => "",
            GetPhoneRecordingsResponseOwnerType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for GetPhoneRecordingsResponseOwnerType {
    fn default() -> GetPhoneRecordingsResponseOwnerType {
        GetPhoneRecordingsResponseOwnerType::Noop
    }
}
impl GetPhoneRecordingsResponseOwnerType {
    pub fn is_noop(&self) -> bool {
        matches!(self, GetPhoneRecordingsResponseOwnerType::Noop)
    }
}

/// The owner of the recording.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetPhoneRecordingsResponseOwner {
    /**
     * The owner of the recording.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub extension_number: String,
    /**
     * The owner of the recording.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * The owner of the recording.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * The owner of the recording.
     */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<GetPhoneRecordingsResponseOwnerType>,
}

/**
 * The direction of the call. Values: `inbound` or `outbound`.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum Direction {
    #[serde(rename = "inbound")]
    Inbound,
    #[serde(rename = "outbound")]
    Outbound,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for Direction {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Direction::Inbound => "inbound",
            Direction::Outbound => "outbound",
            Direction::Noop => "",
            Direction::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for Direction {
    fn default() -> Direction {
        Direction::Noop
    }
}
impl Direction {
    pub fn is_noop(&self) -> bool {
        matches!(self, Direction::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetPhoneRecordingsResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub call_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub call_log_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub callee_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub callee_number: String,
    /**
     * Audio types:<br>`1` - Toll-free Call-in & Call-out.<br>`2` - Toll <br>
     *  `3` - SIP Connected Audio
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub callee_number_type: Option<serde_json::Value>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub caller_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub caller_number: String,
    /**
     * Audio types:<br>`1` - Toll-free Call-in & Call-out.<br>`2` - Toll <br>
     *  `3` - SIP Connected Audio
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub caller_number_type: Option<serde_json::Value>,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub date_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * The direction of the call. Values: `inbound` or `outbound`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub direction: Option<Direction>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub download_url: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub end_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * The owner of the recording.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub owner: Option<GetPhoneRecordingsResponseOwner>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub recording_type: String,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub site: Option<Site>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetPhoneRecordingsResponseData {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub recordings: Vec<GetPhoneRecordingsResponse>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ByocSipTrunk {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub carrier: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub carrier_account: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub region: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub sbc_label: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListByocsipTrunkResponse {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub byoc_sip_trunk: Vec<ByocSipTrunk>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SipTrunks {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub carrier_account: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PostPhoneSipTrunkRequest {
    /**
     * SIP trunk configurations.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub sip_trunks: Vec<SipTrunks>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdatePhoneSipTrunkRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub carrier_account: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountPlanBaseDeleteRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub action: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub comment: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub reason: i64,
}

/**
 * The action that needs to be taken for this sub account. Value must be set to "cancel".
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AccountPlanAddonCancelRequestAction {
    #[serde(rename = "cancel")]
    Cancel,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for AccountPlanAddonCancelRequestAction {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AccountPlanAddonCancelRequestAction::Cancel => "cancel",
            AccountPlanAddonCancelRequestAction::Noop => "",
            AccountPlanAddonCancelRequestAction::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for AccountPlanAddonCancelRequestAction {
    fn default() -> AccountPlanAddonCancelRequestAction {
        AccountPlanAddonCancelRequestAction::Noop
    }
}
impl AccountPlanAddonCancelRequestAction {
    pub fn is_noop(&self) -> bool {
        matches!(self, AccountPlanAddonCancelRequestAction::Noop)
    }
}

/// Account plan object
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AccountPlanAddonCancelRequest {
    /**
     * Account plan object
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub action: Option<AccountPlanAddonCancelRequestAction>,
    /**
     * Account plan object
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub comment: String,
    /**
     * Account plan object
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub reason: i64,
    /**
     * Account plan object
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ActivityLogs {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub client_type: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub ip_address: String,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub version: String,
}

/// Report object
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ReportSignInOutActivitiesResponse {
    /**
     * Report object
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub activity_logs: Vec<ActivityLogs>,
    /**
     * Report object
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub from: String,
    /**
     * Report object
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Report object
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub page_size: String,
    /**
     * Report object
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub to: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetGroupSettingsResponseScheduleMeeting {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub audio_type: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub embed_password_in_join_link: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub force_pmi_jbh_password: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub host_video: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub join_before_host: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub mute_upon_entry: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub participant_video: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub personal_meeting: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub pstn_password_protected: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_instant_meetings: bool,
    /**
     * Require a password for a meeting held using Personal Meeting ID (PMI) This setting is always enabled for free accounts and Pro accounts with a single host and cannot be modified for these accounts.
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub require_password_for_pmi_meetings: Option<RequirePasswordPmiMeetings>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_scheduled_meetings: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_scheduling_new_meetings: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub upcoming_meeting_reminder: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub use_pmi_for_instant_meetings: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub use_pmi_for_schedule_meetings: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetGroupSettingsResponseInMeeting {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub alert_guest_join: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_show_zoom_windows: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub annotation: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub attendee_on_hold: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub auto_answer: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub auto_saving_chat: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub breakout_room: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub chat: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub closed_caption: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub co_host: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub custom_data_center_regions: bool,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub data_center_regions: Vec<String>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub e_2e_encryption: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub entry_exit_chime: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub far_end_camera_control: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub feedback: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub file_transfer: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub group_hd: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub join_from_desktop: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub join_from_mobile: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub non_verbal_feedback: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub only_host_view_device_list: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub original_audio: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub polling: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub post_meeting_feedback: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub private_chat: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub record_play_own_voice: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub remote_control: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub remote_support: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub request_permission_to_unmute: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub screen_sharing: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub sending_default_email_invites: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_a_join_from_your_browser_link: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_browser_join_link: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_device_list: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_meeting_control_toolbar: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub stereo_audio: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub use_html_format_email: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub virtual_background: bool,
    /**
     * Settings to manage virtual background.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub virtual_background_settings: Option<VirtualBackgroundSettings>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub waiting_room: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub whiteboard: bool,
    /**
     * Indicates who can share their screen or content during meetings. The value can be one of the following: <br>
     *  `host`: Only host can share the screen.<br>
     *  `all`: Both hosts and attendees can share their screen during meetings. For Webinar, the hosts and panelists can start screen sharing, but not the attendees.
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub who_can_share_screen: Option<WhoCanShareScreen>,
    /**
     * Indicates who can share their screen or content during meetings. The value can be one of the following: <br>
     *  `host`: Only host can share the screen.<br>
     *  `all`: Both hosts and attendees can share their screen during meetings. For Webinar, the hosts and panelists can start screen sharing, but not the attendees.
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub who_can_share_screen_when_someone_is_sharing: Option<WhoCanShareScreen>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetGroupSettingsResponseEmailNotification {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub alternative_host_reminder: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cancel_meeting_reminder: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cloud_recording_available_reminder: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub jbh_reminder: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub schedule_for_host_reminder: bool,
}

/// [Archiving solution](https://support.zoom.us/hc/en-us/articles/360050431572-Archiving-Meeting-and-Webinar-data) settings. This setting can only be used if you have been granted archiving solution access by the Zoom Support team.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetGroupSettingsResponseRecordingArchive {
    /**
     * [Archiving solution](https://support.zoom.us/hc/en-us/articles/360050431572-Archiving-Meeting-and-Webinar-data) settings. This setting can only be used if you have been granted archiving solution access by the Zoom Support team.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
    /**
     * [Archiving solution](https://support.zoom.us/hc/en-us/articles/360050431572-Archiving-Meeting-and-Webinar-data) settings. This setting can only be used if you have been granted archiving solution access by the Zoom Support team.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub settings: Option<Settings>,
    /**
     * [Archiving solution](https://support.zoom.us/hc/en-us/articles/360050431572-Archiving-Meeting-and-Webinar-data) settings. This setting can only be used if you have been granted archiving solution access by the Zoom Support team.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetGroupSettingsResponseRecording {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub account_user_access_recording: bool,
    /**
     * [Archiving solution](https://support.zoom.us/hc/en-us/articles/360050431572-Archiving-Meeting-and-Webinar-data) settings. This setting can only be used if you have been granted archiving solution access by the Zoom Support team.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub archive: Option<GetGroupSettingsResponseRecordingArchive>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub auto_recording: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cloud_recording: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cloud_recording_download: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cloud_recording_download_host: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub host_delete_cloud_recording: bool,
    /**
     * Setting to allow cloud recording access only from specific IP address ranges.
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub ip_address_access_control: Option<IpAddressAccessControl>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub local_recording: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub prevent_host_access_recording: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub record_audio_file: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub record_audio_file_each_participant: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub record_gallery_view: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub record_speaker_view: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub recording_audio_transcript: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub save_chat_text: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_timestamp: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetGroupSettingsResponseTelephony {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub audio_conference_info: String,
    /**
     * Indicates where most of the participants call into or call from duriing a meeting.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub telephony_regions: Option<UpdateAccountSettingsTelephonyRegions>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub third_party_audio: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetGroupSettingsResponse {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub email_notification: Option<GetGroupSettingsResponseEmailNotification>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub in_meeting: Option<GetGroupSettingsResponseInMeeting>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub profile: Option<Profile>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recording: Option<GetGroupSettingsResponseRecording>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub schedule_meeting: Option<GetGroupSettingsResponseScheduleMeeting>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub telephony: Option<GetGroupSettingsResponseTelephony>,
}

/// All of the following types:
///
/// - `Domains`
/// - `MeetingSecuritySettings`
/// - `GetGroupSettingsResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum GetGroupSettingsResponseOneOf {
    Domains(Domains),
    MeetingSecuritySettings(MeetingSecuritySettings),
    GetGroupSettingsResponse(GetGroupSettingsResponse),
}

impl GetGroupSettingsResponseOneOf {
    pub fn domains(&self) -> Option<&Domains> {
        if let GetGroupSettingsResponseOneOf::Domains(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn get_group_settings_response(&self) -> Option<&GetGroupSettingsResponse> {
        if let GetGroupSettingsResponseOneOf::GetGroupSettingsResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn meeting_security_settings(&self) -> Option<&MeetingSecuritySettings> {
        if let GetGroupSettingsResponseOneOf::MeetingSecuritySettings(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateGroupSettingsRequestScheduleMeeting {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub audio_type: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub embed_password_in_join_link: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub force_pmi_jbh_password: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub host_video: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub join_before_host: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub mute_upon_entry: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub participant_video: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub pstn_password_protected: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_all_meetings: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_instant_meetings: bool,
    /**
     * Require a password for a meeting held using Personal Meeting ID (PMI) This setting is always enabled for free accounts and Pro accounts with a single host and cannot be modified for these accounts.
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub require_password_for_pmi_meetings: Option<RequirePasswordPmiMeetings>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_scheduled_meetings: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_scheduling_new_meetings: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub upcoming_meeting_reminder: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateGroupSettingsRequestInMeeting {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub alert_guest_join: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_show_zoom_windows: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub annotation: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub attendee_on_hold: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub auto_answer: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub auto_saving_chat: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub breakout_room: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub chat: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub closed_caption: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub co_host: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub custom_data_center_regions: bool,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub data_center_regions: Vec<String>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub e_2e_encryption: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub entry_exit_chime: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub far_end_camera_control: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub feedback: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub file_transfer: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub group_hd: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub join_from_desktop: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub join_from_mobile: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub non_verbal_feedback: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub only_host_view_device_list: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub original_audio: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub polling: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub post_meeting_feedback: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub private_chat: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub record_play_own_voice: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub remote_control: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub remote_support: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub request_permission_to_unmute: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub screen_sharing: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub sending_default_email_invites: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_a_join_from_your_browser_link: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_browser_join_link: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_device_list: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_meeting_control_toolbar: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub stereo_audio: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub use_html_format_email: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub virtual_background: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub waiting_room: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub whiteboard: bool,
    /**
     * Indicates who can share their screen or content during meetings. The value can be one of the following: <br>
     *  `host`: Only host can share the screen.<br>
     *  `all`: Both hosts and attendees can share their screen during meetings. For Webinar, the hosts and panelists can start screen sharing, but not the attendees.
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub who_can_share_screen: Option<WhoCanShareScreen>,
    /**
     * Indicates who can share their screen or content during meetings. The value can be one of the following: <br>
     *  `host`: Only host can share the screen.<br>
     *  `all`: Both hosts and attendees can share their screen during meetings. For Webinar, the hosts and panelists can start screen sharing, but not the attendees.
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub who_can_share_screen_when_someone_is_sharing: Option<WhoCanShareScreen>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateGroupSettingsRequestTelephony {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub audio_conference_info: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub third_party_audio: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateGroupSettingsRequest {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub email_notification: Option<GetGroupSettingsResponseEmailNotification>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub in_meeting: Option<UpdateGroupSettingsRequestInMeeting>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub profile: Option<Profile>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recording: Option<GetGroupSettingsResponseRecording>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub schedule_meeting: Option<UpdateGroupSettingsRequestScheduleMeeting>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub telephony: Option<UpdateGroupSettingsRequestTelephony>,
}

/// All of the following types:
///
/// - `Domains`
/// - `MeetingSecuritySettings`
/// - `UpdateGroupSettingsRequest`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum UpdateGroupSettingsRequestOneOf {
    Domains(Domains),
    MeetingSecuritySettings(MeetingSecuritySettings),
    UpdateGroupSettingsRequest(UpdateGroupSettingsRequest),
}

impl UpdateGroupSettingsRequestOneOf {
    pub fn domains(&self) -> Option<&Domains> {
        if let UpdateGroupSettingsRequestOneOf::Domains(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn meeting_security_settings(&self) -> Option<&MeetingSecuritySettings> {
        if let UpdateGroupSettingsRequestOneOf::MeetingSecuritySettings(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn update_group_settings_request(&self) -> Option<&UpdateGroupSettingsRequest> {
        if let UpdateGroupSettingsRequestOneOf::UpdateGroupSettingsRequest(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum UpdateGroupSettingsOption {
    #[serde(rename = "meeting_authentication")]
    MeetingAuthentication,
    #[serde(rename = "meeting_security")]
    MeetingSecurity,
    #[serde(rename = "recording_authentication")]
    RecordingAuthentication,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for UpdateGroupSettingsOption {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            UpdateGroupSettingsOption::MeetingAuthentication => "meeting_authentication",
            UpdateGroupSettingsOption::MeetingSecurity => "meeting_security",
            UpdateGroupSettingsOption::RecordingAuthentication => "recording_authentication",
            UpdateGroupSettingsOption::Noop => "",
            UpdateGroupSettingsOption::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for UpdateGroupSettingsOption {
    fn default() -> UpdateGroupSettingsOption {
        UpdateGroupSettingsOption::Noop
    }
}
impl UpdateGroupSettingsOption {
    pub fn is_noop(&self) -> bool {
        matches!(self, UpdateGroupSettingsOption::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetGroupLockSettingsResponseScheduleMeeting {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub audio_type: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub embed_password_in_join_link: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub force_pmi_jbh_password: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub host_video: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub join_before_host: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub meeting_authentication: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub mute_upon_entry: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub participant_video: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub pstn_password_protected: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_instant_meetings: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_pmi_meetings: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_scheduling_new_meetings: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub upcoming_meeting_reminder: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetGroupLockSettingsResponseInMeeting {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub alert_guest_join: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_show_zoom_windows: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub annotation: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub attendee_on_hold: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub auto_answer: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub auto_saving_chat: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub breakout_room: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub chat: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub closed_caption: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub co_host: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub custom_data_center_regions: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub e_2e_encryption: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub entry_exit_chime: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub far_end_camera_control: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub feedback: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub file_transfer: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub group_hd: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub non_verbal_feedback: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub original_audio: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub polling: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub post_meeting_feedback: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub private_chat: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub remote_control: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub remote_support: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub request_permission_to_unmute: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub screen_sharing: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub sending_default_email_invites: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_a_join_from_your_browser_link: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_browser_join_link: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub show_meeting_control_toolbar: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub stereo_audio: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub use_html_format_email: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub virtual_background: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub waiting_room: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub whiteboard: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetGroupLockSettingsResponseRecording {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub account_user_access_recording: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub auto_delete_cmr: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub auto_recording: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cloud_recording: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cloud_recording_download: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub host_delete_cloud_recording: bool,
    /**
     * Setting to allow cloud recording access only from specific IP address ranges.
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub ip_address_access_control: Option<IpAddressAccessControl>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub local_recording: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub prevent_host_access_recording: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub recording_authentication: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetGroupLockSettingsResponseTelephony {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub telephony_regions: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub third_party_audio: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetGroupLockSettingsResponse {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub email_notification: Option<GetGroupSettingsResponseEmailNotification>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub in_meeting: Option<GetGroupLockSettingsResponseInMeeting>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recording: Option<GetGroupLockSettingsResponseRecording>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub schedule_meeting: Option<GetGroupLockSettingsResponseScheduleMeeting>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub telephony: Option<GetGroupLockSettingsResponseTelephony>,
}

/// All of the following types:
///
/// - `MeetingSecuritySettings`
/// - `GetGroupLockSettingsResponse`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum GetGroupLockSettingsResponseOneOf {
    MeetingSecuritySettings(MeetingSecuritySettings),
    GetGroupLockSettingsResponse(GetGroupLockSettingsResponse),
}

impl GetGroupLockSettingsResponseOneOf {
    pub fn get_group_lock_settings_response(&self) -> Option<&GetGroupLockSettingsResponse> {
        if let GetGroupLockSettingsResponseOneOf::GetGroupLockSettingsResponse(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn meeting_security_settings(&self) -> Option<&MeetingSecuritySettings> {
        if let GetGroupLockSettingsResponseOneOf::MeetingSecuritySettings(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GroupLockedSettingsRequestScheduleMeeting {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub audio_type: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub embed_password_in_join_link: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub force_pmi_jbh_password: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub host_video: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub join_before_host: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub meeting_authentication: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub mute_upon_entry: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub participant_video: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub personal_meeting: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub pstn_password_protected: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_instant_meetings: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_pmi_meetings: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_password_for_scheduling_new_meetings: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub upcoming_meeting_reminder: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GroupLockedSettingsRequestRecording {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub account_user_access_recording: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub auto_delete_cmr: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub auto_recording: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cloud_recording: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cloud_recording_download: bool,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub host_delete_cloud_recording: bool,
    /**
     * Setting to allow cloud recording access only from specific IP address ranges.
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub ip_address_access_control: Option<IpAddressAccessControl>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub local_recording: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub recording_authentication: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GroupLockedSettingsRequest {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub email_notification: Option<GetGroupSettingsResponseEmailNotification>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub in_meeting: Option<GetGroupLockSettingsResponseInMeeting>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recording: Option<GroupLockedSettingsRequestRecording>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub schedule_meeting: Option<GroupLockedSettingsRequestScheduleMeeting>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub telephony: Option<GetGroupLockSettingsResponseTelephony>,
}

/// All of the following types:
///
/// - `MeetingSecuritySettings`
/// - `GroupLockedSettingsRequest`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum GroupLockedSettingsRequestOneOf {
    MeetingSecuritySettings(MeetingSecuritySettings),
    GroupLockedSettingsRequest(GroupLockedSettingsRequest),
}

impl GroupLockedSettingsRequestOneOf {
    pub fn group_locked_settings_request(&self) -> Option<&GroupLockedSettingsRequest> {
        if let GroupLockedSettingsRequestOneOf::GroupLockedSettingsRequest(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn meeting_security_settings(&self) -> Option<&MeetingSecuritySettings> {
        if let GroupLockedSettingsRequestOneOf::MeetingSecuritySettings(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// This field will only be displayed to accounts that are enrolled in a partner plan and follow the master accounts and sub accounts structure.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SubAccountPrivileges {
    /**
     * This field will only be displayed to accounts that are enrolled in a partner plan and follow the master accounts and sub accounts structure.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub second_level: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetRoleInformationResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub privileges: Vec<String>,
    /**
     * This field will only be displayed to accounts that are enrolled in a partner plan and follow the master accounts and sub accounts structure.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sub_account_privileges: Option<SubAccountPrivileges>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_members: i64,
}

/// This field will only be displayed to accounts that are enrolled in the partner plan and follow master accounts and sub accounts structure.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateRoleRequestSubAccountPrivileges {
    /**
     * This field will only be displayed to accounts that are enrolled in the partner plan and follow master accounts and sub accounts structure.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub second_level: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateRoleRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub privileges: Vec<String>,
    /**
     * This field will only be displayed to accounts that are enrolled in the partner plan and follow master accounts and sub accounts structure.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sub_account_privileges: Option<UpdateRoleRequestSubAccountPrivileges>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_members: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetPlanUsageResponseBase {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub hosts: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub usage: i64,
}

/// Recording Plan
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PlanRecording {
    /**
     * Recording Plan
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub free_storage: String,
    /**
     * Recording Plan
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub free_storage_usage: String,
    /**
     * Recording Plan
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub plan_storage: String,
    /**
     * Recording Plan
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub plan_storage_exceed: String,
    /**
     * Recording Plan
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub plan_storage_usage: String,
    /**
     * Recording Plan
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

/// [Zoom United](https://zoom.us/pricing/zoom-bundles) plan.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct PlanUnited {
    /**
     * [Zoom United](https://zoom.us/pricing/zoom-bundles) plan.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub hosts: i64,
    /**
     * [Zoom United](https://zoom.us/pricing/zoom-bundles) plan.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * [Zoom United](https://zoom.us/pricing/zoom-bundles) plan.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
    /**
     * [Zoom United](https://zoom.us/pricing/zoom-bundles) plan.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub usage: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetPlanUsageResponse {
    /**
     * The base plan subscribed for the sub account.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub plan_base: Vec<GetPlanUsageResponseBase>,
    /**
     * The base plan subscribed for the sub account.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub plan_large_meeting: Vec<GetPlanUsageResponseBase>,
    /**
     * Recording Plan
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub plan_recording: Option<PlanRecording>,
    /**
     * [Zoom United](https://zoom.us/pricing/zoom-bundles) plan.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub plan_united: Option<PlanUnited>,
    /**
     * The base plan subscribed for the sub account.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub plan_webinar: Vec<GetPlanUsageResponseBase>,
    /**
     * The base plan subscribed for the sub account.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub plan_zoom_rooms: Vec<GetPlanUsageResponseBase>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SwitchUserAccountRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub account_id: String,
}

/**
 * The recording file type. The value of this field could be one of the following:<br>
 *   * `MP4`: Video file of the recording.<br>
 *   * `M4A` Audio-only file of the recording.<br>
 *   * `TB`: Timestamp file of the recording in JSON format.<br>
 *   * `TRANSCRIPT`: Transcription file of the recording in VTT format.<br>
 *   * `CHAT`: A TXT file containing in-meeting chat messages that were sent during the meeting.<br>
 *   * `CC`: File containing closed captions of the recording in VTT file format.<br>
 *   * `CSV`: File containing polling data in csv format.
 *   
 *   A recording file object with file type of either `CC` or `TB` **does not have** the following properties:<br>
 *   	`id`, `status`, `file_size`, `recording_type`, and `play_url`.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum GetAccountCloudRecordingResponseMeetingsFilesFileType {
    #[serde(rename = "CC")]
    Cc,
    #[serde(rename = "CHAT")]
    Chat,
    #[serde(rename = "CSV")]
    Csv,
    #[serde(rename = "M4A")]
    M4A,
    #[serde(rename = "MP4")]
    Mp4,
    #[serde(rename = "TB")]
    Tb,
    #[serde(rename = "TRANSCRIPT")]
    Transcript,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for GetAccountCloudRecordingResponseMeetingsFilesFileType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GetAccountCloudRecordingResponseMeetingsFilesFileType::Cc => "CC",
            GetAccountCloudRecordingResponseMeetingsFilesFileType::Chat => "CHAT",
            GetAccountCloudRecordingResponseMeetingsFilesFileType::Csv => "CSV",
            GetAccountCloudRecordingResponseMeetingsFilesFileType::M4A => "M4A",
            GetAccountCloudRecordingResponseMeetingsFilesFileType::Mp4 => "MP4",
            GetAccountCloudRecordingResponseMeetingsFilesFileType::Tb => "TB",
            GetAccountCloudRecordingResponseMeetingsFilesFileType::Transcript => "TRANSCRIPT",
            GetAccountCloudRecordingResponseMeetingsFilesFileType::Noop => "",
            GetAccountCloudRecordingResponseMeetingsFilesFileType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for GetAccountCloudRecordingResponseMeetingsFilesFileType {
    fn default() -> GetAccountCloudRecordingResponseMeetingsFilesFileType {
        GetAccountCloudRecordingResponseMeetingsFilesFileType::Noop
    }
}
impl GetAccountCloudRecordingResponseMeetingsFilesFileType {
    pub fn is_noop(&self) -> bool {
        matches!(
            self,
            GetAccountCloudRecordingResponseMeetingsFilesFileType::Noop
        )
    }
}

/**
 * The status of the recording, either `completed` or `processing`.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum GetAccountCloudRecordingResponseMeetingsFilesStatus {
    #[serde(rename = "completed")]
    Completed,
    #[serde(rename = "processing")]
    Processing,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for GetAccountCloudRecordingResponseMeetingsFilesStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GetAccountCloudRecordingResponseMeetingsFilesStatus::Completed => "completed",
            GetAccountCloudRecordingResponseMeetingsFilesStatus::Processing => "processing",
            GetAccountCloudRecordingResponseMeetingsFilesStatus::Noop => "",
            GetAccountCloudRecordingResponseMeetingsFilesStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for GetAccountCloudRecordingResponseMeetingsFilesStatus {
    fn default() -> GetAccountCloudRecordingResponseMeetingsFilesStatus {
        GetAccountCloudRecordingResponseMeetingsFilesStatus::Noop
    }
}
impl GetAccountCloudRecordingResponseMeetingsFilesStatus {
    pub fn is_noop(&self) -> bool {
        matches!(
            self,
            GetAccountCloudRecordingResponseMeetingsFilesStatus::Noop
        )
    }
}

/**
 * The recording type. The value of this field can be one of the following:<br>
 *   * `shared_screen_with_speaker_view(CC)`
 *   * `shared_screen_with_speaker_view`
 *   * `shared_screen_with_gallery_view`
 *   * `shared_screen`
 *   * `active_speaker`
 *   * `gallery_view`
 *   * `audio_only`
 *   * `audio_transcript`
 *   * `chat_file`
 *   * `host_video`
 *   * `closed_caption`
 *   * `poll`
 *   * `timeline`
 *   * `thumbnail`
 *   
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum GetAccountCloudRecordingResponseMeetingsFilesType {
    #[serde(rename = "active_speaker")]
    ActiveSpeaker,
    #[serde(rename = "audio_only")]
    AudioOnly,
    #[serde(rename = "audio_transcript")]
    AudioTranscript,
    #[serde(rename = "chat_file")]
    ChatFile,
    #[serde(rename = "closed_caption")]
    ClosedCaption,
    #[serde(rename = "gallery_view")]
    GalleryView,
    #[serde(rename = "host_video")]
    HostVideo,
    #[serde(rename = "poll")]
    Poll,
    #[serde(rename = "shared_screen")]
    SharedScreen,
    #[serde(rename = "shared_screen_with_gallery_view")]
    SharedScreenWithGalleryView,
    #[serde(rename = "shared_screen_with_speaker_view")]
    SharedScreenWithSpeakerView,
    #[serde(rename = "shared_screen_with_speaker_view(CC)")]
    SharedScreenWithSpeakerViewCc,
    #[serde(rename = "thumbnail")]
    Thumbnail,
    #[serde(rename = "timeline")]
    Timeline,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for GetAccountCloudRecordingResponseMeetingsFilesType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GetAccountCloudRecordingResponseMeetingsFilesType::ActiveSpeaker => "active_speaker",
            GetAccountCloudRecordingResponseMeetingsFilesType::AudioOnly => "audio_only",
            GetAccountCloudRecordingResponseMeetingsFilesType::AudioTranscript => {
                "audio_transcript"
            }
            GetAccountCloudRecordingResponseMeetingsFilesType::ChatFile => "chat_file",
            GetAccountCloudRecordingResponseMeetingsFilesType::ClosedCaption => "closed_caption",
            GetAccountCloudRecordingResponseMeetingsFilesType::GalleryView => "gallery_view",
            GetAccountCloudRecordingResponseMeetingsFilesType::HostVideo => "host_video",
            GetAccountCloudRecordingResponseMeetingsFilesType::Poll => "poll",
            GetAccountCloudRecordingResponseMeetingsFilesType::SharedScreen => "shared_screen",
            GetAccountCloudRecordingResponseMeetingsFilesType::SharedScreenWithGalleryView => {
                "shared_screen_with_gallery_view"
            }
            GetAccountCloudRecordingResponseMeetingsFilesType::SharedScreenWithSpeakerView => {
                "shared_screen_with_speaker_view"
            }
            GetAccountCloudRecordingResponseMeetingsFilesType::SharedScreenWithSpeakerViewCc => {
                "shared_screen_with_speaker_view(CC)"
            }
            GetAccountCloudRecordingResponseMeetingsFilesType::Thumbnail => "thumbnail",
            GetAccountCloudRecordingResponseMeetingsFilesType::Timeline => "timeline",
            GetAccountCloudRecordingResponseMeetingsFilesType::Noop => "",
            GetAccountCloudRecordingResponseMeetingsFilesType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for GetAccountCloudRecordingResponseMeetingsFilesType {
    fn default() -> GetAccountCloudRecordingResponseMeetingsFilesType {
        GetAccountCloudRecordingResponseMeetingsFilesType::Noop
    }
}
impl GetAccountCloudRecordingResponseMeetingsFilesType {
    pub fn is_noop(&self) -> bool {
        matches!(
            self,
            GetAccountCloudRecordingResponseMeetingsFilesType::Noop
        )
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct RecordingFiles {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub download_url: String,
    /**
     * The recording file size.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_f64",
        deserialize_with = "crate::utils::deserialize_null_f64::deserialize"
    )]
    pub file_size: f64,
    /**
     * The recording file type. The value of this field could be one of the following:<br>
     *  \* `MP4`: Video file of the recording.<br>
     *  \* `M4A` Audio-only file of the recording.<br>
     *  \* `TB`: Timestamp file of the recording in JSON format.<br>
     *  \* `TRANSCRIPT`: Transcription file of the recording in VTT format.<br>
     *  \* `CHAT`: A TXT file containing in-meeting chat messages that were sent during the meeting.<br>
     *  \* `CC`: File containing closed captions of the recording in VTT file format.<br>
     *  \* `CSV`: File containing polling data in csv format.
     *  
     *  A recording file object with file type of either `CC` or `TB` \*\*does not have\*\* the following properties:<br>
     *  	`id`, `status`, `file_size`, `recording_type`, and `play_url`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub file_type: Option<GetAccountCloudRecordingResponseMeetingsFilesFileType>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub meeting_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub play_url: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub recording_end: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub recording_start: String,
    /**
     * The recording type. The value of this field can be one of the following:<br>
     *  \* `shared_screen_with_speaker_view(CC)`
     *  \* `shared_screen_with_speaker_view`
     *  \* `shared_screen_with_gallery_view`
     *  \* `shared_screen`
     *  \* `active_speaker`
     *  \* `gallery_view`
     *  \* `audio_only`
     *  \* `audio_transcript`
     *  \* `chat_file`
     *  \* `host_video`
     *  \* `closed_caption`
     *  \* `poll`
     *  \* `timeline`
     *  \* `thumbnail`
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub recording_type: Option<GetAccountCloudRecordingResponseMeetingsFilesType>,
    /**
     * The status of the recording, either `completed` or `processing`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<GetAccountCloudRecordingResponseMeetingsFilesStatus>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetAccountCloudRecordingResponseMeetings {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub host_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub recording_count: i64,
    /**
     * Recording files object
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub recording_files: Vec<RecordingFiles>,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub start_time: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub topic: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_size: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub uuid: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetAccountCloudRecordingResponse {
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub from: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Meetings Object
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub meetings: Vec<GetAccountCloudRecordingResponseMeetings>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * Account subscription start date and time.
     */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_time_format::deserialize"
    )]
    pub to: Option<chrono::DateTime<chrono::Utc>>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct TrackingSources {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub registration_count: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub source_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub tracking_url: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub visitor_count: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetTrackingSourcesResponse {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
    /**
     * Tracking Sources object.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub tracking_sources: Vec<TrackingSources>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Basic {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub required_code_to_ext: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub room_passcode: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub support_email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub support_phone: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateZrAccProfileRequest {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub basic: Option<Basic>,
}

/**
 * The type of setting that you would like to retrieve.<br> `alert`: Alert Settings applied on the Zoom Rooms Account.<br>
 *   `meeting`: Meeting settings of the Zoom Rooms Account. <br>
 *   `signage`: View digital signage settings of the Zoom Rooms Account.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum SettingType {
    #[serde(rename = "alert")]
    Alert,
    #[serde(rename = "meeting")]
    Meeting,
    FallthroughString(String),
}

impl std::fmt::Display for SettingType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            SettingType::Alert => "alert",
            SettingType::Meeting => "meeting",
            SettingType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for SettingType {
    fn default() -> SettingType {
        SettingType::Meeting
    }
}

/**
 * The type of location. The value can be one of the following:
 *   `country`, `states`, `city`, `campus`, `building`, `floor`.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ListZrLocationsResponseType {
    #[serde(rename = "building")]
    Building,
    #[serde(rename = "campus")]
    Campus,
    #[serde(rename = "city")]
    City,
    #[serde(rename = "country")]
    Country,
    #[serde(rename = "floor")]
    Floor,
    #[serde(rename = "states")]
    States,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for ListZrLocationsResponseType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ListZrLocationsResponseType::Building => "building",
            ListZrLocationsResponseType::Campus => "campus",
            ListZrLocationsResponseType::City => "city",
            ListZrLocationsResponseType::Country => "country",
            ListZrLocationsResponseType::Floor => "floor",
            ListZrLocationsResponseType::States => "states",
            ListZrLocationsResponseType::Noop => "",
            ListZrLocationsResponseType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for ListZrLocationsResponseType {
    fn default() -> ListZrLocationsResponseType {
        ListZrLocationsResponseType::Noop
    }
}
impl ListZrLocationsResponseType {
    pub fn is_noop(&self) -> bool {
        matches!(self, ListZrLocationsResponseType::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddAzrLocationResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub parent_location_id: String,
    /**
     * The type of location. The value can be one of the following:
     *  `country`, `states`, `city`, `campus`, `building`, `floor`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<ListZrLocationsResponseType>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListZrLocationsResponseData {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub locations: Vec<AddAzrLocationResponse>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddAzrLocationRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub parent_location_id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetZrLocationProfileResponseBasic {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub required_code_to_ext: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub room_passcode: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub support_email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub support_phone: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetZrLocationProfileResponse {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub basic: Option<GetZrLocationProfileResponseBasic>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetZrLocationStructureResponse {
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub structures: Vec<String>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetZrProfileResponseBasic {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub activation_code: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub hide_room_in_contacts: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub required_code_to_ext: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub room_passcode: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub support_email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub support_phone: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetZrProfileResponse {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub basic: Option<GetZrProfileResponseBasic>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateRoomProfileRequestBasic {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub hide_room_in_contacts: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub required_code_to_ext: bool,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub room_passcode: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub support_email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub support_phone: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateRoomProfileRequest {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub basic: Option<UpdateRoomProfileRequestBasic>,
}

/**
 * Type of the device. The value of this field can be one of the following:<br>`Zoom Rooms Computer`,<br> `Controller`, `Scheduling Display`, `Zoom Rooms Control System`, `Companion Whiteboard`
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DeviceType {
    #[serde(rename = "Companion Whiteboard")]
    CompanionWhiteboard,
    #[serde(rename = "Controller")]
    Controller,
    #[serde(rename = "Scheduling Display")]
    SchedulingDisplay,
    #[serde(rename = "Zoom Rooms Computer")]
    ZoomRoomsComputer,
    #[serde(rename = "Zoom Rooms Control System")]
    ZoomRoomsControlSystem,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for DeviceType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DeviceType::CompanionWhiteboard => "Companion Whiteboard",
            DeviceType::Controller => "Controller",
            DeviceType::SchedulingDisplay => "Scheduling Display",
            DeviceType::ZoomRoomsComputer => "Zoom Rooms Computer",
            DeviceType::ZoomRoomsControlSystem => "Zoom Rooms Control System",
            DeviceType::Noop => "",
            DeviceType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for DeviceType {
    fn default() -> DeviceType {
        DeviceType::Noop
    }
}
impl DeviceType {
    pub fn is_noop(&self) -> bool {
        matches!(self, DeviceType::Noop)
    }
}

/**
 * Status of the device. The value can be either `Online` or `Offline`.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ListZrDevicesResponseStatus {
    #[serde(rename = "Offline")]
    Offline,
    #[serde(rename = "Online")]
    Online,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for ListZrDevicesResponseStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ListZrDevicesResponseStatus::Offline => "Offline",
            ListZrDevicesResponseStatus::Online => "Online",
            ListZrDevicesResponseStatus::Noop => "",
            ListZrDevicesResponseStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for ListZrDevicesResponseStatus {
    fn default() -> ListZrDevicesResponseStatus {
        ListZrDevicesResponseStatus::Noop
    }
}
impl ListZrDevicesResponseStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, ListZrDevicesResponseStatus::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Devices {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub app_version: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub device_system: String,
    /**
     * Type of the device. The value of this field can be one of the following:<br>`Zoom Rooms Computer`,<br> `Controller`, `Scheduling Display`, `Zoom Rooms Control System`, `Companion Whiteboard`
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub device_type: Option<DeviceType>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub room_name: String,
    /**
     * Status of the device. The value can be either `Online` or `Offline`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<ListZrDevicesResponseStatus>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListZrDevicesResponse {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub devices: Vec<Devices>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListSipTrunkNumbersResponse {
    /**
     * List of dedicated dial-in numbers.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub phone_numbers: Vec<DedicatedDialInNumber>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AssignSipConfigRequest {
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub enable: Option<bool>,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub show_callout_internal_number: Option<bool>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub show_zoom_provided_callout_countries: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub show_zoom_provided_numbers: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AssignSipTrunkNumbersRequest {
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub phone_numbers: Vec<String>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct InMeetingFiles {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub download_url: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub file_name: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub file_size: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListPastMeetingFilesResponse {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub in_meeting_files: Vec<InMeetingFiles>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ClientSatisfaction {
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub date: Option<chrono::NaiveDate>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub good_count: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub none_count: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub not_good_count: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub satisfaction_percent: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListMeetingSatisfactionResponse {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub client_satisfaction: Vec<ClientSatisfaction>,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub from: Option<chrono::NaiveDate>,
    /**
     * Start Date.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub to: Option<chrono::NaiveDate>,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

/**
 * State of the device. The value should be either `assigned` to list devices that have been assigned to user(s) or `unassigned` to list devices that have not yet been assigned to any user in the Zoom account.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ListPhoneDevicesType {
    #[serde(rename = "assigned")]
    Assigned,
    #[serde(rename = "unassigned")]
    Unassigned,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for ListPhoneDevicesType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ListPhoneDevicesType::Assigned => "assigned",
            ListPhoneDevicesType::Unassigned => "unassigned",
            ListPhoneDevicesType::Noop => "",
            ListPhoneDevicesType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for ListPhoneDevicesType {
    fn default() -> ListPhoneDevicesType {
        ListPhoneDevicesType::Noop
    }
}
impl ListPhoneDevicesType {
    pub fn is_noop(&self) -> bool {
        matches!(self, ListPhoneDevicesType::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddAutoReceptionistResponse {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

/**
 * Status of the device. The value is either `online` or `offline`.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ListPhoneDevicesResponseStatus {
    #[serde(rename = "offline")]
    Offline,
    #[serde(rename = "online")]
    Online,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for ListPhoneDevicesResponseStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ListPhoneDevicesResponseStatus::Offline => "offline",
            ListPhoneDevicesResponseStatus::Online => "online",
            ListPhoneDevicesResponseStatus::Noop => "",
            ListPhoneDevicesResponseStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for ListPhoneDevicesResponseStatus {
    fn default() -> ListPhoneDevicesResponseStatus {
        ListPhoneDevicesResponseStatus::Noop
    }
}
impl ListPhoneDevicesResponseStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, ListPhoneDevicesResponseStatus::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListPhoneDevicesResponse {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub assignee: Option<AddAutoReceptionistResponse>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub device_type: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub mac_address: String,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub site: Option<Site>,
    /**
     * Status of the device. The value is either `online` or `offline`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<ListPhoneDevicesResponseStatus>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListPhoneDevicesResponseData {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub devices: Vec<ListPhoneDevicesResponse>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub total_records: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddPhoneDeviceRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub assigned_to: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub mac_address: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub model: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

/// User to whom the device has been assigned.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetDeviceResponseAssignee {
    /**
     * User to whom the device has been assigned.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * User to whom the device has been assigned.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User to whom the device has been assigned.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

/**
 * [Provisioning type](https://support.zoom.us/hc/en-us/articles/360033223411). The value can be one of the following:
 *   
 *   * `ztp` : Zero touch provisioning.
 *   * `assisted`: Assisted provisioning.
 *   * `manual`: Manual provisioning.
 *   
 *   
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum GetDeviceResponseProvisionType {
    #[serde(rename = "assisted")]
    Assisted,
    #[serde(rename = "manual")]
    Manual,
    #[serde(rename = "ztp")]
    Ztp,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for GetDeviceResponseProvisionType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GetDeviceResponseProvisionType::Assisted => "assisted",
            GetDeviceResponseProvisionType::Manual => "manual",
            GetDeviceResponseProvisionType::Ztp => "ztp",
            GetDeviceResponseProvisionType::Noop => "",
            GetDeviceResponseProvisionType::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for GetDeviceResponseProvisionType {
    fn default() -> GetDeviceResponseProvisionType {
        GetDeviceResponseProvisionType::Noop
    }
}
impl GetDeviceResponseProvisionType {
    pub fn is_noop(&self) -> bool {
        matches!(self, GetDeviceResponseProvisionType::Noop)
    }
}

/// Line subscription.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct LineSubscription {
    /**
     * Line subscription.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * Line subscription.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * Line subscription.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number: String,
}

/// Return additional provisioning information with generic device SIP credentials.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SharedLine {
    /**
     * Return additional provisioning information with generic device SIP credentials.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub alias: String,
    /**
     * Return additional provisioning information with generic device SIP credentials.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub line_subscription: Option<LineSubscription>,
    /**
     * Return additional provisioning information with generic device SIP credentials.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub outbound_caller_id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct SipAccounts {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authorization_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub outbound_proxy: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub secondary_outbound_proxy: String,
    /**
     * Return additional provisioning information with generic device SIP credentials.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub shared_line: Option<SharedLine>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub sip_domain: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_name: String,
}

/// Provisioning information of a device.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct Provision {
    /**
     * Provisioning information of a device.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub sip_accounts: Vec<SipAccounts>,
    /**
     * Provisioning information of a device.
     */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<GetDeviceResponseProvisionType>,
    /**
     * Provisioning information of a device.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetDeviceResponse {
    /**
     * User to whom the device has been assigned.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub assignee: Option<GetDeviceResponseAssignee>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub device_type: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub mac_address: String,
    /**
     * Provisioning information of a device.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub provision: Option<Provision>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub site: Option<Site>,
    /**
     * Status of the device. The value is either `online` or `offline`.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<ListPhoneDevicesResponseStatus>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateDeviceRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub assigned_to: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub mac_address: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ExternalContacts {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub extension_number: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub external_contact_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub phone_numbers: Vec<String>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListExternalContactsResponse {
    /**
     * External contacts information.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub external_contacts: Vec<ExternalContacts>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddExternalContactRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub extension_number: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub phone_numbers: Vec<String>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub routing_path: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateExternalContactRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub extension_number: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub phone_numbers: Vec<String>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub routing_path: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ChangeZrLocationRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location_id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ChangeParentLocationRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub parent_location_id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListCallQueuesResponsePhoneNumbers {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
    /**
     * Source of phone number.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub source: Option<Source>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CallQueues {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Phone number(s) assigned to the call queue.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub phone_numbers: Vec<ListCallQueuesResponsePhoneNumbers>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub site: Option<Site>,
    /**
     * Active or inactive hosts.<br>`active` - Active hosts. <br>`inactive` - Inactive hosts.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<ReportUsersType>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListCallQueuesResponse {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub call_queues: Vec<CallQueues>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

/// A list of one or more phone users to be included in the call queue. Provide either users or common area phone(s). Provide at least one user in the users object.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CreateCallQueueRequestMembers {
    /**
     * A list of one or more phone users to be included in the call queue. Provide either users or common area phone(s). Provide at least one user in the users object.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub common_area_phone_ids: Vec<String>,
    /**
     * A list of one or more phone users to be included in the call queue. Provide either users or common area phone(s). Provide at least one user in the users object.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub users: Vec<Assistants>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CreateCallQueueRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * A list of one or more phone users to be included in the call queue. Provide either users or common area phone(s). Provide at least one user in the users object.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub members: Option<CreateCallQueueRequestMembers>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub site_id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CreateCallQueueResponse {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub status: String,
}

/**
 * Level of the user. The value can be one of the following:<br>
 *   `manager`: A call queue manager has the privilege to change call queue settings, policy settings and manage recordings and voicemail inbox. There can only be one manager for each call queue.<br><br>
 *   `user`: Regular user without the privileges of a manager.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum Level {
    #[serde(rename = "manager")]
    Manager,
    #[serde(rename = "user")]
    User,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for Level {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Level::Manager => "manager",
            Level::User => "user",
            Level::Noop => "",
            Level::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for Level {
    fn default() -> Level {
        Level::Noop
    }
}
impl Level {
    pub fn is_noop(&self) -> bool {
        matches!(self, Level::Noop)
    }
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetCallQueueResponseMembersUsers {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Level of the user. The value can be one of the following:<br>
     *  `manager`: A call queue manager has the privilege to change call queue settings, policy settings and manage recordings and voicemail inbox. There can only be one manager for each call queue.<br><br>
     *  `user`: Regular user without the privileges of a manager.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub level: Option<Level>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub receive_call: bool,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetCallQueueResponseMembers {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub common_area_phones: Vec<Site>,
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub users: Vec<GetCallQueueResponseMembersUsers>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetCallQueueResponse {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub members: Option<GetCallQueueResponseMembers>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Phone number(s) assigned to the call queue.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub phone_numbers: Vec<ListCallQueuesResponsePhoneNumbers>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub site: Option<Site>,
    /**
     * Active or inactive hosts.<br>`active` - Active hosts. <br>`inactive` - Inactive hosts.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<ReportUsersType>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateCallQueueRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub site_id: String,
    /**
     * Active or inactive hosts.<br>`active` - Active hosts. <br>`inactive` - Inactive hosts.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<ReportUsersType>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
}

/// A maximum of 10 members can be added at a time.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddMembersCallQueueRequest {
    /**
     * A maximum of 10 members can be added at a time.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub common_area_phone_ids: Vec<String>,
    /**
     * A maximum of 10 members can be added at a time.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub users: Vec<Assistants>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddMembersCallQueueRequestData {
    /**
     * A maximum of 10 members can be added at a time.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub members: Option<AddMembersCallQueueRequest>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ChangeCallQueueManagerRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub member_id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetPhoneNumberDetailsResponseCarrier {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub code: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddCommonAreaPhoneResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetPhoneNumberDetailsResponse {
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub assignee: Option<Assignee>,
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub capability: Vec<String>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub carrier: Option<GetPhoneNumberDetailsResponseCarrier>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub location: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
    /**
     * Type of number.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub number_type: Option<Type>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sip_group: Option<AddCommonAreaPhoneResponse>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub site: Option<Site>,
    /**
     * Source of phone number.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub source: Option<Source>,
    /**
     * Status of the number.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<ListAccountPhoneNumbersResponseStatus>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdatePhoneNumberDetailsRequest {
    /**
     * Specify the IP addresses of the Room Connectors that you would like to share with the sub account. Multiple values can be separated by comma. If no value is provided in this field, all the Room Connectors of a master account will be shared with the sub account.
     *  
     *  
     *  \*\*Note:\*\* This option can only be used if the value of `share_rc` is set to `true`.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub capability: Vec<String>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub sip_group_id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ChangeMainCompanyNumberRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListCallingPlansResponse {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub assigned: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub available: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub subscribed: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListCallingPlansResponseData {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub calling_plans: Vec<ListCallingPlansResponse>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListPhoneUsersResponseCallingPlans {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListPhoneUsersResponse {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub calling_plans: Vec<ListPhoneUsersResponseCallingPlans>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_user_id: String,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub site: Option<Site>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub status: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListPhoneUsersResponseData {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub users: Vec<ListPhoneUsersResponse>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateAutoReceptionistRequest {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct CommonAreaPhones {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub calling_plans: Vec<ListPhoneUsersResponseCallingPlans>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub device_type: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub mac_address: String,
    /**
     * Phone number(s) assigned to the call queue.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub phone_numbers: Vec<ListCallQueuesResponsePhoneNumbers>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub site: Option<Site>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub status: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ListCommonAreaPhonesResponse {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub common_area_phones: Vec<CommonAreaPhones>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub next_page_token: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub page_size: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total_records: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AddCommonAreaPhoneRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub mac_address: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub model: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub provision_template_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub site_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub time_zone: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetCommonAreaPhoneResponseProvisionSipAccounts {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub authorization_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub outbound_proxy: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub password: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub secondary_outbound_proxy: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub sip_domain: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_name: String,
}

/// Provisioning information of the common area phone.
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetCommonAreaPhoneResponseProvision {
    /**
     * Provisioning information of the common area phone.
     */
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub sip_accounts: Vec<GetCommonAreaPhoneResponseProvisionSipAccounts>,
    /**
     * Provisioning information of the common area phone.
     */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<GetDeviceResponseProvisionType>,
    /**
     * Provisioning information of the common area phone.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetCommonAreaPhoneResponseNumbers {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
    /**
     * Source of phone number.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub source: Option<Source>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct ProvisionTemplate {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
     * Enable/disable the option for a sub account to use shared [Virtual Room Connector(s)](https://support.zoom.us/hc/en-us/articles/202134758-Getting-Started-With-Virtual-Room-Connector) that are set up by the master account. Virtual Room Connectors can only be used by On-prem users.
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub supported: bool,
}

///
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct InternationalCall {
    /**
     *
     */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enable: bool,
}

/**
 * Allow Hot Desking feature to current device.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum GetCommonAreaPhoneResponsePolicyHotDeskingStatus {
    #[serde(rename = "off")]
    Off,
    #[serde(rename = "on")]
    On,
    #[serde(rename = "unsupported")]
    Unsupported,
    #[serde(rename = "")]
    Noop,
    FallthroughString(String),
}

impl std::fmt::Display for GetCommonAreaPhoneResponsePolicyHotDeskingStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GetCommonAreaPhoneResponsePolicyHotDeskingStatus::Off => "off",
            GetCommonAreaPhoneResponsePolicyHotDeskingStatus::On => "on",
            GetCommonAreaPhoneResponsePolicyHotDeskingStatus::Unsupported => "unsupported",
            GetCommonAreaPhoneResponsePolicyHotDeskingStatus::Noop => "",
            GetCommonAreaPhoneResponsePolicyHotDeskingStatus::FallthroughString(s) => s,
        }
        .fmt(f)
    }
}

impl Default for GetCommonAreaPhoneResponsePolicyHotDeskingStatus {
    fn default() -> GetCommonAreaPhoneResponsePolicyHotDeskingStatus {
        GetCommonAreaPhoneResponsePolicyHotDeskingStatus::Noop
    }
}
impl GetCommonAreaPhoneResponsePolicyHotDeskingStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, GetCommonAreaPhoneResponsePolicyHotDeskingStatus::Noop)
    }
}

///
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct HotDesking {
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<GetCommonAreaPhoneResponsePolicyHotDeskingStatus>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetCommonAreaPhoneResponsePolicy {
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub hot_desking: Option<HotDesking>,
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub international_call: Option<InternationalCall>,
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub select_outbound_caller_id: Option<InternationalCall>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct GetCommonAreaPhoneResponse {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub calling_plans: Vec<ListPhoneUsersResponseCallingPlans>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub device_type: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub mac_address: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub phone_numbers: Vec<GetCommonAreaPhoneResponseNumbers>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub policy: Option<GetCommonAreaPhoneResponsePolicy>,
    /**
     * Provisioning information of the common area phone.
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub provision: Option<GetCommonAreaPhoneResponseProvision>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub provision_template: Option<ProvisionTemplate>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub site: Option<Site>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub status: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateCommonAreaPhoneRequestPolicy {
    /**
     *
     */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub hot_desking: Option<HotDesking>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub international_call: Option<ShowDeviceIpCallLog>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub select_outbound_caller_id: Option<ShowDeviceIpCallLog>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct UpdateCommonAreaPhoneRequest {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub extension_number: i64,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub mac_address: String,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub policy: Option<UpdateCommonAreaPhoneRequestPolicy>,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub provision_template_id: String,
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub site_id: String,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AssignPhoneNumbersCommonAreaRequest {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub phone_numbers: Vec<PhoneUserResponseNumbers>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AssignPhoneNumbersCommonAreaResponse {
    /**
     * User's first name.
     */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub number: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AssignPhoneNumbersCommonAreaResponseData {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub phone_numbers: Vec<AssignPhoneNumbersCommonAreaResponse>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AssignCallingPlansCommonAreaPhoneRequest {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AssignCallingPlansCommonAreaPhoneRequestData {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub calling_plans: Vec<AssignCallingPlansCommonAreaPhoneRequest>,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AssignCallingPlansCommonAreaPhoneResponse {
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub name: i64,
    /**
     * Account seats.
     */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize",
        rename = "type"
    )]
    pub type_: i64,
}

#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
pub struct AssignCallingPlansCommonAreaPhoneResponseData {
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub calling_plans: Vec<AssignCallingPlansCommonAreaPhoneResponse>,
}

/**
 * Indicates the match type for the blocked list. The values can be one of the following:<br>
 *   `phoneNumber`: Indicates that only a specific phone number that is shown in the `phone_number` field is blocked.<br><br>
 *   `prefix`: Indicates that all numbers starting with prefix that is shown in the `phone_number` field are blocked.
 */
#[derive(Serialize, Deserialize, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum MatchType {
    #[serde(rename = "phoneNumber")]
    PhoneNumber,
    #[serde(rename