use clap::{Arg, App};
use anyhow::Result;
use walkdir::WalkDir;
use surf::http::{Url, Method, Mime};
use std::str::FromStr;
use serde_json::json;
use serde_derive::{Deserialize, Serialize};

#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct LoginResponse {
    jwt: String,
}
#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct Object {
    pub last_modified: String,
    pub e_tag: String,
    pub storage_class: String,
    pub key: String,
    pub size: u64,
}
#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct ListResponse {
    pub objects: Vec<Object>,
}

#[async_std::main]
async fn main() -> Result<()> {
    let matches = App::new("zn")
        .version("1.0")
        .author("APIBillMe <Bevan Hunt>")
        .about("CLI for Files")
        .arg(Arg::with_name("upload")
            .long("upload")
            .short("u")
            .help("directory to upload")
            .takes_value(true))
        .arg(Arg::with_name("prefix")
            .long("prefix")
            .short("x")
            .help("prefix for object_path")
            .takes_value(true))
        .arg(Arg::with_name("delimiter")
            .long("delimiter")
            .short("l")
            .help("delimiter for bucket_list")
            .takes_value(true))    
        .arg(Arg::with_name("download")
            .long("download")
            .short("d")
            .help("directory to download to")
            .takes_value(true))
        .arg(Arg::with_name("region_in")
            .long("region_in")
            .short("a")
            .help("region for bucket in")
            .takes_value(true))
        .arg(Arg::with_name("region_out")
            .long("region_out")
            .short("b")
            .help("region for bucket out")
            .takes_value(true))
        .arg(Arg::with_name("bucket_in")
            .long("bucket_in")
            .short("j")
            .help("bucket files are going to")
            .takes_value(true))
        .arg(Arg::with_name("bucket_out")
            .long("bucket_out")
            .short("k")
            .help("bucket files are coming from")
            .takes_value(true))
        .arg(Arg::with_name("files_endpoint")
            .long("files_endpoint")
            .short("w")
            .required(true)
            .help("endpoint for the files service")
            .takes_value(true))
        .arg(Arg::with_name("broker_endpoint")
            .long("broker_endpoint")
            .short("q")
            .required(true)
            .help("endpoint for the broker service")
            .takes_value(true))
        .arg(Arg::with_name("user")
            .long("user")
            .short("r")
            .required(true)
            .help("username")
            .takes_value(true))
        .arg(Arg::with_name("password")
            .long("password")
            .short("p")
            .required(true)
            .help("password")
            .takes_value(true))
    .get_matches();

    // upload files
    match matches.value_of("upload") {
        Some(u) => {

            let body = json!({"username": matches.value_of("user").unwrap(), "password": matches.value_of("password").unwrap()}).to_string();

            let broker_url = format!("{}/login", matches.value_of("broker_endpoint").unwrap());
            let url = Url::parse(&broker_url)?;
            let mime = Mime::from_str("application/json").unwrap();
            let request = surf::Request::builder(Method::Post, url.clone())
            .content_type(mime)
            .body(body)
            .build();
    
            let mut res = surf::client().send(request).await.unwrap();
            if res.status() == 200 {
                let lr: LoginResponse = res.body_json().await.unwrap();
                let jwt = lr.jwt;

                match matches.value_of("region_in") {
                    Some(ri) => {
                        match matches.value_of("bucket_in") {
                            Some(bi) => {
                                let walk = WalkDir::new(u).into_iter();
                                for entry in walk {
                                    let de = entry?;
                                    let path = de.path();
                                    if path.is_file() {
                                        let file_data = base64::encode(async_std::fs::read(path).await?);
                                        let object_path: String;
                                        let file_name = path.file_name().unwrap().to_str().unwrap().to_string();
                                        match matches.value_of("prefix") {
                                            Some(prefix) => {
                                                object_path = format!("{}{}", prefix, file_name.clone());
                                            },
                                            None => {
                                                object_path = file_name.clone();
                                            }
                                        }

                                        let body = json!({"file": file_data, "region": ri, "bucket": bi, "object_path": object_path}).to_string();

                                        let files_url = format!("{}/put", matches.value_of("files_endpoint").unwrap());
                                        let url = Url::parse(&files_url)?;
                                        let auth = format!("Bearer {}", jwt);
                                        let mime = Mime::from_str("application/json").unwrap();
                                        let request = surf::Request::builder(Method::Post, url.clone())
                                        .content_type(mime)
                                        .header("authorization", &auth)
                                        .body(body)
                                        .build();

                                        let res = surf::client().send(request).await.unwrap();
                                        if res.status() == 200 {
                                            println!("uploaded file: {}", file_name.clone());
                                        }
                                        else {
                                            println!("error - could not upload file: {}", file_name.clone());
                                        }
                                    }
                                }
                            },
                            None => {
                                println!("bucket_in must be specified");
                            }
                        }
                    },
                    None => {
                        println!("region_in must be specified");
                    }
                }
            } else {
                println!("cannot login");
            }
        },
        None => {
             // download files
            match matches.value_of("download") {
                Some(d) => {
                    let body = json!({"username": matches.value_of("user").unwrap(), "password": matches.value_of("password").unwrap()}).to_string();

                    let broker_url = format!("{}/login", matches.value_of("broker_endpoint").unwrap());
                    let url = Url::parse(&broker_url)?;
                    let mime = Mime::from_str("application/json").unwrap();
                    let request = surf::Request::builder(Method::Post, url.clone())
                    .content_type(mime)
                    .body(body)
                    .build();
            
                    let mut res = surf::client().send(request).await.unwrap();
                    if res.status() == 200 {
                        let lr: LoginResponse = res.body_json().await.unwrap();
                        let jwt = lr.jwt;

                        match matches.value_of("region_out") {
                            Some(ro) => {
                                match matches.value_of("bucket_out") {
                                    Some(bo) => {

                                        let body: String;

                                        match matches.value_of("prefix") {
                                            Some(prefix) => {

                                                match matches.value_of("delimiter") {
                                                    Some(delimiter) => {
                                                        body = json!({"region": ro, "bucket": bo, "prefix": prefix, "delimiter": delimiter}).to_string();
                                                    },
                                                    None => {
                                                        body = json!({"region": ro, "bucket": bo, "prefix": prefix}).to_string();
                                                    }
                                                }
                                            },
                                            None => {
                                                body = json!({"region": ro, "bucket": bo, "prefix": ""}).to_string();
                                            }     
                                        }

                                        let files_url = format!("{}/list_bucket", matches.value_of("files_endpoint").unwrap());
                                        let url = Url::parse(&files_url)?;
                                        let auth = format!("Bearer {}", jwt);
                                        let mime = Mime::from_str("application/json").unwrap();
                                        let request = surf::Request::builder(Method::Post, url.clone())
                                        .content_type(mime)
                                        .header("authorization", &auth)
                                        .body(body)
                                        .build();

                                        let mut res = surf::client().send(request).await.unwrap();
                                        if res.status() == 200 {
                                            let list: ListResponse = res.body_json().await.unwrap();
                                            for obj in list.objects {

                                                let object_path = obj.key;
                                                let body = json!({"region": ro, "bucket": bo, "object_path": object_path}).to_string();

                                                let files_url = format!("{}/get", matches.value_of("files_endpoint").unwrap());
                                                let url = Url::parse(&files_url)?;
                                                let auth = format!("Bearer {}", jwt);
                                                let mime = Mime::from_str("application/json").unwrap();
                                                let request = surf::Request::builder(Method::Post, url.clone())
                                                .content_type(mime)
                                                .header("authorization", &auth)
                                                .body(body)
                                                .build();

                                                let mut res = surf::client().send(request).await.unwrap();
                                                if res.status() == 200 {
                                                    // download dir must have a trailing slash
                                                    let file_path = format!("{}{}", d, object_path);
                                                    async_std::fs::write(file_path, res.body_bytes().await.unwrap()).await?;
                                                    println!("downloaded file: {}", object_path.clone());
                                                }
                                                else {
                                                    println!("error - could not download file: {}", object_path.clone());
                                                }
                                            }
                                        }
                                        else {
                                            println!("cannot list files from the bucket");
                                        }
                                    },
                                    None => {
                                        println!("bucket_out must be specified");
                                    }
                                }
                            },
                            None => {
                                println!("region_out must be specified");
                            }
                        }
                    } else {
                        println!("cannot login");
                    }
                },
                None => {
                    // copy files
                    let body = json!({"username": matches.value_of("user").unwrap(), "password": matches.value_of("password").unwrap()}).to_string();

                    let broker_url = format!("{}/login", matches.value_of("broker_endpoint").unwrap());
                    let url = Url::parse(&broker_url)?;
                    let mime = Mime::from_str("application/json").unwrap();
                    let request = surf::Request::builder(Method::Post, url.clone())
                    .content_type(mime)
                    .body(body)
                    .build();
            
                    let mut res = surf::client().send(request).await.unwrap();
                    if res.status() == 200 {
                        let lr: LoginResponse = res.body_json().await.unwrap();
                        let jwt = lr.jwt;

                        match matches.value_of("region_out") {
                            Some(ro) => {
                                match matches.value_of("region_in") {
                                    Some(ri) => {
                                        match matches.value_of("bucket_out") {
                                            Some(bo) => {
                                                match matches.value_of("bucket_in") {
                                                    Some(bi) => {
                                                        let body: String;

                                                        match matches.value_of("prefix") {
                                                            Some(prefix) => {
        
                                                                match matches.value_of("delimiter") {
                                                                    Some(delimiter) => {
                                                                        body = json!({"region": ro, "bucket": bo, "prefix": prefix, "delimiter": delimiter}).to_string();
                                                                    },
                                                                    None => {
                                                                        body = json!({"region": ro, "bucket": bo, "prefix": prefix}).to_string();
                                                                    }
                                                                }
                                                            },
                                                            None => {
                                                                body = json!({"region": ro, "bucket": bo, "prefix": ""}).to_string();
                                                            }     
                                                        }
        
                                                        let files_url = format!("{}/list_bucket", matches.value_of("files_endpoint").unwrap());
                                                        let url = Url::parse(&files_url)?;
                                                        let auth = format!("Bearer {}", jwt);
                                                        let mime = Mime::from_str("application/json").unwrap();
                                                        let request = surf::Request::builder(Method::Post, url.clone())
                                                        .content_type(mime)
                                                        .header("authorization", &auth)
                                                        .body(body)
                                                        .build();
        
                                                        let mut res = surf::client().send(request).await.unwrap();
                                                        if res.status() == 200 {
                                                            let list: ListResponse = res.body_json().await.unwrap();
                                                            for obj in list.objects {
        
                                                                let object_path = obj.key;
                                                                let body = json!({"region": ro, "bucket": bo, "object_path": object_path}).to_string();
        
                                                                let files_url = format!("{}/get", matches.value_of("files_endpoint").unwrap());
                                                                let url = Url::parse(&files_url)?;
                                                                let auth = format!("Bearer {}", jwt);
                                                                let mime = Mime::from_str("application/json").unwrap();
                                                                let request = surf::Request::builder(Method::Post, url.clone())
                                                                .content_type(mime)
                                                                .header("authorization", &auth)
                                                                .body(body)
                                                                .build();
        
                                                                let mut res = surf::client().send(request).await.unwrap();
                                                                if res.status() == 200 {
                                                                    
                                                                    let file_data = base64::encode(res.body_bytes().await.unwrap());
                                                                    let body = json!({"region": ri, "bucket": bi, "object_path": object_path, "file": file_data}).to_string();
            
                                                                    let files_url = format!("{}/put", matches.value_of("files_endpoint").unwrap());
                                                                    let url = Url::parse(&files_url)?;
                                                                    let auth = format!("Bearer {}", jwt);
                                                                    let mime = Mime::from_str("application/json").unwrap();
                                                                    let request = surf::Request::builder(Method::Post, url.clone())
                                                                    .content_type(mime)
                                                                    .header("authorization", &auth)
                                                                    .body(body)
                                                                    .build();
            
                                                                    let res = surf::client().send(request).await.unwrap();
                                                                    if res.status() == 200 {
                                                                        println!("copied file: {}", object_path.clone());
                                                                    } else {
                                                                        println!("error - could not copy file: {}", object_path.clone());
                                                                    }
                                                                }
                                                                else {
                                                                    println!("error - could not download file: {}", object_path.clone());
                                                                }
                                                            }
                                                        }
                                                        else {
                                                            println!("cannot list files from the bucket");
                                                        }
                                                    },
                                                    None => {
                                                        println!("bucket_in must be specified");
                                                    }
                                                }
                                            },
                                            None => {
                                                println!("bucket_out must be specified");
                                            }
                                        }
                                    },
                                    None => {
                                        println!("region_in must be specified");
                                    }          
                                }
                            },
                            None => {
                                println!("region_out must be specified");
                            }
                        }
                    } else {
                        println!("cannot login");
                    }
                }
            }      
        }
    }     
    Ok(())
}