// - STD
use std::{
    fs::{File},
    process::exit,
    path::PathBuf,
    collections::{HashMap},
};

// - extern crates
extern crate clap;
extern crate rand;
extern crate zff;

// - modules
mod lib;

// - internal
use crate::lib::{
    hrs_parser,
    constants::*,
};
use zff::{
    header::{CompressionHeader, EncryptionHeader, DescriptionHeader, MainHeader, ObjectHeader, ObjectType},
    header::{KDFParameters, PBKDF2SHA256Parameters, ScryptParameters, PBEHeader},
    EncryptionAlgorithm,
    CompressionAlgorithm,
    HashType,
    KDFScheme,
    PBEScheme,
    ZffCreator,
    ZffCreatorMetadataParams,
    ZffExtender,
    Encryption,
    Signature,
    SignatureFlag,
    constants::{
        DEFAULT_COMPRESSION_RATIO_THRESHOLD,
        DEFAULT_HEADER_VERSION_COMPRESSION_HEADER,
        DEFAULT_HEADER_VERSION_DESCRIPTION_HEADER,
        DEFAULT_HEADER_VERSION_PBE_HEADER,
        DEFAULT_HEADER_VERSION_ENCRYPTION_HEADER,
        DEFAULT_HEADER_VERSION_MAIN_HEADER,
        DEFAULT_HEADER_VERSION_OBJECT_HEADER,
        INITIAL_OBJECT_NUMBER,
    },
};

// - external
use clap::{
    Parser,
    Subcommand,
    ArgEnum,
};
use rand::{Rng};
use ed25519_dalek::Keypair;

#[derive(Parser)]
#[clap(about, version, author)]
struct Cli {

    /// A general description of all data, which are inside the zff file(s).
    #[clap(short='D', long="description-notes", global=true, required=false)]
    description_notes: Option<String>,

    /// sets the compression algorithm. Default is zstd.
    #[clap(short='z', long="compression-algorithm", global=true, required=false, arg_enum, default_value="zstd")]
    compression_algorithm: CompressionAlgorithmValues,

    /// sets the compression level. Default is 3. This option doesn't has any effect while using the lz4 compression algorithm.
    #[clap(short='l', long="compression-level", global=true, required=false, default_value="3")]
    compression_level: u8,

    /// The compression ratio threshold. Default is 1.05.
    #[clap(short='T', long="compression-threshold", global=true, required=false, default_value=DEFAULT_COMPRESSION_RATIO_THRESHOLD)]
    compression_threshold: f32,

    /// The segment size of the output-file(s). Default is 0 (=the output image will never be splitted into segments).
    #[clap(short='s', long="segment-size", global=true, required=false, default_value="0")]
    segment_size: String,

    /// The chunk size. Default is 32kB. The chunksize have to be greater than the segment size.
    #[clap(short='C', long, global=true, required=false, default_value="32KB", arg_enum)]
    chunk_size: ChunkSize,

    /// Sets an encryption password
    #[clap(short='p', long="encryption-password", global=true, required=false)]
    encryption_password: Option<String>,

    /// Sets the key derivation function for the password. Default is [scrypt-aes256].
    #[clap(short='K', long="password-kdf", global=true, required=false, arg_enum, default_value="scrypt-aes256")]
    password_kdf: PasswordKdfValues,

    /// Sets the encryption algorithm. Default is [aes256-gcm-siv].
    #[clap(short='E', long="encryption-algorithm", global=true, required=false, arg_enum, default_value="aes256gcmsiv")]
    encryption_algorithm: EncryptionAlgorithmValues,

    /// Encrypts the data AND parts of the main header (e.g. the "description fields", like 'examiner name', 'case number', ...)
    #[clap(short='H', long="encrypted-header", global=true, requires("encryption-password"))]
    encrypted_header: bool,

    /// This option adds an additional hash algorithm to calculate. You can use this option multiple times.
    #[clap(short='d', long="hash-algorithm", global=true, required=false, arg_enum, multiple_values=true)]
    hash_algorithm: Vec<HashAlgorithmValues>,

    /// Sign all data with an autogenerated or given secret EdDSA key. You have to set, if only the hash values should be signed, or if every chunk should be signed.
    #[clap(short='S', long="sign-data", global=true, arg_enum, default_value="none")]
    sign_data: SignatureFlagValues,

    /// Your secret EdDSA key, base64 formatted.
    #[clap(short='k', long="eddsa-keypair", global=true, required=false)]
    sign_keypair: Option<String>,

    /// The case number.
    #[clap(short='c', long="case-number", global=true, required=false)]
    case_number: Option<String>,

    /// The evidence number.
    #[clap(short='e', long="evidence-number", global=true, required=false)]
    evidence_number: Option<String>,

    /// Examiner's name.
    #[clap(short='x', long="examiner-name", global=true, required=false)]
    examiner_name: Option<String>,

    /// Some notes.
    #[clap(short='n', long="notes", global=true, required=false)]
    notes: Option<String>,

    #[clap(subcommand)]
    command: Commands,
}

#[derive(Subcommand)]
enum Commands {
    /// acquire a physical image
    #[clap(arg_required_else_help=true)]
    Physical {
        /// The input file. This should be your device to dump. This field is REQUIRED.
        #[clap(short='i', long="inputfile", required=true)]
        inputfile: PathBuf,

        /// The the name/path of the output-file WITHOUT file extension. E.g. \"/home/ph0llux/sda_dump\". File extension will be added automatically. This field is REQUIRED.
        #[clap(short='o', long="outputfile", global=true, required=false)]
        outputfile: String,
    },
    /// acquire logical folder
    #[clap(arg_required_else_help=true)]
    Logical {
        /// The input folders. You can use this option multiple times. This field is REQUIRED.
        #[clap(short='i', long="inputfiles", required=true, multiple_values=true)]
        inputfiles: Vec<PathBuf>,

        /// The the name/path of the output-file WITHOUT file extension. E.g. \"/home/ph0llux/sda_dump\". File extension will be added automatically. This field is REQUIRED.
        #[clap(short='o', long="outputfile", global=true, required=false)]
        outputfile: String,
    },

    /// extends an existing zff file
    #[clap(arg_required_else_help=true)]
    Extend {
        /// Your zXX files, which should be extended.
        #[clap(short='a', long="append", global=true, multiple_values=true)]
        append_files: Vec<PathBuf>,

        #[clap(subcommand)]
        extend_command: ExtendSubcommands,
    },
}

#[derive(Subcommand)]
enum ExtendSubcommands {
    /// acquire a physical image. 
    #[clap(arg_required_else_help=true)]
    Physical {
        /// The input file. This should be your device to dump. This field is REQUIRED.
        #[clap(short='i', long="inputfile", required=true)]
        inputfile: PathBuf,
    },
    /// acquire logical folder
    #[clap(arg_required_else_help=true)]
    Logical {
        /// The input folders. You can use this option multiple times. This field is REQUIRED.
        #[clap(short='i', long="inputfiles", required=true)]
        inputfiles: Vec<PathBuf>,
    },
}

#[derive(ArgEnum, Clone)]
enum HashAlgorithmValues {
    Blake2b512,
    SHA256,
    SHA512,
    SHA3_256,
}

#[derive(ArgEnum, Clone)]
enum CompressionAlgorithmValues {
    /// No compression is used
    None,
    /// The zstd algorithm
    Zstd,
    /// The lz4 algorithm
    Lz4,
}

#[derive(ArgEnum, Clone)]
enum PasswordKdfValues {
    Pbkdf2Sha256Aes128,
    Pbkdf2Sha256Aes256,
    ScryptAes128,
    ScryptAes256,
}

#[derive(ArgEnum, Clone)]
enum EncryptionAlgorithmValues {
    AES128GCMSIV,
    AES256GCMSIV,
}

#[derive(ArgEnum, Clone)]
enum ChunkSize {
    #[clap(name="256B")]
    CS8,
    #[clap(name="512B")]
    CS9,
    #[clap(name="1KB")]
    CS10,
    #[clap(name="2KB")]
    CS11,
    #[clap(name="4KB")]
    CS12,
    #[clap(name="8KB")]
    CS13,
    #[clap(name="16KB")]
    CS14,
    #[clap(name="32KB")]
    CS15,
    #[clap(name="64KB")]
    CS16,
    #[clap(name="128KB")]
    CS17,
    #[clap(name="256KB")]
    CS18,
    #[clap(name="512KB")]
    CS19,
    #[clap(name="1MB")]
    CS20,
}

impl ChunkSize {
    fn get_size(&self) -> u8 {
        match self {
            ChunkSize::CS8 => 8,
            ChunkSize::CS9 => 9,
            ChunkSize::CS10 => 10,
            ChunkSize::CS11 => 11,
            ChunkSize::CS12 => 12,
            ChunkSize::CS13 => 13,
            ChunkSize::CS14 => 14,
            ChunkSize::CS15 => 15,
            ChunkSize::CS16 => 16,
            ChunkSize::CS17 => 17,
            ChunkSize::CS18 => 18,
            ChunkSize::CS19 => 19,
            ChunkSize::CS20 => 20,
        }
    }
}

#[derive(ArgEnum, Clone)]
enum SignatureFlagValues {
    #[clap(name="none")]
    NoSignatures,
    #[clap(name="hash_value_signature_only")]
    HashValueSignatureOnly,
    #[clap(name="per_chunk_signatures")]
    PerChunkSignatures,
}

fn signer(args: &Cli) -> Option<Keypair> {
    if let SignatureFlagValues::NoSignatures = args.sign_data { return None }

    match &args.sign_keypair {
        None => Some(Signature::new_keypair()),
        Some(value) => match Signature::new_keypair_from_base64(value.trim()) {
            Ok(keypair) => Some(keypair),
            Err(_) => {
                println!("{}", ERROR_PARSE_KEY);
                exit(EXIT_STATUS_ERROR);
            }
        }
    }
}

fn compression_header(args: &Cli) -> CompressionHeader {
    let compression_algorithm = match args.compression_algorithm {
        CompressionAlgorithmValues::None => CompressionAlgorithm::None,
        CompressionAlgorithmValues::Zstd => CompressionAlgorithm::Zstd,
        CompressionAlgorithmValues::Lz4 => CompressionAlgorithm::Lz4,
    };
    if args.compression_level > 9 {
        println!("error: Invalid value for '--compression-level <COMPRESSION_LEVEL>': number <{}> too large to fit in target type. (Possible values are 1-9)", args.compression_level);
        exit(EXIT_STATUS_ERROR);    
    } else if args.compression_level  < 1 {
        println!("error: Invalid value for '--compression-level <COMPRESSION_LEVEL>': number <{}> too small to fit in target type. (Possible values are 1-9)", args.compression_level);
        exit(EXIT_STATUS_ERROR);
    }
    CompressionHeader::new(DEFAULT_HEADER_VERSION_COMPRESSION_HEADER, compression_algorithm, args.compression_level, args.compression_threshold)
}

/// returns the encryption header and the encryption key.
fn encryption_header(args: &Cli) -> Option<(EncryptionHeader, Vec<u8>)> {
    let password = args.encryption_password.as_ref()?;
    let (kdf, pbes) = match args.password_kdf {
        PasswordKdfValues::Pbkdf2Sha256Aes128 => (KDFScheme::PBKDF2SHA256, PBEScheme::AES128CBC),
        PasswordKdfValues::Pbkdf2Sha256Aes256 => (KDFScheme::PBKDF2SHA256, PBEScheme::AES256CBC),
        PasswordKdfValues::ScryptAes128 => (KDFScheme::Scrypt, PBEScheme::AES128CBC),
        PasswordKdfValues::ScryptAes256 => (KDFScheme::Scrypt, PBEScheme::AES256CBC),
    };
    let encryption_algorithm = match args.encryption_algorithm {
        EncryptionAlgorithmValues::AES128GCMSIV => EncryptionAlgorithm::AES128GCMSIV,
        EncryptionAlgorithmValues::AES256GCMSIV => EncryptionAlgorithm::AES256GCMSIV,
    };
    let encryption_key = match encryption_algorithm {
        EncryptionAlgorithm::AES128GCMSIV => Encryption::gen_random_key(128),
        EncryptionAlgorithm::AES256GCMSIV => Encryption::gen_random_key(256),
        _ => {
            println!("{}", ERROR_UNKNOWN_ENCRYPTION_ALGORITHM);
            exit(EXIT_STATUS_ERROR)
        },
    };
    let pbe_nonce = Encryption::gen_random_iv();
    let salt = Encryption::gen_random_salt();
    let (pbe_header, encrypted_encryption_key) = match kdf {
        KDFScheme::PBKDF2SHA256 => {
            let iterations = 256000;
            let kdf_parameters = KDFParameters::PBKDF2SHA256Parameters(PBKDF2SHA256Parameters::new(iterations, salt));
            let pbe_header = PBEHeader::new(DEFAULT_HEADER_VERSION_PBE_HEADER, kdf, pbes.clone(), kdf_parameters, pbe_nonce);
            let encrypted_encryption_key = match pbes {
                PBEScheme::AES128CBC => match Encryption::encrypt_pbkdf2sha256_aes128cbc(
                    iterations,
                    &salt,
                    &pbe_nonce,
                    password.trim(),
                    &encryption_key,
                    ) {
                    Ok(val) => val,
                    Err(_) => {
                        println!("{}", ERROR_ENCRYPT_KEY);
                        exit(EXIT_STATUS_ERROR);
                    },
                },
                PBEScheme::AES256CBC => match Encryption::encrypt_pbkdf2sha256_aes256cbc(
                    iterations,
                    &salt,
                    &pbe_nonce,
                    password.trim(),
                    &encryption_key,
                    ) {
                    Ok(val) => val,
                    Err(_) => {
                        println!("{}", ERROR_ENCRYPT_KEY);
                        exit(EXIT_STATUS_ERROR);
                    }
                },
                _ => {
                    println!("{}", ERROR_UNKNOWN_PASSWORD_KDF);
                    exit(EXIT_STATUS_ERROR)
                },
            };
            (pbe_header, encrypted_encryption_key)
        },
        KDFScheme::Scrypt => {
            let logn = SCRYPT_LOGN_RECOMMENDED;
            let r = SCRYPT_R_RECOMMENDED;
            let p = SCRYPT_P_RECOMMENDED;
            let kdf_parameters = KDFParameters::ScryptParameters(ScryptParameters::new(logn, r, p, salt));
            let pbe_header = PBEHeader::new(DEFAULT_HEADER_VERSION_PBE_HEADER, kdf, pbes.clone(), kdf_parameters, pbe_nonce);
            let encrypted_encryption_key = match pbes {
                PBEScheme::AES128CBC => match Encryption::encrypt_scrypt_aes128cbc(logn, r, p, &salt, &pbe_nonce, password.trim(), &encryption_key) {
                    Ok(val) => val,
                    Err(_) => {
                        println!("{}", ERROR_ENCRYPT_KEY);
                        exit(EXIT_STATUS_ERROR);
                    }
                },
                PBEScheme::AES256CBC => match Encryption::encrypt_scrypt_aes256cbc(logn, r, p, &salt, &pbe_nonce, password.trim(), &encryption_key) {
                    Ok(val) => val,
                    Err(_) => {
                        println!("{}", ERROR_ENCRYPT_KEY);
                        exit(EXIT_STATUS_ERROR);
                    }
                },
                _ => {
                    println!("{}", ERROR_UNKNOWN_PASSWORD_KDF);
                    exit(EXIT_STATUS_ERROR)
                },
            };
            (pbe_header, encrypted_encryption_key)
        },
        _ => {
            println!("{}", ERROR_UNKNOWN_PASSWORD_KDF);
            exit(EXIT_STATUS_ERROR)
        },
    };
    let encryption_header = EncryptionHeader::new(
        DEFAULT_HEADER_VERSION_ENCRYPTION_HEADER,
        pbe_header,
        encryption_algorithm,
        encrypted_encryption_key,
        Encryption::gen_random_header_nonce()
        );
    Some((encryption_header, encryption_key))
}

fn description_header(args: &Cli) -> DescriptionHeader {
    let mut description_header = DescriptionHeader::new_empty(DEFAULT_HEADER_VERSION_DESCRIPTION_HEADER);
    if let Some(value) = &args.case_number {
        description_header.set_case_number(value);
    };
    if let Some(value) = &args.evidence_number {
        description_header.set_evidence_number(value);
    };
    if let Some(value) = &args.examiner_name {
        description_header.set_examiner_name(value);
    };
    if let Some(value) = &args.notes {
        description_header.set_notes(value);
    };
    description_header
}

fn main() {
    let args = Cli::parse();

    // -- MainHeader:
    let chunk_size = &args.chunk_size.get_size();
    let segment_size = match hrs_parser(&args.segment_size) {
        Some(val) => val,
        None => {
            println!("{}{}", ERROR_UNPARSABLE_SEGMENT_SIZE_VALUE, &args.segment_size);
            exit(EXIT_STATUS_ERROR);
        }
    };
    let unique_segment_identifier: i64 = {
        let mut rng = rand::thread_rng();
        rng.gen()
    };

    let main_header = MainHeader::new(
        DEFAULT_HEADER_VERSION_MAIN_HEADER,
        *chunk_size,
        segment_size,
        unique_segment_identifier);
    // --

    // -- ZffCreator
    // --- object header
    let (encryption_header, encryption_key) = match encryption_header(&args) {
        Some((header, key)) => (Some(header), Some(key)),
        None => (None, None)
    };
    let compression_header = compression_header(&args);
    let signature_flag = match &args.sign_data {
        SignatureFlagValues::NoSignatures => SignatureFlag::NoSignatures,
        SignatureFlagValues::HashValueSignatureOnly => SignatureFlag::HashValueSignatureOnly,
        SignatureFlagValues::PerChunkSignatures => SignatureFlag::PerChunkSignatures,
    };

    let object_header = {
        let object_number = match &args.command {
            Commands::Physical { inputfile: _, outputfile: _ } => INITIAL_OBJECT_NUMBER,
            Commands::Logical { inputfiles: _, outputfile: _ } => INITIAL_OBJECT_NUMBER,
            Commands::Extend { extend_command: _, append_files: _ } => INITIAL_OBJECT_NUMBER,
        };
        let description_header = description_header(&args);
        let object_type = match &args.command {
            Commands::Physical { inputfile: _, outputfile: _ } => ObjectType::Physical,
            Commands::Logical { inputfiles: _, outputfile: _ } => ObjectType::Logical,
            Commands::Extend { extend_command, append_files: _ } => match extend_command {
                ExtendSubcommands::Physical { inputfile: _ } => ObjectType::Physical,
                ExtendSubcommands::Logical { inputfiles: _ } => ObjectType::Logical,
            },
        };
        ObjectHeader::new(DEFAULT_HEADER_VERSION_OBJECT_HEADER, object_number, encryption_header, compression_header, signature_flag, description_header, object_type)
    };

    let mut hash_types = Vec::new();
    for htype in &args.hash_algorithm {
        match htype {
            HashAlgorithmValues::Blake2b512 => hash_types.push(HashType::Blake2b512),
            HashAlgorithmValues::SHA256 => hash_types.push(HashType::SHA256),
            HashAlgorithmValues::SHA512 => hash_types.push(HashType::SHA512),
            HashAlgorithmValues::SHA3_256 => hash_types.push(HashType::SHA3_256),
        }
    }

    let sign_keypair = signer(&args);
    match args.command {
        // Physical acquisition
        Commands::Physical { inputfile, outputfile } => {
            let creator_params = ZffCreatorMetadataParams::with_data(encryption_key, sign_keypair, main_header, args.encrypted_header, args.description_notes);
            let output_filepath = outputfile;
            let input_data = match File::open(&inputfile) {
                Ok(f) => {
                    let mut input_data_map = HashMap::new();
                    input_data_map.insert(object_header, f);
                    input_data_map
                },
                Err(e) => {
                    println!("{}{}\n{}", ERROR_OPEN_INPUT_FILE, inputfile.to_string_lossy(), e);
                    exit(EXIT_STATUS_ERROR);
                }
            };
            let mut zffcreator = match ZffCreator::new(
                                            input_data,
                                            HashMap::new(), 
                                            hash_types,
                                            output_filepath,
                                            creator_params) {
                Ok(zffcreator) => zffcreator,
                Err(e) => {
                    eprintln!("{ERROR_CREATE_OBJECT_ENCODER}{e}");
                    exit(EXIT_STATUS_ERROR);
                }

            };
            match zffcreator.generate_files() {
                Ok(()) => exit(EXIT_STATUS_SUCCESS),
                Err(e) => {
                    println!("{ERROR_GENERATE_FILES}{e}");
                    exit(EXIT_STATUS_ERROR);
                }
            }
        },
        //logical acquisition
        Commands::Logical { inputfiles, outputfile } => {
            let creator_params = ZffCreatorMetadataParams::with_data(encryption_key, sign_keypair, main_header, args.encrypted_header, args.description_notes);
            let output_filepath = outputfile;
            let mut input_data_map = HashMap::new();
            input_data_map.insert(object_header, inputfiles);
            let mut zffcreator = match ZffCreator::new(
                HashMap::<ObjectHeader, std::fs::File>::new(),
                input_data_map,
                hash_types,
                output_filepath,
                creator_params) {
                Ok(zffcreator) => zffcreator,
                Err(e) => {
                    eprintln!("{ERROR_CREATE_OBJECT_ENCODER}{e}");
                    exit(EXIT_STATUS_ERROR);
                }
            };
            for unaccessable_file in zffcreator.unaccessable_files() {
                eprintln!("{WARNING_UNACCESSABLE_LOGICAL_FILE}{unaccessable_file}");
            }
            match zffcreator.generate_files() {
                Ok(()) => exit(EXIT_STATUS_SUCCESS),
                Err(e) => {
                    eprintln!("{ERROR_GENERATE_FILES}{e}");
                    exit(EXIT_STATUS_ERROR);
                }
            }
        },
        //Extension
        Commands::Extend { extend_command, append_files } => {
            let mut zffextender = match extend_command {
                ExtendSubcommands::Physical { inputfile } => {
                    let input_data = match File::open(&inputfile) {
                        Ok(f) => {
                            let mut input_data_map = HashMap::new();
                            input_data_map.insert(object_header, f);
                            input_data_map
                        },
                        Err(e) => {
                            println!("{}{}\n{}", ERROR_OPEN_INPUT_FILE, inputfile.to_string_lossy(), e);
                            exit(EXIT_STATUS_ERROR);
                        }
                    };
                    match ZffExtender::new(append_files, input_data, HashMap::new(), hash_types, encryption_key, sign_keypair, args.encrypted_header) {
                        Ok(extender) => extender,
                        Err(e) => {
                            eprintln!("{e}");
                            exit(EXIT_STATUS_ERROR);
                        }
                    }
                },
                ExtendSubcommands::Logical { inputfiles } => {
                    let mut input_data_map = HashMap::new();
                    input_data_map.insert(object_header, inputfiles);
                    match ZffExtender::new(append_files, HashMap::<ObjectHeader, std::fs::File>::new(), input_data_map, hash_types, encryption_key, sign_keypair, args.encrypted_header) {
                        Ok(extender) => extender,
                        Err(e) => {
                            eprintln!("{e}");
                            exit(EXIT_STATUS_ERROR);
                        }
                    }
                }
            };
            match zffextender.extend() {
                Ok(()) => exit(EXIT_STATUS_SUCCESS),
                Err(e) => {
                    eprintln!("An error occurred while trying to extend the current files: {e}");
                    exit(EXIT_STATUS_ERROR);
                }
            }
        }
    }
}