use std::intrinsics::transmute;
use std::num::Wrapping;
use std::ops::{Add, AddAssign, BitXor, BitXorAssign, Index, IndexMut};

#[derive(Clone, Copy, Debug)]
pub struct ZestBlock {
    pub data: [u64; 64],
}
impl ZestBlock {
    //Blank construction. See "From"
    #[inline(always)]
    pub fn new() -> ZestBlock {
        ZestBlock { data: [0_u64; 64] }
    }
    //Increment Block as if it were a number, rightmost bytes least significant.
    #[inline(always)]
    pub fn increment_block(&mut self) {
        let mut carry: u64 = 1;
        for elem in self.data.iter_mut().rev() {
            *elem = (Wrapping(*elem) + Wrapping(carry)).0;
            carry = (*elem < carry) as u64;
        }
    }
    //Act on data as if it were u8 array.
    #[inline(always)]
    pub fn data_u8_mut(&mut self) -> &mut [u8] {
        unsafe {
            transmute(std::slice::from_raw_parts_mut(
                self.data.as_mut_ptr() as *mut u8,
                self.data.len() * std::mem::size_of::<u64>(),
            ))
        }
    }
    #[inline(always)]
    pub fn data_u8(&self) -> &[u8] {
        unsafe {
            transmute(std::slice::from_raw_parts(
                self.data.as_ptr() as *const u8,
                self.data.len() * std::mem::size_of::<u64>(),
            ))
        }
    }
}

//Addition, u64 wise. Not meant to treat as a single large number,
//But adding seperately each u64 element!
impl Add<ZestBlock> for ZestBlock {
    type Output = ZestBlock;
    #[inline(always)]
    fn add(self, rhs: ZestBlock) -> Self::Output {
        let mut total = ZestBlock::new();
        for (index, chunk) in total.data.iter_mut().enumerate() {
            *chunk = (Wrapping(self[index]) + Wrapping(rhs[index])).0;
        }
        total
    }
}
impl AddAssign<ZestBlock> for ZestBlock {
    #[inline(always)]
    fn add_assign(&mut self, rhs: ZestBlock) {
        for (index, chunk) in self.data.iter_mut().enumerate() {
            *chunk = (Wrapping(*chunk) + Wrapping(rhs[index])).0;
        }
    }
}

//XOR, bitwise.
impl BitXor<ZestBlock> for ZestBlock {
    type Output = ZestBlock;
    #[inline(always)]
    fn bitxor(self, rhs: ZestBlock) -> Self::Output {
        let mut xored = ZestBlock::new();
        for (index, chunk) in xored.data.iter_mut().enumerate() {
            *chunk ^= rhs[index];
        }
        xored
    }
}
impl BitXorAssign<ZestBlock> for ZestBlock {
    #[inline(always)]
    fn bitxor_assign(&mut self, rhs: ZestBlock) {
        for (index, chunk) in self.data.iter_mut().enumerate() {
            *chunk ^= rhs[index];
        }
    }
}

//Indexing
impl Index<usize> for ZestBlock {
    type Output = u64;
    #[inline(always)]
    fn index(&self, index: usize) -> &Self::Output {
        &self.data[index]
    }
}
impl IndexMut<usize> for ZestBlock {
    #[inline(always)]
    fn index_mut(&mut self, index: usize) -> &mut Self::Output {
        &mut self.data[index]
    }
}

//Construction
impl From<[u64; 64]> for ZestBlock {
    #[inline(always)]
    fn from(data: [u64; 64]) -> Self {
        ZestBlock { data }
    }
}
impl From<[u8; 512]> for ZestBlock {
    #[inline(always)]
    fn from(data: [u8; 512]) -> Self {
        let u64_ptr = data.as_ptr() as *const u64;
        // Safety: Cast using same number of bytes.
        let owned_slice = unsafe { std::slice::from_raw_parts(u64_ptr, 64) };
        let converted_data = owned_slice.try_into().unwrap();
        ZestBlock {
            data: converted_data,
        }
    }
}

// Default
impl Default for ZestBlock {
    #[inline(always)]
    fn default() -> Self {
        ZestBlock::new()
    }
}
