mod cli;
use std::process::exit;
use std::process::{Command, Stdio};

use cli::Cli;
use zenv::Zenv;

const HELP: &str = "\
zenv - Dotenv (.env) loader written in rust

USAGE:
    zenv [FLAGS] [OPTIONS] -- <binary> [args]...

FLAGS:
    -h, --help            Prints help information
    -x, --expand          Enable variable expansion

OPTIONS:
    -f, --file            Path to .env file

ARGS:
    <binary>            Command that needs to be executed
    [args]...           Arguments for the command

Examples:
    zenv -f .env -- node index.js
    zenv -f .env -- npm run dev
    zenv -f .env -- terraform apply
";

fn bootstrap() -> Result<i32, String> {
    let args = Cli::parse()?;

    if args.help {
        print!("{}", HELP);
        return Ok(0);
    }

    let fpath = args.path()?;

    let binary = args.binary()?;

    let vars = Zenv::new(fpath.to_owned(), args.expand)
        .parse()
        .map_err(|e| e.to_string())?;

    let mut program = Command::new(&binary)
        .args(&args.bin_args)
        .stdin(Stdio::inherit())
        .stdout(Stdio::inherit())
        .stderr(Stdio::inherit())
        .envs(vars)
        .spawn()
        .map_err(|_| format!("Unable to spawn program - `{}`", binary.to_str().unwrap()))?;

    let code = program
        .wait()
        .map_err(|e| e.to_string())?
        .code()
        .ok_or_else(|| "Failed to grab the exit code".to_string())?;

    Ok(code)
}

fn main() {
    match bootstrap() {
        Ok(code) => exit(code),
        Err(e) => {
            eprintln!("{}", e);
            exit(1)
        }
    };
}
