mod config;
mod pam;
mod yubi;
mod zfs;

use clap::{crate_authors, crate_version, App, Arg};
use core::panic;
use std::{env, process::exit};

use crate::config::{config_mode_file, config_mode_yubi, unlock_zfs_file, unlock_zfs_yubi};
use crate::pam::{pam_mode_file, pam_mode_yubi};

fn main() {
    let arg = App::new("zauthrs")
        .about("zauthrs is a simple program designed as a ZFS 2FA encryption helper for PAM") // Define APP and args
        .author(crate_authors!())
        .version(crate_version!())
        .subcommand(
            App::new("config")
                .about("config mode")
                .arg(
                    Arg::with_name("pass")
                        .short("p")
                        .long("pass")
                        .takes_value(true)
                        .help("<Password>")
                        .required(true),
                )
                .arg(
                    Arg::with_name("yubikey")
                        .long("yubi")
                        .short("y")
                        .help("Use Yubikey HMAC as Second factor.")
                        .required(false)
                        .conflicts_with("pfile")
                        .takes_value(false),
                )
                .arg(
                    Arg::with_name("pfile")
                        .short("f")
                        .long("file")
                        .help("<Keyfile location>")
                        .required(false)
                        .takes_value(true),
                )
                .arg(
                    Arg::with_name("zfs")
                        .short("z")
                        .long("zfs")
                        .help("Unlocks and mounts given dataset using the derived key")
                        .required(false)
                        .takes_value(true),
                ),
        )
        .subcommand(
            App::new("pam")
                .about("PAM mode")
                .arg(
                    Arg::with_name("yubikey")
                        .long("yubi")
                        .short("y")
                        .help("Use Yubikey HMAC as Second factor.")
                        .required(false)
                        .takes_value(false),
                )
                .arg(
                    Arg::with_name("pfile")
                        .short("f")
                        .long("file")
                        .help("<Keyfile location>")
                        .required(false)
                        .takes_value(true)
                        .conflicts_with("yubikey"),
                )
                .arg(
                    Arg::with_name("base-dir")
                        .short("b")
                        .long("base-dir")
                        .takes_value(true)
                        .value_name("base_dir")
                        .required(true)
                        .help("Base home Directory eg. \"zroot/data/home/\""),
                ),
        )
        .get_matches();

    match arg.subcommand() {
        ("config", Some(config_matches)) => {
            if config_matches.is_present("zfs") && config_matches.is_present("yubi") {
                unlock_zfs_yubi(
                    config_matches.value_of("pass").unwrap().to_string(),
                    config_matches.value_of("zfs").unwrap().to_string(),
                )
            } else if config_matches.is_present("zfs") && config_matches.is_present("pfile") {
                unlock_zfs_file(
                    config_matches.value_of("pass").unwrap().to_string(),
                    config_matches.value_of("pfile").unwrap().to_string(),
                    config_matches.value_of("zfs").unwrap().to_string(),
                )
            } else if config_matches.is_present("pfile") {
                config_mode_file(
                    &config_matches.value_of("pass").unwrap().to_string(),
                    &config_matches.value_of("pfile").unwrap().to_string(),
                );
            } else if config_matches.is_present("yubikey") {
                config_mode_yubi(&config_matches.value_of("pass").unwrap().to_string());
            } else {
                print!("Select a 2FA mode with either -y (Yubikey) or -f (File)\n");
                exit(1)
            }
        }
        ("pam", Some(pam_matches)) => {
            if pam_matches.is_present("pfile") {
                pam_mode_file(
                    &pam_matches.value_of("base-dir").unwrap().to_string(),
                    &pam_matches.value_of("pfile").unwrap().to_string(),
                )
            } else if pam_matches.is_present("yubikey") {
                pam_mode_yubi(&pam_matches.value_of("base-dir").unwrap().to_string());
            } else {
                panic!("ERROR: Select a 2FA mode with either -y (Yubikey) or -f (File)\n");
            }
        }
        _ => unreachable!(),
    };
}
