use crate::buttonlogic::*;
use zaplib::*;

define_string_with_filename!(MAIN_SHADER);

#[derive(Clone)]
#[repr(C)]
struct DrawTabClose {
    base: DrawQuad,
    color: Vec4,
    hover: f32,
    down: f32,
}
impl DrawTabClose {
    fn new(cx: &mut Cx) -> Self {
        Self {
            base: DrawQuad::with_slots(
                cx,
                cx.get_shader(StringHash::new(MAIN_SHADER), location_hash!()),
                Vec4::slots() + f32::slots() + f32::slots(),
            )
            .with_draw_depth(1.3),
            color: Default::default(),
            hover: Default::default(),
            down: Default::default(),
        }
    }
}

#[derive(Clone)]
pub struct TabClose {
    bg: DrawTabClose,
    animator: Animator,
}

const ANIM_DEFAULT: Anim = Anim {
    duration: 0.2,
    tracks: &[
        // DrawTabClose::color
        Track::Vec4 { key_frames: &[(1.0, vec4(0.62, 0.62, 0.62, 1.))], ease: Ease::DEFAULT },
        // DrawTabClose::hover
        Track::Float { key_frames: &[(1.0, 0.0)], ease: Ease::DEFAULT },
        // DrawTabClose::down
        Track::Float { key_frames: &[(1.0, 0.0)], ease: Ease::DEFAULT },
    ],
    ..Anim::DEFAULT
};

const ANIM_OVER: Anim = Anim {
    duration: 0.1,
    tracks: &[
        // DrawTabClose::color
        Track::Vec4 { key_frames: &[(0.0, Vec4::all(1.))], ease: Ease::DEFAULT },
        // DrawTabClose::hover
        Track::Float { key_frames: &[(1.0, 1.0)], ease: Ease::DEFAULT },
        // DrawTabClose::down
        Track::Float { key_frames: &[(1.0, 0.0)], ease: Ease::DEFAULT },
    ],
    ..Anim::DEFAULT
};

const ANIM_DOWN: Anim = Anim {
    duration: 0.2,
    tracks: &[
        // DrawTabClose::color
        Track::Vec4 { key_frames: &[(0.0, Vec4::all(1.))], ease: Ease::DEFAULT },
        // DrawTabClose::hover
        Track::Float { key_frames: &[(1.0, 1.0)], ease: Ease::DEFAULT },
        // DrawTabClose::down
        Track::Float { key_frames: &[(0.0, 0.0), (1.0, 0.0)], ease: Ease::DEFAULT },
    ],
    ..Anim::DEFAULT
};

impl TabClose {
    pub fn new(cx: &mut Cx) -> Self {
        Self { bg: DrawTabClose::new(cx), animator: Animator::new(ANIM_DEFAULT) }
    }

    pub fn app_load(cx: &mut Cx) {
        cx.register_shader(
            MAIN_SHADER,
            Some(GEOM_QUAD2D),
            &[STD_SHADER_PRELUDE, DRAWQUAD_SHADER_PRELUDE],
            &code_fragment!(
                r#"
                instance color: vec4;
                instance hover: float;
                instance down: float;

                fn pixel() -> vec4 {
                    let cx = Df::viewport(pos * rect_size);
                    let hover_max: float = (hover * 0.5 + 0.3) * 0.5;
                    let hover_min: float = 1. - hover_max;
                    let c: vec2 = rect_size * 0.5;
                    cx.circle(c.x, c.y, 12.);
                    cx.stroke_keep(#4000,1.);
                    cx.fill(mix(#3332,#555f,hover));
                    cx.rotate(down, c.x, c.y);
                    cx.move_to(c.x * hover_min, c.y * hover_min);
                    cx.line_to(c.x + c.x * hover_max, c.y + c.y * hover_max);
                    cx.move_to(c.x + c.x * hover_max, c.y * hover_min);
                    cx.line_to(c.x * hover_min, c.y + c.y * hover_max);
                    return cx.stroke(color, 1. + hover*0.2);
                    //return df_fill(color);
                }"#
            ),
        );
    }

    fn animate(&mut self, cx: &mut Cx) {
        self.bg.color = self.animator.get_vec4(0);
        self.animator.get_vec4(0).write_shader_value(cx, self.bg.base.area(), "color");
        self.bg.hover = self.animator.get_float(1);
        self.animator.get_float(1).write_shader_value(cx, self.bg.base.area(), "hover");
        self.bg.down = self.animator.get_float(2);
        self.animator.get_float(2).write_shader_value(cx, self.bg.base.area(), "down");
    }

    pub fn handle_tab_close(&mut self, cx: &mut Cx, event: &mut Event) -> ButtonEvent {
        if self.animator.handle_animator(cx, event) {
            self.animate(cx);
        }

        match event.hits(
            cx,
            self.bg.base.area(),
            HitOpt { margin: Some(Margin { l: 5., t: 5., r: 5., b: 5. }), ..Default::default() },
        ) {
            Event::FingerDown(_fe) => {
                self.animator.play_anim(cx, ANIM_DOWN);
                cx.set_down_mouse_cursor(MouseCursor::Hand);
                return ButtonEvent::Down;
            }
            Event::FingerHover(fe) => {
                cx.set_hover_mouse_cursor(MouseCursor::Hand);
                match fe.hover_state {
                    HoverState::In => {
                        if fe.any_down {
                            self.animator.play_anim(cx, ANIM_DOWN)
                        } else {
                            self.animator.play_anim(cx, ANIM_OVER)
                        }
                    }
                    HoverState::Out => self.animator.play_anim(cx, ANIM_DEFAULT),
                    _ => (),
                }
            }
            Event::FingerUp(fe) => {
                if fe.is_over {
                    if fe.input_type.has_hovers() {
                        self.animator.play_anim(cx, ANIM_OVER)
                    } else {
                        self.animator.play_anim(cx, ANIM_DEFAULT)
                    }
                    return ButtonEvent::Clicked;
                } else {
                    self.animator.play_anim(cx, ANIM_DEFAULT);
                    return ButtonEvent::Up;
                }
            }
            _ => (),
        };
        ButtonEvent::None
    }

    pub fn draw_tab_close(&mut self, cx: &mut Cx) {
        if self.animator.process_animator(cx) {
            self.animate(cx);
        }

        self.bg.base.draw_quad_rel(cx, Rect { pos: vec2(1.0, 1.0), size: vec2(25.0, 25.0) });
    }
}
