use zaplib::*;

define_string_with_filename!(MAIN_SHADER);

#[derive(Clone)]
pub struct Tab {
    bg: DrawTab,
    text: DrawText,
    pub label: String,
    animator: Animator,
    draw_depth: f32,
    pub abs_origin: Option<Vec2>,
    is_selected: bool,
    is_focussed: bool,
    close_anim_rect: Rect,
    is_down: bool,
    is_drag: bool,
    close_signal: Signal,
}

#[derive(Clone)]
#[repr(C)]
struct DrawTab {
    base: DrawQuad,
    color: Vec4,
    border_color: Vec4,
}
impl DrawTab {
    fn new(cx: &mut Cx) -> Self {
        Self {
            base: DrawQuad::with_slots(
                cx,
                cx.get_shader(StringHash::new(MAIN_SHADER), location_hash!()),
                Vec4::slots() + Vec4::slots(),
            ),
            color: Default::default(),
            border_color: Default::default(),
        }
    }
}

#[derive(Clone, PartialEq)]
pub enum TabEvent {
    None,
    DragMove(FingerMoveEvent),
    DragEnd(FingerUpEvent),
    Closing,
    Close,
    Select,
}

const COLOR_BG_SELECTED: Vec4 = vec4(0.16, 0.16, 0.16, 1.0);
const COLOR_BG_NORMAL: Vec4 = vec4(0.2, 0.2, 0.2, 1.0);

const COLOR_TEXT_SELECTED_FOCUS: Vec4 = Vec4::all(1.);
const COLOR_TEXT_DESELECTED_FOCUS: Vec4 = vec4(0.62, 0.62, 0.62, 1.0);
const COLOR_TEXT_SELECTED_DEFOCUS: Vec4 = vec4(0.62, 0.62, 0.62, 1.0);
const COLOR_TEXT_DESELECTED_DEFOCUS: Vec4 = vec4(0.51, 0.51, 0.51, 1.0);

const ANIM_DESELECTED_DEFOCUS: Anim = Anim {
    duration: 0.05,
    tracks: &[
        // DrawTab::color
        Track::Vec4 { ease: Ease::Lin, key_frames: &[(1.0, COLOR_BG_NORMAL)] },
        // DrawTab::border_color
        Track::Vec4 { ease: Ease::Lin, key_frames: &[(1.0, COLOR_BG_SELECTED)] },
        // DrawText::color
        Track::Vec4 { ease: Ease::Lin, key_frames: &[(1.0, COLOR_TEXT_DESELECTED_DEFOCUS)] },
    ],
    ..Anim::DEFAULT
};

const ANIM_DESELECTED_FOCUS: Anim = Anim {
    duration: 0.05,
    tracks: &[
        // DrawTab::color
        Track::Vec4 { ease: Ease::Lin, key_frames: &[(1.0, COLOR_BG_NORMAL)] },
        // DrawTab::border_color
        Track::Vec4 { ease: Ease::Lin, key_frames: &[(1.0, COLOR_BG_SELECTED)] },
        // DrawText::color
        Track::Vec4 { ease: Ease::Lin, key_frames: &[(1.0, COLOR_TEXT_DESELECTED_FOCUS)] },
    ],
    ..Anim::DEFAULT
};

const ANIM_SELECTED_DEFOCUS: Anim = Anim {
    duration: 0.05,
    tracks: &[
        // DrawTab::color
        Track::Vec4 { ease: Ease::Lin, key_frames: &[(1.0, COLOR_BG_SELECTED)] },
        // DrawTab::border_color
        Track::Vec4 { ease: Ease::Lin, key_frames: &[(1.0, COLOR_BG_SELECTED)] },
        // DrawText::color
        Track::Vec4 { ease: Ease::Lin, key_frames: &[(1.0, COLOR_TEXT_SELECTED_DEFOCUS)] },
    ],
    ..Anim::DEFAULT
};

const ANIM_SELECTED_FOCUS: Anim = Anim {
    duration: 0.05,
    tracks: &[
        // DrawTab::color
        Track::Vec4 { ease: Ease::Lin, key_frames: &[(1.0, COLOR_BG_SELECTED)] },
        // DrawTab::border_color
        Track::Vec4 { ease: Ease::Lin, key_frames: &[(1.0, COLOR_BG_SELECTED)] },
        // DrawText::color
        Track::Vec4 { ease: Ease::Lin, key_frames: &[(1.0, COLOR_TEXT_SELECTED_FOCUS)] },
    ],
    ..Anim::DEFAULT
};

impl Tab {
    pub fn new(cx: &mut Cx) -> Self {
        Self {
            label: "Tab".to_string(),
            //is_closeable: true,
            draw_depth: 0.,
            bg: DrawTab::new(cx),
            //tab_close: TabClose::new(cx),
            text: DrawText::new(cx, location_hash!()).with_draw_depth(0.1),
            animator: Animator::new(ANIM_DESELECTED_DEFOCUS),
            abs_origin: None,
            is_selected: false,
            is_focussed: false,
            is_down: false,
            is_drag: false,
            close_anim_rect: Rect::default(),
            // TODO(JP): This is really terrible, but we have to set the [`Signal`] later,
            // because [`Elements`] just clones this new tab over and over, and so it would
            // otherwise get the same `signal_id` multiple times. We should really get rid
            // of [`Elements`]; it's not a great abstraction!
            close_signal: Signal { signal_id: 0 },
        }
    }

    pub fn with_draw_depth(self, draw_depth: f32) -> Self {
        Self { draw_depth, ..self }
    }

    pub fn app_load(cx: &mut Cx) {
        cx.register_shader(
            MAIN_SHADER,
            Some(GEOM_QUAD2D),
            &[STD_SHADER_PRELUDE, DRAWQUAD_SHADER_PRELUDE],
            &code_fragment!(
                r#"
                instance color: vec4;
                instance border_color: vec4;
                const border_width: float = 1.0;

                fn pixel() -> vec4 {
                    let cx = Df::viewport(pos * rect_size);
                    cx.rect(-1., -1., rect_size.x + 2., rect_size.y + 2.);
                    cx.fill(color);
                    cx.move_to(rect_size.x, 0.);
                    cx.line_to(rect_size.x, rect_size.y);
                    cx.move_to(0., 0.);
                    cx.line_to(0., rect_size.y);
                    return cx.stroke(border_color, 1.);
                }"#
            ),
        )
    }

    fn animate(&mut self, cx: &mut Cx) {
        self.bg.color = self.animator.get_vec4(0);
        self.animator.get_vec4(0).write_shader_value(cx, self.bg.base.area(), "color");
        self.bg.border_color = self.animator.get_vec4(1);
        self.animator.get_vec4(1).write_shader_value(cx, self.bg.base.area(), "border_color");
        self.text.set_color(cx, self.animator.get_vec4(2));
    }

    fn play_anim(&mut self, cx: &mut Cx) {
        if self.is_selected {
            if self.is_focussed {
                self.animator.play_anim(cx, ANIM_SELECTED_FOCUS);
            } else {
                self.animator.play_anim(cx, ANIM_SELECTED_DEFOCUS);
            }
        } else {
            if self.is_focussed {
                self.animator.play_anim(cx, ANIM_DESELECTED_FOCUS);
            } else {
                self.animator.play_anim(cx, ANIM_DESELECTED_DEFOCUS);
            }
        }
    }

    pub fn set_tab_focus(&mut self, cx: &mut Cx, focus: bool) {
        if focus != self.is_focussed {
            self.is_focussed = focus;
            self.play_anim(cx);
        }
    }

    pub fn set_tab_selected(&mut self, cx: &mut Cx, selected: bool) {
        if selected != self.is_selected {
            self.is_selected = selected;
            self.play_anim(cx);
        }
    }

    pub fn set_tab_state(&mut self, cx: &mut Cx, selected: bool, focus: bool) {
        self.is_selected = selected;
        self.is_focussed = focus;
        self.play_anim(cx);
    }

    pub fn close_tab(&self, cx: &mut Cx) {
        if self.close_signal.signal_id != 0 {
            cx.send_signal(self.close_signal, uid!());
        }
    }

    pub fn handle_tab(&mut self, cx: &mut Cx, event: &mut Event) -> TabEvent {
        if self.animator.handle_animator(cx, event) {
            self.animate(cx);
        }

        if let Event::Signal(sig) = &event {
            if let Some(_) = sig.signals.get(&self.close_signal) {
                return TabEvent::Close;
            }
        }

        match event.hits(cx, self.bg.base.area(), HitOpt::default()) {
            Event::FingerDown(_fe) => {
                cx.set_down_mouse_cursor(MouseCursor::Hand);
                self.is_down = true;
                self.is_drag = false;
                self.is_selected = true;
                self.is_focussed = true;
                self.play_anim(cx);
                return TabEvent::Select;
            }
            Event::FingerHover(_fe) => {
                cx.set_hover_mouse_cursor(MouseCursor::Hand);
            }
            Event::FingerUp(fe) => {
                self.is_down = false;

                if self.is_drag {
                    self.is_drag = false;
                    return TabEvent::DragEnd(fe);
                }
            }
            Event::FingerMove(fe) => {
                if !self.is_drag && fe.move_distance() > 50. {
                    //cx.set_down_mouse_cursor(MouseCursor::Hidden);
                    self.is_drag = true;
                }
                if self.is_drag {
                    return TabEvent::DragMove(fe);
                }
            }
            _ => (),
        };
        TabEvent::None
    }

    pub fn get_tab_rect(&mut self, cx: &Cx) -> Rect {
        self.bg.base.area().get_rect_for_first_instance(cx)
    }

    pub fn begin_tab(&mut self, cx: &mut Cx) -> Result<(), ()> {
        // TODO(JP): This is really terrible, but we have to set the [`Signal`] later,
        // because [`Elements`] just clones this new tab over and over, and so it would
        // otherwise get the same `signal_id` multiple times. We should really get rid
        // of [`Elements`]; it's not a great abstraction!
        if self.close_signal.signal_id == 0 {
            self.close_signal = cx.new_signal();
        }

        if self.animator.process_animator(cx) {
            self.animate(cx);
        }

        self.bg.base.draw_depth = self.draw_depth;
        self.text.info.draw_depth = self.draw_depth;

        let base_layout = Layout {
            align: Align::LEFT_CENTER,
            walk: Walk::wh(Width::Compute, Height::Fix(40.)),
            padding: Padding { l: 16.0, t: 1.0, r: 16.0, b: 0.0 },
            ..Layout::default()
        };
        let layout = if let Some(abs_origin) = self.abs_origin {
            Layout { abs_origin: Some(abs_origin), ..base_layout }
        } else {
            base_layout
        };
        self.bg.base.begin_quad(cx, layout);

        self.text.info.text_style = TEXT_STYLE_NORMAL;
        self.text.draw_text_walk(cx, &self.label);

        cx.turtle_align_y();

        Ok(())
    }

    pub fn end_tab(&mut self, cx: &mut Cx) {
        self.bg.base.end_quad(cx);
    }

    pub fn draw_tab(&mut self, cx: &mut Cx) {
        if self.begin_tab(cx).is_err() {
            return;
        };
        self.end_tab(cx);
    }

    pub fn area(&self) -> Area {
        return self.bg.base.area();
    }

    pub fn selected(&self) -> bool {
        return self.is_selected;
    }
}
