use zaplib::*;

define_string_with_filename!(MAIN_SHADER);

#[derive(Clone)]
pub struct SkyBox {
    cube: DrawCube,
}

impl SkyBox {
    pub fn new(cx: &mut Cx) -> Self {
        Self { cube: DrawCube::new(cx, cx.get_shader(StringHash::new(MAIN_SHADER), location_hash!())) }
    }

    pub fn app_load(cx: &mut Cx) {
        cx.register_shader(
            MAIN_SHADER,
            Some(GEOM_CUBE3D),
            &[STD_SHADER_PRELUDE, GEOM_CUBE3D_SHADER_FIELDS, DRAWCUBE_SHADER_PRELUDE],
            &code_fragment!(
                r#"
                const sky_color: vec4 = #000000;
                const edge_color: vec4 = #111111;
                const floor_color: vec4 = #8;

                fn color_form_id() -> vec4 {
                    if geom_id>4.5 {
                        return #f00;
                    }
                    if geom_id>3.5 {
                        return #0f0;
                    }
                    if geom_id>2.5 {
                        return #00f;
                    }
                    if geom_id>1.5 {
                        return #0ff;
                    }
                    return #f0f;
                }
                varying t:float;
                fn vertex() -> vec4 {

                    let model_view = camera_view * transform ;
                    return camera_projection * (model_view * vec4(
                        geom_pos.x * cube_size.x + cube_pos.x,
                        geom_pos.y * cube_size.y + cube_pos.y,
                        geom_pos.z * cube_size.z + cube_pos.z + draw_zbias,
                        1.
                    ));
                }

                fn pixel() -> vec4 {
                    let x = geom_uv.x;
                    let y = geom_uv.y;
                    // walls
                    let sky = sky_color;
                    let edge = edge_color;
                    if geom_id>4.5 || geom_id > 3.5 || geom_id < 1.5 {
                        return mix(edge, sky, y);
                    }
                    // floor
                    if geom_id>2.5 {
                        let coord = geom_uv * 150.0;
                        let grid = abs(
                            fract(coord - 0.5) - 0.5
                        ) / (abs(dFdx(coord)) + abs(dFdy(coord)));
                        let line = min(grid.x, grid.y);
                        let grid2 = floor_color + 0.4 * vec4(vec3(1.0 - min(line, 1.0)), 1.0);
                        let uv2 = abs(2.0 * geom_uv - 1.0);
                        return mix(grid2, edge, min(max(uv2.x, uv2.y) + 0.7, 1.0));
                    }
                    return sky;
                }"#
            ),
        )
    }

    pub fn draw_sky_box(&mut self, cx: &mut Cx) {
        self.cube.cube_size = vec3(200., 100., 200.);
        self.cube.cube_pos = vec3(0., 50., 0.);
        self.cube.draw_cube(cx);
    }
}
