use zaplib::*;

define_string_with_filename!(MAIN_SHADER);

#[derive(Clone)]
#[repr(C)]
struct DrawScrollShadow {
    base: DrawQuad,
    shadow_top: f32,
}
impl DrawScrollShadow {
    fn new(cx: &mut Cx) -> Self {
        Self {
            base: DrawQuad::with_slots(cx, cx.get_shader(StringHash::new(MAIN_SHADER), location_hash!()), f32::slots()),
            shadow_top: Default::default(),
        }
    }
    fn with_draw_depth(self, depth: f32) -> Self {
        Self { base: self.base.with_draw_depth(depth), ..self }
    }
}

#[derive(Clone)]
pub struct ScrollShadow {
    bg: DrawScrollShadow,
}

const SHADOW_SIZE: f32 = 4.;

impl ScrollShadow {
    pub fn new(cx: &mut Cx) -> Self {
        Self { bg: DrawScrollShadow::new(cx).with_draw_depth(10.) }
    }

    pub fn with_draw_depth(self, depth: f32) -> Self {
        Self { bg: self.bg.with_draw_depth(depth) }
    }

    pub fn app_load(cx: &mut Cx) {
        cx.register_shader(
            MAIN_SHADER,
            Some(GEOM_QUAD2D),
            &[STD_SHADER_PRELUDE, DRAWQUAD_SHADER_PRELUDE],
            &code_fragment!(
                r#"
                instance shadow_top: float;
                varying is_viz: float;

                fn scroll() -> vec2 {
                    if shadow_top > 0.5 {
                        is_viz = clamp(draw_local_scroll.y * 0.1, 0., 1.);
                    }
                    else {
                        is_viz = clamp(draw_local_scroll.x * 0.1, 0., 1.);
                    }
                    return draw_scroll;
                }

                // TODO make the corner overlap properly with a distance field eq.
                fn pixel() -> vec4 {
                    if shadow_top > 0.5 {
                        return mix(vec4(0., 0., 0., is_viz), vec4(0., 0., 0., 0.), pow(geom.y, 0.5));
                    }
                    return mix(vec4(0., 0., 0., is_viz), vec4(0., 0., 0., 0.), pow(geom.x, 0.5));
                }"#
            ),
        );
    }

    pub fn draw_shadow_top(&mut self, cx: &mut Cx) {
        self.draw_shadow_top_at(cx, Rect { pos: vec2(0., 0.), size: vec2(cx.get_width_total(), 0.) });
    }

    pub fn draw_shadow_top_at(&mut self, cx: &mut Cx, rect: Rect) {
        self.bg.shadow_top = 1.0;
        self.bg.base.draw_quad_rel(cx, Rect { pos: rect.pos, size: vec2(rect.size.x, SHADOW_SIZE) });
        self.bg.base.area().get_draw_call(cx).set_scroll_sticky(true, true);
    }

    pub fn draw_shadow_left(&mut self, cx: &mut Cx) {
        self.draw_shadow_left_at(cx, Rect { pos: vec2(0., 0.), size: vec2(0., cx.get_height_total()) });
    }

    pub fn draw_shadow_left_at(&mut self, cx: &mut Cx, rect: Rect) {
        self.bg.shadow_top = 0.0;
        self.bg.base.draw_quad_rel(cx, Rect { pos: rect.pos, size: vec2(SHADOW_SIZE, rect.size.y) });
        self.bg.base.area().get_draw_call(cx).set_scroll_sticky(true, true);
    }
}
