use crate::buttonlogic::*;
use zaplib::*;

define_string_with_filename!(MAIN_SHADER);

#[derive(Clone)]
#[repr(C)]
struct DrawNormalButton {
    base: DrawQuad,
    hover: f32,
    down: f32,
}
impl DrawNormalButton {
    fn new(cx: &mut Cx) -> Self {
        Self {
            base: DrawQuad::with_slots(
                cx,
                cx.get_shader(StringHash::new(MAIN_SHADER), location_hash!()),
                f32::slots() + f32::slots(),
            ),
            hover: Default::default(),
            down: Default::default(),
        }
    }
}

#[derive(Clone)]
pub struct NormalButton {
    button: ButtonLogic,
    bg: DrawNormalButton,
    text: DrawText,
    animator: Animator,
}

const ANIM_DEFAULT: Anim = Anim {
    duration: 0.2,
    tracks: &[
        // DrawNormalButton::hover
        Track::Float { key_frames: &[(1.0, 0.0)], ease: Ease::DEFAULT },
        // DrawNormalButton::down
        Track::Float { key_frames: &[(1.0, 0.0)], ease: Ease::DEFAULT },
        // DrawText::color
        Track::Vec4 { key_frames: &[(1.0, vec4(0.6, 0.6, 0.6, 1.))], ease: Ease::DEFAULT },
    ],
    ..Anim::DEFAULT
};

const ANIM_HOVER: Anim = Anim {
    duration: 0.2,
    tracks: &[
        // DrawNormalButton::hover
        Track::Float { key_frames: &[(0.0, 1.0)], ease: Ease::DEFAULT },
        // DrawNormalButton::down
        Track::Float { key_frames: &[(1.0, 0.0)], ease: Ease::DEFAULT },
        // DrawText::color
        Track::Vec4 { key_frames: &[(1.0, vec4(1., 1., 1., 1.))], ease: Ease::DEFAULT },
    ],
    ..Anim::DEFAULT
};

const ANIM_DOWN: Anim = Anim {
    duration: 0.2,
    tracks: &[
        // DrawNormalButton::hover
        Track::Float { key_frames: &[(0.0, 1.0)], ease: Ease::DEFAULT },
        // DrawNormalButton::down
        Track::Float { key_frames: &[(0.0, 0.0), (1.0, 1.0)], ease: Ease::DEFAULT },
        // DrawText::color
        Track::Vec4 { key_frames: &[(1.0, vec4(0.8, 0.8, 0.8, 1.))], ease: Ease::DEFAULT },
    ],
    ..Anim::DEFAULT
};

impl NormalButton {
    pub fn new(cx: &mut Cx) -> Self {
        Self {
            button: ButtonLogic::default(),
            bg: DrawNormalButton::new(cx),
            text: DrawText::new(cx, location_hash!()),
            animator: Animator::new(ANIM_DEFAULT),
        }
    }

    fn animate(&mut self, cx: &mut Cx) {
        self.animator.get_float(0).write_shader_value(cx, self.bg.base.area(), "hover");
        self.animator.get_float(1).write_shader_value(cx, self.bg.base.area(), "down");
        self.text.set_color(cx, self.animator.get_vec4(2));
    }

    pub fn app_load(cx: &mut Cx) {
        cx.register_shader(
            MAIN_SHADER,
            Some(GEOM_QUAD2D),
            &[STD_SHADER_PRELUDE, DRAWQUAD_SHADER_PRELUDE],
            &code_fragment!(
                r#"
                instance hover: float;
                instance down: float;

                const shadow: float = 3.0;
                const border_radius: float = 2.5;

                fn pixel() -> vec4 {
                    let cx = Df::viewport(pos * rect_size);
                    cx.box(shadow, shadow, rect_size.x - shadow * (1. + down), rect_size.y - shadow * (1. + down), border_radius);
                    cx.blur = 6.0;
                    cx.fill(mix(#0007, #0, hover));
                    cx.blur = 0.001;
                    cx.box(shadow, shadow, rect_size.x - shadow * 2., rect_size.y - shadow * 2., border_radius);
                    return cx.fill(mix(mix(#3, #4, hover), #2a, down));
                }"#
            ),
        );
    }

    pub fn handle_normal_button(&mut self, cx: &mut Cx, event: &mut Event) -> ButtonEvent {
        if self.animator.handle_animator(cx, event) {
            self.animate(cx);
        }
        let animator = &mut self.animator;
        self.button.handle_button_logic(cx, event, self.bg.base.area(), |cx, logic_event, _| match logic_event {
            ButtonLogicEvent::Down => animator.play_anim(cx, ANIM_DOWN),
            ButtonLogicEvent::Default => animator.play_anim(cx, ANIM_DEFAULT),
            ButtonLogicEvent::Over => animator.play_anim(cx, ANIM_HOVER),
        })
    }

    pub fn draw_normal_button(&mut self, cx: &mut Cx, label: &str) {
        if self.animator.process_animator(cx) {
            self.animate(cx);
        }

        self.bg.base.begin_quad(
            cx,
            Layout {
                align: Align::CENTER,
                walk: Walk { width: Width::Compute, height: Height::Compute, margin: Margin::all(1.0) },
                padding: Padding { l: 16.0, t: 12.0, r: 16.0, b: 12.0 },
                ..Layout::default()
            },
        );

        self.text.info.text_style = TEXT_STYLE_NORMAL;
        self.text.draw_text_walk(cx, label);

        self.bg.base.end_quad(cx);
    }
}
