use crate::buttonlogic::*;
use zaplib::*;

define_string_with_filename!(MAIN_SHADER);

#[derive(Clone)]
pub enum FoldOpenState {
    Open,
    Opening(f32),
    Closed,
    Closing(f32),
}

impl FoldOpenState {
    fn get_value(&self) -> f32 {
        match self {
            FoldOpenState::Opening(fac) => 1.0 - *fac,
            FoldOpenState::Closing(fac) => *fac,
            FoldOpenState::Open => 1.0,
            FoldOpenState::Closed => 0.0,
        }
    }
    pub fn is_open(&self) -> bool {
        match self {
            FoldOpenState::Opening(_) => true,
            FoldOpenState::Closing(_) => false,
            FoldOpenState::Open => true,
            FoldOpenState::Closed => false,
        }
    }
    pub fn toggle(&mut self) {
        *self = match self {
            FoldOpenState::Opening(fac) => FoldOpenState::Closing(1.0 - *fac),
            FoldOpenState::Closing(fac) => FoldOpenState::Opening(1.0 - *fac),
            FoldOpenState::Open => FoldOpenState::Closing(1.0),
            FoldOpenState::Closed => FoldOpenState::Opening(1.0),
        };
    }
    pub fn do_open(&mut self) {
        *self = match self {
            FoldOpenState::Opening(fac) => FoldOpenState::Opening(*fac),
            FoldOpenState::Closing(fac) => FoldOpenState::Opening(1.0 - *fac),
            FoldOpenState::Open => FoldOpenState::Open,
            FoldOpenState::Closed => FoldOpenState::Opening(1.0),
        };
    }
    pub fn do_close(&mut self) {
        *self = match self {
            FoldOpenState::Opening(fac) => FoldOpenState::Closing(1.0 - *fac),
            FoldOpenState::Closing(fac) => FoldOpenState::Closing(*fac),
            FoldOpenState::Open => FoldOpenState::Closing(1.0),
            FoldOpenState::Closed => FoldOpenState::Closed,
        };
    }
    pub fn do_time_step(&mut self, mul: f32) -> bool {
        let mut redraw = false;
        *self = match self {
            FoldOpenState::Opening(fac) => {
                redraw = true;
                if *fac < 0.001 {
                    FoldOpenState::Open
                } else {
                    FoldOpenState::Opening(*fac * mul)
                }
            }
            FoldOpenState::Closing(fac) => {
                redraw = true;
                if *fac < 0.001 {
                    FoldOpenState::Closed
                } else {
                    FoldOpenState::Closing(*fac * mul)
                }
            }
            FoldOpenState::Open => FoldOpenState::Open,
            FoldOpenState::Closed => FoldOpenState::Closed,
        };
        redraw
    }
}

#[derive(Clone)]
#[repr(C)]
struct DrawFoldCaption {
    base: DrawQuad,
    hover: f32,
    down: f32,
    open: f32,
}

impl DrawFoldCaption {
    fn new(cx: &mut Cx) -> Self {
        Self {
            base: DrawQuad::with_slots(
                cx,
                cx.get_shader(StringHash::new(MAIN_SHADER), location_hash!()),
                f32::slots() + f32::slots() + f32::slots(),
            )
            .with_draw_depth(0.5),
            hover: Default::default(),
            down: Default::default(),
            open: Default::default(),
        }
    }
}

#[derive(Clone)]
pub struct FoldCaption {
    button: ButtonLogic,
    bg: DrawFoldCaption,
    text: DrawText,
    animator: Animator,
    open_state: FoldOpenState,
}

const ANIM_DEFAULT: Anim = Anim {
    duration: 0.1,
    tracks: &[
        // DrawFoldCaption::hover
        Track::Float { key_frames: &[(1.0, 0.0)], ease: Ease::DEFAULT },
        // DrawFoldCaption::down
        Track::Float { key_frames: &[(1.0, 0.0)], ease: Ease::DEFAULT },
        // DrawText::color
        Track::Vec4 { key_frames: &[(1.0, vec4(0.6, 0.6, 0.6, 1.0))], ease: Ease::DEFAULT },
    ],
    ..Anim::DEFAULT
};

const ANIM_OVER: Anim = Anim {
    duration: 0.1,
    tracks: &[
        // DrawFoldCaption::hover
        Track::Float { key_frames: &[(0.0, 1.0), (1.0, 1.0)], ease: Ease::DEFAULT },
        // DrawFoldCaption::down
        Track::Float { key_frames: &[(1.0, 0.0)], ease: Ease::DEFAULT },
        // DrawText::color
        Track::Vec4 { key_frames: &[(0.0, Vec4::all(1.))], ease: Ease::DEFAULT },
    ],
    ..Anim::DEFAULT
};

const ANIM_DOWN: Anim = Anim {
    duration: 0.2,
    tracks: &[
        // DrawFoldCaption::hover
        Track::Float { key_frames: &[(0.0, 1.0), (1.0, 1.0)], ease: Ease::DEFAULT },
        // DrawFoldCaption::down
        Track::Float { key_frames: &[(1.0, 1.0)], ease: Ease::DEFAULT },
        // DrawText::color
        Track::Vec4 { key_frames: &[(0.0, vec4(0.8, 0.8, 0.8, 1.0))], ease: Ease::DEFAULT },
    ],
    ..Anim::DEFAULT
};

impl FoldCaption {
    pub fn new(cx: &mut Cx) -> Self {
        Self {
            button: ButtonLogic::default(),
            bg: DrawFoldCaption::new(cx),
            text: DrawText::new(cx, location_hash!()).with_draw_depth(1.0),
            open_state: FoldOpenState::Open,
            animator: Animator::new(ANIM_DEFAULT),
        }
    }

    pub fn app_load(cx: &mut Cx) {
        cx.register_shader(
            MAIN_SHADER,
            Some(GEOM_QUAD2D),
            &[STD_SHADER_PRELUDE, DRAWQUAD_SHADER_PRELUDE],
            &code_fragment!(
                r#"
                instance hover: float;
                instance down: float;
                instance open: float;

                const shadow: float = 3.0;
                const border_radius: float = 2.5;

                fn pixel() -> vec4 {
                    let sz = 3.;
                    let c = vec2(5.0,0.5*rect_size.y);
                    let df = Df::viewport(pos * rect_size);
                    df.clear(#2);
                    // we have 3 points, and need to rotate around its center
                    df.rotate(open*0.5*PI+0.5*PI, c.x, c.y);
                    df.move_to(c.x - sz, c.y + sz);
                    df.line_to(c.x, c.y - sz);
                    df.line_to(c.x + sz, c.y + sz);
                    df.close_path();
                    df.fill(mix(#a,#f,hover));

                    return df.result;
                }"#
            ),
        );
    }

    fn animate(&mut self, cx: &mut Cx) {
        self.bg.hover = self.animator.get_float(0);
        self.animator.get_float(0).write_shader_value(cx, self.bg.base.area(), "hover");
        self.bg.down = self.animator.get_float(1);
        self.animator.get_float(1).write_shader_value(cx, self.bg.base.area(), "down");
        self.text.set_color(cx, self.animator.get_vec4(2));
    }

    pub fn handle_fold_caption(&mut self, cx: &mut Cx, event: &mut Event) -> ButtonEvent {
        if self.animator.handle_animator(cx, event) {
            self.animate(cx);
        }

        let animator = &mut self.animator;
        let open_state = &mut self.open_state;
        self.button.handle_button_logic(cx, event, self.bg.base.area(), |cx, logic_event, area| {
            match logic_event {
                ButtonLogicEvent::Down => {
                    // lets toggle our anim state
                    open_state.toggle();
                    cx.redraw_child_area(area);
                    animator.play_anim(cx, ANIM_DOWN);
                }
                ButtonLogicEvent::Default => animator.play_anim(cx, ANIM_DEFAULT),
                ButtonLogicEvent::Over => animator.play_anim(cx, ANIM_OVER),
            }
        })
    }

    pub fn begin_fold_caption(&mut self, cx: &mut Cx) -> f32 {
        if self.animator.process_animator(cx) {
            self.animate(cx);
        }

        let open_value = self.open_state.get_value();
        self.bg.open = open_value;
        self.bg.base.begin_quad(
            cx,
            Layout {
                align: Align::RIGHT_CENTER,
                walk: Walk { width: Width::Fill, height: Height::Compute, margin: Margin::all(1.0) },
                padding: Padding { l: 14.0, t: 8.0, r: 14.0, b: 8.0 },
                ..Layout::default()
            },
        );

        if self.open_state.do_time_step(0.6) {
            cx.redraw_child_area(self.bg.base.area());
        }

        open_value
    }

    pub fn end_fold_caption(&mut self, cx: &mut Cx, label: &str) {
        cx.change_turtle_align_x_cab(0.0);
        cx.reset_turtle_pos();

        self.text.info.text_style = TEXT_STYLE_NORMAL;
        let wleft = cx.get_width_left();
        self.text.info.wrapping = Wrapping::Ellipsis(wleft - 10.0);
        self.text.draw_text_walk(cx, label);

        self.bg.base.end_quad(cx);
    }
}
