use crate::buttonlogic::*;
use crate::desktopbutton::*;
use zaplib::*;

#[derive(Clone)]
pub struct DesktopWindow {
    pub window: Window,
    pub pass: Pass,
    pub clear_color: Vec4,
    pub color_texture: Texture,
    pub depth_texture: Texture,
    pub caption_view: View, // we have a root view otherwise is_overlay subviews can't attach topmost
    pub main_view: View,    // we have a root view otherwise is_overlay subviews can't attach topmost
    pub inner_view: View,
    pub inner_layout: Layout,
    //pub caption_bg_color: ColorId,
    pub min_btn: DesktopButton,
    pub max_btn: DesktopButton,
    pub close_btn: DesktopButton,
    pub xr_btn: DesktopButton,
    pub fullscreen_btn: DesktopButton,
    pub caption_text: DrawText,
    pub caption_bg: DrawColor,
    pub caption_size: Vec2,
    pub caption: String,

    pub default_menu: Menu,

    last_menu: Option<Menu>,

    // testing
    pub inner_over_chrome: bool,
}

#[derive(Clone, PartialEq)]
pub enum DesktopWindowEvent {
    EventForOtherWindow,
    WindowClosed,
    WindowGeomChange(WindowGeomChangeEvent),
    None,
}

impl DesktopWindow {
    pub fn new(cx: &mut Cx) -> Self {
        Self {
            window: Window::new(cx),
            pass: Pass::default(),
            clear_color: Vec4::color("1e"),
            color_texture: Texture::new(cx),
            depth_texture: Texture::new(cx),
            main_view: View::new(),
            caption_view: View::new(),
            inner_view: View::new(),
            inner_layout: Layout::default(),
            min_btn: DesktopButton::new(cx),
            max_btn: DesktopButton::new(cx),
            close_btn: DesktopButton::new(cx),
            xr_btn: DesktopButton::new(cx),
            fullscreen_btn: DesktopButton::new(cx),

            default_menu: Menu::main(vec![Menu::sub("App", vec![Menu::item("Quit App", Cx::command_quit())])]),
            caption_text: DrawText::new(cx, location_hash!()),
            //caption_bg_color: Color_bg_selected_over::id(cx),
            caption_bg: DrawColor::new(cx, DrawColorShader::Default(location_hash!())),
            caption_size: Vec2::default(),
            caption: "Bigedit".to_string(),
            inner_over_chrome: false,
            last_menu: None,
        }
    }

    pub fn with_window(self, window: Window) -> Self {
        Self { window, ..self }
    }
    pub fn with_caption(self, caption: &str) -> Self {
        Self { caption: caption.to_string(), ..self }
    }

    pub fn with_inner_layout(self, inner_layout: Layout) -> Self {
        Self { inner_layout, ..self }
    }

    pub fn handle_desktop_window(&mut self, cx: &mut Cx, event: &mut Event) -> DesktopWindowEvent {
        //self.main_view.handle_scroll_bars(cx, event);
        //self.inner_view.handle_scroll_bars(cx, event);
        if let ButtonEvent::Clicked = self.xr_btn.handle_button(cx, event) {
            if self.window.xr_is_presenting(cx) {
                self.window.xr_stop_presenting(cx);
            } else {
                self.window.xr_start_presenting(cx);
            }
        }

        if let ButtonEvent::Clicked = self.fullscreen_btn.handle_button(cx, event) {
            if self.window.is_fullscreen(cx) {
                self.window.normal_window(cx);
            } else {
                self.window.fullscreen_window(cx);
            }
        }
        if let ButtonEvent::Clicked = self.min_btn.handle_button(cx, event) {
            self.window.minimize_window(cx);
        }
        if let ButtonEvent::Clicked = self.max_btn.handle_button(cx, event) {
            if self.window.is_fullscreen(cx) {
                self.window.restore_window(cx);
            } else {
                self.window.maximize_window(cx);
            }
        }
        if let ButtonEvent::Clicked = self.close_btn.handle_button(cx, event) {
            self.window.close_window(cx);
        }
        if let Some(window_id) = self.window.window_id {
            let is_for_other_window = match event {
                Event::WindowCloseRequested(ev) => ev.window_id != window_id,
                Event::WindowClosed(ev) => {
                    if ev.window_id == window_id {
                        return DesktopWindowEvent::WindowClosed;
                    }
                    true
                }
                Event::WindowGeomChange(ev) => {
                    if ev.window_id == window_id {
                        return DesktopWindowEvent::WindowGeomChange(ev.clone());
                    }
                    true
                }
                Event::WindowDragQuery(dq) => {
                    if dq.window_id == window_id && dq.abs.x < self.caption_size.x && dq.abs.y < self.caption_size.y {
                        if dq.abs.x < 50. {
                            dq.response = WindowDragQueryResponse::SysMenu;
                        } else {
                            dq.response = WindowDragQueryResponse::Caption;
                        }
                    }
                    true
                }
                Event::FingerDown(ev) => ev.window_id != window_id,
                Event::FingerMove(ev) => ev.window_id != window_id,
                Event::FingerHover(ev) => ev.window_id != window_id,
                Event::FingerUp(ev) => ev.window_id != window_id,
                Event::FingerScroll(ev) => ev.window_id != window_id,
                _ => false,
            };
            if is_for_other_window {
                DesktopWindowEvent::EventForOtherWindow
            } else {
                DesktopWindowEvent::None
            }
        } else {
            DesktopWindowEvent::None
        }
    }

    pub fn begin_desktop_window(&mut self, cx: &mut Cx, menu: Option<&Menu>) -> ViewRedraw {
        if !self.main_view.view_will_redraw(cx) {
            return Err(());
        }

        self.window.begin_window(cx);
        self.pass.begin_pass(cx);
        self.pass.add_color_texture(cx, self.color_texture, ClearColor::ClearWith(self.clear_color));
        self.pass.set_depth_texture(cx, self.depth_texture, ClearDepth::ClearWith(1.0));

        let _ = self.main_view.begin_view(cx, Layout::default());

        if self.caption_view.begin_view(cx, Layout { walk: Walk::wh(Width::Fill, Height::Compute), ..Layout::default() }).is_ok()
        {
            self.caption_text.info.text_style = TEXT_STYLE_NORMAL;
            self.caption_bg.color = vec4(0.24, 0.24, 0.24, 1.0);

            let process_chrome = match cx.platform_type {
                PlatformType::Linux { custom_window_chrome } => custom_window_chrome,
                _ => true,
            };
            if process_chrome {
                match cx.platform_type {
                    PlatformType::Windows | PlatformType::Unknown | PlatformType::Linux { .. } => {
                        self.caption_bg.quad.begin_quad(
                            cx,
                            Layout {
                                align: Align::RIGHT_TOP,
                                walk: Walk::wh(Width::Fill, Height::Compute),
                                ..Default::default()
                            },
                        );

                        // we need to draw the window menu here.
                        if let Some(_menu) = menu {
                            // lets draw the thing, check with the clone if it changed
                            // then draw it
                        }

                        self.min_btn.draw_desktop_button(cx, DesktopButtonType::WindowsMin);
                        if self.window.is_fullscreen(cx) {
                            self.max_btn.draw_desktop_button(cx, DesktopButtonType::WindowsMaxToggled);
                        } else {
                            self.max_btn.draw_desktop_button(cx, DesktopButtonType::WindowsMax);
                        }
                        self.close_btn.draw_desktop_button(cx, DesktopButtonType::WindowsClose);

                        // change alignment
                        cx.change_turtle_align_x_cab(0.5); //Align::CENTER);
                        cx.compute_turtle_height();
                        cx.change_turtle_align_y_cab(0.5); //Align::CENTER);
                        cx.reset_turtle_pos();
                        cx.move_turtle(50., 0.);
                        // we need to store our caption rect somewhere.
                        self.caption_size = Vec2 { x: cx.get_width_left(), y: cx.get_height_left() };
                        self.caption_text.draw_text_walk(cx, &self.caption);
                        self.caption_bg.quad.end_quad(cx);
                        cx.turtle_new_line();
                    }

                    PlatformType::OSX => {
                        // mac still uses the built in buttons, TODO, replace that.
                        if let Some(menu) = menu {
                            cx.update_menu(menu);
                        } else {
                            cx.update_menu(&self.default_menu);
                        }
                        self.caption_bg.quad.begin_quad(
                            cx,
                            Layout { align: Align::CENTER, walk: Walk::wh(Width::Fill, Height::Fix(22.)), ..Default::default() },
                        );
                        self.caption_size = Vec2 { x: cx.get_width_left(), y: cx.get_height_left() };
                        self.caption_text.draw_text_walk(cx, &self.caption);
                        self.caption_bg.quad.end_quad(cx);
                        cx.turtle_new_line();
                    }
                    PlatformType::Web { .. } => {
                        if self.window.is_fullscreen(cx) {
                            // put a bar at the top
                            self.caption_bg.quad.begin_quad(
                                cx,
                                Layout {
                                    align: Align::CENTER,
                                    walk: Walk::wh(Width::Fill, Height::Fix(22.)),
                                    ..Default::default()
                                },
                            );
                            self.caption_bg.quad.end_quad(cx);
                            cx.turtle_new_line();
                        }
                    }
                }
            }
            self.caption_view.end_view(cx);
        }
        cx.turtle_new_line();

        if self.inner_over_chrome {
            let _ = self.inner_view.begin_view(cx, Layout { abs_origin: Some(Vec2::default()), ..self.inner_layout });
        } else {
            let _ = self.inner_view.begin_view(cx, self.inner_layout);
        }
        Ok(())
    }

    pub fn end_desktop_window(&mut self, cx: &mut Cx) {
        self.inner_view.end_view(cx);
        // lets draw a VR button top right over the UI.
        // window fullscreen?

        // only support fullscreen on web atm
        if !cx.platform_type.is_desktop() && !self.window.is_fullscreen(cx) {
            cx.reset_turtle_pos();
            cx.move_turtle(cx.get_width_total() - 50.0, 0.);
            self.fullscreen_btn.draw_desktop_button(cx, DesktopButtonType::Fullscreen);
        }

        if self.window.xr_can_present(cx) {
            // show a switch-to-VRMode button
            cx.reset_turtle_pos();
            cx.move_turtle(cx.get_width_total() - 100.0, 0.);
            self.xr_btn.draw_desktop_button(cx, DesktopButtonType::XRMode);
        }

        self.main_view.end_view(cx);

        self.pass.end_pass(cx);

        self.window.end_window(cx);
    }
}
