use crate::math::*;

pub(crate) fn hex_bytes_to_vec4(bytes: &[u8]) -> Result<Vec4, ()> {
    fn hex_to_int(c: u32) -> Result<u32, ()> {
        if (48..=57).contains(&c) {
            return Ok(c - 48);
        }
        if (65..=70).contains(&c) {
            return Ok(c - 65 + 10);
        }
        if (97..=102).contains(&c) {
            return Ok(c - 97 + 10);
        }
        Err(())
    }

    match bytes.len() {
        1 => {
            // #w
            let val = hex_to_int(bytes[0] as u32)? as f32 / 15.0;
            return Ok(vec4(val, val, val, 1.0));
        }
        2 => {
            //#ww
            let w = ((hex_to_int(bytes[0] as u32)? << 4) + hex_to_int(bytes[1] as u32)?) as f32 / 255.0;
            return Ok(vec4(w, w, w, 1.0));
        }
        3 => {
            // #rgb
            let r = hex_to_int(bytes[0] as u32)? as f32 / 15.0;
            let g = hex_to_int(bytes[1] as u32)? as f32 / 15.0;
            let b = hex_to_int(bytes[2] as u32)? as f32 / 15.0;
            return Ok(vec4(r, g, b, 1.0));
        }
        4 => {
            // #rgba
            let r = hex_to_int(bytes[0] as u32)? as f32 / 15.0;
            let g = hex_to_int(bytes[1] as u32)? as f32 / 15.0;
            let b = hex_to_int(bytes[2] as u32)? as f32 / 15.0;
            let a = hex_to_int(bytes[3] as u32)? as f32 / 15.0;
            return Ok(vec4(r, g, b, a));
        }
        6 => {
            // #rrggbb
            let r = ((hex_to_int(bytes[0] as u32)? << 4) + hex_to_int(bytes[1] as u32)?) as f32 / 255.0;
            let g = ((hex_to_int(bytes[2] as u32)? << 4) + hex_to_int(bytes[3] as u32)?) as f32 / 255.0;
            let b = ((hex_to_int(bytes[4] as u32)? << 4) + hex_to_int(bytes[5] as u32)?) as f32 / 255.0;
            return Ok(vec4(r, g, b, 1.0));
        }
        8 => {
            // #rrggbbaa
            let r = ((hex_to_int(bytes[0] as u32)? << 4) + hex_to_int(bytes[1] as u32)?) as f32 / 255.0;
            let g = ((hex_to_int(bytes[2] as u32)? << 4) + hex_to_int(bytes[3] as u32)?) as f32 / 255.0;
            let b = ((hex_to_int(bytes[4] as u32)? << 4) + hex_to_int(bytes[5] as u32)?) as f32 / 255.0;
            let a = ((hex_to_int(bytes[6] as u32)? << 4) + hex_to_int(bytes[7] as u32)?) as f32 / 255.0;
            return Ok(vec4(r, g, b, a));
        }
        _ => (),
    }
    Err(())
}

pub(crate) fn color_name_to_vec4(name: &str) -> Result<Vec4, ()> {
    Ok(match name.to_lowercase().as_ref() {
        "aliceblue" => vec4(0.941_176_5, 0.972_549, 1.0, 1.0),
        "antiquewhite" => vec4(0.980_392_16, 0.921_568_63, 0.843_137_26, 1.0),
        "aqua" => vec4(0.0, 1.0, 1.0, 1.0),
        "aquamarine" => vec4(0.498_039_22, 1.0, 0.831_372_56, 1.0),
        "azure" => vec4(0.941_176_5, 1.0, 1.0, 1.0),
        "beige" => vec4(0.960_784_3, 0.960_784_3, 0.862_745_1, 1.0),
        "bisque" => vec4(1.0, 0.894_117_65, 0.768_627_46, 1.0),
        "black" => vec4(0.0, 0.0, 0.0, 1.0),
        "blanchedalmond" => vec4(1.0, 0.921_568_63, 0.803_921_6, 1.0),
        "blue" => vec4(0.0, 0.0, 1.0, 1.0),
        "blueviolet" => vec4(0.541_176_5, 0.168_627_46, 0.886_274_5, 1.0),
        "brown" => vec4(0.647_058_84, 0.164_705_89, 0.164_705_89, 1.0),
        "burlywood" => vec4(0.870_588_24, 0.721_568_64, 0.529_411_8, 1.0),
        "cadetblue" => vec4(0.372_549_03, 0.619_607_87, 0.627_451, 1.0),
        "chartreuse" => vec4(0.498_039_22, 1.0, 0.0, 1.0),
        "chocolate" => vec4(0.823_529_4, 0.411_764_7, 0.117_647_06, 1.0),
        "coral" => vec4(1.0, 0.498_039_22, 0.313_725_5, 1.0),
        "cornflower" => vec4(0.392_156_87, 0.584_313_75, 0.929_411_77, 1.0),
        "cornsilk" => vec4(1.0, 0.972_549, 0.862_745_1, 1.0),
        "crimson" => vec4(0.862_745_1, 0.078_431_375, 0.235_294_12, 1.0),
        "cyan" => vec4(0.0, 1.0, 1.0, 1.0),
        "darkblue" => vec4(0.0, 0.0, 0.545_098_07, 1.0),
        "darkcyan" => vec4(0.0, 0.545_098_07, 0.545_098_07, 1.0),
        "darkgoldenrod" => vec4(0.721_568_64, 0.525_490_2, 0.043_137_256, 1.0),
        "darkgray" => vec4(0.662_745_1, 0.662_745_1, 0.662_745_1, 1.0),
        "darkgreen" => vec4(0.0, 0.392_156_87, 0.0, 1.0),
        "darkkhaki" => vec4(0.741_176_5, 0.717_647_1, 0.419_607_85, 1.0),
        "darkmagenta" => vec4(0.545_098_07, 0.0, 0.545_098_07, 1.0),
        "darkolivegreen" => vec4(0.333_333_34, 0.419_607_85, 0.184_313_73, 1.0),
        "darkorange" => vec4(1.0, 0.549_019_63, 0.0, 1.0),
        "darkorchid" => vec4(0.6, 0.196_078_43, 0.8, 1.0),
        "darkred" => vec4(0.545_098_07, 0.0, 0.0, 1.0),
        "darksalmon" => vec4(0.913_725_5, 0.588_235_3, 0.478_431_37, 1.0),
        "darkseagreen" => vec4(0.560_784_34, 0.737_254_9, 0.560_784_34, 1.0),
        "darkslateblue" => vec4(0.282_352_95, 0.239_215_69, 0.545_098_07, 1.0),
        "darkslategray" => vec4(0.184_313_73, 0.309_803_93, 0.309_803_93, 1.0),
        "darkturquoise" => vec4(0.0, 0.807_843_15, 0.819_607_85, 1.0),
        "darkviolet" => vec4(0.580_392_2, 0.0, 0.827_451, 1.0),
        "deeppink" => vec4(1.0, 0.078_431_375, 0.576_470_6, 1.0),
        "deepskyblue" => vec4(0.0, 0.749_019_6, 1.0, 1.0),
        "dimgray" => vec4(0.411_764_7, 0.411_764_7, 0.411_764_7, 1.0),
        "dodgerblue" => vec4(0.117_647_06, 0.564_705_9, 1.0, 1.0),
        "firebrick" => vec4(0.698_039_23, 0.133_333_34, 0.133_333_34, 1.0),
        "floralwhite" => vec4(1.0, 0.980_392_16, 0.941_176_5, 1.0),
        "forestgreen" => vec4(0.133_333_34, 0.545_098_07, 0.133_333_34, 1.0),
        "fuchsia" => vec4(1.0, 0.0, 1.0, 1.0),
        "gainsboro" => vec4(0.862_745_1, 0.862_745_1, 0.862_745_1, 1.0),
        "ghostwhite" => vec4(0.972_549, 0.972_549, 1.0, 1.0),
        "gold" => vec4(1.0, 0.843_137_26, 0.0, 1.0),
        "goldenrod" => vec4(0.854_901_97, 0.647_058_84, 0.125_490_2, 1.0),
        "gray" => vec4(0.745_098_05, 0.745_098_05, 0.745_098_05, 1.0),
        "webgray" => vec4(0.501_960_8, 0.501_960_8, 0.501_960_8, 1.0),
        "green" => vec4(0.0, 1.0, 0.0, 1.0),
        "webgreen" => vec4(0.0, 0.501_960_8, 0.0, 1.0),
        "greenyellow" => vec4(0.678_431_4, 1.0, 0.184_313_73, 1.0),
        "honeydew" => vec4(0.941_176_5, 1.0, 0.941_176_5, 1.0),
        "hotpink" => vec4(1.0, 0.411_764_7, 0.705_882_4, 1.0),
        "indianred" => vec4(0.803_921_6, 0.360_784_32, 0.360_784_32, 1.0),
        "indigo" => vec4(0.294_117_66, 0.0, 0.509_803_95, 1.0),
        "ivory" => vec4(1.0, 1.0, 0.941_176_5, 1.0),
        "khaki" => vec4(0.941_176_5, 0.901_960_8, 0.549_019_63, 1.0),
        "lavender" => vec4(0.901_960_8, 0.901_960_8, 0.980_392_16, 1.0),
        "lavenderblush" => vec4(1.0, 0.941_176_5, 0.960_784_3, 1.0),
        "lawngreen" => vec4(0.486_274_5, 0.988_235_3, 0.0, 1.0),
        "lemonchiffon" => vec4(1.0, 0.980_392_16, 0.803_921_6, 1.0),
        "lightblue" => vec4(0.678_431_4, 0.847_058_83, 0.901_960_8, 1.0),
        "lightcoral" => vec4(0.941_176_5, 0.501_960_8, 0.501_960_8, 1.0),
        "lightcyan" => vec4(0.878_431_4, 1.0, 1.0, 1.0),
        "lightgoldenrod" => vec4(0.980_392_16, 0.980_392_16, 0.823_529_4, 1.0),
        "lightgray" => vec4(0.827_451, 0.827_451, 0.827_451, 1.0),
        "lightgreen" => vec4(0.564_705_9, 0.933_333_34, 0.564_705_9, 1.0),
        "lightpink" => vec4(1.0, 0.713_725_5, 0.756_862_76, 1.0),
        "lightsalmon" => vec4(1.0, 0.627_451, 0.478_431_37, 1.0),
        "lightseagreen" => vec4(0.125_490_2, 0.698_039_23, 0.666_666_7, 1.0),
        "lightskyblue" => vec4(0.529_411_8, 0.807_843_15, 0.980_392_16, 1.0),
        "lightslategray" => vec4(0.466_666_67, 0.533_333_36, 0.6, 1.0),
        "lightsteelblue" => vec4(0.690_196_1, 0.768_627_46, 0.870_588_24, 1.0),
        "lightyellow" => vec4(1.0, 1.0, 0.878_431_4, 1.0),
        "lime" => vec4(0.0, 1.0, 0.0, 1.0),
        "limegreen" => vec4(0.196_078_43, 0.803_921_6, 0.196_078_43, 1.0),
        "linen" => vec4(0.980_392_16, 0.941_176_5, 0.901_960_8, 1.0),
        "magenta" => vec4(1.0, 0.0, 1.0, 1.0),
        "maroon" => vec4(0.690_196_1, 0.188_235_3, 0.376_470_6, 1.0),
        "webmaroon" => vec4(0.498_039_22, 0.0, 0.0, 1.0),
        "mediumaquamarine" => vec4(0.4, 0.803_921_6, 0.666_666_7, 1.0),
        "mediumblue" => vec4(0.0, 0.0, 0.803_921_6, 1.0),
        "mediumorchid" => vec4(0.729_411_8, 0.333_333_34, 0.827_451, 1.0),
        "mediumpurple" => vec4(0.576_470_6, 0.439_215_7, 0.858_823_54, 1.0),
        "mediumseagreen" => vec4(0.235_294_12, 0.701_960_8, 0.443_137_26, 1.0),
        "mediumslateblue" => vec4(0.482_352_94, 0.407_843_14, 0.933_333_34, 1.0),
        "mediumspringgreen" => vec4(0.0, 0.980_392_16, 0.603_921_6, 1.0),
        "mediumturquoise" => vec4(0.282_352_95, 0.819_607_85, 0.8, 1.0),
        "mediumvioletred" => vec4(0.780_392_17, 0.082_352_94, 0.521_568_66, 1.0),
        "midnightblue" => vec4(0.098_039_22, 0.098_039_22, 0.439_215_7, 1.0),
        "mintcream" => vec4(0.960_784_3, 1.0, 0.980_392_16, 1.0),
        "mistyrose" => vec4(1.0, 0.894_117_65, 0.882_352_95, 1.0),
        "moccasin" => vec4(1.0, 0.894_117_65, 0.709_803_94, 1.0),
        "navajowhite" => vec4(1.0, 0.870_588_24, 0.678_431_4, 1.0),
        "navyblue" => vec4(0.0, 0.0, 0.501_960_8, 1.0),
        "oldlace" => vec4(0.992_156_86, 0.960_784_3, 0.901_960_8, 1.0),
        "olive" => vec4(0.501_960_8, 0.501_960_8, 0.0, 1.0),
        "olivedrab" => vec4(0.419_607_85, 0.556_862_8, 0.137_254_91, 1.0),
        "orange" => vec4(1.0, 0.647_058_84, 0.0, 1.0),
        "orangered" => vec4(1.0, 0.270_588_25, 0.0, 1.0),
        "orchid" => vec4(0.854_901_97, 0.439_215_7, 0.839_215_7, 1.0),
        "palegoldenrod" => vec4(0.933_333_34, 0.909_803_9, 0.666_666_7, 1.0),
        "palegreen" => vec4(0.596_078_46, 0.984_313_7, 0.596_078_46, 1.0),
        "paleturquoise" => vec4(0.686_274_5, 0.933_333_34, 0.933_333_34, 1.0),
        "palevioletred" => vec4(0.858_823_54, 0.439_215_7, 0.576_470_6, 1.0),
        "papayawhip" => vec4(1.0, 0.937_254_9, 0.835_294_1, 1.0),
        "peachpuff" => vec4(1.0, 0.854_901_97, 0.725_490_2, 1.0),
        "peru" => vec4(0.803_921_6, 0.521_568_66, 0.247_058_82, 1.0),
        "pink" => vec4(1.0, 0.752_941_2, 0.796_078_44, 1.0),
        "plum" => vec4(0.866_666_7, 0.627_451, 0.866_666_7, 1.0),
        "powderblue" => vec4(0.690_196_1, 0.878_431_4, 0.901_960_8, 1.0),
        "purple" => vec4(0.627_451, 0.125_490_2, 0.941_176_5, 1.0),
        "webpurple" => vec4(0.498_039_22, 0.0, 0.498_039_22, 1.0),
        "rebeccapurple" => vec4(0.4, 0.2, 0.6, 1.0),
        "red" => vec4(1.0, 0.0, 0.0, 1.0),
        "rosybrown" => vec4(0.737_254_9, 0.560_784_34, 0.560_784_34, 1.0),
        "royalblue" => vec4(0.254_901_98, 0.411_764_7, 0.882_352_95, 1.0),
        "saddlebrown" => vec4(0.545_098_07, 0.270_588_25, 0.074_509_81, 1.0),
        "salmon" => vec4(0.980_392_16, 0.501_960_8, 0.447_058_83, 1.0),
        "sandybrown" => vec4(0.956_862_75, 0.643_137_3, 0.376_470_6, 1.0),
        "seagreen" => vec4(0.180_392_16, 0.545_098_07, 0.341_176_48, 1.0),
        "seashell" => vec4(1.0, 0.960_784_3, 0.933_333_34, 1.0),
        "sienna" => vec4(0.627_451, 0.321_568_64, 0.176_470_6, 1.0),
        "silver" => vec4(0.752_941_2, 0.752_941_2, 0.752_941_2, 1.0),
        "skyblue" => vec4(0.529_411_8, 0.807_843_15, 0.921_568_63, 1.0),
        "slateblue" => vec4(0.415_686_28, 0.352_941_2, 0.803_921_6, 1.0),
        "slategray" => vec4(0.439_215_7, 0.501_960_8, 0.564_705_9, 1.0),
        "snow" => vec4(1.0, 0.980_392_16, 0.980_392_16, 1.0),
        "springgreen" => vec4(0.0, 1.0, 0.498_039_22, 1.0),
        "steelblue" => vec4(0.274_509_82, 0.509_803_95, 0.705_882_4, 1.0),
        "tan" => vec4(0.823_529_4, 0.705_882_4, 0.549_019_63, 1.0),
        "teal" => vec4(0.0, 0.501_960_8, 0.501_960_8, 1.0),
        "thistle" => vec4(0.847_058_83, 0.749_019_6, 0.847_058_83, 1.0),
        "tomato" => vec4(1.0, 0.388_235_3, 0.278_431_4, 1.0),
        "turquoise" => vec4(0.250_980_4, 0.878_431_4, 0.815_686_3, 1.0),
        "violet" => vec4(0.933_333_34, 0.509_803_95, 0.933_333_34, 1.0),
        "wheat" => vec4(0.960_784_3, 0.870_588_24, 0.701_960_8, 1.0),
        "white" => vec4(1.0, 1.0, 1.0, 1.0),
        "whitesmoke" => vec4(0.960_784_3, 0.960_784_3, 0.960_784_3, 1.0),
        "yellow" => vec4(1.0, 1.0, 0.0, 1.0),
        "yellowgreen" => vec4(0.603_921_6, 0.803_921_6, 0.196_078_43, 1.0),
        "red500" => vec4(0.956_862_75, 0.262_745_1, 0.211_764_71, 1.0),
        "red50" => vec4(1.0, 0.921_568_63, 0.933_333_34, 1.0),
        "red100" => vec4(1.0, 0.803_921_6, 0.823_529_4, 1.0),
        "red200" => vec4(0.937_254_9, 0.603_921_6, 0.603_921_6, 1.0),
        "red300" => vec4(0.898_039_2, 0.450_980_4, 0.450_980_4, 1.0),
        "red400" => vec4(0.937_254_9, 0.325_490_2, 0.313_725_5, 1.0),
        "red600" => vec4(0.898_039_2, 0.223_529_41, 0.207_843_14, 1.0),
        "red700" => vec4(0.827_451, 0.184_313_73, 0.184_313_73, 1.0),
        "red800" => vec4(0.776_470_6, 0.156_862_75, 0.156_862_75, 1.0),
        "red900" => vec4(0.717_647_1, 0.109_803_92, 0.109_803_92, 1.0),
        "reda100" => vec4(1.0, 0.541_176_5, 0.501_960_8, 1.0),
        "reda200" => vec4(1.0, 0.321_568_64, 0.321_568_64, 1.0),
        "reda400" => vec4(1.0, 0.090_196_08, 0.266_666_68, 1.0),
        "reda700" => vec4(0.835_294_1, 0.0, 0.0, 1.0),
        "pink500" => vec4(0.913_725_5, 0.117_647_06, 0.388_235_3, 1.0),
        "pink50" => vec4(0.988_235_3, 0.894_117_65, 0.925_490_2, 1.0),
        "pink100" => vec4(0.972_549, 0.733_333_35, 0.815_686_3, 1.0),
        "pink200" => vec4(0.956_862_75, 0.560_784_34, 0.694_117_67, 1.0),
        "pink300" => vec4(0.941_176_5, 0.384_313_73, 0.572_549_05, 1.0),
        "pink400" => vec4(0.925_490_2, 0.250_980_4, 0.478_431_37, 1.0),
        "pink600" => vec4(0.847_058_83, 0.105_882_354, 0.376_470_6, 1.0),
        "pink700" => vec4(0.760_784_3, 0.094_117_65, 0.356_862_75, 1.0),
        "pink800" => vec4(0.678_431_4, 0.078_431_375, 0.341_176_48, 1.0),
        "pink900" => vec4(0.533_333_36, 0.054_901_96, 0.309_803_93, 1.0),
        "pinka100" => vec4(1.0, 0.501_960_8, 0.670_588_25, 1.0),
        "pinka200" => vec4(1.0, 0.250_980_4, 0.505_882_4, 1.0),
        "pinka400" => vec4(0.960_784_3, 0.0, 0.341_176_48, 1.0),
        "pinka700" => vec4(0.772_549_03, 0.066_666_67, 0.384_313_73, 1.0),
        "purple500" => vec4(0.611_764_7, 0.152_941_18, 0.690_196_1, 1.0),
        "purple50" => vec4(0.952_941_2, 0.898_039_2, 0.960_784_3, 1.0),
        "purple100" => vec4(0.882_352_95, 0.745_098_05, 0.905_882_36, 1.0),
        "purple200" => vec4(0.807_843_15, 0.576_470_6, 0.847_058_83, 1.0),
        "purple300" => vec4(0.729_411_8, 0.407_843_14, 0.784_313_74, 1.0),
        "purple400" => vec4(0.670_588_25, 0.278_431_4, 0.737_254_9, 1.0),
        "purple600" => vec4(0.556_862_8, 0.141_176_48, 0.666_666_7, 1.0),
        "purple700" => vec4(0.482_352_94, 0.121_568_63, 0.635_294_14, 1.0),
        "purple800" => vec4(0.415_686_28, 0.105_882_354, 0.603_921_6, 1.0),
        "purple900" => vec4(0.290_196_1, 0.078_431_375, 0.549_019_63, 1.0),
        "purplea100" => vec4(0.917_647_06, 0.501_960_8, 0.988_235_3, 1.0),
        "purplea200" => vec4(0.878_431_4, 0.250_980_4, 0.984_313_7, 1.0),
        "purplea400" => vec4(0.835_294_1, 0.0, 0.976_470_6, 1.0),
        "purplea700" => vec4(0.666_666_7, 0.0, 1.0, 1.0),
        "deeppurple500" => vec4(0.403_921_57, 0.227_450_98, 0.717_647_1, 1.0),
        "deeppurple50" => vec4(0.929_411_77, 0.905_882_36, 0.964_705_9, 1.0),
        "deeppurple100" => vec4(0.819_607_85, 0.768_627_46, 0.913_725_5, 1.0),
        "deeppurple200" => vec4(0.701_960_8, 0.615_686_3, 0.858_823_54, 1.0),
        "deeppurple300" => vec4(0.584_313_75, 0.458_823_53, 0.803_921_6, 1.0),
        "deeppurple400" => vec4(0.494_117_65, 0.341_176_48, 0.760_784_3, 1.0),
        "deeppurple600" => vec4(0.368_627_46, 0.207_843_14, 0.694_117_67, 1.0),
        "deeppurple700" => vec4(0.317_647_07, 0.176_470_6, 0.658_823_55, 1.0),
        "deeppurple800" => vec4(0.270_588_25, 0.152_941_18, 0.627_451, 1.0),
        "deeppurple900" => vec4(0.192_156_87, 0.105_882_354, 0.572_549_05, 1.0),
        "deeppurplea100" => vec4(0.701_960_8, 0.533_333_36, 1.0, 1.0),
        "deeppurplea200" => vec4(0.486_274_5, 0.301_960_8, 1.0, 1.0),
        "deeppurplea400" => vec4(0.396_078_44, 0.121_568_63, 1.0, 1.0),
        "deeppurplea700" => vec4(0.384_313_73, 0.0, 0.917_647_06, 1.0),
        "indigo500" => vec4(0.247_058_82, 0.317_647_07, 0.709_803_94, 1.0),
        "indigo50" => vec4(0.909_803_9, 0.917_647_06, 0.964_705_9, 1.0),
        "indigo100" => vec4(0.772_549_03, 0.792_156_9, 0.913_725_5, 1.0),
        "indigo200" => vec4(0.623_529_43, 0.658_823_55, 0.854_901_97, 1.0),
        "indigo300" => vec4(0.474_509_8, 0.525_490_2, 0.796_078_44, 1.0),
        "indigo400" => vec4(0.360_784_32, 0.419_607_85, 0.752_941_2, 1.0),
        "indigo600" => vec4(0.223_529_41, 0.286_274_52, 0.670_588_25, 1.0),
        "indigo700" => vec4(0.188_235_3, 0.247_058_82, 0.623_529_43, 1.0),
        "indigo800" => vec4(0.156_862_75, 0.207_843_14, 0.576_470_6, 1.0),
        "indigo900" => vec4(0.101_960_786, 0.137_254_91, 0.494_117_65, 1.0),
        "indigoa100" => vec4(0.549_019_63, 0.619_607_87, 1.0, 1.0),
        "indigoa200" => vec4(0.325_490_2, 0.427_450_98, 0.996_078_43, 1.0),
        "indigoa400" => vec4(0.239_215_69, 0.352_941_2, 0.996_078_43, 1.0),
        "indigoa700" => vec4(0.188_235_3, 0.309_803_93, 0.996_078_43, 1.0),
        "blue500" => vec4(0.129_411_77, 0.588_235_3, 0.952_941_2, 1.0),
        "blue50" => vec4(0.890_196_1, 0.949_019_6, 0.992_156_86, 1.0),
        "blue100" => vec4(0.733_333_35, 0.870_588_24, 0.984_313_7, 1.0),
        "blue200" => vec4(0.564_705_9, 0.792_156_9, 0.976_470_6, 1.0),
        "blue300" => vec4(0.392_156_87, 0.709_803_94, 0.964_705_9, 1.0),
        "blue400" => vec4(0.258_823_54, 0.647_058_84, 0.960_784_3, 1.0),
        "blue600" => vec4(0.117_647_06, 0.533_333_36, 0.898_039_2, 1.0),
        "blue700" => vec4(0.098_039_22, 0.462_745_1, 0.823_529_4, 1.0),
        "blue800" => vec4(0.082_352_94, 0.396_078_44, 0.752_941_2, 1.0),
        "blue900" => vec4(0.050_980_393, 0.278_431_4, 0.631_372_6, 1.0),
        "bluea100" => vec4(0.509_803_95, 0.694_117_67, 1.0, 1.0),
        "bluea200" => vec4(0.266_666_68, 0.541_176_5, 1.0, 1.0),
        "bluea400" => vec4(0.160_784_32, 0.474_509_8, 1.0, 1.0),
        "bluea700" => vec4(0.160_784_32, 0.384_313_73, 1.0, 1.0),
        "lightblue500" => vec4(0.011_764_706, 0.662_745_1, 0.956_862_75, 1.0),
        "lightblue50" => vec4(0.882_352_95, 0.960_784_3, 0.996_078_43, 1.0),
        "lightblue100" => vec4(0.701_960_8, 0.898_039_2, 0.988_235_3, 1.0),
        "lightblue200" => vec4(0.505_882_4, 0.831_372_56, 0.980_392_16, 1.0),
        "lightblue300" => vec4(0.309_803_93, 0.764_705_9, 0.968_627_45, 1.0),
        "lightblue400" => vec4(0.160_784_32, 0.713_725_5, 0.964_705_9, 1.0),
        "lightblue600" => vec4(0.011_764_706, 0.607_843_16, 0.898_039_2, 1.0),
        "lightblue700" => vec4(0.007_843_138, 0.533_333_36, 0.819_607_85, 1.0),
        "lightblue800" => vec4(0.007_843_138, 0.466_666_67, 0.741_176_5, 1.0),
        "lightblue900" => vec4(0.003_921_569, 0.341_176_48, 0.607_843_16, 1.0),
        "lightbluea100" => vec4(0.501_960_8, 0.847_058_83, 1.0, 1.0),
        "lightbluea200" => vec4(0.250_980_4, 0.768_627_46, 1.0, 1.0),
        "lightbluea400" => vec4(0.0, 0.690_196_1, 1.0, 1.0),
        "lightbluea700" => vec4(0.0, 0.568_627_5, 0.917_647_06, 1.0),
        "cyan500" => vec4(0.0, 0.737_254_9, 0.831_372_56, 1.0),
        "cyan50" => vec4(0.878_431_4, 0.968_627_45, 0.980_392_16, 1.0),
        "cyan100" => vec4(0.698_039_23, 0.921_568_63, 0.949_019_6, 1.0),
        "cyan200" => vec4(0.501_960_8, 0.870_588_24, 0.917_647_06, 1.0),
        "cyan300" => vec4(0.301_960_8, 0.815_686_3, 0.882_352_95, 1.0),
        "cyan400" => vec4(0.149_019_61, 0.776_470_6, 0.854_901_97, 1.0),
        "cyan600" => vec4(0.0, 0.674_509_8, 0.756_862_76, 1.0),
        "cyan700" => vec4(0.0, 0.592_156_9, 0.654_902, 1.0),
        "cyan800" => vec4(0.0, 0.513_725_5, 0.560_784_34, 1.0),
        "cyan900" => vec4(0.0, 0.376_470_6, 0.392_156_87, 1.0),
        "cyana100" => vec4(0.517_647_1, 1.0, 1.0, 1.0),
        "cyana200" => vec4(0.094_117_65, 1.0, 1.0, 1.0),
        "cyana400" => vec4(0.0, 0.898_039_2, 1.0, 1.0),
        "cyana700" => vec4(0.0, 0.721_568_64, 0.831_372_56, 1.0),
        "teal500" => vec4(0.0, 0.588_235_3, 0.533_333_36, 1.0),
        "teal50" => vec4(0.878_431_4, 0.949_019_6, 0.945_098_04, 1.0),
        "teal100" => vec4(0.698_039_23, 0.874_509_8, 0.858_823_54, 1.0),
        "teal200" => vec4(0.501_960_8, 0.796_078_44, 0.768_627_46, 1.0),
        "teal300" => vec4(0.301_960_8, 0.713_725_5, 0.674_509_8, 1.0),
        "teal400" => vec4(0.149_019_61, 0.650_980_4, 0.603_921_6, 1.0),
        "teal600" => vec4(0.0, 0.537_254_9, 0.482_352_94, 1.0),
        "teal700" => vec4(0.0, 0.474_509_8, 0.419_607_85, 1.0),
        "teal800" => vec4(0.0, 0.411_764_7, 0.360_784_32, 1.0),
        "teal900" => vec4(0.0, 0.301_960_8, 0.250_980_4, 1.0),
        "teala100" => vec4(0.654_902, 1.0, 0.921_568_63, 1.0),
        "teala200" => vec4(0.392_156_87, 1.0, 0.854_901_97, 1.0),
        "teala400" => vec4(0.113_725_49, 0.913_725_5, 0.713_725_5, 1.0),
        "teala700" => vec4(0.0, 0.749_019_6, 0.647_058_84, 1.0),
        "green500" => vec4(0.298_039_23, 0.686_274_5, 0.313_725_5, 1.0),
        "green50" => vec4(0.909_803_9, 0.960_784_3, 0.913_725_5, 1.0),
        "green100" => vec4(0.784_313_74, 0.901_960_8, 0.788_235_3, 1.0),
        "green200" => vec4(0.647_058_84, 0.839_215_7, 0.654_902, 1.0),
        "green300" => vec4(0.505_882_4, 0.780_392_17, 0.517_647_1, 1.0),
        "green400" => vec4(0.4, 0.733_333_35, 0.415_686_28, 1.0),
        "green600" => vec4(0.262_745_1, 0.627_451, 0.278_431_4, 1.0),
        "green700" => vec4(0.219_607_84, 0.556_862_8, 0.235_294_12, 1.0),
        "green800" => vec4(0.180_392_16, 0.490_196_08, 0.196_078_43, 1.0),
        "green900" => vec4(0.105_882_354, 0.368_627_46, 0.125_490_2, 1.0),
        "greena100" => vec4(0.725_490_2, 0.964_705_9, 0.792_156_9, 1.0),
        "greena200" => vec4(0.411_764_7, 0.941_176_5, 0.682_352_96, 1.0),
        "greena400" => vec4(0.0, 0.901_960_8, 0.462_745_1, 1.0),
        "greena700" => vec4(0.0, 0.784_313_74, 0.325_490_2, 1.0),
        "lightgreen500" => vec4(0.545_098_07, 0.764_705_9, 0.290_196_1, 1.0),
        "lightgreen50" => vec4(0.945_098_04, 0.972_549, 0.913_725_5, 1.0),
        "lightgreen100" => vec4(0.862_745_1, 0.929_411_77, 0.784_313_74, 1.0),
        "lightgreen200" => vec4(0.772_549_03, 0.882_352_95, 0.647_058_84, 1.0),
        "lightgreen300" => vec4(0.682_352_96, 0.835_294_1, 0.505_882_4, 1.0),
        "lightgreen400" => vec4(0.611_764_7, 0.8, 0.396_078_44, 1.0),
        "lightgreen600" => vec4(0.486_274_5, 0.701_960_8, 0.258_823_54, 1.0),
        "lightgreen700" => vec4(0.407_843_14, 0.623_529_43, 0.219_607_84, 1.0),
        "lightgreen800" => vec4(0.333_333_34, 0.545_098_07, 0.184_313_73, 1.0),
        "lightgreen900" => vec4(0.2, 0.411_764_7, 0.117_647_06, 1.0),
        "lightgreena100" => vec4(0.8, 1.0, 0.564_705_9, 1.0),
        "lightgreena200" => vec4(0.698_039_23, 1.0, 0.349_019_62, 1.0),
        "lightgreena400" => vec4(0.462_745_1, 1.0, 0.011_764_706, 1.0),
        "lightgreena700" => vec4(0.392_156_87, 0.866_666_7, 0.090_196_08, 1.0),
        "lime500" => vec4(0.803_921_6, 0.862_745_1, 0.223_529_41, 1.0),
        "lime50" => vec4(0.976_470_6, 0.984_313_7, 0.905_882_36, 1.0),
        "lime100" => vec4(0.941_176_5, 0.956_862_75, 0.764_705_9, 1.0),
        "lime200" => vec4(0.901_960_8, 0.933_333_34, 0.611_764_7, 1.0),
        "lime300" => vec4(0.862_745_1, 0.905_882_36, 0.458_823_53, 1.0),
        "lime400" => vec4(0.831_372_56, 0.882_352_95, 0.341_176_48, 1.0),
        "lime600" => vec4(0.752_941_2, 0.792_156_9, 0.2, 1.0),
        "lime700" => vec4(0.686_274_5, 0.705_882_4, 0.168_627_46, 1.0),
        "lime800" => vec4(0.619_607_87, 0.615_686_3, 0.141_176_48, 1.0),
        "lime900" => vec4(0.509_803_95, 0.466_666_67, 0.090_196_08, 1.0),
        "limea100" => vec4(0.956_862_75, 1.0, 0.505_882_4, 1.0),
        "limea200" => vec4(0.933_333_34, 1.0, 0.254_901_98, 1.0),
        "limea400" => vec4(0.776_470_6, 1.0, 0.0, 1.0),
        "limea700" => vec4(0.682_352_96, 0.917_647_06, 0.0, 1.0),
        "yellow500" => vec4(1.0, 0.921_568_63, 0.231_372_55, 1.0),
        "yellow50" => vec4(1.0, 0.992_156_86, 0.905_882_36, 1.0),
        "yellow100" => vec4(1.0, 0.976_470_6, 0.768_627_46, 1.0),
        "yellow200" => vec4(1.0, 0.960_784_3, 0.615_686_3, 1.0),
        "yellow300" => vec4(1.0, 0.945_098_04, 0.462_745_1, 1.0),
        "yellow400" => vec4(1.0, 0.933_333_34, 0.345_098_05, 1.0),
        "yellow600" => vec4(0.992_156_86, 0.847_058_83, 0.207_843_14, 1.0),
        "yellow700" => vec4(0.984_313_7, 0.752_941_2, 0.176_470_6, 1.0),
        "yellow800" => vec4(0.976_470_6, 0.658_823_55, 0.145_098_05, 1.0),
        "yellow900" => vec4(0.960_784_3, 0.498_039_22, 0.090_196_08, 1.0),
        "yellowa100" => vec4(1.0, 1.0, 0.552_941_2, 1.0),
        "yellowa200" => vec4(1.0, 1.0, 0.0, 1.0),
        "yellowa400" => vec4(1.0, 0.917_647_06, 0.0, 1.0),
        "yellowa700" => vec4(1.0, 0.839_215_7, 0.0, 1.0),
        "amber500" => vec4(1.0, 0.756_862_76, 0.027_450_98, 1.0),
        "amber50" => vec4(1.0, 0.972_549, 0.882_352_95, 1.0),
        "amber100" => vec4(1.0, 0.925_490_2, 0.701_960_8, 1.0),
        "amber200" => vec4(1.0, 0.878_431_4, 0.509_803_95, 1.0),
        "amber300" => vec4(1.0, 0.835_294_1, 0.309_803_93, 1.0),
        "amber400" => vec4(1.0, 0.792_156_9, 0.156_862_75, 1.0),
        "amber600" => vec4(1.0, 0.701_960_8, 0.0, 1.0),
        "amber700" => vec4(1.0, 0.627_451, 0.0, 1.0),
        "amber800" => vec4(1.0, 0.560_784_34, 0.0, 1.0),
        "amber900" => vec4(1.0, 0.435_294_12, 0.0, 1.0),
        "ambera100" => vec4(1.0, 0.898_039_2, 0.498_039_22, 1.0),
        "ambera200" => vec4(1.0, 0.843_137_26, 0.250_980_4, 1.0),
        "ambera400" => vec4(1.0, 0.768_627_46, 0.0, 1.0),
        "ambera700" => vec4(1.0, 0.670_588_25, 0.0, 1.0),
        "orange500" => vec4(1.0, 0.596_078_46, 0.0, 1.0),
        "orange50" => vec4(1.0, 0.952_941_2, 0.878_431_4, 1.0),
        "orange100" => vec4(1.0, 0.878_431_4, 0.698_039_23, 1.0),
        "orange200" => vec4(1.0, 0.8, 0.501_960_8, 1.0),
        "orange300" => vec4(1.0, 0.717_647_1, 0.301_960_8, 1.0),
        "orange400" => vec4(1.0, 0.654_902, 0.149_019_61, 1.0),
        "orange600" => vec4(0.984_313_7, 0.549_019_63, 0.0, 1.0),
        "orange700" => vec4(0.960_784_3, 0.486_274_5, 0.0, 1.0),
        "orange800" => vec4(0.937_254_9, 0.423_529_42, 0.0, 1.0),
        "orange900" => vec4(0.901_960_8, 0.317_647_07, 0.0, 1.0),
        "orangea100" => vec4(1.0, 0.819_607_85, 0.501_960_8, 1.0),
        "orangea200" => vec4(1.0, 0.670_588_25, 0.250_980_4, 1.0),
        "orangea400" => vec4(1.0, 0.568_627_5, 0.0, 1.0),
        "orangea700" => vec4(1.0, 0.427_450_98, 0.0, 1.0),
        "deeporange500" => vec4(1.0, 0.341_176_48, 0.133_333_34, 1.0),
        "deeporange50" => vec4(0.984_313_7, 0.913_725_5, 0.905_882_36, 1.0),
        "deeporange100" => vec4(1.0, 0.8, 0.737_254_9, 1.0),
        "deeporange200" => vec4(1.0, 0.670_588_25, 0.568_627_5, 1.0),
        "deeporange300" => vec4(1.0, 0.541_176_5, 0.396_078_44, 1.0),
        "deeporange400" => vec4(1.0, 0.439_215_7, 0.262_745_1, 1.0),
        "deeporange600" => vec4(0.956_862_75, 0.317_647_07, 0.117_647_06, 1.0),
        "deeporange700" => vec4(0.901_960_8, 0.290_196_1, 0.098_039_22, 1.0),
        "deeporange800" => vec4(0.847_058_83, 0.262_745_1, 0.082_352_94, 1.0),
        "deeporange900" => vec4(0.749_019_6, 0.211_764_71, 0.047_058_824, 1.0),
        "deeporangea100" => vec4(1.0, 0.619_607_87, 0.501_960_8, 1.0),
        "deeporangea200" => vec4(1.0, 0.431_372_55, 0.250_980_4, 1.0),
        "deeporangea400" => vec4(1.0, 0.239_215_69, 0.0, 1.0),
        "deeporangea700" => vec4(0.866_666_7, 0.172_549_02, 0.0, 1.0),
        "brown500" => vec4(0.474_509_8, 0.333_333_34, 0.282_352_95, 1.0),
        "brown50" => vec4(0.937_254_9, 0.921_568_63, 0.913_725_5, 1.0),
        "brown100" => vec4(0.843_137_26, 0.8, 0.784_313_74, 1.0),
        "brown200" => vec4(0.737_254_9, 0.666_666_7, 0.643_137_3, 1.0),
        "brown300" => vec4(0.631_372_6, 0.533_333_36, 0.498_039_22, 1.0),
        "brown400" => vec4(0.552_941_2, 0.431_372_55, 0.388_235_3, 1.0),
        "brown600" => vec4(0.427_450_98, 0.298_039_23, 0.254_901_98, 1.0),
        "brown700" => vec4(0.364_705_9, 0.250_980_4, 0.215_686_28, 1.0),
        "brown800" => vec4(0.305_882_36, 0.203_921_57, 0.180_392_16, 1.0),
        "brown900" => vec4(0.243_137_26, 0.152_941_18, 0.137_254_91, 1.0),
        "grey500" => vec4(0.619_607_87, 0.619_607_87, 0.619_607_87, 1.0),
        "grey50" => vec4(0.980_392_16, 0.980_392_16, 0.980_392_16, 1.0),
        "grey100" => vec4(0.960_784_3, 0.960_784_3, 0.960_784_3, 1.0),
        "grey200" => vec4(0.933_333_34, 0.933_333_34, 0.933_333_34, 1.0),
        "grey300" => vec4(0.878_431_4, 0.878_431_4, 0.878_431_4, 1.0),
        "grey400" => vec4(0.741_176_5, 0.741_176_5, 0.741_176_5, 1.0),
        "grey600" => vec4(0.458_823_53, 0.458_823_53, 0.458_823_53, 1.0),
        "grey700" => vec4(0.380_392_16, 0.380_392_16, 0.380_392_16, 1.0),
        "grey800" => vec4(0.258_823_54, 0.258_823_54, 0.258_823_54, 1.0),
        "grey850" => vec4(0.192_156_87, 0.192_156_87, 0.192_156_87, 1.0),
        "grey900" => vec4(0.129_411_77, 0.129_411_77, 0.129_411_77, 1.0),
        "bluegrey500" => vec4(0.376_470_6, 0.490_196_08, 0.545_098_07, 1.0),
        "bluegrey50" => vec4(0.925_490_2, 0.937_254_9, 0.945_098_04, 1.0),
        "bluegrey100" => vec4(0.811_764_7, 0.847_058_83, 0.862_745_1, 1.0),
        "bluegrey200" => vec4(0.690_196_1, 0.745_098_05, 0.772_549_03, 1.0),
        "bluegrey300" => vec4(0.564_705_9, 0.643_137_3, 0.682_352_96, 1.0),
        "bluegrey400" => vec4(0.470_588_24, 0.564_705_9, 0.611_764_7, 1.0),
        "bluegrey600" => vec4(0.329_411_77, 0.431_372_55, 0.478_431_37, 1.0),
        "bluegrey700" => vec4(0.270_588_25, 0.352_941_2, 0.392_156_87, 1.0),
        "bluegrey800" => vec4(0.215_686_28, 0.278_431_4, 0.309_803_93, 1.0),
        "bluegrey900" => vec4(0.149_019_61, 0.196_078_43, 0.219_607_84, 1.0),
        _ => return Err(()),
    })
}
