//! Version of [`std::thread`] that also works in WebAssembly.

#[cfg(not(target_arch = "wasm32"))]
use std::thread;

/// See [`Thread`].
struct UniversalThread();

/// Encapsulates the public API for threads, so that UniversalThread
/// can implement it differently on native vs. WASM. All methods from here
/// will get exposed at the module level as well lower in this file.
trait Thread {
    /// Run function in a non-blocking thread.
    ///
    /// Check out the `multithread_example` for sample usage.
    ///
    /// TODO(Paras): Implement join handles. When we do, we can use the same
    /// function signature here as std::thread::spawn.
    fn spawn(f: impl FnOnce() + Send + 'static);
}

#[cfg(not(target_arch = "wasm32"))]
impl Thread for UniversalThread {
    /// See [`Thread::spawn`].
    fn spawn(f: impl FnOnce() + Send + 'static) {
        thread::spawn(f);
    }
}

#[cfg(target_arch = "wasm32")]
struct WorkerContext {
    func: Box<dyn FnOnce() + Send>,
}

#[cfg(target_arch = "wasm32")]
impl Thread for UniversalThread {
    /// See [`Thread::spawn`].
    fn spawn(f: impl FnOnce() + Send + 'static) {
        let context = Box::into_raw(Box::new(WorkerContext { func: Box::new(f) })) as usize;

        unsafe {
            thread_spawn(context);
        }
    }
}

#[cfg(target_arch = "wasm32")]
extern "C" {
    fn thread_spawn(context: usize);
}

#[cfg(target_arch = "wasm32")]
#[export_name = "run_function_pointer"]
unsafe extern "C" fn fn_to_run_in_worker(ctx_ptr: usize) {
    let ctx = Box::from_raw(ctx_ptr as *mut WorkerContext);
    (ctx.func)();
}

/// Version of [`std::thread::spawn`] that also works in WebAssembly.
///
/// See also [`Thread::spawn`].
pub fn spawn(f: impl FnOnce() + Send + 'static) {
    UniversalThread::spawn(f);
}
