//! Layout system. 🐢

use crate::cx::*;

/// A [`Turtle`] is a structure used for layouting: laying out different widgets on the screen.
///
/// It is roughly modeled after the [Logo Turtle](https://en.wikipedia.org/wiki/Logo_(programming_language))
/// of old, but our version has a lot more state, and behaves differently in many ways.
///
/// At the core you can imagine the turtle as having a position ([`Turtle::pos`]), a "sandbox" that
/// it can move in (delineated by [`Turtle::origin`] and [`Turtle::width`] and [`Turtle::height`]).
///
/// Its movement is determined primarily by the [`Layout`] that you pass in, and you can modify it
/// ad-hoc by calling various functions.
///
/// Turtles can be nested, so we have a stack of turtles in [`Cx::turtles`]. The last [`Turtle`] on
/// the stack is the "current" or "active" turtle. When you call [`Cx::end_turtle`], the last turtle's
/// "sandbox" [`Rect`] will be used to walk its parent turtle. It truly is [turtles all the way
/// down](https://en.wikipedia.org/wiki/Turtles_all_the_way_down)!
///
/// A core philosophy of the turtle model is its simplicity and speed, by having only a single pass
/// to do layouting. Contrast this with systems like [CSS Flexbox](https://en.wikipedia.org/wiki/CSS_Flexible_Box_Layout),
/// which use a constraint satisfaction system to lay out your widgets. Instead, we make a single
/// pass, but do sometimes shift over individual elements after the fact, typically using
/// [`Cx::turtle_align_list`]. When doing this we can regard it as a "1.5-pass" rendering. Currently
/// we have to go through every individual element if we want to move it, but in the future we could
/// exploit groupings of elements in [`View`]s and [`DrawCall`]s, and set uniforms on them.
///
/// TODO(JP): The way the turtle moves around is quite confusing in a lot of cases! This model
/// probably requires a complete rework. We can take inspiration from other layouting systems (e.g.
/// the [CSS box model](https://developer.mozilla.org/en-US/docs/Learn/CSS/Building_blocks/The_box_model))
/// while retaining the philosophical core of the current turtle approach.
#[derive(Clone, Default, Debug)]
pub(crate) struct Turtle {
    /// The layout that is associated directly with this turtle, which determines a lot of its
    /// behavior.
    layout: Layout,

    /// The index within Cx::turtle_align_list, which contains all the things that we draw within this
    /// turtle, and which needs to get aligned at some point. We have a separate list for x/y
    /// because you can manually trigger an alignment (aside from it happening automatically at the
    /// end), which resets this list to no longer align those areas again.
    align_list_x_start_index: usize,

    /// Same as [`Turtle::align_list_x_start_index`] but for vertical alignment.
    align_list_y_start_index: usize,

    /// The current position of the turtle. This is the only field that seems to actually correspond
    /// to the "turtle graphics" metaphor! This is an absolute position, and starts out at [`Turtle::origin`]
    /// plus padding.
    pos: Vec2,

    /// The origin of the current turtle's walking area. Starts off at the parent's turtle [`Turtle::pos`]
    /// plus this turtle's [`Walk::margin`].
    origin: Vec2,

    /// The inherent width of the current turtle's walking area. Is [`f32::NAN`] if the width is computed,
    /// and can get set explicitly later.
    width: f32,

    /// The inherent height of the current turtle's walking area. Is [`f32::NAN`] if the height is computed,
    /// and can get set explicitly later.
    height: f32,

    /// Seems to only be used to be passed down to child turtles, so if one of them gets an absolute
    /// origin passed in, we can just use the entire remaining absolute canvas as the width/height.
    ///
    /// TODO(JP): seems pretty unnecessary; why not just grab this field from the current [`Pass`
    /// directly if necessary? Or just always have the caller pass it in (they can take it from the
    /// current [`Pass`] if they want)?
    abs_size: Vec2,

    /// Keeps track of the bottom right corner where we have walked so far, including the width/height
    /// of the walk, whereas [`Turtle::pos`] stays in the top left position of what we have last drawn.
    ///
    /// TODO(JP): [`Turtle::pos`] and [`Turtle::bound_right_bottom`] can (and seem to regularly and intentionally do)
    /// get out of sync, which makes things more confusing.
    bound_right_bottom: Vec2,

    /// When you horizontally align some elements, you can choose to "use up" that width, which gets
    /// set in this field. This width is excluded from future alignment and "width left" calculations.
    width_used: f32,

    /// When you vertically align some elements, you can choose to "use up" that height, which gets
    /// set in this field. This height is excluded from future alignment and "height left" calculations.
    height_used: f32,

    /// We keep track of the [`Walk`] with the greatest height (or width, when walking down), so that
    /// we know how much to move the turtle's y-position when wrapping to the next line. When
    /// wrapping to the next line, this value is reset back to 0.
    ///
    /// See also [`Padding`].
    biggest: f32,

    /// TODO(JP): The [`Turtle::guard_area`] seems to only be used for making sure the right turtle gets closed
    /// again, because unlike with a [`View`] we don't have an object on the application side here. We
    /// can probably make a nicer interface here, to avoid passing in [`Area::Empty`] which seems to be
    /// common if using turtles manually.
    guard_area: Area,
}

/// Indicates when to wrap the current line to a new line. See also [`Direction`].
#[derive(Copy, Clone, Debug)]
pub enum LineWrap {
    /// Never wrap to a new line.
    None,

    /// Wrap to a new line when the available width is exhausted.
    /// TODO(JP): This is a bit of a bit of a misnomer, maybe "overflow"
    /// or so would be better?
    NewLine,

    /// Wrap to a new line when a given width is exceeded (OR when the
    /// available width is exhausted; it includes the [`LineWrap::NewLine`]
    /// behavior too).
    /// TODO(JP): I'm not sure if this is the right place to have this; might
    /// be better to express this as a "max width" on the layout?
    MaxSize(f32),
}
impl LineWrap {
    /// TODO(JP): Replace these with LineWrap::default() when
    /// <https://github.com/rust-lang/rust/pull/86808> gets done
    pub const DEFAULT: LineWrap = LineWrap::None;
}
impl Default for LineWrap {
    fn default() -> Self {
        LineWrap::DEFAULT
    }
}

/// Configure how a [`Turtle`] is going to walk, typically bounded by the
/// dimensions of a parent [`Turtle`].
#[derive(Copy, Clone, Debug)]
pub struct Layout {
    /// See [`Walk`].
    pub walk: Walk,
    /// See [`Padding`].
    pub padding: Padding,
    /// See [`Align`].
    pub align: Align,
    /// See [`Direction`].
    pub direction: Direction,
    /// See [`LineWrap`].
    pub line_wrap: LineWrap,
    /// The amount of extra spacing when wrapping a line.
    ///
    /// TODO(JP): Should we make this part of [`LineWrap`] instead?
    pub new_line_padding: f32,
    /// Override the [`Turtle::origin`], instead of using the parent's
    /// current position.
    pub abs_origin: Option<Vec2>,
    /// Override the maximum size of the [`Window`]/[`Pass`]. Should typically
    /// not be used; instead set [`Turtle::width`] and [`Turtle::height`]
    /// through [`Layout::walk`].
    pub abs_size: Option<Vec2>,
}

impl Layout {
    /// TODO(JP): Replace these with Layout::default() when
    /// <https://github.com/rust-lang/rust/pull/86808> gets done
    pub const DEFAULT: Layout = Layout {
        walk: Walk::DEFAULT,
        align: Align::DEFAULT,
        padding: Padding::DEFAULT,
        direction: Direction::DEFAULT,
        line_wrap: LineWrap::DEFAULT,
        new_line_padding: 0.,
        abs_origin: None,
        abs_size: None,
    };

    pub fn abs_origin_zero() -> Self {
        Layout { abs_origin: Some(Vec2::default()), ..Default::default() }
    }
}
impl Default for Layout {
    fn default() -> Self {
        Layout::DEFAULT
    }
}

/// Determines how a [`Turtle`] should walk. Can be applied to a new [`Turtle`]
/// through [`Layout::walk`], or directly to move an existing [`Turtle`] by
/// using [`Cx::walk_turtle`].
#[derive(Copy, Clone, Debug)]
pub struct Walk {
    pub margin: Margin,
    pub width: Width,
    pub height: Height,
}

impl Walk {
    /// TODO(JP): Replace these with Align::default() when
    /// <https://github.com/rust-lang/rust/pull/86808> gets done
    pub const DEFAULT: Walk = Walk { width: Width::DEFAULT, height: Height::DEFAULT, margin: Margin::DEFAULT };

    pub const fn wh(w: Width, h: Height) -> Self {
        Self { width: w, height: h, margin: Margin::ZERO }
    }
}
impl Default for Walk {
    fn default() -> Self {
        Walk::DEFAULT
    }
}

/// Defines how elements on [`Cx::turtle_align_list`] should be moved when
/// the [`Turtle`] ends.
#[derive(Clone, Copy, Debug)]
pub struct Align {
    pub fx: f32,
    pub fy: f32,
}

impl Align {
    pub const LEFT_TOP: Align = Align { fx: 0., fy: 0. };
    pub const CENTER_TOP: Align = Align { fx: 0.5, fy: 0.0 };
    pub const RIGHT_TOP: Align = Align { fx: 1.0, fy: 0.0 };
    pub const LEFT_CENTER: Align = Align { fx: 0.0, fy: 0.5 };
    pub const CENTER: Align = Align { fx: 0.5, fy: 0.5 };
    pub const RIGHT_CENTER: Align = Align { fx: 1.0, fy: 0.5 };
    pub const LEFT_BOTTOM: Align = Align { fx: 0., fy: 1.0 };
    pub const CENTER_BOTTOM: Align = Align { fx: 0.5, fy: 1.0 };
    pub const RIGHT_BOTTOM: Align = Align { fx: 1.0, fy: 1.0 };

    /// TODO(JP): Replace these with Align::default() when
    /// <https://github.com/rust-lang/rust/pull/86808> gets done
    pub const DEFAULT: Align = Align::LEFT_TOP;
}
impl Default for Align {
    fn default() -> Self {
        Align::DEFAULT
    }
}

/// A margin that should be added around a [`Walk`].
///
/// TODO(JP): these values can be negative, which can be quite confusing, but we
/// seem to actually honor that in the turtle code. Might be good to look into that
/// and see if we should forbid that or not (we seem to never actually do that yet).
#[derive(Clone, Copy, Debug)]
pub struct Margin {
    pub l: f32,
    pub t: f32,
    pub r: f32,
    pub b: f32,
}
impl Margin {
    pub const ZERO: Margin = Margin { l: 0.0, t: 0.0, r: 0.0, b: 0.0 };

    /// TODO(JP): Replace these with Margin::default() when
    /// <https://github.com/rust-lang/rust/pull/86808> gets done
    pub const DEFAULT: Margin = Margin::ZERO;

    pub const fn all(v: f32) -> Margin {
        Margin { l: v, t: v, r: v, b: v }
    }
}
impl Default for Margin {
    fn default() -> Self {
        Margin::DEFAULT
    }
}

/// Inner padding dimensions that should be applied on top of the width/height
/// from the parent [`Turtle`].
#[derive(Clone, Copy, Debug)]
pub struct Padding {
    pub l: f32,
    pub t: f32,
    pub r: f32,
    pub b: f32,
}
impl Padding {
    pub const ZERO: Padding = Padding { l: 0.0, t: 0.0, r: 0.0, b: 0.0 };

    /// TODO(JP): Replace these with Padding::default() when
    /// <https://github.com/rust-lang/rust/pull/86808> gets done
    pub const DEFAULT: Padding = Padding::ZERO;

    pub const fn all(v: f32) -> Padding {
        Padding { l: v, t: v, r: v, b: v }
    }
}
impl Default for Padding {
    fn default() -> Self {
        Padding::DEFAULT
    }
}

/// The direction in which the [`Turtle`] should walk. It will typically walk
/// in a straight line in this direction. E.g. when walking to [`Direction::Right`],
/// it will only walk horizontally, not vertically, until it hits the [`Turtle::width`],
/// at which point it will wrap around using [`LineWrap`], based on the maximum
/// height of widgets that have been drawn so far, which is registered in
/// [`Turtle::biggest`].
///
/// TODO(JP): This line wrapping behavior makes sense for [`Direction::Right`],
/// but not so much for [`Direction::Down`].. Maybe we should split [`Turtle`]
/// into different kinds of behavior?
#[derive(Copy, Clone, Debug, PartialEq)]
pub enum Direction {
    Right,
    Down,
}
impl Direction {
    /// TODO(JP): Replace these with Direction::default() when
    /// <https://github.com/rust-lang/rust/pull/86808> gets done
    pub const DEFAULT: Direction = Direction::Right;
}
impl Default for Direction {
    fn default() -> Self {
        Direction::DEFAULT
    }
}

/// Different ways in which a [`Walk`] can get a width.
///
/// TODO(JP): Something like `FillUpTo(f32)` or `FillMax(f32)` might be useful here, to mimic
/// CSS'es `max-width`. For now you can manually use `Cx::get_width_left` with
/// `Width::Fix` as a workaround.
///
/// TODO(JP): See [`Height::DEFAULT`] for a related TODO.
#[derive(Copy, Clone, Debug)]
pub enum Width {
    /// Fill up as much of the available space as possible.
    Fill,
    /// Use a fixed width.
    Fix(f32),
    /// Will defer computation of [`Turtle::width`] by setting it to [`f32::NAN`],
    /// and only properly computing it later on.
    ///
    /// TODO(JP): This can also be passed into [`Cx::walk_turtle`] but there it
    /// makes no sense!
    Compute,
}
impl Width {
    /// TODO(JP): Replace these with Width::default() when
    /// <https://github.com/rust-lang/rust/pull/86808> gets done
    pub const DEFAULT: Width = Width::Fill;

    pub fn fixed(&self) -> f32 {
        match self {
            Width::Fix(v) => *v,
            _ => 0.,
        }
    }
}
impl Default for Width {
    fn default() -> Self {
        Width::Fill
    }
}

/// Different ways in which a [`Walk`] can get a height.
///
/// See [`Width`] for more documentation, since it's analogous.
#[derive(Copy, Clone, Debug)]
pub enum Height {
    // See [`Width::Fill`].
    Fill,
    // See [`Width::Fix`].
    Fix(f32),
    // See [`Width::Compute`].
    Compute,
}
impl Height {
    /// TODO(JP): [`Height::Fill`] might be a bad default, because if you use
    /// [`Direction::Down`] it will push out everything out it below.
    /// HTML/CSS uses something more like [`Height::Compute`] by default for height,
    /// and only [`Height::Fill`] for width (for block-layout elements).
    ///
    /// TODO(JP): Replace these with Height::default() when
    /// <https://github.com/rust-lang/rust/pull/86808> gets done
    pub const DEFAULT: Height = Height::Fill;

    pub fn fixed(&self) -> f32 {
        match self {
            Height::Fix(v) => *v,
            _ => 0.,
        }
    }
}
impl Default for Height {
    fn default() -> Self {
        Height::Fill
    }
}

impl Cx {
    /// Begin a new [`Turtle`] with a given [`Layout`]. This new [`Turtle`] will be added to the
    /// [`Cx::turtles`] stack.
    ///
    /// TODO(JP): The `guard_area` seems to only be used for making sure the right turtle gets closed
    /// again, because unlike with a [`View`] we don't have an object on the application side here. We
    /// can probably make a nicer interface here, to avoid passing in [`Area::Empty`] which seems to be
    /// common if using turtles manually.
    pub fn begin_turtle(&mut self, layout: Layout, guard_area: Area) {
        if !self.in_redraw_cycle {
            panic!("calling begin_turtle outside of redraw cycle is not possible!");
        }

        // fetch origin and size from parent
        let (mut origin, mut abs_size) = if let Some(parent) = self.turtles.last() {
            (Vec2 { x: layout.walk.margin.l + parent.pos.x, y: layout.walk.margin.t + parent.pos.y }, parent.abs_size)
        } else {
            (Vec2 { x: layout.walk.margin.l, y: layout.walk.margin.t }, Vec2::default())
        };

        // see if layout overrode size
        if let Some(layout_abs_size) = layout.abs_size {
            abs_size = layout_abs_size;
        }

        // same for origin
        let is_abs_origin;
        if let Some(abs_origin) = layout.abs_origin {
            origin = abs_origin;
            is_abs_origin = true;
        } else {
            is_abs_origin = false;
        }

        // abs origin overrides the computation of width/height to use the parent abs_origin
        let width = self.eval_width(&layout.walk.width, layout.walk.margin, is_abs_origin, abs_size.x);
        let height = self.eval_height(&layout.walk.height, layout.walk.margin, is_abs_origin, abs_size.y);

        let turtle = Turtle {
            align_list_x_start_index: self.turtle_align_list.len(),
            align_list_y_start_index: self.turtle_align_list.len(),
            origin,
            pos: Vec2 { x: origin.x + layout.padding.l, y: origin.y + layout.padding.t },
            layout,
            biggest: 0.0,
            bound_right_bottom: Vec2 { x: std::f32::NEG_INFINITY, y: std::f32::NEG_INFINITY },
            width,
            height,
            width_used: 0.,
            height_used: 0.,
            abs_size,
            guard_area,
            //..Default::default()
        };

        self.turtles.push(turtle);
    }

    /// Pop the current [`Turtle`] from the [`Cx::turtles`] stack, returning a [`Rect`] that the turtle walked
    /// during its lifetime. The parent [`Turtle`] will be made to walk this [`Rect`].
    pub fn end_turtle(&mut self, guard_area: Area) -> Rect {
        let old = self.turtles.pop().unwrap();
        if guard_area != old.guard_area {
            panic!(
                "End turtle guard area misaligned!, begin/end pair not matched begin {:?} end {:?}",
                old.guard_area, guard_area
            )
        }

        let w = if old.width.is_nan() {
            if old.bound_right_bottom.x == std::f32::NEG_INFINITY {
                // nothing happened, use padding
                Width::Fix(old.layout.padding.l + old.layout.padding.r)
            } else {
                // use the bounding box
                Width::Fix(max_zero_keep_nan(old.bound_right_bottom.x - old.origin.x + old.layout.padding.r))
            }
        } else {
            Width::Fix(old.width)
        };

        let h = if old.height.is_nan() {
            if old.bound_right_bottom.y == std::f32::NEG_INFINITY {
                // nothing happened use the padding
                Height::Fix(old.layout.padding.t + old.layout.padding.b)
            } else {
                // use the bounding box
                Height::Fix(max_zero_keep_nan(old.bound_right_bottom.y - old.origin.y + old.layout.padding.b))
            }
        } else {
            Height::Fix(old.height)
        };

        let margin = old.layout.walk.margin;
        // if we have alignment set, we should now align our childnodes
        let dx = Self::compute_align_turtle_x(&old);
        if dx > 0.0 {
            self.do_align_x(dx, old.align_list_x_start_index);
        }
        let dy = Self::compute_align_turtle_y(&old);
        if dy > 0.0 {
            self.do_align_y(dy, old.align_list_y_start_index);
        }

        // when a turtle is x-abs / y-abs you dont walk the parent
        if old.layout.abs_origin.is_some() {
            let abs_origin = if let Some(abs_origin) = old.layout.abs_origin { abs_origin } else { Vec2::default() };
            let w = if let Width::Fix(vw) = w { vw } else { 0. };
            let h = if let Height::Fix(vh) = h { vh } else { 0. };
            return Rect { pos: abs_origin, size: vec2(w, h) };
        }

        self.walk_turtle_with_old(Walk { width: w, height: h, margin }, Some(&old))
    }

    /// Walk the current [`Turtle`], returning a [`Rect`] that it ended up walking.
    pub fn walk_turtle(&mut self, walk: Walk) -> Rect {
        self.walk_turtle_with_old(walk, None)
    }

    /// Walk the turtle with a 'w/h' and a margin.
    ///
    /// Returns a [`Rect`] containing the area that the turtle walked, excluding the [`Walk::margin`].
    ///
    /// TODO(JP): This `old_turtle` stuff is a bit awkward (awkward turtle..) and only used for the
    /// alignment stuff at the end. We can probably structure this in a nicer way.
    fn walk_turtle_with_old(&mut self, walk: Walk, old_turtle: Option<&Turtle>) -> Rect {
        let mut align_dx = 0.0;
        let mut align_dy = 0.0;

        // TODO(JP): This seems a bit weird: you can technically pass in Width::Compute, which would
        // return a NaN for `w`, but that doesn't make much sense when you explicitly do a walk.
        // It looks like it's assumed that that never gets passed in here, but it would be better to
        // verify that.
        let w = self.eval_width(&walk.width, walk.margin, false, 0.0);
        let h = self.eval_height(&walk.height, walk.margin, false, 0.0);

        let ret = if let Some(turtle) = self.turtles.last_mut() {
            let (x, y) = match turtle.layout.direction {
                Direction::Right => {
                    match turtle.layout.line_wrap {
                        LineWrap::NewLine => {
                            if (turtle.pos.x + walk.margin.l + w)
                                > (turtle.origin.x + turtle.width - turtle.layout.padding.r) + 0.01
                            {
                                // what is the move delta.
                                let old_x = turtle.pos.x;
                                let old_y = turtle.pos.y;
                                turtle.pos.x = turtle.origin.x + turtle.layout.padding.l;
                                turtle.pos.y += turtle.biggest;
                                turtle.biggest = 0.0;
                                align_dx = turtle.pos.x - old_x;
                                align_dy = turtle.pos.y - old_y;
                            }
                        }
                        LineWrap::MaxSize(max_size) => {
                            let new_size = turtle.pos.x + walk.margin.l + w;
                            if new_size > (turtle.origin.x + turtle.width - turtle.layout.padding.r)
                                || new_size > (turtle.origin.x + max_size - turtle.layout.padding.r)
                            {
                                // what is the move delta.
                                let old_x = turtle.pos.x;
                                let old_y = turtle.pos.y;
                                turtle.pos.x = turtle.origin.x + turtle.layout.padding.l;
                                turtle.pos.y += turtle.biggest;
                                turtle.biggest = 0.0;
                                align_dx = turtle.pos.x - old_x;
                                align_dy = turtle.pos.y - old_y;
                            }
                        }
                        LineWrap::None => {}
                    }

                    let x = turtle.pos.x + walk.margin.l;
                    let y = turtle.pos.y + walk.margin.t;
                    // walk it normally
                    turtle.pos.x += w + walk.margin.l + walk.margin.r;

                    // keep track of biggest item in the line (include item margin bottom)
                    let biggest = h + walk.margin.t + walk.margin.b;
                    if biggest > turtle.biggest {
                        turtle.biggest = biggest;
                    }
                    (x, y)
                }
                Direction::Down => {
                    match turtle.layout.line_wrap {
                        LineWrap::NewLine => {
                            if (turtle.pos.y + walk.margin.t + h)
                                > (turtle.origin.y + turtle.height - turtle.layout.padding.b) + 0.01
                            {
                                // what is the move delta.
                                let old_x = turtle.pos.x;
                                let old_y = turtle.pos.y;
                                turtle.pos.y = turtle.origin.y + turtle.layout.padding.t;
                                turtle.pos.x += turtle.biggest;
                                turtle.biggest = 0.0;
                                align_dx = turtle.pos.x - old_x;
                                align_dy = turtle.pos.y - old_y;
                            }
                        }
                        LineWrap::MaxSize(max_size) => {
                            let new_size = turtle.pos.y + walk.margin.t + h;
                            if new_size > (turtle.origin.y + turtle.height - turtle.layout.padding.b)
                                || new_size > (turtle.origin.y + max_size - turtle.layout.padding.b)
                            {
                                // what is the move delta.
                                let old_x = turtle.pos.x;
                                let old_y = turtle.pos.y;
                                turtle.pos.y = turtle.origin.y + turtle.layout.padding.t;
                                turtle.pos.x += turtle.biggest;
                                turtle.biggest = 0.0;
                                align_dx = turtle.pos.x - old_x;
                                align_dy = turtle.pos.y - old_y;
                            }
                        }
                        LineWrap::None => {}
                    }

                    let x = turtle.pos.x + walk.margin.l;
                    let y = turtle.pos.y + walk.margin.t;
                    // walk it normally
                    turtle.pos.y += h + walk.margin.t + walk.margin.b;

                    // keep track of biggest item in the line (include item margin bottom)
                    let biggest = w + walk.margin.r + walk.margin.l;
                    if biggest > turtle.biggest {
                        turtle.biggest = biggest;
                    }
                    (x, y)
                }
            };

            let bound_x2 = x + w + if walk.margin.r < 0. { walk.margin.r } else { 0. };
            if bound_x2 > turtle.bound_right_bottom.x {
                turtle.bound_right_bottom.x = bound_x2;
            }
            // update y2 bounds (margin bottom is only added if its negative)
            // TODO(JP): We seem to never actually use negative margins (yet),
            // maybe we should forbid that altogether since it can be confusing?
            let bound_y2 = y + h + walk.margin.t + if walk.margin.b < 0. { walk.margin.b } else { 0. };
            if bound_y2 > turtle.bound_right_bottom.y {
                turtle.bound_right_bottom.y = bound_y2;
            }

            Rect { pos: vec2(x, y), size: vec2(w, h) }
        } else {
            Rect { pos: vec2(0.0, 0.0), size: vec2(w, h) }
        };

        if align_dx != 0.0 {
            if let Some(old_turtle) = old_turtle {
                self.do_align_x(align_dx, old_turtle.align_list_x_start_index);
            }
        };
        if align_dy != 0.0 {
            if let Some(old_turtle) = old_turtle {
                self.do_align_y(align_dy, old_turtle.align_list_y_start_index);
            }
        };

        ret
    }

    /// High performance turtle walk with no indirections and compute visibility.
    pub fn walk_turtle_right_no_wrap(&mut self, w: f32, h: f32, scroll: Vec2) -> Option<Rect> {
        if let Some(turtle) = self.turtles.last_mut() {
            let x = turtle.pos.x;
            let y = turtle.pos.y;
            // walk it normally
            turtle.pos.x += w;

            // keep track of biggest item in the line (include item margin bottom)
            let biggest = h;
            if biggest > turtle.biggest {
                turtle.biggest = biggest;
            }
            // update x2 bounds (margin right is only added if its negative)
            let bound_x2 = x + w;
            if bound_x2 > turtle.bound_right_bottom.x {
                turtle.bound_right_bottom.x = bound_x2;
            }
            // update y2 bounds (margin bottom is only added if its negative)
            let bound_y2 = turtle.pos.y + h;
            if bound_y2 > turtle.bound_right_bottom.y {
                turtle.bound_right_bottom.y = bound_y2;
            }

            let vx = turtle.origin.x + scroll.x;
            let vy = turtle.origin.y + scroll.y;
            let vw = turtle.width;
            let vh = turtle.height;

            if x > vx + vw || x + w < vx || y > vy + vh || y + h < vy {
                None
            } else {
                Some(Rect { pos: vec2(x, y), size: vec2(w, h) })
            }
        } else {
            None
        }
    }

    /// Explicitly move the current [`Turtle`] to a new line.
    ///
    /// TODO(JP): Mostly relevant for [`Direction::Right`], should we just disable
    /// this for [`Direction::Down`] to avoid confusion?
    pub fn turtle_new_line(&mut self) {
        if let Some(turtle) = self.turtles.last_mut() {
            match turtle.layout.direction {
                Direction::Right => {
                    turtle.pos.x = turtle.origin.x + turtle.layout.padding.l;
                    turtle.pos.y += turtle.biggest + turtle.layout.new_line_padding;
                    turtle.biggest = 0.0;
                }
                Direction::Down => {
                    turtle.pos.y = turtle.origin.y + turtle.layout.padding.t;
                    turtle.pos.x += turtle.biggest + turtle.layout.new_line_padding;
                    turtle.biggest = 0.0;
                }
            }
        }
    }

    /// [`Cx::turtle_new_line`] but allows setting a minimum height for the line.
    ///
    /// TODO(JP): Should we instead include `min_height` in [`Layout`]?
    pub fn turtle_new_line_min_height(&mut self, min_height: f32) {
        if let Some(turtle) = self.turtles.last_mut() {
            assert_eq!(turtle.layout.direction, Direction::Right);
            turtle.pos.x = turtle.origin.x + turtle.layout.padding.l;
            turtle.pos.y += turtle.biggest.max(min_height);
            turtle.biggest = 0.0;
        }
    }

    /// Check if a particular line is visible.
    ///
    /// TODO(JP): Only used in one place currently; should we instead expose
    /// more low-level primitives so the user can compute this themselves?
    pub fn turtle_line_is_visible(&mut self, min_height: f32, scroll: Vec2) -> bool {
        if let Some(turtle) = self.turtles.last_mut() {
            assert_eq!(turtle.layout.direction, Direction::Right);
            let y = turtle.pos.y;
            let h = turtle.biggest.max(min_height);
            let vy = turtle.origin.y + scroll.y;
            let vh = turtle.height;

            return !(y > vy + vh || y + h < vy);
        }
        false
    }

    /// Actually perform a horizontal movement of items in [`Cx::turtle_align_list`].
    ///
    /// TODO(JP): Should we move some of this stuff to [`Area`], where we already seem to do a bunch
    /// of rectangle and position calculations?
    fn do_align_x(&mut self, dx: f32, align_start: usize) {
        let dx = (dx * self.current_dpi_factor).floor() / self.current_dpi_factor;
        for i in align_start..self.turtle_align_list.len() {
            let align_item = &self.turtle_align_list[i];
            match align_item {
                Area::InstanceRange(inst) => {
                    let cxview = &mut self.views[inst.view_id];
                    let draw_call = &mut cxview.draw_calls[inst.draw_call_id];
                    let sh = &self.shaders[draw_call.shader.shader_id];
                    for i in 0..inst.instance_count {
                        if let Some(rect_pos) = sh.mapping.rect_instance_props.rect_pos {
                            draw_call.instances[inst.instance_offset + rect_pos + i * sh.mapping.instance_props.total_slots] +=
                                dx;
                        }
                    }
                }
                // TODO(JP): Would be nice to implement this for [`Align::View`], which would
                // probably require some offset field on [`CxView`] that gets used during rendering.
                _ => unreachable!(),
            }
        }
    }

    /// Actually perform a vertical movement of items in [`Cx::turtle_align_list`].
    ///
    /// TODO(JP): Should we move some of this stuff to [`Area`], where we already seem to do a bunch
    /// of rectangle and position calculations?
    fn do_align_y(&mut self, dy: f32, align_start: usize) {
        let dy = (dy * self.current_dpi_factor).floor() / self.current_dpi_factor;
        for i in align_start..self.turtle_align_list.len() {
            let align_item = &self.turtle_align_list[i];
            match align_item {
                Area::InstanceRange(inst) => {
                    let cxview = &mut self.views[inst.view_id];
                    let draw_call = &mut cxview.draw_calls[inst.draw_call_id];
                    let sh = &self.shaders[draw_call.shader.shader_id];
                    for i in 0..inst.instance_count {
                        if let Some(rect_pos) = sh.mapping.rect_instance_props.rect_pos {
                            draw_call.instances
                                [inst.instance_offset + rect_pos + 1 + i * sh.mapping.instance_props.total_slots] += dy;
                        }
                    }
                }
                // TODO(JP): Would be nice to implement this for `Align::View`, which would
                // probably require some offset field on `CxView` that gets used during rendering.
                _ => unreachable!(),
            }
        }
    }

    /// Get the [`Rect`] that contains the current [`Turtle::origin`], [`Turtle::width], and
    /// [`Turtle::height`]. Note that these are the inherent dimensions of the [`Turtle`], not
    /// what the [`Turtle`] has walked so far. See [`Cx::get_turtle_bounds`] for that.
    ///
    /// TODO(JP): When using [`Width::Compute`] or [`Height::Compute`], this [`Rect`] may include
    /// [`f32::NAN`]s, which is unexpected.
    pub fn get_turtle_rect(&self) -> Rect {
        if let Some(turtle) = self.turtles.last() {
            return Rect { pos: turtle.origin, size: vec2(turtle.width, turtle.height) };
        };
        Rect::default()
    }

    /// Get the bounds of what the turtle has *actually* walked (not just its
    /// inherent width/height as given by [`Cx::get_turtle_rect`]), including any padding that the
    /// layout of the current turtle specifies.
    pub fn get_turtle_bounds(&self) -> Vec2 {
        if let Some(turtle) = self.turtles.last() {
            return Vec2 {
                x: if turtle.bound_right_bottom.x < 0. { 0. } else { turtle.bound_right_bottom.x } + turtle.layout.padding.r
                    - turtle.origin.x,
                y: if turtle.bound_right_bottom.y < 0. { 0. } else { turtle.bound_right_bottom.y } + turtle.layout.padding.b
                    - turtle.origin.y,
            };
        }
        Vec2::default()
    }

    /// Overwrite the turtle bounds to pretend that it has actually walked
    /// a bunch.
    ///
    /// TODO(JP): this seems.. bad? Can we restructure all this?
    pub fn set_turtle_bounds(&mut self, bound: Vec2) {
        if let Some(turtle) = self.turtles.last_mut() {
            turtle.bound_right_bottom = Vec2 {
                x: bound.x - turtle.layout.padding.r + turtle.origin.x,
                y: bound.y - turtle.layout.padding.b + turtle.origin.y,
            }
        }
    }

    /// Same as [`Cx::get_turtle_rect().pos`].
    ///
    /// TODO(JP): Do we really need two different methods to get to the same data?
    pub fn get_turtle_origin(&self) -> Vec2 {
        if let Some(turtle) = self.turtles.last() {
            return turtle.origin;
        }
        Vec2::default()
    }

    /// Get the current [`Turtle::pos`] in absolute coordinates.
    ///
    /// See also [`Cx::get_rel_turtle_pos`].
    ///
    /// TODO(JP): Only used in two places currently; do we really need this?
    pub fn get_turtle_pos(&self) -> Vec2 {
        if let Some(turtle) = self.turtles.last() {
            turtle.pos
        } else {
            Vec2::default()
        }
    }

    /// Get the current [`Turtle::pos`] in coordinates relative to [`Turtle::origin`].
    ///
    /// See also [`Cx::get_turtle_pos`].
    ///
    /// TODO(JP): Only used in one place currently; do we really need this?
    pub fn get_rel_turtle_pos(&self) -> Vec2 {
        if let Some(turtle) = self.turtles.last() {
            Vec2 { x: turtle.pos.x - turtle.origin.x, y: turtle.pos.y - turtle.origin.y }
        } else {
            Vec2::default()
        }
    }

    /// Manually change [`Turtle::pos`]. Warning! Does not update [`Turtle::bound_right_bottom`],
    /// like [`Cx::walk_turtle`] does; might result in unexpected behavior.
    ///
    /// TODO(JP): Should we delete this and just always use [`Cx::walk_turtle`] instead?
    pub fn move_turtle(&mut self, dx: f32, dy: f32) {
        if let Some(turtle) = self.turtles.last_mut() {
            turtle.pos.x += dx;
            turtle.pos.y += dy;
        }
    }

    /// Manually change [`Turtle::pos`]. Warning! Does not update [`Turtle::bound_right_bottom`],
    /// like [`Cx::walk_turtle`] does; might result in unexpected behavior.
    ///
    /// TODO(JP): Should we delete this and just always use [`Cx::walk_turtle`] instead?
    pub fn set_turtle_pos(&mut self, pos: Vec2) {
        if let Some(turtle) = self.turtles.last_mut() {
            turtle.pos = pos
        }
    }

    /// Returns how many pixels we should move over based on the [`Align::fx`] ratio
    /// (which is between 0 and 1). We do this by looking at the bound
    /// ([`Turtle::bound_right_bottom`]) to see how much we have actually drawn, and how
    /// subtract that from the width of this turtle. That "remaining width" is
    /// then multiplied with the ratio. If there is no inherent width then this
    /// will return 0.
    fn compute_align_turtle_x(turtle: &Turtle) -> f32 {
        if turtle.layout.align.fx > 0.0 {
            let dx = turtle.layout.align.fx
                * ((turtle.width - turtle.width_used - (turtle.layout.padding.l + turtle.layout.padding.r))
                    - (turtle.bound_right_bottom.x - (turtle.origin.x + turtle.layout.padding.l)));
            if dx.is_nan() {
                return 0.0;
            }
            dx
        } else {
            0.
        }
    }

    /// Returns how many pixels we should move over based on the [`Align::fy`] ratio
    /// (which is between 0 and 1). We do this by looking at the bound
    /// ([`Turtle::bound_right_bottom`]) to see how much we have actually drawn, and how
    /// subtract that from the height of this turtle. That "remaining height" is
    /// then multiplied with the ratio. If there is no inherent height then this
    /// will return 0.
    fn compute_align_turtle_y(turtle: &Turtle) -> f32 {
        if turtle.layout.align.fy > 0.0 {
            let dy = turtle.layout.align.fy
                * ((turtle.height - turtle.height_used - (turtle.layout.padding.t + turtle.layout.padding.b))
                    - (turtle.bound_right_bottom.y - (turtle.origin.y + turtle.layout.padding.t)));
            if dy.is_nan() {
                return 0.0;
            }
            dy
        } else {
            0.
        }
    }

    /// If the height is computed, then this will set the height to however much is
    /// drawn so far, using [`Turtle::bound_right_bottom`].
    ///
    /// TODO(JP): This also resets the [`Turtle::bound_right_bottom.y`] back to 0 as if
    /// nothing has been drawn. How does that make sense exactly? Is that wrong?
    ///
    /// TODO(JP): This function is currently only used once..
    pub fn compute_turtle_height(&mut self) {
        if let Some(turtle) = self.turtles.last_mut() {
            if turtle.height.is_nan() && turtle.bound_right_bottom.y != std::f32::NEG_INFINITY {
                // nothing happened use the padding
                turtle.height = max_zero_keep_nan(turtle.bound_right_bottom.y - turtle.origin.y + turtle.layout.padding.b);
                turtle.height_used = 0.;
                turtle.bound_right_bottom.y = 0.;
            }
        }
    }

    /// Used for `a<b>c` layouts horizontally.
    pub fn change_turtle_align_x_cab(&mut self, fx: f32) {
        self.change_turtle_align_x(fx, true);
    }

    /// Used for `a<b>c` layouts horizontally.
    ///
    /// This moves all elements in the align list from the start of the turtle
    /// (or since the last time this function was called; it uses
    /// [`turtle.align_list_x_start_index`]) based on the available width and
    /// the given ratio `fx`. If `width_used` is set, then the width of the
    /// elements that were drawn (using [`Turtle::bound_right_bottom`]) is set in [`Turtle::width_used`]
    /// so it will be excluded from future alignments.
    ///
    /// TODO(JP): [`Turtle::width_used`] is set, not *added*, so you can't use this multiple
    /// times in a row cumulatively.
    fn change_turtle_align_x(&mut self, fx: f32, width_used: bool) {
        let (dx, align_origin_x) = if let Some(turtle) = self.turtles.last_mut() {
            (Self::compute_align_turtle_x(turtle), turtle.align_list_x_start_index)
        } else {
            (0., 0)
        };
        if dx > 0.0 {
            self.do_align_x(dx, align_origin_x);
        }
        // reset turtle props
        if let Some(turtle) = self.turtles.last_mut() {
            turtle.align_list_x_start_index = self.turtle_align_list.len();
            // TODO(JP): why do we overwrite [`Align::fx`] here?
            turtle.layout.align.fx = fx;
            // TODO(JP): should this be additive?
            turtle.width_used = if width_used { turtle.bound_right_bottom.x - turtle.origin.x } else { 0.0 };
            turtle.bound_right_bottom.x = std::f32::NEG_INFINITY;
        }
    }

    /// Used for `a<b>c` layouts vertically.
    pub fn change_turtle_align_y_cab(&mut self, fx: f32) {
        self.change_turtle_align_y(fx, true);
    }

    /// Used for `a<b>c` layouts vertically.
    ///
    /// This moves all elements in the align list from the start of the turtle
    /// (or since the last time this function was called; it uses
    /// [`turtle.align_list_y_start_index`]) based on the available height and
    /// the given ratio `fy`. If `height_used` is set, then the height of the
    /// elements that were drawn (using bound_right_bottom) is set in [`Turtle::height_used`]
    /// so it will be excluded from future alignments.
    ///
    /// TODO(JP): [`Turtle::height_used`] is set, not *added*, so you can't use this multiple
    /// times in a row cumulatively.
    fn change_turtle_align_y(&mut self, fy: f32, height_used: bool) {
        let (dy, align_origin_y) = if let Some(turtle) = self.turtles.last_mut() {
            (Self::compute_align_turtle_y(turtle), turtle.align_list_y_start_index)
        } else {
            (0.0, 0)
        };
        if dy > 0.0 {
            self.do_align_y(dy, align_origin_y);
        }
        // reset turtle props
        if let Some(turtle) = self.turtles.last_mut() {
            turtle.align_list_y_start_index = self.turtle_align_list.len();
            turtle.layout.align.fy = fy;
            turtle.height_used = if height_used { turtle.bound_right_bottom.y - turtle.origin.y } else { 0.0 };
            turtle.bound_right_bottom.y = std::f32::NEG_INFINITY;
        }
    }

    /// Horizontally aligns elements that have been drawn so far, using [`Layout::align`].
    pub fn turtle_align_x(&mut self) {
        let fx = if let Some(turtle) = self.turtles.last_mut() {
            turtle.layout.align.fx
        } else {
            return;
        };
        self.change_turtle_align_x(fx, false);
    }

    /// Vertically aligns elements that have been drawn so far, using [`Layout::align`].
    pub fn turtle_align_y(&mut self) {
        let fy = if let Some(turtle) = self.turtles.last_mut() {
            turtle.layout.align.fy
        } else {
            return;
        };
        self.change_turtle_align_y(fy, false);
    }

    /// Reset the current position of the current [`Turtle`] to the starting position
    /// ([`Turtle::origin`] + [`Layout::padding`]).
    ///
    /// TODO(JP): Note that this does *not* reset [`Turtle::bound_right_bottom`] or
    /// [`Turtle::biggest`] or stuff like that, so there is still some leftover state which
    /// might be confusing.
    pub fn reset_turtle_pos(&mut self) {
        if let Some(turtle) = self.turtles.last_mut() {
            // subtract used size so 'fill' works
            turtle.pos = Vec2 { x: turtle.origin.x + turtle.layout.padding.l, y: turtle.origin.y + turtle.layout.padding.t };
        }
    }

    fn _get_width_left(&self, abs: bool, abs_size: f32) -> f32 {
        if !abs {
            self.get_width_left()
        } else {
            abs_size
        }
    }

    /// Get some notion of the width that is "left" for the current [`Turtle`].
    ///
    /// See also [`Cx::get_width_total`].
    pub fn get_width_left(&self) -> f32 {
        if let Some(turtle) = self.turtles.last() {
            let nan_val = max_zero_keep_nan(turtle.width - turtle.width_used - (turtle.pos.x - turtle.origin.x));
            if nan_val.is_nan() {
                // if we are a computed width, if some value is known, use that
                // TODO(JP): this makes no sense to me. This kind of makes sense for the total
                // width, but not for the remaining width? Shouldn't we just be returning NaN here?
                if turtle.bound_right_bottom.x != std::f32::NEG_INFINITY {
                    return turtle.bound_right_bottom.x - turtle.origin.x;
                }
            }
            return nan_val;
        }
        0.
    }

    fn _get_width_total(&self, abs: bool, abs_size: f32) -> f32 {
        if !abs {
            self.get_width_total()
        } else {
            abs_size
        }
    }

    /// Get some notion of the total width of the current turtle. If the width
    /// is well defined, then we return it. If it's computed, then we return the
    /// bound (including padding) of how much we've drawn so far. And if we haven't
    /// drawn anything, we return NaN.
    pub fn get_width_total(&self) -> f32 {
        if let Some(turtle) = self.turtles.last() {
            let nan_val = max_zero_keep_nan(turtle.width /* - (turtle.layout.padding.l + turtle.layout.padding.r)*/);
            if nan_val.is_nan() {
                // if we are a computed width, if some value is known, use that
                if turtle.bound_right_bottom.x != std::f32::NEG_INFINITY {
                    return turtle.bound_right_bottom.x - turtle.origin.x + turtle.layout.padding.r;
                }
            }
            return nan_val;
        }
        0.
    }

    fn _get_height_left(&self, abs: bool, abs_size: f32) -> f32 {
        if !abs {
            self.get_height_left()
        } else {
            abs_size
        }
    }

    /// Get some notion of the height that is "left" for the current [`Turtle`].
    ///
    /// See also [`Cx::get_height_total`].
    pub fn get_height_left(&self) -> f32 {
        if let Some(turtle) = self.turtles.last() {
            let nan_val = max_zero_keep_nan(turtle.height - turtle.height_used - (turtle.pos.y - turtle.origin.y));
            if nan_val.is_nan() {
                // if we are a computed height, if some value is known, use that
                // TODO(JP): this makes no sense to me. This kind of makes sense for the total
                // height, but not for the remaining height? Shouldn't we just be returning NaN here?
                if turtle.bound_right_bottom.y != std::f32::NEG_INFINITY {
                    return turtle.bound_right_bottom.y - turtle.origin.y;
                }
            }
            return nan_val;
        }
        0.
    }

    fn _get_height_total(&self, abs: bool, abs_size: f32) -> f32 {
        if !abs {
            self.get_height_total()
        } else {
            abs_size
        }
    }

    /// Get some notion of the total height of the current turtle. If the height
    /// is well defined, then we return it. If it's computed, then we return the
    /// bound (including padding) of how much we've drawn so far. And if we haven't
    /// drawn anything, we return NaN.
    pub fn get_height_total(&self) -> f32 {
        if let Some(turtle) = self.turtles.last() {
            let nan_val = max_zero_keep_nan(turtle.height /*- (turtle.layout.padding.t + turtle.layout.padding.b)*/);
            if nan_val.is_nan() {
                // if we are a computed height, if some value is known, use that
                if turtle.bound_right_bottom.y != std::f32::NEG_INFINITY {
                    return turtle.bound_right_bottom.y - turtle.origin.y + turtle.layout.padding.b;
                }
            }
            return nan_val;
        }
        0.
    }

    /// Whether the current [`Turtle::layout`] uses [`Height::Compute`].
    ///
    /// TODO(JP): We only use this in one place; is this really necessary?
    pub fn is_height_computed(&self) -> bool {
        if let Some(turtle) = self.turtles.last() {
            if let Height::Compute = turtle.layout.walk.height {
                return true;
            }
        }
        false
    }

    fn eval_width(&self, width: &Width, margin: Margin, abs: bool, abs_pos: f32) -> f32 {
        match width {
            Width::Compute => std::f32::NAN,
            Width::Fix(v) => max_zero_keep_nan(*v),
            Width::Fill => max_zero_keep_nan(self._get_width_left(abs, abs_pos) - (margin.l + margin.r)),
        }
    }

    fn eval_height(&self, height: &Height, margin: Margin, abs: bool, abs_pos: f32) -> f32 {
        match height {
            Height::Compute => std::f32::NAN,
            Height::Fix(v) => max_zero_keep_nan(*v),
            Height::Fill => max_zero_keep_nan(self._get_height_left(abs, abs_pos) - (margin.t + margin.b)),
        }
    }

    /// Add an `Area::InstanceRange` to the [`Cx::turtle_align_list`], so that it will get aligned,
    /// e.g. when you call [`Cx::end_turtle`].
    pub fn add_to_turtle_align_list(&mut self, area: Area) {
        match area {
            Area::InstanceRange(_) => self.turtle_align_list.push(area),
            _ => panic!("Only Area::InstanceRange can be aligned currently"),
        }
    }
}

fn max_zero_keep_nan(v: f32) -> f32 {
    if v.is_nan() {
        v
    } else {
        f32::max(v, 0.0)
    }
}
