//! Managing GPU textures.

use crate::cx::*;

/// A pointer to a [`CxTexture`] (indexed in [`Cx::textures`] using [`Texture::texture_id`]),
#[derive(Copy, Clone, PartialEq, Debug)]
pub struct Texture {
    pub texture_id: u32,
}

/// A reinterpretation of [`Texture`] as a 2D texture, for use in [`Shader`]s.
#[derive(Copy, Clone, Default, PartialEq, Debug)]
pub struct Texture2D(pub Option<u32>);

impl Into<Texture2D> for Texture {
    fn into(self) -> Texture2D {
        Texture2D(Some(self.texture_id as u32))
    }
}

impl Texture {
    pub fn new(cx: &mut Cx) -> Texture {
        Texture {
            texture_id: {
                cx.textures.push(CxTexture::default());
                (cx.textures.len() - 1) as u32
            },
        }
    }
}

#[derive(Clone, PartialEq)]
pub(crate) enum TextureFormat {
    Default,
    ImageBGRA,
    #[allow(dead_code)] // currently never used
    Depth32Stencil8,
    #[allow(dead_code)] // currently never used
    RenderBGRA,
    #[allow(dead_code)] // currently never used
    RenderBGRAf16,
    #[allow(dead_code)] // currently never used
    RenderBGRAf32,
    //    ImageBGRAf32,
    //    ImageRf32,
    //    ImageRGf32,
    //    MappedBGRA,
    //    MappedBGRAf32,
    //    MappedRf32,
    //    MappedRGf32,
}

#[derive(Clone, PartialEq)]
pub(crate) struct TextureDesc {
    pub(crate) format: TextureFormat,
    pub(crate) width: Option<usize>,
    pub(crate) height: Option<usize>,
    pub(crate) multisample: Option<usize>,
}

impl Default for TextureDesc {
    fn default() -> Self {
        TextureDesc { format: TextureFormat::Default, width: None, height: None, multisample: None }
    }
}

/// Texture data, which you can render as an image in your shaders. See e.g. [`crate::DrawImage`].
#[derive(Default)]
pub struct CxTexture {
    pub(crate) desc: TextureDesc,
    pub(crate) image_u32: Vec<u32>,
    pub(crate) update_image: bool,
    // Not used on wasm
    #[cfg_attr(target_arch = "wasm32", allow(dead_code))]
    pub(crate) platform: CxPlatformTexture,
}
