//! Various macros.

/// Define the entry point of your application.
///
/// Your `$app` should implement the `app_load`, `draw_app` and `handle_app`
/// functions. Refer to the examples to get started.
#[macro_export]
macro_rules! main_app {
    ( $ app: ident) => {
        #[cfg(not(target_arch = "wasm32"))]
        fn main() {
            //TODO do this with a macro to generate both entrypoints for App and Cx
            let mut cx = Cx::default();
            cx.app_load();
            $app::app_load(&mut cx);
            let mut app = $app::new(&mut cx);
            let mut cxafterdraw = CxAfterDraw::new(&mut cx);
            cx.event_loop(|cx, mut event| {
                if let Event::Draw = event {
                    app.draw_app(cx);
                    cxafterdraw.after_draw(cx);
                    return;
                }
                app.handle_app(cx, &mut event);
            });
        }

        #[cfg(target_arch = "wasm32")]
        fn main() {}

        #[cfg(target_arch = "wasm32")]
        #[export_name = "create_wasm_app"]
        pub extern "C" fn create_wasm_app() -> u32 {
            std::panic::set_hook(Box::new(|info: &std::panic::PanicInfo| {
                log!("Panic: {}", info.to_string());
            }));
            let mut cx = Box::new(Cx::default());
            cx.app_load();
            $app::app_load(&mut cx);
            let app = Box::new($app::new(&mut cx));
            let cxafterdraw = Box::new(CxAfterDraw::new(&mut cx));
            Box::into_raw(Box::new((Box::into_raw(app), Box::into_raw(cx), Box::into_raw(cxafterdraw)))) as u32
        }

        #[cfg(target_arch = "wasm32")]
        #[export_name = "process_to_wasm"]
        pub unsafe extern "C" fn process_to_wasm(appcx: u32, msg_bytes: u32) -> u32 {
            let appcx = &*(appcx as *mut (*mut $app, *mut Cx, *mut CxAfterDraw));
            (*appcx.1).process_to_wasm(msg_bytes, |cx, mut event| {
                if let Event::Draw = event {
                    (*appcx.0).draw_app(cx);
                    (*appcx.2).after_draw(cx);
                    return;
                };
                (*appcx.0).handle_app(cx, &mut event);
            })
        }
    };
}

/// Generates a unique number.
/// TODO(JP): might be better to replace this with [`crate::hash::LocationHash`]?
#[macro_export]
macro_rules! uid {
    () => {{
        struct Unique {}
        std::any::TypeId::of::<Unique>().into()
    }};
}

/// Generates a [`crate::hash::LocationHash`] based on the current file/line/column.
#[macro_export]
macro_rules! location_hash {
    () => {
        LocationHash::new(file!(), line!() as u64, column!() as u64)
    };
}

/// Generates a constant that contains the current filename in it.
///
/// Example: `define_string_with_filename!(MY_SHADER)` expands to
/// `const MY_SHADER: &str = "myfile.rs::MY_SHADER";`.
///
/// This is useful for giving names to things like [`crate::shader::Shader`]s, since the name
/// will be unique, and easily recognizable during debugging.
#[macro_export]
macro_rules! define_string_with_filename {
    ( $ id: ident ) => {
        const $id: &str = concat!(file!(), "::", stringify!($id));
    };
}

/// Tag a piece of code with filename+line+col. The line+col are done in a hacky
/// way, exploiting the fact that rustfmt usually puts the multiline string
/// start on a newline, with one greater indentation level. This doesn't always
/// work but it's close enough!
///
/// We could at some point use something like this:
/// <https://github.com/makepad/makepad/blob/719012b9348815acdcaec6365a99443b9208aecc/main/live_body/src/lib.rs#L47>
/// But that depends on the nightly `proc_macro_span` feature (<https://github.com/rust-lang/rust/issues/54725>).
#[macro_export]
macro_rules! code_fragment {
    ( $ code: expr ) => {
        CodeFragment { filename: file!(), line: line!() as usize + 1, col: column!() as usize + 7, code: $code }
    };
}

/// Logging helper that works both on native and WebAssembly targets.
#[cfg(not(target_arch = "wasm32"))]
#[macro_export]
macro_rules! log {
    ( $ ( $t: tt) *) => {
        println!("{}:{} - {}",file!(),line!(),format!($($t)*))
    }
}

/// Logging helper that works both on native and WebAssembly targets.
///
/// TODO(JP): Would be better to integrate this with the normal println! dbg! etc macros,
/// so we don't need special treatment.
#[cfg(target_arch = "wasm32")]
#[macro_export]
macro_rules! log {
    ( $ ( $ t: tt) *) => {
        console_log(&format!("{}:{} - {}", file!(), line!(), format!( $ ( $ t) *)))
    }
}
