//! Events coming from user actions, system calls, and so on.

use crate::cx::*;
use std::any::TypeId;
use std::collections::{BTreeSet, HashMap};

/// Modifiers that were held when a key event was fired.
#[derive(Clone, Debug, PartialEq, Default)]
pub struct KeyModifiers {
    pub shift: bool,
    pub control: bool,
    pub alt: bool,
    pub logo: bool,
}

/// The type of input that was used to trigger a finger event.
#[derive(Clone, Debug, PartialEq)]
pub enum FingerInputType {
    Mouse,
    Touch,
    XR,
}

impl FingerInputType {
    pub fn is_touch(&self) -> bool {
        *self == FingerInputType::Touch
    }
    pub fn is_mouse(&self) -> bool {
        *self == FingerInputType::Mouse
    }
    pub fn is_xr(&self) -> bool {
        *self == FingerInputType::XR
    }
    pub fn has_hovers(&self) -> bool {
        *self == FingerInputType::Mouse || *self == FingerInputType::XR
    }
}

impl Default for FingerInputType {
    fn default() -> Self {
        Self::Mouse
    }
}

/// A conceptual "finger" (mouse, actual finger, etc) was pressed down.
#[derive(Clone, Default, Debug, PartialEq)]
pub struct FingerDownEvent {
    pub window_id: usize,
    pub abs: Vec2,
    pub rel: Vec2,
    pub rect: Rect,
    pub digit: usize,
    pub tap_count: u32,
    pub(crate) handled: bool,
    pub input_type: FingerInputType,
    pub modifiers: KeyModifiers,
    pub time: f64,
}

/// A conceptual "finger" (mouse, actual finger, etc) was moved.
#[derive(Clone, Default, Debug, PartialEq)]
pub struct FingerMoveEvent {
    pub window_id: usize,
    pub abs: Vec2,
    pub abs_start: Vec2,
    pub rel: Vec2,
    pub rel_start: Vec2,
    pub rect: Rect,
    pub is_over: bool,
    pub digit: usize,
    pub input_type: FingerInputType,
    pub modifiers: KeyModifiers,
    pub time: f64,
}

impl FingerMoveEvent {
    pub fn move_distance(&self) -> f32 {
        ((self.abs_start.x - self.abs.x).powf(2.) + (self.abs_start.y - self.abs.y).powf(2.)).sqrt()
    }
}

/// A conceptual "finger" (mouse, actual finger, etc) was released.
#[derive(Clone, Default, Debug, PartialEq)]
pub struct FingerUpEvent {
    pub window_id: usize,
    pub abs: Vec2,
    pub abs_start: Vec2,
    pub rel: Vec2,
    pub rel_start: Vec2,
    pub rect: Rect,
    pub digit: usize,
    pub is_over: bool,
    pub input_type: FingerInputType,
    pub modifiers: KeyModifiers,
    pub time: f64,
}

/// The type of [`FingerHoverEvent`].
#[derive(Clone, Debug, PartialEq)]
pub enum HoverState {
    In,
    Over,
    Out,
}

impl Default for HoverState {
    fn default() -> HoverState {
        HoverState::Over
    }
}

/// A conceptual "finger" (mouse, actual finger, etc) was hovered over the screen.
#[derive(Clone, Default, Debug, PartialEq)]
pub struct FingerHoverEvent {
    pub window_id: usize,
    pub digit: usize,
    pub abs: Vec2,
    pub rel: Vec2,
    pub rect: Rect,
    pub any_down: bool,
    pub(crate) handled: bool,
    pub hover_state: HoverState,
    pub modifiers: KeyModifiers,
    pub time: f64,
}

/// A conceptual "finger" (mouse, actual finger, etc) triggered a scroll.
#[derive(Clone, Default, Debug, PartialEq)]
pub struct FingerScrollEvent {
    pub window_id: usize,
    pub digit: usize,
    pub abs: Vec2,
    pub rel: Vec2,
    pub rect: Rect,
    pub scroll: Vec2,
    pub input_type: FingerInputType,
    //pub is_wheel: bool,
    pub handled_x: bool,
    pub handled_y: bool,
    pub modifiers: KeyModifiers,
    pub time: f64,
}

/// Geometry of a [`Window`] changed (position, size, etc).
#[derive(Clone, Default, Debug, PartialEq)]
pub struct WindowGeomChangeEvent {
    pub window_id: usize,
    pub old_geom: WindowGeom,
    pub new_geom: WindowGeom,
}

/// A `FileRead` that was requested using [`Cx::file_read`] has been completed.
#[derive(Clone, Debug, PartialEq)]
pub struct FileReadEvent {
    pub read_id: u64,
    pub universal_file_handle: Result<UniversalFileHandle, String>,
}

/// A [`Timer`] that was requested using [`Cx::start_timer`] has fired.
#[derive(Clone, Debug, PartialEq)]
pub struct TimerEvent {
    pub timer_id: u64,
}

/// Represents a signal that was fired from [`Cx::send_signal`]. Can be captured
/// with [`Signal`].
///
/// TODO(JP): Is this a bit too complicated of an API? What about if we just
/// send `pub signal: u64`? Then you can use it for anything, including pointers,
/// [`StringHash`]es, etc.
#[derive(Clone, Debug, PartialEq)]
pub struct SignalEvent {
    pub signals: HashMap<Signal, BTreeSet<StatusId>>,
}

/// Data for various kinds of key-based events ([`Event::KeyDown`], [`Event::KeyUp`], etc).
#[derive(Clone, Debug, PartialEq)]
pub struct KeyEvent {
    pub key_code: KeyCode,
    //pub key_char: char,
    pub is_repeat: bool,
    pub modifiers: KeyModifiers,
    pub time: f64,
}

/// Called when [`Cx::key_focus`] changes.
#[derive(Clone, Debug, PartialEq)]
pub struct KeyFocusEvent {
    pub prev: Area,
    pub focus: Area,
}

/// Some text was inputted. Rely on this for text input instead of [`KeyEvent`]s.
#[derive(Clone, Debug, PartialEq)]
pub struct TextInputEvent {
    pub input: String,
    pub replace_last: bool,
    pub was_paste: bool,
}

/// The user requested text to be copied to the clipboard.
#[derive(Clone, Debug, PartialEq)]
pub struct TextCopyEvent {
    pub response: Option<String>,
}

/// The user requested to close the [`Window`].
#[derive(Clone, Debug, PartialEq)]
pub struct WindowCloseRequestedEvent {
    pub window_id: usize,
    pub accept_close: bool,
}

/// The [`Window`] actually closed.
#[derive(Clone, Debug, PartialEq)]
pub struct WindowClosedEvent {
    pub window_id: usize,
}

/// The user started or ended resizing the [`Window`].
///
/// TODO(JP): Mostly for internal use in Windows; we might not want to expose this
/// to end users?
#[derive(Clone, Debug, PartialEq)]
pub struct WindowResizeLoopEvent {
    pub was_started: bool,
    pub window_id: usize,
}

/// Response to operating system inquiry if a [`Window`] can be dragged.
#[derive(Clone, Debug, PartialEq)]
pub enum WindowDragQueryResponse {
    NoAnswer,
    Client,
    Caption,
    SysMenu, // windows only
}

/// The operating system inquired if a [`Window`] can be dragged.
#[derive(Clone, Debug, PartialEq)]
pub struct WindowDragQueryEvent {
    pub window_id: usize,
    pub abs: Vec2,
    pub response: WindowDragQueryResponse,
}

/// A websocket message was received.
#[derive(Clone, Debug, PartialEq)]
pub struct WebSocketMessageEvent {
    pub url: String,
    pub result: Result<Vec<u8>, String>,
}

/// A file handle that abstracts over the different ways we have to deal with different kinds of
/// files (local files, file URLs, dragged in files).
///
/// You can read these different types of file handles using [`Cx::get_file_reader`].
///
/// You get these e.g. through [`FileReadEvent`] or [`AppOpenFilesEvent`].
#[derive(Clone, Debug, PartialEq)]
pub enum UniversalFileHandle {
    /// Actually resolved data; contains the entire file.
    Data(Vec<u8>),

    /// Full path. Only available on native targets.
    #[cfg(not(target_arch = "wasm32"))]
    Path(String),

    /// A handle to a file in Javascript.
    #[cfg(target_arch = "wasm32")]
    WasmFile { id: usize, size: u64 },

    /// Used to be [`UniversalFileHandle::Data`], but we consumed it, so it's gone.
    Consumed,
}

/// A file that was supplied by a user, as opposed to by the application itself (like font resources
/// and such).
#[derive(Clone, Debug, PartialEq)]
pub struct UserFile {
    /// Per UNIX convention, basename is the filename (including extension) part.
    /// This is the only part of the filename that is exposed on all platforms (Wasm hides the
    /// full path).
    pub basename: String,
    /// The actual file handle.
    pub universal_file_handle: UniversalFileHandle,
}

/// This is an application-level event intended for platforms that have some sort of notion of
/// globally opening a file.
#[derive(Clone, Debug, PartialEq)]
pub struct AppOpenFilesEvent {
    pub user_files: Vec<UserFile>,
}

/// Global event that gets passed into `handle_app`, and which you can pass down to your own widgets.
///
/// TODO(JP): This is kind of a mishmash of events that are communicated between the lower level
/// windowing layers, and the actual user application. Would be good to split out the low level
/// events to a `SystemEvent` or `PlatformEvent` or so, and keep [`Event`] as just user-level events.
#[derive(Clone, Debug, PartialEq)]
pub enum Event {
    /// No event, to avoid `[Option<Event>]` all over the place.
    None,
    /// App gets started. Should be the very first event that gets fired.
    Construct,
    /// We're going to call `draw_app`.
    ///
    /// TODO(JP): Seems to be for internal use only; let's maybe not expose this to users?
    Draw,
    /// We're going to repaint our draw tree.
    ///
    /// TODO(JP): Seems to be for internal use only; let's maybe not expose this to users?
    Paint,
    /// App gained focus.
    ///
    /// TODO(JP): Rename to `AppFocusGained` to be more symmetric with [`Event::AppFocusLost`]?
    AppFocus,
    /// App lost focus.
    AppFocusLost,
    /// We're going to paint a new frame. Useful for animations; you can request this using [`Cx::request_next_frame`].
    NextFrame,
    /// The system wants us to set a different mouse cursor.
    ///
    /// TODO(JP): Seems to be for internal use only; let's maybe not expose this to users?
    WindowSetHoverCursor(MouseCursor),
    // See [`WindowDragQueryEvent`]
    WindowDragQuery(WindowDragQueryEvent),
    // See [`WindowCloseRequestedEvent`]
    WindowCloseRequested(WindowCloseRequestedEvent),
    // See [`WindowClosedEvent`]
    WindowClosed(WindowClosedEvent),
    // See [`WindowGeomChangeEvent`]
    WindowGeomChange(WindowGeomChangeEvent),
    // See [`WindowResizeLoopEvent`]
    WindowResizeLoop(WindowResizeLoopEvent),
    // See [`FingerDownEvent`]
    FingerDown(FingerDownEvent),
    // See [`FingerMoveEvent`]
    FingerMove(FingerMoveEvent),
    // See [`FingerHoverEvent`]
    FingerHover(FingerHoverEvent),
    // See [`FingerUpEvent`]
    FingerUp(FingerUpEvent),
    // See [`FingerScrollEvent`]
    FingerScroll(FingerScrollEvent),
    // See [`FileReadEvent`]
    FileRead(FileReadEvent),
    // See [`TimerEvent`]
    Timer(TimerEvent),
    // See [`SignalEvent`]
    Signal(SignalEvent),
    // See [`CommandId`]
    Command(CommandId),
    // See [`KeyFocusEvent`]
    KeyFocus(KeyFocusEvent),
    // See [`KeyFocusEvent`]
    KeyFocusLost(KeyFocusEvent),
    // User pressed down a key. See also [`KeyEvent`]
    KeyDown(KeyEvent),
    // User released a key. See also [`KeyEvent`]
    KeyUp(KeyEvent),
    // See [`TextInputEvent`]
    TextInput(TextInputEvent),
    // See [`TextCopyEvent`]
    TextCopy(TextCopyEvent),
    // See [`WebSocketMessageEvent`]
    WebSocketMessage(WebSocketMessageEvent),
    // See [`AppOpenFilesEvent`]
    AppOpenFiles(AppOpenFilesEvent),
}

impl Default for Event {
    fn default() -> Event {
        Event::None
    }
}

/// Modify the behavior of [`Event::hits`].
#[derive(Clone, Debug, Default)]
pub struct HitOpt {
    pub use_multi_touch: bool,
    pub margin: Option<Margin>,
}

impl Event {
    /// Checks if an [`Event`] matches an [`Area`].
    ///
    /// For key/text events, it checks if the given [`Area`] matches _exactly_
    /// the one currently set in [`Cx::key_focus`] (which gets set through
    /// [`Cx::set_key_focus`]).
    ///
    /// For mouse events, it checks if the event coordinates fall in the [`Rect`]
    /// of the _first instance_ of the [`Area`].
    ///
    /// TODO(JP): Only checking the "first instance" for mouse events is
    /// confusing. Ideally we would check if the event falls within any of the
    /// instances covered by the [`Area`].
    pub fn hits(&mut self, cx: &mut Cx, area: Area, opt: HitOpt) -> Event {
        if area.is_empty() {
            return Event::None;
        }
        match self {
            Event::KeyFocus(kf) => {
                if area == kf.prev {
                    return Event::KeyFocusLost(kf.clone());
                } else if area == kf.focus {
                    return Event::KeyFocus(kf.clone());
                }
            }
            Event::KeyDown(_) => {
                if area == cx.key_focus {
                    return self.clone();
                }
            }
            Event::KeyUp(_) => {
                if area == cx.key_focus {
                    return self.clone();
                }
            }
            Event::TextInput(_) => {
                if area == cx.key_focus {
                    return self.clone();
                }
            }
            Event::TextCopy(_) => {
                if area == cx.key_focus {
                    return Event::TextCopy(TextCopyEvent { response: None });
                }
            }
            Event::FingerScroll(fe) => {
                let rect = area.get_rect_for_first_instance(cx);
                if rect_contains_with_margin(&rect, fe.abs, &opt.margin) {
                    //fe.handled = true;
                    return Event::FingerScroll(FingerScrollEvent { rel: fe.abs - rect.pos, rect, ..fe.clone() });
                }
            }
            Event::FingerHover(fe) => {
                let rect = area.get_rect_for_first_instance(cx);

                if cx.fingers[fe.digit]._over_last == area {
                    let mut any_down = false;
                    for finger in &cx.fingers {
                        if finger.captured == area {
                            any_down = true;
                            break;
                        }
                    }
                    if !fe.handled && rect_contains_with_margin(&rect, fe.abs, &opt.margin) {
                        fe.handled = true;
                        if let HoverState::Out = fe.hover_state {
                            //    cx.finger_over_last_area = Area::Empty;
                        } else {
                            cx.fingers[fe.digit].over_last = area;
                        }
                        return Event::FingerHover(FingerHoverEvent {
                            rel: area.get_relative_to_first_instance(cx, fe.abs),
                            rect,
                            any_down,
                            ..fe.clone()
                        });
                    } else {
                        //self.was_over_last_call = false;
                        return Event::FingerHover(FingerHoverEvent {
                            rel: area.get_relative_to_first_instance(cx, fe.abs),
                            rect,
                            any_down,
                            hover_state: HoverState::Out,
                            ..fe.clone()
                        });
                    }
                } else if !fe.handled && rect_contains_with_margin(&rect, fe.abs, &opt.margin) {
                    let mut any_down = false;
                    for finger in &cx.fingers {
                        if finger.captured == area {
                            any_down = true;
                            break;
                        }
                    }
                    cx.fingers[fe.digit].over_last = area;
                    fe.handled = true;
                    //self.was_over_last_call = true;
                    return Event::FingerHover(FingerHoverEvent {
                        rel: area.get_relative_to_first_instance(cx, fe.abs),
                        rect,
                        any_down,
                        hover_state: HoverState::In,
                        ..fe.clone()
                    });
                }
            }
            Event::FingerMove(fe) => {
                // check wether our digit is captured, otherwise don't send
                if cx.fingers[fe.digit].captured == area {
                    let abs_start = cx.fingers[fe.digit].down_abs_start;
                    let rel_start = cx.fingers[fe.digit].down_rel_start;
                    let rect = area.get_rect_for_first_instance(cx);
                    return Event::FingerMove(FingerMoveEvent {
                        abs_start,
                        rel: area.get_relative_to_first_instance(cx, fe.abs),
                        rel_start,
                        rect,
                        is_over: rect_contains_with_margin(&rect, fe.abs, &opt.margin),
                        ..fe.clone()
                    });
                }
            }
            Event::FingerDown(fe) => {
                if !fe.handled {
                    let rect = area.get_rect_for_first_instance(cx);
                    if rect_contains_with_margin(&rect, fe.abs, &opt.margin) {
                        // scan if any of the fingers already captured this area
                        if !opt.use_multi_touch {
                            for finger in &cx.fingers {
                                if finger.captured == area {
                                    return Event::None;
                                }
                            }
                        }
                        cx.fingers[fe.digit].captured = area;
                        let rel = area.get_relative_to_first_instance(cx, fe.abs);
                        cx.fingers[fe.digit].down_abs_start = fe.abs;
                        cx.fingers[fe.digit].down_rel_start = rel;
                        fe.handled = true;
                        return Event::FingerDown(FingerDownEvent { rel, rect, ..fe.clone() });
                    }
                }
            }
            Event::FingerUp(fe) => {
                if cx.fingers[fe.digit].captured == area {
                    cx.fingers[fe.digit].captured = Area::Empty;
                    let abs_start = cx.fingers[fe.digit].down_abs_start;
                    let rel_start = cx.fingers[fe.digit].down_rel_start;
                    let rect = area.get_rect_for_first_instance(cx);
                    return Event::FingerUp(FingerUpEvent {
                        is_over: rect.contains(fe.abs),
                        abs_start,
                        rel_start,
                        rel: area.get_relative_to_first_instance(cx, fe.abs),
                        rect,
                        ..fe.clone()
                    });
                }
            }
            _ => (),
        };
        Event::None
    }
}

fn rect_contains_with_margin(rect: &Rect, pos: Vec2, margin: &Option<Margin>) -> bool {
    if let Some(margin) = margin {
        pos.x >= rect.pos.x - margin.l
            && pos.x <= rect.pos.x + rect.size.x + margin.r
            && pos.y >= rect.pos.y - margin.t
            && pos.y <= rect.pos.y + rect.size.y + margin.b
    } else {
        rect.contains(pos)
    }
}

/// For firing and capturing custom events. Can even be fired from different
/// threads using [`Cx::post_signal`].
#[derive(Hash, Eq, PartialEq, Clone, Copy, Debug, Default)]
pub struct Signal {
    pub signal_id: usize,
}

/// Status field to send with a [`Signal`].
///
/// TODO(JP): Either remove this altogether, or replace it with something like
/// [`StringHash`] or [`LocationHash`] or even just [`u64`].
#[derive(PartialEq, Ord, PartialOrd, Copy, Clone, Hash, Eq, Debug)]
pub struct StatusId(pub TypeId);

impl Into<StatusId> for TypeId {
    fn into(self) -> StatusId {
        StatusId(self)
    }
}

/// Created using [`Cx::file_read`].
#[derive(Clone, Debug, Default)]
pub struct FileRead {
    pub path: String,
    pub read_id: u64,
}

impl FileRead {
    pub fn is_pending(&self) -> bool {
        self.read_id != 0
    }

    /// Synchronously read a [`FileReadEvent`] into a [`&str`], if [`FileReadEvent::read_id`]
    /// matches with [`FileRead::read_id`].
    pub fn resolve_utf8<'a>(&mut self, fr: &'a mut FileReadEvent) -> Option<Result<&'a str, String>> {
        if fr.read_id == self.read_id {
            self.read_id = 0;

            #[cfg(not(target_arch = "wasm32"))]
            if let Ok(UniversalFileHandle::Path(path)) = &fr.universal_file_handle {
                if let Ok(new_data) = std::fs::read(path) {
                    fr.universal_file_handle = Ok(UniversalFileHandle::Data(new_data))
                } else {
                    return Some(Err(format!("Failed to open file {}", path)));
                }
            }

            if let Ok(UniversalFileHandle::Data(str_data)) = &fr.universal_file_handle {
                if let Ok(utf8_string) = std::str::from_utf8(str_data) {
                    return Some(Ok(utf8_string));
                } else {
                    return Some(Err(format!("can't parse file as utf8 {}", self.path)));
                }
            } else if let Err(err) = &fr.universal_file_handle {
                return Some(Err(format!("can't load file as utf8 {} {}", self.path, err)));
            }
        }
        None
    }
}

/// Created using [`Cx::start_timer`].
#[derive(Clone, Debug, Default)]
pub struct Timer {
    pub timer_id: u64,
}

impl Timer {
    pub fn empty() -> Timer {
        Timer { timer_id: 0 }
    }

    pub fn is_timer(&mut self, te: &TimerEvent) -> bool {
        te.timer_id == self.timer_id
    }
}

/// Lowest common denominator keymap between desktop and web.
#[derive(Clone, Copy, PartialEq, Debug)]
pub enum KeyCode {
    Escape,

    Backtick,
    Key0,
    Key1,
    Key2,
    Key3,
    Key4,
    Key5,
    Key6,
    Key7,
    Key8,
    Key9,
    Minus,
    Equals,

    Backspace,
    Tab,

    KeyQ,
    KeyW,
    KeyE,
    KeyR,
    KeyT,
    KeyY,
    KeyU,
    KeyI,
    KeyO,
    KeyP,
    LBracket,
    RBracket,
    Return,

    KeyA,
    KeyS,
    KeyD,
    KeyF,
    KeyG,
    KeyH,
    KeyJ,
    KeyK,
    KeyL,
    Semicolon,
    Quote,
    Backslash,

    KeyZ,
    KeyX,
    KeyC,
    KeyV,
    KeyB,
    KeyN,
    KeyM,
    Comma,
    Period,
    Slash,

    Control,
    Alt,
    Shift,
    Logo,

    //RightControl,
    //RightShift,
    //RightAlt,
    //RightLogo,
    Space,
    Capslock,
    F1,
    F2,
    F3,
    F4,
    F5,
    F6,
    F7,
    F8,
    F9,
    F10,
    F11,
    F12,

    PrintScreen,
    Scrolllock,
    Pause,

    Insert,
    Delete,
    Home,
    End,
    PageUp,
    PageDown,

    Numpad0,
    Numpad1,
    Numpad2,
    Numpad3,
    Numpad4,
    Numpad5,
    Numpad6,
    Numpad7,
    Numpad8,
    Numpad9,

    NumpadEquals,
    NumpadSubtract,
    NumpadAdd,
    NumpadDecimal,
    NumpadMultiply,
    NumpadDivide,
    Numlock,
    NumpadEnter,

    ArrowUp,
    ArrowDown,
    ArrowLeft,
    ArrowRight,

    Unknown,
}

impl Default for KeyCode {
    fn default() -> Self {
        KeyCode::Unknown
    }
}
