//! Drawing rectangles; by far the most commonly used `Draw*` struct.

use crate::cx::*;

/// [`DrawQuad`] is the basis for most draw structs. It contains a bunch of info
/// in [`DrawQuadInfo`] for how to render. Then the actual render data starts at
/// [`DrawQuad::rect_pos`]. We take extra care to properly align this struct, so that you
/// can add your own fields afterwards and have them be contiguous in memory
/// after this [`DrawQuad`].
///
/// This renders a rectangle. There are some default shaders available at
/// [`DRAWQUAD_SHADER_PRELUDE`].
///
/// Example usage with your own struct:
///
/// ```
/// struct MyStruct {
///   pub base: DrawQuad,
///   pub field1: f32,
///   pub field2: f32,
/// }
/// ```
///
/// And then initialize using:
///
/// ```
/// MyStruct {
///   base: DrawQuad::with_slots(cx, my_shader, f32::slots() + f32::slots()),
///   field1: 0.0,
///   field2: 0.0,
/// }
/// ```
///
/// TODO(JP): We duplicate state between your draw struct (typically a [`DrawQuad`]
/// at the base, and then your own fields) and [`DrawCall::instances`]. Could we
/// perhaps get rid of the data being stored in here at all, and instead make
/// it easier to manipulate the data inside [`DrawCall::instances`]?
///
/// TODO(JP): We also sometimes use a single [`DrawQuad`] to instantiate many
/// different instances, sometimes by mutating the fields on [`DrawQuad`] and then
/// copying it to the [`DrawCall::instances`]. That is particularly confusing!
#[derive(Debug)]
#[repr(C, align(8))]
pub struct DrawQuad {
    /// A bunch of info in a separate struct, so we know that this field aligns
    /// to 8 bytes / 2 slots.
    pub info: DrawQuadInfo,

    /// One "slot" of padding, because we have 5 slots below and have to make it
    /// to an alignment of 2 slots (8 bytes) if we don't want any auto-generated
    /// padding at the end of this struct (since other structs will use this as
    /// a base and adding fields afterwards).
    _padding: [u8; 4],

    /// The top-left corner position of the quad, in absolute coordinates.
    /// The first "slot"; data will get copied from here.
    pub rect_pos: Vec2,
    /// The size of the quad.
    pub rect_size: Vec2,
    /// Z-index.
    pub draw_depth: f32,
}

/// Some info on how to render the quad.
#[derive(Debug)]
#[repr(C, align(8))]
pub struct DrawQuadInfo {
    /// A pointer to the actual shader that should be used for [`DrawCall`]s.
    shader: Shader,
    /// The number of slots that we should use for rendering, on top of the slots
    /// inherent in [`DrawQuad`] itself.
    slots: usize,
    /// The `Area::InstanceRange` that describes the [`DrawCall::instances`] corresponding
    /// to this [`DrawQuad`].
    area: Area,
    /// The [`ManyInstances`] object that you get from [`DrawQuad::begin_many`].
    many: Option<ManyInstances>,
    /// The old [`ManyInstances`] object so we can call [`Cx::update_area_refs`] in
    /// [`DrawQuad::end_many`].
    many_old_area: Area,
}

impl Clone for DrawQuad {
    fn clone(&self) -> Self {
        Self {
            info: DrawQuadInfo {
                shader: self.info.shader,
                area: Area::Empty,
                many: None,
                many_old_area: Area::Empty,
                slots: self.info.slots,
            },
            _padding: Default::default(),
            rect_pos: self.rect_pos,
            rect_size: self.rect_size,
            draw_depth: self.draw_depth,
        }
    }
}

/// Common [`Shader`] code for using [`DrawQuad`].
pub const DRAWQUAD_SHADER_PRELUDE: CodeFragment = code_fragment!(
    r#"
    instance rect_pos: vec2;
    instance rect_size: vec2;
    instance draw_depth: float;
    geometry geom: vec2;
    varying pos: vec2;

    fn scroll() -> vec2 {
        return draw_scroll;
    }

    fn vertex() -> vec4 {
        let scr = scroll();

        let clipped: vec2 = clamp(
            geom * rect_size + rect_pos - scr,
            draw_clip.xy,
            draw_clip.zw
        );
        pos = (clipped + scr - rect_pos) / rect_size;
        // only pass the clipped position forward
        return camera_projection * (camera_view * vec4(
            clipped.x,
            clipped.y,
            draw_depth + draw_zbias,
            1.
        ));
    }
"#
);

impl DrawQuad {
    /// Initialize with a shader, and with the number of slots that we should
    /// use for rendering, on top of the slots inherent in [`DrawQuad`] itself.
    pub fn with_slots(_cx: &mut Cx, shader: Shader, slots: usize) -> Self {
        Self {
            info: DrawQuadInfo {
                shader,
                slots: slots + Vec2::slots() + Vec2::slots() + f32::slots(),
                area: Area::Empty,
                many: None,
                many_old_area: Area::Empty,
            },
            _padding: Default::default(),
            rect_pos: Vec2::default(),
            rect_size: Vec2::default(),
            draw_depth: 0.0,
        }
    }

    pub fn with_draw_depth(mut self, draw_depth: f32) -> Self {
        self.draw_depth = draw_depth;
        self
    }

    /// Adds a the current [`DrawQuad`] data to the [`DrawCall::instances`],
    /// sets [`DrawQuadInfo::area`], adds it to [`Cx::turtle_align_list`],
    /// and calls [`Cx::begin_turtle].
    ///
    /// You can then modify the data in the [`DrawCall::instances`] directly,
    /// and at the end call [`DrawQuad::end_quad`] which will write
    /// [`DrawQuad::rect_pos`] and [`DrawQuad::rect_size`] to the
    /// [`DrawCall::instances`].
    ///
    /// TODO(JP): This is a bit convoluted and does way too much. There's only
    /// a few places in which we use this, so we should probably try to get rid
    /// of it.
    pub fn begin_quad(&mut self, cx: &mut Cx, layout: Layout) {
        if self.info.many.is_some() {
            panic!("Cannot use begin_quad inside a many block");
        }
        let new_area = cx.add_instance(self.info.shader, self.as_slice());
        cx.add_to_turtle_align_list(new_area);
        self.info.area = cx.update_area_refs(self.info.area, new_area);
        cx.begin_turtle(layout, self.info.area);
    }

    /// See [`DrawQuad::begin_quad`].
    pub fn end_quad(&mut self, cx: &mut Cx) {
        let rect = cx.end_turtle(self.info.area);
        rect.pos.write_shader_value(cx, self.info.area, "rect_pos");
        rect.size.write_shader_value(cx, self.info.area, "rect_size");
    }

    /// Calls [`Cx::walk_turtle`], sets the resulting [`Rect`] as [`DrawQuad::rect_pos`]
    /// and [`DrawQuad::rect_size`], and calls [`DrawQuad::draw_quad`].
    ///
    /// TODO(JP): Let's get rid of it and just have the user explicitly do those steps.
    pub fn draw_quad_walk(&mut self, cx: &mut Cx, walk: Walk) {
        let rect = cx.walk_turtle(walk);
        self.rect_pos = rect.pos;
        self.rect_size = rect.size;
        self.draw_quad(cx);
    }

    /// Convenience function for setting [`DrawQuad::rect_pos`] and [`DrawQuad::rect_size`],
    /// and calling [`DrawQuad::draw_quad`].
    ///
    /// TODO(JP): Let's get rid of it and just have the user explicitly do those steps.
    pub fn draw_quad_abs(&mut self, cx: &mut Cx, rect: Rect) {
        self.rect_pos = rect.pos;
        self.rect_size = rect.size;
        self.draw_quad(cx);
    }

    /// Convenience function for setting [`DrawQuad::rect_pos`] and [`DrawQuad::rect_size`],
    /// (relative to the current [`Turtle::origin`]) and calling [`DrawQuad::draw_quad`].
    ///
    /// TODO(JP): Let's get rid of it and just have the user explicitly do those steps.
    pub fn draw_quad_rel(&mut self, cx: &mut Cx, rect: Rect) {
        let rect = rect.translate(cx.get_turtle_origin());
        self.rect_pos = rect.pos;
        self.rect_size = rect.size;
        self.draw_quad(cx);
    }

    /// Calls [`Cx::add_instance`], [`Cx::add_to_turtle_align_list`], and updates [`DrawQuadInfo::area`].
    ///
    /// If called between [`DrawQuad::begin_many`] and [`DrawQuad::end_many`] it will append to
    /// [`ManyInstances::instances`] instead of calling [`Cx::add_instance`] and in that case
    /// [`DrawQuad::end_many`] will call [`Cx::add_to_turtle_align_list`]).
    pub fn draw_quad(&mut self, cx: &mut Cx) {
        unsafe {
            if let Some(mi) = &mut self.info.many {
                let new_area = if let Area::InstanceRange(ia) = &mut self.info.area {
                    // we need to update the area pointer
                    if mi.instance_area.redraw_id != ia.redraw_id {
                        Some(Area::InstanceRange(InstanceRangeArea {
                            instance_count: 1,
                            instance_offset: mi.instances.len(),
                            ..mi.instance_area
                        }))
                    } else {
                        // just patch up the area without notifying Cx
                        ia.instance_count = 1;
                        ia.instance_offset = mi.instances.len();
                        None
                    }
                } else {
                    None
                };
                mi.instances
                    .extend_from_slice(std::slice::from_raw_parts(&self.rect_pos as *const _ as *const f32, self.info.slots));

                if let Some(new_area) = new_area {
                    self.info.area = cx.update_area_refs(self.info.area, new_area);
                }
                return;
            }
        }
        let new_area = cx.add_instance(self.info.shader, self.as_slice());
        cx.add_to_turtle_align_list(new_area);
        self.info.area = cx.update_area_refs(self.info.area, new_area);
    }

    /// Get the [`DrawQuadInfo::area`].
    pub fn area(&self) -> Area {
        self.info.area
    }

    /// Manually set the [`DrawQuadInfo::area`].
    pub fn set_area(&mut self, area: Area) {
        self.info.area = area;
    }

    /// Get the [`DrawQuadInfo::shader`].
    pub fn shader(&self) -> Shader {
        self.info.shader
    }

    /// Sets [`DrawQuadInfo::many`], so subsequent calls to [`DrawQuad::draw_quad`] get appended to
    /// [`ManyInstances::instances`]. Call [`DrawQuad::end_many`] to finish.
    pub fn begin_many(&mut self, cx: &mut Cx) {
        let mi = cx.begin_many_instances(self.info.shader, self.info.slots);
        self.info.many_old_area = self.info.area;
        self.info.area =
            Area::InstanceRange(InstanceRangeArea { instance_count: 0, instance_offset: mi.instances.len(), ..mi.instance_area });
        self.info.many = Some(mi);
    }

    /// See [`DrawQuad::begin_many`].
    pub fn end_many(&mut self, cx: &mut Cx) {
        if let Some(mi) = self.info.many.take() {
            // update area pointer
            let new_area = cx.end_many_instances(mi);
            cx.add_to_turtle_align_list(new_area);
            self.info.area = cx.update_area_refs(self.info.many_old_area, new_area);
        }
    }

    /// Get a raw slice containing the data in [`DrawQuad`] and fields that come after it (e.g. in
    /// your own struct) as defined in [`DrawQuadInfo::slots`].
    fn as_slice<'a>(&'a self) -> &'a [f32] {
        unsafe { std::slice::from_raw_parts(&self.rect_pos as *const _ as *const f32, self.info.slots) }
    }
}
