//! Drawing [`Texture`]s.

use crate::cx::*;
use crate::drawquad::*;
use crate::std_shader::*;

define_string_with_filename!(MAIN_SHADER);

/// For drawing a [`Texture2D`].
#[derive(Clone)]
#[repr(C)]
pub struct DrawImage {
    pub texture: Texture2D,
    pub base: DrawQuad,
    pub pt1: Vec2,
    pub pt2: Vec2,
    pub alpha: f32,
}

impl DrawImage {
    pub fn new(cx: &mut Cx) -> Self {
        Self {
            base: DrawQuad::with_slots(
                cx,
                cx.get_shader(StringHash::new(MAIN_SHADER), location_hash!()),
                Vec2::slots() + Vec2::slots() + f32::slots(),
            ),
            texture: Default::default(),
            pt1: vec2(0., 0.),
            pt2: vec2(1., 1.),
            alpha: 1.0,
        }
    }

    pub fn app_load(cx: &mut Cx) {
        cx.register_shader(
            MAIN_SHADER,
            Some(GEOM_QUAD2D),
            &[STD_SHADER_PRELUDE, DRAWQUAD_SHADER_PRELUDE],
            &code_fragment!(
                r#"
                texture texture: texture2D;
                instance pt1: vec2;
                instance pt2: vec2;
                instance alpha: float;
                varying tc: vec2;
                varying v_pixel: vec2;
                //let dpi_dilate: float<Uniform>;

                fn vertex() -> vec4 {
                    // return vec4(geom.x-0.5, geom.y, 0., 1.);
                    let shift: vec2 = -draw_scroll;
                    let clipped: vec2 = clamp(
                        geom * rect_size + rect_pos + shift,
                        draw_clip.xy,
                        draw_clip.zw
                    );
                    let pos = (clipped - shift - rect_pos) / rect_size;
                    tc = mix(pt1, pt2, pos);
                    v_pixel = clipped;
                    // only pass the clipped position forward
                    return camera_projection * vec4(clipped.x, clipped.y, draw_depth, 1.);
                }

                fn pixel() -> vec4 {
                    return vec4(sample2d(texture, tc.xy).rgb * alpha, alpha);
                }"#
            ),
        );
    }

    pub fn write_uniforms(&self, cx: &mut Cx) {
        if self.base.area().is_first_instance() {
            self.texture.write_shader_value(cx, self.base.area(), "texture");
        }
    }
    pub fn draw_quad_rel(&mut self, cx: &mut Cx, rect: Rect) {
        self.base.draw_quad_rel(cx, rect);
        self.write_uniforms(cx)
    }
}
