//! Drawing 3d cubes.

use crate::cx::*;

/// Draw a cube; similar to [`crate::DrawQuad`]. Is currently not used much, so mostly
/// for demonstration purposes.
#[repr(C, align(8))]
pub struct DrawCube {
    /// A bunch of info in a separate struct, so we know that this field aligns
    /// to 8 bytes / 2 slots.
    pub info: DrawCubeInfo,

    // Note: no padding, since we use 22 slots below, which is aligned to
    // 2 slots / 8 bytes already.
    /// Raw transform matrix for the cube.
    /// The first "slot"; data will get copied from here.
    pub transform: Mat4,
    /// Dimensions of the cube.
    pub cube_size: Vec3,
    /// Position in 3d space.
    pub cube_pos: Vec3,
}

/// Some info on how to render the cube.
#[derive(Debug)]
#[repr(C, align(8))]
pub struct DrawCubeInfo {
    /// A pointer to the actual shader that should be used for [`DrawCall`]s.
    shader: Shader,
    /// The number of slots that we should use for rendering, on top of the slots
    /// inherent in [`DrawCube`] itself.
    slots: usize,
    /// TODO(JP): document.
    area: Area,
}

impl Clone for DrawCube {
    fn clone(&self) -> Self {
        Self {
            info: DrawCubeInfo { shader: self.info.shader, area: Area::Empty, slots: self.info.slots },
            transform: self.transform,
            cube_size: self.cube_size,
            cube_pos: self.cube_pos,
        }
    }
}

/// Common [`Shader`] code for using [`DrawCube`].
pub const DRAWCUBE_SHADER_PRELUDE: CodeFragment = code_fragment!(
    r#"
    instance transform: mat4;
    instance cube_size: vec3;
    instance cube_pos: vec3;
"#
);

/*
 Example shader (using cube 3d geometry):
    varying lit_col: vec4;

    fn vertex() -> vec4 {
        let normal_matrix = mat3(transform);
        let normal = normalize(normal_matrix * geom_normal);
        let dp = abs(normal.z);

        lit_col = vec4(color.rgb * dp, color.a);
        return camera_projection * (camera_view * transform * vec4(
            geom_pos.x * cube_size.x + cube_pos.x,
            geom_pos.y * cube_size.y + cube_pos.y,
            geom_pos.z * cube_size.z + cube_pos.z + draw_zbias,
            1.
        ));
    }

    fn pixel() -> vec4 {
        return lit_col;
    }
*/

impl DrawCube {
    pub fn new(cx: &mut Cx, shader: Shader) -> Self {
        Self::with_slots(cx, shader, 0)
    }

    pub fn with_slots(_cx: &mut Cx, shader: Shader, slots: usize) -> Self {
        Self {
            info: DrawCubeInfo { shader, slots: slots + Mat4::slots() + Vec3::slots() + Vec3::slots(), area: Area::Empty },
            transform: Mat4::identity(),
            cube_size: vec3(0.1, 0.1, 0.1),
            cube_pos: vec3(0.0, 0.0, 0.0),
        }
    }

    pub fn area(&self) -> Area {
        self.info.area
    }

    pub fn draw_cube(&mut self, cx: &mut Cx) {
        self.info.area = cx.add_instance(self.info.shader, self.as_slice());
    }

    fn as_slice<'a>(&'a self) -> &'a [f32] {
        unsafe { std::slice::from_raw_parts(&self.transform as *const _ as *const f32, self.info.slots) }
    }
}
