//! Convenient way of drawing colored rectangles, built on top of [`DrawQuad`].

use crate::cx::*;
use crate::drawquad::*;
use crate::std_shader::*;

define_string_with_filename!(MAIN_SHADER);

/// Convenient type of [`DrawQuad`] which has a single `color` field, which is
/// drawn as the background by default. You can pass in your own [`Shader`] to
/// override.
///
/// Note that it is not recommended to use this struct as the base for your own
/// "draw struct", since it adds unnecessary indirection. Instead just add a
/// `color: Vec4` field on your struct yourself.
#[derive(Clone)]
#[repr(C)]
pub struct DrawColor {
    pub quad: DrawQuad,
    pub color: Vec4,
}

/// You can provide your own [`Shader`] if you want to customize how the [`DrawColor::color`]
/// is being rendered. If you want to use the default [`Shader`], you have to pass
/// in a [`LocationHash`] so that you get a more predictable render order. See
/// documentation for [`LocationHash`] for more information.
pub enum DrawColorShader {
    Default(LocationHash),
    Shader(Shader),
}

impl DrawColor {
    pub fn new(cx: &mut Cx, draw_color_shader: DrawColorShader) -> Self {
        let shader = match draw_color_shader {
            DrawColorShader::Shader(s) => s,
            DrawColorShader::Default(location_hash) => cx.get_shader(StringHash::new(MAIN_SHADER), location_hash),
        };

        Self { quad: DrawQuad::with_slots(cx, shader, Vec4::slots()), color: Default::default() }
    }
    pub fn app_load(cx: &mut Cx) {
        cx.register_shader(
            MAIN_SHADER,
            Some(GEOM_QUAD2D),
            &[STD_SHADER_PRELUDE, DRAWQUAD_SHADER_PRELUDE],
            &code_fragment!(
                r#"
                instance color: vec4;
                fn pixel() -> vec4 {
                    return vec4(color.rgb*color.a, color.a);
                }"#
            ),
        );
    }
    pub fn set_color(&mut self, cx: &mut Cx, v: Vec4) {
        self.color = v;
        self.color.write_shader_value(cx, self.quad.area(), "color");
    }
    pub fn with_draw_depth(self, depth: f32) -> Self {
        Self { quad: self.quad.with_draw_depth(depth), ..self }
    }
}
