//! The main primitives for rendering to the screen.
//!
//! A draw tree has two kinds of nodes: [`View`] and [`DrawCall`]. It might look like this:
//! * `View` - root
//!   * `DrawCall` - rendering some text
//!   * `DrawCall` - rendering some buttons
//!   * `View` - scrollable container
//!     * `DrawCall` - rendering some more buttons
//!     * `DrawCall` - rendering some more text
//!   * `DrawCall` - maybe even more buttons
//!
//! [`DrawCall`]s contain the actual data that needs to be drawn to the screen.
//!
//! [`View`]s are groups of draw calls, have some special features, such as
//! scrolling/clipping and caching.
//!
//! Note that one level higher, we have a hierarchy of [`Pass`]es.

use crate::cx::*;

/// Gets returned by [`View::begin_view`].
///
/// TODO(JP): maybe better to make this an enum with descriptive names?
/// It looks like this is done as a [`Result`] so you can use the [`?`] shorthand,
/// but not sure if that is really clear enough?
pub type ViewRedraw = Result<(), ()>;

/// A group of [`DrawCall`]s.
///
/// This is really a pointer to a [`CxView`] (indexed in [`Cx::views`] using [`View::view_id`]),
/// so you can find more information there.
///
/// A [`View`] has a few special features:
/// * It has its own
///
/// See also [`ViewArea`], which is an [`Area`] pointer to a [`View`].
#[derive(Clone)]
pub struct View {
    /// The index of the corresponding [`CxView`] in [`Cx::views`].
    pub view_id: Option<usize>,
    /// The most recent [`Cx::redraw_id`] that this [`View`] was drawn with.
    pub(crate) redraw_id: u64,
    /// Whether this [`View`] is an overlay/popup, which means all [`DrawCall`]s underneath it
    /// will get rendered last.
    pub(crate) is_overlay: bool,
    /// Whether this [`View`] will always redraw. Currently only used internally in fonts.
    ///
    /// TODO(JP): We can probably do this some other way? Like just explicitly mark it for redraw
    /// before the [`View::begin_view`] or so?
    pub(crate) always_redraw: bool,
}

impl View {
    /// Creates a new, empty [`View`].
    pub fn new() -> Self {
        Self { redraw_id: 0, is_overlay: false, always_redraw: false, view_id: None }
    }
    /// See [`View::is_overlay`].
    pub fn with_is_overlay(self, is_overlay: bool) -> Self {
        Self { is_overlay, ..self }
    }
    /// See [`View::always_redraw`].
    pub(crate) fn with_always_redraw(self, always_redraw: bool) -> Self {
        Self { always_redraw, ..self }
    }

    /// Register the [`View`] in the draw tree. Returns [`Result::Err`] if rendering
    /// should be skipped since it wasn't marked for redrawing. Returns [`Result::Ok`]
    /// when drawing should happen, ie. when it was marked for redrawing, or when
    /// drawing the [`View`] for the first time (when it doesn't have a
    /// [`View::view_id`] yet).
    ///
    /// This also creates a new [`Turtle`] with the [`Layout`] that you pass in.
    /// Note that you should not create a [`View`] just in order to get a new
    /// [`Turtle`], since creating a [`View`] is relatively expensive -- no
    /// [`DrawCall`]s inside this [`View`] will get merged with ones outside of
    /// it, so adding too many [`View`]s will create too many individual calls to
    /// the GPU.
    ///
    /// TODO(JP): Perhaps we should decouple [`Turtle`] and [`View`] altogether?
    ///
    /// TODO(JP): If you move where a [`View`] gets rendered in the hierarchy,
    /// then you have to be really sure that you mark the [`View`] that it gets
    /// moved from as needing a redraw (e.g. using [`View::redraw_view`]), otherwise
    /// the [`CxView`] might end up with two parents!
    pub fn begin_view(&mut self, cx: &mut Cx, layout: Layout) -> ViewRedraw {
        if !cx.in_redraw_cycle {
            panic!("calling begin_view outside of redraw cycle is not possible!");
        }

        // check if we have a pass id parent
        let pass_id = *cx.pass_stack.last().expect("No pass found when begin_view");

        let nesting_view_id = if !cx.view_stack.is_empty() {
            *cx.view_stack.last().unwrap()
        } else {
            // return the root draw list
            0
        };

        if self.view_id.is_none() {
            // we need a draw_list_id
            self.view_id = Some(cx.views.len());
            cx.views.push(CxView::default());
            let cxview = &mut cx.views[self.view_id.unwrap()];
            cxview.redraw_id = cx.redraw_id;
            cxview.pass_id = pass_id;
            // set nesting draw list id for incremental repaint scanning
            cx.views[self.view_id.unwrap()].nesting_view_id = nesting_view_id;
        }

        let view_id = self.view_id.unwrap();

        let (override_layout, is_root_for_pass) = if cx.passes[pass_id].main_view_id.is_none() {
            // we are the first view on a window
            let cxpass = &mut cx.passes[pass_id];
            cxpass.main_view_id = Some(view_id);
            // we should take the window geometry and abs position as our turtle layout
            (Layout { abs_origin: Some(Vec2 { x: 0., y: 0. }), abs_size: Some(cxpass.pass_size), ..layout }, true)
        } else {
            (layout, false)
        };

        let cxpass = &mut cx.passes[pass_id];
        // find the parent draw list id
        let parent_view_id = if self.is_overlay {
            if cxpass.main_view_id.is_none() {
                panic!("Cannot make overlay inside window without root view")
            };

            cxpass.main_view_id.unwrap()
        } else if is_root_for_pass {
            view_id
        } else if let Some(last_view_id) = cx.view_stack.last() {
            *last_view_id
        } else {
            // we have no parent
            view_id
        };

        // push ourselves up the parent draw_stack
        if view_id != parent_view_id {
            // we need a new draw
            let parent_cxview = &mut cx.views[parent_view_id];

            let id = parent_cxview.draw_calls_len;
            parent_cxview.draw_calls_len += 1;

            // see if we need to add a new one
            if parent_cxview.draw_calls_len > parent_cxview.draw_calls.len() {
                parent_cxview.draw_calls.push({
                    DrawCall {
                        view_id: parent_view_id,
                        draw_call_id: parent_cxview.draw_calls.len(),
                        redraw_id: cx.redraw_id,
                        sub_view_id: view_id,
                        ..Default::default()
                    }
                })
            } else {
                // or reuse a sub list node
                let draw = &mut parent_cxview.draw_calls[id];
                draw.sub_view_id = view_id;
                draw.redraw_id = cx.redraw_id;
            }
        }

        if !self.always_redraw && cx.views[view_id].draw_calls_len != 0 && !cx.view_will_redraw(view_id) {
            // walk the turtle because we aren't drawing
            let w = Width::Fix(cx.views[view_id].rect.size.x);
            let h = Height::Fix(cx.views[view_id].rect.size.y);
            cx.walk_turtle(Walk { width: w, height: h, margin: override_layout.walk.margin });
            return Err(());
        }

        // prepare drawlist for drawing
        let cxview = &mut cx.views[view_id];

        // TODO(JP): We don't seem to currently support moving a `View` to a different pass. Do we
        // want to?
        assert_eq!(cxview.pass_id, pass_id);

        // update drawlist ids
        let last_redraw_id = cxview.redraw_id;
        self.redraw_id = cx.redraw_id;
        cxview.redraw_id = cx.redraw_id;
        cxview.draw_calls_len = 0;

        cx.view_stack.push(view_id);

        let old_area = Area::View(ViewArea { view_id, redraw_id: last_redraw_id });
        let new_area = Area::View(ViewArea { view_id, redraw_id: cx.redraw_id });
        cx.update_area_refs(old_area, new_area);

        cx.begin_turtle(override_layout, new_area);

        if is_root_for_pass {
            cx.passes[pass_id].paint_dirty = true;
        }

        Ok(())
    }

    /// End the [`View`], by ending the [`Turtle`]. Returns a [`ViewArea`] that
    /// you can hold onto.
    ///
    /// Should only be called if [`View::begin_view`] returned [`Result::Ok`].
    ///
    /// TODO(JP): Is the [`ViewArea`] redundant, since it basically contains the
    /// same information as the [`View`] itself?
    pub fn end_view(&mut self, cx: &mut Cx) -> Area {
        let view_id = self.view_id.unwrap();
        let view_area = Area::View(ViewArea { view_id, redraw_id: cx.redraw_id });
        let rect = cx.end_turtle(view_area);
        let cxview = &mut cx.views[view_id];
        cxview.rect = rect;
        cx.view_stack.pop();
        view_area
    }

    /// Whether or not the [`View`] will redraw.
    ///
    /// See [`Cx::view_will_redraw`].
    pub fn view_will_redraw(&self, cx: &mut Cx) -> bool {
        if let Some(view_id) = self.view_id {
            cx.view_will_redraw(view_id)
        } else {
            true
        }
    }

    /// Get the [`Rect`] that the [`Turtle`] associated with the [`View`]
    /// returned.
    ///
    /// TODO(JP): Should we return an [`Option<Rect>`] instead of just
    /// returning a zero-sized [`Rect`] when the [`View`] has never been
    /// drawn yet?
    ///
    /// TODO(JP): Doesn't check if the [`View::redraw_id`] is still up to
    /// date, so we might be returning an outdated [`Rect`] here.
    pub fn get_rect(&self, cx: &Cx) -> Rect {
        if let Some(view_id) = self.view_id {
            let cxview = &cx.views[view_id];
            return cxview.rect;
        }
        Rect::default()
    }

    /// Print debug information about this [`View`].
    pub fn set_view_debug(&self, cx: &mut Cx, view_debug: CxViewDebug) {
        if let Some(view_id) = self.view_id {
            let cxview = &mut cx.views[view_id];
            cxview.debug = Some(view_debug);
        }
    }

    /// Redraws the [`View`] using [`Cx::redraw_child_area`].
    ///
    /// If the [`View`] hasn't been instantiated yet, we're conservative and
    /// just redraw everything using [`Area::All`].
    ///
    /// TODO(JP): Should we drop this function and just have users call
    /// `Cx::redraw_child_area(view.area())`? That won't do the [`Area::All`]
    /// thing mentioned above, but maybe that is bad anyway? Or maybe it's good,
    /// and we should just change `[Cx::redraw_child_area`] to redraw everything
    /// on [`Area::Empty`] (and get rid of [`Area::All`] completely)?
    pub fn redraw_view(&self, cx: &mut Cx) {
        if let Some(view_id) = self.view_id {
            let cxview = &cx.views[view_id];
            let area = Area::View(ViewArea { view_id, redraw_id: cxview.redraw_id });
            cx.redraw_child_area(area);
        } else {
            cx.redraw_child_area(Area::All)
        }
    }

    /// Returns an [`Area::View`] for this [`View`], or [`Area::Empty`] if the
    /// [`View`] hasn't been instantiated yet.
    pub fn area(&self) -> Area {
        if let Some(view_id) = self.view_id {
            Area::View(ViewArea { view_id, redraw_id: self.redraw_id })
        } else {
            Area::Empty
        }
    }

    /// Get the current [`CxView::unsnapped_scroll`] if the [`View`] has been
    /// instantiated.
    pub fn get_scroll_pos(&self, cx: &Cx) -> Vec2 {
        if let Some(view_id) = self.view_id {
            let cxview = &cx.views[view_id];
            cxview.unsnapped_scroll
        } else {
            Vec2::default()
        }
    }
}

impl Cx {
    /// Create and returns a new [`DrawCall`] for the given [`Shader`], even if
    /// one already exists.
    ///
    /// Will be nested in the current [`CxView`].
    ///
    /// You don't have to use the [`DrawCall`] that gets returned. Instead you
    /// may just call this to force a new [`DrawCall`] to be created for the
    /// [`Shader`], and then use other functions to append to it.
    pub fn new_draw_call(&mut self, shader: Shader) -> &mut DrawCall {
        self.get_draw_call(false, shader, None)
    }

    /// Returns an existing [`DrawCall`] based on the given [`Shader`], or
    /// creates a new one if none can be found in the current [`CxView`].
    fn append_to_draw_call(&mut self, shader: Shader, slots: usize) -> &mut DrawCall {
        self.get_draw_call(true, shader, Some(slots))
    }

    /// Common code between [`Cx::new_draw_call`] and [`Cx::append_to_draw_call`].
    /// Either creates a new [`DrawCall`] or appends to an existing one.
    ///
    /// Reuses an existing [`DrawCall`] if [`CxView::draw_calls_len`] is less than
    /// [`CxView::draw_calls.len()`], so we can reuse existing GPU resources.
    ///
    /// TODO(JP): It's unclear to me if the reusing of GPU resources in this way
    /// is beneficial. And if it is, if it should instead be done in the
    /// platform-specific code instead.
    fn get_draw_call(&mut self, append: bool, shader: Shader, slots: Option<usize>) -> &mut DrawCall {
        let sh = &self.shaders[shader.shader_id];

        let current_view_id = *self.view_stack.last().unwrap();
        let cxview = &mut self.views[current_view_id];
        let draw_call_id = cxview.draw_calls_len;

        if append {
            if let Some(index) = cxview.find_appendable_drawcall(shader) {
                return &mut cxview.draw_calls[index];
            }
        }

        // add one
        cxview.draw_calls_len += 1;

        if let Some(slots) = slots {
            if slots != sh.mapping.instance_props.total_slots {
                log!("Warning, instance misalignment between struct and shader in {}", sh.name)
            }
        }
        // see if we need to add a new one
        if draw_call_id >= cxview.draw_calls.len() {
            cxview.draw_calls.push(DrawCall {
                geometry: None,
                draw_call_id,
                view_id: current_view_id,
                redraw_id: self.redraw_id,
                scroll_sticky_horizontal: false,
                scroll_sticky_vertical: false,
                in_many_instances: false,
                sub_view_id: 0,
                shader,
                instances: Vec::new(),
                draw_uniforms: DrawUniforms::default(),
                user_uniforms: {
                    let mut f = Vec::new();
                    f.resize(sh.mapping.user_uniform_props.total_slots, 0.0);
                    f
                },
                textures_2d: {
                    let mut f = Vec::new();
                    f.resize(sh.mapping.textures.len(), 0);
                    f
                },
                //current_instance_offset: 0,
                instance_dirty: true,
                uniforms_dirty: true,
                platform: CxPlatformDrawCall::default(),
            });
            let dc = &mut cxview.draw_calls[draw_call_id];
            return dc;
        }
        // reuse an older one, keeping all GPU resources attached
        let dc = &mut cxview.draw_calls[draw_call_id];
        dc.shader = shader;
        dc.geometry = None;
        dc.sub_view_id = 0; // make sure its recognised as a draw call
                            // truncate buffers and set update frame
        dc.redraw_id = self.redraw_id;
        dc.instances.truncate(0);
        dc.user_uniforms.truncate(0);
        dc.user_uniforms.resize(sh.mapping.user_uniform_props.total_slots, 0.0);
        dc.textures_2d.truncate(0);
        dc.textures_2d.resize(sh.mapping.textures.len(), 0);
        dc.instance_dirty = true;
        dc.uniforms_dirty = true;
        dc.scroll_sticky_horizontal = false;
        dc.scroll_sticky_vertical = false;
        dc
    }

    /// Start manually editing [`DrawCall::instances`], through a [`ManyInstances`]
    /// temporary object.
    ///
    /// TODO(JP): See [`ManyInstances`] for a TODO on whether we even need
    /// [`ManyInstances`].
    pub fn begin_many_instances(&mut self, shader: Shader, slots: usize) -> ManyInstances {
        let dc = self.append_to_draw_call(shader, slots);
        let mut instances = Vec::new();
        if dc.in_many_instances {
            panic!("please call end_many_instances before calling begin_many_instances again")
        }
        dc.in_many_instances = true;
        std::mem::swap(&mut instances, &mut dc.instances);
        ManyInstances {
            instance_area: InstanceRangeArea {
                view_id: dc.view_id,
                draw_call_id: dc.draw_call_id,
                instance_count: 0,
                instance_offset: instances.len(),
                redraw_id: dc.redraw_id,
            },
            instances,
        }
    }

    /// End manual editing of [`DrawCall::instances`]. Returns an [`Area::InstanceRange`]
    /// that points to the range of instance you just added.
    ///
    /// TODO(JP): See [`ManyInstances`] for a TODO on whether we even need
    /// [`ManyInstances`].
    ///
    /// TODO(JP): Removing or editing instances doesn't seem to be supported; those won't
    /// be described in the [`Area`] (in case of removing you might even get a crash
    /// since [`InstanceRangeArea::instance_count` is unsigned]).
    pub fn end_many_instances(&mut self, mut many_instances: ManyInstances) -> Area {
        let mut ia = many_instances.instance_area;
        let cxview = &mut self.views[ia.view_id];
        let dc = &mut cxview.draw_calls[ia.draw_call_id];

        if !dc.in_many_instances {
            panic!("please call begin_many_instances before calling end_many_instances")
        }
        dc.in_many_instances = false;
        std::mem::swap(&mut many_instances.instances, &mut dc.instances);
        let sh = &self.shaders[dc.shader.shader_id];
        ia.instance_count = (dc.instances.len() - ia.instance_offset) / sh.mapping.instance_props.total_slots;
        Area::InstanceRange(ia)
    }

    /// Add a single instance to [`DrawCall::instances`].
    ///
    /// Uses [`Cx::append_to_draw_call`] under the hood to find the [`DrawCall`]
    /// to add to.
    ///
    /// TODO(JP): This is convenient, but might be a bit too much indirection for
    /// my taste. Why not have this function on [`DrawCall`] directly, and have
    /// users call [`Cx::append_to_draw_call`] directly?
    pub fn add_instance(&mut self, shader: Shader, data: &[f32]) -> Area {
        let total_instance_slots = self.shaders[shader.shader_id].mapping.instance_props.total_slots;
        let dc = self.append_to_draw_call(shader, data.len());
        let instance_count = data.len() / total_instance_slots;
        let check = data.len() % total_instance_slots;
        if check > 0 {
            panic!("Data not multiple of total slots");
        }
        let ia = InstanceRangeArea {
            view_id: dc.view_id,
            draw_call_id: dc.draw_call_id,
            instance_count,
            instance_offset: dc.instances.len(),
            redraw_id: dc.redraw_id,
        };
        dc.instances.extend_from_slice(data);
        Area::InstanceRange(ia)
    }

    /// Sets the horizontal scroll position for a [`View`]/[`CxView`].
    pub fn set_view_scroll_x(&mut self, view_id: usize, scroll_pos: f32) {
        let fac = self.get_delegated_dpi_factor(self.views[view_id].pass_id);
        let cxview = &mut self.views[view_id];
        cxview.unsnapped_scroll.x = scroll_pos;
        let snapped = scroll_pos - scroll_pos % (1.0 / fac);
        if cxview.snapped_scroll.x != snapped {
            cxview.snapped_scroll.x = snapped;
            self.passes[cxview.pass_id].paint_dirty = true;
        }
    }

    /// Sets the vertical scroll position for a [`View`]/[`CxView`].
    pub fn set_view_scroll_y(&mut self, view_id: usize, scroll_pos: f32) {
        let fac = self.get_delegated_dpi_factor(self.views[view_id].pass_id);
        let cxview = &mut self.views[view_id];
        cxview.unsnapped_scroll.y = scroll_pos;
        let snapped = scroll_pos - scroll_pos % (1.0 / fac);
        if cxview.snapped_scroll.y != snapped {
            cxview.snapped_scroll.y = snapped;
            self.passes[cxview.pass_id].paint_dirty = true;
        }
    }
}

/// Temporarily get the [`DrawCall::instances`] array, so you can manually add
/// stuff to it.
///
/// TODO(JP): Why do we need an object like this to be exposed to the user? Why
/// can't we just keep the [`ManyInstances::instance_area`] directly on [`Cx`],
/// like we do with most other `begin_` and `end_` functions (e.g. for [`Turtle`],
/// [`View`], etc)?
#[derive(Debug)]
pub struct ManyInstances {
    /// Temporarily borrowed from [`DrawCall::instances`]. You can manually add
    /// your own instance data to it.
    pub instances: Vec<f32>,
    /// The [`InstanceRangeArea`] from when [`Cx::begin_many_instances`] was called.
    /// Will be updated to include the correct count when you call
    /// [`Cx::end_many_instances`], and returned from there as well.
    pub(crate) instance_area: InstanceRangeArea,
}

/// Hardcoded set of uniforms that are present on every [`DrawCall`].
///
/// TODO(JP): Should we just use [`Vec4`]s and [`Vec2`] here instead of individual
/// [`f32`]s?
#[derive(Default, Clone)]
#[repr(C, align(8))]
pub(crate) struct DrawUniforms {
    /// Clip region top left x-position.
    draw_clip_x1: f32,
    /// Clip region top left y-position.
    draw_clip_y1: f32,
    /// Clip region bottom right x-position.
    draw_clip_x2: f32,
    /// Clip region bottom right y-position.
    draw_clip_y2: f32,
    /// The total horizontal scroll offset, including all its parents.
    pub(crate) draw_scroll_x: f32,
    /// The total vertical scroll offset, including all its parents.
    pub(crate) draw_scroll_y: f32,
    /// The horizontal scroll offset of just the containing [`View`].
    pub(crate) draw_local_scroll_x: f32,
    /// The vertical scroll offset of just the containing [`View`].
    pub(crate) draw_local_scroll_y: f32,
    /// A small increment that you can add to the z-axis of your vertices, which is based on the
    /// position of the [`DrawCall`] in the draw tree.
    ///
    /// TODO(JP): Not entirely sure why we need this, given that we're already drawing everything
    /// in order?
    draw_zbias: f32,
}

impl DrawUniforms {
    /// Get as a raw `[f32]` slice.
    pub fn as_slice(&self) -> &[f32; std::mem::size_of::<DrawUniforms>()] {
        unsafe { std::mem::transmute(self) }
    }
}

/// This represents an actual call to the GPU, _or_ it can represent a
/// sub-[`View`], in case [`DrawCall::sub_view_id`] is set. Note that all of this behaves
/// completely differently if [`DrawCall::sub_view_id`] is set; all regular drawing fields
/// are ignored in that case!
///
/// TODO(JP): this sub-[`View`] behavior is confusing, and we should instead
/// split this out to something like [`enum DrawTreeItem { DrawCall(DrawCall),
/// NestedView(usize) }`] or so.
///
/// That said, for a regular [`DrawCall`], this contains all the information that
/// you need to make a draw call on the GPU: the [`Shader`], [`DrawCall::instances`],
/// [`DrawCall::draw_uniforms`], and so on.
///
/// It is always kept in [`CxView::draw_calls`], and as said, is part of a tree
/// structure, called the "draw tree". To print a textual representation of the
/// draw tree, use [`View::set_view_debug`].
#[derive(Default, Clone)]
pub struct DrawCall {
    /// The index of this [`DrawCall`] within its parent [`CxView::draw_calls`].
    pub(crate) draw_call_id: usize,
    /// The parent [`CxView`]/[`View`] that this [`DrawCall`] is a part of.
    pub(crate) view_id: usize,
    /// The [`Cx::redraw_id`] of the last time this [`DrawCall`] was accessed.
    pub(crate) redraw_id: u64,
    /// If not 0, this [`DrawCall`] actually represents a nested sub-[`View`].
    /// See [`DrawCall`] for a TODO on fixing this, because this is confusing!
    pub(crate) sub_view_id: usize,
    /// The actual [`Shader`] to use when drawing.
    pub(crate) shader: Shader,
    /// The instance buffer that will be sent directly to the GPU.
    pub(crate) instances: Vec<f32>,
    /// Buffer of user-defined uniforms (in addition to the [`draw_uniforms`
    /// below.)
    pub(crate) user_uniforms: Vec<f32>,
    /// Buffer of texture IDs.
    pub(crate) textures_2d: Vec<u32>,
    /// Whether or not the draw call has been accessed since the last paint.
    /// Should currently always be the same as [`DrawCall::uniforms_dirty`] below.
    pub(crate) instance_dirty: bool,
    /// Whether or not the draw call has been accessed since the last paint.
    /// Should currently always be the same as [`DrawCall::instance_dirty`] above.
    pub(crate) uniforms_dirty: bool,
    /// Hardcoded set of uniforms that are present on every [`DrawCall`].
    pub(crate) draw_uniforms: DrawUniforms,
    /// The base [`Geometry`] object that will be used for generating the initial
    /// vertex locations for every instance, such as a rectangle or cube.
    ///
    /// TODO(JP): This geometry override seems to currently never get used, but
    /// I can see how it could be potentially useful in the future. However, if
    /// it doesn't turn out to be useful, let's remove it.
    pub(crate) geometry: Option<Geometry>,
    /// See [`DrawCall::set_scroll_sticky`].
    pub(crate) scroll_sticky_vertical: bool,
    /// See [`DrawCall::set_scroll_sticky`].
    pub(crate) scroll_sticky_horizontal: bool,
    /// Gets set to [`true`] in [`Cx::begin_many_instances`], and to [`false`] in
    /// [`Cx::end_many_instances`], and is used to guard against doing incorrect
    /// things while in this "many instances" mode.
    pub(crate) in_many_instances: bool,
    /// Platform-specific data for use during painting.
    pub(crate) platform: CxPlatformDrawCall,
}

impl DrawCall {
    /// By default, [`DrawCall`] gets horizontal and vertical scrolling applied to
    /// its uniforms, but you can disable that by calling this method. This only
    /// applies to scrolling from its direct parent [`View`].
    ///
    /// TODO(JP): The fact that this only applies to the direct parent [`View`]
    /// makes it so you can't just arbitrarily wrap [`DrawCall`]s inside [`View`]s,
    /// which is somewhat unexpected. It might be better to have this apply to
    /// the nearest `zaplib_widget::ScrollView` instead?
    ///
    /// TODO(JP): Do we need to track as fields on [`DrawCall`? The same behavior
    /// can also be accomplished by overriding the [`Shader`]'s [`scroll`
    /// function, by doing [`draw_scroll - draw_local_scroll`]. It's not as
    /// convenient, but then again, it might not be used very often, and it would
    /// encourage people to do more stuff in shaders.
    pub fn set_scroll_sticky(&mut self, horizontal: bool, vertical: bool) {
        self.scroll_sticky_horizontal = horizontal;
        self.scroll_sticky_vertical = vertical;
    }

    /// Set the scroll uniforms in [`DrawCall::draw_uniforms`], as computed when
    /// walking the draw tree during painting.
    pub(crate) fn set_local_scroll(&mut self, scroll: Vec2, local_scroll: Vec2) {
        self.draw_uniforms.draw_scroll_x = scroll.x;
        if !self.scroll_sticky_horizontal {
            self.draw_uniforms.draw_scroll_x += local_scroll.x;
        }
        self.draw_uniforms.draw_scroll_y = scroll.y;
        if !self.scroll_sticky_vertical {
            self.draw_uniforms.draw_scroll_y += local_scroll.y;
        }
        self.draw_uniforms.draw_local_scroll_x = local_scroll.x;
        self.draw_uniforms.draw_local_scroll_y = local_scroll.y;
    }

    /// Set the zbias in [`DrawCall::draw_uniforms`], as computed when
    /// walking the draw tree during painting.
    pub(crate) fn set_zbias(&mut self, zbias: f32) {
        self.draw_uniforms.draw_zbias = zbias;
    }

    /// Set the clip dimensions in [`DrawCall::draw_uniforms`], as computed when
    /// walking the draw tree during painting.
    pub(crate) fn set_clip(&mut self, clip: (Vec2, Vec2)) {
        self.draw_uniforms.draw_clip_x1 = clip.0.x;
        self.draw_uniforms.draw_clip_y1 = clip.0.y;
        self.draw_uniforms.draw_clip_x2 = clip.1.x;
        self.draw_uniforms.draw_clip_y2 = clip.1.y;
    }

    /// Get the actual position on the screen given the scroll and clip uniforms
    /// in [`DrawCall::draw_uniforms`].
    pub(crate) fn clip_and_scroll_rect(&self, x: f32, y: f32, w: f32, h: f32) -> Rect {
        let mut x1 = x - self.draw_uniforms.draw_scroll_x;
        let mut y1 = y - self.draw_uniforms.draw_scroll_y;
        let mut x2 = x1 + w;
        let mut y2 = y1 + h;
        x1 = self.draw_uniforms.draw_clip_x1.max(x1).min(self.draw_uniforms.draw_clip_x2);
        y1 = self.draw_uniforms.draw_clip_y1.max(y1).min(self.draw_uniforms.draw_clip_y2);
        x2 = self.draw_uniforms.draw_clip_x1.max(x2).min(self.draw_uniforms.draw_clip_x2);
        y2 = self.draw_uniforms.draw_clip_y1.max(y2).min(self.draw_uniforms.draw_clip_y2);
        Rect { pos: vec2(x1, y1), size: vec2(x2 - x1, y2 - y1) }
    }
}

/// Uniforms that can be set on the [`View`] that wraps a [`DrawCall`].
///
/// TODO(JP): Currently empty, but I can see this be potentially useful, so I left
/// the code around. Might want to either make use of this directly, or expose it
/// as something users can configure, or just remove altogether.
#[derive(Default, Clone)]
#[repr(C)]
pub struct ViewUniforms {}

impl ViewUniforms {
    pub fn as_slice(&self) -> &[f32; std::mem::size_of::<ViewUniforms>()] {
        unsafe { std::mem::transmute(self) }
    }
}

/// What kind of debug information should be printed?
///
/// Set using [`View::set_view_debug`].
#[derive(Clone, PartialEq)]
pub enum CxViewDebug {
    /// Print the draw tree.
    DrawTree,
    /// Print the draw tree and also information on the individual instances.
    Instances,
}

/// An actual instantiation of a [`View`]. It's a node in the draw tree with
/// children, which can be either [`DrawCall`]s or other [`View`]s.
///
/// Child [`View`]s are represented by [`DrawCall`]s that have [`DrawCall::sub_view_id`] set.
///
/// TODO(JP): this sub-[`View`] behavior is confusing, and we should instead
/// split out [`DrawCall`] into something like [`enum DrawTreeItem { DrawCall(DrawCall),
/// NestedView(usize) }`] or so.
///
/// See also [`View`] and [`ViewArea`].
#[derive(Default, Clone)]
pub struct CxView {
    /// The actual children, which are always [`DrawCall`] objects, but those can
    /// represent either actual draw calls or child [`CxView`]s (see [`DrawCall`] and
    /// [`CxView`] for more documentation).
    pub(crate) draw_calls: Vec<DrawCall>,
    /// The [`Rect`] of the [`Turtle`] that we created in [`View::begin_view`].
    ///
    /// TODO(JP): We want to decouple [`Turtle`] more from [`CxView`], so we have to
    /// figure out what to do with this. For [`View`]s that are actively used for
    /// scrolling and clipping, having this `rect` makes sense, but maybe not for
    /// other uses?
    pub(crate) rect: Rect,
    /// The id of the parent we nest in, codeflow wise.
    ///
    /// TODO(JP): we now have a tree in two directions: [`CxView::draw_calls`]
    /// contains the child [`CxView`]s, and [`CxView::nesting_view_id`] points to the
    /// parent. Is there any chance that they can get out of sync? Do we really
    /// need both? This one here seems only to be used in [`Cx::view_will_redraw`].
    /// If we do need to keep them, we might need some more runtime assertions to make
    /// sure they stay in sync.
    pub(crate) nesting_view_id: usize,
    /// The [`Cx::redraw_id`] of the last time this [`CxView`] was drawn. Can be used
    /// to see if an [`ViewArea`] pointer is still valid.
    ///
    /// TODO(JP): There is no way to tell if a [`CxView`] is still part of the draw tree,
    /// since merely comparing [`CxView::redraw_id`] and [`Cx::redraw_id`] is not
    /// enough, since those can also be different if the [`CxView`] was simply not
    /// marked for redrawing recently. It would be good to have some way to clean up
    /// old [`CxView`]s.
    pub(crate) redraw_id: u64,
    /// The [`Pass`]/[`CxPass`] that this is part of.
    ///
    /// TODO(JP): What happens if you change this after instantiating a [`CxView`]?
    /// Does that even work? Should it be supported?
    pub(crate) pass_id: usize,
    /// The actual number of fields in [`CxView::draw_calls`] that we use, so we can keep
    /// GPU resources associated with each [`DrawCall`] associated even when not in use.
    ///
    /// TODO(JP): Is this actually useful? Is caching of resources like that worth it, or
    /// should we do it on a per-platform basis, and only where it's really necessary?
    pub(crate) draw_calls_len: usize,
    /// The cumulative scroll offset from all of the parents. Gets set during painting.
    pub(crate) parent_scroll: Vec2,
    /// See [`ViewUniforms`].
    pub(crate) view_uniforms: ViewUniforms,
    /// The actual scroll position, including fractional offsets.
    pub(crate) unsnapped_scroll: Vec2,
    /// The scroll position that gets snapped to actual pixel values (taking into account
    /// the device pixel ratio; called `dpi_factor` internally).
    pub(crate) snapped_scroll: Vec2,
    /// Platform-specific fields.
    pub(crate) platform: CxPlatformView,
    /// Whether to print debug information. Set using [`View::set_view_debug`].
    pub(crate) debug: Option<CxViewDebug>,
}

impl CxView {
    /// Returns the intersection of clip coordinates and [`CxView::rect`], taking
    /// into account [`CxView::parent_scroll`].
    ///
    /// TODO(JP): Should this instead take and return a [`Rect`]?
    pub(crate) fn intersect_clip(&self, clip: (Vec2, Vec2)) -> (Vec2, Vec2) {
        let min_x = self.rect.pos.x - self.parent_scroll.x;
        let min_y = self.rect.pos.y - self.parent_scroll.y;
        let max_x = self.rect.pos.x + self.rect.size.x - self.parent_scroll.x;
        let max_y = self.rect.pos.y + self.rect.size.y - self.parent_scroll.y;

        (Vec2 { x: min_x.max(clip.0.x), y: min_y.max(clip.0.y) }, Vec2 { x: max_x.min(clip.1.x), y: max_y.min(clip.1.y) })
    }

    /// Find the last [`DrawCall`] within [`CxView::draw_calls`] that matches
    /// the given [`Shader`]. Returns an index.
    fn find_appendable_drawcall(&mut self, shader: Shader) -> Option<usize> {
        if self.draw_calls_len > 0 {
            for i in (0..self.draw_calls_len).rev() {
                let dc = &mut self.draw_calls[i];
                if dc.sub_view_id == 0 && dc.shader == shader {
                    return Some(i);
                }
            }
        }
        None
    }
}
