//! Convenience functions for writing common types to [`Shader`]s.

use crate::cx::*;

/// Convenience functions for writing common types to [`Shader`]s.
pub trait DrawInputType {
    /// Number of slots this value takes.
    fn slots() -> usize;
    /// This writes a value to the area (whether texture, uniform or instance).
    fn write_shader_value(self, cx: &mut Cx, area: Area, name: &str);
}

impl DrawInputType for f32 {
    #[inline]
    fn slots() -> usize {
        1
    }

    fn write_shader_value(self, cx: &mut Cx, area: Area, name: &str) {
        if let Some(wr) = area.get_write_ref(cx, StringHash::new(name), Ty::Float) {
            for i in 0..wr.repeat {
                wr.buffer[i * wr.stride] = self;
            }
        }
    }
}

impl DrawInputType for Vec2 {
    #[inline]
    fn slots() -> usize {
        2
    }

    fn write_shader_value(self, cx: &mut Cx, area: Area, name: &str) {
        if let Some(wr) = area.get_write_ref(cx, StringHash::new(name), Ty::Vec2) {
            for i in 0..wr.repeat {
                wr.buffer[i * wr.stride + 0] = self.x;
                wr.buffer[i * wr.stride + 1] = self.y;
            }
        }
    }
}

impl DrawInputType for Vec3 {
    #[inline]
    fn slots() -> usize {
        3
    }

    fn write_shader_value(self, cx: &mut Cx, area: Area, name: &str) {
        if let Some(wr) = area.get_write_ref(cx, StringHash::new(name), Ty::Vec3) {
            for i in 0..wr.repeat {
                wr.buffer[i * wr.stride + 0] = self.x;
                wr.buffer[i * wr.stride + 1] = self.y;
                wr.buffer[i * wr.stride + 2] = self.z;
            }
        }
    }
}

impl DrawInputType for Vec4 {
    #[inline]
    fn slots() -> usize {
        4
    }

    fn write_shader_value(self, cx: &mut Cx, area: Area, name: &str) {
        if let Some(wr) = area.get_write_ref(cx, StringHash::new(name), Ty::Vec4) {
            for i in 0..wr.repeat {
                wr.buffer[i * wr.stride + 0] = self.x;
                wr.buffer[i * wr.stride + 1] = self.y;
                wr.buffer[i * wr.stride + 2] = self.z;
                wr.buffer[i * wr.stride + 3] = self.w;
            }
        }
    }
}

impl DrawInputType for Mat4 {
    #[inline]
    fn slots() -> usize {
        16
    }

    // find uniform, then find instance prop
    fn write_shader_value(self, cx: &mut Cx, area: Area, name: &str) {
        if let Some(wr) = area.get_write_ref(cx, StringHash::new(name), Ty::Mat4) {
            for i in 0..wr.repeat {
                for j in 0..16 {
                    wr.buffer[i * wr.stride + j] = self.v[j];
                }
            }
        }
    }
}

impl DrawInputType for Texture2D {
    #[inline]
    fn slots() -> usize {
        0
    }

    fn write_shader_value(self, cx: &mut Cx, area: Area, name: &str) {
        if let Some(u) = self.0 {
            area.write_texture_2d_id(cx, StringHash::new(name), u as usize)
        }
    }
}
