(function() {
    self.importScripts(new URL("cx_webgl_Rpc.js", location.href).href);

    const rpc = new Rpc(self);

    var is_firefox = self.navigator.userAgent.toLowerCase().indexOf('firefox') > -1;
    // var is_add_to_homescreen_safari = is_mobile_safari && navigator.standalone;
    //var is_oculus_browser = navigator.userAgent.indexOf('OculusBrowser') > -1;

    const file_reader_sync = new FileReaderSync();

    // message we can send to wasm
    class ToWasm {
        constructor(wasm_app) {
            this.wasm_app = wasm_app;
            this.exports = wasm_app.exports;
            this.slots = 1024;
            this.used = 2; // skip 8 byte header
            this.buffer_len = 0;
            this.last_pointer = 0;
            // lets write
            this.pointer = this.exports.alloc_wasm_message(this.slots * 4);
            this.update_refs();
            this.fit(2); // Fit a f64 for the timestamp of when we send the message.
        }

        update_refs() {
            if (this.last_ptr != this.pointer || this.buffer_len != this.wasm_app.memory.buffer.byteLength) {
                this.last_pointer = this.pointer;
                this.buffer_len = this.wasm_app.memory.buffer.byteLength;
                this.mf32 = new Float32Array(this.wasm_app.memory.buffer, this.pointer, this.slots);
                this.mu32 = new Uint32Array(this.wasm_app.memory.buffer, this.pointer, this.slots);
                this.mf64 = new Float64Array(this.wasm_app.memory.buffer, this.pointer, this.slots >> 1);
                this.mu64 = new BigUint64Array(this.wasm_app.memory.buffer, this.pointer, this.slots >> 1);
            }
        }

        fit(slots) {
            this.update_refs(); // its possible our mf32/mu32 refs are dead because of realloccing of wasm heap inbetween calls
            if (this.used + slots > this.slots) {
                let new_slots = Math.max(this.used + slots, this.slots * 2) // exp alloc algo
                if (new_slots & 1)new_slots ++; // float64 align it
                let new_bytes = new_slots * 4;
                this.pointer = this.exports.realloc_wasm_message(this.pointer, new_bytes); // by
                this.slots = new_slots
                this.update_refs()
            }
            let pos = this.used;
            this.used += slots;
            return pos;
        }

        fetch_deps() {
            let port;
            if (!location.port) {
                if (location.protocol == "https:") {
                    port = 443;
                }
                else {
                    port = 80;
                }
            }
            else {
                port = parseInt(location.port);
            }
            let pos = this.fit(1);
            this.mu32[pos ++] = 1;

            pos = this.fit(1);
            this.mu32[pos ++] = port;
            this.send_string(location.protocol);
            this.send_string(location.hostname);
            this.send_string(location.pathname);
            this.send_string(location.search);
            this.send_string(location.hash);
        }

        // i forgot how to do memcpy with typed arrays. so, we'll do this.
        copy_to_wasm(input_buffer, output_ptr) {
            let u8len = input_buffer.byteLength;

            if ((u8len & 3) != 0 || (output_ptr & 3) != 0) { // not u32 aligned, do a byte copy
                var u8out = new Uint8Array(this.wasm_app.memory.buffer, output_ptr, u8len)
                var u8in = new Uint8Array(input_buffer)
                for (let i = 0; i < u8len; i ++) {
                    u8out[i] = u8in[i];
                }
            }
            else { // do a u32 copy
                let u32len = u8len >> 2; //4 bytes at a time.
                var u32out = new Uint32Array(this.wasm_app.memory.buffer, output_ptr, u32len)
                var u32in = new Uint32Array(input_buffer)
                for (let i = 0; i < u32len; i ++) {
                    u32out[i] = u32in[i];
                }
            }
        }

        alloc_wasm_vec(vec_len) {
            let ret = this.exports.alloc_wasm_vec(vec_len);
            this.update_refs();
            return ret
        }

        send_string(str) {
            let pos = this.fit(str.length + 1)
            this.mu32[pos ++] = str.length
            for (let i = 0; i < str.length; i ++) {
                this.mu32[pos ++] = str.charCodeAt(i)
            }
        }

        send_f64(value) {
            if (this.used & 1) { // float64 align, need to fit another
                let pos = this.fit(3);
                pos ++;
                this.mf64[pos >> 1] = value;
            }
            else {
                let pos = this.fit(2);
                this.mf64[pos >> 1] = value;
            }
        }

        send_u64(value) {
            if (this.used & 1) { // u64 align, need to fit another
                let pos = this.fit(3);
                pos ++;
                this.mu64[pos >> 1] = value;
            }
            else {
                let pos = this.fit(2);
                this.mu64[pos >> 1] = value;
            }
        }

        send_optional_mat4(matrix) {
            if (matrix == null) {
                let pos = this.fit(1);
                this.mu32[pos ++] = 0;
            }
            else {
                let pos = this.fit(17);
                this.mu32[pos ++] = 1;
                for (let i = 0; i < 16; i ++) {
                    this.mf32[pos ++] = matrix[i];
                }
            }
        }

        send_pose_transform(pose_transform) {
            // lets send over a pose.
            let pos = this.fit(7)
            let inv_orient = pose_transform.inverse.orientation;
            this.mf32[pos ++] = inv_orient.x;
            this.mf32[pos ++] = inv_orient.y;
            this.mf32[pos ++] = inv_orient.z;
            this.mf32[pos ++] = inv_orient.w;
            let tpos = pose_transform.position;
            this.mf32[pos ++] = tpos.x;
            this.mf32[pos ++] = tpos.y;
            this.mf32[pos ++] = tpos.z;
        }

        deps_loaded(deps) {
            let pos = this.fit(2);
            this.mu32[pos ++] = 2
            this.mu32[pos ++] = deps.length
            for (let i = 0; i < deps.length; i ++) {
                let dep = deps[i];
                this.send_string(dep.name);
                pos = this.fit(2);
                this.mu32[pos ++] = dep.vec_ptr
                this.mu32[pos ++] = dep.vec_len
            }
        }

        init(info) {
            let pos = this.fit(6);
            this.mu32[pos ++] = 3;
            this.mf32[pos ++] = info.width;
            this.mf32[pos ++] = info.height;
            this.mf32[pos ++] = info.dpi_factor;
            this.mu32[pos ++] = info.xr_can_present? 1: 0;
            this.mu32[pos ++] = info.can_fullscreen? 1: 0;
        }

        resize(info) {
            let pos = this.fit(7);
            this.mu32[pos ++] = 4;
            this.mf32[pos ++] = info.width;
            this.mf32[pos ++] = info.height;
            this.mf32[pos ++] = info.dpi_factor;
            this.mu32[pos ++] = info.xr_is_presenting? 1: 0;
            this.mu32[pos ++] = info.xr_can_present? 1: 0;
            this.mu32[pos ++] = info.is_fullscreen? 1: 0;
        }

        animation_frame() {
            let pos = this.fit(1);
            this.mu32[pos ++] = 5;
        }

        finger_down(finger) {
            let pos = this.fit(6);
            this.mu32[pos ++] = 6;
            this.mf32[pos ++] = finger.x
            this.mf32[pos ++] = finger.y
            this.mu32[pos ++] = finger.digit
            this.mu32[pos ++] = finger.touch? 1: 0
            this.mu32[pos ++] = finger.modifiers
            this.send_f64(finger.time);
        }

        finger_up(finger) {
            let pos = this.fit(6);
            this.mu32[pos ++] = 7;
            this.mf32[pos ++] = finger.x
            this.mf32[pos ++] = finger.y
            this.mu32[pos ++] = finger.digit
            this.mu32[pos ++] = finger.touch? 1: 0
            this.mu32[pos ++] = finger.modifiers
            this.send_f64(finger.time);
        }

        finger_move(finger) {
            let pos = this.fit(6);
            this.mu32[pos ++] = 8;
            this.mf32[pos ++] = finger.x
            this.mf32[pos ++] = finger.y
            this.mu32[pos ++] = finger.digit
            this.mu32[pos ++] = finger.touch? 1: 0
            this.mu32[pos ++] = finger.modifiers
            this.send_f64(finger.time);
        }

        finger_hover(finger) {
            let pos = this.fit(4);
            this.mu32[pos ++] = 9;
            this.mf32[pos ++] = finger.x
            this.mf32[pos ++] = finger.y
            this.mu32[pos ++] = finger.modifiers
            this.send_f64(finger.time);
        }

        finger_scroll(finger) {
            let pos = this.fit(7);
            this.mu32[pos ++] = 10;
            this.mf32[pos ++] = finger.x
            this.mf32[pos ++] = finger.y
            this.mf32[pos ++] = finger.scroll_x
            this.mf32[pos ++] = finger.scroll_y
            this.mu32[pos ++] = finger.is_wheel? 1: 0
            this.mu32[pos ++] = finger.modifiers
            this.send_f64(finger.time);
        }

        finger_out(finger) {
            let pos = this.fit(4);
            this.mu32[pos ++] = 11;
            this.mf32[pos ++] = finger.x
            this.mf32[pos ++] = finger.y
            this.mu32[pos ++] = finger.modifiers
            this.send_f64(finger.time);
        }

        key_down(key) {
            let pos = this.fit(4);
            this.mu32[pos ++] = 12;
            this.mu32[pos ++] = key.key_code;
            this.mu32[pos ++] = key.is_repeat? 1: 0;
            this.mu32[pos ++] = key.modifiers;
            this.send_f64(key.time);
        }

        key_up(key) {
            let pos = this.fit(4);
            this.mu32[pos ++] = 13;
            this.mu32[pos ++] = key.key_code;
            this.mu32[pos ++] = key.is_repeat? 1: 0;
            this.mu32[pos ++] = key.modifiers;
            this.send_f64(key.time);
        }

        text_input(data) {
            let pos = this.fit(3);
            this.mu32[pos ++] = 14;
            this.mu32[pos ++] = data.was_paste? 1: 0,
            this.mu32[pos ++] = data.replace_last? 1: 0,
            this.send_string(data.input);
        }

        read_file_data(id, buf_ptr, buf_len) {
            let pos = this.fit(4);
            this.mu32[pos ++] = 15;
            this.mu32[pos ++] = id;
            this.mu32[pos ++] = buf_ptr;
            this.mu32[pos ++] = buf_len;
        }

        read_file_error(id) {
            let pos = this.fit(2);
            this.mu32[pos ++] = 16;
            this.mu32[pos ++] = id;
        }


        text_copy() {
            let pos = this.fit(1);
            this.mu32[pos ++] = 17;
        }

        timer(id) {
            let pos = this.fit(1);
            this.mu32[pos ++] = 18;
            this.send_f64(id);
        }

        window_focus(is_focus) { // TODO CALL THIS
            let pos = this.fit(2);
            this.mu32[pos ++] = 19;
            this.mu32[pos ++] = is_focus? 1: 0;
        }

        xr_update_head(inputs_len, time) {
            //this.send_f64(time);
        }

        xr_update_inputs(xr_frame, xr_session, time, xr_pose, xr_reference_space) {
            // let input_len = xr_session.inputSources.length;

            // let pos = this.fit(2);
            // this.mu32[pos ++] = 20;
            // this.mu32[pos ++] = input_len;

            // this.send_f64(time / 1000.0);
            // this.send_pose_transform(xr_pose.transform);

            // for (let i = 0; i < input_len; i ++) {
            //     let input = xr_session.inputSources[i];
            //     let grip_pose = xr_frame.getPose(input.gripSpace, xr_reference_space);
            //     let ray_pose = xr_frame.getPose(input.targetRaySpace, xr_reference_space);
            //     if (grip_pose == null || ray_pose == null) {
            //         let pos = this.fit(1);
            //         this.mu32[pos ++] = 0;
            //         continue
            //     }
            //     let pos = this.fit(1);
            //     this.mu32[pos ++] = 1;

            //     this.send_pose_transform(grip_pose.transform)
            //     this.send_pose_transform(ray_pose.transform)
            //     let buttons = input.gamepad.buttons;
            //     let axes = input.gamepad.axes;
            //     let buttons_len = buttons.length;
            //     let axes_len = axes.length;

            //     pos = this.fit(3 + buttons_len * 2 + axes_len);
            //     this.mu32[pos ++] = input.handedness == "left"? 1: input.handedness == "right"? 2: 0;
            //     this.mu32[pos ++] = buttons_len;
            //     for (let i = 0; i < buttons_len; i ++) {
            //         this.mu32[pos ++] = buttons[i].pressed? 1: 0;
            //         this.mf32[pos ++] = buttons[i].value;
            //     }
            //     this.mu32[pos ++] = axes_len;
            //     for (let i = 0; i < axes_len; i ++) {
            //         this.mf32[pos ++] = axes[i]
            //     }
            // }
        }

        paint_dirty(time, frame_data) {
            let pos = this.fit(1);
            this.mu32[pos ++] = 21;
        }

        http_send_response(signal_id, success) {
            let pos = this.fit(3);
            this.mu32[pos ++] = 22;
            this.mu32[pos ++] = signal_id;
            this.mu32[pos ++] = success? 1: 2;
        }

        websocket_message(url, data) {
            let vec_len = data.byteLength;
            let vec_ptr = this.alloc_wasm_vec(vec_len);
            this.copy_to_wasm(data, vec_ptr);
            let pos = this.fit(3);
            this.mu32[pos ++] = 23;
            this.mu32[pos ++] = vec_ptr;
            this.mu32[pos ++] = vec_len;
            this.send_string(url);
        }

        websocket_error(url, error) {
            let pos = this.fit(1);
            this.mu32[pos ++] = 24;
            this.send_string(url);
            this.send_string(error);
        }

        app_open_files(file_handles) {
            let pos = this.fit(2);
            this.mu32[pos ++] = 25;
            this.mu32[pos ++] = file_handles.length;
            for (var file_handle of file_handles) {
                let pos = this.fit(1);
                this.mu32[pos ++] = file_handle.id;
                this.send_u64(BigInt(file_handle.file.size))
                this.send_string(file_handle.basename);
            }
        }

        end() {
            let pos = this.fit(1);
            this.mu32[pos] = 0;
        }
    }


    class WasmApp {
        constructor(canvas, webasm, memory, can_fullscreen, base_uri, wasm_path) {
            this.canvas = canvas;
            this.webasm = webasm;
            this.exports = webasm.instance.exports;
            this.memory = memory
            this.can_fullscreen = can_fullscreen;
            this.base_uri = base_uri;
            this.wasmPath = wasm_path;

            // local webgl resources
            this.shaders = [];
            this.index_buffers = [];
            this.array_buffers = [];
            this.timers = [];
            this.vaos = [];
            this.textures = [];
            this.framebuffers = [];
            this.resources = [];
            this.req_anim_frame_id = 0;
            this.websockets = {};
            this.file_handles = [];

            this.cursor_map = [
                "none", //Hidden=>0
                "default", //Default=>1,
                "crosshair", //CrossHair=>2,
                "pointer", //Hand=>3,
                "default", //Arrow=>4,
                "move", //Move=>5,
                "text", //Text=>6,
                "wait", //Wait=>7,
                "help", //Help=>8,
                "not-allowed", //NotAllowed=>9,
                "n-resize", // NResize=>10,
                "ne-resize", // NeResize=>11,
                "e-resize", // EResize=>12,
                "se-resize", // SeResize=>13,
                "s-resize", // SResize=>14,
                "sw-resize", // SwResize=>15,
                "w-resize", // WResize=>16,
                "nw-resize", // NwResize=>17,
                "ns-resize", //NsResize=>18,
                "nesw-resize", //NeswResize=>19,
                "ew-resize", //EwResize=>20,
                "nwse-resize", //NwseResize=>21,
                "col-resize", //ColResize=>22,
                "row-resize", //RowResize=>23,
            ];

            this.init_webgl_context();
            // this.run_async_webxr_check();
            this.bind_mouse_and_touch();
            this.bind_keyboard();

            rpc.receive("window_focus", () => {
                this.to_wasm.window_focus(true);
                this.do_wasm_io();
            });
            rpc.receive("window_blur", () => {
                this.to_wasm.window_focus(false);
                this.do_wasm_io();
            });

            // lets create the wasm app and cx
            this.app = this.exports.create_wasm_app();

            // create initial to_wasm
            this.to_wasm = new ToWasm(this);

            // fetch dependencies
            this.to_wasm.fetch_deps();

            this.do_wasm_io();

            this.do_wasm_block = true;

            // ok now, we wait for our resources to load.
            Promise.all(this.resources).then(this.do_dep_results.bind(this))
        }

        do_dep_results(results) {
            let deps = []
            // copy our reslts into wasm pointers
            for (let i = 0; i < results.length; i ++) {
                var result = results[i]
                // allocate pointer, do +8 because of the u64 length at the head of the buffer
                let vec_len = result.buffer.byteLength;
                let vec_ptr = this.to_wasm.alloc_wasm_vec(vec_len);
                this.to_wasm.copy_to_wasm(result.buffer, vec_ptr);
                deps.push({
                    name: result.name,
                    vec_ptr: vec_ptr,
                    vec_len: vec_len
                });
            }
            // pass wasm the deps
            this.to_wasm.deps_loaded(deps);
            // initialize the application
            this.to_wasm.init({
                width: this.width,
                height: this.height,
                dpi_factor: this.dpi_factor,
                xr_can_present: this.xr_can_present,
                can_fullscreen: this.can_fullscreen,
                xr_is_presenting: false,
            })
            this.do_wasm_block = false;
            this.do_wasm_io();

            rpc.send("remove_loading_indicators", {});
        }

        do_wasm_io() {
            if (this.do_wasm_block) {
                return
            }

            this.to_wasm.end();
            this.to_wasm.mf64[1] = performance.now() / 1000.0; // Fill in the slot that we reserved at the start.
            let from_wasm_ptr = this.exports.process_to_wasm(this.app, this.to_wasm.pointer)

            // get a clean to_wasm set up immediately
            this.to_wasm = new ToWasm(this);

            // set up local shortcuts to the from_wasm memory chunk for faster parsing
            this.parse = 2; // skip the 8 byte header
            this.mf32 = new Float32Array(this.memory.buffer, from_wasm_ptr);
            this.mu32 = new Uint32Array(this.memory.buffer, from_wasm_ptr);
            this.mf64 = new Float64Array(this.memory.buffer, from_wasm_ptr);
            this.mu64 = new BigUint64Array(this.memory.buffer, from_wasm_ptr);
            this.basef32 = new Float32Array(this.memory.buffer);
            this.baseu32 = new Uint32Array(this.memory.buffer);
            this.basef64 = new Float64Array(this.memory.buffer);
            this.baseu64 = new BigUint64Array(this.memory.buffer);

            // process all messages
            var send_fn_table = this.send_fn_table;

            while (1) {
                let msg_type = this.mu32[this.parse ++];
                if (send_fn_table[msg_type](this)) {
                    break;
                }
            }
            // destroy from_wasm_ptr object
            this.exports.dealloc_wasm_message(from_wasm_ptr);
        }


        parse_string() {
            var str = "";
            var len = this.mu32[this.parse ++];
            for (let i = 0; i < len; i ++) {
                var c = this.mu32[this.parse ++];
                if (c != 0) str += String.fromCharCode(c);
            }
            return str
        }

        parse_u8slice() {
            var str = "";
            var u8_len = this.mu32[this.parse ++];
            let len = u8_len >> 2;
            let data = new Uint8Array(u8_len);
            let spare = u8_len & 3;
            for (let i = 0; i < len; i ++) {
                let u8_pos = i << 2;
                let u32 = this.mu32[this.parse ++];
                data[u8_pos + 0] = u32 & 0xff;
                data[u8_pos + 1] = (u32 >> 8) & 0xff;
                data[u8_pos + 2] = (u32 >> 16) & 0xff;
                data[u8_pos + 3] = (u32 >> 24) & 0xff;
            }
            let u8_pos = len << 2;
            if (spare == 1) {
                let u32 = this.mu32[this.parse ++];
                data[u8_pos + 0] = u32 & 0xff;
            }
            else if (spare == 2) {
                let u32 = this.mu32[this.parse ++];
                data[u8_pos + 0] = u32 & 0xff;
                data[u8_pos + 1] = (u32 >> 8) & 0xff;
            }
            else if (spare == 3) {
                let u32 = this.mu32[this.parse ++];
                data[u8_pos + 0] = u32 & 0xff;
                data[u8_pos + 1] = (u32 >> 8) & 0xff;
                data[u8_pos + 2] = (u32 >> 16) & 0xff;
            }
            return data
        }

        parse_f64() {
            if (this.parse & 1) {
                this.parse ++;
            }
            var ret = this.mf64[this.parse >> 1];
            this.parse += 2;
            return ret
        }

        parse_shvarvec() {
            var len = this.mu32[this.parse ++];
            var vars = []
            for (let i = 0; i < len; i ++) {
                vars.push({ty: this.parse_string(), name: this.parse_string()})
            }
            return vars
        }


        load_deps(deps) {
            for (var i = 0; i < deps.length; i ++) {
                let file_path = deps[i];
                this.resources.push(this.fetch_path(file_path))
            }
        }

        set_document_title(title) {
            rpc.send("set_document_title", { title });
        }

        bind_mouse_and_touch() {
            let last_mouse_finger;
            // TODO(JP): Some day bring back touch support..
            // let use_touch_scroll_overlay = window.ontouchstart === null;
            // if (use_touch_scroll_overlay) {
            //     var ts = this.touch_scroll_overlay = document.createElement('div')
            //     ts.className = "cx_webgl_scroll_overlay"
            //     var ts_inner = document.createElement('div')
            //     var style = document.createElement('style')
            //     style.innerHTML = "\n"
            //         + "div.cx_webgl_scroll_overlay {\n"
            //         + "z-index: 10000;\n"
            //         + "margin:0;\n"
            //         + "overflow:scroll;\n"
            //         + "top:0;\n"
            //         + "left:0;\n"
            //         + "width:100%;\n"
            //         + "height:100%;\n"
            //         + "position:fixed;\n"
            //         + "background-color:transparent\n"
            //         + "}\n"
            //         + "div.cx_webgl_scroll_overlay div{\n"
            //         + "margin:0;\n"
            //         + "width:400000px;\n"
            //         + "height:400000px;\n"
            //         + "background-color:transparent\n"
            //         + "}\n"

            //     document.body.appendChild(style)
            //     ts.appendChild(ts_inner);
            //     document.body.appendChild(ts);
            //     canvas = ts;

            //     ts.scrollTop = 200000;
            //     ts.scrollLeft = 200000;
            //     let last_scroll_top = ts.scrollTop;
            //     let last_scroll_left = ts.scrollLeft;
            //     let scroll_timeout = null;
            //     ts.addEventListener('scroll', e => {
            //         let new_scroll_top = ts.scrollTop;
            //         let new_scroll_left = ts.scrollLeft;
            //         let dx = new_scroll_left - last_scroll_left;
            //         let dy = new_scroll_top - last_scroll_top;
            //         last_scroll_top = new_scroll_top;
            //         last_scroll_left = new_scroll_left;
            //         self.clearTimeout(scroll_timeout);
            //         scroll_timeout = self.setTimeout(_ => {
            //             ts.scrollTop = 200000;
            //             ts.scrollLeft = 200000;
            //             last_scroll_top = ts.scrollTop;
            //             last_scroll_left = ts.scrollLeft;
            //         }, 200);

            //         let finger = last_mouse_finger;
            //         if (finger) {
            //             finger.scroll_x = dx;
            //             finger.scroll_y = dy;
            //             finger.is_wheel = true;
            //             this.to_wasm.finger_scroll(finger);
            //             this.do_wasm_io();
            //         }
            //     })
            // }

            var mouse_fingers = [];
            function mouse_to_finger(e) {
                let mf = mouse_fingers[e.button] || (mouse_fingers[e.button] = {});
                mf.x = e.pageX;
                mf.y = e.pageY;
                mf.digit = e.button;
                mf.time = performance.now() / 1000.0;
                mf.modifiers = pack_key_modifier(e);
                mf.touch = false;
                return mf
            }

            // var digit_map = {}
            // var digit_alloc = 0;

            // function touch_to_finger_alloc(e) {
            //     var f = []
            //     for (let i = 0; i < e.changedTouches.length; i ++) {
            //         var t = e.changedTouches[i]
            //         // find an unused digit
            //         var digit = undefined;
            //         for (digit in digit_map) {
            //             if (!digit_map[digit]) break
            //         }
            //         // we need to alloc a new one
            //         if (digit === undefined || digit_map[digit]) digit = digit_alloc ++;
            //         // store it
            //         digit_map[digit] = {identifier: t.identifier};
            //         // return allocated digit
            //         digit = parseInt(digit);

            //         f.push({
            //             x: t.pageX,
            //             y: t.pageY,
            //             digit: digit,
            //             time: e.timeStamp / 1000.0,
            //             modifiers: 0,
            //             touch: true,
            //         })
            //     }
            //     return f
            // }

            // function lookup_digit(identifier) {
            //     for (let digit in digit_map) {
            //         var digit_id = digit_map[digit]
            //         if (!digit_id) continue
            //         if (digit_id.identifier == identifier) {
            //             return digit
            //         }
            //     }
            // }

            // function touch_to_finger_lookup(e) {
            //     var f = []
            //     for (let i = 0; i < e.changedTouches.length; i ++) {
            //         var t = e.changedTouches[i]
            //         f.push({
            //             x: t.pageX,
            //             y: t.pageY,
            //             digit: lookup_digit(t.identifier),
            //             time: e.timeStamp / 1000.0,
            //             modifiers: {},
            //             touch: true,
            //         })
            //     }
            //     return f
            // }

            // function touch_to_finger_free(e) {
            //     var f = []
            //     for (let i = 0; i < e.changedTouches.length; i ++) {
            //         var t = e.changedTouches[i]
            //         var digit = lookup_digit(t.identifier)
            //         if (!digit) {
            //             console.log("Undefined state in free_digit");
            //             digit = 0
            //         }
            //         else {
            //             digit_map[digit] = undefined
            //         }

            //         f.push({
            //             x: t.pageX,
            //             y: t.pageY,
            //             time: e.timeStamp / 1000.0,
            //             digit: digit,
            //             modifiers: 0,
            //             touch: true,
            //         })
            //     }
            //     return f
            // }

            // var easy_xr_presenting_toggle = window.localStorage.getItem("xr_presenting") == "true"

            var mouse_buttons_down = [];
            rpc.receive("canvas_mousedown", ({ event }) => {
                mouse_buttons_down[event.button] = true;
                this.to_wasm.finger_down(mouse_to_finger(event))
                this.do_wasm_io();
            });

            rpc.receive("window_mouseup", ({ event }) => {
                mouse_buttons_down[event.button] = false;
                this.to_wasm.finger_up(mouse_to_finger(event))
                this.do_wasm_io();
            });

            rpc.receive("window_mousemove", ({ event }) => {
                for (var i = 0; i < mouse_buttons_down.length; i ++) {
                    if (mouse_buttons_down[i]) {
                        let mf = mouse_to_finger(event);
                        mf.digit = i;
                        this.to_wasm.finger_move(mf);
                    }
                }
                last_mouse_finger = mouse_to_finger(event);
                this.to_wasm.finger_hover(last_mouse_finger);
                this.do_wasm_io();
                //console.log("Redraw cycle "+(end-begin)+" ms");
            });

            rpc.receive("window_mouseout", ({ event }) => {
                this.to_wasm.finger_out(mouse_to_finger(event)) //e.pageX, e.pageY, pa;
                this.do_wasm_io();
            });
            // canvas.addEventListener('touchstart', e => {
            //     e.preventDefault()

            //     let fingers = touch_to_finger_alloc(e);
            //     for (let i = 0; i < fingers.length; i ++) {
            //         this.to_wasm.finger_down(fingers[i])
            //     }
            //     this.do_wasm_io();
            //     return false
            // })
            // canvas.addEventListener('touchmove', e => {
            //     //e.preventDefault();
            //     var fingers = touch_to_finger_lookup(e);
            //     for (let i = 0; i < fingers.length; i ++) {
            //         this.to_wasm.finger_move(fingers[i])
            //     }
            //     this.do_wasm_io();
            //     return false
            // }, {passive: false})

            // var end_cancel_leave = e => {
            //     //if (easy_xr_presenting_toggle) {
            //     //    easy_xr_presenting_toggle = false;
            //     //    this.xr_start_presenting();
            //     //};

            //     e.preventDefault();
            //     var fingers = touch_to_finger_free(e);
            //     for (let i = 0; i < fingers.length; i ++) {
            //         this.to_wasm.finger_up(fingers[i])
            //     }
            //     this.do_wasm_io();
            //     return false
            // }

            // canvas.addEventListener('touchend', end_cancel_leave);
            // canvas.addEventListener('touchcancel', end_cancel_leave);
            // canvas.addEventListener('touchleave', end_cancel_leave);

            var last_wheel_time;
            var last_was_wheel;
            rpc.receive("canvas_wheel", ({ event, offsetHeight }) => {
                var finger = mouse_to_finger(event)
                let delta = event.timeStamp - last_wheel_time;
                last_wheel_time = event.timeStamp;
                // typical web bullshit. this reliably detects mousewheel or touchpad on mac in safari
                if (is_firefox) {
                    last_was_wheel = event.deltaMode == 1
                }
                else { // detect it
                    if (Math.abs(Math.abs((event.deltaY / event.wheelDeltaY)) - (1. / 3.)) < 0.00001 || !last_was_wheel && delta < 250) {
                        last_was_wheel = false;
                    }
                    else {
                        last_was_wheel = true;
                    }
                }
                //console.log(event.deltaY / event.wheelDeltaY);
                //last_delta = delta;
                var fac = 1
                if (event.deltaMode === 1) fac = 40
                else if (event.deltaMode === 2) fac = offsetHeight
                finger.scroll_x = event.deltaX * fac
                finger.scroll_y = event.deltaY * fac
                finger.is_wheel = last_was_wheel;
                this.to_wasm.finger_scroll(finger);
                this.do_wasm_io();
            });

            //window.addEventListener('webkitmouseforcewillbegin', this.onCheckMacForce.bind(this), false)
            //window.addEventListener('webkitmouseforcechanged', this.onCheckMacForce.bind(this), false)
        }

        bind_keyboard() {
            rpc.receive("text_input", ({ was_paste, input, replace_last }) => {
                this.to_wasm.text_input({ was_paste, input, replace_last });
                this.do_wasm_io();
            });
            rpc.receive("text_copy", () => {
                this.to_wasm.text_copy();
                this.do_wasm_io();
            });
            rpc.receive("key_down", ({ event }) => {
                this.to_wasm.key_down({
                    key_code: event.keyCode,
                    char_code: event.charCode,
                    is_repeat: event.repeat,
                    time: performance.now() / 1000.0,
                    modifiers: pack_key_modifier(event),
                });
                this.do_wasm_io();
            });
            rpc.receive("key_up", ({ event }) => {
                this.to_wasm.key_up({
                    key_code: event.keyCode,
                    char_code: event.charCode,
                    is_repeat: event.repeat,
                    time: performance.now() / 1000.0,
                    modifiers: pack_key_modifier(event),
                })
                this.do_wasm_io();
            });
        }

        set_mouse_cursor(id) {
            rpc.send("set_mouse_cursor", { style: this.cursor_map[id] || 'default' });
        }

        read_file(id, file_path) {
            this.fetch_path(file_path).then(result => {
                let byte_len = result.buffer.byteLength
                let output_ptr = this.to_wasm.alloc_wasm_vec(byte_len);
                this.to_wasm.copy_to_wasm(result.buffer, output_ptr);
                this.to_wasm.read_file_data(id, output_ptr, byte_len)
                this.do_wasm_io();
            }, err => {
                this.to_wasm.read_file_error(id)
                this.do_wasm_io();
            })
        }

        start_timer(id, interval, repeats) {
            for (let i = 0; i < this.timers.length; i ++) {
                if (this.timers[i].id == id) {
                    console.log("Timer ID collision!")
                    return
                }
            }
            var obj = {id: id, repeats: repeats};
            if (repeats !== 0) {
                obj.sys_id = self.setInterval(e => {
                    this.to_wasm.timer(id);
                    this.do_wasm_io();
                }, interval * 1000.0);
            }
            else {
                obj.sys_id = self.setTimeout(e => {
                    for (let i = 0; i < this.timers.length; i ++) {
                        let timer = this.timers[i];
                        if (timer.id == id) {
                            this.timers.splice(i, 1);
                            break;
                        }
                    }
                    this.to_wasm.timer(id);
                    this.do_wasm_io();
                }, interval * 1000.0);
            }
            this.timers.push(obj)
        }

        stop_timer(id) {
            for (let i = 0; i < this.timers.length; i ++) {
                let timer = this.timers[i];
                if (timer.id == id) {
                    if (timer.repeats) {
                        self.clearInterval(timer.sys_id);
                    }
                    else {
                        self.clearTimeout(timer.sys_id);
                    }
                    this.timers.splice(i, 1);
                    return
                }
            }
            //console.log("Timer ID not found!")
        }

        http_send(verb, path, proto, domain, port, content_type, body, signal_id) {

            var req = new XMLHttpRequest()
            req.addEventListener("error", _ => {
                // signal fail
                this.to_wasm.http_send_response(signal_id, 2);
                this.do_wasm_io();
            })
            req.addEventListener("load", _ => {
                if (req.status !== 200) {
                    // signal fail
                    this.to_wasm.http_send_response(signal_id, 2);
                }
                else {
                    //signal success
                    this.to_wasm.http_send_response(signal_id, 1);
                }
                this.do_wasm_io();
            })
            req.open(verb, proto + "://" + domain + ":" + port + path, true);
            console.log(verb, proto + "://" + domain + ":" + port + path, body);
            req.send(body.buffer);
        }

        websocket_send(url, data) {
            let socket = this.websockets[url];
            if (!socket) {
                let socket = new WebSocket(url);
                this.websockets[url] = socket;
                socket.send_stack = [data];
                socket.addEventListener('close', event => {
                    this.websockets[url] = null;
                })
                socket.addEventListener('error', event => {
                    this.websockets[url] = null;
                    this.to_wasm.websocket_error(url, "" + event);
                    this.do_wasm_io();
                })
                socket.addEventListener('message', event => {
                    event.data.arrayBuffer().then(data => {
                        this.to_wasm.websocket_message(url, data);
                        this.do_wasm_io();
                    })
                })
                socket.addEventListener('open', event => {
                    let send_stack = socket.send_stack;
                    socket.send_stack = null;
                    for (data of send_stack) {
                        socket.send(data);
                    }
                })
            }
            else {
                if (socket.send_stack) {
                    socket.send_stack.push(data);
                }
                else {
                    socket.send(data);
                }
            }
        }

        enable_global_file_drop_target() {
            rpc.send("enable_global_file_drop_target", {});
            rpc.receive("drop", ({ files }) => {
                let file_handles_to_send = [];
                for (const file of files) {
                    let file_handle = { id: this.file_handles.length, basename: file.name, file, last_read_start: -1, last_read_end: -1 };
                    file_handles_to_send.push(file_handle);
                    this.file_handles.push(file_handle);
                }
                this.to_wasm.app_open_files(file_handles_to_send);
                this.do_wasm_io();
            });
        }

        init_webgl_context() {
            rpc.receive("on_screen_resize", ({ dpi_factor, width, height, is_fullscreen }) => {
                this.dpi_factor = dpi_factor;
                this.width = width;
                this.height = height;

                this.canvas.width = width * dpi_factor;
                this.canvas.height = height * dpi_factor;
                this.gl.viewport(0, 0, this.canvas.width, this.canvas.height);

                this.to_wasm.resize({
                    width: this.width,
                    height: this.height,
                    dpi_factor: this.dpi_factor,
                    xr_is_presenting: this.xr_is_presenting,
                    xr_can_present: this.xr_can_present,
                    is_fullscreen,
                });
                this.request_animation_frame();
            });

            var options = {
                preferLowPowerToHighPerformance: true,
                // xrCompatible: true // TODO(JP): Bring back some day?
            };
            var gl = this.gl = this.canvas.getContext('webgl', options)
                || this.canvas.getContext('webgl-experimental', options)
                || this.canvas.getContext('experimental-webgl', options);

            if (!gl) {
                rpc.send("show_incompatible_browser_notification", {});
                return
            }
            this.OES_standard_derivatives = gl.getExtension('OES_standard_derivatives')
            this.OES_vertex_array_object = gl.getExtension('OES_vertex_array_object')
            this.OES_element_index_uint = gl.getExtension("OES_element_index_uint")
            this.ANGLE_instanced_arrays = gl.getExtension('ANGLE_instanced_arrays')
        }

        request_animation_frame() {
            if (this.xr_is_presenting || this.req_anim_frame_id) {
                return;
            }
            this.req_anim_frame_id = self.requestAnimationFrame(time => {
                this.req_anim_frame_id = 0;
                if (this.xr_is_presenting) {
                    return
                }
                this.to_wasm.animation_frame();
                this.in_animation_frame = true;
                this.do_wasm_io();
                this.in_animation_frame = false;

            })
        }

        run_async_webxr_check() {
            this.xr_can_present = false;
            this.xr_is_presenting = false;

            // ok this changes a bunch in how the renderflow works.
            // first thing we are going to do is get the vr displays.
            let xr_system = self.navigator.xr;
            if (xr_system) {

                xr_system.isSessionSupported('immersive-vr').then(supported => {

                    if (supported) {
                        this.xr_can_present = true;
                    }
                });
            }
            else {
                console.log("No webVR support found")
            }
        }


        xr_start_presenting() {
            // TODO(JP): Some day bring back XR support?
            // if (this.xr_can_present) {
            //     navigator.xr.requestSession('immersive-vr', {requiredFeatures: ['local-floor']}).then(xr_session => {

            //         let xr_layer = new XRWebGLLayer(xr_session, this.gl, {
            //             antialias: false,
            //             depth: true,
            //             stencil: false,
            //             alpha: false,
            //             ignoreDepthValues: false,
            //             framebufferScaleFactor: 1.5
            //         });

            //         xr_session.updateRenderState({baseLayer: xr_layer});
            //         xr_session.requestReferenceSpace("local-floor").then(xr_reference_space => {
            //             window.localStorage.setItem("xr_presenting", "true");

            //             this.xr_reference_space = xr_reference_space;
            //             this.xr_session = xr_session;
            //             this.xr_is_presenting = true;
            //             let first_on_resize = true;

            //             // read shit off the gamepad
            //             xr_session.gamepad;

            //             // lets start the loop
            //             let inputs = [];
            //             let alternate = false;
            //             let last_time;
            //             let xr_on_request_animation_frame = (time, xr_frame) => {

            //                 if (first_on_resize) {
            //                     this.on_screen_resize();
            //                     first_on_resize = false;
            //                 }
            //                 if (!this.xr_is_presenting) {
            //                     return;
            //                 }
            //                 this.xr_session.requestAnimationFrame(xr_on_request_animation_frame);
            //                 this.xr_pose = xr_frame.getViewerPose(this.xr_reference_space);
            //                 if (!this.xr_pose) {
            //                     return;
            //                 }

            //                 this.to_wasm.xr_update_inputs(xr_frame, xr_session, time, this.xr_pose, this.xr_reference_space)
            //                 this.to_wasm.animation_frame(time / 1000.0);
            //                 this.in_animation_frame = true;
            //                 let start = performance.now();
            //                 this.do_wasm_io();
            //                 this.in_animation_frame = false;
            //                 this.xr_pose = null;
            //                 //let new_time = performance.now();
            //                 //if (new_time - last_time > 13.) {
            //                 //    console.log(new_time - last_time);
            //                 // }
            //                 //last_time = new_time;
            //             }
            //             this.xr_session.requestAnimationFrame(xr_on_request_animation_frame);

            //             this.xr_session.addEventListener("end", () => {
            //                 window.localStorage.setItem("xr_presenting", "false");
            //                 this.xr_is_presenting = false;
            //                 this.on_screen_resize();
            //                 this.to_wasm.paint_dirty();
            //                 this.request_animation_frame();
            //             })
            //         })
            //     })
            // }
        }

        xr_stop_presenting() {

        }


        begin_main_canvas(r, g, b, a, depth) {
            let gl = this.gl
            this.is_main_canvas = true;
            if (this.xr_is_presenting) {
                // let xr_webgllayer = this.xr_session.renderState.baseLayer;
                // this.gl.bindFramebuffer(gl.FRAMEBUFFER, xr_webgllayer.framebuffer);
                // gl.viewport(0, 0, xr_webgllayer.framebufferWidth, xr_webgllayer.framebufferHeight);

                // // quest 1 is 3648
                // // quest 2 is 4096

                // let left_view = this.xr_pose.views[0];
                // let right_view = this.xr_pose.views[1];

                // this.xr_left_viewport = xr_webgllayer.getViewport(left_view);
                // this.xr_right_viewport = xr_webgllayer.getViewport(right_view);

                // this.xr_left_projection_matrix = left_view.projectionMatrix;
                // this.xr_left_transform_matrix = left_view.transform.inverse.matrix;
                // this.xr_left_invtransform_matrix = left_view.transform.matrix;

                // this.xr_right_projection_matrix = right_view.projectionMatrix;
                // this.xr_right_transform_matrix = right_view.transform.inverse.matrix;
                // this.xr_right_camera_pos = right_view.transform.inverse.position;
                // this.xr_right_invtransform_matrix = right_view.transform.matrix;
            }
            else {
                gl.bindFramebuffer(gl.FRAMEBUFFER, null);
                gl.viewport(0, 0, this.canvas.width, this.canvas.height);
            }

            gl.clearColor(r, g, b, a);
            gl.clearDepth(depth);
            gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
        }


        begin_render_targets(pass_id, width, height) {
            let gl = this.gl
            this.target_width = width;
            this.target_height = height;
            this.color_targets = 0;
            this.clear_flags = 0;
            this.is_main_canvas = false;
            var gl_framebuffer = this.framebuffers[pass_id] || (this.framebuffers[pass_id] = gl.createFramebuffer());
            gl.bindFramebuffer(gl.FRAMEBUFFER, gl_framebuffer);
        }

        add_color_target(texture_id, init_only, r, g, b, a) {
            // if use_default
            this.clear_r = r;
            this.clear_g = g;
            this.clear_b = b;
            this.clear_a = a;
            var gl = this.gl;

            var gl_tex = this.textures[texture_id] || (this.textures[texture_id] = gl.createTexture());

            // resize or create texture
            if (gl_tex.mp_width != this.target_width || gl_tex.mp_height != this.target_height) {
                gl.bindTexture(gl.TEXTURE_2D, gl_tex)
                this.clear_flags = gl.COLOR_BUFFER_BIT;

                gl_tex.mp_width = this.target_width
                gl_tex.mp_height = this.target_height
                gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.LINEAR)
                gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.LINEAR)
                gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE)
                gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE)

                gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, gl_tex.mp_width, gl_tex.mp_height, 0, gl.RGBA, gl.UNSIGNED_BYTE, null);
            }
            else if (!init_only) {
                this.clear_flags = gl.COLOR_BUFFER_BIT;
            }

            gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0, gl.TEXTURE_2D, gl_tex, 0)
            this.color_targets += 1;
        }

        set_depth_target(texture_id, init_only, depth) {
            this.clear_depth = depth;
            //console.log("IMPLEMENT DEPTH TEXTURE TARGETS ON WEBGL")
        }

        end_render_targets() {
            var gl = this.gl;

            // process the actual 'clear'
            gl.viewport(0, 0, this.target_width, this.target_height);

            // check if we need to clear color, and depth
            // clear it
            if (this.clear_flags) {
                gl.clearColor(this.clear_r, this.clear_g, this.clear_b, this.clear_a);
                gl.clearDepth(this.clear_depth);
                gl.clear(this.clear_flags);
            }
        }

        set_default_depth_and_blend_mode() {
            let gl = this.gl
            gl.enable(gl.DEPTH_TEST);
            gl.depthFunc(gl.LEQUAL);
            gl.blendEquationSeparate(gl.FUNC_ADD, gl.FUNC_ADD);
            gl.blendFuncSeparate(gl.ONE, gl.ONE_MINUS_SRC_ALPHA, gl.ONE, gl.ONE_MINUS_SRC_ALPHA);
            gl.enable(gl.BLEND);
        }

        // new shader helpers
        get_attrib_locations(program, base, slots) {
            var gl = this.gl;
            let attrib_locs = [];
            let attribs = slots >> 2;
            let stride = slots * 4;
            if ((slots & 3) != 0) attribs ++;
            for (let i = 0; i < attribs; i ++) {
                let size = (slots - i * 4);
                if (size > 4) size = 4;
                attrib_locs.push({
                    loc: gl.getAttribLocation(program, base + i),
                    offset: i * 16,
                    size: size,
                    stride: slots * 4
                });
            }
            return attrib_locs
        }

        get_uniform_locations(program, uniforms) {
            var gl = this.gl;
            let uniform_locs = [];
            let offset = 0;
            for (let i = 0; i < uniforms.length; i ++) {
                let uniform = uniforms[i];
                // lets align the uniform
                let slots = this.uniform_size_table[uniform.ty];

                if ((offset & 3) != 0 && (offset & 3) + slots > 4) { // goes over the boundary
                    offset += 4 - (offset & 3); // make jump to new slot
                }
                uniform_locs.push({
                    name: uniform.name,
                    offset: offset << 2,
                    ty: uniform.ty,
                    loc: gl.getUniformLocation(program, uniform.name),
                    fn: this.uniform_fn_table[uniform.ty]
                });
                offset += slots
            }
            return uniform_locs;
        }

        compile_webgl_shader(ash) {
            var gl = this.gl
            var vsh = gl.createShader(gl.VERTEX_SHADER)

            gl.shaderSource(vsh, ash.vertex)
            gl.compileShader(vsh)
            if (!gl.getShaderParameter(vsh, gl.COMPILE_STATUS)) {
                return console.log(
                    gl.getShaderInfoLog(vsh),
                    add_line_numbers_to_string(ash.vertex)
                )
            }

            // compile pixelshader
            var fsh = gl.createShader(gl.FRAGMENT_SHADER)
            gl.shaderSource(fsh, ash.fragment)
            gl.compileShader(fsh)
            if (!gl.getShaderParameter(fsh, gl.COMPILE_STATUS)) {
                return console.log(
                    gl.getShaderInfoLog(fsh),
                    add_line_numbers_to_string(ash.fragment)
                )
            }

            var program = gl.createProgram()
            gl.attachShader(program, vsh)
            gl.attachShader(program, fsh)
            gl.linkProgram(program)
            if (!gl.getProgramParameter(program, gl.LINK_STATUS)) {
                return console.log(
                    gl.getProgramInfoLog(program),
                    add_line_numbers_to_string(ash.vertex),
                    add_line_numbers_to_string(ash.fragment)
                )
            }
            // fetch all attribs and uniforms
            this.shaders[ash.shader_id] = {
                geom_attribs: this.get_attrib_locations(program, "mpsc_packed_geometry_", ash.geometry_slots),
                inst_attribs: this.get_attrib_locations(program, "mpsc_packed_instance_", ash.instance_slots),
                pass_uniforms: this.get_uniform_locations(program, ash.pass_uniforms),
                view_uniforms: this.get_uniform_locations(program, ash.view_uniforms),
                draw_uniforms: this.get_uniform_locations(program, ash.draw_uniforms),
                user_uniforms: this.get_uniform_locations(program, ash.user_uniforms),
                texture_slots: this.get_uniform_locations(program, ash.texture_slots),
                instance_slots: ash.instance_slots,
                program: program,
                ash: ash
            };
        }

        alloc_array_buffer(array_buffer_id, array) {
            var gl = this.gl;
            let buf = this.array_buffers[array_buffer_id];
            if (buf === undefined) {
                buf = this.array_buffers[array_buffer_id] = {
                    gl_buf: gl.createBuffer(),
                };
            }
            buf.length = array.length;
            gl.bindBuffer(gl.ARRAY_BUFFER, buf.gl_buf);
            gl.bufferData(gl.ARRAY_BUFFER, array, gl.STATIC_DRAW);
            gl.bindBuffer(gl.ARRAY_BUFFER, null);
        }

        alloc_index_buffer(index_buffer_id, array) {
            var gl = this.gl;

            let buf = this.index_buffers[index_buffer_id];
            if (buf === undefined) {
                buf = this.index_buffers[index_buffer_id] = {
                    gl_buf: gl.createBuffer()
                };
            };
            buf.length = array.length;
            gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, buf.gl_buf);
            gl.bufferData(gl.ELEMENT_ARRAY_BUFFER, array, gl.STATIC_DRAW);
            gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, null);
        }

        alloc_texture(texture_id, width, height, data_ptr) {
            var gl = this.gl;
            var gl_tex = this.textures[texture_id] || gl.createTexture()

            gl.bindTexture(gl.TEXTURE_2D, gl_tex)
            gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.LINEAR)
            gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.LINEAR)
            gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE)
            gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE)

            let data = new Uint8Array(this.memory.buffer, data_ptr, width * height * 4);
            gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, width, height, 0, gl.RGBA, gl.UNSIGNED_BYTE, data);
            //gl.bindTexture(gl.TEXTURE_2D,0);
            this.textures[texture_id] = gl_tex;
        }

        alloc_vao(vao_id, shader_id, geom_ib_id, geom_vb_id, inst_vb_id) {
            let gl = this.gl;
            let old_vao = this.vaos[vao_id];
            if (old_vao) {
                this.OES_vertex_array_object.deleteVertexArrayOES(old_vao.gl);
            }
            let gl_vao = this.OES_vertex_array_object.createVertexArrayOES();
            let vao = this.vaos[vao_id] = {
                gl_vao: gl_vao,
                geom_ib_id,
                geom_vb_id,
                inst_vb_id
            };

            this.OES_vertex_array_object.bindVertexArrayOES(vao.gl_vao)
            gl.bindBuffer(gl.ARRAY_BUFFER, this.array_buffers[geom_vb_id].gl_buf);

            let shader = this.shaders[shader_id];

            for (let i = 0; i < shader.geom_attribs.length; i ++) {
                let attr = shader.geom_attribs[i];
                if (attr.loc < 0) {
                    continue;
                }
                gl.vertexAttribPointer(attr.loc, attr.size, gl.FLOAT, false, attr.stride, attr.offset);
                gl.enableVertexAttribArray(attr.loc);
                this.ANGLE_instanced_arrays.vertexAttribDivisorANGLE(attr.loc, 0);
            }

            gl.bindBuffer(gl.ARRAY_BUFFER, this.array_buffers[inst_vb_id].gl_buf);
            for (let i = 0; i < shader.inst_attribs.length; i ++) {
                let attr = shader.inst_attribs[i];
                if (attr.loc < 0) {
                    continue;
                }
                gl.vertexAttribPointer(attr.loc, attr.size, gl.FLOAT, false, attr.stride, attr.offset);
                gl.enableVertexAttribArray(attr.loc);
                this.ANGLE_instanced_arrays.vertexAttribDivisorANGLE(attr.loc, 1);
            }

            gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, this.index_buffers[geom_ib_id].gl_buf);
            this.OES_vertex_array_object.bindVertexArrayOES(null);
        }

        draw_call(
            shader_id,
            vao_id,
            pass_uniforms_ptr,
            view_uniforms_ptr,
            draw_uniforms_ptr,
            user_uniforms_ptr,
            textures_ptr
        ) {
            var gl = this.gl;

            let shader = this.shaders[shader_id];
            gl.useProgram(shader.program);

            let vao = this.vaos[vao_id];

            this.OES_vertex_array_object.bindVertexArrayOES(vao.gl_vao);

            let index_buffer = this.index_buffers[vao.geom_ib_id];
            let instance_buffer = this.array_buffers[vao.inst_vb_id];
            // set up uniforms TODO do this a bit more incremental based on uniform layer
            // also possibly use webGL2 uniform buffers. For now this will suffice for webGL 1 compat
            let pass_uniforms = shader.pass_uniforms;
            // if vr_presenting

            let view_uniforms = shader.view_uniforms;
            for (let i = 0; i < view_uniforms.length; i ++) {
                let uni = view_uniforms[i];
                uni.fn(this, uni.loc, uni.offset + view_uniforms_ptr);
            }
            let draw_uniforms = shader.draw_uniforms;
            for (let i = 0; i < draw_uniforms.length; i ++) {
                let uni = draw_uniforms[i];
                uni.fn(this, uni.loc, uni.offset + draw_uniforms_ptr);
            }
            let user_uniforms = shader.user_uniforms;
            for (let i = 0; i < user_uniforms.length; i ++) {
                let uni = user_uniforms[i];
                uni.fn(this, uni.loc, uni.offset + user_uniforms_ptr);
            }
            let texture_slots = shader.texture_slots;
            for (let i = 0; i < texture_slots.length; i ++) {
                let tex_slot = texture_slots[i];
                let tex_id = this.baseu32[(textures_ptr >> 2) + i];
                let tex_obj = this.textures[tex_id];
                gl.activeTexture(gl.TEXTURE0 + i);
                gl.bindTexture(gl.TEXTURE_2D, tex_obj);
                gl.uniform1i(tex_slot.loc, i);
            }
            let indices = index_buffer.length;
            let instances = instance_buffer.length / shader.instance_slots;
            // lets do a drawcall!

            if (this.is_main_canvas && this.xr_is_presenting) {
                // for (let i = 3; i < pass_uniforms.length; i ++) {
                //     let uni = pass_uniforms[i];
                //     uni.fn(this, uni.loc, uni.offset + pass_uniforms_ptr);
                // }

                // // the first 2 matrices are project and view
                // let left_viewport = this.xr_left_viewport;
                // gl.viewport(left_viewport.x, left_viewport.y, left_viewport.width, left_viewport.height);
                // gl.uniformMatrix4fv(pass_uniforms[0].loc, false, this.xr_left_projection_matrix);
                // gl.uniformMatrix4fv(pass_uniforms[1].loc, false, this.xr_left_transform_matrix);
                // gl.uniformMatrix4fv(pass_uniforms[2].loc, false, this.xr_left_invtransform_matrix);

                // this.ANGLE_instanced_arrays.drawElementsInstancedANGLE(gl.TRIANGLES, indices, gl.UNSIGNED_INT, 0, instances);
                // let right_viewport = this.xr_right_viewport;
                // gl.viewport(right_viewport.x, right_viewport.y, right_viewport.width, right_viewport.height);

                // gl.uniformMatrix4fv(pass_uniforms[0].loc, false, this.xr_right_projection_matrix);
                // gl.uniformMatrix4fv(pass_uniforms[1].loc, false, this.xr_right_transform_matrix);
                // gl.uniformMatrix4fv(pass_uniforms[2].loc, false, this.xr_right_invtransform_matrix);

                // this.ANGLE_instanced_arrays.drawElementsInstancedANGLE(gl.TRIANGLES, indices, gl.UNSIGNED_INT, 0, instances);
            }
            else {
                for (let i = 0; i < pass_uniforms.length; i ++) {
                    let uni = pass_uniforms[i];
                    uni.fn(this, uni.loc, uni.offset + pass_uniforms_ptr);
                }
                this.ANGLE_instanced_arrays.drawElementsInstancedANGLE(gl.TRIANGLES, indices, gl.UNSIGNED_INT, 0, instances);
            }
            this.OES_vertex_array_object.bindVertexArrayOES(null);
        }

        fetch_path(file_path) {
            return new Promise((resolve, reject) => {
                var req = new XMLHttpRequest()
                req.addEventListener("error", function() {
                    reject(resource)
                })
                req.responseType = 'arraybuffer'
                req.addEventListener("load", function() {
                    if (req.status !== 200) {
                        return reject(req.status)
                    }
                    resolve({
                        name: file_path,
                        buffer: req.response
                    })
                })
                req.open("GET", new URL(file_path, this.base_uri).href)
                req.send()
            })
        }

        read_user_file_range(user_file_id, buf_ptr, buf_len, file_offset) {
            const file = this.file_handles[user_file_id];
            const start = Number(file_offset);
            const end = start + Number(buf_len);
            if (file.last_read_start <= start && start < file.last_read_end) {
                console.warn(`Read start (${start}) fell in the range of the last read (${file.last_read_start}-${file.last_read_end}); ` +
                    "this usually happens if you don't use BufReader or if you don't use BufReader.seek_relative.");
            }
            file.last_read_start = start;
            file.last_read_end = end;
            // TODO(JP): This creates a new buffer instead of reading directly into the wasm memory.
            // Maybe we can avoid this by using a stream with a ReadableStreamBYOBReader, but that is
            // asynchronous, so we'd have to do a dance with another thread and atomics and all that,
            // and I don't know if that overhead would be worth it..
            const buffer = file_reader_sync.readAsArrayBuffer(file.file.slice(start, end));
            this.to_wasm.copy_to_wasm(buffer, Number(buf_ptr));
            return BigInt(buffer.byteLength);
        }
    }

    // array of function id's wasm can call on us, self is pointer to WasmApp
    WasmApp.prototype.send_fn_table = [
        function end_0(self) {
            return true;
        },
        function log_1(self) {
            console.log(self.parse_string());
        },
        function compile_webgl_shader_2(self) {
            let ash = {
                shader_id: self.mu32[self.parse ++],
                fragment: self.parse_string(),
                vertex: self.parse_string(),
                geometry_slots: self.mu32[self.parse ++],
                instance_slots: self.mu32[self.parse ++],
                pass_uniforms: self.parse_shvarvec(),
                view_uniforms: self.parse_shvarvec(),
                draw_uniforms: self.parse_shvarvec(),
                user_uniforms: self.parse_shvarvec(),
                texture_slots: self.parse_shvarvec()
            }
            self.compile_webgl_shader(ash);
        },
        function alloc_array_buffer_3(self) {
            let array_buffer_id = self.mu32[self.parse ++];
            let len = self.mu32[self.parse ++];
            let pointer = self.mu32[self.parse ++];
            let array = new Float32Array(self.memory.buffer, pointer, len);
            self.alloc_array_buffer(array_buffer_id, array);
        },
        function alloc_index_buffer_4(self) {
            let index_buffer_id = self.mu32[self.parse ++];
            let len = self.mu32[self.parse ++];
            let pointer = self.mu32[self.parse ++];
            let array = new Uint32Array(self.memory.buffer, pointer, len);
            self.alloc_index_buffer(index_buffer_id, array);
        },
        function alloc_vao_5(self) {
            let vao_id = self.mu32[self.parse ++];
            let shader_id = self.mu32[self.parse ++];
            let geom_ib_id = self.mu32[self.parse ++];
            let geom_vb_id = self.mu32[self.parse ++];
            let inst_vb_id = self.mu32[self.parse ++];
            self.alloc_vao(vao_id, shader_id, geom_ib_id, geom_vb_id, inst_vb_id)
        },
        function draw_call_6(self) {
            let shader_id = self.mu32[self.parse ++];
            let vao_id = self.mu32[self.parse ++];
            let uniforms_pass_ptr = self.mu32[self.parse ++];
            let uniforms_view_ptr = self.mu32[self.parse ++];
            let uniforms_draw_ptr = self.mu32[self.parse ++];
            let uniforms_user_ptr = self.mu32[self.parse ++];
            let textures = self.mu32[self.parse ++];
            self.draw_call(
                shader_id,
                vao_id,
                uniforms_pass_ptr,
                uniforms_view_ptr,
                uniforms_draw_ptr,
                uniforms_user_ptr,
                textures
            );
        },
        function clear_7(self) {
            let r = self.mf32[self.parse ++];
            let g = self.mf32[self.parse ++];
            let b = self.mf32[self.parse ++];
            let a = self.mf32[self.parse ++];
            self.clear(r, g, b, a);
        },
        function load_deps_8(self) {
            let deps = []
            let num_deps = self.mu32[self.parse ++];
            for (let i = 0; i < num_deps; i ++) {
                deps.push(self.parse_string());
            }
            self.load_deps(deps);
        },
        function alloc_texture_9(self) {
            let texture_id = self.mu32[self.parse ++];
            let width = self.mu32[self.parse ++];
            let height = self.mu32[self.parse ++];
            let data_ptr = self.mu32[self.parse ++];
            self.alloc_texture(texture_id, width, height, data_ptr);
        },
        function request_animation_frame_10(self) {
            self.request_animation_frame()
        },
        function set_document_title_11(self) {
            self.set_document_title(self.parse_string())
        },
        function set_mouse_cursor_12(self) {
            self.set_mouse_cursor(self.mu32[self.parse ++]);
        },
        function read_file_13(self) {
            self.read_file(self.mu32[self.parse ++], self.parse_string());
        },
        function show_text_ime_14(self) {
            const x = self.mf32[self.parse ++];
            const y = self.mf32[self.parse ++];
            rpc.send("show_text_ime", { x, y });
        },
        function hide_text_ime_15(self) {
            // TODO(JP): doesn't seem to do anything, is that intentional?
        },
        function text_copy_response_16(self) {
            const text_copy_response = self.parse_string();
            rpc.send("text_copy_response", { text_copy_response });

        },
        function start_timer_17(self) {
            var repeats = self.mu32[self.parse ++]
            var id = self.parse_f64();
            var interval = self.parse_f64();
            self.start_timer(id, interval, repeats);
        },
        function stop_timer_18(self) {
            var id = self.parse_f64();
            self.stop_timer(id);
        },
        function xr_start_presenting_19(self) {
            self.xr_start_presenting();
        },
        function xr_stop_presenting_20(self) {
            self.xr_stop_presenting();
        },
        function begin_render_targets_21(self) {
            let pass_id = self.mu32[self.parse ++];
            let width = self.mu32[self.parse ++];
            let height = self.mu32[self.parse ++];
            self.begin_render_targets(pass_id, width, height);
        },
        function add_color_target_22(self) {
            let texture_id = self.mu32[self.parse ++];
            let init_only = self.mu32[self.parse ++];
            let r = self.mf32[self.parse ++];
            let g = self.mf32[self.parse ++];
            let b = self.mf32[self.parse ++];
            let a = self.mf32[self.parse ++];
            self.add_color_target(texture_id, init_only, r, g, b, a)
        },
        function set_depth_target_23(self) {
            let texture_id = self.mu32[self.parse ++];
            let init_only = self.mu32[self.parse ++];
            let depth = self.mf32[self.parse ++];
            self.set_depth_target(texture_id, init_only, depth);
        },
        function end_render_targets_24(self) {
            self.end_render_targets();
        },
        function set_default_depth_and_blend_mode_25(self) {
            self.set_default_depth_and_blend_mode();
        },
        function begin_main_canvas_26(self) {
            let r = self.mf32[self.parse ++];
            let g = self.mf32[self.parse ++];
            let b = self.mf32[self.parse ++];
            let a = self.mf32[self.parse ++];
            let depth = self.mf32[self.parse ++];
            self.begin_main_canvas(r, g, b, a, depth);
        },
        function http_send_27(self) {
            let port = self.mu32[self.parse ++];
            let signal_id = self.mu32[self.parse ++];
            let verb = self.parse_string();
            let path = self.parse_string();
            let proto = self.parse_string();
            let domain = self.parse_string();
            let content_type = self.parse_string();
            let body = self.parse_u8slice();
            // do XHR.
            self.http_send(verb, path, proto, domain, port, content_type, body, signal_id);
        },
        function fullscreen_28(self) {
            rpc.send("fullscreen", {});
        },
        function normalscreen_29(self) {
            rpc.send("normalscreen", {});
        },
        function websocket_send_30(self) {
            let url = self.parse_string();
            let data = self.parse_u8slice();
            self.websocket_send(url, data);
        },
        function enable_global_file_drop_target_31(self) {
            self.enable_global_file_drop_target()
        },
    ]

    WasmApp.prototype.uniform_fn_table = {
        "float": function set_float(self, loc, off) {
            let slot = off >> 2;
            self.gl.uniform1f(loc, self.basef32[slot])
        },
        "vec2": function set_vec2(self, loc, off) {
            let slot = off >> 2;
            let basef32 = self.basef32;
            self.gl.uniform2f(loc, basef32[slot], basef32[slot + 1])
        },
        "vec3": function set_vec3(self, loc, off) {
            let slot = off >> 2;
            let basef32 = self.basef32;
            self.gl.uniform3f(loc, basef32[slot], basef32[slot + 1], basef32[slot + 2])
        },
        "vec4": function set_vec4(self, loc, off) {
            let slot = off >> 2;
            let basef32 = self.basef32;
            self.gl.uniform4f(loc, basef32[slot], basef32[slot + 1], basef32[slot + 2], basef32[slot + 3])
        },
        "mat2": function set_mat2(self, loc, off) {
            self.gl.uniformMatrix2fv(loc, false, new Float32Array(self.memory.buffer, off, 4))
        },
        "mat3": function set_mat3(self, loc, off) {
            self.gl.uniformMatrix3fv(loc, false, new Float32Array(self.memory.buffer, off, 9))
        },
        "mat4": function set_mat4(self, loc, off) {
            let mat4 = new Float32Array(self.memory.buffer, off, 16);
            self.gl.uniformMatrix4fv(loc, false, mat4)
        },
    };

    WasmApp.prototype.uniform_size_table = {
        "float": 1,
        "vec2": 2,
        "vec3": 3,
        "vec4": 4,
        "mat2": 4,
        "mat3": 9,
        "mat4": 16
    }

    function add_line_numbers_to_string(code) {
        var lines = code.split('\n')
        var out = ''
        for (let i = 0; i < lines.length; i ++) {
            out += (i + 1) + ': ' + lines[i] + '\n'
        }
        return out
    }

    //var firefox_logo_key = false;
    function pack_key_modifier(e) {
        return (e.shiftKey? 1: 0) | (e.ctrlKey? 2: 0) | (e.altKey? 4: 0) | (e.metaKey? 8: 0)
    }

    rpc.receive("init", ({ offscreenCanvas, wasmFilename, can_fullscreen, base_uri }) => {
        // Initial has to be equal to or higher than required by the app (which at the time of writing
        // is around 20 pages).
        // Maximum has to be equal to or lower than that of the app.
        // TODO(JP): Set this to the browser's maximum of 65536 here and in the Rust code,
        // and make sure that that doesn't have a negative performance effect (should just
        // be a virtual allocation).
        const memory = new WebAssembly.Memory({initial: 40, maximum: 16384, shared: true});

        function _console_log(chars_ptr, len) {
            let out = "";
            let array = new Uint32Array(memory.buffer, chars_ptr, len);
            for (let i = 0; i < len; i ++) {
                out += String.fromCharCode(array[i]);
            }
            console.log(out);
        }

        const wasmPath = new URL(wasmFilename, base_uri).href;

        let wasmapp;
        return new Promise((resolve, reject) => {
            const env = {
                _console_log,
                memory,
                read_user_file_range: (user_file_id, buf_ptr, buf_len, file_offset) => {
                    return wasmapp.read_user_file_range(user_file_id, buf_ptr, buf_len, file_offset);
                },
                performance_now: () => {
                    return performance.now();
                },
                thread_spawn: (ctx_ptr) => {
                    const worker = new Worker(new URL('zaplib/main/src/cx_async_worker.js', base_uri).href);
                    const workerRpc = new Rpc(worker);
                    workerRpc.send("run", { wasmPath, memory, ctx_ptr }).finally(() => {
                        worker.terminate();
                    });
                },
            };

            WebAssembly.instantiateStreaming(fetch(wasmPath), { env }).then(results => {
                wasmapp = new WasmApp(offscreenCanvas, results, memory, can_fullscreen, base_uri, wasmPath);
                resolve();
            }, reject);
        });
    });
})();
