// Taken from https://github.com/cruise-automation/webviz/blob/6a4226bc2959444704d650d8c55cea4f4220c75c/packages/webviz-core/src/util/Rpc.js
// TODO(JP): Maybe convert to Typescript and release as a package?
// TODO(JP): Also be sure to include the tests at some point: https://github.com/cruise-automation/webviz/blob/6a4226bc2959444704d650d8c55cea4f4220c75c/packages/webviz-core/src/util/Rpc.test.js

// this type mirrors the MessageChannel and MessagePort APIs which are available on
// instances of web-workers and shared-workers respectively, as well as avaiable on
// 'global' within them.
// export interface Channel {
//   postMessage(data: any, transfer?: any[]): void;
//   onmessage: null | ((ev: MessageEvent) => mixed);
// }

const RESPONSE = "$$RESPONSE";
const ERROR = "$$ERROR";

// helper function to create linked channels for testing
function createLinkedChannels() {
  const local = {
    onmessage,

    postMessage(data, _transfer) {
      const ev = new MessageEvent("message", { data });
      if (remote.onmessage) {
        remote.onmessage(ev);
      }
    },
    terminate: () => {},
  };

  const remote = {
    onmessage,

    postMessage(data, _transfer) {
      const ev = new MessageEvent("message", { data });
      if (local.onmessage) {
        local.onmessage(ev);
      }
    },
    terminate: () => {},
  };
  return { local, remote };
}

// This class allows you to hook up bi-directional async calls across web-worker
// boundaries where a single call to or from a worker can 'wait' on the response.
// Errors in receivers are propigated back to the caller as a rejection.
// It also supports returning transferrables over the web-worker postMessage api,
// which was the main shortcomming with the worker-rpc npm module.
// To attach rpc to an instance of a worker in the main thread:
//   const rpc = new Rpc(workerInstace);
// To attach rpc within an a web worker:
//   const rpc = new Rpc(global);
// Check out the tests for more examples.
class Rpc {
  static transferrables = "$$TRANSFERRABLES";
  _channel = undefined;
  _messageId = 0;
  _pendingCallbacks = {};
  _receivers = new Map();

  constructor(channel) {
    this._channel = channel;
    if (this._channel.onmessage) {
      throw new Error("channel.onmessage is already set. Can only use one Rpc instance per channel.");
    }
    this._channel.onmessage = this._onChannelMessage;
  }

  _onChannelMessage = (ev) => {
    const { id, topic, data } = (ev.data);
    if (topic === RESPONSE) {
      this._pendingCallbacks[id](ev.data);
      delete this._pendingCallbacks[id];
      return;
    }
    // invoke the receive handler in a promise so if it throws synchronously we can reject
    new Promise((resolve) => {
      const handler = this._receivers.get(topic);
      if (!handler) {
        throw new Error(`no receiver registered for ${topic}`);
      }
      // This works both when `handler` returns a value or a Promise.
      resolve(handler(data));
    })
      .then((result) => {
        if (!result) {
          return this._channel.postMessage({ topic: RESPONSE, id });
        }
        const transferrables = result[Rpc.transferrables];
        delete result[Rpc.transferrables];
        const message = {
          topic: RESPONSE,
          id,
          data: result,
        };
        this._channel.postMessage(message, transferrables);
      })
      .catch((err) => {
        const message = {
          topic: RESPONSE,
          id,
          data: {
            [ERROR]: true,
            name: err.name,
            message: err.message,
            stack: err.stack,
          },
        };
        this._channel.postMessage(message);
      });
  };

  // send a message across the rpc boundary to a receiver on the other side
  // this returns a promise for the receiver's response.  If there is no registered
  // receiver for the given topic, this method throws
  send(topic, data, transfer) {
    const id = this._messageId++;
    const message = { topic, id, data };
    const result = new Promise((resolve, reject) => {
      this._pendingCallbacks[id] = (info) => {
        if (info.data && info.data[ERROR]) {
          const error = new Error(info.data.message);
          error.name = info.data.name;
          error.stack = info.data.stack;
          reject(error);
        } else {
          resolve(info.data);
        }
      };
    });
    this._channel.postMessage(message, transfer);
    return result;
  }

  // register a receiver for a given message on a topic
  // only one receiver can be registered per topic and currently
  // 'deregistering' a receiver is not supported since this is not common
  receive(topic, handler) {
    if (this._receivers.has(topic)) {
      throw new Error(`Receiver already registered for topic: ${topic}`);
    }
    this._receivers.set(topic, handler);
  }
}
