(function() {
    const scriptPath = document.currentScript.src.replace(/\/[^/]+$/, '/');

    document.addEventListener('DOMContentLoaded', () => {
        var is_mobile_safari = self.navigator.platform.match(/iPhone|iPad/i);
        var is_android = self.navigator.userAgent.match(/Android/i);

        const worker = new Worker(scriptPath + 'cx_webgl_worker.js');
        let rpcInitialized = false;
        const rpc = new Rpc(worker);

        rpc.receive("show_incompatible_browser_notification", () => {
            const span = document.createElement('span');
            span.style.color = 'white';
            canvas.parentNode.replaceChild(span, canvas);
            span.innerHTML = "Sorry, we need browser support for WebGL to run<br/>Please update your browser to a more modern one<br/>Update to at least iOS 10, Safari 10, latest Chrome, Edge or Firefox<br/>Go and update and come back, your browser will be better, faster and more secure!<br/>If you are using chrome on OSX on a 2011/2012 mac please enable your GPU at: Override software rendering list:Enable (the top item) in: <a href='about://flags'>about://flags</a>. Or switch to Firefox or Safari.";
        });

        rpc.receive("remove_loading_indicators", () => {
            var loaders = document.getElementsByClassName('cx_webgl_loader');
            for (var i = 0; i < loaders.length; i ++) {
                loaders[i].parentNode.removeChild(loaders[i])
            }
        });

        rpc.receive("set_document_title", ({ title }) => {
            document.title = title;
        });

        rpc.receive("set_mouse_cursor", ({ style }) => {
            document.body.style.cursor = style;
        });

        rpc.receive("fullscreen", ({ style }) => {
            if (document.body.requestFullscreen) {
                document.body.requestFullscreen();
            } else if (document.body.webkitRequestFullscreen) {
                document.body.webkitRequestFullscreen();
            } else if (document.body.mozRequestFullscreen) {
                document.body.mozRequestFullscreen();
            }
        });

        rpc.receive("normalscreen", ({ style }) => {
            if (document.exitFullscreen) {
                document.exitFullscreen();
            } else if (document.webkitExitFullscreen) {
                document.webkitExitFullscreen();
            } else if (document.mozExitFullscreen) {
                document.mozExitFullscreen();
            }
        });

        rpc.receive("text_copy_response", ({ text_copy_response }) => {
            navigator.clipboard.writeText(text_copy_response);
        });

        rpc.receive("enable_global_file_drop_target", () => {
            document.addEventListener("dragover", (ev) => {
                var dataTransfer = ev.dataTransfer;
                // dataTransfer isn't guaranteed to exist by spec, so it must be checked
                if (dataTransfer && dataTransfer.types.length === 1 && dataTransfer.types[0] === "Files") {
                  ev.stopPropagation();
                  ev.preventDefault();
                  dataTransfer.dropEffect = "copy";
                }
            });
            document.addEventListener("drop", (ev) => {
                if (!ev.dataTransfer) {
                  return;
                }
                const files = Array.from(ev.dataTransfer.files);
                if (!files.length) {
                  return;
                }
                ev.preventDefault();
                ev.stopPropagation();
                if (rpcInitialized) rpc.send("drop", { files });
            });
        });

        function make_rpc_event(event) {
            return {
                button: event.button,
                pageX: event.pageX,
                pageY: event.pageY,
                timeStamp: event.timeStamp,
                shiftKey: event.shiftKey,
                ctrlKey: event.ctrlKey,
                altKey: event.altKey,
                metaKey: event.metaKey,
                deltaMode: event.deltaMode,
                deltaX: event.deltaX,
                deltaY: event.deltaY,
                wheelDeltaY: event.wheelDeltaY,
                keyCode: event.keyCode,
                charCode: event.charCode,
                repeat: event.repeat,
            };
        }

        const canvas = document.querySelector('canvas');
        const wasmFilename = canvas.getAttribute("wasm");

        let ta;

        canvas.addEventListener('contextmenu', (event) => {
            event.preventDefault()
            return false;
        });

        const mouse_down_handler = (event) => {
            if (ta) ta.focus();
            event.preventDefault();
            if (rpcInitialized) rpc.send('canvas_mousedown', { event: make_rpc_event(event) });
        };
        canvas.addEventListener('mousedown', mouse_down_handler);
        const mouse_up_handler = (event) => {
            event.preventDefault();
            if (rpcInitialized) rpc.send('window_mouseup', { event: make_rpc_event(event) });
        };
        window.addEventListener('mouseup', mouse_up_handler);
        window.addEventListener('mousemove', (event) => {
            document.body.scrollTop = 0;
            document.body.scrollLeft = 0;
            if (rpcInitialized) rpc.send('window_mousemove', { event: make_rpc_event(event) });
        });
        window.addEventListener('mouseout', (event) => {
            if (rpcInitialized) rpc.send('window_mouseout', { event: make_rpc_event(event) });
        });
        const mouse_wheel_handler = (event) => {
            event.preventDefault();
            if (rpcInitialized) rpc.send('canvas_wheel', { event: make_rpc_event(event), offsetHeight: window.offsetHeight });
        };
        canvas.addEventListener('wheel', mouse_wheel_handler);
        window.addEventListener('focus', () => {
            if (rpcInitialized) rpc.send('window_focus', {});
        });
        window.addEventListener('blur', () => {
            if (rpcInitialized) rpc.send('window_blur', {});
        });

        if (!is_mobile_safari && !is_android) { // mobile keyboards are unusable on a UI like this
            let text_area_pos;
            function update_text_area_pos() {
                if (!text_area_pos) {
                    ta.style.left = -100 + 'px';
                    ta.style.top = -100 + 'px';
                } else {
                    ta.style.left = (Math.round(text_area_pos.x) - 4) + "px";
                    ta.style.top = Math.round(text_area_pos.y) + "px";
                }
            }
            rpc.receive("show_text_ime", ({ x, y }) => {
                text_area_pos = { x, y };
                update_text_area_pos();
            })

            let was_paste = false;
            let last_len = 0;
            let ugly_ime_hack = false;

            let recreate_textarea = function() {
                if (ta) document.body.removeChild(ta);

                ta = document.createElement('textarea');
                ta.className = "cx_webgl_textinput";
                ta.setAttribute('autocomplete', 'off');
                ta.setAttribute('autocorrect', 'off');
                ta.setAttribute('autocapitalize', 'off');
                ta.setAttribute('spellcheck', 'false');
                var style = document.createElement('style');
                style.innerHTML = "\n"
                    + "textarea.cx_webgl_textinput {\n"
                    + "z-index: 1000;\n"
                    + "position: absolute;\n"
                    + "opacity: 0;\n"
                    + "border-radius: 4px;\n"
                    + "color:white;\n"
                    + "font-size: 6;\n"
                    + "background: gray;\n"
                    + "-moz-appearance: none;\n"
                    + "appearance:none;\n"
                    + "border:none;\n"
                    + "resize: none;\n"
                    + "outline: none;\n"
                    + "overflow: hidden;\n"
                    + "text-indent: 0px;\n"
                    + "padding: 0 0px;\n"
                    + "margin: 0 -1px;\n"
                    + "text-indent: 0px;\n"
                    + "-ms-user-select: text;\n"
                    + "-moz-user-select: text;\n"
                    + "-webkit-user-select: text;\n"
                    + "user-select: text;\n"
                    + "white-space: pre!important;\n"
                    + "}\n"
                    + "textarea: focus.cx_webgl_textinput {\n"
                    + "outline: 0px !important;\n"
                    + "-webkit-appearance: none;\n"
                    + "}";
                document.body.appendChild(style);
                ta.style.left = -100 + 'px';
                ta.style.top = -100 + 'px';
                ta.style.height = 1;
                ta.style.width = 1;

                ta.addEventListener('mousedown', mouse_down_handler);
                ta.addEventListener('mouseup', mouse_up_handler);
                ta.addEventListener('wheel', mouse_wheel_handler);
                ta.addEventListener('contextmenu', (event) => {
                    event.preventDefault();
                    return false;
                });
                ta.addEventListener('blur', () => {
                    ta.focus();
                });
                document.body.appendChild(ta);
                ta.focus();
                update_text_area_pos();

                ta.addEventListener('cut', () => {
                    setTimeout(() => {
                        ta.value = "";
                        last_len = 0;
                    });
                });
                ta.addEventListener('copy', () => {
                    setTimeout(() => {
                        ta.value = "";
                        last_len = 0;
                    });
                });
                ta.addEventListener('paste', () => {
                    was_paste = true;
                });

                ta.addEventListener('input', () => {
                    if (ta.value.length > 0) {
                        if (was_paste) {
                            was_paste = false;
                            const input = ta.value.substring(last_len);
                            ta.value = "";
                            if (rpcInitialized) rpc.send("text_input", { was_paste: true, input, replace_last: false });
                        }
                        else {
                            var replace_last = false;
                            var text_value = ta.value;
                            if (ta.value.length >= 2) { // we want the second char
                                text_value = ta.value.substring(1, 2);
                                ta.value = text_value;
                            }
                            else if (ta.value.length == 1 && last_len == ta.value.length) { // its an IME replace
                                replace_last = true;
                            }
                            // we should send a replace last
                            if (replace_last || text_value != '\n') {
                                if (rpcInitialized) rpc.send("text_input", { was_paste: false, input: text_value, replace_last: replace_last });
                            }
                        }
                    }
                    last_len = ta.value.length;
                });

                ta.addEventListener('keydown', (event) => {
                    const code = event.keyCode;

                    if (code == 18 || code == 17 || code == 16) event.preventDefault(); // alt
                    if (code === 8 || code === 9) event.preventDefault(); // backspace/tab
                    if (code === 89 && (event.metaKey || event.ctrlKey)) event.preventDefault(); // all (select all)
                    if (code === 83 && (event.metaKey || event.ctrlKey)) event.preventDefault(); // ctrl s
                    if (code >= 33 && code <= 40) { // if we are using arrow keys, home or end
                        ta.value = "";
                        last_len = ta.value.length;
                    }
                    if ((code === 88 || code == 67) && (event.metaKey || event.ctrlKey)) { // copy or cut
                        // we need to request the clipboard
                        if (rpcInitialized) rpc.send("text_copy", {})
                        event.preventDefault();
                    }
                    if (code === 90 && (event.metaKey || event.ctrlKey)) { // ctrl/cmd + z
                        update_text_area_pos();
                        ta.value = "";
                        ugly_ime_hack = true;
                        ta.readOnly = true;
                        event.preventDefault()
                    }
                    if (rpcInitialized) rpc.send("key_down", { event: make_rpc_event(event) });
                });

                ta.addEventListener('keyup', (event) => {
                    const code = event.keyCode;
                    if (code == 18 || code == 17 || code == 16) event.preventDefault(); // alt
                    if (code == 91) event.preventDefault(); // left window key
                    if (ugly_ime_hack) {
                        ugly_ime_hack = false;
                        recreate_textarea();
                    }
                    if (rpcInitialized) rpc.send("key_up", { event: make_rpc_event(event) });
                });
            }
            recreate_textarea();
        }

        function on_screen_resize() {
            var dpi_factor = window.devicePixelRatio;
            var w, h;

            if (this.xr_is_presenting) {
                // let xr_webgllayer = this.xr_session.renderState.baseLayer;
                // this.dpi_factor = 3.0;
                // this.width = 2560.0 / this.dpi_factor;
                // this.height = 2000.0 / this.dpi_factor;
            }
            else {
                if (canvas.getAttribute("fullpage")) {
                    // TODO(JP): Some day bring this back?
                    // if (is_add_to_homescreen_safari) { // extremely ugly. but whatever.
                    //     if (window.orientation == 90 || window.orientation == -90) {
                    //         h = screen.width;
                    //         w = screen.height - 90;
                    //     }
                    //     else {
                    //         w = screen.width;
                    //         h = screen.height - 80;
                    //     }
                    // }
                    // else {
                    w = window.innerWidth;
                    h = window.innerHeight;
                    // }
                }
                else {
                    w = canvas.offsetWidth;
                    h = canvas.offsetHeight;
                }

                const is_fullscreen = !!(document.fullscreenElement || document.webkitFullscreenElement || document.mozFullscreenElement);
                if (rpcInitialized) rpc.send("on_screen_resize", { width: w, height: h, dpi_factor, is_fullscreen})
            }
        }
        window.addEventListener('resize', () => on_screen_resize());
        window.addEventListener('orientationchange', () => on_screen_resize());

        let dpi_factor = window.devicePixelRatio;
        let mqString = '(resolution: ' + window.devicePixelRatio + 'dppx)'
        let mq = matchMedia(mqString);
        if (mq && mq.addEventListener) {
            mq.addEventListener('change', () => on_screen_resize());
        } else { // poll for it. yes. its terrible
            self.setInterval(_ => {
                if (window.devicePixelRatio != dpi_factor) {
                    dpi_factor = window.devicePixelRatio;
                    on_screen_resize();
                }
            }, 1000);
        }

        var offscreenCanvas = canvas.transferControlToOffscreen();
        const can_fullscreen = !!(document.fullscreenEnabled || document.webkitFullscreenEnabled || document.mozFullscreenEnabled);
        const base_uri = document.baseURI;
        rpc.send("init", { offscreenCanvas, wasmFilename, can_fullscreen, base_uri }, [offscreenCanvas]).then(() => {
            rpcInitialized = true;
            on_screen_resize();
        });
    });
})();
