//! Mac OS X platform-specific entry point.

use crate::cx::*;
use crate::cx_cocoa::*;

impl Cx {
    pub fn event_loop<F>(&mut self, mut event_handler: F)
    where
        F: FnMut(&mut Cx, &mut Event),
    {
        self.event_handler =
            Some(&mut event_handler as *const dyn FnMut(&mut Cx, &mut Event) as *mut dyn FnMut(&mut Cx, &mut Event));
        self.event_loop_core();
        self.event_handler = None;
    }

    fn event_loop_core(&mut self) {
        self.platform_type = PlatformType::OSX;

        let mut cocoa_app = CocoaApp::new();

        cocoa_app.init();

        let mut metal_cx = MetalCx::new();

        let mut metal_windows: Vec<MetalWindow> = Vec::new();

        self.mtl_compile_shaders(&metal_cx);

        self.desktop_load_fonts();

        self.call_event_handler(&mut Event::Construct);

        self.redraw_child_area(Area::All);

        let mut passes_todo = Vec::new();

        cocoa_app.event_loop(|cocoa_app, events| {
            //let mut paint_dirty = false;
            self.last_event_time = cocoa_app.time_now();

            for mut event in events {
                self.process_desktop_pre_event(&mut event);

                match &event {
                    Event::WindowGeomChange(re) => {
                        // do this here because mac
                        for metal_window in &mut metal_windows {
                            if metal_window.window_id == re.window_id {
                                metal_window.window_geom = re.new_geom.clone();
                                self.windows[re.window_id].window_geom = re.new_geom.clone();
                                if re.old_geom.inner_size != re.new_geom.inner_size {
                                    self.redraw_child_area(Area::All);
                                }
                                break;
                            }
                        }
                        // ok lets not redraw all, just this window
                        self.call_event_handler(&mut event);
                    }
                    Event::WindowClosed(wc) => {
                        // lets remove the window from the set
                        self.windows[wc.window_id].window_state = CxWindowState::Closed;
                        self.windows_free.push(wc.window_id);
                        // remove the d3d11/win32 window

                        for index in 0..metal_windows.len() {
                            if metal_windows[index].window_id == wc.window_id {
                                metal_windows.remove(index);
                                if metal_windows.is_empty() {
                                    cocoa_app.terminate_event_loop();
                                }
                                for metal_window in &mut metal_windows {
                                    metal_window.cocoa_window.update_ptrs();
                                }
                            }
                        }
                        self.call_event_handler(&mut event);
                    }
                    Event::Paint => {
                        let vsync = self.process_desktop_paint_callbacks();

                        // construct or destruct windows
                        for (index, window) in self.windows.iter_mut().enumerate() {
                            window.window_state = match &window.window_state {
                                CxWindowState::Create { inner_size, position, title, add_drop_target_for_app_open_files } => {
                                    // lets create a platformwindow
                                    let metal_window = MetalWindow::new(
                                        index,
                                        &metal_cx,
                                        cocoa_app,
                                        *inner_size,
                                        *position,
                                        title,
                                        *add_drop_target_for_app_open_files,
                                    );
                                    window.window_geom = metal_window.window_geom.clone();
                                    metal_windows.push(metal_window);
                                    for metal_window in &mut metal_windows {
                                        metal_window.cocoa_window.update_ptrs();
                                    }
                                    CxWindowState::Created
                                }
                                CxWindowState::Close => {
                                    for metal_window in &mut metal_windows {
                                        if metal_window.window_id == index {
                                            metal_window.cocoa_window.close_window();
                                            break;
                                        }
                                    }
                                    CxWindowState::Closed
                                }
                                CxWindowState::Created => CxWindowState::Created,
                                CxWindowState::Closed => CxWindowState::Closed,
                            };

                            window.window_command = match &window.window_command {
                                CxWindowCmd::Restore => {
                                    for metal_window in &mut metal_windows {
                                        if metal_window.window_id == index {
                                            metal_window.cocoa_window.restore();
                                        }
                                    }
                                    CxWindowCmd::None
                                }
                                CxWindowCmd::Maximize => {
                                    for metal_window in &mut metal_windows {
                                        if metal_window.window_id == index {
                                            metal_window.cocoa_window.maximize();
                                        }
                                    }
                                    CxWindowCmd::None
                                }
                                CxWindowCmd::Minimize => {
                                    for metal_window in &mut metal_windows {
                                        if metal_window.window_id == index {
                                            metal_window.cocoa_window.minimize();
                                        }
                                    }
                                    CxWindowCmd::None
                                }
                                _ => CxWindowCmd::None,
                            };

                            if let Some(topmost) = window.window_topmost {
                                for metal_window in &mut metal_windows {
                                    if metal_window.window_id == index {
                                        metal_window.cocoa_window.set_topmost(topmost);
                                    }
                                }
                            }
                        }

                        // set a cursor
                        if self.down_mouse_cursor.is_some() {
                            cocoa_app.set_mouse_cursor(self.down_mouse_cursor.as_ref().unwrap().clone())
                        } else if self.hover_mouse_cursor.is_some() {
                            cocoa_app.set_mouse_cursor(self.hover_mouse_cursor.as_ref().unwrap().clone())
                        } else {
                            cocoa_app.set_mouse_cursor(MouseCursor::Default)
                        }

                        if let Some(set_ime_position) = self.platform.set_ime_position {
                            self.platform.set_ime_position = None;
                            for metal_window in &mut metal_windows {
                                metal_window.cocoa_window.set_ime_spot(set_ime_position);
                            }
                        }

                        while !self.platform.start_timer.is_empty() {
                            let (timer_id, interval, repeats) = self.platform.start_timer.pop().unwrap();
                            cocoa_app.start_timer(timer_id, interval, repeats);
                        }

                        while !self.platform.stop_timer.is_empty() {
                            let timer_id = self.platform.stop_timer.pop().unwrap();
                            cocoa_app.stop_timer(timer_id);
                        }

                        if self.platform.set_menu {
                            self.platform.set_menu = false;
                            if let Some(menu) = &self.platform.last_menu {
                                cocoa_app.update_app_menu(menu, &self.command_settings)
                            }
                        }

                        // build a list of renderpasses to repaint
                        let mut windows_need_repaint = 0;
                        self.compute_passes_to_repaint(&mut passes_todo, &mut windows_need_repaint);

                        if !passes_todo.is_empty() {
                            for pass_id in &passes_todo {
                                match self.passes[*pass_id].dep_of.clone() {
                                    CxPassDepOf::Window(window_id) => {
                                        // find the accompanying render window
                                        // its a render window
                                        windows_need_repaint -= 1;
                                        for metal_window in &mut metal_windows {
                                            if metal_window.window_id == window_id {
                                                metal_window.set_vsync_enable(windows_need_repaint == 0 && vsync);
                                                metal_window.set_buffer_count(if metal_window.window_geom.is_fullscreen {
                                                    3
                                                } else {
                                                    2
                                                });

                                                let dpi_factor = metal_window.window_geom.dpi_factor;

                                                metal_window.resize_core_animation_layer(&metal_cx);

                                                self.draw_pass_to_layer(
                                                    *pass_id,
                                                    dpi_factor,
                                                    metal_window.ca_layer,
                                                    &mut metal_cx,
                                                );
                                                // call redraw if we guessed the dpi wrong on startup
                                                if metal_window.first_draw {
                                                    metal_window.first_draw = false;
                                                    if dpi_factor != self.default_dpi_factor {
                                                        self.redraw_child_area(Area::All);
                                                    }
                                                }
                                            }
                                        }
                                    }
                                    CxPassDepOf::Pass(parent_pass_id) => {
                                        let dpi_factor = self.get_delegated_dpi_factor(parent_pass_id);
                                        self.draw_pass_to_texture(*pass_id, dpi_factor, &mut metal_cx);
                                    }
                                    CxPassDepOf::None => {
                                        self.draw_pass_to_texture(*pass_id, 1.0, &mut metal_cx);
                                    }
                                }
                            }
                        }
                    }
                    Event::None => {}
                    Event::Signal { .. } => {
                        self.call_event_handler(&mut event);
                        self.call_signals();
                    }
                    _ => {
                        self.call_event_handler(&mut event);
                    }
                }
                if self.process_desktop_post_event(event) {
                    cocoa_app.terminate_event_loop();
                }
            }

            !(!self.redraw_child_areas.is_empty() || self.requested_next_frame)
        })
    }
}

impl CxPlatformCommon for Cx {
    /// See [`CxPlatformCommon::show_text_ime`] for documentation.
    fn show_text_ime(&mut self, x: f32, y: f32) {
        self.platform.set_ime_position = Some(Vec2 { x, y });
    }

    /// See [`CxPlatformCommon::hide_text_ime`] for documentation.
    fn hide_text_ime(&mut self) {}

    /// See [`CxPlatformCommon::start_timer`] for documentation.
    fn start_timer(&mut self, interval: f64, repeats: bool) -> Timer {
        self.last_timer_id += 1;
        self.platform.start_timer.push((self.last_timer_id, interval, repeats));
        Timer { timer_id: self.last_timer_id }
    }

    /// See [`CxPlatformCommon::stop_timer`] for documentation.
    fn stop_timer(&mut self, timer: &mut Timer) {
        if timer.timer_id != 0 {
            self.platform.stop_timer.push(timer.timer_id);
            timer.timer_id = 0;
        }
    }

    /// See [`CxPlatformCommon::post_signal`] for documentation.
    fn post_signal(signal: Signal, status: StatusId) {
        if signal.signal_id != 0 {
            CocoaApp::post_signal(signal.signal_id, status);
        }
    }

    /// See [`CxPlatformCommon::update_menu`] for documentation.
    fn update_menu(&mut self, menu: &Menu) {
        // lets walk the menu and do the cocoa equivalents
        let platform = &mut self.platform;
        if platform.last_menu.is_none() || platform.last_menu.as_ref().unwrap() != menu {
            platform.last_menu = Some(menu.clone());
            platform.set_menu = true;
        }
    }
}

#[derive(Clone, Default)]
pub(crate) struct CxPlatform {
    pub(crate) bytes_written: usize,
    pub(crate) draw_calls_done: usize,
    pub(crate) last_menu: Option<Menu>,
    pub(crate) set_menu: bool,
    pub(crate) set_ime_position: Option<Vec2>,
    pub(crate) start_timer: Vec<(u64, f64, bool)>,
    pub(crate) stop_timer: Vec<u64>,
    pub(crate) text_clipboard_response: Option<String>,
    pub(crate) desktop: CxDesktop,
}
