//! Common code between all native desktop platforms. The counterpart is `cx_wasm32`.

use crate::cx::*;
use std::fs::File;
use std::io::prelude::*;
use std::io::Cursor;
use std::net::TcpStream;

#[derive(Clone)]
pub(crate) struct CxDesktop {
    pub(crate) repaint_via_scroll_event: bool,
    pub(crate) file_read_id: u64,
    pub(crate) file_reads: Vec<FileRead>,
    pub(crate) user_file_id_to_path: Vec<String>,
}

impl Default for CxDesktop {
    fn default() -> CxDesktop {
        CxDesktop { repaint_via_scroll_event: false, file_read_id: 1, file_reads: Vec::new(), user_file_id_to_path: Vec::new() }
    }
}

impl CxDesktopVsWasmCommon for Cx {
    /// See [`CxDesktopVsWasmCommon::get_default_window_size`] for documentation.
    fn get_default_window_size(&self) -> Vec2 {
        Vec2 { x: 800., y: 600. }
    }

    /// See [`CxDesktopVsWasmCommon::file_read`] for documentation.
    fn file_read(&mut self, path: &str) -> FileRead {
        let desktop = &mut self.platform.desktop;
        desktop.file_read_id += 1;
        let read_id = desktop.file_read_id;
        let file_read = FileRead { read_id, path: path.to_string() };
        desktop.file_reads.push(file_read.clone());
        file_read
    }

    /// See [`CxDesktopVsWasmCommon::get_file_reader`] for documentation.
    fn get_file_reader<'a>(&mut self, universal_file_handle: &'a UniversalFileHandle) -> Result<Box<dyn ReadSeek + 'a>, String> {
        match universal_file_handle {
            UniversalFileHandle::Path(path) => {
                if let Ok(file) = File::open(path) {
                    Ok(Box::new(file))
                } else {
                    Err(format!("Failed to open file {}", path))
                }
            }
            UniversalFileHandle::Data(data) => Ok(Box::new(Cursor::new(data))),
            UniversalFileHandle::Consumed => {
                unreachable!("UniversalFileHandle::Consumed encountered")
            }
        }
    }

    /// See [`CxDesktopVsWasmCommon::get_file_reader_consume`] for documentation.
    fn get_file_reader_consume(&mut self, universal_file_handle: &mut UniversalFileHandle) -> Result<Box<dyn ReadSeek>, String> {
        match universal_file_handle {
            UniversalFileHandle::Path(path) => {
                if let Ok(file) = File::open(path.clone()) {
                    Ok(Box::new(file))
                } else {
                    Err(format!("Failed to open file {}", path))
                }
            }
            UniversalFileHandle::Data(data) => {
                let mut new_data = Vec::<u8>::new();
                std::mem::swap(data, &mut new_data);
                Ok(Box::new(Cursor::new(new_data)))
            }
            UniversalFileHandle::Consumed => {
                unreachable!("UniversalFileHandle::Consumed encountered")
            }
        }
    }

    /// See [`CxDesktopVsWasmCommon::file_write`] for documentation.
    fn file_write(&mut self, path: &str, data: &[u8]) {
        // just write it right now
        if let Ok(mut file) = File::create(path) {
            if let Ok(_) = file.write_all(data) {
            } else {
                println!("ERROR WRITING FILE {}", path);
            }
        } else {
            println!("ERROR WRITING FILE {}", path);
        }
    }

    /// See [`CxDesktopVsWasmCommon::websocket_send`] for documentation.
    fn websocket_send(&mut self, _url: &str, _data: &[u8]) {}

    /// See [`CxDesktopVsWasmCommon::http_send`] for documentation.
    fn http_send(
        &mut self,
        verb: &str,
        path: &str,
        _proto: &str,
        domain: &str,
        port: u16,
        content_type: &str,
        body: &[u8],
        signal: Signal,
    ) {
        fn write_bytes_to_tcp_stream(tcp_stream: &mut TcpStream, bytes: &[u8]) -> bool {
            let bytes_total = bytes.len();
            let mut bytes_left = bytes_total;
            while bytes_left > 0 {
                let buf = &bytes[(bytes_total - bytes_left)..bytes_total];
                if let Ok(bytes_written) = tcp_stream.write(buf) {
                    if bytes_written == 0 {
                        return false;
                    }
                    bytes_left -= bytes_written;
                } else {
                    return true;
                }
            }
            false
        }

        // start a thread, connect, and report back.
        let data = body.to_vec();
        let byte_len = data.len();
        let header = format!(
            "{} {} HTTP/1.1\r\nHost: {}\r\nConnect: close\r\nContent-Type:{}\r\nContent-Length:{}\r\n\r\n",
            verb, path, domain, content_type, byte_len
        );
        let host = format!("{}:{}", domain, port);
        let _connect_thread = {
            std::thread::spawn(move || {
                let stream = TcpStream::connect(&host);
                if let Ok(mut stream) = stream {
                    if !write_bytes_to_tcp_stream(&mut stream, header.as_bytes())
                        && !write_bytes_to_tcp_stream(&mut stream, &data)
                    {
                        Cx::post_signal(signal, Cx::status_http_send_ok());
                        return;
                    }
                }
                Cx::post_signal(signal, Cx::status_http_send_fail());
            })
        };
    }
}

impl Cx {
    pub(crate) fn process_desktop_pre_event(&mut self, event: &mut Event) {
        match event {
            Event::FingerHover(fe) => {
                self.fingers[fe.digit].over_last = Area::Empty;
                //self.hover_mouse_cursor = None;
            }
            Event::FingerUp(_fe) => {
                self.down_mouse_cursor = None;
            }
            Event::WindowCloseRequested(_cr) => {}
            Event::FingerDown(fe) => {
                // lets set the finger tap count
                fe.tap_count = self.process_tap_count(fe.digit, fe.abs, fe.time);
            }
            Event::KeyDown(ke) => {
                self.process_key_down(ke.clone());
                if ke.key_code == KeyCode::PrintScreen {
                    self.panic_redraw = true;
                }
            }
            Event::KeyUp(ke) => {
                self.process_key_up(ke);
            }
            Event::AppFocusLost => {
                self.call_all_keys_up();
            }
            _ => (),
        };
    }

    pub(crate) fn process_desktop_post_event(&mut self, event: &mut Event) -> bool {
        match event {
            Event::FingerUp(fe) => {
                // decapture automatically
                self.fingers[fe.digit].captured = Area::Empty;
            }
            Event::FingerHover(fe) => {
                // new last area finger over
                self.fingers[fe.digit]._over_last = self.fingers[fe.digit].over_last;
                //if fe.hover_state == HoverState::Out{
                //    self.hover_mouse_cursor = None;
                //}
            }
            Event::FingerScroll(_) => {
                // check for anything being paint or dra dirty
                if !self.redraw_child_areas.is_empty() {
                    self.platform.desktop.repaint_via_scroll_event = true;
                }
            }
            _ => {}
        }
        false
    }

    pub(crate) fn process_desktop_paint_callbacks(&mut self) -> bool {
        let mut vsync = false; //self.platform.desktop.repaint_via_scroll_event;
        self.platform.desktop.repaint_via_scroll_event = false;
        if self.requested_next_frame {
            self.call_next_frame_event();
            if self.requested_next_frame {
                vsync = true;
            }
        }

        self.call_signals();

        // call redraw event
        if !self.redraw_child_areas.is_empty() {
            self.call_draw_event();
        }
        if !self.redraw_child_areas.is_empty() {
            vsync = true;
        }

        self.process_desktop_file_reads();

        self.call_signals();

        vsync
    }

    pub(crate) fn process_desktop_file_reads(&mut self) {
        if self.platform.desktop.file_reads.is_empty() {
            return;
        }

        let file_read_requests = self.platform.desktop.file_reads.clone();
        self.platform.desktop.file_reads.truncate(0);

        for read_req in file_read_requests {
            if read_req.path.starts_with("http://") || read_req.path.starts_with("https://") {
                let resp = ureq::get(&read_req.path).call();
                if resp.ok() {
                    let mut buffer: Vec<u8> = Vec::new();
                    if resp.into_reader().read_to_end(&mut buffer).is_ok() {
                        self.call_event_handler(&mut Event::FileRead(FileReadEvent {
                            read_id: read_req.read_id,
                            universal_file_handle: Ok(UniversalFileHandle::Data(buffer)),
                        }))
                    } else {
                        self.call_event_handler(&mut Event::FileRead(FileReadEvent {
                            read_id: read_req.read_id,
                            universal_file_handle: Err(format!("Failed to read {}", read_req.path)),
                        }))
                    }
                } else {
                    self.call_event_handler(&mut Event::FileRead(FileReadEvent {
                        read_id: read_req.read_id,
                        universal_file_handle: Err(format!("Failed to open {}", read_req.path)),
                    }))
                }
            } else {
                // Local file.
                self.call_event_handler(&mut Event::FileRead(FileReadEvent {
                    read_id: read_req.read_id,
                    universal_file_handle: Ok(UniversalFileHandle::Path(read_req.path)),
                }))
            }
        }

        if !self.platform.desktop.file_reads.is_empty() {
            self.process_desktop_file_reads();
        }
    }

    pub(crate) fn desktop_load_fonts(&mut self) {
        self.fonts.resize(FONT_FILENAMES.len(), CxFont::default());
        // lets load all fonts that aren't loaded yet
        for (font_id, filename) in FONT_FILENAMES.iter().enumerate() {
            let cxfont = &mut self.fonts[font_id];
            if let Ok(mut file_handle) = File::open(filename) {
                let mut buffer = Vec::<u8>::new();
                if file_handle.read_to_end(&mut buffer).is_ok() {
                    if cxfont.load_from_ttf_bytes(&buffer).is_err() {
                        println!("Error loading font {} ", filename);
                    } else {
                        cxfont.file = filename.to_string();
                    }
                }
            } else {
                println!("Error loading font {} ", filename);
            }
        }
    }
}
