//! The main module of this crate. Contains [`Cx`], which you will see
//! used all over the place.
//!
//! This re-exports a lot of stuff, and so internally you
//! will often see us write `use crate::cx::*` (external users should
//! instead write `use zaplib::*`, which will include all of
//! the exports in here as well).

use std::collections::{BTreeSet, HashMap};
use std::fmt::Write;
use zaplib_shader_compiler::generate_shader_ast::*;

pub use zaplib_shader_compiler::generate_shader_ast::CodeFragment;
pub use zaplib_shader_compiler::math::*;
pub use zaplib_shader_compiler::ty::Ty;

pub use crate::animator::*;
pub use crate::area::*;
pub use crate::cursor::*;
pub use crate::draw_input_type::*;
pub use crate::draw_tree::*;
pub use crate::events::*;
pub use crate::fonts::*;
pub use crate::geometry::*;
pub use crate::hash::*;
pub use crate::macros::*;
pub use crate::menu::*;
pub use crate::pass::*;
pub use crate::read_seek::*;
pub use crate::shader::*;
pub use crate::texture::*;
pub use crate::turtle::*;
pub use crate::universal_instant::*;
pub use crate::universal_thread::*;
pub use crate::window::*;

#[cfg(all(not(feature = "ipc"), target_os = "linux"))]
pub(crate) use crate::cx_linux::*;
#[cfg(all(not(feature = "ipc"), target_os = "linux"))]
pub(crate) use crate::cx_opengl::*;

#[cfg(all(not(feature = "ipc"), target_os = "macos"))]
pub(crate) use crate::cx_macos::*;
#[cfg(all(not(feature = "ipc"), target_os = "macos"))]
pub(crate) use crate::cx_metal::*;

#[cfg(all(not(feature = "ipc"), target_os = "windows"))]
pub(crate) use crate::cx_dx11::*;
#[cfg(all(not(feature = "ipc"), target_os = "windows"))]
pub(crate) use crate::cx_windows::*;

#[cfg(all(not(feature = "ipc"), target_arch = "wasm32"))]
pub(crate) use crate::cx_webgl::*;

#[cfg(all(not(feature = "ipc"), any(target_os = "linux", target_os = "macos", target_os = "windows")))]
pub(crate) use crate::cx_desktop::*;

#[cfg(all(not(feature = "ipc"), target_arch = "wasm32"))]
pub use crate::cx_wasm32::*;

#[cfg(feature = "ipc")]
pub use crate::cx_ipc_child::*;

#[cfg(all(feature = "ipc", target_arch = "wasm32"))]
pub use crate::cx_ipc_wasm32::*;

#[cfg(all(feature = "ipc", any(target_os = "linux", target_os = "macos")))]
pub use crate::cx_ipc_posix::*;

#[cfg(all(feature = "ipc", target_os = "windows"))]
pub use crate::cx_ipc_win32::*;

/// Contains information about the platform (operating system) that we're running on.
#[derive(Clone)]
pub enum PlatformType {
    Unknown,
    Windows,
    OSX,
    Linux { custom_window_chrome: bool },
    Web { protocol: String, hostname: String, port: u16, pathname: String, search: String, hash: String },
}

impl PlatformType {
    pub fn is_desktop(&self) -> bool {
        match self {
            PlatformType::Unknown => true,
            PlatformType::Windows => true,
            PlatformType::OSX => true,
            PlatformType::Linux { .. } => true,
            PlatformType::Web { .. } => false,
        }
    }
}

/// The main "context" object which contains pretty much everything we need within the framework.
pub struct Cx {
    /// See [`PlatformType`].
    pub platform_type: PlatformType,

    /// List of actual [`CxWindow`] objects. [`Window::window_id`] represents an index in this list.
    pub(crate) windows: Vec<CxWindow>,
    /// Indices in [`Cx::windows`] that have been closed and can be reused.
    ///
    /// TODO(JP): Should we be more explicit and use [`Option`] in [`Cx::windows`]?
    pub(crate) windows_free: Vec<usize>,

    /// List of actual [`CxPass`] objects. [`Pass::pass_id`] represents an index in this list.
    ///
    /// TODO(JP): We currently never remove old [`CxPass`]es.
    pub(crate) passes: Vec<CxPass>,

    /// The [`CxView`] objects that make up the draw tree. [`View::view_id`] represents an index in this list.
    ///
    /// TODO(JP): The first element is a dummy element, since we use `view_id == 0`
    /// as a sort of null pointer, which is pretty gross and can be confusing. Might
    /// be better to use `Option` wherever we need that instead..
    ///
    /// TODO(JP): We currently never remove old [`CxView`]s.
    pub(crate) views: Vec<CxView>,

    /// The compiled [`CxShader`]s. [`Shader::shader_id`] represents an index in this list.
    pub(crate) shaders: Vec<CxShader>,
    /// Map from hashed names (using [`StringHash`] to shader_id / indices in [`Cx::shaders`].
    pub(crate) shader_id_by_hashed_name: HashMap<u64, usize>,
    /// List of actual [`CxTexture`] objects. [`Texture::texture_id`] represents an index in this list.
    pub(crate) textures: Vec<CxTexture>,
    /// List of actual [`CxGeometry`] objects. [`Geometry::geometry_id`] represents an index in this list.
    pub(crate) geometries: Vec<CxGeometry>,

    /// Whether we are currently (re)drawing, ie. we called the app's `draw_app` function.
    pub(crate) in_redraw_cycle: bool,
    /// An auto-incrementing ID representing the current (re)draw cycle.
    ///
    /// TODO(JP): This value probably shouldn't be used when [`Cx::in_redraw_cycle`] is false, but
    /// currently it sticks around. We could combine the two variables in an [`Option<u64>`]?
    pub(crate) redraw_id: u64,
    /// Stack of [`Window::window_id`]s / indices into [`Cx::windows`], using [`Window::begin_window`]
    /// and [`Window::end_window`].
    pub(crate) window_stack: Vec<usize>,
    /// Stack of [`Pass::pass_id`]s / indices into [`Cx::passes`], using [`Pass::begin_pass`]
    /// and [`Pass::end_pass`].
    pub(crate) pass_stack: Vec<usize>,
    /// Stack of [`View::view_id`]s / indices into [`Cx::views`], using [`View::begin_view`]
    /// and [`View::end_view`].
    pub(crate) view_stack: Vec<usize>,
    /// A stack of [`Turtle`]s, using [`Cx::begin_turtle`] and [`Cx::end_turtle`]. Turtles all the
    /// way down!
    pub(crate) turtles: Vec<Turtle>,

    /// A list of [`Area`]s that we want to align later on. This is kept separate
    /// from [`Cx::turtles`] (even though this is part of the turtle layout
    /// system) so that a parent [`Turtle`] can align a bunch of stuff that was
    /// drawn using child [`Turtle`]s.
    ///
    /// TODO(JP): This may currently only contain [`Area::InstanceRange`], so
    /// maybe we should change the type of this? It might also be nice to be able
    /// explicitly push [`Area::View`] on this list though, and then set a uniform
    /// on the entire [`CxView`], for better performance?
    pub(crate) turtle_align_list: Vec<Area>,

    /// List of [`Area`]s that need to be redrawn on the next draw cycle.
    /// See [`Cx::redraw_child_area`].
    pub(crate) redraw_child_areas: Vec<Area>,
    /// List of [`Area`]s that need to be redrawn in the current draw cycle.
    /// This is kept separate from [`Cx::redraw_child_areas`] so that we can mark
    /// new [`Area`]s for redraw _during_ a draw cycle.
    ///
    /// TODO(JP): See [`Cx::redraw_child_area`] for a TODO involving this field.
    ///
    /// See [`Cx::redraw_child_area`].
    _redraw_child_areas: Vec<Area>,

    /// The system-default `dpi_factor`. See also [`PassUniforms::dpi_factor`].
    ///
    /// More commonly known as the "device pixel ratio". TODO(JP): Rename?
    pub(crate) default_dpi_factor: f32,

    /// The `dpi_factor` used during the current [`Pass`] (since you can override the system default).
    /// See also [`PassUniforms::dpi_factor`].
    ///
    /// More commonly known as the "device pixel ratio". TODO(JP): Rename?
    pub(crate) current_dpi_factor: f32,

    /// Last timestamp from when an event was fired, in second since the application
    /// was started. Typically you want to use this instead of making a system call.
    pub last_event_time: f64,

    /// The last [`Timer::timer_id`] that was issued.
    pub(crate) last_timer_id: u64,
    /// The last [`Signal::signal_id`] that was issued.
    pub(crate) last_signal_id: usize,

    /// The current [`Area`] that has keyboard focus, so it can register key input [`Event`]s.
    ///
    /// See also [`Cx::prev_key_focus`] and [`Cx::next_key_focus`].
    pub(crate) key_focus: Area,
    /// The [`Area`] that previously was [`Cx::key_focus`], so you can revert it using
    /// [`Cx::revert_key_focus`].
    ///
    /// See also [`Cx::key_focus`] and [`Cx::next_key_focus`].
    prev_key_focus: Area,
    /// The [`Area`] that will become [`Cx::key_focus`] when the current events are handled.
    /// Gets set using [`Cx::set_key_focus`] or [`Cx::revert_key_focus`].
    ///
    /// See also [`Cx::prev_key_focus`] and [`Cx::next_key_focus`].
    ///
    /// TODO(JP): It's possible to set this during the draw cycle instead of during an
    /// event handler, and then it won't update [`Cx::key_focus`] until the next event
    /// is handled. We should probably guard against that.
    next_key_focus: Area,
    pub(crate) keys_down: Vec<KeyEvent>,

    /// The cursor type that the user sees while holding the mouse down. Gets reset to [`None`] when
    /// you release the mouse button ([`Event::FingerUp`]).
    pub(crate) down_mouse_cursor: Option<MouseCursor>,

    /// The cursor type that the user sees while hovering (not holding the mouse down).
    /// Gets reset when there's a new [`Event::FingerHover`], so you have to periodically set this.
    pub(crate) hover_mouse_cursor: Option<MouseCursor>,

    /// The current state of each "finger" that we track.
    ///
    /// TODO(JP): This seems mostly relevant for multi-touch, which we don't really support very
    /// well yet. Should we keep this?
    pub(crate) fingers: Vec<CxPerFinger>,

    /// Whether [`Cx::request_next_frame`] was called.
    pub(crate) requested_next_frame: bool,

    /// The local "signals", which are like custom events that also work across threads.
    ///
    /// See also [`Signal`] and [`SignalEvent`].
    pub(crate) signals: HashMap<Signal, BTreeSet<StatusId>>,

    /// A map from profile IDs to [`UniversalInstant`], for keeping track of how long things
    /// take.
    pub(crate) profiles: HashMap<u64, UniversalInstant>,

    /// For compiling [`Shader`]s.
    pub(crate) shader_ast_generator: ShaderAstGenerator,

    /// Settings per command; see [`CommandId`] and [`CxCommandSetting`].
    pub(crate) command_settings: HashMap<CommandId, CxCommandSetting>,

    /// When set to true, will trigger a panic on the next redraw. Can be useful
    /// for debugging unwanted redraws. Can be set to true by pressing the "print
    /// screen" button on the keyboard.
    pub(crate) panic_redraw: bool,

    /// Platform-specific fields.
    pub(crate) platform: CxPlatform,

    /// The user's event handler. Storing it like this cuts the compile time of an end-user application in half.
    pub(crate) event_handler: Option<*mut dyn FnMut(&mut Cx, &mut Event)>,

    /// List of actual [`CxFont`] objects. [`Font::font_id`] represents an index in this list.
    pub(crate) fonts: Vec<CxFont>,
    /// See [`CxFontsAtlas`].
    pub(crate) fonts_atlas: CxFontsAtlas,
}

/// Settings for "commands"; see [`CommandId`].
///
/// Only supported on OSX for now.
#[cfg_attr(not(target_os = "macos"), allow(dead_code))]
#[derive(Clone, Copy, Default)]
pub(crate) struct CxCommandSetting {
    pub(crate) shift: bool,
    pub(crate) key_code: KeyCode,
    pub(crate) enabled: bool,
}

#[derive(Default, Clone)]
pub(crate) struct CxPerFinger {
    pub(crate) captured: Area,
    pub(crate) tap_count: (Vec2, f64, u32),
    pub(crate) down_abs_start: Vec2,
    pub(crate) down_rel_start: Vec2,
    pub(crate) over_last: Area,
    pub(crate) _over_last: Area,
}

pub(crate) const NUM_FINGERS: usize = 10;

impl Default for Cx {
    fn default() -> Self {
        let mut fingers = Vec::new();
        fingers.resize(NUM_FINGERS, CxPerFinger::default());

        let textures = vec![CxTexture {
            desc: TextureDesc { format: TextureFormat::ImageBGRA, width: Some(4), height: Some(4), multisample: None },
            image_u32: vec![0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            update_image: true,
            platform: CxPlatformTexture::default(),
        }];

        Self {
            platform_type: PlatformType::Unknown,

            windows: Vec::new(),
            windows_free: Vec::new(),
            passes: Vec::new(),
            // TODO(JP): See my note up at [`Cx::views`].
            views: vec![CxView { ..Default::default() }],
            fonts: Vec::new(),
            fonts_atlas: CxFontsAtlas::default(),
            textures,
            shaders: Vec::new(),
            shader_id_by_hashed_name: HashMap::new(),
            geometries: Vec::new(),

            default_dpi_factor: 1.0,
            current_dpi_factor: 1.0,
            in_redraw_cycle: false,
            window_stack: Vec::new(),
            pass_stack: Vec::new(),
            view_stack: Vec::new(),
            turtles: Vec::new(),
            turtle_align_list: Vec::new(),

            last_event_time: 0.0,

            redraw_child_areas: Vec::new(),
            _redraw_child_areas: Vec::new(),

            redraw_id: 1,
            last_timer_id: 1,
            last_signal_id: 1,

            next_key_focus: Area::Empty,
            prev_key_focus: Area::Empty,
            key_focus: Area::Empty,
            keys_down: Vec::new(),

            down_mouse_cursor: None,
            hover_mouse_cursor: None,
            fingers,

            shader_ast_generator: ShaderAstGenerator::new(),

            command_settings: HashMap::new(),

            requested_next_frame: false,

            profiles: HashMap::new(),

            signals: HashMap::new(),

            panic_redraw: false,

            platform: CxPlatform { ..Default::default() },

            event_handler: None,
        }
    }
}

impl Cx {
    pub(crate) fn process_tap_count(&mut self, digit: usize, pos: Vec2, time: f64) -> u32 {
        if digit >= self.fingers.len() {
            return 0;
        };
        let (last_pos, last_time, count) = self.fingers[digit].tap_count;

        if (time - last_time) < 0.5 && pos.distance(&last_pos) < 10. {
            self.fingers[digit].tap_count = (pos, time, count + 1);
            count + 1
        } else {
            self.fingers[digit].tap_count = (pos, time, 1);
            1
        }
    }

    pub fn get_dpi_factor_of(&mut self, area: &Area) -> f32 {
        match area {
            Area::InstanceRange(ia) => {
                let pass_id = self.views[ia.view_id].pass_id;
                return self.get_delegated_dpi_factor(pass_id);
            }
            Area::View(va) => {
                let pass_id = self.views[va.view_id].pass_id;
                return self.get_delegated_dpi_factor(pass_id);
            }
            _ => (),
        }
        1.0
    }

    pub(crate) fn get_delegated_dpi_factor(&mut self, pass_id: usize) -> f32 {
        let mut dpi_factor = 1.0;
        let mut pass_id_walk = pass_id;
        for _ in 0..25 {
            match self.passes[pass_id_walk].dep_of {
                CxPassDepOf::Window(window_id) => {
                    dpi_factor = match self.windows[window_id].window_state {
                        CxWindowState::Create { .. } => self.default_dpi_factor,
                        CxWindowState::Created => self.windows[window_id].window_geom.dpi_factor,
                        _ => 1.0,
                    };
                    break;
                }
                CxPassDepOf::Pass(next_pass_id) => {
                    pass_id_walk = next_pass_id;
                }
                _ => {
                    break;
                }
            }
        }
        dpi_factor
    }

    pub(crate) fn compute_passes_to_repaint(&mut self, passes_todo: &mut Vec<usize>, windows_need_repaint: &mut usize) {
        passes_todo.truncate(0);

        loop {
            let mut altered = false; // yes this is horrible but im tired and i dont know why recursion fails
            for pass_id in 0..self.passes.len() {
                if self.passes[pass_id].paint_dirty {
                    let other = match self.passes[pass_id].dep_of {
                        CxPassDepOf::Pass(dep_of_pass_id) => Some(dep_of_pass_id),
                        _ => None,
                    };
                    if let Some(other) = other {
                        if !self.passes[other].paint_dirty {
                            self.passes[other].paint_dirty = true;
                            altered = true;
                        }
                    }
                }
            }
            if !altered {
                break;
            }
        }

        for (pass_id, cxpass) in self.passes.iter().enumerate() {
            if cxpass.paint_dirty {
                let mut inserted = false;
                match cxpass.dep_of {
                    CxPassDepOf::Window(_) => *windows_need_repaint += 1,
                    CxPassDepOf::Pass(dep_of_pass_id) => {
                        if pass_id == dep_of_pass_id {
                            eprintln!("WHAAAT");
                        }
                        for insert_before in 0..passes_todo.len() {
                            if passes_todo[insert_before] == dep_of_pass_id {
                                passes_todo.insert(insert_before, pass_id);
                                inserted = true;
                                break;
                            }
                        }
                    }
                    CxPassDepOf::None => {
                        // we need to be first
                        passes_todo.insert(0, pass_id);
                        inserted = true;
                    }
                }
                if !inserted {
                    passes_todo.push(pass_id);
                }
            }
        }
    }

    /// Mark a particular `Area` to be redrawn.
    ///
    /// All parent [`View`]s will also be redrawn, since rendering always starts
    /// at the root of a [`Window`] or [`Pass`].
    ///
    /// If you pass in [`Area::All`], we'll redraw everything. This should
    /// generally be OK to do, though it is slower than only redrawing
    /// particular [`Area`]s.
    ///
    /// TODO(JP): When called _during_ a draw cycle, this will always trigger a
    /// new draw cycle, even if the [`Area`] that was marked hasn't been drawn
    /// yet. It would be better to see if we can still redraw the [`Area`] (and
    /// its parents) during the current draw cycle, and then modify
    /// [`Cx::_redraw_child_areas`] instead of [`Cx::redraw_child_areas`].
    ///
    /// See also [`View::redraw_view`], [`Cx::view_will_redraw`].
    pub fn redraw_child_area(&mut self, area: Area) {
        if self.panic_redraw {
            #[cfg(debug_assertions)]
            panic!("Panic Redraw triggered")
        }

        assert_ne!(area, Area::Empty);

        // if we are redrawing all, clear the rest
        if area == Area::All {
            self.redraw_child_areas.truncate(0);
        }
        // check if we are already redrawing all
        else if self.redraw_child_areas.len() == 1 && self.redraw_child_areas[0] == Area::All {
            return;
        };
        // only add it if we dont have it already
        if self.redraw_child_areas.iter().any(|a| *a == area) {
            return;
        }
        self.redraw_child_areas.push(area);
    }

    /// Checks if a [`View`] will redraw.
    ///
    /// By walking the tree upwards from any [`Area`]s that have been marked for
    /// redraw in [`Cx::redraw_child_area`].
    ///
    /// TODO(JP): This will often visit the same draw tree nodes multiple times,
    /// so it would be better to cache this somewhere, e.g. in a `Vec<bool>` of
    /// same size as [`Cx::views`]. We could even precompute that in
    /// [`Cx::redraw_child_area`] and have that replace [`Cx::redraw_child_areas`]
    /// entirely.
    ///
    /// See also [`View::view_will_redraw`].
    pub(crate) fn view_will_redraw(&self, view_id: usize) -> bool {
        // figure out if areas are in some way a child of draw_list_id, then we need to redraw
        for area in &self._redraw_child_areas {
            match area {
                Area::All => {
                    return true;
                }
                Area::Empty => (),
                Area::InstanceRange(instance) => {
                    let mut vw = instance.view_id;
                    if vw == view_id {
                        return true;
                    }
                    while vw != 0 {
                        vw = self.views[vw].nesting_view_id;
                        if vw == view_id {
                            return true;
                        }
                    }
                }
                Area::View(viewarea) => {
                    let mut vw = viewarea.view_id;
                    if vw == view_id {
                        return true;
                    }
                    while vw != 0 {
                        vw = self.views[vw].nesting_view_id;
                        if vw == view_id {
                            return true;
                        }
                    }
                }
            }
        }
        false
    }

    /// Update any global pointers that were referring to an old [`Area`].
    ///
    /// TODO(JP): It's easy to forget to call this! Should we add a field on
    /// [`Area`] (or use an existing one, like [`ViewArea::redraw_id`]) to
    /// only mark a new [`Area`] as valid once you pass it through this function?
    pub fn update_area_refs(&mut self, old_area: Area, new_area: Area) -> Area {
        if old_area == Area::Empty || old_area == Area::All {
            return new_area;
        }

        for finger in &mut self.fingers {
            if finger.captured == old_area {
                finger.captured = new_area;
            }
            if finger._over_last == old_area {
                finger._over_last = new_area;
            }
        }
        // update capture keyboard
        if self.key_focus == old_area {
            self.key_focus = new_area
        }

        // update capture keyboard
        if self.prev_key_focus == old_area {
            self.prev_key_focus = new_area
        }
        if self.next_key_focus == old_area {
            self.next_key_focus = new_area
        }

        new_area
    }

    /// Sets an [`Area`] that will become [`Cx::key_focus`] when the current events are handled.
    ///
    /// TODO(JP): It's possible to set this during the draw cycle instead of during an
    /// event handler, and then it won't update [`Cx::key_focus`] until the next event
    /// is handled. We should probably guard against that.
    pub fn set_key_focus(&mut self, focus_area: Area) {
        self.next_key_focus = focus_area;
    }

    /// Reverts back to the previous [`Cx::key_focus`] value.
    ///
    /// TODO(JP): It's possible to set this during the draw cycle instead of during an
    /// event handler, and then it won't update [`Cx::key_focus`] until the next event
    /// is handled. We should probably guard against that.
    pub fn revert_key_focus(&mut self) {
        self.next_key_focus = self.prev_key_focus;
    }

    /// Check if an [`Area`] currently is the [Cx::key_focus] area.
    pub fn has_key_focus(&self, focus_area: Area) -> bool {
        self.key_focus == focus_area
    }

    pub(crate) fn process_key_down(&mut self, key_event: KeyEvent) {
        if self.keys_down.iter().any(|k| k.key_code == key_event.key_code) {
            return;
        }
        self.keys_down.push(key_event);
    }

    pub(crate) fn process_key_up(&mut self, key_event: &KeyEvent) {
        for i in 0..self.keys_down.len() {
            if self.keys_down[i].key_code == key_event.key_code {
                self.keys_down.remove(i);
                return;
            }
        }
    }

    pub(crate) fn call_all_keys_up(&mut self) {
        let mut keys_down = Vec::new();
        std::mem::swap(&mut keys_down, &mut self.keys_down);
        for key_event in keys_down {
            self.call_event_handler(&mut Event::KeyUp(key_event))
        }
    }

    pub(crate) fn call_event_handler(&mut self, event: &mut Event) {
        let event_handler = self.event_handler.unwrap();

        unsafe {
            (*event_handler)(self, event);
        }

        if self.next_key_focus != self.key_focus {
            self.prev_key_focus = self.key_focus;
            self.key_focus = self.next_key_focus;
            unsafe {
                (*event_handler)(self, &mut Event::KeyFocus(KeyFocusEvent { prev: self.prev_key_focus, focus: self.key_focus }));
            }
        }
    }

    pub(crate) fn call_draw_event(&mut self) {
        // self.profile();
        self.in_redraw_cycle = true;
        self.redraw_id += 1;
        std::mem::swap(&mut self._redraw_child_areas, &mut self.redraw_child_areas);
        self.turtle_align_list.truncate(0);
        self.redraw_child_areas.truncate(0);
        self.call_event_handler(&mut Event::Draw);
        self.in_redraw_cycle = false;
        if !self.view_stack.is_empty() {
            panic!("View stack disaligned, forgot an end_view(cx)");
        }
        if !self.pass_stack.is_empty() {
            panic!("Pass stack disaligned, forgot an end_pass(cx)");
        }
        if !self.window_stack.is_empty() {
            panic!("Window stack disaligned, forgot an end_window(cx)");
        }
        if !self.turtles.is_empty() {
            panic!("Turtle stack disaligned, forgot an end_turtle()");
        }
        //self.profile();
    }

    pub(crate) fn call_next_frame_event(&mut self) {
        self.requested_next_frame = false;
        self.call_event_handler(&mut Event::NextFrame);
    }

    /// Request an [`Event::NextFrame`].
    pub fn request_next_frame(&mut self) {
        self.requested_next_frame = true;
    }

    /// Create a new [`Signal`], which is used to send and capture custom
    /// events.
    ///
    /// See also [`SignalEvent`].
    pub fn new_signal(&mut self) -> Signal {
        self.last_signal_id += 1;
        Signal { signal_id: self.last_signal_id }
    }

    /// Triggers a new [`SignalEvent`] with the same ID as [`Signal`]. You can
    /// post custom data with it using `status`.
    ///
    /// If you want to fire a [`SignalEvent`] from a thread, use [`Cx::post_signal`]
    /// instead.
    ///
    /// See also [`SignalEvent`].
    pub fn send_signal(&mut self, signal: Signal, status: StatusId) {
        if signal.signal_id == 0 {
            return;
        }
        if let Some(statusses) = self.signals.get_mut(&signal) {
            if !statusses.contains(&status) {
                statusses.insert(status);
            }
        } else {
            let mut new_set = BTreeSet::new();
            new_set.insert(status);
            self.signals.insert(signal, new_set);
        }
    }

    pub(crate) fn call_signals(&mut self) {
        let mut counter = 0;
        while !self.signals.is_empty() {
            counter += 1;
            let mut signals = HashMap::new();
            std::mem::swap(&mut self.signals, &mut signals);

            self.call_event_handler(&mut Event::Signal(SignalEvent { signals }));

            if counter > 100 {
                println!("Signal feedback loop detected");
                break;
            }
        }
    }

    pub fn status_http_send_ok() -> StatusId {
        uid!()
    }
    pub fn status_http_send_fail() -> StatusId {
        uid!()
    }

    pub(crate) fn debug_draw_tree(&mut self, view_id: usize) {
        if let Some(debug_type) = &self.views[view_id].debug {
            let mut s = String::new();
            let dump_instances = *debug_type == CxViewDebug::Instances;
            self.debug_draw_tree_recur(dump_instances, &mut s, view_id, 0);
            crate::log!("{}", &s);
        }
    }

    fn debug_draw_tree_recur(&mut self, dump_instances: bool, s: &mut String, view_id: usize, depth: usize) {
        if view_id >= self.views.len() {
            writeln!(s, "---------- Drawlist still empty ---------").unwrap();
            return;
        }
        let mut indent = String::new();
        for _i in 0..depth {
            indent.push_str("  ");
        }
        let draw_calls_len = self.views[view_id].draw_calls_len;
        if view_id == 0 {
            writeln!(s, "---------- Begin Debug draw tree for redraw_id: {} ---------", self.redraw_id).unwrap();
        }
        writeln!(
            s,
            "{}view {}: len:{} rect:{:?} scroll:{:?}",
            indent, view_id, draw_calls_len, self.views[view_id].rect, self.views[view_id].snapped_scroll
        )
        .unwrap();
        indent.push_str("  ");
        for draw_call_id in 0..draw_calls_len {
            let sub_view_id = self.views[view_id].draw_calls[draw_call_id].sub_view_id;
            if sub_view_id != 0 {
                self.debug_draw_tree_recur(dump_instances, s, sub_view_id, depth + 1);
            } else {
                let cxview = &mut self.views[view_id];
                let draw_call = &mut cxview.draw_calls[draw_call_id];
                let sh = &self.shaders[draw_call.shader.shader_id];
                let slots = sh.mapping.instance_props.total_slots;
                let instances = draw_call.instances.len() / slots;
                writeln!(
                    s,
                    "{}call {}: {}({}) *:{} scroll: {} draw_local_scroll: {}",
                    indent,
                    draw_call_id,
                    sh.name,
                    draw_call.shader.shader_id,
                    instances,
                    vec2(draw_call.draw_uniforms.draw_scroll_x, draw_call.draw_uniforms.draw_scroll_y),
                    vec2(draw_call.draw_uniforms.draw_local_scroll_x, draw_call.draw_uniforms.draw_local_scroll_y)
                )
                .unwrap();
                if dump_instances {
                    for inst in 0..instances.min(1) {
                        let mut out = String::new();
                        let mut off = 0;
                        for prop in &sh.mapping.instance_props.props {
                            match prop.slots {
                                1 => out.push_str(&format!("{}:{} ", prop.name, draw_call.instances[inst * slots + off])),
                                2 => out.push_str(&format!(
                                    "{}:v2({},{}) ",
                                    prop.name,
                                    draw_call.instances[inst * slots + off],
                                    draw_call.instances[inst * slots + 1 + off]
                                )),
                                3 => out.push_str(&format!(
                                    "{}:v3({},{},{}) ",
                                    prop.name,
                                    draw_call.instances[inst * slots + off],
                                    draw_call.instances[inst * slots + 1 + off],
                                    draw_call.instances[inst * slots + 1 + off]
                                )),
                                4 => out.push_str(&format!(
                                    "{}:v4({},{},{},{}) ",
                                    prop.name,
                                    draw_call.instances[inst * slots + off],
                                    draw_call.instances[inst * slots + 1 + off],
                                    draw_call.instances[inst * slots + 2 + off],
                                    draw_call.instances[inst * slots + 3 + off]
                                )),
                                _ => {}
                            }
                            off += prop.slots;
                        }
                        writeln!(s, "  {}instance {}: {}", indent, inst, out).unwrap();
                    }
                }
            }
        }
        if view_id == 0 {
            writeln!(s, "---------- End Debug draw tree for redraw_id: {} ---------", self.redraw_id).unwrap();
        }
    }
}

/// A bunch of traits that are common between the native platforms and the WebAssembly platform. This trait makes sure
/// that there is consistency in the interface, and provides one place for documentation.
pub trait CxDesktopVsWasmCommon {
    /// Get a default window size for new windows.
    /// TODO(JP): This doesn't make too much sense for Wasm; maybe just omit this method there?
    fn get_default_window_size(&self) -> Vec2;

    /// Request for a file to be read. Makes some effort to defer actually reading of the file until
    /// later. Currently the behavior is as follows:
    /// * Wasm32 target: Always loads the entire file over HTTP(S).
    /// * Native targets: If `path` is a relative file name, we don't load the file but instead open
    //    a read handle. If `path` is an HTTP URL we load the whole file. HTTPS is not supported yet.
    fn file_read(&mut self, path: &str) -> FileRead;

    /// Get an actual file reader based on a [`UniversalFileHandle`].
    fn get_file_reader<'a>(&mut self, universal_file_handle: &'a UniversalFileHandle) -> Result<Box<dyn ReadSeek + 'a>, String>;

    /// Get a file reader, but consumes the UniversalFileHandle if it contains
    /// [`UniversalFileHandle::Data`]. Useful if you want to store the resulting reader somewhere.
    fn get_file_reader_consume(&mut self, universal_file_handle: &mut UniversalFileHandle) -> Result<Box<dyn ReadSeek>, String>;

    /// Write `data` to `path`.
    fn file_write(&mut self, path: &str, data: &[u8]);

    /// Send data over a Websocket.
    fn websocket_send(&mut self, url: &str, data: &[u8]);

    /// Make an HTTP request. When done, you get a [`SignalEvent`] corresponding to the provided
    /// [`Signal`] with [`Cx::status_http_send_ok`] or [`Cx::status_http_send_fail`] as the status.
    fn http_send(
        &mut self,
        verb: &str,
        path: &str,
        _proto: &str,
        domain: &str,
        port: u16,
        content_type: &str,
        body: &[u8],
        signal: Signal,
    );
}

/// A bunch of traits that are common between the different target platforms. This trait makes sure
/// that there is consistency in the interface, and provides one place for documentation.
pub trait CxPlatformCommon {
    /// Show an [Input Method Editor (IME)](https://en.wikipedia.org/wiki/Input_method) at a particular
    /// location, typically with everything but the cursor hidden.
    fn show_text_ime(&mut self, x: f32, y: f32);
    /// Hide the IME shown by [`CxPlatformCommon::show_text_ime`].
    fn hide_text_ime(&mut self);
    /// Start a new [`Timer`] with the given `interval`, and which may `repeat` if required.
    fn start_timer(&mut self, interval: f64, repeats: bool) -> Timer;
    /// Stop a [`Timer`] given by [`CxPlatformCommon::start_timer`].
    fn stop_timer(&mut self, timer: &mut Timer);
    /// Post a [`Signal`] from any thread. If you don't need to use this from a thread, you may
    /// instead use [`Cx::send_signal`], which might be faster.
    fn post_signal(signal: Signal, status: StatusId);
    /// Set a [`Menu`].
    fn update_menu(&mut self, menu: &Menu);
}
