//! "Pointer" into the draw tree.

use crate::cx::*;
use zaplib_shader_compiler::ty::Ty;

/// An [`Area`] can be thought of as a "pointer" into the draw tree. You typically
/// get one as the result of a draw command, like [`Cx::add_instance`],
/// or [`View::end_view`].
///
/// Note that an [`Area`] might point to an old element in the draw tree, so use
/// [`Area::is_valid`] to check if it points to the latest version. [`Area`
/// typically gets updated using [`Cx::update_area_refs`] under the hood.
///
/// You can use an [`Area`] pointer to write fields, e.g. using [`Area::get_write_ref`],
/// [`DrawInputType::write_shader_value`], and so on. You
/// can also use it for rerendering part of the application (using
/// [`Cx::redraw_child_area`]), checking if an event was fired on a certain part
/// of the draw tree (using [`Event::hits`], [`Cx::key_focus`], etc), and more.
///
/// TODO(JP): this can currently point to a [`View`]/[`CxView`] that isn't
/// actually in the draw tree any more (ie. there the corresponding [`CxView`] is
/// never referenced in the draw tree), or which just doesn't exist at all any
/// more (ie. the [`View`] object has also been removed). There is currently no
/// was of telling if any of this is the case, since there is no "garbage
/// collection" of views. [`CxView`] just sticks around in [`Cx::views`] forever.
#[derive(Clone, Debug, Hash, PartialEq, Ord, PartialOrd, Eq, Copy)]
pub enum Area {
    /// A "null pointer", doesn't point to anything yet.
    Empty,
    /// Used to represent the entire application at once. Mostly useful for
    /// rerendering everything, by passing this into [`Cx::redraw_child_area`].
    ///
    /// TODO(JP): this isn't really useful outside of rerendering everything;
    /// should we maybe remove this and do that some other way?
    All,
    /// See [`ViewArea`].
    View(ViewArea),
    /// See [`InstanceRangeArea`].
    InstanceRange(InstanceRangeArea),
}

impl Default for Area {
    fn default() -> Area {
        Area::Empty
    }
}

/// Pointer to a particular view in the draw tree, using a [`ViewArea::view_id`]. Note
/// that a [`View::view_id`] only gets set when it gets drawn.
///
/// See also [`Area`].
#[derive(Clone, Default, Hash, Ord, PartialOrd, Eq, Debug, PartialEq, Copy)]
pub struct ViewArea {
    /// Corresponds to [`View::view_id`] of the [`View`] this points to, which
    /// is the same as the index of [`Cx::views`] of the corresponding [`CxView`].
    pub(crate) view_id: usize,
    /// The [`Cx::redraw_id`] during which this [`Area`] was created. If it
    /// doesn't match the corresponding [`View::redraw_id`], then this pointer is
    /// stale; it likely wasn't properly updated with [`Cx::redraw_child_area`].
    ///
    /// Note that if [`ViewArea::redraw_id`] doesn't match [`Cx::redraw_id`] then that doesn't
    /// necessarily mean that the pointer is stale, since instead rendering for
    /// the corresponding [`View`] could have been skipped (if nothing had
    /// changed).
    ///
    /// See also [`View::view_id`].
    ///
    /// TODO(JP): Is the [`ViewArea`] redundant, since it basically contains the
    /// same information as the [`View`] itself?
    pub(crate) redraw_id: u64,
}

/// Pointer to a part of a [`DrawCall`], e.g. from [`crate::DrawQuad::area()`] or
/// [`Cx::end_many_instances`]. This pointer points to a range of instances,
/// where the first index is indicated by [`InstanceRangeArea::instance_offset`], and the size of the
/// range by [`InstanceRangeArea::instance_count`].
///
/// See also [`Area`].
#[derive(Clone, Default, Hash, Ord, PartialOrd, Eq, Debug, PartialEq, Copy)]
pub struct InstanceRangeArea {
    /// Corresponds to [`View::view_id`] of the [`View`] this points to, which
    /// is the same as the index of [`Cx::views`] of the corresponding [`CxView`].
    pub(crate) view_id: usize,
    /// Index of [`CxView::draw_calls`] of the corresponding [`DrawCall`].
    pub(crate) draw_call_id: usize,
    /// Offset in "slots"/nibbles/4 bytes from the start of [`DrawCall::instances`]
    /// to the first instance that this pointer describes.
    pub(crate) instance_offset: usize,
    /// Number of instances that this pointer describes.
    pub(crate) instance_count: usize,
    /// See [`ViewArea::redraw_id`].
    pub(crate) redraw_id: u64,
}

/// Immutable reference to the raw data in a [`DrawCall`].
pub struct DrawReadRef<'a> {
    pub buffer: &'a [f32],
    pub repeat: usize,
    pub stride: usize,
}

/// Mutable reference to the raw data in a [`DrawCall`].
pub struct DrawWriteRef<'a> {
    pub buffer: &'a mut [f32],
    pub repeat: usize,
    pub stride: usize,
}

impl Area {
    /// Shorthand for `if let Area::Empty = area`.
    pub fn is_empty(&self) -> bool {
        if let Area::Empty = self {
            return true;
        }
        false
    }

    /// Check if this is an [`Area::InstanceRange`] that points to the first instance
    /// in its corresponding [`DrawCall`]. Useful for setting uniforms on a
    /// [`DrawCall`] only once, when handling the first instance.
    pub fn is_first_instance(&self) -> bool {
        match self {
            Area::InstanceRange(inst) => inst.instance_offset == 0,
            _ => false,
        }
    }

    /// Check if this [`Area`] points to valid data. Will return false for
    /// [`Area::Empty`], [`Area::All`], and if the [`View::redraw_id`] is different
    /// than the [`ViewArea::redraw_id`] or [`InstanceRangeArea::redraw_id`].
    ///
    /// TODO(JP): this will return [`true`] when the [`Area`] points to data that
    /// is not visible on the screen / when it's gone from the draw tree, or
    /// even when the original [`View`] object is gone. That's probably bad, or at
    /// least confusing. See [`Area`] for more information.
    fn is_valid(&self, cx: &Cx) -> bool {
        match self {
            Area::InstanceRange(inst) => {
                let cxview = &cx.views[inst.view_id];
                if cxview.redraw_id != inst.redraw_id {
                    return false;
                }
                true
            }
            Area::View(view_area) => {
                let cxview = &cx.views[view_area.view_id];
                if cxview.redraw_id != view_area.redraw_id {
                    return false;
                }
                true
            }
            _ => false,
        }
    }

    /// The scroll position of an [`Area`] is the cumulative offset of all scroll
    /// containers (compared to if there had not been scrolling at all).
    pub fn get_scroll_pos(&self, cx: &Cx) -> Vec2 {
        if !self.is_valid(cx) {
            panic!("get_scroll_pos was called for an invalid Area");
        }
        match self {
            Area::InstanceRange(inst) => {
                // Pull it directly out of the draw uniforms.
                let cxview = &cx.views[inst.view_id];
                let draw_call = &cxview.draw_calls[inst.draw_call_id];
                Vec2 { x: draw_call.draw_uniforms.draw_scroll_x, y: draw_call.draw_uniforms.draw_scroll_y }
            }
            Area::View(view_area) => {
                let cxview = &cx.views[view_area.view_id];
                cxview.parent_scroll
            }
            _ => unreachable!(),
        }
    }

    /// Get a mutable reference to a [`DrawCall`], so you can modify it. Only
    /// works for a valid [`Area::InstanceRange`].
    ///
    /// Note that any changes that you make modify the entire [`DrawCall`], not
    /// just the data/elements described by your specific [`Area`]!
    pub fn get_draw_call<'a>(&self, cx: &'a mut Cx) -> &'a mut DrawCall {
        if !self.is_valid(cx) {
            panic!("get_draw_call was called for an invalid Area");
        }
        match self {
            Area::InstanceRange(inst) => {
                let cxview = &mut cx.views[inst.view_id];
                &mut cxview.draw_calls[inst.draw_call_id]
            }
            _ => panic!("get_draw_call only supports Area::InstanceRange"),
        }
    }

    /// Returns the final screen [`Rect`] for the first instance of the [`Area`].
    ///
    /// TODO(JP): The "first instance" bit is confusing; in most (if not all)
    /// cases you'd want to get something that covers the entire [`Area`]. Maybe
    /// returning a single [`Rect`] isn't the right thing then, since the
    /// individual rectangles can be all over the place. We could return a [`Vec`]
    /// instead?
    ///
    /// TODO(JP): Specifically, this seems to return very weird values for
    /// [`crate::DrawText`] (only the first character, and offset to the bottom it seems).
    pub fn get_rect_for_first_instance(&self, cx: &Cx) -> Rect {
        if !self.is_valid(cx) {
            panic!("get_rect was called for an invalid Area");
        }
        return match self {
            Area::InstanceRange(inst) => {
                let cxview = &cx.views[inst.view_id];
                let draw_call = &cxview.draw_calls[inst.draw_call_id];
                if draw_call.in_many_instances {
                    panic!("get_rect called whilst in many instances");
                }
                assert!(!draw_call.instances.is_empty());
                let sh = &cx.shaders[draw_call.shader.shader_id];
                if let Some(rect_pos) = sh.mapping.rect_instance_props.rect_pos {
                    let x = draw_call.instances[inst.instance_offset + rect_pos + 0];
                    let y = draw_call.instances[inst.instance_offset + rect_pos + 1];
                    if let Some(rect_size) = sh.mapping.rect_instance_props.rect_size {
                        let w = draw_call.instances[inst.instance_offset + rect_size + 0];
                        let h = draw_call.instances[inst.instance_offset + rect_size + 1];
                        return draw_call.clip_and_scroll_rect(x, y, w, h);
                    }
                }
                Rect::default()
            }
            Area::View(view_area) => {
                let cxview = &cx.views[view_area.view_id];
                Rect { pos: cxview.rect.pos - cxview.parent_scroll, size: cxview.rect.size }
            }
            _ => unreachable!(),
        };
    }

    /// Given an absolute coordinate, get a coordinate relative to the first instance in this [`Area`].
    pub fn get_relative_to_first_instance(&self, cx: &Cx, abs: Vec2) -> Vec2 {
        if !self.is_valid(cx) {
            panic!("get_relative_to_first_instance was called for an invalid Area");
        }
        return match self {
            Area::InstanceRange(inst) => {
                let cxview = &cx.views[inst.view_id];
                let draw_call = &cxview.draw_calls[inst.draw_call_id];
                let sh = &cx.shaders[draw_call.shader.shader_id];
                if let Some(rect_pos) = sh.mapping.rect_instance_props.rect_pos {
                    let first_instance_x = draw_call.instances[inst.instance_offset + rect_pos + 0];
                    let first_instance_y = draw_call.instances[inst.instance_offset + rect_pos + 1];
                    return Vec2 {
                        x: abs.x - first_instance_x + draw_call.draw_uniforms.draw_scroll_x,
                        y: abs.y - first_instance_y + draw_call.draw_uniforms.draw_scroll_y,
                    };
                }
                abs
            }
            Area::View(view_area) => {
                let cxview = &cx.views[view_area.view_id];
                Vec2 {
                    x: abs.x - cxview.rect.pos.x + cxview.parent_scroll.x + cxview.unsnapped_scroll.x,
                    y: abs.y - cxview.rect.pos.y - cxview.parent_scroll.y + cxview.unsnapped_scroll.y,
                }
            }
            _ => unreachable!(),
        };
    }

    /// Get a [`DrawReadRef`] object, which contains information on how to extract
    /// the raw instance data from the [`Area::InstanceRange`].
    pub fn get_read_ref<'a>(&self, cx: &'a Cx, name_hash: StringHash, ty: Ty) -> Option<DrawReadRef<'a>> {
        if !self.is_valid(cx) {
            return None;
        }
        match self {
            Area::InstanceRange(inst) => {
                let cxview = &cx.views[inst.view_id];
                let draw_call = &cxview.draw_calls[inst.draw_call_id];
                let sh = &cx.shaders[draw_call.shader.shader_id];
                if let Some(prop_id) = sh.mapping.user_uniform_props.prop_map.get(&name_hash.hash) {
                    let prop = &sh.mapping.user_uniform_props.props[*prop_id];
                    if prop.ty != ty {
                        panic!("get_read_ref wrong uniform type, expected {:?} got: {:?}!", prop.ty, ty);
                    }
                    return Some(DrawReadRef { repeat: 1, stride: 0, buffer: &draw_call.user_uniforms[prop.offset..] });
                }
                if let Some(prop_id) = sh.mapping.instance_props.prop_map.get(&name_hash.hash) {
                    let prop = &sh.mapping.instance_props.props[*prop_id];
                    if prop.ty != ty {
                        panic!("get_read_ref wrong instance type, expected {:?} got: {:?}!", prop.ty, ty);
                    }
                    if inst.instance_count == 0 {
                        return None;
                    }
                    return Some(DrawReadRef {
                        repeat: inst.instance_count,
                        stride: sh.mapping.instance_props.total_slots,
                        buffer: &draw_call.instances[(inst.instance_offset + prop.offset)..],
                    });
                }
                panic!("get_read_ref property not found! {}", name_hash.string);
            }
            _ => None,
        }
    }

    /// Get a [`DrawWriteRef`] object, which contains information on how to modify
    /// the raw instance data in the [`Area::InstanceRange`]. Typically used through
    /// [`DrawInputType::write_shader_value`].
    pub fn get_write_ref<'a>(&self, cx: &'a mut Cx, name_hash: StringHash, ty: Ty) -> Option<DrawWriteRef<'a>> {
        if !self.is_valid(cx) {
            return None;
        }
        match self {
            Area::InstanceRange(inst) => {
                let cxview = &mut cx.views[inst.view_id];
                let draw_call = &mut cxview.draw_calls[inst.draw_call_id];
                let sh = &cx.shaders[draw_call.shader.shader_id];
                if let Some(prop_id) = sh.mapping.user_uniform_props.prop_map.get(&name_hash.hash) {
                    let prop = &sh.mapping.user_uniform_props.props[*prop_id];
                    if prop.ty != ty {
                        panic!("get_write_ref {} wrong uniform type, expected {:?} got: {:?}!", name_hash.string, prop.ty, ty);
                    }

                    cx.passes[cxview.pass_id].paint_dirty = true;
                    draw_call.uniforms_dirty = true;

                    return Some(DrawWriteRef { repeat: 1, stride: 0, buffer: &mut draw_call.user_uniforms[prop.offset..] });
                }
                if let Some(prop_id) = sh.mapping.instance_props.prop_map.get(&name_hash.hash) {
                    let prop = &sh.mapping.instance_props.props[*prop_id];
                    if prop.ty != ty {
                        panic!("get_write_ref {} wrong instance type, expected {:?} got: {:?}!", name_hash.string, prop.ty, ty);
                    }

                    cx.passes[cxview.pass_id].paint_dirty = true;
                    draw_call.instance_dirty = true;
                    if inst.instance_count == 0 {
                        return None;
                    }
                    return Some(DrawWriteRef {
                        repeat: inst.instance_count,
                        stride: sh.mapping.instance_props.total_slots,
                        buffer: &mut draw_call.instances[(inst.instance_offset + prop.offset)..],
                    });
                }
                panic!("get_write_ref {} property not found!", name_hash.string);
            }
            _ => None,
        }
    }

    /// Write a `texture_id` value into the the [`CxShader`] associated with this
    /// [`Area::InstanceRange`]. Should only be used through
    /// [`DrawInputType::write_shader_value`].
    pub(crate) fn write_texture_2d_id(&self, cx: &mut Cx, name_hash: StringHash, texture_id: usize) {
        match self {
            Area::InstanceRange(inst) => {
                let cxview = &mut cx.views[inst.view_id];
                let draw_call = &mut cxview.draw_calls[inst.draw_call_id];
                let sh = &cx.shaders[draw_call.shader.shader_id];
                for (index, prop) in sh.mapping.textures.iter().enumerate() {
                    if prop.name_hash == name_hash.hash {
                        draw_call.textures_2d[index] = texture_id as u32;
                        return;
                    }
                }
            }
            _ => (),
        }
        panic!("Cannot find texture2D prop {}", name_hash.string)
    }
}
