use crate::ResourceData;
use serde::{Deserialize, Serialize};
use smart_default::SmartDefault;

#[derive(Debug, Serialize, Deserialize, SmartDefault, PartialEq, Clone)]
#[serde(rename_all = "camelCase")]
pub struct Script {
    /// Is this used in DragNDrop? Hopefully not! that would get messy.
    pub is_dn_d: bool,

    /// Is this an autogenerated compatibility script?
    pub is_compatibility: bool,

    /// Common resource data.
    #[serde(flatten)]
    pub resource_data: ResourceData,

    /// Const id tag of the object, given by Gms2.
    pub resource_type: ConstGmScript,
}

#[derive(Debug, Copy, Serialize, Deserialize, SmartDefault, PartialEq, Eq, Clone)]
pub enum ConstGmScript {
    #[serde(rename = "GMScript")]
    #[default]
    Const,
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::{utils::TrailingCommaUtility, ResourceVersion, ViewPath, ViewPathLocation};
    use include_dir::{include_dir, Dir, DirEntry};
    use pretty_assertions::assert_eq;

    static ALL_SCRIPTS: Dir = include_dir!("$CARGO_MANIFEST_DIR/data/scripts");

    #[test]
    fn trivial_sprite_parsing() {
        let tcu = TrailingCommaUtility::new();

        for object_file in ALL_SCRIPTS.find("**/*.yy").unwrap() {
            if let DirEntry::File(file) = object_file {
                println!("parsing {}", file.path().display());
                let our_str = std::str::from_utf8(file.contents()).unwrap();
                let our_str = tcu.clear_trailing_comma(our_str);
                serde_json::from_str::<Script>(&our_str).unwrap();
            }
        }
    }

    #[test]
    fn deep_equality() {
        let script_raw = include_str!("../../data/scripts/CameraClass.yy");

        let script_parsed: Script =
            serde_json::from_str(&TrailingCommaUtility::clear_trailing_comma_once(script_raw))
                .unwrap();

        let script = Script {
            is_dn_d: false,
            is_compatibility: false,
            resource_data: ResourceData {
                parent: ViewPath {
                    name: "Camera".to_string(),
                    path: ViewPathLocation(
                        "folders/Scripts/Gameplay Systems/Camera.yy".to_string(),
                    ),
                },
                resource_version: ResourceVersion::default(),
                name: "CameraClass".to_string(),
                tags: vec![],
            },
            resource_type: ConstGmScript::Const,
        };

        assert_eq!(script_parsed, script);
    }
}
