use anyhow::Result;
use async_trait::async_trait;
use regex::RegexSet;
use std::{
    path::{Path, PathBuf},
    time::Duration,
};
use tokio::fs;
use url::Url;

pub mod client;
pub use client::Client;
pub mod extractors;

pub type HttpClient = reqwest::Client;

pub enum Info {
    Video {
        title: String,
        duration: Duration,
        tags: Vec<String>,
        description: Option<String>,
        author: Option<String>,
        url: Url, // Yeah this is bad, definitly needs a struct with multiple quality/resolution
    },
    Image {
        width: Option<u32>,
        height: Option<u32>,
        url: Url,
    },
}

impl Info {
    pub async fn download(self, http_client: &HttpClient, path: &PathBuf) -> Result<()> {
        match self {
            Info::Video { url, .. } => {
                let resp = http_client.get(url).send().await?.bytes().await?;
                fs::write(path.join("video.mp4"), resp).await?;
            }
            Info::Image { .. } => {}
        }

        Ok(())
    }
}

#[async_trait]
pub trait Extractor {
    fn url_tester(&self) -> RegexSet;
    async fn get_info(&self, http_client: &HttpClient, url: &str) -> Result<Info>;
}

/// A struct containing vital information about a comic.
#[derive(Debug)]
pub struct ImageAlbum {
    /// The authors of the comic
    pub authors: Vec<String>,
    pub categories: Vec<String>,
    /// The characters appearing in the comic.
    pub characters: Vec<String>,
    pub cover: Page,
    /// The unique identifier of the comic.
    ///
    /// This is guaranteed to be unique only on the original site,
    /// as such it should always be used in conjunction with the site property.
    pub id: String,
    pub groups: Vec<String>,
    /// The language this comic is avaible.
    ///
    /// This is usually one but it is a vec for api reason.
    pub languages: Vec<String>, // TODO: Explain why it's a vec in detail
    pub pages: Vec<Page>,
    /// The site this comic was downloaded from
    pub site: String,
    pub tags: Vec<String>,
    pub title: String,
    /// Whether or not the comic was translated from another language.
    pub translated: bool,
    /// Upload date in UNIX time-stamp format.
    pub upload_date: f64,
}

#[derive(Debug)]
pub struct Page {
    pub file_name: String,
    pub heigth: Option<u32>,
    pub url: Url,
    pub width: Option<u32>,
}

impl ImageAlbum {
    pub async fn download(&self) -> Result<()> {
        for page in self.pages.iter() {
            let path = format!("temp/{}/{}", self.site, self.id);
            println!("{}", &path);

            fs::create_dir_all(&path).await?;

            let full_path = format!("{}/{}", &path, page.file_name);
            let full_path = Path::new(&full_path);

            println!("{:#?}", full_path);

            let image = reqwest::get(page.url.to_owned()).await?.bytes().await?;
            fs::write(full_path, image).await?;
        }

        Ok(())
    }
}
