use std::path::PathBuf;

use anyhow::{bail, Result};

use crate::{
    extractors::{hentai_foundry::HentaiFoundry, youtube::Youtube},
    Extractor, HttpClient, Info,
};

pub struct Client {
    extractors: Vec<Box<dyn Extractor>>,
    http_client: HttpClient,
}

impl Client {
    pub fn builder() -> ClientBuilder {
        ClientBuilder::new()
    }

    fn choose_extractor(&self, url: &str) -> Option<&Box<dyn Extractor>> {
        let mut extractor: Option<&Box<dyn Extractor>> = None;

        for e in &self.extractors {
            if e.url_tester().is_match(url) {
                extractor = Some(e);
                break;
            }
        }

        extractor
    }

    pub fn is_supported(&self, url: &str) -> bool {
        self.choose_extractor(url).is_some()
    }

    pub async fn get_info(&self, url: &str) -> Result<Info> {
        if let Some(extractor) = self.choose_extractor(url) {
            return Ok(extractor.get_info(&self.http_client, url).await?);
        } else {
            bail!("No extractor found")
        }
    }

    pub async fn download(&self, path: &PathBuf, url: &str) -> Result<()> {
        let info = self.get_info(url).await?;
        info.download(&self.http_client, path).await?;
        Ok(())
    }
}

pub struct ClientBuilder {
    extractors: Vec<Box<dyn Extractor>>,
    http_client: HttpClient,
}

impl ClientBuilder {
    pub fn new() -> ClientBuilder {
        let http_client = HttpClient::builder()
            .cookie_store(true)
            .build()
            .expect("Client::new()");
        let extractors: Vec<Box<dyn Extractor>> = vec![Box::new(Youtube), Box::new(HentaiFoundry)];
        ClientBuilder {
            extractors,
            http_client,
        }
    }

    pub fn build(self) -> Result<Client> {
        Ok(Client {
            extractors: self.extractors,
            http_client: self.http_client,
        })
    }

    pub fn http_client(mut self, http_client: HttpClient) -> ClientBuilder {
        self.http_client = http_client;
        self
    }
}

impl Default for ClientBuilder {
    fn default() -> Self {
        Self::new()
    }
}
