use ureq::{Agent, SerdeValue, Response, Error};
use serde::{Deserialize, Serialize, de::DeserializeOwned};
use chrono::{NaiveDateTime, DateTime, Utc};
use crate::structs::oanda_instrument::OandaInstrument;

pub struct Oanda {
    pub agent:Agent,
    pub account:String,
    pub token:String
}

#[derive(Serialize, Deserialize, Debug )]
pub struct OandaTrade {
    pub trades:Vec<OandaTradeDetail>,
}

#[derive(Serialize, Deserialize, Debug )]
pub struct OandaTradeDetail {
    #[serde(rename(serialize = "openTime", deserialize = "openTime"))]
    pub open_time: DateTime<Utc>,
}

#[derive(Serialize, Deserialize, Debug, Clone )]
pub struct OandaOrderTakeProfitStopLoss {
    pub price: String,
}

#[derive(Serialize, Deserialize, Debug, Clone )]
pub struct OandaOrderRequest {
    pub order: OandaOrderDetail,
}

#[derive(Serialize, Deserialize, Debug, Clone )]
pub struct OandaOrderDetail {
    #[serde(rename(serialize = "type", deserialize = "type"))]
    pub order_type: String,
    pub instrument: String,
    pub units: f64,
    #[serde(rename(serialize = "takeProfitOnFill", deserialize = "takeProfitOnFill"))]
    pub tp: OandaOrderTakeProfitStopLoss,
    #[serde(rename(serialize = "stopLossOnFill", deserialize = "stopLossOnFill"))]
    pub sl: OandaOrderTakeProfitStopLoss,
}

#[derive(Serialize, Deserialize, Debug )]
pub struct OandaResponseBasic {
    #[serde(rename(serialize = "lastTransactionID", deserialize = "lastTransactionID"))]
    pub last_transaction_id: String,
}


impl Oanda {
    pub fn call<T>(&self, url:&str) -> T
    where T:DeserializeOwned{
        let resp = self.agent
            .get(url)
            .set("Authorization", format!("Bearer {}",self.token).as_str())
            .call()
            .unwrap()
            .into_json::<T>()
            .unwrap();
        resp
    }
    pub fn post<T>(&self, url:&str, body:SerdeValue) -> T
        where T:DeserializeOwned{
        let resp = self.agent
            .post(url)
            .set("Authorization", format!("Bearer {}",self.token).as_str())
            .send_json(body)
            .unwrap()
            .into_json::<T>()
            .unwrap();
        resp
    }
    pub fn post_with_resp<T>(&self, url:&str, body:SerdeValue) -> Result<Response, Error>
        where T:DeserializeOwned{
        let resp = self.agent
            .post(url)
            .set("Authorization", format!("Bearer {}",self.token).as_str())
            .send_json(body);
        resp
    }
    pub fn getTrades(&self) -> OandaTrade {
        let url = format!("https://api-fxtrade.oanda.com/v3/accounts/{}/trades",self.account);
        self.call::<OandaTrade>(url.as_str())
    }
    pub fn getInstrumentLatest(&self, instrument:&str) -> OandaInstrument{
        let url = format!(
            "https://api-fxtrade.oanda.com/v3/instruments/{}/candles?alignmentTimezone=UTC&count=1&smooth=false&price=BAM",instrument);
        self.call::<OandaInstrument>(url.as_str())
    }
    pub fn order(&self, request:OandaOrderRequest){

        let url = format!(
            "https://api-fxtrade.oanda.com/v3/accounts/{}/orders",self.account);
        let request_map = serde_json::to_value(request).expect("could convert to hashmap");
        println!("{:?}",request_map);

        let resp = self.post_with_resp::<OandaResponseBasic>(url.as_str(), request_map);
        match resp {
            Ok(res) => {
                println!("{}",res.status())
            }
            Err(Error::Status(code,res)) => {
                println!("{:?}",res.into_string());
            }
            _ => {}
        }
    }
    pub fn hasTrades(&self) -> bool{
        !self.getTrades().trades.is_empty()
    }
}
