use crate::html_compiler;
use crate::settings_serializer;
use crate::template_filler;
use crate::template_finder;
use crate::video_serializer;

pub fn call() {
    let video_section = get_all_category_sections();
    let mut fillings = Vec::new();
    fillings.push(("{{video_section}}", video_section.as_str()));
    let template_index = template_finder::call("index.html");
    let index_page_content = template_filler::call(&template_index, fillings);
    html_compiler::call(index_page_content, "public/index.html");
}

fn get_all_category_sections() -> String {
    let sections = settings_serializer::call("config/config.txt");
    let mut section_content: String = "".to_string();
    for section_data in sections {
        let config_file = get_config_file_path(&section_data.value);
        let videos = get_videos_section_html(&config_file);
        let mut fillings = Vec::new();
        fillings.push(("{{title}}", section_data.key.as_str()));
        fillings.push(("{{videos}}", videos.as_str()));
        let section = template_filler::call("src/templates/category_section.html", fillings);
        section_content.push_str(&section);
    }
    section_content
}

fn get_config_file_path(config_filename: &str) -> String {
    let mut config_path: String = "config/".to_owned();
    config_path.push_str(config_filename);
    config_path
}

fn get_videos_section_html(file: &str) -> String {
    let mut html_content: String = "".to_owned();
    let video_serializer = video_serializer::Videos::new(file);
    let categories = settings_serializer::call("config/categories.txt");
    for year in video_serializer.ordered_years() {
        let mut year_fillings = Vec::new();
        let mut yearly_videos: String = "".to_owned();
        for video in video_serializer.select_by_year(year).videos {
            let mut fillings = Vec::new();
            let category = find_category_name(&categories, video.category_id);
            fillings.push(("{{title}}", video.title.as_str()));
            fillings.push(("{{youtube_id}}", video.youtube_id.as_str()));
            fillings.push(("{{category}}", category.as_str()));
            let video_html = template_filler::call("src/templates/youtube_video.html", fillings);
            yearly_videos.push_str(&video_html)
        }
        let year_string = year.to_string();
        year_fillings.push(("{{year}}", year_string.as_str()));
        year_fillings.push(("{{yearly_videos}}", yearly_videos.as_str()));
        let year_html = template_filler::call("src/templates/year_section.html", year_fillings);
        html_content.push_str(&year_html);
    }
    return html_content;
}

fn find_category_name(categories: &Vec<settings_serializer::Settings>, category_id: String) -> String {
    let matching_category = categories.iter().find( |s| s.key == category_id);
    match matching_category {
        Some(c) => return c.value.to_string(),
        None => return "".to_string(),
    }
}
