//! This module contains the `App` struct, which is used to bootstrap
//! a component in an isolated scope.

use std::ops::Deref;

use crate::html::{Component, NodeRef, Scope, Scoped};
use gloo_utils::document;
use std::rc::Rc;
use web_sys::Element;

/// An instance of an application.
#[derive(Debug)]
pub struct AppHandle<COMP: Component> {
    /// `Scope` holder
    pub(crate) scope: Scope<COMP>,
}

impl<COMP> AppHandle<COMP>
where
    COMP: Component,
{
    /// The main entry point of a Yew program which also allows passing properties. It works
    /// similarly to the `program` function in Elm. You should provide an initial model, `update`
    /// function which will update the state of the model and a `view` function which
    /// will render the model to a virtual DOM tree.
    pub(crate) fn mount_with_props(element: Element, props: Rc<COMP::Properties>) -> Self {
        clear_element(&element);
        let app = Self {
            scope: Scope::new(None),
        };
        app.scope
            .mount_in_place(element, NodeRef::default(), NodeRef::default(), props);

        app
    }

    /// Alternative to `mount_with_props` which replaces the body element with a component which
    /// has a body element at the root of the HTML generated by its `view` method. Use this method
    /// when you need to manipulate the body element. For example, adding/removing app-wide
    /// CSS classes of the body element.
    pub(crate) fn mount_as_body_with_props(props: Rc<COMP::Properties>) -> Self {
        let html_element = document().document_element().unwrap();
        let body_element = document().body().expect("no body node found");
        html_element
            .remove_child(&body_element)
            .expect("can't remove body child");

        Self::mount_with_props(html_element, props)
    }

    /// Schedule the app for destruction
    pub fn destroy(mut self) {
        self.scope.destroy()
    }
}

impl<COMP> Deref for AppHandle<COMP>
where
    COMP: Component,
{
    type Target = Scope<COMP>;

    fn deref(&self) -> &Self::Target {
        &self.scope
    }
}

/// Removes anything from the given element.
fn clear_element(element: &Element) {
    while let Some(child) = element.last_child() {
        element.remove_child(&child).expect("can't remove a child");
    }
}
