use crate::common::{calc_height, calc_width};
use crate::*;
use css_style::{box_align::*, flexbox::*, prelude::*, Display, FlexDirection};
use yew::prelude::*;

// ======================== Column Component ========================

/// Column properties.
///
/// Here you can find all properties that can be used with
/// [Column](crate::column::Column) component.
#[derive(Properties, Clone, PartialEq)]
pub struct Props {
    pub children: Children,

    /// Fill the available space vertically.
    ///
    /// The default is `true`
    #[prop_or(true)]
    pub fill: bool,

    /// Fill the available space horizontally.
    ///
    /// The default is `false`
    #[prop_or(false)]
    pub cross_fill: bool,

    /// Expand factor used to expand this column in direction relevant to it's
    /// parent layout direction.
    ///
    /// When the parent is `Row` it will expand horizontally, when the parent is
    /// `Column` it will expand vertically.
    ///
    /// Note: This only works when this `Column` inside another layout (e.g. Row/Column).
    ///
    /// The default is `None`
    #[prop_or_default]
    pub expand_by: Option<f32>,

    /// Shrink factor used to shrink this column in direction relevant to it's
    /// parent layout direction when needed.
    ///
    /// When the parent is `Row` it will shrink horizontally, when the parent is
    /// `Column` it will shrink vertically.
    ///
    /// Note: This only works when this `Column` inside another layout (e.g. Row/Column).
    ///
    /// The default is `None`
    #[prop_or_default]
    pub shrink_by: Option<f32>,

    /// Make this layout inline
    ///
    /// The default is `false`
    #[prop_or(false)]
    pub inline: bool,

    /// Reverse the order of the children
    ///
    /// The default is `false`
    #[prop_or(false)]
    pub reverse: bool,

    /// Wrap into another column when there is no more vertical space.
    ///
    /// The default is `false`
    #[prop_or(false)]
    pub wrap: bool,

    /// Align the children inside this column in main direction (horizontally).
    ///
    /// The default is `None`
    #[prop_or_default]
    pub align: Option<Align>,

    /// Align the children inside this column in the cross direction
    /// (vertically).
    ///
    /// The default is `None`
    #[prop_or_default]
    pub cross_align: Option<CrossAlign>,

    /// Align this column when it's inside another layout, the alignment
    /// direction is relevant to the parent layout direction
    ///
    /// When the parent is `Row` it will align horizontally, when the parent is
    /// `Column` it will align vertically.
    ///
    /// Note: This only works when this `Column` inside another layout (e.g. Row/Column).
    ///
    /// The default is `None`
    #[prop_or_default]
    pub align_self: Option<AlignSelf>,

    /// Gap between children.
    ///
    /// This take `Gap` value, which can take either one value that defines the
    /// gap for both the columns (if there is any) and rows, or two values one
    /// for rows and the other for columns.
    ///
    /// The default is `None`
    #[prop_or_default]
    pub gap: Option<Gap>,

    /// Reverse columns if there is more than one column within this `Column`.
    ///
    /// This only works when used with `wrap=true`.
    ///
    /// The default is `false`
    #[prop_or(false)]
    pub reverse_columns: bool,

    /// Align columns in the cross direction (horizontally) if there is more
    /// than one column within this `Column`.
    ///
    /// The default is `None`
    #[prop_or_default]
    pub align_columns: Option<AlignColumns>,

    /// Scroll bars for this Column
    ///
    /// By default any child that get oversized will be hidden/cutted off.
    /// Change this property if you like to make the content scrollable.
    ///
    /// The default is `Scroll::off()`
    #[prop_or(Scroll::off())]
    pub scroll: Scroll,

    /// Padding for the `Column`
    ///
    /// The default is `None`
    #[prop_or_default]
    pub padding: Option<Padding>,

    /// Margin for the `Column`
    ///
    /// The default is `None`
    #[prop_or_default]
    pub margin: Option<Margin>,
}

/// Column layout component.
///
/// See [column properties docs](crate::column::Props) for more details on
/// how to use them.
///
/// # Usage
///
/// ```rust
/// use yew_layout::{Column, Align};
/// # use yew::prelude::*;
///
/// # fn view() -> Html {
/// html! {
///     <Column align={ Align::Center } cross_fill=true wrap=true>
///         { "Column children.." }
///     </Column>
/// }
/// # }
/// ```
#[function_component(Column)]
pub fn column(props: &Props) -> Html {
    let style = Style::default()
        .display(match props.inline {
            true => Display::InlineFlex,
            false => Display::Flex,
        })
        .and_size(|mut size| {
            if props.fill && !props.inline {
                let height = calc_height(props.padding.as_ref(), props.margin.as_ref());
                size = size.height(height);
            }
            if props.cross_fill && !props.inline {
                let width = calc_width(props.padding.as_ref(), props.margin.as_ref());
                size = size.width(width);
            }
            size
        })
        .flex_direction(match props.reverse {
            true => FlexDirection::ColumnReverse,
            false => FlexDirection::Column,
        })
        .try_flex_wrap(match (props.wrap, props.reverse_columns) {
            (true, true) => Some(Wrap::WrapReverse),
            (true, false) => Some(Wrap::Wrap),
            _ => None,
        })
        .try_justify_content(props.align)
        .try_align_items(props.cross_align)
        .try_align_content(props.align_columns)
        .try_align_self(props.align_self)
        .try_flex_grow(props.expand_by)
        .try_flex_shrink(props.shrink_by)
        .try_gap(props.gap.clone())
        .try_padding(props.padding.clone())
        .try_margin(props.margin.clone())
        .merge(props.scroll);

    html! {
        <div class="yew-layout-column" style={ style.to_string() }>
            { props.children.clone() }
        </div>
    }
}
