#![doc = include_str!("../README.md")]

use css_style::{
    box_align::*, flexbox::*, prelude::*, Display, FlexDirection, Gap, Margin, Padding,
};
use yew::prelude::*;

pub type Align = JustifyContent;
pub type CrossAlign = AlignItems;
pub type AlignColumns = AlignContent;
pub type AlignRows = AlignContent;
pub use css_style::box_align::AlignSelf;

// ======================== Row Component ========================

/// Column properties.
///
/// Here you can find all properties that can be used with
/// [Column](crate::Column) component.
#[derive(Properties, Clone, PartialEq)]
pub struct ColumnProps {
    pub children: Children,
    /// Fill the available sapce vertically.
    ///
    /// The default is `true`
    #[prop_or(true)]
    pub fill: bool,
    /// Fill the available sapce horizontally.
    ///
    /// The default is `false`
    #[prop_or(false)]
    pub cross_fill: bool,
    /// Expand factor used to exapnd this column in direction relevant to it's
    /// parent layout direction.
    ///
    /// When the parent is `Row` it will expand horizontally, when the parent is
    /// `Column` it will expand vertically.
    ///
    /// Note: This only works when this `Column` inside another flex layout.
    ///
    /// The default is `None`
    #[prop_or_default]
    pub expand_by: Option<f32>,
    /// Shrink factor used to shrink this column in direction relevant to it's
    /// parent layout direction when needed.
    ///
    /// When the parent is `Row` it will shrink horizontally, when the parent is
    /// `Column` it will shrink vertically.
    ///
    /// Note: This only works when this `Column` inside another flex layout.
    ///
    /// The default is `None`
    #[prop_or_default]
    pub shrink_by: Option<f32>,
    /// Make this layout inline
    ///
    /// The default is `false`
    #[prop_or(false)]
    pub inline: bool,
    /// Reverse the order of the children
    ///
    /// The default is `false`
    #[prop_or(false)]
    pub reverse: bool,
    /// Wrap into another column when there is no more vertical space.
    ///
    /// The default is `false`
    #[prop_or(false)]
    pub wrap: bool,
    /// Align the children inside this column in main direction (horizontally).
    ///
    /// The default is `None`
    #[prop_or_default]
    pub align: Option<Align>,
    /// Align the children inside this column in the cross direction
    /// (vertically).
    ///
    /// The default is `None`
    #[prop_or_default]
    pub cross_align: Option<CrossAlign>,
    /// Align this column when it's inside another layout, the alignment
    /// direction is relevant to the parent layout direction
    ///
    /// When the parent is `Row` it will align horizontally, when the parent is
    /// `Column` it will align vertically.
    ///
    /// Note: This only works when this `Column` inside another layout.
    ///
    /// The default is `None`
    #[prop_or_default]
    pub align_self: Option<AlignSelf>,
    /// Gap between children.
    ///
    /// this take `css_style::Gap` value, which can take either one value that
    /// defines the gap for both the columns (if there is any) and rows, or two
    /// values one for rows and the other for columns.
    ///
    /// The default is `None`
    #[prop_or_default]
    pub gap: Option<Gap>,
    /// Reverse columns if there is more than one column within this `Column`.
    ///
    /// The default is `false`
    #[prop_or(false)]
    pub reverse_columns: bool,
    /// Align columns in the corss direction (horizontally) if there is more
    /// than one column within this `Column`.
    ///
    /// The default is `None`
    #[prop_or_default]
    pub align_columns: Option<AlignColumns>,
    /// Padding for the `Column`
    ///
    /// The default is `None`
    #[prop_or_default]
    pub padding: Option<Padding>,
    /// Margin for the `Column`
    ///
    /// The default is `None`
    #[prop_or_default]
    pub margin: Option<Margin>,
}

/// Column layout component.
///
/// See [column properties docs](crate::ColumnProps) for more details
/// on how to use them.
///
/// # Usage
///
/// ```rust
/// use yew_layout::{Column, Align};
/// # use yew::prelude::*;
///
/// # fn view() -> Html {
/// html! {
///     <Column align={ Align::Center } cross_fill=true wrap=true>
///         { "Column children.." }
///     </Column>
/// }
/// # }
/// ```
#[function_component(Column)]
pub fn column(props: &ColumnProps) -> Html {
    let style = Style::default()
        .display(match props.inline {
            true => Display::InlineFlex,
            false => Display::Flex,
        })
        .and_size(|mut size| {
            if props.fill && !props.inline {
                size = size.height(1.0);
            }
            if props.cross_fill && !props.inline {
                size = size.width(1.0);
            }
            size
        })
        .flex_direction(match props.reverse {
            true => FlexDirection::ColumnReverse,
            false => FlexDirection::Column,
        })
        .try_flex_wrap(match (props.wrap, props.reverse_columns) {
            (true, true) => Some(Wrap::WrapReverse),
            (true, false) => Some(Wrap::Wrap),
            _ => None,
        })
        .try_justify_content(props.align)
        .try_align_items(props.cross_align)
        .try_align_content(props.align_columns)
        .try_align_self(props.align_self)
        .try_flex_grow(props.expand_by)
        .try_flex_shrink(props.shrink_by)
        .try_gap(props.gap.clone())
        .try_padding(props.padding.clone())
        .try_margin(props.margin.clone());

    html! {
        <div class="yew-layout-column" style={ style.to_string() }>
            { props.children.clone() }
        </div>
    }
}

// ======================== Row Component ========================

/// Row properties.
///
/// Here you can find all properties that can be used with [Row](crate::Row)
/// component.
#[derive(Properties, Clone, PartialEq)]
pub struct RowProps {
    pub children: Children,
    /// Fill the available sapce horizontally.
    ///
    /// The default is `true`
    #[prop_or(true)]
    pub fill: bool,
    /// Fill the available sapce vertically.
    ///
    /// The default is `false`
    #[prop_or(false)]
    pub cross_fill: bool,
    /// Expand factor used to exapnd this row in direction relevant to it's
    /// parent layout direction.
    ///
    /// When the parent is `Row` it will expand horizontally, when the parent is
    /// `Column` it will expand vertically.
    ///
    /// Note: This only works when this `Row` inside another flex layout.
    ///
    /// The default is `None`
    #[prop_or_default]
    pub expand_by: Option<f32>,
    /// Shrink factor used to shrink this row in direction relevant to it's
    /// parent layout direction when needed.
    ///
    /// When the parent is `Row` it will shrink horizontally, when the parent is
    /// `Column` it will shrink vertically.
    ///
    /// Note: This only works when this `Row` inside another flex layout.
    ///
    /// The default is `None`
    #[prop_or_default]
    pub shrink_by: Option<f32>,
    /// Make this layout inline
    ///
    /// The default is `false`
    #[prop_or(false)]
    pub inline: bool,
    /// Reverse the order of the children
    ///
    /// The default is `false`
    #[prop_or(false)]
    pub reverse: bool,
    /// Wrap into another row when there is no more horizontal space.
    ///
    /// The default is `false`
    #[prop_or(false)]
    pub wrap: bool,
    /// Align the children inside this row in main direction (vertically).
    ///
    /// The default is `None`
    #[prop_or_default]
    pub align: Option<Align>,
    /// Align the children inside this row in the cross direction
    /// (horizontally).
    ///
    /// The default is `None`
    #[prop_or_default]
    pub cross_align: Option<CrossAlign>,
    /// Align this row when it's inside another layout, the alignment direction
    /// is relevant to the parent layout direction
    ///
    /// When the parent is `Row` it will align horizontally, when the parent is
    /// `Column` it will align vertically.
    ///
    /// Note: This only works when this `Row` inside another layout.
    ///
    /// The default is `None`
    #[prop_or_default]
    pub align_self: Option<AlignSelf>,
    /// Gap between children.
    ///
    /// this take `css_style::Gap` value, which can take either one value that
    /// defines the gap for both the columns and rows (if there is any), or two
    /// values one for rows and the other for columns.
    ///
    /// The default is `None`
    #[prop_or_default]
    pub gap: Option<Gap>,
    /// Reverse rows if there is more than one column within this `Column`.
    ///
    /// The default is `false`
    #[prop_or(false)]
    pub reverse_rows: bool,
    /// Align rows in the corss direction (vertically) if there is more than one
    /// row within this `Row`.
    ///
    /// The default is `None`
    #[prop_or_default]
    pub align_rows: Option<AlignRows>,
    /// Padding for the `Row`
    ///
    /// The default is `None`
    #[prop_or_default]
    pub padding: Option<Padding>,
    /// Margin for the `Row`
    ///
    /// The default is `None`
    #[prop_or_default]
    pub margin: Option<Margin>,
}

/// Row layout component.
///
/// See [row properties docs](crate::RowProps) for more details on how to use
/// them.
///
/// # Usage
///
/// ```rust
/// use yew_layout::{Row, Align};
/// # use yew::prelude::*;
///
/// # fn view() -> Html {
/// html! {
///     <Row align={ Align::Center } cross_fill=true wrap=true>
///         { "Row children.." }
///     </Row>
/// }
/// # }
/// ```
#[function_component(Row)]
pub fn row(props: &RowProps) -> Html {
    let style = Style::default()
        .display(match props.inline {
            true => Display::InlineFlex,
            false => Display::Flex,
        })
        .and_size(|mut size| {
            if props.fill && !props.inline {
                size = size.width(1.0);
            }
            if props.cross_fill && !props.inline {
                size = size.height(1.0);
            }
            size
        })
        .flex_direction(match props.reverse {
            true => FlexDirection::RowReverse,
            false => FlexDirection::Row,
        })
        .try_flex_wrap(match (props.wrap, props.reverse_rows) {
            (true, true) => Some(Wrap::WrapReverse),
            (true, false) => Some(Wrap::Wrap),
            _ => None,
        })
        .try_justify_content(props.align)
        .try_align_items(props.cross_align)
        .try_align_content(props.align_rows)
        .try_align_self(props.align_self)
        .try_flex_grow(props.expand_by)
        .try_flex_shrink(props.shrink_by)
        .try_gap(props.gap.clone())
        .try_padding(props.padding.clone())
        .try_margin(props.margin.clone());

    html! {
        <div class="yew-layout-row" style={ style.to_string() }>
            { props.children.clone() }
        </div>
    }
}
