use lazy_static::lazy_static;

use tokio::fs;
use tokio::io::{AsyncReadExt, AsyncWriteExt};
use tokio::net::{TcpListener, TcpStream, UnixListener, UnixStream};
use tokio::signal::unix::{signal, SignalKind};
use tokio::sync::RwLock;

use std::vec::Vec;

use yedb::common::JSONRpcRequest;
use yedb::*;

use rmp_serde;
use serde_json;

use serde_json::Value;

use log::LevelFilter;
use syslog::{BasicLogger, Facility, Formatter3164};

use chrono::prelude::*;
use colored::Colorize;

use clap::Clap;

use log::{debug, error, info, Level, Metadata, Record};

lazy_static! {
    pub static ref DBCELL: RwLock<Database> = RwLock::new(yedb::Database::new());
}

struct SimpleLogger;

#[derive(Debug, Eq, PartialEq)]
pub enum YedbServerErrorKind {
    Critical,
    #[allow(dead_code)]
    Other,
}

#[macro_export]
macro_rules! parse_jsonrpc_request_param {
    ($r:expr, $k:expr, $p:path) => {
        match $r.params.get($k) {
            Some(v) => match v {
                $p(v) => Some(v),
                _ => None,
            },
            None => None,
        }
    };
}

#[macro_export]
macro_rules! encode_jsonrpc_response {
    ($v:expr) => {
        match rmp_serde::to_vec_named(&$v) {
            Ok(v) => v,
            Err(e) => {
                error!("Response encode error {}", e);
                return Err(YedbServerErrorKind::Critical);
            }
        }
    };
}

#[macro_export]
macro_rules! parse_result_for_jsonrpc {
    ($v:expr, $r:expr) => {
        match $v {
            Ok(value) => encode_jsonrpc_response!($r.respond(value)),
            Err(e) => encode_jsonrpc_response!($r.error(e)),
        }
    };
}

#[macro_export]
macro_rules! parse_result_for_jsonrpc_ok {
    ($v:expr, $r:expr) => {
        match $v {
            Ok(_) => encode_jsonrpc_response!($r.respond_ok()),
            Err(e) => encode_jsonrpc_response!($r.error(e)),
        }
    };
}

impl log::Log for SimpleLogger {
    fn enabled(&self, metadata: &Metadata) -> bool {
        metadata.level() <= Level::Debug
    }

    fn log(&self, record: &Record) {
        if self.enabled(record.metadata()) {
            let s = format!(
                "{}  {}",
                Local::now().to_rfc3339_opts(SecondsFormat::Secs, false),
                record.args()
            );
            println!(
                "{}",
                match record.level() {
                    Level::Debug => s.dimmed(),
                    Level::Warn => s.yellow().bold(),
                    Level::Error => s.red(),
                    _ => s.normal(),
                }
            );
        }
    }

    fn flush(&self) {}
}

static LOGGER: SimpleLogger = SimpleLogger;

enum Listener {
    TCP(TcpListener),
    UNIX(UnixListener),
}

pub struct ServerData {
    pub pid_path: String,
    pub socket_path: Option<String>,
}

lazy_static! {
    pub static ref SDATA: RwLock<ServerData> = RwLock::new(ServerData {
        pid_path: String::new(),
        socket_path: None
    });
}

macro_rules! handle_term {
    ($s:expr) => {
        loop {
            $s.recv().await;
            info!("terminating");
            let mut dbobj = DBCELL.write().await;
            if dbobj.is_open() {
                dbobj.close().unwrap();
            }
            let s = SDATA.read().await;
            let _ = std::fs::remove_file(&s.pid_path);
            match s.socket_path {
                Some(ref f) => {
                    let _ = std::fs::remove_file(f);
                }
                None => {}
            };
            std::process::exit(0);
        }
    };
}

#[derive(Clap)]
struct Opts {
    #[clap(about = "database directory")]
    path: String,
    #[clap(short = 'B', long = "bind", default_value = "tcp://127.0.0.1:8870")]
    bind: String,
    #[clap(long, default_value = "/tmp/yedb-server.pid")]
    pid_file: String,
    #[clap(long)]
    lock_path: Option<String>,
    #[clap(long, default_value = "json")]
    default_fmt: SerializationFormat,
    #[clap(short = 'v', about = "Verbose logging")]
    verbose: bool,
    #[clap(long)]
    disable_auto_flush: bool,
    #[clap(long)]
    disable_auto_repair: bool,
    #[clap(long, default_value = "1000")]
    cache_size: usize,
    #[clap(long, default_value = "0")]
    auto_bak: u64,
    #[clap(long, default_value = "2")]
    workers: usize,
}

enum SerializationFormat {
    Json,
    Yaml,
    Msgpack,
    Cbor,
}

impl std::str::FromStr for SerializationFormat {
    type Err = Error;
    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s {
            "json" => Ok(Self::Json),
            "yaml" => Ok(Self::Yaml),
            "msgpack" => Ok(Self::Msgpack),
            "cbor" => Ok(Self::Cbor),
            _ => Err(Error::new(
                ErrorKind::UnsupportedFormat,
                format!("{}, valid values: json|yaml|msgpack|cbor", s),
            )),
        }
    }
}

impl SerializationFormat {
    pub fn to_string(&self) -> String {
        use SerializationFormat::*;
        match self {
            Json => "json".to_owned(),
            Msgpack => "msgpack".to_owned(),
            Cbor => "cbor".to_owned(),
            Yaml => "yaml".to_owned(),
        }
    }
}

fn main() {
    let opts: Opts = Opts::parse();
    if opts.verbose {
        log::set_logger(&LOGGER)
            .map(|()| log::set_max_level(LevelFilter::Debug))
            .unwrap();
    } else {
        let formatter = Formatter3164 {
            facility: Facility::LOG_USER,
            hostname: None,
            process: "yedb-server".into(),
            pid: 0,
        };
        let logger = syslog::unix(formatter).expect("could not connect to syslog");
        log::set_boxed_logger(Box::new(BasicLogger::new(logger)))
            .map(|()| log::set_max_level(LevelFilter::Info))
            .unwrap();
    }
    let rt = tokio::runtime::Builder::new_multi_thread()
        .worker_threads(opts.workers)
        .enable_all()
        .build()
        .unwrap();
    rt.block_on(async move {
        let mut dbobj = DBCELL.write().await;
        dbobj.set_db_path(&opts.path).unwrap();
        match opts.lock_path {
            Some(path) => dbobj.set_lock_path(&path).unwrap(),
            None => {}
        }
        dbobj.auto_flush = !opts.disable_auto_flush;
        dbobj.auto_repair = !opts.disable_auto_repair;
        dbobj
            .set_default_fmt(&opts.default_fmt.to_string(), true)
            .unwrap();
        dbobj.set_cache_size(opts.cache_size);
        debug!("Auto bak: {}", opts.auto_bak);
        dbobj.auto_bak = opts.auto_bak as u64;
        debug!("Workers: {}", opts.workers);
        drop(dbobj);
        run_server(&opts.bind, &opts.pid_file).await;
    });
}

async fn run_server(bind_to: &String, pidfile: &String) {
    let mut dbobj = DBCELL.write().await;
    let _ = fs::remove_file(&bind_to).await;
    let listener = match bind_to.starts_with("tcp://") {
        true => Listener::TCP(TcpListener::bind(&bind_to[6..]).await.unwrap()),
        false => {
            let _ = fs::remove_file(&bind_to).await;
            SDATA.write().await.socket_path = Some(bind_to.to_owned());
            Listener::UNIX(UnixListener::bind(&bind_to).unwrap())
        }
    };
    let server_info = dbobj.open().unwrap();
    debug!("Engine version: {}", server_info.version);
    let dbinfo = dbobj.info().unwrap();
    debug!("Library: {}, version {}", dbinfo.server.0, dbinfo.server.1);
    debug!("Database: {}, format: {}", dbinfo.path, dbinfo.fmt);
    tokio::spawn(async move { handle_term!(signal(SignalKind::interrupt()).unwrap()) });
    tokio::spawn(async move { handle_term!(signal(SignalKind::terminate()).unwrap()) });
    {
        let mut f = fs::File::create(&pidfile).await.unwrap();
        f.write_all(std::process::id().to_string().as_bytes())
            .await
            .unwrap();
        SDATA.write().await.pid_path = pidfile.clone();
    }
    drop(dbobj);
    info!("Started, listening at {}", bind_to);
    loop {
        match listener {
            Listener::UNIX(ref socket) => match socket.accept().await {
                Ok((mut stream, _addr)) => {
                    tokio::spawn(async move {
                        unix_worker(&mut stream).await;
                    });
                }
                Err(e) => {
                    error!("API connect error {}", e);
                }
            },
            Listener::TCP(ref socket) => match socket.accept().await {
                Ok((mut stream, _addr)) => {
                    stream.set_nodelay(true).unwrap();
                    tokio::spawn(async move {
                        tcp_worker(&mut stream).await;
                    });
                }
                Err(e) => {
                    error!("API connect error {}", e);
                }
            },
        };
    }
}

macro_rules! parse_request_meta {
    ($s:expr, $b:expr, $l:expr) => {
        match $s.read_exact(&mut $b).await {
            Ok(_) => $l = u32::from_le_bytes([$b[2], $b[3], $b[4], $b[5]]) as usize,
            Err(ref e) if e.kind() == std::io::ErrorKind::UnexpectedEof => {
                break;
            }
            Err(e) => {
                debug!("API read error {}", e);
                break;
            }
        };
        if $b[0] != yedb::ENGINE_VERSION || $b[1] != 2 || $l == 0 {
            debug!("Invalid packet");
            break;
        };
    };
}

macro_rules! handle_request {
    ($s:expr, $b:expr) => {
        match $s.read_exact(&mut $b).await {
            Ok(_) => match process_request(&$b).await {
                Ok(response_buf) => {
                    let mut response_frame = vec![yedb::ENGINE_VERSION, 2u8];
                    response_frame.extend(&(response_buf.len() as u32).to_le_bytes());
                    response_frame.extend(&response_buf);
                    match $s.write_all(&response_frame).await {
                        Ok(_) => {}
                        Err(e) => {
                            debug!("API write error {}", e);
                            break;
                        }
                    };
                }
                Err(e) if e == YedbServerErrorKind::Critical => break,
                Err(_) => continue,
            },
            Err(e) => {
                error!("Socket error {}", e);
                break;
            }
        }
    };
}

async fn unix_worker(stream: &mut UnixStream) {
    loop {
        let mut buf = [0_u8; 6];
        let frame_len: usize;
        parse_request_meta!(stream, buf, frame_len);
        let mut buf: Vec<u8> = vec![0; frame_len];

        handle_request!(stream, buf);
    }
}

async fn tcp_worker(stream: &mut TcpStream) {
    loop {
        let mut buf = [0_u8; 6];
        let frame_len: usize;
        parse_request_meta!(stream, buf, frame_len);
        let mut buf: Vec<u8> = vec![0; frame_len];

        handle_request!(stream, buf);
    }
}

async fn process_request(buf: &[u8]) -> Result<Vec<u8>, YedbServerErrorKind> {
    let request: JSONRpcRequest = match rmp_serde::from_read_ref(&buf) {
        Ok(v) => v,
        Err(e) => {
            error!("API decode error {}", e);
            return Err(YedbServerErrorKind::Critical);
        }
    };
    if !request.is_valid() {
        error!("API error: invalid request");
        return Err(YedbServerErrorKind::Critical);
    }
    Ok(match request.method.as_str() {
        "test" => {
            debug!("API request: test");
            match request.params_valid(vec![]) {
                true => encode_jsonrpc_response!(request.respond(yedb::ServerInfo::new())),
                false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            }
        }
        "info" => {
            debug!("API request: info");
            match request.params_valid(vec![]) {
                true => match DBCELL.write().await.info() {
                    Ok(v) => encode_jsonrpc_response!(request.respond(v)),
                    Err(e) => encode_jsonrpc_response!(request.error(e)),
                },
                false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            }
        }
        "server_set" => match request.params_valid(vec!["name", "value"]) {
            true => match parse_jsonrpc_request_param!(request, "name", Value::String) {
                Some(name) => {
                    let value = request.params.get("value").unwrap();
                    debug!("API request: server_set {}={}", name, value);
                    parse_result_for_jsonrpc_ok!(
                        DBCELL.write().await.server_set(&name, value.clone()),
                        request
                    )
                }
                None => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            },
            false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
        },
        "key_get" => match request.params_valid(vec!["key"]) {
            true => match parse_jsonrpc_request_param!(request, "key", Value::String) {
                Some(v) => {
                    debug!("API request: key_get {}", v);
                    parse_result_for_jsonrpc!(DBCELL.write().await.key_get(&v), request)
                }
                None => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            },
            false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
        },
        "key_get_field" => match request.params_valid(vec!["key", "field"]) {
            true => {
                let key = parse_jsonrpc_request_param!(request, "key", Value::String);
                let field = parse_jsonrpc_request_param!(request, "field", Value::String);
                if key.is_some() && field.is_some() {
                    let k = key.unwrap();
                    let f = field.unwrap();
                    debug!("API request: key_get_field {}:{}", k, f);
                    parse_result_for_jsonrpc!(DBCELL.write().await.key_get_field(k, f), request)
                } else {
                    encode_jsonrpc_response!(request.error(Error::err_invalid_parameter()))
                }
            }
            false => {
                encode_jsonrpc_response!(request.error(Error::err_invalid_parameter()))
            }
        },
        "key_get_recursive" => match request.params_valid(vec!["key"]) {
            true => match parse_jsonrpc_request_param!(request, "key", Value::String) {
                Some(v) => {
                    debug!("API request: key_get_recursive {}", v);
                    parse_result_for_jsonrpc!(DBCELL.write().await.key_get_recursive(v), request)
                }
                None => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            },
            false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
        },
        "key_explain" => match request.params_valid(vec!["key"]) {
            true => match parse_jsonrpc_request_param!(request, "key", Value::String) {
                Some(v) => {
                    debug!("API request: key_explain {}", v);
                    parse_result_for_jsonrpc!(DBCELL.write().await.key_explain(v), request)
                }
                None => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            },
            false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
        },
        "key_list" => match request.params_valid(vec!["key"]) {
            true => match parse_jsonrpc_request_param!(request, "key", Value::String) {
                Some(v) => {
                    debug!("API request: key_list {}", v);
                    parse_result_for_jsonrpc!(DBCELL.write().await.key_list(v), request)
                }
                None => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            },
            false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
        },
        "key_list_all" => match request.params_valid(vec!["key"]) {
            true => match parse_jsonrpc_request_param!(request, "key", Value::String) {
                Some(v) => {
                    debug!("API request: key_list_all {}", v);
                    parse_result_for_jsonrpc!(DBCELL.write().await.key_list_all(v), request)
                }
                None => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            },
            false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
        },
        "key_set" => match request.params_valid(vec!["key", "value"]) {
            true => {
                let key = parse_jsonrpc_request_param!(request, "key", Value::String);
                let value = match request.params.get("value") {
                    Some(v) => Some(v.clone()),
                    None => None,
                };
                if key.is_some() && value.is_some() {
                    let k = key.unwrap();
                    debug!("API request: key_set {}", k);
                    let result = DBCELL.write().await.key_set(k, value.unwrap());
                    parse_result_for_jsonrpc_ok!(result, request)
                } else {
                    encode_jsonrpc_response!(request.error(Error::err_invalid_parameter()))
                }
            }
            false => {
                encode_jsonrpc_response!(request.error(Error::err_invalid_parameter()))
            }
        },
        "key_set_field" => match request.params_valid(vec!["key", "field", "value"]) {
            true => {
                let key = parse_jsonrpc_request_param!(request, "key", Value::String);
                let field = parse_jsonrpc_request_param!(request, "field", Value::String);
                let value = match request.params.get("value") {
                    Some(v) => Some(v.clone()),
                    None => None,
                };
                if key.is_some() && field.is_some() && value.is_some() {
                    let k = key.unwrap();
                    let f = field.unwrap();
                    debug!("API request: key_set_field {}:{}", k, f);
                    let result = DBCELL.write().await.key_set_field(k, f, value.unwrap());
                    parse_result_for_jsonrpc_ok!(result, request)
                } else {
                    encode_jsonrpc_response!(request.error(Error::err_invalid_parameter()))
                }
            }
            false => {
                encode_jsonrpc_response!(request.error(Error::err_invalid_parameter()))
            }
        },
        "key_increment" => match request.params_valid(vec!["key"]) {
            true => match parse_jsonrpc_request_param!(request, "key", Value::String) {
                Some(v) => {
                    debug!("API request: key_get {}", v);
                    parse_result_for_jsonrpc!(DBCELL.write().await.key_increment(v), request)
                }
                None => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            },
            false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
        },
        "key_decrement" => match request.params_valid(vec!["key"]) {
            true => match parse_jsonrpc_request_param!(request, "key", Value::String) {
                Some(v) => {
                    debug!("API request: key_get {}", v);
                    parse_result_for_jsonrpc!(DBCELL.write().await.key_decrement(v), request)
                }
                None => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            },
            false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
        },
        "key_copy" => match request.params_valid(vec!["key", "dst_key"]) {
            true => {
                let key = parse_jsonrpc_request_param!(request, "key", Value::String);
                let dst_key = parse_jsonrpc_request_param!(request, "dst_key", Value::String);
                if key.is_some() && dst_key.is_some() {
                    let k = key.unwrap();
                    let dk = dst_key.unwrap();
                    debug!("API request: key_copy {} -> {}", k, dk);
                    let result = DBCELL.write().await.key_copy(k, dk);
                    parse_result_for_jsonrpc_ok!(result, request)
                } else {
                    encode_jsonrpc_response!(request.error(Error::err_invalid_parameter()))
                }
            }
            false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
        },
        "key_rename" => match request.params_valid(vec!["key", "dst_key"]) {
            true => {
                let key = parse_jsonrpc_request_param!(request, "key", Value::String);
                let dst_key = parse_jsonrpc_request_param!(request, "dst_key", Value::String);
                if key.is_some() && dst_key.is_some() {
                    let k = key.unwrap();
                    let dk = dst_key.unwrap();
                    debug!("API request: key_rename {} -> {}", k, dk);
                    let result = DBCELL.write().await.key_rename(k, dk);
                    parse_result_for_jsonrpc_ok!(result, request)
                } else {
                    encode_jsonrpc_response!(request.error(Error::err_invalid_parameter()))
                }
            }
            false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
        },
        "key_delete" => match request.params_valid(vec!["key"]) {
            true => match parse_jsonrpc_request_param!(request, "key", Value::String) {
                Some(v) => {
                    debug!("API request: key_delete {}", v);
                    let result = DBCELL.write().await.key_delete(v);
                    parse_result_for_jsonrpc_ok!(result, request)
                }
                None => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            },
            false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
        },
        "key_delete_recursive" => match request.params_valid(vec!["key"]) {
            true => match parse_jsonrpc_request_param!(request, "key", Value::String) {
                Some(v) => {
                    debug!("API request: key_delete_recursive {}", v);
                    let result = DBCELL.write().await.key_delete_recursive(&v);
                    parse_result_for_jsonrpc_ok!(result, request)
                }
                None => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            },
            false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
        },
        "check" => {
            debug!("API request: check");
            match request.params_valid(vec![]) {
                true => {
                    let result = DBCELL.write().await.check();
                    parse_result_for_jsonrpc!(result, request)
                }
                false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            }
        }
        "repair" => {
            debug!("API request: repair");
            match request.params_valid(vec![]) {
                true => {
                    let result = DBCELL.write().await.repair();
                    parse_result_for_jsonrpc!(result, request)
                }
                false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            }
        }
        "purge" => {
            debug!("API request: purge");
            match request.params_valid(vec![]) {
                true => {
                    let result = DBCELL.write().await.purge();
                    parse_result_for_jsonrpc!(result, request)
                }
                false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            }
        }
        "purge_cache" => {
            debug!("API request: safe purge");
            match request.params_valid(vec![]) {
                true => {
                    let result = DBCELL.write().await.purge_cache();
                    parse_result_for_jsonrpc_ok!(result, request)
                }
                false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            }
        }
        "safe_purge" => {
            debug!("API request: safe purge");
            match request.params_valid(vec![]) {
                true => {
                    let result = DBCELL.write().await.safe_purge();
                    parse_result_for_jsonrpc_ok!(result, request)
                }
                false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            }
        }
        "key_dump" => match request.params_valid(vec!["key"]) {
            true => match parse_jsonrpc_request_param!(request, "key", Value::String) {
                Some(v) => {
                    debug!("API request: key_dump {}", v);
                    parse_result_for_jsonrpc!(DBCELL.write().await.key_dump(&v), request)
                }
                None => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            },
            false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
        },
        "key_load" => match request.params_valid(vec!["data"]) {
            true => match parse_jsonrpc_request_param!(request, "data", Value::Array) {
                Some(v) => {
                    debug!("API request: key_load");
                    let result = parse_result_for_jsonrpc!(
                        DBCELL.write().await.key_load_from_serialized(&v),
                        request
                    );
                    result
                }
                None => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
            },
            false => encode_jsonrpc_response!(request.error(Error::err_invalid_parameter())),
        },
        _ => encode_jsonrpc_response!(request.error(Error::err_method_not_found())),
    })
}
