mod operand;

use std::fmt::Write;

use yaxpeax_arch::{AddressBase, Decoder, U8Reader, LengthedInstruction};
use yaxpeax_x86::real_mode::InstDecoder;

fn test_invalid(data: &[u8]) {
    test_invalid_under(&InstDecoder::default(), data);
}

fn test_invalid_under(decoder: &InstDecoder, data: &[u8]) {
    let mut reader = U8Reader::new(data);
    if let Ok(inst) = decoder.decode(&mut reader) {
        panic!("decoded {:?} from {:02x?} under decoder {}", inst.opcode(), data, decoder);
    } else {
        // this is fine
    }
}

#[allow(dead_code)]
fn test_display(data: &[u8], expected: &'static str) {
    test_display_under(&InstDecoder::default(), data, expected);
}

fn test_display_under(decoder: &InstDecoder, data: &[u8], expected: &'static str) {
    let mut hex = String::new();
    for b in data {
        write!(hex, "{:02x}", b).unwrap();
    }
    let mut reader = U8Reader::new(data);
    match decoder.decode(&mut reader) {
        Ok(instr) => {
            let text = format!("{}", instr);
            assert!(
                text == expected,
                "display error for {}:\n  decoded: {:?} under decoder {}\n displayed: {}\n expected: {}\n",
                hex,
                instr,
                decoder,
                text,
                expected
            );
            // while we're at it, test that the instruction is as long, and no longer, than its
            // input
            assert_eq!((0u32.wrapping_offset(instr.len()).to_linear()) as usize, data.len(), "instruction length is incorrect, wanted instruction {}", expected);
        },
        Err(e) => {
            assert!(false, "decode error ({}) for {} under decoder {}:\n  expected: {}\n", e, hex, decoder, expected);
        }
    }
}

#[test]
fn only_16bit() {
    test_display(&[0xac], "lods al, byte ds:[si]");
    test_display(&[0xae], "scas byte es:[di], al");
    test_display(&[0x67, 0xac], "lods al, byte ds:[esi]");
    test_display(&[0x67, 0xae], "scas byte es:[edi], al");
    test_display(&[0xf3, 0xa4], "rep movs byte es:[di], byte ds:[si]");
    test_display(&[0x67, 0xf3, 0xa4], "rep movs byte es:[edi], byte ds:[esi]");
    test_display(&[0xf3, 0xa5], "rep movs word es:[di], word ds:[si]");
    test_display(&[0x67, 0xf3, 0xa5], "rep movs word es:[edi], word ds:[esi]");
    test_display(&[0x8b, 0x0e, 0x55, 0xaa], "mov cx, word [0xaa55]");
    test_display(&[0x66, 0x8b, 0x0e, 0x55, 0xaa], "mov ecx, dword [0xaa55]");
    test_display(&[0x67, 0x8b, 0x0e], "mov cx, word [esi]");
    test_display(&[0x66, 0x67, 0x8b, 0x0e], "mov ecx, dword [esi]");
}

#[test]
fn test_real_mode() {
    test_display(&[0x00, 0xcc], "add ah, cl");
    test_display(&[0x03, 0x0b], "add cx, word [bp + di]");
    test_display(&[0x06], "push es");
    test_display(&[0x07], "pop es");
    test_display(&[0x0e], "push cs");
    test_display(&[0x0f, 0x01, 0x38], "invlpg byte [bx + si]");
    test_display(&[0x0f, 0x01, 0x3f], "invlpg byte [bx]");
    test_display(&[0x0f, 0x01, 0x40, 0xff], "sgdt ptr [bx + si - 0x1]");
    test_display(&[0x0f, 0x01, 0x41, 0xff], "sgdt ptr [bx + di - 0x1]");
    test_display(&[0x0f, 0x01, 0x49, 0xff], "sidt ptr [bx + di - 0x1]");
    test_display(&[0x0f, 0x01, 0x51, 0xff], "lgdt ptr [bx + di - 0x1]");
    test_display(&[0x0f, 0x01, 0x59, 0xff], "lidt ptr [bx + di - 0x1]");
    test_display(&[0x0f, 0x01, 0x61, 0xff], "smsw word [bx + di - 0x1]");
    test_display(&[0x0f, 0x01, 0x71, 0xff], "lmsw word [bx + di - 0x1]");
    test_display(&[0x0f, 0x01, 0x79, 0xff], "invlpg byte [bx + di - 0x1]");
    test_display(&[0x0f, 0x01, 0xc0], "enclv");
    test_display(&[0x0f, 0x01, 0xc1], "vmcall");
    test_display(&[0x0f, 0x01, 0xc2], "vmlaunch");
    test_display(&[0x0f, 0x01, 0xc3], "vmresume");
    test_display(&[0x0f, 0x01, 0xc4], "vmxoff");
    test_display(&[0x0f, 0x01, 0xc8], "monitor");
    test_display(&[0x0f, 0x01, 0xc8], "monitor");
    test_display(&[0x0f, 0x01, 0xc9], "mwait");
    test_display(&[0x0f, 0x01, 0xc9], "mwait");
    test_display(&[0x0f, 0x01, 0xca], "clac");
    test_display(&[0x0f, 0x01, 0xcb], "stac");
    test_display(&[0x0f, 0x01, 0xcf], "encls");
    test_display(&[0x0f, 0x01, 0xd0], "xgetbv");
    test_display(&[0x0f, 0x01, 0xd1], "xsetbv");
    test_display(&[0x0f, 0x01, 0xd4], "vmfunc");
    test_display(&[0x0f, 0x01, 0xd5], "xend");
    test_display(&[0x0f, 0x01, 0xd6], "xtest");
    test_display(&[0x0f, 0x01, 0xd7], "enclu");
    test_display(&[0x0f, 0x01, 0xd8], "vmrun ax");
    test_display(&[0x0f, 0x01, 0xd9], "vmmcall");
    test_display(&[0x0f, 0x01, 0xda], "vmload ax");
    test_display(&[0x0f, 0x01, 0xdb], "vmsave ax");
    test_display(&[0x0f, 0x01, 0xdc], "stgi");
    test_display(&[0x0f, 0x01, 0xdd], "clgi");
    test_display(&[0x0f, 0x01, 0xde], "skinit eax");
    test_display(&[0x0f, 0x01, 0xdf], "invlpga ax, ecx");
    test_display(&[0x0f, 0x01, 0xe0], "smsw ax");
    test_display(&[0x0f, 0x01, 0xe1], "smsw cx");
    test_display(&[0x0f, 0x01, 0xe2], "smsw dx");
    test_display(&[0x0f, 0x01, 0xe3], "smsw bx");
    test_display(&[0x0f, 0x01, 0xe4], "smsw sp");
    test_display(&[0x0f, 0x01, 0xe5], "smsw bp");
    test_display(&[0x0f, 0x01, 0xe6], "smsw si");
    test_display(&[0x0f, 0x01, 0xe7], "smsw di");
    test_display(&[0x0f, 0x01, 0xee], "rdpkru");
    test_display(&[0x0f, 0x01, 0xef], "wrpkru");
    test_display(&[0x0f, 0x01, 0xf0], "lmsw ax");
    test_display(&[0x0f, 0x01, 0xf1], "lmsw cx");
    test_display(&[0x0f, 0x01, 0xf2], "lmsw dx");
    test_display(&[0x0f, 0x01, 0xf3], "lmsw bx");
    test_display(&[0x0f, 0x01, 0xf4], "lmsw sp");
    test_display(&[0x0f, 0x01, 0xf5], "lmsw bp");
    test_display(&[0x0f, 0x01, 0xf6], "lmsw si");
    test_display(&[0x0f, 0x01, 0xf7], "lmsw di");
    test_display(&[0x0f, 0x01, 0xf9], "rdtscp");
    test_display(&[0x0f, 0x01, 0xfa], "monitorx");
    test_display(&[0x0f, 0x01, 0xfb], "mwaitx");
    test_display(&[0x0f, 0x01, 0xfc], "clzero");
    test_display(&[0x0f, 0x01, 0xfd], "rdpru ecx");
    test_display(&[0x0f, 0x02, 0xc0], "lar ax, ax");
    test_display(&[0x0f, 0x03, 0xc0], "lsl ax, ax");
    test_display(&[0x0f, 0x05], "syscall");
    test_display(&[0x0f, 0x06], "clts");
    test_display(&[0x0f, 0x07], "sysret");
    test_display(&[0x0f, 0x0d, 0x08], "prefetchw zmmword [bx + si]");
    test_display(&[0x0f, 0x0f, 0x38, 0x8e], "pfpnacc mm7, qword [bx + si]");
    test_display(&[0x0f, 0x0f, 0xc6, 0xb7], "pmulhrw mm0, mm6");
    test_display(&[0x0f, 0x0f, 0xe0, 0x8a], "pfnacc mm4, mm0");
    test_display(&[0x0f, 0x12, 0x0f], "movlps xmm1, qword [bx]");
    test_display(&[0x0f, 0x12, 0xc0], "movhlps xmm0, xmm0");
    test_display(&[0x0f, 0x12, 0xcf], "movhlps xmm1, xmm7");
    test_display(&[0x0f, 0x13, 0x00], "movlps qword [bx + si], xmm0");
    test_display(&[0x0f, 0x14, 0x08], "unpcklps xmm1, xmmword [bx + si]");
    test_display(&[0x0f, 0x15, 0x08], "unpckhps xmm1, xmmword [bx + si]");
    test_display(&[0x0f, 0x16, 0x0f], "movhps xmm1, qword [bx]");
    test_display(&[0x0f, 0x16, 0xc0], "movlhps xmm0, xmm0");
    test_display(&[0x0f, 0x16, 0xcf], "movlhps xmm1, xmm7");
    test_display(&[0x0f, 0x17, 0x00], "movhps qword [bx + si], xmm0");
    test_display(&[0x0f, 0x18, 0x00], "prefetchnta zmmword [bx + si]");
    test_display(&[0x0f, 0x18, 0x08], "prefetch0 zmmword [bx + si]");
    test_display(&[0x0f, 0x18, 0x10], "prefetch1 zmmword [bx + si]");
    test_display(&[0x0f, 0x18, 0x18], "prefetch2 zmmword [bx + si]");
    test_display(&[0x0f, 0x18, 0x20], "nop zmmword [bx + si]");
    test_display(&[0x0f, 0x18, 0xc0], "nop ax");
    test_display(&[0x0f, 0x18, 0xcc], "nop sp");
    test_display(&[0x0f, 0x19, 0x20], "nop word [bx + si]");
    test_display(&[0x0f, 0x1a, 0x20], "nop word [bx + si]");
    test_display(&[0x0f, 0x1b, 0x20], "nop word [bx + si]");
    test_display(&[0x0f, 0x1c, 0x20], "nop word [bx + si]");
    test_display(&[0x0f, 0x1d, 0x20], "nop word [bx + si]");
    test_display(&[0x0f, 0x1e, 0x20], "nop word [bx + si]");
    test_display(&[0x0f, 0x1f, 0x20], "nop word [bx + si]");
    test_display(&[0x0f, 0x20, 0xc0], "mov eax, cr0");
    test_display(&[0x0f, 0x21, 0xc8], "mov eax, dr1");
    test_display(&[0x0f, 0x22, 0xc0], "mov cr0, eax");
    test_display(&[0x0f, 0x22, 0xc7], "mov cr0, edi");
    test_display(&[0x0f, 0x23, 0xc8], "mov dr1, eax");
    test_display(&[0x0f, 0x23, 0xcf], "mov dr1, edi");
    test_display(&[0x0f, 0x28, 0x00], "movaps xmm0, xmmword [bx + si]");
    test_display(&[0x0f, 0x28, 0xd0], "movaps xmm2, xmm0");
    test_display(&[0x0f, 0x29, 0x00], "movaps xmmword [bx + si], xmm0");
    test_display(&[0x0f, 0x2a, 0x00], "cvtpi2ps xmm0, qword [bx + si]");
    test_display(&[0x0f, 0x2a, 0xcf], "cvtpi2ps xmm1, mm7");
    test_display(&[0x0f, 0x2b, 0x00], "movntps xmmword [bx + si], xmm0");
    test_display(&[0x0f, 0x2c, 0xcf], "cvttps2pi mm1, xmm7");
    test_display(&[0x0f, 0x2e, 0x00], "ucomiss xmm0, dword [bx + si]");
    test_display(&[0x0f, 0x2f, 0x00], "comiss xmm0, dword [bx + si]");
    test_display(&[0x0f, 0x30], "wrmsr");
    test_display(&[0x0f, 0x31], "rdtsc");
    test_display(&[0x0f, 0x32], "rdmsr");
    test_display(&[0x0f, 0x33], "rdpmc");
    test_display(&[0x0f, 0x34], "sysenter");
    test_display(&[0x0f, 0x35], "sysexit");
    test_display(&[0x0f, 0x37], "getsec");
    test_display(&[0x0f, 0x38, 0x00, 0xda], "pshufb mm3, mm2");
    test_display(&[0x0f, 0x38, 0xc8, 0x12], "sha1nexte xmm2, xmmword [bp + si]");
    test_display(&[0x0f, 0x38, 0xc9, 0x12], "sha1msg1 xmm2, xmmword [bp + si]");
    test_display(&[0x0f, 0x38, 0xca, 0x12], "sha1msg2 xmm2, xmmword [bp + si]");
    test_display(&[0x0f, 0x38, 0xcb, 0x12], "sha256rnds2 xmm2, xmmword [bp + si]");
    test_display(&[0x0f, 0x38, 0xcc, 0x12], "sha256msg1 xmm2, xmmword [bp + si]");
    test_display(&[0x0f, 0x38, 0xcd, 0x12], "sha256msg2 xmm2, xmmword [bp + si]");
    test_display(&[0x0f, 0x3a, 0x0f, 0xc1, 0x23], "palignr mm0, mm1, 0x23");
    test_display(&[0x0f, 0x3a, 0xcc, 0x12, 0x40], "sha1rnds4 xmm2, xmmword [bp + si], 0x40");
    test_display(&[0x0f, 0x3a, 0xcc, 0x12, 0xff], "sha1rnds4 xmm2, xmmword [bp + si], 0xff");
    test_display(&[0x0f, 0x43, 0xec], "cmovnb bp, sp");
    test_display(&[0x0f, 0x50, 0xc1], "movmskps eax, xmm1");
    test_display(&[0x0f, 0x51, 0x01], "sqrtps xmm0, xmmword [bx + di]");
    test_display(&[0x0f, 0x52, 0x01], "rsqrtps xmm0, xmmword [bx + di]");
    test_display(&[0x0f, 0x53, 0x01], "rcpps xmm0, xmmword [bx + di]");
    test_display(&[0x0f, 0x54, 0x01], "andps xmm0, xmmword [bx + di]");
    test_display(&[0x0f, 0x55, 0x01], "andnps xmm0, xmmword [bx + di]");
    test_display(&[0x0f, 0x56, 0x01], "orps xmm0, xmmword [bx + di]");
    test_display(&[0x0f, 0x57, 0x01], "xorps xmm0, xmmword [bx + di]");
    test_display(&[0x0f, 0x58, 0x01], "addps xmm0, xmmword [bx + di]");
    test_display(&[0x0f, 0x59, 0x01], "mulps xmm0, xmmword [bx + di]");
    test_display(&[0x0f, 0x5a, 0x01], "cvtps2pd xmm0, qword [bx + di]");
    test_display(&[0x0f, 0x5b, 0x01], "cvtdq2ps xmm0, xmmword [bx + di]");
    test_display(&[0x0f, 0x5c, 0x01], "subps xmm0, xmmword [bx + di]");
    test_display(&[0x0f, 0x5d, 0x01], "minps xmm0, xmmword [bx + di]");
    test_display(&[0x0f, 0x5e, 0x01], "divps xmm0, xmmword [bx + di]");
    test_display(&[0x0f, 0x5f, 0x01], "maxps xmm0, xmmword [bx + di]");
    test_display(&[0x0f, 0x60, 0x00], "punpcklbw mm0, dword [bx + si]");
    test_display(&[0x0f, 0x60, 0xc2], "punpcklbw mm0, mm2");
    test_display(&[0x0f, 0x61, 0x00], "punpcklwd mm0, dword [bx + si]");
    test_display(&[0x0f, 0x61, 0xc2], "punpcklwd mm0, mm2");
    test_display(&[0x0f, 0x62, 0x00], "punpckldq mm0, dword [bx + si]");
    test_display(&[0x0f, 0x62, 0xc2], "punpckldq mm0, mm2");
    test_display(&[0x0f, 0x63, 0x00], "packsswb mm0, qword [bx + si]");
    test_display(&[0x0f, 0x63, 0xc2], "packsswb mm0, mm2");
    test_display(&[0x0f, 0x64, 0x00], "pcmpgtb mm0, qword [bx + si]");
    test_display(&[0x0f, 0x64, 0xc2], "pcmpgtb mm0, mm2");
    test_display(&[0x0f, 0x65, 0x00], "pcmpgtw mm0, qword [bx + si]");
    test_display(&[0x0f, 0x65, 0xc2], "pcmpgtw mm0, mm2");
    test_display(&[0x0f, 0x66, 0x00], "pcmpgtd mm0, qword [bx + si]");
    test_display(&[0x0f, 0x66, 0xc2], "pcmpgtd mm0, mm2");
    test_display(&[0x0f, 0x67, 0x00], "packuswb mm0, qword [bx + si]");
    test_display(&[0x0f, 0x67, 0xc2], "packuswb mm0, mm2");
    test_display(&[0x0f, 0x68, 0x00], "punpckhbw mm0, qword [bx + si]");
    test_display(&[0x0f, 0x68, 0xc2], "punpckhbw mm0, mm2");
    test_display(&[0x0f, 0x69, 0x00], "punpckhwd mm0, qword [bx + si]");
    test_display(&[0x0f, 0x69, 0xc2], "punpckhwd mm0, mm2");
    test_display(&[0x0f, 0x6a, 0x00], "punpckhdq mm0, qword [bx + si]");
    test_display(&[0x0f, 0x6a, 0xc2], "punpckhdq mm0, mm2");
    test_display(&[0x0f, 0x6b, 0x00], "packssdw mm0, qword [bx + si]");
    test_display(&[0x0f, 0x6b, 0xc2], "packssdw mm0, mm2");
    test_display(&[0x0f, 0x6e, 0x00], "movd mm0, dword [bx + si]");
    test_display(&[0x0f, 0x6e, 0xc2], "movd mm0, edx");
    test_display(&[0x0f, 0x6f, 0x00], "movq mm0, qword [bx + si]");
    test_display(&[0x0f, 0x6f, 0xc2], "movq mm0, mm2");
    test_display(&[0x0f, 0x6f, 0xe9], "movq mm5, mm1");
    test_display(&[0x0f, 0x6f, 0xfb], "movq mm7, mm3");
    test_display(&[0x0f, 0x70, 0x00, 0x7f], "pshufw mm0, qword [bx + si], 0x7f");
    test_display(&[0x0f, 0x71, 0xd0, 0x7f], "psrlw mm0, 0x7f");
    test_display(&[0x0f, 0x71, 0xe0, 0x7f], "psraw mm0, 0x7f");
    test_display(&[0x0f, 0x71, 0xf0, 0x7f], "psllw mm0, 0x7f");
    test_display(&[0x0f, 0x72, 0xd0, 0x7f], "psrld mm0, 0x7f");
    test_display(&[0x0f, 0x72, 0xe0, 0x7f], "psrad mm0, 0x7f");
    test_display(&[0x0f, 0x72, 0xf0, 0x7f], "pslld mm0, 0x7f");
    test_display(&[0x0f, 0x73, 0xd0, 0x7f], "psrlq mm0, 0x7f");
    test_display(&[0x0f, 0x73, 0xf0, 0x7f], "psllq mm0, 0x7f");
    test_display(&[0x0f, 0x74, 0xc2], "pcmpeqb mm0, mm2");
    test_display(&[0x0f, 0x75, 0xc2], "pcmpeqw mm0, mm2");
    test_display(&[0x0f, 0x76, 0xc2], "pcmpeqd mm0, mm2");
    test_display(&[0x0f, 0x78, 0x0b], "vmread dword [bp + di], ecx");
    test_display(&[0x0f, 0x78, 0xc4], "vmread esp, eax");
    test_display(&[0x0f, 0x79, 0x0b], "vmwrite ecx, dword [bp + di]");
    test_display(&[0x0f, 0x79, 0xc5], "vmwrite eax, ebp");
    test_display(&[0x0f, 0x7e, 0xcf], "movd edi, mm1");
    test_display(&[0x0f, 0x7f, 0x0f], "movq qword [bx], mm1");
    test_display(&[0x0f, 0x7f, 0xcf], "movq mm7, mm1");
    test_display(&[0x0f, 0x86, 0x8b, 0x01], "jna $+0x18b");
    test_display(&[0x0f, 0x97, 0x00], "seta byte [bx + si]");
    test_display(&[0x0f, 0x97, 0x08], "seta byte [bx + si]");
    test_display(&[0x0f, 0x97, 0xc0], "seta al");
    test_display(&[0x0f, 0x97, 0xc8], "seta al");
    test_display(&[0x0f, 0xa0], "push fs");
    test_display(&[0x0f, 0xa1], "pop fs");
    test_display(&[0x0f, 0xa2], "cpuid");
    test_display(&[0x0f, 0xa3, 0xd0], "bt ax, dx");
    test_display(&[0x0f, 0xa4, 0xc0, 0x11], "shld ax, ax, 0x11");
    test_display(&[0x0f, 0xa5, 0xc0], "shld ax, ax, cl");
    test_display(&[0x0f, 0xa5, 0xc9], "shld cx, cx, cl");
    test_display(&[0x0f, 0xab, 0xd0], "bts ax, dx");
    test_display(&[0x0f, 0xac, 0xc0, 0x11], "shrd ax, ax, 0x11");
    test_display(&[0x0f, 0xad, 0xc9], "shrd cx, cx, cl");
    test_display(&[0x0f, 0xae, 0x04], "fxsave ptr [si]");
    test_display(&[0x0f, 0xae, 0x0c], "fxrstor ptr [si]");
    test_display(&[0x0f, 0xae, 0x14], "ldmxcsr dword [si]");
    test_display(&[0x0f, 0xae, 0x1c], "stmxcsr dword [si]");
    test_display(&[0x0f, 0xae, 0x24], "xsave ptr [si]");
    test_display(&[0x0f, 0xae, 0x2c], "xrstor ptr [si]");
    test_display(&[0x0f, 0xae, 0x34], "xsaveopt ptr [si]");
    test_display(&[0x0f, 0xae, 0x3c], "clflush zmmword [si]");
    test_display(&[0x0f, 0xaf, 0xc2], "imul ax, dx");
    test_display(&[0x0f, 0xb3, 0xd0], "btr ax, dx");
    test_display(&[0x0f, 0xbb, 0x17], "btc word [bx], dx");
    test_display(&[0x0f, 0xbc, 0xd3], "bsf dx, bx");
    test_display(&[0x0f, 0xbc, 0xd3], "bsf dx, bx");
    test_display(&[0x0f, 0xbe, 0x83, 0xb4, 0x00], "movsx ax, byte [bp + di + 0xb4]");
    test_display(&[0x0f, 0xc0, 0xcc], "xadd ah, cl");
    test_display(&[0x0f, 0xc1, 0xcc], "xadd sp, cx");
    test_display(&[0x0f, 0xc3, 0x03], "movnti dword [bp + di], eax");
    test_display(&[0x0f, 0xc4, 0x00, 0x14], "pinsrw mm0, word [bx + si], 0x14");
    test_display(&[0x0f, 0xc4, 0xc0, 0x14], "pinsrw mm0, eax, 0x14");
    test_display(&[0x0f, 0xc5, 0xd1, 0x00], "pextrw edx, mm1, 0x0");
    test_display(&[0x0f, 0xc7, 0x0f], "cmpxchg8b qword [bx]");
    test_display(&[0x0f, 0xc7, 0x37], "vmptrld qword [bx]");
    test_display(&[0x0f, 0xc7, 0x3f], "vmptrst qword [bx]");
    test_display(&[0x0f, 0xc7, 0x5c, 0x24], "xrstors ptr [si + 0x24]");
    test_display(&[0x0f, 0xc7, 0x64, 0x24], "xsavec ptr [si + 0x24]");
    test_display(&[0x0f, 0xc7, 0x6c, 0x24], "xsaves ptr [si + 0x24]");
    test_display(&[0x0f, 0xc7, 0x74, 0x24], "vmptrld qword [si + 0x24]");
    test_display(&[0x0f, 0xc7, 0x7c, 0x24], "vmptrst qword [si + 0x24]");
    test_display(&[0x0f, 0xc7, 0xf5], "rdrand bp");
    test_display(&[0x0f, 0xc7, 0xfd], "rdseed bp");
    test_display(&[0x0f, 0xd1, 0x00], "psrlw mm0, qword [bx + si]");
    test_display(&[0x0f, 0xd1, 0xcf], "psrlw mm1, mm7");
    test_display(&[0x0f, 0xd7, 0xcf], "pmovmskb ecx, mm7");
    test_display(&[0x0f, 0xd8, 0xc2], "psubusb mm0, mm2");
    test_display(&[0x0f, 0xd9, 0xc2], "psubusw mm0, mm2");
    test_display(&[0x0f, 0xda, 0xc2], "pminub mm0, mm2");
    test_display(&[0x0f, 0xdb, 0xc2], "pand mm0, mm2");
    test_display(&[0x0f, 0xdc, 0xc2], "paddusb mm0, mm2");
    test_display(&[0x0f, 0xdd, 0xc2], "paddusw mm0, mm2");
    test_display(&[0x0f, 0xde, 0xc2], "pmaxub mm0, mm2");
    test_display(&[0x0f, 0xdf, 0xc2], "pandn mm0, mm2");
    test_display(&[0x0f, 0xe5, 0x3d], "pmulhw mm7, qword [di]");
    test_display(&[0x0f, 0xe7, 0x03], "movntq qword [bp + di], mm0");
    test_display(&[0x0f, 0xe8, 0xc2], "psubsb mm0, mm2");
    test_display(&[0x0f, 0xe9, 0xc2], "psubsw mm0, mm2");
    test_display(&[0x0f, 0xea, 0xc2], "pminsw mm0, mm2");
    test_display(&[0x0f, 0xeb, 0xc2], "por mm0, mm2");
    test_display(&[0x0f, 0xec, 0xc2], "paddsb mm0, mm2");
    test_display(&[0x0f, 0xed, 0xc2], "paddsw mm0, mm2");
    test_display(&[0x0f, 0xee, 0xc2], "pmaxsw mm0, mm2");
    test_display(&[0x0f, 0xef, 0xc2], "pxor mm0, mm2");
    test_display(&[0x0f, 0xf1, 0x02], "psllw mm0, qword [bp + si]");
    test_display(&[0x0f, 0xf1, 0xc2], "psllw mm0, mm2");
    test_display(&[0x0f, 0xf2, 0x02], "pslld mm0, qword [bp + si]");
    test_display(&[0x0f, 0xf2, 0xc2], "pslld mm0, mm2");
    test_display(&[0x0f, 0xf3, 0x02], "psllq mm0, qword [bp + si]");
    test_display(&[0x0f, 0xf3, 0xc2], "psllq mm0, mm2");
    test_display(&[0x0f, 0xf4, 0x02], "pmuludq mm0, qword [bp + si]");
    test_display(&[0x0f, 0xf4, 0xc2], "pmuludq mm0, mm2");
    test_display(&[0x0f, 0xf5, 0x02], "pmaddwd mm0, qword [bp + si]");
    test_display(&[0x0f, 0xf5, 0xc2], "pmaddwd mm0, mm2");
    test_display(&[0x0f, 0xf6, 0x02], "psadbw mm0, qword [bp + si]");
    test_display(&[0x0f, 0xf6, 0xc2], "psadbw mm0, mm2");
    test_display(&[0x0f, 0xf7, 0xc1], "maskmovq mm0, mm1");
    test_display(&[0x0f, 0xf8, 0x02], "psubb mm0, qword [bp + si]");
    test_display(&[0x0f, 0xf8, 0xc2], "psubb mm0, mm2");
    test_display(&[0x0f, 0xf9, 0x02], "psubw mm0, qword [bp + si]");
    test_display(&[0x0f, 0xf9, 0xc2], "psubw mm0, mm2");
    test_display(&[0x0f, 0xfa, 0x02], "psubd mm0, qword [bp + si]");
    test_display(&[0x0f, 0xfa, 0xc2], "psubd mm0, mm2");
    test_display(&[0x0f, 0xfb, 0x02], "psubq mm0, qword [bp + si]");
    test_display(&[0x0f, 0xfb, 0xc2], "psubq mm0, mm2");
    test_display(&[0x0f, 0xfc, 0x02], "paddb mm0, qword [bp + si]");
    test_display(&[0x0f, 0xfc, 0xc2], "paddb mm0, mm2");
    test_display(&[0x0f, 0xfd, 0x02], "paddw mm0, qword [bp + si]");
    test_display(&[0x0f, 0xfd, 0xc2], "paddw mm0, mm2");
    test_display(&[0x0f, 0xfd, 0xd2], "paddw mm2, mm2");
    test_display(&[0x0f, 0xfe, 0x02], "paddd mm0, qword [bp + si]");
    test_display(&[0x0f, 0xfe, 0xc2], "paddd mm0, mm2");
    test_display(&[0x0f, 0xff, 0x6b, 0xac], "ud0 ebp, dword [bp + di - 0x54]");
    test_display(&[0x16], "push ss");
    test_display(&[0x17], "pop ss");
    test_display(&[0x1e], "push ds");
    test_display(&[0x1f], "pop ds");
    test_display(&[0x26, 0x36, 0x0f, 0x0f, 0x70, 0xfb, 0x0c], "pi2fw mm6, qword ss:[bx + si - 0x5]");
    test_display(&[0x26, 0x66, 0x67, 0x0f, 0x38, 0xdf, 0xe4], "aesdeclast xmm4, xmm4");
    test_display(&[0x27], "daa");
    test_display(&[0x29, 0xc8], "sub ax, cx");
    test_display(&[0x2e, 0x36, 0x0f, 0x18, 0xe7], "nop di");
    test_display(&[0x2e, 0x3e, 0x66, 0x3e, 0x0f, 0x3a, 0x41, 0x30, 0x48], "dppd xmm6, xmmword [bx + si], 0x48");
    test_display(&[0x2e, 0x66, 0x0f, 0x3a, 0x0d, 0x40, 0x2d, 0x57], "blendpd xmm0, xmmword cs:[bx + si + 0x2d], 0x57");
    test_display(&[0x2e, 0x66, 0x26, 0x64, 0x0f, 0x3a, 0x21, 0x0b, 0xb1], "insertps xmm1, dword fs:[bp + di], -0x4f");
    test_display(&[0x2f], "das");
    test_display(&[0x31, 0xc9], "xor cx, cx");
    test_display(&[0x33, 0x04], "xor ax, word [si]");
    test_display(&[0x33, 0x05], "xor ax, word [di]");
    test_display(&[0x33, 0x08], "xor cx, word [bx + si]");
    test_display(&[0x33, 0x20], "xor sp, word [bx + si]");
    test_display(&[0x33, 0x34], "xor si, word [si]");
    test_display(&[0x33, 0x41, 0x23], "xor ax, word [bx + di + 0x23]");
    test_display(&[0x33, 0x81, 0x23, 0x01], "xor ax, word [bx + di + 0x123]");
    test_display(&[0x33, 0x84, 0xa5, 0x11], "xor ax, word [si + 0x11a5]");
    test_display(&[0x33, 0xb4, 0x25, 0x20], "xor si, word [si + 0x2025]");
    test_display(&[0x30, 0x40, 0x50], "xor byte [bx + si + 0x50], al");
    test_display(&[0x33, 0xc0], "xor ax, ax");
    test_display(&[0x33, 0xc1], "xor ax, cx");
    test_display(&[0x36, 0x36, 0x2e, 0x0f, 0x38, 0xf9, 0x55, 0x3e], "movdiri dword cs:[di + 0x3e], edx");
    test_display(&[0x36, 0x36, 0x2e, 0x0f, 0x38, 0xf9, 0x55, 0x3e], "movdiri dword cs:[di + 0x3e], edx");
    test_display(&[0x37], "aaa");
    test_display(&[0x39, 0xc6], "cmp si, ax");
    test_display(&[0x3e, 0x0f, 0x38, 0xf6, 0x23], "wrss dword [bp + di], esp");
    test_display(&[0x3e, 0x64, 0xf3, 0x64, 0x0f, 0x38, 0xf8, 0x72, 0x54], "enqcmds si, zmmword fs:[bp + si + 0x54]");
    test_display(&[0x3f], "aas");
    test_display(&[0x40], "inc ax");
    test_display(&[0x41], "inc cx");
    test_display(&[0x47], "inc di");
    test_display(&[0x48], "dec ax");
    test_display(&[0x4f], "dec di");
    test_display(&[0x5b], "pop bx");
    test_display(&[0x5e], "pop si");
    test_display(&[0x60], "pusha");
    test_display(&[0x61], "popa");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x10, 0x0a], "vmovups xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x10, 0x4a, 0x01], "vmovups xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x10, 0xca], "vmovups xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x11, 0x0a], "vmovups xmmword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x11, 0x4a, 0x01], "vmovups xmmword [bp + si + 0x10], xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x11, 0xca], "vmovups xmm2, xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x12, 0x0a], "vmovlps xmm1, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x12, 0x4a, 0x01], "vmovlps xmm1, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x12, 0xca], "vmovhlps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x13, 0x0a], "vmovlps qword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x13, 0x4a, 0x01], "vmovlps qword [bp + si + 0x8], xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x14, 0x0a], "vunpcklps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x14, 0x4a, 0x01], "vunpcklps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x14, 0xca], "vunpcklps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x15, 0x0a], "vunpckhps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x15, 0x4a, 0x01], "vunpckhps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x15, 0xca], "vunpckhps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x16, 0x0a], "vmovhps xmm1, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x16, 0x4a, 0x01], "vmovhps xmm1, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x16, 0xca], "vmovlhps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x17, 0x0a], "vmovhps qword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x17, 0x4a, 0x01], "vmovhps qword [bp + si + 0x8], xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x28, 0x0a], "vmovaps xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x28, 0x4a, 0x01], "vmovaps xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x28, 0xca], "vmovaps xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x29, 0x0a], "vmovaps xmmword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x29, 0x4a, 0x01], "vmovaps xmmword [bp + si + 0x10], xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x29, 0xca], "vmovaps xmm2, xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x2b, 0x0a], "vmovntps xmmword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x2b, 0x4a, 0x01], "vmovntps xmmword [bp + si + 0x10], xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x51, 0x0a], "vsqrtps xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x51, 0x4a, 0x01], "vsqrtps xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x51, 0xca], "vsqrtps xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x54, 0x0a], "vandps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x54, 0x4a, 0x01], "vandps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x54, 0xca], "vandps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x55, 0x0a], "vandnps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x55, 0x4a, 0x01], "vandnps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x55, 0xca], "vandnps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x56, 0x0a], "vorps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x56, 0x4a, 0x01], "vorps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x56, 0xca], "vorps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x57, 0x0a], "vxorps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x57, 0x4a, 0x01], "vxorps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x57, 0xca], "vxorps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x58, 0x0a], "vaddps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x58, 0x4a, 0x01], "vaddps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x58, 0xca], "vaddps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x59, 0x0a], "vmulps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x59, 0x4a, 0x01], "vmulps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x59, 0xca], "vmulps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5a, 0x0a], "vcvtps2pd xmm1, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5a, 0x4a, 0x01], "vcvtps2pd xmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5a, 0xca], "vcvtps2pd xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5b, 0x0a], "vcvtdq2ps xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5b, 0x4a, 0x01], "vcvtdq2ps xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5b, 0xca], "vcvtdq2ps xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5c, 0x0a], "vsubps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5c, 0x4a, 0x01], "vsubps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5c, 0xca], "vsubps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5d, 0x0a], "vminps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5d, 0x4a, 0x01], "vminps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5d, 0xca], "vminps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5e, 0x0a], "vdivps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5e, 0x4a, 0x01], "vdivps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5e, 0xca], "vdivps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5f, 0x0a], "vmaxps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5f, 0x4a, 0x01], "vmaxps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x5f, 0xca], "vmaxps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x78, 0x0a], "vcvttps2udq xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x78, 0x4a, 0x01], "vcvttps2udq xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x78, 0xca], "vcvttps2udq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x79, 0x0a], "vcvtps2udq xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x79, 0x4a, 0x01], "vcvtps2udq xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0x79, 0xca], "vcvtps2udq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0xc2, 0x0a, 0xcc], "vcmpps k1, xmm0, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1, xmm0, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0xc2, 0xca, 0xcc], "vcmpps k1, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0xc6, 0x0a, 0xcc], "vshufps xmm1, xmm0, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0xc6, 0x4a, 0x01, 0xcc], "vshufps xmm1, xmm0, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x08, 0xc6, 0xca, 0xcc], "vshufps xmm1, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x10, 0x0a], "vmovups xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x10, 0x4a, 0x01], "vmovups xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x10, 0xca], "vmovups xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x11, 0x0a], "vmovups xmmword [bp + si]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x11, 0x4a, 0x01], "vmovups xmmword [bp + si + 0x10]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x11, 0xca], "vmovups xmm2{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x14, 0x0a], "vunpcklps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x14, 0x4a, 0x01], "vunpcklps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x14, 0xca], "vunpcklps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x15, 0x0a], "vunpckhps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x15, 0x4a, 0x01], "vunpckhps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x15, 0xca], "vunpckhps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x28, 0x0a], "vmovaps xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x28, 0x4a, 0x01], "vmovaps xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x28, 0xca], "vmovaps xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x29, 0x0a], "vmovaps xmmword [bp + si]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x29, 0x4a, 0x01], "vmovaps xmmword [bp + si + 0x10]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x29, 0xca], "vmovaps xmm2{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x51, 0x0a], "vsqrtps xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x51, 0x4a, 0x01], "vsqrtps xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x51, 0xca], "vsqrtps xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x54, 0x0a], "vandps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x54, 0x4a, 0x01], "vandps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x54, 0xca], "vandps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x55, 0x0a], "vandnps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x55, 0x4a, 0x01], "vandnps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x55, 0xca], "vandnps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x56, 0x0a], "vorps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x56, 0x4a, 0x01], "vorps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x56, 0xca], "vorps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x57, 0x0a], "vxorps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x57, 0x4a, 0x01], "vxorps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x57, 0xca], "vxorps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x58, 0x0a], "vaddps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x58, 0x4a, 0x01], "vaddps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x58, 0xca], "vaddps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x59, 0x0a], "vmulps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x59, 0x4a, 0x01], "vmulps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x59, 0xca], "vmulps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5a, 0x0a], "vcvtps2pd xmm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5a, 0x4a, 0x01], "vcvtps2pd xmm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5a, 0xca], "vcvtps2pd xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5b, 0x0a], "vcvtdq2ps xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5b, 0x4a, 0x01], "vcvtdq2ps xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5b, 0xca], "vcvtdq2ps xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5c, 0x0a], "vsubps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5c, 0x4a, 0x01], "vsubps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5c, 0xca], "vsubps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5d, 0x0a], "vminps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5d, 0x4a, 0x01], "vminps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5d, 0xca], "vminps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5e, 0x0a], "vdivps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5e, 0x4a, 0x01], "vdivps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5e, 0xca], "vdivps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5f, 0x0a], "vmaxps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5f, 0x4a, 0x01], "vmaxps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x5f, 0xca], "vmaxps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x78, 0x0a], "vcvttps2udq xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x78, 0x4a, 0x01], "vcvttps2udq xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x78, 0xca], "vcvttps2udq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x79, 0x0a], "vcvtps2udq xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x79, 0x4a, 0x01], "vcvtps2udq xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0x79, 0xca], "vcvtps2udq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, xmm0, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1{k5}, xmm0, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0xc2, 0xca, 0xcc], "vcmpps k1{k5}, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0xc6, 0x0a, 0xcc], "vshufps xmm1{k5}, xmm0, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0xc6, 0x4a, 0x01, 0xcc], "vshufps xmm1{k5}, xmm0, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x0d, 0xc6, 0xca, 0xcc], "vshufps xmm1{k5}, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x14, 0x0a], "vunpcklps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x14, 0x4a, 0x01], "vunpcklps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x15, 0x0a], "vunpckhps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x15, 0x4a, 0x01], "vunpckhps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x51, 0x0a], "vsqrtps xmm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x51, 0x4a, 0x01], "vsqrtps xmm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x51, 0xca], "vsqrtps zmm1{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x54, 0x0a], "vandps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x54, 0x4a, 0x01], "vandps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x55, 0x0a], "vandnps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x55, 0x4a, 0x01], "vandnps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x56, 0x0a], "vorps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x56, 0x4a, 0x01], "vorps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x57, 0x0a], "vxorps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x57, 0x4a, 0x01], "vxorps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x58, 0x0a], "vaddps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x58, 0x4a, 0x01], "vaddps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x58, 0xca], "vaddps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x59, 0x0a], "vmulps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x59, 0x4a, 0x01], "vmulps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x59, 0xca], "vmulps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x5a, 0x0a], "vcvtps2pd xmm1, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x5a, 0x4a, 0x01], "vcvtps2pd xmm1, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x5b, 0x0a], "vcvtdq2ps xmm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x5b, 0x4a, 0x01], "vcvtdq2ps xmm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x5b, 0xca], "vcvtdq2ps zmm1{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x5c, 0x0a], "vsubps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x5c, 0x4a, 0x01], "vsubps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x5c, 0xca], "vsubps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x5d, 0x0a], "vminps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x5d, 0x4a, 0x01], "vminps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x5e, 0x0a], "vdivps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x5e, 0x4a, 0x01], "vdivps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x5e, 0xca], "vdivps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x5f, 0x0a], "vmaxps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x5f, 0x4a, 0x01], "vmaxps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x78, 0x0a], "vcvttps2udq xmm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x78, 0x4a, 0x01], "vcvttps2udq xmm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x79, 0x0a], "vcvtps2udq xmm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x79, 0x4a, 0x01], "vcvtps2udq xmm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0x79, 0xca], "vcvtps2udq zmm1{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0xc2, 0x0a, 0xcc], "vcmpps k1, xmm0, dword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1, xmm0, dword [bp + si + 0x4]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0xc6, 0x0a, 0xcc], "vshufps xmm1, xmm0, dword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x18, 0xc6, 0x4a, 0x01, 0xcc], "vshufps xmm1, xmm0, dword [bp + si + 0x4]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x14, 0x0a], "vunpcklps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x14, 0x4a, 0x01], "vunpcklps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x15, 0x0a], "vunpckhps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x15, 0x4a, 0x01], "vunpckhps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x51, 0x0a], "vsqrtps xmm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x51, 0x4a, 0x01], "vsqrtps xmm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x51, 0xca], "vsqrtps zmm1{k5}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x54, 0x0a], "vandps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x54, 0x4a, 0x01], "vandps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x55, 0x0a], "vandnps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x55, 0x4a, 0x01], "vandnps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x56, 0x0a], "vorps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x56, 0x4a, 0x01], "vorps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x57, 0x0a], "vxorps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x57, 0x4a, 0x01], "vxorps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x58, 0x0a], "vaddps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x58, 0x4a, 0x01], "vaddps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x58, 0xca], "vaddps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x59, 0x0a], "vmulps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x59, 0x4a, 0x01], "vmulps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x59, 0xca], "vmulps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x5a, 0x0a], "vcvtps2pd xmm1{k5}, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x5a, 0x4a, 0x01], "vcvtps2pd xmm1{k5}, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x5b, 0x0a], "vcvtdq2ps xmm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x5b, 0x4a, 0x01], "vcvtdq2ps xmm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x5c, 0x0a], "vsubps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x5c, 0x4a, 0x01], "vsubps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x5c, 0xca], "vsubps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x5d, 0x0a], "vminps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x5d, 0x4a, 0x01], "vminps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x5e, 0x0a], "vdivps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x5e, 0x4a, 0x01], "vdivps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x5e, 0xca], "vdivps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x5f, 0x0a], "vmaxps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x5f, 0x4a, 0x01], "vmaxps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x78, 0x0a], "vcvttps2udq xmm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x78, 0x4a, 0x01], "vcvttps2udq xmm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x79, 0x0a], "vcvtps2udq xmm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x79, 0x4a, 0x01], "vcvtps2udq xmm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0x79, 0xca], "vcvtps2udq zmm1{k5}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, xmm0, dword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1{k5}, xmm0, dword [bp + si + 0x4]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0xc6, 0x0a, 0xcc], "vshufps xmm1{k5}, xmm0, dword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x1d, 0xc6, 0x4a, 0x01, 0xcc], "vshufps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x10, 0x0a], "vmovups ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x10, 0x4a, 0x01], "vmovups ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x10, 0xca], "vmovups ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x11, 0x0a], "vmovups ymmword [bp + si], ymm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x11, 0x4a, 0x01], "vmovups ymmword [bp + si + 0x20], ymm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x11, 0xca], "vmovups ymm2, ymm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x14, 0x0a], "vunpcklps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x14, 0x4a, 0x01], "vunpcklps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x14, 0xca], "vunpcklps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x15, 0x0a], "vunpckhps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x15, 0x4a, 0x01], "vunpckhps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x15, 0xca], "vunpckhps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x28, 0x0a], "vmovaps ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x28, 0x4a, 0x01], "vmovaps ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x28, 0xca], "vmovaps ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x29, 0x0a], "vmovaps ymmword [bp + si], ymm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x29, 0x4a, 0x01], "vmovaps ymmword [bp + si + 0x20], ymm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x29, 0xca], "vmovaps ymm2, ymm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x2b, 0x0a], "vmovntps ymmword [bp + si], ymm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x2b, 0x4a, 0x01], "vmovntps ymmword [bp + si + 0x20], ymm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x2e, 0x0a], "vucomiss xmm1, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x2e, 0x4a, 0x01], "vucomiss xmm1, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x2e, 0xca], "vucomiss xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x2f, 0x0a], "vcomiss xmm1, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x2f, 0x4a, 0x01], "vcomiss xmm1, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x2f, 0xca], "vcomiss xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x51, 0x0a], "vsqrtps ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x51, 0x4a, 0x01], "vsqrtps ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x51, 0xca], "vsqrtps ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x54, 0x0a], "vandps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x54, 0x4a, 0x01], "vandps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x54, 0xca], "vandps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x55, 0x0a], "vandnps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x55, 0x4a, 0x01], "vandnps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x55, 0xca], "vandnps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x56, 0x0a], "vorps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x56, 0x4a, 0x01], "vorps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x56, 0xca], "vorps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x57, 0x0a], "vxorps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x57, 0x4a, 0x01], "vxorps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x57, 0xca], "vxorps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x58, 0x0a], "vaddps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x58, 0x4a, 0x01], "vaddps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x58, 0xca], "vaddps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x59, 0x0a], "vmulps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x59, 0x4a, 0x01], "vmulps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x59, 0xca], "vmulps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5a, 0x0a], "vcvtps2pd ymm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5a, 0x4a, 0x01], "vcvtps2pd ymm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5a, 0xca], "vcvtps2pd ymm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5b, 0x0a], "vcvtdq2ps ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5b, 0x4a, 0x01], "vcvtdq2ps ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5b, 0xca], "vcvtdq2ps ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5c, 0x0a], "vsubps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5c, 0x4a, 0x01], "vsubps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5c, 0xca], "vsubps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5d, 0x0a], "vminps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5d, 0x4a, 0x01], "vminps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5d, 0xca], "vminps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5e, 0x0a], "vdivps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5e, 0x4a, 0x01], "vdivps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5e, 0xca], "vdivps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5f, 0x0a], "vmaxps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5f, 0x4a, 0x01], "vmaxps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x5f, 0xca], "vmaxps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x78, 0x0a], "vcvttps2udq ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x78, 0x4a, 0x01], "vcvttps2udq ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x78, 0xca], "vcvttps2udq ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x79, 0x0a], "vcvtps2udq ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x79, 0x4a, 0x01], "vcvtps2udq ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0x79, 0xca], "vcvtps2udq ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0xc2, 0x0a, 0xcc], "vcmpps k1, ymm0, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1, ymm0, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0xc2, 0xca, 0xcc], "vcmpps k1, ymm0, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0xc6, 0x0a, 0xcc], "vshufps ymm1, ymm0, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0xc6, 0x4a, 0x01, 0xcc], "vshufps ymm1, ymm0, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x28, 0xc6, 0xca, 0xcc], "vshufps ymm1, ymm0, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x10, 0x0a], "vmovups ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x10, 0x4a, 0x01], "vmovups ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x10, 0xca], "vmovups ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x11, 0x0a], "vmovups ymmword [bp + si]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x11, 0x4a, 0x01], "vmovups ymmword [bp + si + 0x20]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x11, 0xca], "vmovups ymm2{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x14, 0x0a], "vunpcklps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x14, 0x4a, 0x01], "vunpcklps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x14, 0xca], "vunpcklps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x15, 0x0a], "vunpckhps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x15, 0x4a, 0x01], "vunpckhps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x15, 0xca], "vunpckhps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x28, 0x0a], "vmovaps ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x28, 0x4a, 0x01], "vmovaps ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x28, 0xca], "vmovaps ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x29, 0x0a], "vmovaps ymmword [bp + si]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x29, 0x4a, 0x01], "vmovaps ymmword [bp + si + 0x20]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x29, 0xca], "vmovaps ymm2{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x51, 0x0a], "vsqrtps ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x51, 0x4a, 0x01], "vsqrtps ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x51, 0xca], "vsqrtps ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x54, 0x0a], "vandps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x54, 0x4a, 0x01], "vandps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x54, 0xca], "vandps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x55, 0x0a], "vandnps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x55, 0x4a, 0x01], "vandnps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x55, 0xca], "vandnps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x56, 0x0a], "vorps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x56, 0x4a, 0x01], "vorps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x56, 0xca], "vorps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x57, 0x0a], "vxorps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x57, 0x4a, 0x01], "vxorps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x57, 0xca], "vxorps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x58, 0x0a], "vaddps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x58, 0x4a, 0x01], "vaddps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x58, 0xca], "vaddps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x59, 0x0a], "vmulps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x59, 0x4a, 0x01], "vmulps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x59, 0xca], "vmulps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5a, 0x0a], "vcvtps2pd ymm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5a, 0x4a, 0x01], "vcvtps2pd ymm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5a, 0xca], "vcvtps2pd ymm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5b, 0x0a], "vcvtdq2ps ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5b, 0x4a, 0x01], "vcvtdq2ps ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5b, 0xca], "vcvtdq2ps ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5c, 0x0a], "vsubps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5c, 0x4a, 0x01], "vsubps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5c, 0xca], "vsubps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5d, 0x0a], "vminps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5d, 0x4a, 0x01], "vminps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5d, 0xca], "vminps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5e, 0x0a], "vdivps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5e, 0x4a, 0x01], "vdivps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5e, 0xca], "vdivps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5f, 0x0a], "vmaxps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5f, 0x4a, 0x01], "vmaxps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x5f, 0xca], "vmaxps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x78, 0x0a], "vcvttps2udq ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x78, 0x4a, 0x01], "vcvttps2udq ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x78, 0xca], "vcvttps2udq ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x79, 0x0a], "vcvtps2udq ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x79, 0x4a, 0x01], "vcvtps2udq ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0x79, 0xca], "vcvtps2udq ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, ymm0, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1{k5}, ymm0, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0xc2, 0xca, 0xcc], "vcmpps k1{k5}, ymm0, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0xc6, 0x0a, 0xcc], "vshufps ymm1{k5}, ymm0, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0xc6, 0x4a, 0x01, 0xcc], "vshufps ymm1{k5}, ymm0, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x2d, 0xc6, 0xca, 0xcc], "vshufps ymm1{k5}, ymm0, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x14, 0x0a], "vunpcklps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x14, 0x4a, 0x01], "vunpcklps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x15, 0x0a], "vunpckhps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x15, 0x4a, 0x01], "vunpckhps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x51, 0x0a], "vsqrtps ymm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x51, 0x4a, 0x01], "vsqrtps ymm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x51, 0xca], "vsqrtps zmm1{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x54, 0x0a], "vandps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x54, 0x4a, 0x01], "vandps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x55, 0x0a], "vandnps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x55, 0x4a, 0x01], "vandnps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x56, 0x0a], "vorps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x56, 0x4a, 0x01], "vorps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x57, 0x0a], "vxorps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x57, 0x4a, 0x01], "vxorps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x58, 0x0a], "vaddps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x58, 0x4a, 0x01], "vaddps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x58, 0xca], "vaddps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x59, 0x0a], "vmulps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x59, 0x4a, 0x01], "vmulps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x59, 0xca], "vmulps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x5a, 0x0a], "vcvtps2pd ymm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x5a, 0x4a, 0x01], "vcvtps2pd ymm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x5b, 0x0a], "vcvtdq2ps ymm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x5b, 0x4a, 0x01], "vcvtdq2ps ymm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x5b, 0xca], "vcvtdq2ps zmm1{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x5c, 0x0a], "vsubps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x5c, 0x4a, 0x01], "vsubps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x5c, 0xca], "vsubps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x5d, 0x0a], "vminps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x5d, 0x4a, 0x01], "vminps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x5e, 0x0a], "vdivps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x5e, 0x4a, 0x01], "vdivps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x5e, 0xca], "vdivps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x5f, 0x0a], "vmaxps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x5f, 0x4a, 0x01], "vmaxps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x78, 0x0a], "vcvttps2udq ymm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x78, 0x4a, 0x01], "vcvttps2udq ymm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x79, 0x0a], "vcvtps2udq ymm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x79, 0x4a, 0x01], "vcvtps2udq ymm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0x79, 0xca], "vcvtps2udq zmm1{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0xc2, 0x0a, 0xcc], "vcmpps k1, ymm0, dword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1, ymm0, dword [bp + si + 0x4]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0xc6, 0x0a, 0xcc], "vshufps ymm1, ymm0, dword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x38, 0xc6, 0x4a, 0x01, 0xcc], "vshufps ymm1, ymm0, dword [bp + si + 0x4]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x14, 0x0a], "vunpcklps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x14, 0x4a, 0x01], "vunpcklps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x15, 0x0a], "vunpckhps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x15, 0x4a, 0x01], "vunpckhps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x51, 0x0a], "vsqrtps ymm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x51, 0x4a, 0x01], "vsqrtps ymm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x51, 0xca], "vsqrtps zmm1{k5}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x54, 0x0a], "vandps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x54, 0x4a, 0x01], "vandps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x55, 0x0a], "vandnps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x55, 0x4a, 0x01], "vandnps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x56, 0x0a], "vorps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x56, 0x4a, 0x01], "vorps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x57, 0x0a], "vxorps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x57, 0x4a, 0x01], "vxorps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x58, 0x0a], "vaddps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x58, 0x4a, 0x01], "vaddps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x58, 0xca], "vaddps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x59, 0x0a], "vmulps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x59, 0x4a, 0x01], "vmulps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x59, 0xca], "vmulps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x5a, 0x0a], "vcvtps2pd ymm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x5a, 0x4a, 0x01], "vcvtps2pd ymm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x5b, 0x0a], "vcvtdq2ps ymm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x5b, 0x4a, 0x01], "vcvtdq2ps ymm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x5c, 0x0a], "vsubps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x5c, 0x4a, 0x01], "vsubps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x5c, 0xca], "vsubps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x5d, 0x0a], "vminps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x5d, 0x4a, 0x01], "vminps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x5e, 0x0a], "vdivps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x5e, 0x4a, 0x01], "vdivps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x5e, 0xca], "vdivps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x5f, 0x0a], "vmaxps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x5f, 0x4a, 0x01], "vmaxps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x78, 0x0a], "vcvttps2udq ymm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x78, 0x4a, 0x01], "vcvttps2udq ymm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x79, 0x0a], "vcvtps2udq ymm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x79, 0x4a, 0x01], "vcvtps2udq ymm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0x79, 0xca], "vcvtps2udq zmm1{k5}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, ymm0, dword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1{k5}, ymm0, dword [bp + si + 0x4]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0xc6, 0x0a, 0xcc], "vshufps ymm1{k5}, ymm0, dword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x3d, 0xc6, 0x4a, 0x01, 0xcc], "vshufps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x10, 0x0a], "vmovups zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x10, 0x4a, 0x01], "vmovups zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x10, 0xca], "vmovups zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x11, 0x0a], "vmovups zmmword [bp + si], zmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x11, 0x4a, 0x01], "vmovups zmmword [bp + si + 0x40], zmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x11, 0xca], "vmovups zmm2, zmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x14, 0x0a], "vunpcklps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x14, 0x4a, 0x01], "vunpcklps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x14, 0xca], "vunpcklps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x15, 0x0a], "vunpckhps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x15, 0x4a, 0x01], "vunpckhps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x15, 0xca], "vunpckhps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x28, 0x0a], "vmovaps zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x28, 0x4a, 0x01], "vmovaps zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x28, 0xca], "vmovaps zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x29, 0x0a], "vmovaps zmmword [bp + si], zmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x29, 0x4a, 0x01], "vmovaps zmmword [bp + si + 0x40], zmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x29, 0xca], "vmovaps zmm2, zmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x2b, 0x0a], "vmovntps zmmword [bp + si], zmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x2b, 0x4a, 0x01], "vmovntps zmmword [bp + si + 0x40], zmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x51, 0x0a], "vsqrtps zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x51, 0x4a, 0x01], "vsqrtps zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x51, 0xca], "vsqrtps zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x54, 0x0a], "vandps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x54, 0x4a, 0x01], "vandps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x54, 0xca], "vandps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x55, 0x0a], "vandnps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x55, 0x4a, 0x01], "vandnps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x55, 0xca], "vandnps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x56, 0x0a], "vorps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x56, 0x4a, 0x01], "vorps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x56, 0xca], "vorps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x57, 0x0a], "vxorps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x57, 0x4a, 0x01], "vxorps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x57, 0xca], "vxorps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x58, 0x0a], "vaddps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x58, 0x4a, 0x01], "vaddps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x58, 0xca], "vaddps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x59, 0x0a], "vmulps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x59, 0x4a, 0x01], "vmulps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x59, 0xca], "vmulps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5a, 0x0a], "vcvtps2pd zmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5a, 0x4a, 0x01], "vcvtps2pd zmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5a, 0xca], "vcvtps2pd zmm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5b, 0x0a], "vcvtdq2ps zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5b, 0x4a, 0x01], "vcvtdq2ps zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5b, 0xca], "vcvtdq2ps zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5c, 0x0a], "vsubps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5c, 0x4a, 0x01], "vsubps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5c, 0xca], "vsubps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5d, 0x0a], "vminps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5d, 0x4a, 0x01], "vminps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5d, 0xca], "vminps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5e, 0x0a], "vdivps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5e, 0x4a, 0x01], "vdivps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5e, 0xca], "vdivps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5f, 0x0a], "vmaxps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5f, 0x4a, 0x01], "vmaxps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x5f, 0xca], "vmaxps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x78, 0x0a], "vcvttps2udq zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x78, 0x4a, 0x01], "vcvttps2udq zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x78, 0xca], "vcvttps2udq zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x79, 0x0a], "vcvtps2udq zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x79, 0x4a, 0x01], "vcvtps2udq zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0x79, 0xca], "vcvtps2udq zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0xc2, 0x0a, 0xcc], "vcmpps k1, zmm0, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1, zmm0, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0xc2, 0xca, 0xcc], "vcmpps k1, zmm0, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0xc6, 0x0a, 0xcc], "vshufps zmm1, zmm0, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0xc6, 0x4a, 0x01, 0xcc], "vshufps zmm1, zmm0, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x48, 0xc6, 0xca, 0xcc], "vshufps zmm1, zmm0, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x10, 0x0a], "vmovups zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x10, 0x4a, 0x01], "vmovups zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x10, 0xca], "vmovups zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x11, 0x0a], "vmovups zmmword [bp + si]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x11, 0x4a, 0x01], "vmovups zmmword [bp + si + 0x40]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x11, 0xca], "vmovups zmm2{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x14, 0x0a], "vunpcklps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x14, 0x4a, 0x01], "vunpcklps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x14, 0xca], "vunpcklps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x15, 0x0a], "vunpckhps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x15, 0x4a, 0x01], "vunpckhps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x15, 0xca], "vunpckhps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x28, 0x0a], "vmovaps zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x28, 0x4a, 0x01], "vmovaps zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x28, 0xca], "vmovaps zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x29, 0x0a], "vmovaps zmmword [bp + si]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x29, 0x4a, 0x01], "vmovaps zmmword [bp + si + 0x40]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x29, 0xca], "vmovaps zmm2{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x51, 0x0a], "vsqrtps zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x51, 0x4a, 0x01], "vsqrtps zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x51, 0xca], "vsqrtps zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x54, 0x0a], "vandps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x54, 0x4a, 0x01], "vandps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x54, 0xca], "vandps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x55, 0x0a], "vandnps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x55, 0x4a, 0x01], "vandnps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x55, 0xca], "vandnps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x56, 0x0a], "vorps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x56, 0x4a, 0x01], "vorps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x56, 0xca], "vorps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x57, 0x0a], "vxorps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x57, 0x4a, 0x01], "vxorps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x57, 0xca], "vxorps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x58, 0x0a], "vaddps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x58, 0x4a, 0x01], "vaddps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x58, 0xca], "vaddps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x59, 0x0a], "vmulps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x59, 0x4a, 0x01], "vmulps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x59, 0xca], "vmulps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5a, 0x0a], "vcvtps2pd zmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5a, 0x4a, 0x01], "vcvtps2pd zmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5a, 0xca], "vcvtps2pd zmm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5b, 0x0a], "vcvtdq2ps zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5b, 0x4a, 0x01], "vcvtdq2ps zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5c, 0x0a], "vsubps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5c, 0x4a, 0x01], "vsubps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5c, 0xca], "vsubps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5d, 0x0a], "vminps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5d, 0x4a, 0x01], "vminps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5d, 0xca], "vminps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5e, 0x0a], "vdivps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5e, 0x4a, 0x01], "vdivps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5e, 0xca], "vdivps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5f, 0x0a], "vmaxps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5f, 0x4a, 0x01], "vmaxps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x5f, 0xca], "vmaxps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x78, 0x0a], "vcvttps2udq zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x78, 0x4a, 0x01], "vcvttps2udq zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x78, 0xca], "vcvttps2udq zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x79, 0x0a], "vcvtps2udq zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x79, 0x4a, 0x01], "vcvtps2udq zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0x79, 0xca], "vcvtps2udq zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, zmm0, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1{k5}, zmm0, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0xc2, 0xca, 0xcc], "vcmpps k1{k5}, zmm0, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0xc6, 0x0a, 0xcc], "vshufps zmm1{k5}, zmm0, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0xc6, 0x4a, 0x01, 0xcc], "vshufps zmm1{k5}, zmm0, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x4d, 0xc6, 0xca, 0xcc], "vshufps zmm1{k5}, zmm0, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x14, 0x0a], "vunpcklps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x14, 0x4a, 0x01], "vunpcklps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x15, 0x0a], "vunpckhps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x15, 0x4a, 0x01], "vunpckhps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x51, 0x0a], "vsqrtps zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x51, 0x4a, 0x01], "vsqrtps zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x51, 0xca], "vsqrtps zmm1{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x54, 0x0a], "vandps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x54, 0x4a, 0x01], "vandps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x55, 0x0a], "vandnps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x55, 0x4a, 0x01], "vandnps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x56, 0x0a], "vorps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x56, 0x4a, 0x01], "vorps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x57, 0x0a], "vxorps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x57, 0x4a, 0x01], "vxorps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x58, 0x0a], "vaddps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x58, 0x4a, 0x01], "vaddps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x58, 0xca], "vaddps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x59, 0x0a], "vmulps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x59, 0x4a, 0x01], "vmulps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x59, 0xca], "vmulps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x5a, 0x0a], "vcvtps2pd zmm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x5a, 0x4a, 0x01], "vcvtps2pd zmm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x5b, 0x0a], "vcvtdq2ps zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x5b, 0x4a, 0x01], "vcvtdq2ps zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x5b, 0xca], "vcvtdq2ps zmm1{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x5c, 0x0a], "vsubps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x5c, 0x4a, 0x01], "vsubps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x5c, 0xca], "vsubps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x5d, 0x0a], "vminps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x5d, 0x4a, 0x01], "vminps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x5e, 0x0a], "vdivps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x5e, 0x4a, 0x01], "vdivps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x5e, 0xca], "vdivps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x5f, 0x0a], "vmaxps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x5f, 0x4a, 0x01], "vmaxps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x78, 0x0a], "vcvttps2udq zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x78, 0x4a, 0x01], "vcvttps2udq zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x79, 0x0a], "vcvtps2udq zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x79, 0x4a, 0x01], "vcvtps2udq zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0x79, 0xca], "vcvtps2udq zmm1{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0xc2, 0x0a, 0xcc], "vcmpps k1, zmm0, dword [bp + si]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1, zmm0, dword [bp + si + 0x4]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0xc6, 0x0a, 0xcc], "vshufps zmm1, zmm0, dword [bp + si]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x58, 0xc6, 0x4a, 0x01, 0xcc], "vshufps zmm1, zmm0, dword [bp + si + 0x4]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x14, 0x0a], "vunpcklps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x14, 0x4a, 0x01], "vunpcklps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x15, 0x0a], "vunpckhps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x15, 0x4a, 0x01], "vunpckhps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x51, 0x0a], "vsqrtps zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x51, 0x4a, 0x01], "vsqrtps zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x51, 0xca], "vsqrtps zmm1{k5}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x54, 0x0a], "vandps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x54, 0x4a, 0x01], "vandps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x55, 0x0a], "vandnps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x55, 0x4a, 0x01], "vandnps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x56, 0x0a], "vorps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x56, 0x4a, 0x01], "vorps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x57, 0x0a], "vxorps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x57, 0x4a, 0x01], "vxorps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x58, 0x0a], "vaddps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x58, 0x4a, 0x01], "vaddps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x58, 0xca], "vaddps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x59, 0x0a], "vmulps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x59, 0x4a, 0x01], "vmulps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x59, 0xca], "vmulps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x5a, 0x0a], "vcvtps2pd zmm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x5a, 0x4a, 0x01], "vcvtps2pd zmm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x5b, 0x0a], "vcvtdq2ps zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x5b, 0x4a, 0x01], "vcvtdq2ps zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x5c, 0x0a], "vsubps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x5c, 0x4a, 0x01], "vsubps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x5c, 0xca], "vsubps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x5d, 0x0a], "vminps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x5d, 0x4a, 0x01], "vminps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x5e, 0x0a], "vdivps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x5e, 0x4a, 0x01], "vdivps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x5e, 0xca], "vdivps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x5f, 0x0a], "vmaxps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x5f, 0x4a, 0x01], "vmaxps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x78, 0x0a], "vcvttps2udq zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x78, 0x4a, 0x01], "vcvttps2udq zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x79, 0x0a], "vcvtps2udq zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x79, 0x4a, 0x01], "vcvtps2udq zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0x79, 0xca], "vcvtps2udq zmm1{k5}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, zmm0, dword [bp + si]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0xc2, 0x4a, 0x01, 0xcc], "vcmpps k1{k5}, zmm0, dword [bp + si + 0x4]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0xc6, 0x0a, 0xcc], "vshufps zmm1{k5}, zmm0, dword [bp + si]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x5d, 0xc6, 0x4a, 0x01, 0xcc], "vshufps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x78, 0x2e, 0xca], "vucomiss xmm1{sae}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x78, 0x2f, 0xca], "vcomiss xmm1{sae}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x78, 0x51, 0xca], "vsqrtps zmm1{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x78, 0x58, 0xca], "vaddps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x78, 0x59, 0xca], "vmulps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x78, 0x5a, 0xca], "vcvtps2pd zmm1{sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x78, 0x5b, 0xca], "vcvtdq2ps zmm1{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x78, 0x5c, 0xca], "vsubps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x78, 0x5d, 0xca], "vminps zmm1{sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x78, 0x5e, 0xca], "vdivps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x78, 0x5f, 0xca], "vmaxps zmm1{sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x78, 0x78, 0xca], "vcvttps2udq zmm1{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x78, 0x79, 0xca], "vcvtps2udq zmm1{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x78, 0xc2, 0xca, 0xcc], "vcmpps k1{sae}, zmm0, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x7d, 0x51, 0xca], "vsqrtps zmm1{k5}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x7d, 0x58, 0xca], "vaddps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x7d, 0x59, 0xca], "vmulps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x7d, 0x5a, 0xca], "vcvtps2pd zmm1{k5}{sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x7d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x7d, 0x5c, 0xca], "vsubps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x7d, 0x5d, 0xca], "vminps zmm1{k5}{sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x7d, 0x5e, 0xca], "vdivps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x7d, 0x5f, 0xca], "vmaxps zmm1{k5}{sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x7d, 0x78, 0xca], "vcvttps2udq zmm1{k5}{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x7d, 0x79, 0xca], "vcvtps2udq zmm1{k5}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x7d, 0xc2, 0xca, 0xcc], "vcmpps k1{k5}{sae}, zmm0, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x10, 0x0a], "vmovups xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x10, 0x4a, 0x01], "vmovups xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x10, 0xca], "vmovups xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x11, 0xca], "vmovups xmm2{k5}{z}, xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x14, 0x0a], "vunpcklps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x14, 0x4a, 0x01], "vunpcklps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x14, 0xca], "vunpcklps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x15, 0x0a], "vunpckhps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x15, 0x4a, 0x01], "vunpckhps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x15, 0xca], "vunpckhps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x28, 0x0a], "vmovaps xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x28, 0x4a, 0x01], "vmovaps xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x28, 0xca], "vmovaps xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x29, 0xca], "vmovaps xmm2{k5}{z}, xmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x51, 0x0a], "vsqrtps xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x51, 0x4a, 0x01], "vsqrtps xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x51, 0xca], "vsqrtps xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x54, 0x0a], "vandps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x54, 0x4a, 0x01], "vandps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x54, 0xca], "vandps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x55, 0x0a], "vandnps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x55, 0x4a, 0x01], "vandnps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x55, 0xca], "vandnps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x56, 0x0a], "vorps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x56, 0x4a, 0x01], "vorps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x56, 0xca], "vorps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x57, 0x0a], "vxorps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x57, 0x4a, 0x01], "vxorps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x57, 0xca], "vxorps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x58, 0x0a], "vaddps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x58, 0x4a, 0x01], "vaddps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x58, 0xca], "vaddps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x59, 0x0a], "vmulps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x59, 0x4a, 0x01], "vmulps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x59, 0xca], "vmulps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5a, 0x0a], "vcvtps2pd xmm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5a, 0x4a, 0x01], "vcvtps2pd xmm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5a, 0xca], "vcvtps2pd xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5b, 0x0a], "vcvtdq2ps xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5b, 0x4a, 0x01], "vcvtdq2ps xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5b, 0xca], "vcvtdq2ps xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5c, 0x0a], "vsubps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5c, 0x4a, 0x01], "vsubps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5c, 0xca], "vsubps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5d, 0x0a], "vminps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5d, 0x4a, 0x01], "vminps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5d, 0xca], "vminps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5e, 0x0a], "vdivps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5e, 0x4a, 0x01], "vdivps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5e, 0xca], "vdivps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5f, 0x0a], "vmaxps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5f, 0x4a, 0x01], "vmaxps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x5f, 0xca], "vmaxps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x78, 0x0a], "vcvttps2udq xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x78, 0x4a, 0x01], "vcvttps2udq xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x78, 0xca], "vcvttps2udq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x79, 0x0a], "vcvtps2udq xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x79, 0x4a, 0x01], "vcvtps2udq xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0x79, 0xca], "vcvtps2udq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0xc6, 0x0a, 0xcc], "vshufps xmm1{k5}{z}, xmm0, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0xc6, 0x4a, 0x01, 0xcc], "vshufps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x8d, 0xc6, 0xca, 0xcc], "vshufps xmm1{k5}{z}, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x14, 0x0a], "vunpcklps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x14, 0x4a, 0x01], "vunpcklps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x15, 0x0a], "vunpckhps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x15, 0x4a, 0x01], "vunpckhps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x51, 0x0a], "vsqrtps xmm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x51, 0x4a, 0x01], "vsqrtps xmm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x51, 0xca], "vsqrtps zmm1{k5}{z}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x54, 0x0a], "vandps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x54, 0x4a, 0x01], "vandps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x55, 0x0a], "vandnps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x55, 0x4a, 0x01], "vandnps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x56, 0x0a], "vorps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x56, 0x4a, 0x01], "vorps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x57, 0x0a], "vxorps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x57, 0x4a, 0x01], "vxorps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x58, 0x0a], "vaddps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x58, 0x4a, 0x01], "vaddps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x58, 0xca], "vaddps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x59, 0x0a], "vmulps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x59, 0x4a, 0x01], "vmulps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x59, 0xca], "vmulps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x5a, 0x0a], "vcvtps2pd xmm1{k5}{z}, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x5a, 0x4a, 0x01], "vcvtps2pd xmm1{k5}{z}, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x5b, 0x0a], "vcvtdq2ps xmm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x5b, 0x4a, 0x01], "vcvtdq2ps xmm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{z}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x5c, 0x0a], "vsubps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x5c, 0x4a, 0x01], "vsubps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x5c, 0xca], "vsubps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x5d, 0x0a], "vminps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x5d, 0x4a, 0x01], "vminps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x5e, 0x0a], "vdivps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x5e, 0x4a, 0x01], "vdivps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x5e, 0xca], "vdivps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x5f, 0x0a], "vmaxps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x5f, 0x4a, 0x01], "vmaxps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x78, 0x0a], "vcvttps2udq xmm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x78, 0x4a, 0x01], "vcvttps2udq xmm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x79, 0x0a], "vcvtps2udq xmm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x79, 0x4a, 0x01], "vcvtps2udq xmm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0x79, 0xca], "vcvtps2udq zmm1{k5}{z}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0xc6, 0x0a, 0xcc], "vshufps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0x9d, 0xc6, 0x4a, 0x01, 0xcc], "vshufps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x10, 0x0a], "vmovups ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x10, 0x4a, 0x01], "vmovups ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x10, 0xca], "vmovups ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x11, 0xca], "vmovups ymm2{k5}{z}, ymm1");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x14, 0x0a], "vunpcklps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x14, 0x4a, 0x01], "vunpcklps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x14, 0xca], "vunpcklps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x15, 0x0a], "vunpckhps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x15, 0x4a, 0x01], "vunpckhps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x15, 0xca], "vunpckhps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x28, 0x0a], "vmovaps ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x28, 0x4a, 0x01], "vmovaps ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x28, 0xca], "vmovaps ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x29, 0xca], "vmovaps ymm2{k5}{z}, ymm1");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x51, 0x0a], "vsqrtps ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x51, 0x4a, 0x01], "vsqrtps ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x51, 0xca], "vsqrtps ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x54, 0x0a], "vandps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x54, 0x4a, 0x01], "vandps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x54, 0xca], "vandps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x55, 0x0a], "vandnps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x55, 0x4a, 0x01], "vandnps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x55, 0xca], "vandnps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x56, 0x0a], "vorps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x56, 0x4a, 0x01], "vorps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x56, 0xca], "vorps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x57, 0x0a], "vxorps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x57, 0x4a, 0x01], "vxorps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x57, 0xca], "vxorps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x58, 0x0a], "vaddps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x58, 0x4a, 0x01], "vaddps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x58, 0xca], "vaddps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x59, 0x0a], "vmulps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x59, 0x4a, 0x01], "vmulps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x59, 0xca], "vmulps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5a, 0x0a], "vcvtps2pd ymm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5a, 0x4a, 0x01], "vcvtps2pd ymm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5a, 0xca], "vcvtps2pd ymm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5b, 0x0a], "vcvtdq2ps ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5b, 0x4a, 0x01], "vcvtdq2ps ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5b, 0xca], "vcvtdq2ps ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5c, 0x0a], "vsubps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5c, 0x4a, 0x01], "vsubps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5c, 0xca], "vsubps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5d, 0x0a], "vminps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5d, 0x4a, 0x01], "vminps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5d, 0xca], "vminps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5e, 0x0a], "vdivps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5e, 0x4a, 0x01], "vdivps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5e, 0xca], "vdivps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5f, 0x0a], "vmaxps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5f, 0x4a, 0x01], "vmaxps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x5f, 0xca], "vmaxps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x78, 0x0a], "vcvttps2udq ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x78, 0x4a, 0x01], "vcvttps2udq ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x78, 0xca], "vcvttps2udq ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x79, 0x0a], "vcvtps2udq ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x79, 0x4a, 0x01], "vcvtps2udq ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0x79, 0xca], "vcvtps2udq ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0xc6, 0x0a, 0xcc], "vshufps ymm1{k5}{z}, ymm0, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0xc6, 0x4a, 0x01, 0xcc], "vshufps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0xad, 0xc6, 0xca, 0xcc], "vshufps ymm1{k5}{z}, ymm0, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x14, 0x0a], "vunpcklps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x14, 0x4a, 0x01], "vunpcklps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x15, 0x0a], "vunpckhps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x15, 0x4a, 0x01], "vunpckhps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x51, 0x0a], "vsqrtps ymm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x51, 0x4a, 0x01], "vsqrtps ymm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x51, 0xca], "vsqrtps zmm1{k5}{z}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x54, 0x0a], "vandps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x54, 0x4a, 0x01], "vandps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x55, 0x0a], "vandnps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x55, 0x4a, 0x01], "vandnps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x56, 0x0a], "vorps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x56, 0x4a, 0x01], "vorps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x57, 0x0a], "vxorps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x57, 0x4a, 0x01], "vxorps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x58, 0x0a], "vaddps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x58, 0x4a, 0x01], "vaddps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x58, 0xca], "vaddps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x59, 0x0a], "vmulps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x59, 0x4a, 0x01], "vmulps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x59, 0xca], "vmulps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x5a, 0x0a], "vcvtps2pd ymm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x5a, 0x4a, 0x01], "vcvtps2pd ymm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x5b, 0x0a], "vcvtdq2ps ymm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x5b, 0x4a, 0x01], "vcvtdq2ps ymm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{z}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x5c, 0x0a], "vsubps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x5c, 0x4a, 0x01], "vsubps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x5c, 0xca], "vsubps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x5d, 0x0a], "vminps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x5d, 0x4a, 0x01], "vminps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x5e, 0x0a], "vdivps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x5e, 0x4a, 0x01], "vdivps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x5e, 0xca], "vdivps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x5f, 0x0a], "vmaxps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x5f, 0x4a, 0x01], "vmaxps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x78, 0x0a], "vcvttps2udq ymm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x78, 0x4a, 0x01], "vcvttps2udq ymm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x79, 0x0a], "vcvtps2udq ymm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x79, 0x4a, 0x01], "vcvtps2udq ymm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0x79, 0xca], "vcvtps2udq zmm1{k5}{z}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0xc6, 0x0a, 0xcc], "vshufps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0xbd, 0xc6, 0x4a, 0x01, 0xcc], "vshufps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x10, 0x0a], "vmovups zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x10, 0x4a, 0x01], "vmovups zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x10, 0xca], "vmovups zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x11, 0xca], "vmovups zmm2{k5}{z}, zmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x14, 0x0a], "vunpcklps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x14, 0x4a, 0x01], "vunpcklps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x14, 0xca], "vunpcklps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x15, 0x0a], "vunpckhps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x15, 0x4a, 0x01], "vunpckhps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x15, 0xca], "vunpckhps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x28, 0x0a], "vmovaps zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x28, 0x4a, 0x01], "vmovaps zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x28, 0xca], "vmovaps zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x29, 0xca], "vmovaps zmm2{k5}{z}, zmm1");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x51, 0x0a], "vsqrtps zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x51, 0x4a, 0x01], "vsqrtps zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x51, 0xca], "vsqrtps zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x54, 0x0a], "vandps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x54, 0x4a, 0x01], "vandps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x54, 0xca], "vandps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x55, 0x0a], "vandnps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x55, 0x4a, 0x01], "vandnps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x55, 0xca], "vandnps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x56, 0x0a], "vorps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x56, 0x4a, 0x01], "vorps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x56, 0xca], "vorps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x57, 0x0a], "vxorps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x57, 0x4a, 0x01], "vxorps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x57, 0xca], "vxorps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x58, 0x0a], "vaddps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x58, 0x4a, 0x01], "vaddps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x58, 0xca], "vaddps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x59, 0x0a], "vmulps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x59, 0x4a, 0x01], "vmulps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x59, 0xca], "vmulps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5a, 0x0a], "vcvtps2pd zmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5a, 0x4a, 0x01], "vcvtps2pd zmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5a, 0xca], "vcvtps2pd zmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5b, 0x0a], "vcvtdq2ps zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5b, 0x4a, 0x01], "vcvtdq2ps zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5c, 0x0a], "vsubps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5c, 0x4a, 0x01], "vsubps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5c, 0xca], "vsubps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5d, 0x0a], "vminps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5d, 0x4a, 0x01], "vminps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5d, 0xca], "vminps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5e, 0x0a], "vdivps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5e, 0x4a, 0x01], "vdivps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5e, 0xca], "vdivps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5f, 0x0a], "vmaxps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5f, 0x4a, 0x01], "vmaxps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x5f, 0xca], "vmaxps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x78, 0x0a], "vcvttps2udq zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x78, 0x4a, 0x01], "vcvttps2udq zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x78, 0xca], "vcvttps2udq zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x79, 0x0a], "vcvtps2udq zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x79, 0x4a, 0x01], "vcvtps2udq zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0x79, 0xca], "vcvtps2udq zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0xc6, 0x0a, 0xcc], "vshufps zmm1{k5}{z}, zmm0, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0xc6, 0x4a, 0x01, 0xcc], "vshufps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0xcd, 0xc6, 0xca, 0xcc], "vshufps zmm1{k5}{z}, zmm0, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x14, 0x0a], "vunpcklps zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x14, 0x4a, 0x01], "vunpcklps zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x15, 0x0a], "vunpckhps zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x15, 0x4a, 0x01], "vunpckhps zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x51, 0x0a], "vsqrtps zmm1{k5}{z}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x51, 0x4a, 0x01], "vsqrtps zmm1{k5}{z}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x51, 0xca], "vsqrtps zmm1{k5}{z}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x54, 0x0a], "vandps zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x54, 0x4a, 0x01], "vandps zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x55, 0x0a], "vandnps zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x55, 0x4a, 0x01], "vandnps zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x56, 0x0a], "vorps zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x56, 0x4a, 0x01], "vorps zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x57, 0x0a], "vxorps zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x57, 0x4a, 0x01], "vxorps zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x58, 0x0a], "vaddps zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x58, 0x4a, 0x01], "vaddps zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x58, 0xca], "vaddps zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x59, 0x0a], "vmulps zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x59, 0x4a, 0x01], "vmulps zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x59, 0xca], "vmulps zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x5a, 0x0a], "vcvtps2pd zmm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x5a, 0x4a, 0x01], "vcvtps2pd zmm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x5b, 0x0a], "vcvtdq2ps zmm1{k5}{z}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x5b, 0x4a, 0x01], "vcvtdq2ps zmm1{k5}{z}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{z}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x5c, 0x0a], "vsubps zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x5c, 0x4a, 0x01], "vsubps zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x5c, 0xca], "vsubps zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x5d, 0x0a], "vminps zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x5d, 0x4a, 0x01], "vminps zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x5e, 0x0a], "vdivps zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x5e, 0x4a, 0x01], "vdivps zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x5e, 0xca], "vdivps zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x5f, 0x0a], "vmaxps zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x5f, 0x4a, 0x01], "vmaxps zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x78, 0x0a], "vcvttps2udq zmm1{k5}{z}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x78, 0x4a, 0x01], "vcvttps2udq zmm1{k5}{z}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x79, 0x0a], "vcvtps2udq zmm1{k5}{z}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x79, 0x4a, 0x01], "vcvtps2udq zmm1{k5}{z}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0x79, 0xca], "vcvtps2udq zmm1{k5}{z}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0xc6, 0x0a, 0xcc], "vshufps zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0xdd, 0xc6, 0x4a, 0x01, 0xcc], "vshufps zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7c, 0xfd, 0x51, 0xca], "vsqrtps zmm1{k5}{z}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xfd, 0x58, 0xca], "vaddps zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xfd, 0x59, 0xca], "vmulps zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xfd, 0x5a, 0xca], "vcvtps2pd zmm1{k5}{z}{sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xfd, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{z}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xfd, 0x5c, 0xca], "vsubps zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xfd, 0x5d, 0xca], "vminps zmm1{k5}{z}{sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xfd, 0x5e, 0xca], "vdivps zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xfd, 0x5f, 0xca], "vmaxps zmm1{k5}{z}{sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xfd, 0x78, 0xca], "vcvttps2udq zmm1{k5}{z}{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7c, 0xfd, 0x79, 0xca], "vcvtps2udq zmm1{k5}{z}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x5b, 0x0a], "vcvtps2dq xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x5b, 0x4a, 0x01], "vcvtps2dq xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x5b, 0xca], "vcvtps2dq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x62, 0x0a], "vpunpckldq xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x62, 0x4a, 0x01], "vpunpckldq xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x62, 0xca], "vpunpckldq xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x66, 0x0a], "vpcmpgtd k1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x66, 0x4a, 0x01], "vpcmpgtd k1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x66, 0xca], "vpcmpgtd k1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x6a, 0x0a], "vpunpckhdq xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x6a, 0x4a, 0x01], "vpunpckhdq xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x6a, 0xca], "vpunpckhdq xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x6b, 0x0a], "vpackssdw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x6b, 0x4a, 0x01], "vpackssdw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x6b, 0xca], "vpackssdw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x6f, 0x0a], "vmovdqa32 xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x6f, 0x4a, 0x01], "vmovdqa32 xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x6f, 0xca], "vmovdqa32 xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x70, 0x0a, 0xcc], "vpshufd xmm1, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x70, 0x4a, 0x01, 0xcc], "vpshufd xmm1, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x70, 0xca, 0xcc], "vpshufd xmm1, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x72, 0x0a, 0xcc], "vprold xmm0, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x72, 0x4a, 0x01, 0xcc], "vprold xmm0, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x72, 0xca, 0xcc], "vprold xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x76, 0x0a], "vpcmpeqd k1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x76, 0x4a, 0x01], "vpcmpeqd k1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x76, 0xca], "vpcmpeqd k1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x78, 0x0a], "vcvttps2uqq xmm1, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x78, 0x4a, 0x01], "vcvttps2uqq xmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x78, 0xca], "vcvttps2uqq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x79, 0x0a], "vcvtps2uqq xmm1, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x79, 0x4a, 0x01], "vcvtps2uqq xmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x79, 0xca], "vcvtps2uqq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x7a, 0x0a], "vcvttps2qq xmm1, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x7a, 0x4a, 0x01], "vcvttps2qq xmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x7a, 0xca], "vcvttps2qq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x7b, 0x0a], "vcvtps2qq xmm1, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x7b, 0x4a, 0x01], "vcvtps2qq xmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x7b, 0xca], "vcvtps2qq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x7f, 0x0a], "vmovdqa32 xmmword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x7f, 0x4a, 0x01], "vmovdqa32 xmmword [bp + si + 0x10], xmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0x7f, 0xca], "vmovdqa32 xmm2, xmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xd2, 0x0a], "vpsrld xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xd2, 0x4a, 0x01], "vpsrld xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xd2, 0xca], "vpsrld xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xdb, 0x0a], "vpandd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xdb, 0x4a, 0x01], "vpandd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xdb, 0xca], "vpandd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xdf, 0x0a], "vpandnd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xdf, 0x4a, 0x01], "vpandnd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xdf, 0xca], "vpandnd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xe2, 0x0a], "vpsrad xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xe2, 0x4a, 0x01], "vpsrad xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xe2, 0xca], "vpsrad xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xe7, 0x0a], "vmovntdq xmmword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xe7, 0x4a, 0x01], "vmovntdq xmmword [bp + si + 0x10], xmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xeb, 0x0a], "vpord xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xeb, 0x4a, 0x01], "vpord xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xeb, 0xca], "vpord xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xef, 0x0a], "vpxord xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xef, 0x4a, 0x01], "vpxord xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xef, 0xca], "vpxord xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xf2, 0x0a], "vpslld xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xf2, 0x4a, 0x01], "vpslld xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xf2, 0xca], "vpslld xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xfa, 0x0a], "vpsubd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xfa, 0x4a, 0x01], "vpsubd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xfa, 0xca], "vpsubd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xfe, 0x0a], "vpaddd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xfe, 0x4a, 0x01], "vpaddd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x08, 0xfe, 0xca], "vpaddd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x5b, 0x0a], "vcvtps2dq xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x5b, 0x4a, 0x01], "vcvtps2dq xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x5b, 0xca], "vcvtps2dq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x62, 0x0a], "vpunpckldq xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x62, 0x4a, 0x01], "vpunpckldq xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x62, 0xca], "vpunpckldq xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x66, 0x0a], "vpcmpgtd k1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x66, 0x4a, 0x01], "vpcmpgtd k1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x66, 0xca], "vpcmpgtd k1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x6a, 0x0a], "vpunpckhdq xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x6a, 0x4a, 0x01], "vpunpckhdq xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x6a, 0xca], "vpunpckhdq xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x6b, 0x0a], "vpackssdw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x6b, 0x4a, 0x01], "vpackssdw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x6b, 0xca], "vpackssdw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x6f, 0x0a], "vmovdqa32 xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x6f, 0x4a, 0x01], "vmovdqa32 xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x6f, 0xca], "vmovdqa32 xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x70, 0x0a, 0xcc], "vpshufd xmm1{k5}, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x70, 0x4a, 0x01, 0xcc], "vpshufd xmm1{k5}, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x70, 0xca, 0xcc], "vpshufd xmm1{k5}, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x72, 0x0a, 0xcc], "vprold xmm0{k5}, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x72, 0x4a, 0x01, 0xcc], "vprold xmm0{k5}, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x72, 0xca, 0xcc], "vprold xmm0{k5}, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x76, 0x0a], "vpcmpeqd k1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x76, 0x4a, 0x01], "vpcmpeqd k1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x76, 0xca], "vpcmpeqd k1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x78, 0x0a], "vcvttps2uqq xmm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x78, 0x4a, 0x01], "vcvttps2uqq xmm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x78, 0xca], "vcvttps2uqq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x79, 0x0a], "vcvtps2uqq xmm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x79, 0x4a, 0x01], "vcvtps2uqq xmm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x79, 0xca], "vcvtps2uqq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x7a, 0x0a], "vcvttps2qq xmm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x7a, 0x4a, 0x01], "vcvttps2qq xmm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x7a, 0xca], "vcvttps2qq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x7b, 0x0a], "vcvtps2qq xmm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x7b, 0x4a, 0x01], "vcvtps2qq xmm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x7b, 0xca], "vcvtps2qq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x7f, 0x0a], "vmovdqa32 xmmword [bp + si]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x7f, 0x4a, 0x01], "vmovdqa32 xmmword [bp + si + 0x10]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0x7f, 0xca], "vmovdqa32 xmm2{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xd2, 0x0a], "vpsrld xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xd2, 0x4a, 0x01], "vpsrld xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xd2, 0xca], "vpsrld xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xdb, 0x0a], "vpandd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xdb, 0x4a, 0x01], "vpandd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xdb, 0xca], "vpandd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xdf, 0x0a], "vpandnd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xdf, 0x4a, 0x01], "vpandnd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xdf, 0xca], "vpandnd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xe2, 0x0a], "vpsrad xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xe2, 0x4a, 0x01], "vpsrad xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xe2, 0xca], "vpsrad xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xeb, 0x0a], "vpord xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xeb, 0x4a, 0x01], "vpord xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xeb, 0xca], "vpord xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xef, 0x0a], "vpxord xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xef, 0x4a, 0x01], "vpxord xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xef, 0xca], "vpxord xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xf2, 0x0a], "vpslld xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xf2, 0x4a, 0x01], "vpslld xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xf2, 0xca], "vpslld xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xfa, 0x0a], "vpsubd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xfa, 0x4a, 0x01], "vpsubd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xfa, 0xca], "vpsubd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xfe, 0x0a], "vpaddd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xfe, 0x4a, 0x01], "vpaddd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x0d, 0xfe, 0xca], "vpaddd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x5b, 0x0a], "vcvtps2dq xmm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x5b, 0x4a, 0x01], "vcvtps2dq xmm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x5b, 0xca], "vcvtps2dq zmm1{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x62, 0x0a], "vpunpckldq xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x62, 0x4a, 0x01], "vpunpckldq xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x66, 0x0a], "vpcmpgtd k1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x66, 0x4a, 0x01], "vpcmpgtd k1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x6a, 0x0a], "vpunpckhdq xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x6a, 0x4a, 0x01], "vpunpckhdq xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x6b, 0x0a], "vpackssdw xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x6b, 0x4a, 0x01], "vpackssdw xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x70, 0x0a, 0xcc], "vpshufd xmm1, dword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x70, 0x4a, 0x01, 0xcc], "vpshufd xmm1, dword [bp + si + 0x4]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x72, 0x0a, 0xcc], "vprold xmm0, dword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x72, 0x4a, 0x01, 0xcc], "vprold xmm0, dword [bp + si + 0x4]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x76, 0x0a], "vpcmpeqd k1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x76, 0x4a, 0x01], "vpcmpeqd k1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x78, 0x0a], "vcvttps2uqq xmm1, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x78, 0x4a, 0x01], "vcvttps2uqq xmm1, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x79, 0x0a], "vcvtps2uqq xmm1, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x79, 0x4a, 0x01], "vcvtps2uqq xmm1, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x79, 0xca], "vcvtps2uqq zmm1{rne-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x7a, 0x0a], "vcvttps2qq xmm1, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x7a, 0x4a, 0x01], "vcvttps2qq xmm1, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x7b, 0x0a], "vcvtps2qq xmm1, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x7b, 0x4a, 0x01], "vcvtps2qq xmm1, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0x7b, 0xca], "vcvtps2qq zmm1{rne-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0xdb, 0x0a], "vpandd xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0xdb, 0x4a, 0x01], "vpandd xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0xdf, 0x0a], "vpandnd xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0xdf, 0x4a, 0x01], "vpandnd xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0xeb, 0x0a], "vpord xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0xeb, 0x4a, 0x01], "vpord xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0xef, 0x0a], "vpxord xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0xef, 0x4a, 0x01], "vpxord xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0xfa, 0x0a], "vpsubd xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0xfa, 0x4a, 0x01], "vpsubd xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0xfe, 0x0a], "vpaddd xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x18, 0xfe, 0x4a, 0x01], "vpaddd xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x5b, 0x0a], "vcvtps2dq xmm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x5b, 0x4a, 0x01], "vcvtps2dq xmm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x62, 0x0a], "vpunpckldq xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x62, 0x4a, 0x01], "vpunpckldq xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x66, 0x0a], "vpcmpgtd k1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x66, 0x4a, 0x01], "vpcmpgtd k1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x6a, 0x0a], "vpunpckhdq xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x6a, 0x4a, 0x01], "vpunpckhdq xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x6b, 0x0a], "vpackssdw xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x6b, 0x4a, 0x01], "vpackssdw xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x70, 0x0a, 0xcc], "vpshufd xmm1{k5}, dword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x70, 0x4a, 0x01, 0xcc], "vpshufd xmm1{k5}, dword [bp + si + 0x4]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x72, 0x0a, 0xcc], "vprold xmm0{k5}, dword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x72, 0x4a, 0x01, 0xcc], "vprold xmm0{k5}, dword [bp + si + 0x4]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x76, 0x0a], "vpcmpeqd k1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x76, 0x4a, 0x01], "vpcmpeqd k1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x78, 0x0a], "vcvttps2uqq xmm1{k5}, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x78, 0x4a, 0x01], "vcvttps2uqq xmm1{k5}, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x79, 0x0a], "vcvtps2uqq xmm1{k5}, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x79, 0x4a, 0x01], "vcvtps2uqq xmm1{k5}, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{rne-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x7a, 0x0a], "vcvttps2qq xmm1{k5}, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x7a, 0x4a, 0x01], "vcvttps2qq xmm1{k5}, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x7b, 0x0a], "vcvtps2qq xmm1{k5}, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x7b, 0x4a, 0x01], "vcvtps2qq xmm1{k5}, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{rne-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0xdb, 0x0a], "vpandd xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0xdb, 0x4a, 0x01], "vpandd xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0xdf, 0x0a], "vpandnd xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0xdf, 0x4a, 0x01], "vpandnd xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0xeb, 0x0a], "vpord xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0xeb, 0x4a, 0x01], "vpord xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0xef, 0x0a], "vpxord xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0xef, 0x4a, 0x01], "vpxord xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0xfa, 0x0a], "vpsubd xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0xfa, 0x4a, 0x01], "vpsubd xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0xfe, 0x0a], "vpaddd xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x1d, 0xfe, 0x4a, 0x01], "vpaddd xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x5b, 0x0a], "vcvtps2dq ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x5b, 0x4a, 0x01], "vcvtps2dq ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x5b, 0xca], "vcvtps2dq ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x62, 0x0a], "vpunpckldq ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x62, 0x4a, 0x01], "vpunpckldq ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x62, 0xca], "vpunpckldq ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x66, 0x0a], "vpcmpgtd k1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x66, 0x4a, 0x01], "vpcmpgtd k1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x66, 0xca], "vpcmpgtd k1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x6a, 0x0a], "vpunpckhdq ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x6a, 0x4a, 0x01], "vpunpckhdq ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x6a, 0xca], "vpunpckhdq ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x6b, 0x0a], "vpackssdw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x6b, 0x4a, 0x01], "vpackssdw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x6b, 0xca], "vpackssdw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x6f, 0x0a], "vmovdqa32 ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x6f, 0x4a, 0x01], "vmovdqa32 ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x6f, 0xca], "vmovdqa32 ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x70, 0x0a, 0xcc], "vpshufd ymm1, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x70, 0x4a, 0x01, 0xcc], "vpshufd ymm1, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x70, 0xca, 0xcc], "vpshufd ymm1, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x72, 0x0a, 0xcc], "vprold ymm0, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x72, 0x4a, 0x01, 0xcc], "vprold ymm0, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x72, 0xca, 0xcc], "vprold ymm0, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x76, 0x0a], "vpcmpeqd k1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x76, 0x4a, 0x01], "vpcmpeqd k1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x76, 0xca], "vpcmpeqd k1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x78, 0x0a], "vcvttps2uqq ymm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x78, 0x4a, 0x01], "vcvttps2uqq ymm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x78, 0xca], "vcvttps2uqq ymm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x79, 0x0a], "vcvtps2uqq ymm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x79, 0x4a, 0x01], "vcvtps2uqq ymm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x79, 0xca], "vcvtps2uqq ymm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x7a, 0x0a], "vcvttps2qq ymm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x7a, 0x4a, 0x01], "vcvttps2qq ymm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x7a, 0xca], "vcvttps2qq ymm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x7b, 0x0a], "vcvtps2qq ymm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x7b, 0x4a, 0x01], "vcvtps2qq ymm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x7b, 0xca], "vcvtps2qq ymm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x7f, 0x0a], "vmovdqa32 ymmword [bp + si], ymm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x7f, 0x4a, 0x01], "vmovdqa32 ymmword [bp + si + 0x20], ymm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0x7f, 0xca], "vmovdqa32 ymm2, ymm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xd2, 0x0a], "vpsrld ymm1, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xd2, 0x4a, 0x01], "vpsrld ymm1, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xd2, 0xca], "vpsrld ymm1, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xdb, 0x0a], "vpandd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xdb, 0x4a, 0x01], "vpandd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xdb, 0xca], "vpandd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xdf, 0x0a], "vpandnd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xdf, 0x4a, 0x01], "vpandnd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xdf, 0xca], "vpandnd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xe2, 0x0a], "vpsrad ymm1, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xe2, 0x4a, 0x01], "vpsrad ymm1, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xe2, 0xca], "vpsrad ymm1, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xe7, 0x0a], "vmovntdq ymmword [bp + si], ymm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xe7, 0x4a, 0x01], "vmovntdq ymmword [bp + si + 0x20], ymm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xeb, 0x0a], "vpord ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xeb, 0x4a, 0x01], "vpord ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xeb, 0xca], "vpord ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xef, 0x0a], "vpxord ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xef, 0x4a, 0x01], "vpxord ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xef, 0xca], "vpxord ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xf2, 0x0a], "vpslld ymm1, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xf2, 0x4a, 0x01], "vpslld ymm1, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xf2, 0xca], "vpslld ymm1, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xfa, 0x0a], "vpsubd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xfa, 0x4a, 0x01], "vpsubd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xfa, 0xca], "vpsubd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xfe, 0x0a], "vpaddd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xfe, 0x4a, 0x01], "vpaddd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x28, 0xfe, 0xca], "vpaddd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x5b, 0x0a], "vcvtps2dq ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x5b, 0x4a, 0x01], "vcvtps2dq ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x5b, 0xca], "vcvtps2dq ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x62, 0x0a], "vpunpckldq ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x62, 0x4a, 0x01], "vpunpckldq ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x62, 0xca], "vpunpckldq ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x66, 0x0a], "vpcmpgtd k1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x66, 0x4a, 0x01], "vpcmpgtd k1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x66, 0xca], "vpcmpgtd k1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x6a, 0x0a], "vpunpckhdq ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x6a, 0x4a, 0x01], "vpunpckhdq ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x6a, 0xca], "vpunpckhdq ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x6b, 0x0a], "vpackssdw ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x6b, 0x4a, 0x01], "vpackssdw ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x6b, 0xca], "vpackssdw ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x6f, 0x0a], "vmovdqa32 ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x6f, 0x4a, 0x01], "vmovdqa32 ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x6f, 0xca], "vmovdqa32 ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x70, 0x0a, 0xcc], "vpshufd ymm1{k5}, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x70, 0x4a, 0x01, 0xcc], "vpshufd ymm1{k5}, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x70, 0xca, 0xcc], "vpshufd ymm1{k5}, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x72, 0x0a, 0xcc], "vprold ymm0{k5}, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x72, 0x4a, 0x01, 0xcc], "vprold ymm0{k5}, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x72, 0xca, 0xcc], "vprold ymm0{k5}, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x76, 0x0a], "vpcmpeqd k1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x76, 0x4a, 0x01], "vpcmpeqd k1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x76, 0xca], "vpcmpeqd k1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x78, 0x0a], "vcvttps2uqq ymm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x78, 0x4a, 0x01], "vcvttps2uqq ymm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x78, 0xca], "vcvttps2uqq ymm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x79, 0x0a], "vcvtps2uqq ymm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x79, 0x4a, 0x01], "vcvtps2uqq ymm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x79, 0xca], "vcvtps2uqq ymm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x7a, 0x0a], "vcvttps2qq ymm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x7a, 0x4a, 0x01], "vcvttps2qq ymm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x7a, 0xca], "vcvttps2qq ymm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x7b, 0x0a], "vcvtps2qq ymm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x7b, 0x4a, 0x01], "vcvtps2qq ymm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x7b, 0xca], "vcvtps2qq ymm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x7f, 0x0a], "vmovdqa32 ymmword [bp + si]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x7f, 0x4a, 0x01], "vmovdqa32 ymmword [bp + si + 0x20]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0x7f, 0xca], "vmovdqa32 ymm2{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xd2, 0x0a], "vpsrld ymm1{k5}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xd2, 0x4a, 0x01], "vpsrld ymm1{k5}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xd2, 0xca], "vpsrld ymm1{k5}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xdb, 0x0a], "vpandd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xdb, 0x4a, 0x01], "vpandd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xdb, 0xca], "vpandd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xdf, 0x0a], "vpandnd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xdf, 0x4a, 0x01], "vpandnd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xdf, 0xca], "vpandnd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xe2, 0x0a], "vpsrad ymm1{k5}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xe2, 0x4a, 0x01], "vpsrad ymm1{k5}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xe2, 0xca], "vpsrad ymm1{k5}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xeb, 0x0a], "vpord ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xeb, 0x4a, 0x01], "vpord ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xeb, 0xca], "vpord ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xef, 0x0a], "vpxord ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xef, 0x4a, 0x01], "vpxord ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xef, 0xca], "vpxord ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xf2, 0x0a], "vpslld ymm1{k5}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xf2, 0x4a, 0x01], "vpslld ymm1{k5}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xf2, 0xca], "vpslld ymm1{k5}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xfa, 0x0a], "vpsubd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xfa, 0x4a, 0x01], "vpsubd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xfa, 0xca], "vpsubd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xfe, 0x0a], "vpaddd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xfe, 0x4a, 0x01], "vpaddd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x2d, 0xfe, 0xca], "vpaddd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x5b, 0x0a], "vcvtps2dq ymm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x5b, 0x4a, 0x01], "vcvtps2dq ymm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x5b, 0xca], "vcvtps2dq zmm1{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x62, 0x0a], "vpunpckldq ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x62, 0x4a, 0x01], "vpunpckldq ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x66, 0x0a], "vpcmpgtd k1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x66, 0x4a, 0x01], "vpcmpgtd k1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x6a, 0x0a], "vpunpckhdq ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x6a, 0x4a, 0x01], "vpunpckhdq ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x6b, 0x0a], "vpackssdw ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x6b, 0x4a, 0x01], "vpackssdw ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x70, 0x0a, 0xcc], "vpshufd ymm1, dword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x70, 0x4a, 0x01, 0xcc], "vpshufd ymm1, dword [bp + si + 0x4]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x72, 0x0a, 0xcc], "vprold ymm0, dword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x72, 0x4a, 0x01, 0xcc], "vprold ymm0, dword [bp + si + 0x4]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x76, 0x0a], "vpcmpeqd k1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x76, 0x4a, 0x01], "vpcmpeqd k1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x78, 0x0a], "vcvttps2uqq ymm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x78, 0x4a, 0x01], "vcvttps2uqq ymm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x79, 0x0a], "vcvtps2uqq ymm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x79, 0x4a, 0x01], "vcvtps2uqq ymm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x79, 0xca], "vcvtps2uqq zmm1{rd-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x7a, 0x0a], "vcvttps2qq ymm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x7a, 0x4a, 0x01], "vcvttps2qq ymm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x7b, 0x0a], "vcvtps2qq ymm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x7b, 0x4a, 0x01], "vcvtps2qq ymm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0x7b, 0xca], "vcvtps2qq zmm1{rd-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0xdb, 0x0a], "vpandd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0xdb, 0x4a, 0x01], "vpandd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0xdf, 0x0a], "vpandnd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0xdf, 0x4a, 0x01], "vpandnd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0xeb, 0x0a], "vpord ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0xeb, 0x4a, 0x01], "vpord ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0xef, 0x0a], "vpxord ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0xef, 0x4a, 0x01], "vpxord ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0xfa, 0x0a], "vpsubd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0xfa, 0x4a, 0x01], "vpsubd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0xfe, 0x0a], "vpaddd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x38, 0xfe, 0x4a, 0x01], "vpaddd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x5b, 0x0a], "vcvtps2dq ymm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x5b, 0x4a, 0x01], "vcvtps2dq ymm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x62, 0x0a], "vpunpckldq ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x62, 0x4a, 0x01], "vpunpckldq ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x66, 0x0a], "vpcmpgtd k1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x66, 0x4a, 0x01], "vpcmpgtd k1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x6a, 0x0a], "vpunpckhdq ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x6a, 0x4a, 0x01], "vpunpckhdq ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x6b, 0x0a], "vpackssdw ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x6b, 0x4a, 0x01], "vpackssdw ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x70, 0x0a, 0xcc], "vpshufd ymm1{k5}, dword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x70, 0x4a, 0x01, 0xcc], "vpshufd ymm1{k5}, dword [bp + si + 0x4]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x72, 0x0a, 0xcc], "vprold ymm0{k5}, dword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x72, 0x4a, 0x01, 0xcc], "vprold ymm0{k5}, dword [bp + si + 0x4]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x76, 0x0a], "vpcmpeqd k1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x76, 0x4a, 0x01], "vpcmpeqd k1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x78, 0x0a], "vcvttps2uqq ymm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x78, 0x4a, 0x01], "vcvttps2uqq ymm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x79, 0x0a], "vcvtps2uqq ymm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x79, 0x4a, 0x01], "vcvtps2uqq ymm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{rd-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x7a, 0x0a], "vcvttps2qq ymm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x7a, 0x4a, 0x01], "vcvttps2qq ymm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x7b, 0x0a], "vcvtps2qq ymm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x7b, 0x4a, 0x01], "vcvtps2qq ymm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{rd-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0xdb, 0x0a], "vpandd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0xdb, 0x4a, 0x01], "vpandd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0xdf, 0x0a], "vpandnd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0xdf, 0x4a, 0x01], "vpandnd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0xeb, 0x0a], "vpord ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0xeb, 0x4a, 0x01], "vpord ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0xef, 0x0a], "vpxord ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0xef, 0x4a, 0x01], "vpxord ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0xfa, 0x0a], "vpsubd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0xfa, 0x4a, 0x01], "vpsubd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0xfe, 0x0a], "vpaddd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x3d, 0xfe, 0x4a, 0x01], "vpaddd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x5b, 0x0a], "vcvtps2dq zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x5b, 0x4a, 0x01], "vcvtps2dq zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x5b, 0xca], "vcvtps2dq zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x62, 0x0a], "vpunpckldq zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x62, 0x4a, 0x01], "vpunpckldq zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x62, 0xca], "vpunpckldq zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x66, 0x0a], "vpcmpgtd k1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x66, 0x4a, 0x01], "vpcmpgtd k1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x66, 0xca], "vpcmpgtd k1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x6a, 0x0a], "vpunpckhdq zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x6a, 0x4a, 0x01], "vpunpckhdq zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x6a, 0xca], "vpunpckhdq zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x6b, 0x0a], "vpackssdw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x6b, 0x4a, 0x01], "vpackssdw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x6b, 0xca], "vpackssdw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x6f, 0x0a], "vmovdqa32 zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x6f, 0x4a, 0x01], "vmovdqa32 zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x6f, 0xca], "vmovdqa32 zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x70, 0x0a, 0xcc], "vpshufd zmm1, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x70, 0x4a, 0x01, 0xcc], "vpshufd zmm1, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x70, 0xca, 0xcc], "vpshufd zmm1, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x72, 0x0a, 0xcc], "vprold zmm0, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x72, 0x4a, 0x01, 0xcc], "vprold zmm0, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x72, 0xca, 0xcc], "vprold zmm0, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x76, 0x0a], "vpcmpeqd k1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x76, 0x4a, 0x01], "vpcmpeqd k1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x76, 0xca], "vpcmpeqd k1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x78, 0x0a], "vcvttps2uqq zmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x78, 0x4a, 0x01], "vcvttps2uqq zmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x78, 0xca], "vcvttps2uqq zmm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x79, 0x0a], "vcvtps2uqq zmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x79, 0x4a, 0x01], "vcvtps2uqq zmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x79, 0xca], "vcvtps2uqq zmm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x7a, 0x0a], "vcvttps2qq zmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x7a, 0x4a, 0x01], "vcvttps2qq zmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x7a, 0xca], "vcvttps2qq zmm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x7b, 0x0a], "vcvtps2qq zmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x7b, 0x4a, 0x01], "vcvtps2qq zmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x7b, 0xca], "vcvtps2qq zmm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x7f, 0x0a], "vmovdqa32 zmmword [bp + si], zmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x7f, 0x4a, 0x01], "vmovdqa32 zmmword [bp + si + 0x40], zmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0x7f, 0xca], "vmovdqa32 zmm2, zmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xd2, 0x0a], "vpsrld zmm1, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xd2, 0x4a, 0x01], "vpsrld zmm1, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xd2, 0xca], "vpsrld zmm1, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xdb, 0x0a], "vpandd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xdb, 0x4a, 0x01], "vpandd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xdb, 0xca], "vpandd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xdf, 0x0a], "vpandnd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xdf, 0x4a, 0x01], "vpandnd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xdf, 0xca], "vpandnd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xe2, 0x0a], "vpsrad zmm1, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xe2, 0x4a, 0x01], "vpsrad zmm1, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xe2, 0xca], "vpsrad zmm1, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xe7, 0x0a], "vmovntdq zmmword [bp + si], zmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xe7, 0x4a, 0x01], "vmovntdq zmmword [bp + si + 0x40], zmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xeb, 0x0a], "vpord zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xeb, 0x4a, 0x01], "vpord zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xeb, 0xca], "vpord zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xef, 0x0a], "vpxord zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xef, 0x4a, 0x01], "vpxord zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xef, 0xca], "vpxord zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xf2, 0x0a], "vpslld zmm1, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xf2, 0x4a, 0x01], "vpslld zmm1, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xf2, 0xca], "vpslld zmm1, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xfa, 0x0a], "vpsubd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xfa, 0x4a, 0x01], "vpsubd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xfa, 0xca], "vpsubd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xfe, 0x0a], "vpaddd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xfe, 0x4a, 0x01], "vpaddd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x48, 0xfe, 0xca], "vpaddd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x5b, 0x0a], "vcvtps2dq zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x5b, 0x4a, 0x01], "vcvtps2dq zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x62, 0x0a], "vpunpckldq zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x62, 0x4a, 0x01], "vpunpckldq zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x62, 0xca], "vpunpckldq zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x66, 0x0a], "vpcmpgtd k1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x66, 0x4a, 0x01], "vpcmpgtd k1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x66, 0xca], "vpcmpgtd k1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x6a, 0x0a], "vpunpckhdq zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x6a, 0x4a, 0x01], "vpunpckhdq zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x6a, 0xca], "vpunpckhdq zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x6b, 0x0a], "vpackssdw zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x6b, 0x4a, 0x01], "vpackssdw zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x6b, 0xca], "vpackssdw zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x6f, 0x0a], "vmovdqa32 zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x6f, 0x4a, 0x01], "vmovdqa32 zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x6f, 0xca], "vmovdqa32 zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x70, 0x0a, 0xcc], "vpshufd zmm1{k5}, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x70, 0x4a, 0x01, 0xcc], "vpshufd zmm1{k5}, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x70, 0xca, 0xcc], "vpshufd zmm1{k5}, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x72, 0x0a, 0xcc], "vprold zmm0{k5}, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x72, 0x4a, 0x01, 0xcc], "vprold zmm0{k5}, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x72, 0xca, 0xcc], "vprold zmm0{k5}, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x76, 0x0a], "vpcmpeqd k1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x76, 0x4a, 0x01], "vpcmpeqd k1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x76, 0xca], "vpcmpeqd k1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x78, 0x0a], "vcvttps2uqq zmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x78, 0x4a, 0x01], "vcvttps2uqq zmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x78, 0xca], "vcvttps2uqq zmm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x79, 0x0a], "vcvtps2uqq zmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x79, 0x4a, 0x01], "vcvtps2uqq zmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x7a, 0x0a], "vcvttps2qq zmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x7a, 0x4a, 0x01], "vcvttps2qq zmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x7a, 0xca], "vcvttps2qq zmm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x7b, 0x0a], "vcvtps2qq zmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x7b, 0x4a, 0x01], "vcvtps2qq zmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x7f, 0x0a], "vmovdqa32 zmmword [bp + si]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x7f, 0x4a, 0x01], "vmovdqa32 zmmword [bp + si + 0x40]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0x7f, 0xca], "vmovdqa32 zmm2{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xd2, 0x0a], "vpsrld zmm1{k5}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xd2, 0x4a, 0x01], "vpsrld zmm1{k5}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xd2, 0xca], "vpsrld zmm1{k5}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xdb, 0x0a], "vpandd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xdb, 0x4a, 0x01], "vpandd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xdb, 0xca], "vpandd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xdf, 0x0a], "vpandnd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xdf, 0x4a, 0x01], "vpandnd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xdf, 0xca], "vpandnd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xe2, 0x0a], "vpsrad zmm1{k5}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xe2, 0x4a, 0x01], "vpsrad zmm1{k5}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xe2, 0xca], "vpsrad zmm1{k5}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xeb, 0x0a], "vpord zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xeb, 0x4a, 0x01], "vpord zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xeb, 0xca], "vpord zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xef, 0x0a], "vpxord zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xef, 0x4a, 0x01], "vpxord zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xef, 0xca], "vpxord zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xf2, 0x0a], "vpslld zmm1{k5}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xf2, 0x4a, 0x01], "vpslld zmm1{k5}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xf2, 0xca], "vpslld zmm1{k5}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xfa, 0x0a], "vpsubd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xfa, 0x4a, 0x01], "vpsubd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xfa, 0xca], "vpsubd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xfe, 0x0a], "vpaddd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xfe, 0x4a, 0x01], "vpaddd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0x4d, 0xfe, 0xca], "vpaddd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x5b, 0x0a], "vcvtps2dq zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x5b, 0x4a, 0x01], "vcvtps2dq zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x5b, 0xca], "vcvtps2dq zmm1{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x62, 0x0a], "vpunpckldq zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x62, 0x4a, 0x01], "vpunpckldq zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x66, 0x0a], "vpcmpgtd k1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x66, 0x4a, 0x01], "vpcmpgtd k1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x6a, 0x0a], "vpunpckhdq zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x6a, 0x4a, 0x01], "vpunpckhdq zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x6b, 0x0a], "vpackssdw zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x6b, 0x4a, 0x01], "vpackssdw zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x70, 0x0a, 0xcc], "vpshufd zmm1, dword [bp + si]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x70, 0x4a, 0x01, 0xcc], "vpshufd zmm1, dword [bp + si + 0x4]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x72, 0x0a, 0xcc], "vprold zmm0, dword [bp + si]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x72, 0x4a, 0x01, 0xcc], "vprold zmm0, dword [bp + si + 0x4]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x76, 0x0a], "vpcmpeqd k1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x76, 0x4a, 0x01], "vpcmpeqd k1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x78, 0x0a], "vcvttps2uqq zmm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x78, 0x4a, 0x01], "vcvttps2uqq zmm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x79, 0x0a], "vcvtps2uqq zmm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x79, 0x4a, 0x01], "vcvtps2uqq zmm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x79, 0xca], "vcvtps2uqq zmm1{ru-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x7a, 0x0a], "vcvttps2qq zmm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x7a, 0x4a, 0x01], "vcvttps2qq zmm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x7b, 0x0a], "vcvtps2qq zmm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x7b, 0x4a, 0x01], "vcvtps2qq zmm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0x7b, 0xca], "vcvtps2qq zmm1{ru-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0xdb, 0x0a], "vpandd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0xdb, 0x4a, 0x01], "vpandd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0xdf, 0x0a], "vpandnd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0xdf, 0x4a, 0x01], "vpandnd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0xeb, 0x0a], "vpord zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0xeb, 0x4a, 0x01], "vpord zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0xef, 0x0a], "vpxord zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0xef, 0x4a, 0x01], "vpxord zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0xfa, 0x0a], "vpsubd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0xfa, 0x4a, 0x01], "vpsubd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0xfe, 0x0a], "vpaddd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x58, 0xfe, 0x4a, 0x01], "vpaddd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x5b, 0x0a], "vcvtps2dq zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x5b, 0x4a, 0x01], "vcvtps2dq zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x62, 0x0a], "vpunpckldq zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x62, 0x4a, 0x01], "vpunpckldq zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x66, 0x0a], "vpcmpgtd k1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x66, 0x4a, 0x01], "vpcmpgtd k1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x6a, 0x0a], "vpunpckhdq zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x6a, 0x4a, 0x01], "vpunpckhdq zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x6b, 0x0a], "vpackssdw zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x6b, 0x4a, 0x01], "vpackssdw zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x70, 0x0a, 0xcc], "vpshufd zmm1{k5}, dword [bp + si]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x70, 0x4a, 0x01, 0xcc], "vpshufd zmm1{k5}, dword [bp + si + 0x4]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x72, 0x0a, 0xcc], "vprold zmm0{k5}, dword [bp + si]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x72, 0x4a, 0x01, 0xcc], "vprold zmm0{k5}, dword [bp + si + 0x4]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x76, 0x0a], "vpcmpeqd k1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x76, 0x4a, 0x01], "vpcmpeqd k1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x78, 0x0a], "vcvttps2uqq zmm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x78, 0x4a, 0x01], "vcvttps2uqq zmm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x79, 0x0a], "vcvtps2uqq zmm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x79, 0x4a, 0x01], "vcvtps2uqq zmm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{ru-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x7a, 0x0a], "vcvttps2qq zmm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x7a, 0x4a, 0x01], "vcvttps2qq zmm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x7b, 0x0a], "vcvtps2qq zmm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x7b, 0x4a, 0x01], "vcvtps2qq zmm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{ru-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0xdb, 0x0a], "vpandd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0xdb, 0x4a, 0x01], "vpandd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0xdf, 0x0a], "vpandnd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0xdf, 0x4a, 0x01], "vpandnd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0xeb, 0x0a], "vpord zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0xeb, 0x4a, 0x01], "vpord zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0xef, 0x0a], "vpxord zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0xef, 0x4a, 0x01], "vpxord zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0xfa, 0x0a], "vpsubd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0xfa, 0x4a, 0x01], "vpsubd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0xfe, 0x0a], "vpaddd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x5d, 0xfe, 0x4a, 0x01], "vpaddd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0x78, 0x5b, 0xca], "vcvtps2dq zmm1{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x78, 0x78, 0xca], "vcvttps2uqq zmm1{sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x78, 0x79, 0xca], "vcvtps2uqq zmm1{rz-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x78, 0x7a, 0xca], "vcvttps2qq zmm1{sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x78, 0x7b, 0xca], "vcvtps2qq zmm1{rz-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x7d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x7d, 0x78, 0xca], "vcvttps2uqq zmm1{k5}{sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x7d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{rz-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x7d, 0x7a, 0xca], "vcvttps2qq zmm1{k5}{sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x7d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{rz-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x5b, 0x0a], "vcvtps2dq xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x5b, 0x4a, 0x01], "vcvtps2dq xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x5b, 0xca], "vcvtps2dq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x62, 0x0a], "vpunpckldq xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x62, 0x4a, 0x01], "vpunpckldq xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x62, 0xca], "vpunpckldq xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x6a, 0x0a], "vpunpckhdq xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x6a, 0x4a, 0x01], "vpunpckhdq xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x6a, 0xca], "vpunpckhdq xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x6b, 0x0a], "vpackssdw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x6b, 0x4a, 0x01], "vpackssdw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x6b, 0xca], "vpackssdw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x6f, 0x0a], "vmovdqa32 xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x6f, 0x4a, 0x01], "vmovdqa32 xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x6f, 0xca], "vmovdqa32 xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x70, 0x0a, 0xcc], "vpshufd xmm1{k5}{z}, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x70, 0x4a, 0x01, 0xcc], "vpshufd xmm1{k5}{z}, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x70, 0xca, 0xcc], "vpshufd xmm1{k5}{z}, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x72, 0x0a, 0xcc], "vprold xmm0{k5}{z}, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x72, 0x4a, 0x01, 0xcc], "vprold xmm0{k5}{z}, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x72, 0xca, 0xcc], "vprold xmm0{k5}{z}, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x78, 0x0a], "vcvttps2uqq xmm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x78, 0x4a, 0x01], "vcvttps2uqq xmm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x78, 0xca], "vcvttps2uqq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x79, 0x0a], "vcvtps2uqq xmm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x79, 0x4a, 0x01], "vcvtps2uqq xmm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x79, 0xca], "vcvtps2uqq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x7a, 0x0a], "vcvttps2qq xmm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x7a, 0x4a, 0x01], "vcvttps2qq xmm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x7a, 0xca], "vcvttps2qq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x7b, 0x0a], "vcvtps2qq xmm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x7b, 0x4a, 0x01], "vcvtps2qq xmm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x7b, 0xca], "vcvtps2qq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0x7f, 0xca], "vmovdqa32 xmm2{k5}{z}, xmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xd2, 0x0a], "vpsrld xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xd2, 0x4a, 0x01], "vpsrld xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xd2, 0xca], "vpsrld xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xdb, 0x0a], "vpandd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xdb, 0x4a, 0x01], "vpandd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xdb, 0xca], "vpandd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xdf, 0x0a], "vpandnd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xdf, 0x4a, 0x01], "vpandnd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xdf, 0xca], "vpandnd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xe2, 0x0a], "vpsrad xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xe2, 0x4a, 0x01], "vpsrad xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xe2, 0xca], "vpsrad xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xeb, 0x0a], "vpord xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xeb, 0x4a, 0x01], "vpord xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xeb, 0xca], "vpord xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xef, 0x0a], "vpxord xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xef, 0x4a, 0x01], "vpxord xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xef, 0xca], "vpxord xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xf2, 0x0a], "vpslld xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xf2, 0x4a, 0x01], "vpslld xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xf2, 0xca], "vpslld xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xfa, 0x0a], "vpsubd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xfa, 0x4a, 0x01], "vpsubd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xfa, 0xca], "vpsubd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xfe, 0x0a], "vpaddd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xfe, 0x4a, 0x01], "vpaddd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0x8d, 0xfe, 0xca], "vpaddd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x5b, 0x0a], "vcvtps2dq xmm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x5b, 0x4a, 0x01], "vcvtps2dq xmm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{z}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x62, 0x0a], "vpunpckldq xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x62, 0x4a, 0x01], "vpunpckldq xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x6a, 0x0a], "vpunpckhdq xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x6a, 0x4a, 0x01], "vpunpckhdq xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x6b, 0x0a], "vpackssdw xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x6b, 0x4a, 0x01], "vpackssdw xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x70, 0x0a, 0xcc], "vpshufd xmm1{k5}{z}, dword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x70, 0x4a, 0x01, 0xcc], "vpshufd xmm1{k5}{z}, dword [bp + si + 0x4]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x72, 0x0a, 0xcc], "vprold xmm0{k5}{z}, dword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x72, 0x4a, 0x01, 0xcc], "vprold xmm0{k5}{z}, dword [bp + si + 0x4]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x78, 0x0a], "vcvttps2uqq xmm1{k5}{z}, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x78, 0x4a, 0x01], "vcvttps2uqq xmm1{k5}{z}, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x79, 0x0a], "vcvtps2uqq xmm1{k5}{z}, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x79, 0x4a, 0x01], "vcvtps2uqq xmm1{k5}{z}, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{z}{rne-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x7a, 0x0a], "vcvttps2qq xmm1{k5}{z}, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x7a, 0x4a, 0x01], "vcvttps2qq xmm1{k5}{z}, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x7b, 0x0a], "vcvtps2qq xmm1{k5}{z}, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x7b, 0x4a, 0x01], "vcvtps2qq xmm1{k5}{z}, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{z}{rne-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0xdb, 0x0a], "vpandd xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0xdb, 0x4a, 0x01], "vpandd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0xdf, 0x0a], "vpandnd xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0xdf, 0x4a, 0x01], "vpandnd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0xeb, 0x0a], "vpord xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0xeb, 0x4a, 0x01], "vpord xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0xef, 0x0a], "vpxord xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0xef, 0x4a, 0x01], "vpxord xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0xfa, 0x0a], "vpsubd xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0xfa, 0x4a, 0x01], "vpsubd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0xfe, 0x0a], "vpaddd xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0x9d, 0xfe, 0x4a, 0x01], "vpaddd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x5b, 0x0a], "vcvtps2dq ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x5b, 0x4a, 0x01], "vcvtps2dq ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x5b, 0xca], "vcvtps2dq ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x62, 0x0a], "vpunpckldq ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x62, 0x4a, 0x01], "vpunpckldq ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x62, 0xca], "vpunpckldq ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x6a, 0x0a], "vpunpckhdq ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x6a, 0x4a, 0x01], "vpunpckhdq ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x6a, 0xca], "vpunpckhdq ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x6b, 0x0a], "vpackssdw ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x6b, 0x4a, 0x01], "vpackssdw ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x6b, 0xca], "vpackssdw ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x6f, 0x0a], "vmovdqa32 ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x6f, 0x4a, 0x01], "vmovdqa32 ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x6f, 0xca], "vmovdqa32 ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x70, 0x0a, 0xcc], "vpshufd ymm1{k5}{z}, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x70, 0x4a, 0x01, 0xcc], "vpshufd ymm1{k5}{z}, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x70, 0xca, 0xcc], "vpshufd ymm1{k5}{z}, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x72, 0x0a, 0xcc], "vprold ymm0{k5}{z}, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x72, 0x4a, 0x01, 0xcc], "vprold ymm0{k5}{z}, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x72, 0xca, 0xcc], "vprold ymm0{k5}{z}, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x78, 0x0a], "vcvttps2uqq ymm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x78, 0x4a, 0x01], "vcvttps2uqq ymm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x78, 0xca], "vcvttps2uqq ymm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x79, 0x0a], "vcvtps2uqq ymm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x79, 0x4a, 0x01], "vcvtps2uqq ymm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x79, 0xca], "vcvtps2uqq ymm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x7a, 0x0a], "vcvttps2qq ymm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x7a, 0x4a, 0x01], "vcvttps2qq ymm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x7a, 0xca], "vcvttps2qq ymm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x7b, 0x0a], "vcvtps2qq ymm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x7b, 0x4a, 0x01], "vcvtps2qq ymm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x7b, 0xca], "vcvtps2qq ymm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0x7f, 0xca], "vmovdqa32 ymm2{k5}{z}, ymm1");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xd2, 0x0a], "vpsrld ymm1{k5}{z}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xd2, 0x4a, 0x01], "vpsrld ymm1{k5}{z}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xd2, 0xca], "vpsrld ymm1{k5}{z}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xdb, 0x0a], "vpandd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xdb, 0x4a, 0x01], "vpandd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xdb, 0xca], "vpandd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xdf, 0x0a], "vpandnd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xdf, 0x4a, 0x01], "vpandnd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xdf, 0xca], "vpandnd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xe2, 0x0a], "vpsrad ymm1{k5}{z}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xe2, 0x4a, 0x01], "vpsrad ymm1{k5}{z}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xe2, 0xca], "vpsrad ymm1{k5}{z}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xeb, 0x0a], "vpord ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xeb, 0x4a, 0x01], "vpord ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xeb, 0xca], "vpord ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xef, 0x0a], "vpxord ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xef, 0x4a, 0x01], "vpxord ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xef, 0xca], "vpxord ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xf2, 0x0a], "vpslld ymm1{k5}{z}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xf2, 0x4a, 0x01], "vpslld ymm1{k5}{z}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xf2, 0xca], "vpslld ymm1{k5}{z}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xfa, 0x0a], "vpsubd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xfa, 0x4a, 0x01], "vpsubd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xfa, 0xca], "vpsubd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xfe, 0x0a], "vpaddd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xfe, 0x4a, 0x01], "vpaddd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0xad, 0xfe, 0xca], "vpaddd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x5b, 0x0a], "vcvtps2dq ymm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x5b, 0x4a, 0x01], "vcvtps2dq ymm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{z}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x62, 0x0a], "vpunpckldq ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x62, 0x4a, 0x01], "vpunpckldq ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x6a, 0x0a], "vpunpckhdq ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x6a, 0x4a, 0x01], "vpunpckhdq ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x6b, 0x0a], "vpackssdw ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x6b, 0x4a, 0x01], "vpackssdw ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x70, 0x0a, 0xcc], "vpshufd ymm1{k5}{z}, dword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x70, 0x4a, 0x01, 0xcc], "vpshufd ymm1{k5}{z}, dword [bp + si + 0x4]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x72, 0x0a, 0xcc], "vprold ymm0{k5}{z}, dword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x72, 0x4a, 0x01, 0xcc], "vprold ymm0{k5}{z}, dword [bp + si + 0x4]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x78, 0x0a], "vcvttps2uqq ymm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x78, 0x4a, 0x01], "vcvttps2uqq ymm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x79, 0x0a], "vcvtps2uqq ymm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x79, 0x4a, 0x01], "vcvtps2uqq ymm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{z}{rd-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x7a, 0x0a], "vcvttps2qq ymm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x7a, 0x4a, 0x01], "vcvttps2qq ymm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x7b, 0x0a], "vcvtps2qq ymm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x7b, 0x4a, 0x01], "vcvtps2qq ymm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{z}{rd-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0xdb, 0x0a], "vpandd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0xdb, 0x4a, 0x01], "vpandd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0xdf, 0x0a], "vpandnd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0xdf, 0x4a, 0x01], "vpandnd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0xeb, 0x0a], "vpord ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0xeb, 0x4a, 0x01], "vpord ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0xef, 0x0a], "vpxord ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0xef, 0x4a, 0x01], "vpxord ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0xfa, 0x0a], "vpsubd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0xfa, 0x4a, 0x01], "vpsubd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0xfe, 0x0a], "vpaddd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xbd, 0xfe, 0x4a, 0x01], "vpaddd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x5b, 0x0a], "vcvtps2dq zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x5b, 0x4a, 0x01], "vcvtps2dq zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x62, 0x0a], "vpunpckldq zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x62, 0x4a, 0x01], "vpunpckldq zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x62, 0xca], "vpunpckldq zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x6a, 0x0a], "vpunpckhdq zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x6a, 0x4a, 0x01], "vpunpckhdq zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x6a, 0xca], "vpunpckhdq zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x6b, 0x0a], "vpackssdw zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x6b, 0x4a, 0x01], "vpackssdw zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x6b, 0xca], "vpackssdw zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x6f, 0x0a], "vmovdqa32 zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x6f, 0x4a, 0x01], "vmovdqa32 zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x6f, 0xca], "vmovdqa32 zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x70, 0x0a, 0xcc], "vpshufd zmm1{k5}{z}, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x70, 0x4a, 0x01, 0xcc], "vpshufd zmm1{k5}{z}, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x70, 0xca, 0xcc], "vpshufd zmm1{k5}{z}, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x72, 0x0a, 0xcc], "vprold zmm0{k5}{z}, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x72, 0x4a, 0x01, 0xcc], "vprold zmm0{k5}{z}, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x72, 0xca, 0xcc], "vprold zmm0{k5}{z}, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x78, 0x0a], "vcvttps2uqq zmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x78, 0x4a, 0x01], "vcvttps2uqq zmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x78, 0xca], "vcvttps2uqq zmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x79, 0x0a], "vcvtps2uqq zmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x79, 0x4a, 0x01], "vcvtps2uqq zmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x7a, 0x0a], "vcvttps2qq zmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x7a, 0x4a, 0x01], "vcvttps2qq zmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x7a, 0xca], "vcvttps2qq zmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x7b, 0x0a], "vcvtps2qq zmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x7b, 0x4a, 0x01], "vcvtps2qq zmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0x7f, 0xca], "vmovdqa32 zmm2{k5}{z}, zmm1");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xd2, 0x0a], "vpsrld zmm1{k5}{z}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xd2, 0x4a, 0x01], "vpsrld zmm1{k5}{z}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xd2, 0xca], "vpsrld zmm1{k5}{z}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xdb, 0x0a], "vpandd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xdb, 0x4a, 0x01], "vpandd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xdb, 0xca], "vpandd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xdf, 0x0a], "vpandnd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xdf, 0x4a, 0x01], "vpandnd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xdf, 0xca], "vpandnd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xe2, 0x0a], "vpsrad zmm1{k5}{z}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xe2, 0x4a, 0x01], "vpsrad zmm1{k5}{z}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xe2, 0xca], "vpsrad zmm1{k5}{z}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xeb, 0x0a], "vpord zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xeb, 0x4a, 0x01], "vpord zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xeb, 0xca], "vpord zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xef, 0x0a], "vpxord zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xef, 0x4a, 0x01], "vpxord zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xef, 0xca], "vpxord zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xf2, 0x0a], "vpslld zmm1{k5}{z}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xf2, 0x4a, 0x01], "vpslld zmm1{k5}{z}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xf2, 0xca], "vpslld zmm1{k5}{z}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xfa, 0x0a], "vpsubd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xfa, 0x4a, 0x01], "vpsubd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xfa, 0xca], "vpsubd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xfe, 0x0a], "vpaddd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xfe, 0x4a, 0x01], "vpaddd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7d, 0xcd, 0xfe, 0xca], "vpaddd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x5b, 0x0a], "vcvtps2dq zmm1{k5}{z}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x5b, 0x4a, 0x01], "vcvtps2dq zmm1{k5}{z}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{z}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x62, 0x0a], "vpunpckldq zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x62, 0x4a, 0x01], "vpunpckldq zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x6a, 0x0a], "vpunpckhdq zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x6a, 0x4a, 0x01], "vpunpckhdq zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x6b, 0x0a], "vpackssdw zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x6b, 0x4a, 0x01], "vpackssdw zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x70, 0x0a, 0xcc], "vpshufd zmm1{k5}{z}, dword [bp + si]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x70, 0x4a, 0x01, 0xcc], "vpshufd zmm1{k5}{z}, dword [bp + si + 0x4]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x72, 0x0a, 0xcc], "vprold zmm0{k5}{z}, dword [bp + si]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x72, 0x4a, 0x01, 0xcc], "vprold zmm0{k5}{z}, dword [bp + si + 0x4]{1to16}, 0xcc");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x78, 0x0a], "vcvttps2uqq zmm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x78, 0x4a, 0x01], "vcvttps2uqq zmm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x79, 0x0a], "vcvtps2uqq zmm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x79, 0x4a, 0x01], "vcvtps2uqq zmm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{z}{ru-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x7a, 0x0a], "vcvttps2qq zmm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x7a, 0x4a, 0x01], "vcvttps2qq zmm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x7b, 0x0a], "vcvtps2qq zmm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x7b, 0x4a, 0x01], "vcvtps2qq zmm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{z}{ru-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0xdb, 0x0a], "vpandd zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0xdb, 0x4a, 0x01], "vpandd zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0xdf, 0x0a], "vpandnd zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0xdf, 0x4a, 0x01], "vpandnd zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0xeb, 0x0a], "vpord zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0xeb, 0x4a, 0x01], "vpord zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0xef, 0x0a], "vpxord zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0xef, 0x4a, 0x01], "vpxord zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0xfa, 0x0a], "vpsubd zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0xfa, 0x4a, 0x01], "vpsubd zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0xfe, 0x0a], "vpaddd zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xdd, 0xfe, 0x4a, 0x01], "vpaddd zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7d, 0xfd, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{z}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xfd, 0x78, 0xca], "vcvttps2uqq zmm1{k5}{z}{sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xfd, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{z}{rz-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xfd, 0x7a, 0xca], "vcvttps2qq zmm1{k5}{z}{sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7d, 0xfd, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{z}{rz-sae}, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x12, 0x0a], "vmovsldup xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x12, 0x4a, 0x01], "vmovsldup xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x12, 0xca], "vmovsldup xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x16, 0x0a], "vmovshdup xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x16, 0x4a, 0x01], "vmovshdup xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x16, 0xca], "vmovshdup xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x5b, 0x0a], "vcvttps2dq xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x5b, 0x4a, 0x01], "vcvttps2dq xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x5b, 0xca], "vcvttps2dq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x6f, 0x0a], "vmovdqu32 xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x6f, 0x4a, 0x01], "vmovdqu32 xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x6f, 0xca], "vmovdqu32 xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x7a, 0x0a], "vcvtudq2pd xmm1, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x7a, 0x4a, 0x01], "vcvtudq2pd xmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x7a, 0xca], "vcvtudq2pd xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x7f, 0x0a], "vmovdqu32 xmmword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x7f, 0x4a, 0x01], "vmovdqu32 xmmword [bp + si + 0x10], xmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0x7f, 0xca], "vmovdqu32 xmm2, xmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0xe6, 0x0a], "vcvtdq2pd xmm1, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0xe6, 0x4a, 0x01], "vcvtdq2pd xmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7e, 0x08, 0xe6, 0xca], "vcvtdq2pd xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x12, 0x0a], "vmovsldup xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x12, 0x4a, 0x01], "vmovsldup xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x12, 0xca], "vmovsldup xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x16, 0x0a], "vmovshdup xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x16, 0x4a, 0x01], "vmovshdup xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x16, 0xca], "vmovshdup xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x5b, 0x0a], "vcvttps2dq xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x5b, 0x4a, 0x01], "vcvttps2dq xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x5b, 0xca], "vcvttps2dq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x6f, 0x0a], "vmovdqu32 xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x6f, 0x4a, 0x01], "vmovdqu32 xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x6f, 0xca], "vmovdqu32 xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x7a, 0x0a], "vcvtudq2pd xmm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x7a, 0x4a, 0x01], "vcvtudq2pd xmm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x7a, 0xca], "vcvtudq2pd xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x7f, 0x0a], "vmovdqu32 xmmword [bp + si]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x7f, 0x4a, 0x01], "vmovdqu32 xmmword [bp + si + 0x10]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0x7f, 0xca], "vmovdqu32 xmm2{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0xe6, 0x0a], "vcvtdq2pd xmm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0xe6, 0x4a, 0x01], "vcvtdq2pd xmm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7e, 0x0d, 0xe6, 0xca], "vcvtdq2pd xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x18, 0x51, 0xca], "vsqrtss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x18, 0x58, 0xca], "vaddss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x18, 0x59, 0xca], "vmulss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x18, 0x5b, 0x0a], "vcvttps2dq xmm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0x18, 0x5b, 0x4a, 0x01], "vcvttps2dq xmm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0x18, 0x5c, 0xca], "vsubss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x18, 0x5e, 0xca], "vdivss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x18, 0x7a, 0x0a], "vcvtudq2pd xmm1, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7e, 0x18, 0x7a, 0x4a, 0x01], "vcvtudq2pd xmm1, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7e, 0x18, 0xe6, 0x0a], "vcvtdq2pd xmm1, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7e, 0x18, 0xe6, 0x4a, 0x01], "vcvtdq2pd xmm1, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7e, 0x1d, 0x51, 0xca], "vsqrtss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x1d, 0x58, 0xca], "vaddss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x1d, 0x59, 0xca], "vmulss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x1d, 0x5b, 0x0a], "vcvttps2dq xmm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0x1d, 0x5b, 0x4a, 0x01], "vcvttps2dq xmm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0x1d, 0x5c, 0xca], "vsubss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x1d, 0x5e, 0xca], "vdivss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x1d, 0x7a, 0x0a], "vcvtudq2pd xmm1{k5}, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7e, 0x1d, 0x7a, 0x4a, 0x01], "vcvtudq2pd xmm1{k5}, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7e, 0x1d, 0xe6, 0x0a], "vcvtdq2pd xmm1{k5}, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7e, 0x1d, 0xe6, 0x4a, 0x01], "vcvtdq2pd xmm1{k5}, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x10, 0x0a], "vmovss xmm1, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x10, 0x4a, 0x01], "vmovss xmm1, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x10, 0xca], "vmovss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x11, 0x0a], "vmovss dword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x11, 0x4a, 0x01], "vmovss dword [bp + si + 0x4], xmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x11, 0xca], "vmovss xmm2, xmm0, xmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x12, 0x0a], "vmovsldup ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x12, 0x4a, 0x01], "vmovsldup ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x12, 0xca], "vmovsldup ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x16, 0x0a], "vmovshdup ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x16, 0x4a, 0x01], "vmovshdup ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x16, 0xca], "vmovshdup ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x51, 0x0a], "vsqrtss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x51, 0x4a, 0x01], "vsqrtss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x51, 0xca], "vsqrtss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x58, 0x0a], "vaddss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x58, 0x4a, 0x01], "vaddss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x58, 0xca], "vaddss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x59, 0x0a], "vmulss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x59, 0x4a, 0x01], "vmulss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x59, 0xca], "vmulss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5a, 0x0a], "vcvtss2sd xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5a, 0x4a, 0x01], "vcvtss2sd xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5a, 0xca], "vcvtss2sd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5b, 0x0a], "vcvttps2dq ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5b, 0x4a, 0x01], "vcvttps2dq ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5b, 0xca], "vcvttps2dq ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5c, 0x0a], "vsubss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5c, 0x4a, 0x01], "vsubss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5c, 0xca], "vsubss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5d, 0x0a], "vminss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5d, 0x4a, 0x01], "vminss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5d, 0xca], "vminss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5e, 0x0a], "vdivss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5e, 0x4a, 0x01], "vdivss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5e, 0xca], "vdivss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5f, 0x0a], "vmaxss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5f, 0x4a, 0x01], "vmaxss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x5f, 0xca], "vmaxss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x6f, 0x0a], "vmovdqu32 ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x6f, 0x4a, 0x01], "vmovdqu32 ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x6f, 0xca], "vmovdqu32 ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x7a, 0x0a], "vcvtudq2pd ymm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x7a, 0x4a, 0x01], "vcvtudq2pd ymm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x7a, 0xca], "vcvtudq2pd ymm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x7f, 0x0a], "vmovdqu32 ymmword [bp + si], ymm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x7f, 0x4a, 0x01], "vmovdqu32 ymmword [bp + si + 0x20], ymm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0x7f, 0xca], "vmovdqu32 ymm2, ymm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0xc2, 0x0a, 0xcc], "vcmpss k1, xmm0, dword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0xc2, 0x4a, 0x01, 0xcc], "vcmpss k1, xmm0, dword [bp + si + 0x4], 0xcc");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0xc2, 0xca, 0xcc], "vcmpss k1, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0xe6, 0x0a], "vcvtdq2pd ymm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0xe6, 0x4a, 0x01], "vcvtdq2pd ymm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0x28, 0xe6, 0xca], "vcvtdq2pd ymm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x10, 0x0a], "vmovss xmm1{k5}, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x10, 0x4a, 0x01], "vmovss xmm1{k5}, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x10, 0xca], "vmovss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x11, 0x0a], "vmovss dword [bp + si]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x11, 0x4a, 0x01], "vmovss dword [bp + si + 0x4]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x11, 0xca], "vmovss xmm2{k5}, xmm0, xmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x12, 0x0a], "vmovsldup ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x12, 0x4a, 0x01], "vmovsldup ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x12, 0xca], "vmovsldup ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x16, 0x0a], "vmovshdup ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x16, 0x4a, 0x01], "vmovshdup ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x16, 0xca], "vmovshdup ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x51, 0x0a], "vsqrtss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x51, 0x4a, 0x01], "vsqrtss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x51, 0xca], "vsqrtss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x58, 0x0a], "vaddss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x58, 0x4a, 0x01], "vaddss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x58, 0xca], "vaddss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x59, 0x0a], "vmulss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x59, 0x4a, 0x01], "vmulss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x59, 0xca], "vmulss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5a, 0x0a], "vcvtss2sd xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5a, 0x4a, 0x01], "vcvtss2sd xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5a, 0xca], "vcvtss2sd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5b, 0x0a], "vcvttps2dq ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5b, 0x4a, 0x01], "vcvttps2dq ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5b, 0xca], "vcvttps2dq ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5c, 0x0a], "vsubss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5c, 0x4a, 0x01], "vsubss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5c, 0xca], "vsubss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5d, 0x0a], "vminss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5d, 0x4a, 0x01], "vminss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5d, 0xca], "vminss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5e, 0x0a], "vdivss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5e, 0x4a, 0x01], "vdivss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5e, 0xca], "vdivss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5f, 0x0a], "vmaxss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5f, 0x4a, 0x01], "vmaxss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x5f, 0xca], "vmaxss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x6f, 0x0a], "vmovdqu32 ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x6f, 0x4a, 0x01], "vmovdqu32 ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x6f, 0xca], "vmovdqu32 ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x7a, 0x0a], "vcvtudq2pd ymm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x7a, 0x4a, 0x01], "vcvtudq2pd ymm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x7a, 0xca], "vcvtudq2pd ymm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x7f, 0x0a], "vmovdqu32 ymmword [bp + si]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x7f, 0x4a, 0x01], "vmovdqu32 ymmword [bp + si + 0x20]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0x7f, 0xca], "vmovdqu32 ymm2{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0xc2, 0x0a, 0xcc], "vcmpss k1{k5}, xmm0, dword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0xc2, 0x4a, 0x01, 0xcc], "vcmpss k1{k5}, xmm0, dword [bp + si + 0x4], 0xcc");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0xc2, 0xca, 0xcc], "vcmpss k1{k5}, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0xe6, 0x0a], "vcvtdq2pd ymm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0xe6, 0x4a, 0x01], "vcvtdq2pd ymm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0x2d, 0xe6, 0xca], "vcvtdq2pd ymm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x38, 0x51, 0xca], "vsqrtss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x38, 0x58, 0xca], "vaddss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x38, 0x59, 0xca], "vmulss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x38, 0x5b, 0x0a], "vcvttps2dq ymm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0x38, 0x5b, 0x4a, 0x01], "vcvttps2dq ymm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0x38, 0x5c, 0xca], "vsubss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x38, 0x5e, 0xca], "vdivss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x38, 0x7a, 0x0a], "vcvtudq2pd ymm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0x38, 0x7a, 0x4a, 0x01], "vcvtudq2pd ymm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0x38, 0xe6, 0x0a], "vcvtdq2pd ymm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0x38, 0xe6, 0x4a, 0x01], "vcvtdq2pd ymm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0x3d, 0x51, 0xca], "vsqrtss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x3d, 0x58, 0xca], "vaddss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x3d, 0x59, 0xca], "vmulss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x3d, 0x5b, 0x0a], "vcvttps2dq ymm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0x3d, 0x5b, 0x4a, 0x01], "vcvttps2dq ymm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0x3d, 0x5c, 0xca], "vsubss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x3d, 0x5e, 0xca], "vdivss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x3d, 0x7a, 0x0a], "vcvtudq2pd ymm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0x3d, 0x7a, 0x4a, 0x01], "vcvtudq2pd ymm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0x3d, 0xe6, 0x0a], "vcvtdq2pd ymm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0x3d, 0xe6, 0x4a, 0x01], "vcvtdq2pd ymm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x12, 0x0a], "vmovsldup zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x12, 0x4a, 0x01], "vmovsldup zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x12, 0xca], "vmovsldup zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x16, 0x0a], "vmovshdup zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x16, 0x4a, 0x01], "vmovshdup zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x16, 0xca], "vmovshdup zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x5b, 0x0a], "vcvttps2dq zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x5b, 0x4a, 0x01], "vcvttps2dq zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x5b, 0xca], "vcvttps2dq zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x6f, 0x0a], "vmovdqu32 zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x6f, 0x4a, 0x01], "vmovdqu32 zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x6f, 0xca], "vmovdqu32 zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x7a, 0x0a], "vcvtudq2pd zmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x7a, 0x4a, 0x01], "vcvtudq2pd zmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x7a, 0xca], "vcvtudq2pd zmm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x7f, 0x0a], "vmovdqu32 zmmword [bp + si], zmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x7f, 0x4a, 0x01], "vmovdqu32 zmmword [bp + si + 0x40], zmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0x7f, 0xca], "vmovdqu32 zmm2, zmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0xe6, 0x0a], "vcvtdq2pd zmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0xe6, 0x4a, 0x01], "vcvtdq2pd zmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0x48, 0xe6, 0xca], "vcvtdq2pd zmm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x12, 0x0a], "vmovsldup zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x12, 0x4a, 0x01], "vmovsldup zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x12, 0xca], "vmovsldup zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x16, 0x0a], "vmovshdup zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x16, 0x4a, 0x01], "vmovshdup zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x16, 0xca], "vmovshdup zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x5b, 0x0a], "vcvttps2dq zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x5b, 0x4a, 0x01], "vcvttps2dq zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x5b, 0xca], "vcvttps2dq zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x6f, 0x0a], "vmovdqu32 zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x6f, 0x4a, 0x01], "vmovdqu32 zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x6f, 0xca], "vmovdqu32 zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x7a, 0x0a], "vcvtudq2pd zmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x7a, 0x4a, 0x01], "vcvtudq2pd zmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x7a, 0xca], "vcvtudq2pd zmm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x7f, 0x0a], "vmovdqu32 zmmword [bp + si]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x7f, 0x4a, 0x01], "vmovdqu32 zmmword [bp + si + 0x40]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0x7f, 0xca], "vmovdqu32 zmm2{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0xe6, 0x0a], "vcvtdq2pd zmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0xe6, 0x4a, 0x01], "vcvtdq2pd zmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0x4d, 0xe6, 0xca], "vcvtdq2pd zmm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x58, 0x51, 0xca], "vsqrtss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x58, 0x58, 0xca], "vaddss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x58, 0x59, 0xca], "vmulss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x58, 0x5b, 0x0a], "vcvttps2dq zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7e, 0x58, 0x5b, 0x4a, 0x01], "vcvttps2dq zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7e, 0x58, 0x5c, 0xca], "vsubss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x58, 0x5e, 0xca], "vdivss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x58, 0x7a, 0x0a], "vcvtudq2pd zmm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0x58, 0x7a, 0x4a, 0x01], "vcvtudq2pd zmm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0x58, 0xe6, 0x0a], "vcvtdq2pd zmm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0x58, 0xe6, 0x4a, 0x01], "vcvtdq2pd zmm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0x5d, 0x51, 0xca], "vsqrtss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x5d, 0x58, 0xca], "vaddss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x5d, 0x59, 0xca], "vmulss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x5d, 0x5b, 0x0a], "vcvttps2dq zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7e, 0x5d, 0x5b, 0x4a, 0x01], "vcvttps2dq zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7e, 0x5d, 0x5c, 0xca], "vsubss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x5d, 0x5e, 0xca], "vdivss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x5d, 0x7a, 0x0a], "vcvtudq2pd zmm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0x5d, 0x7a, 0x4a, 0x01], "vcvtudq2pd zmm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0x5d, 0xe6, 0x0a], "vcvtdq2pd zmm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0x5d, 0xe6, 0x4a, 0x01], "vcvtdq2pd zmm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0x78, 0x51, 0xca], "vsqrtss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x78, 0x58, 0xca], "vaddss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x78, 0x59, 0xca], "vmulss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x78, 0x5a, 0xca], "vcvtss2sd xmm1{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x78, 0x5b, 0xca], "vcvttps2dq zmm1{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x78, 0x5c, 0xca], "vsubss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x78, 0x5d, 0xca], "vminss xmm1{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x78, 0x5e, 0xca], "vdivss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x78, 0x5f, 0xca], "vmaxss xmm1{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x78, 0xc2, 0xca, 0xcc], "vcmpss k1{sae}, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7e, 0x7d, 0x51, 0xca], "vsqrtss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x7d, 0x58, 0xca], "vaddss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x7d, 0x59, 0xca], "vmulss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x7d, 0x5a, 0xca], "vcvtss2sd xmm1{k5}{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x7d, 0x5b, 0xca], "vcvttps2dq zmm1{k5}{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x7d, 0x5c, 0xca], "vsubss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x7d, 0x5d, 0xca], "vminss xmm1{k5}{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x7d, 0x5e, 0xca], "vdivss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x7d, 0x5f, 0xca], "vmaxss xmm1{k5}{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x7d, 0xc2, 0xca, 0xcc], "vcmpss k1{k5}{sae}, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0x12, 0x0a], "vmovsldup xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0x12, 0x4a, 0x01], "vmovsldup xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0x12, 0xca], "vmovsldup xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0x16, 0x0a], "vmovshdup xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0x16, 0x4a, 0x01], "vmovshdup xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0x16, 0xca], "vmovshdup xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0x5b, 0x0a], "vcvttps2dq xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0x5b, 0x4a, 0x01], "vcvttps2dq xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0x5b, 0xca], "vcvttps2dq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0x6f, 0x0a], "vmovdqu32 xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0x6f, 0x4a, 0x01], "vmovdqu32 xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0x6f, 0xca], "vmovdqu32 xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0x7a, 0x0a], "vcvtudq2pd xmm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0x7a, 0x4a, 0x01], "vcvtudq2pd xmm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0x7a, 0xca], "vcvtudq2pd xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0x7f, 0xca], "vmovdqu32 xmm2{k5}{z}, xmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0xe6, 0x0a], "vcvtdq2pd xmm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0xe6, 0x4a, 0x01], "vcvtdq2pd xmm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0x7e, 0x8d, 0xe6, 0xca], "vcvtdq2pd xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x9d, 0x51, 0xca], "vsqrtss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x9d, 0x58, 0xca], "vaddss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x9d, 0x59, 0xca], "vmulss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x9d, 0x5b, 0x0a], "vcvttps2dq xmm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0x9d, 0x5b, 0x4a, 0x01], "vcvttps2dq xmm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0x9d, 0x5c, 0xca], "vsubss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x9d, 0x5e, 0xca], "vdivss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0x9d, 0x7a, 0x0a], "vcvtudq2pd xmm1{k5}{z}, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7e, 0x9d, 0x7a, 0x4a, 0x01], "vcvtudq2pd xmm1{k5}{z}, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7e, 0x9d, 0xe6, 0x0a], "vcvtdq2pd xmm1{k5}{z}, dword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0x7e, 0x9d, 0xe6, 0x4a, 0x01], "vcvtdq2pd xmm1{k5}{z}, dword [bp + si + 0x4]{1to2}");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x10, 0x0a], "vmovss xmm1{k5}{z}, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x10, 0x4a, 0x01], "vmovss xmm1{k5}{z}, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x10, 0xca], "vmovss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x11, 0xca], "vmovss xmm2{k5}{z}, xmm0, xmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x12, 0x0a], "vmovsldup ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x12, 0x4a, 0x01], "vmovsldup ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x12, 0xca], "vmovsldup ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x16, 0x0a], "vmovshdup ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x16, 0x4a, 0x01], "vmovshdup ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x16, 0xca], "vmovshdup ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x51, 0x0a], "vsqrtss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x51, 0x4a, 0x01], "vsqrtss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x51, 0xca], "vsqrtss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x58, 0x0a], "vaddss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x58, 0x4a, 0x01], "vaddss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x58, 0xca], "vaddss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x59, 0x0a], "vmulss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x59, 0x4a, 0x01], "vmulss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x59, 0xca], "vmulss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5a, 0x0a], "vcvtss2sd xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5a, 0x4a, 0x01], "vcvtss2sd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5a, 0xca], "vcvtss2sd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5b, 0x0a], "vcvttps2dq ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5b, 0x4a, 0x01], "vcvttps2dq ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5b, 0xca], "vcvttps2dq ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5c, 0x0a], "vsubss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5c, 0x4a, 0x01], "vsubss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5c, 0xca], "vsubss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5d, 0x0a], "vminss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5d, 0x4a, 0x01], "vminss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5d, 0xca], "vminss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5e, 0x0a], "vdivss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5e, 0x4a, 0x01], "vdivss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5e, 0xca], "vdivss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5f, 0x0a], "vmaxss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5f, 0x4a, 0x01], "vmaxss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x5f, 0xca], "vmaxss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x6f, 0x0a], "vmovdqu32 ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x6f, 0x4a, 0x01], "vmovdqu32 ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x6f, 0xca], "vmovdqu32 ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x7a, 0x0a], "vcvtudq2pd ymm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x7a, 0x4a, 0x01], "vcvtudq2pd ymm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x7a, 0xca], "vcvtudq2pd ymm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0x7f, 0xca], "vmovdqu32 ymm2{k5}{z}, ymm1");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0xe6, 0x0a], "vcvtdq2pd ymm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0xe6, 0x4a, 0x01], "vcvtdq2pd ymm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7e, 0xad, 0xe6, 0xca], "vcvtdq2pd ymm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xbd, 0x51, 0xca], "vsqrtss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xbd, 0x58, 0xca], "vaddss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xbd, 0x59, 0xca], "vmulss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xbd, 0x5b, 0x0a], "vcvttps2dq ymm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0xbd, 0x5b, 0x4a, 0x01], "vcvttps2dq ymm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0xbd, 0x5c, 0xca], "vsubss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xbd, 0x5e, 0xca], "vdivss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xbd, 0x7a, 0x0a], "vcvtudq2pd ymm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0xbd, 0x7a, 0x4a, 0x01], "vcvtudq2pd ymm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0xbd, 0xe6, 0x0a], "vcvtdq2pd ymm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0xbd, 0xe6, 0x4a, 0x01], "vcvtdq2pd ymm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0x12, 0x0a], "vmovsldup zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0x12, 0x4a, 0x01], "vmovsldup zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0x12, 0xca], "vmovsldup zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0x16, 0x0a], "vmovshdup zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0x16, 0x4a, 0x01], "vmovshdup zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0x16, 0xca], "vmovshdup zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0x5b, 0x0a], "vcvttps2dq zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0x5b, 0x4a, 0x01], "vcvttps2dq zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0x5b, 0xca], "vcvttps2dq zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0x6f, 0x0a], "vmovdqu32 zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0x6f, 0x4a, 0x01], "vmovdqu32 zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0x6f, 0xca], "vmovdqu32 zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0x7a, 0x0a], "vcvtudq2pd zmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0x7a, 0x4a, 0x01], "vcvtudq2pd zmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0x7a, 0xca], "vcvtudq2pd zmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0x7f, 0xca], "vmovdqu32 zmm2{k5}{z}, zmm1");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0xe6, 0x0a], "vcvtdq2pd zmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0xe6, 0x4a, 0x01], "vcvtdq2pd zmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7e, 0xcd, 0xe6, 0xca], "vcvtdq2pd zmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xdd, 0x51, 0xca], "vsqrtss xmm1{k5}{z}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xdd, 0x58, 0xca], "vaddss xmm1{k5}{z}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xdd, 0x59, 0xca], "vmulss xmm1{k5}{z}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xdd, 0x5b, 0x0a], "vcvttps2dq zmm1{k5}{z}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7e, 0xdd, 0x5b, 0x4a, 0x01], "vcvttps2dq zmm1{k5}{z}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7e, 0xdd, 0x5c, 0xca], "vsubss xmm1{k5}{z}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xdd, 0x5e, 0xca], "vdivss xmm1{k5}{z}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xdd, 0x7a, 0x0a], "vcvtudq2pd zmm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0xdd, 0x7a, 0x4a, 0x01], "vcvtudq2pd zmm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0xdd, 0xe6, 0x0a], "vcvtdq2pd zmm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0xdd, 0xe6, 0x4a, 0x01], "vcvtdq2pd zmm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7e, 0xfd, 0x51, 0xca], "vsqrtss xmm1{k5}{z}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xfd, 0x58, 0xca], "vaddss xmm1{k5}{z}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xfd, 0x59, 0xca], "vmulss xmm1{k5}{z}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xfd, 0x5a, 0xca], "vcvtss2sd xmm1{k5}{z}{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xfd, 0x5b, 0xca], "vcvttps2dq zmm1{k5}{z}{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xfd, 0x5c, 0xca], "vsubss xmm1{k5}{z}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xfd, 0x5d, 0xca], "vminss xmm1{k5}{z}{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xfd, 0x5e, 0xca], "vdivss xmm1{k5}{z}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7e, 0xfd, 0x5f, 0xca], "vmaxss xmm1{k5}{z}{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x08, 0x6f, 0x0a], "vmovdqu8 xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0x08, 0x6f, 0x4a, 0x01], "vmovdqu8 xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7f, 0x08, 0x6f, 0xca], "vmovdqu8 xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x08, 0x7a, 0x0a], "vcvtudq2ps xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0x08, 0x7a, 0x4a, 0x01], "vcvtudq2ps xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7f, 0x08, 0x7a, 0xca], "vcvtudq2ps xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x08, 0x7f, 0x0a], "vmovdqu8 xmmword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x08, 0x7f, 0x4a, 0x01], "vmovdqu8 xmmword [bp + si + 0x10], xmm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x08, 0x7f, 0xca], "vmovdqu8 xmm2, xmm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x0d, 0x6f, 0x0a], "vmovdqu8 xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0x0d, 0x6f, 0x4a, 0x01], "vmovdqu8 xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7f, 0x0d, 0x6f, 0xca], "vmovdqu8 xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x0d, 0x7a, 0x0a], "vcvtudq2ps xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0x0d, 0x7a, 0x4a, 0x01], "vcvtudq2ps xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7f, 0x0d, 0x7a, 0xca], "vcvtudq2ps xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x0d, 0x7f, 0x0a], "vmovdqu8 xmmword [bp + si]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x0d, 0x7f, 0x4a, 0x01], "vmovdqu8 xmmword [bp + si + 0x10]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x0d, 0x7f, 0xca], "vmovdqu8 xmm2{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x18, 0x7a, 0x0a], "vcvtudq2ps xmm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7f, 0x18, 0x7a, 0x4a, 0x01], "vcvtudq2ps xmm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7f, 0x18, 0x7a, 0xca], "vcvtudq2ps zmm1{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x1d, 0x7a, 0x0a], "vcvtudq2ps xmm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7f, 0x1d, 0x7a, 0x4a, 0x01], "vcvtudq2ps xmm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7f, 0x1d, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x28, 0x6f, 0x0a], "vmovdqu8 ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0x28, 0x6f, 0x4a, 0x01], "vmovdqu8 ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7f, 0x28, 0x6f, 0xca], "vmovdqu8 ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x28, 0x7a, 0x0a], "vcvtudq2ps ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0x28, 0x7a, 0x4a, 0x01], "vcvtudq2ps ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7f, 0x28, 0x7a, 0xca], "vcvtudq2ps ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x28, 0x7f, 0x0a], "vmovdqu8 ymmword [bp + si], ymm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x28, 0x7f, 0x4a, 0x01], "vmovdqu8 ymmword [bp + si + 0x20], ymm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x28, 0x7f, 0xca], "vmovdqu8 ymm2, ymm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x2d, 0x6f, 0x0a], "vmovdqu8 ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0x2d, 0x6f, 0x4a, 0x01], "vmovdqu8 ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7f, 0x2d, 0x6f, 0xca], "vmovdqu8 ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x2d, 0x7a, 0x0a], "vcvtudq2ps ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0x2d, 0x7a, 0x4a, 0x01], "vcvtudq2ps ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7f, 0x2d, 0x7a, 0xca], "vcvtudq2ps ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x2d, 0x7f, 0x0a], "vmovdqu8 ymmword [bp + si]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x2d, 0x7f, 0x4a, 0x01], "vmovdqu8 ymmword [bp + si + 0x20]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x2d, 0x7f, 0xca], "vmovdqu8 ymm2{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x38, 0x7a, 0x0a], "vcvtudq2ps ymm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7f, 0x38, 0x7a, 0x4a, 0x01], "vcvtudq2ps ymm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7f, 0x38, 0x7a, 0xca], "vcvtudq2ps zmm1{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x3d, 0x7a, 0x0a], "vcvtudq2ps ymm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7f, 0x3d, 0x7a, 0x4a, 0x01], "vcvtudq2ps ymm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7f, 0x3d, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x48, 0x6f, 0x0a], "vmovdqu8 zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0x48, 0x6f, 0x4a, 0x01], "vmovdqu8 zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7f, 0x48, 0x6f, 0xca], "vmovdqu8 zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x48, 0x7a, 0x0a], "vcvtudq2ps zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0x48, 0x7a, 0x4a, 0x01], "vcvtudq2ps zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7f, 0x48, 0x7a, 0xca], "vcvtudq2ps zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x48, 0x7f, 0x0a], "vmovdqu8 zmmword [bp + si], zmm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x48, 0x7f, 0x4a, 0x01], "vmovdqu8 zmmword [bp + si + 0x40], zmm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x48, 0x7f, 0xca], "vmovdqu8 zmm2, zmm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x4d, 0x6f, 0x0a], "vmovdqu8 zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0x4d, 0x6f, 0x4a, 0x01], "vmovdqu8 zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7f, 0x4d, 0x6f, 0xca], "vmovdqu8 zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x4d, 0x7a, 0x0a], "vcvtudq2ps zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0x4d, 0x7a, 0x4a, 0x01], "vcvtudq2ps zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7f, 0x4d, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x4d, 0x7f, 0x0a], "vmovdqu8 zmmword [bp + si]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x4d, 0x7f, 0x4a, 0x01], "vmovdqu8 zmmword [bp + si + 0x40]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x4d, 0x7f, 0xca], "vmovdqu8 zmm2{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x58, 0x7a, 0x0a], "vcvtudq2ps zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7f, 0x58, 0x7a, 0x4a, 0x01], "vcvtudq2ps zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7f, 0x58, 0x7a, 0xca], "vcvtudq2ps zmm1{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x5d, 0x7a, 0x0a], "vcvtudq2ps zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7f, 0x5d, 0x7a, 0x4a, 0x01], "vcvtudq2ps zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7f, 0x5d, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x78, 0x7a, 0xca], "vcvtudq2ps zmm1{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x7d, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x8d, 0x6f, 0x0a], "vmovdqu8 xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0x8d, 0x6f, 0x4a, 0x01], "vmovdqu8 xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7f, 0x8d, 0x6f, 0xca], "vmovdqu8 xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x8d, 0x7a, 0x0a], "vcvtudq2ps xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0x8d, 0x7a, 0x4a, 0x01], "vcvtudq2ps xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0x7f, 0x8d, 0x7a, 0xca], "vcvtudq2ps xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0x8d, 0x7f, 0xca], "vmovdqu8 xmm2{k5}{z}, xmm1");
    test_display(&[0x62, 0xf1, 0x7f, 0x9d, 0x7a, 0x0a], "vcvtudq2ps xmm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0x7f, 0x9d, 0x7a, 0x4a, 0x01], "vcvtudq2ps xmm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf1, 0x7f, 0x9d, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{z}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0xad, 0x6f, 0x0a], "vmovdqu8 ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0xad, 0x6f, 0x4a, 0x01], "vmovdqu8 ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7f, 0xad, 0x6f, 0xca], "vmovdqu8 ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7f, 0xad, 0x7a, 0x0a], "vcvtudq2ps ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0xad, 0x7a, 0x4a, 0x01], "vcvtudq2ps ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0x7f, 0xad, 0x7a, 0xca], "vcvtudq2ps ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0x7f, 0xad, 0x7f, 0xca], "vmovdqu8 ymm2{k5}{z}, ymm1");
    test_display(&[0x62, 0xf1, 0x7f, 0xbd, 0x7a, 0x0a], "vcvtudq2ps ymm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0x7f, 0xbd, 0x7a, 0x4a, 0x01], "vcvtudq2ps ymm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf1, 0x7f, 0xbd, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{z}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0xcd, 0x6f, 0x0a], "vmovdqu8 zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0xcd, 0x6f, 0x4a, 0x01], "vmovdqu8 zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7f, 0xcd, 0x6f, 0xca], "vmovdqu8 zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0xcd, 0x7a, 0x0a], "vcvtudq2ps zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0x7f, 0xcd, 0x7a, 0x4a, 0x01], "vcvtudq2ps zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0x7f, 0xcd, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0xcd, 0x7f, 0xca], "vmovdqu8 zmm2{k5}{z}, zmm1");
    test_display(&[0x62, 0xf1, 0x7f, 0xdd, 0x7a, 0x0a], "vcvtudq2ps zmm1{k5}{z}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf1, 0x7f, 0xdd, 0x7a, 0x4a, 0x01], "vcvtudq2ps zmm1{k5}{z}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf1, 0x7f, 0xdd, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{z}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0x7f, 0xfd, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{z}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x08, 0x5b, 0x0a], "vcvtqq2ps xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x08, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfc, 0x08, 0x5b, 0xca], "vcvtqq2ps xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x08, 0x78, 0x0a], "vcvttpd2udq xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x08, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfc, 0x08, 0x78, 0xca], "vcvttpd2udq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x08, 0x79, 0x0a], "vcvtpd2udq xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x08, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfc, 0x08, 0x79, 0xca], "vcvtpd2udq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x0d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x0d, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfc, 0x0d, 0x5b, 0xca], "vcvtqq2ps xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x0d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x0d, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfc, 0x0d, 0x78, 0xca], "vcvttpd2udq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x0d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x0d, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfc, 0x0d, 0x79, 0xca], "vcvtpd2udq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x18, 0x5b, 0x0a], "vcvtqq2ps xmm1, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x18, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x18, 0x5b, 0xca], "vcvtqq2ps ymm1{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x18, 0x78, 0x0a], "vcvttpd2udq xmm1, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x18, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x18, 0x79, 0x0a], "vcvtpd2udq xmm1, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x18, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x18, 0x79, 0xca], "vcvtpd2udq ymm1{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x1d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x1d, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1{k5}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x1d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x1d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x1d, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1{k5}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x1d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x1d, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1{k5}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x1d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x28, 0x5b, 0x0a], "vcvtqq2ps xmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x28, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfc, 0x28, 0x5b, 0xca], "vcvtqq2ps xmm1, ymm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x28, 0x78, 0x0a], "vcvttpd2udq xmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x28, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfc, 0x28, 0x78, 0xca], "vcvttpd2udq xmm1, ymm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x28, 0x79, 0x0a], "vcvtpd2udq xmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x28, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfc, 0x28, 0x79, 0xca], "vcvtpd2udq xmm1, ymm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x2d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x2d, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfc, 0x2d, 0x5b, 0xca], "vcvtqq2ps xmm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x2d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x2d, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfc, 0x2d, 0x78, 0xca], "vcvttpd2udq xmm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x2d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x2d, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfc, 0x2d, 0x79, 0xca], "vcvtpd2udq xmm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x38, 0x5b, 0x0a], "vcvtqq2ps xmm1, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0x38, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0x38, 0x5b, 0xca], "vcvtqq2ps ymm1{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x38, 0x78, 0x0a], "vcvttpd2udq xmm1, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0x38, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0x38, 0x79, 0x0a], "vcvtpd2udq xmm1, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0x38, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0x38, 0x79, 0xca], "vcvtpd2udq ymm1{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x3d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0x3d, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1{k5}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0x3d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x3d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0x3d, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1{k5}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0x3d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0x3d, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1{k5}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0x3d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x48, 0x5b, 0x0a], "vcvtqq2ps ymm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x48, 0x5b, 0x4a, 0x01], "vcvtqq2ps ymm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfc, 0x48, 0x5b, 0xca], "vcvtqq2ps ymm1, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x48, 0x78, 0x0a], "vcvttpd2udq ymm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x48, 0x78, 0x4a, 0x01], "vcvttpd2udq ymm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfc, 0x48, 0x78, 0xca], "vcvttpd2udq ymm1, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x48, 0x79, 0x0a], "vcvtpd2udq ymm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x48, 0x79, 0x4a, 0x01], "vcvtpd2udq ymm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfc, 0x48, 0x79, 0xca], "vcvtpd2udq ymm1, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x4d, 0x5b, 0x0a], "vcvtqq2ps ymm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x4d, 0x5b, 0x4a, 0x01], "vcvtqq2ps ymm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfc, 0x4d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x4d, 0x78, 0x0a], "vcvttpd2udq ymm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x4d, 0x78, 0x4a, 0x01], "vcvttpd2udq ymm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfc, 0x4d, 0x78, 0xca], "vcvttpd2udq ymm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x4d, 0x79, 0x0a], "vcvtpd2udq ymm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x4d, 0x79, 0x4a, 0x01], "vcvtpd2udq ymm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfc, 0x4d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x58, 0x5b, 0x0a], "vcvtqq2ps ymm1, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0x58, 0x5b, 0x4a, 0x01], "vcvtqq2ps ymm1, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0x58, 0x5b, 0xca], "vcvtqq2ps ymm1{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x58, 0x78, 0x0a], "vcvttpd2udq ymm1, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0x58, 0x78, 0x4a, 0x01], "vcvttpd2udq ymm1, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0x58, 0x79, 0x0a], "vcvtpd2udq ymm1, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0x58, 0x79, 0x4a, 0x01], "vcvtpd2udq ymm1, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0x58, 0x79, 0xca], "vcvtpd2udq ymm1{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x5d, 0x5b, 0x0a], "vcvtqq2ps ymm1{k5}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0x5d, 0x5b, 0x4a, 0x01], "vcvtqq2ps ymm1{k5}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0x5d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x5d, 0x78, 0x0a], "vcvttpd2udq ymm1{k5}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0x5d, 0x78, 0x4a, 0x01], "vcvttpd2udq ymm1{k5}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0x5d, 0x79, 0x0a], "vcvtpd2udq ymm1{k5}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0x5d, 0x79, 0x4a, 0x01], "vcvtpd2udq ymm1{k5}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0x5d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x78, 0x5b, 0xca], "vcvtqq2ps ymm1{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x78, 0x78, 0xca], "vcvttpd2udq ymm1{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x78, 0x79, 0xca], "vcvtpd2udq ymm1{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x7d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x7d, 0x78, 0xca], "vcvttpd2udq ymm1{k5}{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x7d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x8d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x8d, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfc, 0x8d, 0x5b, 0xca], "vcvtqq2ps xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x8d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x8d, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfc, 0x8d, 0x78, 0xca], "vcvttpd2udq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x8d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0x8d, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfc, 0x8d, 0x79, 0xca], "vcvtpd2udq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x9d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}{z}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x9d, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1{k5}{z}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x9d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{z}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0x9d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}{z}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x9d, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1{k5}{z}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x9d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}{z}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x9d, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1{k5}{z}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfc, 0x9d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{z}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0xad, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0xad, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfc, 0xad, 0x5b, 0xca], "vcvtqq2ps xmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xfc, 0xad, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0xad, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfc, 0xad, 0x78, 0xca], "vcvttpd2udq xmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xfc, 0xad, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0xad, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfc, 0xad, 0x79, 0xca], "vcvtpd2udq xmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xfc, 0xbd, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}{z}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0xbd, 0x5b, 0x4a, 0x01], "vcvtqq2ps xmm1{k5}{z}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0xbd, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{z}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0xbd, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}{z}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0xbd, 0x78, 0x4a, 0x01], "vcvttpd2udq xmm1{k5}{z}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0xbd, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}{z}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0xbd, 0x79, 0x4a, 0x01], "vcvtpd2udq xmm1{k5}{z}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfc, 0xbd, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{z}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0xcd, 0x5b, 0x0a], "vcvtqq2ps ymm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0xcd, 0x5b, 0x4a, 0x01], "vcvtqq2ps ymm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfc, 0xcd, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0xcd, 0x78, 0x0a], "vcvttpd2udq ymm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0xcd, 0x78, 0x4a, 0x01], "vcvttpd2udq ymm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfc, 0xcd, 0x78, 0xca], "vcvttpd2udq ymm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0xcd, 0x79, 0x0a], "vcvtpd2udq ymm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfc, 0xcd, 0x79, 0x4a, 0x01], "vcvtpd2udq ymm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfc, 0xcd, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0xdd, 0x5b, 0x0a], "vcvtqq2ps ymm1{k5}{z}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0xdd, 0x5b, 0x4a, 0x01], "vcvtqq2ps ymm1{k5}{z}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0xdd, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{z}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0xdd, 0x78, 0x0a], "vcvttpd2udq ymm1{k5}{z}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0xdd, 0x78, 0x4a, 0x01], "vcvttpd2udq ymm1{k5}{z}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0xdd, 0x79, 0x0a], "vcvtpd2udq ymm1{k5}{z}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0xdd, 0x79, 0x4a, 0x01], "vcvtpd2udq ymm1{k5}{z}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfc, 0xdd, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{z}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0xfd, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{z}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0xfd, 0x78, 0xca], "vcvttpd2udq ymm1{k5}{z}{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfc, 0xfd, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{z}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x10, 0x0a], "vmovupd xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x10, 0x4a, 0x01], "vmovupd xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x10, 0xca], "vmovupd xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x11, 0x0a], "vmovupd xmmword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x11, 0x4a, 0x01], "vmovupd xmmword [bp + si + 0x10], xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x11, 0xca], "vmovupd xmm2, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x12, 0x0a], "vmovlpd xmm1, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x12, 0x4a, 0x01], "vmovlpd xmm1, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x13, 0x0a], "vmovlpd qword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x13, 0x4a, 0x01], "vmovlpd qword [bp + si + 0x8], xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x14, 0x0a], "vunpcklpd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x14, 0x4a, 0x01], "vunpcklpd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x14, 0xca], "vunpcklpd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x15, 0x0a], "vunpckhpd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x15, 0x4a, 0x01], "vunpckhpd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x15, 0xca], "vunpckhpd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x16, 0x0a], "vmovhpd xmm1, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x16, 0x4a, 0x01], "vmovhpd xmm1, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x17, 0x0a], "vmovhpd qword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x17, 0x4a, 0x01], "vmovhpd qword [bp + si + 0x8], xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x28, 0x0a], "vmovapd xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x28, 0x4a, 0x01], "vmovapd xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x28, 0xca], "vmovapd xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x29, 0x0a], "vmovapd xmmword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x29, 0x4a, 0x01], "vmovapd xmmword [bp + si + 0x10], xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x29, 0xca], "vmovapd xmm2, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x2b, 0x0a], "vmovntpd xmmword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x2b, 0x4a, 0x01], "vmovntpd xmmword [bp + si + 0x10], xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x51, 0x0a], "vsqrtpd xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x51, 0x4a, 0x01], "vsqrtpd xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x51, 0xca], "vsqrtpd xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x54, 0x0a], "vandpd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x54, 0x4a, 0x01], "vandpd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x54, 0xca], "vandpd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x55, 0x0a], "vandnpd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x55, 0x4a, 0x01], "vandnpd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x55, 0xca], "vandnpd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x56, 0x0a], "vorpd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x56, 0x4a, 0x01], "vorpd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x56, 0xca], "vorpd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x57, 0x0a], "vxorpd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x57, 0x4a, 0x01], "vxorpd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x57, 0xca], "vxorpd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x58, 0x0a], "vaddpd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x58, 0x4a, 0x01], "vaddpd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x58, 0xca], "vaddpd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x59, 0x0a], "vmulpd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x59, 0x4a, 0x01], "vmulpd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x59, 0xca], "vmulpd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x5a, 0x0a], "vcvtpd2ps xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x5a, 0xca], "vcvtpd2ps xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x5c, 0x0a], "vsubpd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x5c, 0x4a, 0x01], "vsubpd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x5c, 0xca], "vsubpd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x5d, 0x0a], "vminpd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x5d, 0x4a, 0x01], "vminpd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x5d, 0xca], "vminpd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x5e, 0x0a], "vdivpd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x5e, 0x4a, 0x01], "vdivpd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x5e, 0xca], "vdivpd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x5f, 0x0a], "vmaxpd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x5f, 0x4a, 0x01], "vmaxpd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x5f, 0xca], "vmaxpd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x60, 0x0a], "vpunpcklbw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x60, 0x4a, 0x01], "vpunpcklbw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x60, 0xca], "vpunpcklbw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x61, 0x0a], "vpunpcklwd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x61, 0x4a, 0x01], "vpunpcklwd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x61, 0xca], "vpunpcklwd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x63, 0x0a], "vpacksswb xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x63, 0x4a, 0x01], "vpacksswb xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x63, 0xca], "vpacksswb xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x64, 0x0a], "vpcmpgtb k1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x64, 0x4a, 0x01], "vpcmpgtb k1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x64, 0xca], "vpcmpgtb k1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x65, 0x0a], "vpcmpgtw k1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x65, 0x4a, 0x01], "vpcmpgtw k1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x65, 0xca], "vpcmpgtw k1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x67, 0x0a], "vpackuswb xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x67, 0x4a, 0x01], "vpackuswb xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x67, 0xca], "vpackuswb xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x68, 0x0a], "vpunpckhbw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x68, 0x4a, 0x01], "vpunpckhbw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x68, 0xca], "vpunpckhbw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x69, 0x0a], "vpunpckhwd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x69, 0x4a, 0x01], "vpunpckhwd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x69, 0xca], "vpunpckhwd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x6c, 0x0a], "vpunpcklqdq xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x6c, 0x4a, 0x01], "vpunpcklqdq xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x6c, 0xca], "vpunpcklqdq xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x6d, 0x0a], "vpunpckhqdq xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x6d, 0x4a, 0x01], "vpunpckhqdq xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x6d, 0xca], "vpunpckhqdq xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x6e, 0x0a], "vmovd xmm1, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x6e, 0x4a, 0x01], "vmovd xmm1, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x6e, 0xca], "vmovd xmm1, edx");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x6f, 0x0a], "vmovdqa64 xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x6f, 0x4a, 0x01], "vmovdqa64 xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x6f, 0xca], "vmovdqa64 xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x72, 0x0a, 0xcc], "vprolq xmm0, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x72, 0x4a, 0x01, 0xcc], "vprolq xmm0, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x72, 0xca, 0xcc], "vprolq xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x74, 0x0a], "vpcmpeqb k1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x74, 0x4a, 0x01], "vpcmpeqb k1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x74, 0xca], "vpcmpeqb k1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x75, 0x0a], "vpcmpeqw k1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x75, 0x4a, 0x01], "vpcmpeqw k1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x75, 0xca], "vpcmpeqw k1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x78, 0x0a], "vcvttpd2uqq xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x78, 0x4a, 0x01], "vcvttpd2uqq xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x78, 0xca], "vcvttpd2uqq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x79, 0x0a], "vcvtpd2uqq xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x79, 0x4a, 0x01], "vcvtpd2uqq xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x79, 0xca], "vcvtpd2uqq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x7a, 0x0a], "vcvttpd2qq xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x7a, 0x4a, 0x01], "vcvttpd2qq xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x7a, 0xca], "vcvttpd2qq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x7b, 0x0a], "vcvtpd2qq xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x7b, 0x4a, 0x01], "vcvtpd2qq xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x7b, 0xca], "vcvtpd2qq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x7e, 0x0a], "vmovd dword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x7e, 0x4a, 0x01], "vmovd dword [bp + si + 0x4], xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x7e, 0xca], "vmovd edx, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x7f, 0x0a], "vmovdqa64 xmmword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x7f, 0x4a, 0x01], "vmovdqa64 xmmword [bp + si + 0x10], xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0x7f, 0xca], "vmovdqa64 xmm2, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xc2, 0x0a, 0xcc], "vcmppd k1, xmm0, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1, xmm0, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xc2, 0xca, 0xcc], "vcmppd k1, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xc4, 0x0a, 0xcc], "vpinsrw xmm1, xmm0, word [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xc4, 0x4a, 0x01, 0xcc], "vpinsrw xmm1, xmm0, word [bp + si + 0x2], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xc4, 0xca, 0xcc], "vpinsrw xmm1, xmm0, edx, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xc5, 0xca, 0xcc], "vpextrw ecx, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xc6, 0x0a, 0xcc], "vshufpd xmm1, xmm0, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd xmm1, xmm0, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xc6, 0xca, 0xcc], "vshufpd xmm1, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd1, 0x0a], "vpsrlw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd1, 0x4a, 0x01], "vpsrlw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd1, 0xca], "vpsrlw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd3, 0x0a], "vpsrlq xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd3, 0x4a, 0x01], "vpsrlq xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd3, 0xca], "vpsrlq xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd4, 0x0a], "vpaddq xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd4, 0x4a, 0x01], "vpaddq xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd4, 0xca], "vpaddq xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd5, 0x0a], "vpmullw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd5, 0x4a, 0x01], "vpmullw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd5, 0xca], "vpmullw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd6, 0x0a], "vmovq qword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd6, 0x4a, 0x01], "vmovq qword [bp + si + 0x8], xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd6, 0xca], "vmovq xmm2, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd8, 0x0a], "vpsubusb xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd8, 0x4a, 0x01], "vpsubusb xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd8, 0xca], "vpsubusb xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd9, 0x0a], "vpsubusw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd9, 0x4a, 0x01], "vpsubusw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xd9, 0xca], "vpsubusw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xda, 0x0a], "vpminub xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xda, 0x4a, 0x01], "vpminub xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xda, 0xca], "vpminub xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xdb, 0x0a], "vpandq xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xdb, 0x4a, 0x01], "vpandq xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xdb, 0xca], "vpandq xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xdc, 0x0a], "vpaddusb xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xdc, 0x4a, 0x01], "vpaddusb xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xdc, 0xca], "vpaddusb xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xdd, 0x0a], "vpaddusw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xdd, 0x4a, 0x01], "vpaddusw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xdd, 0xca], "vpaddusw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xde, 0x0a], "vpmaxub xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xde, 0x4a, 0x01], "vpmaxub xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xde, 0xca], "vpmaxub xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xdf, 0x0a], "vpandnq xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xdf, 0x4a, 0x01], "vpandnq xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xdf, 0xca], "vpandnq xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe0, 0x0a], "vpavgb xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe0, 0x4a, 0x01], "vpavgb xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe0, 0xca], "vpavgb xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe1, 0x0a], "vpsraw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe1, 0x4a, 0x01], "vpsraw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe1, 0xca], "vpsraw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe2, 0x0a], "vpsraq xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe2, 0x4a, 0x01], "vpsraq xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe2, 0xca], "vpsraq xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe3, 0x0a], "vpavgw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe3, 0x4a, 0x01], "vpavgw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe3, 0xca], "vpavgw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe4, 0x0a], "vpmulhuw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe4, 0x4a, 0x01], "vpmulhuw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe4, 0xca], "vpmulhuw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe5, 0x0a], "vpmulhw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe5, 0x4a, 0x01], "vpmulhw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe5, 0xca], "vpmulhw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe6, 0x0a], "vcvttpd2dq xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe6, 0xca], "vcvttpd2dq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe8, 0x0a], "vpsubsb xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe8, 0x4a, 0x01], "vpsubsb xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe8, 0xca], "vpsubsb xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe9, 0x0a], "vpsubsw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe9, 0x4a, 0x01], "vpsubsw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xe9, 0xca], "vpsubsw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xea, 0x0a], "vpminsw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xea, 0x4a, 0x01], "vpminsw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xea, 0xca], "vpminsw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xeb, 0x0a], "vporq xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xeb, 0x4a, 0x01], "vporq xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xeb, 0xca], "vporq xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xec, 0x0a], "vpaddsb xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xec, 0x4a, 0x01], "vpaddsb xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xec, 0xca], "vpaddsb xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xed, 0x0a], "vpaddsw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xed, 0x4a, 0x01], "vpaddsw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xed, 0xca], "vpaddsw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xee, 0x0a], "vpmaxsw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xee, 0x4a, 0x01], "vpmaxsw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xee, 0xca], "vpmaxsw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xef, 0x0a], "vpxorq xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xef, 0x4a, 0x01], "vpxorq xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xef, 0xca], "vpxorq xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf1, 0x0a], "vpsllw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf1, 0x4a, 0x01], "vpsllw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf1, 0xca], "vpsllw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf3, 0x0a], "vpsllq xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf3, 0x4a, 0x01], "vpsllq xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf3, 0xca], "vpsllq xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf4, 0x0a], "vpmuludq xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf4, 0x4a, 0x01], "vpmuludq xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf4, 0xca], "vpmuludq xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf5, 0x0a], "vpmaddwd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf5, 0x4a, 0x01], "vpmaddwd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf5, 0xca], "vpmaddwd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf6, 0x0a], "vpsadbw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf6, 0x4a, 0x01], "vpsadbw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf6, 0xca], "vpsadbw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf8, 0x0a], "vpsubb xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf8, 0x4a, 0x01], "vpsubb xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf8, 0xca], "vpsubb xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf9, 0x0a], "vpsubw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf9, 0x4a, 0x01], "vpsubw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xf9, 0xca], "vpsubw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xfb, 0x0a], "vpsubq xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xfb, 0x4a, 0x01], "vpsubq xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xfb, 0xca], "vpsubq xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xfc, 0x0a], "vpaddb xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xfc, 0x4a, 0x01], "vpaddb xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xfc, 0xca], "vpaddb xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xfd, 0x0a], "vpaddw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xfd, 0x4a, 0x01], "vpaddw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x08, 0xfd, 0xca], "vpaddw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x10, 0x0a], "vmovupd xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x10, 0x4a, 0x01], "vmovupd xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x10, 0xca], "vmovupd xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x11, 0x0a], "vmovupd xmmword [bp + si]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x11, 0x4a, 0x01], "vmovupd xmmword [bp + si + 0x10]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x11, 0xca], "vmovupd xmm2{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x14, 0x0a], "vunpcklpd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x14, 0x4a, 0x01], "vunpcklpd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x14, 0xca], "vunpcklpd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x15, 0x0a], "vunpckhpd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x15, 0x4a, 0x01], "vunpckhpd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x15, 0xca], "vunpckhpd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x28, 0x0a], "vmovapd xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x28, 0x4a, 0x01], "vmovapd xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x28, 0xca], "vmovapd xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x29, 0x0a], "vmovapd xmmword [bp + si]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x29, 0x4a, 0x01], "vmovapd xmmword [bp + si + 0x10]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x29, 0xca], "vmovapd xmm2{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x51, 0x0a], "vsqrtpd xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x51, 0x4a, 0x01], "vsqrtpd xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x51, 0xca], "vsqrtpd xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x54, 0x0a], "vandpd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x54, 0x4a, 0x01], "vandpd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x54, 0xca], "vandpd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x55, 0x0a], "vandnpd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x55, 0x4a, 0x01], "vandnpd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x55, 0xca], "vandnpd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x56, 0x0a], "vorpd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x56, 0x4a, 0x01], "vorpd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x56, 0xca], "vorpd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x57, 0x0a], "vxorpd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x57, 0x4a, 0x01], "vxorpd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x57, 0xca], "vxorpd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x58, 0x0a], "vaddpd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x58, 0x4a, 0x01], "vaddpd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x58, 0xca], "vaddpd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x59, 0x0a], "vmulpd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x59, 0x4a, 0x01], "vmulpd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x59, 0xca], "vmulpd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x5a, 0xca], "vcvtpd2ps xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x5c, 0x0a], "vsubpd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x5c, 0x4a, 0x01], "vsubpd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x5c, 0xca], "vsubpd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x5d, 0x0a], "vminpd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x5d, 0x4a, 0x01], "vminpd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x5d, 0xca], "vminpd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x5e, 0x0a], "vdivpd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x5e, 0x4a, 0x01], "vdivpd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x5e, 0xca], "vdivpd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x5f, 0x0a], "vmaxpd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x5f, 0x4a, 0x01], "vmaxpd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x5f, 0xca], "vmaxpd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x60, 0x0a], "vpunpcklbw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x60, 0x4a, 0x01], "vpunpcklbw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x60, 0xca], "vpunpcklbw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x61, 0x0a], "vpunpcklwd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x61, 0x4a, 0x01], "vpunpcklwd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x61, 0xca], "vpunpcklwd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x63, 0x0a], "vpacksswb xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x63, 0x4a, 0x01], "vpacksswb xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x63, 0xca], "vpacksswb xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x64, 0x0a], "vpcmpgtb k1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x64, 0x4a, 0x01], "vpcmpgtb k1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x64, 0xca], "vpcmpgtb k1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x65, 0x0a], "vpcmpgtw k1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x65, 0x4a, 0x01], "vpcmpgtw k1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x65, 0xca], "vpcmpgtw k1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x67, 0x0a], "vpackuswb xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x67, 0x4a, 0x01], "vpackuswb xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x67, 0xca], "vpackuswb xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x68, 0x0a], "vpunpckhbw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x68, 0x4a, 0x01], "vpunpckhbw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x68, 0xca], "vpunpckhbw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x69, 0x0a], "vpunpckhwd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x69, 0x4a, 0x01], "vpunpckhwd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x69, 0xca], "vpunpckhwd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x6c, 0x0a], "vpunpcklqdq xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x6c, 0x4a, 0x01], "vpunpcklqdq xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x6c, 0xca], "vpunpcklqdq xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x6d, 0x0a], "vpunpckhqdq xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x6d, 0x4a, 0x01], "vpunpckhqdq xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x6d, 0xca], "vpunpckhqdq xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x6f, 0x0a], "vmovdqa64 xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x6f, 0x4a, 0x01], "vmovdqa64 xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x6f, 0xca], "vmovdqa64 xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x72, 0x0a, 0xcc], "vprolq xmm0{k5}, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x72, 0x4a, 0x01, 0xcc], "vprolq xmm0{k5}, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x72, 0xca, 0xcc], "vprolq xmm0{k5}, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x74, 0x0a], "vpcmpeqb k1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x74, 0x4a, 0x01], "vpcmpeqb k1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x74, 0xca], "vpcmpeqb k1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x75, 0x0a], "vpcmpeqw k1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x75, 0x4a, 0x01], "vpcmpeqw k1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x75, 0xca], "vpcmpeqw k1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x78, 0x0a], "vcvttpd2uqq xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x78, 0x4a, 0x01], "vcvttpd2uqq xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x78, 0xca], "vcvttpd2uqq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x79, 0x0a], "vcvtpd2uqq xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x79, 0x4a, 0x01], "vcvtpd2uqq xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x79, 0xca], "vcvtpd2uqq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x7a, 0x0a], "vcvttpd2qq xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x7a, 0x4a, 0x01], "vcvttpd2qq xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x7a, 0xca], "vcvttpd2qq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x7b, 0x0a], "vcvtpd2qq xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x7b, 0x4a, 0x01], "vcvtpd2qq xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x7b, 0xca], "vcvtpd2qq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x7f, 0x0a], "vmovdqa64 xmmword [bp + si]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x7f, 0x4a, 0x01], "vmovdqa64 xmmword [bp + si + 0x10]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0x7f, 0xca], "vmovdqa64 xmm2{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, xmm0, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1{k5}, xmm0, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xc2, 0xca, 0xcc], "vcmppd k1{k5}, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xc6, 0x0a, 0xcc], "vshufpd xmm1{k5}, xmm0, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd xmm1{k5}, xmm0, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xc6, 0xca, 0xcc], "vshufpd xmm1{k5}, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd1, 0x0a], "vpsrlw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd1, 0x4a, 0x01], "vpsrlw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd1, 0xca], "vpsrlw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd3, 0x0a], "vpsrlq xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd3, 0x4a, 0x01], "vpsrlq xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd3, 0xca], "vpsrlq xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd4, 0x0a], "vpaddq xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd4, 0x4a, 0x01], "vpaddq xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd4, 0xca], "vpaddq xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd5, 0x0a], "vpmullw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd5, 0x4a, 0x01], "vpmullw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd5, 0xca], "vpmullw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd8, 0x0a], "vpsubusb xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd8, 0x4a, 0x01], "vpsubusb xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd8, 0xca], "vpsubusb xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd9, 0x0a], "vpsubusw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd9, 0x4a, 0x01], "vpsubusw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xd9, 0xca], "vpsubusw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xda, 0x0a], "vpminub xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xda, 0x4a, 0x01], "vpminub xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xda, 0xca], "vpminub xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xdb, 0x0a], "vpandq xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xdb, 0x4a, 0x01], "vpandq xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xdb, 0xca], "vpandq xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xdc, 0x0a], "vpaddusb xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xdc, 0x4a, 0x01], "vpaddusb xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xdc, 0xca], "vpaddusb xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xdd, 0x0a], "vpaddusw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xdd, 0x4a, 0x01], "vpaddusw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xdd, 0xca], "vpaddusw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xde, 0x0a], "vpmaxub xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xde, 0x4a, 0x01], "vpmaxub xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xde, 0xca], "vpmaxub xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xdf, 0x0a], "vpandnq xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xdf, 0x4a, 0x01], "vpandnq xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xdf, 0xca], "vpandnq xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe0, 0x0a], "vpavgb xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe0, 0x4a, 0x01], "vpavgb xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe0, 0xca], "vpavgb xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe1, 0x0a], "vpsraw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe1, 0x4a, 0x01], "vpsraw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe1, 0xca], "vpsraw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe2, 0x0a], "vpsraq xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe2, 0x4a, 0x01], "vpsraq xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe2, 0xca], "vpsraq xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe3, 0x0a], "vpavgw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe3, 0x4a, 0x01], "vpavgw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe3, 0xca], "vpavgw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe4, 0x0a], "vpmulhuw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe4, 0x4a, 0x01], "vpmulhuw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe4, 0xca], "vpmulhuw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe5, 0x0a], "vpmulhw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe5, 0x4a, 0x01], "vpmulhw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe5, 0xca], "vpmulhw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe6, 0xca], "vcvttpd2dq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe8, 0x0a], "vpsubsb xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe8, 0x4a, 0x01], "vpsubsb xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe8, 0xca], "vpsubsb xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe9, 0x0a], "vpsubsw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe9, 0x4a, 0x01], "vpsubsw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xe9, 0xca], "vpsubsw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xea, 0x0a], "vpminsw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xea, 0x4a, 0x01], "vpminsw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xea, 0xca], "vpminsw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xeb, 0x0a], "vporq xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xeb, 0x4a, 0x01], "vporq xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xeb, 0xca], "vporq xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xec, 0x0a], "vpaddsb xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xec, 0x4a, 0x01], "vpaddsb xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xec, 0xca], "vpaddsb xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xed, 0x0a], "vpaddsw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xed, 0x4a, 0x01], "vpaddsw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xed, 0xca], "vpaddsw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xee, 0x0a], "vpmaxsw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xee, 0x4a, 0x01], "vpmaxsw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xee, 0xca], "vpmaxsw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xef, 0x0a], "vpxorq xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xef, 0x4a, 0x01], "vpxorq xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xef, 0xca], "vpxorq xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf1, 0x0a], "vpsllw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf1, 0x4a, 0x01], "vpsllw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf1, 0xca], "vpsllw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf3, 0x0a], "vpsllq xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf3, 0x4a, 0x01], "vpsllq xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf3, 0xca], "vpsllq xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf4, 0x0a], "vpmuludq xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf4, 0x4a, 0x01], "vpmuludq xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf4, 0xca], "vpmuludq xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf5, 0x0a], "vpmaddwd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf5, 0x4a, 0x01], "vpmaddwd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf5, 0xca], "vpmaddwd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf8, 0x0a], "vpsubb xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf8, 0x4a, 0x01], "vpsubb xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf8, 0xca], "vpsubb xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf9, 0x0a], "vpsubw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf9, 0x4a, 0x01], "vpsubw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xf9, 0xca], "vpsubw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xfb, 0x0a], "vpsubq xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xfb, 0x4a, 0x01], "vpsubq xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xfb, 0xca], "vpsubq xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xfc, 0x0a], "vpaddb xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xfc, 0x4a, 0x01], "vpaddb xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xfc, 0xca], "vpaddb xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xfd, 0x0a], "vpaddw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xfd, 0x4a, 0x01], "vpaddw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x0d, 0xfd, 0xca], "vpaddw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x14, 0x0a], "vunpcklpd xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x14, 0x4a, 0x01], "vunpcklpd xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x15, 0x0a], "vunpckhpd xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x15, 0x4a, 0x01], "vunpckhpd xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x51, 0x0a], "vsqrtpd xmm1, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x51, 0x4a, 0x01], "vsqrtpd xmm1, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x51, 0xca], "vsqrtpd zmm1{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x54, 0x0a], "vandpd xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x54, 0x4a, 0x01], "vandpd xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x55, 0x0a], "vandnpd xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x55, 0x4a, 0x01], "vandnpd xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x56, 0x0a], "vorpd xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x56, 0x4a, 0x01], "vorpd xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x57, 0x0a], "vxorpd xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x57, 0x4a, 0x01], "vxorpd xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x58, 0x0a], "vaddpd xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x58, 0x4a, 0x01], "vaddpd xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x58, 0xca], "vaddpd zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x59, 0x0a], "vmulpd xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x59, 0x4a, 0x01], "vmulpd xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x59, 0xca], "vmulpd zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x5a, 0x0a], "vcvtpd2ps xmm1, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x5a, 0xca], "vcvtpd2ps ymm1{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x5c, 0x0a], "vsubpd xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x5c, 0x4a, 0x01], "vsubpd xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x5c, 0xca], "vsubpd zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x5d, 0x0a], "vminpd xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x5d, 0x4a, 0x01], "vminpd xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x5e, 0x0a], "vdivpd xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x5e, 0x4a, 0x01], "vdivpd xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x5e, 0xca], "vdivpd zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x5f, 0x0a], "vmaxpd xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x5f, 0x4a, 0x01], "vmaxpd xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x6c, 0x0a], "vpunpcklqdq xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x6c, 0x4a, 0x01], "vpunpcklqdq xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x6d, 0x0a], "vpunpckhqdq xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x6d, 0x4a, 0x01], "vpunpckhqdq xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x72, 0x0a, 0xcc], "vprolq xmm0, qword [bp + si]{1to2}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x72, 0x4a, 0x01, 0xcc], "vprolq xmm0, qword [bp + si + 0x8]{1to2}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x78, 0x0a], "vcvttpd2uqq xmm1, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x78, 0x4a, 0x01], "vcvttpd2uqq xmm1, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x79, 0x0a], "vcvtpd2uqq xmm1, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x79, 0x4a, 0x01], "vcvtpd2uqq xmm1, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x79, 0xca], "vcvtpd2uqq zmm1{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x7a, 0x0a], "vcvttpd2qq xmm1, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x7a, 0x4a, 0x01], "vcvttpd2qq xmm1, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x7b, 0x0a], "vcvtpd2qq xmm1, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x7b, 0x4a, 0x01], "vcvtpd2qq xmm1, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0x7b, 0xca], "vcvtpd2qq zmm1{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xc2, 0x0a, 0xcc], "vcmppd k1, xmm0, qword [bp + si]{1to2}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1, xmm0, qword [bp + si + 0x8]{1to2}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xc6, 0x0a, 0xcc], "vshufpd xmm1, xmm0, qword [bp + si]{1to2}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd xmm1, xmm0, qword [bp + si + 0x8]{1to2}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xd4, 0x0a], "vpaddq xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xd4, 0x4a, 0x01], "vpaddq xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xdb, 0x0a], "vpandq xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xdb, 0x4a, 0x01], "vpandq xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xdf, 0x0a], "vpandnq xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xdf, 0x4a, 0x01], "vpandnq xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xe6, 0x0a], "vcvttpd2dq xmm1, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xeb, 0x0a], "vporq xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xeb, 0x4a, 0x01], "vporq xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xef, 0x0a], "vpxorq xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xef, 0x4a, 0x01], "vpxorq xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xf4, 0x0a], "vpmuludq xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xf4, 0x4a, 0x01], "vpmuludq xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xfb, 0x0a], "vpsubq xmm1, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x18, 0xfb, 0x4a, 0x01], "vpsubq xmm1, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x14, 0x0a], "vunpcklpd xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x14, 0x4a, 0x01], "vunpcklpd xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x15, 0x0a], "vunpckhpd xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x15, 0x4a, 0x01], "vunpckhpd xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x51, 0x0a], "vsqrtpd xmm1{k5}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x51, 0x4a, 0x01], "vsqrtpd xmm1{k5}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x51, 0xca], "vsqrtpd zmm1{k5}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x54, 0x0a], "vandpd xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x54, 0x4a, 0x01], "vandpd xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x55, 0x0a], "vandnpd xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x55, 0x4a, 0x01], "vandnpd xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x56, 0x0a], "vorpd xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x56, 0x4a, 0x01], "vorpd xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x57, 0x0a], "vxorpd xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x57, 0x4a, 0x01], "vxorpd xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x58, 0x0a], "vaddpd xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x58, 0x4a, 0x01], "vaddpd xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x58, 0xca], "vaddpd zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x59, 0x0a], "vmulpd xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x59, 0x4a, 0x01], "vmulpd xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x59, 0xca], "vmulpd zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1{k5}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x5c, 0x0a], "vsubpd xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x5c, 0x4a, 0x01], "vsubpd xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x5c, 0xca], "vsubpd zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x5d, 0x0a], "vminpd xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x5d, 0x4a, 0x01], "vminpd xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x5e, 0x0a], "vdivpd xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x5e, 0x4a, 0x01], "vdivpd xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x5e, 0xca], "vdivpd zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x5f, 0x0a], "vmaxpd xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x5f, 0x4a, 0x01], "vmaxpd xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x6c, 0x0a], "vpunpcklqdq xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x6c, 0x4a, 0x01], "vpunpcklqdq xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x6d, 0x0a], "vpunpckhqdq xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x6d, 0x4a, 0x01], "vpunpckhqdq xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x72, 0x0a, 0xcc], "vprolq xmm0{k5}, qword [bp + si]{1to2}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x72, 0x4a, 0x01, 0xcc], "vprolq xmm0{k5}, qword [bp + si + 0x8]{1to2}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x78, 0x0a], "vcvttpd2uqq xmm1{k5}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x78, 0x4a, 0x01], "vcvttpd2uqq xmm1{k5}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x79, 0x0a], "vcvtpd2uqq xmm1{k5}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x79, 0x4a, 0x01], "vcvtpd2uqq xmm1{k5}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x7a, 0x0a], "vcvttpd2qq xmm1{k5}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x7a, 0x4a, 0x01], "vcvttpd2qq xmm1{k5}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x7b, 0x0a], "vcvtpd2qq xmm1{k5}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x7b, 0x4a, 0x01], "vcvtpd2qq xmm1{k5}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, xmm0, qword [bp + si]{1to2}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1{k5}, xmm0, qword [bp + si + 0x8]{1to2}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xc6, 0x0a, 0xcc], "vshufpd xmm1{k5}, xmm0, qword [bp + si]{1to2}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xd4, 0x0a], "vpaddq xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xd4, 0x4a, 0x01], "vpaddq xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xdb, 0x0a], "vpandq xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xdb, 0x4a, 0x01], "vpandq xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xdf, 0x0a], "vpandnq xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xdf, 0x4a, 0x01], "vpandnq xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1{k5}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xeb, 0x0a], "vporq xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xeb, 0x4a, 0x01], "vporq xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xef, 0x0a], "vpxorq xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xef, 0x4a, 0x01], "vpxorq xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xf4, 0x0a], "vpmuludq xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xf4, 0x4a, 0x01], "vpmuludq xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xfb, 0x0a], "vpsubq xmm1{k5}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x1d, 0xfb, 0x4a, 0x01], "vpsubq xmm1{k5}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x10, 0x0a], "vmovupd ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x10, 0x4a, 0x01], "vmovupd ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x10, 0xca], "vmovupd ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x11, 0x0a], "vmovupd ymmword [bp + si], ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x11, 0x4a, 0x01], "vmovupd ymmword [bp + si + 0x20], ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x11, 0xca], "vmovupd ymm2, ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x14, 0x0a], "vunpcklpd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x14, 0x4a, 0x01], "vunpcklpd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x14, 0xca], "vunpcklpd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x15, 0x0a], "vunpckhpd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x15, 0x4a, 0x01], "vunpckhpd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x15, 0xca], "vunpckhpd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x28, 0x0a], "vmovapd ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x28, 0x4a, 0x01], "vmovapd ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x28, 0xca], "vmovapd ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x29, 0x0a], "vmovapd ymmword [bp + si], ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x29, 0x4a, 0x01], "vmovapd ymmword [bp + si + 0x20], ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x29, 0xca], "vmovapd ymm2, ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x2b, 0x0a], "vmovntpd ymmword [bp + si], ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x2b, 0x4a, 0x01], "vmovntpd ymmword [bp + si + 0x20], ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x2e, 0x0a], "vucomisd xmm1, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x2e, 0x4a, 0x01], "vucomisd xmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x2e, 0xca], "vucomisd xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x2f, 0x0a], "vcomisd xmm1, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x2f, 0x4a, 0x01], "vcomisd xmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x2f, 0xca], "vcomisd xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x51, 0x0a], "vsqrtpd ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x51, 0x4a, 0x01], "vsqrtpd ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x51, 0xca], "vsqrtpd ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x54, 0x0a], "vandpd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x54, 0x4a, 0x01], "vandpd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x54, 0xca], "vandpd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x55, 0x0a], "vandnpd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x55, 0x4a, 0x01], "vandnpd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x55, 0xca], "vandnpd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x56, 0x0a], "vorpd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x56, 0x4a, 0x01], "vorpd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x56, 0xca], "vorpd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x57, 0x0a], "vxorpd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x57, 0x4a, 0x01], "vxorpd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x57, 0xca], "vxorpd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x58, 0x0a], "vaddpd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x58, 0x4a, 0x01], "vaddpd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x58, 0xca], "vaddpd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x59, 0x0a], "vmulpd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x59, 0x4a, 0x01], "vmulpd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x59, 0xca], "vmulpd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x5a, 0x0a], "vcvtpd2ps xmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x5a, 0xca], "vcvtpd2ps xmm1, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x5c, 0x0a], "vsubpd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x5c, 0x4a, 0x01], "vsubpd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x5c, 0xca], "vsubpd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x5d, 0x0a], "vminpd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x5d, 0x4a, 0x01], "vminpd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x5d, 0xca], "vminpd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x5e, 0x0a], "vdivpd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x5e, 0x4a, 0x01], "vdivpd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x5e, 0xca], "vdivpd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x5f, 0x0a], "vmaxpd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x5f, 0x4a, 0x01], "vmaxpd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x5f, 0xca], "vmaxpd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x60, 0x0a], "vpunpcklbw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x60, 0x4a, 0x01], "vpunpcklbw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x60, 0xca], "vpunpcklbw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x61, 0x0a], "vpunpcklwd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x61, 0x4a, 0x01], "vpunpcklwd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x61, 0xca], "vpunpcklwd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x63, 0x0a], "vpacksswb ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x63, 0x4a, 0x01], "vpacksswb ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x63, 0xca], "vpacksswb ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x64, 0x0a], "vpcmpgtb k1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x64, 0x4a, 0x01], "vpcmpgtb k1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x64, 0xca], "vpcmpgtb k1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x65, 0x0a], "vpcmpgtw k1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x65, 0x4a, 0x01], "vpcmpgtw k1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x65, 0xca], "vpcmpgtw k1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x67, 0x0a], "vpackuswb ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x67, 0x4a, 0x01], "vpackuswb ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x67, 0xca], "vpackuswb ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x68, 0x0a], "vpunpckhbw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x68, 0x4a, 0x01], "vpunpckhbw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x68, 0xca], "vpunpckhbw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x69, 0x0a], "vpunpckhwd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x69, 0x4a, 0x01], "vpunpckhwd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x69, 0xca], "vpunpckhwd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x6c, 0x0a], "vpunpcklqdq ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x6c, 0x4a, 0x01], "vpunpcklqdq ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x6c, 0xca], "vpunpcklqdq ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x6d, 0x0a], "vpunpckhqdq ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x6d, 0x4a, 0x01], "vpunpckhqdq ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x6d, 0xca], "vpunpckhqdq ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x6f, 0x0a], "vmovdqa64 ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x6f, 0x4a, 0x01], "vmovdqa64 ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x6f, 0xca], "vmovdqa64 ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x72, 0x0a, 0xcc], "vprolq ymm0, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x72, 0x4a, 0x01, 0xcc], "vprolq ymm0, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x72, 0xca, 0xcc], "vprolq ymm0, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x74, 0x0a], "vpcmpeqb k1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x74, 0x4a, 0x01], "vpcmpeqb k1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x74, 0xca], "vpcmpeqb k1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x75, 0x0a], "vpcmpeqw k1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x75, 0x4a, 0x01], "vpcmpeqw k1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x75, 0xca], "vpcmpeqw k1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x78, 0x0a], "vcvttpd2uqq ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x78, 0x4a, 0x01], "vcvttpd2uqq ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x78, 0xca], "vcvttpd2uqq ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x79, 0x0a], "vcvtpd2uqq ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x79, 0x4a, 0x01], "vcvtpd2uqq ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x79, 0xca], "vcvtpd2uqq ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x7a, 0x0a], "vcvttpd2qq ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x7a, 0x4a, 0x01], "vcvttpd2qq ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x7a, 0xca], "vcvttpd2qq ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x7b, 0x0a], "vcvtpd2qq ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x7b, 0x4a, 0x01], "vcvtpd2qq ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x7b, 0xca], "vcvtpd2qq ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x7f, 0x0a], "vmovdqa64 ymmword [bp + si], ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x7f, 0x4a, 0x01], "vmovdqa64 ymmword [bp + si + 0x20], ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0x7f, 0xca], "vmovdqa64 ymm2, ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xc2, 0x0a, 0xcc], "vcmppd k1, ymm0, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1, ymm0, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xc2, 0xca, 0xcc], "vcmppd k1, ymm0, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xc6, 0x0a, 0xcc], "vshufpd ymm1, ymm0, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd ymm1, ymm0, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xc6, 0xca, 0xcc], "vshufpd ymm1, ymm0, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd1, 0x0a], "vpsrlw ymm1, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd1, 0x4a, 0x01], "vpsrlw ymm1, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd1, 0xca], "vpsrlw ymm1, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd3, 0x0a], "vpsrlq ymm1, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd3, 0x4a, 0x01], "vpsrlq ymm1, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd3, 0xca], "vpsrlq ymm1, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd4, 0x0a], "vpaddq ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd4, 0x4a, 0x01], "vpaddq ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd4, 0xca], "vpaddq ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd5, 0x0a], "vpmullw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd5, 0x4a, 0x01], "vpmullw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd5, 0xca], "vpmullw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd8, 0x0a], "vpsubusb ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd8, 0x4a, 0x01], "vpsubusb ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd8, 0xca], "vpsubusb ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd9, 0x0a], "vpsubusw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd9, 0x4a, 0x01], "vpsubusw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xd9, 0xca], "vpsubusw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xda, 0x0a], "vpminub ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xda, 0x4a, 0x01], "vpminub ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xda, 0xca], "vpminub ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xdb, 0x0a], "vpandq ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xdb, 0x4a, 0x01], "vpandq ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xdb, 0xca], "vpandq ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xdc, 0x0a], "vpaddusb ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xdc, 0x4a, 0x01], "vpaddusb ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xdc, 0xca], "vpaddusb ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xdd, 0x0a], "vpaddusw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xdd, 0x4a, 0x01], "vpaddusw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xdd, 0xca], "vpaddusw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xde, 0x0a], "vpmaxub ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xde, 0x4a, 0x01], "vpmaxub ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xde, 0xca], "vpmaxub ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xdf, 0x0a], "vpandnq ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xdf, 0x4a, 0x01], "vpandnq ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xdf, 0xca], "vpandnq ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe0, 0x0a], "vpavgb ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe0, 0x4a, 0x01], "vpavgb ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe0, 0xca], "vpavgb ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe1, 0x0a], "vpsraw ymm1, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe1, 0x4a, 0x01], "vpsraw ymm1, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe1, 0xca], "vpsraw ymm1, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe2, 0x0a], "vpsraq ymm1, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe2, 0x4a, 0x01], "vpsraq ymm1, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe2, 0xca], "vpsraq ymm1, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe3, 0x0a], "vpavgw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe3, 0x4a, 0x01], "vpavgw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe3, 0xca], "vpavgw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe4, 0x0a], "vpmulhuw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe4, 0x4a, 0x01], "vpmulhuw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe4, 0xca], "vpmulhuw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe5, 0x0a], "vpmulhw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe5, 0x4a, 0x01], "vpmulhw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe5, 0xca], "vpmulhw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe6, 0x0a], "vcvttpd2dq xmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe6, 0xca], "vcvttpd2dq xmm1, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe8, 0x0a], "vpsubsb ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe8, 0x4a, 0x01], "vpsubsb ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe8, 0xca], "vpsubsb ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe9, 0x0a], "vpsubsw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe9, 0x4a, 0x01], "vpsubsw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xe9, 0xca], "vpsubsw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xea, 0x0a], "vpminsw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xea, 0x4a, 0x01], "vpminsw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xea, 0xca], "vpminsw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xeb, 0x0a], "vporq ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xeb, 0x4a, 0x01], "vporq ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xeb, 0xca], "vporq ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xec, 0x0a], "vpaddsb ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xec, 0x4a, 0x01], "vpaddsb ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xec, 0xca], "vpaddsb ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xed, 0x0a], "vpaddsw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xed, 0x4a, 0x01], "vpaddsw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xed, 0xca], "vpaddsw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xee, 0x0a], "vpmaxsw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xee, 0x4a, 0x01], "vpmaxsw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xee, 0xca], "vpmaxsw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xef, 0x0a], "vpxorq ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xef, 0x4a, 0x01], "vpxorq ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xef, 0xca], "vpxorq ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf1, 0x0a], "vpsllw ymm1, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf1, 0x4a, 0x01], "vpsllw ymm1, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf1, 0xca], "vpsllw ymm1, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf3, 0x0a], "vpsllq ymm1, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf3, 0x4a, 0x01], "vpsllq ymm1, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf3, 0xca], "vpsllq ymm1, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf4, 0x0a], "vpmuludq ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf4, 0x4a, 0x01], "vpmuludq ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf4, 0xca], "vpmuludq ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf5, 0x0a], "vpmaddwd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf5, 0x4a, 0x01], "vpmaddwd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf5, 0xca], "vpmaddwd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf6, 0x0a], "vpsadbw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf6, 0x4a, 0x01], "vpsadbw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf6, 0xca], "vpsadbw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf8, 0x0a], "vpsubb ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf8, 0x4a, 0x01], "vpsubb ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf8, 0xca], "vpsubb ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf9, 0x0a], "vpsubw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf9, 0x4a, 0x01], "vpsubw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xf9, 0xca], "vpsubw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xfb, 0x0a], "vpsubq ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xfb, 0x4a, 0x01], "vpsubq ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xfb, 0xca], "vpsubq ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xfc, 0x0a], "vpaddb ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xfc, 0x4a, 0x01], "vpaddb ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xfc, 0xca], "vpaddb ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xfd, 0x0a], "vpaddw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xfd, 0x4a, 0x01], "vpaddw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x28, 0xfd, 0xca], "vpaddw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x10, 0x0a], "vmovupd ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x10, 0x4a, 0x01], "vmovupd ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x10, 0xca], "vmovupd ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x11, 0x0a], "vmovupd ymmword [bp + si]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x11, 0x4a, 0x01], "vmovupd ymmword [bp + si + 0x20]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x11, 0xca], "vmovupd ymm2{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x14, 0x0a], "vunpcklpd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x14, 0x4a, 0x01], "vunpcklpd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x14, 0xca], "vunpcklpd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x15, 0x0a], "vunpckhpd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x15, 0x4a, 0x01], "vunpckhpd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x15, 0xca], "vunpckhpd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x28, 0x0a], "vmovapd ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x28, 0x4a, 0x01], "vmovapd ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x28, 0xca], "vmovapd ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x29, 0x0a], "vmovapd ymmword [bp + si]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x29, 0x4a, 0x01], "vmovapd ymmword [bp + si + 0x20]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x29, 0xca], "vmovapd ymm2{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x51, 0x0a], "vsqrtpd ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x51, 0x4a, 0x01], "vsqrtpd ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x51, 0xca], "vsqrtpd ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x54, 0x0a], "vandpd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x54, 0x4a, 0x01], "vandpd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x54, 0xca], "vandpd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x55, 0x0a], "vandnpd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x55, 0x4a, 0x01], "vandnpd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x55, 0xca], "vandnpd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x56, 0x0a], "vorpd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x56, 0x4a, 0x01], "vorpd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x56, 0xca], "vorpd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x57, 0x0a], "vxorpd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x57, 0x4a, 0x01], "vxorpd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x57, 0xca], "vxorpd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x58, 0x0a], "vaddpd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x58, 0x4a, 0x01], "vaddpd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x58, 0xca], "vaddpd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x59, 0x0a], "vmulpd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x59, 0x4a, 0x01], "vmulpd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x59, 0xca], "vmulpd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x5a, 0xca], "vcvtpd2ps xmm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x5c, 0x0a], "vsubpd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x5c, 0x4a, 0x01], "vsubpd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x5c, 0xca], "vsubpd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x5d, 0x0a], "vminpd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x5d, 0x4a, 0x01], "vminpd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x5d, 0xca], "vminpd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x5e, 0x0a], "vdivpd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x5e, 0x4a, 0x01], "vdivpd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x5e, 0xca], "vdivpd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x5f, 0x0a], "vmaxpd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x5f, 0x4a, 0x01], "vmaxpd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x5f, 0xca], "vmaxpd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x60, 0x0a], "vpunpcklbw ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x60, 0x4a, 0x01], "vpunpcklbw ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x60, 0xca], "vpunpcklbw ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x61, 0x0a], "vpunpcklwd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x61, 0x4a, 0x01], "vpunpcklwd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x61, 0xca], "vpunpcklwd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x63, 0x0a], "vpacksswb ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x63, 0x4a, 0x01], "vpacksswb ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x63, 0xca], "vpacksswb ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x64, 0x0a], "vpcmpgtb k1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x64, 0x4a, 0x01], "vpcmpgtb k1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x64, 0xca], "vpcmpgtb k1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x65, 0x0a], "vpcmpgtw k1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x65, 0x4a, 0x01], "vpcmpgtw k1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x65, 0xca], "vpcmpgtw k1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x67, 0x0a], "vpackuswb ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x67, 0x4a, 0x01], "vpackuswb ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x67, 0xca], "vpackuswb ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x68, 0x0a], "vpunpckhbw ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x68, 0x4a, 0x01], "vpunpckhbw ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x68, 0xca], "vpunpckhbw ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x69, 0x0a], "vpunpckhwd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x69, 0x4a, 0x01], "vpunpckhwd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x69, 0xca], "vpunpckhwd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x6c, 0x0a], "vpunpcklqdq ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x6c, 0x4a, 0x01], "vpunpcklqdq ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x6c, 0xca], "vpunpcklqdq ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x6d, 0x0a], "vpunpckhqdq ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x6d, 0x4a, 0x01], "vpunpckhqdq ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x6d, 0xca], "vpunpckhqdq ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x6f, 0x0a], "vmovdqa64 ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x6f, 0x4a, 0x01], "vmovdqa64 ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x6f, 0xca], "vmovdqa64 ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x72, 0x0a, 0xcc], "vprolq ymm0{k5}, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x72, 0x4a, 0x01, 0xcc], "vprolq ymm0{k5}, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x72, 0xca, 0xcc], "vprolq ymm0{k5}, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x74, 0x0a], "vpcmpeqb k1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x74, 0x4a, 0x01], "vpcmpeqb k1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x74, 0xca], "vpcmpeqb k1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x75, 0x0a], "vpcmpeqw k1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x75, 0x4a, 0x01], "vpcmpeqw k1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x75, 0xca], "vpcmpeqw k1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x78, 0x0a], "vcvttpd2uqq ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x78, 0x4a, 0x01], "vcvttpd2uqq ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x78, 0xca], "vcvttpd2uqq ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x79, 0x0a], "vcvtpd2uqq ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x79, 0x4a, 0x01], "vcvtpd2uqq ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x79, 0xca], "vcvtpd2uqq ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x7a, 0x0a], "vcvttpd2qq ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x7a, 0x4a, 0x01], "vcvttpd2qq ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x7a, 0xca], "vcvttpd2qq ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x7b, 0x0a], "vcvtpd2qq ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x7b, 0x4a, 0x01], "vcvtpd2qq ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x7b, 0xca], "vcvtpd2qq ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x7f, 0x0a], "vmovdqa64 ymmword [bp + si]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x7f, 0x4a, 0x01], "vmovdqa64 ymmword [bp + si + 0x20]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0x7f, 0xca], "vmovdqa64 ymm2{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, ymm0, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1{k5}, ymm0, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xc2, 0xca, 0xcc], "vcmppd k1{k5}, ymm0, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xc6, 0x0a, 0xcc], "vshufpd ymm1{k5}, ymm0, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd ymm1{k5}, ymm0, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xc6, 0xca, 0xcc], "vshufpd ymm1{k5}, ymm0, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd1, 0x0a], "vpsrlw ymm1{k5}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd1, 0x4a, 0x01], "vpsrlw ymm1{k5}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd1, 0xca], "vpsrlw ymm1{k5}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd3, 0x0a], "vpsrlq ymm1{k5}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd3, 0x4a, 0x01], "vpsrlq ymm1{k5}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd3, 0xca], "vpsrlq ymm1{k5}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd4, 0x0a], "vpaddq ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd4, 0x4a, 0x01], "vpaddq ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd4, 0xca], "vpaddq ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd5, 0x0a], "vpmullw ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd5, 0x4a, 0x01], "vpmullw ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd5, 0xca], "vpmullw ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd8, 0x0a], "vpsubusb ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd8, 0x4a, 0x01], "vpsubusb ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd8, 0xca], "vpsubusb ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd9, 0x0a], "vpsubusw ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd9, 0x4a, 0x01], "vpsubusw ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xd9, 0xca], "vpsubusw ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xda, 0x0a], "vpminub ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xda, 0x4a, 0x01], "vpminub ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xda, 0xca], "vpminub ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xdb, 0x0a], "vpandq ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xdb, 0x4a, 0x01], "vpandq ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xdb, 0xca], "vpandq ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xdc, 0x0a], "vpaddusb ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xdc, 0x4a, 0x01], "vpaddusb ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xdc, 0xca], "vpaddusb ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xdd, 0x0a], "vpaddusw ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xdd, 0x4a, 0x01], "vpaddusw ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xdd, 0xca], "vpaddusw ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xde, 0x0a], "vpmaxub ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xde, 0x4a, 0x01], "vpmaxub ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xde, 0xca], "vpmaxub ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xdf, 0x0a], "vpandnq ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xdf, 0x4a, 0x01], "vpandnq ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xdf, 0xca], "vpandnq ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe0, 0x0a], "vpavgb ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe0, 0x4a, 0x01], "vpavgb ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe0, 0xca], "vpavgb ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe1, 0x0a], "vpsraw ymm1{k5}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe1, 0x4a, 0x01], "vpsraw ymm1{k5}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe1, 0xca], "vpsraw ymm1{k5}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe2, 0x0a], "vpsraq ymm1{k5}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe2, 0x4a, 0x01], "vpsraq ymm1{k5}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe2, 0xca], "vpsraq ymm1{k5}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe3, 0x0a], "vpavgw ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe3, 0x4a, 0x01], "vpavgw ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe3, 0xca], "vpavgw ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe4, 0x0a], "vpmulhuw ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe4, 0x4a, 0x01], "vpmulhuw ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe4, 0xca], "vpmulhuw ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe5, 0x0a], "vpmulhw ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe5, 0x4a, 0x01], "vpmulhw ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe5, 0xca], "vpmulhw ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe6, 0xca], "vcvttpd2dq xmm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe8, 0x0a], "vpsubsb ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe8, 0x4a, 0x01], "vpsubsb ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe8, 0xca], "vpsubsb ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe9, 0x0a], "vpsubsw ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe9, 0x4a, 0x01], "vpsubsw ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xe9, 0xca], "vpsubsw ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xea, 0x0a], "vpminsw ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xea, 0x4a, 0x01], "vpminsw ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xea, 0xca], "vpminsw ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xeb, 0x0a], "vporq ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xeb, 0x4a, 0x01], "vporq ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xeb, 0xca], "vporq ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xec, 0x0a], "vpaddsb ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xec, 0x4a, 0x01], "vpaddsb ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xec, 0xca], "vpaddsb ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xed, 0x0a], "vpaddsw ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xed, 0x4a, 0x01], "vpaddsw ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xed, 0xca], "vpaddsw ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xee, 0x0a], "vpmaxsw ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xee, 0x4a, 0x01], "vpmaxsw ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xee, 0xca], "vpmaxsw ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xef, 0x0a], "vpxorq ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xef, 0x4a, 0x01], "vpxorq ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xef, 0xca], "vpxorq ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf1, 0x0a], "vpsllw ymm1{k5}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf1, 0x4a, 0x01], "vpsllw ymm1{k5}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf1, 0xca], "vpsllw ymm1{k5}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf3, 0x0a], "vpsllq ymm1{k5}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf3, 0x4a, 0x01], "vpsllq ymm1{k5}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf3, 0xca], "vpsllq ymm1{k5}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf4, 0x0a], "vpmuludq ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf4, 0x4a, 0x01], "vpmuludq ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf4, 0xca], "vpmuludq ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf5, 0x0a], "vpmaddwd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf5, 0x4a, 0x01], "vpmaddwd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf5, 0xca], "vpmaddwd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf8, 0x0a], "vpsubb ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf8, 0x4a, 0x01], "vpsubb ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf8, 0xca], "vpsubb ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf9, 0x0a], "vpsubw ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf9, 0x4a, 0x01], "vpsubw ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xf9, 0xca], "vpsubw ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xfb, 0x0a], "vpsubq ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xfb, 0x4a, 0x01], "vpsubq ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xfb, 0xca], "vpsubq ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xfc, 0x0a], "vpaddb ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xfc, 0x4a, 0x01], "vpaddb ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xfc, 0xca], "vpaddb ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xfd, 0x0a], "vpaddw ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xfd, 0x4a, 0x01], "vpaddw ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0x2d, 0xfd, 0xca], "vpaddw ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x14, 0x0a], "vunpcklpd ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x14, 0x4a, 0x01], "vunpcklpd ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x15, 0x0a], "vunpckhpd ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x15, 0x4a, 0x01], "vunpckhpd ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x51, 0x0a], "vsqrtpd ymm1, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x51, 0x4a, 0x01], "vsqrtpd ymm1, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x51, 0xca], "vsqrtpd zmm1{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x54, 0x0a], "vandpd ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x54, 0x4a, 0x01], "vandpd ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x55, 0x0a], "vandnpd ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x55, 0x4a, 0x01], "vandnpd ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x56, 0x0a], "vorpd ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x56, 0x4a, 0x01], "vorpd ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x57, 0x0a], "vxorpd ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x57, 0x4a, 0x01], "vxorpd ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x58, 0x0a], "vaddpd ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x58, 0x4a, 0x01], "vaddpd ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x58, 0xca], "vaddpd zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x59, 0x0a], "vmulpd ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x59, 0x4a, 0x01], "vmulpd ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x59, 0xca], "vmulpd zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x5a, 0x0a], "vcvtpd2ps xmm1, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x5a, 0xca], "vcvtpd2ps ymm1{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x5c, 0x0a], "vsubpd ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x5c, 0x4a, 0x01], "vsubpd ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x5c, 0xca], "vsubpd zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x5d, 0x0a], "vminpd ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x5d, 0x4a, 0x01], "vminpd ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x5e, 0x0a], "vdivpd ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x5e, 0x4a, 0x01], "vdivpd ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x5e, 0xca], "vdivpd zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x5f, 0x0a], "vmaxpd ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x5f, 0x4a, 0x01], "vmaxpd ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x6c, 0x0a], "vpunpcklqdq ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x6c, 0x4a, 0x01], "vpunpcklqdq ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x6d, 0x0a], "vpunpckhqdq ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x6d, 0x4a, 0x01], "vpunpckhqdq ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x72, 0x0a, 0xcc], "vprolq ymm0, qword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x72, 0x4a, 0x01, 0xcc], "vprolq ymm0, qword [bp + si + 0x8]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x78, 0x0a], "vcvttpd2uqq ymm1, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x78, 0x4a, 0x01], "vcvttpd2uqq ymm1, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x79, 0x0a], "vcvtpd2uqq ymm1, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x79, 0x4a, 0x01], "vcvtpd2uqq ymm1, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x79, 0xca], "vcvtpd2uqq zmm1{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x7a, 0x0a], "vcvttpd2qq ymm1, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x7a, 0x4a, 0x01], "vcvttpd2qq ymm1, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x7b, 0x0a], "vcvtpd2qq ymm1, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x7b, 0x4a, 0x01], "vcvtpd2qq ymm1, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0x7b, 0xca], "vcvtpd2qq zmm1{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xc2, 0x0a, 0xcc], "vcmppd k1, ymm0, qword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1, ymm0, qword [bp + si + 0x8]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xc6, 0x0a, 0xcc], "vshufpd ymm1, ymm0, qword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd ymm1, ymm0, qword [bp + si + 0x8]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xd4, 0x0a], "vpaddq ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xd4, 0x4a, 0x01], "vpaddq ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xdb, 0x0a], "vpandq ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xdb, 0x4a, 0x01], "vpandq ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xdf, 0x0a], "vpandnq ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xdf, 0x4a, 0x01], "vpandnq ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xe6, 0x0a], "vcvttpd2dq xmm1, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xeb, 0x0a], "vporq ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xeb, 0x4a, 0x01], "vporq ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xef, 0x0a], "vpxorq ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xef, 0x4a, 0x01], "vpxorq ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xf4, 0x0a], "vpmuludq ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xf4, 0x4a, 0x01], "vpmuludq ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xfb, 0x0a], "vpsubq ymm1, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x38, 0xfb, 0x4a, 0x01], "vpsubq ymm1, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x14, 0x0a], "vunpcklpd ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x14, 0x4a, 0x01], "vunpcklpd ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x15, 0x0a], "vunpckhpd ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x15, 0x4a, 0x01], "vunpckhpd ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x51, 0x0a], "vsqrtpd ymm1{k5}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x51, 0x4a, 0x01], "vsqrtpd ymm1{k5}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x51, 0xca], "vsqrtpd zmm1{k5}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x54, 0x0a], "vandpd ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x54, 0x4a, 0x01], "vandpd ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x55, 0x0a], "vandnpd ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x55, 0x4a, 0x01], "vandnpd ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x56, 0x0a], "vorpd ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x56, 0x4a, 0x01], "vorpd ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x57, 0x0a], "vxorpd ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x57, 0x4a, 0x01], "vxorpd ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x58, 0x0a], "vaddpd ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x58, 0x4a, 0x01], "vaddpd ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x58, 0xca], "vaddpd zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x59, 0x0a], "vmulpd ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x59, 0x4a, 0x01], "vmulpd ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x59, 0xca], "vmulpd zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1{k5}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x5c, 0x0a], "vsubpd ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x5c, 0x4a, 0x01], "vsubpd ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x5c, 0xca], "vsubpd zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x5d, 0x0a], "vminpd ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x5d, 0x4a, 0x01], "vminpd ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x5e, 0x0a], "vdivpd ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x5e, 0x4a, 0x01], "vdivpd ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x5e, 0xca], "vdivpd zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x5f, 0x0a], "vmaxpd ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x5f, 0x4a, 0x01], "vmaxpd ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x6c, 0x0a], "vpunpcklqdq ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x6c, 0x4a, 0x01], "vpunpcklqdq ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x6d, 0x0a], "vpunpckhqdq ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x6d, 0x4a, 0x01], "vpunpckhqdq ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x72, 0x0a, 0xcc], "vprolq ymm0{k5}, qword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x72, 0x4a, 0x01, 0xcc], "vprolq ymm0{k5}, qword [bp + si + 0x8]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x78, 0x0a], "vcvttpd2uqq ymm1{k5}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x78, 0x4a, 0x01], "vcvttpd2uqq ymm1{k5}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x79, 0x0a], "vcvtpd2uqq ymm1{k5}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x79, 0x4a, 0x01], "vcvtpd2uqq ymm1{k5}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x7a, 0x0a], "vcvttpd2qq ymm1{k5}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x7a, 0x4a, 0x01], "vcvttpd2qq ymm1{k5}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x7b, 0x0a], "vcvtpd2qq ymm1{k5}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x7b, 0x4a, 0x01], "vcvtpd2qq ymm1{k5}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, ymm0, qword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1{k5}, ymm0, qword [bp + si + 0x8]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xc6, 0x0a, 0xcc], "vshufpd ymm1{k5}, ymm0, qword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xd4, 0x0a], "vpaddq ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xd4, 0x4a, 0x01], "vpaddq ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xdb, 0x0a], "vpandq ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xdb, 0x4a, 0x01], "vpandq ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xdf, 0x0a], "vpandnq ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xdf, 0x4a, 0x01], "vpandnq ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1{k5}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xeb, 0x0a], "vporq ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xeb, 0x4a, 0x01], "vporq ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xef, 0x0a], "vpxorq ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xef, 0x4a, 0x01], "vpxorq ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xf4, 0x0a], "vpmuludq ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xf4, 0x4a, 0x01], "vpmuludq ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xfb, 0x0a], "vpsubq ymm1{k5}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x3d, 0xfb, 0x4a, 0x01], "vpsubq ymm1{k5}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x10, 0x0a], "vmovupd zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x10, 0x4a, 0x01], "vmovupd zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x10, 0xca], "vmovupd zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x11, 0x0a], "vmovupd zmmword [bp + si], zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x11, 0x4a, 0x01], "vmovupd zmmword [bp + si + 0x40], zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x11, 0xca], "vmovupd zmm2, zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x14, 0x0a], "vunpcklpd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x14, 0x4a, 0x01], "vunpcklpd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x14, 0xca], "vunpcklpd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x15, 0x0a], "vunpckhpd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x15, 0x4a, 0x01], "vunpckhpd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x15, 0xca], "vunpckhpd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x28, 0x0a], "vmovapd zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x28, 0x4a, 0x01], "vmovapd zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x28, 0xca], "vmovapd zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x29, 0x0a], "vmovapd zmmword [bp + si], zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x29, 0x4a, 0x01], "vmovapd zmmword [bp + si + 0x40], zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x29, 0xca], "vmovapd zmm2, zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x2b, 0x0a], "vmovntpd zmmword [bp + si], zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x2b, 0x4a, 0x01], "vmovntpd zmmword [bp + si + 0x40], zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x51, 0x0a], "vsqrtpd zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x51, 0x4a, 0x01], "vsqrtpd zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x51, 0xca], "vsqrtpd zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x54, 0x0a], "vandpd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x54, 0x4a, 0x01], "vandpd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x54, 0xca], "vandpd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x55, 0x0a], "vandnpd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x55, 0x4a, 0x01], "vandnpd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x55, 0xca], "vandnpd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x56, 0x0a], "vorpd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x56, 0x4a, 0x01], "vorpd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x56, 0xca], "vorpd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x57, 0x0a], "vxorpd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x57, 0x4a, 0x01], "vxorpd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x57, 0xca], "vxorpd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x58, 0x0a], "vaddpd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x58, 0x4a, 0x01], "vaddpd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x58, 0xca], "vaddpd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x59, 0x0a], "vmulpd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x59, 0x4a, 0x01], "vmulpd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x59, 0xca], "vmulpd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x5a, 0x0a], "vcvtpd2ps ymm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x5a, 0x4a, 0x01], "vcvtpd2ps ymm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x5a, 0xca], "vcvtpd2ps ymm1, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x5c, 0x0a], "vsubpd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x5c, 0x4a, 0x01], "vsubpd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x5c, 0xca], "vsubpd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x5d, 0x0a], "vminpd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x5d, 0x4a, 0x01], "vminpd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x5d, 0xca], "vminpd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x5e, 0x0a], "vdivpd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x5e, 0x4a, 0x01], "vdivpd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x5e, 0xca], "vdivpd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x5f, 0x0a], "vmaxpd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x5f, 0x4a, 0x01], "vmaxpd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x5f, 0xca], "vmaxpd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x60, 0x0a], "vpunpcklbw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x60, 0x4a, 0x01], "vpunpcklbw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x60, 0xca], "vpunpcklbw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x61, 0x0a], "vpunpcklwd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x61, 0x4a, 0x01], "vpunpcklwd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x61, 0xca], "vpunpcklwd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x63, 0x0a], "vpacksswb zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x63, 0x4a, 0x01], "vpacksswb zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x63, 0xca], "vpacksswb zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x64, 0x0a], "vpcmpgtb k1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x64, 0x4a, 0x01], "vpcmpgtb k1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x64, 0xca], "vpcmpgtb k1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x65, 0x0a], "vpcmpgtw k1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x65, 0x4a, 0x01], "vpcmpgtw k1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x65, 0xca], "vpcmpgtw k1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x67, 0x0a], "vpackuswb zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x67, 0x4a, 0x01], "vpackuswb zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x67, 0xca], "vpackuswb zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x68, 0x0a], "vpunpckhbw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x68, 0x4a, 0x01], "vpunpckhbw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x68, 0xca], "vpunpckhbw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x69, 0x0a], "vpunpckhwd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x69, 0x4a, 0x01], "vpunpckhwd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x69, 0xca], "vpunpckhwd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x6c, 0x0a], "vpunpcklqdq zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x6c, 0x4a, 0x01], "vpunpcklqdq zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x6c, 0xca], "vpunpcklqdq zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x6d, 0x0a], "vpunpckhqdq zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x6d, 0x4a, 0x01], "vpunpckhqdq zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x6d, 0xca], "vpunpckhqdq zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x6f, 0x0a], "vmovdqa64 zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x6f, 0x4a, 0x01], "vmovdqa64 zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x6f, 0xca], "vmovdqa64 zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x72, 0x0a, 0xcc], "vprolq zmm0, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x72, 0x4a, 0x01, 0xcc], "vprolq zmm0, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x72, 0xca, 0xcc], "vprolq zmm0, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x74, 0x0a], "vpcmpeqb k1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x74, 0x4a, 0x01], "vpcmpeqb k1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x74, 0xca], "vpcmpeqb k1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x75, 0x0a], "vpcmpeqw k1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x75, 0x4a, 0x01], "vpcmpeqw k1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x75, 0xca], "vpcmpeqw k1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x78, 0x0a], "vcvttpd2uqq zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x78, 0x4a, 0x01], "vcvttpd2uqq zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x78, 0xca], "vcvttpd2uqq zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x79, 0x0a], "vcvtpd2uqq zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x79, 0x4a, 0x01], "vcvtpd2uqq zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x79, 0xca], "vcvtpd2uqq zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x7a, 0x0a], "vcvttpd2qq zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x7a, 0x4a, 0x01], "vcvttpd2qq zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x7a, 0xca], "vcvttpd2qq zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x7b, 0x0a], "vcvtpd2qq zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x7b, 0x4a, 0x01], "vcvtpd2qq zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x7b, 0xca], "vcvtpd2qq zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x7f, 0x0a], "vmovdqa64 zmmword [bp + si], zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x7f, 0x4a, 0x01], "vmovdqa64 zmmword [bp + si + 0x40], zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0x7f, 0xca], "vmovdqa64 zmm2, zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xc2, 0x0a, 0xcc], "vcmppd k1, zmm0, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1, zmm0, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xc2, 0xca, 0xcc], "vcmppd k1, zmm0, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xc6, 0x0a, 0xcc], "vshufpd zmm1, zmm0, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd zmm1, zmm0, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xc6, 0xca, 0xcc], "vshufpd zmm1, zmm0, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd1, 0x0a], "vpsrlw zmm1, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd1, 0x4a, 0x01], "vpsrlw zmm1, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd1, 0xca], "vpsrlw zmm1, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd3, 0x0a], "vpsrlq zmm1, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd3, 0x4a, 0x01], "vpsrlq zmm1, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd3, 0xca], "vpsrlq zmm1, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd4, 0x0a], "vpaddq zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd4, 0x4a, 0x01], "vpaddq zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd4, 0xca], "vpaddq zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd5, 0x0a], "vpmullw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd5, 0x4a, 0x01], "vpmullw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd5, 0xca], "vpmullw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd8, 0x0a], "vpsubusb zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd8, 0x4a, 0x01], "vpsubusb zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd8, 0xca], "vpsubusb zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd9, 0x0a], "vpsubusw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd9, 0x4a, 0x01], "vpsubusw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xd9, 0xca], "vpsubusw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xda, 0x0a], "vpminub zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xda, 0x4a, 0x01], "vpminub zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xda, 0xca], "vpminub zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xdb, 0x0a], "vpandq zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xdb, 0x4a, 0x01], "vpandq zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xdb, 0xca], "vpandq zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xdc, 0x0a], "vpaddusb zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xdc, 0x4a, 0x01], "vpaddusb zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xdc, 0xca], "vpaddusb zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xdd, 0x0a], "vpaddusw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xdd, 0x4a, 0x01], "vpaddusw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xdd, 0xca], "vpaddusw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xde, 0x0a], "vpmaxub zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xde, 0x4a, 0x01], "vpmaxub zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xde, 0xca], "vpmaxub zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xdf, 0x0a], "vpandnq zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xdf, 0x4a, 0x01], "vpandnq zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xdf, 0xca], "vpandnq zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe0, 0x0a], "vpavgb zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe0, 0x4a, 0x01], "vpavgb zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe0, 0xca], "vpavgb zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe1, 0x0a], "vpsraw zmm1, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe1, 0x4a, 0x01], "vpsraw zmm1, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe1, 0xca], "vpsraw zmm1, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe2, 0x0a], "vpsraq zmm1, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe2, 0x4a, 0x01], "vpsraq zmm1, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe2, 0xca], "vpsraq zmm1, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe3, 0x0a], "vpavgw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe3, 0x4a, 0x01], "vpavgw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe3, 0xca], "vpavgw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe4, 0x0a], "vpmulhuw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe4, 0x4a, 0x01], "vpmulhuw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe4, 0xca], "vpmulhuw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe5, 0x0a], "vpmulhw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe5, 0x4a, 0x01], "vpmulhw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe5, 0xca], "vpmulhw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe6, 0x0a], "vcvttpd2dq ymm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe6, 0x4a, 0x01], "vcvttpd2dq ymm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe6, 0xca], "vcvttpd2dq ymm1, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe8, 0x0a], "vpsubsb zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe8, 0x4a, 0x01], "vpsubsb zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe8, 0xca], "vpsubsb zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe9, 0x0a], "vpsubsw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe9, 0x4a, 0x01], "vpsubsw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xe9, 0xca], "vpsubsw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xea, 0x0a], "vpminsw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xea, 0x4a, 0x01], "vpminsw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xea, 0xca], "vpminsw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xeb, 0x0a], "vporq zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xeb, 0x4a, 0x01], "vporq zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xeb, 0xca], "vporq zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xec, 0x0a], "vpaddsb zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xec, 0x4a, 0x01], "vpaddsb zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xec, 0xca], "vpaddsb zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xed, 0x0a], "vpaddsw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xed, 0x4a, 0x01], "vpaddsw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xed, 0xca], "vpaddsw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xee, 0x0a], "vpmaxsw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xee, 0x4a, 0x01], "vpmaxsw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xee, 0xca], "vpmaxsw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xef, 0x0a], "vpxorq zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xef, 0x4a, 0x01], "vpxorq zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xef, 0xca], "vpxorq zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf1, 0x0a], "vpsllw zmm1, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf1, 0x4a, 0x01], "vpsllw zmm1, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf1, 0xca], "vpsllw zmm1, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf3, 0x0a], "vpsllq zmm1, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf3, 0x4a, 0x01], "vpsllq zmm1, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf3, 0xca], "vpsllq zmm1, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf4, 0x0a], "vpmuludq zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf4, 0x4a, 0x01], "vpmuludq zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf4, 0xca], "vpmuludq zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf5, 0x0a], "vpmaddwd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf5, 0x4a, 0x01], "vpmaddwd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf5, 0xca], "vpmaddwd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf6, 0x0a], "vpsadbw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf6, 0x4a, 0x01], "vpsadbw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf6, 0xca], "vpsadbw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf8, 0x0a], "vpsubb zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf8, 0x4a, 0x01], "vpsubb zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf8, 0xca], "vpsubb zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf9, 0x0a], "vpsubw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf9, 0x4a, 0x01], "vpsubw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xf9, 0xca], "vpsubw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xfb, 0x0a], "vpsubq zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xfb, 0x4a, 0x01], "vpsubq zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xfb, 0xca], "vpsubq zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xfc, 0x0a], "vpaddb zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xfc, 0x4a, 0x01], "vpaddb zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xfc, 0xca], "vpaddb zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xfd, 0x0a], "vpaddw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xfd, 0x4a, 0x01], "vpaddw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x48, 0xfd, 0xca], "vpaddw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x10, 0x0a], "vmovupd zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x10, 0x4a, 0x01], "vmovupd zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x10, 0xca], "vmovupd zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x11, 0x0a], "vmovupd zmmword [bp + si]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x11, 0x4a, 0x01], "vmovupd zmmword [bp + si + 0x40]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x11, 0xca], "vmovupd zmm2{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x14, 0x0a], "vunpcklpd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x14, 0x4a, 0x01], "vunpcklpd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x14, 0xca], "vunpcklpd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x15, 0x0a], "vunpckhpd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x15, 0x4a, 0x01], "vunpckhpd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x15, 0xca], "vunpckhpd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x28, 0x0a], "vmovapd zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x28, 0x4a, 0x01], "vmovapd zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x28, 0xca], "vmovapd zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x29, 0x0a], "vmovapd zmmword [bp + si]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x29, 0x4a, 0x01], "vmovapd zmmword [bp + si + 0x40]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x29, 0xca], "vmovapd zmm2{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x51, 0x0a], "vsqrtpd zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x51, 0x4a, 0x01], "vsqrtpd zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x51, 0xca], "vsqrtpd zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x54, 0x0a], "vandpd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x54, 0x4a, 0x01], "vandpd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x54, 0xca], "vandpd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x55, 0x0a], "vandnpd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x55, 0x4a, 0x01], "vandnpd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x55, 0xca], "vandnpd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x56, 0x0a], "vorpd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x56, 0x4a, 0x01], "vorpd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x56, 0xca], "vorpd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x57, 0x0a], "vxorpd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x57, 0x4a, 0x01], "vxorpd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x57, 0xca], "vxorpd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x58, 0x0a], "vaddpd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x58, 0x4a, 0x01], "vaddpd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x58, 0xca], "vaddpd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x59, 0x0a], "vmulpd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x59, 0x4a, 0x01], "vmulpd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x59, 0xca], "vmulpd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x5a, 0x0a], "vcvtpd2ps ymm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x5a, 0x4a, 0x01], "vcvtpd2ps ymm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x5c, 0x0a], "vsubpd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x5c, 0x4a, 0x01], "vsubpd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x5c, 0xca], "vsubpd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x5d, 0x0a], "vminpd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x5d, 0x4a, 0x01], "vminpd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x5d, 0xca], "vminpd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x5e, 0x0a], "vdivpd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x5e, 0x4a, 0x01], "vdivpd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x5e, 0xca], "vdivpd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x5f, 0x0a], "vmaxpd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x5f, 0x4a, 0x01], "vmaxpd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x5f, 0xca], "vmaxpd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x60, 0x0a], "vpunpcklbw zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x60, 0x4a, 0x01], "vpunpcklbw zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x60, 0xca], "vpunpcklbw zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x61, 0x0a], "vpunpcklwd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x61, 0x4a, 0x01], "vpunpcklwd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x61, 0xca], "vpunpcklwd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x63, 0x0a], "vpacksswb zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x63, 0x4a, 0x01], "vpacksswb zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x63, 0xca], "vpacksswb zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x64, 0x0a], "vpcmpgtb k1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x64, 0x4a, 0x01], "vpcmpgtb k1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x64, 0xca], "vpcmpgtb k1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x65, 0x0a], "vpcmpgtw k1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x65, 0x4a, 0x01], "vpcmpgtw k1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x65, 0xca], "vpcmpgtw k1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x67, 0x0a], "vpackuswb zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x67, 0x4a, 0x01], "vpackuswb zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x67, 0xca], "vpackuswb zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x68, 0x0a], "vpunpckhbw zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x68, 0x4a, 0x01], "vpunpckhbw zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x68, 0xca], "vpunpckhbw zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x69, 0x0a], "vpunpckhwd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x69, 0x4a, 0x01], "vpunpckhwd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x69, 0xca], "vpunpckhwd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x6c, 0x0a], "vpunpcklqdq zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x6c, 0x4a, 0x01], "vpunpcklqdq zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x6c, 0xca], "vpunpcklqdq zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x6d, 0x0a], "vpunpckhqdq zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x6d, 0x4a, 0x01], "vpunpckhqdq zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x6d, 0xca], "vpunpckhqdq zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x6f, 0x0a], "vmovdqa64 zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x6f, 0x4a, 0x01], "vmovdqa64 zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x6f, 0xca], "vmovdqa64 zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x72, 0x0a, 0xcc], "vprolq zmm0{k5}, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x72, 0x4a, 0x01, 0xcc], "vprolq zmm0{k5}, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x72, 0xca, 0xcc], "vprolq zmm0{k5}, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x74, 0x0a], "vpcmpeqb k1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x74, 0x4a, 0x01], "vpcmpeqb k1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x74, 0xca], "vpcmpeqb k1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x75, 0x0a], "vpcmpeqw k1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x75, 0x4a, 0x01], "vpcmpeqw k1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x75, 0xca], "vpcmpeqw k1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x78, 0x0a], "vcvttpd2uqq zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x78, 0x4a, 0x01], "vcvttpd2uqq zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x78, 0xca], "vcvttpd2uqq zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x79, 0x0a], "vcvtpd2uqq zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x79, 0x4a, 0x01], "vcvtpd2uqq zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x7a, 0x0a], "vcvttpd2qq zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x7a, 0x4a, 0x01], "vcvttpd2qq zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x7a, 0xca], "vcvttpd2qq zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x7b, 0x0a], "vcvtpd2qq zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x7b, 0x4a, 0x01], "vcvtpd2qq zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x7f, 0x0a], "vmovdqa64 zmmword [bp + si]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x7f, 0x4a, 0x01], "vmovdqa64 zmmword [bp + si + 0x40]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0x7f, 0xca], "vmovdqa64 zmm2{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, zmm0, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1{k5}, zmm0, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xc2, 0xca, 0xcc], "vcmppd k1{k5}, zmm0, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xc6, 0x0a, 0xcc], "vshufpd zmm1{k5}, zmm0, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd zmm1{k5}, zmm0, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xc6, 0xca, 0xcc], "vshufpd zmm1{k5}, zmm0, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd1, 0x0a], "vpsrlw zmm1{k5}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd1, 0x4a, 0x01], "vpsrlw zmm1{k5}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd1, 0xca], "vpsrlw zmm1{k5}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd3, 0x0a], "vpsrlq zmm1{k5}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd3, 0x4a, 0x01], "vpsrlq zmm1{k5}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd3, 0xca], "vpsrlq zmm1{k5}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd4, 0x0a], "vpaddq zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd4, 0x4a, 0x01], "vpaddq zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd4, 0xca], "vpaddq zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd5, 0x0a], "vpmullw zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd5, 0x4a, 0x01], "vpmullw zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd5, 0xca], "vpmullw zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd8, 0x0a], "vpsubusb zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd8, 0x4a, 0x01], "vpsubusb zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd8, 0xca], "vpsubusb zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd9, 0x0a], "vpsubusw zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd9, 0x4a, 0x01], "vpsubusw zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xd9, 0xca], "vpsubusw zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xda, 0x0a], "vpminub zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xda, 0x4a, 0x01], "vpminub zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xda, 0xca], "vpminub zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xdb, 0x0a], "vpandq zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xdb, 0x4a, 0x01], "vpandq zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xdb, 0xca], "vpandq zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xdc, 0x0a], "vpaddusb zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xdc, 0x4a, 0x01], "vpaddusb zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xdc, 0xca], "vpaddusb zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xdd, 0x0a], "vpaddusw zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xdd, 0x4a, 0x01], "vpaddusw zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xdd, 0xca], "vpaddusw zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xde, 0x0a], "vpmaxub zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xde, 0x4a, 0x01], "vpmaxub zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xde, 0xca], "vpmaxub zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xdf, 0x0a], "vpandnq zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xdf, 0x4a, 0x01], "vpandnq zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xdf, 0xca], "vpandnq zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe0, 0x0a], "vpavgb zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe0, 0x4a, 0x01], "vpavgb zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe0, 0xca], "vpavgb zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe1, 0x0a], "vpsraw zmm1{k5}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe1, 0x4a, 0x01], "vpsraw zmm1{k5}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe1, 0xca], "vpsraw zmm1{k5}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe2, 0x0a], "vpsraq zmm1{k5}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe2, 0x4a, 0x01], "vpsraq zmm1{k5}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe2, 0xca], "vpsraq zmm1{k5}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe3, 0x0a], "vpavgw zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe3, 0x4a, 0x01], "vpavgw zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe3, 0xca], "vpavgw zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe4, 0x0a], "vpmulhuw zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe4, 0x4a, 0x01], "vpmulhuw zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe4, 0xca], "vpmulhuw zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe5, 0x0a], "vpmulhw zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe5, 0x4a, 0x01], "vpmulhw zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe5, 0xca], "vpmulhw zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe6, 0x0a], "vcvttpd2dq ymm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe6, 0x4a, 0x01], "vcvttpd2dq ymm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe6, 0xca], "vcvttpd2dq ymm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe8, 0x0a], "vpsubsb zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe8, 0x4a, 0x01], "vpsubsb zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe8, 0xca], "vpsubsb zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe9, 0x0a], "vpsubsw zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe9, 0x4a, 0x01], "vpsubsw zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xe9, 0xca], "vpsubsw zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xea, 0x0a], "vpminsw zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xea, 0x4a, 0x01], "vpminsw zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xea, 0xca], "vpminsw zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xeb, 0x0a], "vporq zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xeb, 0x4a, 0x01], "vporq zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xeb, 0xca], "vporq zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xec, 0x0a], "vpaddsb zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xec, 0x4a, 0x01], "vpaddsb zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xec, 0xca], "vpaddsb zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xed, 0x0a], "vpaddsw zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xed, 0x4a, 0x01], "vpaddsw zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xed, 0xca], "vpaddsw zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xee, 0x0a], "vpmaxsw zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xee, 0x4a, 0x01], "vpmaxsw zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xee, 0xca], "vpmaxsw zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xef, 0x0a], "vpxorq zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xef, 0x4a, 0x01], "vpxorq zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xef, 0xca], "vpxorq zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf1, 0x0a], "vpsllw zmm1{k5}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf1, 0x4a, 0x01], "vpsllw zmm1{k5}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf1, 0xca], "vpsllw zmm1{k5}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf3, 0x0a], "vpsllq zmm1{k5}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf3, 0x4a, 0x01], "vpsllq zmm1{k5}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf3, 0xca], "vpsllq zmm1{k5}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf4, 0x0a], "vpmuludq zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf4, 0x4a, 0x01], "vpmuludq zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf4, 0xca], "vpmuludq zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf5, 0x0a], "vpmaddwd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf5, 0x4a, 0x01], "vpmaddwd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf5, 0xca], "vpmaddwd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf8, 0x0a], "vpsubb zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf8, 0x4a, 0x01], "vpsubb zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf8, 0xca], "vpsubb zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf9, 0x0a], "vpsubw zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf9, 0x4a, 0x01], "vpsubw zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xf9, 0xca], "vpsubw zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xfb, 0x0a], "vpsubq zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xfb, 0x4a, 0x01], "vpsubq zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xfb, 0xca], "vpsubq zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xfc, 0x0a], "vpaddb zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xfc, 0x4a, 0x01], "vpaddb zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xfc, 0xca], "vpaddb zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xfd, 0x0a], "vpaddw zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xfd, 0x4a, 0x01], "vpaddw zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0x4d, 0xfd, 0xca], "vpaddw zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x14, 0x0a], "vunpcklpd zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x14, 0x4a, 0x01], "vunpcklpd zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x15, 0x0a], "vunpckhpd zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x15, 0x4a, 0x01], "vunpckhpd zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x51, 0x0a], "vsqrtpd zmm1, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x51, 0x4a, 0x01], "vsqrtpd zmm1, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x51, 0xca], "vsqrtpd zmm1{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x54, 0x0a], "vandpd zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x54, 0x4a, 0x01], "vandpd zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x55, 0x0a], "vandnpd zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x55, 0x4a, 0x01], "vandnpd zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x56, 0x0a], "vorpd zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x56, 0x4a, 0x01], "vorpd zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x57, 0x0a], "vxorpd zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x57, 0x4a, 0x01], "vxorpd zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x58, 0x0a], "vaddpd zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x58, 0x4a, 0x01], "vaddpd zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x58, 0xca], "vaddpd zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x59, 0x0a], "vmulpd zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x59, 0x4a, 0x01], "vmulpd zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x59, 0xca], "vmulpd zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x5a, 0x0a], "vcvtpd2ps ymm1, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x5a, 0x4a, 0x01], "vcvtpd2ps ymm1, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x5a, 0xca], "vcvtpd2ps ymm1{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x5c, 0x0a], "vsubpd zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x5c, 0x4a, 0x01], "vsubpd zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x5c, 0xca], "vsubpd zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x5d, 0x0a], "vminpd zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x5d, 0x4a, 0x01], "vminpd zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x5e, 0x0a], "vdivpd zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x5e, 0x4a, 0x01], "vdivpd zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x5e, 0xca], "vdivpd zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x5f, 0x0a], "vmaxpd zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x5f, 0x4a, 0x01], "vmaxpd zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x6c, 0x0a], "vpunpcklqdq zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x6c, 0x4a, 0x01], "vpunpcklqdq zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x6d, 0x0a], "vpunpckhqdq zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x6d, 0x4a, 0x01], "vpunpckhqdq zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x72, 0x0a, 0xcc], "vprolq zmm0, qword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x72, 0x4a, 0x01, 0xcc], "vprolq zmm0, qword [bp + si + 0x8]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x78, 0x0a], "vcvttpd2uqq zmm1, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x78, 0x4a, 0x01], "vcvttpd2uqq zmm1, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x79, 0x0a], "vcvtpd2uqq zmm1, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x79, 0x4a, 0x01], "vcvtpd2uqq zmm1, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x79, 0xca], "vcvtpd2uqq zmm1{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x7a, 0x0a], "vcvttpd2qq zmm1, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x7a, 0x4a, 0x01], "vcvttpd2qq zmm1, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x7b, 0x0a], "vcvtpd2qq zmm1, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x7b, 0x4a, 0x01], "vcvtpd2qq zmm1, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0x7b, 0xca], "vcvtpd2qq zmm1{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xc2, 0x0a, 0xcc], "vcmppd k1, zmm0, qword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1, zmm0, qword [bp + si + 0x8]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xc6, 0x0a, 0xcc], "vshufpd zmm1, zmm0, qword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd zmm1, zmm0, qword [bp + si + 0x8]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xd4, 0x0a], "vpaddq zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xd4, 0x4a, 0x01], "vpaddq zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xdb, 0x0a], "vpandq zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xdb, 0x4a, 0x01], "vpandq zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xdf, 0x0a], "vpandnq zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xdf, 0x4a, 0x01], "vpandnq zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xe6, 0x0a], "vcvttpd2dq ymm1, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xe6, 0x4a, 0x01], "vcvttpd2dq ymm1, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xeb, 0x0a], "vporq zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xeb, 0x4a, 0x01], "vporq zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xef, 0x0a], "vpxorq zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xef, 0x4a, 0x01], "vpxorq zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xf4, 0x0a], "vpmuludq zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xf4, 0x4a, 0x01], "vpmuludq zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xfb, 0x0a], "vpsubq zmm1, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x58, 0xfb, 0x4a, 0x01], "vpsubq zmm1, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x14, 0x0a], "vunpcklpd zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x14, 0x4a, 0x01], "vunpcklpd zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x15, 0x0a], "vunpckhpd zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x15, 0x4a, 0x01], "vunpckhpd zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x51, 0x0a], "vsqrtpd zmm1{k5}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x51, 0x4a, 0x01], "vsqrtpd zmm1{k5}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x51, 0xca], "vsqrtpd zmm1{k5}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x54, 0x0a], "vandpd zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x54, 0x4a, 0x01], "vandpd zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x55, 0x0a], "vandnpd zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x55, 0x4a, 0x01], "vandnpd zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x56, 0x0a], "vorpd zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x56, 0x4a, 0x01], "vorpd zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x57, 0x0a], "vxorpd zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x57, 0x4a, 0x01], "vxorpd zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x58, 0x0a], "vaddpd zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x58, 0x4a, 0x01], "vaddpd zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x58, 0xca], "vaddpd zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x59, 0x0a], "vmulpd zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x59, 0x4a, 0x01], "vmulpd zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x59, 0xca], "vmulpd zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x5a, 0x0a], "vcvtpd2ps ymm1{k5}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x5a, 0x4a, 0x01], "vcvtpd2ps ymm1{k5}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x5c, 0x0a], "vsubpd zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x5c, 0x4a, 0x01], "vsubpd zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x5c, 0xca], "vsubpd zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x5d, 0x0a], "vminpd zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x5d, 0x4a, 0x01], "vminpd zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x5e, 0x0a], "vdivpd zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x5e, 0x4a, 0x01], "vdivpd zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x5e, 0xca], "vdivpd zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x5f, 0x0a], "vmaxpd zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x5f, 0x4a, 0x01], "vmaxpd zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x6c, 0x0a], "vpunpcklqdq zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x6c, 0x4a, 0x01], "vpunpcklqdq zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x6d, 0x0a], "vpunpckhqdq zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x6d, 0x4a, 0x01], "vpunpckhqdq zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x72, 0x0a, 0xcc], "vprolq zmm0{k5}, qword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x72, 0x4a, 0x01, 0xcc], "vprolq zmm0{k5}, qword [bp + si + 0x8]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x78, 0x0a], "vcvttpd2uqq zmm1{k5}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x78, 0x4a, 0x01], "vcvttpd2uqq zmm1{k5}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x79, 0x0a], "vcvtpd2uqq zmm1{k5}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x79, 0x4a, 0x01], "vcvtpd2uqq zmm1{k5}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x7a, 0x0a], "vcvttpd2qq zmm1{k5}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x7a, 0x4a, 0x01], "vcvttpd2qq zmm1{k5}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x7b, 0x0a], "vcvtpd2qq zmm1{k5}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x7b, 0x4a, 0x01], "vcvtpd2qq zmm1{k5}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, zmm0, qword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xc2, 0x4a, 0x01, 0xcc], "vcmppd k1{k5}, zmm0, qword [bp + si + 0x8]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xc6, 0x0a, 0xcc], "vshufpd zmm1{k5}, zmm0, qword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xd4, 0x0a], "vpaddq zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xd4, 0x4a, 0x01], "vpaddq zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xdb, 0x0a], "vpandq zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xdb, 0x4a, 0x01], "vpandq zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xdf, 0x0a], "vpandnq zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xdf, 0x4a, 0x01], "vpandnq zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xe6, 0x0a], "vcvttpd2dq ymm1{k5}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xe6, 0x4a, 0x01], "vcvttpd2dq ymm1{k5}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xeb, 0x0a], "vporq zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xeb, 0x4a, 0x01], "vporq zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xef, 0x0a], "vpxorq zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xef, 0x4a, 0x01], "vpxorq zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xf4, 0x0a], "vpmuludq zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xf4, 0x4a, 0x01], "vpmuludq zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xfb, 0x0a], "vpsubq zmm1{k5}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x5d, 0xfb, 0x4a, 0x01], "vpsubq zmm1{k5}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0x78, 0x2e, 0xca], "vucomisd xmm1{sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x78, 0x2f, 0xca], "vcomisd xmm1{sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x78, 0x51, 0xca], "vsqrtpd zmm1{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x78, 0x58, 0xca], "vaddpd zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x78, 0x59, 0xca], "vmulpd zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x78, 0x5a, 0xca], "vcvtpd2ps ymm1{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x78, 0x5c, 0xca], "vsubpd zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x78, 0x5d, 0xca], "vminpd zmm1{sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x78, 0x5e, 0xca], "vdivpd zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x78, 0x5f, 0xca], "vmaxpd zmm1{sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x78, 0x78, 0xca], "vcvttpd2uqq zmm1{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x78, 0x79, 0xca], "vcvtpd2uqq zmm1{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x78, 0x7a, 0xca], "vcvttpd2qq zmm1{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x78, 0x7b, 0xca], "vcvtpd2qq zmm1{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x78, 0xc2, 0xca, 0xcc], "vcmppd k1{sae}, zmm0, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x78, 0xe6, 0xca], "vcvttpd2dq ymm1{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x7d, 0x51, 0xca], "vsqrtpd zmm1{k5}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x7d, 0x58, 0xca], "vaddpd zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x7d, 0x59, 0xca], "vmulpd zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x7d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x7d, 0x5c, 0xca], "vsubpd zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x7d, 0x5d, 0xca], "vminpd zmm1{k5}{sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x7d, 0x5e, 0xca], "vdivpd zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x7d, 0x5f, 0xca], "vmaxpd zmm1{k5}{sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x7d, 0x78, 0xca], "vcvttpd2uqq zmm1{k5}{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x7d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x7d, 0x7a, 0xca], "vcvttpd2qq zmm1{k5}{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x7d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x7d, 0xc2, 0xca, 0xcc], "vcmppd k1{k5}{sae}, zmm0, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x7d, 0xe6, 0xca], "vcvttpd2dq ymm1{k5}{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x10, 0x0a], "vmovupd xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x10, 0x4a, 0x01], "vmovupd xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x10, 0xca], "vmovupd xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x11, 0xca], "vmovupd xmm2{k5}{z}, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x14, 0x0a], "vunpcklpd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x14, 0x4a, 0x01], "vunpcklpd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x14, 0xca], "vunpcklpd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x15, 0x0a], "vunpckhpd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x15, 0x4a, 0x01], "vunpckhpd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x15, 0xca], "vunpckhpd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x28, 0x0a], "vmovapd xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x28, 0x4a, 0x01], "vmovapd xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x28, 0xca], "vmovapd xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x29, 0xca], "vmovapd xmm2{k5}{z}, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x51, 0x0a], "vsqrtpd xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x51, 0x4a, 0x01], "vsqrtpd xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x51, 0xca], "vsqrtpd xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x54, 0x0a], "vandpd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x54, 0x4a, 0x01], "vandpd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x54, 0xca], "vandpd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x55, 0x0a], "vandnpd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x55, 0x4a, 0x01], "vandnpd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x55, 0xca], "vandnpd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x56, 0x0a], "vorpd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x56, 0x4a, 0x01], "vorpd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x56, 0xca], "vorpd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x57, 0x0a], "vxorpd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x57, 0x4a, 0x01], "vxorpd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x57, 0xca], "vxorpd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x58, 0x0a], "vaddpd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x58, 0x4a, 0x01], "vaddpd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x58, 0xca], "vaddpd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x59, 0x0a], "vmulpd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x59, 0x4a, 0x01], "vmulpd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x59, 0xca], "vmulpd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x5a, 0xca], "vcvtpd2ps xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x5c, 0x0a], "vsubpd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x5c, 0x4a, 0x01], "vsubpd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x5c, 0xca], "vsubpd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x5d, 0x0a], "vminpd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x5d, 0x4a, 0x01], "vminpd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x5d, 0xca], "vminpd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x5e, 0x0a], "vdivpd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x5e, 0x4a, 0x01], "vdivpd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x5e, 0xca], "vdivpd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x5f, 0x0a], "vmaxpd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x5f, 0x4a, 0x01], "vmaxpd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x5f, 0xca], "vmaxpd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x60, 0x0a], "vpunpcklbw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x60, 0x4a, 0x01], "vpunpcklbw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x60, 0xca], "vpunpcklbw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x61, 0x0a], "vpunpcklwd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x61, 0x4a, 0x01], "vpunpcklwd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x61, 0xca], "vpunpcklwd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x63, 0x0a], "vpacksswb xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x63, 0x4a, 0x01], "vpacksswb xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x63, 0xca], "vpacksswb xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x67, 0x0a], "vpackuswb xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x67, 0x4a, 0x01], "vpackuswb xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x67, 0xca], "vpackuswb xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x68, 0x0a], "vpunpckhbw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x68, 0x4a, 0x01], "vpunpckhbw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x68, 0xca], "vpunpckhbw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x69, 0x0a], "vpunpckhwd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x69, 0x4a, 0x01], "vpunpckhwd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x69, 0xca], "vpunpckhwd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x6c, 0x0a], "vpunpcklqdq xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x6c, 0x4a, 0x01], "vpunpcklqdq xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x6c, 0xca], "vpunpcklqdq xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x6d, 0x0a], "vpunpckhqdq xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x6d, 0x4a, 0x01], "vpunpckhqdq xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x6d, 0xca], "vpunpckhqdq xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x6f, 0x0a], "vmovdqa64 xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x6f, 0x4a, 0x01], "vmovdqa64 xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x6f, 0xca], "vmovdqa64 xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x72, 0x0a, 0xcc], "vprolq xmm0{k5}{z}, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x72, 0x4a, 0x01, 0xcc], "vprolq xmm0{k5}{z}, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x72, 0xca, 0xcc], "vprolq xmm0{k5}{z}, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x78, 0x0a], "vcvttpd2uqq xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x78, 0x4a, 0x01], "vcvttpd2uqq xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x78, 0xca], "vcvttpd2uqq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x79, 0x0a], "vcvtpd2uqq xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x79, 0x4a, 0x01], "vcvtpd2uqq xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x79, 0xca], "vcvtpd2uqq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x7a, 0x0a], "vcvttpd2qq xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x7a, 0x4a, 0x01], "vcvttpd2qq xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x7a, 0xca], "vcvttpd2qq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x7b, 0x0a], "vcvtpd2qq xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x7b, 0x4a, 0x01], "vcvtpd2qq xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x7b, 0xca], "vcvtpd2qq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0x7f, 0xca], "vmovdqa64 xmm2{k5}{z}, xmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xc6, 0x0a, 0xcc], "vshufpd xmm1{k5}{z}, xmm0, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xc6, 0xca, 0xcc], "vshufpd xmm1{k5}{z}, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd1, 0x0a], "vpsrlw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd1, 0x4a, 0x01], "vpsrlw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd1, 0xca], "vpsrlw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd3, 0x0a], "vpsrlq xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd3, 0x4a, 0x01], "vpsrlq xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd3, 0xca], "vpsrlq xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd4, 0x0a], "vpaddq xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd4, 0x4a, 0x01], "vpaddq xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd4, 0xca], "vpaddq xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd5, 0x0a], "vpmullw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd5, 0x4a, 0x01], "vpmullw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd5, 0xca], "vpmullw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd8, 0x0a], "vpsubusb xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd8, 0x4a, 0x01], "vpsubusb xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd8, 0xca], "vpsubusb xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd9, 0x0a], "vpsubusw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd9, 0x4a, 0x01], "vpsubusw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xd9, 0xca], "vpsubusw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xda, 0x0a], "vpminub xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xda, 0x4a, 0x01], "vpminub xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xda, 0xca], "vpminub xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xdb, 0x0a], "vpandq xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xdb, 0x4a, 0x01], "vpandq xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xdb, 0xca], "vpandq xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xdc, 0x0a], "vpaddusb xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xdc, 0x4a, 0x01], "vpaddusb xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xdc, 0xca], "vpaddusb xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xdd, 0x0a], "vpaddusw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xdd, 0x4a, 0x01], "vpaddusw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xdd, 0xca], "vpaddusw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xde, 0x0a], "vpmaxub xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xde, 0x4a, 0x01], "vpmaxub xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xde, 0xca], "vpmaxub xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xdf, 0x0a], "vpandnq xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xdf, 0x4a, 0x01], "vpandnq xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xdf, 0xca], "vpandnq xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe0, 0x0a], "vpavgb xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe0, 0x4a, 0x01], "vpavgb xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe0, 0xca], "vpavgb xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe1, 0x0a], "vpsraw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe1, 0x4a, 0x01], "vpsraw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe1, 0xca], "vpsraw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe2, 0x0a], "vpsraq xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe2, 0x4a, 0x01], "vpsraq xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe2, 0xca], "vpsraq xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe3, 0x0a], "vpavgw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe3, 0x4a, 0x01], "vpavgw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe3, 0xca], "vpavgw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe4, 0x0a], "vpmulhuw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe4, 0x4a, 0x01], "vpmulhuw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe4, 0xca], "vpmulhuw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe5, 0x0a], "vpmulhw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe5, 0x4a, 0x01], "vpmulhw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe5, 0xca], "vpmulhw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe6, 0xca], "vcvttpd2dq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe8, 0x0a], "vpsubsb xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe8, 0x4a, 0x01], "vpsubsb xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe8, 0xca], "vpsubsb xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe9, 0x0a], "vpsubsw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe9, 0x4a, 0x01], "vpsubsw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xe9, 0xca], "vpsubsw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xea, 0x0a], "vpminsw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xea, 0x4a, 0x01], "vpminsw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xea, 0xca], "vpminsw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xeb, 0x0a], "vporq xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xeb, 0x4a, 0x01], "vporq xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xeb, 0xca], "vporq xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xec, 0x0a], "vpaddsb xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xec, 0x4a, 0x01], "vpaddsb xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xec, 0xca], "vpaddsb xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xed, 0x0a], "vpaddsw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xed, 0x4a, 0x01], "vpaddsw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xed, 0xca], "vpaddsw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xee, 0x0a], "vpmaxsw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xee, 0x4a, 0x01], "vpmaxsw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xee, 0xca], "vpmaxsw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xef, 0x0a], "vpxorq xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xef, 0x4a, 0x01], "vpxorq xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xef, 0xca], "vpxorq xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf1, 0x0a], "vpsllw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf1, 0x4a, 0x01], "vpsllw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf1, 0xca], "vpsllw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf3, 0x0a], "vpsllq xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf3, 0x4a, 0x01], "vpsllq xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf3, 0xca], "vpsllq xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf4, 0x0a], "vpmuludq xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf4, 0x4a, 0x01], "vpmuludq xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf4, 0xca], "vpmuludq xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf5, 0x0a], "vpmaddwd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf5, 0x4a, 0x01], "vpmaddwd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf5, 0xca], "vpmaddwd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf8, 0x0a], "vpsubb xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf8, 0x4a, 0x01], "vpsubb xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf8, 0xca], "vpsubb xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf9, 0x0a], "vpsubw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf9, 0x4a, 0x01], "vpsubw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xf9, 0xca], "vpsubw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xfb, 0x0a], "vpsubq xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xfb, 0x4a, 0x01], "vpsubq xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xfb, 0xca], "vpsubq xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xfc, 0x0a], "vpaddb xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xfc, 0x4a, 0x01], "vpaddb xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xfc, 0xca], "vpaddb xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xfd, 0x0a], "vpaddw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xfd, 0x4a, 0x01], "vpaddw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0x8d, 0xfd, 0xca], "vpaddw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x14, 0x0a], "vunpcklpd xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x14, 0x4a, 0x01], "vunpcklpd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x15, 0x0a], "vunpckhpd xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x15, 0x4a, 0x01], "vunpckhpd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x51, 0x0a], "vsqrtpd xmm1{k5}{z}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x51, 0x4a, 0x01], "vsqrtpd xmm1{k5}{z}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x51, 0xca], "vsqrtpd zmm1{k5}{z}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x54, 0x0a], "vandpd xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x54, 0x4a, 0x01], "vandpd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x55, 0x0a], "vandnpd xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x55, 0x4a, 0x01], "vandnpd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x56, 0x0a], "vorpd xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x56, 0x4a, 0x01], "vorpd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x57, 0x0a], "vxorpd xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x57, 0x4a, 0x01], "vxorpd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x58, 0x0a], "vaddpd xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x58, 0x4a, 0x01], "vaddpd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x58, 0xca], "vaddpd zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x59, 0x0a], "vmulpd xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x59, 0x4a, 0x01], "vmulpd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x59, 0xca], "vmulpd zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}{z}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1{k5}{z}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{z}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x5c, 0x0a], "vsubpd xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x5c, 0x4a, 0x01], "vsubpd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x5c, 0xca], "vsubpd zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x5d, 0x0a], "vminpd xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x5d, 0x4a, 0x01], "vminpd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x5e, 0x0a], "vdivpd xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x5e, 0x4a, 0x01], "vdivpd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x5e, 0xca], "vdivpd zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x5f, 0x0a], "vmaxpd xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x5f, 0x4a, 0x01], "vmaxpd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x6c, 0x0a], "vpunpcklqdq xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x6c, 0x4a, 0x01], "vpunpcklqdq xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x6d, 0x0a], "vpunpckhqdq xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x6d, 0x4a, 0x01], "vpunpckhqdq xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x72, 0x0a, 0xcc], "vprolq xmm0{k5}{z}, qword [bp + si]{1to2}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x72, 0x4a, 0x01, 0xcc], "vprolq xmm0{k5}{z}, qword [bp + si + 0x8]{1to2}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x78, 0x0a], "vcvttpd2uqq xmm1{k5}{z}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x78, 0x4a, 0x01], "vcvttpd2uqq xmm1{k5}{z}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x79, 0x0a], "vcvtpd2uqq xmm1{k5}{z}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x79, 0x4a, 0x01], "vcvtpd2uqq xmm1{k5}{z}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{z}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x7a, 0x0a], "vcvttpd2qq xmm1{k5}{z}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x7a, 0x4a, 0x01], "vcvttpd2qq xmm1{k5}{z}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x7b, 0x0a], "vcvtpd2qq xmm1{k5}{z}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x7b, 0x4a, 0x01], "vcvtpd2qq xmm1{k5}{z}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{z}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xc6, 0x0a, 0xcc], "vshufpd xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xd4, 0x0a], "vpaddq xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xd4, 0x4a, 0x01], "vpaddq xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xdb, 0x0a], "vpandq xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xdb, 0x4a, 0x01], "vpandq xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xdf, 0x0a], "vpandnq xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xdf, 0x4a, 0x01], "vpandnq xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}{z}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1{k5}{z}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xeb, 0x0a], "vporq xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xeb, 0x4a, 0x01], "vporq xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xef, 0x0a], "vpxorq xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xef, 0x4a, 0x01], "vpxorq xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xf4, 0x0a], "vpmuludq xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xf4, 0x4a, 0x01], "vpmuludq xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xfb, 0x0a], "vpsubq xmm1{k5}{z}, xmm0, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0x9d, 0xfb, 0x4a, 0x01], "vpsubq xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x10, 0x0a], "vmovupd ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x10, 0x4a, 0x01], "vmovupd ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x10, 0xca], "vmovupd ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x11, 0xca], "vmovupd ymm2{k5}{z}, ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x14, 0x0a], "vunpcklpd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x14, 0x4a, 0x01], "vunpcklpd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x14, 0xca], "vunpcklpd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x15, 0x0a], "vunpckhpd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x15, 0x4a, 0x01], "vunpckhpd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x15, 0xca], "vunpckhpd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x28, 0x0a], "vmovapd ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x28, 0x4a, 0x01], "vmovapd ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x28, 0xca], "vmovapd ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x29, 0xca], "vmovapd ymm2{k5}{z}, ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x51, 0x0a], "vsqrtpd ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x51, 0x4a, 0x01], "vsqrtpd ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x51, 0xca], "vsqrtpd ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x54, 0x0a], "vandpd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x54, 0x4a, 0x01], "vandpd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x54, 0xca], "vandpd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x55, 0x0a], "vandnpd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x55, 0x4a, 0x01], "vandnpd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x55, 0xca], "vandnpd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x56, 0x0a], "vorpd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x56, 0x4a, 0x01], "vorpd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x56, 0xca], "vorpd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x57, 0x0a], "vxorpd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x57, 0x4a, 0x01], "vxorpd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x57, 0xca], "vxorpd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x58, 0x0a], "vaddpd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x58, 0x4a, 0x01], "vaddpd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x58, 0xca], "vaddpd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x59, 0x0a], "vmulpd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x59, 0x4a, 0x01], "vmulpd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x59, 0xca], "vmulpd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x5a, 0xca], "vcvtpd2ps xmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x5c, 0x0a], "vsubpd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x5c, 0x4a, 0x01], "vsubpd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x5c, 0xca], "vsubpd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x5d, 0x0a], "vminpd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x5d, 0x4a, 0x01], "vminpd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x5d, 0xca], "vminpd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x5e, 0x0a], "vdivpd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x5e, 0x4a, 0x01], "vdivpd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x5e, 0xca], "vdivpd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x5f, 0x0a], "vmaxpd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x5f, 0x4a, 0x01], "vmaxpd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x5f, 0xca], "vmaxpd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x60, 0x0a], "vpunpcklbw ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x60, 0x4a, 0x01], "vpunpcklbw ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x60, 0xca], "vpunpcklbw ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x61, 0x0a], "vpunpcklwd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x61, 0x4a, 0x01], "vpunpcklwd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x61, 0xca], "vpunpcklwd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x63, 0x0a], "vpacksswb ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x63, 0x4a, 0x01], "vpacksswb ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x63, 0xca], "vpacksswb ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x67, 0x0a], "vpackuswb ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x67, 0x4a, 0x01], "vpackuswb ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x67, 0xca], "vpackuswb ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x68, 0x0a], "vpunpckhbw ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x68, 0x4a, 0x01], "vpunpckhbw ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x68, 0xca], "vpunpckhbw ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x69, 0x0a], "vpunpckhwd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x69, 0x4a, 0x01], "vpunpckhwd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x69, 0xca], "vpunpckhwd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x6c, 0x0a], "vpunpcklqdq ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x6c, 0x4a, 0x01], "vpunpcklqdq ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x6c, 0xca], "vpunpcklqdq ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x6d, 0x0a], "vpunpckhqdq ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x6d, 0x4a, 0x01], "vpunpckhqdq ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x6d, 0xca], "vpunpckhqdq ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x6f, 0x0a], "vmovdqa64 ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x6f, 0x4a, 0x01], "vmovdqa64 ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x6f, 0xca], "vmovdqa64 ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x72, 0x0a, 0xcc], "vprolq ymm0{k5}{z}, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x72, 0x4a, 0x01, 0xcc], "vprolq ymm0{k5}{z}, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x72, 0xca, 0xcc], "vprolq ymm0{k5}{z}, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x78, 0x0a], "vcvttpd2uqq ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x78, 0x4a, 0x01], "vcvttpd2uqq ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x78, 0xca], "vcvttpd2uqq ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x79, 0x0a], "vcvtpd2uqq ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x79, 0x4a, 0x01], "vcvtpd2uqq ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x79, 0xca], "vcvtpd2uqq ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x7a, 0x0a], "vcvttpd2qq ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x7a, 0x4a, 0x01], "vcvttpd2qq ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x7a, 0xca], "vcvttpd2qq ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x7b, 0x0a], "vcvtpd2qq ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x7b, 0x4a, 0x01], "vcvtpd2qq ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x7b, 0xca], "vcvtpd2qq ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0x7f, 0xca], "vmovdqa64 ymm2{k5}{z}, ymm1");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xc6, 0x0a, 0xcc], "vshufpd ymm1{k5}{z}, ymm0, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xc6, 0xca, 0xcc], "vshufpd ymm1{k5}{z}, ymm0, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd1, 0x0a], "vpsrlw ymm1{k5}{z}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd1, 0x4a, 0x01], "vpsrlw ymm1{k5}{z}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd1, 0xca], "vpsrlw ymm1{k5}{z}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd3, 0x0a], "vpsrlq ymm1{k5}{z}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd3, 0x4a, 0x01], "vpsrlq ymm1{k5}{z}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd3, 0xca], "vpsrlq ymm1{k5}{z}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd4, 0x0a], "vpaddq ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd4, 0x4a, 0x01], "vpaddq ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd4, 0xca], "vpaddq ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd5, 0x0a], "vpmullw ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd5, 0x4a, 0x01], "vpmullw ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd5, 0xca], "vpmullw ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd8, 0x0a], "vpsubusb ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd8, 0x4a, 0x01], "vpsubusb ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd8, 0xca], "vpsubusb ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd9, 0x0a], "vpsubusw ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd9, 0x4a, 0x01], "vpsubusw ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xd9, 0xca], "vpsubusw ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xda, 0x0a], "vpminub ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xda, 0x4a, 0x01], "vpminub ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xda, 0xca], "vpminub ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xdb, 0x0a], "vpandq ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xdb, 0x4a, 0x01], "vpandq ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xdb, 0xca], "vpandq ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xdc, 0x0a], "vpaddusb ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xdc, 0x4a, 0x01], "vpaddusb ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xdc, 0xca], "vpaddusb ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xdd, 0x0a], "vpaddusw ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xdd, 0x4a, 0x01], "vpaddusw ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xdd, 0xca], "vpaddusw ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xde, 0x0a], "vpmaxub ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xde, 0x4a, 0x01], "vpmaxub ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xde, 0xca], "vpmaxub ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xdf, 0x0a], "vpandnq ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xdf, 0x4a, 0x01], "vpandnq ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xdf, 0xca], "vpandnq ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe0, 0x0a], "vpavgb ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe0, 0x4a, 0x01], "vpavgb ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe0, 0xca], "vpavgb ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe1, 0x0a], "vpsraw ymm1{k5}{z}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe1, 0x4a, 0x01], "vpsraw ymm1{k5}{z}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe1, 0xca], "vpsraw ymm1{k5}{z}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe2, 0x0a], "vpsraq ymm1{k5}{z}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe2, 0x4a, 0x01], "vpsraq ymm1{k5}{z}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe2, 0xca], "vpsraq ymm1{k5}{z}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe3, 0x0a], "vpavgw ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe3, 0x4a, 0x01], "vpavgw ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe3, 0xca], "vpavgw ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe4, 0x0a], "vpmulhuw ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe4, 0x4a, 0x01], "vpmulhuw ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe4, 0xca], "vpmulhuw ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe5, 0x0a], "vpmulhw ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe5, 0x4a, 0x01], "vpmulhw ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe5, 0xca], "vpmulhw ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe6, 0xca], "vcvttpd2dq xmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe8, 0x0a], "vpsubsb ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe8, 0x4a, 0x01], "vpsubsb ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe8, 0xca], "vpsubsb ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe9, 0x0a], "vpsubsw ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe9, 0x4a, 0x01], "vpsubsw ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xe9, 0xca], "vpsubsw ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xea, 0x0a], "vpminsw ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xea, 0x4a, 0x01], "vpminsw ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xea, 0xca], "vpminsw ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xeb, 0x0a], "vporq ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xeb, 0x4a, 0x01], "vporq ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xeb, 0xca], "vporq ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xec, 0x0a], "vpaddsb ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xec, 0x4a, 0x01], "vpaddsb ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xec, 0xca], "vpaddsb ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xed, 0x0a], "vpaddsw ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xed, 0x4a, 0x01], "vpaddsw ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xed, 0xca], "vpaddsw ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xee, 0x0a], "vpmaxsw ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xee, 0x4a, 0x01], "vpmaxsw ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xee, 0xca], "vpmaxsw ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xef, 0x0a], "vpxorq ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xef, 0x4a, 0x01], "vpxorq ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xef, 0xca], "vpxorq ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf1, 0x0a], "vpsllw ymm1{k5}{z}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf1, 0x4a, 0x01], "vpsllw ymm1{k5}{z}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf1, 0xca], "vpsllw ymm1{k5}{z}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf3, 0x0a], "vpsllq ymm1{k5}{z}, ymm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf3, 0x4a, 0x01], "vpsllq ymm1{k5}{z}, ymm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf3, 0xca], "vpsllq ymm1{k5}{z}, ymm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf4, 0x0a], "vpmuludq ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf4, 0x4a, 0x01], "vpmuludq ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf4, 0xca], "vpmuludq ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf5, 0x0a], "vpmaddwd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf5, 0x4a, 0x01], "vpmaddwd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf5, 0xca], "vpmaddwd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf8, 0x0a], "vpsubb ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf8, 0x4a, 0x01], "vpsubb ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf8, 0xca], "vpsubb ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf9, 0x0a], "vpsubw ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf9, 0x4a, 0x01], "vpsubw ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xf9, 0xca], "vpsubw ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xfb, 0x0a], "vpsubq ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xfb, 0x4a, 0x01], "vpsubq ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xfb, 0xca], "vpsubq ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xfc, 0x0a], "vpaddb ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xfc, 0x4a, 0x01], "vpaddb ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xfc, 0xca], "vpaddb ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xfd, 0x0a], "vpaddw ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xfd, 0x4a, 0x01], "vpaddw ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfd, 0xad, 0xfd, 0xca], "vpaddw ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x14, 0x0a], "vunpcklpd ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x14, 0x4a, 0x01], "vunpcklpd ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x15, 0x0a], "vunpckhpd ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x15, 0x4a, 0x01], "vunpckhpd ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x51, 0x0a], "vsqrtpd ymm1{k5}{z}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x51, 0x4a, 0x01], "vsqrtpd ymm1{k5}{z}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x51, 0xca], "vsqrtpd zmm1{k5}{z}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x54, 0x0a], "vandpd ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x54, 0x4a, 0x01], "vandpd ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x55, 0x0a], "vandnpd ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x55, 0x4a, 0x01], "vandnpd ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x56, 0x0a], "vorpd ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x56, 0x4a, 0x01], "vorpd ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x57, 0x0a], "vxorpd ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x57, 0x4a, 0x01], "vxorpd ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x58, 0x0a], "vaddpd ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x58, 0x4a, 0x01], "vaddpd ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x58, 0xca], "vaddpd zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x59, 0x0a], "vmulpd ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x59, 0x4a, 0x01], "vmulpd ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x59, 0xca], "vmulpd zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}{z}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x5a, 0x4a, 0x01], "vcvtpd2ps xmm1{k5}{z}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{z}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x5c, 0x0a], "vsubpd ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x5c, 0x4a, 0x01], "vsubpd ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x5c, 0xca], "vsubpd zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x5d, 0x0a], "vminpd ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x5d, 0x4a, 0x01], "vminpd ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x5e, 0x0a], "vdivpd ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x5e, 0x4a, 0x01], "vdivpd ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x5e, 0xca], "vdivpd zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x5f, 0x0a], "vmaxpd ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x5f, 0x4a, 0x01], "vmaxpd ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x6c, 0x0a], "vpunpcklqdq ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x6c, 0x4a, 0x01], "vpunpcklqdq ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x6d, 0x0a], "vpunpckhqdq ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x6d, 0x4a, 0x01], "vpunpckhqdq ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x72, 0x0a, 0xcc], "vprolq ymm0{k5}{z}, qword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x72, 0x4a, 0x01, 0xcc], "vprolq ymm0{k5}{z}, qword [bp + si + 0x8]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x78, 0x0a], "vcvttpd2uqq ymm1{k5}{z}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x78, 0x4a, 0x01], "vcvttpd2uqq ymm1{k5}{z}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x79, 0x0a], "vcvtpd2uqq ymm1{k5}{z}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x79, 0x4a, 0x01], "vcvtpd2uqq ymm1{k5}{z}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{z}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x7a, 0x0a], "vcvttpd2qq ymm1{k5}{z}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x7a, 0x4a, 0x01], "vcvttpd2qq ymm1{k5}{z}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x7b, 0x0a], "vcvtpd2qq ymm1{k5}{z}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x7b, 0x4a, 0x01], "vcvtpd2qq ymm1{k5}{z}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{z}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xc6, 0x0a, 0xcc], "vshufpd ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xd4, 0x0a], "vpaddq ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xd4, 0x4a, 0x01], "vpaddq ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xdb, 0x0a], "vpandq ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xdb, 0x4a, 0x01], "vpandq ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xdf, 0x0a], "vpandnq ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xdf, 0x4a, 0x01], "vpandnq ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}{z}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xe6, 0x4a, 0x01], "vcvttpd2dq xmm1{k5}{z}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xeb, 0x0a], "vporq ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xeb, 0x4a, 0x01], "vporq ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xef, 0x0a], "vpxorq ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xef, 0x4a, 0x01], "vpxorq ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xf4, 0x0a], "vpmuludq ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xf4, 0x4a, 0x01], "vpmuludq ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xfb, 0x0a], "vpsubq ymm1{k5}{z}, ymm0, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xbd, 0xfb, 0x4a, 0x01], "vpsubq ymm1{k5}{z}, ymm0, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x10, 0x0a], "vmovupd zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x10, 0x4a, 0x01], "vmovupd zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x10, 0xca], "vmovupd zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x11, 0xca], "vmovupd zmm2{k5}{z}, zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x14, 0x0a], "vunpcklpd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x14, 0x4a, 0x01], "vunpcklpd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x14, 0xca], "vunpcklpd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x15, 0x0a], "vunpckhpd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x15, 0x4a, 0x01], "vunpckhpd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x15, 0xca], "vunpckhpd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x28, 0x0a], "vmovapd zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x28, 0x4a, 0x01], "vmovapd zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x28, 0xca], "vmovapd zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x29, 0xca], "vmovapd zmm2{k5}{z}, zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x51, 0x0a], "vsqrtpd zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x51, 0x4a, 0x01], "vsqrtpd zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x51, 0xca], "vsqrtpd zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x54, 0x0a], "vandpd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x54, 0x4a, 0x01], "vandpd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x54, 0xca], "vandpd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x55, 0x0a], "vandnpd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x55, 0x4a, 0x01], "vandnpd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x55, 0xca], "vandnpd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x56, 0x0a], "vorpd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x56, 0x4a, 0x01], "vorpd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x56, 0xca], "vorpd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x57, 0x0a], "vxorpd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x57, 0x4a, 0x01], "vxorpd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x57, 0xca], "vxorpd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x58, 0x0a], "vaddpd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x58, 0x4a, 0x01], "vaddpd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x58, 0xca], "vaddpd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x59, 0x0a], "vmulpd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x59, 0x4a, 0x01], "vmulpd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x59, 0xca], "vmulpd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x5a, 0x0a], "vcvtpd2ps ymm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x5a, 0x4a, 0x01], "vcvtpd2ps ymm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x5c, 0x0a], "vsubpd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x5c, 0x4a, 0x01], "vsubpd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x5c, 0xca], "vsubpd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x5d, 0x0a], "vminpd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x5d, 0x4a, 0x01], "vminpd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x5d, 0xca], "vminpd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x5e, 0x0a], "vdivpd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x5e, 0x4a, 0x01], "vdivpd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x5e, 0xca], "vdivpd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x5f, 0x0a], "vmaxpd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x5f, 0x4a, 0x01], "vmaxpd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x5f, 0xca], "vmaxpd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x60, 0x0a], "vpunpcklbw zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x60, 0x4a, 0x01], "vpunpcklbw zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x60, 0xca], "vpunpcklbw zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x61, 0x0a], "vpunpcklwd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x61, 0x4a, 0x01], "vpunpcklwd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x61, 0xca], "vpunpcklwd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x63, 0x0a], "vpacksswb zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x63, 0x4a, 0x01], "vpacksswb zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x63, 0xca], "vpacksswb zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x67, 0x0a], "vpackuswb zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x67, 0x4a, 0x01], "vpackuswb zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x67, 0xca], "vpackuswb zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x68, 0x0a], "vpunpckhbw zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x68, 0x4a, 0x01], "vpunpckhbw zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x68, 0xca], "vpunpckhbw zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x69, 0x0a], "vpunpckhwd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x69, 0x4a, 0x01], "vpunpckhwd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x69, 0xca], "vpunpckhwd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x6c, 0x0a], "vpunpcklqdq zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x6c, 0x4a, 0x01], "vpunpcklqdq zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x6c, 0xca], "vpunpcklqdq zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x6d, 0x0a], "vpunpckhqdq zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x6d, 0x4a, 0x01], "vpunpckhqdq zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x6d, 0xca], "vpunpckhqdq zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x6f, 0x0a], "vmovdqa64 zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x6f, 0x4a, 0x01], "vmovdqa64 zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x6f, 0xca], "vmovdqa64 zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x72, 0x0a, 0xcc], "vprolq zmm0{k5}{z}, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x72, 0x4a, 0x01, 0xcc], "vprolq zmm0{k5}{z}, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x72, 0xca, 0xcc], "vprolq zmm0{k5}{z}, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x78, 0x0a], "vcvttpd2uqq zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x78, 0x4a, 0x01], "vcvttpd2uqq zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x78, 0xca], "vcvttpd2uqq zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x79, 0x0a], "vcvtpd2uqq zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x79, 0x4a, 0x01], "vcvtpd2uqq zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x7a, 0x0a], "vcvttpd2qq zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x7a, 0x4a, 0x01], "vcvttpd2qq zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x7a, 0xca], "vcvttpd2qq zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x7b, 0x0a], "vcvtpd2qq zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x7b, 0x4a, 0x01], "vcvtpd2qq zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0x7f, 0xca], "vmovdqa64 zmm2{k5}{z}, zmm1");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xc6, 0x0a, 0xcc], "vshufpd zmm1{k5}{z}, zmm0, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xc6, 0xca, 0xcc], "vshufpd zmm1{k5}{z}, zmm0, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd1, 0x0a], "vpsrlw zmm1{k5}{z}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd1, 0x4a, 0x01], "vpsrlw zmm1{k5}{z}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd1, 0xca], "vpsrlw zmm1{k5}{z}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd3, 0x0a], "vpsrlq zmm1{k5}{z}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd3, 0x4a, 0x01], "vpsrlq zmm1{k5}{z}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd3, 0xca], "vpsrlq zmm1{k5}{z}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd4, 0x0a], "vpaddq zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd4, 0x4a, 0x01], "vpaddq zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd4, 0xca], "vpaddq zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd5, 0x0a], "vpmullw zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd5, 0x4a, 0x01], "vpmullw zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd5, 0xca], "vpmullw zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd8, 0x0a], "vpsubusb zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd8, 0x4a, 0x01], "vpsubusb zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd8, 0xca], "vpsubusb zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd9, 0x0a], "vpsubusw zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd9, 0x4a, 0x01], "vpsubusw zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xd9, 0xca], "vpsubusw zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xda, 0x0a], "vpminub zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xda, 0x4a, 0x01], "vpminub zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xda, 0xca], "vpminub zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xdb, 0x0a], "vpandq zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xdb, 0x4a, 0x01], "vpandq zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xdb, 0xca], "vpandq zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xdc, 0x0a], "vpaddusb zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xdc, 0x4a, 0x01], "vpaddusb zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xdc, 0xca], "vpaddusb zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xdd, 0x0a], "vpaddusw zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xdd, 0x4a, 0x01], "vpaddusw zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xdd, 0xca], "vpaddusw zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xde, 0x0a], "vpmaxub zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xde, 0x4a, 0x01], "vpmaxub zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xde, 0xca], "vpmaxub zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xdf, 0x0a], "vpandnq zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xdf, 0x4a, 0x01], "vpandnq zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xdf, 0xca], "vpandnq zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe0, 0x0a], "vpavgb zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe0, 0x4a, 0x01], "vpavgb zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe0, 0xca], "vpavgb zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe1, 0x0a], "vpsraw zmm1{k5}{z}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe1, 0x4a, 0x01], "vpsraw zmm1{k5}{z}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe1, 0xca], "vpsraw zmm1{k5}{z}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe2, 0x0a], "vpsraq zmm1{k5}{z}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe2, 0x4a, 0x01], "vpsraq zmm1{k5}{z}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe2, 0xca], "vpsraq zmm1{k5}{z}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe3, 0x0a], "vpavgw zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe3, 0x4a, 0x01], "vpavgw zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe3, 0xca], "vpavgw zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe4, 0x0a], "vpmulhuw zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe4, 0x4a, 0x01], "vpmulhuw zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe4, 0xca], "vpmulhuw zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe5, 0x0a], "vpmulhw zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe5, 0x4a, 0x01], "vpmulhw zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe5, 0xca], "vpmulhw zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe6, 0x0a], "vcvttpd2dq ymm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe6, 0x4a, 0x01], "vcvttpd2dq ymm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe6, 0xca], "vcvttpd2dq ymm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe8, 0x0a], "vpsubsb zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe8, 0x4a, 0x01], "vpsubsb zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe8, 0xca], "vpsubsb zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe9, 0x0a], "vpsubsw zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe9, 0x4a, 0x01], "vpsubsw zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xe9, 0xca], "vpsubsw zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xea, 0x0a], "vpminsw zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xea, 0x4a, 0x01], "vpminsw zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xea, 0xca], "vpminsw zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xeb, 0x0a], "vporq zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xeb, 0x4a, 0x01], "vporq zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xeb, 0xca], "vporq zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xec, 0x0a], "vpaddsb zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xec, 0x4a, 0x01], "vpaddsb zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xec, 0xca], "vpaddsb zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xed, 0x0a], "vpaddsw zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xed, 0x4a, 0x01], "vpaddsw zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xed, 0xca], "vpaddsw zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xee, 0x0a], "vpmaxsw zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xee, 0x4a, 0x01], "vpmaxsw zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xee, 0xca], "vpmaxsw zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xef, 0x0a], "vpxorq zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xef, 0x4a, 0x01], "vpxorq zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xef, 0xca], "vpxorq zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf1, 0x0a], "vpsllw zmm1{k5}{z}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf1, 0x4a, 0x01], "vpsllw zmm1{k5}{z}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf1, 0xca], "vpsllw zmm1{k5}{z}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf3, 0x0a], "vpsllq zmm1{k5}{z}, zmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf3, 0x4a, 0x01], "vpsllq zmm1{k5}{z}, zmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf3, 0xca], "vpsllq zmm1{k5}{z}, zmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf4, 0x0a], "vpmuludq zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf4, 0x4a, 0x01], "vpmuludq zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf4, 0xca], "vpmuludq zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf5, 0x0a], "vpmaddwd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf5, 0x4a, 0x01], "vpmaddwd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf5, 0xca], "vpmaddwd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf8, 0x0a], "vpsubb zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf8, 0x4a, 0x01], "vpsubb zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf8, 0xca], "vpsubb zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf9, 0x0a], "vpsubw zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf9, 0x4a, 0x01], "vpsubw zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xf9, 0xca], "vpsubw zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xfb, 0x0a], "vpsubq zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xfb, 0x4a, 0x01], "vpsubq zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xfb, 0xca], "vpsubq zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xfc, 0x0a], "vpaddb zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xfc, 0x4a, 0x01], "vpaddb zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xfc, 0xca], "vpaddb zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xfd, 0x0a], "vpaddw zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xfd, 0x4a, 0x01], "vpaddw zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfd, 0xcd, 0xfd, 0xca], "vpaddw zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x14, 0x0a], "vunpcklpd zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x14, 0x4a, 0x01], "vunpcklpd zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x15, 0x0a], "vunpckhpd zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x15, 0x4a, 0x01], "vunpckhpd zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x51, 0x0a], "vsqrtpd zmm1{k5}{z}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x51, 0x4a, 0x01], "vsqrtpd zmm1{k5}{z}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x51, 0xca], "vsqrtpd zmm1{k5}{z}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x54, 0x0a], "vandpd zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x54, 0x4a, 0x01], "vandpd zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x55, 0x0a], "vandnpd zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x55, 0x4a, 0x01], "vandnpd zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x56, 0x0a], "vorpd zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x56, 0x4a, 0x01], "vorpd zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x57, 0x0a], "vxorpd zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x57, 0x4a, 0x01], "vxorpd zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x58, 0x0a], "vaddpd zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x58, 0x4a, 0x01], "vaddpd zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x58, 0xca], "vaddpd zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x59, 0x0a], "vmulpd zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x59, 0x4a, 0x01], "vmulpd zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x59, 0xca], "vmulpd zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x5a, 0x0a], "vcvtpd2ps ymm1{k5}{z}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x5a, 0x4a, 0x01], "vcvtpd2ps ymm1{k5}{z}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{z}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x5c, 0x0a], "vsubpd zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x5c, 0x4a, 0x01], "vsubpd zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x5c, 0xca], "vsubpd zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x5d, 0x0a], "vminpd zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x5d, 0x4a, 0x01], "vminpd zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x5e, 0x0a], "vdivpd zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x5e, 0x4a, 0x01], "vdivpd zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x5e, 0xca], "vdivpd zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x5f, 0x0a], "vmaxpd zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x5f, 0x4a, 0x01], "vmaxpd zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x6c, 0x0a], "vpunpcklqdq zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x6c, 0x4a, 0x01], "vpunpcklqdq zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x6d, 0x0a], "vpunpckhqdq zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x6d, 0x4a, 0x01], "vpunpckhqdq zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x72, 0x0a, 0xcc], "vprolq zmm0{k5}{z}, qword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x72, 0x4a, 0x01, 0xcc], "vprolq zmm0{k5}{z}, qword [bp + si + 0x8]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x78, 0x0a], "vcvttpd2uqq zmm1{k5}{z}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x78, 0x4a, 0x01], "vcvttpd2uqq zmm1{k5}{z}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x79, 0x0a], "vcvtpd2uqq zmm1{k5}{z}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x79, 0x4a, 0x01], "vcvtpd2uqq zmm1{k5}{z}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{z}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x7a, 0x0a], "vcvttpd2qq zmm1{k5}{z}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x7a, 0x4a, 0x01], "vcvttpd2qq zmm1{k5}{z}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x7b, 0x0a], "vcvtpd2qq zmm1{k5}{z}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x7b, 0x4a, 0x01], "vcvtpd2qq zmm1{k5}{z}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{z}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xc6, 0x0a, 0xcc], "vshufpd zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xc6, 0x4a, 0x01, 0xcc], "vshufpd zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}, 0xcc");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xd4, 0x0a], "vpaddq zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xd4, 0x4a, 0x01], "vpaddq zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xdb, 0x0a], "vpandq zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xdb, 0x4a, 0x01], "vpandq zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xdf, 0x0a], "vpandnq zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xdf, 0x4a, 0x01], "vpandnq zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xe6, 0x0a], "vcvttpd2dq ymm1{k5}{z}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xe6, 0x4a, 0x01], "vcvttpd2dq ymm1{k5}{z}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xeb, 0x0a], "vporq zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xeb, 0x4a, 0x01], "vporq zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xef, 0x0a], "vpxorq zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xef, 0x4a, 0x01], "vpxorq zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xf4, 0x0a], "vpmuludq zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xf4, 0x4a, 0x01], "vpmuludq zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xfb, 0x0a], "vpsubq zmm1{k5}{z}, zmm0, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xdd, 0xfb, 0x4a, 0x01], "vpsubq zmm1{k5}{z}, zmm0, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfd, 0xfd, 0x51, 0xca], "vsqrtpd zmm1{k5}{z}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xfd, 0x58, 0xca], "vaddpd zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xfd, 0x59, 0xca], "vmulpd zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xfd, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{z}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xfd, 0x5c, 0xca], "vsubpd zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xfd, 0x5d, 0xca], "vminpd zmm1{k5}{z}{sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xfd, 0x5e, 0xca], "vdivpd zmm1{k5}{z}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xfd, 0x5f, 0xca], "vmaxpd zmm1{k5}{z}{sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xfd, 0x78, 0xca], "vcvttpd2uqq zmm1{k5}{z}{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xfd, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{z}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xfd, 0x7a, 0xca], "vcvttpd2qq zmm1{k5}{z}{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xfd, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{z}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfd, 0xfd, 0xe6, 0xca], "vcvttpd2dq ymm1{k5}{z}{sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0x6f, 0x0a], "vmovdqu64 xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0x6f, 0x4a, 0x01], "vmovdqu64 xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0x6f, 0xca], "vmovdqu64 xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0x70, 0x0a, 0xcc], "vpshufhw xmm1, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0x70, 0x4a, 0x01, 0xcc], "vpshufhw xmm1, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0x70, 0xca, 0xcc], "vpshufhw xmm1, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0x7a, 0x0a], "vcvtuqq2pd xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0x7a, 0x4a, 0x01], "vcvtuqq2pd xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0x7a, 0xca], "vcvtuqq2pd xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0x7e, 0x0a], "vmovq xmm1, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0x7e, 0x4a, 0x01], "vmovq xmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0x7e, 0xca], "vmovq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0x7f, 0x0a], "vmovdqu64 xmmword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0x7f, 0x4a, 0x01], "vmovdqu64 xmmword [bp + si + 0x10], xmm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0x7f, 0xca], "vmovdqu64 xmm2, xmm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0xe6, 0x0a], "vcvtqq2pd xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0xe6, 0x4a, 0x01], "vcvtqq2pd xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfe, 0x08, 0xe6, 0xca], "vcvtqq2pd xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x0d, 0x6f, 0x0a], "vmovdqu64 xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x0d, 0x6f, 0x4a, 0x01], "vmovdqu64 xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfe, 0x0d, 0x6f, 0xca], "vmovdqu64 xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x0d, 0x70, 0x0a, 0xcc], "vpshufhw xmm1{k5}, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x0d, 0x70, 0x4a, 0x01, 0xcc], "vpshufhw xmm1{k5}, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x0d, 0x70, 0xca, 0xcc], "vpshufhw xmm1{k5}, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x0d, 0x7a, 0x0a], "vcvtuqq2pd xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x0d, 0x7a, 0x4a, 0x01], "vcvtuqq2pd xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfe, 0x0d, 0x7a, 0xca], "vcvtuqq2pd xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x0d, 0x7f, 0x0a], "vmovdqu64 xmmword [bp + si]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x0d, 0x7f, 0x4a, 0x01], "vmovdqu64 xmmword [bp + si + 0x10]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x0d, 0x7f, 0xca], "vmovdqu64 xmm2{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x0d, 0xe6, 0x0a], "vcvtqq2pd xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x0d, 0xe6, 0x4a, 0x01], "vcvtqq2pd xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfe, 0x0d, 0xe6, 0xca], "vcvtqq2pd xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x18, 0x2a, 0xca], "vcvtsi2ss xmm1{rne-sae}, xmm0, edx");
    test_display(&[0x62, 0xf1, 0xfe, 0x18, 0x2d, 0xca], "vcvtss2si ecx{rne-sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x18, 0x79, 0xca], "vcvtss2usi ecx{rne-sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x18, 0x7a, 0x0a], "vcvtuqq2pd xmm1, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfe, 0x18, 0x7a, 0x4a, 0x01], "vcvtuqq2pd xmm1, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfe, 0x18, 0x7a, 0xca], "vcvtuqq2pd zmm1{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x18, 0x7b, 0xca], "vcvtusi2ss xmm1{rne-sae}, xmm0, edx");
    test_display(&[0x62, 0xf1, 0xfe, 0x18, 0xe6, 0x0a], "vcvtqq2pd xmm1, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfe, 0x18, 0xe6, 0x4a, 0x01], "vcvtqq2pd xmm1, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfe, 0x18, 0xe6, 0xca], "vcvtqq2pd zmm1{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x1d, 0x7a, 0x0a], "vcvtuqq2pd xmm1{k5}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfe, 0x1d, 0x7a, 0x4a, 0x01], "vcvtuqq2pd xmm1{k5}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfe, 0x1d, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x1d, 0xe6, 0x0a], "vcvtqq2pd xmm1{k5}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfe, 0x1d, 0xe6, 0x4a, 0x01], "vcvtqq2pd xmm1{k5}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfe, 0x1d, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x2a, 0x0a], "vcvtsi2ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x2a, 0x4a, 0x01], "vcvtsi2ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x2a, 0xca], "vcvtsi2ss xmm1, xmm0, edx");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x2c, 0x0a], "vcvttss2si ecx, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x2c, 0x4a, 0x01], "vcvttss2si ecx, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x2c, 0xca], "vcvttss2si ecx, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x2d, 0x0a], "vcvtss2si ecx, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x2d, 0x4a, 0x01], "vcvtss2si ecx, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x2d, 0xca], "vcvtss2si ecx, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x6f, 0x0a], "vmovdqu64 ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x6f, 0x4a, 0x01], "vmovdqu64 ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x6f, 0xca], "vmovdqu64 ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x70, 0x0a, 0xcc], "vpshufhw ymm1, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x70, 0x4a, 0x01, 0xcc], "vpshufhw ymm1, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x70, 0xca, 0xcc], "vpshufhw ymm1, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x78, 0x0a], "vcvttss2usi ecx, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x78, 0x4a, 0x01], "vcvttss2usi ecx, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x78, 0xca], "vcvttss2usi ecx, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x79, 0x0a], "vcvtss2usi ecx, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x79, 0x4a, 0x01], "vcvtss2usi ecx, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x79, 0xca], "vcvtss2usi ecx, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x7a, 0x0a], "vcvtuqq2pd ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x7a, 0x4a, 0x01], "vcvtuqq2pd ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x7a, 0xca], "vcvtuqq2pd ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x7b, 0x0a], "vcvtusi2ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x7b, 0x4a, 0x01], "vcvtusi2ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x7b, 0xca], "vcvtusi2ss xmm1, xmm0, edx");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x7f, 0x0a], "vmovdqu64 ymmword [bp + si], ymm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x7f, 0x4a, 0x01], "vmovdqu64 ymmword [bp + si + 0x20], ymm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0x7f, 0xca], "vmovdqu64 ymm2, ymm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0xe6, 0x0a], "vcvtqq2pd ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0xe6, 0x4a, 0x01], "vcvtqq2pd ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfe, 0x28, 0xe6, 0xca], "vcvtqq2pd ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x2d, 0x6f, 0x0a], "vmovdqu64 ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x2d, 0x6f, 0x4a, 0x01], "vmovdqu64 ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfe, 0x2d, 0x6f, 0xca], "vmovdqu64 ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x2d, 0x70, 0x0a, 0xcc], "vpshufhw ymm1{k5}, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x2d, 0x70, 0x4a, 0x01, 0xcc], "vpshufhw ymm1{k5}, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x2d, 0x70, 0xca, 0xcc], "vpshufhw ymm1{k5}, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x2d, 0x7a, 0x0a], "vcvtuqq2pd ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x2d, 0x7a, 0x4a, 0x01], "vcvtuqq2pd ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfe, 0x2d, 0x7a, 0xca], "vcvtuqq2pd ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x2d, 0x7f, 0x0a], "vmovdqu64 ymmword [bp + si]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x2d, 0x7f, 0x4a, 0x01], "vmovdqu64 ymmword [bp + si + 0x20]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x2d, 0x7f, 0xca], "vmovdqu64 ymm2{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x2d, 0xe6, 0x0a], "vcvtqq2pd ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x2d, 0xe6, 0x4a, 0x01], "vcvtqq2pd ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfe, 0x2d, 0xe6, 0xca], "vcvtqq2pd ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x38, 0x2a, 0xca], "vcvtsi2ss xmm1{rd-sae}, xmm0, edx");
    test_display(&[0x62, 0xf1, 0xfe, 0x38, 0x2d, 0xca], "vcvtss2si ecx{rd-sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x38, 0x79, 0xca], "vcvtss2usi ecx{rd-sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x38, 0x7a, 0x0a], "vcvtuqq2pd ymm1, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfe, 0x38, 0x7a, 0x4a, 0x01], "vcvtuqq2pd ymm1, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfe, 0x38, 0x7a, 0xca], "vcvtuqq2pd zmm1{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x38, 0x7b, 0xca], "vcvtusi2ss xmm1{rd-sae}, xmm0, edx");
    test_display(&[0x62, 0xf1, 0xfe, 0x38, 0xe6, 0x0a], "vcvtqq2pd ymm1, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfe, 0x38, 0xe6, 0x4a, 0x01], "vcvtqq2pd ymm1, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfe, 0x38, 0xe6, 0xca], "vcvtqq2pd zmm1{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x3d, 0x7a, 0x0a], "vcvtuqq2pd ymm1{k5}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfe, 0x3d, 0x7a, 0x4a, 0x01], "vcvtuqq2pd ymm1{k5}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfe, 0x3d, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x3d, 0xe6, 0x0a], "vcvtqq2pd ymm1{k5}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfe, 0x3d, 0xe6, 0x4a, 0x01], "vcvtqq2pd ymm1{k5}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfe, 0x3d, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x48, 0x6f, 0x0a], "vmovdqu64 zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x48, 0x6f, 0x4a, 0x01], "vmovdqu64 zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfe, 0x48, 0x6f, 0xca], "vmovdqu64 zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x48, 0x70, 0x0a, 0xcc], "vpshufhw zmm1, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x48, 0x70, 0x4a, 0x01, 0xcc], "vpshufhw zmm1, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x48, 0x70, 0xca, 0xcc], "vpshufhw zmm1, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x48, 0x7a, 0x0a], "vcvtuqq2pd zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x48, 0x7a, 0x4a, 0x01], "vcvtuqq2pd zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfe, 0x48, 0x7a, 0xca], "vcvtuqq2pd zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x48, 0x7f, 0x0a], "vmovdqu64 zmmword [bp + si], zmm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x48, 0x7f, 0x4a, 0x01], "vmovdqu64 zmmword [bp + si + 0x40], zmm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x48, 0x7f, 0xca], "vmovdqu64 zmm2, zmm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x48, 0xe6, 0x0a], "vcvtqq2pd zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x48, 0xe6, 0x4a, 0x01], "vcvtqq2pd zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfe, 0x48, 0xe6, 0xca], "vcvtqq2pd zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x4d, 0x6f, 0x0a], "vmovdqu64 zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x4d, 0x6f, 0x4a, 0x01], "vmovdqu64 zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfe, 0x4d, 0x6f, 0xca], "vmovdqu64 zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x4d, 0x70, 0x0a, 0xcc], "vpshufhw zmm1{k5}, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x4d, 0x70, 0x4a, 0x01, 0xcc], "vpshufhw zmm1{k5}, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x4d, 0x70, 0xca, 0xcc], "vpshufhw zmm1{k5}, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x4d, 0x7a, 0x0a], "vcvtuqq2pd zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x4d, 0x7a, 0x4a, 0x01], "vcvtuqq2pd zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfe, 0x4d, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x4d, 0x7f, 0x0a], "vmovdqu64 zmmword [bp + si]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x4d, 0x7f, 0x4a, 0x01], "vmovdqu64 zmmword [bp + si + 0x40]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x4d, 0x7f, 0xca], "vmovdqu64 zmm2{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x4d, 0xe6, 0x0a], "vcvtqq2pd zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x4d, 0xe6, 0x4a, 0x01], "vcvtqq2pd zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfe, 0x4d, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x58, 0x2a, 0xca], "vcvtsi2ss xmm1{ru-sae}, xmm0, edx");
    test_display(&[0x62, 0xf1, 0xfe, 0x58, 0x2d, 0xca], "vcvtss2si ecx{ru-sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x58, 0x79, 0xca], "vcvtss2usi ecx{ru-sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x58, 0x7a, 0x0a], "vcvtuqq2pd zmm1, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfe, 0x58, 0x7a, 0x4a, 0x01], "vcvtuqq2pd zmm1, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfe, 0x58, 0x7a, 0xca], "vcvtuqq2pd zmm1{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x58, 0x7b, 0xca], "vcvtusi2ss xmm1{ru-sae}, xmm0, edx");
    test_display(&[0x62, 0xf1, 0xfe, 0x58, 0xe6, 0x0a], "vcvtqq2pd zmm1, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfe, 0x58, 0xe6, 0x4a, 0x01], "vcvtqq2pd zmm1, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfe, 0x58, 0xe6, 0xca], "vcvtqq2pd zmm1{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x5d, 0x7a, 0x0a], "vcvtuqq2pd zmm1{k5}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfe, 0x5d, 0x7a, 0x4a, 0x01], "vcvtuqq2pd zmm1{k5}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfe, 0x5d, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x5d, 0xe6, 0x0a], "vcvtqq2pd zmm1{k5}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfe, 0x5d, 0xe6, 0x4a, 0x01], "vcvtqq2pd zmm1{k5}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfe, 0x5d, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x78, 0x2a, 0xca], "vcvtsi2ss xmm1{rz-sae}, xmm0, edx");
    test_display(&[0x62, 0xf1, 0xfe, 0x78, 0x2c, 0xca], "vcvttss2si ecx{sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x78, 0x2d, 0xca], "vcvtss2si ecx{rz-sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x78, 0x78, 0xca], "vcvttss2usi ecx{sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x78, 0x79, 0xca], "vcvtss2usi ecx{rz-sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x78, 0x7a, 0xca], "vcvtuqq2pd zmm1{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x78, 0x7b, 0xca], "vcvtusi2ss xmm1{rz-sae}, xmm0, edx");
    test_display(&[0x62, 0xf1, 0xfe, 0x78, 0xe6, 0xca], "vcvtqq2pd zmm1{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x7d, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x7d, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x8d, 0x6f, 0x0a], "vmovdqu64 xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x8d, 0x6f, 0x4a, 0x01], "vmovdqu64 xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfe, 0x8d, 0x6f, 0xca], "vmovdqu64 xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x8d, 0x70, 0x0a, 0xcc], "vpshufhw xmm1{k5}{z}, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x8d, 0x70, 0x4a, 0x01, 0xcc], "vpshufhw xmm1{k5}{z}, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x8d, 0x70, 0xca, 0xcc], "vpshufhw xmm1{k5}{z}, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0x8d, 0x7a, 0x0a], "vcvtuqq2pd xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x8d, 0x7a, 0x4a, 0x01], "vcvtuqq2pd xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfe, 0x8d, 0x7a, 0xca], "vcvtuqq2pd xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x8d, 0x7f, 0xca], "vmovdqu64 xmm2{k5}{z}, xmm1");
    test_display(&[0x62, 0xf1, 0xfe, 0x8d, 0xe6, 0x0a], "vcvtqq2pd xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0x8d, 0xe6, 0x4a, 0x01], "vcvtqq2pd xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xfe, 0x8d, 0xe6, 0xca], "vcvtqq2pd xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x9d, 0x7a, 0x0a], "vcvtuqq2pd xmm1{k5}{z}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfe, 0x9d, 0x7a, 0x4a, 0x01], "vcvtuqq2pd xmm1{k5}{z}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfe, 0x9d, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{z}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0x9d, 0xe6, 0x0a], "vcvtqq2pd xmm1{k5}{z}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xfe, 0x9d, 0xe6, 0x4a, 0x01], "vcvtqq2pd xmm1{k5}{z}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xfe, 0x9d, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{z}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0xad, 0x6f, 0x0a], "vmovdqu64 ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0xad, 0x6f, 0x4a, 0x01], "vmovdqu64 ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfe, 0xad, 0x6f, 0xca], "vmovdqu64 ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xfe, 0xad, 0x70, 0x0a, 0xcc], "vpshufhw ymm1{k5}{z}, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0xad, 0x70, 0x4a, 0x01, 0xcc], "vpshufhw ymm1{k5}{z}, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0xad, 0x70, 0xca, 0xcc], "vpshufhw ymm1{k5}{z}, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0xad, 0x7a, 0x0a], "vcvtuqq2pd ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0xad, 0x7a, 0x4a, 0x01], "vcvtuqq2pd ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfe, 0xad, 0x7a, 0xca], "vcvtuqq2pd ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xfe, 0xad, 0x7f, 0xca], "vmovdqu64 ymm2{k5}{z}, ymm1");
    test_display(&[0x62, 0xf1, 0xfe, 0xad, 0xe6, 0x0a], "vcvtqq2pd ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0xad, 0xe6, 0x4a, 0x01], "vcvtqq2pd ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xfe, 0xad, 0xe6, 0xca], "vcvtqq2pd ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xfe, 0xbd, 0x7a, 0x0a], "vcvtuqq2pd ymm1{k5}{z}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfe, 0xbd, 0x7a, 0x4a, 0x01], "vcvtuqq2pd ymm1{k5}{z}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfe, 0xbd, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{z}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0xbd, 0xe6, 0x0a], "vcvtqq2pd ymm1{k5}{z}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xfe, 0xbd, 0xe6, 0x4a, 0x01], "vcvtqq2pd ymm1{k5}{z}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xfe, 0xbd, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{z}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0xcd, 0x6f, 0x0a], "vmovdqu64 zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0xcd, 0x6f, 0x4a, 0x01], "vmovdqu64 zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfe, 0xcd, 0x6f, 0xca], "vmovdqu64 zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0xcd, 0x70, 0x0a, 0xcc], "vpshufhw zmm1{k5}{z}, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0xcd, 0x70, 0x4a, 0x01, 0xcc], "vpshufhw zmm1{k5}{z}, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0xcd, 0x70, 0xca, 0xcc], "vpshufhw zmm1{k5}{z}, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xfe, 0xcd, 0x7a, 0x0a], "vcvtuqq2pd zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0xcd, 0x7a, 0x4a, 0x01], "vcvtuqq2pd zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfe, 0xcd, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0xcd, 0x7f, 0xca], "vmovdqu64 zmm2{k5}{z}, zmm1");
    test_display(&[0x62, 0xf1, 0xfe, 0xcd, 0xe6, 0x0a], "vcvtqq2pd zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xfe, 0xcd, 0xe6, 0x4a, 0x01], "vcvtqq2pd zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xfe, 0xcd, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0xdd, 0x7a, 0x0a], "vcvtuqq2pd zmm1{k5}{z}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfe, 0xdd, 0x7a, 0x4a, 0x01], "vcvtuqq2pd zmm1{k5}{z}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfe, 0xdd, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{z}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0xdd, 0xe6, 0x0a], "vcvtqq2pd zmm1{k5}{z}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xfe, 0xdd, 0xe6, 0x4a, 0x01], "vcvtqq2pd zmm1{k5}{z}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xfe, 0xdd, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{z}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0xfd, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{z}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xfe, 0xfd, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{z}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0x12, 0x0a], "vmovddup xmm1, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0x12, 0x4a, 0x01], "vmovddup xmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0x12, 0xca], "vmovddup xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0x6f, 0x0a], "vmovdqu16 xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0x6f, 0x4a, 0x01], "vmovdqu16 xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0x6f, 0xca], "vmovdqu16 xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0x70, 0x0a, 0xcc], "vpshuflw xmm1, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0x70, 0x4a, 0x01, 0xcc], "vpshuflw xmm1, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0x70, 0xca, 0xcc], "vpshuflw xmm1, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0x7a, 0x0a], "vcvtuqq2ps xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0x7a, 0x4a, 0x01], "vcvtuqq2ps xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0x7a, 0xca], "vcvtuqq2ps xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0x7f, 0x0a], "vmovdqu16 xmmword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0x7f, 0x4a, 0x01], "vmovdqu16 xmmword [bp + si + 0x10], xmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0x7f, 0xca], "vmovdqu16 xmm2, xmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0xe6, 0x0a], "vcvtpd2dq xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0xe6, 0x4a, 0x01], "vcvtpd2dq xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xff, 0x08, 0xe6, 0xca], "vcvtpd2dq xmm1, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0x12, 0x0a], "vmovddup xmm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0x12, 0x4a, 0x01], "vmovddup xmm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0x12, 0xca], "vmovddup xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0x6f, 0x0a], "vmovdqu16 xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0x6f, 0x4a, 0x01], "vmovdqu16 xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0x6f, 0xca], "vmovdqu16 xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0x70, 0x0a, 0xcc], "vpshuflw xmm1{k5}, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0x70, 0x4a, 0x01, 0xcc], "vpshuflw xmm1{k5}, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0x70, 0xca, 0xcc], "vpshuflw xmm1{k5}, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0x7a, 0x0a], "vcvtuqq2ps xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0x7a, 0x4a, 0x01], "vcvtuqq2ps xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0x7a, 0xca], "vcvtuqq2ps xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0x7f, 0x0a], "vmovdqu16 xmmword [bp + si]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0x7f, 0x4a, 0x01], "vmovdqu16 xmmword [bp + si + 0x10]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0x7f, 0xca], "vmovdqu16 xmm2{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0xe6, 0x0a], "vcvtpd2dq xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0xe6, 0x4a, 0x01], "vcvtpd2dq xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xff, 0x0d, 0xe6, 0xca], "vcvtpd2dq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x18, 0x2d, 0xca], "vcvtsd2si ecx{rne-sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x18, 0x51, 0xca], "vsqrtsd xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x18, 0x58, 0xca], "vaddsd xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x18, 0x59, 0xca], "vmulsd xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x18, 0x5a, 0xca], "vcvtsd2ss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x18, 0x5c, 0xca], "vsubsd xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x18, 0x5e, 0xca], "vdivsd xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x18, 0x79, 0xca], "vcvtsd2usi ecx{rne-sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x18, 0x7a, 0x0a], "vcvtuqq2ps xmm1, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xff, 0x18, 0x7a, 0x4a, 0x01], "vcvtuqq2ps xmm1, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xff, 0x18, 0x7a, 0xca], "vcvtuqq2ps ymm1{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x18, 0xe6, 0x0a], "vcvtpd2dq xmm1, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xff, 0x18, 0xe6, 0x4a, 0x01], "vcvtpd2dq xmm1, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xff, 0x18, 0xe6, 0xca], "vcvtpd2dq ymm1{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x1d, 0x51, 0xca], "vsqrtsd xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x1d, 0x58, 0xca], "vaddsd xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x1d, 0x59, 0xca], "vmulsd xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x1d, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x1d, 0x5c, 0xca], "vsubsd xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x1d, 0x5e, 0xca], "vdivsd xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x1d, 0x7a, 0x0a], "vcvtuqq2ps xmm1{k5}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xff, 0x1d, 0x7a, 0x4a, 0x01], "vcvtuqq2ps xmm1{k5}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xff, 0x1d, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x1d, 0xe6, 0x0a], "vcvtpd2dq xmm1{k5}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xff, 0x1d, 0xe6, 0x4a, 0x01], "vcvtpd2dq xmm1{k5}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xff, 0x1d, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x10, 0x0a], "vmovsd xmm1, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x10, 0x4a, 0x01], "vmovsd xmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x10, 0xca], "vmovsd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x11, 0x0a], "vmovsd qword [bp + si], xmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x11, 0x4a, 0x01], "vmovsd qword [bp + si + 0x8], xmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x11, 0xca], "vmovsd xmm2, xmm0, xmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x12, 0x0a], "vmovddup ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x12, 0x4a, 0x01], "vmovddup ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x12, 0xca], "vmovddup ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x2a, 0x0a], "vcvtsi2sd xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x2a, 0x4a, 0x01], "vcvtsi2sd xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x2c, 0x0a], "vcvttsd2si ecx, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x2c, 0x4a, 0x01], "vcvttsd2si ecx, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x2c, 0xca], "vcvttsd2si ecx, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x2d, 0x0a], "vcvtsd2si ecx, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x2d, 0x4a, 0x01], "vcvtsd2si ecx, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x2d, 0xca], "vcvtsd2si ecx, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x51, 0x0a], "vsqrtsd xmm1, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x51, 0x4a, 0x01], "vsqrtsd xmm1, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x51, 0xca], "vsqrtsd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x58, 0x0a], "vaddsd xmm1, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x58, 0x4a, 0x01], "vaddsd xmm1, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x58, 0xca], "vaddsd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x59, 0x0a], "vmulsd xmm1, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x59, 0x4a, 0x01], "vmulsd xmm1, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x59, 0xca], "vmulsd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x5a, 0x0a], "vcvtsd2ss xmm1, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x5a, 0x4a, 0x01], "vcvtsd2ss xmm1, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x5a, 0xca], "vcvtsd2ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x5c, 0x0a], "vsubsd xmm1, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x5c, 0x4a, 0x01], "vsubsd xmm1, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x5c, 0xca], "vsubsd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x5d, 0x0a], "vminsd xmm1, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x5d, 0x4a, 0x01], "vminsd xmm1, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x5d, 0xca], "vminsd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x5e, 0x0a], "vdivsd xmm1, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x5e, 0x4a, 0x01], "vdivsd xmm1, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x5e, 0xca], "vdivsd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x5f, 0x0a], "vmaxsd xmm1, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x5f, 0x4a, 0x01], "vmaxsd xmm1, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x5f, 0xca], "vmaxsd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x6f, 0x0a], "vmovdqu16 ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x6f, 0x4a, 0x01], "vmovdqu16 ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x6f, 0xca], "vmovdqu16 ymm1, ymm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x70, 0x0a, 0xcc], "vpshuflw ymm1, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x70, 0x4a, 0x01, 0xcc], "vpshuflw ymm1, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x70, 0xca, 0xcc], "vpshuflw ymm1, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x78, 0x0a], "vcvttsd2usi ecx, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x78, 0x4a, 0x01], "vcvttsd2usi ecx, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x78, 0xca], "vcvttsd2usi ecx, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x79, 0x0a], "vcvtsd2usi ecx, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x79, 0x4a, 0x01], "vcvtsd2usi ecx, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x79, 0xca], "vcvtsd2usi ecx, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x7a, 0x0a], "vcvtuqq2ps xmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x7a, 0x4a, 0x01], "vcvtuqq2ps xmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x7a, 0xca], "vcvtuqq2ps xmm1, ymm2");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x7b, 0x0a], "vcvtusi2sd xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x7b, 0x4a, 0x01], "vcvtusi2sd xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x7f, 0x0a], "vmovdqu16 ymmword [bp + si], ymm1");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x7f, 0x4a, 0x01], "vmovdqu16 ymmword [bp + si + 0x20], ymm1");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0x7f, 0xca], "vmovdqu16 ymm2, ymm1");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0xc2, 0x0a, 0xcc], "vcmpsd k1, xmm0, qword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0xc2, 0x4a, 0x01, 0xcc], "vcmpsd k1, xmm0, qword [bp + si + 0x8], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0xc2, 0xca, 0xcc], "vcmpsd k1, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0xe6, 0x0a], "vcvtpd2dq xmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0xe6, 0x4a, 0x01], "vcvtpd2dq xmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xff, 0x28, 0xe6, 0xca], "vcvtpd2dq xmm1, ymm2");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x10, 0x0a], "vmovsd xmm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x10, 0x4a, 0x01], "vmovsd xmm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x10, 0xca], "vmovsd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x11, 0x0a], "vmovsd qword [bp + si]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x11, 0x4a, 0x01], "vmovsd qword [bp + si + 0x8]{k5}, xmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x11, 0xca], "vmovsd xmm2{k5}, xmm0, xmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x12, 0x0a], "vmovddup ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x12, 0x4a, 0x01], "vmovddup ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x12, 0xca], "vmovddup ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x51, 0x0a], "vsqrtsd xmm1{k5}, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x51, 0x4a, 0x01], "vsqrtsd xmm1{k5}, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x51, 0xca], "vsqrtsd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x58, 0x0a], "vaddsd xmm1{k5}, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x58, 0x4a, 0x01], "vaddsd xmm1{k5}, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x58, 0xca], "vaddsd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x59, 0x0a], "vmulsd xmm1{k5}, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x59, 0x4a, 0x01], "vmulsd xmm1{k5}, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x59, 0xca], "vmulsd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x5a, 0x0a], "vcvtsd2ss xmm1{k5}, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x5a, 0x4a, 0x01], "vcvtsd2ss xmm1{k5}, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x5c, 0x0a], "vsubsd xmm1{k5}, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x5c, 0x4a, 0x01], "vsubsd xmm1{k5}, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x5c, 0xca], "vsubsd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x5d, 0x0a], "vminsd xmm1{k5}, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x5d, 0x4a, 0x01], "vminsd xmm1{k5}, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x5d, 0xca], "vminsd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x5e, 0x0a], "vdivsd xmm1{k5}, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x5e, 0x4a, 0x01], "vdivsd xmm1{k5}, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x5e, 0xca], "vdivsd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x5f, 0x0a], "vmaxsd xmm1{k5}, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x5f, 0x4a, 0x01], "vmaxsd xmm1{k5}, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x5f, 0xca], "vmaxsd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x6f, 0x0a], "vmovdqu16 ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x6f, 0x4a, 0x01], "vmovdqu16 ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x6f, 0xca], "vmovdqu16 ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x70, 0x0a, 0xcc], "vpshuflw ymm1{k5}, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x70, 0x4a, 0x01, 0xcc], "vpshuflw ymm1{k5}, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x70, 0xca, 0xcc], "vpshuflw ymm1{k5}, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x7a, 0x0a], "vcvtuqq2ps xmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x7a, 0x4a, 0x01], "vcvtuqq2ps xmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x7a, 0xca], "vcvtuqq2ps xmm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x7f, 0x0a], "vmovdqu16 ymmword [bp + si]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x7f, 0x4a, 0x01], "vmovdqu16 ymmword [bp + si + 0x20]{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0x7f, 0xca], "vmovdqu16 ymm2{k5}, ymm1");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0xc2, 0x0a, 0xcc], "vcmpsd k1{k5}, xmm0, qword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0xc2, 0x4a, 0x01, 0xcc], "vcmpsd k1{k5}, xmm0, qword [bp + si + 0x8], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0xc2, 0xca, 0xcc], "vcmpsd k1{k5}, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0xe6, 0x0a], "vcvtpd2dq xmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0xe6, 0x4a, 0x01], "vcvtpd2dq xmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xff, 0x2d, 0xe6, 0xca], "vcvtpd2dq xmm1{k5}, ymm2");
    test_display(&[0x62, 0xf1, 0xff, 0x38, 0x2d, 0xca], "vcvtsd2si ecx{rd-sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x38, 0x51, 0xca], "vsqrtsd xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x38, 0x58, 0xca], "vaddsd xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x38, 0x59, 0xca], "vmulsd xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x38, 0x5a, 0xca], "vcvtsd2ss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x38, 0x5c, 0xca], "vsubsd xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x38, 0x5e, 0xca], "vdivsd xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x38, 0x79, 0xca], "vcvtsd2usi ecx{rd-sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x38, 0x7a, 0x0a], "vcvtuqq2ps xmm1, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xff, 0x38, 0x7a, 0x4a, 0x01], "vcvtuqq2ps xmm1, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xff, 0x38, 0x7a, 0xca], "vcvtuqq2ps ymm1{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x38, 0xe6, 0x0a], "vcvtpd2dq xmm1, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xff, 0x38, 0xe6, 0x4a, 0x01], "vcvtpd2dq xmm1, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xff, 0x38, 0xe6, 0xca], "vcvtpd2dq ymm1{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x3d, 0x51, 0xca], "vsqrtsd xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x3d, 0x58, 0xca], "vaddsd xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x3d, 0x59, 0xca], "vmulsd xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x3d, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x3d, 0x5c, 0xca], "vsubsd xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x3d, 0x5e, 0xca], "vdivsd xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x3d, 0x7a, 0x0a], "vcvtuqq2ps xmm1{k5}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xff, 0x3d, 0x7a, 0x4a, 0x01], "vcvtuqq2ps xmm1{k5}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xff, 0x3d, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x3d, 0xe6, 0x0a], "vcvtpd2dq xmm1{k5}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xff, 0x3d, 0xe6, 0x4a, 0x01], "vcvtpd2dq xmm1{k5}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xff, 0x3d, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0x12, 0x0a], "vmovddup zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0x12, 0x4a, 0x01], "vmovddup zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0x12, 0xca], "vmovddup zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0x6f, 0x0a], "vmovdqu16 zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0x6f, 0x4a, 0x01], "vmovdqu16 zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0x6f, 0xca], "vmovdqu16 zmm1, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0x70, 0x0a, 0xcc], "vpshuflw zmm1, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0x70, 0x4a, 0x01, 0xcc], "vpshuflw zmm1, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0x70, 0xca, 0xcc], "vpshuflw zmm1, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0x7a, 0x0a], "vcvtuqq2ps ymm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0x7a, 0x4a, 0x01], "vcvtuqq2ps ymm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0x7a, 0xca], "vcvtuqq2ps ymm1, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0x7f, 0x0a], "vmovdqu16 zmmword [bp + si], zmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0x7f, 0x4a, 0x01], "vmovdqu16 zmmword [bp + si + 0x40], zmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0x7f, 0xca], "vmovdqu16 zmm2, zmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0xe6, 0x0a], "vcvtpd2dq ymm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0xe6, 0x4a, 0x01], "vcvtpd2dq ymm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xff, 0x48, 0xe6, 0xca], "vcvtpd2dq ymm1, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0x12, 0x0a], "vmovddup zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0x12, 0x4a, 0x01], "vmovddup zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0x12, 0xca], "vmovddup zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0x6f, 0x0a], "vmovdqu16 zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0x6f, 0x4a, 0x01], "vmovdqu16 zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0x6f, 0xca], "vmovdqu16 zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0x70, 0x0a, 0xcc], "vpshuflw zmm1{k5}, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0x70, 0x4a, 0x01, 0xcc], "vpshuflw zmm1{k5}, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0x70, 0xca, 0xcc], "vpshuflw zmm1{k5}, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0x7a, 0x0a], "vcvtuqq2ps ymm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0x7a, 0x4a, 0x01], "vcvtuqq2ps ymm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0x7f, 0x0a], "vmovdqu16 zmmword [bp + si]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0x7f, 0x4a, 0x01], "vmovdqu16 zmmword [bp + si + 0x40]{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0x7f, 0xca], "vmovdqu16 zmm2{k5}, zmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0xe6, 0x0a], "vcvtpd2dq ymm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0xe6, 0x4a, 0x01], "vcvtpd2dq ymm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xff, 0x4d, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x58, 0x2d, 0xca], "vcvtsd2si ecx{ru-sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x58, 0x51, 0xca], "vsqrtsd xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x58, 0x58, 0xca], "vaddsd xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x58, 0x59, 0xca], "vmulsd xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x58, 0x5a, 0xca], "vcvtsd2ss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x58, 0x5c, 0xca], "vsubsd xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x58, 0x5e, 0xca], "vdivsd xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x58, 0x79, 0xca], "vcvtsd2usi ecx{ru-sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x58, 0x7a, 0x0a], "vcvtuqq2ps ymm1, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xff, 0x58, 0x7a, 0x4a, 0x01], "vcvtuqq2ps ymm1, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xff, 0x58, 0x7a, 0xca], "vcvtuqq2ps ymm1{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x58, 0xe6, 0x0a], "vcvtpd2dq ymm1, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xff, 0x58, 0xe6, 0x4a, 0x01], "vcvtpd2dq ymm1, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xff, 0x58, 0xe6, 0xca], "vcvtpd2dq ymm1{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x5d, 0x51, 0xca], "vsqrtsd xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x5d, 0x58, 0xca], "vaddsd xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x5d, 0x59, 0xca], "vmulsd xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x5d, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x5d, 0x5c, 0xca], "vsubsd xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x5d, 0x5e, 0xca], "vdivsd xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x5d, 0x7a, 0x0a], "vcvtuqq2ps ymm1{k5}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xff, 0x5d, 0x7a, 0x4a, 0x01], "vcvtuqq2ps ymm1{k5}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xff, 0x5d, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x5d, 0xe6, 0x0a], "vcvtpd2dq ymm1{k5}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xff, 0x5d, 0xe6, 0x4a, 0x01], "vcvtpd2dq ymm1{k5}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xff, 0x5d, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0x2a, 0xca], "vcvtsi2sd xmm1, xmm0, edx");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0x2c, 0xca], "vcvttsd2si ecx{sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0x2d, 0xca], "vcvtsd2si ecx{rz-sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0x51, 0xca], "vsqrtsd xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0x58, 0xca], "vaddsd xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0x59, 0xca], "vmulsd xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0x5a, 0xca], "vcvtsd2ss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0x5c, 0xca], "vsubsd xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0x5d, 0xca], "vminsd xmm1{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0x5e, 0xca], "vdivsd xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0x5f, 0xca], "vmaxsd xmm1{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0x78, 0xca], "vcvttsd2usi ecx{sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0x79, 0xca], "vcvtsd2usi ecx{rz-sae}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0x7a, 0xca], "vcvtuqq2ps ymm1{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0x7b, 0xca], "vcvtusi2sd xmm1, xmm0, edx");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0xc2, 0xca, 0xcc], "vcmpsd k1{sae}, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x78, 0xe6, 0xca], "vcvtpd2dq ymm1{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x7d, 0x51, 0xca], "vsqrtsd xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x7d, 0x58, 0xca], "vaddsd xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x7d, 0x59, 0xca], "vmulsd xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x7d, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x7d, 0x5c, 0xca], "vsubsd xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x7d, 0x5d, 0xca], "vminsd xmm1{k5}{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x7d, 0x5e, 0xca], "vdivsd xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x7d, 0x5f, 0xca], "vmaxsd xmm1{k5}{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x7d, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x7d, 0xc2, 0xca, 0xcc], "vcmpsd k1{k5}{sae}, xmm0, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x7d, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x8d, 0x12, 0x0a], "vmovddup xmm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x8d, 0x12, 0x4a, 0x01], "vmovddup xmm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0x8d, 0x12, 0xca], "vmovddup xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x8d, 0x6f, 0x0a], "vmovdqu16 xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x8d, 0x6f, 0x4a, 0x01], "vmovdqu16 xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xff, 0x8d, 0x6f, 0xca], "vmovdqu16 xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x8d, 0x70, 0x0a, 0xcc], "vpshuflw xmm1{k5}{z}, xmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x8d, 0x70, 0x4a, 0x01, 0xcc], "vpshuflw xmm1{k5}{z}, xmmword [bp + si + 0x10], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x8d, 0x70, 0xca, 0xcc], "vpshuflw xmm1{k5}{z}, xmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0x8d, 0x7a, 0x0a], "vcvtuqq2ps xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x8d, 0x7a, 0x4a, 0x01], "vcvtuqq2ps xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xff, 0x8d, 0x7a, 0xca], "vcvtuqq2ps xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x8d, 0x7f, 0xca], "vmovdqu16 xmm2{k5}{z}, xmm1");
    test_display(&[0x62, 0xf1, 0xff, 0x8d, 0xe6, 0x0a], "vcvtpd2dq xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0x8d, 0xe6, 0x4a, 0x01], "vcvtpd2dq xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf1, 0xff, 0x8d, 0xe6, 0xca], "vcvtpd2dq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x9d, 0x51, 0xca], "vsqrtsd xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x9d, 0x58, 0xca], "vaddsd xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x9d, 0x59, 0xca], "vmulsd xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x9d, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x9d, 0x5c, 0xca], "vsubsd xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x9d, 0x5e, 0xca], "vdivsd xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x9d, 0x7a, 0x0a], "vcvtuqq2ps xmm1{k5}{z}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xff, 0x9d, 0x7a, 0x4a, 0x01], "vcvtuqq2ps xmm1{k5}{z}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xff, 0x9d, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{z}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0x9d, 0xe6, 0x0a], "vcvtpd2dq xmm1{k5}{z}, qword [bp + si]{1to2}");
    test_display(&[0x62, 0xf1, 0xff, 0x9d, 0xe6, 0x4a, 0x01], "vcvtpd2dq xmm1{k5}{z}, qword [bp + si + 0x8]{1to2}");
    test_display(&[0x62, 0xf1, 0xff, 0x9d, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{z}{rne-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x10, 0x0a], "vmovsd xmm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x10, 0x4a, 0x01], "vmovsd xmm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x10, 0xca], "vmovsd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x11, 0xca], "vmovsd xmm2{k5}{z}, xmm0, xmm1");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x12, 0x0a], "vmovddup ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x12, 0x4a, 0x01], "vmovddup ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x12, 0xca], "vmovddup ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x51, 0x0a], "vsqrtsd xmm1{k5}{z}, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x51, 0x4a, 0x01], "vsqrtsd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x51, 0xca], "vsqrtsd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x58, 0x0a], "vaddsd xmm1{k5}{z}, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x58, 0x4a, 0x01], "vaddsd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x58, 0xca], "vaddsd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x59, 0x0a], "vmulsd xmm1{k5}{z}, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x59, 0x4a, 0x01], "vmulsd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x59, 0xca], "vmulsd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x5a, 0x0a], "vcvtsd2ss xmm1{k5}{z}, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x5a, 0x4a, 0x01], "vcvtsd2ss xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x5c, 0x0a], "vsubsd xmm1{k5}{z}, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x5c, 0x4a, 0x01], "vsubsd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x5c, 0xca], "vsubsd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x5d, 0x0a], "vminsd xmm1{k5}{z}, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x5d, 0x4a, 0x01], "vminsd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x5d, 0xca], "vminsd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x5e, 0x0a], "vdivsd xmm1{k5}{z}, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x5e, 0x4a, 0x01], "vdivsd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x5e, 0xca], "vdivsd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x5f, 0x0a], "vmaxsd xmm1{k5}{z}, xmm0, qword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x5f, 0x4a, 0x01], "vmaxsd xmm1{k5}{z}, xmm0, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x5f, 0xca], "vmaxsd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x6f, 0x0a], "vmovdqu16 ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x6f, 0x4a, 0x01], "vmovdqu16 ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x6f, 0xca], "vmovdqu16 ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x70, 0x0a, 0xcc], "vpshuflw ymm1{k5}{z}, ymmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x70, 0x4a, 0x01, 0xcc], "vpshuflw ymm1{k5}{z}, ymmword [bp + si + 0x20], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x70, 0xca, 0xcc], "vpshuflw ymm1{k5}{z}, ymm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x7a, 0x0a], "vcvtuqq2ps xmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x7a, 0x4a, 0x01], "vcvtuqq2ps xmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x7a, 0xca], "vcvtuqq2ps xmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0x7f, 0xca], "vmovdqu16 ymm2{k5}{z}, ymm1");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0xe6, 0x0a], "vcvtpd2dq xmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0xe6, 0x4a, 0x01], "vcvtpd2dq xmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf1, 0xff, 0xad, 0xe6, 0xca], "vcvtpd2dq xmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf1, 0xff, 0xbd, 0x51, 0xca], "vsqrtsd xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xbd, 0x58, 0xca], "vaddsd xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xbd, 0x59, 0xca], "vmulsd xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xbd, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xbd, 0x5c, 0xca], "vsubsd xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xbd, 0x5e, 0xca], "vdivsd xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xbd, 0x7a, 0x0a], "vcvtuqq2ps xmm1{k5}{z}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xff, 0xbd, 0x7a, 0x4a, 0x01], "vcvtuqq2ps xmm1{k5}{z}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xff, 0xbd, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{z}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xbd, 0xe6, 0x0a], "vcvtpd2dq xmm1{k5}{z}, qword [bp + si]{1to4}");
    test_display(&[0x62, 0xf1, 0xff, 0xbd, 0xe6, 0x4a, 0x01], "vcvtpd2dq xmm1{k5}{z}, qword [bp + si + 0x8]{1to4}");
    test_display(&[0x62, 0xf1, 0xff, 0xbd, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{z}{rd-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xcd, 0x12, 0x0a], "vmovddup zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xcd, 0x12, 0x4a, 0x01], "vmovddup zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xff, 0xcd, 0x12, 0xca], "vmovddup zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xcd, 0x6f, 0x0a], "vmovdqu16 zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xcd, 0x6f, 0x4a, 0x01], "vmovdqu16 zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xff, 0xcd, 0x6f, 0xca], "vmovdqu16 zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xcd, 0x70, 0x0a, 0xcc], "vpshuflw zmm1{k5}{z}, zmmword [bp + si], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0xcd, 0x70, 0x4a, 0x01, 0xcc], "vpshuflw zmm1{k5}{z}, zmmword [bp + si + 0x40], 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0xcd, 0x70, 0xca, 0xcc], "vpshuflw zmm1{k5}{z}, zmm2, 0xcc");
    test_display(&[0x62, 0xf1, 0xff, 0xcd, 0x7a, 0x0a], "vcvtuqq2ps ymm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xcd, 0x7a, 0x4a, 0x01], "vcvtuqq2ps ymm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xff, 0xcd, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xcd, 0x7f, 0xca], "vmovdqu16 zmm2{k5}{z}, zmm1");
    test_display(&[0x62, 0xf1, 0xff, 0xcd, 0xe6, 0x0a], "vcvtpd2dq ymm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf1, 0xff, 0xcd, 0xe6, 0x4a, 0x01], "vcvtpd2dq ymm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf1, 0xff, 0xcd, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xdd, 0x51, 0xca], "vsqrtsd xmm1{k5}{z}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xdd, 0x58, 0xca], "vaddsd xmm1{k5}{z}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xdd, 0x59, 0xca], "vmulsd xmm1{k5}{z}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xdd, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{z}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xdd, 0x5c, 0xca], "vsubsd xmm1{k5}{z}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xdd, 0x5e, 0xca], "vdivsd xmm1{k5}{z}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xdd, 0x7a, 0x0a], "vcvtuqq2ps ymm1{k5}{z}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xff, 0xdd, 0x7a, 0x4a, 0x01], "vcvtuqq2ps ymm1{k5}{z}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xff, 0xdd, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{z}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xdd, 0xe6, 0x0a], "vcvtpd2dq ymm1{k5}{z}, qword [bp + si]{1to8}");
    test_display(&[0x62, 0xf1, 0xff, 0xdd, 0xe6, 0x4a, 0x01], "vcvtpd2dq ymm1{k5}{z}, qword [bp + si + 0x8]{1to8}");
    test_display(&[0x62, 0xf1, 0xff, 0xdd, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{z}{ru-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xfd, 0x51, 0xca], "vsqrtsd xmm1{k5}{z}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xfd, 0x58, 0xca], "vaddsd xmm1{k5}{z}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xfd, 0x59, 0xca], "vmulsd xmm1{k5}{z}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xfd, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{z}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xfd, 0x5c, 0xca], "vsubsd xmm1{k5}{z}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xfd, 0x5d, 0xca], "vminsd xmm1{k5}{z}{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xfd, 0x5e, 0xca], "vdivsd xmm1{k5}{z}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xfd, 0x5f, 0xca], "vmaxsd xmm1{k5}{z}{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xfd, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{z}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf1, 0xff, 0xfd, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{z}{rz-sae}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x0c, 0x0a], "vpermilps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x0c, 0x4a, 0x01], "vpermilps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x0c, 0xca], "vpermilps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x13, 0x0a], "vcvtph2ps xmm1, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x13, 0x4a, 0x01], "vcvtph2ps xmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x13, 0xca], "vcvtph2ps xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x14, 0x0a], "vprorvd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x14, 0x4a, 0x01], "vprorvd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x14, 0xca], "vprorvd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x15, 0x0a], "vprolvd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x15, 0x4a, 0x01], "vprolvd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x15, 0xca], "vprolvd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x18, 0x0a], "vbroadcastss xmm1, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x18, 0x4a, 0x01], "vbroadcastss xmm1, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x18, 0xca], "vbroadcastss xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x1e, 0x0a], "vpabsd xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x1e, 0x4a, 0x01], "vpabsd xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x1e, 0xca], "vpabsd xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x25, 0x0a], "vpmovsxdq xmm1, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x25, 0x4a, 0x01], "vpmovsxdq xmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x25, 0xca], "vpmovsxdq xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x26, 0x0a], "vptestmb k1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x26, 0x4a, 0x01], "vptestmb k1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x26, 0xca], "vptestmb k1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x27, 0x0a], "vptestmd k1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x27, 0x4a, 0x01], "vptestmd k1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x27, 0xca], "vptestmd k1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x2a, 0x0a], "vmovntdqa xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x2a, 0x4a, 0x01], "vmovntdqa xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x2b, 0x0a], "vpackusdw xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x2b, 0x4a, 0x01], "vpackusdw xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x2b, 0xca], "vpackusdw xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x2c, 0x0a], "vscalefps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x2c, 0x4a, 0x01], "vscalefps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x2c, 0xca], "vscalefps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x35, 0x0a], "vpmovzxdq xmm1, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x35, 0x4a, 0x01], "vpmovzxdq xmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x35, 0xca], "vpmovzxdq xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x39, 0x0a], "vpminsd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x39, 0x4a, 0x01], "vpminsd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x39, 0xca], "vpminsd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x3b, 0x0a], "vpminud xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x3b, 0x4a, 0x01], "vpminud xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x3b, 0xca], "vpminud xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x3d, 0x0a], "vpmaxsd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x3d, 0x4a, 0x01], "vpmaxsd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x3d, 0xca], "vpmaxsd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x3f, 0x0a], "vpmaxud xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x3f, 0x4a, 0x01], "vpmaxud xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x3f, 0xca], "vpmaxud xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x40, 0x0a], "vpmulld xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x40, 0x4a, 0x01], "vpmulld xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x40, 0xca], "vpmulld xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x42, 0x0a], "vgetexpps xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x42, 0x4a, 0x01], "vgetexpps xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x42, 0xca], "vgetexpps xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x44, 0x0a], "vplzcntd xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x44, 0x4a, 0x01], "vplzcntd xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x44, 0xca], "vplzcntd xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x45, 0x0a], "vpsrlvd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x45, 0x4a, 0x01], "vpsrlvd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x45, 0xca], "vpsrlvd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x46, 0x0a], "vpsravd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x46, 0x4a, 0x01], "vpsravd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x46, 0xca], "vpsravd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x47, 0x0a], "vpsllvd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x47, 0x4a, 0x01], "vpsllvd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x47, 0xca], "vpsllvd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x4c, 0x0a], "vrcp14ps xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x4c, 0x4a, 0x01], "vrcp14ps xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x4c, 0xca], "vrcp14ps xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x4e, 0x0a], "vrsqrt14ps xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x4e, 0x4a, 0x01], "vrsqrt14ps xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x4e, 0xca], "vrsqrt14ps xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x50, 0x0a], "vpdpbusd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x50, 0x4a, 0x01], "vpdpbusd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x50, 0xca], "vpdpbusd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x51, 0x0a], "vpdpbusds xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x51, 0x4a, 0x01], "vpdpbusds xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x51, 0xca], "vpdpbusds xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x52, 0x0a], "vpdpwssd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x52, 0x4a, 0x01], "vpdpwssd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x52, 0xca], "vpdpwssd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x53, 0x0a], "vpdpwssds xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x53, 0x4a, 0x01], "vpdpwssds xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x53, 0xca], "vpdpwssds xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x54, 0x0a], "vpopcntb xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x54, 0x4a, 0x01], "vpopcntb xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x54, 0xca], "vpopcntb xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x55, 0x0a], "vpopcntd xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x55, 0x4a, 0x01], "vpopcntd xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x55, 0xca], "vpopcntd xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x58, 0x0a], "vpbroadcastd xmm1, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x58, 0x4a, 0x01], "vpbroadcastd xmm1, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x58, 0xca], "vpbroadcastd xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x59, 0x0a], "vbroadcasti32x2 xmm1, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x59, 0x4a, 0x01], "vbroadcasti32x2 xmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x59, 0xca], "vbroadcasti32x2 xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x62, 0x0a], "vpexpandb xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x62, 0x4a, 0x01], "vpexpandb xmm1, xmmword [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x62, 0xca], "vpexpandb xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x63, 0x0a], "vpcompressb xmmword [bp + si], xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x63, 0x4a, 0x01], "vpcompressb xmmword [bp + si + 0x1], xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x63, 0xca], "vpcompressb xmm2, xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x64, 0x0a], "vpblendmd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x64, 0x4a, 0x01], "vpblendmd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x64, 0xca], "vpblendmd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x65, 0x0a], "vblendmps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x65, 0x4a, 0x01], "vblendmps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x65, 0xca], "vblendmps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x66, 0x0a], "vpblendmb xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x66, 0x4a, 0x01], "vpblendmb xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x66, 0xca], "vpblendmb xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x71, 0x0a], "vpshldvd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x71, 0x4a, 0x01], "vpshldvd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x71, 0xca], "vpshldvd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x73, 0x0a], "vpshrdvd xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x73, 0x4a, 0x01], "vpshrdvd xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x73, 0xca], "vpshrdvd xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x75, 0x0a], "vpermi2b xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x75, 0x4a, 0x01], "vpermi2b xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x75, 0xca], "vpermi2b xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x76, 0x0a], "vpermi2d xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x76, 0x4a, 0x01], "vpermi2d xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x76, 0xca], "vpermi2d xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x77, 0x0a], "vpermi2ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x77, 0x4a, 0x01], "vpermi2ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x77, 0xca], "vpermi2ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x78, 0x0a], "vpbroadcastb xmm1, byte [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x78, 0x4a, 0x01], "vpbroadcastb xmm1, byte [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x78, 0xca], "vpbroadcastb xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x79, 0x0a], "vpbroadcastw xmm1, word [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x79, 0x4a, 0x01], "vpbroadcastw xmm1, word [bp + si + 0x2]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x79, 0xca], "vpbroadcastw xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x7a, 0xca], "vpbroadcastb xmm1, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x7b, 0xca], "vpbroadcastw xmm1, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x7d, 0x0a], "vpermt2b xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x7d, 0x4a, 0x01], "vpermt2b xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x7d, 0xca], "vpermt2b xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x7e, 0x0a], "vpermt2d xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x7e, 0x4a, 0x01], "vpermt2d xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x7e, 0xca], "vpermt2d xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x7f, 0x0a], "vpermt2ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x7f, 0x4a, 0x01], "vpermt2ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x7f, 0xca], "vpermt2ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x88, 0x0a], "vexpandps xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x88, 0x4a, 0x01], "vexpandps xmm1, xmmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x88, 0xca], "vexpandps xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x89, 0x0a], "vpexpandd xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x89, 0x4a, 0x01], "vpexpandd xmm1, xmmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x89, 0xca], "vpexpandd xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x8a, 0x0a], "vcompressps xmmword [bp + si], xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x8a, 0x4a, 0x01], "vcompressps xmmword [bp + si + 0x4], xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x8a, 0xca], "vcompressps xmm2, xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x8b, 0x0a], "vpcompressd xmmword [bp + si], xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x8b, 0x4a, 0x01], "vpcompressd xmmword [bp + si + 0x4], xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x8b, 0xca], "vpcompressd xmm2, xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x8d, 0x0a], "vpermb xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x8d, 0x4a, 0x01], "vpermb xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x8d, 0xca], "vpermb xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x8f, 0x0a], "vpshufbitqmb k1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x8f, 0x4a, 0x01], "vpshufbitqmb k1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x8f, 0xca], "vpshufbitqmb k1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x96, 0x0a], "vfmaddsub132ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x96, 0x4a, 0x01], "vfmaddsub132ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x96, 0xca], "vfmaddsub132ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x97, 0x0a], "vfmsubadd132ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x97, 0x4a, 0x01], "vfmsubadd132ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x97, 0xca], "vfmsubadd132ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x98, 0x0a], "vfmadd132ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x98, 0x4a, 0x01], "vfmadd132ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x98, 0xca], "vfmadd132ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x9a, 0x0a], "vfmsub132ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x9a, 0x4a, 0x01], "vfmsub132ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x9a, 0xca], "vfmsub132ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x9c, 0x0a], "vfnmadd132ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x9c, 0x4a, 0x01], "vfnmadd132ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x9c, 0xca], "vfnmadd132ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x9e, 0x0a], "vfnmsub132ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x9e, 0x4a, 0x01], "vfnmsub132ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0x9e, 0xca], "vfnmsub132ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xa6, 0x0a], "vfmaddsub213ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xa6, 0x4a, 0x01], "vfmaddsub213ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xa6, 0xca], "vfmaddsub213ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xa7, 0x0a], "vfmsubadd213ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xa7, 0x4a, 0x01], "vfmsubadd213ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xa7, 0xca], "vfmsubadd213ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xa8, 0x0a], "vfmadd213ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xa8, 0x4a, 0x01], "vfmadd213ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xa8, 0xca], "vfmadd213ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xaa, 0x0a], "vfmsub213ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xaa, 0x4a, 0x01], "vfmsub213ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xaa, 0xca], "vfmsub213ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xac, 0x0a], "vfnmadd213ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xac, 0x4a, 0x01], "vfnmadd213ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xac, 0xca], "vfnmadd213ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xae, 0x0a], "vfnmsub213ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xae, 0x4a, 0x01], "vfnmsub213ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xae, 0xca], "vfnmsub213ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xb6, 0x0a], "vfmaddsub231ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xb6, 0x4a, 0x01], "vfmaddsub231ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xb6, 0xca], "vfmaddsub231ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xb7, 0x0a], "vfmsubadd231ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xb7, 0x4a, 0x01], "vfmsubadd231ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xb7, 0xca], "vfmsubadd231ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xb8, 0x0a], "vfmadd231ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xb8, 0x4a, 0x01], "vfmadd231ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xb8, 0xca], "vfmadd231ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xba, 0x0a], "vfmsub231ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xba, 0x4a, 0x01], "vfmsub231ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xba, 0xca], "vfmsub231ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xbc, 0x0a], "vfnmadd231ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xbc, 0x4a, 0x01], "vfnmadd231ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xbc, 0xca], "vfnmadd231ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xbe, 0x0a], "vfnmsub231ps xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xbe, 0x4a, 0x01], "vfnmsub231ps xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xbe, 0xca], "vfnmsub231ps xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xc4, 0x0a], "vpconflictd xmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xc4, 0x4a, 0x01], "vpconflictd xmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xc4, 0xca], "vpconflictd xmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xcf, 0x0a], "vgf2p8mulb xmm1, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xcf, 0x4a, 0x01], "vgf2p8mulb xmm1, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x08, 0xcf, 0xca], "vgf2p8mulb xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x0c, 0x0a], "vpermilps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x0c, 0x4a, 0x01], "vpermilps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x0c, 0xca], "vpermilps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x13, 0x0a], "vcvtph2ps xmm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x13, 0x4a, 0x01], "vcvtph2ps xmm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x13, 0xca], "vcvtph2ps xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x14, 0x0a], "vprorvd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x14, 0x4a, 0x01], "vprorvd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x14, 0xca], "vprorvd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x15, 0x0a], "vprolvd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x15, 0x4a, 0x01], "vprolvd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x15, 0xca], "vprolvd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x18, 0x0a], "vbroadcastss xmm1{k5}, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x18, 0x4a, 0x01], "vbroadcastss xmm1{k5}, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x18, 0xca], "vbroadcastss xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x1e, 0x0a], "vpabsd xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x1e, 0x4a, 0x01], "vpabsd xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x1e, 0xca], "vpabsd xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x25, 0x0a], "vpmovsxdq xmm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x25, 0x4a, 0x01], "vpmovsxdq xmm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x25, 0xca], "vpmovsxdq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x26, 0x0a], "vptestmb k1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x26, 0x4a, 0x01], "vptestmb k1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x26, 0xca], "vptestmb k1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x27, 0x0a], "vptestmd k1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x27, 0x4a, 0x01], "vptestmd k1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x27, 0xca], "vptestmd k1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x2b, 0x0a], "vpackusdw xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x2b, 0x4a, 0x01], "vpackusdw xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x2b, 0xca], "vpackusdw xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x2c, 0x0a], "vscalefps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x2c, 0x4a, 0x01], "vscalefps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x2c, 0xca], "vscalefps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x35, 0x0a], "vpmovzxdq xmm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x35, 0x4a, 0x01], "vpmovzxdq xmm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x35, 0xca], "vpmovzxdq xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x39, 0x0a], "vpminsd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x39, 0x4a, 0x01], "vpminsd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x39, 0xca], "vpminsd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x3b, 0x0a], "vpminud xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x3b, 0x4a, 0x01], "vpminud xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x3b, 0xca], "vpminud xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x3d, 0x0a], "vpmaxsd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x3d, 0x4a, 0x01], "vpmaxsd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x3d, 0xca], "vpmaxsd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x3f, 0x0a], "vpmaxud xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x3f, 0x4a, 0x01], "vpmaxud xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x3f, 0xca], "vpmaxud xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x40, 0x0a], "vpmulld xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x40, 0x4a, 0x01], "vpmulld xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x40, 0xca], "vpmulld xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x42, 0x0a], "vgetexpps xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x42, 0x4a, 0x01], "vgetexpps xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x42, 0xca], "vgetexpps xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x44, 0x0a], "vplzcntd xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x44, 0x4a, 0x01], "vplzcntd xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x44, 0xca], "vplzcntd xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x45, 0x0a], "vpsrlvd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x45, 0x4a, 0x01], "vpsrlvd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x45, 0xca], "vpsrlvd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x46, 0x0a], "vpsravd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x46, 0x4a, 0x01], "vpsravd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x46, 0xca], "vpsravd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x47, 0x0a], "vpsllvd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x47, 0x4a, 0x01], "vpsllvd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x47, 0xca], "vpsllvd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x4c, 0x0a], "vrcp14ps xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x4c, 0x4a, 0x01], "vrcp14ps xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x4c, 0xca], "vrcp14ps xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x4e, 0x0a], "vrsqrt14ps xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x4e, 0x4a, 0x01], "vrsqrt14ps xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x4e, 0xca], "vrsqrt14ps xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x50, 0x0a], "vpdpbusd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x50, 0x4a, 0x01], "vpdpbusd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x50, 0xca], "vpdpbusd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x51, 0x0a], "vpdpbusds xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x51, 0x4a, 0x01], "vpdpbusds xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x51, 0xca], "vpdpbusds xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x52, 0x0a], "vpdpwssd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x52, 0x4a, 0x01], "vpdpwssd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x52, 0xca], "vpdpwssd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x53, 0x0a], "vpdpwssds xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x53, 0x4a, 0x01], "vpdpwssds xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x53, 0xca], "vpdpwssds xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x54, 0x0a], "vpopcntb xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x54, 0x4a, 0x01], "vpopcntb xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x54, 0xca], "vpopcntb xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x55, 0x0a], "vpopcntd xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x55, 0x4a, 0x01], "vpopcntd xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x55, 0xca], "vpopcntd xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x58, 0x0a], "vpbroadcastd xmm1{k5}, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x58, 0x4a, 0x01], "vpbroadcastd xmm1{k5}, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x58, 0xca], "vpbroadcastd xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x59, 0x0a], "vbroadcasti32x2 xmm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x59, 0x4a, 0x01], "vbroadcasti32x2 xmm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x59, 0xca], "vbroadcasti32x2 xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x62, 0x0a], "vpexpandb xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x62, 0x4a, 0x01], "vpexpandb xmm1{k5}, xmmword [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x62, 0xca], "vpexpandb xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x63, 0x0a], "vpcompressb xmmword [bp + si]{k5}, xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x63, 0x4a, 0x01], "vpcompressb xmmword [bp + si + 0x1]{k5}, xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x63, 0xca], "vpcompressb xmm2{k5}, xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x64, 0x0a], "vpblendmd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x64, 0x4a, 0x01], "vpblendmd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x64, 0xca], "vpblendmd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x65, 0x0a], "vblendmps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x65, 0x4a, 0x01], "vblendmps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x65, 0xca], "vblendmps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x66, 0x0a], "vpblendmb xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x66, 0x4a, 0x01], "vpblendmb xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x66, 0xca], "vpblendmb xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x71, 0x0a], "vpshldvd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x71, 0x4a, 0x01], "vpshldvd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x71, 0xca], "vpshldvd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x73, 0x0a], "vpshrdvd xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x73, 0x4a, 0x01], "vpshrdvd xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x73, 0xca], "vpshrdvd xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x75, 0x0a], "vpermi2b xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x75, 0x4a, 0x01], "vpermi2b xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x75, 0xca], "vpermi2b xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x76, 0x0a], "vpermi2d xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x76, 0x4a, 0x01], "vpermi2d xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x76, 0xca], "vpermi2d xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x77, 0x0a], "vpermi2ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x77, 0x4a, 0x01], "vpermi2ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x77, 0xca], "vpermi2ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x78, 0x0a], "vpbroadcastb xmm1{k5}, byte [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x78, 0x4a, 0x01], "vpbroadcastb xmm1{k5}, byte [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x78, 0xca], "vpbroadcastb xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x79, 0x0a], "vpbroadcastw xmm1{k5}, word [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x79, 0x4a, 0x01], "vpbroadcastw xmm1{k5}, word [bp + si + 0x2]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x79, 0xca], "vpbroadcastw xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x7a, 0xca], "vpbroadcastb xmm1{k5}, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x7b, 0xca], "vpbroadcastw xmm1{k5}, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x7d, 0x0a], "vpermt2b xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x7d, 0x4a, 0x01], "vpermt2b xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x7d, 0xca], "vpermt2b xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x7e, 0x0a], "vpermt2d xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x7e, 0x4a, 0x01], "vpermt2d xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x7e, 0xca], "vpermt2d xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x7f, 0x0a], "vpermt2ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x7f, 0x4a, 0x01], "vpermt2ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x7f, 0xca], "vpermt2ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x88, 0x0a], "vexpandps xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x88, 0x4a, 0x01], "vexpandps xmm1{k5}, xmmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x88, 0xca], "vexpandps xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x89, 0x0a], "vpexpandd xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x89, 0x4a, 0x01], "vpexpandd xmm1{k5}, xmmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x89, 0xca], "vpexpandd xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x8a, 0x0a], "vcompressps xmmword [bp + si]{k5}, xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x8a, 0x4a, 0x01], "vcompressps xmmword [bp + si + 0x4]{k5}, xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x8a, 0xca], "vcompressps xmm2{k5}, xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x8b, 0x0a], "vpcompressd xmmword [bp + si]{k5}, xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x8b, 0x4a, 0x01], "vpcompressd xmmword [bp + si + 0x4]{k5}, xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x8b, 0xca], "vpcompressd xmm2{k5}, xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x8d, 0x0a], "vpermb xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x8d, 0x4a, 0x01], "vpermb xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x8d, 0xca], "vpermb xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x8f, 0x0a], "vpshufbitqmb k1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x8f, 0x4a, 0x01], "vpshufbitqmb k1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x8f, 0xca], "vpshufbitqmb k1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x96, 0x0a], "vfmaddsub132ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x96, 0x4a, 0x01], "vfmaddsub132ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x96, 0xca], "vfmaddsub132ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x97, 0x0a], "vfmsubadd132ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x97, 0x4a, 0x01], "vfmsubadd132ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x97, 0xca], "vfmsubadd132ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x98, 0x0a], "vfmadd132ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x98, 0x4a, 0x01], "vfmadd132ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x98, 0xca], "vfmadd132ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x9a, 0x0a], "vfmsub132ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x9a, 0x4a, 0x01], "vfmsub132ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x9a, 0xca], "vfmsub132ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x9c, 0x0a], "vfnmadd132ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x9c, 0x4a, 0x01], "vfnmadd132ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x9c, 0xca], "vfnmadd132ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x9e, 0x0a], "vfnmsub132ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x9e, 0x4a, 0x01], "vfnmsub132ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0x9e, 0xca], "vfnmsub132ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xa6, 0x0a], "vfmaddsub213ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xa6, 0x4a, 0x01], "vfmaddsub213ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xa6, 0xca], "vfmaddsub213ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xa7, 0x0a], "vfmsubadd213ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xa7, 0x4a, 0x01], "vfmsubadd213ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xa7, 0xca], "vfmsubadd213ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xa8, 0x0a], "vfmadd213ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xa8, 0x4a, 0x01], "vfmadd213ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xa8, 0xca], "vfmadd213ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xaa, 0x0a], "vfmsub213ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xaa, 0x4a, 0x01], "vfmsub213ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xaa, 0xca], "vfmsub213ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xac, 0x0a], "vfnmadd213ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xac, 0x4a, 0x01], "vfnmadd213ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xac, 0xca], "vfnmadd213ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xae, 0x0a], "vfnmsub213ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xae, 0x4a, 0x01], "vfnmsub213ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xae, 0xca], "vfnmsub213ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xb6, 0x0a], "vfmaddsub231ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xb6, 0x4a, 0x01], "vfmaddsub231ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xb6, 0xca], "vfmaddsub231ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xb7, 0x0a], "vfmsubadd231ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xb7, 0x4a, 0x01], "vfmsubadd231ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xb7, 0xca], "vfmsubadd231ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xb8, 0x0a], "vfmadd231ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xb8, 0x4a, 0x01], "vfmadd231ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xb8, 0xca], "vfmadd231ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xba, 0x0a], "vfmsub231ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xba, 0x4a, 0x01], "vfmsub231ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xba, 0xca], "vfmsub231ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xbc, 0x0a], "vfnmadd231ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xbc, 0x4a, 0x01], "vfnmadd231ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xbc, 0xca], "vfnmadd231ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xbe, 0x0a], "vfnmsub231ps xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xbe, 0x4a, 0x01], "vfnmsub231ps xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xbe, 0xca], "vfnmsub231ps xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xc4, 0x0a], "vpconflictd xmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xc4, 0x4a, 0x01], "vpconflictd xmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xc4, 0xca], "vpconflictd xmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xcf, 0x0a], "vgf2p8mulb xmm1{k5}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xcf, 0x4a, 0x01], "vgf2p8mulb xmm1{k5}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x0d, 0xcf, 0xca], "vgf2p8mulb xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x0c, 0x0a], "vpermilps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x0c, 0x4a, 0x01], "vpermilps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x14, 0x0a], "vprorvd xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x14, 0x4a, 0x01], "vprorvd xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x15, 0x0a], "vprolvd xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x15, 0x4a, 0x01], "vprolvd xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x1e, 0x0a], "vpabsd xmm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x1e, 0x4a, 0x01], "vpabsd xmm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x27, 0x0a], "vptestmd k1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x27, 0x4a, 0x01], "vptestmd k1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x2b, 0x0a], "vpackusdw xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x2b, 0x4a, 0x01], "vpackusdw xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x2c, 0x0a], "vscalefps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x2c, 0x4a, 0x01], "vscalefps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x2c, 0xca], "vscalefps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x2d, 0xca], "vscalefss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x39, 0x0a], "vpminsd xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x39, 0x4a, 0x01], "vpminsd xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x3b, 0x0a], "vpminud xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x3b, 0x4a, 0x01], "vpminud xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x3d, 0x0a], "vpmaxsd xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x3d, 0x4a, 0x01], "vpmaxsd xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x3f, 0x0a], "vpmaxud xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x3f, 0x4a, 0x01], "vpmaxud xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x40, 0x0a], "vpmulld xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x40, 0x4a, 0x01], "vpmulld xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x42, 0x0a], "vgetexpps xmm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x42, 0x4a, 0x01], "vgetexpps xmm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x44, 0x0a], "vplzcntd xmm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x44, 0x4a, 0x01], "vplzcntd xmm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x45, 0x0a], "vpsrlvd xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x45, 0x4a, 0x01], "vpsrlvd xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x46, 0x0a], "vpsravd xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x46, 0x4a, 0x01], "vpsravd xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x47, 0x0a], "vpsllvd xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x47, 0x4a, 0x01], "vpsllvd xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x4c, 0x0a], "vrcp14ps xmm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x4c, 0x4a, 0x01], "vrcp14ps xmm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x4e, 0x0a], "vrsqrt14ps xmm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x4e, 0x4a, 0x01], "vrsqrt14ps xmm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x50, 0x0a], "vpdpbusd xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x50, 0x4a, 0x01], "vpdpbusd xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x51, 0x0a], "vpdpbusds xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x51, 0x4a, 0x01], "vpdpbusds xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x52, 0x0a], "vpdpwssd xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x52, 0x4a, 0x01], "vpdpwssd xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x53, 0x0a], "vpdpwssds xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x53, 0x4a, 0x01], "vpdpwssds xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x55, 0x0a], "vpopcntd xmm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x55, 0x4a, 0x01], "vpopcntd xmm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x64, 0x0a], "vpblendmd xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x64, 0x4a, 0x01], "vpblendmd xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x65, 0x0a], "vblendmps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x65, 0x4a, 0x01], "vblendmps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x71, 0x0a], "vpshldvd xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x71, 0x4a, 0x01], "vpshldvd xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x73, 0x0a], "vpshrdvd xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x73, 0x4a, 0x01], "vpshrdvd xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x76, 0x0a], "vpermi2d xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x76, 0x4a, 0x01], "vpermi2d xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x77, 0x0a], "vpermi2ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x77, 0x4a, 0x01], "vpermi2ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x7e, 0x0a], "vpermt2d xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x7e, 0x4a, 0x01], "vpermt2d xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x7f, 0x0a], "vpermt2ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x7f, 0x4a, 0x01], "vpermt2ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x96, 0x0a], "vfmaddsub132ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x96, 0x4a, 0x01], "vfmaddsub132ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x96, 0xca], "vfmaddsub132ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x97, 0x0a], "vfmsubadd132ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x97, 0x4a, 0x01], "vfmsubadd132ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x97, 0xca], "vfmsubadd132ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x98, 0x0a], "vfmadd132ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x98, 0x4a, 0x01], "vfmadd132ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x98, 0xca], "vfmadd132ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x99, 0xca], "vfmadd132ss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x9a, 0x0a], "vfmsub132ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x9a, 0x4a, 0x01], "vfmsub132ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x9a, 0xca], "vfmsub132ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x9b, 0xca], "vfmsub132ss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x9c, 0x0a], "vfnmadd132ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x9c, 0x4a, 0x01], "vfnmadd132ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x9c, 0xca], "vfnmadd132ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x9d, 0xca], "vfnmadd132ss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x9e, 0x0a], "vfnmsub132ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x9e, 0x4a, 0x01], "vfnmsub132ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x9e, 0xca], "vfnmsub132ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0x9f, 0xca], "vfnmsub132ss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xa6, 0x0a], "vfmaddsub213ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xa6, 0x4a, 0x01], "vfmaddsub213ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xa6, 0xca], "vfmaddsub213ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xa7, 0x0a], "vfmsubadd213ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xa7, 0x4a, 0x01], "vfmsubadd213ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xa7, 0xca], "vfmsubadd213ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xa8, 0x0a], "vfmadd213ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xa8, 0x4a, 0x01], "vfmadd213ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xa8, 0xca], "vfmadd213ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xa9, 0xca], "vfmadd213ss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xaa, 0x0a], "vfmsub213ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xaa, 0x4a, 0x01], "vfmsub213ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xaa, 0xca], "vfmsub213ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xab, 0xca], "vfmsub213ss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xac, 0x0a], "vfnmadd213ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xac, 0x4a, 0x01], "vfnmadd213ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xac, 0xca], "vfnmadd213ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xad, 0xca], "vfnmadd213ss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xae, 0x0a], "vfnmsub213ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xae, 0x4a, 0x01], "vfnmsub213ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xae, 0xca], "vfnmsub213ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xaf, 0xca], "vfnmsub213ss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xb6, 0x0a], "vfmaddsub231ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xb6, 0x4a, 0x01], "vfmaddsub231ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xb6, 0xca], "vfmaddsub231ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xb7, 0x0a], "vfmsubadd231ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xb7, 0x4a, 0x01], "vfmsubadd231ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xb7, 0xca], "vfmsubadd231ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xb8, 0x0a], "vfmadd231ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xb8, 0x4a, 0x01], "vfmadd231ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xb8, 0xca], "vfmadd231ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xb9, 0xca], "vfmadd231ss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xba, 0x0a], "vfmsub231ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xba, 0x4a, 0x01], "vfmsub231ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xba, 0xca], "vfmsub231ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xbb, 0xca], "vfmsub231ss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xbc, 0x0a], "vfnmadd231ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xbc, 0x4a, 0x01], "vfnmadd231ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xbc, 0xca], "vfnmadd231ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xbd, 0xca], "vfnmadd231ss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xbe, 0x0a], "vfnmsub231ps xmm1, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xbe, 0x4a, 0x01], "vfnmsub231ps xmm1, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xbe, 0xca], "vfnmsub231ps zmm1{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xbf, 0xca], "vfnmsub231ss xmm1{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xc4, 0x0a], "vpconflictd xmm1, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x18, 0xc4, 0x4a, 0x01], "vpconflictd xmm1, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x0c, 0x0a], "vpermilps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x0c, 0x4a, 0x01], "vpermilps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x14, 0x0a], "vprorvd xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x14, 0x4a, 0x01], "vprorvd xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x15, 0x0a], "vprolvd xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x15, 0x4a, 0x01], "vprolvd xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x1e, 0x0a], "vpabsd xmm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x1e, 0x4a, 0x01], "vpabsd xmm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x27, 0x0a], "vptestmd k1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x27, 0x4a, 0x01], "vptestmd k1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x2b, 0x0a], "vpackusdw xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x2b, 0x4a, 0x01], "vpackusdw xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x2c, 0x0a], "vscalefps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x2c, 0x4a, 0x01], "vscalefps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x2c, 0xca], "vscalefps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x2d, 0xca], "vscalefss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x39, 0x0a], "vpminsd xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x39, 0x4a, 0x01], "vpminsd xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x3b, 0x0a], "vpminud xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x3b, 0x4a, 0x01], "vpminud xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x3d, 0x0a], "vpmaxsd xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x3d, 0x4a, 0x01], "vpmaxsd xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x3f, 0x0a], "vpmaxud xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x3f, 0x4a, 0x01], "vpmaxud xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x40, 0x0a], "vpmulld xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x40, 0x4a, 0x01], "vpmulld xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x42, 0x0a], "vgetexpps xmm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x42, 0x4a, 0x01], "vgetexpps xmm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x44, 0x0a], "vplzcntd xmm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x44, 0x4a, 0x01], "vplzcntd xmm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x45, 0x0a], "vpsrlvd xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x45, 0x4a, 0x01], "vpsrlvd xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x46, 0x0a], "vpsravd xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x46, 0x4a, 0x01], "vpsravd xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x47, 0x0a], "vpsllvd xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x47, 0x4a, 0x01], "vpsllvd xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x4c, 0x0a], "vrcp14ps xmm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x4c, 0x4a, 0x01], "vrcp14ps xmm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x4e, 0x0a], "vrsqrt14ps xmm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x4e, 0x4a, 0x01], "vrsqrt14ps xmm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x50, 0x0a], "vpdpbusd xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x50, 0x4a, 0x01], "vpdpbusd xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x51, 0x0a], "vpdpbusds xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x51, 0x4a, 0x01], "vpdpbusds xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x52, 0x0a], "vpdpwssd xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x52, 0x4a, 0x01], "vpdpwssd xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x53, 0x0a], "vpdpwssds xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x53, 0x4a, 0x01], "vpdpwssds xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x55, 0x0a], "vpopcntd xmm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x55, 0x4a, 0x01], "vpopcntd xmm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x64, 0x0a], "vpblendmd xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x64, 0x4a, 0x01], "vpblendmd xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x65, 0x0a], "vblendmps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x65, 0x4a, 0x01], "vblendmps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x71, 0x0a], "vpshldvd xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x71, 0x4a, 0x01], "vpshldvd xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x73, 0x0a], "vpshrdvd xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x73, 0x4a, 0x01], "vpshrdvd xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x76, 0x0a], "vpermi2d xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x76, 0x4a, 0x01], "vpermi2d xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x77, 0x0a], "vpermi2ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x77, 0x4a, 0x01], "vpermi2ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x7e, 0x0a], "vpermt2d xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x7e, 0x4a, 0x01], "vpermt2d xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x7f, 0x0a], "vpermt2ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x7f, 0x4a, 0x01], "vpermt2ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x96, 0x0a], "vfmaddsub132ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x96, 0x4a, 0x01], "vfmaddsub132ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x97, 0x0a], "vfmsubadd132ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x97, 0x4a, 0x01], "vfmsubadd132ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x98, 0x0a], "vfmadd132ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x98, 0x4a, 0x01], "vfmadd132ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x98, 0xca], "vfmadd132ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x99, 0xca], "vfmadd132ss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x9a, 0x0a], "vfmsub132ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x9a, 0x4a, 0x01], "vfmsub132ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x9a, 0xca], "vfmsub132ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x9b, 0xca], "vfmsub132ss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x9c, 0x0a], "vfnmadd132ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x9c, 0x4a, 0x01], "vfnmadd132ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x9c, 0xca], "vfnmadd132ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x9d, 0xca], "vfnmadd132ss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x9e, 0x0a], "vfnmsub132ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x9e, 0x4a, 0x01], "vfnmsub132ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x9e, 0xca], "vfnmsub132ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0x9f, 0xca], "vfnmsub132ss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xa6, 0x0a], "vfmaddsub213ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xa6, 0x4a, 0x01], "vfmaddsub213ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xa6, 0xca], "vfmaddsub213ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xa7, 0x0a], "vfmsubadd213ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xa7, 0x4a, 0x01], "vfmsubadd213ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xa7, 0xca], "vfmsubadd213ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xa8, 0x0a], "vfmadd213ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xa8, 0x4a, 0x01], "vfmadd213ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xa8, 0xca], "vfmadd213ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xa9, 0xca], "vfmadd213ss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xaa, 0x0a], "vfmsub213ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xaa, 0x4a, 0x01], "vfmsub213ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xaa, 0xca], "vfmsub213ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xab, 0xca], "vfmsub213ss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xac, 0x0a], "vfnmadd213ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xac, 0x4a, 0x01], "vfnmadd213ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xac, 0xca], "vfnmadd213ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xad, 0xca], "vfnmadd213ss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xae, 0x0a], "vfnmsub213ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xae, 0x4a, 0x01], "vfnmsub213ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xae, 0xca], "vfnmsub213ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xaf, 0xca], "vfnmsub213ss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xb6, 0x0a], "vfmaddsub231ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xb6, 0x4a, 0x01], "vfmaddsub231ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xb6, 0xca], "vfmaddsub231ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xb7, 0x0a], "vfmsubadd231ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xb7, 0x4a, 0x01], "vfmsubadd231ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xb7, 0xca], "vfmsubadd231ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xb8, 0x0a], "vfmadd231ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xb8, 0x4a, 0x01], "vfmadd231ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xb8, 0xca], "vfmadd231ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xb9, 0xca], "vfmadd231ss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xba, 0x0a], "vfmsub231ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xba, 0x4a, 0x01], "vfmsub231ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xba, 0xca], "vfmsub231ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xbb, 0xca], "vfmsub231ss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xbc, 0x0a], "vfnmadd231ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xbc, 0x4a, 0x01], "vfnmadd231ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xbc, 0xca], "vfnmadd231ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xbd, 0xca], "vfnmadd231ss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xbe, 0x0a], "vfnmsub231ps xmm1{k5}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xbe, 0x4a, 0x01], "vfnmsub231ps xmm1{k5}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xbe, 0xca], "vfnmsub231ps zmm1{k5}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xbf, 0xca], "vfnmsub231ss xmm1{k5}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xc4, 0x0a], "vpconflictd xmm1{k5}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x1d, 0xc4, 0x4a, 0x01], "vpconflictd xmm1{k5}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x0c, 0x0a], "vpermilps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x0c, 0x4a, 0x01], "vpermilps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x0c, 0xca], "vpermilps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x13, 0x0a], "vcvtph2ps ymm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x13, 0x4a, 0x01], "vcvtph2ps ymm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x13, 0xca], "vcvtph2ps ymm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x14, 0x0a], "vprorvd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x14, 0x4a, 0x01], "vprorvd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x14, 0xca], "vprorvd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x15, 0x0a], "vprolvd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x15, 0x4a, 0x01], "vprolvd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x15, 0xca], "vprolvd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x16, 0x0a], "vpermps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x16, 0x4a, 0x01], "vpermps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x16, 0xca], "vpermps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x18, 0x0a], "vbroadcastss ymm1, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x18, 0x4a, 0x01], "vbroadcastss ymm1, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x18, 0xca], "vbroadcastss ymm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x19, 0x0a], "vbroadcastf32x2 ymm1, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x19, 0x4a, 0x01], "vbroadcastf32x2 ymm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x19, 0xca], "vbroadcastf32x2 ymm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x1a, 0x0a], "vbroadcastf32x4 ymm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x1a, 0x4a, 0x01], "vbroadcastf32x4 ymm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x1e, 0x0a], "vpabsd ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x1e, 0x4a, 0x01], "vpabsd ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x1e, 0xca], "vpabsd ymm1, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x25, 0x0a], "vpmovsxdq ymm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x25, 0x4a, 0x01], "vpmovsxdq ymm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x25, 0xca], "vpmovsxdq ymm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x26, 0x0a], "vptestmb k1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x26, 0x4a, 0x01], "vptestmb k1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x26, 0xca], "vptestmb k1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x27, 0x0a], "vptestmd k1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x27, 0x4a, 0x01], "vptestmd k1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x27, 0xca], "vptestmd k1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x2a, 0x0a], "vmovntdqa ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x2a, 0x4a, 0x01], "vmovntdqa ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x2b, 0x0a], "vpackusdw ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x2b, 0x4a, 0x01], "vpackusdw ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x2b, 0xca], "vpackusdw ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x2c, 0x0a], "vscalefps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x2c, 0x4a, 0x01], "vscalefps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x2c, 0xca], "vscalefps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x2d, 0x0a], "vscalefss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x2d, 0x4a, 0x01], "vscalefss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x2d, 0xca], "vscalefss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x35, 0x0a], "vpmovzxdq ymm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x35, 0x4a, 0x01], "vpmovzxdq ymm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x35, 0xca], "vpmovzxdq ymm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x36, 0x0a], "vpermd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x36, 0x4a, 0x01], "vpermd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x36, 0xca], "vpermd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x39, 0x0a], "vpminsd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x39, 0x4a, 0x01], "vpminsd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x39, 0xca], "vpminsd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x3b, 0x0a], "vpminud ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x3b, 0x4a, 0x01], "vpminud ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x3b, 0xca], "vpminud ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x3d, 0x0a], "vpmaxsd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x3d, 0x4a, 0x01], "vpmaxsd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x3d, 0xca], "vpmaxsd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x3f, 0x0a], "vpmaxud ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x3f, 0x4a, 0x01], "vpmaxud ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x3f, 0xca], "vpmaxud ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x40, 0x0a], "vpmulld ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x40, 0x4a, 0x01], "vpmulld ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x40, 0xca], "vpmulld ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x42, 0x0a], "vgetexpps ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x42, 0x4a, 0x01], "vgetexpps ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x42, 0xca], "vgetexpps ymm1, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x43, 0x0a], "vgetexpss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x43, 0x4a, 0x01], "vgetexpss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x43, 0xca], "vgetexpss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x44, 0x0a], "vplzcntd ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x44, 0x4a, 0x01], "vplzcntd ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x44, 0xca], "vplzcntd ymm1, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x45, 0x0a], "vpsrlvd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x45, 0x4a, 0x01], "vpsrlvd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x45, 0xca], "vpsrlvd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x46, 0x0a], "vpsravd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x46, 0x4a, 0x01], "vpsravd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x46, 0xca], "vpsravd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x47, 0x0a], "vpsllvd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x47, 0x4a, 0x01], "vpsllvd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x47, 0xca], "vpsllvd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x4c, 0x0a], "vrcp14ps ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x4c, 0x4a, 0x01], "vrcp14ps ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x4c, 0xca], "vrcp14ps ymm1, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x4d, 0x0a], "vrcp14ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x4d, 0x4a, 0x01], "vrcp14ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x4d, 0xca], "vrcp14ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x4e, 0x0a], "vrsqrt14ps ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x4e, 0x4a, 0x01], "vrsqrt14ps ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x4e, 0xca], "vrsqrt14ps ymm1, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x4f, 0x0a], "vrsqrt14ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x4f, 0x4a, 0x01], "vrsqrt14ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x4f, 0xca], "vrsqrt14ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x50, 0x0a], "vpdpbusd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x50, 0x4a, 0x01], "vpdpbusd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x50, 0xca], "vpdpbusd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x51, 0x0a], "vpdpbusds ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x51, 0x4a, 0x01], "vpdpbusds ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x51, 0xca], "vpdpbusds ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x52, 0x0a], "vpdpwssd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x52, 0x4a, 0x01], "vpdpwssd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x52, 0xca], "vpdpwssd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x53, 0x0a], "vpdpwssds ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x53, 0x4a, 0x01], "vpdpwssds ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x53, 0xca], "vpdpwssds ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x54, 0x0a], "vpopcntb ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x54, 0x4a, 0x01], "vpopcntb ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x54, 0xca], "vpopcntb ymm1, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x55, 0x0a], "vpopcntd ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x55, 0x4a, 0x01], "vpopcntd ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x55, 0xca], "vpopcntd ymm1, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x58, 0x0a], "vpbroadcastd ymm1, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x58, 0x4a, 0x01], "vpbroadcastd ymm1, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x58, 0xca], "vpbroadcastd ymm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x59, 0x0a], "vbroadcasti32x2 ymm1, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x59, 0x4a, 0x01], "vbroadcasti32x2 ymm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x59, 0xca], "vbroadcasti32x2 ymm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x5a, 0x0a], "vbroadcasti32x4 ymm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x5a, 0x4a, 0x01], "vbroadcasti32x4 ymm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x62, 0x0a], "vpexpandb ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x62, 0x4a, 0x01], "vpexpandb ymm1, ymmword [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x62, 0xca], "vpexpandb ymm1, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x63, 0x0a], "vpcompressb ymmword [bp + si], ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x63, 0x4a, 0x01], "vpcompressb ymmword [bp + si + 0x1], ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x63, 0xca], "vpcompressb ymm2, ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x64, 0x0a], "vpblendmd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x64, 0x4a, 0x01], "vpblendmd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x64, 0xca], "vpblendmd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x65, 0x0a], "vblendmps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x65, 0x4a, 0x01], "vblendmps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x65, 0xca], "vblendmps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x66, 0x0a], "vpblendmb ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x66, 0x4a, 0x01], "vpblendmb ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x66, 0xca], "vpblendmb ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x71, 0x0a], "vpshldvd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x71, 0x4a, 0x01], "vpshldvd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x71, 0xca], "vpshldvd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x73, 0x0a], "vpshrdvd ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x73, 0x4a, 0x01], "vpshrdvd ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x73, 0xca], "vpshrdvd ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x75, 0x0a], "vpermi2b ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x75, 0x4a, 0x01], "vpermi2b ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x75, 0xca], "vpermi2b ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x76, 0x0a], "vpermi2d ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x76, 0x4a, 0x01], "vpermi2d ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x76, 0xca], "vpermi2d ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x77, 0x0a], "vpermi2ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x77, 0x4a, 0x01], "vpermi2ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x77, 0xca], "vpermi2ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x78, 0x0a], "vpbroadcastb ymm1, byte [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x78, 0x4a, 0x01], "vpbroadcastb ymm1, byte [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x78, 0xca], "vpbroadcastb ymm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x79, 0x0a], "vpbroadcastw ymm1, word [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x79, 0x4a, 0x01], "vpbroadcastw ymm1, word [bp + si + 0x2]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x79, 0xca], "vpbroadcastw ymm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x7a, 0xca], "vpbroadcastb ymm1, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x7b, 0xca], "vpbroadcastw ymm1, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x7d, 0x0a], "vpermt2b ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x7d, 0x4a, 0x01], "vpermt2b ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x7d, 0xca], "vpermt2b ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x7e, 0x0a], "vpermt2d ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x7e, 0x4a, 0x01], "vpermt2d ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x7e, 0xca], "vpermt2d ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x7f, 0x0a], "vpermt2ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x7f, 0x4a, 0x01], "vpermt2ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x7f, 0xca], "vpermt2ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x88, 0x0a], "vexpandps ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x88, 0x4a, 0x01], "vexpandps ymm1, ymmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x88, 0xca], "vexpandps ymm1, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x89, 0x0a], "vpexpandd ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x89, 0x4a, 0x01], "vpexpandd ymm1, ymmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x89, 0xca], "vpexpandd ymm1, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x8a, 0x0a], "vcompressps ymmword [bp + si], ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x8a, 0x4a, 0x01], "vcompressps ymmword [bp + si + 0x4], ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x8a, 0xca], "vcompressps ymm2, ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x8b, 0x0a], "vpcompressd ymmword [bp + si], ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x8b, 0x4a, 0x01], "vpcompressd ymmword [bp + si + 0x4], ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x8b, 0xca], "vpcompressd ymm2, ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x8d, 0x0a], "vpermb ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x8d, 0x4a, 0x01], "vpermb ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x8d, 0xca], "vpermb ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x8f, 0x0a], "vpshufbitqmb k1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x8f, 0x4a, 0x01], "vpshufbitqmb k1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x8f, 0xca], "vpshufbitqmb k1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x96, 0x0a], "vfmaddsub132ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x96, 0x4a, 0x01], "vfmaddsub132ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x96, 0xca], "vfmaddsub132ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x97, 0x0a], "vfmsubadd132ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x97, 0x4a, 0x01], "vfmsubadd132ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x97, 0xca], "vfmsubadd132ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x98, 0x0a], "vfmadd132ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x98, 0x4a, 0x01], "vfmadd132ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x98, 0xca], "vfmadd132ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x99, 0x0a], "vfmadd132ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x99, 0x4a, 0x01], "vfmadd132ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x99, 0xca], "vfmadd132ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9a, 0x0a], "vfmsub132ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9a, 0x4a, 0x01], "vfmsub132ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9a, 0xca], "vfmsub132ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9b, 0x0a], "vfmsub132ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9b, 0x4a, 0x01], "vfmsub132ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9b, 0xca], "vfmsub132ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9c, 0x0a], "vfnmadd132ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9c, 0x4a, 0x01], "vfnmadd132ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9c, 0xca], "vfnmadd132ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9d, 0x0a], "vfnmadd132ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9d, 0x4a, 0x01], "vfnmadd132ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9d, 0xca], "vfnmadd132ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9e, 0x0a], "vfnmsub132ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9e, 0x4a, 0x01], "vfnmsub132ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9e, 0xca], "vfnmsub132ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9f, 0x0a], "vfnmsub132ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9f, 0x4a, 0x01], "vfnmsub132ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0x9f, 0xca], "vfnmsub132ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xa6, 0x0a], "vfmaddsub213ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xa6, 0x4a, 0x01], "vfmaddsub213ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xa6, 0xca], "vfmaddsub213ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xa7, 0x0a], "vfmsubadd213ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xa7, 0x4a, 0x01], "vfmsubadd213ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xa7, 0xca], "vfmsubadd213ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xa8, 0x0a], "vfmadd213ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xa8, 0x4a, 0x01], "vfmadd213ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xa8, 0xca], "vfmadd213ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xa9, 0x0a], "vfmadd213ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xa9, 0x4a, 0x01], "vfmadd213ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xa9, 0xca], "vfmadd213ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xaa, 0x0a], "vfmsub213ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xaa, 0x4a, 0x01], "vfmsub213ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xaa, 0xca], "vfmsub213ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xab, 0x0a], "vfmsub213ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xab, 0x4a, 0x01], "vfmsub213ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xab, 0xca], "vfmsub213ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xac, 0x0a], "vfnmadd213ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xac, 0x4a, 0x01], "vfnmadd213ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xac, 0xca], "vfnmadd213ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xad, 0x0a], "vfnmadd213ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xad, 0x4a, 0x01], "vfnmadd213ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xad, 0xca], "vfnmadd213ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xae, 0x0a], "vfnmsub213ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xae, 0x4a, 0x01], "vfnmsub213ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xae, 0xca], "vfnmsub213ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xaf, 0x0a], "vfnmsub213ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xaf, 0x4a, 0x01], "vfnmsub213ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xaf, 0xca], "vfnmsub213ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xb6, 0x0a], "vfmaddsub231ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xb6, 0x4a, 0x01], "vfmaddsub231ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xb6, 0xca], "vfmaddsub231ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xb7, 0x0a], "vfmsubadd231ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xb7, 0x4a, 0x01], "vfmsubadd231ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xb7, 0xca], "vfmsubadd231ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xb8, 0x0a], "vfmadd231ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xb8, 0x4a, 0x01], "vfmadd231ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xb8, 0xca], "vfmadd231ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xb9, 0x0a], "vfmadd231ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xb9, 0x4a, 0x01], "vfmadd231ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xb9, 0xca], "vfmadd231ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xba, 0x0a], "vfmsub231ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xba, 0x4a, 0x01], "vfmsub231ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xba, 0xca], "vfmsub231ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xbb, 0x0a], "vfmsub231ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xbb, 0x4a, 0x01], "vfmsub231ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xbb, 0xca], "vfmsub231ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xbc, 0x0a], "vfnmadd231ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xbc, 0x4a, 0x01], "vfnmadd231ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xbc, 0xca], "vfnmadd231ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xbd, 0x0a], "vfnmadd231ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xbd, 0x4a, 0x01], "vfnmadd231ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xbd, 0xca], "vfnmadd231ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xbe, 0x0a], "vfnmsub231ps ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xbe, 0x4a, 0x01], "vfnmsub231ps ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xbe, 0xca], "vfnmsub231ps ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xbf, 0x0a], "vfnmsub231ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xbf, 0x4a, 0x01], "vfnmsub231ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xbf, 0xca], "vfnmsub231ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xc4, 0x0a], "vpconflictd ymm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xc4, 0x4a, 0x01], "vpconflictd ymm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xc4, 0xca], "vpconflictd ymm1, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xcb, 0x0a], "vrcp28ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xcb, 0x4a, 0x01], "vrcp28ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xcb, 0xca], "vrcp28ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xcd, 0x0a], "vrsqrt28ss xmm1, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xcd, 0x4a, 0x01], "vrsqrt28ss xmm1, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xcd, 0xca], "vrsqrt28ss xmm1, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xcf, 0x0a], "vgf2p8mulb ymm1, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xcf, 0x4a, 0x01], "vgf2p8mulb ymm1, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x28, 0xcf, 0xca], "vgf2p8mulb ymm1, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x0c, 0x0a], "vpermilps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x0c, 0x4a, 0x01], "vpermilps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x0c, 0xca], "vpermilps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x13, 0x0a], "vcvtph2ps ymm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x13, 0x4a, 0x01], "vcvtph2ps ymm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x13, 0xca], "vcvtph2ps ymm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x14, 0x0a], "vprorvd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x14, 0x4a, 0x01], "vprorvd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x14, 0xca], "vprorvd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x15, 0x0a], "vprolvd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x15, 0x4a, 0x01], "vprolvd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x15, 0xca], "vprolvd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x16, 0x0a], "vpermps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x16, 0x4a, 0x01], "vpermps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x16, 0xca], "vpermps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x18, 0x0a], "vbroadcastss ymm1{k5}, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x18, 0x4a, 0x01], "vbroadcastss ymm1{k5}, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x18, 0xca], "vbroadcastss ymm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x19, 0x0a], "vbroadcastf32x2 ymm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x19, 0x4a, 0x01], "vbroadcastf32x2 ymm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x19, 0xca], "vbroadcastf32x2 ymm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x1a, 0x0a], "vbroadcastf32x4 ymm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x1a, 0x4a, 0x01], "vbroadcastf32x4 ymm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x1e, 0x0a], "vpabsd ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x1e, 0x4a, 0x01], "vpabsd ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x1e, 0xca], "vpabsd ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x25, 0x0a], "vpmovsxdq ymm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x25, 0x4a, 0x01], "vpmovsxdq ymm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x25, 0xca], "vpmovsxdq ymm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x26, 0x0a], "vptestmb k1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x26, 0x4a, 0x01], "vptestmb k1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x26, 0xca], "vptestmb k1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x27, 0x0a], "vptestmd k1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x27, 0x4a, 0x01], "vptestmd k1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x27, 0xca], "vptestmd k1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x2b, 0x0a], "vpackusdw ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x2b, 0x4a, 0x01], "vpackusdw ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x2b, 0xca], "vpackusdw ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x2c, 0x0a], "vscalefps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x2c, 0x4a, 0x01], "vscalefps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x2c, 0xca], "vscalefps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x2d, 0x0a], "vscalefss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x2d, 0x4a, 0x01], "vscalefss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x2d, 0xca], "vscalefss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x35, 0x0a], "vpmovzxdq ymm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x35, 0x4a, 0x01], "vpmovzxdq ymm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x35, 0xca], "vpmovzxdq ymm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x36, 0x0a], "vpermd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x36, 0x4a, 0x01], "vpermd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x36, 0xca], "vpermd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x39, 0x0a], "vpminsd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x39, 0x4a, 0x01], "vpminsd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x39, 0xca], "vpminsd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x3b, 0x0a], "vpminud ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x3b, 0x4a, 0x01], "vpminud ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x3b, 0xca], "vpminud ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x3d, 0x0a], "vpmaxsd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x3d, 0x4a, 0x01], "vpmaxsd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x3d, 0xca], "vpmaxsd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x3f, 0x0a], "vpmaxud ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x3f, 0x4a, 0x01], "vpmaxud ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x3f, 0xca], "vpmaxud ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x40, 0x0a], "vpmulld ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x40, 0x4a, 0x01], "vpmulld ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x40, 0xca], "vpmulld ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x42, 0x0a], "vgetexpps ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x42, 0x4a, 0x01], "vgetexpps ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x42, 0xca], "vgetexpps ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x43, 0x0a], "vgetexpss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x43, 0x4a, 0x01], "vgetexpss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x43, 0xca], "vgetexpss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x44, 0x0a], "vplzcntd ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x44, 0x4a, 0x01], "vplzcntd ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x44, 0xca], "vplzcntd ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x45, 0x0a], "vpsrlvd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x45, 0x4a, 0x01], "vpsrlvd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x45, 0xca], "vpsrlvd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x46, 0x0a], "vpsravd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x46, 0x4a, 0x01], "vpsravd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x46, 0xca], "vpsravd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x47, 0x0a], "vpsllvd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x47, 0x4a, 0x01], "vpsllvd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x47, 0xca], "vpsllvd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x4c, 0x0a], "vrcp14ps ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x4c, 0x4a, 0x01], "vrcp14ps ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x4c, 0xca], "vrcp14ps ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x4d, 0x0a], "vrcp14ss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x4d, 0x4a, 0x01], "vrcp14ss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x4d, 0xca], "vrcp14ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x4e, 0x0a], "vrsqrt14ps ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x4e, 0x4a, 0x01], "vrsqrt14ps ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x4e, 0xca], "vrsqrt14ps ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x4f, 0x0a], "vrsqrt14ss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x4f, 0x4a, 0x01], "vrsqrt14ss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x4f, 0xca], "vrsqrt14ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x50, 0x0a], "vpdpbusd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x50, 0x4a, 0x01], "vpdpbusd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x50, 0xca], "vpdpbusd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x51, 0x0a], "vpdpbusds ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x51, 0x4a, 0x01], "vpdpbusds ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x51, 0xca], "vpdpbusds ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x52, 0x0a], "vpdpwssd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x52, 0x4a, 0x01], "vpdpwssd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x52, 0xca], "vpdpwssd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x53, 0x0a], "vpdpwssds ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x53, 0x4a, 0x01], "vpdpwssds ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x53, 0xca], "vpdpwssds ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x54, 0x0a], "vpopcntb ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x54, 0x4a, 0x01], "vpopcntb ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x54, 0xca], "vpopcntb ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x55, 0x0a], "vpopcntd ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x55, 0x4a, 0x01], "vpopcntd ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x55, 0xca], "vpopcntd ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x58, 0x0a], "vpbroadcastd ymm1{k5}, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x58, 0x4a, 0x01], "vpbroadcastd ymm1{k5}, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x58, 0xca], "vpbroadcastd ymm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x59, 0x0a], "vbroadcasti32x2 ymm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x59, 0x4a, 0x01], "vbroadcasti32x2 ymm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x59, 0xca], "vbroadcasti32x2 ymm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x5a, 0x0a], "vbroadcasti32x4 ymm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x5a, 0x4a, 0x01], "vbroadcasti32x4 ymm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x62, 0x0a], "vpexpandb ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x62, 0x4a, 0x01], "vpexpandb ymm1{k5}, ymmword [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x62, 0xca], "vpexpandb ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x63, 0x0a], "vpcompressb ymmword [bp + si]{k5}, ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x63, 0x4a, 0x01], "vpcompressb ymmword [bp + si + 0x1]{k5}, ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x63, 0xca], "vpcompressb ymm2{k5}, ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x64, 0x0a], "vpblendmd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x64, 0x4a, 0x01], "vpblendmd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x64, 0xca], "vpblendmd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x65, 0x0a], "vblendmps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x65, 0x4a, 0x01], "vblendmps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x65, 0xca], "vblendmps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x66, 0x0a], "vpblendmb ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x66, 0x4a, 0x01], "vpblendmb ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x66, 0xca], "vpblendmb ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x71, 0x0a], "vpshldvd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x71, 0x4a, 0x01], "vpshldvd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x71, 0xca], "vpshldvd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x73, 0x0a], "vpshrdvd ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x73, 0x4a, 0x01], "vpshrdvd ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x73, 0xca], "vpshrdvd ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x75, 0x0a], "vpermi2b ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x75, 0x4a, 0x01], "vpermi2b ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x75, 0xca], "vpermi2b ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x76, 0x0a], "vpermi2d ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x76, 0x4a, 0x01], "vpermi2d ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x76, 0xca], "vpermi2d ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x77, 0x0a], "vpermi2ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x77, 0x4a, 0x01], "vpermi2ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x77, 0xca], "vpermi2ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x78, 0x0a], "vpbroadcastb ymm1{k5}, byte [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x78, 0x4a, 0x01], "vpbroadcastb ymm1{k5}, byte [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x78, 0xca], "vpbroadcastb ymm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x79, 0x0a], "vpbroadcastw ymm1{k5}, word [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x79, 0x4a, 0x01], "vpbroadcastw ymm1{k5}, word [bp + si + 0x2]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x79, 0xca], "vpbroadcastw ymm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x7a, 0xca], "vpbroadcastb ymm1{k5}, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x7b, 0xca], "vpbroadcastw ymm1{k5}, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x7d, 0x0a], "vpermt2b ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x7d, 0x4a, 0x01], "vpermt2b ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x7d, 0xca], "vpermt2b ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x7e, 0x0a], "vpermt2d ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x7e, 0x4a, 0x01], "vpermt2d ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x7e, 0xca], "vpermt2d ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x7f, 0x0a], "vpermt2ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x7f, 0x4a, 0x01], "vpermt2ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x7f, 0xca], "vpermt2ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x88, 0x0a], "vexpandps ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x88, 0x4a, 0x01], "vexpandps ymm1{k5}, ymmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x88, 0xca], "vexpandps ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x89, 0x0a], "vpexpandd ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x89, 0x4a, 0x01], "vpexpandd ymm1{k5}, ymmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x89, 0xca], "vpexpandd ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x8a, 0x0a], "vcompressps ymmword [bp + si]{k5}, ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x8a, 0x4a, 0x01], "vcompressps ymmword [bp + si + 0x4]{k5}, ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x8a, 0xca], "vcompressps ymm2{k5}, ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x8b, 0x0a], "vpcompressd ymmword [bp + si]{k5}, ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x8b, 0x4a, 0x01], "vpcompressd ymmword [bp + si + 0x4]{k5}, ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x8b, 0xca], "vpcompressd ymm2{k5}, ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x8d, 0x0a], "vpermb ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x8d, 0x4a, 0x01], "vpermb ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x8d, 0xca], "vpermb ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x8f, 0x0a], "vpshufbitqmb k1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x8f, 0x4a, 0x01], "vpshufbitqmb k1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x8f, 0xca], "vpshufbitqmb k1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x96, 0x0a], "vfmaddsub132ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x96, 0x4a, 0x01], "vfmaddsub132ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x96, 0xca], "vfmaddsub132ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x97, 0x0a], "vfmsubadd132ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x97, 0x4a, 0x01], "vfmsubadd132ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x97, 0xca], "vfmsubadd132ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x98, 0x0a], "vfmadd132ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x98, 0x4a, 0x01], "vfmadd132ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x98, 0xca], "vfmadd132ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x99, 0x0a], "vfmadd132ss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x99, 0x4a, 0x01], "vfmadd132ss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x99, 0xca], "vfmadd132ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9a, 0x0a], "vfmsub132ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9a, 0x4a, 0x01], "vfmsub132ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9a, 0xca], "vfmsub132ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9b, 0x0a], "vfmsub132ss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9b, 0x4a, 0x01], "vfmsub132ss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9b, 0xca], "vfmsub132ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9c, 0x0a], "vfnmadd132ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9c, 0x4a, 0x01], "vfnmadd132ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9c, 0xca], "vfnmadd132ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9d, 0x0a], "vfnmadd132ss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9d, 0x4a, 0x01], "vfnmadd132ss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9d, 0xca], "vfnmadd132ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9e, 0x0a], "vfnmsub132ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9e, 0x4a, 0x01], "vfnmsub132ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9e, 0xca], "vfnmsub132ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9f, 0x0a], "vfnmsub132ss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9f, 0x4a, 0x01], "vfnmsub132ss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0x9f, 0xca], "vfnmsub132ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xa6, 0x0a], "vfmaddsub213ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xa6, 0x4a, 0x01], "vfmaddsub213ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xa6, 0xca], "vfmaddsub213ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xa7, 0x0a], "vfmsubadd213ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xa7, 0x4a, 0x01], "vfmsubadd213ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xa7, 0xca], "vfmsubadd213ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xa8, 0x0a], "vfmadd213ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xa8, 0x4a, 0x01], "vfmadd213ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xa8, 0xca], "vfmadd213ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xa9, 0x0a], "vfmadd213ss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xa9, 0x4a, 0x01], "vfmadd213ss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xa9, 0xca], "vfmadd213ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xaa, 0x0a], "vfmsub213ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xaa, 0x4a, 0x01], "vfmsub213ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xaa, 0xca], "vfmsub213ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xab, 0x0a], "vfmsub213ss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xab, 0x4a, 0x01], "vfmsub213ss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xab, 0xca], "vfmsub213ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xac, 0x0a], "vfnmadd213ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xac, 0x4a, 0x01], "vfnmadd213ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xac, 0xca], "vfnmadd213ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xad, 0x0a], "vfnmadd213ss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xad, 0x4a, 0x01], "vfnmadd213ss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xad, 0xca], "vfnmadd213ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xae, 0x0a], "vfnmsub213ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xae, 0x4a, 0x01], "vfnmsub213ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xae, 0xca], "vfnmsub213ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xaf, 0x0a], "vfnmsub213ss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xaf, 0x4a, 0x01], "vfnmsub213ss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xaf, 0xca], "vfnmsub213ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xb6, 0x0a], "vfmaddsub231ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xb6, 0x4a, 0x01], "vfmaddsub231ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xb6, 0xca], "vfmaddsub231ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xb7, 0x0a], "vfmsubadd231ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xb7, 0x4a, 0x01], "vfmsubadd231ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xb7, 0xca], "vfmsubadd231ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xb8, 0x0a], "vfmadd231ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xb8, 0x4a, 0x01], "vfmadd231ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xb8, 0xca], "vfmadd231ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xb9, 0x0a], "vfmadd231ss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xb9, 0x4a, 0x01], "vfmadd231ss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xb9, 0xca], "vfmadd231ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xba, 0x0a], "vfmsub231ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xba, 0x4a, 0x01], "vfmsub231ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xba, 0xca], "vfmsub231ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xbb, 0x0a], "vfmsub231ss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xbb, 0x4a, 0x01], "vfmsub231ss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xbb, 0xca], "vfmsub231ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xbc, 0x0a], "vfnmadd231ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xbc, 0x4a, 0x01], "vfnmadd231ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xbc, 0xca], "vfnmadd231ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xbd, 0x0a], "vfnmadd231ss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xbd, 0x4a, 0x01], "vfnmadd231ss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xbd, 0xca], "vfnmadd231ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xbe, 0x0a], "vfnmsub231ps ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xbe, 0x4a, 0x01], "vfnmsub231ps ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xbe, 0xca], "vfnmsub231ps ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xbf, 0x0a], "vfnmsub231ss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xbf, 0x4a, 0x01], "vfnmsub231ss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xbf, 0xca], "vfnmsub231ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xc4, 0x0a], "vpconflictd ymm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xc4, 0x4a, 0x01], "vpconflictd ymm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xc4, 0xca], "vpconflictd ymm1{k5}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xcb, 0x0a], "vrcp28ss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xcb, 0x4a, 0x01], "vrcp28ss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xcb, 0xca], "vrcp28ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xcd, 0x0a], "vrsqrt28ss xmm1{k5}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xcd, 0x4a, 0x01], "vrsqrt28ss xmm1{k5}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xcd, 0xca], "vrsqrt28ss xmm1{k5}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xcf, 0x0a], "vgf2p8mulb ymm1{k5}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xcf, 0x4a, 0x01], "vgf2p8mulb ymm1{k5}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x2d, 0xcf, 0xca], "vgf2p8mulb ymm1{k5}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x0c, 0x0a], "vpermilps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x0c, 0x4a, 0x01], "vpermilps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x14, 0x0a], "vprorvd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x14, 0x4a, 0x01], "vprorvd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x15, 0x0a], "vprolvd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x15, 0x4a, 0x01], "vprolvd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x16, 0x0a], "vpermps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x16, 0x4a, 0x01], "vpermps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x1e, 0x0a], "vpabsd ymm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x1e, 0x4a, 0x01], "vpabsd ymm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x27, 0x0a], "vptestmd k1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x27, 0x4a, 0x01], "vptestmd k1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x2b, 0x0a], "vpackusdw ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x2b, 0x4a, 0x01], "vpackusdw ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x2c, 0x0a], "vscalefps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x2c, 0x4a, 0x01], "vscalefps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x2c, 0xca], "vscalefps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x2d, 0xca], "vscalefss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x36, 0x0a], "vpermd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x36, 0x4a, 0x01], "vpermd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x39, 0x0a], "vpminsd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x39, 0x4a, 0x01], "vpminsd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x3b, 0x0a], "vpminud ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x3b, 0x4a, 0x01], "vpminud ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x3d, 0x0a], "vpmaxsd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x3d, 0x4a, 0x01], "vpmaxsd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x3f, 0x0a], "vpmaxud ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x3f, 0x4a, 0x01], "vpmaxud ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x40, 0x0a], "vpmulld ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x40, 0x4a, 0x01], "vpmulld ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x42, 0x0a], "vgetexpps ymm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x42, 0x4a, 0x01], "vgetexpps ymm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x44, 0x0a], "vplzcntd ymm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x44, 0x4a, 0x01], "vplzcntd ymm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x45, 0x0a], "vpsrlvd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x45, 0x4a, 0x01], "vpsrlvd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x46, 0x0a], "vpsravd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x46, 0x4a, 0x01], "vpsravd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x47, 0x0a], "vpsllvd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x47, 0x4a, 0x01], "vpsllvd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x4c, 0x0a], "vrcp14ps ymm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x4c, 0x4a, 0x01], "vrcp14ps ymm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x4e, 0x0a], "vrsqrt14ps ymm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x4e, 0x4a, 0x01], "vrsqrt14ps ymm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x50, 0x0a], "vpdpbusd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x50, 0x4a, 0x01], "vpdpbusd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x51, 0x0a], "vpdpbusds ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x51, 0x4a, 0x01], "vpdpbusds ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x52, 0x0a], "vpdpwssd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x52, 0x4a, 0x01], "vpdpwssd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x53, 0x0a], "vpdpwssds ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x53, 0x4a, 0x01], "vpdpwssds ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x55, 0x0a], "vpopcntd ymm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x55, 0x4a, 0x01], "vpopcntd ymm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x64, 0x0a], "vpblendmd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x64, 0x4a, 0x01], "vpblendmd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x65, 0x0a], "vblendmps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x65, 0x4a, 0x01], "vblendmps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x71, 0x0a], "vpshldvd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x71, 0x4a, 0x01], "vpshldvd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x73, 0x0a], "vpshrdvd ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x73, 0x4a, 0x01], "vpshrdvd ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x76, 0x0a], "vpermi2d ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x76, 0x4a, 0x01], "vpermi2d ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x77, 0x0a], "vpermi2ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x77, 0x4a, 0x01], "vpermi2ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x7e, 0x0a], "vpermt2d ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x7e, 0x4a, 0x01], "vpermt2d ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x7f, 0x0a], "vpermt2ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x7f, 0x4a, 0x01], "vpermt2ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x96, 0x0a], "vfmaddsub132ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x96, 0x4a, 0x01], "vfmaddsub132ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x96, 0xca], "vfmaddsub132ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x97, 0x0a], "vfmsubadd132ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x97, 0x4a, 0x01], "vfmsubadd132ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x97, 0xca], "vfmsubadd132ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x98, 0x0a], "vfmadd132ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x98, 0x4a, 0x01], "vfmadd132ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x98, 0xca], "vfmadd132ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x99, 0xca], "vfmadd132ss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x9a, 0x0a], "vfmsub132ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x9a, 0x4a, 0x01], "vfmsub132ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x9a, 0xca], "vfmsub132ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x9b, 0xca], "vfmsub132ss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x9c, 0x0a], "vfnmadd132ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x9c, 0x4a, 0x01], "vfnmadd132ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x9c, 0xca], "vfnmadd132ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x9d, 0xca], "vfnmadd132ss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x9e, 0x0a], "vfnmsub132ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x9e, 0x4a, 0x01], "vfnmsub132ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x9e, 0xca], "vfnmsub132ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0x9f, 0xca], "vfnmsub132ss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xa6, 0x0a], "vfmaddsub213ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xa6, 0x4a, 0x01], "vfmaddsub213ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xa6, 0xca], "vfmaddsub213ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xa7, 0x0a], "vfmsubadd213ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xa7, 0x4a, 0x01], "vfmsubadd213ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xa7, 0xca], "vfmsubadd213ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xa8, 0x0a], "vfmadd213ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xa8, 0x4a, 0x01], "vfmadd213ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xa8, 0xca], "vfmadd213ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xa9, 0xca], "vfmadd213ss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xaa, 0x0a], "vfmsub213ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xaa, 0x4a, 0x01], "vfmsub213ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xaa, 0xca], "vfmsub213ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xab, 0xca], "vfmsub213ss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xac, 0x0a], "vfnmadd213ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xac, 0x4a, 0x01], "vfnmadd213ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xac, 0xca], "vfnmadd213ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xad, 0xca], "vfnmadd213ss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xae, 0x0a], "vfnmsub213ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xae, 0x4a, 0x01], "vfnmsub213ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xae, 0xca], "vfnmsub213ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xaf, 0xca], "vfnmsub213ss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xb6, 0x0a], "vfmaddsub231ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xb6, 0x4a, 0x01], "vfmaddsub231ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xb6, 0xca], "vfmaddsub231ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xb7, 0x0a], "vfmsubadd231ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xb7, 0x4a, 0x01], "vfmsubadd231ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xb7, 0xca], "vfmsubadd231ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xb8, 0x0a], "vfmadd231ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xb8, 0x4a, 0x01], "vfmadd231ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xb8, 0xca], "vfmadd231ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xb9, 0xca], "vfmadd231ss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xba, 0x0a], "vfmsub231ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xba, 0x4a, 0x01], "vfmsub231ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xba, 0xca], "vfmsub231ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xbb, 0xca], "vfmsub231ss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xbc, 0x0a], "vfnmadd231ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xbc, 0x4a, 0x01], "vfnmadd231ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xbc, 0xca], "vfnmadd231ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xbd, 0xca], "vfnmadd231ss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xbe, 0x0a], "vfnmsub231ps ymm1, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xbe, 0x4a, 0x01], "vfnmsub231ps ymm1, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xbe, 0xca], "vfnmsub231ps zmm1{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xbf, 0xca], "vfnmsub231ss xmm1{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xc4, 0x0a], "vpconflictd ymm1, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x38, 0xc4, 0x4a, 0x01], "vpconflictd ymm1, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x0c, 0x0a], "vpermilps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x0c, 0x4a, 0x01], "vpermilps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x14, 0x0a], "vprorvd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x14, 0x4a, 0x01], "vprorvd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x15, 0x0a], "vprolvd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x15, 0x4a, 0x01], "vprolvd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x16, 0x0a], "vpermps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x16, 0x4a, 0x01], "vpermps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x1e, 0x0a], "vpabsd ymm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x1e, 0x4a, 0x01], "vpabsd ymm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x27, 0x0a], "vptestmd k1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x27, 0x4a, 0x01], "vptestmd k1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x2b, 0x0a], "vpackusdw ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x2b, 0x4a, 0x01], "vpackusdw ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x2c, 0x0a], "vscalefps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x2c, 0x4a, 0x01], "vscalefps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x2c, 0xca], "vscalefps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x2d, 0xca], "vscalefss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x36, 0x0a], "vpermd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x36, 0x4a, 0x01], "vpermd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x39, 0x0a], "vpminsd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x39, 0x4a, 0x01], "vpminsd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x3b, 0x0a], "vpminud ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x3b, 0x4a, 0x01], "vpminud ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x3d, 0x0a], "vpmaxsd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x3d, 0x4a, 0x01], "vpmaxsd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x3f, 0x0a], "vpmaxud ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x3f, 0x4a, 0x01], "vpmaxud ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x40, 0x0a], "vpmulld ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x40, 0x4a, 0x01], "vpmulld ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x42, 0x0a], "vgetexpps ymm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x42, 0x4a, 0x01], "vgetexpps ymm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x44, 0x0a], "vplzcntd ymm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x44, 0x4a, 0x01], "vplzcntd ymm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x45, 0x0a], "vpsrlvd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x45, 0x4a, 0x01], "vpsrlvd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x46, 0x0a], "vpsravd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x46, 0x4a, 0x01], "vpsravd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x47, 0x0a], "vpsllvd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x47, 0x4a, 0x01], "vpsllvd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x4c, 0x0a], "vrcp14ps ymm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x4c, 0x4a, 0x01], "vrcp14ps ymm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x4e, 0x0a], "vrsqrt14ps ymm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x4e, 0x4a, 0x01], "vrsqrt14ps ymm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x50, 0x0a], "vpdpbusd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x50, 0x4a, 0x01], "vpdpbusd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x51, 0x0a], "vpdpbusds ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x51, 0x4a, 0x01], "vpdpbusds ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x52, 0x0a], "vpdpwssd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x52, 0x4a, 0x01], "vpdpwssd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x53, 0x0a], "vpdpwssds ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x53, 0x4a, 0x01], "vpdpwssds ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x55, 0x0a], "vpopcntd ymm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x55, 0x4a, 0x01], "vpopcntd ymm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x64, 0x0a], "vpblendmd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x64, 0x4a, 0x01], "vpblendmd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x65, 0x0a], "vblendmps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x65, 0x4a, 0x01], "vblendmps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x71, 0x0a], "vpshldvd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x71, 0x4a, 0x01], "vpshldvd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x73, 0x0a], "vpshrdvd ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x73, 0x4a, 0x01], "vpshrdvd ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x76, 0x0a], "vpermi2d ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x76, 0x4a, 0x01], "vpermi2d ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x77, 0x0a], "vpermi2ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x77, 0x4a, 0x01], "vpermi2ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x7e, 0x0a], "vpermt2d ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x7e, 0x4a, 0x01], "vpermt2d ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x7f, 0x0a], "vpermt2ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x7f, 0x4a, 0x01], "vpermt2ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x96, 0x0a], "vfmaddsub132ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x96, 0x4a, 0x01], "vfmaddsub132ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x97, 0x0a], "vfmsubadd132ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x97, 0x4a, 0x01], "vfmsubadd132ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x98, 0x0a], "vfmadd132ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x98, 0x4a, 0x01], "vfmadd132ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x98, 0xca], "vfmadd132ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x99, 0xca], "vfmadd132ss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x9a, 0x0a], "vfmsub132ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x9a, 0x4a, 0x01], "vfmsub132ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x9a, 0xca], "vfmsub132ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x9b, 0xca], "vfmsub132ss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x9c, 0x0a], "vfnmadd132ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x9c, 0x4a, 0x01], "vfnmadd132ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x9c, 0xca], "vfnmadd132ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x9d, 0xca], "vfnmadd132ss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x9e, 0x0a], "vfnmsub132ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x9e, 0x4a, 0x01], "vfnmsub132ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x9e, 0xca], "vfnmsub132ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0x9f, 0xca], "vfnmsub132ss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xa6, 0x0a], "vfmaddsub213ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xa6, 0x4a, 0x01], "vfmaddsub213ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xa6, 0xca], "vfmaddsub213ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xa7, 0x0a], "vfmsubadd213ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xa7, 0x4a, 0x01], "vfmsubadd213ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xa7, 0xca], "vfmsubadd213ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xa8, 0x0a], "vfmadd213ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xa8, 0x4a, 0x01], "vfmadd213ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xa8, 0xca], "vfmadd213ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xa9, 0xca], "vfmadd213ss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xaa, 0x0a], "vfmsub213ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xaa, 0x4a, 0x01], "vfmsub213ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xaa, 0xca], "vfmsub213ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xab, 0xca], "vfmsub213ss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xac, 0x0a], "vfnmadd213ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xac, 0x4a, 0x01], "vfnmadd213ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xac, 0xca], "vfnmadd213ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xad, 0xca], "vfnmadd213ss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xae, 0x0a], "vfnmsub213ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xae, 0x4a, 0x01], "vfnmsub213ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xae, 0xca], "vfnmsub213ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xaf, 0xca], "vfnmsub213ss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xb6, 0x0a], "vfmaddsub231ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xb6, 0x4a, 0x01], "vfmaddsub231ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xb6, 0xca], "vfmaddsub231ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xb7, 0x0a], "vfmsubadd231ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xb7, 0x4a, 0x01], "vfmsubadd231ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xb7, 0xca], "vfmsubadd231ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xb8, 0x0a], "vfmadd231ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xb8, 0x4a, 0x01], "vfmadd231ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xb8, 0xca], "vfmadd231ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xb9, 0xca], "vfmadd231ss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xba, 0x0a], "vfmsub231ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xba, 0x4a, 0x01], "vfmsub231ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xba, 0xca], "vfmsub231ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xbb, 0xca], "vfmsub231ss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xbc, 0x0a], "vfnmadd231ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xbc, 0x4a, 0x01], "vfnmadd231ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xbc, 0xca], "vfnmadd231ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xbd, 0xca], "vfnmadd231ss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xbe, 0x0a], "vfnmsub231ps ymm1{k5}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xbe, 0x4a, 0x01], "vfnmsub231ps ymm1{k5}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xbe, 0xca], "vfnmsub231ps zmm1{k5}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xbf, 0xca], "vfnmsub231ss xmm1{k5}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xc4, 0x0a], "vpconflictd ymm1{k5}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x3d, 0xc4, 0x4a, 0x01], "vpconflictd ymm1{k5}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x0c, 0x0a], "vpermilps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x0c, 0x4a, 0x01], "vpermilps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x0c, 0xca], "vpermilps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x13, 0x0a], "vcvtph2ps zmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x13, 0x4a, 0x01], "vcvtph2ps zmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x13, 0xca], "vcvtph2ps zmm1, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x14, 0x0a], "vprorvd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x14, 0x4a, 0x01], "vprorvd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x14, 0xca], "vprorvd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x15, 0x0a], "vprolvd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x15, 0x4a, 0x01], "vprolvd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x15, 0xca], "vprolvd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x16, 0x0a], "vpermps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x16, 0x4a, 0x01], "vpermps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x16, 0xca], "vpermps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x18, 0x0a], "vbroadcastss zmm1, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x18, 0x4a, 0x01], "vbroadcastss zmm1, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x18, 0xca], "vbroadcastss zmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x19, 0x0a], "vbroadcastf32x2 zmm1, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x19, 0x4a, 0x01], "vbroadcastf32x2 zmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x19, 0xca], "vbroadcastf32x2 zmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x1a, 0x0a], "vbroadcastf32x4 zmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x1a, 0x4a, 0x01], "vbroadcastf32x4 zmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x1b, 0x0a], "vbroadcastf32x8 zmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x1b, 0x4a, 0x01], "vbroadcastf32x8 zmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x1e, 0x0a], "vpabsd zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x1e, 0x4a, 0x01], "vpabsd zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x1e, 0xca], "vpabsd zmm1, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x25, 0x0a], "vpmovsxdq zmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x25, 0x4a, 0x01], "vpmovsxdq zmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x25, 0xca], "vpmovsxdq zmm1, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x26, 0x0a], "vptestmb k1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x26, 0x4a, 0x01], "vptestmb k1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x26, 0xca], "vptestmb k1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x27, 0x0a], "vptestmd k1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x27, 0x4a, 0x01], "vptestmd k1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x27, 0xca], "vptestmd k1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x2a, 0x0a], "vmovntdqa zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x2a, 0x4a, 0x01], "vmovntdqa zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x2b, 0x0a], "vpackusdw zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x2b, 0x4a, 0x01], "vpackusdw zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x2b, 0xca], "vpackusdw zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x2c, 0x0a], "vscalefps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x2c, 0x4a, 0x01], "vscalefps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x2c, 0xca], "vscalefps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x35, 0x0a], "vpmovzxdq zmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x35, 0x4a, 0x01], "vpmovzxdq zmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x35, 0xca], "vpmovzxdq zmm1, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x36, 0x0a], "vpermd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x36, 0x4a, 0x01], "vpermd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x36, 0xca], "vpermd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x39, 0x0a], "vpminsd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x39, 0x4a, 0x01], "vpminsd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x39, 0xca], "vpminsd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x3b, 0x0a], "vpminud zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x3b, 0x4a, 0x01], "vpminud zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x3b, 0xca], "vpminud zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x3d, 0x0a], "vpmaxsd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x3d, 0x4a, 0x01], "vpmaxsd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x3d, 0xca], "vpmaxsd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x3f, 0x0a], "vpmaxud zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x3f, 0x4a, 0x01], "vpmaxud zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x3f, 0xca], "vpmaxud zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x40, 0x0a], "vpmulld zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x40, 0x4a, 0x01], "vpmulld zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x40, 0xca], "vpmulld zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x42, 0x0a], "vgetexpps zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x42, 0x4a, 0x01], "vgetexpps zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x42, 0xca], "vgetexpps zmm1, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x44, 0x0a], "vplzcntd zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x44, 0x4a, 0x01], "vplzcntd zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x44, 0xca], "vplzcntd zmm1, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x45, 0x0a], "vpsrlvd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x45, 0x4a, 0x01], "vpsrlvd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x45, 0xca], "vpsrlvd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x46, 0x0a], "vpsravd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x46, 0x4a, 0x01], "vpsravd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x46, 0xca], "vpsravd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x47, 0x0a], "vpsllvd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x47, 0x4a, 0x01], "vpsllvd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x47, 0xca], "vpsllvd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x4c, 0x0a], "vrcp14ps zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x4c, 0x4a, 0x01], "vrcp14ps zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x4c, 0xca], "vrcp14ps zmm1, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x4e, 0x0a], "vrsqrt14ps zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x4e, 0x4a, 0x01], "vrsqrt14ps zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x4e, 0xca], "vrsqrt14ps zmm1, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x50, 0x0a], "vpdpbusd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x50, 0x4a, 0x01], "vpdpbusd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x50, 0xca], "vpdpbusd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x51, 0x0a], "vpdpbusds zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x51, 0x4a, 0x01], "vpdpbusds zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x51, 0xca], "vpdpbusds zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x52, 0x0a], "vpdpwssd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x52, 0x4a, 0x01], "vpdpwssd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x52, 0xca], "vpdpwssd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x53, 0x0a], "vpdpwssds zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x53, 0x4a, 0x01], "vpdpwssds zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x53, 0xca], "vpdpwssds zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x54, 0x0a], "vpopcntb zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x54, 0x4a, 0x01], "vpopcntb zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x54, 0xca], "vpopcntb zmm1, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x55, 0x0a], "vpopcntd zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x55, 0x4a, 0x01], "vpopcntd zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x55, 0xca], "vpopcntd zmm1, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x58, 0x0a], "vpbroadcastd zmm1, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x58, 0x4a, 0x01], "vpbroadcastd zmm1, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x58, 0xca], "vpbroadcastd zmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x59, 0x0a], "vbroadcasti32x2 zmm1, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x59, 0x4a, 0x01], "vbroadcasti32x2 zmm1, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x59, 0xca], "vbroadcasti32x2 zmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x5a, 0x0a], "vbroadcasti32x4 zmm1, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x5a, 0x4a, 0x01], "vbroadcasti32x4 zmm1, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x5b, 0x0a], "vbroadcasti32x8 zmm1, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x5b, 0x4a, 0x01], "vbroadcasti32x8 zmm1, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x62, 0x0a], "vpexpandb zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x62, 0x4a, 0x01], "vpexpandb zmm1, zmmword [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x62, 0xca], "vpexpandb zmm1, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x63, 0x0a], "vpcompressb zmmword [bp + si], zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x63, 0x4a, 0x01], "vpcompressb zmmword [bp + si + 0x1], zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x63, 0xca], "vpcompressb zmm2, zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x64, 0x0a], "vpblendmd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x64, 0x4a, 0x01], "vpblendmd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x64, 0xca], "vpblendmd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x65, 0x0a], "vblendmps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x65, 0x4a, 0x01], "vblendmps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x65, 0xca], "vblendmps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x66, 0x0a], "vpblendmb zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x66, 0x4a, 0x01], "vpblendmb zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x66, 0xca], "vpblendmb zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x71, 0x0a], "vpshldvd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x71, 0x4a, 0x01], "vpshldvd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x71, 0xca], "vpshldvd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x73, 0x0a], "vpshrdvd zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x73, 0x4a, 0x01], "vpshrdvd zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x73, 0xca], "vpshrdvd zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x75, 0x0a], "vpermi2b zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x75, 0x4a, 0x01], "vpermi2b zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x75, 0xca], "vpermi2b zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x76, 0x0a], "vpermi2d zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x76, 0x4a, 0x01], "vpermi2d zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x76, 0xca], "vpermi2d zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x77, 0x0a], "vpermi2ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x77, 0x4a, 0x01], "vpermi2ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x77, 0xca], "vpermi2ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x78, 0x0a], "vpbroadcastb zmm1, byte [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x78, 0x4a, 0x01], "vpbroadcastb zmm1, byte [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x78, 0xca], "vpbroadcastb zmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x79, 0x0a], "vpbroadcastw zmm1, word [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x79, 0x4a, 0x01], "vpbroadcastw zmm1, word [bp + si + 0x2]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x79, 0xca], "vpbroadcastw zmm1, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x7a, 0xca], "vpbroadcastb zmm1, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x7b, 0xca], "vpbroadcastw zmm1, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x7d, 0x0a], "vpermt2b zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x7d, 0x4a, 0x01], "vpermt2b zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x7d, 0xca], "vpermt2b zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x7e, 0x0a], "vpermt2d zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x7e, 0x4a, 0x01], "vpermt2d zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x7e, 0xca], "vpermt2d zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x7f, 0x0a], "vpermt2ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x7f, 0x4a, 0x01], "vpermt2ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x7f, 0xca], "vpermt2ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x88, 0x0a], "vexpandps zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x88, 0x4a, 0x01], "vexpandps zmm1, zmmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x88, 0xca], "vexpandps zmm1, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x89, 0x0a], "vpexpandd zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x89, 0x4a, 0x01], "vpexpandd zmm1, zmmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x89, 0xca], "vpexpandd zmm1, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x8a, 0x0a], "vcompressps zmmword [bp + si], zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x8a, 0x4a, 0x01], "vcompressps zmmword [bp + si + 0x4], zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x8a, 0xca], "vcompressps zmm2, zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x8b, 0x0a], "vpcompressd zmmword [bp + si], zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x8b, 0x4a, 0x01], "vpcompressd zmmword [bp + si + 0x4], zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x8b, 0xca], "vpcompressd zmm2, zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x8d, 0x0a], "vpermb zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x8d, 0x4a, 0x01], "vpermb zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x8d, 0xca], "vpermb zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x8f, 0x0a], "vpshufbitqmb k1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x8f, 0x4a, 0x01], "vpshufbitqmb k1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x8f, 0xca], "vpshufbitqmb k1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x96, 0x0a], "vfmaddsub132ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x96, 0x4a, 0x01], "vfmaddsub132ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x96, 0xca], "vfmaddsub132ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x97, 0x0a], "vfmsubadd132ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x97, 0x4a, 0x01], "vfmsubadd132ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x97, 0xca], "vfmsubadd132ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x98, 0x0a], "vfmadd132ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x98, 0x4a, 0x01], "vfmadd132ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x98, 0xca], "vfmadd132ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x9a, 0x0a], "vfmsub132ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x9a, 0x4a, 0x01], "vfmsub132ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x9a, 0xca], "vfmsub132ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x9c, 0x0a], "vfnmadd132ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x9c, 0x4a, 0x01], "vfnmadd132ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x9c, 0xca], "vfnmadd132ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x9e, 0x0a], "vfnmsub132ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x9e, 0x4a, 0x01], "vfnmsub132ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0x9e, 0xca], "vfnmsub132ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xa6, 0x0a], "vfmaddsub213ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xa6, 0x4a, 0x01], "vfmaddsub213ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xa6, 0xca], "vfmaddsub213ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xa7, 0x0a], "vfmsubadd213ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xa7, 0x4a, 0x01], "vfmsubadd213ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xa7, 0xca], "vfmsubadd213ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xa8, 0x0a], "vfmadd213ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xa8, 0x4a, 0x01], "vfmadd213ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xa8, 0xca], "vfmadd213ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xaa, 0x0a], "vfmsub213ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xaa, 0x4a, 0x01], "vfmsub213ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xaa, 0xca], "vfmsub213ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xac, 0x0a], "vfnmadd213ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xac, 0x4a, 0x01], "vfnmadd213ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xac, 0xca], "vfnmadd213ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xae, 0x0a], "vfnmsub213ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xae, 0x4a, 0x01], "vfnmsub213ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xae, 0xca], "vfnmsub213ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xb6, 0x0a], "vfmaddsub231ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xb6, 0x4a, 0x01], "vfmaddsub231ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xb6, 0xca], "vfmaddsub231ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xb7, 0x0a], "vfmsubadd231ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xb7, 0x4a, 0x01], "vfmsubadd231ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xb7, 0xca], "vfmsubadd231ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xb8, 0x0a], "vfmadd231ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xb8, 0x4a, 0x01], "vfmadd231ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xb8, 0xca], "vfmadd231ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xba, 0x0a], "vfmsub231ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xba, 0x4a, 0x01], "vfmsub231ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xba, 0xca], "vfmsub231ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xbc, 0x0a], "vfnmadd231ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xbc, 0x4a, 0x01], "vfnmadd231ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xbc, 0xca], "vfnmadd231ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xbe, 0x0a], "vfnmsub231ps zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xbe, 0x4a, 0x01], "vfnmsub231ps zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xbe, 0xca], "vfnmsub231ps zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xc4, 0x0a], "vpconflictd zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xc4, 0x4a, 0x01], "vpconflictd zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xc4, 0xca], "vpconflictd zmm1, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xc8, 0x0a], "vexp2ps zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xc8, 0x4a, 0x01], "vexp2ps zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xc8, 0xca], "vexp2ps zmm1, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xca, 0x0a], "vrcp28ps zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xca, 0x4a, 0x01], "vrcp28ps zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xca, 0xca], "vrcp28ps zmm1, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xcc, 0x0a], "vrsqrt28ps zmm1, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xcc, 0x4a, 0x01], "vrsqrt28ps zmm1, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xcc, 0xca], "vrsqrt28ps zmm1, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xcf, 0x0a], "vgf2p8mulb zmm1, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xcf, 0x4a, 0x01], "vgf2p8mulb zmm1, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x48, 0xcf, 0xca], "vgf2p8mulb zmm1, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x0c, 0x0a], "vpermilps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x0c, 0x4a, 0x01], "vpermilps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x0c, 0xca], "vpermilps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x13, 0x0a], "vcvtph2ps zmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x13, 0x4a, 0x01], "vcvtph2ps zmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x13, 0xca], "vcvtph2ps zmm1{k5}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x14, 0x0a], "vprorvd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x14, 0x4a, 0x01], "vprorvd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x14, 0xca], "vprorvd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x15, 0x0a], "vprolvd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x15, 0x4a, 0x01], "vprolvd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x15, 0xca], "vprolvd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x16, 0x0a], "vpermps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x16, 0x4a, 0x01], "vpermps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x16, 0xca], "vpermps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x18, 0x0a], "vbroadcastss zmm1{k5}, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x18, 0x4a, 0x01], "vbroadcastss zmm1{k5}, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x18, 0xca], "vbroadcastss zmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x19, 0x0a], "vbroadcastf32x2 zmm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x19, 0x4a, 0x01], "vbroadcastf32x2 zmm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x19, 0xca], "vbroadcastf32x2 zmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x1a, 0x0a], "vbroadcastf32x4 zmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x1a, 0x4a, 0x01], "vbroadcastf32x4 zmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x1b, 0x0a], "vbroadcastf32x8 zmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x1b, 0x4a, 0x01], "vbroadcastf32x8 zmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x1e, 0x0a], "vpabsd zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x1e, 0x4a, 0x01], "vpabsd zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x1e, 0xca], "vpabsd zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x25, 0x0a], "vpmovsxdq zmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x25, 0x4a, 0x01], "vpmovsxdq zmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x25, 0xca], "vpmovsxdq zmm1{k5}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x26, 0x0a], "vptestmb k1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x26, 0x4a, 0x01], "vptestmb k1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x26, 0xca], "vptestmb k1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x27, 0x0a], "vptestmd k1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x27, 0x4a, 0x01], "vptestmd k1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x27, 0xca], "vptestmd k1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x2b, 0x0a], "vpackusdw zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x2b, 0x4a, 0x01], "vpackusdw zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x2b, 0xca], "vpackusdw zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x2c, 0x0a], "vscalefps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x2c, 0x4a, 0x01], "vscalefps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x2c, 0xca], "vscalefps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x35, 0x0a], "vpmovzxdq zmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x35, 0x4a, 0x01], "vpmovzxdq zmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x35, 0xca], "vpmovzxdq zmm1{k5}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x36, 0x0a], "vpermd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x36, 0x4a, 0x01], "vpermd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x36, 0xca], "vpermd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x39, 0x0a], "vpminsd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x39, 0x4a, 0x01], "vpminsd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x39, 0xca], "vpminsd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x3b, 0x0a], "vpminud zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x3b, 0x4a, 0x01], "vpminud zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x3b, 0xca], "vpminud zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x3d, 0x0a], "vpmaxsd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x3d, 0x4a, 0x01], "vpmaxsd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x3d, 0xca], "vpmaxsd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x3f, 0x0a], "vpmaxud zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x3f, 0x4a, 0x01], "vpmaxud zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x3f, 0xca], "vpmaxud zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x40, 0x0a], "vpmulld zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x40, 0x4a, 0x01], "vpmulld zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x40, 0xca], "vpmulld zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x42, 0x0a], "vgetexpps zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x42, 0x4a, 0x01], "vgetexpps zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x42, 0xca], "vgetexpps zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x44, 0x0a], "vplzcntd zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x44, 0x4a, 0x01], "vplzcntd zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x44, 0xca], "vplzcntd zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x45, 0x0a], "vpsrlvd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x45, 0x4a, 0x01], "vpsrlvd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x45, 0xca], "vpsrlvd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x46, 0x0a], "vpsravd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x46, 0x4a, 0x01], "vpsravd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x46, 0xca], "vpsravd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x47, 0x0a], "vpsllvd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x47, 0x4a, 0x01], "vpsllvd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x47, 0xca], "vpsllvd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x4c, 0x0a], "vrcp14ps zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x4c, 0x4a, 0x01], "vrcp14ps zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x4c, 0xca], "vrcp14ps zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x4e, 0x0a], "vrsqrt14ps zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x4e, 0x4a, 0x01], "vrsqrt14ps zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x4e, 0xca], "vrsqrt14ps zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x50, 0x0a], "vpdpbusd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x50, 0x4a, 0x01], "vpdpbusd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x50, 0xca], "vpdpbusd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x51, 0x0a], "vpdpbusds zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x51, 0x4a, 0x01], "vpdpbusds zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x51, 0xca], "vpdpbusds zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x52, 0x0a], "vpdpwssd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x52, 0x4a, 0x01], "vpdpwssd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x52, 0xca], "vpdpwssd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x53, 0x0a], "vpdpwssds zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x53, 0x4a, 0x01], "vpdpwssds zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x53, 0xca], "vpdpwssds zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x54, 0x0a], "vpopcntb zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x54, 0x4a, 0x01], "vpopcntb zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x54, 0xca], "vpopcntb zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x55, 0x0a], "vpopcntd zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x55, 0x4a, 0x01], "vpopcntd zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x55, 0xca], "vpopcntd zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x58, 0x0a], "vpbroadcastd zmm1{k5}, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x58, 0x4a, 0x01], "vpbroadcastd zmm1{k5}, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x58, 0xca], "vpbroadcastd zmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x59, 0x0a], "vbroadcasti32x2 zmm1{k5}, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x59, 0x4a, 0x01], "vbroadcasti32x2 zmm1{k5}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x59, 0xca], "vbroadcasti32x2 zmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x5a, 0x0a], "vbroadcasti32x4 zmm1{k5}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x5a, 0x4a, 0x01], "vbroadcasti32x4 zmm1{k5}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x5b, 0x0a], "vbroadcasti32x8 zmm1{k5}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x5b, 0x4a, 0x01], "vbroadcasti32x8 zmm1{k5}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x62, 0x0a], "vpexpandb zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x62, 0x4a, 0x01], "vpexpandb zmm1{k5}, zmmword [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x62, 0xca], "vpexpandb zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x63, 0x0a], "vpcompressb zmmword [bp + si]{k5}, zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x63, 0x4a, 0x01], "vpcompressb zmmword [bp + si + 0x1]{k5}, zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x63, 0xca], "vpcompressb zmm2{k5}, zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x64, 0x0a], "vpblendmd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x64, 0x4a, 0x01], "vpblendmd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x64, 0xca], "vpblendmd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x65, 0x0a], "vblendmps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x65, 0x4a, 0x01], "vblendmps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x65, 0xca], "vblendmps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x66, 0x0a], "vpblendmb zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x66, 0x4a, 0x01], "vpblendmb zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x66, 0xca], "vpblendmb zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x71, 0x0a], "vpshldvd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x71, 0x4a, 0x01], "vpshldvd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x71, 0xca], "vpshldvd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x73, 0x0a], "vpshrdvd zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x73, 0x4a, 0x01], "vpshrdvd zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x73, 0xca], "vpshrdvd zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x75, 0x0a], "vpermi2b zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x75, 0x4a, 0x01], "vpermi2b zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x75, 0xca], "vpermi2b zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x76, 0x0a], "vpermi2d zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x76, 0x4a, 0x01], "vpermi2d zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x76, 0xca], "vpermi2d zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x77, 0x0a], "vpermi2ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x77, 0x4a, 0x01], "vpermi2ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x77, 0xca], "vpermi2ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x78, 0x0a], "vpbroadcastb zmm1{k5}, byte [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x78, 0x4a, 0x01], "vpbroadcastb zmm1{k5}, byte [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x78, 0xca], "vpbroadcastb zmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x79, 0x0a], "vpbroadcastw zmm1{k5}, word [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x79, 0x4a, 0x01], "vpbroadcastw zmm1{k5}, word [bp + si + 0x2]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x79, 0xca], "vpbroadcastw zmm1{k5}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x7a, 0xca], "vpbroadcastb zmm1{k5}, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x7b, 0xca], "vpbroadcastw zmm1{k5}, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x7d, 0x0a], "vpermt2b zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x7d, 0x4a, 0x01], "vpermt2b zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x7d, 0xca], "vpermt2b zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x7e, 0x0a], "vpermt2d zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x7e, 0x4a, 0x01], "vpermt2d zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x7e, 0xca], "vpermt2d zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x7f, 0x0a], "vpermt2ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x7f, 0x4a, 0x01], "vpermt2ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x7f, 0xca], "vpermt2ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x88, 0x0a], "vexpandps zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x88, 0x4a, 0x01], "vexpandps zmm1{k5}, zmmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x88, 0xca], "vexpandps zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x89, 0x0a], "vpexpandd zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x89, 0x4a, 0x01], "vpexpandd zmm1{k5}, zmmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x89, 0xca], "vpexpandd zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x8a, 0x0a], "vcompressps zmmword [bp + si]{k5}, zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x8a, 0x4a, 0x01], "vcompressps zmmword [bp + si + 0x4]{k5}, zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x8a, 0xca], "vcompressps zmm2{k5}, zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x8b, 0x0a], "vpcompressd zmmword [bp + si]{k5}, zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x8b, 0x4a, 0x01], "vpcompressd zmmword [bp + si + 0x4]{k5}, zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x8b, 0xca], "vpcompressd zmm2{k5}, zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x8d, 0x0a], "vpermb zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x8d, 0x4a, 0x01], "vpermb zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x8d, 0xca], "vpermb zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x8f, 0x0a], "vpshufbitqmb k1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x8f, 0x4a, 0x01], "vpshufbitqmb k1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x8f, 0xca], "vpshufbitqmb k1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x96, 0x0a], "vfmaddsub132ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x96, 0x4a, 0x01], "vfmaddsub132ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x97, 0x0a], "vfmsubadd132ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x97, 0x4a, 0x01], "vfmsubadd132ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x98, 0x0a], "vfmadd132ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x98, 0x4a, 0x01], "vfmadd132ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x98, 0xca], "vfmadd132ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x9a, 0x0a], "vfmsub132ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x9a, 0x4a, 0x01], "vfmsub132ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x9a, 0xca], "vfmsub132ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x9c, 0x0a], "vfnmadd132ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x9c, 0x4a, 0x01], "vfnmadd132ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x9c, 0xca], "vfnmadd132ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x9e, 0x0a], "vfnmsub132ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x9e, 0x4a, 0x01], "vfnmsub132ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0x9e, 0xca], "vfnmsub132ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xa6, 0x0a], "vfmaddsub213ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xa6, 0x4a, 0x01], "vfmaddsub213ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xa6, 0xca], "vfmaddsub213ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xa7, 0x0a], "vfmsubadd213ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xa7, 0x4a, 0x01], "vfmsubadd213ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xa7, 0xca], "vfmsubadd213ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xa8, 0x0a], "vfmadd213ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xa8, 0x4a, 0x01], "vfmadd213ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xa8, 0xca], "vfmadd213ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xaa, 0x0a], "vfmsub213ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xaa, 0x4a, 0x01], "vfmsub213ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xaa, 0xca], "vfmsub213ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xac, 0x0a], "vfnmadd213ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xac, 0x4a, 0x01], "vfnmadd213ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xac, 0xca], "vfnmadd213ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xae, 0x0a], "vfnmsub213ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xae, 0x4a, 0x01], "vfnmsub213ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xae, 0xca], "vfnmsub213ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xb6, 0x0a], "vfmaddsub231ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xb6, 0x4a, 0x01], "vfmaddsub231ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xb6, 0xca], "vfmaddsub231ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xb7, 0x0a], "vfmsubadd231ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xb7, 0x4a, 0x01], "vfmsubadd231ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xb7, 0xca], "vfmsubadd231ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xb8, 0x0a], "vfmadd231ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xb8, 0x4a, 0x01], "vfmadd231ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xb8, 0xca], "vfmadd231ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xba, 0x0a], "vfmsub231ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xba, 0x4a, 0x01], "vfmsub231ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xba, 0xca], "vfmsub231ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xbc, 0x0a], "vfnmadd231ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xbc, 0x4a, 0x01], "vfnmadd231ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xbc, 0xca], "vfnmadd231ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xbe, 0x0a], "vfnmsub231ps zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xbe, 0x4a, 0x01], "vfnmsub231ps zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xbe, 0xca], "vfnmsub231ps zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xc4, 0x0a], "vpconflictd zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xc4, 0x4a, 0x01], "vpconflictd zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xc4, 0xca], "vpconflictd zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xc8, 0x0a], "vexp2ps zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xc8, 0x4a, 0x01], "vexp2ps zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xc8, 0xca], "vexp2ps zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xca, 0x0a], "vrcp28ps zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xca, 0x4a, 0x01], "vrcp28ps zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xca, 0xca], "vrcp28ps zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xcc, 0x0a], "vrsqrt28ps zmm1{k5}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xcc, 0x4a, 0x01], "vrsqrt28ps zmm1{k5}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xcc, 0xca], "vrsqrt28ps zmm1{k5}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xcf, 0x0a], "vgf2p8mulb zmm1{k5}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xcf, 0x4a, 0x01], "vgf2p8mulb zmm1{k5}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0x4d, 0xcf, 0xca], "vgf2p8mulb zmm1{k5}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x0c, 0x0a], "vpermilps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x0c, 0x4a, 0x01], "vpermilps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x14, 0x0a], "vprorvd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x14, 0x4a, 0x01], "vprorvd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x15, 0x0a], "vprolvd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x15, 0x4a, 0x01], "vprolvd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x16, 0x0a], "vpermps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x16, 0x4a, 0x01], "vpermps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x1e, 0x0a], "vpabsd zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x1e, 0x4a, 0x01], "vpabsd zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x27, 0x0a], "vptestmd k1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x27, 0x4a, 0x01], "vptestmd k1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x2b, 0x0a], "vpackusdw zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x2b, 0x4a, 0x01], "vpackusdw zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x2c, 0x0a], "vscalefps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x2c, 0x4a, 0x01], "vscalefps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x2c, 0xca], "vscalefps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x2d, 0xca], "vscalefss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x36, 0x0a], "vpermd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x36, 0x4a, 0x01], "vpermd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x39, 0x0a], "vpminsd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x39, 0x4a, 0x01], "vpminsd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x3b, 0x0a], "vpminud zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x3b, 0x4a, 0x01], "vpminud zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x3d, 0x0a], "vpmaxsd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x3d, 0x4a, 0x01], "vpmaxsd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x3f, 0x0a], "vpmaxud zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x3f, 0x4a, 0x01], "vpmaxud zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x40, 0x0a], "vpmulld zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x40, 0x4a, 0x01], "vpmulld zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x42, 0x0a], "vgetexpps zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x42, 0x4a, 0x01], "vgetexpps zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x44, 0x0a], "vplzcntd zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x44, 0x4a, 0x01], "vplzcntd zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x45, 0x0a], "vpsrlvd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x45, 0x4a, 0x01], "vpsrlvd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x46, 0x0a], "vpsravd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x46, 0x4a, 0x01], "vpsravd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x47, 0x0a], "vpsllvd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x47, 0x4a, 0x01], "vpsllvd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x4c, 0x0a], "vrcp14ps zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x4c, 0x4a, 0x01], "vrcp14ps zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x4e, 0x0a], "vrsqrt14ps zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x4e, 0x4a, 0x01], "vrsqrt14ps zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x50, 0x0a], "vpdpbusd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x50, 0x4a, 0x01], "vpdpbusd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x51, 0x0a], "vpdpbusds zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x51, 0x4a, 0x01], "vpdpbusds zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x52, 0x0a], "vpdpwssd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x52, 0x4a, 0x01], "vpdpwssd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x53, 0x0a], "vpdpwssds zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x53, 0x4a, 0x01], "vpdpwssds zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x55, 0x0a], "vpopcntd zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x55, 0x4a, 0x01], "vpopcntd zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x64, 0x0a], "vpblendmd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x64, 0x4a, 0x01], "vpblendmd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x65, 0x0a], "vblendmps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x65, 0x4a, 0x01], "vblendmps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x71, 0x0a], "vpshldvd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x71, 0x4a, 0x01], "vpshldvd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x73, 0x0a], "vpshrdvd zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x73, 0x4a, 0x01], "vpshrdvd zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x76, 0x0a], "vpermi2d zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x76, 0x4a, 0x01], "vpermi2d zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x77, 0x0a], "vpermi2ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x77, 0x4a, 0x01], "vpermi2ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x7e, 0x0a], "vpermt2d zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x7e, 0x4a, 0x01], "vpermt2d zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x7f, 0x0a], "vpermt2ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x7f, 0x4a, 0x01], "vpermt2ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x96, 0x0a], "vfmaddsub132ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x96, 0x4a, 0x01], "vfmaddsub132ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x96, 0xca], "vfmaddsub132ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x97, 0x0a], "vfmsubadd132ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x97, 0x4a, 0x01], "vfmsubadd132ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x97, 0xca], "vfmsubadd132ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x98, 0x0a], "vfmadd132ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x98, 0x4a, 0x01], "vfmadd132ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x98, 0xca], "vfmadd132ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x99, 0xca], "vfmadd132ss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x9a, 0x0a], "vfmsub132ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x9a, 0x4a, 0x01], "vfmsub132ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x9a, 0xca], "vfmsub132ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x9b, 0xca], "vfmsub132ss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x9c, 0x0a], "vfnmadd132ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x9c, 0x4a, 0x01], "vfnmadd132ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x9c, 0xca], "vfnmadd132ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x9d, 0xca], "vfnmadd132ss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x9e, 0x0a], "vfnmsub132ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x9e, 0x4a, 0x01], "vfnmsub132ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x9e, 0xca], "vfnmsub132ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0x9f, 0xca], "vfnmsub132ss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xa6, 0x0a], "vfmaddsub213ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xa6, 0x4a, 0x01], "vfmaddsub213ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xa6, 0xca], "vfmaddsub213ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xa7, 0x0a], "vfmsubadd213ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xa7, 0x4a, 0x01], "vfmsubadd213ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xa7, 0xca], "vfmsubadd213ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xa8, 0x0a], "vfmadd213ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xa8, 0x4a, 0x01], "vfmadd213ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xa8, 0xca], "vfmadd213ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xa9, 0xca], "vfmadd213ss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xaa, 0x0a], "vfmsub213ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xaa, 0x4a, 0x01], "vfmsub213ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xaa, 0xca], "vfmsub213ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xab, 0xca], "vfmsub213ss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xac, 0x0a], "vfnmadd213ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xac, 0x4a, 0x01], "vfnmadd213ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xac, 0xca], "vfnmadd213ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xad, 0xca], "vfnmadd213ss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xae, 0x0a], "vfnmsub213ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xae, 0x4a, 0x01], "vfnmsub213ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xae, 0xca], "vfnmsub213ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xaf, 0xca], "vfnmsub213ss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xb6, 0x0a], "vfmaddsub231ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xb6, 0x4a, 0x01], "vfmaddsub231ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xb6, 0xca], "vfmaddsub231ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xb7, 0x0a], "vfmsubadd231ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xb7, 0x4a, 0x01], "vfmsubadd231ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xb7, 0xca], "vfmsubadd231ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xb8, 0x0a], "vfmadd231ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xb8, 0x4a, 0x01], "vfmadd231ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xb8, 0xca], "vfmadd231ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xb9, 0xca], "vfmadd231ss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xba, 0x0a], "vfmsub231ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xba, 0x4a, 0x01], "vfmsub231ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xba, 0xca], "vfmsub231ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xbb, 0xca], "vfmsub231ss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xbc, 0x0a], "vfnmadd231ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xbc, 0x4a, 0x01], "vfnmadd231ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xbc, 0xca], "vfnmadd231ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xbd, 0xca], "vfnmadd231ss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xbe, 0x0a], "vfnmsub231ps zmm1, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xbe, 0x4a, 0x01], "vfnmsub231ps zmm1, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xbe, 0xca], "vfnmsub231ps zmm1{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xbf, 0xca], "vfnmsub231ss xmm1{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xc4, 0x0a], "vpconflictd zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xc4, 0x4a, 0x01], "vpconflictd zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xc8, 0x0a], "vexp2ps zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xc8, 0x4a, 0x01], "vexp2ps zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xca, 0x0a], "vrcp28ps zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xca, 0x4a, 0x01], "vrcp28ps zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xcc, 0x0a], "vrsqrt28ps zmm1, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x58, 0xcc, 0x4a, 0x01], "vrsqrt28ps zmm1, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x0c, 0x0a], "vpermilps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x0c, 0x4a, 0x01], "vpermilps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x14, 0x0a], "vprorvd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x14, 0x4a, 0x01], "vprorvd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x15, 0x0a], "vprolvd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x15, 0x4a, 0x01], "vprolvd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x16, 0x0a], "vpermps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x16, 0x4a, 0x01], "vpermps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x1e, 0x0a], "vpabsd zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x1e, 0x4a, 0x01], "vpabsd zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x27, 0x0a], "vptestmd k1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x27, 0x4a, 0x01], "vptestmd k1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x2b, 0x0a], "vpackusdw zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x2b, 0x4a, 0x01], "vpackusdw zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x2c, 0x0a], "vscalefps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x2c, 0x4a, 0x01], "vscalefps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x2c, 0xca], "vscalefps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x2d, 0xca], "vscalefss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x36, 0x0a], "vpermd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x36, 0x4a, 0x01], "vpermd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x39, 0x0a], "vpminsd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x39, 0x4a, 0x01], "vpminsd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x3b, 0x0a], "vpminud zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x3b, 0x4a, 0x01], "vpminud zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x3d, 0x0a], "vpmaxsd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x3d, 0x4a, 0x01], "vpmaxsd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x3f, 0x0a], "vpmaxud zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x3f, 0x4a, 0x01], "vpmaxud zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x40, 0x0a], "vpmulld zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x40, 0x4a, 0x01], "vpmulld zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x42, 0x0a], "vgetexpps zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x42, 0x4a, 0x01], "vgetexpps zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x44, 0x0a], "vplzcntd zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x44, 0x4a, 0x01], "vplzcntd zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x45, 0x0a], "vpsrlvd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x45, 0x4a, 0x01], "vpsrlvd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x46, 0x0a], "vpsravd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x46, 0x4a, 0x01], "vpsravd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x47, 0x0a], "vpsllvd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x47, 0x4a, 0x01], "vpsllvd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x4c, 0x0a], "vrcp14ps zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x4c, 0x4a, 0x01], "vrcp14ps zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x4e, 0x0a], "vrsqrt14ps zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x4e, 0x4a, 0x01], "vrsqrt14ps zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x50, 0x0a], "vpdpbusd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x50, 0x4a, 0x01], "vpdpbusd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x51, 0x0a], "vpdpbusds zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x51, 0x4a, 0x01], "vpdpbusds zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x52, 0x0a], "vpdpwssd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x52, 0x4a, 0x01], "vpdpwssd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x53, 0x0a], "vpdpwssds zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x53, 0x4a, 0x01], "vpdpwssds zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x55, 0x0a], "vpopcntd zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x55, 0x4a, 0x01], "vpopcntd zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x64, 0x0a], "vpblendmd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x64, 0x4a, 0x01], "vpblendmd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x65, 0x0a], "vblendmps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x65, 0x4a, 0x01], "vblendmps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x71, 0x0a], "vpshldvd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x71, 0x4a, 0x01], "vpshldvd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x73, 0x0a], "vpshrdvd zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x73, 0x4a, 0x01], "vpshrdvd zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x76, 0x0a], "vpermi2d zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x76, 0x4a, 0x01], "vpermi2d zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x77, 0x0a], "vpermi2ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x77, 0x4a, 0x01], "vpermi2ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x7e, 0x0a], "vpermt2d zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x7e, 0x4a, 0x01], "vpermt2d zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x7f, 0x0a], "vpermt2ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x7f, 0x4a, 0x01], "vpermt2ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x96, 0x0a], "vfmaddsub132ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x96, 0x4a, 0x01], "vfmaddsub132ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x97, 0x0a], "vfmsubadd132ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x97, 0x4a, 0x01], "vfmsubadd132ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x98, 0x0a], "vfmadd132ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x98, 0x4a, 0x01], "vfmadd132ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x98, 0xca], "vfmadd132ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x99, 0xca], "vfmadd132ss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x9a, 0x0a], "vfmsub132ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x9a, 0x4a, 0x01], "vfmsub132ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x9a, 0xca], "vfmsub132ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x9b, 0xca], "vfmsub132ss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x9c, 0x0a], "vfnmadd132ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x9c, 0x4a, 0x01], "vfnmadd132ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x9c, 0xca], "vfnmadd132ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x9d, 0xca], "vfnmadd132ss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x9e, 0x0a], "vfnmsub132ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x9e, 0x4a, 0x01], "vfnmsub132ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x9e, 0xca], "vfnmsub132ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0x9f, 0xca], "vfnmsub132ss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xa6, 0x0a], "vfmaddsub213ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xa6, 0x4a, 0x01], "vfmaddsub213ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xa6, 0xca], "vfmaddsub213ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xa7, 0x0a], "vfmsubadd213ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xa7, 0x4a, 0x01], "vfmsubadd213ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xa7, 0xca], "vfmsubadd213ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xa8, 0x0a], "vfmadd213ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xa8, 0x4a, 0x01], "vfmadd213ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xa8, 0xca], "vfmadd213ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xa9, 0xca], "vfmadd213ss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xaa, 0x0a], "vfmsub213ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xaa, 0x4a, 0x01], "vfmsub213ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xaa, 0xca], "vfmsub213ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xab, 0xca], "vfmsub213ss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xac, 0x0a], "vfnmadd213ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xac, 0x4a, 0x01], "vfnmadd213ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xac, 0xca], "vfnmadd213ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xad, 0xca], "vfnmadd213ss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xae, 0x0a], "vfnmsub213ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xae, 0x4a, 0x01], "vfnmsub213ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xae, 0xca], "vfnmsub213ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xaf, 0xca], "vfnmsub213ss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xb6, 0x0a], "vfmaddsub231ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xb6, 0x4a, 0x01], "vfmaddsub231ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xb6, 0xca], "vfmaddsub231ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xb7, 0x0a], "vfmsubadd231ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xb7, 0x4a, 0x01], "vfmsubadd231ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xb7, 0xca], "vfmsubadd231ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xb8, 0x0a], "vfmadd231ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xb8, 0x4a, 0x01], "vfmadd231ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xb8, 0xca], "vfmadd231ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xb9, 0xca], "vfmadd231ss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xba, 0x0a], "vfmsub231ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xba, 0x4a, 0x01], "vfmsub231ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xba, 0xca], "vfmsub231ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xbb, 0xca], "vfmsub231ss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xbc, 0x0a], "vfnmadd231ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xbc, 0x4a, 0x01], "vfnmadd231ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xbc, 0xca], "vfnmadd231ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xbd, 0xca], "vfnmadd231ss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xbe, 0x0a], "vfnmsub231ps zmm1{k5}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xbe, 0x4a, 0x01], "vfnmsub231ps zmm1{k5}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xbe, 0xca], "vfnmsub231ps zmm1{k5}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xbf, 0xca], "vfnmsub231ss xmm1{k5}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xc4, 0x0a], "vpconflictd zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xc4, 0x4a, 0x01], "vpconflictd zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xc8, 0x0a], "vexp2ps zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xc8, 0x4a, 0x01], "vexp2ps zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xca, 0x0a], "vrcp28ps zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xca, 0x4a, 0x01], "vrcp28ps zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xcc, 0x0a], "vrsqrt28ps zmm1{k5}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x5d, 0xcc, 0x4a, 0x01], "vrsqrt28ps zmm1{k5}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0x13, 0xca], "vcvtph2ps zmm1{sae}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0x2c, 0xca], "vscalefps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0x2d, 0xca], "vscalefss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0x42, 0xca], "vgetexpps zmm1{sae}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0x43, 0xca], "vgetexpss xmm1{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0x96, 0xca], "vfmaddsub132ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0x97, 0xca], "vfmsubadd132ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0x98, 0xca], "vfmadd132ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0x99, 0xca], "vfmadd132ss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0x9a, 0xca], "vfmsub132ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0x9b, 0xca], "vfmsub132ss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0x9c, 0xca], "vfnmadd132ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0x9d, 0xca], "vfnmadd132ss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0x9e, 0xca], "vfnmsub132ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0x9f, 0xca], "vfnmsub132ss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xa6, 0xca], "vfmaddsub213ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xa7, 0xca], "vfmsubadd213ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xa8, 0xca], "vfmadd213ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xa9, 0xca], "vfmadd213ss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xaa, 0xca], "vfmsub213ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xab, 0xca], "vfmsub213ss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xac, 0xca], "vfnmadd213ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xad, 0xca], "vfnmadd213ss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xae, 0xca], "vfnmsub213ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xaf, 0xca], "vfnmsub213ss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xb6, 0xca], "vfmaddsub231ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xb7, 0xca], "vfmsubadd231ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xb8, 0xca], "vfmadd231ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xb9, 0xca], "vfmadd231ss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xba, 0xca], "vfmsub231ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xbb, 0xca], "vfmsub231ss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xbc, 0xca], "vfnmadd231ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xbd, 0xca], "vfnmadd231ss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xbe, 0xca], "vfnmsub231ps zmm1{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xbf, 0xca], "vfnmsub231ss xmm1{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xc8, 0xca], "vexp2ps zmm1{sae}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xca, 0xca], "vrcp28ps zmm1{sae}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xcb, 0xca], "vrcp28ss xmm1{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xcc, 0xca], "vrsqrt28ps zmm1{sae}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x78, 0xcd, 0xca], "vrsqrt28ss xmm1{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0x13, 0xca], "vcvtph2ps zmm1{k5}{sae}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0x2c, 0xca], "vscalefps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0x2d, 0xca], "vscalefss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0x42, 0xca], "vgetexpps zmm1{k5}{sae}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0x43, 0xca], "vgetexpss xmm1{k5}{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0x98, 0xca], "vfmadd132ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0x99, 0xca], "vfmadd132ss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0x9a, 0xca], "vfmsub132ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0x9b, 0xca], "vfmsub132ss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0x9c, 0xca], "vfnmadd132ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0x9d, 0xca], "vfnmadd132ss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0x9e, 0xca], "vfnmsub132ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0x9f, 0xca], "vfnmsub132ss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xa6, 0xca], "vfmaddsub213ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xa7, 0xca], "vfmsubadd213ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xa8, 0xca], "vfmadd213ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xa9, 0xca], "vfmadd213ss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xaa, 0xca], "vfmsub213ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xab, 0xca], "vfmsub213ss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xac, 0xca], "vfnmadd213ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xad, 0xca], "vfnmadd213ss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xae, 0xca], "vfnmsub213ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xaf, 0xca], "vfnmsub213ss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xb6, 0xca], "vfmaddsub231ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xb7, 0xca], "vfmsubadd231ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xb8, 0xca], "vfmadd231ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xb9, 0xca], "vfmadd231ss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xba, 0xca], "vfmsub231ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xbb, 0xca], "vfmsub231ss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xbc, 0xca], "vfnmadd231ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xbd, 0xca], "vfnmadd231ss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xbe, 0xca], "vfnmsub231ps zmm1{k5}{rz-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xbf, 0xca], "vfnmsub231ss xmm1{k5}{rz-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xc8, 0xca], "vexp2ps zmm1{k5}{sae}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xca, 0xca], "vrcp28ps zmm1{k5}{sae}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xcb, 0xca], "vrcp28ss xmm1{k5}{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xcc, 0xca], "vrsqrt28ps zmm1{k5}{sae}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x7d, 0xcd, 0xca], "vrsqrt28ss xmm1{k5}{sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x0c, 0x0a], "vpermilps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x0c, 0x4a, 0x01], "vpermilps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x0c, 0xca], "vpermilps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x13, 0x0a], "vcvtph2ps xmm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x13, 0x4a, 0x01], "vcvtph2ps xmm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x13, 0xca], "vcvtph2ps xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x14, 0x0a], "vprorvd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x14, 0x4a, 0x01], "vprorvd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x14, 0xca], "vprorvd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x15, 0x0a], "vprolvd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x15, 0x4a, 0x01], "vprolvd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x15, 0xca], "vprolvd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x18, 0x0a], "vbroadcastss xmm1{k5}{z}, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x18, 0x4a, 0x01], "vbroadcastss xmm1{k5}{z}, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x18, 0xca], "vbroadcastss xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x1e, 0x0a], "vpabsd xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x1e, 0x4a, 0x01], "vpabsd xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x1e, 0xca], "vpabsd xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x25, 0x0a], "vpmovsxdq xmm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x25, 0x4a, 0x01], "vpmovsxdq xmm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x25, 0xca], "vpmovsxdq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x2b, 0x0a], "vpackusdw xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x2b, 0x4a, 0x01], "vpackusdw xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x2b, 0xca], "vpackusdw xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x2c, 0x0a], "vscalefps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x2c, 0x4a, 0x01], "vscalefps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x2c, 0xca], "vscalefps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x35, 0x0a], "vpmovzxdq xmm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x35, 0x4a, 0x01], "vpmovzxdq xmm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x35, 0xca], "vpmovzxdq xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x39, 0x0a], "vpminsd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x39, 0x4a, 0x01], "vpminsd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x39, 0xca], "vpminsd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x3b, 0x0a], "vpminud xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x3b, 0x4a, 0x01], "vpminud xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x3b, 0xca], "vpminud xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x3d, 0x0a], "vpmaxsd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x3d, 0x4a, 0x01], "vpmaxsd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x3d, 0xca], "vpmaxsd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x3f, 0x0a], "vpmaxud xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x3f, 0x4a, 0x01], "vpmaxud xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x3f, 0xca], "vpmaxud xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x40, 0x0a], "vpmulld xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x40, 0x4a, 0x01], "vpmulld xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x40, 0xca], "vpmulld xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x42, 0x0a], "vgetexpps xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x42, 0x4a, 0x01], "vgetexpps xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x42, 0xca], "vgetexpps xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x44, 0x0a], "vplzcntd xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x44, 0x4a, 0x01], "vplzcntd xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x44, 0xca], "vplzcntd xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x45, 0x0a], "vpsrlvd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x45, 0x4a, 0x01], "vpsrlvd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x45, 0xca], "vpsrlvd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x46, 0x0a], "vpsravd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x46, 0x4a, 0x01], "vpsravd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x46, 0xca], "vpsravd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x47, 0x0a], "vpsllvd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x47, 0x4a, 0x01], "vpsllvd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x47, 0xca], "vpsllvd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x4c, 0x0a], "vrcp14ps xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x4c, 0x4a, 0x01], "vrcp14ps xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x4c, 0xca], "vrcp14ps xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x4e, 0x0a], "vrsqrt14ps xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x4e, 0x4a, 0x01], "vrsqrt14ps xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x4e, 0xca], "vrsqrt14ps xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x50, 0x0a], "vpdpbusd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x50, 0x4a, 0x01], "vpdpbusd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x50, 0xca], "vpdpbusd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x51, 0x0a], "vpdpbusds xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x51, 0x4a, 0x01], "vpdpbusds xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x51, 0xca], "vpdpbusds xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x52, 0x0a], "vpdpwssd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x52, 0x4a, 0x01], "vpdpwssd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x52, 0xca], "vpdpwssd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x53, 0x0a], "vpdpwssds xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x53, 0x4a, 0x01], "vpdpwssds xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x53, 0xca], "vpdpwssds xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x54, 0x0a], "vpopcntb xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x54, 0x4a, 0x01], "vpopcntb xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x54, 0xca], "vpopcntb xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x55, 0x0a], "vpopcntd xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x55, 0x4a, 0x01], "vpopcntd xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x55, 0xca], "vpopcntd xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x58, 0x0a], "vpbroadcastd xmm1{k5}{z}, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x58, 0x4a, 0x01], "vpbroadcastd xmm1{k5}{z}, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x58, 0xca], "vpbroadcastd xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x59, 0x0a], "vbroadcasti32x2 xmm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x59, 0x4a, 0x01], "vbroadcasti32x2 xmm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x59, 0xca], "vbroadcasti32x2 xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x62, 0x0a], "vpexpandb xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x62, 0x4a, 0x01], "vpexpandb xmm1{k5}{z}, xmmword [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x62, 0xca], "vpexpandb xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x63, 0xca], "vpcompressb xmm2{k5}{z}, xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x64, 0x0a], "vpblendmd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x64, 0x4a, 0x01], "vpblendmd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x64, 0xca], "vpblendmd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x65, 0x0a], "vblendmps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x65, 0x4a, 0x01], "vblendmps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x65, 0xca], "vblendmps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x66, 0x0a], "vpblendmb xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x66, 0x4a, 0x01], "vpblendmb xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x66, 0xca], "vpblendmb xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x71, 0x0a], "vpshldvd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x71, 0x4a, 0x01], "vpshldvd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x71, 0xca], "vpshldvd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x73, 0x0a], "vpshrdvd xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x73, 0x4a, 0x01], "vpshrdvd xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x73, 0xca], "vpshrdvd xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x75, 0x0a], "vpermi2b xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x75, 0x4a, 0x01], "vpermi2b xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x75, 0xca], "vpermi2b xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x76, 0x0a], "vpermi2d xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x76, 0x4a, 0x01], "vpermi2d xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x76, 0xca], "vpermi2d xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x77, 0x0a], "vpermi2ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x77, 0x4a, 0x01], "vpermi2ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x77, 0xca], "vpermi2ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x78, 0x0a], "vpbroadcastb xmm1{k5}{z}, byte [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x78, 0x4a, 0x01], "vpbroadcastb xmm1{k5}{z}, byte [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x78, 0xca], "vpbroadcastb xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x79, 0x0a], "vpbroadcastw xmm1{k5}{z}, word [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x79, 0x4a, 0x01], "vpbroadcastw xmm1{k5}{z}, word [bp + si + 0x2]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x79, 0xca], "vpbroadcastw xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x7a, 0xca], "vpbroadcastb xmm1{k5}{z}, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x7b, 0xca], "vpbroadcastw xmm1{k5}{z}, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x7d, 0x0a], "vpermt2b xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x7d, 0x4a, 0x01], "vpermt2b xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x7d, 0xca], "vpermt2b xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x7e, 0x0a], "vpermt2d xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x7e, 0x4a, 0x01], "vpermt2d xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x7e, 0xca], "vpermt2d xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x7f, 0x0a], "vpermt2ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x7f, 0x4a, 0x01], "vpermt2ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x7f, 0xca], "vpermt2ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x88, 0x0a], "vexpandps xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x88, 0x4a, 0x01], "vexpandps xmm1{k5}{z}, xmmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x88, 0xca], "vexpandps xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x89, 0x0a], "vpexpandd xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x89, 0x4a, 0x01], "vpexpandd xmm1{k5}{z}, xmmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x89, 0xca], "vpexpandd xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x8a, 0xca], "vcompressps xmm2{k5}{z}, xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x8b, 0xca], "vpcompressd xmm2{k5}{z}, xmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x8d, 0x0a], "vpermb xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x8d, 0x4a, 0x01], "vpermb xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x8d, 0xca], "vpermb xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x96, 0x0a], "vfmaddsub132ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x96, 0x4a, 0x01], "vfmaddsub132ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x96, 0xca], "vfmaddsub132ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x97, 0x0a], "vfmsubadd132ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x97, 0x4a, 0x01], "vfmsubadd132ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x97, 0xca], "vfmsubadd132ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x98, 0x0a], "vfmadd132ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x98, 0x4a, 0x01], "vfmadd132ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x98, 0xca], "vfmadd132ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x9a, 0x0a], "vfmsub132ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x9a, 0x4a, 0x01], "vfmsub132ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x9a, 0xca], "vfmsub132ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x9c, 0x0a], "vfnmadd132ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x9c, 0x4a, 0x01], "vfnmadd132ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x9c, 0xca], "vfnmadd132ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x9e, 0x0a], "vfnmsub132ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x9e, 0x4a, 0x01], "vfnmsub132ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0x9e, 0xca], "vfnmsub132ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xa6, 0x0a], "vfmaddsub213ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xa6, 0x4a, 0x01], "vfmaddsub213ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xa6, 0xca], "vfmaddsub213ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xa7, 0x0a], "vfmsubadd213ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xa7, 0x4a, 0x01], "vfmsubadd213ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xa7, 0xca], "vfmsubadd213ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xa8, 0x0a], "vfmadd213ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xa8, 0x4a, 0x01], "vfmadd213ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xa8, 0xca], "vfmadd213ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xaa, 0x0a], "vfmsub213ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xaa, 0x4a, 0x01], "vfmsub213ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xaa, 0xca], "vfmsub213ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xac, 0x0a], "vfnmadd213ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xac, 0x4a, 0x01], "vfnmadd213ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xac, 0xca], "vfnmadd213ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xae, 0x0a], "vfnmsub213ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xae, 0x4a, 0x01], "vfnmsub213ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xae, 0xca], "vfnmsub213ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xb6, 0x0a], "vfmaddsub231ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xb6, 0x4a, 0x01], "vfmaddsub231ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xb6, 0xca], "vfmaddsub231ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xb7, 0x0a], "vfmsubadd231ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xb7, 0x4a, 0x01], "vfmsubadd231ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xb7, 0xca], "vfmsubadd231ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xb8, 0x0a], "vfmadd231ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xb8, 0x4a, 0x01], "vfmadd231ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xb8, 0xca], "vfmadd231ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xba, 0x0a], "vfmsub231ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xba, 0x4a, 0x01], "vfmsub231ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xba, 0xca], "vfmsub231ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xbc, 0x0a], "vfnmadd231ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xbc, 0x4a, 0x01], "vfnmadd231ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xbc, 0xca], "vfnmadd231ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xbe, 0x0a], "vfnmsub231ps xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xbe, 0x4a, 0x01], "vfnmsub231ps xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xbe, 0xca], "vfnmsub231ps xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xc4, 0x0a], "vpconflictd xmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xc4, 0x4a, 0x01], "vpconflictd xmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xc4, 0xca], "vpconflictd xmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xcf, 0x0a], "vgf2p8mulb xmm1{k5}{z}, xmm0, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xcf, 0x4a, 0x01], "vgf2p8mulb xmm1{k5}{z}, xmm0, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0x8d, 0xcf, 0xca], "vgf2p8mulb xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x0c, 0x0a], "vpermilps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x0c, 0x4a, 0x01], "vpermilps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x14, 0x0a], "vprorvd xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x14, 0x4a, 0x01], "vprorvd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x15, 0x0a], "vprolvd xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x15, 0x4a, 0x01], "vprolvd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x1e, 0x0a], "vpabsd xmm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x1e, 0x4a, 0x01], "vpabsd xmm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x2b, 0x0a], "vpackusdw xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x2b, 0x4a, 0x01], "vpackusdw xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x2c, 0x0a], "vscalefps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x2c, 0x4a, 0x01], "vscalefps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x2c, 0xca], "vscalefps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x2d, 0xca], "vscalefss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x39, 0x0a], "vpminsd xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x39, 0x4a, 0x01], "vpminsd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x3b, 0x0a], "vpminud xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x3b, 0x4a, 0x01], "vpminud xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x3d, 0x0a], "vpmaxsd xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x3d, 0x4a, 0x01], "vpmaxsd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x3f, 0x0a], "vpmaxud xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x3f, 0x4a, 0x01], "vpmaxud xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x40, 0x0a], "vpmulld xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x40, 0x4a, 0x01], "vpmulld xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x42, 0x0a], "vgetexpps xmm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x42, 0x4a, 0x01], "vgetexpps xmm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x44, 0x0a], "vplzcntd xmm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x44, 0x4a, 0x01], "vplzcntd xmm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x45, 0x0a], "vpsrlvd xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x45, 0x4a, 0x01], "vpsrlvd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x46, 0x0a], "vpsravd xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x46, 0x4a, 0x01], "vpsravd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x47, 0x0a], "vpsllvd xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x47, 0x4a, 0x01], "vpsllvd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x4c, 0x0a], "vrcp14ps xmm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x4c, 0x4a, 0x01], "vrcp14ps xmm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x4e, 0x0a], "vrsqrt14ps xmm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x4e, 0x4a, 0x01], "vrsqrt14ps xmm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x50, 0x0a], "vpdpbusd xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x50, 0x4a, 0x01], "vpdpbusd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x51, 0x0a], "vpdpbusds xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x51, 0x4a, 0x01], "vpdpbusds xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x52, 0x0a], "vpdpwssd xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x52, 0x4a, 0x01], "vpdpwssd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x53, 0x0a], "vpdpwssds xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x53, 0x4a, 0x01], "vpdpwssds xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x55, 0x0a], "vpopcntd xmm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x55, 0x4a, 0x01], "vpopcntd xmm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x64, 0x0a], "vpblendmd xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x64, 0x4a, 0x01], "vpblendmd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x65, 0x0a], "vblendmps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x65, 0x4a, 0x01], "vblendmps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x71, 0x0a], "vpshldvd xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x71, 0x4a, 0x01], "vpshldvd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x73, 0x0a], "vpshrdvd xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x73, 0x4a, 0x01], "vpshrdvd xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x76, 0x0a], "vpermi2d xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x76, 0x4a, 0x01], "vpermi2d xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x77, 0x0a], "vpermi2ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x77, 0x4a, 0x01], "vpermi2ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x7e, 0x0a], "vpermt2d xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x7e, 0x4a, 0x01], "vpermt2d xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x7f, 0x0a], "vpermt2ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x7f, 0x4a, 0x01], "vpermt2ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x96, 0x0a], "vfmaddsub132ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x96, 0x4a, 0x01], "vfmaddsub132ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x97, 0x0a], "vfmsubadd132ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x97, 0x4a, 0x01], "vfmsubadd132ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x98, 0x0a], "vfmadd132ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x98, 0x4a, 0x01], "vfmadd132ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x98, 0xca], "vfmadd132ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x99, 0xca], "vfmadd132ss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x9a, 0x0a], "vfmsub132ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x9a, 0x4a, 0x01], "vfmsub132ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x9a, 0xca], "vfmsub132ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x9b, 0xca], "vfmsub132ss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x9c, 0x0a], "vfnmadd132ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x9c, 0x4a, 0x01], "vfnmadd132ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x9c, 0xca], "vfnmadd132ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x9d, 0xca], "vfnmadd132ss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x9e, 0x0a], "vfnmsub132ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x9e, 0x4a, 0x01], "vfnmsub132ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x9e, 0xca], "vfnmsub132ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0x9f, 0xca], "vfnmsub132ss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xa6, 0x0a], "vfmaddsub213ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xa6, 0x4a, 0x01], "vfmaddsub213ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xa6, 0xca], "vfmaddsub213ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xa7, 0x0a], "vfmsubadd213ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xa7, 0x4a, 0x01], "vfmsubadd213ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xa7, 0xca], "vfmsubadd213ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xa8, 0x0a], "vfmadd213ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xa8, 0x4a, 0x01], "vfmadd213ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xa8, 0xca], "vfmadd213ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xa9, 0xca], "vfmadd213ss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xaa, 0x0a], "vfmsub213ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xaa, 0x4a, 0x01], "vfmsub213ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xaa, 0xca], "vfmsub213ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xab, 0xca], "vfmsub213ss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xac, 0x0a], "vfnmadd213ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xac, 0x4a, 0x01], "vfnmadd213ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xac, 0xca], "vfnmadd213ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xad, 0xca], "vfnmadd213ss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xae, 0x0a], "vfnmsub213ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xae, 0x4a, 0x01], "vfnmsub213ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xae, 0xca], "vfnmsub213ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xaf, 0xca], "vfnmsub213ss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xb6, 0x0a], "vfmaddsub231ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xb6, 0x4a, 0x01], "vfmaddsub231ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xb6, 0xca], "vfmaddsub231ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xb7, 0x0a], "vfmsubadd231ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xb7, 0x4a, 0x01], "vfmsubadd231ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xb7, 0xca], "vfmsubadd231ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xb8, 0x0a], "vfmadd231ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xb8, 0x4a, 0x01], "vfmadd231ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xb8, 0xca], "vfmadd231ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xb9, 0xca], "vfmadd231ss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xba, 0x0a], "vfmsub231ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xba, 0x4a, 0x01], "vfmsub231ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xba, 0xca], "vfmsub231ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xbb, 0xca], "vfmsub231ss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xbc, 0x0a], "vfnmadd231ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xbc, 0x4a, 0x01], "vfnmadd231ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xbc, 0xca], "vfnmadd231ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xbd, 0xca], "vfnmadd231ss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xbe, 0x0a], "vfnmsub231ps xmm1{k5}{z}, xmm0, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xbe, 0x4a, 0x01], "vfnmsub231ps xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xbe, 0xca], "vfnmsub231ps zmm1{k5}{z}{rne-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xbf, 0xca], "vfnmsub231ss xmm1{k5}{z}{rne-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xc4, 0x0a], "vpconflictd xmm1{k5}{z}, dword [bp + si]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0x9d, 0xc4, 0x4a, 0x01], "vpconflictd xmm1{k5}{z}, dword [bp + si + 0x4]{1to4}");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x0c, 0x0a], "vpermilps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x0c, 0x4a, 0x01], "vpermilps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x0c, 0xca], "vpermilps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x13, 0x0a], "vcvtph2ps ymm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x13, 0x4a, 0x01], "vcvtph2ps ymm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x13, 0xca], "vcvtph2ps ymm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x14, 0x0a], "vprorvd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x14, 0x4a, 0x01], "vprorvd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x14, 0xca], "vprorvd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x15, 0x0a], "vprolvd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x15, 0x4a, 0x01], "vprolvd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x15, 0xca], "vprolvd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x16, 0x0a], "vpermps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x16, 0x4a, 0x01], "vpermps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x16, 0xca], "vpermps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x18, 0x0a], "vbroadcastss ymm1{k5}{z}, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x18, 0x4a, 0x01], "vbroadcastss ymm1{k5}{z}, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x18, 0xca], "vbroadcastss ymm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x19, 0x0a], "vbroadcastf32x2 ymm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x19, 0x4a, 0x01], "vbroadcastf32x2 ymm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x19, 0xca], "vbroadcastf32x2 ymm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x1a, 0x0a], "vbroadcastf32x4 ymm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x1a, 0x4a, 0x01], "vbroadcastf32x4 ymm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x1e, 0x0a], "vpabsd ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x1e, 0x4a, 0x01], "vpabsd ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x1e, 0xca], "vpabsd ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x25, 0x0a], "vpmovsxdq ymm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x25, 0x4a, 0x01], "vpmovsxdq ymm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x25, 0xca], "vpmovsxdq ymm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x2b, 0x0a], "vpackusdw ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x2b, 0x4a, 0x01], "vpackusdw ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x2b, 0xca], "vpackusdw ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x2c, 0x0a], "vscalefps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x2c, 0x4a, 0x01], "vscalefps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x2c, 0xca], "vscalefps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x2d, 0x0a], "vscalefss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x2d, 0x4a, 0x01], "vscalefss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x2d, 0xca], "vscalefss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x35, 0x0a], "vpmovzxdq ymm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x35, 0x4a, 0x01], "vpmovzxdq ymm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x35, 0xca], "vpmovzxdq ymm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x36, 0x0a], "vpermd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x36, 0x4a, 0x01], "vpermd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x36, 0xca], "vpermd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x39, 0x0a], "vpminsd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x39, 0x4a, 0x01], "vpminsd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x39, 0xca], "vpminsd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x3b, 0x0a], "vpminud ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x3b, 0x4a, 0x01], "vpminud ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x3b, 0xca], "vpminud ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x3d, 0x0a], "vpmaxsd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x3d, 0x4a, 0x01], "vpmaxsd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x3d, 0xca], "vpmaxsd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x3f, 0x0a], "vpmaxud ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x3f, 0x4a, 0x01], "vpmaxud ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x3f, 0xca], "vpmaxud ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x40, 0x0a], "vpmulld ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x40, 0x4a, 0x01], "vpmulld ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x40, 0xca], "vpmulld ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x42, 0x0a], "vgetexpps ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x42, 0x4a, 0x01], "vgetexpps ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x42, 0xca], "vgetexpps ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x43, 0x0a], "vgetexpss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x43, 0x4a, 0x01], "vgetexpss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x43, 0xca], "vgetexpss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x44, 0x0a], "vplzcntd ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x44, 0x4a, 0x01], "vplzcntd ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x44, 0xca], "vplzcntd ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x45, 0x0a], "vpsrlvd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x45, 0x4a, 0x01], "vpsrlvd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x45, 0xca], "vpsrlvd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x46, 0x0a], "vpsravd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x46, 0x4a, 0x01], "vpsravd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x46, 0xca], "vpsravd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x47, 0x0a], "vpsllvd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x47, 0x4a, 0x01], "vpsllvd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x47, 0xca], "vpsllvd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x4c, 0x0a], "vrcp14ps ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x4c, 0x4a, 0x01], "vrcp14ps ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x4c, 0xca], "vrcp14ps ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x4d, 0x0a], "vrcp14ss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x4d, 0x4a, 0x01], "vrcp14ss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x4d, 0xca], "vrcp14ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x4e, 0x0a], "vrsqrt14ps ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x4e, 0x4a, 0x01], "vrsqrt14ps ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x4e, 0xca], "vrsqrt14ps ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x4f, 0x0a], "vrsqrt14ss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x4f, 0x4a, 0x01], "vrsqrt14ss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x4f, 0xca], "vrsqrt14ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x50, 0x0a], "vpdpbusd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x50, 0x4a, 0x01], "vpdpbusd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x50, 0xca], "vpdpbusd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x51, 0x0a], "vpdpbusds ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x51, 0x4a, 0x01], "vpdpbusds ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x51, 0xca], "vpdpbusds ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x52, 0x0a], "vpdpwssd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x52, 0x4a, 0x01], "vpdpwssd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x52, 0xca], "vpdpwssd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x53, 0x0a], "vpdpwssds ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x53, 0x4a, 0x01], "vpdpwssds ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x53, 0xca], "vpdpwssds ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x54, 0x0a], "vpopcntb ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x54, 0x4a, 0x01], "vpopcntb ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x54, 0xca], "vpopcntb ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x55, 0x0a], "vpopcntd ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x55, 0x4a, 0x01], "vpopcntd ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x55, 0xca], "vpopcntd ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x58, 0x0a], "vpbroadcastd ymm1{k5}{z}, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x58, 0x4a, 0x01], "vpbroadcastd ymm1{k5}{z}, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x58, 0xca], "vpbroadcastd ymm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x59, 0x0a], "vbroadcasti32x2 ymm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x59, 0x4a, 0x01], "vbroadcasti32x2 ymm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x59, 0xca], "vbroadcasti32x2 ymm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x5a, 0x0a], "vbroadcasti32x4 ymm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x5a, 0x4a, 0x01], "vbroadcasti32x4 ymm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x62, 0x0a], "vpexpandb ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x62, 0x4a, 0x01], "vpexpandb ymm1{k5}{z}, ymmword [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x62, 0xca], "vpexpandb ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x63, 0xca], "vpcompressb ymm2{k5}{z}, ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x64, 0x0a], "vpblendmd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x64, 0x4a, 0x01], "vpblendmd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x64, 0xca], "vpblendmd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x65, 0x0a], "vblendmps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x65, 0x4a, 0x01], "vblendmps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x65, 0xca], "vblendmps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x66, 0x0a], "vpblendmb ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x66, 0x4a, 0x01], "vpblendmb ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x66, 0xca], "vpblendmb ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x71, 0x0a], "vpshldvd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x71, 0x4a, 0x01], "vpshldvd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x71, 0xca], "vpshldvd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x73, 0x0a], "vpshrdvd ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x73, 0x4a, 0x01], "vpshrdvd ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x73, 0xca], "vpshrdvd ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x75, 0x0a], "vpermi2b ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x75, 0x4a, 0x01], "vpermi2b ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x75, 0xca], "vpermi2b ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x76, 0x0a], "vpermi2d ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x76, 0x4a, 0x01], "vpermi2d ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x76, 0xca], "vpermi2d ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x77, 0x0a], "vpermi2ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x77, 0x4a, 0x01], "vpermi2ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x77, 0xca], "vpermi2ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x78, 0x0a], "vpbroadcastb ymm1{k5}{z}, byte [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x78, 0x4a, 0x01], "vpbroadcastb ymm1{k5}{z}, byte [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x78, 0xca], "vpbroadcastb ymm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x79, 0x0a], "vpbroadcastw ymm1{k5}{z}, word [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x79, 0x4a, 0x01], "vpbroadcastw ymm1{k5}{z}, word [bp + si + 0x2]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x79, 0xca], "vpbroadcastw ymm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x7a, 0xca], "vpbroadcastb ymm1{k5}{z}, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x7b, 0xca], "vpbroadcastw ymm1{k5}{z}, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x7d, 0x0a], "vpermt2b ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x7d, 0x4a, 0x01], "vpermt2b ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x7d, 0xca], "vpermt2b ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x7e, 0x0a], "vpermt2d ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x7e, 0x4a, 0x01], "vpermt2d ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x7e, 0xca], "vpermt2d ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x7f, 0x0a], "vpermt2ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x7f, 0x4a, 0x01], "vpermt2ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x7f, 0xca], "vpermt2ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x88, 0x0a], "vexpandps ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x88, 0x4a, 0x01], "vexpandps ymm1{k5}{z}, ymmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x88, 0xca], "vexpandps ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x89, 0x0a], "vpexpandd ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x89, 0x4a, 0x01], "vpexpandd ymm1{k5}{z}, ymmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x89, 0xca], "vpexpandd ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x8a, 0xca], "vcompressps ymm2{k5}{z}, ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x8b, 0xca], "vpcompressd ymm2{k5}{z}, ymm1");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x8d, 0x0a], "vpermb ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x8d, 0x4a, 0x01], "vpermb ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x8d, 0xca], "vpermb ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x96, 0x0a], "vfmaddsub132ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x96, 0x4a, 0x01], "vfmaddsub132ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x96, 0xca], "vfmaddsub132ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x97, 0x0a], "vfmsubadd132ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x97, 0x4a, 0x01], "vfmsubadd132ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x97, 0xca], "vfmsubadd132ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x98, 0x0a], "vfmadd132ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x98, 0x4a, 0x01], "vfmadd132ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x98, 0xca], "vfmadd132ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x99, 0x0a], "vfmadd132ss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x99, 0x4a, 0x01], "vfmadd132ss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x99, 0xca], "vfmadd132ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9a, 0x0a], "vfmsub132ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9a, 0x4a, 0x01], "vfmsub132ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9a, 0xca], "vfmsub132ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9b, 0x0a], "vfmsub132ss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9b, 0x4a, 0x01], "vfmsub132ss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9b, 0xca], "vfmsub132ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9c, 0x0a], "vfnmadd132ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9c, 0x4a, 0x01], "vfnmadd132ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9c, 0xca], "vfnmadd132ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9d, 0x0a], "vfnmadd132ss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9d, 0x4a, 0x01], "vfnmadd132ss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9d, 0xca], "vfnmadd132ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9e, 0x0a], "vfnmsub132ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9e, 0x4a, 0x01], "vfnmsub132ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9e, 0xca], "vfnmsub132ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9f, 0x0a], "vfnmsub132ss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9f, 0x4a, 0x01], "vfnmsub132ss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0x9f, 0xca], "vfnmsub132ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xa6, 0x0a], "vfmaddsub213ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xa6, 0x4a, 0x01], "vfmaddsub213ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xa6, 0xca], "vfmaddsub213ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xa7, 0x0a], "vfmsubadd213ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xa7, 0x4a, 0x01], "vfmsubadd213ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xa7, 0xca], "vfmsubadd213ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xa8, 0x0a], "vfmadd213ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xa8, 0x4a, 0x01], "vfmadd213ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xa8, 0xca], "vfmadd213ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xa9, 0x0a], "vfmadd213ss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xa9, 0x4a, 0x01], "vfmadd213ss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xa9, 0xca], "vfmadd213ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xaa, 0x0a], "vfmsub213ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xaa, 0x4a, 0x01], "vfmsub213ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xaa, 0xca], "vfmsub213ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xab, 0x0a], "vfmsub213ss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xab, 0x4a, 0x01], "vfmsub213ss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xab, 0xca], "vfmsub213ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xac, 0x0a], "vfnmadd213ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xac, 0x4a, 0x01], "vfnmadd213ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xac, 0xca], "vfnmadd213ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xad, 0x0a], "vfnmadd213ss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xad, 0x4a, 0x01], "vfnmadd213ss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xad, 0xca], "vfnmadd213ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xae, 0x0a], "vfnmsub213ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xae, 0x4a, 0x01], "vfnmsub213ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xae, 0xca], "vfnmsub213ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xaf, 0x0a], "vfnmsub213ss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xaf, 0x4a, 0x01], "vfnmsub213ss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xaf, 0xca], "vfnmsub213ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xb6, 0x0a], "vfmaddsub231ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xb6, 0x4a, 0x01], "vfmaddsub231ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xb6, 0xca], "vfmaddsub231ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xb7, 0x0a], "vfmsubadd231ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xb7, 0x4a, 0x01], "vfmsubadd231ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xb7, 0xca], "vfmsubadd231ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xb8, 0x0a], "vfmadd231ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xb8, 0x4a, 0x01], "vfmadd231ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xb8, 0xca], "vfmadd231ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xb9, 0x0a], "vfmadd231ss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xb9, 0x4a, 0x01], "vfmadd231ss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xb9, 0xca], "vfmadd231ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xba, 0x0a], "vfmsub231ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xba, 0x4a, 0x01], "vfmsub231ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xba, 0xca], "vfmsub231ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xbb, 0x0a], "vfmsub231ss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xbb, 0x4a, 0x01], "vfmsub231ss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xbb, 0xca], "vfmsub231ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xbc, 0x0a], "vfnmadd231ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xbc, 0x4a, 0x01], "vfnmadd231ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xbc, 0xca], "vfnmadd231ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xbd, 0x0a], "vfnmadd231ss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xbd, 0x4a, 0x01], "vfnmadd231ss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xbd, 0xca], "vfnmadd231ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xbe, 0x0a], "vfnmsub231ps ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xbe, 0x4a, 0x01], "vfnmsub231ps ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xbe, 0xca], "vfnmsub231ps ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xbf, 0x0a], "vfnmsub231ss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xbf, 0x4a, 0x01], "vfnmsub231ss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xbf, 0xca], "vfnmsub231ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xc4, 0x0a], "vpconflictd ymm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xc4, 0x4a, 0x01], "vpconflictd ymm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xc4, 0xca], "vpconflictd ymm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xcb, 0x0a], "vrcp28ss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xcb, 0x4a, 0x01], "vrcp28ss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xcb, 0xca], "vrcp28ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xcd, 0x0a], "vrsqrt28ss xmm1{k5}{z}, xmm0, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xcd, 0x4a, 0x01], "vrsqrt28ss xmm1{k5}{z}, xmm0, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xcd, 0xca], "vrsqrt28ss xmm1{k5}{z}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xcf, 0x0a], "vgf2p8mulb ymm1{k5}{z}, ymm0, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xcf, 0x4a, 0x01], "vgf2p8mulb ymm1{k5}{z}, ymm0, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xad, 0xcf, 0xca], "vgf2p8mulb ymm1{k5}{z}, ymm0, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x0c, 0x0a], "vpermilps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x0c, 0x4a, 0x01], "vpermilps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x14, 0x0a], "vprorvd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x14, 0x4a, 0x01], "vprorvd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x15, 0x0a], "vprolvd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x15, 0x4a, 0x01], "vprolvd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x16, 0x0a], "vpermps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x16, 0x4a, 0x01], "vpermps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x1e, 0x0a], "vpabsd ymm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x1e, 0x4a, 0x01], "vpabsd ymm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x2b, 0x0a], "vpackusdw ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x2b, 0x4a, 0x01], "vpackusdw ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x2c, 0x0a], "vscalefps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x2c, 0x4a, 0x01], "vscalefps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x2c, 0xca], "vscalefps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x2d, 0xca], "vscalefss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x36, 0x0a], "vpermd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x36, 0x4a, 0x01], "vpermd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x39, 0x0a], "vpminsd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x39, 0x4a, 0x01], "vpminsd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x3b, 0x0a], "vpminud ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x3b, 0x4a, 0x01], "vpminud ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x3d, 0x0a], "vpmaxsd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x3d, 0x4a, 0x01], "vpmaxsd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x3f, 0x0a], "vpmaxud ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x3f, 0x4a, 0x01], "vpmaxud ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x40, 0x0a], "vpmulld ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x40, 0x4a, 0x01], "vpmulld ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x42, 0x0a], "vgetexpps ymm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x42, 0x4a, 0x01], "vgetexpps ymm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x44, 0x0a], "vplzcntd ymm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x44, 0x4a, 0x01], "vplzcntd ymm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x45, 0x0a], "vpsrlvd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x45, 0x4a, 0x01], "vpsrlvd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x46, 0x0a], "vpsravd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x46, 0x4a, 0x01], "vpsravd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x47, 0x0a], "vpsllvd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x47, 0x4a, 0x01], "vpsllvd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x4c, 0x0a], "vrcp14ps ymm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x4c, 0x4a, 0x01], "vrcp14ps ymm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x4e, 0x0a], "vrsqrt14ps ymm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x4e, 0x4a, 0x01], "vrsqrt14ps ymm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x50, 0x0a], "vpdpbusd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x50, 0x4a, 0x01], "vpdpbusd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x51, 0x0a], "vpdpbusds ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x51, 0x4a, 0x01], "vpdpbusds ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x52, 0x0a], "vpdpwssd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x52, 0x4a, 0x01], "vpdpwssd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x53, 0x0a], "vpdpwssds ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x53, 0x4a, 0x01], "vpdpwssds ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x55, 0x0a], "vpopcntd ymm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x55, 0x4a, 0x01], "vpopcntd ymm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x64, 0x0a], "vpblendmd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x64, 0x4a, 0x01], "vpblendmd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x65, 0x0a], "vblendmps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x65, 0x4a, 0x01], "vblendmps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x71, 0x0a], "vpshldvd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x71, 0x4a, 0x01], "vpshldvd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x73, 0x0a], "vpshrdvd ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x73, 0x4a, 0x01], "vpshrdvd ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x76, 0x0a], "vpermi2d ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x76, 0x4a, 0x01], "vpermi2d ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x77, 0x0a], "vpermi2ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x77, 0x4a, 0x01], "vpermi2ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x7e, 0x0a], "vpermt2d ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x7e, 0x4a, 0x01], "vpermt2d ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x7f, 0x0a], "vpermt2ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x7f, 0x4a, 0x01], "vpermt2ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x96, 0x0a], "vfmaddsub132ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x96, 0x4a, 0x01], "vfmaddsub132ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x97, 0x0a], "vfmsubadd132ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x97, 0x4a, 0x01], "vfmsubadd132ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x98, 0x0a], "vfmadd132ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x98, 0x4a, 0x01], "vfmadd132ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x98, 0xca], "vfmadd132ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x99, 0xca], "vfmadd132ss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x9a, 0x0a], "vfmsub132ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x9a, 0x4a, 0x01], "vfmsub132ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x9a, 0xca], "vfmsub132ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x9b, 0xca], "vfmsub132ss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x9c, 0x0a], "vfnmadd132ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x9c, 0x4a, 0x01], "vfnmadd132ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x9c, 0xca], "vfnmadd132ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x9d, 0xca], "vfnmadd132ss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x9e, 0x0a], "vfnmsub132ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x9e, 0x4a, 0x01], "vfnmsub132ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x9e, 0xca], "vfnmsub132ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0x9f, 0xca], "vfnmsub132ss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xa6, 0x0a], "vfmaddsub213ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xa6, 0x4a, 0x01], "vfmaddsub213ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xa6, 0xca], "vfmaddsub213ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xa7, 0x0a], "vfmsubadd213ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xa7, 0x4a, 0x01], "vfmsubadd213ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xa7, 0xca], "vfmsubadd213ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xa8, 0x0a], "vfmadd213ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xa8, 0x4a, 0x01], "vfmadd213ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xa8, 0xca], "vfmadd213ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xa9, 0xca], "vfmadd213ss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xaa, 0x0a], "vfmsub213ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xaa, 0x4a, 0x01], "vfmsub213ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xaa, 0xca], "vfmsub213ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xab, 0xca], "vfmsub213ss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xac, 0x0a], "vfnmadd213ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xac, 0x4a, 0x01], "vfnmadd213ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xac, 0xca], "vfnmadd213ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xad, 0xca], "vfnmadd213ss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xae, 0x0a], "vfnmsub213ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xae, 0x4a, 0x01], "vfnmsub213ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xae, 0xca], "vfnmsub213ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xaf, 0xca], "vfnmsub213ss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xb6, 0x0a], "vfmaddsub231ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xb6, 0x4a, 0x01], "vfmaddsub231ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xb6, 0xca], "vfmaddsub231ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xb7, 0x0a], "vfmsubadd231ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xb7, 0x4a, 0x01], "vfmsubadd231ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xb7, 0xca], "vfmsubadd231ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xb8, 0x0a], "vfmadd231ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xb8, 0x4a, 0x01], "vfmadd231ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xb8, 0xca], "vfmadd231ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xb9, 0xca], "vfmadd231ss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xba, 0x0a], "vfmsub231ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xba, 0x4a, 0x01], "vfmsub231ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xba, 0xca], "vfmsub231ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xbb, 0xca], "vfmsub231ss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xbc, 0x0a], "vfnmadd231ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xbc, 0x4a, 0x01], "vfnmadd231ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xbc, 0xca], "vfnmadd231ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xbd, 0xca], "vfnmadd231ss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xbe, 0x0a], "vfnmsub231ps ymm1{k5}{z}, ymm0, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xbe, 0x4a, 0x01], "vfnmsub231ps ymm1{k5}{z}, ymm0, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xbe, 0xca], "vfnmsub231ps zmm1{k5}{z}{rd-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xbf, 0xca], "vfnmsub231ss xmm1{k5}{z}{rd-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xc4, 0x0a], "vpconflictd ymm1{k5}{z}, dword [bp + si]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xbd, 0xc4, 0x4a, 0x01], "vpconflictd ymm1{k5}{z}, dword [bp + si + 0x4]{1to8}");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x0c, 0x0a], "vpermilps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x0c, 0x4a, 0x01], "vpermilps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x0c, 0xca], "vpermilps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x13, 0x0a], "vcvtph2ps zmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x13, 0x4a, 0x01], "vcvtph2ps zmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x13, 0xca], "vcvtph2ps zmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x14, 0x0a], "vprorvd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x14, 0x4a, 0x01], "vprorvd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x14, 0xca], "vprorvd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x15, 0x0a], "vprolvd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x15, 0x4a, 0x01], "vprolvd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x15, 0xca], "vprolvd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x16, 0x0a], "vpermps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x16, 0x4a, 0x01], "vpermps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x16, 0xca], "vpermps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x18, 0x0a], "vbroadcastss zmm1{k5}{z}, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x18, 0x4a, 0x01], "vbroadcastss zmm1{k5}{z}, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x18, 0xca], "vbroadcastss zmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x19, 0x0a], "vbroadcastf32x2 zmm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x19, 0x4a, 0x01], "vbroadcastf32x2 zmm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x19, 0xca], "vbroadcastf32x2 zmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x1a, 0x0a], "vbroadcastf32x4 zmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x1a, 0x4a, 0x01], "vbroadcastf32x4 zmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x1b, 0x0a], "vbroadcastf32x8 zmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x1b, 0x4a, 0x01], "vbroadcastf32x8 zmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x1e, 0x0a], "vpabsd zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x1e, 0x4a, 0x01], "vpabsd zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x1e, 0xca], "vpabsd zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x25, 0x0a], "vpmovsxdq zmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x25, 0x4a, 0x01], "vpmovsxdq zmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x25, 0xca], "vpmovsxdq zmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x2b, 0x0a], "vpackusdw zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x2b, 0x4a, 0x01], "vpackusdw zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x2b, 0xca], "vpackusdw zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x2c, 0x0a], "vscalefps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x2c, 0x4a, 0x01], "vscalefps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x2c, 0xca], "vscalefps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x35, 0x0a], "vpmovzxdq zmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x35, 0x4a, 0x01], "vpmovzxdq zmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x35, 0xca], "vpmovzxdq zmm1{k5}{z}, ymm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x36, 0x0a], "vpermd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x36, 0x4a, 0x01], "vpermd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x36, 0xca], "vpermd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x39, 0x0a], "vpminsd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x39, 0x4a, 0x01], "vpminsd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x39, 0xca], "vpminsd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x3b, 0x0a], "vpminud zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x3b, 0x4a, 0x01], "vpminud zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x3b, 0xca], "vpminud zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x3d, 0x0a], "vpmaxsd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x3d, 0x4a, 0x01], "vpmaxsd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x3d, 0xca], "vpmaxsd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x3f, 0x0a], "vpmaxud zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x3f, 0x4a, 0x01], "vpmaxud zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x3f, 0xca], "vpmaxud zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x40, 0x0a], "vpmulld zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x40, 0x4a, 0x01], "vpmulld zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x40, 0xca], "vpmulld zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x42, 0x0a], "vgetexpps zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x42, 0x4a, 0x01], "vgetexpps zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x42, 0xca], "vgetexpps zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x44, 0x0a], "vplzcntd zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x44, 0x4a, 0x01], "vplzcntd zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x44, 0xca], "vplzcntd zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x45, 0x0a], "vpsrlvd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x45, 0x4a, 0x01], "vpsrlvd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x45, 0xca], "vpsrlvd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x46, 0x0a], "vpsravd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x46, 0x4a, 0x01], "vpsravd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x46, 0xca], "vpsravd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x47, 0x0a], "vpsllvd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x47, 0x4a, 0x01], "vpsllvd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x47, 0xca], "vpsllvd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x4c, 0x0a], "vrcp14ps zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x4c, 0x4a, 0x01], "vrcp14ps zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x4c, 0xca], "vrcp14ps zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x4e, 0x0a], "vrsqrt14ps zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x4e, 0x4a, 0x01], "vrsqrt14ps zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x4e, 0xca], "vrsqrt14ps zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x50, 0x0a], "vpdpbusd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x50, 0x4a, 0x01], "vpdpbusd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x50, 0xca], "vpdpbusd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x51, 0x0a], "vpdpbusds zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x51, 0x4a, 0x01], "vpdpbusds zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x51, 0xca], "vpdpbusds zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x52, 0x0a], "vpdpwssd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x52, 0x4a, 0x01], "vpdpwssd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x52, 0xca], "vpdpwssd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x53, 0x0a], "vpdpwssds zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x53, 0x4a, 0x01], "vpdpwssds zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x53, 0xca], "vpdpwssds zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x54, 0x0a], "vpopcntb zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x54, 0x4a, 0x01], "vpopcntb zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x54, 0xca], "vpopcntb zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x55, 0x0a], "vpopcntd zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x55, 0x4a, 0x01], "vpopcntd zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x55, 0xca], "vpopcntd zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x58, 0x0a], "vpbroadcastd zmm1{k5}{z}, dword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x58, 0x4a, 0x01], "vpbroadcastd zmm1{k5}{z}, dword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x58, 0xca], "vpbroadcastd zmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x59, 0x0a], "vbroadcasti32x2 zmm1{k5}{z}, qword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x59, 0x4a, 0x01], "vbroadcasti32x2 zmm1{k5}{z}, qword [bp + si + 0x8]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x59, 0xca], "vbroadcasti32x2 zmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x5a, 0x0a], "vbroadcasti32x4 zmm1{k5}{z}, xmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x5a, 0x4a, 0x01], "vbroadcasti32x4 zmm1{k5}{z}, xmmword [bp + si + 0x10]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x5b, 0x0a], "vbroadcasti32x8 zmm1{k5}{z}, ymmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x5b, 0x4a, 0x01], "vbroadcasti32x8 zmm1{k5}{z}, ymmword [bp + si + 0x20]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x62, 0x0a], "vpexpandb zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x62, 0x4a, 0x01], "vpexpandb zmm1{k5}{z}, zmmword [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x62, 0xca], "vpexpandb zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x63, 0xca], "vpcompressb zmm2{k5}{z}, zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x64, 0x0a], "vpblendmd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x64, 0x4a, 0x01], "vpblendmd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x64, 0xca], "vpblendmd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x65, 0x0a], "vblendmps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x65, 0x4a, 0x01], "vblendmps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x65, 0xca], "vblendmps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x66, 0x0a], "vpblendmb zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x66, 0x4a, 0x01], "vpblendmb zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x66, 0xca], "vpblendmb zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x71, 0x0a], "vpshldvd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x71, 0x4a, 0x01], "vpshldvd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x71, 0xca], "vpshldvd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x73, 0x0a], "vpshrdvd zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x73, 0x4a, 0x01], "vpshrdvd zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x73, 0xca], "vpshrdvd zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x75, 0x0a], "vpermi2b zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x75, 0x4a, 0x01], "vpermi2b zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x75, 0xca], "vpermi2b zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x76, 0x0a], "vpermi2d zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x76, 0x4a, 0x01], "vpermi2d zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x76, 0xca], "vpermi2d zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x77, 0x0a], "vpermi2ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x77, 0x4a, 0x01], "vpermi2ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x77, 0xca], "vpermi2ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x78, 0x0a], "vpbroadcastb zmm1{k5}{z}, byte [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x78, 0x4a, 0x01], "vpbroadcastb zmm1{k5}{z}, byte [bp + si + 0x1]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x78, 0xca], "vpbroadcastb zmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x79, 0x0a], "vpbroadcastw zmm1{k5}{z}, word [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x79, 0x4a, 0x01], "vpbroadcastw zmm1{k5}{z}, word [bp + si + 0x2]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x79, 0xca], "vpbroadcastw zmm1{k5}{z}, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x7a, 0xca], "vpbroadcastb zmm1{k5}{z}, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x7b, 0xca], "vpbroadcastw zmm1{k5}{z}, edx");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x7d, 0x0a], "vpermt2b zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x7d, 0x4a, 0x01], "vpermt2b zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x7d, 0xca], "vpermt2b zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x7e, 0x0a], "vpermt2d zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x7e, 0x4a, 0x01], "vpermt2d zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x7e, 0xca], "vpermt2d zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x7f, 0x0a], "vpermt2ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x7f, 0x4a, 0x01], "vpermt2ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x7f, 0xca], "vpermt2ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x88, 0x0a], "vexpandps zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x88, 0x4a, 0x01], "vexpandps zmm1{k5}{z}, zmmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x88, 0xca], "vexpandps zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x89, 0x0a], "vpexpandd zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x89, 0x4a, 0x01], "vpexpandd zmm1{k5}{z}, zmmword [bp + si + 0x4]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x89, 0xca], "vpexpandd zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x8a, 0xca], "vcompressps zmm2{k5}{z}, zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x8b, 0xca], "vpcompressd zmm2{k5}{z}, zmm1");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x8d, 0x0a], "vpermb zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x8d, 0x4a, 0x01], "vpermb zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x8d, 0xca], "vpermb zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x96, 0x0a], "vfmaddsub132ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x96, 0x4a, 0x01], "vfmaddsub132ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x96, 0xca], "vfmaddsub132ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x97, 0x0a], "vfmsubadd132ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x97, 0x4a, 0x01], "vfmsubadd132ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x97, 0xca], "vfmsubadd132ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x98, 0x0a], "vfmadd132ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x98, 0x4a, 0x01], "vfmadd132ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x98, 0xca], "vfmadd132ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x9a, 0x0a], "vfmsub132ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x9a, 0x4a, 0x01], "vfmsub132ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x9a, 0xca], "vfmsub132ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x9c, 0x0a], "vfnmadd132ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x9c, 0x4a, 0x01], "vfnmadd132ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x9c, 0xca], "vfnmadd132ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x9e, 0x0a], "vfnmsub132ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x9e, 0x4a, 0x01], "vfnmsub132ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0x9e, 0xca], "vfnmsub132ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xa6, 0x0a], "vfmaddsub213ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xa6, 0x4a, 0x01], "vfmaddsub213ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xa6, 0xca], "vfmaddsub213ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xa7, 0x0a], "vfmsubadd213ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xa7, 0x4a, 0x01], "vfmsubadd213ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xa7, 0xca], "vfmsubadd213ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xa8, 0x0a], "vfmadd213ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xa8, 0x4a, 0x01], "vfmadd213ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xa8, 0xca], "vfmadd213ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xaa, 0x0a], "vfmsub213ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xaa, 0x4a, 0x01], "vfmsub213ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xaa, 0xca], "vfmsub213ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xac, 0x0a], "vfnmadd213ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xac, 0x4a, 0x01], "vfnmadd213ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xac, 0xca], "vfnmadd213ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xae, 0x0a], "vfnmsub213ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xae, 0x4a, 0x01], "vfnmsub213ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xae, 0xca], "vfnmsub213ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xb6, 0x0a], "vfmaddsub231ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xb6, 0x4a, 0x01], "vfmaddsub231ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xb6, 0xca], "vfmaddsub231ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xb7, 0x0a], "vfmsubadd231ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xb7, 0x4a, 0x01], "vfmsubadd231ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xb7, 0xca], "vfmsubadd231ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xb8, 0x0a], "vfmadd231ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xb8, 0x4a, 0x01], "vfmadd231ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xb8, 0xca], "vfmadd231ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xba, 0x0a], "vfmsub231ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xba, 0x4a, 0x01], "vfmsub231ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xba, 0xca], "vfmsub231ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xbc, 0x0a], "vfnmadd231ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xbc, 0x4a, 0x01], "vfnmadd231ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xbc, 0xca], "vfnmadd231ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xbe, 0x0a], "vfnmsub231ps zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xbe, 0x4a, 0x01], "vfnmsub231ps zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xbe, 0xca], "vfnmsub231ps zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xc4, 0x0a], "vpconflictd zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xc4, 0x4a, 0x01], "vpconflictd zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xc4, 0xca], "vpconflictd zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xc8, 0x0a], "vexp2ps zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xc8, 0x4a, 0x01], "vexp2ps zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xc8, 0xca], "vexp2ps zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xca, 0x0a], "vrcp28ps zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xca, 0x4a, 0x01], "vrcp28ps zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xca, 0xca], "vrcp28ps zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xcc, 0x0a], "vrsqrt28ps zmm1{k5}{z}, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xcc, 0x4a, 0x01], "vrsqrt28ps zmm1{k5}{z}, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xcc, 0xca], "vrsqrt28ps zmm1{k5}{z}, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xcf, 0x0a], "vgf2p8mulb zmm1{k5}{z}, zmm0, zmmword [bp + si]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xcf, 0x4a, 0x01], "vgf2p8mulb zmm1{k5}{z}, zmm0, zmmword [bp + si + 0x40]");
    test_display(&[0x62, 0xf2, 0x7d, 0xcd, 0xcf, 0xca], "vgf2p8mulb zmm1{k5}{z}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x0c, 0x0a], "vpermilps zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x0c, 0x4a, 0x01], "vpermilps zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x14, 0x0a], "vprorvd zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x14, 0x4a, 0x01], "vprorvd zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x15, 0x0a], "vprolvd zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x15, 0x4a, 0x01], "vprolvd zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x16, 0x0a], "vpermps zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x16, 0x4a, 0x01], "vpermps zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x1e, 0x0a], "vpabsd zmm1{k5}{z}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x1e, 0x4a, 0x01], "vpabsd zmm1{k5}{z}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x2b, 0x0a], "vpackusdw zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x2b, 0x4a, 0x01], "vpackusdw zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x2c, 0x0a], "vscalefps zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x2c, 0x4a, 0x01], "vscalefps zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x2c, 0xca], "vscalefps zmm1{k5}{z}{ru-sae}, zmm0, zmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x2d, 0xca], "vscalefss xmm1{k5}{z}{ru-sae}, xmm0, xmm2");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x36, 0x0a], "vpermd zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x36, 0x4a, 0x01], "vpermd zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x39, 0x0a], "vpminsd zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x39, 0x4a, 0x01], "vpminsd zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x3b, 0x0a], "vpminud zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x3b, 0x4a, 0x01], "vpminud zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x3d, 0x0a], "vpmaxsd zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x3d, 0x4a, 0x01], "vpmaxsd zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x3f, 0x0a], "vpmaxud zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x3f, 0x4a, 0x01], "vpmaxud zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x40, 0x0a], "vpmulld zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x40, 0x4a, 0x01], "vpmulld zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x42, 0x0a], "vgetexpps zmm1{k5}{z}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x42, 0x4a, 0x01], "vgetexpps zmm1{k5}{z}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x44, 0x0a], "vplzcntd zmm1{k5}{z}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x44, 0x4a, 0x01], "vplzcntd zmm1{k5}{z}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x45, 0x0a], "vpsrlvd zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x45, 0x4a, 0x01], "vpsrlvd zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x46, 0x0a], "vpsravd zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x46, 0x4a, 0x01], "vpsravd zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x47, 0x0a], "vpsllvd zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x47, 0x4a, 0x01], "vpsllvd zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x4c, 0x0a], "vrcp14ps zmm1{k5}{z}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x4c, 0x4a, 0x01], "vrcp14ps zmm1{k5}{z}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x4e, 0x0a], "vrsqrt14ps zmm1{k5}{z}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x4e, 0x4a, 0x01], "vrsqrt14ps zmm1{k5}{z}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x50, 0x0a], "vpdpbusd zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x50, 0x4a, 0x01], "vpdpbusd zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x51, 0x0a], "vpdpbusds zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x51, 0x4a, 0x01], "vpdpbusds zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x52, 0x0a], "vpdpwssd zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x52, 0x4a, 0x01], "vpdpwssd zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x53, 0x0a], "vpdpwssds zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x53, 0x4a, 0x01], "vpdpwssds zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x55, 0x0a], "vpopcntd zmm1{k5}{z}, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x55, 0x4a, 0x01], "vpopcntd zmm1{k5}{z}, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x64, 0x0a], "vpblendmd zmm1{k5}{z}, zmm0, dword [bp + si]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x64, 0x4a, 0x01], "vpblendmd zmm1{k5}{z}, zmm0, dword [bp + si + 0x4]{1to16}");
    test_display(&[0x62, 0xf2, 0x7d, 0xdd, 0x65, 0x0a], "vblendmps zm