use std::fmt::Write;

use yaxpeax_arch::{AddressBase, Decoder, U8Reader, LengthedInstruction};
use yaxpeax_x86::long_mode::InstDecoder;

#[allow(dead_code)]
fn test_invalid(data: &[u8]) {
    test_invalid_under(&InstDecoder::default(), data);
}

fn test_invalid_under(decoder: &InstDecoder, data: &[u8]) {
    let mut reader = U8Reader::new(data);
    if let Ok(inst) = decoder.decode(&mut reader) {
        panic!("decoded {:?} from {:02x?} under decoder {}", inst.opcode(), data, decoder);
    } else {
        // this is fine
    }
}

#[allow(dead_code)]
fn test_display(data: &[u8], expected: &'static str) {
    test_display_under(&InstDecoder::default(), data, expected);
}

fn test_display_under(decoder: &InstDecoder, data: &[u8], expected: &'static str) {
    let mut hex = String::new();
    for b in data {
        write!(hex, "{:02x}", b).unwrap();
    }
    let mut reader = U8Reader::new(data);
    match decoder.decode(&mut reader) {
        Ok(instr) => {
            let text = format!("{}", instr);
            assert!(
                text == expected,
                "display error for {}:\n  decoded: {:?} under decoder {}\n displayed: {}\n expected: {}\n",
                hex,
                instr,
                decoder,
                text,
                expected
            );
            // while we're at it, test that the instruction is as long, and no longer, than its
            // input
            assert_eq!((0u64.wrapping_offset(instr.len()).to_linear()) as usize, data.len(), "instruction length is incorrect, wanted instruction {}", expected);
        },
        Err(e) => {
            assert!(false, "decode error ({}) for {} under decoder {}:\n  expected: {}\n", e, hex, decoder, expected);
        }
    }
}

fn test_avx_full(bytes: &[u8], text: &'static str) {
    // test with a hypothetical CPU that supports all of AVX512. at time of writing, no such CPU
    // exists.
    test_display_under(&InstDecoder::minimal().with_avx512(), bytes, text);
    test_display_under(&InstDecoder::default(), bytes, text);
    test_invalid_under(&InstDecoder::minimal(), bytes);
}

#[allow(non_snake_case)]
#[test]
fn tests_None_0f() {
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x10, 0xca], "vmovups ymm1{k5}{z}, ymm2"); // VMOVUPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x10, 0x0a], "vmovups ymm1{k5}{z}, ymmword [rdx]"); // VMOVUPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x10, 0xca], "vmovups ymm1, ymm2"); // VMOVUPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x10, 0xca], "vmovups ymm1{k5}, ymm2"); // VMOVUPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x10, 0x0a], "vmovups ymm1, ymmword [rdx]"); // VMOVUPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x10, 0x0a], "vmovups ymm1{k5}, ymmword [rdx]"); // VMOVUPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x10, 0xca], "vmovups zmm1{k5}{z}, zmm2"); // VMOVUPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x10, 0x0a], "vmovups zmm1{k5}{z}, zmmword [rdx]"); // VMOVUPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x10, 0xca], "vmovups zmm1, zmm2"); // VMOVUPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x10, 0xca], "vmovups zmm1{k5}, zmm2"); // VMOVUPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x10, 0x0a], "vmovups zmm1, zmmword [rdx]"); // VMOVUPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x10, 0x0a], "vmovups zmm1{k5}, zmmword [rdx]"); // VMOVUPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x10, 0xca], "vmovups xmm1{k5}{z}, xmm2"); // VMOVUPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x10, 0x0a], "vmovups xmm1{k5}{z}, xmmword [rdx]"); // VMOVUPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x10, 0xca], "vmovups xmm1, xmm2"); // VMOVUPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x10, 0xca], "vmovups xmm1{k5}, xmm2"); // VMOVUPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x10, 0x0a], "vmovups xmm1, xmmword [rdx]"); // VMOVUPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x10, 0x0a], "vmovups xmm1{k5}, xmmword [rdx]"); // VMOVUPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x11, 0xca], "vmovups ymm2{k5}{z}, ymm1"); // VMOVUPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x11, 0xca], "vmovups ymm2, ymm1"); // VMOVUPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x11, 0xca], "vmovups ymm2{k5}, ymm1"); // VMOVUPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x11, 0x0a], "vmovups ymmword [rdx], ymm1"); // VMOVUPS_MEMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x11, 0x0a], "vmovups ymmword [rdx]{k5}, ymm1"); // VMOVUPS_MEMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x11, 0xca], "vmovups zmm2{k5}{z}, zmm1"); // VMOVUPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x11, 0xca], "vmovups zmm2, zmm1"); // VMOVUPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x11, 0xca], "vmovups zmm2{k5}, zmm1"); // VMOVUPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x11, 0x0a], "vmovups zmmword [rdx], zmm1"); // VMOVUPS_MEMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x11, 0x0a], "vmovups zmmword [rdx]{k5}, zmm1"); // VMOVUPS_MEMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x11, 0xca], "vmovups xmm2{k5}{z}, xmm1"); // VMOVUPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x11, 0xca], "vmovups xmm2, xmm1"); // VMOVUPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x11, 0xca], "vmovups xmm2{k5}, xmm1"); // VMOVUPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x11, 0x0a], "vmovups xmmword [rdx], xmm1"); // VMOVUPS_MEMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x11, 0x0a], "vmovups xmmword [rdx]{k5}, xmm1"); // VMOVUPS_MEMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x12, 0xca], "vmovhlps xmm1, xmm0, xmm2"); // VMOVHLPS_XMMf32_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x12, 0x0a], "vmovlps xmm1, xmm0, qword [rdx]"); // VMOVLPS_XMMf32_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x13, 0x0a], "vmovlps qword [rdx], xmm1"); // VMOVLPS_MEMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x14, 0x0a], "vunpcklps ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VUNPCKLPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x14, 0x0a], "vunpcklps ymm1, ymm0, dword [rdx]{1to8}"); // VUNPCKLPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x14, 0x0a], "vunpcklps ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VUNPCKLPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x14, 0xca], "vunpcklps ymm1{k5}{z}, ymm0, ymm2"); // VUNPCKLPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x14, 0x0a], "vunpcklps ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VUNPCKLPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x14, 0xca], "vunpcklps ymm1, ymm0, ymm2"); // VUNPCKLPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x14, 0xca], "vunpcklps ymm1{k5}, ymm0, ymm2"); // VUNPCKLPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x14, 0x0a], "vunpcklps ymm1, ymm0, ymmword [rdx]"); // VUNPCKLPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x14, 0x0a], "vunpcklps ymm1{k5}, ymm0, ymmword [rdx]"); // VUNPCKLPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x14, 0x0a], "vunpcklps zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VUNPCKLPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x14, 0x0a], "vunpcklps zmm1, zmm0, dword [rdx]{1to16}"); // VUNPCKLPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x14, 0x0a], "vunpcklps zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VUNPCKLPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x14, 0x0a], "vunpcklps xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VUNPCKLPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x14, 0x0a], "vunpcklps xmm1, xmm0, dword [rdx]{1to4}"); // VUNPCKLPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x14, 0x0a], "vunpcklps xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VUNPCKLPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x14, 0xca], "vunpcklps zmm1{k5}{z}, zmm0, zmm2"); // VUNPCKLPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x14, 0x0a], "vunpcklps zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VUNPCKLPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x14, 0xca], "vunpcklps zmm1, zmm0, zmm2"); // VUNPCKLPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x14, 0xca], "vunpcklps zmm1{k5}, zmm0, zmm2"); // VUNPCKLPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x14, 0x0a], "vunpcklps zmm1, zmm0, zmmword [rdx]"); // VUNPCKLPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x14, 0x0a], "vunpcklps zmm1{k5}, zmm0, zmmword [rdx]"); // VUNPCKLPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x14, 0xca], "vunpcklps xmm1{k5}{z}, xmm0, xmm2"); // VUNPCKLPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x14, 0x0a], "vunpcklps xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VUNPCKLPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x14, 0xca], "vunpcklps xmm1, xmm0, xmm2"); // VUNPCKLPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x14, 0xca], "vunpcklps xmm1{k5}, xmm0, xmm2"); // VUNPCKLPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x14, 0x0a], "vunpcklps xmm1, xmm0, xmmword [rdx]"); // VUNPCKLPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x14, 0x0a], "vunpcklps xmm1{k5}, xmm0, xmmword [rdx]"); // VUNPCKLPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x15, 0x0a], "vunpckhps ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VUNPCKHPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x15, 0x0a], "vunpckhps ymm1, ymm0, dword [rdx]{1to8}"); // VUNPCKHPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x15, 0x0a], "vunpckhps ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VUNPCKHPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x15, 0xca], "vunpckhps ymm1{k5}{z}, ymm0, ymm2"); // VUNPCKHPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x15, 0x0a], "vunpckhps ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VUNPCKHPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x15, 0xca], "vunpckhps ymm1, ymm0, ymm2"); // VUNPCKHPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x15, 0xca], "vunpckhps ymm1{k5}, ymm0, ymm2"); // VUNPCKHPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x15, 0x0a], "vunpckhps ymm1, ymm0, ymmword [rdx]"); // VUNPCKHPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x15, 0x0a], "vunpckhps ymm1{k5}, ymm0, ymmword [rdx]"); // VUNPCKHPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x15, 0x0a], "vunpckhps zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VUNPCKHPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x15, 0x0a], "vunpckhps zmm1, zmm0, dword [rdx]{1to16}"); // VUNPCKHPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x15, 0x0a], "vunpckhps zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VUNPCKHPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x15, 0x0a], "vunpckhps xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VUNPCKHPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x15, 0x0a], "vunpckhps xmm1, xmm0, dword [rdx]{1to4}"); // VUNPCKHPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x15, 0x0a], "vunpckhps xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VUNPCKHPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x15, 0xca], "vunpckhps zmm1{k5}{z}, zmm0, zmm2"); // VUNPCKHPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x15, 0x0a], "vunpckhps zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VUNPCKHPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x15, 0xca], "vunpckhps zmm1, zmm0, zmm2"); // VUNPCKHPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x15, 0xca], "vunpckhps zmm1{k5}, zmm0, zmm2"); // VUNPCKHPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x15, 0x0a], "vunpckhps zmm1, zmm0, zmmword [rdx]"); // VUNPCKHPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x15, 0x0a], "vunpckhps zmm1{k5}, zmm0, zmmword [rdx]"); // VUNPCKHPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x15, 0xca], "vunpckhps xmm1{k5}{z}, xmm0, xmm2"); // VUNPCKHPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x15, 0x0a], "vunpckhps xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VUNPCKHPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x15, 0xca], "vunpckhps xmm1, xmm0, xmm2"); // VUNPCKHPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x15, 0xca], "vunpckhps xmm1{k5}, xmm0, xmm2"); // VUNPCKHPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x15, 0x0a], "vunpckhps xmm1, xmm0, xmmword [rdx]"); // VUNPCKHPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x15, 0x0a], "vunpckhps xmm1{k5}, xmm0, xmmword [rdx]"); // VUNPCKHPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x16, 0xca], "vmovlhps xmm1, xmm0, xmm2"); // VMOVLHPS_XMMf32_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x16, 0x0a], "vmovhps xmm1, xmm0, qword [rdx]"); // VMOVHPS_XMMf32_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x17, 0x0a], "vmovhps qword [rdx], xmm1"); // VMOVHPS_MEMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x28, 0xca], "vmovaps ymm1{k5}{z}, ymm2"); // VMOVAPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x28, 0x0a], "vmovaps ymm1{k5}{z}, ymmword [rdx]"); // VMOVAPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x28, 0xca], "vmovaps ymm1, ymm2"); // VMOVAPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x28, 0xca], "vmovaps ymm1{k5}, ymm2"); // VMOVAPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x28, 0x0a], "vmovaps ymm1, ymmword [rdx]"); // VMOVAPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x28, 0x0a], "vmovaps ymm1{k5}, ymmword [rdx]"); // VMOVAPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x28, 0xca], "vmovaps zmm1{k5}{z}, zmm2"); // VMOVAPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x28, 0x0a], "vmovaps zmm1{k5}{z}, zmmword [rdx]"); // VMOVAPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x28, 0xca], "vmovaps zmm1, zmm2"); // VMOVAPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x28, 0xca], "vmovaps zmm1{k5}, zmm2"); // VMOVAPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x28, 0x0a], "vmovaps zmm1, zmmword [rdx]"); // VMOVAPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x28, 0x0a], "vmovaps zmm1{k5}, zmmword [rdx]"); // VMOVAPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x28, 0xca], "vmovaps xmm1{k5}{z}, xmm2"); // VMOVAPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x28, 0x0a], "vmovaps xmm1{k5}{z}, xmmword [rdx]"); // VMOVAPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x28, 0xca], "vmovaps xmm1, xmm2"); // VMOVAPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x28, 0xca], "vmovaps xmm1{k5}, xmm2"); // VMOVAPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x28, 0x0a], "vmovaps xmm1, xmmword [rdx]"); // VMOVAPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x28, 0x0a], "vmovaps xmm1{k5}, xmmword [rdx]"); // VMOVAPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x29, 0xca], "vmovaps ymm2{k5}{z}, ymm1"); // VMOVAPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x29, 0xca], "vmovaps ymm2, ymm1"); // VMOVAPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x29, 0xca], "vmovaps ymm2{k5}, ymm1"); // VMOVAPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x29, 0x0a], "vmovaps ymmword [rdx], ymm1"); // VMOVAPS_MEMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x29, 0x0a], "vmovaps ymmword [rdx]{k5}, ymm1"); // VMOVAPS_MEMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x29, 0xca], "vmovaps zmm2{k5}{z}, zmm1"); // VMOVAPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x29, 0xca], "vmovaps zmm2, zmm1"); // VMOVAPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x29, 0xca], "vmovaps zmm2{k5}, zmm1"); // VMOVAPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x29, 0x0a], "vmovaps zmmword [rdx], zmm1"); // VMOVAPS_MEMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x29, 0x0a], "vmovaps zmmword [rdx]{k5}, zmm1"); // VMOVAPS_MEMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x29, 0xca], "vmovaps xmm2{k5}{z}, xmm1"); // VMOVAPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x29, 0xca], "vmovaps xmm2, xmm1"); // VMOVAPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x29, 0xca], "vmovaps xmm2{k5}, xmm1"); // VMOVAPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x29, 0x0a], "vmovaps xmmword [rdx], xmm1"); // VMOVAPS_MEMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x29, 0x0a], "vmovaps xmmword [rdx]{k5}, xmm1"); // VMOVAPS_MEMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x2b, 0x0a], "vmovntps ymmword [rdx], ymm1"); // VMOVNTPS_MEMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x2b, 0x0a], "vmovntps zmmword [rdx], zmm1"); // VMOVNTPS_MEMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x2b, 0x0a], "vmovntps xmmword [rdx], xmm1"); // VMOVNTPS_MEMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x2e, 0xca], "vucomiss xmm1{sae}, xmm2"); // VUCOMISS_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x2e, 0xca], "vucomiss xmm1, xmm2"); // VUCOMISS_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x2e, 0x0a], "vucomiss xmm1, dword [rdx]"); // VUCOMISS_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x2f, 0xca], "vcomiss xmm1{sae}, xmm2"); // VCOMISS_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x2f, 0xca], "vcomiss xmm1, xmm2"); // VCOMISS_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x2f, 0x0a], "vcomiss xmm1, dword [rdx]"); // VCOMISS_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x51, 0xca], "vsqrtps zmm1{k5}{z}{rz-sae}, zmm2"); // VSQRTPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x51, 0xca], "vsqrtps zmm1{rz-sae}, zmm2"); // VSQRTPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x51, 0xca], "vsqrtps zmm1{k5}{rz-sae}, zmm2"); // VSQRTPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x51, 0xca], "vsqrtps zmm1{k5}{z}{rd-sae}, zmm2"); // VSQRTPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x51, 0x0a], "vsqrtps ymm1{k5}{z}, dword [rdx]{1to8}"); // VSQRTPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x51, 0xca], "vsqrtps zmm1{rd-sae}, zmm2"); // VSQRTPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x51, 0xca], "vsqrtps zmm1{k5}{rd-sae}, zmm2"); // VSQRTPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x51, 0x0a], "vsqrtps ymm1, dword [rdx]{1to8}"); // VSQRTPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x51, 0x0a], "vsqrtps ymm1{k5}, dword [rdx]{1to8}"); // VSQRTPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x51, 0xca], "vsqrtps ymm1{k5}{z}, ymm2"); // VSQRTPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x51, 0x0a], "vsqrtps ymm1{k5}{z}, ymmword [rdx]"); // VSQRTPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x51, 0xca], "vsqrtps ymm1, ymm2"); // VSQRTPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x51, 0xca], "vsqrtps ymm1{k5}, ymm2"); // VSQRTPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x51, 0x0a], "vsqrtps ymm1, ymmword [rdx]"); // VSQRTPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x51, 0x0a], "vsqrtps ymm1{k5}, ymmword [rdx]"); // VSQRTPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x51, 0xca], "vsqrtps zmm1{k5}{z}{ru-sae}, zmm2"); // VSQRTPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x51, 0x0a], "vsqrtps zmm1{k5}{z}, dword [rdx]{1to16}"); // VSQRTPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x51, 0xca], "vsqrtps zmm1{ru-sae}, zmm2"); // VSQRTPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x51, 0xca], "vsqrtps zmm1{k5}{ru-sae}, zmm2"); // VSQRTPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x51, 0x0a], "vsqrtps zmm1, dword [rdx]{1to16}"); // VSQRTPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x51, 0x0a], "vsqrtps zmm1{k5}, dword [rdx]{1to16}"); // VSQRTPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x51, 0xca], "vsqrtps zmm1{k5}{z}{rne-sae}, zmm2"); // VSQRTPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x51, 0x0a], "vsqrtps xmm1{k5}{z}, dword [rdx]{1to4}"); // VSQRTPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x51, 0xca], "vsqrtps zmm1{rne-sae}, zmm2"); // VSQRTPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x51, 0xca], "vsqrtps zmm1{k5}{rne-sae}, zmm2"); // VSQRTPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x51, 0x0a], "vsqrtps xmm1, dword [rdx]{1to4}"); // VSQRTPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x51, 0x0a], "vsqrtps xmm1{k5}, dword [rdx]{1to4}"); // VSQRTPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x51, 0xca], "vsqrtps zmm1{k5}{z}, zmm2"); // VSQRTPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x51, 0x0a], "vsqrtps zmm1{k5}{z}, zmmword [rdx]"); // VSQRTPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x51, 0xca], "vsqrtps zmm1, zmm2"); // VSQRTPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x51, 0xca], "vsqrtps zmm1{k5}, zmm2"); // VSQRTPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x51, 0x0a], "vsqrtps zmm1, zmmword [rdx]"); // VSQRTPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x51, 0x0a], "vsqrtps zmm1{k5}, zmmword [rdx]"); // VSQRTPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x51, 0xca], "vsqrtps xmm1{k5}{z}, xmm2"); // VSQRTPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x51, 0x0a], "vsqrtps xmm1{k5}{z}, xmmword [rdx]"); // VSQRTPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x51, 0xca], "vsqrtps xmm1, xmm2"); // VSQRTPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x51, 0xca], "vsqrtps xmm1{k5}, xmm2"); // VSQRTPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x51, 0x0a], "vsqrtps xmm1, xmmword [rdx]"); // VSQRTPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x51, 0x0a], "vsqrtps xmm1{k5}, xmmword [rdx]"); // VSQRTPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x54, 0x0a], "vandps ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VANDPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x54, 0x0a], "vandps ymm1, ymm0, dword [rdx]{1to8}"); // VANDPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x54, 0x0a], "vandps ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VANDPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x54, 0xca], "vandps ymm1{k5}{z}, ymm0, ymm2"); // VANDPS_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x54, 0x0a], "vandps ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VANDPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x54, 0xca], "vandps ymm1, ymm0, ymm2"); // VANDPS_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x54, 0xca], "vandps ymm1{k5}, ymm0, ymm2"); // VANDPS_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x54, 0x0a], "vandps ymm1, ymm0, ymmword [rdx]"); // VANDPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x54, 0x0a], "vandps ymm1{k5}, ymm0, ymmword [rdx]"); // VANDPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x54, 0x0a], "vandps zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VANDPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x54, 0x0a], "vandps zmm1, zmm0, dword [rdx]{1to16}"); // VANDPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x54, 0x0a], "vandps zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VANDPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x54, 0x0a], "vandps xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VANDPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x54, 0x0a], "vandps xmm1, xmm0, dword [rdx]{1to4}"); // VANDPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x54, 0x0a], "vandps xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VANDPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x54, 0xca], "vandps zmm1{k5}{z}, zmm0, zmm2"); // VANDPS_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x54, 0x0a], "vandps zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VANDPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x54, 0xca], "vandps zmm1, zmm0, zmm2"); // VANDPS_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x54, 0xca], "vandps zmm1{k5}, zmm0, zmm2"); // VANDPS_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x54, 0x0a], "vandps zmm1, zmm0, zmmword [rdx]"); // VANDPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x54, 0x0a], "vandps zmm1{k5}, zmm0, zmmword [rdx]"); // VANDPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x54, 0xca], "vandps xmm1{k5}{z}, xmm0, xmm2"); // VANDPS_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x54, 0x0a], "vandps xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VANDPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x54, 0xca], "vandps xmm1, xmm0, xmm2"); // VANDPS_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x54, 0xca], "vandps xmm1{k5}, xmm0, xmm2"); // VANDPS_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x54, 0x0a], "vandps xmm1, xmm0, xmmword [rdx]"); // VANDPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x54, 0x0a], "vandps xmm1{k5}, xmm0, xmmword [rdx]"); // VANDPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x55, 0x0a], "vandnps ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VANDNPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x55, 0x0a], "vandnps ymm1, ymm0, dword [rdx]{1to8}"); // VANDNPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x55, 0x0a], "vandnps ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VANDNPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x55, 0xca], "vandnps ymm1{k5}{z}, ymm0, ymm2"); // VANDNPS_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x55, 0x0a], "vandnps ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VANDNPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x55, 0xca], "vandnps ymm1, ymm0, ymm2"); // VANDNPS_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x55, 0xca], "vandnps ymm1{k5}, ymm0, ymm2"); // VANDNPS_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x55, 0x0a], "vandnps ymm1, ymm0, ymmword [rdx]"); // VANDNPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x55, 0x0a], "vandnps ymm1{k5}, ymm0, ymmword [rdx]"); // VANDNPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x55, 0x0a], "vandnps zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VANDNPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x55, 0x0a], "vandnps zmm1, zmm0, dword [rdx]{1to16}"); // VANDNPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x55, 0x0a], "vandnps zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VANDNPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x55, 0x0a], "vandnps xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VANDNPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x55, 0x0a], "vandnps xmm1, xmm0, dword [rdx]{1to4}"); // VANDNPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x55, 0x0a], "vandnps xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VANDNPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x55, 0xca], "vandnps zmm1{k5}{z}, zmm0, zmm2"); // VANDNPS_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x55, 0x0a], "vandnps zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VANDNPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x55, 0xca], "vandnps zmm1, zmm0, zmm2"); // VANDNPS_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x55, 0xca], "vandnps zmm1{k5}, zmm0, zmm2"); // VANDNPS_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x55, 0x0a], "vandnps zmm1, zmm0, zmmword [rdx]"); // VANDNPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x55, 0x0a], "vandnps zmm1{k5}, zmm0, zmmword [rdx]"); // VANDNPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x55, 0xca], "vandnps xmm1{k5}{z}, xmm0, xmm2"); // VANDNPS_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x55, 0x0a], "vandnps xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VANDNPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x55, 0xca], "vandnps xmm1, xmm0, xmm2"); // VANDNPS_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x55, 0xca], "vandnps xmm1{k5}, xmm0, xmm2"); // VANDNPS_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x55, 0x0a], "vandnps xmm1, xmm0, xmmword [rdx]"); // VANDNPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x55, 0x0a], "vandnps xmm1{k5}, xmm0, xmmword [rdx]"); // VANDNPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x56, 0x0a], "vorps ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VORPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x56, 0x0a], "vorps ymm1, ymm0, dword [rdx]{1to8}"); // VORPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x56, 0x0a], "vorps ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VORPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x56, 0xca], "vorps ymm1{k5}{z}, ymm0, ymm2"); // VORPS_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x56, 0x0a], "vorps ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VORPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x56, 0xca], "vorps ymm1, ymm0, ymm2"); // VORPS_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x56, 0xca], "vorps ymm1{k5}, ymm0, ymm2"); // VORPS_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x56, 0x0a], "vorps ymm1, ymm0, ymmword [rdx]"); // VORPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x56, 0x0a], "vorps ymm1{k5}, ymm0, ymmword [rdx]"); // VORPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x56, 0x0a], "vorps zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VORPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x56, 0x0a], "vorps zmm1, zmm0, dword [rdx]{1to16}"); // VORPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x56, 0x0a], "vorps zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VORPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x56, 0x0a], "vorps xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VORPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x56, 0x0a], "vorps xmm1, xmm0, dword [rdx]{1to4}"); // VORPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x56, 0x0a], "vorps xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VORPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x56, 0xca], "vorps zmm1{k5}{z}, zmm0, zmm2"); // VORPS_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x56, 0x0a], "vorps zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VORPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x56, 0xca], "vorps zmm1, zmm0, zmm2"); // VORPS_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x56, 0xca], "vorps zmm1{k5}, zmm0, zmm2"); // VORPS_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x56, 0x0a], "vorps zmm1, zmm0, zmmword [rdx]"); // VORPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x56, 0x0a], "vorps zmm1{k5}, zmm0, zmmword [rdx]"); // VORPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x56, 0xca], "vorps xmm1{k5}{z}, xmm0, xmm2"); // VORPS_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x56, 0x0a], "vorps xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VORPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x56, 0xca], "vorps xmm1, xmm0, xmm2"); // VORPS_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x56, 0xca], "vorps xmm1{k5}, xmm0, xmm2"); // VORPS_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x56, 0x0a], "vorps xmm1, xmm0, xmmword [rdx]"); // VORPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x56, 0x0a], "vorps xmm1{k5}, xmm0, xmmword [rdx]"); // VORPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x57, 0x0a], "vxorps ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VXORPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x57, 0x0a], "vxorps ymm1, ymm0, dword [rdx]{1to8}"); // VXORPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x57, 0x0a], "vxorps ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VXORPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x57, 0xca], "vxorps ymm1{k5}{z}, ymm0, ymm2"); // VXORPS_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x57, 0x0a], "vxorps ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VXORPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x57, 0xca], "vxorps ymm1, ymm0, ymm2"); // VXORPS_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x57, 0xca], "vxorps ymm1{k5}, ymm0, ymm2"); // VXORPS_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x57, 0x0a], "vxorps ymm1, ymm0, ymmword [rdx]"); // VXORPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x57, 0x0a], "vxorps ymm1{k5}, ymm0, ymmword [rdx]"); // VXORPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x57, 0x0a], "vxorps zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VXORPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x57, 0x0a], "vxorps zmm1, zmm0, dword [rdx]{1to16}"); // VXORPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x57, 0x0a], "vxorps zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VXORPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x57, 0x0a], "vxorps xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VXORPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x57, 0x0a], "vxorps xmm1, xmm0, dword [rdx]{1to4}"); // VXORPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x57, 0x0a], "vxorps xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VXORPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x57, 0xca], "vxorps zmm1{k5}{z}, zmm0, zmm2"); // VXORPS_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x57, 0x0a], "vxorps zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VXORPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x57, 0xca], "vxorps zmm1, zmm0, zmm2"); // VXORPS_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x57, 0xca], "vxorps zmm1{k5}, zmm0, zmm2"); // VXORPS_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x57, 0x0a], "vxorps zmm1, zmm0, zmmword [rdx]"); // VXORPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x57, 0x0a], "vxorps zmm1{k5}, zmm0, zmmword [rdx]"); // VXORPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x57, 0xca], "vxorps xmm1{k5}{z}, xmm0, xmm2"); // VXORPS_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x57, 0x0a], "vxorps xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VXORPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x57, 0xca], "vxorps xmm1, xmm0, xmm2"); // VXORPS_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x57, 0xca], "vxorps xmm1{k5}, xmm0, xmm2"); // VXORPS_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x57, 0x0a], "vxorps xmm1, xmm0, xmmword [rdx]"); // VXORPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x57, 0x0a], "vxorps xmm1{k5}, xmm0, xmmword [rdx]"); // VXORPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x58, 0xca], "vaddps zmm1{k5}{z}{rz-sae}, zmm0, zmm2"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x58, 0xca], "vaddps zmm1{rz-sae}, zmm0, zmm2"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x58, 0xca], "vaddps zmm1{k5}{rz-sae}, zmm0, zmm2"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x58, 0xca], "vaddps zmm1{k5}{z}{rd-sae}, zmm0, zmm2"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x58, 0x0a], "vaddps ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VADDPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x58, 0xca], "vaddps zmm1{rd-sae}, zmm0, zmm2"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x58, 0xca], "vaddps zmm1{k5}{rd-sae}, zmm0, zmm2"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x58, 0x0a], "vaddps ymm1, ymm0, dword [rdx]{1to8}"); // VADDPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x58, 0x0a], "vaddps ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VADDPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x58, 0xca], "vaddps ymm1{k5}{z}, ymm0, ymm2"); // VADDPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x58, 0x0a], "vaddps ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VADDPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x58, 0xca], "vaddps ymm1, ymm0, ymm2"); // VADDPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x58, 0xca], "vaddps ymm1{k5}, ymm0, ymm2"); // VADDPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x58, 0x0a], "vaddps ymm1, ymm0, ymmword [rdx]"); // VADDPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x58, 0x0a], "vaddps ymm1{k5}, ymm0, ymmword [rdx]"); // VADDPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x58, 0xca], "vaddps zmm1{k5}{z}{ru-sae}, zmm0, zmm2"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x58, 0x0a], "vaddps zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x58, 0xca], "vaddps zmm1{ru-sae}, zmm0, zmm2"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x58, 0xca], "vaddps zmm1{k5}{ru-sae}, zmm0, zmm2"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x58, 0x0a], "vaddps zmm1, zmm0, dword [rdx]{1to16}"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x58, 0x0a], "vaddps zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x58, 0xca], "vaddps zmm1{k5}{z}{rne-sae}, zmm0, zmm2"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x58, 0x0a], "vaddps xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VADDPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x58, 0xca], "vaddps zmm1{rne-sae}, zmm0, zmm2"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x58, 0xca], "vaddps zmm1{k5}{rne-sae}, zmm0, zmm2"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x58, 0x0a], "vaddps xmm1, xmm0, dword [rdx]{1to4}"); // VADDPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x58, 0x0a], "vaddps xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VADDPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x58, 0xca], "vaddps zmm1{k5}{z}, zmm0, zmm2"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x58, 0x0a], "vaddps zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x58, 0xca], "vaddps zmm1, zmm0, zmm2"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x58, 0xca], "vaddps zmm1{k5}, zmm0, zmm2"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x58, 0x0a], "vaddps zmm1, zmm0, zmmword [rdx]"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x58, 0x0a], "vaddps zmm1{k5}, zmm0, zmmword [rdx]"); // VADDPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x58, 0xca], "vaddps xmm1{k5}{z}, xmm0, xmm2"); // VADDPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x58, 0x0a], "vaddps xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VADDPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x58, 0xca], "vaddps xmm1, xmm0, xmm2"); // VADDPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x58, 0xca], "vaddps xmm1{k5}, xmm0, xmm2"); // VADDPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x58, 0x0a], "vaddps xmm1, xmm0, xmmword [rdx]"); // VADDPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x58, 0x0a], "vaddps xmm1{k5}, xmm0, xmmword [rdx]"); // VADDPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x59, 0xca], "vmulps zmm1{k5}{z}{rz-sae}, zmm0, zmm2"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x59, 0xca], "vmulps zmm1{rz-sae}, zmm0, zmm2"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x59, 0xca], "vmulps zmm1{k5}{rz-sae}, zmm0, zmm2"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x59, 0xca], "vmulps zmm1{k5}{z}{rd-sae}, zmm0, zmm2"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x59, 0x0a], "vmulps ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VMULPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x59, 0xca], "vmulps zmm1{rd-sae}, zmm0, zmm2"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x59, 0xca], "vmulps zmm1{k5}{rd-sae}, zmm0, zmm2"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x59, 0x0a], "vmulps ymm1, ymm0, dword [rdx]{1to8}"); // VMULPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x59, 0x0a], "vmulps ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VMULPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x59, 0xca], "vmulps ymm1{k5}{z}, ymm0, ymm2"); // VMULPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x59, 0x0a], "vmulps ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VMULPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x59, 0xca], "vmulps ymm1, ymm0, ymm2"); // VMULPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x59, 0xca], "vmulps ymm1{k5}, ymm0, ymm2"); // VMULPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x59, 0x0a], "vmulps ymm1, ymm0, ymmword [rdx]"); // VMULPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x59, 0x0a], "vmulps ymm1{k5}, ymm0, ymmword [rdx]"); // VMULPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x59, 0xca], "vmulps zmm1{k5}{z}{ru-sae}, zmm0, zmm2"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x59, 0x0a], "vmulps zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x59, 0xca], "vmulps zmm1{ru-sae}, zmm0, zmm2"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x59, 0xca], "vmulps zmm1{k5}{ru-sae}, zmm0, zmm2"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x59, 0x0a], "vmulps zmm1, zmm0, dword [rdx]{1to16}"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x59, 0x0a], "vmulps zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x59, 0xca], "vmulps zmm1{k5}{z}{rne-sae}, zmm0, zmm2"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x59, 0x0a], "vmulps xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VMULPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x59, 0xca], "vmulps zmm1{rne-sae}, zmm0, zmm2"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x59, 0xca], "vmulps zmm1{k5}{rne-sae}, zmm0, zmm2"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x59, 0x0a], "vmulps xmm1, xmm0, dword [rdx]{1to4}"); // VMULPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x59, 0x0a], "vmulps xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VMULPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x59, 0xca], "vmulps zmm1{k5}{z}, zmm0, zmm2"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x59, 0x0a], "vmulps zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x59, 0xca], "vmulps zmm1, zmm0, zmm2"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x59, 0xca], "vmulps zmm1{k5}, zmm0, zmm2"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x59, 0x0a], "vmulps zmm1, zmm0, zmmword [rdx]"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x59, 0x0a], "vmulps zmm1{k5}, zmm0, zmmword [rdx]"); // VMULPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x59, 0xca], "vmulps xmm1{k5}{z}, xmm0, xmm2"); // VMULPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x59, 0x0a], "vmulps xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VMULPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x59, 0xca], "vmulps xmm1, xmm0, xmm2"); // VMULPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x59, 0xca], "vmulps xmm1{k5}, xmm0, xmm2"); // VMULPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x59, 0x0a], "vmulps xmm1, xmm0, xmmword [rdx]"); // VMULPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x59, 0x0a], "vmulps xmm1{k5}, xmm0, xmmword [rdx]"); // VMULPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x5a, 0xca], "vcvtps2pd zmm1{k5}{z}{sae}, ymm2"); // VCVTPS2PD_ZMMf64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x5a, 0xca], "vcvtps2pd zmm1{sae}, ymm2"); // VCVTPS2PD_ZMMf64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x5a, 0xca], "vcvtps2pd zmm1{k5}{sae}, ymm2"); // VCVTPS2PD_ZMMf64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5a, 0x0a], "vcvtps2pd ymm1{k5}{z}, dword [rdx]{1to4}"); // VCVTPS2PD_YMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5a, 0x0a], "vcvtps2pd ymm1, dword [rdx]{1to4}"); // VCVTPS2PD_YMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5a, 0x0a], "vcvtps2pd ymm1{k5}, dword [rdx]{1to4}"); // VCVTPS2PD_YMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5a, 0xca], "vcvtps2pd ymm1{k5}{z}, xmm2"); // VCVTPS2PD_YMMf64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5a, 0x0a], "vcvtps2pd ymm1{k5}{z}, xmmword [rdx]"); // VCVTPS2PD_YMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5a, 0xca], "vcvtps2pd ymm1, xmm2"); // VCVTPS2PD_YMMf64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5a, 0xca], "vcvtps2pd ymm1{k5}, xmm2"); // VCVTPS2PD_YMMf64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5a, 0x0a], "vcvtps2pd ymm1, xmmword [rdx]"); // VCVTPS2PD_YMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5a, 0x0a], "vcvtps2pd ymm1{k5}, xmmword [rdx]"); // VCVTPS2PD_YMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5a, 0x0a], "vcvtps2pd zmm1{k5}{z}, dword [rdx]{1to8}"); // VCVTPS2PD_ZMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5a, 0x0a], "vcvtps2pd zmm1, dword [rdx]{1to8}"); // VCVTPS2PD_ZMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5a, 0x0a], "vcvtps2pd zmm1{k5}, dword [rdx]{1to8}"); // VCVTPS2PD_ZMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5a, 0x0a], "vcvtps2pd xmm1{k5}{z}, dword [rdx]{1to2}"); // VCVTPS2PD_XMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5a, 0x0a], "vcvtps2pd xmm1, dword [rdx]{1to2}"); // VCVTPS2PD_XMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5a, 0x0a], "vcvtps2pd xmm1{k5}, dword [rdx]{1to2}"); // VCVTPS2PD_XMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5a, 0xca], "vcvtps2pd zmm1{k5}{z}, ymm2"); // VCVTPS2PD_ZMMf64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5a, 0x0a], "vcvtps2pd zmm1{k5}{z}, ymmword [rdx]"); // VCVTPS2PD_ZMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5a, 0xca], "vcvtps2pd zmm1, ymm2"); // VCVTPS2PD_ZMMf64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5a, 0xca], "vcvtps2pd zmm1{k5}, ymm2"); // VCVTPS2PD_ZMMf64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5a, 0x0a], "vcvtps2pd zmm1, ymmword [rdx]"); // VCVTPS2PD_ZMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5a, 0x0a], "vcvtps2pd zmm1{k5}, ymmword [rdx]"); // VCVTPS2PD_ZMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5a, 0xca], "vcvtps2pd xmm1{k5}{z}, xmm2"); // VCVTPS2PD_XMMf64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5a, 0x0a], "vcvtps2pd xmm1{k5}{z}, qword [rdx]"); // VCVTPS2PD_XMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5a, 0xca], "vcvtps2pd xmm1, xmm2"); // VCVTPS2PD_XMMf64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5a, 0xca], "vcvtps2pd xmm1{k5}, xmm2"); // VCVTPS2PD_XMMf64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5a, 0x0a], "vcvtps2pd xmm1, qword [rdx]"); // VCVTPS2PD_XMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5a, 0x0a], "vcvtps2pd xmm1{k5}, qword [rdx]"); // VCVTPS2PD_XMMf64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xfd, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{z}{rz-sae}, zmm2"); // VCVTQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x78, 0x5b, 0xca], "vcvtqq2ps ymm1{rz-sae}, zmm2"); // VCVTQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x7d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{rz-sae}, zmm2"); // VCVTQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xbd, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{z}{rd-sae}, zmm2"); // VCVTQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xbd, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}{z}, qword [rdx]{1to4}"); // VCVTQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x38, 0x5b, 0xca], "vcvtqq2ps ymm1{rd-sae}, zmm2"); // VCVTQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x3d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{rd-sae}, zmm2"); // VCVTQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x38, 0x5b, 0x0a], "vcvtqq2ps xmm1, qword [rdx]{1to4}"); // VCVTQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x3d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}, qword [rdx]{1to4}"); // VCVTQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xad, 0x5b, 0xca], "vcvtqq2ps xmm1{k5}{z}, ymm2"); // VCVTQQ2PS_XMMf32_MASKmskw_YMMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xad, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}{z}, ymmword [rdx]"); // VCVTQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x28, 0x5b, 0xca], "vcvtqq2ps xmm1, ymm2"); // VCVTQQ2PS_XMMf32_MASKmskw_YMMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x2d, 0x5b, 0xca], "vcvtqq2ps xmm1{k5}, ymm2"); // VCVTQQ2PS_XMMf32_MASKmskw_YMMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x28, 0x5b, 0x0a], "vcvtqq2ps xmm1, ymmword [rdx]"); // VCVTQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x2d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}, ymmword [rdx]"); // VCVTQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{z}{rz-sae}, zmm2"); // VCVTDQ2PS_ZMMf32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x5b, 0xca], "vcvtdq2ps zmm1{rz-sae}, zmm2"); // VCVTDQ2PS_ZMMf32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{rz-sae}, zmm2"); // VCVTDQ2PS_ZMMf32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{z}{rd-sae}, zmm2"); // VCVTDQ2PS_ZMMf32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5b, 0x0a], "vcvtdq2ps ymm1{k5}{z}, dword [rdx]{1to8}"); // VCVTDQ2PS_YMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5b, 0xca], "vcvtdq2ps zmm1{rd-sae}, zmm2"); // VCVTDQ2PS_ZMMf32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{rd-sae}, zmm2"); // VCVTDQ2PS_ZMMf32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5b, 0x0a], "vcvtdq2ps ymm1, dword [rdx]{1to8}"); // VCVTDQ2PS_YMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5b, 0x0a], "vcvtdq2ps ymm1{k5}, dword [rdx]{1to8}"); // VCVTDQ2PS_YMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5b, 0xca], "vcvtdq2ps ymm1{k5}{z}, ymm2"); // VCVTDQ2PS_YMMf32_MASKmskw_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5b, 0x0a], "vcvtdq2ps ymm1{k5}{z}, ymmword [rdx]"); // VCVTDQ2PS_YMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5b, 0xca], "vcvtdq2ps ymm1, ymm2"); // VCVTDQ2PS_YMMf32_MASKmskw_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5b, 0xca], "vcvtdq2ps ymm1{k5}, ymm2"); // VCVTDQ2PS_YMMf32_MASKmskw_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5b, 0x0a], "vcvtdq2ps ymm1, ymmword [rdx]"); // VCVTDQ2PS_YMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5b, 0x0a], "vcvtdq2ps ymm1{k5}, ymmword [rdx]"); // VCVTDQ2PS_YMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xdd, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{z}{ru-sae}, zmm2"); // VCVTQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xdd, 0x5b, 0x0a], "vcvtqq2ps ymm1{k5}{z}, qword [rdx]{1to8}"); // VCVTQQ2PS_YMMf32_MASKmskw_MEMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x58, 0x5b, 0xca], "vcvtqq2ps ymm1{ru-sae}, zmm2"); // VCVTQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x5d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{ru-sae}, zmm2"); // VCVTQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x58, 0x5b, 0x0a], "vcvtqq2ps ymm1, qword [rdx]{1to8}"); // VCVTQQ2PS_YMMf32_MASKmskw_MEMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x5d, 0x5b, 0x0a], "vcvtqq2ps ymm1{k5}, qword [rdx]{1to8}"); // VCVTQQ2PS_YMMf32_MASKmskw_MEMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x9d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{z}{rne-sae}, zmm2"); // VCVTQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x9d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}{z}, qword [rdx]{1to2}"); // VCVTQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x18, 0x5b, 0xca], "vcvtqq2ps ymm1{rne-sae}, zmm2"); // VCVTQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x1d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{rne-sae}, zmm2"); // VCVTQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x18, 0x5b, 0x0a], "vcvtqq2ps xmm1, qword [rdx]{1to2}"); // VCVTQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x1d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}, qword [rdx]{1to2}"); // VCVTQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xcd, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}{z}, zmm2"); // VCVTQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xcd, 0x5b, 0x0a], "vcvtqq2ps ymm1{k5}{z}, zmmword [rdx]"); // VCVTQQ2PS_YMMf32_MASKmskw_MEMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x48, 0x5b, 0xca], "vcvtqq2ps ymm1, zmm2"); // VCVTQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x4d, 0x5b, 0xca], "vcvtqq2ps ymm1{k5}, zmm2"); // VCVTQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x48, 0x5b, 0x0a], "vcvtqq2ps ymm1, zmmword [rdx]"); // VCVTQQ2PS_YMMf32_MASKmskw_MEMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x4d, 0x5b, 0x0a], "vcvtqq2ps ymm1{k5}, zmmword [rdx]"); // VCVTQQ2PS_YMMf32_MASKmskw_MEMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x8d, 0x5b, 0xca], "vcvtqq2ps xmm1{k5}{z}, xmm2"); // VCVTQQ2PS_XMMf32_MASKmskw_XMMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x8d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}{z}, xmmword [rdx]"); // VCVTQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x08, 0x5b, 0xca], "vcvtqq2ps xmm1, xmm2"); // VCVTQQ2PS_XMMf32_MASKmskw_XMMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x0d, 0x5b, 0xca], "vcvtqq2ps xmm1{k5}, xmm2"); // VCVTQQ2PS_XMMf32_MASKmskw_XMMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x08, 0x5b, 0x0a], "vcvtqq2ps xmm1, xmmword [rdx]"); // VCVTQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x0d, 0x5b, 0x0a], "vcvtqq2ps xmm1{k5}, xmmword [rdx]"); // VCVTQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{z}{ru-sae}, zmm2"); // VCVTDQ2PS_ZMMf32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5b, 0x0a], "vcvtdq2ps zmm1{k5}{z}, dword [rdx]{1to16}"); // VCVTDQ2PS_ZMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5b, 0xca], "vcvtdq2ps zmm1{ru-sae}, zmm2"); // VCVTDQ2PS_ZMMf32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{ru-sae}, zmm2"); // VCVTDQ2PS_ZMMf32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5b, 0x0a], "vcvtdq2ps zmm1, dword [rdx]{1to16}"); // VCVTDQ2PS_ZMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5b, 0x0a], "vcvtdq2ps zmm1{k5}, dword [rdx]{1to16}"); // VCVTDQ2PS_ZMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{z}{rne-sae}, zmm2"); // VCVTDQ2PS_ZMMf32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5b, 0x0a], "vcvtdq2ps xmm1{k5}{z}, dword [rdx]{1to4}"); // VCVTDQ2PS_XMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5b, 0xca], "vcvtdq2ps zmm1{rne-sae}, zmm2"); // VCVTDQ2PS_ZMMf32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{rne-sae}, zmm2"); // VCVTDQ2PS_ZMMf32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5b, 0x0a], "vcvtdq2ps xmm1, dword [rdx]{1to4}"); // VCVTDQ2PS_XMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5b, 0x0a], "vcvtdq2ps xmm1{k5}, dword [rdx]{1to4}"); // VCVTDQ2PS_XMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}{z}, zmm2"); // VCVTDQ2PS_ZMMf32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5b, 0x0a], "vcvtdq2ps zmm1{k5}{z}, zmmword [rdx]"); // VCVTDQ2PS_ZMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5b, 0xca], "vcvtdq2ps zmm1, zmm2"); // VCVTDQ2PS_ZMMf32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5b, 0xca], "vcvtdq2ps zmm1{k5}, zmm2"); // VCVTDQ2PS_ZMMf32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5b, 0x0a], "vcvtdq2ps zmm1, zmmword [rdx]"); // VCVTDQ2PS_ZMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5b, 0x0a], "vcvtdq2ps zmm1{k5}, zmmword [rdx]"); // VCVTDQ2PS_ZMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5b, 0xca], "vcvtdq2ps xmm1{k5}{z}, xmm2"); // VCVTDQ2PS_XMMf32_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5b, 0x0a], "vcvtdq2ps xmm1{k5}{z}, xmmword [rdx]"); // VCVTDQ2PS_XMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5b, 0xca], "vcvtdq2ps xmm1, xmm2"); // VCVTDQ2PS_XMMf32_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5b, 0xca], "vcvtdq2ps xmm1{k5}, xmm2"); // VCVTDQ2PS_XMMf32_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5b, 0x0a], "vcvtdq2ps xmm1, xmmword [rdx]"); // VCVTDQ2PS_XMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5b, 0x0a], "vcvtdq2ps xmm1{k5}, xmmword [rdx]"); // VCVTDQ2PS_XMMf32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x5c, 0xca], "vsubps zmm1{k5}{z}{rz-sae}, zmm0, zmm2"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x5c, 0xca], "vsubps zmm1{rz-sae}, zmm0, zmm2"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x5c, 0xca], "vsubps zmm1{k5}{rz-sae}, zmm0, zmm2"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5c, 0xca], "vsubps zmm1{k5}{z}{rd-sae}, zmm0, zmm2"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5c, 0x0a], "vsubps ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VSUBPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5c, 0xca], "vsubps zmm1{rd-sae}, zmm0, zmm2"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5c, 0xca], "vsubps zmm1{k5}{rd-sae}, zmm0, zmm2"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5c, 0x0a], "vsubps ymm1, ymm0, dword [rdx]{1to8}"); // VSUBPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5c, 0x0a], "vsubps ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VSUBPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5c, 0xca], "vsubps ymm1{k5}{z}, ymm0, ymm2"); // VSUBPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5c, 0x0a], "vsubps ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VSUBPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5c, 0xca], "vsubps ymm1, ymm0, ymm2"); // VSUBPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5c, 0xca], "vsubps ymm1{k5}, ymm0, ymm2"); // VSUBPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5c, 0x0a], "vsubps ymm1, ymm0, ymmword [rdx]"); // VSUBPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5c, 0x0a], "vsubps ymm1{k5}, ymm0, ymmword [rdx]"); // VSUBPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5c, 0xca], "vsubps zmm1{k5}{z}{ru-sae}, zmm0, zmm2"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5c, 0x0a], "vsubps zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5c, 0xca], "vsubps zmm1{ru-sae}, zmm0, zmm2"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5c, 0xca], "vsubps zmm1{k5}{ru-sae}, zmm0, zmm2"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5c, 0x0a], "vsubps zmm1, zmm0, dword [rdx]{1to16}"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5c, 0x0a], "vsubps zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5c, 0xca], "vsubps zmm1{k5}{z}{rne-sae}, zmm0, zmm2"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5c, 0x0a], "vsubps xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VSUBPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5c, 0xca], "vsubps zmm1{rne-sae}, zmm0, zmm2"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5c, 0xca], "vsubps zmm1{k5}{rne-sae}, zmm0, zmm2"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5c, 0x0a], "vsubps xmm1, xmm0, dword [rdx]{1to4}"); // VSUBPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5c, 0x0a], "vsubps xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VSUBPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5c, 0xca], "vsubps zmm1{k5}{z}, zmm0, zmm2"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5c, 0x0a], "vsubps zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5c, 0xca], "vsubps zmm1, zmm0, zmm2"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5c, 0xca], "vsubps zmm1{k5}, zmm0, zmm2"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5c, 0x0a], "vsubps zmm1, zmm0, zmmword [rdx]"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5c, 0x0a], "vsubps zmm1{k5}, zmm0, zmmword [rdx]"); // VSUBPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5c, 0xca], "vsubps xmm1{k5}{z}, xmm0, xmm2"); // VSUBPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5c, 0x0a], "vsubps xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VSUBPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5c, 0xca], "vsubps xmm1, xmm0, xmm2"); // VSUBPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5c, 0xca], "vsubps xmm1{k5}, xmm0, xmm2"); // VSUBPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5c, 0x0a], "vsubps xmm1, xmm0, xmmword [rdx]"); // VSUBPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5c, 0x0a], "vsubps xmm1{k5}, xmm0, xmmword [rdx]"); // VSUBPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x5d, 0xca], "vminps zmm1{k5}{z}{sae}, zmm0, zmm2"); // VMINPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x5d, 0xca], "vminps zmm1{sae}, zmm0, zmm2"); // VMINPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x5d, 0xca], "vminps zmm1{k5}{sae}, zmm0, zmm2"); // VMINPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5d, 0x0a], "vminps ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VMINPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5d, 0x0a], "vminps ymm1, ymm0, dword [rdx]{1to8}"); // VMINPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5d, 0x0a], "vminps ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VMINPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5d, 0xca], "vminps ymm1{k5}{z}, ymm0, ymm2"); // VMINPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5d, 0x0a], "vminps ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VMINPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5d, 0xca], "vminps ymm1, ymm0, ymm2"); // VMINPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5d, 0xca], "vminps ymm1{k5}, ymm0, ymm2"); // VMINPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5d, 0x0a], "vminps ymm1, ymm0, ymmword [rdx]"); // VMINPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5d, 0x0a], "vminps ymm1{k5}, ymm0, ymmword [rdx]"); // VMINPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5d, 0x0a], "vminps zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VMINPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5d, 0x0a], "vminps zmm1, zmm0, dword [rdx]{1to16}"); // VMINPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5d, 0x0a], "vminps zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VMINPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5d, 0x0a], "vminps xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VMINPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5d, 0x0a], "vminps xmm1, xmm0, dword [rdx]{1to4}"); // VMINPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5d, 0x0a], "vminps xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VMINPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5d, 0xca], "vminps zmm1{k5}{z}, zmm0, zmm2"); // VMINPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5d, 0x0a], "vminps zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VMINPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5d, 0xca], "vminps zmm1, zmm0, zmm2"); // VMINPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5d, 0xca], "vminps zmm1{k5}, zmm0, zmm2"); // VMINPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5d, 0x0a], "vminps zmm1, zmm0, zmmword [rdx]"); // VMINPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5d, 0x0a], "vminps zmm1{k5}, zmm0, zmmword [rdx]"); // VMINPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5d, 0xca], "vminps xmm1{k5}{z}, xmm0, xmm2"); // VMINPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5d, 0x0a], "vminps xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VMINPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5d, 0xca], "vminps xmm1, xmm0, xmm2"); // VMINPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5d, 0xca], "vminps xmm1{k5}, xmm0, xmm2"); // VMINPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5d, 0x0a], "vminps xmm1, xmm0, xmmword [rdx]"); // VMINPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5d, 0x0a], "vminps xmm1{k5}, xmm0, xmmword [rdx]"); // VMINPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x5e, 0xca], "vdivps zmm1{k5}{z}{rz-sae}, zmm0, zmm2"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x5e, 0xca], "vdivps zmm1{rz-sae}, zmm0, zmm2"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x5e, 0xca], "vdivps zmm1{k5}{rz-sae}, zmm0, zmm2"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5e, 0xca], "vdivps zmm1{k5}{z}{rd-sae}, zmm0, zmm2"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5e, 0x0a], "vdivps ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VDIVPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5e, 0xca], "vdivps zmm1{rd-sae}, zmm0, zmm2"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5e, 0xca], "vdivps zmm1{k5}{rd-sae}, zmm0, zmm2"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5e, 0x0a], "vdivps ymm1, ymm0, dword [rdx]{1to8}"); // VDIVPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5e, 0x0a], "vdivps ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VDIVPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5e, 0xca], "vdivps ymm1{k5}{z}, ymm0, ymm2"); // VDIVPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5e, 0x0a], "vdivps ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VDIVPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5e, 0xca], "vdivps ymm1, ymm0, ymm2"); // VDIVPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5e, 0xca], "vdivps ymm1{k5}, ymm0, ymm2"); // VDIVPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5e, 0x0a], "vdivps ymm1, ymm0, ymmword [rdx]"); // VDIVPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5e, 0x0a], "vdivps ymm1{k5}, ymm0, ymmword [rdx]"); // VDIVPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5e, 0xca], "vdivps zmm1{k5}{z}{ru-sae}, zmm0, zmm2"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5e, 0x0a], "vdivps zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5e, 0xca], "vdivps zmm1{ru-sae}, zmm0, zmm2"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5e, 0xca], "vdivps zmm1{k5}{ru-sae}, zmm0, zmm2"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5e, 0x0a], "vdivps zmm1, zmm0, dword [rdx]{1to16}"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5e, 0x0a], "vdivps zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5e, 0xca], "vdivps zmm1{k5}{z}{rne-sae}, zmm0, zmm2"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5e, 0x0a], "vdivps xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VDIVPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5e, 0xca], "vdivps zmm1{rne-sae}, zmm0, zmm2"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5e, 0xca], "vdivps zmm1{k5}{rne-sae}, zmm0, zmm2"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5e, 0x0a], "vdivps xmm1, xmm0, dword [rdx]{1to4}"); // VDIVPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5e, 0x0a], "vdivps xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VDIVPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5e, 0xca], "vdivps zmm1{k5}{z}, zmm0, zmm2"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5e, 0x0a], "vdivps zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5e, 0xca], "vdivps zmm1, zmm0, zmm2"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5e, 0xca], "vdivps zmm1{k5}, zmm0, zmm2"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5e, 0x0a], "vdivps zmm1, zmm0, zmmword [rdx]"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5e, 0x0a], "vdivps zmm1{k5}, zmm0, zmmword [rdx]"); // VDIVPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5e, 0xca], "vdivps xmm1{k5}{z}, xmm0, xmm2"); // VDIVPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5e, 0x0a], "vdivps xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VDIVPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5e, 0xca], "vdivps xmm1, xmm0, xmm2"); // VDIVPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5e, 0xca], "vdivps xmm1{k5}, xmm0, xmm2"); // VDIVPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5e, 0x0a], "vdivps xmm1, xmm0, xmmword [rdx]"); // VDIVPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5e, 0x0a], "vdivps xmm1{k5}, xmm0, xmmword [rdx]"); // VDIVPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x5f, 0xca], "vmaxps zmm1{k5}{z}{sae}, zmm0, zmm2"); // VMAXPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x5f, 0xca], "vmaxps zmm1{sae}, zmm0, zmm2"); // VMAXPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x5f, 0xca], "vmaxps zmm1{k5}{sae}, zmm0, zmm2"); // VMAXPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x5f, 0x0a], "vmaxps ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VMAXPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x5f, 0x0a], "vmaxps ymm1, ymm0, dword [rdx]{1to8}"); // VMAXPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x5f, 0x0a], "vmaxps ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VMAXPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5f, 0xca], "vmaxps ymm1{k5}{z}, ymm0, ymm2"); // VMAXPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x5f, 0x0a], "vmaxps ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VMAXPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5f, 0xca], "vmaxps ymm1, ymm0, ymm2"); // VMAXPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5f, 0xca], "vmaxps ymm1{k5}, ymm0, ymm2"); // VMAXPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x5f, 0x0a], "vmaxps ymm1, ymm0, ymmword [rdx]"); // VMAXPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x5f, 0x0a], "vmaxps ymm1{k5}, ymm0, ymmword [rdx]"); // VMAXPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x5f, 0x0a], "vmaxps zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VMAXPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x5f, 0x0a], "vmaxps zmm1, zmm0, dword [rdx]{1to16}"); // VMAXPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x5f, 0x0a], "vmaxps zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VMAXPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x5f, 0x0a], "vmaxps xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VMAXPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x5f, 0x0a], "vmaxps xmm1, xmm0, dword [rdx]{1to4}"); // VMAXPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x5f, 0x0a], "vmaxps xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VMAXPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5f, 0xca], "vmaxps zmm1{k5}{z}, zmm0, zmm2"); // VMAXPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x5f, 0x0a], "vmaxps zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VMAXPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5f, 0xca], "vmaxps zmm1, zmm0, zmm2"); // VMAXPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5f, 0xca], "vmaxps zmm1{k5}, zmm0, zmm2"); // VMAXPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x5f, 0x0a], "vmaxps zmm1, zmm0, zmmword [rdx]"); // VMAXPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x5f, 0x0a], "vmaxps zmm1{k5}, zmm0, zmmword [rdx]"); // VMAXPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5f, 0xca], "vmaxps xmm1{k5}{z}, xmm0, xmm2"); // VMAXPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x5f, 0x0a], "vmaxps xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VMAXPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5f, 0xca], "vmaxps xmm1, xmm0, xmm2"); // VMAXPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5f, 0xca], "vmaxps xmm1{k5}, xmm0, xmm2"); // VMAXPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x5f, 0x0a], "vmaxps xmm1, xmm0, xmmword [rdx]"); // VMAXPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x5f, 0x0a], "vmaxps xmm1{k5}, xmm0, xmmword [rdx]"); // VMAXPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xfd, 0x78, 0xca], "vcvttpd2udq ymm1{k5}{z}{sae}, zmm2"); // VCVTTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x78, 0x78, 0xca], "vcvttpd2udq ymm1{sae}, zmm2"); // VCVTTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x7d, 0x78, 0xca], "vcvttpd2udq ymm1{k5}{sae}, zmm2"); // VCVTTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xbd, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}{z}, qword [rdx]{1to4}"); // VCVTTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x38, 0x78, 0x0a], "vcvttpd2udq xmm1, qword [rdx]{1to4}"); // VCVTTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x3d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}, qword [rdx]{1to4}"); // VCVTTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xad, 0x78, 0xca], "vcvttpd2udq xmm1{k5}{z}, ymm2"); // VCVTTPD2UDQ_XMMu32_MASKmskw_YMMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xad, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}{z}, ymmword [rdx]"); // VCVTTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x28, 0x78, 0xca], "vcvttpd2udq xmm1, ymm2"); // VCVTTPD2UDQ_XMMu32_MASKmskw_YMMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x2d, 0x78, 0xca], "vcvttpd2udq xmm1{k5}, ymm2"); // VCVTTPD2UDQ_XMMu32_MASKmskw_YMMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x28, 0x78, 0x0a], "vcvttpd2udq xmm1, ymmword [rdx]"); // VCVTTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x2d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}, ymmword [rdx]"); // VCVTTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x78, 0xca], "vcvttps2udq zmm1{k5}{z}{sae}, zmm2"); // VCVTTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x78, 0xca], "vcvttps2udq zmm1{sae}, zmm2"); // VCVTTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x78, 0xca], "vcvttps2udq zmm1{k5}{sae}, zmm2"); // VCVTTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x78, 0x0a], "vcvttps2udq ymm1{k5}{z}, dword [rdx]{1to8}"); // VCVTTPS2UDQ_YMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x78, 0x0a], "vcvttps2udq ymm1, dword [rdx]{1to8}"); // VCVTTPS2UDQ_YMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x78, 0x0a], "vcvttps2udq ymm1{k5}, dword [rdx]{1to8}"); // VCVTTPS2UDQ_YMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x78, 0xca], "vcvttps2udq ymm1{k5}{z}, ymm2"); // VCVTTPS2UDQ_YMMu32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x78, 0x0a], "vcvttps2udq ymm1{k5}{z}, ymmword [rdx]"); // VCVTTPS2UDQ_YMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x78, 0xca], "vcvttps2udq ymm1, ymm2"); // VCVTTPS2UDQ_YMMu32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x78, 0xca], "vcvttps2udq ymm1{k5}, ymm2"); // VCVTTPS2UDQ_YMMu32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x78, 0x0a], "vcvttps2udq ymm1, ymmword [rdx]"); // VCVTTPS2UDQ_YMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x78, 0x0a], "vcvttps2udq ymm1{k5}, ymmword [rdx]"); // VCVTTPS2UDQ_YMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xdd, 0x78, 0x0a], "vcvttpd2udq ymm1{k5}{z}, qword [rdx]{1to8}"); // VCVTTPD2UDQ_YMMu32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x58, 0x78, 0x0a], "vcvttpd2udq ymm1, qword [rdx]{1to8}"); // VCVTTPD2UDQ_YMMu32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x5d, 0x78, 0x0a], "vcvttpd2udq ymm1{k5}, qword [rdx]{1to8}"); // VCVTTPD2UDQ_YMMu32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x9d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}{z}, qword [rdx]{1to2}"); // VCVTTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x18, 0x78, 0x0a], "vcvttpd2udq xmm1, qword [rdx]{1to2}"); // VCVTTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x1d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}, qword [rdx]{1to2}"); // VCVTTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xcd, 0x78, 0xca], "vcvttpd2udq ymm1{k5}{z}, zmm2"); // VCVTTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xcd, 0x78, 0x0a], "vcvttpd2udq ymm1{k5}{z}, zmmword [rdx]"); // VCVTTPD2UDQ_YMMu32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x48, 0x78, 0xca], "vcvttpd2udq ymm1, zmm2"); // VCVTTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x4d, 0x78, 0xca], "vcvttpd2udq ymm1{k5}, zmm2"); // VCVTTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x48, 0x78, 0x0a], "vcvttpd2udq ymm1, zmmword [rdx]"); // VCVTTPD2UDQ_YMMu32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x4d, 0x78, 0x0a], "vcvttpd2udq ymm1{k5}, zmmword [rdx]"); // VCVTTPD2UDQ_YMMu32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x8d, 0x78, 0xca], "vcvttpd2udq xmm1{k5}{z}, xmm2"); // VCVTTPD2UDQ_XMMu32_MASKmskw_XMMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x8d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}{z}, xmmword [rdx]"); // VCVTTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x08, 0x78, 0xca], "vcvttpd2udq xmm1, xmm2"); // VCVTTPD2UDQ_XMMu32_MASKmskw_XMMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x0d, 0x78, 0xca], "vcvttpd2udq xmm1{k5}, xmm2"); // VCVTTPD2UDQ_XMMu32_MASKmskw_XMMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x08, 0x78, 0x0a], "vcvttpd2udq xmm1, xmmword [rdx]"); // VCVTTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x0d, 0x78, 0x0a], "vcvttpd2udq xmm1{k5}, xmmword [rdx]"); // VCVTTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x78, 0x0a], "vcvttps2udq zmm1{k5}{z}, dword [rdx]{1to16}"); // VCVTTPS2UDQ_ZMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x78, 0x0a], "vcvttps2udq zmm1, dword [rdx]{1to16}"); // VCVTTPS2UDQ_ZMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x78, 0x0a], "vcvttps2udq zmm1{k5}, dword [rdx]{1to16}"); // VCVTTPS2UDQ_ZMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x78, 0x0a], "vcvttps2udq xmm1{k5}{z}, dword [rdx]{1to4}"); // VCVTTPS2UDQ_XMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x78, 0x0a], "vcvttps2udq xmm1, dword [rdx]{1to4}"); // VCVTTPS2UDQ_XMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x78, 0x0a], "vcvttps2udq xmm1{k5}, dword [rdx]{1to4}"); // VCVTTPS2UDQ_XMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x78, 0xca], "vcvttps2udq zmm1{k5}{z}, zmm2"); // VCVTTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x78, 0x0a], "vcvttps2udq zmm1{k5}{z}, zmmword [rdx]"); // VCVTTPS2UDQ_ZMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x78, 0xca], "vcvttps2udq zmm1, zmm2"); // VCVTTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x78, 0xca], "vcvttps2udq zmm1{k5}, zmm2"); // VCVTTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x78, 0x0a], "vcvttps2udq zmm1, zmmword [rdx]"); // VCVTTPS2UDQ_ZMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x78, 0x0a], "vcvttps2udq zmm1{k5}, zmmword [rdx]"); // VCVTTPS2UDQ_ZMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x78, 0xca], "vcvttps2udq xmm1{k5}{z}, xmm2"); // VCVTTPS2UDQ_XMMu32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x78, 0x0a], "vcvttps2udq xmm1{k5}{z}, xmmword [rdx]"); // VCVTTPS2UDQ_XMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x78, 0xca], "vcvttps2udq xmm1, xmm2"); // VCVTTPS2UDQ_XMMu32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x78, 0xca], "vcvttps2udq xmm1{k5}, xmm2"); // VCVTTPS2UDQ_XMMu32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x78, 0x0a], "vcvttps2udq xmm1, xmmword [rdx]"); // VCVTTPS2UDQ_XMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x78, 0x0a], "vcvttps2udq xmm1{k5}, xmmword [rdx]"); // VCVTTPS2UDQ_XMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xfd, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{z}{rz-sae}, zmm2"); // VCVTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x78, 0x79, 0xca], "vcvtpd2udq ymm1{rz-sae}, zmm2"); // VCVTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x7d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{rz-sae}, zmm2"); // VCVTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xbd, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{z}{rd-sae}, zmm2"); // VCVTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xbd, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}{z}, qword [rdx]{1to4}"); // VCVTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x38, 0x79, 0xca], "vcvtpd2udq ymm1{rd-sae}, zmm2"); // VCVTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x3d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{rd-sae}, zmm2"); // VCVTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x38, 0x79, 0x0a], "vcvtpd2udq xmm1, qword [rdx]{1to4}"); // VCVTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x3d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}, qword [rdx]{1to4}"); // VCVTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xad, 0x79, 0xca], "vcvtpd2udq xmm1{k5}{z}, ymm2"); // VCVTPD2UDQ_XMMu32_MASKmskw_YMMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xad, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}{z}, ymmword [rdx]"); // VCVTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x28, 0x79, 0xca], "vcvtpd2udq xmm1, ymm2"); // VCVTPD2UDQ_XMMu32_MASKmskw_YMMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x2d, 0x79, 0xca], "vcvtpd2udq xmm1{k5}, ymm2"); // VCVTPD2UDQ_XMMu32_MASKmskw_YMMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x28, 0x79, 0x0a], "vcvtpd2udq xmm1, ymmword [rdx]"); // VCVTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x2d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}, ymmword [rdx]"); // VCVTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xfd, 0x79, 0xca], "vcvtps2udq zmm1{k5}{z}{rz-sae}, zmm2"); // VCVTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0x79, 0xca], "vcvtps2udq zmm1{rz-sae}, zmm2"); // VCVTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0x79, 0xca], "vcvtps2udq zmm1{k5}{rz-sae}, zmm2"); // VCVTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x79, 0xca], "vcvtps2udq zmm1{k5}{z}{rd-sae}, zmm2"); // VCVTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0x79, 0x0a], "vcvtps2udq ymm1{k5}{z}, dword [rdx]{1to8}"); // VCVTPS2UDQ_YMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x79, 0xca], "vcvtps2udq zmm1{rd-sae}, zmm2"); // VCVTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x79, 0xca], "vcvtps2udq zmm1{k5}{rd-sae}, zmm2"); // VCVTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0x79, 0x0a], "vcvtps2udq ymm1, dword [rdx]{1to8}"); // VCVTPS2UDQ_YMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0x79, 0x0a], "vcvtps2udq ymm1{k5}, dword [rdx]{1to8}"); // VCVTPS2UDQ_YMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x79, 0xca], "vcvtps2udq ymm1{k5}{z}, ymm2"); // VCVTPS2UDQ_YMMu32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0x79, 0x0a], "vcvtps2udq ymm1{k5}{z}, ymmword [rdx]"); // VCVTPS2UDQ_YMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x79, 0xca], "vcvtps2udq ymm1, ymm2"); // VCVTPS2UDQ_YMMu32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x79, 0xca], "vcvtps2udq ymm1{k5}, ymm2"); // VCVTPS2UDQ_YMMu32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0x79, 0x0a], "vcvtps2udq ymm1, ymmword [rdx]"); // VCVTPS2UDQ_YMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0x79, 0x0a], "vcvtps2udq ymm1{k5}, ymmword [rdx]"); // VCVTPS2UDQ_YMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xdd, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{z}{ru-sae}, zmm2"); // VCVTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xdd, 0x79, 0x0a], "vcvtpd2udq ymm1{k5}{z}, qword [rdx]{1to8}"); // VCVTPD2UDQ_YMMu32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x58, 0x79, 0xca], "vcvtpd2udq ymm1{ru-sae}, zmm2"); // VCVTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x5d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{ru-sae}, zmm2"); // VCVTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x58, 0x79, 0x0a], "vcvtpd2udq ymm1, qword [rdx]{1to8}"); // VCVTPD2UDQ_YMMu32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x5d, 0x79, 0x0a], "vcvtpd2udq ymm1{k5}, qword [rdx]{1to8}"); // VCVTPD2UDQ_YMMu32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x9d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{z}{rne-sae}, zmm2"); // VCVTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x9d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}{z}, qword [rdx]{1to2}"); // VCVTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x18, 0x79, 0xca], "vcvtpd2udq ymm1{rne-sae}, zmm2"); // VCVTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x1d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{rne-sae}, zmm2"); // VCVTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x18, 0x79, 0x0a], "vcvtpd2udq xmm1, qword [rdx]{1to2}"); // VCVTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x1d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}, qword [rdx]{1to2}"); // VCVTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xcd, 0x79, 0xca], "vcvtpd2udq ymm1{k5}{z}, zmm2"); // VCVTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0xcd, 0x79, 0x0a], "vcvtpd2udq ymm1{k5}{z}, zmmword [rdx]"); // VCVTPD2UDQ_YMMu32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x48, 0x79, 0xca], "vcvtpd2udq ymm1, zmm2"); // VCVTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x4d, 0x79, 0xca], "vcvtpd2udq ymm1{k5}, zmm2"); // VCVTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x48, 0x79, 0x0a], "vcvtpd2udq ymm1, zmmword [rdx]"); // VCVTPD2UDQ_YMMu32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x4d, 0x79, 0x0a], "vcvtpd2udq ymm1{k5}, zmmword [rdx]"); // VCVTPD2UDQ_YMMu32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x8d, 0x79, 0xca], "vcvtpd2udq xmm1{k5}{z}, xmm2"); // VCVTPD2UDQ_XMMu32_MASKmskw_XMMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x8d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}{z}, xmmword [rdx]"); // VCVTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x08, 0x79, 0xca], "vcvtpd2udq xmm1, xmm2"); // VCVTPD2UDQ_XMMu32_MASKmskw_XMMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x0d, 0x79, 0xca], "vcvtpd2udq xmm1{k5}, xmm2"); // VCVTPD2UDQ_XMMu32_MASKmskw_XMMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x08, 0x79, 0x0a], "vcvtpd2udq xmm1, xmmword [rdx]"); // VCVTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfc, 0x0d, 0x79, 0x0a], "vcvtpd2udq xmm1{k5}, xmmword [rdx]"); // VCVTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x79, 0xca], "vcvtps2udq zmm1{k5}{z}{ru-sae}, zmm2"); // VCVTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0x79, 0x0a], "vcvtps2udq zmm1{k5}{z}, dword [rdx]{1to16}"); // VCVTPS2UDQ_ZMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x79, 0xca], "vcvtps2udq zmm1{ru-sae}, zmm2"); // VCVTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x79, 0xca], "vcvtps2udq zmm1{k5}{ru-sae}, zmm2"); // VCVTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0x79, 0x0a], "vcvtps2udq zmm1, dword [rdx]{1to16}"); // VCVTPS2UDQ_ZMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0x79, 0x0a], "vcvtps2udq zmm1{k5}, dword [rdx]{1to16}"); // VCVTPS2UDQ_ZMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x79, 0xca], "vcvtps2udq zmm1{k5}{z}{rne-sae}, zmm2"); // VCVTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0x79, 0x0a], "vcvtps2udq xmm1{k5}{z}, dword [rdx]{1to4}"); // VCVTPS2UDQ_XMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x79, 0xca], "vcvtps2udq zmm1{rne-sae}, zmm2"); // VCVTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x79, 0xca], "vcvtps2udq zmm1{k5}{rne-sae}, zmm2"); // VCVTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0x79, 0x0a], "vcvtps2udq xmm1, dword [rdx]{1to4}"); // VCVTPS2UDQ_XMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0x79, 0x0a], "vcvtps2udq xmm1{k5}, dword [rdx]{1to4}"); // VCVTPS2UDQ_XMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x79, 0xca], "vcvtps2udq zmm1{k5}{z}, zmm2"); // VCVTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0x79, 0x0a], "vcvtps2udq zmm1{k5}{z}, zmmword [rdx]"); // VCVTPS2UDQ_ZMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x79, 0xca], "vcvtps2udq zmm1, zmm2"); // VCVTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x79, 0xca], "vcvtps2udq zmm1{k5}, zmm2"); // VCVTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0x79, 0x0a], "vcvtps2udq zmm1, zmmword [rdx]"); // VCVTPS2UDQ_ZMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0x79, 0x0a], "vcvtps2udq zmm1{k5}, zmmword [rdx]"); // VCVTPS2UDQ_ZMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x79, 0xca], "vcvtps2udq xmm1{k5}{z}, xmm2"); // VCVTPS2UDQ_XMMu32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0x79, 0x0a], "vcvtps2udq xmm1{k5}{z}, xmmword [rdx]"); // VCVTPS2UDQ_XMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x79, 0xca], "vcvtps2udq xmm1, xmm2"); // VCVTPS2UDQ_XMMu32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x79, 0xca], "vcvtps2udq xmm1{k5}, xmm2"); // VCVTPS2UDQ_XMMu32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0x79, 0x0a], "vcvtps2udq xmm1, xmmword [rdx]"); // VCVTPS2UDQ_XMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0x79, 0x0a], "vcvtps2udq xmm1{k5}, xmmword [rdx]"); // VCVTPS2UDQ_XMMu32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x78, 0xc2, 0xca, 0xcc], "vcmpps k1{sae}, zmm0, zmm2, 0xcc"); // VCMPPS_MASKmskw_MASKmskw_ZMMf32_ZMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x7d, 0xc2, 0xca, 0xcc], "vcmpps k1{k5}{sae}, zmm0, zmm2, 0xcc"); // VCMPPS_MASKmskw_MASKmskw_ZMMf32_ZMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0xc2, 0x0a, 0xcc], "vcmpps k1, ymm0, dword [rdx]{1to8}, 0xcc"); // VCMPPS_MASKmskw_MASKmskw_YMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, ymm0, dword [rdx]{1to8}, 0xcc"); // VCMPPS_MASKmskw_MASKmskw_YMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0xc2, 0xca, 0xcc], "vcmpps k1, ymm0, ymm2, 0xcc"); // VCMPPS_MASKmskw_MASKmskw_YMMf32_YMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0xc2, 0xca, 0xcc], "vcmpps k1{k5}, ymm0, ymm2, 0xcc"); // VCMPPS_MASKmskw_MASKmskw_YMMf32_YMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0xc2, 0x0a, 0xcc], "vcmpps k1, ymm0, ymmword [rdx], 0xcc"); // VCMPPS_MASKmskw_MASKmskw_YMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, ymm0, ymmword [rdx], 0xcc"); // VCMPPS_MASKmskw_MASKmskw_YMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0xc2, 0x0a, 0xcc], "vcmpps k1, zmm0, dword [rdx]{1to16}, 0xcc"); // VCMPPS_MASKmskw_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, zmm0, dword [rdx]{1to16}, 0xcc"); // VCMPPS_MASKmskw_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0xc2, 0x0a, 0xcc], "vcmpps k1, xmm0, dword [rdx]{1to4}, 0xcc"); // VCMPPS_MASKmskw_MASKmskw_XMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, xmm0, dword [rdx]{1to4}, 0xcc"); // VCMPPS_MASKmskw_MASKmskw_XMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0xc2, 0xca, 0xcc], "vcmpps k1, zmm0, zmm2, 0xcc"); // VCMPPS_MASKmskw_MASKmskw_ZMMf32_ZMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0xc2, 0xca, 0xcc], "vcmpps k1{k5}, zmm0, zmm2, 0xcc"); // VCMPPS_MASKmskw_MASKmskw_ZMMf32_ZMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0xc2, 0x0a, 0xcc], "vcmpps k1, zmm0, zmmword [rdx], 0xcc"); // VCMPPS_MASKmskw_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, zmm0, zmmword [rdx], 0xcc"); // VCMPPS_MASKmskw_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0xc2, 0xca, 0xcc], "vcmpps k1, xmm0, xmm2, 0xcc"); // VCMPPS_MASKmskw_MASKmskw_XMMf32_XMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0xc2, 0xca, 0xcc], "vcmpps k1{k5}, xmm0, xmm2, 0xcc"); // VCMPPS_MASKmskw_MASKmskw_XMMf32_XMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0xc2, 0x0a, 0xcc], "vcmpps k1, xmm0, xmmword [rdx], 0xcc"); // VCMPPS_MASKmskw_MASKmskw_XMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0xc2, 0x0a, 0xcc], "vcmpps k1{k5}, xmm0, xmmword [rdx], 0xcc"); // VCMPPS_MASKmskw_MASKmskw_XMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xbd, 0xc6, 0x0a, 0xcc], "vshufps ymm1{k5}{z}, ymm0, dword [rdx]{1to8}, 0xcc"); // VSHUFPS_YMMf32_MASKmskw_YMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x38, 0xc6, 0x0a, 0xcc], "vshufps ymm1, ymm0, dword [rdx]{1to8}, 0xcc"); // VSHUFPS_YMMf32_MASKmskw_YMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x3d, 0xc6, 0x0a, 0xcc], "vshufps ymm1{k5}, ymm0, dword [rdx]{1to8}, 0xcc"); // VSHUFPS_YMMf32_MASKmskw_YMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0xc6, 0xca, 0xcc], "vshufps ymm1{k5}{z}, ymm0, ymm2, 0xcc"); // VSHUFPS_YMMf32_MASKmskw_YMMf32_YMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xad, 0xc6, 0x0a, 0xcc], "vshufps ymm1{k5}{z}, ymm0, ymmword [rdx], 0xcc"); // VSHUFPS_YMMf32_MASKmskw_YMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0xc6, 0xca, 0xcc], "vshufps ymm1, ymm0, ymm2, 0xcc"); // VSHUFPS_YMMf32_MASKmskw_YMMf32_YMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0xc6, 0xca, 0xcc], "vshufps ymm1{k5}, ymm0, ymm2, 0xcc"); // VSHUFPS_YMMf32_MASKmskw_YMMf32_YMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x28, 0xc6, 0x0a, 0xcc], "vshufps ymm1, ymm0, ymmword [rdx], 0xcc"); // VSHUFPS_YMMf32_MASKmskw_YMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x2d, 0xc6, 0x0a, 0xcc], "vshufps ymm1{k5}, ymm0, ymmword [rdx], 0xcc"); // VSHUFPS_YMMf32_MASKmskw_YMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xdd, 0xc6, 0x0a, 0xcc], "vshufps zmm1{k5}{z}, zmm0, dword [rdx]{1to16}, 0xcc"); // VSHUFPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x58, 0xc6, 0x0a, 0xcc], "vshufps zmm1, zmm0, dword [rdx]{1to16}, 0xcc"); // VSHUFPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x5d, 0xc6, 0x0a, 0xcc], "vshufps zmm1{k5}, zmm0, dword [rdx]{1to16}, 0xcc"); // VSHUFPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x9d, 0xc6, 0x0a, 0xcc], "vshufps xmm1{k5}{z}, xmm0, dword [rdx]{1to4}, 0xcc"); // VSHUFPS_XMMf32_MASKmskw_XMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x18, 0xc6, 0x0a, 0xcc], "vshufps xmm1, xmm0, dword [rdx]{1to4}, 0xcc"); // VSHUFPS_XMMf32_MASKmskw_XMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x1d, 0xc6, 0x0a, 0xcc], "vshufps xmm1{k5}, xmm0, dword [rdx]{1to4}, 0xcc"); // VSHUFPS_XMMf32_MASKmskw_XMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0xc6, 0xca, 0xcc], "vshufps zmm1{k5}{z}, zmm0, zmm2, 0xcc"); // VSHUFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0xcd, 0xc6, 0x0a, 0xcc], "vshufps zmm1{k5}{z}, zmm0, zmmword [rdx], 0xcc"); // VSHUFPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0xc6, 0xca, 0xcc], "vshufps zmm1, zmm0, zmm2, 0xcc"); // VSHUFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0xc6, 0xca, 0xcc], "vshufps zmm1{k5}, zmm0, zmm2, 0xcc"); // VSHUFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x48, 0xc6, 0x0a, 0xcc], "vshufps zmm1, zmm0, zmmword [rdx], 0xcc"); // VSHUFPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x4d, 0xc6, 0x0a, 0xcc], "vshufps zmm1{k5}, zmm0, zmmword [rdx], 0xcc"); // VSHUFPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0xc6, 0xca, 0xcc], "vshufps xmm1{k5}{z}, xmm0, xmm2, 0xcc"); // VSHUFPS_XMMf32_MASKmskw_XMMf32_XMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x8d, 0xc6, 0x0a, 0xcc], "vshufps xmm1{k5}{z}, xmm0, xmmword [rdx], 0xcc"); // VSHUFPS_XMMf32_MASKmskw_XMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0xc6, 0xca, 0xcc], "vshufps xmm1, xmm0, xmm2, 0xcc"); // VSHUFPS_XMMf32_MASKmskw_XMMf32_XMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0xc6, 0xca, 0xcc], "vshufps xmm1{k5}, xmm0, xmm2, 0xcc"); // VSHUFPS_XMMf32_MASKmskw_XMMf32_XMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x08, 0xc6, 0x0a, 0xcc], "vshufps xmm1, xmm0, xmmword [rdx], 0xcc"); // VSHUFPS_XMMf32_MASKmskw_XMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7c, 0x0d, 0xc6, 0x0a, 0xcc], "vshufps xmm1{k5}, xmm0, xmmword [rdx], 0xcc"); // VSHUFPS_XMMf32_MASKmskw_XMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
}

#[test]
fn tests_66_0f() {
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x10, 0xca], "vmovupd ymm1{k5}{z}, ymm2"); // VMOVUPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x10, 0x0a], "vmovupd ymm1{k5}{z}, ymmword [rdx]"); // VMOVUPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x10, 0xca], "vmovupd ymm1, ymm2"); // VMOVUPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x10, 0xca], "vmovupd ymm1{k5}, ymm2"); // VMOVUPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x10, 0x0a], "vmovupd ymm1, ymmword [rdx]"); // VMOVUPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x10, 0x0a], "vmovupd ymm1{k5}, ymmword [rdx]"); // VMOVUPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x10, 0xca], "vmovupd zmm1{k5}{z}, zmm2"); // VMOVUPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x10, 0x0a], "vmovupd zmm1{k5}{z}, zmmword [rdx]"); // VMOVUPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x10, 0xca], "vmovupd zmm1, zmm2"); // VMOVUPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x10, 0xca], "vmovupd zmm1{k5}, zmm2"); // VMOVUPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x10, 0x0a], "vmovupd zmm1, zmmword [rdx]"); // VMOVUPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x10, 0x0a], "vmovupd zmm1{k5}, zmmword [rdx]"); // VMOVUPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x10, 0xca], "vmovupd xmm1{k5}{z}, xmm2"); // VMOVUPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x10, 0x0a], "vmovupd xmm1{k5}{z}, xmmword [rdx]"); // VMOVUPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x10, 0xca], "vmovupd xmm1, xmm2"); // VMOVUPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x10, 0xca], "vmovupd xmm1{k5}, xmm2"); // VMOVUPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x10, 0x0a], "vmovupd xmm1, xmmword [rdx]"); // VMOVUPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x10, 0x0a], "vmovupd xmm1{k5}, xmmword [rdx]"); // VMOVUPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x11, 0xca], "vmovupd ymm2{k5}{z}, ymm1"); // VMOVUPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x11, 0xca], "vmovupd ymm2, ymm1"); // VMOVUPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x11, 0xca], "vmovupd ymm2{k5}, ymm1"); // VMOVUPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x11, 0x0a], "vmovupd ymmword [rdx], ymm1"); // VMOVUPD_MEMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x11, 0x0a], "vmovupd ymmword [rdx]{k5}, ymm1"); // VMOVUPD_MEMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x11, 0xca], "vmovupd zmm2{k5}{z}, zmm1"); // VMOVUPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x11, 0xca], "vmovupd zmm2, zmm1"); // VMOVUPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x11, 0xca], "vmovupd zmm2{k5}, zmm1"); // VMOVUPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x11, 0x0a], "vmovupd zmmword [rdx], zmm1"); // VMOVUPD_MEMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x11, 0x0a], "vmovupd zmmword [rdx]{k5}, zmm1"); // VMOVUPD_MEMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x11, 0xca], "vmovupd xmm2{k5}{z}, xmm1"); // VMOVUPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x11, 0xca], "vmovupd xmm2, xmm1"); // VMOVUPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x11, 0xca], "vmovupd xmm2{k5}, xmm1"); // VMOVUPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x11, 0x0a], "vmovupd xmmword [rdx], xmm1"); // VMOVUPD_MEMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x11, 0x0a], "vmovupd xmmword [rdx]{k5}, xmm1"); // VMOVUPD_MEMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x12, 0x0a], "vmovlpd xmm1, xmm0, qword [rdx]"); // VMOVLPD_XMMf64_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x13, 0x0a], "vmovlpd qword [rdx], xmm1"); // VMOVLPD_MEMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x14, 0x0a], "vunpcklpd ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VUNPCKLPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x14, 0x0a], "vunpcklpd ymm1, ymm0, qword [rdx]{1to4}"); // VUNPCKLPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x14, 0x0a], "vunpcklpd ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VUNPCKLPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x14, 0xca], "vunpcklpd ymm1{k5}{z}, ymm0, ymm2"); // VUNPCKLPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x14, 0x0a], "vunpcklpd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VUNPCKLPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x14, 0xca], "vunpcklpd ymm1, ymm0, ymm2"); // VUNPCKLPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x14, 0xca], "vunpcklpd ymm1{k5}, ymm0, ymm2"); // VUNPCKLPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x14, 0x0a], "vunpcklpd ymm1, ymm0, ymmword [rdx]"); // VUNPCKLPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x14, 0x0a], "vunpcklpd ymm1{k5}, ymm0, ymmword [rdx]"); // VUNPCKLPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x14, 0x0a], "vunpcklpd zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VUNPCKLPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x14, 0x0a], "vunpcklpd zmm1, zmm0, qword [rdx]{1to8}"); // VUNPCKLPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x14, 0x0a], "vunpcklpd zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VUNPCKLPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x14, 0x0a], "vunpcklpd xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VUNPCKLPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x14, 0x0a], "vunpcklpd xmm1, xmm0, qword [rdx]{1to2}"); // VUNPCKLPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x14, 0x0a], "vunpcklpd xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VUNPCKLPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x14, 0xca], "vunpcklpd zmm1{k5}{z}, zmm0, zmm2"); // VUNPCKLPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x14, 0x0a], "vunpcklpd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VUNPCKLPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x14, 0xca], "vunpcklpd zmm1, zmm0, zmm2"); // VUNPCKLPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x14, 0xca], "vunpcklpd zmm1{k5}, zmm0, zmm2"); // VUNPCKLPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x14, 0x0a], "vunpcklpd zmm1, zmm0, zmmword [rdx]"); // VUNPCKLPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x14, 0x0a], "vunpcklpd zmm1{k5}, zmm0, zmmword [rdx]"); // VUNPCKLPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x14, 0xca], "vunpcklpd xmm1{k5}{z}, xmm0, xmm2"); // VUNPCKLPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x14, 0x0a], "vunpcklpd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VUNPCKLPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x14, 0xca], "vunpcklpd xmm1, xmm0, xmm2"); // VUNPCKLPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x14, 0xca], "vunpcklpd xmm1{k5}, xmm0, xmm2"); // VUNPCKLPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x14, 0x0a], "vunpcklpd xmm1, xmm0, xmmword [rdx]"); // VUNPCKLPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x14, 0x0a], "vunpcklpd xmm1{k5}, xmm0, xmmword [rdx]"); // VUNPCKLPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x15, 0x0a], "vunpckhpd ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VUNPCKHPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x15, 0x0a], "vunpckhpd ymm1, ymm0, qword [rdx]{1to4}"); // VUNPCKHPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x15, 0x0a], "vunpckhpd ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VUNPCKHPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x15, 0xca], "vunpckhpd ymm1{k5}{z}, ymm0, ymm2"); // VUNPCKHPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x15, 0x0a], "vunpckhpd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VUNPCKHPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x15, 0xca], "vunpckhpd ymm1, ymm0, ymm2"); // VUNPCKHPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x15, 0xca], "vunpckhpd ymm1{k5}, ymm0, ymm2"); // VUNPCKHPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x15, 0x0a], "vunpckhpd ymm1, ymm0, ymmword [rdx]"); // VUNPCKHPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x15, 0x0a], "vunpckhpd ymm1{k5}, ymm0, ymmword [rdx]"); // VUNPCKHPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x15, 0x0a], "vunpckhpd zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VUNPCKHPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x15, 0x0a], "vunpckhpd zmm1, zmm0, qword [rdx]{1to8}"); // VUNPCKHPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x15, 0x0a], "vunpckhpd zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VUNPCKHPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x15, 0x0a], "vunpckhpd xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VUNPCKHPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x15, 0x0a], "vunpckhpd xmm1, xmm0, qword [rdx]{1to2}"); // VUNPCKHPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x15, 0x0a], "vunpckhpd xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VUNPCKHPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x15, 0xca], "vunpckhpd zmm1{k5}{z}, zmm0, zmm2"); // VUNPCKHPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x15, 0x0a], "vunpckhpd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VUNPCKHPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x15, 0xca], "vunpckhpd zmm1, zmm0, zmm2"); // VUNPCKHPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x15, 0xca], "vunpckhpd zmm1{k5}, zmm0, zmm2"); // VUNPCKHPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x15, 0x0a], "vunpckhpd zmm1, zmm0, zmmword [rdx]"); // VUNPCKHPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x15, 0x0a], "vunpckhpd zmm1{k5}, zmm0, zmmword [rdx]"); // VUNPCKHPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x15, 0xca], "vunpckhpd xmm1{k5}{z}, xmm0, xmm2"); // VUNPCKHPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x15, 0x0a], "vunpckhpd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VUNPCKHPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x15, 0xca], "vunpckhpd xmm1, xmm0, xmm2"); // VUNPCKHPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x15, 0xca], "vunpckhpd xmm1{k5}, xmm0, xmm2"); // VUNPCKHPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x15, 0x0a], "vunpckhpd xmm1, xmm0, xmmword [rdx]"); // VUNPCKHPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x15, 0x0a], "vunpckhpd xmm1{k5}, xmm0, xmmword [rdx]"); // VUNPCKHPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x16, 0x0a], "vmovhpd xmm1, xmm0, qword [rdx]"); // VMOVHPD_XMMf64_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x17, 0x0a], "vmovhpd qword [rdx], xmm1"); // VMOVHPD_MEMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x28, 0xca], "vmovapd ymm1{k5}{z}, ymm2"); // VMOVAPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x28, 0x0a], "vmovapd ymm1{k5}{z}, ymmword [rdx]"); // VMOVAPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x28, 0xca], "vmovapd ymm1, ymm2"); // VMOVAPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x28, 0xca], "vmovapd ymm1{k5}, ymm2"); // VMOVAPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x28, 0x0a], "vmovapd ymm1, ymmword [rdx]"); // VMOVAPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x28, 0x0a], "vmovapd ymm1{k5}, ymmword [rdx]"); // VMOVAPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x28, 0xca], "vmovapd zmm1{k5}{z}, zmm2"); // VMOVAPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x28, 0x0a], "vmovapd zmm1{k5}{z}, zmmword [rdx]"); // VMOVAPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x28, 0xca], "vmovapd zmm1, zmm2"); // VMOVAPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x28, 0xca], "vmovapd zmm1{k5}, zmm2"); // VMOVAPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x28, 0x0a], "vmovapd zmm1, zmmword [rdx]"); // VMOVAPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x28, 0x0a], "vmovapd zmm1{k5}, zmmword [rdx]"); // VMOVAPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x28, 0xca], "vmovapd xmm1{k5}{z}, xmm2"); // VMOVAPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x28, 0x0a], "vmovapd xmm1{k5}{z}, xmmword [rdx]"); // VMOVAPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x28, 0xca], "vmovapd xmm1, xmm2"); // VMOVAPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x28, 0xca], "vmovapd xmm1{k5}, xmm2"); // VMOVAPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x28, 0x0a], "vmovapd xmm1, xmmword [rdx]"); // VMOVAPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x28, 0x0a], "vmovapd xmm1{k5}, xmmword [rdx]"); // VMOVAPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x29, 0xca], "vmovapd ymm2{k5}{z}, ymm1"); // VMOVAPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x29, 0xca], "vmovapd ymm2, ymm1"); // VMOVAPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x29, 0xca], "vmovapd ymm2{k5}, ymm1"); // VMOVAPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x29, 0x0a], "vmovapd ymmword [rdx], ymm1"); // VMOVAPD_MEMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x29, 0x0a], "vmovapd ymmword [rdx]{k5}, ymm1"); // VMOVAPD_MEMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x29, 0xca], "vmovapd zmm2{k5}{z}, zmm1"); // VMOVAPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x29, 0xca], "vmovapd zmm2, zmm1"); // VMOVAPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x29, 0xca], "vmovapd zmm2{k5}, zmm1"); // VMOVAPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x29, 0x0a], "vmovapd zmmword [rdx], zmm1"); // VMOVAPD_MEMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x29, 0x0a], "vmovapd zmmword [rdx]{k5}, zmm1"); // VMOVAPD_MEMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x29, 0xca], "vmovapd xmm2{k5}{z}, xmm1"); // VMOVAPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x29, 0xca], "vmovapd xmm2, xmm1"); // VMOVAPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x29, 0xca], "vmovapd xmm2{k5}, xmm1"); // VMOVAPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x29, 0x0a], "vmovapd xmmword [rdx], xmm1"); // VMOVAPD_MEMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x29, 0x0a], "vmovapd xmmword [rdx]{k5}, xmm1"); // VMOVAPD_MEMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x2b, 0x0a], "vmovntpd ymmword [rdx], ymm1"); // VMOVNTPD_MEMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x2b, 0x0a], "vmovntpd zmmword [rdx], zmm1"); // VMOVNTPD_MEMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x2b, 0x0a], "vmovntpd xmmword [rdx], xmm1"); // VMOVNTPD_MEMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x2e, 0xca], "vucomisd xmm1{sae}, xmm2"); // VUCOMISD_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x2e, 0xca], "vucomisd xmm1, xmm2"); // VUCOMISD_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x2e, 0x0a], "vucomisd xmm1, qword [rdx]"); // VUCOMISD_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x2f, 0xca], "vcomisd xmm1{sae}, xmm2"); // VCOMISD_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x2f, 0xca], "vcomisd xmm1, xmm2"); // VCOMISD_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x2f, 0x0a], "vcomisd xmm1, qword [rdx]"); // VCOMISD_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x51, 0xca], "vsqrtpd zmm1{k5}{z}{rz-sae}, zmm2"); // VSQRTPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x51, 0xca], "vsqrtpd zmm1{rz-sae}, zmm2"); // VSQRTPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x51, 0xca], "vsqrtpd zmm1{k5}{rz-sae}, zmm2"); // VSQRTPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x51, 0xca], "vsqrtpd zmm1{k5}{z}{rd-sae}, zmm2"); // VSQRTPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x51, 0x0a], "vsqrtpd ymm1{k5}{z}, qword [rdx]{1to4}"); // VSQRTPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x51, 0xca], "vsqrtpd zmm1{rd-sae}, zmm2"); // VSQRTPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x51, 0xca], "vsqrtpd zmm1{k5}{rd-sae}, zmm2"); // VSQRTPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x51, 0x0a], "vsqrtpd ymm1, qword [rdx]{1to4}"); // VSQRTPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x51, 0x0a], "vsqrtpd ymm1{k5}, qword [rdx]{1to4}"); // VSQRTPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x51, 0xca], "vsqrtpd ymm1{k5}{z}, ymm2"); // VSQRTPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x51, 0x0a], "vsqrtpd ymm1{k5}{z}, ymmword [rdx]"); // VSQRTPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x51, 0xca], "vsqrtpd ymm1, ymm2"); // VSQRTPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x51, 0xca], "vsqrtpd ymm1{k5}, ymm2"); // VSQRTPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x51, 0x0a], "vsqrtpd ymm1, ymmword [rdx]"); // VSQRTPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x51, 0x0a], "vsqrtpd ymm1{k5}, ymmword [rdx]"); // VSQRTPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x51, 0xca], "vsqrtpd zmm1{k5}{z}{ru-sae}, zmm2"); // VSQRTPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x51, 0x0a], "vsqrtpd zmm1{k5}{z}, qword [rdx]{1to8}"); // VSQRTPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x51, 0xca], "vsqrtpd zmm1{ru-sae}, zmm2"); // VSQRTPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x51, 0xca], "vsqrtpd zmm1{k5}{ru-sae}, zmm2"); // VSQRTPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x51, 0x0a], "vsqrtpd zmm1, qword [rdx]{1to8}"); // VSQRTPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x51, 0x0a], "vsqrtpd zmm1{k5}, qword [rdx]{1to8}"); // VSQRTPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x51, 0xca], "vsqrtpd zmm1{k5}{z}{rne-sae}, zmm2"); // VSQRTPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x51, 0x0a], "vsqrtpd xmm1{k5}{z}, qword [rdx]{1to2}"); // VSQRTPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x51, 0xca], "vsqrtpd zmm1{rne-sae}, zmm2"); // VSQRTPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x51, 0xca], "vsqrtpd zmm1{k5}{rne-sae}, zmm2"); // VSQRTPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x51, 0x0a], "vsqrtpd xmm1, qword [rdx]{1to2}"); // VSQRTPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x51, 0x0a], "vsqrtpd xmm1{k5}, qword [rdx]{1to2}"); // VSQRTPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x51, 0xca], "vsqrtpd zmm1{k5}{z}, zmm2"); // VSQRTPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x51, 0x0a], "vsqrtpd zmm1{k5}{z}, zmmword [rdx]"); // VSQRTPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x51, 0xca], "vsqrtpd zmm1, zmm2"); // VSQRTPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x51, 0xca], "vsqrtpd zmm1{k5}, zmm2"); // VSQRTPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x51, 0x0a], "vsqrtpd zmm1, zmmword [rdx]"); // VSQRTPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x51, 0x0a], "vsqrtpd zmm1{k5}, zmmword [rdx]"); // VSQRTPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x51, 0xca], "vsqrtpd xmm1{k5}{z}, xmm2"); // VSQRTPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x51, 0x0a], "vsqrtpd xmm1{k5}{z}, xmmword [rdx]"); // VSQRTPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x51, 0xca], "vsqrtpd xmm1, xmm2"); // VSQRTPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x51, 0xca], "vsqrtpd xmm1{k5}, xmm2"); // VSQRTPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x51, 0x0a], "vsqrtpd xmm1, xmmword [rdx]"); // VSQRTPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x51, 0x0a], "vsqrtpd xmm1{k5}, xmmword [rdx]"); // VSQRTPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x54, 0x0a], "vandpd ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VANDPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x54, 0x0a], "vandpd ymm1, ymm0, qword [rdx]{1to4}"); // VANDPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x54, 0x0a], "vandpd ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VANDPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x54, 0xca], "vandpd ymm1{k5}{z}, ymm0, ymm2"); // VANDPD_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x54, 0x0a], "vandpd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VANDPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x54, 0xca], "vandpd ymm1, ymm0, ymm2"); // VANDPD_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x54, 0xca], "vandpd ymm1{k5}, ymm0, ymm2"); // VANDPD_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x54, 0x0a], "vandpd ymm1, ymm0, ymmword [rdx]"); // VANDPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x54, 0x0a], "vandpd ymm1{k5}, ymm0, ymmword [rdx]"); // VANDPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x54, 0x0a], "vandpd zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VANDPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x54, 0x0a], "vandpd zmm1, zmm0, qword [rdx]{1to8}"); // VANDPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x54, 0x0a], "vandpd zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VANDPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x54, 0x0a], "vandpd xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VANDPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x54, 0x0a], "vandpd xmm1, xmm0, qword [rdx]{1to2}"); // VANDPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x54, 0x0a], "vandpd xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VANDPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x54, 0xca], "vandpd zmm1{k5}{z}, zmm0, zmm2"); // VANDPD_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x54, 0x0a], "vandpd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VANDPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x54, 0xca], "vandpd zmm1, zmm0, zmm2"); // VANDPD_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x54, 0xca], "vandpd zmm1{k5}, zmm0, zmm2"); // VANDPD_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x54, 0x0a], "vandpd zmm1, zmm0, zmmword [rdx]"); // VANDPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x54, 0x0a], "vandpd zmm1{k5}, zmm0, zmmword [rdx]"); // VANDPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x54, 0xca], "vandpd xmm1{k5}{z}, xmm0, xmm2"); // VANDPD_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x54, 0x0a], "vandpd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VANDPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x54, 0xca], "vandpd xmm1, xmm0, xmm2"); // VANDPD_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x54, 0xca], "vandpd xmm1{k5}, xmm0, xmm2"); // VANDPD_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x54, 0x0a], "vandpd xmm1, xmm0, xmmword [rdx]"); // VANDPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x54, 0x0a], "vandpd xmm1{k5}, xmm0, xmmword [rdx]"); // VANDPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x55, 0x0a], "vandnpd ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VANDNPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x55, 0x0a], "vandnpd ymm1, ymm0, qword [rdx]{1to4}"); // VANDNPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x55, 0x0a], "vandnpd ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VANDNPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x55, 0xca], "vandnpd ymm1{k5}{z}, ymm0, ymm2"); // VANDNPD_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x55, 0x0a], "vandnpd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VANDNPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x55, 0xca], "vandnpd ymm1, ymm0, ymm2"); // VANDNPD_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x55, 0xca], "vandnpd ymm1{k5}, ymm0, ymm2"); // VANDNPD_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x55, 0x0a], "vandnpd ymm1, ymm0, ymmword [rdx]"); // VANDNPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x55, 0x0a], "vandnpd ymm1{k5}, ymm0, ymmword [rdx]"); // VANDNPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x55, 0x0a], "vandnpd zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VANDNPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x55, 0x0a], "vandnpd zmm1, zmm0, qword [rdx]{1to8}"); // VANDNPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x55, 0x0a], "vandnpd zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VANDNPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x55, 0x0a], "vandnpd xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VANDNPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x55, 0x0a], "vandnpd xmm1, xmm0, qword [rdx]{1to2}"); // VANDNPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x55, 0x0a], "vandnpd xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VANDNPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x55, 0xca], "vandnpd zmm1{k5}{z}, zmm0, zmm2"); // VANDNPD_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x55, 0x0a], "vandnpd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VANDNPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x55, 0xca], "vandnpd zmm1, zmm0, zmm2"); // VANDNPD_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x55, 0xca], "vandnpd zmm1{k5}, zmm0, zmm2"); // VANDNPD_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x55, 0x0a], "vandnpd zmm1, zmm0, zmmword [rdx]"); // VANDNPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x55, 0x0a], "vandnpd zmm1{k5}, zmm0, zmmword [rdx]"); // VANDNPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x55, 0xca], "vandnpd xmm1{k5}{z}, xmm0, xmm2"); // VANDNPD_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x55, 0x0a], "vandnpd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VANDNPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x55, 0xca], "vandnpd xmm1, xmm0, xmm2"); // VANDNPD_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x55, 0xca], "vandnpd xmm1{k5}, xmm0, xmm2"); // VANDNPD_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x55, 0x0a], "vandnpd xmm1, xmm0, xmmword [rdx]"); // VANDNPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x55, 0x0a], "vandnpd xmm1{k5}, xmm0, xmmword [rdx]"); // VANDNPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x56, 0x0a], "vorpd ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VORPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x56, 0x0a], "vorpd ymm1, ymm0, qword [rdx]{1to4}"); // VORPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x56, 0x0a], "vorpd ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VORPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x56, 0xca], "vorpd ymm1{k5}{z}, ymm0, ymm2"); // VORPD_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x56, 0x0a], "vorpd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VORPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x56, 0xca], "vorpd ymm1, ymm0, ymm2"); // VORPD_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x56, 0xca], "vorpd ymm1{k5}, ymm0, ymm2"); // VORPD_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x56, 0x0a], "vorpd ymm1, ymm0, ymmword [rdx]"); // VORPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x56, 0x0a], "vorpd ymm1{k5}, ymm0, ymmword [rdx]"); // VORPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x56, 0x0a], "vorpd zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VORPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x56, 0x0a], "vorpd zmm1, zmm0, qword [rdx]{1to8}"); // VORPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x56, 0x0a], "vorpd zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VORPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x56, 0x0a], "vorpd xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VORPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x56, 0x0a], "vorpd xmm1, xmm0, qword [rdx]{1to2}"); // VORPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x56, 0x0a], "vorpd xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VORPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x56, 0xca], "vorpd zmm1{k5}{z}, zmm0, zmm2"); // VORPD_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x56, 0x0a], "vorpd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VORPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x56, 0xca], "vorpd zmm1, zmm0, zmm2"); // VORPD_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x56, 0xca], "vorpd zmm1{k5}, zmm0, zmm2"); // VORPD_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x56, 0x0a], "vorpd zmm1, zmm0, zmmword [rdx]"); // VORPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x56, 0x0a], "vorpd zmm1{k5}, zmm0, zmmword [rdx]"); // VORPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x56, 0xca], "vorpd xmm1{k5}{z}, xmm0, xmm2"); // VORPD_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x56, 0x0a], "vorpd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VORPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x56, 0xca], "vorpd xmm1, xmm0, xmm2"); // VORPD_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x56, 0xca], "vorpd xmm1{k5}, xmm0, xmm2"); // VORPD_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x56, 0x0a], "vorpd xmm1, xmm0, xmmword [rdx]"); // VORPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x56, 0x0a], "vorpd xmm1{k5}, xmm0, xmmword [rdx]"); // VORPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x57, 0x0a], "vxorpd ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VXORPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x57, 0x0a], "vxorpd ymm1, ymm0, qword [rdx]{1to4}"); // VXORPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x57, 0x0a], "vxorpd ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VXORPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x57, 0xca], "vxorpd ymm1{k5}{z}, ymm0, ymm2"); // VXORPD_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x57, 0x0a], "vxorpd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VXORPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x57, 0xca], "vxorpd ymm1, ymm0, ymm2"); // VXORPD_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x57, 0xca], "vxorpd ymm1{k5}, ymm0, ymm2"); // VXORPD_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x57, 0x0a], "vxorpd ymm1, ymm0, ymmword [rdx]"); // VXORPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x57, 0x0a], "vxorpd ymm1{k5}, ymm0, ymmword [rdx]"); // VXORPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x57, 0x0a], "vxorpd zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VXORPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x57, 0x0a], "vxorpd zmm1, zmm0, qword [rdx]{1to8}"); // VXORPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x57, 0x0a], "vxorpd zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VXORPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x57, 0x0a], "vxorpd xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VXORPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x57, 0x0a], "vxorpd xmm1, xmm0, qword [rdx]{1to2}"); // VXORPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x57, 0x0a], "vxorpd xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VXORPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x57, 0xca], "vxorpd zmm1{k5}{z}, zmm0, zmm2"); // VXORPD_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x57, 0x0a], "vxorpd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VXORPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x57, 0xca], "vxorpd zmm1, zmm0, zmm2"); // VXORPD_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x57, 0xca], "vxorpd zmm1{k5}, zmm0, zmm2"); // VXORPD_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x57, 0x0a], "vxorpd zmm1, zmm0, zmmword [rdx]"); // VXORPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x57, 0x0a], "vxorpd zmm1{k5}, zmm0, zmmword [rdx]"); // VXORPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x57, 0xca], "vxorpd xmm1{k5}{z}, xmm0, xmm2"); // VXORPD_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x57, 0x0a], "vxorpd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VXORPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x57, 0xca], "vxorpd xmm1, xmm0, xmm2"); // VXORPD_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x57, 0xca], "vxorpd xmm1{k5}, xmm0, xmm2"); // VXORPD_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x57, 0x0a], "vxorpd xmm1, xmm0, xmmword [rdx]"); // VXORPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x57, 0x0a], "vxorpd xmm1{k5}, xmm0, xmmword [rdx]"); // VXORPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x58, 0xca], "vaddpd zmm1{k5}{z}{rz-sae}, zmm0, zmm2"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x58, 0xca], "vaddpd zmm1{rz-sae}, zmm0, zmm2"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x58, 0xca], "vaddpd zmm1{k5}{rz-sae}, zmm0, zmm2"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x58, 0xca], "vaddpd zmm1{k5}{z}{rd-sae}, zmm0, zmm2"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x58, 0x0a], "vaddpd ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VADDPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x58, 0xca], "vaddpd zmm1{rd-sae}, zmm0, zmm2"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x58, 0xca], "vaddpd zmm1{k5}{rd-sae}, zmm0, zmm2"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x58, 0x0a], "vaddpd ymm1, ymm0, qword [rdx]{1to4}"); // VADDPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x58, 0x0a], "vaddpd ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VADDPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x58, 0xca], "vaddpd ymm1{k5}{z}, ymm0, ymm2"); // VADDPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x58, 0x0a], "vaddpd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VADDPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x58, 0xca], "vaddpd ymm1, ymm0, ymm2"); // VADDPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x58, 0xca], "vaddpd ymm1{k5}, ymm0, ymm2"); // VADDPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x58, 0x0a], "vaddpd ymm1, ymm0, ymmword [rdx]"); // VADDPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x58, 0x0a], "vaddpd ymm1{k5}, ymm0, ymmword [rdx]"); // VADDPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x58, 0xca], "vaddpd zmm1{k5}{z}{ru-sae}, zmm0, zmm2"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x58, 0x0a], "vaddpd zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x58, 0xca], "vaddpd zmm1{ru-sae}, zmm0, zmm2"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x58, 0xca], "vaddpd zmm1{k5}{ru-sae}, zmm0, zmm2"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x58, 0x0a], "vaddpd zmm1, zmm0, qword [rdx]{1to8}"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x58, 0x0a], "vaddpd zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x58, 0xca], "vaddpd zmm1{k5}{z}{rne-sae}, zmm0, zmm2"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x58, 0x0a], "vaddpd xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VADDPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x58, 0xca], "vaddpd zmm1{rne-sae}, zmm0, zmm2"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x58, 0xca], "vaddpd zmm1{k5}{rne-sae}, zmm0, zmm2"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x58, 0x0a], "vaddpd xmm1, xmm0, qword [rdx]{1to2}"); // VADDPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x58, 0x0a], "vaddpd xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VADDPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x58, 0xca], "vaddpd zmm1{k5}{z}, zmm0, zmm2"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x58, 0x0a], "vaddpd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x58, 0xca], "vaddpd zmm1, zmm0, zmm2"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x58, 0xca], "vaddpd zmm1{k5}, zmm0, zmm2"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x58, 0x0a], "vaddpd zmm1, zmm0, zmmword [rdx]"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x58, 0x0a], "vaddpd zmm1{k5}, zmm0, zmmword [rdx]"); // VADDPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x58, 0xca], "vaddpd xmm1{k5}{z}, xmm0, xmm2"); // VADDPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x58, 0x0a], "vaddpd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VADDPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x58, 0xca], "vaddpd xmm1, xmm0, xmm2"); // VADDPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x58, 0xca], "vaddpd xmm1{k5}, xmm0, xmm2"); // VADDPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x58, 0x0a], "vaddpd xmm1, xmm0, xmmword [rdx]"); // VADDPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x58, 0x0a], "vaddpd xmm1{k5}, xmm0, xmmword [rdx]"); // VADDPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x59, 0xca], "vmulpd zmm1{k5}{z}{rz-sae}, zmm0, zmm2"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x59, 0xca], "vmulpd zmm1{rz-sae}, zmm0, zmm2"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x59, 0xca], "vmulpd zmm1{k5}{rz-sae}, zmm0, zmm2"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x59, 0xca], "vmulpd zmm1{k5}{z}{rd-sae}, zmm0, zmm2"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x59, 0x0a], "vmulpd ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VMULPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x59, 0xca], "vmulpd zmm1{rd-sae}, zmm0, zmm2"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x59, 0xca], "vmulpd zmm1{k5}{rd-sae}, zmm0, zmm2"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x59, 0x0a], "vmulpd ymm1, ymm0, qword [rdx]{1to4}"); // VMULPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x59, 0x0a], "vmulpd ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VMULPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x59, 0xca], "vmulpd ymm1{k5}{z}, ymm0, ymm2"); // VMULPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x59, 0x0a], "vmulpd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VMULPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x59, 0xca], "vmulpd ymm1, ymm0, ymm2"); // VMULPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x59, 0xca], "vmulpd ymm1{k5}, ymm0, ymm2"); // VMULPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x59, 0x0a], "vmulpd ymm1, ymm0, ymmword [rdx]"); // VMULPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x59, 0x0a], "vmulpd ymm1{k5}, ymm0, ymmword [rdx]"); // VMULPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x59, 0xca], "vmulpd zmm1{k5}{z}{ru-sae}, zmm0, zmm2"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x59, 0x0a], "vmulpd zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x59, 0xca], "vmulpd zmm1{ru-sae}, zmm0, zmm2"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x59, 0xca], "vmulpd zmm1{k5}{ru-sae}, zmm0, zmm2"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x59, 0x0a], "vmulpd zmm1, zmm0, qword [rdx]{1to8}"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x59, 0x0a], "vmulpd zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x59, 0xca], "vmulpd zmm1{k5}{z}{rne-sae}, zmm0, zmm2"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x59, 0x0a], "vmulpd xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VMULPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x59, 0xca], "vmulpd zmm1{rne-sae}, zmm0, zmm2"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x59, 0xca], "vmulpd zmm1{k5}{rne-sae}, zmm0, zmm2"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x59, 0x0a], "vmulpd xmm1, xmm0, qword [rdx]{1to2}"); // VMULPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x59, 0x0a], "vmulpd xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VMULPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x59, 0xca], "vmulpd zmm1{k5}{z}, zmm0, zmm2"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x59, 0x0a], "vmulpd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x59, 0xca], "vmulpd zmm1, zmm0, zmm2"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x59, 0xca], "vmulpd zmm1{k5}, zmm0, zmm2"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x59, 0x0a], "vmulpd zmm1, zmm0, zmmword [rdx]"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x59, 0x0a], "vmulpd zmm1{k5}, zmm0, zmmword [rdx]"); // VMULPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x59, 0xca], "vmulpd xmm1{k5}{z}, xmm0, xmm2"); // VMULPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x59, 0x0a], "vmulpd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VMULPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x59, 0xca], "vmulpd xmm1, xmm0, xmm2"); // VMULPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x59, 0xca], "vmulpd xmm1{k5}, xmm0, xmm2"); // VMULPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x59, 0x0a], "vmulpd xmm1, xmm0, xmmword [rdx]"); // VMULPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x59, 0x0a], "vmulpd xmm1{k5}, xmm0, xmmword [rdx]"); // VMULPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{z}{rz-sae}, zmm2"); // VCVTPD2PS_YMMf32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x5a, 0xca], "vcvtpd2ps ymm1{rz-sae}, zmm2"); // VCVTPD2PS_YMMf32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{rz-sae}, zmm2"); // VCVTPD2PS_YMMf32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{z}{rd-sae}, zmm2"); // VCVTPD2PS_YMMf32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}{z}, qword [rdx]{1to4}"); // VCVTPD2PS_XMMf32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5a, 0xca], "vcvtpd2ps ymm1{rd-sae}, zmm2"); // VCVTPD2PS_YMMf32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{rd-sae}, zmm2"); // VCVTPD2PS_YMMf32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5a, 0x0a], "vcvtpd2ps xmm1, qword [rdx]{1to4}"); // VCVTPD2PS_XMMf32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}, qword [rdx]{1to4}"); // VCVTPD2PS_XMMf32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5a, 0xca], "vcvtpd2ps xmm1{k5}{z}, ymm2"); // VCVTPD2PS_XMMf32_MASKmskw_YMMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}{z}, ymmword [rdx]"); // VCVTPD2PS_XMMf32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5a, 0xca], "vcvtpd2ps xmm1, ymm2"); // VCVTPD2PS_XMMf32_MASKmskw_YMMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5a, 0xca], "vcvtpd2ps xmm1{k5}, ymm2"); // VCVTPD2PS_XMMf32_MASKmskw_YMMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5a, 0x0a], "vcvtpd2ps xmm1, ymmword [rdx]"); // VCVTPD2PS_XMMf32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}, ymmword [rdx]"); // VCVTPD2PS_XMMf32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{z}{ru-sae}, zmm2"); // VCVTPD2PS_YMMf32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5a, 0x0a], "vcvtpd2ps ymm1{k5}{z}, qword [rdx]{1to8}"); // VCVTPD2PS_YMMf32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5a, 0xca], "vcvtpd2ps ymm1{ru-sae}, zmm2"); // VCVTPD2PS_YMMf32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{ru-sae}, zmm2"); // VCVTPD2PS_YMMf32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5a, 0x0a], "vcvtpd2ps ymm1, qword [rdx]{1to8}"); // VCVTPD2PS_YMMf32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5a, 0x0a], "vcvtpd2ps ymm1{k5}, qword [rdx]{1to8}"); // VCVTPD2PS_YMMf32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{z}{rne-sae}, zmm2"); // VCVTPD2PS_YMMf32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}{z}, qword [rdx]{1to2}"); // VCVTPD2PS_XMMf32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5a, 0xca], "vcvtpd2ps ymm1{rne-sae}, zmm2"); // VCVTPD2PS_YMMf32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{rne-sae}, zmm2"); // VCVTPD2PS_YMMf32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5a, 0x0a], "vcvtpd2ps xmm1, qword [rdx]{1to2}"); // VCVTPD2PS_XMMf32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}, qword [rdx]{1to2}"); // VCVTPD2PS_XMMf32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}{z}, zmm2"); // VCVTPD2PS_YMMf32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5a, 0x0a], "vcvtpd2ps ymm1{k5}{z}, zmmword [rdx]"); // VCVTPD2PS_YMMf32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5a, 0xca], "vcvtpd2ps ymm1, zmm2"); // VCVTPD2PS_YMMf32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5a, 0xca], "vcvtpd2ps ymm1{k5}, zmm2"); // VCVTPD2PS_YMMf32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5a, 0x0a], "vcvtpd2ps ymm1, zmmword [rdx]"); // VCVTPD2PS_YMMf32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5a, 0x0a], "vcvtpd2ps ymm1{k5}, zmmword [rdx]"); // VCVTPD2PS_YMMf32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5a, 0xca], "vcvtpd2ps xmm1{k5}{z}, xmm2"); // VCVTPD2PS_XMMf32_MASKmskw_XMMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}{z}, xmmword [rdx]"); // VCVTPD2PS_XMMf32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5a, 0xca], "vcvtpd2ps xmm1, xmm2"); // VCVTPD2PS_XMMf32_MASKmskw_XMMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5a, 0xca], "vcvtpd2ps xmm1{k5}, xmm2"); // VCVTPD2PS_XMMf32_MASKmskw_XMMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5a, 0x0a], "vcvtpd2ps xmm1, xmmword [rdx]"); // VCVTPD2PS_XMMf32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5a, 0x0a], "vcvtpd2ps xmm1{k5}, xmmword [rdx]"); // VCVTPD2PS_XMMf32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xfd, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{z}{rz-sae}, zmm2"); // VCVTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x78, 0x5b, 0xca], "vcvtps2dq zmm1{rz-sae}, zmm2"); // VCVTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x7d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{rz-sae}, zmm2"); // VCVTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{z}{rd-sae}, zmm2"); // VCVTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x5b, 0x0a], "vcvtps2dq ymm1{k5}{z}, dword [rdx]{1to8}"); // VCVTPS2DQ_YMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x5b, 0xca], "vcvtps2dq zmm1{rd-sae}, zmm2"); // VCVTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{rd-sae}, zmm2"); // VCVTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x5b, 0x0a], "vcvtps2dq ymm1, dword [rdx]{1to8}"); // VCVTPS2DQ_YMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x5b, 0x0a], "vcvtps2dq ymm1{k5}, dword [rdx]{1to8}"); // VCVTPS2DQ_YMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x5b, 0xca], "vcvtps2dq ymm1{k5}{z}, ymm2"); // VCVTPS2DQ_YMMi32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x5b, 0x0a], "vcvtps2dq ymm1{k5}{z}, ymmword [rdx]"); // VCVTPS2DQ_YMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x5b, 0xca], "vcvtps2dq ymm1, ymm2"); // VCVTPS2DQ_YMMi32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x5b, 0xca], "vcvtps2dq ymm1{k5}, ymm2"); // VCVTPS2DQ_YMMi32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x5b, 0x0a], "vcvtps2dq ymm1, ymmword [rdx]"); // VCVTPS2DQ_YMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x5b, 0x0a], "vcvtps2dq ymm1{k5}, ymmword [rdx]"); // VCVTPS2DQ_YMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{z}{ru-sae}, zmm2"); // VCVTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x5b, 0x0a], "vcvtps2dq zmm1{k5}{z}, dword [rdx]{1to16}"); // VCVTPS2DQ_ZMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x5b, 0xca], "vcvtps2dq zmm1{ru-sae}, zmm2"); // VCVTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{ru-sae}, zmm2"); // VCVTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x5b, 0x0a], "vcvtps2dq zmm1, dword [rdx]{1to16}"); // VCVTPS2DQ_ZMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x5b, 0x0a], "vcvtps2dq zmm1{k5}, dword [rdx]{1to16}"); // VCVTPS2DQ_ZMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{z}{rne-sae}, zmm2"); // VCVTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x5b, 0x0a], "vcvtps2dq xmm1{k5}{z}, dword [rdx]{1to4}"); // VCVTPS2DQ_XMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x5b, 0xca], "vcvtps2dq zmm1{rne-sae}, zmm2"); // VCVTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{rne-sae}, zmm2"); // VCVTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x5b, 0x0a], "vcvtps2dq xmm1, dword [rdx]{1to4}"); // VCVTPS2DQ_XMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x5b, 0x0a], "vcvtps2dq xmm1{k5}, dword [rdx]{1to4}"); // VCVTPS2DQ_XMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x5b, 0xca], "vcvtps2dq zmm1{k5}{z}, zmm2"); // VCVTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x5b, 0x0a], "vcvtps2dq zmm1{k5}{z}, zmmword [rdx]"); // VCVTPS2DQ_ZMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x5b, 0xca], "vcvtps2dq zmm1, zmm2"); // VCVTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x5b, 0xca], "vcvtps2dq zmm1{k5}, zmm2"); // VCVTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x5b, 0x0a], "vcvtps2dq zmm1, zmmword [rdx]"); // VCVTPS2DQ_ZMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x5b, 0x0a], "vcvtps2dq zmm1{k5}, zmmword [rdx]"); // VCVTPS2DQ_ZMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x5b, 0xca], "vcvtps2dq xmm1{k5}{z}, xmm2"); // VCVTPS2DQ_XMMi32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x5b, 0x0a], "vcvtps2dq xmm1{k5}{z}, xmmword [rdx]"); // VCVTPS2DQ_XMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x5b, 0xca], "vcvtps2dq xmm1, xmm2"); // VCVTPS2DQ_XMMi32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x5b, 0xca], "vcvtps2dq xmm1{k5}, xmm2"); // VCVTPS2DQ_XMMi32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x5b, 0x0a], "vcvtps2dq xmm1, xmmword [rdx]"); // VCVTPS2DQ_XMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x5b, 0x0a], "vcvtps2dq xmm1{k5}, xmmword [rdx]"); // VCVTPS2DQ_XMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x5c, 0xca], "vsubpd zmm1{k5}{z}{rz-sae}, zmm0, zmm2"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x5c, 0xca], "vsubpd zmm1{rz-sae}, zmm0, zmm2"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x5c, 0xca], "vsubpd zmm1{k5}{rz-sae}, zmm0, zmm2"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5c, 0xca], "vsubpd zmm1{k5}{z}{rd-sae}, zmm0, zmm2"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5c, 0x0a], "vsubpd ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VSUBPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5c, 0xca], "vsubpd zmm1{rd-sae}, zmm0, zmm2"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5c, 0xca], "vsubpd zmm1{k5}{rd-sae}, zmm0, zmm2"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5c, 0x0a], "vsubpd ymm1, ymm0, qword [rdx]{1to4}"); // VSUBPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5c, 0x0a], "vsubpd ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VSUBPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5c, 0xca], "vsubpd ymm1{k5}{z}, ymm0, ymm2"); // VSUBPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5c, 0x0a], "vsubpd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VSUBPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5c, 0xca], "vsubpd ymm1, ymm0, ymm2"); // VSUBPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5c, 0xca], "vsubpd ymm1{k5}, ymm0, ymm2"); // VSUBPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5c, 0x0a], "vsubpd ymm1, ymm0, ymmword [rdx]"); // VSUBPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5c, 0x0a], "vsubpd ymm1{k5}, ymm0, ymmword [rdx]"); // VSUBPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5c, 0xca], "vsubpd zmm1{k5}{z}{ru-sae}, zmm0, zmm2"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5c, 0x0a], "vsubpd zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5c, 0xca], "vsubpd zmm1{ru-sae}, zmm0, zmm2"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5c, 0xca], "vsubpd zmm1{k5}{ru-sae}, zmm0, zmm2"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5c, 0x0a], "vsubpd zmm1, zmm0, qword [rdx]{1to8}"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5c, 0x0a], "vsubpd zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5c, 0xca], "vsubpd zmm1{k5}{z}{rne-sae}, zmm0, zmm2"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5c, 0x0a], "vsubpd xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VSUBPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5c, 0xca], "vsubpd zmm1{rne-sae}, zmm0, zmm2"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5c, 0xca], "vsubpd zmm1{k5}{rne-sae}, zmm0, zmm2"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5c, 0x0a], "vsubpd xmm1, xmm0, qword [rdx]{1to2}"); // VSUBPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5c, 0x0a], "vsubpd xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VSUBPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5c, 0xca], "vsubpd zmm1{k5}{z}, zmm0, zmm2"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5c, 0x0a], "vsubpd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5c, 0xca], "vsubpd zmm1, zmm0, zmm2"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5c, 0xca], "vsubpd zmm1{k5}, zmm0, zmm2"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5c, 0x0a], "vsubpd zmm1, zmm0, zmmword [rdx]"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5c, 0x0a], "vsubpd zmm1{k5}, zmm0, zmmword [rdx]"); // VSUBPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5c, 0xca], "vsubpd xmm1{k5}{z}, xmm0, xmm2"); // VSUBPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5c, 0x0a], "vsubpd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VSUBPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5c, 0xca], "vsubpd xmm1, xmm0, xmm2"); // VSUBPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5c, 0xca], "vsubpd xmm1{k5}, xmm0, xmm2"); // VSUBPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5c, 0x0a], "vsubpd xmm1, xmm0, xmmword [rdx]"); // VSUBPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5c, 0x0a], "vsubpd xmm1{k5}, xmm0, xmmword [rdx]"); // VSUBPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x5d, 0xca], "vminpd zmm1{k5}{z}{sae}, zmm0, zmm2"); // VMINPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x5d, 0xca], "vminpd zmm1{sae}, zmm0, zmm2"); // VMINPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x5d, 0xca], "vminpd zmm1{k5}{sae}, zmm0, zmm2"); // VMINPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5d, 0x0a], "vminpd ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VMINPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5d, 0x0a], "vminpd ymm1, ymm0, qword [rdx]{1to4}"); // VMINPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5d, 0x0a], "vminpd ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VMINPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5d, 0xca], "vminpd ymm1{k5}{z}, ymm0, ymm2"); // VMINPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5d, 0x0a], "vminpd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VMINPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5d, 0xca], "vminpd ymm1, ymm0, ymm2"); // VMINPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5d, 0xca], "vminpd ymm1{k5}, ymm0, ymm2"); // VMINPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5d, 0x0a], "vminpd ymm1, ymm0, ymmword [rdx]"); // VMINPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5d, 0x0a], "vminpd ymm1{k5}, ymm0, ymmword [rdx]"); // VMINPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5d, 0x0a], "vminpd zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VMINPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5d, 0x0a], "vminpd zmm1, zmm0, qword [rdx]{1to8}"); // VMINPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5d, 0x0a], "vminpd zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VMINPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5d, 0x0a], "vminpd xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VMINPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5d, 0x0a], "vminpd xmm1, xmm0, qword [rdx]{1to2}"); // VMINPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5d, 0x0a], "vminpd xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VMINPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5d, 0xca], "vminpd zmm1{k5}{z}, zmm0, zmm2"); // VMINPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5d, 0x0a], "vminpd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VMINPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5d, 0xca], "vminpd zmm1, zmm0, zmm2"); // VMINPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5d, 0xca], "vminpd zmm1{k5}, zmm0, zmm2"); // VMINPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5d, 0x0a], "vminpd zmm1, zmm0, zmmword [rdx]"); // VMINPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5d, 0x0a], "vminpd zmm1{k5}, zmm0, zmmword [rdx]"); // VMINPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5d, 0xca], "vminpd xmm1{k5}{z}, xmm0, xmm2"); // VMINPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5d, 0x0a], "vminpd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VMINPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5d, 0xca], "vminpd xmm1, xmm0, xmm2"); // VMINPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5d, 0xca], "vminpd xmm1{k5}, xmm0, xmm2"); // VMINPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5d, 0x0a], "vminpd xmm1, xmm0, xmmword [rdx]"); // VMINPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5d, 0x0a], "vminpd xmm1{k5}, xmm0, xmmword [rdx]"); // VMINPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x5e, 0xca], "vdivpd zmm1{k5}{z}{rz-sae}, zmm0, zmm2"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x5e, 0xca], "vdivpd zmm1{rz-sae}, zmm0, zmm2"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x5e, 0xca], "vdivpd zmm1{k5}{rz-sae}, zmm0, zmm2"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5e, 0xca], "vdivpd zmm1{k5}{z}{rd-sae}, zmm0, zmm2"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5e, 0x0a], "vdivpd ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VDIVPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5e, 0xca], "vdivpd zmm1{rd-sae}, zmm0, zmm2"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5e, 0xca], "vdivpd zmm1{k5}{rd-sae}, zmm0, zmm2"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5e, 0x0a], "vdivpd ymm1, ymm0, qword [rdx]{1to4}"); // VDIVPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5e, 0x0a], "vdivpd ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VDIVPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5e, 0xca], "vdivpd ymm1{k5}{z}, ymm0, ymm2"); // VDIVPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5e, 0x0a], "vdivpd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VDIVPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5e, 0xca], "vdivpd ymm1, ymm0, ymm2"); // VDIVPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5e, 0xca], "vdivpd ymm1{k5}, ymm0, ymm2"); // VDIVPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5e, 0x0a], "vdivpd ymm1, ymm0, ymmword [rdx]"); // VDIVPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5e, 0x0a], "vdivpd ymm1{k5}, ymm0, ymmword [rdx]"); // VDIVPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5e, 0xca], "vdivpd zmm1{k5}{z}{ru-sae}, zmm0, zmm2"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5e, 0x0a], "vdivpd zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5e, 0xca], "vdivpd zmm1{ru-sae}, zmm0, zmm2"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5e, 0xca], "vdivpd zmm1{k5}{ru-sae}, zmm0, zmm2"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5e, 0x0a], "vdivpd zmm1, zmm0, qword [rdx]{1to8}"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5e, 0x0a], "vdivpd zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5e, 0xca], "vdivpd zmm1{k5}{z}{rne-sae}, zmm0, zmm2"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5e, 0x0a], "vdivpd xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VDIVPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5e, 0xca], "vdivpd zmm1{rne-sae}, zmm0, zmm2"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5e, 0xca], "vdivpd zmm1{k5}{rne-sae}, zmm0, zmm2"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5e, 0x0a], "vdivpd xmm1, xmm0, qword [rdx]{1to2}"); // VDIVPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5e, 0x0a], "vdivpd xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VDIVPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5e, 0xca], "vdivpd zmm1{k5}{z}, zmm0, zmm2"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5e, 0x0a], "vdivpd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5e, 0xca], "vdivpd zmm1, zmm0, zmm2"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5e, 0xca], "vdivpd zmm1{k5}, zmm0, zmm2"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5e, 0x0a], "vdivpd zmm1, zmm0, zmmword [rdx]"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5e, 0x0a], "vdivpd zmm1{k5}, zmm0, zmmword [rdx]"); // VDIVPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5e, 0xca], "vdivpd xmm1{k5}{z}, xmm0, xmm2"); // VDIVPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5e, 0x0a], "vdivpd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VDIVPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5e, 0xca], "vdivpd xmm1, xmm0, xmm2"); // VDIVPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5e, 0xca], "vdivpd xmm1{k5}, xmm0, xmm2"); // VDIVPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5e, 0x0a], "vdivpd xmm1, xmm0, xmmword [rdx]"); // VDIVPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5e, 0x0a], "vdivpd xmm1{k5}, xmm0, xmmword [rdx]"); // VDIVPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x5f, 0xca], "vmaxpd zmm1{k5}{z}{sae}, zmm0, zmm2"); // VMAXPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x5f, 0xca], "vmaxpd zmm1{sae}, zmm0, zmm2"); // VMAXPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x5f, 0xca], "vmaxpd zmm1{k5}{sae}, zmm0, zmm2"); // VMAXPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x5f, 0x0a], "vmaxpd ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VMAXPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x5f, 0x0a], "vmaxpd ymm1, ymm0, qword [rdx]{1to4}"); // VMAXPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x5f, 0x0a], "vmaxpd ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VMAXPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5f, 0xca], "vmaxpd ymm1{k5}{z}, ymm0, ymm2"); // VMAXPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x5f, 0x0a], "vmaxpd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VMAXPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5f, 0xca], "vmaxpd ymm1, ymm0, ymm2"); // VMAXPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5f, 0xca], "vmaxpd ymm1{k5}, ymm0, ymm2"); // VMAXPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x5f, 0x0a], "vmaxpd ymm1, ymm0, ymmword [rdx]"); // VMAXPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x5f, 0x0a], "vmaxpd ymm1{k5}, ymm0, ymmword [rdx]"); // VMAXPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x5f, 0x0a], "vmaxpd zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VMAXPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x5f, 0x0a], "vmaxpd zmm1, zmm0, qword [rdx]{1to8}"); // VMAXPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x5f, 0x0a], "vmaxpd zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VMAXPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x5f, 0x0a], "vmaxpd xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VMAXPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x5f, 0x0a], "vmaxpd xmm1, xmm0, qword [rdx]{1to2}"); // VMAXPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x5f, 0x0a], "vmaxpd xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VMAXPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5f, 0xca], "vmaxpd zmm1{k5}{z}, zmm0, zmm2"); // VMAXPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x5f, 0x0a], "vmaxpd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VMAXPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5f, 0xca], "vmaxpd zmm1, zmm0, zmm2"); // VMAXPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5f, 0xca], "vmaxpd zmm1{k5}, zmm0, zmm2"); // VMAXPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x5f, 0x0a], "vmaxpd zmm1, zmm0, zmmword [rdx]"); // VMAXPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x5f, 0x0a], "vmaxpd zmm1{k5}, zmm0, zmmword [rdx]"); // VMAXPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5f, 0xca], "vmaxpd xmm1{k5}{z}, xmm0, xmm2"); // VMAXPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x5f, 0x0a], "vmaxpd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VMAXPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5f, 0xca], "vmaxpd xmm1, xmm0, xmm2"); // VMAXPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5f, 0xca], "vmaxpd xmm1{k5}, xmm0, xmm2"); // VMAXPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x5f, 0x0a], "vmaxpd xmm1, xmm0, xmmword [rdx]"); // VMAXPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x5f, 0x0a], "vmaxpd xmm1{k5}, xmm0, xmmword [rdx]"); // VMAXPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x60, 0xca], "vpunpcklbw ymm1{k5}{z}, ymm0, ymm2"); // VPUNPCKLBW_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x60, 0x0a], "vpunpcklbw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPUNPCKLBW_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x60, 0xca], "vpunpcklbw ymm1, ymm0, ymm2"); // VPUNPCKLBW_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x60, 0xca], "vpunpcklbw ymm1{k5}, ymm0, ymm2"); // VPUNPCKLBW_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x60, 0x0a], "vpunpcklbw ymm1, ymm0, ymmword [rdx]"); // VPUNPCKLBW_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x60, 0x0a], "vpunpcklbw ymm1{k5}, ymm0, ymmword [rdx]"); // VPUNPCKLBW_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x60, 0xca], "vpunpcklbw zmm1{k5}{z}, zmm0, zmm2"); // VPUNPCKLBW_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x60, 0x0a], "vpunpcklbw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPUNPCKLBW_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x60, 0xca], "vpunpcklbw zmm1, zmm0, zmm2"); // VPUNPCKLBW_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x60, 0xca], "vpunpcklbw zmm1{k5}, zmm0, zmm2"); // VPUNPCKLBW_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x60, 0x0a], "vpunpcklbw zmm1, zmm0, zmmword [rdx]"); // VPUNPCKLBW_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x60, 0x0a], "vpunpcklbw zmm1{k5}, zmm0, zmmword [rdx]"); // VPUNPCKLBW_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x60, 0xca], "vpunpcklbw xmm1{k5}{z}, xmm0, xmm2"); // VPUNPCKLBW_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x60, 0x0a], "vpunpcklbw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPUNPCKLBW_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x60, 0xca], "vpunpcklbw xmm1, xmm0, xmm2"); // VPUNPCKLBW_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x60, 0xca], "vpunpcklbw xmm1{k5}, xmm0, xmm2"); // VPUNPCKLBW_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x60, 0x0a], "vpunpcklbw xmm1, xmm0, xmmword [rdx]"); // VPUNPCKLBW_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x60, 0x0a], "vpunpcklbw xmm1{k5}, xmm0, xmmword [rdx]"); // VPUNPCKLBW_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x61, 0xca], "vpunpcklwd ymm1{k5}{z}, ymm0, ymm2"); // VPUNPCKLWD_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x61, 0x0a], "vpunpcklwd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPUNPCKLWD_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x61, 0xca], "vpunpcklwd ymm1, ymm0, ymm2"); // VPUNPCKLWD_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x61, 0xca], "vpunpcklwd ymm1{k5}, ymm0, ymm2"); // VPUNPCKLWD_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x61, 0x0a], "vpunpcklwd ymm1, ymm0, ymmword [rdx]"); // VPUNPCKLWD_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x61, 0x0a], "vpunpcklwd ymm1{k5}, ymm0, ymmword [rdx]"); // VPUNPCKLWD_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x61, 0xca], "vpunpcklwd zmm1{k5}{z}, zmm0, zmm2"); // VPUNPCKLWD_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x61, 0x0a], "vpunpcklwd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPUNPCKLWD_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x61, 0xca], "vpunpcklwd zmm1, zmm0, zmm2"); // VPUNPCKLWD_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x61, 0xca], "vpunpcklwd zmm1{k5}, zmm0, zmm2"); // VPUNPCKLWD_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x61, 0x0a], "vpunpcklwd zmm1, zmm0, zmmword [rdx]"); // VPUNPCKLWD_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x61, 0x0a], "vpunpcklwd zmm1{k5}, zmm0, zmmword [rdx]"); // VPUNPCKLWD_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x61, 0xca], "vpunpcklwd xmm1{k5}{z}, xmm0, xmm2"); // VPUNPCKLWD_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x61, 0x0a], "vpunpcklwd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPUNPCKLWD_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x61, 0xca], "vpunpcklwd xmm1, xmm0, xmm2"); // VPUNPCKLWD_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x61, 0xca], "vpunpcklwd xmm1{k5}, xmm0, xmm2"); // VPUNPCKLWD_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x61, 0x0a], "vpunpcklwd xmm1, xmm0, xmmword [rdx]"); // VPUNPCKLWD_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x61, 0x0a], "vpunpcklwd xmm1{k5}, xmm0, xmmword [rdx]"); // VPUNPCKLWD_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x62, 0x0a], "vpunpckldq ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPUNPCKLDQ_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x62, 0x0a], "vpunpckldq ymm1, ymm0, dword [rdx]{1to8}"); // VPUNPCKLDQ_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x62, 0x0a], "vpunpckldq ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPUNPCKLDQ_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x62, 0xca], "vpunpckldq ymm1{k5}{z}, ymm0, ymm2"); // VPUNPCKLDQ_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x62, 0x0a], "vpunpckldq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPUNPCKLDQ_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x62, 0xca], "vpunpckldq ymm1, ymm0, ymm2"); // VPUNPCKLDQ_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x62, 0xca], "vpunpckldq ymm1{k5}, ymm0, ymm2"); // VPUNPCKLDQ_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x62, 0x0a], "vpunpckldq ymm1, ymm0, ymmword [rdx]"); // VPUNPCKLDQ_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x62, 0x0a], "vpunpckldq ymm1{k5}, ymm0, ymmword [rdx]"); // VPUNPCKLDQ_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x62, 0x0a], "vpunpckldq zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPUNPCKLDQ_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x62, 0x0a], "vpunpckldq zmm1, zmm0, dword [rdx]{1to16}"); // VPUNPCKLDQ_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x62, 0x0a], "vpunpckldq zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPUNPCKLDQ_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x62, 0x0a], "vpunpckldq xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPUNPCKLDQ_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x62, 0x0a], "vpunpckldq xmm1, xmm0, dword [rdx]{1to4}"); // VPUNPCKLDQ_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x62, 0x0a], "vpunpckldq xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPUNPCKLDQ_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x62, 0xca], "vpunpckldq zmm1{k5}{z}, zmm0, zmm2"); // VPUNPCKLDQ_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x62, 0x0a], "vpunpckldq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPUNPCKLDQ_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x62, 0xca], "vpunpckldq zmm1, zmm0, zmm2"); // VPUNPCKLDQ_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x62, 0xca], "vpunpckldq zmm1{k5}, zmm0, zmm2"); // VPUNPCKLDQ_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x62, 0x0a], "vpunpckldq zmm1, zmm0, zmmword [rdx]"); // VPUNPCKLDQ_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x62, 0x0a], "vpunpckldq zmm1{k5}, zmm0, zmmword [rdx]"); // VPUNPCKLDQ_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x62, 0xca], "vpunpckldq xmm1{k5}{z}, xmm0, xmm2"); // VPUNPCKLDQ_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x62, 0x0a], "vpunpckldq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPUNPCKLDQ_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x62, 0xca], "vpunpckldq xmm1, xmm0, xmm2"); // VPUNPCKLDQ_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x62, 0xca], "vpunpckldq xmm1{k5}, xmm0, xmm2"); // VPUNPCKLDQ_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x62, 0x0a], "vpunpckldq xmm1, xmm0, xmmword [rdx]"); // VPUNPCKLDQ_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x62, 0x0a], "vpunpckldq xmm1{k5}, xmm0, xmmword [rdx]"); // VPUNPCKLDQ_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x63, 0xca], "vpacksswb ymm1{k5}{z}, ymm0, ymm2"); // VPACKSSWB_YMMi8_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x63, 0x0a], "vpacksswb ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPACKSSWB_YMMi8_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x63, 0xca], "vpacksswb ymm1, ymm0, ymm2"); // VPACKSSWB_YMMi8_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x63, 0xca], "vpacksswb ymm1{k5}, ymm0, ymm2"); // VPACKSSWB_YMMi8_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x63, 0x0a], "vpacksswb ymm1, ymm0, ymmword [rdx]"); // VPACKSSWB_YMMi8_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x63, 0x0a], "vpacksswb ymm1{k5}, ymm0, ymmword [rdx]"); // VPACKSSWB_YMMi8_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x63, 0xca], "vpacksswb zmm1{k5}{z}, zmm0, zmm2"); // VPACKSSWB_ZMMi8_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x63, 0x0a], "vpacksswb zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPACKSSWB_ZMMi8_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x63, 0xca], "vpacksswb zmm1, zmm0, zmm2"); // VPACKSSWB_ZMMi8_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x63, 0xca], "vpacksswb zmm1{k5}, zmm0, zmm2"); // VPACKSSWB_ZMMi8_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x63, 0x0a], "vpacksswb zmm1, zmm0, zmmword [rdx]"); // VPACKSSWB_ZMMi8_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x63, 0x0a], "vpacksswb zmm1{k5}, zmm0, zmmword [rdx]"); // VPACKSSWB_ZMMi8_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x63, 0xca], "vpacksswb xmm1{k5}{z}, xmm0, xmm2"); // VPACKSSWB_XMMi8_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x63, 0x0a], "vpacksswb xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPACKSSWB_XMMi8_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x63, 0xca], "vpacksswb xmm1, xmm0, xmm2"); // VPACKSSWB_XMMi8_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x63, 0xca], "vpacksswb xmm1{k5}, xmm0, xmm2"); // VPACKSSWB_XMMi8_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x63, 0x0a], "vpacksswb xmm1, xmm0, xmmword [rdx]"); // VPACKSSWB_XMMi8_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x63, 0x0a], "vpacksswb xmm1{k5}, xmm0, xmmword [rdx]"); // VPACKSSWB_XMMi8_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x64, 0xca], "vpcmpgtb k1, ymm0, ymm2"); // VPCMPGTB_MASKmskw_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x64, 0xca], "vpcmpgtb k1{k5}, ymm0, ymm2"); // VPCMPGTB_MASKmskw_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x64, 0x0a], "vpcmpgtb k1, ymm0, ymmword [rdx]"); // VPCMPGTB_MASKmskw_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x64, 0x0a], "vpcmpgtb k1{k5}, ymm0, ymmword [rdx]"); // VPCMPGTB_MASKmskw_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x64, 0xca], "vpcmpgtb k1, zmm0, zmm2"); // VPCMPGTB_MASKmskw_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x64, 0xca], "vpcmpgtb k1{k5}, zmm0, zmm2"); // VPCMPGTB_MASKmskw_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x64, 0x0a], "vpcmpgtb k1, zmm0, zmmword [rdx]"); // VPCMPGTB_MASKmskw_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x64, 0x0a], "vpcmpgtb k1{k5}, zmm0, zmmword [rdx]"); // VPCMPGTB_MASKmskw_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x64, 0xca], "vpcmpgtb k1, xmm0, xmm2"); // VPCMPGTB_MASKmskw_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x64, 0xca], "vpcmpgtb k1{k5}, xmm0, xmm2"); // VPCMPGTB_MASKmskw_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x64, 0x0a], "vpcmpgtb k1, xmm0, xmmword [rdx]"); // VPCMPGTB_MASKmskw_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x64, 0x0a], "vpcmpgtb k1{k5}, xmm0, xmmword [rdx]"); // VPCMPGTB_MASKmskw_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x65, 0xca], "vpcmpgtw k1, ymm0, ymm2"); // VPCMPGTW_MASKmskw_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x65, 0xca], "vpcmpgtw k1{k5}, ymm0, ymm2"); // VPCMPGTW_MASKmskw_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x65, 0x0a], "vpcmpgtw k1, ymm0, ymmword [rdx]"); // VPCMPGTW_MASKmskw_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x65, 0x0a], "vpcmpgtw k1{k5}, ymm0, ymmword [rdx]"); // VPCMPGTW_MASKmskw_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x65, 0xca], "vpcmpgtw k1, zmm0, zmm2"); // VPCMPGTW_MASKmskw_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x65, 0xca], "vpcmpgtw k1{k5}, zmm0, zmm2"); // VPCMPGTW_MASKmskw_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x65, 0x0a], "vpcmpgtw k1, zmm0, zmmword [rdx]"); // VPCMPGTW_MASKmskw_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x65, 0x0a], "vpcmpgtw k1{k5}, zmm0, zmmword [rdx]"); // VPCMPGTW_MASKmskw_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x65, 0xca], "vpcmpgtw k1, xmm0, xmm2"); // VPCMPGTW_MASKmskw_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x65, 0xca], "vpcmpgtw k1{k5}, xmm0, xmm2"); // VPCMPGTW_MASKmskw_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x65, 0x0a], "vpcmpgtw k1, xmm0, xmmword [rdx]"); // VPCMPGTW_MASKmskw_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x65, 0x0a], "vpcmpgtw k1{k5}, xmm0, xmmword [rdx]"); // VPCMPGTW_MASKmskw_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x66, 0x0a], "vpcmpgtd k1, ymm0, dword [rdx]{1to8}"); // VPCMPGTD_MASKmskw_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x66, 0x0a], "vpcmpgtd k1{k5}, ymm0, dword [rdx]{1to8}"); // VPCMPGTD_MASKmskw_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x66, 0xca], "vpcmpgtd k1, ymm0, ymm2"); // VPCMPGTD_MASKmskw_MASKmskw_YMMi32_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x66, 0xca], "vpcmpgtd k1{k5}, ymm0, ymm2"); // VPCMPGTD_MASKmskw_MASKmskw_YMMi32_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x66, 0x0a], "vpcmpgtd k1, ymm0, ymmword [rdx]"); // VPCMPGTD_MASKmskw_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x66, 0x0a], "vpcmpgtd k1{k5}, ymm0, ymmword [rdx]"); // VPCMPGTD_MASKmskw_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x66, 0x0a], "vpcmpgtd k1, zmm0, dword [rdx]{1to16}"); // VPCMPGTD_MASKmskw_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x66, 0x0a], "vpcmpgtd k1{k5}, zmm0, dword [rdx]{1to16}"); // VPCMPGTD_MASKmskw_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x66, 0x0a], "vpcmpgtd k1, xmm0, dword [rdx]{1to4}"); // VPCMPGTD_MASKmskw_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x66, 0x0a], "vpcmpgtd k1{k5}, xmm0, dword [rdx]{1to4}"); // VPCMPGTD_MASKmskw_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x66, 0xca], "vpcmpgtd k1, zmm0, zmm2"); // VPCMPGTD_MASKmskw_MASKmskw_ZMMi32_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x66, 0xca], "vpcmpgtd k1{k5}, zmm0, zmm2"); // VPCMPGTD_MASKmskw_MASKmskw_ZMMi32_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x66, 0x0a], "vpcmpgtd k1, zmm0, zmmword [rdx]"); // VPCMPGTD_MASKmskw_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x66, 0x0a], "vpcmpgtd k1{k5}, zmm0, zmmword [rdx]"); // VPCMPGTD_MASKmskw_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x66, 0xca], "vpcmpgtd k1, xmm0, xmm2"); // VPCMPGTD_MASKmskw_MASKmskw_XMMi32_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x66, 0xca], "vpcmpgtd k1{k5}, xmm0, xmm2"); // VPCMPGTD_MASKmskw_MASKmskw_XMMi32_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x66, 0x0a], "vpcmpgtd k1, xmm0, xmmword [rdx]"); // VPCMPGTD_MASKmskw_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x66, 0x0a], "vpcmpgtd k1{k5}, xmm0, xmmword [rdx]"); // VPCMPGTD_MASKmskw_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x67, 0xca], "vpackuswb ymm1{k5}{z}, ymm0, ymm2"); // VPACKUSWB_YMMu8_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x67, 0x0a], "vpackuswb ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPACKUSWB_YMMu8_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x67, 0xca], "vpackuswb ymm1, ymm0, ymm2"); // VPACKUSWB_YMMu8_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x67, 0xca], "vpackuswb ymm1{k5}, ymm0, ymm2"); // VPACKUSWB_YMMu8_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x67, 0x0a], "vpackuswb ymm1, ymm0, ymmword [rdx]"); // VPACKUSWB_YMMu8_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x67, 0x0a], "vpackuswb ymm1{k5}, ymm0, ymmword [rdx]"); // VPACKUSWB_YMMu8_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x67, 0xca], "vpackuswb zmm1{k5}{z}, zmm0, zmm2"); // VPACKUSWB_ZMMu8_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x67, 0x0a], "vpackuswb zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPACKUSWB_ZMMu8_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x67, 0xca], "vpackuswb zmm1, zmm0, zmm2"); // VPACKUSWB_ZMMu8_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x67, 0xca], "vpackuswb zmm1{k5}, zmm0, zmm2"); // VPACKUSWB_ZMMu8_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x67, 0x0a], "vpackuswb zmm1, zmm0, zmmword [rdx]"); // VPACKUSWB_ZMMu8_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x67, 0x0a], "vpackuswb zmm1{k5}, zmm0, zmmword [rdx]"); // VPACKUSWB_ZMMu8_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x67, 0xca], "vpackuswb xmm1{k5}{z}, xmm0, xmm2"); // VPACKUSWB_XMMu8_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x67, 0x0a], "vpackuswb xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPACKUSWB_XMMu8_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x67, 0xca], "vpackuswb xmm1, xmm0, xmm2"); // VPACKUSWB_XMMu8_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x67, 0xca], "vpackuswb xmm1{k5}, xmm0, xmm2"); // VPACKUSWB_XMMu8_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x67, 0x0a], "vpackuswb xmm1, xmm0, xmmword [rdx]"); // VPACKUSWB_XMMu8_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x67, 0x0a], "vpackuswb xmm1{k5}, xmm0, xmmword [rdx]"); // VPACKUSWB_XMMu8_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x68, 0xca], "vpunpckhbw ymm1{k5}{z}, ymm0, ymm2"); // VPUNPCKHBW_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x68, 0x0a], "vpunpckhbw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPUNPCKHBW_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x68, 0xca], "vpunpckhbw ymm1, ymm0, ymm2"); // VPUNPCKHBW_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x68, 0xca], "vpunpckhbw ymm1{k5}, ymm0, ymm2"); // VPUNPCKHBW_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x68, 0x0a], "vpunpckhbw ymm1, ymm0, ymmword [rdx]"); // VPUNPCKHBW_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x68, 0x0a], "vpunpckhbw ymm1{k5}, ymm0, ymmword [rdx]"); // VPUNPCKHBW_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x68, 0xca], "vpunpckhbw zmm1{k5}{z}, zmm0, zmm2"); // VPUNPCKHBW_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x68, 0x0a], "vpunpckhbw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPUNPCKHBW_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x68, 0xca], "vpunpckhbw zmm1, zmm0, zmm2"); // VPUNPCKHBW_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x68, 0xca], "vpunpckhbw zmm1{k5}, zmm0, zmm2"); // VPUNPCKHBW_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x68, 0x0a], "vpunpckhbw zmm1, zmm0, zmmword [rdx]"); // VPUNPCKHBW_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x68, 0x0a], "vpunpckhbw zmm1{k5}, zmm0, zmmword [rdx]"); // VPUNPCKHBW_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x68, 0xca], "vpunpckhbw xmm1{k5}{z}, xmm0, xmm2"); // VPUNPCKHBW_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x68, 0x0a], "vpunpckhbw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPUNPCKHBW_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x68, 0xca], "vpunpckhbw xmm1, xmm0, xmm2"); // VPUNPCKHBW_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x68, 0xca], "vpunpckhbw xmm1{k5}, xmm0, xmm2"); // VPUNPCKHBW_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x68, 0x0a], "vpunpckhbw xmm1, xmm0, xmmword [rdx]"); // VPUNPCKHBW_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x68, 0x0a], "vpunpckhbw xmm1{k5}, xmm0, xmmword [rdx]"); // VPUNPCKHBW_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x69, 0xca], "vpunpckhwd ymm1{k5}{z}, ymm0, ymm2"); // VPUNPCKHWD_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x69, 0x0a], "vpunpckhwd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPUNPCKHWD_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x69, 0xca], "vpunpckhwd ymm1, ymm0, ymm2"); // VPUNPCKHWD_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x69, 0xca], "vpunpckhwd ymm1{k5}, ymm0, ymm2"); // VPUNPCKHWD_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x69, 0x0a], "vpunpckhwd ymm1, ymm0, ymmword [rdx]"); // VPUNPCKHWD_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x69, 0x0a], "vpunpckhwd ymm1{k5}, ymm0, ymmword [rdx]"); // VPUNPCKHWD_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x69, 0xca], "vpunpckhwd zmm1{k5}{z}, zmm0, zmm2"); // VPUNPCKHWD_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x69, 0x0a], "vpunpckhwd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPUNPCKHWD_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x69, 0xca], "vpunpckhwd zmm1, zmm0, zmm2"); // VPUNPCKHWD_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x69, 0xca], "vpunpckhwd zmm1{k5}, zmm0, zmm2"); // VPUNPCKHWD_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x69, 0x0a], "vpunpckhwd zmm1, zmm0, zmmword [rdx]"); // VPUNPCKHWD_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x69, 0x0a], "vpunpckhwd zmm1{k5}, zmm0, zmmword [rdx]"); // VPUNPCKHWD_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x69, 0xca], "vpunpckhwd xmm1{k5}{z}, xmm0, xmm2"); // VPUNPCKHWD_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x69, 0x0a], "vpunpckhwd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPUNPCKHWD_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x69, 0xca], "vpunpckhwd xmm1, xmm0, xmm2"); // VPUNPCKHWD_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x69, 0xca], "vpunpckhwd xmm1{k5}, xmm0, xmm2"); // VPUNPCKHWD_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x69, 0x0a], "vpunpckhwd xmm1, xmm0, xmmword [rdx]"); // VPUNPCKHWD_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x69, 0x0a], "vpunpckhwd xmm1{k5}, xmm0, xmmword [rdx]"); // VPUNPCKHWD_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x6a, 0x0a], "vpunpckhdq ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPUNPCKHDQ_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x6a, 0x0a], "vpunpckhdq ymm1, ymm0, dword [rdx]{1to8}"); // VPUNPCKHDQ_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x6a, 0x0a], "vpunpckhdq ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPUNPCKHDQ_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x6a, 0xca], "vpunpckhdq ymm1{k5}{z}, ymm0, ymm2"); // VPUNPCKHDQ_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x6a, 0x0a], "vpunpckhdq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPUNPCKHDQ_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x6a, 0xca], "vpunpckhdq ymm1, ymm0, ymm2"); // VPUNPCKHDQ_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x6a, 0xca], "vpunpckhdq ymm1{k5}, ymm0, ymm2"); // VPUNPCKHDQ_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x6a, 0x0a], "vpunpckhdq ymm1, ymm0, ymmword [rdx]"); // VPUNPCKHDQ_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x6a, 0x0a], "vpunpckhdq ymm1{k5}, ymm0, ymmword [rdx]"); // VPUNPCKHDQ_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x6a, 0x0a], "vpunpckhdq zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPUNPCKHDQ_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x6a, 0x0a], "vpunpckhdq zmm1, zmm0, dword [rdx]{1to16}"); // VPUNPCKHDQ_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x6a, 0x0a], "vpunpckhdq zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPUNPCKHDQ_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x6a, 0x0a], "vpunpckhdq xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPUNPCKHDQ_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x6a, 0x0a], "vpunpckhdq xmm1, xmm0, dword [rdx]{1to4}"); // VPUNPCKHDQ_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x6a, 0x0a], "vpunpckhdq xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPUNPCKHDQ_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x6a, 0xca], "vpunpckhdq zmm1{k5}{z}, zmm0, zmm2"); // VPUNPCKHDQ_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x6a, 0x0a], "vpunpckhdq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPUNPCKHDQ_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x6a, 0xca], "vpunpckhdq zmm1, zmm0, zmm2"); // VPUNPCKHDQ_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x6a, 0xca], "vpunpckhdq zmm1{k5}, zmm0, zmm2"); // VPUNPCKHDQ_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x6a, 0x0a], "vpunpckhdq zmm1, zmm0, zmmword [rdx]"); // VPUNPCKHDQ_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x6a, 0x0a], "vpunpckhdq zmm1{k5}, zmm0, zmmword [rdx]"); // VPUNPCKHDQ_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x6a, 0xca], "vpunpckhdq xmm1{k5}{z}, xmm0, xmm2"); // VPUNPCKHDQ_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x6a, 0x0a], "vpunpckhdq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPUNPCKHDQ_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6a, 0xca], "vpunpckhdq xmm1, xmm0, xmm2"); // VPUNPCKHDQ_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x6a, 0xca], "vpunpckhdq xmm1{k5}, xmm0, xmm2"); // VPUNPCKHDQ_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6a, 0x0a], "vpunpckhdq xmm1, xmm0, xmmword [rdx]"); // VPUNPCKHDQ_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x6a, 0x0a], "vpunpckhdq xmm1{k5}, xmm0, xmmword [rdx]"); // VPUNPCKHDQ_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x6b, 0x0a], "vpackssdw ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPACKSSDW_YMMi16_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x6b, 0x0a], "vpackssdw ymm1, ymm0, dword [rdx]{1to8}"); // VPACKSSDW_YMMi16_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x6b, 0x0a], "vpackssdw ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPACKSSDW_YMMi16_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x6b, 0xca], "vpackssdw ymm1{k5}{z}, ymm0, ymm2"); // VPACKSSDW_YMMi16_MASKmskw_YMMi32_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x6b, 0x0a], "vpackssdw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPACKSSDW_YMMi16_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x6b, 0xca], "vpackssdw ymm1, ymm0, ymm2"); // VPACKSSDW_YMMi16_MASKmskw_YMMi32_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x6b, 0xca], "vpackssdw ymm1{k5}, ymm0, ymm2"); // VPACKSSDW_YMMi16_MASKmskw_YMMi32_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x6b, 0x0a], "vpackssdw ymm1, ymm0, ymmword [rdx]"); // VPACKSSDW_YMMi16_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x6b, 0x0a], "vpackssdw ymm1{k5}, ymm0, ymmword [rdx]"); // VPACKSSDW_YMMi16_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x6b, 0x0a], "vpackssdw zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPACKSSDW_ZMMi16_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x6b, 0x0a], "vpackssdw zmm1, zmm0, dword [rdx]{1to16}"); // VPACKSSDW_ZMMi16_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x6b, 0x0a], "vpackssdw zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPACKSSDW_ZMMi16_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x6b, 0x0a], "vpackssdw xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPACKSSDW_XMMi16_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x6b, 0x0a], "vpackssdw xmm1, xmm0, dword [rdx]{1to4}"); // VPACKSSDW_XMMi16_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x6b, 0x0a], "vpackssdw xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPACKSSDW_XMMi16_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x6b, 0xca], "vpackssdw zmm1{k5}{z}, zmm0, zmm2"); // VPACKSSDW_ZMMi16_MASKmskw_ZMMi32_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x6b, 0x0a], "vpackssdw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPACKSSDW_ZMMi16_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x6b, 0xca], "vpackssdw zmm1, zmm0, zmm2"); // VPACKSSDW_ZMMi16_MASKmskw_ZMMi32_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x6b, 0xca], "vpackssdw zmm1{k5}, zmm0, zmm2"); // VPACKSSDW_ZMMi16_MASKmskw_ZMMi32_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x6b, 0x0a], "vpackssdw zmm1, zmm0, zmmword [rdx]"); // VPACKSSDW_ZMMi16_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x6b, 0x0a], "vpackssdw zmm1{k5}, zmm0, zmmword [rdx]"); // VPACKSSDW_ZMMi16_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x6b, 0xca], "vpackssdw xmm1{k5}{z}, xmm0, xmm2"); // VPACKSSDW_XMMi16_MASKmskw_XMMi32_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x6b, 0x0a], "vpackssdw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPACKSSDW_XMMi16_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6b, 0xca], "vpackssdw xmm1, xmm0, xmm2"); // VPACKSSDW_XMMi16_MASKmskw_XMMi32_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x6b, 0xca], "vpackssdw xmm1{k5}, xmm0, xmm2"); // VPACKSSDW_XMMi16_MASKmskw_XMMi32_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6b, 0x0a], "vpackssdw xmm1, xmm0, xmmword [rdx]"); // VPACKSSDW_XMMi16_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x6b, 0x0a], "vpackssdw xmm1{k5}, xmm0, xmmword [rdx]"); // VPACKSSDW_XMMi16_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x6c, 0x0a], "vpunpcklqdq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPUNPCKLQDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x6c, 0x0a], "vpunpcklqdq ymm1, ymm0, qword [rdx]{1to4}"); // VPUNPCKLQDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x6c, 0x0a], "vpunpcklqdq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPUNPCKLQDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x6c, 0xca], "vpunpcklqdq ymm1{k5}{z}, ymm0, ymm2"); // VPUNPCKLQDQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x6c, 0x0a], "vpunpcklqdq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPUNPCKLQDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x6c, 0xca], "vpunpcklqdq ymm1, ymm0, ymm2"); // VPUNPCKLQDQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x6c, 0xca], "vpunpcklqdq ymm1{k5}, ymm0, ymm2"); // VPUNPCKLQDQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x6c, 0x0a], "vpunpcklqdq ymm1, ymm0, ymmword [rdx]"); // VPUNPCKLQDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x6c, 0x0a], "vpunpcklqdq ymm1{k5}, ymm0, ymmword [rdx]"); // VPUNPCKLQDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x6c, 0x0a], "vpunpcklqdq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPUNPCKLQDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x6c, 0x0a], "vpunpcklqdq zmm1, zmm0, qword [rdx]{1to8}"); // VPUNPCKLQDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x6c, 0x0a], "vpunpcklqdq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPUNPCKLQDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x6c, 0x0a], "vpunpcklqdq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPUNPCKLQDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x6c, 0x0a], "vpunpcklqdq xmm1, xmm0, qword [rdx]{1to2}"); // VPUNPCKLQDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x6c, 0x0a], "vpunpcklqdq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPUNPCKLQDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x6c, 0xca], "vpunpcklqdq zmm1{k5}{z}, zmm0, zmm2"); // VPUNPCKLQDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x6c, 0x0a], "vpunpcklqdq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPUNPCKLQDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x6c, 0xca], "vpunpcklqdq zmm1, zmm0, zmm2"); // VPUNPCKLQDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x6c, 0xca], "vpunpcklqdq zmm1{k5}, zmm0, zmm2"); // VPUNPCKLQDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x6c, 0x0a], "vpunpcklqdq zmm1, zmm0, zmmword [rdx]"); // VPUNPCKLQDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x6c, 0x0a], "vpunpcklqdq zmm1{k5}, zmm0, zmmword [rdx]"); // VPUNPCKLQDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x6c, 0xca], "vpunpcklqdq xmm1{k5}{z}, xmm0, xmm2"); // VPUNPCKLQDQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x6c, 0x0a], "vpunpcklqdq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPUNPCKLQDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6c, 0xca], "vpunpcklqdq xmm1, xmm0, xmm2"); // VPUNPCKLQDQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x6c, 0xca], "vpunpcklqdq xmm1{k5}, xmm0, xmm2"); // VPUNPCKLQDQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6c, 0x0a], "vpunpcklqdq xmm1, xmm0, xmmword [rdx]"); // VPUNPCKLQDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x6c, 0x0a], "vpunpcklqdq xmm1{k5}, xmm0, xmmword [rdx]"); // VPUNPCKLQDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x6d, 0x0a], "vpunpckhqdq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPUNPCKHQDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x6d, 0x0a], "vpunpckhqdq ymm1, ymm0, qword [rdx]{1to4}"); // VPUNPCKHQDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x6d, 0x0a], "vpunpckhqdq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPUNPCKHQDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x6d, 0xca], "vpunpckhqdq ymm1{k5}{z}, ymm0, ymm2"); // VPUNPCKHQDQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x6d, 0x0a], "vpunpckhqdq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPUNPCKHQDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x6d, 0xca], "vpunpckhqdq ymm1, ymm0, ymm2"); // VPUNPCKHQDQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x6d, 0xca], "vpunpckhqdq ymm1{k5}, ymm0, ymm2"); // VPUNPCKHQDQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x6d, 0x0a], "vpunpckhqdq ymm1, ymm0, ymmword [rdx]"); // VPUNPCKHQDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x6d, 0x0a], "vpunpckhqdq ymm1{k5}, ymm0, ymmword [rdx]"); // VPUNPCKHQDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x6d, 0x0a], "vpunpckhqdq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPUNPCKHQDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x6d, 0x0a], "vpunpckhqdq zmm1, zmm0, qword [rdx]{1to8}"); // VPUNPCKHQDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x6d, 0x0a], "vpunpckhqdq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPUNPCKHQDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x6d, 0x0a], "vpunpckhqdq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPUNPCKHQDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x6d, 0x0a], "vpunpckhqdq xmm1, xmm0, qword [rdx]{1to2}"); // VPUNPCKHQDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x6d, 0x0a], "vpunpckhqdq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPUNPCKHQDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x6d, 0xca], "vpunpckhqdq zmm1{k5}{z}, zmm0, zmm2"); // VPUNPCKHQDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x6d, 0x0a], "vpunpckhqdq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPUNPCKHQDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x6d, 0xca], "vpunpckhqdq zmm1, zmm0, zmm2"); // VPUNPCKHQDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x6d, 0xca], "vpunpckhqdq zmm1{k5}, zmm0, zmm2"); // VPUNPCKHQDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x6d, 0x0a], "vpunpckhqdq zmm1, zmm0, zmmword [rdx]"); // VPUNPCKHQDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x6d, 0x0a], "vpunpckhqdq zmm1{k5}, zmm0, zmmword [rdx]"); // VPUNPCKHQDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x6d, 0xca], "vpunpckhqdq xmm1{k5}{z}, xmm0, xmm2"); // VPUNPCKHQDQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x6d, 0x0a], "vpunpckhqdq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPUNPCKHQDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6d, 0xca], "vpunpckhqdq xmm1, xmm0, xmm2"); // VPUNPCKHQDQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x6d, 0xca], "vpunpckhqdq xmm1{k5}, xmm0, xmm2"); // VPUNPCKHQDQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6d, 0x0a], "vpunpckhqdq xmm1, xmm0, xmmword [rdx]"); // VPUNPCKHQDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x6d, 0x0a], "vpunpckhqdq xmm1{k5}, xmm0, xmmword [rdx]"); // VPUNPCKHQDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6e, 0xca], "vmovq xmm1, rdx"); // VMOVQ_XMMu64_GPR64u64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6e, 0x0a], "vmovq xmm1, qword [rdx]"); // VMOVQ_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6e, 0xca], "vmovd xmm1, edx"); // VMOVD_XMMu32_GPR32u32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6e, 0x0a], "vmovd xmm1, dword [rdx]"); // VMOVD_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x6f, 0xca], "vmovdqa64 ymm1{k5}{z}, ymm2"); // VMOVDQA64_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x6f, 0x0a], "vmovdqa64 ymm1{k5}{z}, ymmword [rdx]"); // VMOVDQA64_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x6f, 0xca], "vmovdqa64 ymm1, ymm2"); // VMOVDQA64_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x6f, 0xca], "vmovdqa64 ymm1{k5}, ymm2"); // VMOVDQA64_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x6f, 0x0a], "vmovdqa64 ymm1, ymmword [rdx]"); // VMOVDQA64_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x6f, 0x0a], "vmovdqa64 ymm1{k5}, ymmword [rdx]"); // VMOVDQA64_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x6f, 0xca], "vmovdqa32 ymm1{k5}{z}, ymm2"); // VMOVDQA32_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x6f, 0x0a], "vmovdqa32 ymm1{k5}{z}, ymmword [rdx]"); // VMOVDQA32_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x6f, 0xca], "vmovdqa32 ymm1, ymm2"); // VMOVDQA32_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x6f, 0xca], "vmovdqa32 ymm1{k5}, ymm2"); // VMOVDQA32_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x6f, 0x0a], "vmovdqa32 ymm1, ymmword [rdx]"); // VMOVDQA32_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x6f, 0x0a], "vmovdqa32 ymm1{k5}, ymmword [rdx]"); // VMOVDQA32_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x6f, 0xca], "vmovdqa64 zmm1{k5}{z}, zmm2"); // VMOVDQA64_ZMMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x6f, 0x0a], "vmovdqa64 zmm1{k5}{z}, zmmword [rdx]"); // VMOVDQA64_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x6f, 0xca], "vmovdqa64 zmm1, zmm2"); // VMOVDQA64_ZMMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x6f, 0xca], "vmovdqa64 zmm1{k5}, zmm2"); // VMOVDQA64_ZMMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x6f, 0x0a], "vmovdqa64 zmm1, zmmword [rdx]"); // VMOVDQA64_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x6f, 0x0a], "vmovdqa64 zmm1{k5}, zmmword [rdx]"); // VMOVDQA64_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x6f, 0xca], "vmovdqa64 xmm1{k5}{z}, xmm2"); // VMOVDQA64_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x6f, 0x0a], "vmovdqa64 xmm1{k5}{z}, xmmword [rdx]"); // VMOVDQA64_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6f, 0xca], "vmovdqa64 xmm1, xmm2"); // VMOVDQA64_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x6f, 0xca], "vmovdqa64 xmm1{k5}, xmm2"); // VMOVDQA64_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x6f, 0x0a], "vmovdqa64 xmm1, xmmword [rdx]"); // VMOVDQA64_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x6f, 0x0a], "vmovdqa64 xmm1{k5}, xmmword [rdx]"); // VMOVDQA64_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x6f, 0xca], "vmovdqa32 zmm1{k5}{z}, zmm2"); // VMOVDQA32_ZMMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x6f, 0x0a], "vmovdqa32 zmm1{k5}{z}, zmmword [rdx]"); // VMOVDQA32_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x6f, 0xca], "vmovdqa32 zmm1, zmm2"); // VMOVDQA32_ZMMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x6f, 0xca], "vmovdqa32 zmm1{k5}, zmm2"); // VMOVDQA32_ZMMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x6f, 0x0a], "vmovdqa32 zmm1, zmmword [rdx]"); // VMOVDQA32_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x6f, 0x0a], "vmovdqa32 zmm1{k5}, zmmword [rdx]"); // VMOVDQA32_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x6f, 0xca], "vmovdqa32 xmm1{k5}{z}, xmm2"); // VMOVDQA32_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x6f, 0x0a], "vmovdqa32 xmm1{k5}{z}, xmmword [rdx]"); // VMOVDQA32_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6f, 0xca], "vmovdqa32 xmm1, xmm2"); // VMOVDQA32_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x6f, 0xca], "vmovdqa32 xmm1{k5}, xmm2"); // VMOVDQA32_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x6f, 0x0a], "vmovdqa32 xmm1, xmmword [rdx]"); // VMOVDQA32_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x6f, 0x0a], "vmovdqa32 xmm1{k5}, xmmword [rdx]"); // VMOVDQA32_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x70, 0x0a, 0xcc], "vpshufd ymm1{k5}{z}, dword [rdx]{1to8}, 0xcc"); // VPSHUFD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x70, 0x0a, 0xcc], "vpshufd ymm1, dword [rdx]{1to8}, 0xcc"); // VPSHUFD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x70, 0x0a, 0xcc], "vpshufd ymm1{k5}, dword [rdx]{1to8}, 0xcc"); // VPSHUFD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x70, 0xca, 0xcc], "vpshufd ymm1{k5}{z}, ymm2, 0xcc"); // VPSHUFD_YMMu32_MASKmskw_YMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x70, 0x0a, 0xcc], "vpshufd ymm1{k5}{z}, ymmword [rdx], 0xcc"); // VPSHUFD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x70, 0xca, 0xcc], "vpshufd ymm1, ymm2, 0xcc"); // VPSHUFD_YMMu32_MASKmskw_YMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x70, 0xca, 0xcc], "vpshufd ymm1{k5}, ymm2, 0xcc"); // VPSHUFD_YMMu32_MASKmskw_YMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x70, 0x0a, 0xcc], "vpshufd ymm1, ymmword [rdx], 0xcc"); // VPSHUFD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x70, 0x0a, 0xcc], "vpshufd ymm1{k5}, ymmword [rdx], 0xcc"); // VPSHUFD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x70, 0x0a, 0xcc], "vpshufd zmm1{k5}{z}, dword [rdx]{1to16}, 0xcc"); // VPSHUFD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x70, 0x0a, 0xcc], "vpshufd zmm1, dword [rdx]{1to16}, 0xcc"); // VPSHUFD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x70, 0x0a, 0xcc], "vpshufd zmm1{k5}, dword [rdx]{1to16}, 0xcc"); // VPSHUFD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x70, 0x0a, 0xcc], "vpshufd xmm1{k5}{z}, dword [rdx]{1to4}, 0xcc"); // VPSHUFD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x70, 0x0a, 0xcc], "vpshufd xmm1, dword [rdx]{1to4}, 0xcc"); // VPSHUFD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x70, 0x0a, 0xcc], "vpshufd xmm1{k5}, dword [rdx]{1to4}, 0xcc"); // VPSHUFD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x70, 0xca, 0xcc], "vpshufd zmm1{k5}{z}, zmm2, 0xcc"); // VPSHUFD_ZMMu32_MASKmskw_ZMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x70, 0x0a, 0xcc], "vpshufd zmm1{k5}{z}, zmmword [rdx], 0xcc"); // VPSHUFD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x70, 0xca, 0xcc], "vpshufd zmm1, zmm2, 0xcc"); // VPSHUFD_ZMMu32_MASKmskw_ZMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x70, 0xca, 0xcc], "vpshufd zmm1{k5}, zmm2, 0xcc"); // VPSHUFD_ZMMu32_MASKmskw_ZMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x70, 0x0a, 0xcc], "vpshufd zmm1, zmmword [rdx], 0xcc"); // VPSHUFD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x70, 0x0a, 0xcc], "vpshufd zmm1{k5}, zmmword [rdx], 0xcc"); // VPSHUFD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x70, 0xca, 0xcc], "vpshufd xmm1{k5}{z}, xmm2, 0xcc"); // VPSHUFD_XMMu32_MASKmskw_XMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x70, 0x0a, 0xcc], "vpshufd xmm1{k5}{z}, xmmword [rdx], 0xcc"); // VPSHUFD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x70, 0xca, 0xcc], "vpshufd xmm1, xmm2, 0xcc"); // VPSHUFD_XMMu32_MASKmskw_XMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x70, 0xca, 0xcc], "vpshufd xmm1{k5}, xmm2, 0xcc"); // VPSHUFD_XMMu32_MASKmskw_XMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x70, 0x0a, 0xcc], "vpshufd xmm1, xmmword [rdx], 0xcc"); // VPSHUFD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x70, 0x0a, 0xcc], "vpshufd xmm1{k5}, xmmword [rdx], 0xcc"); // VPSHUFD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    // TODO: 72 (and 71) are selected by /r. these need more exhaustive (hmmm) testing.
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x72, 0x0a, 0xcc], "vprolq ymm0{k5}{z}, qword [rdx]{1to4}, 0xcc"); // VPROLQ_YMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x72, 0x0a, 0xcc], "vprolq ymm0, qword [rdx]{1to4}, 0xcc"); // VPROLQ_YMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x72, 0x0a, 0xcc], "vprolq ymm0{k5}, qword [rdx]{1to4}, 0xcc"); // VPROLQ_YMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x72, 0xca, 0xcc], "vprolq ymm0{k5}{z}, ymm2, 0xcc"); // VPROLQ_YMMu64_MASKmskw_YMMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x72, 0x0a, 0xcc], "vprolq ymm0{k5}{z}, ymmword [rdx], 0xcc"); // VPROLQ_YMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x72, 0xca, 0xcc], "vprolq ymm0, ymm2, 0xcc"); // VPROLQ_YMMu64_MASKmskw_YMMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x72, 0xca, 0xcc], "vprolq ymm0{k5}, ymm2, 0xcc"); // VPROLQ_YMMu64_MASKmskw_YMMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x72, 0x0a, 0xcc], "vprolq ymm0, ymmword [rdx], 0xcc"); // VPROLQ_YMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x72, 0x0a, 0xcc], "vprolq ymm0{k5}, ymmword [rdx], 0xcc"); // VPROLQ_YMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x72, 0x0a, 0xcc], "vprold ymm0{k5}{z}, dword [rdx]{1to8}, 0xcc"); // VPROLD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x72, 0x0a, 0xcc], "vprold ymm0, dword [rdx]{1to8}, 0xcc"); // VPROLD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x72, 0x0a, 0xcc], "vprold ymm0{k5}, dword [rdx]{1to8}, 0xcc"); // VPROLD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x72, 0xca, 0xcc], "vprold ymm0{k5}{z}, ymm2, 0xcc"); // VPROLD_YMMu32_MASKmskw_YMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x72, 0x0a, 0xcc], "vprold ymm0{k5}{z}, ymmword [rdx], 0xcc"); // VPROLD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x72, 0xca, 0xcc], "vprold ymm0, ymm2, 0xcc"); // VPROLD_YMMu32_MASKmskw_YMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x72, 0xca, 0xcc], "vprold ymm0{k5}, ymm2, 0xcc"); // VPROLD_YMMu32_MASKmskw_YMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x72, 0x0a, 0xcc], "vprold ymm0, ymmword [rdx], 0xcc"); // VPROLD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x72, 0x0a, 0xcc], "vprold ymm0{k5}, ymmword [rdx], 0xcc"); // VPROLD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x72, 0x0a, 0xcc], "vprolq zmm0{k5}{z}, qword [rdx]{1to8}, 0xcc"); // VPROLQ_ZMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x72, 0x0a, 0xcc], "vprolq zmm0, qword [rdx]{1to8}, 0xcc"); // VPROLQ_ZMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x72, 0x0a, 0xcc], "vprolq zmm0{k5}, qword [rdx]{1to8}, 0xcc"); // VPROLQ_ZMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x72, 0x0a, 0xcc], "vprolq xmm0{k5}{z}, qword [rdx]{1to2}, 0xcc"); // VPROLQ_XMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x72, 0x0a, 0xcc], "vprolq xmm0, qword [rdx]{1to2}, 0xcc"); // VPROLQ_XMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x72, 0x0a, 0xcc], "vprolq xmm0{k5}, qword [rdx]{1to2}, 0xcc"); // VPROLQ_XMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x72, 0xca, 0xcc], "vprolq zmm0{k5}{z}, zmm2, 0xcc"); // VPROLQ_ZMMu64_MASKmskw_ZMMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x72, 0x0a, 0xcc], "vprolq zmm0{k5}{z}, zmmword [rdx], 0xcc"); // VPROLQ_ZMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x72, 0xca, 0xcc], "vprolq zmm0, zmm2, 0xcc"); // VPROLQ_ZMMu64_MASKmskw_ZMMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x72, 0xca, 0xcc], "vprolq zmm0{k5}, zmm2, 0xcc"); // VPROLQ_ZMMu64_MASKmskw_ZMMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x72, 0x0a, 0xcc], "vprolq zmm0, zmmword [rdx], 0xcc"); // VPROLQ_ZMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x72, 0x0a, 0xcc], "vprolq zmm0{k5}, zmmword [rdx], 0xcc"); // VPROLQ_ZMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x72, 0xca, 0xcc], "vprolq xmm0{k5}{z}, xmm2, 0xcc"); // VPROLQ_XMMu64_MASKmskw_XMMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x72, 0x0a, 0xcc], "vprolq xmm0{k5}{z}, xmmword [rdx], 0xcc"); // VPROLQ_XMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x72, 0xca, 0xcc], "vprolq xmm0, xmm2, 0xcc"); // VPROLQ_XMMu64_MASKmskw_XMMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x72, 0xca, 0xcc], "vprolq xmm0{k5}, xmm2, 0xcc"); // VPROLQ_XMMu64_MASKmskw_XMMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x72, 0x0a, 0xcc], "vprolq xmm0, xmmword [rdx], 0xcc"); // VPROLQ_XMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x72, 0x0a, 0xcc], "vprolq xmm0{k5}, xmmword [rdx], 0xcc"); // VPROLQ_XMMu64_MASKmskw_MEMu64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x72, 0x0a, 0xcc], "vprold zmm0{k5}{z}, dword [rdx]{1to16}, 0xcc"); // VPROLD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x72, 0x0a, 0xcc], "vprold zmm0, dword [rdx]{1to16}, 0xcc"); // VPROLD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x72, 0x0a, 0xcc], "vprold zmm0{k5}, dword [rdx]{1to16}, 0xcc"); // VPROLD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x72, 0x0a, 0xcc], "vprold xmm0{k5}{z}, dword [rdx]{1to4}, 0xcc"); // VPROLD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x72, 0x0a, 0xcc], "vprold xmm0, dword [rdx]{1to4}, 0xcc"); // VPROLD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x72, 0x0a, 0xcc], "vprold xmm0{k5}, dword [rdx]{1to4}, 0xcc"); // VPROLD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x72, 0xca, 0xcc], "vprold zmm0{k5}{z}, zmm2, 0xcc"); // VPROLD_ZMMu32_MASKmskw_ZMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x72, 0x0a, 0xcc], "vprold zmm0{k5}{z}, zmmword [rdx], 0xcc"); // VPROLD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x72, 0xca, 0xcc], "vprold zmm0, zmm2, 0xcc"); // VPROLD_ZMMu32_MASKmskw_ZMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x72, 0xca, 0xcc], "vprold zmm0{k5}, zmm2, 0xcc"); // VPROLD_ZMMu32_MASKmskw_ZMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x72, 0x0a, 0xcc], "vprold zmm0, zmmword [rdx], 0xcc"); // VPROLD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x72, 0x0a, 0xcc], "vprold zmm0{k5}, zmmword [rdx], 0xcc"); // VPROLD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x72, 0xca, 0xcc], "vprold xmm0{k5}{z}, xmm2, 0xcc"); // VPROLD_XMMu32_MASKmskw_XMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x72, 0x0a, 0xcc], "vprold xmm0{k5}{z}, xmmword [rdx], 0xcc"); // VPROLD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x72, 0xca, 0xcc], "vprold xmm0, xmm2, 0xcc"); // VPROLD_XMMu32_MASKmskw_XMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x72, 0xca, 0xcc], "vprold xmm0{k5}, xmm2, 0xcc"); // VPROLD_XMMu32_MASKmskw_XMMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x72, 0x0a, 0xcc], "vprold xmm0, xmmword [rdx], 0xcc"); // VPROLD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x72, 0x0a, 0xcc], "vprold xmm0{k5}, xmmword [rdx], 0xcc"); // VPROLD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x74, 0xca], "vpcmpeqb k1, ymm0, ymm2"); // VPCMPEQB_MASKmskw_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x74, 0xca], "vpcmpeqb k1{k5}, ymm0, ymm2"); // VPCMPEQB_MASKmskw_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x74, 0x0a], "vpcmpeqb k1, ymm0, ymmword [rdx]"); // VPCMPEQB_MASKmskw_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x74, 0x0a], "vpcmpeqb k1{k5}, ymm0, ymmword [rdx]"); // VPCMPEQB_MASKmskw_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x74, 0xca], "vpcmpeqb k1, zmm0, zmm2"); // VPCMPEQB_MASKmskw_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x74, 0xca], "vpcmpeqb k1{k5}, zmm0, zmm2"); // VPCMPEQB_MASKmskw_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x74, 0x0a], "vpcmpeqb k1, zmm0, zmmword [rdx]"); // VPCMPEQB_MASKmskw_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x74, 0x0a], "vpcmpeqb k1{k5}, zmm0, zmmword [rdx]"); // VPCMPEQB_MASKmskw_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x74, 0xca], "vpcmpeqb k1, xmm0, xmm2"); // VPCMPEQB_MASKmskw_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x74, 0xca], "vpcmpeqb k1{k5}, xmm0, xmm2"); // VPCMPEQB_MASKmskw_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x74, 0x0a], "vpcmpeqb k1, xmm0, xmmword [rdx]"); // VPCMPEQB_MASKmskw_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x74, 0x0a], "vpcmpeqb k1{k5}, xmm0, xmmword [rdx]"); // VPCMPEQB_MASKmskw_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x75, 0xca], "vpcmpeqw k1, ymm0, ymm2"); // VPCMPEQW_MASKmskw_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x75, 0xca], "vpcmpeqw k1{k5}, ymm0, ymm2"); // VPCMPEQW_MASKmskw_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x75, 0x0a], "vpcmpeqw k1, ymm0, ymmword [rdx]"); // VPCMPEQW_MASKmskw_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x75, 0x0a], "vpcmpeqw k1{k5}, ymm0, ymmword [rdx]"); // VPCMPEQW_MASKmskw_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x75, 0xca], "vpcmpeqw k1, zmm0, zmm2"); // VPCMPEQW_MASKmskw_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x75, 0xca], "vpcmpeqw k1{k5}, zmm0, zmm2"); // VPCMPEQW_MASKmskw_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x75, 0x0a], "vpcmpeqw k1, zmm0, zmmword [rdx]"); // VPCMPEQW_MASKmskw_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x75, 0x0a], "vpcmpeqw k1{k5}, zmm0, zmmword [rdx]"); // VPCMPEQW_MASKmskw_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x75, 0xca], "vpcmpeqw k1, xmm0, xmm2"); // VPCMPEQW_MASKmskw_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x75, 0xca], "vpcmpeqw k1{k5}, xmm0, xmm2"); // VPCMPEQW_MASKmskw_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x75, 0x0a], "vpcmpeqw k1, xmm0, xmmword [rdx]"); // VPCMPEQW_MASKmskw_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x75, 0x0a], "vpcmpeqw k1{k5}, xmm0, xmmword [rdx]"); // VPCMPEQW_MASKmskw_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x76, 0x0a], "vpcmpeqd k1, ymm0, dword [rdx]{1to8}"); // VPCMPEQD_MASKmskw_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x76, 0x0a], "vpcmpeqd k1{k5}, ymm0, dword [rdx]{1to8}"); // VPCMPEQD_MASKmskw_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x76, 0xca], "vpcmpeqd k1, ymm0, ymm2"); // VPCMPEQD_MASKmskw_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x76, 0xca], "vpcmpeqd k1{k5}, ymm0, ymm2"); // VPCMPEQD_MASKmskw_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x76, 0x0a], "vpcmpeqd k1, ymm0, ymmword [rdx]"); // VPCMPEQD_MASKmskw_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x76, 0x0a], "vpcmpeqd k1{k5}, ymm0, ymmword [rdx]"); // VPCMPEQD_MASKmskw_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x76, 0x0a], "vpcmpeqd k1, zmm0, dword [rdx]{1to16}"); // VPCMPEQD_MASKmskw_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x76, 0x0a], "vpcmpeqd k1{k5}, zmm0, dword [rdx]{1to16}"); // VPCMPEQD_MASKmskw_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x76, 0x0a], "vpcmpeqd k1, xmm0, dword [rdx]{1to4}"); // VPCMPEQD_MASKmskw_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x76, 0x0a], "vpcmpeqd k1{k5}, xmm0, dword [rdx]{1to4}"); // VPCMPEQD_MASKmskw_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x76, 0xca], "vpcmpeqd k1, zmm0, zmm2"); // VPCMPEQD_MASKmskw_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x76, 0xca], "vpcmpeqd k1{k5}, zmm0, zmm2"); // VPCMPEQD_MASKmskw_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x76, 0x0a], "vpcmpeqd k1, zmm0, zmmword [rdx]"); // VPCMPEQD_MASKmskw_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x76, 0x0a], "vpcmpeqd k1{k5}, zmm0, zmmword [rdx]"); // VPCMPEQD_MASKmskw_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x76, 0xca], "vpcmpeqd k1, xmm0, xmm2"); // VPCMPEQD_MASKmskw_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x76, 0xca], "vpcmpeqd k1{k5}, xmm0, xmm2"); // VPCMPEQD_MASKmskw_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x76, 0x0a], "vpcmpeqd k1, xmm0, xmmword [rdx]"); // VPCMPEQD_MASKmskw_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x76, 0x0a], "vpcmpeqd k1{k5}, xmm0, xmmword [rdx]"); // VPCMPEQD_MASKmskw_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x78, 0xca], "vcvttpd2uqq zmm1{k5}{z}{sae}, zmm2"); // VCVTTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x78, 0xca], "vcvttpd2uqq zmm1{sae}, zmm2"); // VCVTTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x78, 0xca], "vcvttpd2uqq zmm1{k5}{sae}, zmm2"); // VCVTTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x78, 0x0a], "vcvttpd2uqq ymm1{k5}{z}, qword [rdx]{1to4}"); // VCVTTPD2UQQ_YMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x78, 0x0a], "vcvttpd2uqq ymm1, qword [rdx]{1to4}"); // VCVTTPD2UQQ_YMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x78, 0x0a], "vcvttpd2uqq ymm1{k5}, qword [rdx]{1to4}"); // VCVTTPD2UQQ_YMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x78, 0xca], "vcvttpd2uqq ymm1{k5}{z}, ymm2"); // VCVTTPD2UQQ_YMMu64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x78, 0x0a], "vcvttpd2uqq ymm1{k5}{z}, ymmword [rdx]"); // VCVTTPD2UQQ_YMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x78, 0xca], "vcvttpd2uqq ymm1, ymm2"); // VCVTTPD2UQQ_YMMu64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x78, 0xca], "vcvttpd2uqq ymm1{k5}, ymm2"); // VCVTTPD2UQQ_YMMu64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x78, 0x0a], "vcvttpd2uqq ymm1, ymmword [rdx]"); // VCVTTPD2UQQ_YMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x78, 0x0a], "vcvttpd2uqq ymm1{k5}, ymmword [rdx]"); // VCVTTPD2UQQ_YMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xfd, 0x78, 0xca], "vcvttps2uqq zmm1{k5}{z}{sae}, ymm2"); // VCVTTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x78, 0x78, 0xca], "vcvttps2uqq zmm1{sae}, ymm2"); // VCVTTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x7d, 0x78, 0xca], "vcvttps2uqq zmm1{k5}{sae}, ymm2"); // VCVTTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x78, 0x0a], "vcvttps2uqq ymm1{k5}{z}, dword [rdx]{1to4}"); // VCVTTPS2UQQ_YMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x78, 0x0a], "vcvttps2uqq ymm1, dword [rdx]{1to4}"); // VCVTTPS2UQQ_YMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x78, 0x0a], "vcvttps2uqq ymm1{k5}, dword [rdx]{1to4}"); // VCVTTPS2UQQ_YMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x78, 0xca], "vcvttps2uqq ymm1{k5}{z}, xmm2"); // VCVTTPS2UQQ_YMMu64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x78, 0x0a], "vcvttps2uqq ymm1{k5}{z}, xmmword [rdx]"); // VCVTTPS2UQQ_YMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x78, 0xca], "vcvttps2uqq ymm1, xmm2"); // VCVTTPS2UQQ_YMMu64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x78, 0xca], "vcvttps2uqq ymm1{k5}, xmm2"); // VCVTTPS2UQQ_YMMu64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x78, 0x0a], "vcvttps2uqq ymm1, xmmword [rdx]"); // VCVTTPS2UQQ_YMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x78, 0x0a], "vcvttps2uqq ymm1{k5}, xmmword [rdx]"); // VCVTTPS2UQQ_YMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x78, 0x0a], "vcvttpd2uqq zmm1{k5}{z}, qword [rdx]{1to8}"); // VCVTTPD2UQQ_ZMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x78, 0x0a], "vcvttpd2uqq zmm1, qword [rdx]{1to8}"); // VCVTTPD2UQQ_ZMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x78, 0x0a], "vcvttpd2uqq zmm1{k5}, qword [rdx]{1to8}"); // VCVTTPD2UQQ_ZMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x78, 0x0a], "vcvttpd2uqq xmm1{k5}{z}, qword [rdx]{1to2}"); // VCVTTPD2UQQ_XMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x78, 0x0a], "vcvttpd2uqq xmm1, qword [rdx]{1to2}"); // VCVTTPD2UQQ_XMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x78, 0x0a], "vcvttpd2uqq xmm1{k5}, qword [rdx]{1to2}"); // VCVTTPD2UQQ_XMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x78, 0xca], "vcvttpd2uqq zmm1{k5}{z}, zmm2"); // VCVTTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x78, 0x0a], "vcvttpd2uqq zmm1{k5}{z}, zmmword [rdx]"); // VCVTTPD2UQQ_ZMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x78, 0xca], "vcvttpd2uqq zmm1, zmm2"); // VCVTTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x78, 0xca], "vcvttpd2uqq zmm1{k5}, zmm2"); // VCVTTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x78, 0x0a], "vcvttpd2uqq zmm1, zmmword [rdx]"); // VCVTTPD2UQQ_ZMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x78, 0x0a], "vcvttpd2uqq zmm1{k5}, zmmword [rdx]"); // VCVTTPD2UQQ_ZMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x78, 0xca], "vcvttpd2uqq xmm1{k5}{z}, xmm2"); // VCVTTPD2UQQ_XMMu64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x78, 0x0a], "vcvttpd2uqq xmm1{k5}{z}, xmmword [rdx]"); // VCVTTPD2UQQ_XMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x78, 0xca], "vcvttpd2uqq xmm1, xmm2"); // VCVTTPD2UQQ_XMMu64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x78, 0xca], "vcvttpd2uqq xmm1{k5}, xmm2"); // VCVTTPD2UQQ_XMMu64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x78, 0x0a], "vcvttpd2uqq xmm1, xmmword [rdx]"); // VCVTTPD2UQQ_XMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x78, 0x0a], "vcvttpd2uqq xmm1{k5}, xmmword [rdx]"); // VCVTTPD2UQQ_XMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x78, 0x0a], "vcvttps2uqq zmm1{k5}{z}, dword [rdx]{1to8}"); // VCVTTPS2UQQ_ZMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x78, 0x0a], "vcvttps2uqq zmm1, dword [rdx]{1to8}"); // VCVTTPS2UQQ_ZMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x78, 0x0a], "vcvttps2uqq zmm1{k5}, dword [rdx]{1to8}"); // VCVTTPS2UQQ_ZMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x78, 0x0a], "vcvttps2uqq xmm1{k5}{z}, dword [rdx]{1to2}"); // VCVTTPS2UQQ_XMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x78, 0x0a], "vcvttps2uqq xmm1, dword [rdx]{1to2}"); // VCVTTPS2UQQ_XMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x78, 0x0a], "vcvttps2uqq xmm1{k5}, dword [rdx]{1to2}"); // VCVTTPS2UQQ_XMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x78, 0xca], "vcvttps2uqq zmm1{k5}{z}, ymm2"); // VCVTTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x78, 0x0a], "vcvttps2uqq zmm1{k5}{z}, ymmword [rdx]"); // VCVTTPS2UQQ_ZMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x78, 0xca], "vcvttps2uqq zmm1, ymm2"); // VCVTTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x78, 0xca], "vcvttps2uqq zmm1{k5}, ymm2"); // VCVTTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x78, 0x0a], "vcvttps2uqq zmm1, ymmword [rdx]"); // VCVTTPS2UQQ_ZMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x78, 0x0a], "vcvttps2uqq zmm1{k5}, ymmword [rdx]"); // VCVTTPS2UQQ_ZMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x78, 0xca], "vcvttps2uqq xmm1{k5}{z}, xmm2"); // VCVTTPS2UQQ_XMMu64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x78, 0x0a], "vcvttps2uqq xmm1{k5}{z}, qword [rdx]"); // VCVTTPS2UQQ_XMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x78, 0xca], "vcvttps2uqq xmm1, xmm2"); // VCVTTPS2UQQ_XMMu64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x78, 0xca], "vcvttps2uqq xmm1{k5}, xmm2"); // VCVTTPS2UQQ_XMMu64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x78, 0x0a], "vcvttps2uqq xmm1, qword [rdx]"); // VCVTTPS2UQQ_XMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x78, 0x0a], "vcvttps2uqq xmm1{k5}, qword [rdx]"); // VCVTTPS2UQQ_XMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{z}{rz-sae}, zmm2"); // VCVTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x79, 0xca], "vcvtpd2uqq zmm1{rz-sae}, zmm2"); // VCVTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{rz-sae}, zmm2"); // VCVTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{z}{rd-sae}, zmm2"); // VCVTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x79, 0x0a], "vcvtpd2uqq ymm1{k5}{z}, qword [rdx]{1to4}"); // VCVTPD2UQQ_YMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x79, 0xca], "vcvtpd2uqq zmm1{rd-sae}, zmm2"); // VCVTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{rd-sae}, zmm2"); // VCVTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x79, 0x0a], "vcvtpd2uqq ymm1, qword [rdx]{1to4}"); // VCVTPD2UQQ_YMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x79, 0x0a], "vcvtpd2uqq ymm1{k5}, qword [rdx]{1to4}"); // VCVTPD2UQQ_YMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x79, 0xca], "vcvtpd2uqq ymm1{k5}{z}, ymm2"); // VCVTPD2UQQ_YMMu64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x79, 0x0a], "vcvtpd2uqq ymm1{k5}{z}, ymmword [rdx]"); // VCVTPD2UQQ_YMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x79, 0xca], "vcvtpd2uqq ymm1, ymm2"); // VCVTPD2UQQ_YMMu64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x79, 0xca], "vcvtpd2uqq ymm1{k5}, ymm2"); // VCVTPD2UQQ_YMMu64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x79, 0x0a], "vcvtpd2uqq ymm1, ymmword [rdx]"); // VCVTPD2UQQ_YMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x79, 0x0a], "vcvtpd2uqq ymm1{k5}, ymmword [rdx]"); // VCVTPD2UQQ_YMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xfd, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{z}{rz-sae}, ymm2"); // VCVTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x78, 0x79, 0xca], "vcvtps2uqq zmm1{rz-sae}, ymm2"); // VCVTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x7d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{rz-sae}, ymm2"); // VCVTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{z}{rd-sae}, ymm2"); // VCVTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x79, 0x0a], "vcvtps2uqq ymm1{k5}{z}, dword [rdx]{1to4}"); // VCVTPS2UQQ_YMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x79, 0xca], "vcvtps2uqq zmm1{rd-sae}, ymm2"); // VCVTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{rd-sae}, ymm2"); // VCVTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x79, 0x0a], "vcvtps2uqq ymm1, dword [rdx]{1to4}"); // VCVTPS2UQQ_YMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x79, 0x0a], "vcvtps2uqq ymm1{k5}, dword [rdx]{1to4}"); // VCVTPS2UQQ_YMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x79, 0xca], "vcvtps2uqq ymm1{k5}{z}, xmm2"); // VCVTPS2UQQ_YMMu64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x79, 0x0a], "vcvtps2uqq ymm1{k5}{z}, xmmword [rdx]"); // VCVTPS2UQQ_YMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x79, 0xca], "vcvtps2uqq ymm1, xmm2"); // VCVTPS2UQQ_YMMu64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x79, 0xca], "vcvtps2uqq ymm1{k5}, xmm2"); // VCVTPS2UQQ_YMMu64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x79, 0x0a], "vcvtps2uqq ymm1, xmmword [rdx]"); // VCVTPS2UQQ_YMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x79, 0x0a], "vcvtps2uqq ymm1{k5}, xmmword [rdx]"); // VCVTPS2UQQ_YMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{z}{ru-sae}, zmm2"); // VCVTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x79, 0x0a], "vcvtpd2uqq zmm1{k5}{z}, qword [rdx]{1to8}"); // VCVTPD2UQQ_ZMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x79, 0xca], "vcvtpd2uqq zmm1{ru-sae}, zmm2"); // VCVTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{ru-sae}, zmm2"); // VCVTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x79, 0x0a], "vcvtpd2uqq zmm1, qword [rdx]{1to8}"); // VCVTPD2UQQ_ZMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x79, 0x0a], "vcvtpd2uqq zmm1{k5}, qword [rdx]{1to8}"); // VCVTPD2UQQ_ZMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{z}{rne-sae}, zmm2"); // VCVTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x79, 0x0a], "vcvtpd2uqq xmm1{k5}{z}, qword [rdx]{1to2}"); // VCVTPD2UQQ_XMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x79, 0xca], "vcvtpd2uqq zmm1{rne-sae}, zmm2"); // VCVTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{rne-sae}, zmm2"); // VCVTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x79, 0x0a], "vcvtpd2uqq xmm1, qword [rdx]{1to2}"); // VCVTPD2UQQ_XMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x79, 0x0a], "vcvtpd2uqq xmm1{k5}, qword [rdx]{1to2}"); // VCVTPD2UQQ_XMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}{z}, zmm2"); // VCVTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x79, 0x0a], "vcvtpd2uqq zmm1{k5}{z}, zmmword [rdx]"); // VCVTPD2UQQ_ZMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x79, 0xca], "vcvtpd2uqq zmm1, zmm2"); // VCVTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x79, 0xca], "vcvtpd2uqq zmm1{k5}, zmm2"); // VCVTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x79, 0x0a], "vcvtpd2uqq zmm1, zmmword [rdx]"); // VCVTPD2UQQ_ZMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x79, 0x0a], "vcvtpd2uqq zmm1{k5}, zmmword [rdx]"); // VCVTPD2UQQ_ZMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x79, 0xca], "vcvtpd2uqq xmm1{k5}{z}, xmm2"); // VCVTPD2UQQ_XMMu64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x79, 0x0a], "vcvtpd2uqq xmm1{k5}{z}, xmmword [rdx]"); // VCVTPD2UQQ_XMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x79, 0xca], "vcvtpd2uqq xmm1, xmm2"); // VCVTPD2UQQ_XMMu64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x79, 0xca], "vcvtpd2uqq xmm1{k5}, xmm2"); // VCVTPD2UQQ_XMMu64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x79, 0x0a], "vcvtpd2uqq xmm1, xmmword [rdx]"); // VCVTPD2UQQ_XMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x79, 0x0a], "vcvtpd2uqq xmm1{k5}, xmmword [rdx]"); // VCVTPD2UQQ_XMMu64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{z}{ru-sae}, ymm2"); // VCVTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x79, 0x0a], "vcvtps2uqq zmm1{k5}{z}, dword [rdx]{1to8}"); // VCVTPS2UQQ_ZMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x79, 0xca], "vcvtps2uqq zmm1{ru-sae}, ymm2"); // VCVTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{ru-sae}, ymm2"); // VCVTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x79, 0x0a], "vcvtps2uqq zmm1, dword [rdx]{1to8}"); // VCVTPS2UQQ_ZMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x79, 0x0a], "vcvtps2uqq zmm1{k5}, dword [rdx]{1to8}"); // VCVTPS2UQQ_ZMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{z}{rne-sae}, ymm2"); // VCVTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x79, 0x0a], "vcvtps2uqq xmm1{k5}{z}, dword [rdx]{1to2}"); // VCVTPS2UQQ_XMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x79, 0xca], "vcvtps2uqq zmm1{rne-sae}, ymm2"); // VCVTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{rne-sae}, ymm2"); // VCVTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x79, 0x0a], "vcvtps2uqq xmm1, dword [rdx]{1to2}"); // VCVTPS2UQQ_XMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x79, 0x0a], "vcvtps2uqq xmm1{k5}, dword [rdx]{1to2}"); // VCVTPS2UQQ_XMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x79, 0xca], "vcvtps2uqq zmm1{k5}{z}, ymm2"); // VCVTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x79, 0x0a], "vcvtps2uqq zmm1{k5}{z}, ymmword [rdx]"); // VCVTPS2UQQ_ZMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x79, 0xca], "vcvtps2uqq zmm1, ymm2"); // VCVTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x79, 0xca], "vcvtps2uqq zmm1{k5}, ymm2"); // VCVTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x79, 0x0a], "vcvtps2uqq zmm1, ymmword [rdx]"); // VCVTPS2UQQ_ZMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x79, 0x0a], "vcvtps2uqq zmm1{k5}, ymmword [rdx]"); // VCVTPS2UQQ_ZMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x79, 0xca], "vcvtps2uqq xmm1{k5}{z}, xmm2"); // VCVTPS2UQQ_XMMu64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x79, 0x0a], "vcvtps2uqq xmm1{k5}{z}, qword [rdx]"); // VCVTPS2UQQ_XMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x79, 0xca], "vcvtps2uqq xmm1, xmm2"); // VCVTPS2UQQ_XMMu64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x79, 0xca], "vcvtps2uqq xmm1{k5}, xmm2"); // VCVTPS2UQQ_XMMu64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x79, 0x0a], "vcvtps2uqq xmm1, qword [rdx]"); // VCVTPS2UQQ_XMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x79, 0x0a], "vcvtps2uqq xmm1{k5}, qword [rdx]"); // VCVTPS2UQQ_XMMu64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x7a, 0xca], "vcvttpd2qq zmm1{k5}{z}{sae}, zmm2"); // VCVTTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x7a, 0xca], "vcvttpd2qq zmm1{sae}, zmm2"); // VCVTTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x7a, 0xca], "vcvttpd2qq zmm1{k5}{sae}, zmm2"); // VCVTTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x7a, 0x0a], "vcvttpd2qq ymm1{k5}{z}, qword [rdx]{1to4}"); // VCVTTPD2QQ_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x7a, 0x0a], "vcvttpd2qq ymm1, qword [rdx]{1to4}"); // VCVTTPD2QQ_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x7a, 0x0a], "vcvttpd2qq ymm1{k5}, qword [rdx]{1to4}"); // VCVTTPD2QQ_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x7a, 0xca], "vcvttpd2qq ymm1{k5}{z}, ymm2"); // VCVTTPD2QQ_YMMi64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x7a, 0x0a], "vcvttpd2qq ymm1{k5}{z}, ymmword [rdx]"); // VCVTTPD2QQ_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x7a, 0xca], "vcvttpd2qq ymm1, ymm2"); // VCVTTPD2QQ_YMMi64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x7a, 0xca], "vcvttpd2qq ymm1{k5}, ymm2"); // VCVTTPD2QQ_YMMi64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x7a, 0x0a], "vcvttpd2qq ymm1, ymmword [rdx]"); // VCVTTPD2QQ_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x7a, 0x0a], "vcvttpd2qq ymm1{k5}, ymmword [rdx]"); // VCVTTPD2QQ_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xfd, 0x7a, 0xca], "vcvttps2qq zmm1{k5}{z}{sae}, ymm2"); // VCVTTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x78, 0x7a, 0xca], "vcvttps2qq zmm1{sae}, ymm2"); // VCVTTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x7d, 0x7a, 0xca], "vcvttps2qq zmm1{k5}{sae}, ymm2"); // VCVTTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x7a, 0x0a], "vcvttps2qq ymm1{k5}{z}, dword [rdx]{1to4}"); // VCVTTPS2QQ_YMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x7a, 0x0a], "vcvttps2qq ymm1, dword [rdx]{1to4}"); // VCVTTPS2QQ_YMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x7a, 0x0a], "vcvttps2qq ymm1{k5}, dword [rdx]{1to4}"); // VCVTTPS2QQ_YMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x7a, 0xca], "vcvttps2qq ymm1{k5}{z}, xmm2"); // VCVTTPS2QQ_YMMi64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x7a, 0x0a], "vcvttps2qq ymm1{k5}{z}, xmmword [rdx]"); // VCVTTPS2QQ_YMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x7a, 0xca], "vcvttps2qq ymm1, xmm2"); // VCVTTPS2QQ_YMMi64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x7a, 0xca], "vcvttps2qq ymm1{k5}, xmm2"); // VCVTTPS2QQ_YMMi64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x7a, 0x0a], "vcvttps2qq ymm1, xmmword [rdx]"); // VCVTTPS2QQ_YMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x7a, 0x0a], "vcvttps2qq ymm1{k5}, xmmword [rdx]"); // VCVTTPS2QQ_YMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x7a, 0x0a], "vcvttpd2qq zmm1{k5}{z}, qword [rdx]{1to8}"); // VCVTTPD2QQ_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x7a, 0x0a], "vcvttpd2qq zmm1, qword [rdx]{1to8}"); // VCVTTPD2QQ_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x7a, 0x0a], "vcvttpd2qq zmm1{k5}, qword [rdx]{1to8}"); // VCVTTPD2QQ_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x7a, 0x0a], "vcvttpd2qq xmm1{k5}{z}, qword [rdx]{1to2}"); // VCVTTPD2QQ_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x7a, 0x0a], "vcvttpd2qq xmm1, qword [rdx]{1to2}"); // VCVTTPD2QQ_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x7a, 0x0a], "vcvttpd2qq xmm1{k5}, qword [rdx]{1to2}"); // VCVTTPD2QQ_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x7a, 0xca], "vcvttpd2qq zmm1{k5}{z}, zmm2"); // VCVTTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x7a, 0x0a], "vcvttpd2qq zmm1{k5}{z}, zmmword [rdx]"); // VCVTTPD2QQ_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x7a, 0xca], "vcvttpd2qq zmm1, zmm2"); // VCVTTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x7a, 0xca], "vcvttpd2qq zmm1{k5}, zmm2"); // VCVTTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x7a, 0x0a], "vcvttpd2qq zmm1, zmmword [rdx]"); // VCVTTPD2QQ_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x7a, 0x0a], "vcvttpd2qq zmm1{k5}, zmmword [rdx]"); // VCVTTPD2QQ_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x7a, 0xca], "vcvttpd2qq xmm1{k5}{z}, xmm2"); // VCVTTPD2QQ_XMMi64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x7a, 0x0a], "vcvttpd2qq xmm1{k5}{z}, xmmword [rdx]"); // VCVTTPD2QQ_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7a, 0xca], "vcvttpd2qq xmm1, xmm2"); // VCVTTPD2QQ_XMMi64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x7a, 0xca], "vcvttpd2qq xmm1{k5}, xmm2"); // VCVTTPD2QQ_XMMi64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7a, 0x0a], "vcvttpd2qq xmm1, xmmword [rdx]"); // VCVTTPD2QQ_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x7a, 0x0a], "vcvttpd2qq xmm1{k5}, xmmword [rdx]"); // VCVTTPD2QQ_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x7a, 0x0a], "vcvttps2qq zmm1{k5}{z}, dword [rdx]{1to8}"); // VCVTTPS2QQ_ZMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x7a, 0x0a], "vcvttps2qq zmm1, dword [rdx]{1to8}"); // VCVTTPS2QQ_ZMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x7a, 0x0a], "vcvttps2qq zmm1{k5}, dword [rdx]{1to8}"); // VCVTTPS2QQ_ZMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x7a, 0x0a], "vcvttps2qq xmm1{k5}{z}, dword [rdx]{1to2}"); // VCVTTPS2QQ_XMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x7a, 0x0a], "vcvttps2qq xmm1, dword [rdx]{1to2}"); // VCVTTPS2QQ_XMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x7a, 0x0a], "vcvttps2qq xmm1{k5}, dword [rdx]{1to2}"); // VCVTTPS2QQ_XMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x7a, 0xca], "vcvttps2qq zmm1{k5}{z}, ymm2"); // VCVTTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x7a, 0x0a], "vcvttps2qq zmm1{k5}{z}, ymmword [rdx]"); // VCVTTPS2QQ_ZMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x7a, 0xca], "vcvttps2qq zmm1, ymm2"); // VCVTTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x7a, 0xca], "vcvttps2qq zmm1{k5}, ymm2"); // VCVTTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x7a, 0x0a], "vcvttps2qq zmm1, ymmword [rdx]"); // VCVTTPS2QQ_ZMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x7a, 0x0a], "vcvttps2qq zmm1{k5}, ymmword [rdx]"); // VCVTTPS2QQ_ZMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x7a, 0xca], "vcvttps2qq xmm1{k5}{z}, xmm2"); // VCVTTPS2QQ_XMMi64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x7a, 0x0a], "vcvttps2qq xmm1{k5}{z}, qword [rdx]"); // VCVTTPS2QQ_XMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7a, 0xca], "vcvttps2qq xmm1, xmm2"); // VCVTTPS2QQ_XMMi64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x7a, 0xca], "vcvttps2qq xmm1{k5}, xmm2"); // VCVTTPS2QQ_XMMi64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7a, 0x0a], "vcvttps2qq xmm1, qword [rdx]"); // VCVTTPS2QQ_XMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x7a, 0x0a], "vcvttps2qq xmm1{k5}, qword [rdx]"); // VCVTTPS2QQ_XMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{z}{rz-sae}, zmm2"); // VCVTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0x7b, 0xca], "vcvtpd2qq zmm1{rz-sae}, zmm2"); // VCVTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{rz-sae}, zmm2"); // VCVTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{z}{rd-sae}, zmm2"); // VCVTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0x7b, 0x0a], "vcvtpd2qq ymm1{k5}{z}, qword [rdx]{1to4}"); // VCVTPD2QQ_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x7b, 0xca], "vcvtpd2qq zmm1{rd-sae}, zmm2"); // VCVTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{rd-sae}, zmm2"); // VCVTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0x7b, 0x0a], "vcvtpd2qq ymm1, qword [rdx]{1to4}"); // VCVTPD2QQ_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0x7b, 0x0a], "vcvtpd2qq ymm1{k5}, qword [rdx]{1to4}"); // VCVTPD2QQ_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x7b, 0xca], "vcvtpd2qq ymm1{k5}{z}, ymm2"); // VCVTPD2QQ_YMMi64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x7b, 0x0a], "vcvtpd2qq ymm1{k5}{z}, ymmword [rdx]"); // VCVTPD2QQ_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x7b, 0xca], "vcvtpd2qq ymm1, ymm2"); // VCVTPD2QQ_YMMi64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x7b, 0xca], "vcvtpd2qq ymm1{k5}, ymm2"); // VCVTPD2QQ_YMMi64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x7b, 0x0a], "vcvtpd2qq ymm1, ymmword [rdx]"); // VCVTPD2QQ_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x7b, 0x0a], "vcvtpd2qq ymm1{k5}, ymmword [rdx]"); // VCVTPD2QQ_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xfd, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{z}{rz-sae}, ymm2"); // VCVTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x78, 0x7b, 0xca], "vcvtps2qq zmm1{rz-sae}, ymm2"); // VCVTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x7d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{rz-sae}, ymm2"); // VCVTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{z}{rd-sae}, ymm2"); // VCVTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0x7b, 0x0a], "vcvtps2qq ymm1{k5}{z}, dword [rdx]{1to4}"); // VCVTPS2QQ_YMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x7b, 0xca], "vcvtps2qq zmm1{rd-sae}, ymm2"); // VCVTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{rd-sae}, ymm2"); // VCVTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0x7b, 0x0a], "vcvtps2qq ymm1, dword [rdx]{1to4}"); // VCVTPS2QQ_YMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0x7b, 0x0a], "vcvtps2qq ymm1{k5}, dword [rdx]{1to4}"); // VCVTPS2QQ_YMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x7b, 0xca], "vcvtps2qq ymm1{k5}{z}, xmm2"); // VCVTPS2QQ_YMMi64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x7b, 0x0a], "vcvtps2qq ymm1{k5}{z}, xmmword [rdx]"); // VCVTPS2QQ_YMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x7b, 0xca], "vcvtps2qq ymm1, xmm2"); // VCVTPS2QQ_YMMi64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x7b, 0xca], "vcvtps2qq ymm1{k5}, xmm2"); // VCVTPS2QQ_YMMi64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x7b, 0x0a], "vcvtps2qq ymm1, xmmword [rdx]"); // VCVTPS2QQ_YMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x7b, 0x0a], "vcvtps2qq ymm1{k5}, xmmword [rdx]"); // VCVTPS2QQ_YMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{z}{ru-sae}, zmm2"); // VCVTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0x7b, 0x0a], "vcvtpd2qq zmm1{k5}{z}, qword [rdx]{1to8}"); // VCVTPD2QQ_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x7b, 0xca], "vcvtpd2qq zmm1{ru-sae}, zmm2"); // VCVTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{ru-sae}, zmm2"); // VCVTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0x7b, 0x0a], "vcvtpd2qq zmm1, qword [rdx]{1to8}"); // VCVTPD2QQ_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0x7b, 0x0a], "vcvtpd2qq zmm1{k5}, qword [rdx]{1to8}"); // VCVTPD2QQ_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{z}{rne-sae}, zmm2"); // VCVTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0x7b, 0x0a], "vcvtpd2qq xmm1{k5}{z}, qword [rdx]{1to2}"); // VCVTPD2QQ_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x7b, 0xca], "vcvtpd2qq zmm1{rne-sae}, zmm2"); // VCVTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{rne-sae}, zmm2"); // VCVTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0x7b, 0x0a], "vcvtpd2qq xmm1, qword [rdx]{1to2}"); // VCVTPD2QQ_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0x7b, 0x0a], "vcvtpd2qq xmm1{k5}, qword [rdx]{1to2}"); // VCVTPD2QQ_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}{z}, zmm2"); // VCVTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x7b, 0x0a], "vcvtpd2qq zmm1{k5}{z}, zmmword [rdx]"); // VCVTPD2QQ_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x7b, 0xca], "vcvtpd2qq zmm1, zmm2"); // VCVTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x7b, 0xca], "vcvtpd2qq zmm1{k5}, zmm2"); // VCVTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x7b, 0x0a], "vcvtpd2qq zmm1, zmmword [rdx]"); // VCVTPD2QQ_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x7b, 0x0a], "vcvtpd2qq zmm1{k5}, zmmword [rdx]"); // VCVTPD2QQ_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x7b, 0xca], "vcvtpd2qq xmm1{k5}{z}, xmm2"); // VCVTPD2QQ_XMMi64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x7b, 0x0a], "vcvtpd2qq xmm1{k5}{z}, xmmword [rdx]"); // VCVTPD2QQ_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7b, 0xca], "vcvtpd2qq xmm1, xmm2"); // VCVTPD2QQ_XMMi64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x7b, 0xca], "vcvtpd2qq xmm1{k5}, xmm2"); // VCVTPD2QQ_XMMi64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7b, 0x0a], "vcvtpd2qq xmm1, xmmword [rdx]"); // VCVTPD2QQ_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x7b, 0x0a], "vcvtpd2qq xmm1{k5}, xmmword [rdx]"); // VCVTPD2QQ_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{z}{ru-sae}, ymm2"); // VCVTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0x7b, 0x0a], "vcvtps2qq zmm1{k5}{z}, dword [rdx]{1to8}"); // VCVTPS2QQ_ZMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x7b, 0xca], "vcvtps2qq zmm1{ru-sae}, ymm2"); // VCVTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{ru-sae}, ymm2"); // VCVTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0x7b, 0x0a], "vcvtps2qq zmm1, dword [rdx]{1to8}"); // VCVTPS2QQ_ZMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0x7b, 0x0a], "vcvtps2qq zmm1{k5}, dword [rdx]{1to8}"); // VCVTPS2QQ_ZMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{z}{rne-sae}, ymm2"); // VCVTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0x7b, 0x0a], "vcvtps2qq xmm1{k5}{z}, dword [rdx]{1to2}"); // VCVTPS2QQ_XMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x7b, 0xca], "vcvtps2qq zmm1{rne-sae}, ymm2"); // VCVTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{rne-sae}, ymm2"); // VCVTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0x7b, 0x0a], "vcvtps2qq xmm1, dword [rdx]{1to2}"); // VCVTPS2QQ_XMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0x7b, 0x0a], "vcvtps2qq xmm1{k5}, dword [rdx]{1to2}"); // VCVTPS2QQ_XMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x7b, 0xca], "vcvtps2qq zmm1{k5}{z}, ymm2"); // VCVTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x7b, 0x0a], "vcvtps2qq zmm1{k5}{z}, ymmword [rdx]"); // VCVTPS2QQ_ZMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x7b, 0xca], "vcvtps2qq zmm1, ymm2"); // VCVTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x7b, 0xca], "vcvtps2qq zmm1{k5}, ymm2"); // VCVTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x7b, 0x0a], "vcvtps2qq zmm1, ymmword [rdx]"); // VCVTPS2QQ_ZMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x7b, 0x0a], "vcvtps2qq zmm1{k5}, ymmword [rdx]"); // VCVTPS2QQ_ZMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x7b, 0xca], "vcvtps2qq xmm1{k5}{z}, xmm2"); // VCVTPS2QQ_XMMi64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x7b, 0x0a], "vcvtps2qq xmm1{k5}{z}, qword [rdx]"); // VCVTPS2QQ_XMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7b, 0xca], "vcvtps2qq xmm1, xmm2"); // VCVTPS2QQ_XMMi64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x7b, 0xca], "vcvtps2qq xmm1{k5}, xmm2"); // VCVTPS2QQ_XMMi64_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7b, 0x0a], "vcvtps2qq xmm1, qword [rdx]"); // VCVTPS2QQ_XMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x7b, 0x0a], "vcvtps2qq xmm1{k5}, qword [rdx]"); // VCVTPS2QQ_XMMi64_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7e, 0xca], "vmovq rdx, xmm1"); // VMOVQ_GPR64u64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7e, 0x0a], "vmovq qword [rdx], xmm1"); // VMOVQ_MEMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7e, 0xca], "vmovd edx, xmm1"); // VMOVD_GPR32u32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7e, 0x0a], "vmovd dword [rdx], xmm1"); // VMOVD_MEMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0x7f, 0xca], "vmovdqa64 ymm2{k5}{z}, ymm1"); // VMOVDQA64_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x7f, 0xca], "vmovdqa64 ymm2, ymm1"); // VMOVDQA64_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x7f, 0xca], "vmovdqa64 ymm2{k5}, ymm1"); // VMOVDQA64_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0x7f, 0x0a], "vmovdqa64 ymmword [rdx], ymm1"); // VMOVDQA64_MEMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0x7f, 0x0a], "vmovdqa64 ymmword [rdx]{k5}, ymm1"); // VMOVDQA64_MEMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0x7f, 0xca], "vmovdqa32 ymm2{k5}{z}, ymm1"); // VMOVDQA32_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x7f, 0xca], "vmovdqa32 ymm2, ymm1"); // VMOVDQA32_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x7f, 0xca], "vmovdqa32 ymm2{k5}, ymm1"); // VMOVDQA32_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0x7f, 0x0a], "vmovdqa32 ymmword [rdx], ymm1"); // VMOVDQA32_MEMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0x7f, 0x0a], "vmovdqa32 ymmword [rdx]{k5}, ymm1"); // VMOVDQA32_MEMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0x7f, 0xca], "vmovdqa64 zmm2{k5}{z}, zmm1"); // VMOVDQA64_ZMMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x7f, 0xca], "vmovdqa64 zmm2, zmm1"); // VMOVDQA64_ZMMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x7f, 0xca], "vmovdqa64 zmm2{k5}, zmm1"); // VMOVDQA64_ZMMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0x7f, 0x0a], "vmovdqa64 zmmword [rdx], zmm1"); // VMOVDQA64_MEMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0x7f, 0x0a], "vmovdqa64 zmmword [rdx]{k5}, zmm1"); // VMOVDQA64_MEMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0x7f, 0xca], "vmovdqa64 xmm2{k5}{z}, xmm1"); // VMOVDQA64_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7f, 0xca], "vmovdqa64 xmm2, xmm1"); // VMOVDQA64_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x7f, 0xca], "vmovdqa64 xmm2{k5}, xmm1"); // VMOVDQA64_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0x7f, 0x0a], "vmovdqa64 xmmword [rdx], xmm1"); // VMOVDQA64_MEMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0x7f, 0x0a], "vmovdqa64 xmmword [rdx]{k5}, xmm1"); // VMOVDQA64_MEMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0x7f, 0xca], "vmovdqa32 zmm2{k5}{z}, zmm1"); // VMOVDQA32_ZMMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x7f, 0xca], "vmovdqa32 zmm2, zmm1"); // VMOVDQA32_ZMMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x7f, 0xca], "vmovdqa32 zmm2{k5}, zmm1"); // VMOVDQA32_ZMMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0x7f, 0x0a], "vmovdqa32 zmmword [rdx], zmm1"); // VMOVDQA32_MEMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0x7f, 0x0a], "vmovdqa32 zmmword [rdx]{k5}, zmm1"); // VMOVDQA32_MEMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0x7f, 0xca], "vmovdqa32 xmm2{k5}{z}, xmm1"); // VMOVDQA32_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7f, 0xca], "vmovdqa32 xmm2, xmm1"); // VMOVDQA32_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x7f, 0xca], "vmovdqa32 xmm2{k5}, xmm1"); // VMOVDQA32_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0x7f, 0x0a], "vmovdqa32 xmmword [rdx], xmm1"); // VMOVDQA32_MEMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0x7f, 0x0a], "vmovdqa32 xmmword [rdx]{k5}, xmm1"); // VMOVDQA32_MEMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0xc2, 0xca, 0xcc], "vcmppd k1{sae}, zmm0, zmm2, 0xcc"); // VCMPPD_MASKmskw_MASKmskw_ZMMf64_ZMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0xc2, 0xca, 0xcc], "vcmppd k1{k5}{sae}, zmm0, zmm2, 0xcc"); // VCMPPD_MASKmskw_MASKmskw_ZMMf64_ZMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xc2, 0x0a, 0xcc], "vcmppd k1, ymm0, qword [rdx]{1to4}, 0xcc"); // VCMPPD_MASKmskw_MASKmskw_YMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, ymm0, qword [rdx]{1to4}, 0xcc"); // VCMPPD_MASKmskw_MASKmskw_YMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xc2, 0xca, 0xcc], "vcmppd k1, ymm0, ymm2, 0xcc"); // VCMPPD_MASKmskw_MASKmskw_YMMf64_YMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xc2, 0xca, 0xcc], "vcmppd k1{k5}, ymm0, ymm2, 0xcc"); // VCMPPD_MASKmskw_MASKmskw_YMMf64_YMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xc2, 0x0a, 0xcc], "vcmppd k1, ymm0, ymmword [rdx], 0xcc"); // VCMPPD_MASKmskw_MASKmskw_YMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, ymm0, ymmword [rdx], 0xcc"); // VCMPPD_MASKmskw_MASKmskw_YMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xc2, 0x0a, 0xcc], "vcmppd k1, zmm0, qword [rdx]{1to8}, 0xcc"); // VCMPPD_MASKmskw_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, zmm0, qword [rdx]{1to8}, 0xcc"); // VCMPPD_MASKmskw_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xc2, 0x0a, 0xcc], "vcmppd k1, xmm0, qword [rdx]{1to2}, 0xcc"); // VCMPPD_MASKmskw_MASKmskw_XMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, xmm0, qword [rdx]{1to2}, 0xcc"); // VCMPPD_MASKmskw_MASKmskw_XMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xc2, 0xca, 0xcc], "vcmppd k1, zmm0, zmm2, 0xcc"); // VCMPPD_MASKmskw_MASKmskw_ZMMf64_ZMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xc2, 0xca, 0xcc], "vcmppd k1{k5}, zmm0, zmm2, 0xcc"); // VCMPPD_MASKmskw_MASKmskw_ZMMf64_ZMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xc2, 0x0a, 0xcc], "vcmppd k1, zmm0, zmmword [rdx], 0xcc"); // VCMPPD_MASKmskw_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, zmm0, zmmword [rdx], 0xcc"); // VCMPPD_MASKmskw_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc2, 0xca, 0xcc], "vcmppd k1, xmm0, xmm2, 0xcc"); // VCMPPD_MASKmskw_MASKmskw_XMMf64_XMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xc2, 0xca, 0xcc], "vcmppd k1{k5}, xmm0, xmm2, 0xcc"); // VCMPPD_MASKmskw_MASKmskw_XMMf64_XMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc2, 0x0a, 0xcc], "vcmppd k1, xmm0, xmmword [rdx], 0xcc"); // VCMPPD_MASKmskw_MASKmskw_XMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xc2, 0x0a, 0xcc], "vcmppd k1{k5}, xmm0, xmmword [rdx], 0xcc"); // VCMPPD_MASKmskw_MASKmskw_XMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc4, 0xca, 0xcc], "vpinsrw xmm1, xmm0, edx, 0xcc"); // VPINSRW_XMMu16_XMMu16_GPR32u16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc4, 0x0a, 0xcc], "vpinsrw xmm1, xmm0, word [rdx], 0xcc"); // VPINSRW_XMMu16_XMMu16_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc5, 0xca, 0xcc], "vpextrw ecx, xmm2, 0xcc"); // VPEXTRW_GPR32u16_XMMu16_IMM8_AVX512_C5, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xc6, 0x0a, 0xcc], "vshufpd ymm1{k5}{z}, ymm0, qword [rdx]{1to4}, 0xcc"); // VSHUFPD_YMMf64_MASKmskw_YMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xc6, 0x0a, 0xcc], "vshufpd ymm1, ymm0, qword [rdx]{1to4}, 0xcc"); // VSHUFPD_YMMf64_MASKmskw_YMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xc6, 0x0a, 0xcc], "vshufpd ymm1{k5}, ymm0, qword [rdx]{1to4}, 0xcc"); // VSHUFPD_YMMf64_MASKmskw_YMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xc6, 0xca, 0xcc], "vshufpd ymm1{k5}{z}, ymm0, ymm2, 0xcc"); // VSHUFPD_YMMf64_MASKmskw_YMMf64_YMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xc6, 0x0a, 0xcc], "vshufpd ymm1{k5}{z}, ymm0, ymmword [rdx], 0xcc"); // VSHUFPD_YMMf64_MASKmskw_YMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xc6, 0xca, 0xcc], "vshufpd ymm1, ymm0, ymm2, 0xcc"); // VSHUFPD_YMMf64_MASKmskw_YMMf64_YMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xc6, 0xca, 0xcc], "vshufpd ymm1{k5}, ymm0, ymm2, 0xcc"); // VSHUFPD_YMMf64_MASKmskw_YMMf64_YMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xc6, 0x0a, 0xcc], "vshufpd ymm1, ymm0, ymmword [rdx], 0xcc"); // VSHUFPD_YMMf64_MASKmskw_YMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xc6, 0x0a, 0xcc], "vshufpd ymm1{k5}, ymm0, ymmword [rdx], 0xcc"); // VSHUFPD_YMMf64_MASKmskw_YMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xc6, 0x0a, 0xcc], "vshufpd zmm1{k5}{z}, zmm0, qword [rdx]{1to8}, 0xcc"); // VSHUFPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xc6, 0x0a, 0xcc], "vshufpd zmm1, zmm0, qword [rdx]{1to8}, 0xcc"); // VSHUFPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xc6, 0x0a, 0xcc], "vshufpd zmm1{k5}, zmm0, qword [rdx]{1to8}, 0xcc"); // VSHUFPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xc6, 0x0a, 0xcc], "vshufpd xmm1{k5}{z}, xmm0, qword [rdx]{1to2}, 0xcc"); // VSHUFPD_XMMf64_MASKmskw_XMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xc6, 0x0a, 0xcc], "vshufpd xmm1, xmm0, qword [rdx]{1to2}, 0xcc"); // VSHUFPD_XMMf64_MASKmskw_XMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xc6, 0x0a, 0xcc], "vshufpd xmm1{k5}, xmm0, qword [rdx]{1to2}, 0xcc"); // VSHUFPD_XMMf64_MASKmskw_XMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xc6, 0xca, 0xcc], "vshufpd zmm1{k5}{z}, zmm0, zmm2, 0xcc"); // VSHUFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xc6, 0x0a, 0xcc], "vshufpd zmm1{k5}{z}, zmm0, zmmword [rdx], 0xcc"); // VSHUFPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xc6, 0xca, 0xcc], "vshufpd zmm1, zmm0, zmm2, 0xcc"); // VSHUFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xc6, 0xca, 0xcc], "vshufpd zmm1{k5}, zmm0, zmm2, 0xcc"); // VSHUFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xc6, 0x0a, 0xcc], "vshufpd zmm1, zmm0, zmmword [rdx], 0xcc"); // VSHUFPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xc6, 0x0a, 0xcc], "vshufpd zmm1{k5}, zmm0, zmmword [rdx], 0xcc"); // VSHUFPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xc6, 0xca, 0xcc], "vshufpd xmm1{k5}{z}, xmm0, xmm2, 0xcc"); // VSHUFPD_XMMf64_MASKmskw_XMMf64_XMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xc6, 0x0a, 0xcc], "vshufpd xmm1{k5}{z}, xmm0, xmmword [rdx], 0xcc"); // VSHUFPD_XMMf64_MASKmskw_XMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc6, 0xca, 0xcc], "vshufpd xmm1, xmm0, xmm2, 0xcc"); // VSHUFPD_XMMf64_MASKmskw_XMMf64_XMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xc6, 0xca, 0xcc], "vshufpd xmm1{k5}, xmm0, xmm2, 0xcc"); // VSHUFPD_XMMf64_MASKmskw_XMMf64_XMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xc6, 0x0a, 0xcc], "vshufpd xmm1, xmm0, xmmword [rdx], 0xcc"); // VSHUFPD_XMMf64_MASKmskw_XMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xc6, 0x0a, 0xcc], "vshufpd xmm1{k5}, xmm0, xmmword [rdx], 0xcc"); // VSHUFPD_XMMf64_MASKmskw_XMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd1, 0xca], "vpsrlw ymm1{k5}{z}, ymm0, xmm2"); // VPSRLW_YMMu16_MASKmskw_YMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd1, 0x0a], "vpsrlw ymm1{k5}{z}, ymm0, xmmword [rdx]"); // VPSRLW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd1, 0xca], "vpsrlw ymm1, ymm0, xmm2"); // VPSRLW_YMMu16_MASKmskw_YMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd1, 0xca], "vpsrlw ymm1{k5}, ymm0, xmm2"); // VPSRLW_YMMu16_MASKmskw_YMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd1, 0x0a], "vpsrlw ymm1, ymm0, xmmword [rdx]"); // VPSRLW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd1, 0x0a], "vpsrlw ymm1{k5}, ymm0, xmmword [rdx]"); // VPSRLW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd1, 0xca], "vpsrlw zmm1{k5}{z}, zmm0, xmm2"); // VPSRLW_ZMMu16_MASKmskw_ZMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd1, 0x0a], "vpsrlw zmm1{k5}{z}, zmm0, xmmword [rdx]"); // VPSRLW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd1, 0xca], "vpsrlw zmm1, zmm0, xmm2"); // VPSRLW_ZMMu16_MASKmskw_ZMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd1, 0xca], "vpsrlw zmm1{k5}, zmm0, xmm2"); // VPSRLW_ZMMu16_MASKmskw_ZMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd1, 0x0a], "vpsrlw zmm1, zmm0, xmmword [rdx]"); // VPSRLW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd1, 0x0a], "vpsrlw zmm1{k5}, zmm0, xmmword [rdx]"); // VPSRLW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd1, 0xca], "vpsrlw xmm1{k5}{z}, xmm0, xmm2"); // VPSRLW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd1, 0x0a], "vpsrlw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSRLW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd1, 0xca], "vpsrlw xmm1, xmm0, xmm2"); // VPSRLW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd1, 0xca], "vpsrlw xmm1{k5}, xmm0, xmm2"); // VPSRLW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd1, 0x0a], "vpsrlw xmm1, xmm0, xmmword [rdx]"); // VPSRLW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd1, 0x0a], "vpsrlw xmm1{k5}, xmm0, xmmword [rdx]"); // VPSRLW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xd2, 0xca], "vpsrld ymm1{k5}{z}, ymm0, xmm2"); // VPSRLD_YMMu32_MASKmskw_YMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xd2, 0x0a], "vpsrld ymm1{k5}{z}, ymm0, xmmword [rdx]"); // VPSRLD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xd2, 0xca], "vpsrld ymm1, ymm0, xmm2"); // VPSRLD_YMMu32_MASKmskw_YMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xd2, 0xca], "vpsrld ymm1{k5}, ymm0, xmm2"); // VPSRLD_YMMu32_MASKmskw_YMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xd2, 0x0a], "vpsrld ymm1, ymm0, xmmword [rdx]"); // VPSRLD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xd2, 0x0a], "vpsrld ymm1{k5}, ymm0, xmmword [rdx]"); // VPSRLD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xd2, 0xca], "vpsrld zmm1{k5}{z}, zmm0, xmm2"); // VPSRLD_ZMMu32_MASKmskw_ZMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xd2, 0x0a], "vpsrld zmm1{k5}{z}, zmm0, xmmword [rdx]"); // VPSRLD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xd2, 0xca], "vpsrld zmm1, zmm0, xmm2"); // VPSRLD_ZMMu32_MASKmskw_ZMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xd2, 0xca], "vpsrld zmm1{k5}, zmm0, xmm2"); // VPSRLD_ZMMu32_MASKmskw_ZMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xd2, 0x0a], "vpsrld zmm1, zmm0, xmmword [rdx]"); // VPSRLD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xd2, 0x0a], "vpsrld zmm1{k5}, zmm0, xmmword [rdx]"); // VPSRLD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xd2, 0xca], "vpsrld xmm1{k5}{z}, xmm0, xmm2"); // VPSRLD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xd2, 0x0a], "vpsrld xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSRLD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xd2, 0xca], "vpsrld xmm1, xmm0, xmm2"); // VPSRLD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xd2, 0xca], "vpsrld xmm1{k5}, xmm0, xmm2"); // VPSRLD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xd2, 0x0a], "vpsrld xmm1, xmm0, xmmword [rdx]"); // VPSRLD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xd2, 0x0a], "vpsrld xmm1{k5}, xmm0, xmmword [rdx]"); // VPSRLD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd3, 0xca], "vpsrlq ymm1{k5}{z}, ymm0, xmm2"); // VPSRLQ_YMMu64_MASKmskw_YMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd3, 0x0a], "vpsrlq ymm1{k5}{z}, ymm0, xmmword [rdx]"); // VPSRLQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd3, 0xca], "vpsrlq ymm1, ymm0, xmm2"); // VPSRLQ_YMMu64_MASKmskw_YMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd3, 0xca], "vpsrlq ymm1{k5}, ymm0, xmm2"); // VPSRLQ_YMMu64_MASKmskw_YMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd3, 0x0a], "vpsrlq ymm1, ymm0, xmmword [rdx]"); // VPSRLQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd3, 0x0a], "vpsrlq ymm1{k5}, ymm0, xmmword [rdx]"); // VPSRLQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd3, 0xca], "vpsrlq zmm1{k5}{z}, zmm0, xmm2"); // VPSRLQ_ZMMu64_MASKmskw_ZMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd3, 0x0a], "vpsrlq zmm1{k5}{z}, zmm0, xmmword [rdx]"); // VPSRLQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd3, 0xca], "vpsrlq zmm1, zmm0, xmm2"); // VPSRLQ_ZMMu64_MASKmskw_ZMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd3, 0xca], "vpsrlq zmm1{k5}, zmm0, xmm2"); // VPSRLQ_ZMMu64_MASKmskw_ZMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd3, 0x0a], "vpsrlq zmm1, zmm0, xmmword [rdx]"); // VPSRLQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd3, 0x0a], "vpsrlq zmm1{k5}, zmm0, xmmword [rdx]"); // VPSRLQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd3, 0xca], "vpsrlq xmm1{k5}{z}, xmm0, xmm2"); // VPSRLQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd3, 0x0a], "vpsrlq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSRLQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd3, 0xca], "vpsrlq xmm1, xmm0, xmm2"); // VPSRLQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd3, 0xca], "vpsrlq xmm1{k5}, xmm0, xmm2"); // VPSRLQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd3, 0x0a], "vpsrlq xmm1, xmm0, xmmword [rdx]"); // VPSRLQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd3, 0x0a], "vpsrlq xmm1{k5}, xmm0, xmmword [rdx]"); // VPSRLQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xd4, 0x0a], "vpaddq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPADDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xd4, 0x0a], "vpaddq ymm1, ymm0, qword [rdx]{1to4}"); // VPADDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xd4, 0x0a], "vpaddq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPADDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd4, 0xca], "vpaddq ymm1{k5}{z}, ymm0, ymm2"); // VPADDQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd4, 0x0a], "vpaddq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPADDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd4, 0xca], "vpaddq ymm1, ymm0, ymm2"); // VPADDQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd4, 0xca], "vpaddq ymm1{k5}, ymm0, ymm2"); // VPADDQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd4, 0x0a], "vpaddq ymm1, ymm0, ymmword [rdx]"); // VPADDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd4, 0x0a], "vpaddq ymm1{k5}, ymm0, ymmword [rdx]"); // VPADDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xd4, 0x0a], "vpaddq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPADDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xd4, 0x0a], "vpaddq zmm1, zmm0, qword [rdx]{1to8}"); // VPADDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xd4, 0x0a], "vpaddq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPADDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xd4, 0x0a], "vpaddq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPADDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xd4, 0x0a], "vpaddq xmm1, xmm0, qword [rdx]{1to2}"); // VPADDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xd4, 0x0a], "vpaddq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPADDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd4, 0xca], "vpaddq zmm1{k5}{z}, zmm0, zmm2"); // VPADDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd4, 0x0a], "vpaddq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPADDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd4, 0xca], "vpaddq zmm1, zmm0, zmm2"); // VPADDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd4, 0xca], "vpaddq zmm1{k5}, zmm0, zmm2"); // VPADDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd4, 0x0a], "vpaddq zmm1, zmm0, zmmword [rdx]"); // VPADDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd4, 0x0a], "vpaddq zmm1{k5}, zmm0, zmmword [rdx]"); // VPADDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd4, 0xca], "vpaddq xmm1{k5}{z}, xmm0, xmm2"); // VPADDQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd4, 0x0a], "vpaddq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPADDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd4, 0xca], "vpaddq xmm1, xmm0, xmm2"); // VPADDQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd4, 0xca], "vpaddq xmm1{k5}, xmm0, xmm2"); // VPADDQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd4, 0x0a], "vpaddq xmm1, xmm0, xmmword [rdx]"); // VPADDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd4, 0x0a], "vpaddq xmm1{k5}, xmm0, xmmword [rdx]"); // VPADDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd5, 0xca], "vpmullw ymm1{k5}{z}, ymm0, ymm2"); // VPMULLW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd5, 0x0a], "vpmullw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMULLW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd5, 0xca], "vpmullw ymm1, ymm0, ymm2"); // VPMULLW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd5, 0xca], "vpmullw ymm1{k5}, ymm0, ymm2"); // VPMULLW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd5, 0x0a], "vpmullw ymm1, ymm0, ymmword [rdx]"); // VPMULLW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd5, 0x0a], "vpmullw ymm1{k5}, ymm0, ymmword [rdx]"); // VPMULLW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd5, 0xca], "vpmullw zmm1{k5}{z}, zmm0, zmm2"); // VPMULLW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd5, 0x0a], "vpmullw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMULLW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd5, 0xca], "vpmullw zmm1, zmm0, zmm2"); // VPMULLW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd5, 0xca], "vpmullw zmm1{k5}, zmm0, zmm2"); // VPMULLW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd5, 0x0a], "vpmullw zmm1, zmm0, zmmword [rdx]"); // VPMULLW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd5, 0x0a], "vpmullw zmm1{k5}, zmm0, zmmword [rdx]"); // VPMULLW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd5, 0xca], "vpmullw xmm1{k5}{z}, xmm0, xmm2"); // VPMULLW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd5, 0x0a], "vpmullw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMULLW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd5, 0xca], "vpmullw xmm1, xmm0, xmm2"); // VPMULLW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd5, 0xca], "vpmullw xmm1{k5}, xmm0, xmm2"); // VPMULLW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd5, 0x0a], "vpmullw xmm1, xmm0, xmmword [rdx]"); // VPMULLW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd5, 0x0a], "vpmullw xmm1{k5}, xmm0, xmmword [rdx]"); // VPMULLW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd6, 0xca], "vmovq xmm2, xmm1"); // VMOVQ_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd6, 0x0a], "vmovq qword [rdx], xmm1"); // VMOVQ_MEMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd8, 0xca], "vpsubusb ymm1{k5}{z}, ymm0, ymm2"); // VPSUBUSB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd8, 0x0a], "vpsubusb ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSUBUSB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd8, 0xca], "vpsubusb ymm1, ymm0, ymm2"); // VPSUBUSB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd8, 0xca], "vpsubusb ymm1{k5}, ymm0, ymm2"); // VPSUBUSB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd8, 0x0a], "vpsubusb ymm1, ymm0, ymmword [rdx]"); // VPSUBUSB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd8, 0x0a], "vpsubusb ymm1{k5}, ymm0, ymmword [rdx]"); // VPSUBUSB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd8, 0xca], "vpsubusb zmm1{k5}{z}, zmm0, zmm2"); // VPSUBUSB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd8, 0x0a], "vpsubusb zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSUBUSB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd8, 0xca], "vpsubusb zmm1, zmm0, zmm2"); // VPSUBUSB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd8, 0xca], "vpsubusb zmm1{k5}, zmm0, zmm2"); // VPSUBUSB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd8, 0x0a], "vpsubusb zmm1, zmm0, zmmword [rdx]"); // VPSUBUSB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd8, 0x0a], "vpsubusb zmm1{k5}, zmm0, zmmword [rdx]"); // VPSUBUSB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd8, 0xca], "vpsubusb xmm1{k5}{z}, xmm0, xmm2"); // VPSUBUSB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd8, 0x0a], "vpsubusb xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSUBUSB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd8, 0xca], "vpsubusb xmm1, xmm0, xmm2"); // VPSUBUSB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd8, 0xca], "vpsubusb xmm1{k5}, xmm0, xmm2"); // VPSUBUSB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd8, 0x0a], "vpsubusb xmm1, xmm0, xmmword [rdx]"); // VPSUBUSB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd8, 0x0a], "vpsubusb xmm1{k5}, xmm0, xmmword [rdx]"); // VPSUBUSB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd9, 0xca], "vpsubusw ymm1{k5}{z}, ymm0, ymm2"); // VPSUBUSW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xd9, 0x0a], "vpsubusw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSUBUSW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd9, 0xca], "vpsubusw ymm1, ymm0, ymm2"); // VPSUBUSW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd9, 0xca], "vpsubusw ymm1{k5}, ymm0, ymm2"); // VPSUBUSW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xd9, 0x0a], "vpsubusw ymm1, ymm0, ymmword [rdx]"); // VPSUBUSW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xd9, 0x0a], "vpsubusw ymm1{k5}, ymm0, ymmword [rdx]"); // VPSUBUSW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd9, 0xca], "vpsubusw zmm1{k5}{z}, zmm0, zmm2"); // VPSUBUSW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xd9, 0x0a], "vpsubusw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSUBUSW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd9, 0xca], "vpsubusw zmm1, zmm0, zmm2"); // VPSUBUSW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd9, 0xca], "vpsubusw zmm1{k5}, zmm0, zmm2"); // VPSUBUSW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xd9, 0x0a], "vpsubusw zmm1, zmm0, zmmword [rdx]"); // VPSUBUSW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xd9, 0x0a], "vpsubusw zmm1{k5}, zmm0, zmmword [rdx]"); // VPSUBUSW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd9, 0xca], "vpsubusw xmm1{k5}{z}, xmm0, xmm2"); // VPSUBUSW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xd9, 0x0a], "vpsubusw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSUBUSW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd9, 0xca], "vpsubusw xmm1, xmm0, xmm2"); // VPSUBUSW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd9, 0xca], "vpsubusw xmm1{k5}, xmm0, xmm2"); // VPSUBUSW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xd9, 0x0a], "vpsubusw xmm1, xmm0, xmmword [rdx]"); // VPSUBUSW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xd9, 0x0a], "vpsubusw xmm1{k5}, xmm0, xmmword [rdx]"); // VPSUBUSW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xda, 0xca], "vpminub ymm1{k5}{z}, ymm0, ymm2"); // VPMINUB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xda, 0x0a], "vpminub ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMINUB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xda, 0xca], "vpminub ymm1, ymm0, ymm2"); // VPMINUB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xda, 0xca], "vpminub ymm1{k5}, ymm0, ymm2"); // VPMINUB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xda, 0x0a], "vpminub ymm1, ymm0, ymmword [rdx]"); // VPMINUB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xda, 0x0a], "vpminub ymm1{k5}, ymm0, ymmword [rdx]"); // VPMINUB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xda, 0xca], "vpminub zmm1{k5}{z}, zmm0, zmm2"); // VPMINUB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xda, 0x0a], "vpminub zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMINUB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xda, 0xca], "vpminub zmm1, zmm0, zmm2"); // VPMINUB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xda, 0xca], "vpminub zmm1{k5}, zmm0, zmm2"); // VPMINUB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xda, 0x0a], "vpminub zmm1, zmm0, zmmword [rdx]"); // VPMINUB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xda, 0x0a], "vpminub zmm1{k5}, zmm0, zmmword [rdx]"); // VPMINUB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xda, 0xca], "vpminub xmm1{k5}{z}, xmm0, xmm2"); // VPMINUB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xda, 0x0a], "vpminub xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMINUB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xda, 0xca], "vpminub xmm1, xmm0, xmm2"); // VPMINUB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xda, 0xca], "vpminub xmm1{k5}, xmm0, xmm2"); // VPMINUB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xda, 0x0a], "vpminub xmm1, xmm0, xmmword [rdx]"); // VPMINUB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xda, 0x0a], "vpminub xmm1{k5}, xmm0, xmmword [rdx]"); // VPMINUB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xdb, 0x0a], "vpandq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPANDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xdb, 0x0a], "vpandq ymm1, ymm0, qword [rdx]{1to4}"); // VPANDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xdb, 0x0a], "vpandq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPANDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdb, 0xca], "vpandq ymm1{k5}{z}, ymm0, ymm2"); // VPANDQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdb, 0x0a], "vpandq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPANDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdb, 0xca], "vpandq ymm1, ymm0, ymm2"); // VPANDQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdb, 0xca], "vpandq ymm1{k5}, ymm0, ymm2"); // VPANDQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdb, 0x0a], "vpandq ymm1, ymm0, ymmword [rdx]"); // VPANDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdb, 0x0a], "vpandq ymm1{k5}, ymm0, ymmword [rdx]"); // VPANDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xdb, 0x0a], "vpandd ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPANDD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xdb, 0x0a], "vpandd ymm1, ymm0, dword [rdx]{1to8}"); // VPANDD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xdb, 0x0a], "vpandd ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPANDD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xdb, 0xca], "vpandd ymm1{k5}{z}, ymm0, ymm2"); // VPANDD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xdb, 0x0a], "vpandd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPANDD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xdb, 0xca], "vpandd ymm1, ymm0, ymm2"); // VPANDD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xdb, 0xca], "vpandd ymm1{k5}, ymm0, ymm2"); // VPANDD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xdb, 0x0a], "vpandd ymm1, ymm0, ymmword [rdx]"); // VPANDD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xdb, 0x0a], "vpandd ymm1{k5}, ymm0, ymmword [rdx]"); // VPANDD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xdb, 0x0a], "vpandq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPANDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xdb, 0x0a], "vpandq zmm1, zmm0, qword [rdx]{1to8}"); // VPANDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xdb, 0x0a], "vpandq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPANDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xdb, 0x0a], "vpandq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPANDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xdb, 0x0a], "vpandq xmm1, xmm0, qword [rdx]{1to2}"); // VPANDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xdb, 0x0a], "vpandq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPANDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdb, 0xca], "vpandq zmm1{k5}{z}, zmm0, zmm2"); // VPANDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdb, 0x0a], "vpandq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPANDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdb, 0xca], "vpandq zmm1, zmm0, zmm2"); // VPANDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdb, 0xca], "vpandq zmm1{k5}, zmm0, zmm2"); // VPANDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdb, 0x0a], "vpandq zmm1, zmm0, zmmword [rdx]"); // VPANDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdb, 0x0a], "vpandq zmm1{k5}, zmm0, zmmword [rdx]"); // VPANDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdb, 0xca], "vpandq xmm1{k5}{z}, xmm0, xmm2"); // VPANDQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdb, 0x0a], "vpandq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPANDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdb, 0xca], "vpandq xmm1, xmm0, xmm2"); // VPANDQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdb, 0xca], "vpandq xmm1{k5}, xmm0, xmm2"); // VPANDQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdb, 0x0a], "vpandq xmm1, xmm0, xmmword [rdx]"); // VPANDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdb, 0x0a], "vpandq xmm1{k5}, xmm0, xmmword [rdx]"); // VPANDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xdb, 0x0a], "vpandd zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPANDD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xdb, 0x0a], "vpandd zmm1, zmm0, dword [rdx]{1to16}"); // VPANDD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xdb, 0x0a], "vpandd zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPANDD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xdb, 0x0a], "vpandd xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPANDD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xdb, 0x0a], "vpandd xmm1, xmm0, dword [rdx]{1to4}"); // VPANDD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xdb, 0x0a], "vpandd xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPANDD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xdb, 0xca], "vpandd zmm1{k5}{z}, zmm0, zmm2"); // VPANDD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xdb, 0x0a], "vpandd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPANDD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xdb, 0xca], "vpandd zmm1, zmm0, zmm2"); // VPANDD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xdb, 0xca], "vpandd zmm1{k5}, zmm0, zmm2"); // VPANDD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xdb, 0x0a], "vpandd zmm1, zmm0, zmmword [rdx]"); // VPANDD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xdb, 0x0a], "vpandd zmm1{k5}, zmm0, zmmword [rdx]"); // VPANDD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xdb, 0xca], "vpandd xmm1{k5}{z}, xmm0, xmm2"); // VPANDD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xdb, 0x0a], "vpandd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPANDD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xdb, 0xca], "vpandd xmm1, xmm0, xmm2"); // VPANDD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xdb, 0xca], "vpandd xmm1{k5}, xmm0, xmm2"); // VPANDD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xdb, 0x0a], "vpandd xmm1, xmm0, xmmword [rdx]"); // VPANDD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xdb, 0x0a], "vpandd xmm1{k5}, xmm0, xmmword [rdx]"); // VPANDD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdc, 0xca], "vpaddusb ymm1{k5}{z}, ymm0, ymm2"); // VPADDUSB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdc, 0x0a], "vpaddusb ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPADDUSB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdc, 0xca], "vpaddusb ymm1, ymm0, ymm2"); // VPADDUSB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdc, 0xca], "vpaddusb ymm1{k5}, ymm0, ymm2"); // VPADDUSB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdc, 0x0a], "vpaddusb ymm1, ymm0, ymmword [rdx]"); // VPADDUSB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdc, 0x0a], "vpaddusb ymm1{k5}, ymm0, ymmword [rdx]"); // VPADDUSB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdc, 0xca], "vpaddusb zmm1{k5}{z}, zmm0, zmm2"); // VPADDUSB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdc, 0x0a], "vpaddusb zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPADDUSB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdc, 0xca], "vpaddusb zmm1, zmm0, zmm2"); // VPADDUSB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdc, 0xca], "vpaddusb zmm1{k5}, zmm0, zmm2"); // VPADDUSB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdc, 0x0a], "vpaddusb zmm1, zmm0, zmmword [rdx]"); // VPADDUSB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdc, 0x0a], "vpaddusb zmm1{k5}, zmm0, zmmword [rdx]"); // VPADDUSB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdc, 0xca], "vpaddusb xmm1{k5}{z}, xmm0, xmm2"); // VPADDUSB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdc, 0x0a], "vpaddusb xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPADDUSB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdc, 0xca], "vpaddusb xmm1, xmm0, xmm2"); // VPADDUSB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdc, 0xca], "vpaddusb xmm1{k5}, xmm0, xmm2"); // VPADDUSB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdc, 0x0a], "vpaddusb xmm1, xmm0, xmmword [rdx]"); // VPADDUSB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdc, 0x0a], "vpaddusb xmm1{k5}, xmm0, xmmword [rdx]"); // VPADDUSB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdd, 0xca], "vpaddusw ymm1{k5}{z}, ymm0, ymm2"); // VPADDUSW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdd, 0x0a], "vpaddusw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPADDUSW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdd, 0xca], "vpaddusw ymm1, ymm0, ymm2"); // VPADDUSW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdd, 0xca], "vpaddusw ymm1{k5}, ymm0, ymm2"); // VPADDUSW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdd, 0x0a], "vpaddusw ymm1, ymm0, ymmword [rdx]"); // VPADDUSW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdd, 0x0a], "vpaddusw ymm1{k5}, ymm0, ymmword [rdx]"); // VPADDUSW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdd, 0xca], "vpaddusw zmm1{k5}{z}, zmm0, zmm2"); // VPADDUSW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdd, 0x0a], "vpaddusw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPADDUSW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdd, 0xca], "vpaddusw zmm1, zmm0, zmm2"); // VPADDUSW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdd, 0xca], "vpaddusw zmm1{k5}, zmm0, zmm2"); // VPADDUSW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdd, 0x0a], "vpaddusw zmm1, zmm0, zmmword [rdx]"); // VPADDUSW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdd, 0x0a], "vpaddusw zmm1{k5}, zmm0, zmmword [rdx]"); // VPADDUSW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdd, 0xca], "vpaddusw xmm1{k5}{z}, xmm0, xmm2"); // VPADDUSW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdd, 0x0a], "vpaddusw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPADDUSW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdd, 0xca], "vpaddusw xmm1, xmm0, xmm2"); // VPADDUSW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdd, 0xca], "vpaddusw xmm1{k5}, xmm0, xmm2"); // VPADDUSW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdd, 0x0a], "vpaddusw xmm1, xmm0, xmmword [rdx]"); // VPADDUSW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdd, 0x0a], "vpaddusw xmm1{k5}, xmm0, xmmword [rdx]"); // VPADDUSW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xde, 0xca], "vpmaxub ymm1{k5}{z}, ymm0, ymm2"); // VPMAXUB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xde, 0x0a], "vpmaxub ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMAXUB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xde, 0xca], "vpmaxub ymm1, ymm0, ymm2"); // VPMAXUB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xde, 0xca], "vpmaxub ymm1{k5}, ymm0, ymm2"); // VPMAXUB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xde, 0x0a], "vpmaxub ymm1, ymm0, ymmword [rdx]"); // VPMAXUB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xde, 0x0a], "vpmaxub ymm1{k5}, ymm0, ymmword [rdx]"); // VPMAXUB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xde, 0xca], "vpmaxub zmm1{k5}{z}, zmm0, zmm2"); // VPMAXUB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xde, 0x0a], "vpmaxub zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMAXUB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xde, 0xca], "vpmaxub zmm1, zmm0, zmm2"); // VPMAXUB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xde, 0xca], "vpmaxub zmm1{k5}, zmm0, zmm2"); // VPMAXUB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xde, 0x0a], "vpmaxub zmm1, zmm0, zmmword [rdx]"); // VPMAXUB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xde, 0x0a], "vpmaxub zmm1{k5}, zmm0, zmmword [rdx]"); // VPMAXUB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xde, 0xca], "vpmaxub xmm1{k5}{z}, xmm0, xmm2"); // VPMAXUB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xde, 0x0a], "vpmaxub xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMAXUB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xde, 0xca], "vpmaxub xmm1, xmm0, xmm2"); // VPMAXUB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xde, 0xca], "vpmaxub xmm1{k5}, xmm0, xmm2"); // VPMAXUB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xde, 0x0a], "vpmaxub xmm1, xmm0, xmmword [rdx]"); // VPMAXUB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xde, 0x0a], "vpmaxub xmm1{k5}, xmm0, xmmword [rdx]"); // VPMAXUB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xdf, 0x0a], "vpandnq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPANDNQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xdf, 0x0a], "vpandnq ymm1, ymm0, qword [rdx]{1to4}"); // VPANDNQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xdf, 0x0a], "vpandnq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPANDNQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdf, 0xca], "vpandnq ymm1{k5}{z}, ymm0, ymm2"); // VPANDNQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xdf, 0x0a], "vpandnq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPANDNQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdf, 0xca], "vpandnq ymm1, ymm0, ymm2"); // VPANDNQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdf, 0xca], "vpandnq ymm1{k5}, ymm0, ymm2"); // VPANDNQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xdf, 0x0a], "vpandnq ymm1, ymm0, ymmword [rdx]"); // VPANDNQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xdf, 0x0a], "vpandnq ymm1{k5}, ymm0, ymmword [rdx]"); // VPANDNQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xdf, 0x0a], "vpandnd ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPANDND_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xdf, 0x0a], "vpandnd ymm1, ymm0, dword [rdx]{1to8}"); // VPANDND_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xdf, 0x0a], "vpandnd ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPANDND_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xdf, 0xca], "vpandnd ymm1{k5}{z}, ymm0, ymm2"); // VPANDND_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xdf, 0x0a], "vpandnd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPANDND_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xdf, 0xca], "vpandnd ymm1, ymm0, ymm2"); // VPANDND_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xdf, 0xca], "vpandnd ymm1{k5}, ymm0, ymm2"); // VPANDND_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xdf, 0x0a], "vpandnd ymm1, ymm0, ymmword [rdx]"); // VPANDND_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xdf, 0x0a], "vpandnd ymm1{k5}, ymm0, ymmword [rdx]"); // VPANDND_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xdf, 0x0a], "vpandnq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPANDNQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xdf, 0x0a], "vpandnq zmm1, zmm0, qword [rdx]{1to8}"); // VPANDNQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xdf, 0x0a], "vpandnq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPANDNQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xdf, 0x0a], "vpandnq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPANDNQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xdf, 0x0a], "vpandnq xmm1, xmm0, qword [rdx]{1to2}"); // VPANDNQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xdf, 0x0a], "vpandnq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPANDNQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdf, 0xca], "vpandnq zmm1{k5}{z}, zmm0, zmm2"); // VPANDNQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xdf, 0x0a], "vpandnq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPANDNQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdf, 0xca], "vpandnq zmm1, zmm0, zmm2"); // VPANDNQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdf, 0xca], "vpandnq zmm1{k5}, zmm0, zmm2"); // VPANDNQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xdf, 0x0a], "vpandnq zmm1, zmm0, zmmword [rdx]"); // VPANDNQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xdf, 0x0a], "vpandnq zmm1{k5}, zmm0, zmmword [rdx]"); // VPANDNQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdf, 0xca], "vpandnq xmm1{k5}{z}, xmm0, xmm2"); // VPANDNQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xdf, 0x0a], "vpandnq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPANDNQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdf, 0xca], "vpandnq xmm1, xmm0, xmm2"); // VPANDNQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdf, 0xca], "vpandnq xmm1{k5}, xmm0, xmm2"); // VPANDNQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xdf, 0x0a], "vpandnq xmm1, xmm0, xmmword [rdx]"); // VPANDNQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xdf, 0x0a], "vpandnq xmm1{k5}, xmm0, xmmword [rdx]"); // VPANDNQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xdf, 0x0a], "vpandnd zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPANDND_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xdf, 0x0a], "vpandnd zmm1, zmm0, dword [rdx]{1to16}"); // VPANDND_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xdf, 0x0a], "vpandnd zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPANDND_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xdf, 0x0a], "vpandnd xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPANDND_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xdf, 0x0a], "vpandnd xmm1, xmm0, dword [rdx]{1to4}"); // VPANDND_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xdf, 0x0a], "vpandnd xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPANDND_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xdf, 0xca], "vpandnd zmm1{k5}{z}, zmm0, zmm2"); // VPANDND_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xdf, 0x0a], "vpandnd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPANDND_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xdf, 0xca], "vpandnd zmm1, zmm0, zmm2"); // VPANDND_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xdf, 0xca], "vpandnd zmm1{k5}, zmm0, zmm2"); // VPANDND_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xdf, 0x0a], "vpandnd zmm1, zmm0, zmmword [rdx]"); // VPANDND_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xdf, 0x0a], "vpandnd zmm1{k5}, zmm0, zmmword [rdx]"); // VPANDND_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xdf, 0xca], "vpandnd xmm1{k5}{z}, xmm0, xmm2"); // VPANDND_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xdf, 0x0a], "vpandnd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPANDND_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xdf, 0xca], "vpandnd xmm1, xmm0, xmm2"); // VPANDND_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xdf, 0xca], "vpandnd xmm1{k5}, xmm0, xmm2"); // VPANDND_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xdf, 0x0a], "vpandnd xmm1, xmm0, xmmword [rdx]"); // VPANDND_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xdf, 0x0a], "vpandnd xmm1{k5}, xmm0, xmmword [rdx]"); // VPANDND_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe0, 0xca], "vpavgb ymm1{k5}{z}, ymm0, ymm2"); // VPAVGB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe0, 0x0a], "vpavgb ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPAVGB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe0, 0xca], "vpavgb ymm1, ymm0, ymm2"); // VPAVGB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe0, 0xca], "vpavgb ymm1{k5}, ymm0, ymm2"); // VPAVGB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe0, 0x0a], "vpavgb ymm1, ymm0, ymmword [rdx]"); // VPAVGB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe0, 0x0a], "vpavgb ymm1{k5}, ymm0, ymmword [rdx]"); // VPAVGB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe0, 0xca], "vpavgb zmm1{k5}{z}, zmm0, zmm2"); // VPAVGB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe0, 0x0a], "vpavgb zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPAVGB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe0, 0xca], "vpavgb zmm1, zmm0, zmm2"); // VPAVGB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe0, 0xca], "vpavgb zmm1{k5}, zmm0, zmm2"); // VPAVGB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe0, 0x0a], "vpavgb zmm1, zmm0, zmmword [rdx]"); // VPAVGB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe0, 0x0a], "vpavgb zmm1{k5}, zmm0, zmmword [rdx]"); // VPAVGB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe0, 0xca], "vpavgb xmm1{k5}{z}, xmm0, xmm2"); // VPAVGB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe0, 0x0a], "vpavgb xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPAVGB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe0, 0xca], "vpavgb xmm1, xmm0, xmm2"); // VPAVGB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe0, 0xca], "vpavgb xmm1{k5}, xmm0, xmm2"); // VPAVGB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe0, 0x0a], "vpavgb xmm1, xmm0, xmmword [rdx]"); // VPAVGB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe0, 0x0a], "vpavgb xmm1{k5}, xmm0, xmmword [rdx]"); // VPAVGB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe1, 0xca], "vpsraw ymm1{k5}{z}, ymm0, xmm2"); // VPSRAW_YMMu16_MASKmskw_YMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe1, 0x0a], "vpsraw ymm1{k5}{z}, ymm0, xmmword [rdx]"); // VPSRAW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe1, 0xca], "vpsraw ymm1, ymm0, xmm2"); // VPSRAW_YMMu16_MASKmskw_YMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe1, 0xca], "vpsraw ymm1{k5}, ymm0, xmm2"); // VPSRAW_YMMu16_MASKmskw_YMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe1, 0x0a], "vpsraw ymm1, ymm0, xmmword [rdx]"); // VPSRAW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe1, 0x0a], "vpsraw ymm1{k5}, ymm0, xmmword [rdx]"); // VPSRAW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe1, 0xca], "vpsraw zmm1{k5}{z}, zmm0, xmm2"); // VPSRAW_ZMMu16_MASKmskw_ZMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe1, 0x0a], "vpsraw zmm1{k5}{z}, zmm0, xmmword [rdx]"); // VPSRAW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe1, 0xca], "vpsraw zmm1, zmm0, xmm2"); // VPSRAW_ZMMu16_MASKmskw_ZMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe1, 0xca], "vpsraw zmm1{k5}, zmm0, xmm2"); // VPSRAW_ZMMu16_MASKmskw_ZMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe1, 0x0a], "vpsraw zmm1, zmm0, xmmword [rdx]"); // VPSRAW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe1, 0x0a], "vpsraw zmm1{k5}, zmm0, xmmword [rdx]"); // VPSRAW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe1, 0xca], "vpsraw xmm1{k5}{z}, xmm0, xmm2"); // VPSRAW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe1, 0x0a], "vpsraw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSRAW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe1, 0xca], "vpsraw xmm1, xmm0, xmm2"); // VPSRAW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe1, 0xca], "vpsraw xmm1{k5}, xmm0, xmm2"); // VPSRAW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe1, 0x0a], "vpsraw xmm1, xmm0, xmmword [rdx]"); // VPSRAW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe1, 0x0a], "vpsraw xmm1{k5}, xmm0, xmmword [rdx]"); // VPSRAW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe2, 0xca], "vpsraq ymm1{k5}{z}, ymm0, xmm2"); // VPSRAQ_YMMu64_MASKmskw_YMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe2, 0x0a], "vpsraq ymm1{k5}{z}, ymm0, xmmword [rdx]"); // VPSRAQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe2, 0xca], "vpsraq ymm1, ymm0, xmm2"); // VPSRAQ_YMMu64_MASKmskw_YMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe2, 0xca], "vpsraq ymm1{k5}, ymm0, xmm2"); // VPSRAQ_YMMu64_MASKmskw_YMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe2, 0x0a], "vpsraq ymm1, ymm0, xmmword [rdx]"); // VPSRAQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe2, 0x0a], "vpsraq ymm1{k5}, ymm0, xmmword [rdx]"); // VPSRAQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xe2, 0xca], "vpsrad ymm1{k5}{z}, ymm0, xmm2"); // VPSRAD_YMMu32_MASKmskw_YMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xe2, 0x0a], "vpsrad ymm1{k5}{z}, ymm0, xmmword [rdx]"); // VPSRAD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xe2, 0xca], "vpsrad ymm1, ymm0, xmm2"); // VPSRAD_YMMu32_MASKmskw_YMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xe2, 0xca], "vpsrad ymm1{k5}, ymm0, xmm2"); // VPSRAD_YMMu32_MASKmskw_YMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xe2, 0x0a], "vpsrad ymm1, ymm0, xmmword [rdx]"); // VPSRAD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xe2, 0x0a], "vpsrad ymm1{k5}, ymm0, xmmword [rdx]"); // VPSRAD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe2, 0xca], "vpsraq zmm1{k5}{z}, zmm0, xmm2"); // VPSRAQ_ZMMu64_MASKmskw_ZMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe2, 0x0a], "vpsraq zmm1{k5}{z}, zmm0, xmmword [rdx]"); // VPSRAQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe2, 0xca], "vpsraq zmm1, zmm0, xmm2"); // VPSRAQ_ZMMu64_MASKmskw_ZMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe2, 0xca], "vpsraq zmm1{k5}, zmm0, xmm2"); // VPSRAQ_ZMMu64_MASKmskw_ZMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe2, 0x0a], "vpsraq zmm1, zmm0, xmmword [rdx]"); // VPSRAQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe2, 0x0a], "vpsraq zmm1{k5}, zmm0, xmmword [rdx]"); // VPSRAQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe2, 0xca], "vpsraq xmm1{k5}{z}, xmm0, xmm2"); // VPSRAQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe2, 0x0a], "vpsraq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSRAQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe2, 0xca], "vpsraq xmm1, xmm0, xmm2"); // VPSRAQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe2, 0xca], "vpsraq xmm1{k5}, xmm0, xmm2"); // VPSRAQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe2, 0x0a], "vpsraq xmm1, xmm0, xmmword [rdx]"); // VPSRAQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe2, 0x0a], "vpsraq xmm1{k5}, xmm0, xmmword [rdx]"); // VPSRAQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xe2, 0xca], "vpsrad zmm1{k5}{z}, zmm0, xmm2"); // VPSRAD_ZMMu32_MASKmskw_ZMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xe2, 0x0a], "vpsrad zmm1{k5}{z}, zmm0, xmmword [rdx]"); // VPSRAD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xe2, 0xca], "vpsrad zmm1, zmm0, xmm2"); // VPSRAD_ZMMu32_MASKmskw_ZMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xe2, 0xca], "vpsrad zmm1{k5}, zmm0, xmm2"); // VPSRAD_ZMMu32_MASKmskw_ZMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xe2, 0x0a], "vpsrad zmm1, zmm0, xmmword [rdx]"); // VPSRAD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xe2, 0x0a], "vpsrad zmm1{k5}, zmm0, xmmword [rdx]"); // VPSRAD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xe2, 0xca], "vpsrad xmm1{k5}{z}, xmm0, xmm2"); // VPSRAD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xe2, 0x0a], "vpsrad xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSRAD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xe2, 0xca], "vpsrad xmm1, xmm0, xmm2"); // VPSRAD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xe2, 0xca], "vpsrad xmm1{k5}, xmm0, xmm2"); // VPSRAD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xe2, 0x0a], "vpsrad xmm1, xmm0, xmmword [rdx]"); // VPSRAD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xe2, 0x0a], "vpsrad xmm1{k5}, xmm0, xmmword [rdx]"); // VPSRAD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe3, 0xca], "vpavgw ymm1{k5}{z}, ymm0, ymm2"); // VPAVGW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe3, 0x0a], "vpavgw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPAVGW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe3, 0xca], "vpavgw ymm1, ymm0, ymm2"); // VPAVGW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe3, 0xca], "vpavgw ymm1{k5}, ymm0, ymm2"); // VPAVGW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe3, 0x0a], "vpavgw ymm1, ymm0, ymmword [rdx]"); // VPAVGW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe3, 0x0a], "vpavgw ymm1{k5}, ymm0, ymmword [rdx]"); // VPAVGW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe3, 0xca], "vpavgw zmm1{k5}{z}, zmm0, zmm2"); // VPAVGW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe3, 0x0a], "vpavgw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPAVGW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe3, 0xca], "vpavgw zmm1, zmm0, zmm2"); // VPAVGW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe3, 0xca], "vpavgw zmm1{k5}, zmm0, zmm2"); // VPAVGW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe3, 0x0a], "vpavgw zmm1, zmm0, zmmword [rdx]"); // VPAVGW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe3, 0x0a], "vpavgw zmm1{k5}, zmm0, zmmword [rdx]"); // VPAVGW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe3, 0xca], "vpavgw xmm1{k5}{z}, xmm0, xmm2"); // VPAVGW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe3, 0x0a], "vpavgw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPAVGW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe3, 0xca], "vpavgw xmm1, xmm0, xmm2"); // VPAVGW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe3, 0xca], "vpavgw xmm1{k5}, xmm0, xmm2"); // VPAVGW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe3, 0x0a], "vpavgw xmm1, xmm0, xmmword [rdx]"); // VPAVGW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe3, 0x0a], "vpavgw xmm1{k5}, xmm0, xmmword [rdx]"); // VPAVGW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe4, 0xca], "vpmulhuw ymm1{k5}{z}, ymm0, ymm2"); // VPMULHUW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe4, 0x0a], "vpmulhuw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMULHUW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe4, 0xca], "vpmulhuw ymm1, ymm0, ymm2"); // VPMULHUW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe4, 0xca], "vpmulhuw ymm1{k5}, ymm0, ymm2"); // VPMULHUW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe4, 0x0a], "vpmulhuw ymm1, ymm0, ymmword [rdx]"); // VPMULHUW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe4, 0x0a], "vpmulhuw ymm1{k5}, ymm0, ymmword [rdx]"); // VPMULHUW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe4, 0xca], "vpmulhuw zmm1{k5}{z}, zmm0, zmm2"); // VPMULHUW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe4, 0x0a], "vpmulhuw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMULHUW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe4, 0xca], "vpmulhuw zmm1, zmm0, zmm2"); // VPMULHUW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe4, 0xca], "vpmulhuw zmm1{k5}, zmm0, zmm2"); // VPMULHUW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe4, 0x0a], "vpmulhuw zmm1, zmm0, zmmword [rdx]"); // VPMULHUW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe4, 0x0a], "vpmulhuw zmm1{k5}, zmm0, zmmword [rdx]"); // VPMULHUW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe4, 0xca], "vpmulhuw xmm1{k5}{z}, xmm0, xmm2"); // VPMULHUW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe4, 0x0a], "vpmulhuw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMULHUW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe4, 0xca], "vpmulhuw xmm1, xmm0, xmm2"); // VPMULHUW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe4, 0xca], "vpmulhuw xmm1{k5}, xmm0, xmm2"); // VPMULHUW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe4, 0x0a], "vpmulhuw xmm1, xmm0, xmmword [rdx]"); // VPMULHUW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe4, 0x0a], "vpmulhuw xmm1{k5}, xmm0, xmmword [rdx]"); // VPMULHUW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe5, 0xca], "vpmulhw ymm1{k5}{z}, ymm0, ymm2"); // VPMULHW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe5, 0x0a], "vpmulhw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMULHW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe5, 0xca], "vpmulhw ymm1, ymm0, ymm2"); // VPMULHW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe5, 0xca], "vpmulhw ymm1{k5}, ymm0, ymm2"); // VPMULHW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe5, 0x0a], "vpmulhw ymm1, ymm0, ymmword [rdx]"); // VPMULHW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe5, 0x0a], "vpmulhw ymm1{k5}, ymm0, ymmword [rdx]"); // VPMULHW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe5, 0xca], "vpmulhw zmm1{k5}{z}, zmm0, zmm2"); // VPMULHW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe5, 0x0a], "vpmulhw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMULHW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe5, 0xca], "vpmulhw zmm1, zmm0, zmm2"); // VPMULHW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe5, 0xca], "vpmulhw zmm1{k5}, zmm0, zmm2"); // VPMULHW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe5, 0x0a], "vpmulhw zmm1, zmm0, zmmword [rdx]"); // VPMULHW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe5, 0x0a], "vpmulhw zmm1{k5}, zmm0, zmmword [rdx]"); // VPMULHW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe5, 0xca], "vpmulhw xmm1{k5}{z}, xmm0, xmm2"); // VPMULHW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe5, 0x0a], "vpmulhw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMULHW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe5, 0xca], "vpmulhw xmm1, xmm0, xmm2"); // VPMULHW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe5, 0xca], "vpmulhw xmm1{k5}, xmm0, xmm2"); // VPMULHW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe5, 0x0a], "vpmulhw xmm1, xmm0, xmmword [rdx]"); // VPMULHW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe5, 0x0a], "vpmulhw xmm1{k5}, xmm0, xmmword [rdx]"); // VPMULHW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xfd, 0xe6, 0xca], "vcvttpd2dq ymm1{k5}{z}{sae}, zmm2"); // VCVTTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x78, 0xe6, 0xca], "vcvttpd2dq ymm1{sae}, zmm2"); // VCVTTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x7d, 0xe6, 0xca], "vcvttpd2dq ymm1{k5}{sae}, zmm2"); // VCVTTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}{z}, qword [rdx]{1to4}"); // VCVTTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xe6, 0x0a], "vcvttpd2dq xmm1, qword [rdx]{1to4}"); // VCVTTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}, qword [rdx]{1to4}"); // VCVTTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe6, 0xca], "vcvttpd2dq xmm1{k5}{z}, ymm2"); // VCVTTPD2DQ_XMMi32_MASKmskw_YMMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}{z}, ymmword [rdx]"); // VCVTTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe6, 0xca], "vcvttpd2dq xmm1, ymm2"); // VCVTTPD2DQ_XMMi32_MASKmskw_YMMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe6, 0xca], "vcvttpd2dq xmm1{k5}, ymm2"); // VCVTTPD2DQ_XMMi32_MASKmskw_YMMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe6, 0x0a], "vcvttpd2dq xmm1, ymmword [rdx]"); // VCVTTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}, ymmword [rdx]"); // VCVTTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xe6, 0x0a], "vcvttpd2dq ymm1{k5}{z}, qword [rdx]{1to8}"); // VCVTTPD2DQ_YMMi32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xe6, 0x0a], "vcvttpd2dq ymm1, qword [rdx]{1to8}"); // VCVTTPD2DQ_YMMi32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xe6, 0x0a], "vcvttpd2dq ymm1{k5}, qword [rdx]{1to8}"); // VCVTTPD2DQ_YMMi32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}{z}, qword [rdx]{1to2}"); // VCVTTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xe6, 0x0a], "vcvttpd2dq xmm1, qword [rdx]{1to2}"); // VCVTTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}, qword [rdx]{1to2}"); // VCVTTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe6, 0xca], "vcvttpd2dq ymm1{k5}{z}, zmm2"); // VCVTTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe6, 0x0a], "vcvttpd2dq ymm1{k5}{z}, zmmword [rdx]"); // VCVTTPD2DQ_YMMi32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe6, 0xca], "vcvttpd2dq ymm1, zmm2"); // VCVTTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe6, 0xca], "vcvttpd2dq ymm1{k5}, zmm2"); // VCVTTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe6, 0x0a], "vcvttpd2dq ymm1, zmmword [rdx]"); // VCVTTPD2DQ_YMMi32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe6, 0x0a], "vcvttpd2dq ymm1{k5}, zmmword [rdx]"); // VCVTTPD2DQ_YMMi32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe6, 0xca], "vcvttpd2dq xmm1{k5}{z}, xmm2"); // VCVTTPD2DQ_XMMi32_MASKmskw_XMMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}{z}, xmmword [rdx]"); // VCVTTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe6, 0xca], "vcvttpd2dq xmm1, xmm2"); // VCVTTPD2DQ_XMMi32_MASKmskw_XMMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe6, 0xca], "vcvttpd2dq xmm1{k5}, xmm2"); // VCVTTPD2DQ_XMMi32_MASKmskw_XMMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe6, 0x0a], "vcvttpd2dq xmm1, xmmword [rdx]"); // VCVTTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe6, 0x0a], "vcvttpd2dq xmm1{k5}, xmmword [rdx]"); // VCVTTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xe7, 0x0a], "vmovntdq ymmword [rdx], ymm1"); // VMOVNTDQ_MEMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xe7, 0x0a], "vmovntdq zmmword [rdx], zmm1"); // VMOVNTDQ_MEMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xe7, 0x0a], "vmovntdq xmmword [rdx], xmm1"); // VMOVNTDQ_MEMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe8, 0xca], "vpsubsb ymm1{k5}{z}, ymm0, ymm2"); // VPSUBSB_YMMi8_MASKmskw_YMMi8_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe8, 0x0a], "vpsubsb ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSUBSB_YMMi8_MASKmskw_YMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe8, 0xca], "vpsubsb ymm1, ymm0, ymm2"); // VPSUBSB_YMMi8_MASKmskw_YMMi8_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe8, 0xca], "vpsubsb ymm1{k5}, ymm0, ymm2"); // VPSUBSB_YMMi8_MASKmskw_YMMi8_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe8, 0x0a], "vpsubsb ymm1, ymm0, ymmword [rdx]"); // VPSUBSB_YMMi8_MASKmskw_YMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe8, 0x0a], "vpsubsb ymm1{k5}, ymm0, ymmword [rdx]"); // VPSUBSB_YMMi8_MASKmskw_YMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe8, 0xca], "vpsubsb zmm1{k5}{z}, zmm0, zmm2"); // VPSUBSB_ZMMi8_MASKmskw_ZMMi8_ZMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe8, 0x0a], "vpsubsb zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSUBSB_ZMMi8_MASKmskw_ZMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe8, 0xca], "vpsubsb zmm1, zmm0, zmm2"); // VPSUBSB_ZMMi8_MASKmskw_ZMMi8_ZMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe8, 0xca], "vpsubsb zmm1{k5}, zmm0, zmm2"); // VPSUBSB_ZMMi8_MASKmskw_ZMMi8_ZMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe8, 0x0a], "vpsubsb zmm1, zmm0, zmmword [rdx]"); // VPSUBSB_ZMMi8_MASKmskw_ZMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe8, 0x0a], "vpsubsb zmm1{k5}, zmm0, zmmword [rdx]"); // VPSUBSB_ZMMi8_MASKmskw_ZMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe8, 0xca], "vpsubsb xmm1{k5}{z}, xmm0, xmm2"); // VPSUBSB_XMMi8_MASKmskw_XMMi8_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe8, 0x0a], "vpsubsb xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSUBSB_XMMi8_MASKmskw_XMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe8, 0xca], "vpsubsb xmm1, xmm0, xmm2"); // VPSUBSB_XMMi8_MASKmskw_XMMi8_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe8, 0xca], "vpsubsb xmm1{k5}, xmm0, xmm2"); // VPSUBSB_XMMi8_MASKmskw_XMMi8_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe8, 0x0a], "vpsubsb xmm1, xmm0, xmmword [rdx]"); // VPSUBSB_XMMi8_MASKmskw_XMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe8, 0x0a], "vpsubsb xmm1{k5}, xmm0, xmmword [rdx]"); // VPSUBSB_XMMi8_MASKmskw_XMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe9, 0xca], "vpsubsw ymm1{k5}{z}, ymm0, ymm2"); // VPSUBSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xe9, 0x0a], "vpsubsw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSUBSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe9, 0xca], "vpsubsw ymm1, ymm0, ymm2"); // VPSUBSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe9, 0xca], "vpsubsw ymm1{k5}, ymm0, ymm2"); // VPSUBSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xe9, 0x0a], "vpsubsw ymm1, ymm0, ymmword [rdx]"); // VPSUBSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xe9, 0x0a], "vpsubsw ymm1{k5}, ymm0, ymmword [rdx]"); // VPSUBSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe9, 0xca], "vpsubsw zmm1{k5}{z}, zmm0, zmm2"); // VPSUBSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xe9, 0x0a], "vpsubsw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSUBSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe9, 0xca], "vpsubsw zmm1, zmm0, zmm2"); // VPSUBSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe9, 0xca], "vpsubsw zmm1{k5}, zmm0, zmm2"); // VPSUBSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xe9, 0x0a], "vpsubsw zmm1, zmm0, zmmword [rdx]"); // VPSUBSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xe9, 0x0a], "vpsubsw zmm1{k5}, zmm0, zmmword [rdx]"); // VPSUBSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe9, 0xca], "vpsubsw xmm1{k5}{z}, xmm0, xmm2"); // VPSUBSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xe9, 0x0a], "vpsubsw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSUBSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe9, 0xca], "vpsubsw xmm1, xmm0, xmm2"); // VPSUBSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe9, 0xca], "vpsubsw xmm1{k5}, xmm0, xmm2"); // VPSUBSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xe9, 0x0a], "vpsubsw xmm1, xmm0, xmmword [rdx]"); // VPSUBSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xe9, 0x0a], "vpsubsw xmm1{k5}, xmm0, xmmword [rdx]"); // VPSUBSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xea, 0xca], "vpminsw ymm1{k5}{z}, ymm0, ymm2"); // VPMINSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xea, 0x0a], "vpminsw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMINSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xea, 0xca], "vpminsw ymm1, ymm0, ymm2"); // VPMINSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xea, 0xca], "vpminsw ymm1{k5}, ymm0, ymm2"); // VPMINSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xea, 0x0a], "vpminsw ymm1, ymm0, ymmword [rdx]"); // VPMINSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xea, 0x0a], "vpminsw ymm1{k5}, ymm0, ymmword [rdx]"); // VPMINSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xea, 0xca], "vpminsw zmm1{k5}{z}, zmm0, zmm2"); // VPMINSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xea, 0x0a], "vpminsw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMINSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xea, 0xca], "vpminsw zmm1, zmm0, zmm2"); // VPMINSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xea, 0xca], "vpminsw zmm1{k5}, zmm0, zmm2"); // VPMINSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xea, 0x0a], "vpminsw zmm1, zmm0, zmmword [rdx]"); // VPMINSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xea, 0x0a], "vpminsw zmm1{k5}, zmm0, zmmword [rdx]"); // VPMINSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xea, 0xca], "vpminsw xmm1{k5}{z}, xmm0, xmm2"); // VPMINSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xea, 0x0a], "vpminsw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMINSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xea, 0xca], "vpminsw xmm1, xmm0, xmm2"); // VPMINSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xea, 0xca], "vpminsw xmm1{k5}, xmm0, xmm2"); // VPMINSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xea, 0x0a], "vpminsw xmm1, xmm0, xmmword [rdx]"); // VPMINSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xea, 0x0a], "vpminsw xmm1{k5}, xmm0, xmmword [rdx]"); // VPMINSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xeb, 0x0a], "vporq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPORQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xeb, 0x0a], "vporq ymm1, ymm0, qword [rdx]{1to4}"); // VPORQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xeb, 0x0a], "vporq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPORQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xeb, 0xca], "vporq ymm1{k5}{z}, ymm0, ymm2"); // VPORQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xeb, 0x0a], "vporq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPORQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xeb, 0xca], "vporq ymm1, ymm0, ymm2"); // VPORQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xeb, 0xca], "vporq ymm1{k5}, ymm0, ymm2"); // VPORQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xeb, 0x0a], "vporq ymm1, ymm0, ymmword [rdx]"); // VPORQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xeb, 0x0a], "vporq ymm1{k5}, ymm0, ymmword [rdx]"); // VPORQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xeb, 0x0a], "vpord ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPORD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xeb, 0x0a], "vpord ymm1, ymm0, dword [rdx]{1to8}"); // VPORD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xeb, 0x0a], "vpord ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPORD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xeb, 0xca], "vpord ymm1{k5}{z}, ymm0, ymm2"); // VPORD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xeb, 0x0a], "vpord ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPORD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xeb, 0xca], "vpord ymm1, ymm0, ymm2"); // VPORD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xeb, 0xca], "vpord ymm1{k5}, ymm0, ymm2"); // VPORD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xeb, 0x0a], "vpord ymm1, ymm0, ymmword [rdx]"); // VPORD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xeb, 0x0a], "vpord ymm1{k5}, ymm0, ymmword [rdx]"); // VPORD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xeb, 0x0a], "vporq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPORQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xeb, 0x0a], "vporq zmm1, zmm0, qword [rdx]{1to8}"); // VPORQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xeb, 0x0a], "vporq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPORQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xeb, 0x0a], "vporq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPORQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xeb, 0x0a], "vporq xmm1, xmm0, qword [rdx]{1to2}"); // VPORQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xeb, 0x0a], "vporq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPORQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xeb, 0xca], "vporq zmm1{k5}{z}, zmm0, zmm2"); // VPORQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xeb, 0x0a], "vporq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPORQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xeb, 0xca], "vporq zmm1, zmm0, zmm2"); // VPORQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xeb, 0xca], "vporq zmm1{k5}, zmm0, zmm2"); // VPORQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xeb, 0x0a], "vporq zmm1, zmm0, zmmword [rdx]"); // VPORQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xeb, 0x0a], "vporq zmm1{k5}, zmm0, zmmword [rdx]"); // VPORQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xeb, 0xca], "vporq xmm1{k5}{z}, xmm0, xmm2"); // VPORQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xeb, 0x0a], "vporq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPORQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xeb, 0xca], "vporq xmm1, xmm0, xmm2"); // VPORQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xeb, 0xca], "vporq xmm1{k5}, xmm0, xmm2"); // VPORQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xeb, 0x0a], "vporq xmm1, xmm0, xmmword [rdx]"); // VPORQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xeb, 0x0a], "vporq xmm1{k5}, xmm0, xmmword [rdx]"); // VPORQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xeb, 0x0a], "vpord zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPORD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xeb, 0x0a], "vpord zmm1, zmm0, dword [rdx]{1to16}"); // VPORD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xeb, 0x0a], "vpord zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPORD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xeb, 0x0a], "vpord xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPORD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xeb, 0x0a], "vpord xmm1, xmm0, dword [rdx]{1to4}"); // VPORD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xeb, 0x0a], "vpord xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPORD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xeb, 0xca], "vpord zmm1{k5}{z}, zmm0, zmm2"); // VPORD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xeb, 0x0a], "vpord zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPORD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xeb, 0xca], "vpord zmm1, zmm0, zmm2"); // VPORD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xeb, 0xca], "vpord zmm1{k5}, zmm0, zmm2"); // VPORD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xeb, 0x0a], "vpord zmm1, zmm0, zmmword [rdx]"); // VPORD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xeb, 0x0a], "vpord zmm1{k5}, zmm0, zmmword [rdx]"); // VPORD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xeb, 0xca], "vpord xmm1{k5}{z}, xmm0, xmm2"); // VPORD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xeb, 0x0a], "vpord xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPORD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xeb, 0xca], "vpord xmm1, xmm0, xmm2"); // VPORD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xeb, 0xca], "vpord xmm1{k5}, xmm0, xmm2"); // VPORD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xeb, 0x0a], "vpord xmm1, xmm0, xmmword [rdx]"); // VPORD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xeb, 0x0a], "vpord xmm1{k5}, xmm0, xmmword [rdx]"); // VPORD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xec, 0xca], "vpaddsb ymm1{k5}{z}, ymm0, ymm2"); // VPADDSB_YMMi8_MASKmskw_YMMi8_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xec, 0x0a], "vpaddsb ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPADDSB_YMMi8_MASKmskw_YMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xec, 0xca], "vpaddsb ymm1, ymm0, ymm2"); // VPADDSB_YMMi8_MASKmskw_YMMi8_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xec, 0xca], "vpaddsb ymm1{k5}, ymm0, ymm2"); // VPADDSB_YMMi8_MASKmskw_YMMi8_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xec, 0x0a], "vpaddsb ymm1, ymm0, ymmword [rdx]"); // VPADDSB_YMMi8_MASKmskw_YMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xec, 0x0a], "vpaddsb ymm1{k5}, ymm0, ymmword [rdx]"); // VPADDSB_YMMi8_MASKmskw_YMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xec, 0xca], "vpaddsb zmm1{k5}{z}, zmm0, zmm2"); // VPADDSB_ZMMi8_MASKmskw_ZMMi8_ZMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xec, 0x0a], "vpaddsb zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPADDSB_ZMMi8_MASKmskw_ZMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xec, 0xca], "vpaddsb zmm1, zmm0, zmm2"); // VPADDSB_ZMMi8_MASKmskw_ZMMi8_ZMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xec, 0xca], "vpaddsb zmm1{k5}, zmm0, zmm2"); // VPADDSB_ZMMi8_MASKmskw_ZMMi8_ZMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xec, 0x0a], "vpaddsb zmm1, zmm0, zmmword [rdx]"); // VPADDSB_ZMMi8_MASKmskw_ZMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xec, 0x0a], "vpaddsb zmm1{k5}, zmm0, zmmword [rdx]"); // VPADDSB_ZMMi8_MASKmskw_ZMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xec, 0xca], "vpaddsb xmm1{k5}{z}, xmm0, xmm2"); // VPADDSB_XMMi8_MASKmskw_XMMi8_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xec, 0x0a], "vpaddsb xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPADDSB_XMMi8_MASKmskw_XMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xec, 0xca], "vpaddsb xmm1, xmm0, xmm2"); // VPADDSB_XMMi8_MASKmskw_XMMi8_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xec, 0xca], "vpaddsb xmm1{k5}, xmm0, xmm2"); // VPADDSB_XMMi8_MASKmskw_XMMi8_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xec, 0x0a], "vpaddsb xmm1, xmm0, xmmword [rdx]"); // VPADDSB_XMMi8_MASKmskw_XMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xec, 0x0a], "vpaddsb xmm1{k5}, xmm0, xmmword [rdx]"); // VPADDSB_XMMi8_MASKmskw_XMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xed, 0xca], "vpaddsw ymm1{k5}{z}, ymm0, ymm2"); // VPADDSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xed, 0x0a], "vpaddsw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPADDSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xed, 0xca], "vpaddsw ymm1, ymm0, ymm2"); // VPADDSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xed, 0xca], "vpaddsw ymm1{k5}, ymm0, ymm2"); // VPADDSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xed, 0x0a], "vpaddsw ymm1, ymm0, ymmword [rdx]"); // VPADDSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xed, 0x0a], "vpaddsw ymm1{k5}, ymm0, ymmword [rdx]"); // VPADDSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xed, 0xca], "vpaddsw zmm1{k5}{z}, zmm0, zmm2"); // VPADDSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xed, 0x0a], "vpaddsw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPADDSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xed, 0xca], "vpaddsw zmm1, zmm0, zmm2"); // VPADDSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xed, 0xca], "vpaddsw zmm1{k5}, zmm0, zmm2"); // VPADDSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xed, 0x0a], "vpaddsw zmm1, zmm0, zmmword [rdx]"); // VPADDSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xed, 0x0a], "vpaddsw zmm1{k5}, zmm0, zmmword [rdx]"); // VPADDSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xed, 0xca], "vpaddsw xmm1{k5}{z}, xmm0, xmm2"); // VPADDSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xed, 0x0a], "vpaddsw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPADDSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xed, 0xca], "vpaddsw xmm1, xmm0, xmm2"); // VPADDSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xed, 0xca], "vpaddsw xmm1{k5}, xmm0, xmm2"); // VPADDSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xed, 0x0a], "vpaddsw xmm1, xmm0, xmmword [rdx]"); // VPADDSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xed, 0x0a], "vpaddsw xmm1{k5}, xmm0, xmmword [rdx]"); // VPADDSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xee, 0xca], "vpmaxsw ymm1{k5}{z}, ymm0, ymm2"); // VPMAXSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xee, 0x0a], "vpmaxsw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMAXSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xee, 0xca], "vpmaxsw ymm1, ymm0, ymm2"); // VPMAXSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xee, 0xca], "vpmaxsw ymm1{k5}, ymm0, ymm2"); // VPMAXSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xee, 0x0a], "vpmaxsw ymm1, ymm0, ymmword [rdx]"); // VPMAXSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xee, 0x0a], "vpmaxsw ymm1{k5}, ymm0, ymmword [rdx]"); // VPMAXSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xee, 0xca], "vpmaxsw zmm1{k5}{z}, zmm0, zmm2"); // VPMAXSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xee, 0x0a], "vpmaxsw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMAXSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xee, 0xca], "vpmaxsw zmm1, zmm0, zmm2"); // VPMAXSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xee, 0xca], "vpmaxsw zmm1{k5}, zmm0, zmm2"); // VPMAXSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xee, 0x0a], "vpmaxsw zmm1, zmm0, zmmword [rdx]"); // VPMAXSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xee, 0x0a], "vpmaxsw zmm1{k5}, zmm0, zmmword [rdx]"); // VPMAXSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xee, 0xca], "vpmaxsw xmm1{k5}{z}, xmm0, xmm2"); // VPMAXSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xee, 0x0a], "vpmaxsw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMAXSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xee, 0xca], "vpmaxsw xmm1, xmm0, xmm2"); // VPMAXSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xee, 0xca], "vpmaxsw xmm1{k5}, xmm0, xmm2"); // VPMAXSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xee, 0x0a], "vpmaxsw xmm1, xmm0, xmmword [rdx]"); // VPMAXSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xee, 0x0a], "vpmaxsw xmm1{k5}, xmm0, xmmword [rdx]"); // VPMAXSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xef, 0x0a], "vpxorq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPXORQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xef, 0x0a], "vpxorq ymm1, ymm0, qword [rdx]{1to4}"); // VPXORQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xef, 0x0a], "vpxorq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPXORQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xef, 0xca], "vpxorq ymm1{k5}{z}, ymm0, ymm2"); // VPXORQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xef, 0x0a], "vpxorq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPXORQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xef, 0xca], "vpxorq ymm1, ymm0, ymm2"); // VPXORQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xef, 0xca], "vpxorq ymm1{k5}, ymm0, ymm2"); // VPXORQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xef, 0x0a], "vpxorq ymm1, ymm0, ymmword [rdx]"); // VPXORQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xef, 0x0a], "vpxorq ymm1{k5}, ymm0, ymmword [rdx]"); // VPXORQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xef, 0x0a], "vpxord ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPXORD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xef, 0x0a], "vpxord ymm1, ymm0, dword [rdx]{1to8}"); // VPXORD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xef, 0x0a], "vpxord ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPXORD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xef, 0xca], "vpxord ymm1{k5}{z}, ymm0, ymm2"); // VPXORD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xef, 0x0a], "vpxord ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPXORD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xef, 0xca], "vpxord ymm1, ymm0, ymm2"); // VPXORD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xef, 0xca], "vpxord ymm1{k5}, ymm0, ymm2"); // VPXORD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xef, 0x0a], "vpxord ymm1, ymm0, ymmword [rdx]"); // VPXORD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xef, 0x0a], "vpxord ymm1{k5}, ymm0, ymmword [rdx]"); // VPXORD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xef, 0x0a], "vpxorq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPXORQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xef, 0x0a], "vpxorq zmm1, zmm0, qword [rdx]{1to8}"); // VPXORQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xef, 0x0a], "vpxorq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPXORQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xef, 0x0a], "vpxorq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPXORQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xef, 0x0a], "vpxorq xmm1, xmm0, qword [rdx]{1to2}"); // VPXORQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xef, 0x0a], "vpxorq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPXORQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xef, 0xca], "vpxorq zmm1{k5}{z}, zmm0, zmm2"); // VPXORQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xef, 0x0a], "vpxorq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPXORQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xef, 0xca], "vpxorq zmm1, zmm0, zmm2"); // VPXORQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xef, 0xca], "vpxorq zmm1{k5}, zmm0, zmm2"); // VPXORQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xef, 0x0a], "vpxorq zmm1, zmm0, zmmword [rdx]"); // VPXORQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xef, 0x0a], "vpxorq zmm1{k5}, zmm0, zmmword [rdx]"); // VPXORQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xef, 0xca], "vpxorq xmm1{k5}{z}, xmm0, xmm2"); // VPXORQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xef, 0x0a], "vpxorq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPXORQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xef, 0xca], "vpxorq xmm1, xmm0, xmm2"); // VPXORQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xef, 0xca], "vpxorq xmm1{k5}, xmm0, xmm2"); // VPXORQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xef, 0x0a], "vpxorq xmm1, xmm0, xmmword [rdx]"); // VPXORQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xef, 0x0a], "vpxorq xmm1{k5}, xmm0, xmmword [rdx]"); // VPXORQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xef, 0x0a], "vpxord zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPXORD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xef, 0x0a], "vpxord zmm1, zmm0, dword [rdx]{1to16}"); // VPXORD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xef, 0x0a], "vpxord zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPXORD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xef, 0x0a], "vpxord xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPXORD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xef, 0x0a], "vpxord xmm1, xmm0, dword [rdx]{1to4}"); // VPXORD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xef, 0x0a], "vpxord xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPXORD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xef, 0xca], "vpxord zmm1{k5}{z}, zmm0, zmm2"); // VPXORD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xef, 0x0a], "vpxord zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPXORD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xef, 0xca], "vpxord zmm1, zmm0, zmm2"); // VPXORD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xef, 0xca], "vpxord zmm1{k5}, zmm0, zmm2"); // VPXORD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xef, 0x0a], "vpxord zmm1, zmm0, zmmword [rdx]"); // VPXORD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xef, 0x0a], "vpxord zmm1{k5}, zmm0, zmmword [rdx]"); // VPXORD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xef, 0xca], "vpxord xmm1{k5}{z}, xmm0, xmm2"); // VPXORD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xef, 0x0a], "vpxord xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPXORD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xef, 0xca], "vpxord xmm1, xmm0, xmm2"); // VPXORD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xef, 0xca], "vpxord xmm1{k5}, xmm0, xmm2"); // VPXORD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xef, 0x0a], "vpxord xmm1, xmm0, xmmword [rdx]"); // VPXORD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xef, 0x0a], "vpxord xmm1{k5}, xmm0, xmmword [rdx]"); // VPXORD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf1, 0xca], "vpsllw ymm1{k5}{z}, ymm0, xmm2"); // VPSLLW_YMMu16_MASKmskw_YMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf1, 0x0a], "vpsllw ymm1{k5}{z}, ymm0, xmmword [rdx]"); // VPSLLW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf1, 0xca], "vpsllw ymm1, ymm0, xmm2"); // VPSLLW_YMMu16_MASKmskw_YMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf1, 0xca], "vpsllw ymm1{k5}, ymm0, xmm2"); // VPSLLW_YMMu16_MASKmskw_YMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf1, 0x0a], "vpsllw ymm1, ymm0, xmmword [rdx]"); // VPSLLW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf1, 0x0a], "vpsllw ymm1{k5}, ymm0, xmmword [rdx]"); // VPSLLW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf1, 0xca], "vpsllw zmm1{k5}{z}, zmm0, xmm2"); // VPSLLW_ZMMu16_MASKmskw_ZMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf1, 0x0a], "vpsllw zmm1{k5}{z}, zmm0, xmmword [rdx]"); // VPSLLW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf1, 0xca], "vpsllw zmm1, zmm0, xmm2"); // VPSLLW_ZMMu16_MASKmskw_ZMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf1, 0xca], "vpsllw zmm1{k5}, zmm0, xmm2"); // VPSLLW_ZMMu16_MASKmskw_ZMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf1, 0x0a], "vpsllw zmm1, zmm0, xmmword [rdx]"); // VPSLLW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf1, 0x0a], "vpsllw zmm1{k5}, zmm0, xmmword [rdx]"); // VPSLLW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf1, 0xca], "vpsllw xmm1{k5}{z}, xmm0, xmm2"); // VPSLLW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf1, 0x0a], "vpsllw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSLLW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf1, 0xca], "vpsllw xmm1, xmm0, xmm2"); // VPSLLW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf1, 0xca], "vpsllw xmm1{k5}, xmm0, xmm2"); // VPSLLW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf1, 0x0a], "vpsllw xmm1, xmm0, xmmword [rdx]"); // VPSLLW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf1, 0x0a], "vpsllw xmm1{k5}, xmm0, xmmword [rdx]"); // VPSLLW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xf2, 0xca], "vpslld ymm1{k5}{z}, ymm0, xmm2"); // VPSLLD_YMMu32_MASKmskw_YMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xf2, 0x0a], "vpslld ymm1{k5}{z}, ymm0, xmmword [rdx]"); // VPSLLD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xf2, 0xca], "vpslld ymm1, ymm0, xmm2"); // VPSLLD_YMMu32_MASKmskw_YMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xf2, 0xca], "vpslld ymm1{k5}, ymm0, xmm2"); // VPSLLD_YMMu32_MASKmskw_YMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xf2, 0x0a], "vpslld ymm1, ymm0, xmmword [rdx]"); // VPSLLD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xf2, 0x0a], "vpslld ymm1{k5}, ymm0, xmmword [rdx]"); // VPSLLD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xf2, 0xca], "vpslld zmm1{k5}{z}, zmm0, xmm2"); // VPSLLD_ZMMu32_MASKmskw_ZMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xf2, 0x0a], "vpslld zmm1{k5}{z}, zmm0, xmmword [rdx]"); // VPSLLD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xf2, 0xca], "vpslld zmm1, zmm0, xmm2"); // VPSLLD_ZMMu32_MASKmskw_ZMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xf2, 0xca], "vpslld zmm1{k5}, zmm0, xmm2"); // VPSLLD_ZMMu32_MASKmskw_ZMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xf2, 0x0a], "vpslld zmm1, zmm0, xmmword [rdx]"); // VPSLLD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xf2, 0x0a], "vpslld zmm1{k5}, zmm0, xmmword [rdx]"); // VPSLLD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xf2, 0xca], "vpslld xmm1{k5}{z}, xmm0, xmm2"); // VPSLLD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xf2, 0x0a], "vpslld xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSLLD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xf2, 0xca], "vpslld xmm1, xmm0, xmm2"); // VPSLLD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xf2, 0xca], "vpslld xmm1{k5}, xmm0, xmm2"); // VPSLLD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xf2, 0x0a], "vpslld xmm1, xmm0, xmmword [rdx]"); // VPSLLD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xf2, 0x0a], "vpslld xmm1{k5}, xmm0, xmmword [rdx]"); // VPSLLD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf3, 0xca], "vpsllq ymm1{k5}{z}, ymm0, xmm2"); // VPSLLQ_YMMu64_MASKmskw_YMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf3, 0x0a], "vpsllq ymm1{k5}{z}, ymm0, xmmword [rdx]"); // VPSLLQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf3, 0xca], "vpsllq ymm1, ymm0, xmm2"); // VPSLLQ_YMMu64_MASKmskw_YMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf3, 0xca], "vpsllq ymm1{k5}, ymm0, xmm2"); // VPSLLQ_YMMu64_MASKmskw_YMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf3, 0x0a], "vpsllq ymm1, ymm0, xmmword [rdx]"); // VPSLLQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf3, 0x0a], "vpsllq ymm1{k5}, ymm0, xmmword [rdx]"); // VPSLLQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf3, 0xca], "vpsllq zmm1{k5}{z}, zmm0, xmm2"); // VPSLLQ_ZMMu64_MASKmskw_ZMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf3, 0x0a], "vpsllq zmm1{k5}{z}, zmm0, xmmword [rdx]"); // VPSLLQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf3, 0xca], "vpsllq zmm1, zmm0, xmm2"); // VPSLLQ_ZMMu64_MASKmskw_ZMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf3, 0xca], "vpsllq zmm1{k5}, zmm0, xmm2"); // VPSLLQ_ZMMu64_MASKmskw_ZMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf3, 0x0a], "vpsllq zmm1, zmm0, xmmword [rdx]"); // VPSLLQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf3, 0x0a], "vpsllq zmm1{k5}, zmm0, xmmword [rdx]"); // VPSLLQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf3, 0xca], "vpsllq xmm1{k5}{z}, xmm0, xmm2"); // VPSLLQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf3, 0x0a], "vpsllq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSLLQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf3, 0xca], "vpsllq xmm1, xmm0, xmm2"); // VPSLLQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf3, 0xca], "vpsllq xmm1{k5}, xmm0, xmm2"); // VPSLLQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf3, 0x0a], "vpsllq xmm1, xmm0, xmmword [rdx]"); // VPSLLQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf3, 0x0a], "vpsllq xmm1{k5}, xmm0, xmmword [rdx]"); // VPSLLQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xf4, 0x0a], "vpmuludq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPMULUDQ_YMMu64_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xf4, 0x0a], "vpmuludq ymm1, ymm0, qword [rdx]{1to4}"); // VPMULUDQ_YMMu64_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xf4, 0x0a], "vpmuludq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPMULUDQ_YMMu64_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf4, 0xca], "vpmuludq ymm1{k5}{z}, ymm0, ymm2"); // VPMULUDQ_YMMu64_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf4, 0x0a], "vpmuludq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMULUDQ_YMMu64_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf4, 0xca], "vpmuludq ymm1, ymm0, ymm2"); // VPMULUDQ_YMMu64_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf4, 0xca], "vpmuludq ymm1{k5}, ymm0, ymm2"); // VPMULUDQ_YMMu64_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf4, 0x0a], "vpmuludq ymm1, ymm0, ymmword [rdx]"); // VPMULUDQ_YMMu64_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf4, 0x0a], "vpmuludq ymm1{k5}, ymm0, ymmword [rdx]"); // VPMULUDQ_YMMu64_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xf4, 0x0a], "vpmuludq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPMULUDQ_ZMMu64_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xf4, 0x0a], "vpmuludq zmm1, zmm0, qword [rdx]{1to8}"); // VPMULUDQ_ZMMu64_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xf4, 0x0a], "vpmuludq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPMULUDQ_ZMMu64_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xf4, 0x0a], "vpmuludq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPMULUDQ_XMMu64_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xf4, 0x0a], "vpmuludq xmm1, xmm0, qword [rdx]{1to2}"); // VPMULUDQ_XMMu64_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xf4, 0x0a], "vpmuludq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPMULUDQ_XMMu64_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf4, 0xca], "vpmuludq zmm1{k5}{z}, zmm0, zmm2"); // VPMULUDQ_ZMMu64_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf4, 0x0a], "vpmuludq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMULUDQ_ZMMu64_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf4, 0xca], "vpmuludq zmm1, zmm0, zmm2"); // VPMULUDQ_ZMMu64_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf4, 0xca], "vpmuludq zmm1{k5}, zmm0, zmm2"); // VPMULUDQ_ZMMu64_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf4, 0x0a], "vpmuludq zmm1, zmm0, zmmword [rdx]"); // VPMULUDQ_ZMMu64_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf4, 0x0a], "vpmuludq zmm1{k5}, zmm0, zmmword [rdx]"); // VPMULUDQ_ZMMu64_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf4, 0xca], "vpmuludq xmm1{k5}{z}, xmm0, xmm2"); // VPMULUDQ_XMMu64_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf4, 0x0a], "vpmuludq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMULUDQ_XMMu64_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf4, 0xca], "vpmuludq xmm1, xmm0, xmm2"); // VPMULUDQ_XMMu64_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf4, 0xca], "vpmuludq xmm1{k5}, xmm0, xmm2"); // VPMULUDQ_XMMu64_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf4, 0x0a], "vpmuludq xmm1, xmm0, xmmword [rdx]"); // VPMULUDQ_XMMu64_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf4, 0x0a], "vpmuludq xmm1{k5}, xmm0, xmmword [rdx]"); // VPMULUDQ_XMMu64_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf5, 0xca], "vpmaddwd ymm1{k5}{z}, ymm0, ymm2"); // VPMADDWD_YMMi32_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf5, 0x0a], "vpmaddwd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMADDWD_YMMi32_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf5, 0xca], "vpmaddwd ymm1, ymm0, ymm2"); // VPMADDWD_YMMi32_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf5, 0xca], "vpmaddwd ymm1{k5}, ymm0, ymm2"); // VPMADDWD_YMMi32_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf5, 0x0a], "vpmaddwd ymm1, ymm0, ymmword [rdx]"); // VPMADDWD_YMMi32_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf5, 0x0a], "vpmaddwd ymm1{k5}, ymm0, ymmword [rdx]"); // VPMADDWD_YMMi32_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf5, 0xca], "vpmaddwd zmm1{k5}{z}, zmm0, zmm2"); // VPMADDWD_ZMMi32_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf5, 0x0a], "vpmaddwd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMADDWD_ZMMi32_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf5, 0xca], "vpmaddwd zmm1, zmm0, zmm2"); // VPMADDWD_ZMMi32_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf5, 0xca], "vpmaddwd zmm1{k5}, zmm0, zmm2"); // VPMADDWD_ZMMi32_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf5, 0x0a], "vpmaddwd zmm1, zmm0, zmmword [rdx]"); // VPMADDWD_ZMMi32_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf5, 0x0a], "vpmaddwd zmm1{k5}, zmm0, zmmword [rdx]"); // VPMADDWD_ZMMi32_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf5, 0xca], "vpmaddwd xmm1{k5}{z}, xmm0, xmm2"); // VPMADDWD_XMMi32_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf5, 0x0a], "vpmaddwd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMADDWD_XMMi32_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf5, 0xca], "vpmaddwd xmm1, xmm0, xmm2"); // VPMADDWD_XMMi32_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf5, 0xca], "vpmaddwd xmm1{k5}, xmm0, xmm2"); // VPMADDWD_XMMi32_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf5, 0x0a], "vpmaddwd xmm1, xmm0, xmmword [rdx]"); // VPMADDWD_XMMi32_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf5, 0x0a], "vpmaddwd xmm1{k5}, xmm0, xmmword [rdx]"); // VPMADDWD_XMMi32_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf6, 0xca], "vpsadbw ymm1, ymm0, ymm2"); // VPSADBW_YMMu16_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf6, 0x0a], "vpsadbw ymm1, ymm0, ymmword [rdx]"); // VPSADBW_YMMu16_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf6, 0xca], "vpsadbw zmm1, zmm0, zmm2"); // VPSADBW_ZMMu16_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf6, 0x0a], "vpsadbw zmm1, zmm0, zmmword [rdx]"); // VPSADBW_ZMMu16_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf6, 0xca], "vpsadbw xmm1, xmm0, xmm2"); // VPSADBW_XMMu16_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf6, 0x0a], "vpsadbw xmm1, xmm0, xmmword [rdx]"); // VPSADBW_XMMu16_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf8, 0xca], "vpsubb ymm1{k5}{z}, ymm0, ymm2"); // VPSUBB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf8, 0x0a], "vpsubb ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSUBB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf8, 0xca], "vpsubb ymm1, ymm0, ymm2"); // VPSUBB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf8, 0xca], "vpsubb ymm1{k5}, ymm0, ymm2"); // VPSUBB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf8, 0x0a], "vpsubb ymm1, ymm0, ymmword [rdx]"); // VPSUBB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf8, 0x0a], "vpsubb ymm1{k5}, ymm0, ymmword [rdx]"); // VPSUBB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf8, 0xca], "vpsubb zmm1{k5}{z}, zmm0, zmm2"); // VPSUBB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf8, 0x0a], "vpsubb zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSUBB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf8, 0xca], "vpsubb zmm1, zmm0, zmm2"); // VPSUBB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf8, 0xca], "vpsubb zmm1{k5}, zmm0, zmm2"); // VPSUBB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf8, 0x0a], "vpsubb zmm1, zmm0, zmmword [rdx]"); // VPSUBB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf8, 0x0a], "vpsubb zmm1{k5}, zmm0, zmmword [rdx]"); // VPSUBB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf8, 0xca], "vpsubb xmm1{k5}{z}, xmm0, xmm2"); // VPSUBB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf8, 0x0a], "vpsubb xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSUBB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf8, 0xca], "vpsubb xmm1, xmm0, xmm2"); // VPSUBB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf8, 0xca], "vpsubb xmm1{k5}, xmm0, xmm2"); // VPSUBB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf8, 0x0a], "vpsubb xmm1, xmm0, xmmword [rdx]"); // VPSUBB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf8, 0x0a], "vpsubb xmm1{k5}, xmm0, xmmword [rdx]"); // VPSUBB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf9, 0xca], "vpsubw ymm1{k5}{z}, ymm0, ymm2"); // VPSUBW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xf9, 0x0a], "vpsubw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSUBW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf9, 0xca], "vpsubw ymm1, ymm0, ymm2"); // VPSUBW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf9, 0xca], "vpsubw ymm1{k5}, ymm0, ymm2"); // VPSUBW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xf9, 0x0a], "vpsubw ymm1, ymm0, ymmword [rdx]"); // VPSUBW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xf9, 0x0a], "vpsubw ymm1{k5}, ymm0, ymmword [rdx]"); // VPSUBW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf9, 0xca], "vpsubw zmm1{k5}{z}, zmm0, zmm2"); // VPSUBW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xf9, 0x0a], "vpsubw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSUBW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf9, 0xca], "vpsubw zmm1, zmm0, zmm2"); // VPSUBW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf9, 0xca], "vpsubw zmm1{k5}, zmm0, zmm2"); // VPSUBW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xf9, 0x0a], "vpsubw zmm1, zmm0, zmmword [rdx]"); // VPSUBW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xf9, 0x0a], "vpsubw zmm1{k5}, zmm0, zmmword [rdx]"); // VPSUBW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf9, 0xca], "vpsubw xmm1{k5}{z}, xmm0, xmm2"); // VPSUBW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xf9, 0x0a], "vpsubw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSUBW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf9, 0xca], "vpsubw xmm1, xmm0, xmm2"); // VPSUBW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf9, 0xca], "vpsubw xmm1{k5}, xmm0, xmm2"); // VPSUBW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xf9, 0x0a], "vpsubw xmm1, xmm0, xmmword [rdx]"); // VPSUBW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xf9, 0x0a], "vpsubw xmm1{k5}, xmm0, xmmword [rdx]"); // VPSUBW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xfa, 0x0a], "vpsubd ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPSUBD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xfa, 0x0a], "vpsubd ymm1, ymm0, dword [rdx]{1to8}"); // VPSUBD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xfa, 0x0a], "vpsubd ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPSUBD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xfa, 0xca], "vpsubd ymm1{k5}{z}, ymm0, ymm2"); // VPSUBD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xfa, 0x0a], "vpsubd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSUBD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xfa, 0xca], "vpsubd ymm1, ymm0, ymm2"); // VPSUBD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xfa, 0xca], "vpsubd ymm1{k5}, ymm0, ymm2"); // VPSUBD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xfa, 0x0a], "vpsubd ymm1, ymm0, ymmword [rdx]"); // VPSUBD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xfa, 0x0a], "vpsubd ymm1{k5}, ymm0, ymmword [rdx]"); // VPSUBD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xfa, 0x0a], "vpsubd zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPSUBD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xfa, 0x0a], "vpsubd zmm1, zmm0, dword [rdx]{1to16}"); // VPSUBD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xfa, 0x0a], "vpsubd zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPSUBD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xfa, 0x0a], "vpsubd xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPSUBD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xfa, 0x0a], "vpsubd xmm1, xmm0, dword [rdx]{1to4}"); // VPSUBD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xfa, 0x0a], "vpsubd xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPSUBD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xfa, 0xca], "vpsubd zmm1{k5}{z}, zmm0, zmm2"); // VPSUBD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xfa, 0x0a], "vpsubd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSUBD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xfa, 0xca], "vpsubd zmm1, zmm0, zmm2"); // VPSUBD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xfa, 0xca], "vpsubd zmm1{k5}, zmm0, zmm2"); // VPSUBD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xfa, 0x0a], "vpsubd zmm1, zmm0, zmmword [rdx]"); // VPSUBD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xfa, 0x0a], "vpsubd zmm1{k5}, zmm0, zmmword [rdx]"); // VPSUBD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xfa, 0xca], "vpsubd xmm1{k5}{z}, xmm0, xmm2"); // VPSUBD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xfa, 0x0a], "vpsubd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSUBD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xfa, 0xca], "vpsubd xmm1, xmm0, xmm2"); // VPSUBD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xfa, 0xca], "vpsubd xmm1{k5}, xmm0, xmm2"); // VPSUBD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xfa, 0x0a], "vpsubd xmm1, xmm0, xmmword [rdx]"); // VPSUBD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xfa, 0x0a], "vpsubd xmm1{k5}, xmm0, xmmword [rdx]"); // VPSUBD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xbd, 0xfb, 0x0a], "vpsubq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPSUBQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x38, 0xfb, 0x0a], "vpsubq ymm1, ymm0, qword [rdx]{1to4}"); // VPSUBQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x3d, 0xfb, 0x0a], "vpsubq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPSUBQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xfb, 0xca], "vpsubq ymm1{k5}{z}, ymm0, ymm2"); // VPSUBQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xfb, 0x0a], "vpsubq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSUBQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xfb, 0xca], "vpsubq ymm1, ymm0, ymm2"); // VPSUBQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xfb, 0xca], "vpsubq ymm1{k5}, ymm0, ymm2"); // VPSUBQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xfb, 0x0a], "vpsubq ymm1, ymm0, ymmword [rdx]"); // VPSUBQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xfb, 0x0a], "vpsubq ymm1{k5}, ymm0, ymmword [rdx]"); // VPSUBQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xdd, 0xfb, 0x0a], "vpsubq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPSUBQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x58, 0xfb, 0x0a], "vpsubq zmm1, zmm0, qword [rdx]{1to8}"); // VPSUBQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x5d, 0xfb, 0x0a], "vpsubq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPSUBQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x9d, 0xfb, 0x0a], "vpsubq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPSUBQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x18, 0xfb, 0x0a], "vpsubq xmm1, xmm0, qword [rdx]{1to2}"); // VPSUBQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x1d, 0xfb, 0x0a], "vpsubq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPSUBQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xfb, 0xca], "vpsubq zmm1{k5}{z}, zmm0, zmm2"); // VPSUBQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xfb, 0x0a], "vpsubq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSUBQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xfb, 0xca], "vpsubq zmm1, zmm0, zmm2"); // VPSUBQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xfb, 0xca], "vpsubq zmm1{k5}, zmm0, zmm2"); // VPSUBQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xfb, 0x0a], "vpsubq zmm1, zmm0, zmmword [rdx]"); // VPSUBQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xfb, 0x0a], "vpsubq zmm1{k5}, zmm0, zmmword [rdx]"); // VPSUBQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xfb, 0xca], "vpsubq xmm1{k5}{z}, xmm0, xmm2"); // VPSUBQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xfb, 0x0a], "vpsubq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSUBQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xfb, 0xca], "vpsubq xmm1, xmm0, xmm2"); // VPSUBQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xfb, 0xca], "vpsubq xmm1{k5}, xmm0, xmm2"); // VPSUBQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xfb, 0x0a], "vpsubq xmm1, xmm0, xmmword [rdx]"); // VPSUBQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xfb, 0x0a], "vpsubq xmm1{k5}, xmm0, xmmword [rdx]"); // VPSUBQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xfc, 0xca], "vpaddb ymm1{k5}{z}, ymm0, ymm2"); // VPADDB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xfc, 0x0a], "vpaddb ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPADDB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xfc, 0xca], "vpaddb ymm1, ymm0, ymm2"); // VPADDB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xfc, 0xca], "vpaddb ymm1{k5}, ymm0, ymm2"); // VPADDB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xfc, 0x0a], "vpaddb ymm1, ymm0, ymmword [rdx]"); // VPADDB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xfc, 0x0a], "vpaddb ymm1{k5}, ymm0, ymmword [rdx]"); // VPADDB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xfc, 0xca], "vpaddb zmm1{k5}{z}, zmm0, zmm2"); // VPADDB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xfc, 0x0a], "vpaddb zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPADDB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xfc, 0xca], "vpaddb zmm1, zmm0, zmm2"); // VPADDB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xfc, 0xca], "vpaddb zmm1{k5}, zmm0, zmm2"); // VPADDB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xfc, 0x0a], "vpaddb zmm1, zmm0, zmmword [rdx]"); // VPADDB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xfc, 0x0a], "vpaddb zmm1{k5}, zmm0, zmmword [rdx]"); // VPADDB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xfc, 0xca], "vpaddb xmm1{k5}{z}, xmm0, xmm2"); // VPADDB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xfc, 0x0a], "vpaddb xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPADDB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xfc, 0xca], "vpaddb xmm1, xmm0, xmm2"); // VPADDB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xfc, 0xca], "vpaddb xmm1{k5}, xmm0, xmm2"); // VPADDB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xfc, 0x0a], "vpaddb xmm1, xmm0, xmmword [rdx]"); // VPADDB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xfc, 0x0a], "vpaddb xmm1{k5}, xmm0, xmmword [rdx]"); // VPADDB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xfd, 0xca], "vpaddw ymm1{k5}{z}, ymm0, ymm2"); // VPADDW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xad, 0xfd, 0x0a], "vpaddw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPADDW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xfd, 0xca], "vpaddw ymm1, ymm0, ymm2"); // VPADDW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xfd, 0xca], "vpaddw ymm1{k5}, ymm0, ymm2"); // VPADDW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x28, 0xfd, 0x0a], "vpaddw ymm1, ymm0, ymmword [rdx]"); // VPADDW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x2d, 0xfd, 0x0a], "vpaddw ymm1{k5}, ymm0, ymmword [rdx]"); // VPADDW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xfd, 0xca], "vpaddw zmm1{k5}{z}, zmm0, zmm2"); // VPADDW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0xcd, 0xfd, 0x0a], "vpaddw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPADDW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xfd, 0xca], "vpaddw zmm1, zmm0, zmm2"); // VPADDW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xfd, 0xca], "vpaddw zmm1{k5}, zmm0, zmm2"); // VPADDW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x48, 0xfd, 0x0a], "vpaddw zmm1, zmm0, zmmword [rdx]"); // VPADDW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x4d, 0xfd, 0x0a], "vpaddw zmm1{k5}, zmm0, zmmword [rdx]"); // VPADDW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xfd, 0xca], "vpaddw xmm1{k5}{z}, xmm0, xmm2"); // VPADDW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x8d, 0xfd, 0x0a], "vpaddw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPADDW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xfd, 0xca], "vpaddw xmm1, xmm0, xmm2"); // VPADDW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xfd, 0xca], "vpaddw xmm1{k5}, xmm0, xmm2"); // VPADDW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x08, 0xfd, 0x0a], "vpaddw xmm1, xmm0, xmmword [rdx]"); // VPADDW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfd, 0x0d, 0xfd, 0x0a], "vpaddw xmm1{k5}, xmm0, xmmword [rdx]"); // VPADDW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xbd, 0xfe, 0x0a], "vpaddd ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPADDD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x38, 0xfe, 0x0a], "vpaddd ymm1, ymm0, dword [rdx]{1to8}"); // VPADDD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x3d, 0xfe, 0x0a], "vpaddd ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPADDD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xfe, 0xca], "vpaddd ymm1{k5}{z}, ymm0, ymm2"); // VPADDD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xad, 0xfe, 0x0a], "vpaddd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPADDD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xfe, 0xca], "vpaddd ymm1, ymm0, ymm2"); // VPADDD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xfe, 0xca], "vpaddd ymm1{k5}, ymm0, ymm2"); // VPADDD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x28, 0xfe, 0x0a], "vpaddd ymm1, ymm0, ymmword [rdx]"); // VPADDD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x2d, 0xfe, 0x0a], "vpaddd ymm1{k5}, ymm0, ymmword [rdx]"); // VPADDD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xdd, 0xfe, 0x0a], "vpaddd zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPADDD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x58, 0xfe, 0x0a], "vpaddd zmm1, zmm0, dword [rdx]{1to16}"); // VPADDD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x5d, 0xfe, 0x0a], "vpaddd zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPADDD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x9d, 0xfe, 0x0a], "vpaddd xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPADDD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x18, 0xfe, 0x0a], "vpaddd xmm1, xmm0, dword [rdx]{1to4}"); // VPADDD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x1d, 0xfe, 0x0a], "vpaddd xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPADDD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xfe, 0xca], "vpaddd zmm1{k5}{z}, zmm0, zmm2"); // VPADDD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0xcd, 0xfe, 0x0a], "vpaddd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPADDD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xfe, 0xca], "vpaddd zmm1, zmm0, zmm2"); // VPADDD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xfe, 0xca], "vpaddd zmm1{k5}, zmm0, zmm2"); // VPADDD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x48, 0xfe, 0x0a], "vpaddd zmm1, zmm0, zmmword [rdx]"); // VPADDD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x4d, 0xfe, 0x0a], "vpaddd zmm1{k5}, zmm0, zmmword [rdx]"); // VPADDD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xfe, 0xca], "vpaddd xmm1{k5}{z}, xmm0, xmm2"); // VPADDD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x8d, 0xfe, 0x0a], "vpaddd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPADDD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xfe, 0xca], "vpaddd xmm1, xmm0, xmm2"); // VPADDD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xfe, 0xca], "vpaddd xmm1{k5}, xmm0, xmm2"); // VPADDD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x08, 0xfe, 0x0a], "vpaddd xmm1, xmm0, xmmword [rdx]"); // VPADDD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7d, 0x0d, 0xfe, 0x0a], "vpaddd xmm1{k5}, xmm0, xmmword [rdx]"); // VPADDD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
}
#[test]
fn tests_f2_0f() {
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x10, 0xca], "vmovss xmm1{k5}{z}, xmm0, xmm2"); // VMOVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x10, 0x0a], "vmovss xmm1{k5}{z}, dword [rdx]"); // VMOVSS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x10, 0xca], "vmovss xmm1, xmm0, xmm2"); // VMOVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x10, 0xca], "vmovss xmm1{k5}, xmm0, xmm2"); // VMOVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x10, 0x0a], "vmovss xmm1, dword [rdx]"); // VMOVSS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x10, 0x0a], "vmovss xmm1{k5}, dword [rdx]"); // VMOVSS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x11, 0xca], "vmovss xmm2{k5}{z}, xmm0, xmm1"); // VMOVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x11, 0xca], "vmovss xmm2, xmm0, xmm1"); // VMOVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x11, 0xca], "vmovss xmm2{k5}, xmm0, xmm1"); // VMOVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x11, 0x0a], "vmovss dword [rdx], xmm1"); // VMOVSS_MEMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x11, 0x0a], "vmovss dword [rdx]{k5}, xmm1"); // VMOVSS_MEMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x12, 0xca], "vmovsldup ymm1{k5}{z}, ymm2"); // VMOVSLDUP_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x12, 0x0a], "vmovsldup ymm1{k5}{z}, ymmword [rdx]"); // VMOVSLDUP_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x12, 0xca], "vmovsldup ymm1, ymm2"); // VMOVSLDUP_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x12, 0xca], "vmovsldup ymm1{k5}, ymm2"); // VMOVSLDUP_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x12, 0x0a], "vmovsldup ymm1, ymmword [rdx]"); // VMOVSLDUP_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x12, 0x0a], "vmovsldup ymm1{k5}, ymmword [rdx]"); // VMOVSLDUP_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xcd, 0x12, 0xca], "vmovsldup zmm1{k5}{z}, zmm2"); // VMOVSLDUP_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xcd, 0x12, 0x0a], "vmovsldup zmm1{k5}{z}, zmmword [rdx]"); // VMOVSLDUP_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x48, 0x12, 0xca], "vmovsldup zmm1, zmm2"); // VMOVSLDUP_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x4d, 0x12, 0xca], "vmovsldup zmm1{k5}, zmm2"); // VMOVSLDUP_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x48, 0x12, 0x0a], "vmovsldup zmm1, zmmword [rdx]"); // VMOVSLDUP_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x4d, 0x12, 0x0a], "vmovsldup zmm1{k5}, zmmword [rdx]"); // VMOVSLDUP_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x8d, 0x12, 0xca], "vmovsldup xmm1{k5}{z}, xmm2"); // VMOVSLDUP_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x8d, 0x12, 0x0a], "vmovsldup xmm1{k5}{z}, xmmword [rdx]"); // VMOVSLDUP_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x08, 0x12, 0xca], "vmovsldup xmm1, xmm2"); // VMOVSLDUP_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x0d, 0x12, 0xca], "vmovsldup xmm1{k5}, xmm2"); // VMOVSLDUP_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x08, 0x12, 0x0a], "vmovsldup xmm1, xmmword [rdx]"); // VMOVSLDUP_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x0d, 0x12, 0x0a], "vmovsldup xmm1{k5}, xmmword [rdx]"); // VMOVSLDUP_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x16, 0xca], "vmovshdup ymm1{k5}{z}, ymm2"); // VMOVSHDUP_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x16, 0x0a], "vmovshdup ymm1{k5}{z}, ymmword [rdx]"); // VMOVSHDUP_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x16, 0xca], "vmovshdup ymm1, ymm2"); // VMOVSHDUP_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x16, 0xca], "vmovshdup ymm1{k5}, ymm2"); // VMOVSHDUP_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x16, 0x0a], "vmovshdup ymm1, ymmword [rdx]"); // VMOVSHDUP_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x16, 0x0a], "vmovshdup ymm1{k5}, ymmword [rdx]"); // VMOVSHDUP_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xcd, 0x16, 0xca], "vmovshdup zmm1{k5}{z}, zmm2"); // VMOVSHDUP_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xcd, 0x16, 0x0a], "vmovshdup zmm1{k5}{z}, zmmword [rdx]"); // VMOVSHDUP_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x48, 0x16, 0xca], "vmovshdup zmm1, zmm2"); // VMOVSHDUP_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x4d, 0x16, 0xca], "vmovshdup zmm1{k5}, zmm2"); // VMOVSHDUP_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x48, 0x16, 0x0a], "vmovshdup zmm1, zmmword [rdx]"); // VMOVSHDUP_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x4d, 0x16, 0x0a], "vmovshdup zmm1{k5}, zmmword [rdx]"); // VMOVSHDUP_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x8d, 0x16, 0xca], "vmovshdup xmm1{k5}{z}, xmm2"); // VMOVSHDUP_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x8d, 0x16, 0x0a], "vmovshdup xmm1{k5}{z}, xmmword [rdx]"); // VMOVSHDUP_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x08, 0x16, 0xca], "vmovshdup xmm1, xmm2"); // VMOVSHDUP_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x0d, 0x16, 0xca], "vmovshdup xmm1{k5}, xmm2"); // VMOVSHDUP_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x08, 0x16, 0x0a], "vmovshdup xmm1, xmmword [rdx]"); // VMOVSHDUP_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x0d, 0x16, 0x0a], "vmovshdup xmm1{k5}, xmmword [rdx]"); // VMOVSHDUP_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x78, 0x2a, 0xca], "vcvtsi2ss xmm1{rz-sae}, xmm0, rdx"); // VCVTSI2SS_XMMf32_XMMf32_GPR64i64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x38, 0x2a, 0xca], "vcvtsi2ss xmm1{rd-sae}, xmm0, rdx"); // VCVTSI2SS_XMMf32_XMMf32_GPR64i64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x2a, 0xca], "vcvtsi2ss xmm1, xmm0, rdx"); // VCVTSI2SS_XMMf32_XMMf32_GPR64i64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x2a, 0x0a], "vcvtsi2ss xmm1, xmm0, qword [rdx]"); // VCVTSI2SS_XMMf32_XMMf32_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x78, 0x2a, 0xca], "vcvtsi2ss xmm1{rz-sae}, xmm0, edx"); // VCVTSI2SS_XMMf32_XMMf32_GPR32i32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x38, 0x2a, 0xca], "vcvtsi2ss xmm1{rd-sae}, xmm0, edx"); // VCVTSI2SS_XMMf32_XMMf32_GPR32i32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x2a, 0xca], "vcvtsi2ss xmm1, xmm0, edx"); // VCVTSI2SS_XMMf32_XMMf32_GPR32i32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x2a, 0x0a], "vcvtsi2ss xmm1, xmm0, dword [rdx]"); // VCVTSI2SS_XMMf32_XMMf32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x58, 0x2a, 0xca], "vcvtsi2ss xmm1{ru-sae}, xmm0, rdx"); // VCVTSI2SS_XMMf32_XMMf32_GPR64i64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x18, 0x2a, 0xca], "vcvtsi2ss xmm1{rne-sae}, xmm0, rdx"); // VCVTSI2SS_XMMf32_XMMf32_GPR64i64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x58, 0x2a, 0xca], "vcvtsi2ss xmm1{ru-sae}, xmm0, edx"); // VCVTSI2SS_XMMf32_XMMf32_GPR32i32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x18, 0x2a, 0xca], "vcvtsi2ss xmm1{rne-sae}, xmm0, edx"); // VCVTSI2SS_XMMf32_XMMf32_GPR32i32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x78, 0x2c, 0xca], "vcvttss2si rcx{sae}, xmm2"); // VCVTTSS2SI_GPR64i64_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x2c, 0xca], "vcvttss2si rcx, xmm2"); // VCVTTSS2SI_GPR64i64_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x2c, 0x0a], "vcvttss2si rcx, dword [rdx]"); // VCVTTSS2SI_GPR64i64_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x78, 0x2c, 0xca], "vcvttss2si ecx{sae}, xmm2"); // VCVTTSS2SI_GPR32i32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x2c, 0xca], "vcvttss2si ecx, xmm2"); // VCVTTSS2SI_GPR32i32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x2c, 0x0a], "vcvttss2si ecx, dword [rdx]"); // VCVTTSS2SI_GPR32i32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x78, 0x2d, 0xca], "vcvtss2si rcx{rz-sae}, xmm2"); // VCVTSS2SI_GPR64i64_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x38, 0x2d, 0xca], "vcvtss2si rcx{rd-sae}, xmm2"); // VCVTSS2SI_GPR64i64_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x2d, 0xca], "vcvtss2si rcx, xmm2"); // VCVTSS2SI_GPR64i64_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x2d, 0x0a], "vcvtss2si rcx, dword [rdx]"); // VCVTSS2SI_GPR64i64_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x78, 0x2d, 0xca], "vcvtss2si ecx{rz-sae}, xmm2"); // VCVTSS2SI_GPR32i32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x38, 0x2d, 0xca], "vcvtss2si ecx{rd-sae}, xmm2"); // VCVTSS2SI_GPR32i32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x2d, 0xca], "vcvtss2si ecx, xmm2"); // VCVTSS2SI_GPR32i32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x2d, 0x0a], "vcvtss2si ecx, dword [rdx]"); // VCVTSS2SI_GPR32i32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x58, 0x2d, 0xca], "vcvtss2si rcx{ru-sae}, xmm2"); // VCVTSS2SI_GPR64i64_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x18, 0x2d, 0xca], "vcvtss2si rcx{rne-sae}, xmm2"); // VCVTSS2SI_GPR64i64_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x58, 0x2d, 0xca], "vcvtss2si ecx{ru-sae}, xmm2"); // VCVTSS2SI_GPR32i32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x18, 0x2d, 0xca], "vcvtss2si ecx{rne-sae}, xmm2"); // VCVTSS2SI_GPR32i32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xfd, 0x51, 0xca], "vsqrtss xmm1{k5}{z}{rz-sae}, xmm0, xmm2"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x78, 0x51, 0xca], "vsqrtss xmm1{rz-sae}, xmm0, xmm2"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x7d, 0x51, 0xca], "vsqrtss xmm1{k5}{rz-sae}, xmm0, xmm2"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xbd, 0x51, 0xca], "vsqrtss xmm1{k5}{z}{rd-sae}, xmm0, xmm2"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x38, 0x51, 0xca], "vsqrtss xmm1{rd-sae}, xmm0, xmm2"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x3d, 0x51, 0xca], "vsqrtss xmm1{k5}{rd-sae}, xmm0, xmm2"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x51, 0xca], "vsqrtss xmm1{k5}{z}, xmm0, xmm2"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x51, 0x0a], "vsqrtss xmm1{k5}{z}, xmm0, dword [rdx]"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x51, 0xca], "vsqrtss xmm1, xmm0, xmm2"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x51, 0xca], "vsqrtss xmm1{k5}, xmm0, xmm2"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x51, 0x0a], "vsqrtss xmm1, xmm0, dword [rdx]"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x51, 0x0a], "vsqrtss xmm1{k5}, xmm0, dword [rdx]"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xdd, 0x51, 0xca], "vsqrtss xmm1{k5}{z}{ru-sae}, xmm0, xmm2"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x58, 0x51, 0xca], "vsqrtss xmm1{ru-sae}, xmm0, xmm2"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x5d, 0x51, 0xca], "vsqrtss xmm1{k5}{ru-sae}, xmm0, xmm2"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x9d, 0x51, 0xca], "vsqrtss xmm1{k5}{z}{rne-sae}, xmm0, xmm2"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x18, 0x51, 0xca], "vsqrtss xmm1{rne-sae}, xmm0, xmm2"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x1d, 0x51, 0xca], "vsqrtss xmm1{k5}{rne-sae}, xmm0, xmm2"); // VSQRTSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xfd, 0x58, 0xca], "vaddss xmm1{k5}{z}{rz-sae}, xmm0, xmm2"); // VADDSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x78, 0x58, 0xca], "vaddss xmm1{rz-sae}, xmm0, xmm2"); // VADDSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x7d, 0x58, 0xca], "vaddss xmm1{k5}{rz-sae}, xmm0, xmm2"); // VADDSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xbd, 0x58, 0xca], "vaddss xmm1{k5}{z}{rd-sae}, xmm0, xmm2"); // VADDSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x38, 0x58, 0xca], "vaddss xmm1{rd-sae}, xmm0, xmm2"); // VADDSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x3d, 0x58, 0xca], "vaddss xmm1{k5}{rd-sae}, xmm0, xmm2"); // VADDSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x58, 0xca], "vaddss xmm1{k5}{z}, xmm0, xmm2"); // VADDSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x58, 0x0a], "vaddss xmm1{k5}{z}, xmm0, dword [rdx]"); // VADDSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x58, 0xca], "vaddss xmm1, xmm0, xmm2"); // VADDSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x58, 0xca], "vaddss xmm1{k5}, xmm0, xmm2"); // VADDSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x58, 0x0a], "vaddss xmm1, xmm0, dword [rdx]"); // VADDSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x58, 0x0a], "vaddss xmm1{k5}, xmm0, dword [rdx]"); // VADDSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xdd, 0x58, 0xca], "vaddss xmm1{k5}{z}{ru-sae}, xmm0, xmm2"); // VADDSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x58, 0x58, 0xca], "vaddss xmm1{ru-sae}, xmm0, xmm2"); // VADDSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x5d, 0x58, 0xca], "vaddss xmm1{k5}{ru-sae}, xmm0, xmm2"); // VADDSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x9d, 0x58, 0xca], "vaddss xmm1{k5}{z}{rne-sae}, xmm0, xmm2"); // VADDSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x18, 0x58, 0xca], "vaddss xmm1{rne-sae}, xmm0, xmm2"); // VADDSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x1d, 0x58, 0xca], "vaddss xmm1{k5}{rne-sae}, xmm0, xmm2"); // VADDSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xfd, 0x59, 0xca], "vmulss xmm1{k5}{z}{rz-sae}, xmm0, xmm2"); // VMULSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x78, 0x59, 0xca], "vmulss xmm1{rz-sae}, xmm0, xmm2"); // VMULSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x7d, 0x59, 0xca], "vmulss xmm1{k5}{rz-sae}, xmm0, xmm2"); // VMULSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xbd, 0x59, 0xca], "vmulss xmm1{k5}{z}{rd-sae}, xmm0, xmm2"); // VMULSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x38, 0x59, 0xca], "vmulss xmm1{rd-sae}, xmm0, xmm2"); // VMULSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x3d, 0x59, 0xca], "vmulss xmm1{k5}{rd-sae}, xmm0, xmm2"); // VMULSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x59, 0xca], "vmulss xmm1{k5}{z}, xmm0, xmm2"); // VMULSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x59, 0x0a], "vmulss xmm1{k5}{z}, xmm0, dword [rdx]"); // VMULSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x59, 0xca], "vmulss xmm1, xmm0, xmm2"); // VMULSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x59, 0xca], "vmulss xmm1{k5}, xmm0, xmm2"); // VMULSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x59, 0x0a], "vmulss xmm1, xmm0, dword [rdx]"); // VMULSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x59, 0x0a], "vmulss xmm1{k5}, xmm0, dword [rdx]"); // VMULSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xdd, 0x59, 0xca], "vmulss xmm1{k5}{z}{ru-sae}, xmm0, xmm2"); // VMULSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x58, 0x59, 0xca], "vmulss xmm1{ru-sae}, xmm0, xmm2"); // VMULSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x5d, 0x59, 0xca], "vmulss xmm1{k5}{ru-sae}, xmm0, xmm2"); // VMULSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x9d, 0x59, 0xca], "vmulss xmm1{k5}{z}{rne-sae}, xmm0, xmm2"); // VMULSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x18, 0x59, 0xca], "vmulss xmm1{rne-sae}, xmm0, xmm2"); // VMULSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x1d, 0x59, 0xca], "vmulss xmm1{k5}{rne-sae}, xmm0, xmm2"); // VMULSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xfd, 0x5a, 0xca], "vcvtss2sd xmm1{k5}{z}{sae}, xmm0, xmm2"); // VCVTSS2SD_XMMf64_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x78, 0x5a, 0xca], "vcvtss2sd xmm1{sae}, xmm0, xmm2"); // VCVTSS2SD_XMMf64_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x7d, 0x5a, 0xca], "vcvtss2sd xmm1{k5}{sae}, xmm0, xmm2"); // VCVTSS2SD_XMMf64_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x5a, 0xca], "vcvtss2sd xmm1{k5}{z}, xmm0, xmm2"); // VCVTSS2SD_XMMf64_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x5a, 0x0a], "vcvtss2sd xmm1{k5}{z}, xmm0, dword [rdx]"); // VCVTSS2SD_XMMf64_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x5a, 0xca], "vcvtss2sd xmm1, xmm0, xmm2"); // VCVTSS2SD_XMMf64_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x5a, 0xca], "vcvtss2sd xmm1{k5}, xmm0, xmm2"); // VCVTSS2SD_XMMf64_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x5a, 0x0a], "vcvtss2sd xmm1, xmm0, dword [rdx]"); // VCVTSS2SD_XMMf64_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x5a, 0x0a], "vcvtss2sd xmm1{k5}, xmm0, dword [rdx]"); // VCVTSS2SD_XMMf64_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xfd, 0x5b, 0xca], "vcvttps2dq zmm1{k5}{z}{sae}, zmm2"); // VCVTTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x78, 0x5b, 0xca], "vcvttps2dq zmm1{sae}, zmm2"); // VCVTTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x7d, 0x5b, 0xca], "vcvttps2dq zmm1{k5}{sae}, zmm2"); // VCVTTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xbd, 0x5b, 0x0a], "vcvttps2dq ymm1{k5}{z}, dword [rdx]{1to8}"); // VCVTTPS2DQ_YMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x38, 0x5b, 0x0a], "vcvttps2dq ymm1, dword [rdx]{1to8}"); // VCVTTPS2DQ_YMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x3d, 0x5b, 0x0a], "vcvttps2dq ymm1{k5}, dword [rdx]{1to8}"); // VCVTTPS2DQ_YMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x5b, 0xca], "vcvttps2dq ymm1{k5}{z}, ymm2"); // VCVTTPS2DQ_YMMi32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x5b, 0x0a], "vcvttps2dq ymm1{k5}{z}, ymmword [rdx]"); // VCVTTPS2DQ_YMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x5b, 0xca], "vcvttps2dq ymm1, ymm2"); // VCVTTPS2DQ_YMMi32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x5b, 0xca], "vcvttps2dq ymm1{k5}, ymm2"); // VCVTTPS2DQ_YMMi32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x5b, 0x0a], "vcvttps2dq ymm1, ymmword [rdx]"); // VCVTTPS2DQ_YMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x5b, 0x0a], "vcvttps2dq ymm1{k5}, ymmword [rdx]"); // VCVTTPS2DQ_YMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xdd, 0x5b, 0x0a], "vcvttps2dq zmm1{k5}{z}, dword [rdx]{1to16}"); // VCVTTPS2DQ_ZMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x58, 0x5b, 0x0a], "vcvttps2dq zmm1, dword [rdx]{1to16}"); // VCVTTPS2DQ_ZMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x5d, 0x5b, 0x0a], "vcvttps2dq zmm1{k5}, dword [rdx]{1to16}"); // VCVTTPS2DQ_ZMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x9d, 0x5b, 0x0a], "vcvttps2dq xmm1{k5}{z}, dword [rdx]{1to4}"); // VCVTTPS2DQ_XMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x18, 0x5b, 0x0a], "vcvttps2dq xmm1, dword [rdx]{1to4}"); // VCVTTPS2DQ_XMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x1d, 0x5b, 0x0a], "vcvttps2dq xmm1{k5}, dword [rdx]{1to4}"); // VCVTTPS2DQ_XMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xcd, 0x5b, 0xca], "vcvttps2dq zmm1{k5}{z}, zmm2"); // VCVTTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xcd, 0x5b, 0x0a], "vcvttps2dq zmm1{k5}{z}, zmmword [rdx]"); // VCVTTPS2DQ_ZMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x48, 0x5b, 0xca], "vcvttps2dq zmm1, zmm2"); // VCVTTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x4d, 0x5b, 0xca], "vcvttps2dq zmm1{k5}, zmm2"); // VCVTTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x48, 0x5b, 0x0a], "vcvttps2dq zmm1, zmmword [rdx]"); // VCVTTPS2DQ_ZMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x4d, 0x5b, 0x0a], "vcvttps2dq zmm1{k5}, zmmword [rdx]"); // VCVTTPS2DQ_ZMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x8d, 0x5b, 0xca], "vcvttps2dq xmm1{k5}{z}, xmm2"); // VCVTTPS2DQ_XMMi32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x8d, 0x5b, 0x0a], "vcvttps2dq xmm1{k5}{z}, xmmword [rdx]"); // VCVTTPS2DQ_XMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x08, 0x5b, 0xca], "vcvttps2dq xmm1, xmm2"); // VCVTTPS2DQ_XMMi32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x0d, 0x5b, 0xca], "vcvttps2dq xmm1{k5}, xmm2"); // VCVTTPS2DQ_XMMi32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x08, 0x5b, 0x0a], "vcvttps2dq xmm1, xmmword [rdx]"); // VCVTTPS2DQ_XMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x0d, 0x5b, 0x0a], "vcvttps2dq xmm1{k5}, xmmword [rdx]"); // VCVTTPS2DQ_XMMi32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xfd, 0x5c, 0xca], "vsubss xmm1{k5}{z}{rz-sae}, xmm0, xmm2"); // VSUBSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x78, 0x5c, 0xca], "vsubss xmm1{rz-sae}, xmm0, xmm2"); // VSUBSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x7d, 0x5c, 0xca], "vsubss xmm1{k5}{rz-sae}, xmm0, xmm2"); // VSUBSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xbd, 0x5c, 0xca], "vsubss xmm1{k5}{z}{rd-sae}, xmm0, xmm2"); // VSUBSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x38, 0x5c, 0xca], "vsubss xmm1{rd-sae}, xmm0, xmm2"); // VSUBSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x3d, 0x5c, 0xca], "vsubss xmm1{k5}{rd-sae}, xmm0, xmm2"); // VSUBSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x5c, 0xca], "vsubss xmm1{k5}{z}, xmm0, xmm2"); // VSUBSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x5c, 0x0a], "vsubss xmm1{k5}{z}, xmm0, dword [rdx]"); // VSUBSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x5c, 0xca], "vsubss xmm1, xmm0, xmm2"); // VSUBSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x5c, 0xca], "vsubss xmm1{k5}, xmm0, xmm2"); // VSUBSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x5c, 0x0a], "vsubss xmm1, xmm0, dword [rdx]"); // VSUBSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x5c, 0x0a], "vsubss xmm1{k5}, xmm0, dword [rdx]"); // VSUBSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xdd, 0x5c, 0xca], "vsubss xmm1{k5}{z}{ru-sae}, xmm0, xmm2"); // VSUBSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x58, 0x5c, 0xca], "vsubss xmm1{ru-sae}, xmm0, xmm2"); // VSUBSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x5d, 0x5c, 0xca], "vsubss xmm1{k5}{ru-sae}, xmm0, xmm2"); // VSUBSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x9d, 0x5c, 0xca], "vsubss xmm1{k5}{z}{rne-sae}, xmm0, xmm2"); // VSUBSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x18, 0x5c, 0xca], "vsubss xmm1{rne-sae}, xmm0, xmm2"); // VSUBSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x1d, 0x5c, 0xca], "vsubss xmm1{k5}{rne-sae}, xmm0, xmm2"); // VSUBSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xfd, 0x5d, 0xca], "vminss xmm1{k5}{z}{sae}, xmm0, xmm2"); // VMINSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x78, 0x5d, 0xca], "vminss xmm1{sae}, xmm0, xmm2"); // VMINSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x7d, 0x5d, 0xca], "vminss xmm1{k5}{sae}, xmm0, xmm2"); // VMINSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x5d, 0xca], "vminss xmm1{k5}{z}, xmm0, xmm2"); // VMINSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x5d, 0x0a], "vminss xmm1{k5}{z}, xmm0, dword [rdx]"); // VMINSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x5d, 0xca], "vminss xmm1, xmm0, xmm2"); // VMINSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x5d, 0xca], "vminss xmm1{k5}, xmm0, xmm2"); // VMINSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x5d, 0x0a], "vminss xmm1, xmm0, dword [rdx]"); // VMINSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x5d, 0x0a], "vminss xmm1{k5}, xmm0, dword [rdx]"); // VMINSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xfd, 0x5e, 0xca], "vdivss xmm1{k5}{z}{rz-sae}, xmm0, xmm2"); // VDIVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x78, 0x5e, 0xca], "vdivss xmm1{rz-sae}, xmm0, xmm2"); // VDIVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x7d, 0x5e, 0xca], "vdivss xmm1{k5}{rz-sae}, xmm0, xmm2"); // VDIVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xbd, 0x5e, 0xca], "vdivss xmm1{k5}{z}{rd-sae}, xmm0, xmm2"); // VDIVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x38, 0x5e, 0xca], "vdivss xmm1{rd-sae}, xmm0, xmm2"); // VDIVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x3d, 0x5e, 0xca], "vdivss xmm1{k5}{rd-sae}, xmm0, xmm2"); // VDIVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x5e, 0xca], "vdivss xmm1{k5}{z}, xmm0, xmm2"); // VDIVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x5e, 0x0a], "vdivss xmm1{k5}{z}, xmm0, dword [rdx]"); // VDIVSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x5e, 0xca], "vdivss xmm1, xmm0, xmm2"); // VDIVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x5e, 0xca], "vdivss xmm1{k5}, xmm0, xmm2"); // VDIVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x5e, 0x0a], "vdivss xmm1, xmm0, dword [rdx]"); // VDIVSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x5e, 0x0a], "vdivss xmm1{k5}, xmm0, dword [rdx]"); // VDIVSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xdd, 0x5e, 0xca], "vdivss xmm1{k5}{z}{ru-sae}, xmm0, xmm2"); // VDIVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x58, 0x5e, 0xca], "vdivss xmm1{ru-sae}, xmm0, xmm2"); // VDIVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x5d, 0x5e, 0xca], "vdivss xmm1{k5}{ru-sae}, xmm0, xmm2"); // VDIVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x9d, 0x5e, 0xca], "vdivss xmm1{k5}{z}{rne-sae}, xmm0, xmm2"); // VDIVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x18, 0x5e, 0xca], "vdivss xmm1{rne-sae}, xmm0, xmm2"); // VDIVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x1d, 0x5e, 0xca], "vdivss xmm1{k5}{rne-sae}, xmm0, xmm2"); // VDIVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xfd, 0x5f, 0xca], "vmaxss xmm1{k5}{z}{sae}, xmm0, xmm2"); // VMAXSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x78, 0x5f, 0xca], "vmaxss xmm1{sae}, xmm0, xmm2"); // VMAXSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x7d, 0x5f, 0xca], "vmaxss xmm1{k5}{sae}, xmm0, xmm2"); // VMAXSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x5f, 0xca], "vmaxss xmm1{k5}{z}, xmm0, xmm2"); // VMAXSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x5f, 0x0a], "vmaxss xmm1{k5}{z}, xmm0, dword [rdx]"); // VMAXSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x5f, 0xca], "vmaxss xmm1, xmm0, xmm2"); // VMAXSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x5f, 0xca], "vmaxss xmm1{k5}, xmm0, xmm2"); // VMAXSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x5f, 0x0a], "vmaxss xmm1, xmm0, dword [rdx]"); // VMAXSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x5f, 0x0a], "vmaxss xmm1{k5}, xmm0, dword [rdx]"); // VMAXSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xad, 0x6f, 0xca], "vmovdqu64 ymm1{k5}{z}, ymm2"); // VMOVDQU64_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xad, 0x6f, 0x0a], "vmovdqu64 ymm1{k5}{z}, ymmword [rdx]"); // VMOVDQU64_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x6f, 0xca], "vmovdqu64 ymm1, ymm2"); // VMOVDQU64_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x2d, 0x6f, 0xca], "vmovdqu64 ymm1{k5}, ymm2"); // VMOVDQU64_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x6f, 0x0a], "vmovdqu64 ymm1, ymmword [rdx]"); // VMOVDQU64_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x2d, 0x6f, 0x0a], "vmovdqu64 ymm1{k5}, ymmword [rdx]"); // VMOVDQU64_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x6f, 0xca], "vmovdqu32 ymm1{k5}{z}, ymm2"); // VMOVDQU32_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x6f, 0x0a], "vmovdqu32 ymm1{k5}{z}, ymmword [rdx]"); // VMOVDQU32_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x6f, 0xca], "vmovdqu32 ymm1, ymm2"); // VMOVDQU32_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x6f, 0xca], "vmovdqu32 ymm1{k5}, ymm2"); // VMOVDQU32_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x6f, 0x0a], "vmovdqu32 ymm1, ymmword [rdx]"); // VMOVDQU32_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x6f, 0x0a], "vmovdqu32 ymm1{k5}, ymmword [rdx]"); // VMOVDQU32_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xcd, 0x6f, 0xca], "vmovdqu64 zmm1{k5}{z}, zmm2"); // VMOVDQU64_ZMMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xcd, 0x6f, 0x0a], "vmovdqu64 zmm1{k5}{z}, zmmword [rdx]"); // VMOVDQU64_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x48, 0x6f, 0xca], "vmovdqu64 zmm1, zmm2"); // VMOVDQU64_ZMMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x4d, 0x6f, 0xca], "vmovdqu64 zmm1{k5}, zmm2"); // VMOVDQU64_ZMMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x48, 0x6f, 0x0a], "vmovdqu64 zmm1, zmmword [rdx]"); // VMOVDQU64_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x4d, 0x6f, 0x0a], "vmovdqu64 zmm1{k5}, zmmword [rdx]"); // VMOVDQU64_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x8d, 0x6f, 0xca], "vmovdqu64 xmm1{k5}{z}, xmm2"); // VMOVDQU64_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x8d, 0x6f, 0x0a], "vmovdqu64 xmm1{k5}{z}, xmmword [rdx]"); // VMOVDQU64_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x08, 0x6f, 0xca], "vmovdqu64 xmm1, xmm2"); // VMOVDQU64_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x0d, 0x6f, 0xca], "vmovdqu64 xmm1{k5}, xmm2"); // VMOVDQU64_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x08, 0x6f, 0x0a], "vmovdqu64 xmm1, xmmword [rdx]"); // VMOVDQU64_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x0d, 0x6f, 0x0a], "vmovdqu64 xmm1{k5}, xmmword [rdx]"); // VMOVDQU64_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xcd, 0x6f, 0xca], "vmovdqu32 zmm1{k5}{z}, zmm2"); // VMOVDQU32_ZMMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xcd, 0x6f, 0x0a], "vmovdqu32 zmm1{k5}{z}, zmmword [rdx]"); // VMOVDQU32_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x48, 0x6f, 0xca], "vmovdqu32 zmm1, zmm2"); // VMOVDQU32_ZMMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x4d, 0x6f, 0xca], "vmovdqu32 zmm1{k5}, zmm2"); // VMOVDQU32_ZMMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x48, 0x6f, 0x0a], "vmovdqu32 zmm1, zmmword [rdx]"); // VMOVDQU32_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x4d, 0x6f, 0x0a], "vmovdqu32 zmm1{k5}, zmmword [rdx]"); // VMOVDQU32_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x8d, 0x6f, 0xca], "vmovdqu32 xmm1{k5}{z}, xmm2"); // VMOVDQU32_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x8d, 0x6f, 0x0a], "vmovdqu32 xmm1{k5}{z}, xmmword [rdx]"); // VMOVDQU32_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x08, 0x6f, 0xca], "vmovdqu32 xmm1, xmm2"); // VMOVDQU32_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x0d, 0x6f, 0xca], "vmovdqu32 xmm1{k5}, xmm2"); // VMOVDQU32_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x08, 0x6f, 0x0a], "vmovdqu32 xmm1, xmmword [rdx]"); // VMOVDQU32_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x0d, 0x6f, 0x0a], "vmovdqu32 xmm1{k5}, xmmword [rdx]"); // VMOVDQU32_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xad, 0x70, 0xca, 0xcc], "vpshufhw ymm1{k5}{z}, ymm2, 0xcc"); // VPSHUFHW_YMMu16_MASKmskw_YMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xad, 0x70, 0x0a, 0xcc], "vpshufhw ymm1{k5}{z}, ymmword [rdx], 0xcc"); // VPSHUFHW_YMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x70, 0xca, 0xcc], "vpshufhw ymm1, ymm2, 0xcc"); // VPSHUFHW_YMMu16_MASKmskw_YMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x2d, 0x70, 0xca, 0xcc], "vpshufhw ymm1{k5}, ymm2, 0xcc"); // VPSHUFHW_YMMu16_MASKmskw_YMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x70, 0x0a, 0xcc], "vpshufhw ymm1, ymmword [rdx], 0xcc"); // VPSHUFHW_YMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x2d, 0x70, 0x0a, 0xcc], "vpshufhw ymm1{k5}, ymmword [rdx], 0xcc"); // VPSHUFHW_YMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xcd, 0x70, 0xca, 0xcc], "vpshufhw zmm1{k5}{z}, zmm2, 0xcc"); // VPSHUFHW_ZMMu16_MASKmskw_ZMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xcd, 0x70, 0x0a, 0xcc], "vpshufhw zmm1{k5}{z}, zmmword [rdx], 0xcc"); // VPSHUFHW_ZMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x48, 0x70, 0xca, 0xcc], "vpshufhw zmm1, zmm2, 0xcc"); // VPSHUFHW_ZMMu16_MASKmskw_ZMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x4d, 0x70, 0xca, 0xcc], "vpshufhw zmm1{k5}, zmm2, 0xcc"); // VPSHUFHW_ZMMu16_MASKmskw_ZMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x48, 0x70, 0x0a, 0xcc], "vpshufhw zmm1, zmmword [rdx], 0xcc"); // VPSHUFHW_ZMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x4d, 0x70, 0x0a, 0xcc], "vpshufhw zmm1{k5}, zmmword [rdx], 0xcc"); // VPSHUFHW_ZMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x8d, 0x70, 0xca, 0xcc], "vpshufhw xmm1{k5}{z}, xmm2, 0xcc"); // VPSHUFHW_XMMu16_MASKmskw_XMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x8d, 0x70, 0x0a, 0xcc], "vpshufhw xmm1{k5}{z}, xmmword [rdx], 0xcc"); // VPSHUFHW_XMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x08, 0x70, 0xca, 0xcc], "vpshufhw xmm1, xmm2, 0xcc"); // VPSHUFHW_XMMu16_MASKmskw_XMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x0d, 0x70, 0xca, 0xcc], "vpshufhw xmm1{k5}, xmm2, 0xcc"); // VPSHUFHW_XMMu16_MASKmskw_XMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x08, 0x70, 0x0a, 0xcc], "vpshufhw xmm1, xmmword [rdx], 0xcc"); // VPSHUFHW_XMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x0d, 0x70, 0x0a, 0xcc], "vpshufhw xmm1{k5}, xmmword [rdx], 0xcc"); // VPSHUFHW_XMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x78, 0x78, 0xca], "vcvttss2usi rcx{sae}, xmm2"); // VCVTTSS2USI_GPR64u64_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x78, 0xca], "vcvttss2usi rcx, xmm2"); // VCVTTSS2USI_GPR64u64_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x78, 0x0a], "vcvttss2usi rcx, dword [rdx]"); // VCVTTSS2USI_GPR64u64_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x78, 0x78, 0xca], "vcvttss2usi ecx{sae}, xmm2"); // VCVTTSS2USI_GPR32u32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x78, 0xca], "vcvttss2usi ecx, xmm2"); // VCVTTSS2USI_GPR32u32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x78, 0x0a], "vcvttss2usi ecx, dword [rdx]"); // VCVTTSS2USI_GPR32u32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x78, 0x79, 0xca], "vcvtss2usi rcx{rz-sae}, xmm2"); // VCVTSS2USI_GPR64u64_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x38, 0x79, 0xca], "vcvtss2usi rcx{rd-sae}, xmm2"); // VCVTSS2USI_GPR64u64_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x79, 0xca], "vcvtss2usi rcx, xmm2"); // VCVTSS2USI_GPR64u64_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x79, 0x0a], "vcvtss2usi rcx, dword [rdx]"); // VCVTSS2USI_GPR64u64_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x78, 0x79, 0xca], "vcvtss2usi ecx{rz-sae}, xmm2"); // VCVTSS2USI_GPR32u32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x38, 0x79, 0xca], "vcvtss2usi ecx{rd-sae}, xmm2"); // VCVTSS2USI_GPR32u32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x79, 0xca], "vcvtss2usi ecx, xmm2"); // VCVTSS2USI_GPR32u32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x79, 0x0a], "vcvtss2usi ecx, dword [rdx]"); // VCVTSS2USI_GPR32u32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x58, 0x79, 0xca], "vcvtss2usi rcx{ru-sae}, xmm2"); // VCVTSS2USI_GPR64u64_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x18, 0x79, 0xca], "vcvtss2usi rcx{rne-sae}, xmm2"); // VCVTSS2USI_GPR64u64_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x58, 0x79, 0xca], "vcvtss2usi ecx{ru-sae}, xmm2"); // VCVTSS2USI_GPR32u32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x18, 0x79, 0xca], "vcvtss2usi ecx{rne-sae}, xmm2"); // VCVTSS2USI_GPR32u32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xfd, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{z}{rz-sae}, zmm2"); // VCVTUQQ2PD_ZMMf64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x78, 0x7a, 0xca], "vcvtuqq2pd zmm1{rz-sae}, zmm2"); // VCVTUQQ2PD_ZMMf64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x7d, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{rz-sae}, zmm2"); // VCVTUQQ2PD_ZMMf64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xbd, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{z}{rd-sae}, zmm2"); // VCVTUQQ2PD_ZMMf64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xbd, 0x7a, 0x0a], "vcvtuqq2pd ymm1{k5}{z}, qword [rdx]{1to4}"); // VCVTUQQ2PD_YMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x38, 0x7a, 0xca], "vcvtuqq2pd zmm1{rd-sae}, zmm2"); // VCVTUQQ2PD_ZMMf64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x3d, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{rd-sae}, zmm2"); // VCVTUQQ2PD_ZMMf64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x38, 0x7a, 0x0a], "vcvtuqq2pd ymm1, qword [rdx]{1to4}"); // VCVTUQQ2PD_YMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x3d, 0x7a, 0x0a], "vcvtuqq2pd ymm1{k5}, qword [rdx]{1to4}"); // VCVTUQQ2PD_YMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xad, 0x7a, 0xca], "vcvtuqq2pd ymm1{k5}{z}, ymm2"); // VCVTUQQ2PD_YMMf64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xad, 0x7a, 0x0a], "vcvtuqq2pd ymm1{k5}{z}, ymmword [rdx]"); // VCVTUQQ2PD_YMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x7a, 0xca], "vcvtuqq2pd ymm1, ymm2"); // VCVTUQQ2PD_YMMf64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x2d, 0x7a, 0xca], "vcvtuqq2pd ymm1{k5}, ymm2"); // VCVTUQQ2PD_YMMf64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x7a, 0x0a], "vcvtuqq2pd ymm1, ymmword [rdx]"); // VCVTUQQ2PD_YMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x2d, 0x7a, 0x0a], "vcvtuqq2pd ymm1{k5}, ymmword [rdx]"); // VCVTUQQ2PD_YMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xbd, 0x7a, 0x0a], "vcvtudq2pd ymm1{k5}{z}, dword [rdx]{1to4}"); // VCVTUDQ2PD_YMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x38, 0x7a, 0x0a], "vcvtudq2pd ymm1, dword [rdx]{1to4}"); // VCVTUDQ2PD_YMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x3d, 0x7a, 0x0a], "vcvtudq2pd ymm1{k5}, dword [rdx]{1to4}"); // VCVTUDQ2PD_YMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x7a, 0xca], "vcvtudq2pd ymm1{k5}{z}, xmm2"); // VCVTUDQ2PD_YMMf64_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x7a, 0x0a], "vcvtudq2pd ymm1{k5}{z}, xmmword [rdx]"); // VCVTUDQ2PD_YMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x7a, 0xca], "vcvtudq2pd ymm1, xmm2"); // VCVTUDQ2PD_YMMf64_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x7a, 0xca], "vcvtudq2pd ymm1{k5}, xmm2"); // VCVTUDQ2PD_YMMf64_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x7a, 0x0a], "vcvtudq2pd ymm1, xmmword [rdx]"); // VCVTUDQ2PD_YMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x7a, 0x0a], "vcvtudq2pd ymm1{k5}, xmmword [rdx]"); // VCVTUDQ2PD_YMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xdd, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{z}{ru-sae}, zmm2"); // VCVTUQQ2PD_ZMMf64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xdd, 0x7a, 0x0a], "vcvtuqq2pd zmm1{k5}{z}, qword [rdx]{1to8}"); // VCVTUQQ2PD_ZMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x58, 0x7a, 0xca], "vcvtuqq2pd zmm1{ru-sae}, zmm2"); // VCVTUQQ2PD_ZMMf64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x5d, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{ru-sae}, zmm2"); // VCVTUQQ2PD_ZMMf64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x58, 0x7a, 0x0a], "vcvtuqq2pd zmm1, qword [rdx]{1to8}"); // VCVTUQQ2PD_ZMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x5d, 0x7a, 0x0a], "vcvtuqq2pd zmm1{k5}, qword [rdx]{1to8}"); // VCVTUQQ2PD_ZMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x9d, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{z}{rne-sae}, zmm2"); // VCVTUQQ2PD_ZMMf64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x9d, 0x7a, 0x0a], "vcvtuqq2pd xmm1{k5}{z}, qword [rdx]{1to2}"); // VCVTUQQ2PD_XMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x18, 0x7a, 0xca], "vcvtuqq2pd zmm1{rne-sae}, zmm2"); // VCVTUQQ2PD_ZMMf64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x1d, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{rne-sae}, zmm2"); // VCVTUQQ2PD_ZMMf64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x18, 0x7a, 0x0a], "vcvtuqq2pd xmm1, qword [rdx]{1to2}"); // VCVTUQQ2PD_XMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x1d, 0x7a, 0x0a], "vcvtuqq2pd xmm1{k5}, qword [rdx]{1to2}"); // VCVTUQQ2PD_XMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xcd, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}{z}, zmm2"); // VCVTUQQ2PD_ZMMf64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xcd, 0x7a, 0x0a], "vcvtuqq2pd zmm1{k5}{z}, zmmword [rdx]"); // VCVTUQQ2PD_ZMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x48, 0x7a, 0xca], "vcvtuqq2pd zmm1, zmm2"); // VCVTUQQ2PD_ZMMf64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x4d, 0x7a, 0xca], "vcvtuqq2pd zmm1{k5}, zmm2"); // VCVTUQQ2PD_ZMMf64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x48, 0x7a, 0x0a], "vcvtuqq2pd zmm1, zmmword [rdx]"); // VCVTUQQ2PD_ZMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x4d, 0x7a, 0x0a], "vcvtuqq2pd zmm1{k5}, zmmword [rdx]"); // VCVTUQQ2PD_ZMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x8d, 0x7a, 0xca], "vcvtuqq2pd xmm1{k5}{z}, xmm2"); // VCVTUQQ2PD_XMMf64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x8d, 0x7a, 0x0a], "vcvtuqq2pd xmm1{k5}{z}, xmmword [rdx]"); // VCVTUQQ2PD_XMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x08, 0x7a, 0xca], "vcvtuqq2pd xmm1, xmm2"); // VCVTUQQ2PD_XMMf64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x0d, 0x7a, 0xca], "vcvtuqq2pd xmm1{k5}, xmm2"); // VCVTUQQ2PD_XMMf64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x08, 0x7a, 0x0a], "vcvtuqq2pd xmm1, xmmword [rdx]"); // VCVTUQQ2PD_XMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x0d, 0x7a, 0x0a], "vcvtuqq2pd xmm1{k5}, xmmword [rdx]"); // VCVTUQQ2PD_XMMf64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xdd, 0x7a, 0x0a], "vcvtudq2pd zmm1{k5}{z}, dword [rdx]{1to8}"); // VCVTUDQ2PD_ZMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x58, 0x7a, 0x0a], "vcvtudq2pd zmm1, dword [rdx]{1to8}"); // VCVTUDQ2PD_ZMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x5d, 0x7a, 0x0a], "vcvtudq2pd zmm1{k5}, dword [rdx]{1to8}"); // VCVTUDQ2PD_ZMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x9d, 0x7a, 0x0a], "vcvtudq2pd xmm1{k5}{z}, dword [rdx]{1to2}"); // VCVTUDQ2PD_XMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x18, 0x7a, 0x0a], "vcvtudq2pd xmm1, dword [rdx]{1to2}"); // VCVTUDQ2PD_XMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x1d, 0x7a, 0x0a], "vcvtudq2pd xmm1{k5}, dword [rdx]{1to2}"); // VCVTUDQ2PD_XMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xcd, 0x7a, 0xca], "vcvtudq2pd zmm1{k5}{z}, ymm2"); // VCVTUDQ2PD_ZMMf64_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xcd, 0x7a, 0x0a], "vcvtudq2pd zmm1{k5}{z}, ymmword [rdx]"); // VCVTUDQ2PD_ZMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x48, 0x7a, 0xca], "vcvtudq2pd zmm1, ymm2"); // VCVTUDQ2PD_ZMMf64_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x4d, 0x7a, 0xca], "vcvtudq2pd zmm1{k5}, ymm2"); // VCVTUDQ2PD_ZMMf64_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x48, 0x7a, 0x0a], "vcvtudq2pd zmm1, ymmword [rdx]"); // VCVTUDQ2PD_ZMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x4d, 0x7a, 0x0a], "vcvtudq2pd zmm1{k5}, ymmword [rdx]"); // VCVTUDQ2PD_ZMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x8d, 0x7a, 0xca], "vcvtudq2pd xmm1{k5}{z}, xmm2"); // VCVTUDQ2PD_XMMf64_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x8d, 0x7a, 0x0a], "vcvtudq2pd xmm1{k5}{z}, qword [rdx]"); // VCVTUDQ2PD_XMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x08, 0x7a, 0xca], "vcvtudq2pd xmm1, xmm2"); // VCVTUDQ2PD_XMMf64_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x0d, 0x7a, 0xca], "vcvtudq2pd xmm1{k5}, xmm2"); // VCVTUDQ2PD_XMMf64_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x08, 0x7a, 0x0a], "vcvtudq2pd xmm1, qword [rdx]"); // VCVTUDQ2PD_XMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x0d, 0x7a, 0x0a], "vcvtudq2pd xmm1{k5}, qword [rdx]"); // VCVTUDQ2PD_XMMf64_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x78, 0x7b, 0xca], "vcvtusi2ss xmm1{rz-sae}, xmm0, rdx"); // VCVTUSI2SS_XMMf32_XMMf32_GPR64u64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x38, 0x7b, 0xca], "vcvtusi2ss xmm1{rd-sae}, xmm0, rdx"); // VCVTUSI2SS_XMMf32_XMMf32_GPR64u64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x7b, 0xca], "vcvtusi2ss xmm1, xmm0, rdx"); // VCVTUSI2SS_XMMf32_XMMf32_GPR64u64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x7b, 0x0a], "vcvtusi2ss xmm1, xmm0, qword [rdx]"); // VCVTUSI2SS_XMMf32_XMMf32_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x78, 0x7b, 0xca], "vcvtusi2ss xmm1{rz-sae}, xmm0, edx"); // VCVTUSI2SS_XMMf32_XMMf32_GPR32u32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x38, 0x7b, 0xca], "vcvtusi2ss xmm1{rd-sae}, xmm0, edx"); // VCVTUSI2SS_XMMf32_XMMf32_GPR32u32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x7b, 0xca], "vcvtusi2ss xmm1, xmm0, edx"); // VCVTUSI2SS_XMMf32_XMMf32_GPR32u32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x7b, 0x0a], "vcvtusi2ss xmm1, xmm0, dword [rdx]"); // VCVTUSI2SS_XMMf32_XMMf32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x58, 0x7b, 0xca], "vcvtusi2ss xmm1{ru-sae}, xmm0, rdx"); // VCVTUSI2SS_XMMf32_XMMf32_GPR64u64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x18, 0x7b, 0xca], "vcvtusi2ss xmm1{rne-sae}, xmm0, rdx"); // VCVTUSI2SS_XMMf32_XMMf32_GPR64u64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x58, 0x7b, 0xca], "vcvtusi2ss xmm1{ru-sae}, xmm0, edx"); // VCVTUSI2SS_XMMf32_XMMf32_GPR32u32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x18, 0x7b, 0xca], "vcvtusi2ss xmm1{rne-sae}, xmm0, edx"); // VCVTUSI2SS_XMMf32_XMMf32_GPR32u32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x08, 0x7e, 0xca], "vmovq xmm1, xmm2"); // VMOVQ_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x08, 0x7e, 0x0a], "vmovq xmm1, qword [rdx]"); // VMOVQ_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xad, 0x7f, 0xca], "vmovdqu64 ymm2{k5}{z}, ymm1"); // VMOVDQU64_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x7f, 0xca], "vmovdqu64 ymm2, ymm1"); // VMOVDQU64_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x2d, 0x7f, 0xca], "vmovdqu64 ymm2{k5}, ymm1"); // VMOVDQU64_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0x7f, 0x0a], "vmovdqu64 ymmword [rdx], ymm1"); // VMOVDQU64_MEMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x2d, 0x7f, 0x0a], "vmovdqu64 ymmword [rdx]{k5}, ymm1"); // VMOVDQU64_MEMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0x7f, 0xca], "vmovdqu32 ymm2{k5}{z}, ymm1"); // VMOVDQU32_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x7f, 0xca], "vmovdqu32 ymm2, ymm1"); // VMOVDQU32_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x7f, 0xca], "vmovdqu32 ymm2{k5}, ymm1"); // VMOVDQU32_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0x7f, 0x0a], "vmovdqu32 ymmword [rdx], ymm1"); // VMOVDQU32_MEMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0x7f, 0x0a], "vmovdqu32 ymmword [rdx]{k5}, ymm1"); // VMOVDQU32_MEMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xcd, 0x7f, 0xca], "vmovdqu64 zmm2{k5}{z}, zmm1"); // VMOVDQU64_ZMMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x48, 0x7f, 0xca], "vmovdqu64 zmm2, zmm1"); // VMOVDQU64_ZMMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x4d, 0x7f, 0xca], "vmovdqu64 zmm2{k5}, zmm1"); // VMOVDQU64_ZMMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x48, 0x7f, 0x0a], "vmovdqu64 zmmword [rdx], zmm1"); // VMOVDQU64_MEMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x4d, 0x7f, 0x0a], "vmovdqu64 zmmword [rdx]{k5}, zmm1"); // VMOVDQU64_MEMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x8d, 0x7f, 0xca], "vmovdqu64 xmm2{k5}{z}, xmm1"); // VMOVDQU64_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x08, 0x7f, 0xca], "vmovdqu64 xmm2, xmm1"); // VMOVDQU64_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x0d, 0x7f, 0xca], "vmovdqu64 xmm2{k5}, xmm1"); // VMOVDQU64_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x08, 0x7f, 0x0a], "vmovdqu64 xmmword [rdx], xmm1"); // VMOVDQU64_MEMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x0d, 0x7f, 0x0a], "vmovdqu64 xmmword [rdx]{k5}, xmm1"); // VMOVDQU64_MEMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xcd, 0x7f, 0xca], "vmovdqu32 zmm2{k5}{z}, zmm1"); // VMOVDQU32_ZMMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x48, 0x7f, 0xca], "vmovdqu32 zmm2, zmm1"); // VMOVDQU32_ZMMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x4d, 0x7f, 0xca], "vmovdqu32 zmm2{k5}, zmm1"); // VMOVDQU32_ZMMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x48, 0x7f, 0x0a], "vmovdqu32 zmmword [rdx], zmm1"); // VMOVDQU32_MEMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x4d, 0x7f, 0x0a], "vmovdqu32 zmmword [rdx]{k5}, zmm1"); // VMOVDQU32_MEMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x8d, 0x7f, 0xca], "vmovdqu32 xmm2{k5}{z}, xmm1"); // VMOVDQU32_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x08, 0x7f, 0xca], "vmovdqu32 xmm2, xmm1"); // VMOVDQU32_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x0d, 0x7f, 0xca], "vmovdqu32 xmm2{k5}, xmm1"); // VMOVDQU32_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x08, 0x7f, 0x0a], "vmovdqu32 xmmword [rdx], xmm1"); // VMOVDQU32_MEMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x0d, 0x7f, 0x0a], "vmovdqu32 xmmword [rdx]{k5}, xmm1"); // VMOVDQU32_MEMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x78, 0xc2, 0xca, 0xcc], "vcmpss k1{sae}, xmm0, xmm2, 0xcc"); // VCMPSS_MASKmskw_MASKmskw_XMMf32_XMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x7d, 0xc2, 0xca, 0xcc], "vcmpss k1{k5}{sae}, xmm0, xmm2, 0xcc"); // VCMPSS_MASKmskw_MASKmskw_XMMf32_XMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0xc2, 0xca, 0xcc], "vcmpss k1, xmm0, xmm2, 0xcc"); // VCMPSS_MASKmskw_MASKmskw_XMMf32_XMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0xc2, 0xca, 0xcc], "vcmpss k1{k5}, xmm0, xmm2, 0xcc"); // VCMPSS_MASKmskw_MASKmskw_XMMf32_XMMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0xc2, 0x0a, 0xcc], "vcmpss k1, xmm0, dword [rdx], 0xcc"); // VCMPSS_MASKmskw_MASKmskw_XMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0xc2, 0x0a, 0xcc], "vcmpss k1{k5}, xmm0, dword [rdx], 0xcc"); // VCMPSS_MASKmskw_MASKmskw_XMMf32_MEMf32_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xfd, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{z}{rz-sae}, zmm2"); // VCVTQQ2PD_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x78, 0xe6, 0xca], "vcvtqq2pd zmm1{rz-sae}, zmm2"); // VCVTQQ2PD_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x7d, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{rz-sae}, zmm2"); // VCVTQQ2PD_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xbd, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{z}{rd-sae}, zmm2"); // VCVTQQ2PD_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xbd, 0xe6, 0x0a], "vcvtqq2pd ymm1{k5}{z}, qword [rdx]{1to4}"); // VCVTQQ2PD_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x38, 0xe6, 0xca], "vcvtqq2pd zmm1{rd-sae}, zmm2"); // VCVTQQ2PD_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x3d, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{rd-sae}, zmm2"); // VCVTQQ2PD_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x38, 0xe6, 0x0a], "vcvtqq2pd ymm1, qword [rdx]{1to4}"); // VCVTQQ2PD_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x3d, 0xe6, 0x0a], "vcvtqq2pd ymm1{k5}, qword [rdx]{1to4}"); // VCVTQQ2PD_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xad, 0xe6, 0xca], "vcvtqq2pd ymm1{k5}{z}, ymm2"); // VCVTQQ2PD_YMMi64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xad, 0xe6, 0x0a], "vcvtqq2pd ymm1{k5}{z}, ymmword [rdx]"); // VCVTQQ2PD_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0xe6, 0xca], "vcvtqq2pd ymm1, ymm2"); // VCVTQQ2PD_YMMi64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x2d, 0xe6, 0xca], "vcvtqq2pd ymm1{k5}, ymm2"); // VCVTQQ2PD_YMMi64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x28, 0xe6, 0x0a], "vcvtqq2pd ymm1, ymmword [rdx]"); // VCVTQQ2PD_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x2d, 0xe6, 0x0a], "vcvtqq2pd ymm1{k5}, ymmword [rdx]"); // VCVTQQ2PD_YMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xbd, 0xe6, 0x0a], "vcvtdq2pd ymm1{k5}{z}, dword [rdx]{1to4}"); // VCVTDQ2PD_YMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x38, 0xe6, 0x0a], "vcvtdq2pd ymm1, dword [rdx]{1to4}"); // VCVTDQ2PD_YMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x3d, 0xe6, 0x0a], "vcvtdq2pd ymm1{k5}, dword [rdx]{1to4}"); // VCVTDQ2PD_YMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0xe6, 0xca], "vcvtdq2pd ymm1{k5}{z}, xmm2"); // VCVTDQ2PD_YMMf64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xad, 0xe6, 0x0a], "vcvtdq2pd ymm1{k5}{z}, xmmword [rdx]"); // VCVTDQ2PD_YMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0xe6, 0xca], "vcvtdq2pd ymm1, xmm2"); // VCVTDQ2PD_YMMf64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0xe6, 0xca], "vcvtdq2pd ymm1{k5}, xmm2"); // VCVTDQ2PD_YMMf64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x28, 0xe6, 0x0a], "vcvtdq2pd ymm1, xmmword [rdx]"); // VCVTDQ2PD_YMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x2d, 0xe6, 0x0a], "vcvtdq2pd ymm1{k5}, xmmword [rdx]"); // VCVTDQ2PD_YMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xdd, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{z}{ru-sae}, zmm2"); // VCVTQQ2PD_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xdd, 0xe6, 0x0a], "vcvtqq2pd zmm1{k5}{z}, qword [rdx]{1to8}"); // VCVTQQ2PD_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x58, 0xe6, 0xca], "vcvtqq2pd zmm1{ru-sae}, zmm2"); // VCVTQQ2PD_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x5d, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{ru-sae}, zmm2"); // VCVTQQ2PD_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x58, 0xe6, 0x0a], "vcvtqq2pd zmm1, qword [rdx]{1to8}"); // VCVTQQ2PD_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x5d, 0xe6, 0x0a], "vcvtqq2pd zmm1{k5}, qword [rdx]{1to8}"); // VCVTQQ2PD_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x9d, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{z}{rne-sae}, zmm2"); // VCVTQQ2PD_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x9d, 0xe6, 0x0a], "vcvtqq2pd xmm1{k5}{z}, qword [rdx]{1to2}"); // VCVTQQ2PD_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x18, 0xe6, 0xca], "vcvtqq2pd zmm1{rne-sae}, zmm2"); // VCVTQQ2PD_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x1d, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{rne-sae}, zmm2"); // VCVTQQ2PD_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x18, 0xe6, 0x0a], "vcvtqq2pd xmm1, qword [rdx]{1to2}"); // VCVTQQ2PD_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x1d, 0xe6, 0x0a], "vcvtqq2pd xmm1{k5}, qword [rdx]{1to2}"); // VCVTQQ2PD_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xcd, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}{z}, zmm2"); // VCVTQQ2PD_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0xcd, 0xe6, 0x0a], "vcvtqq2pd zmm1{k5}{z}, zmmword [rdx]"); // VCVTQQ2PD_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x48, 0xe6, 0xca], "vcvtqq2pd zmm1, zmm2"); // VCVTQQ2PD_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x4d, 0xe6, 0xca], "vcvtqq2pd zmm1{k5}, zmm2"); // VCVTQQ2PD_ZMMi64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x48, 0xe6, 0x0a], "vcvtqq2pd zmm1, zmmword [rdx]"); // VCVTQQ2PD_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x4d, 0xe6, 0x0a], "vcvtqq2pd zmm1{k5}, zmmword [rdx]"); // VCVTQQ2PD_ZMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x8d, 0xe6, 0xca], "vcvtqq2pd xmm1{k5}{z}, xmm2"); // VCVTQQ2PD_XMMi64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x8d, 0xe6, 0x0a], "vcvtqq2pd xmm1{k5}{z}, xmmword [rdx]"); // VCVTQQ2PD_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x08, 0xe6, 0xca], "vcvtqq2pd xmm1, xmm2"); // VCVTQQ2PD_XMMi64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x0d, 0xe6, 0xca], "vcvtqq2pd xmm1{k5}, xmm2"); // VCVTQQ2PD_XMMi64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x08, 0xe6, 0x0a], "vcvtqq2pd xmm1, xmmword [rdx]"); // VCVTQQ2PD_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xfe, 0x0d, 0xe6, 0x0a], "vcvtqq2pd xmm1{k5}, xmmword [rdx]"); // VCVTQQ2PD_XMMi64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xdd, 0xe6, 0x0a], "vcvtdq2pd zmm1{k5}{z}, dword [rdx]{1to8}"); // VCVTDQ2PD_ZMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x58, 0xe6, 0x0a], "vcvtdq2pd zmm1, dword [rdx]{1to8}"); // VCVTDQ2PD_ZMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x5d, 0xe6, 0x0a], "vcvtdq2pd zmm1{k5}, dword [rdx]{1to8}"); // VCVTDQ2PD_ZMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x9d, 0xe6, 0x0a], "vcvtdq2pd xmm1{k5}{z}, dword [rdx]{1to2}"); // VCVTDQ2PD_XMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x18, 0xe6, 0x0a], "vcvtdq2pd xmm1, dword [rdx]{1to2}"); // VCVTDQ2PD_XMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x1d, 0xe6, 0x0a], "vcvtdq2pd xmm1{k5}, dword [rdx]{1to2}"); // VCVTDQ2PD_XMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xcd, 0xe6, 0xca], "vcvtdq2pd zmm1{k5}{z}, ymm2"); // VCVTDQ2PD_ZMMf64_MASKmskw_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0xcd, 0xe6, 0x0a], "vcvtdq2pd zmm1{k5}{z}, ymmword [rdx]"); // VCVTDQ2PD_ZMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x48, 0xe6, 0xca], "vcvtdq2pd zmm1, ymm2"); // VCVTDQ2PD_ZMMf64_MASKmskw_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x4d, 0xe6, 0xca], "vcvtdq2pd zmm1{k5}, ymm2"); // VCVTDQ2PD_ZMMf64_MASKmskw_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x48, 0xe6, 0x0a], "vcvtdq2pd zmm1, ymmword [rdx]"); // VCVTDQ2PD_ZMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x4d, 0xe6, 0x0a], "vcvtdq2pd zmm1{k5}, ymmword [rdx]"); // VCVTDQ2PD_ZMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x8d, 0xe6, 0xca], "vcvtdq2pd xmm1{k5}{z}, xmm2"); // VCVTDQ2PD_XMMf64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x8d, 0xe6, 0x0a], "vcvtdq2pd xmm1{k5}{z}, qword [rdx]"); // VCVTDQ2PD_XMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x08, 0xe6, 0xca], "vcvtdq2pd xmm1, xmm2"); // VCVTDQ2PD_XMMf64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x0d, 0xe6, 0xca], "vcvtdq2pd xmm1{k5}, xmm2"); // VCVTDQ2PD_XMMf64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x08, 0xe6, 0x0a], "vcvtdq2pd xmm1, qword [rdx]"); // VCVTDQ2PD_XMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7e, 0x0d, 0xe6, 0x0a], "vcvtdq2pd xmm1{k5}, qword [rdx]"); // VCVTDQ2PD_XMMf64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
}
#[test]
fn tests_f3_0f() {
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x10, 0xca], "vmovsd xmm1{k5}{z}, xmm0, xmm2"); // VMOVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x10, 0x0a], "vmovsd xmm1{k5}{z}, qword [rdx]"); // VMOVSD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x10, 0xca], "vmovsd xmm1, xmm0, xmm2"); // VMOVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x10, 0xca], "vmovsd xmm1{k5}, xmm0, xmm2"); // VMOVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x10, 0x0a], "vmovsd xmm1, qword [rdx]"); // VMOVSD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x10, 0x0a], "vmovsd xmm1{k5}, qword [rdx]"); // VMOVSD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x11, 0xca], "vmovsd xmm2{k5}{z}, xmm0, xmm1"); // VMOVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x11, 0xca], "vmovsd xmm2, xmm0, xmm1"); // VMOVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x11, 0xca], "vmovsd xmm2{k5}, xmm0, xmm1"); // VMOVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x11, 0x0a], "vmovsd qword [rdx], xmm1"); // VMOVSD_MEMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x11, 0x0a], "vmovsd qword [rdx]{k5}, xmm1"); // VMOVSD_MEMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x12, 0xca], "vmovddup ymm1{k5}{z}, ymm2"); // VMOVDDUP_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x12, 0x0a], "vmovddup ymm1{k5}{z}, ymmword [rdx]"); // VMOVDDUP_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x12, 0xca], "vmovddup ymm1, ymm2"); // VMOVDDUP_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x12, 0xca], "vmovddup ymm1{k5}, ymm2"); // VMOVDDUP_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x12, 0x0a], "vmovddup ymm1, ymmword [rdx]"); // VMOVDDUP_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x12, 0x0a], "vmovddup ymm1{k5}, ymmword [rdx]"); // VMOVDDUP_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xcd, 0x12, 0xca], "vmovddup zmm1{k5}{z}, zmm2"); // VMOVDDUP_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xcd, 0x12, 0x0a], "vmovddup zmm1{k5}{z}, zmmword [rdx]"); // VMOVDDUP_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x48, 0x12, 0xca], "vmovddup zmm1, zmm2"); // VMOVDDUP_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x4d, 0x12, 0xca], "vmovddup zmm1{k5}, zmm2"); // VMOVDDUP_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x48, 0x12, 0x0a], "vmovddup zmm1, zmmword [rdx]"); // VMOVDDUP_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x4d, 0x12, 0x0a], "vmovddup zmm1{k5}, zmmword [rdx]"); // VMOVDDUP_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x8d, 0x12, 0xca], "vmovddup xmm1{k5}{z}, xmm2"); // VMOVDDUP_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x8d, 0x12, 0x0a], "vmovddup xmm1{k5}{z}, qword [rdx]"); // VMOVDDUP_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x08, 0x12, 0xca], "vmovddup xmm1, xmm2"); // VMOVDDUP_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x0d, 0x12, 0xca], "vmovddup xmm1{k5}, xmm2"); // VMOVDDUP_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x08, 0x12, 0x0a], "vmovddup xmm1, qword [rdx]"); // VMOVDDUP_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x0d, 0x12, 0x0a], "vmovddup xmm1{k5}, qword [rdx]"); // VMOVDDUP_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0x2a, 0xca], "vcvtsi2sd xmm1{rz-sae}, xmm0, rdx"); // VCVTSI2SD_XMMf64_XMMf64_GPR64i64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x38, 0x2a, 0xca], "vcvtsi2sd xmm1{rd-sae}, xmm0, rdx"); // VCVTSI2SD_XMMf64_XMMf64_GPR64i64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x2a, 0xca], "vcvtsi2sd xmm1, xmm0, rdx"); // VCVTSI2SD_XMMf64_XMMf64_GPR64i64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x2a, 0x0a], "vcvtsi2sd xmm1, xmm0, qword [rdx]"); // VCVTSI2SD_XMMf64_XMMf64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x78, 0x2a, 0xca], "vcvtsi2sd xmm1, xmm0, edx"); // VCVTSI2SD_XMMf64_XMMf64_GPR32i32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x28, 0x2a, 0x0a], "vcvtsi2sd xmm1, xmm0, dword [rdx]"); // VCVTSI2SD_XMMf64_XMMf64_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x58, 0x2a, 0xca], "vcvtsi2sd xmm1{ru-sae}, xmm0, rdx"); // VCVTSI2SD_XMMf64_XMMf64_GPR64i64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x18, 0x2a, 0xca], "vcvtsi2sd xmm1{rne-sae}, xmm0, rdx"); // VCVTSI2SD_XMMf64_XMMf64_GPR64i64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0x2c, 0xca], "vcvttsd2si rcx{sae}, xmm2"); // VCVTTSD2SI_GPR64i64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x2c, 0xca], "vcvttsd2si rcx, xmm2"); // VCVTTSD2SI_GPR64i64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x2c, 0x0a], "vcvttsd2si rcx, qword [rdx]"); // VCVTTSD2SI_GPR64i64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x78, 0x2c, 0xca], "vcvttsd2si ecx{sae}, xmm2"); // VCVTTSD2SI_GPR32i32_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x28, 0x2c, 0xca], "vcvttsd2si ecx, xmm2"); // VCVTTSD2SI_GPR32i32_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x28, 0x2c, 0x0a], "vcvttsd2si ecx, qword [rdx]"); // VCVTTSD2SI_GPR32i32_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0x2d, 0xca], "vcvtsd2si rcx{rz-sae}, xmm2"); // VCVTSD2SI_GPR64i64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x38, 0x2d, 0xca], "vcvtsd2si rcx{rd-sae}, xmm2"); // VCVTSD2SI_GPR64i64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x2d, 0xca], "vcvtsd2si rcx, xmm2"); // VCVTSD2SI_GPR64i64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x2d, 0x0a], "vcvtsd2si rcx, qword [rdx]"); // VCVTSD2SI_GPR64i64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x78, 0x2d, 0xca], "vcvtsd2si ecx{rz-sae}, xmm2"); // VCVTSD2SI_GPR32i32_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x38, 0x2d, 0xca], "vcvtsd2si ecx{rd-sae}, xmm2"); // VCVTSD2SI_GPR32i32_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x28, 0x2d, 0xca], "vcvtsd2si ecx, xmm2"); // VCVTSD2SI_GPR32i32_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x28, 0x2d, 0x0a], "vcvtsd2si ecx, qword [rdx]"); // VCVTSD2SI_GPR32i32_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x58, 0x2d, 0xca], "vcvtsd2si rcx{ru-sae}, xmm2"); // VCVTSD2SI_GPR64i64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x18, 0x2d, 0xca], "vcvtsd2si rcx{rne-sae}, xmm2"); // VCVTSD2SI_GPR64i64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x58, 0x2d, 0xca], "vcvtsd2si ecx{ru-sae}, xmm2"); // VCVTSD2SI_GPR32i32_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x18, 0x2d, 0xca], "vcvtsd2si ecx{rne-sae}, xmm2"); // VCVTSD2SI_GPR32i32_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xfd, 0x51, 0xca], "vsqrtsd xmm1{k5}{z}{rz-sae}, xmm0, xmm2"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0x51, 0xca], "vsqrtsd xmm1{rz-sae}, xmm0, xmm2"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x7d, 0x51, 0xca], "vsqrtsd xmm1{k5}{rz-sae}, xmm0, xmm2"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xbd, 0x51, 0xca], "vsqrtsd xmm1{k5}{z}{rd-sae}, xmm0, xmm2"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x38, 0x51, 0xca], "vsqrtsd xmm1{rd-sae}, xmm0, xmm2"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x3d, 0x51, 0xca], "vsqrtsd xmm1{k5}{rd-sae}, xmm0, xmm2"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x51, 0xca], "vsqrtsd xmm1{k5}{z}, xmm0, xmm2"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x51, 0x0a], "vsqrtsd xmm1{k5}{z}, xmm0, qword [rdx]"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x51, 0xca], "vsqrtsd xmm1, xmm0, xmm2"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x51, 0xca], "vsqrtsd xmm1{k5}, xmm0, xmm2"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x51, 0x0a], "vsqrtsd xmm1, xmm0, qword [rdx]"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x51, 0x0a], "vsqrtsd xmm1{k5}, xmm0, qword [rdx]"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xdd, 0x51, 0xca], "vsqrtsd xmm1{k5}{z}{ru-sae}, xmm0, xmm2"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x58, 0x51, 0xca], "vsqrtsd xmm1{ru-sae}, xmm0, xmm2"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x5d, 0x51, 0xca], "vsqrtsd xmm1{k5}{ru-sae}, xmm0, xmm2"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x9d, 0x51, 0xca], "vsqrtsd xmm1{k5}{z}{rne-sae}, xmm0, xmm2"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x18, 0x51, 0xca], "vsqrtsd xmm1{rne-sae}, xmm0, xmm2"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x1d, 0x51, 0xca], "vsqrtsd xmm1{k5}{rne-sae}, xmm0, xmm2"); // VSQRTSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xfd, 0x58, 0xca], "vaddsd xmm1{k5}{z}{rz-sae}, xmm0, xmm2"); // VADDSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0x58, 0xca], "vaddsd xmm1{rz-sae}, xmm0, xmm2"); // VADDSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x7d, 0x58, 0xca], "vaddsd xmm1{k5}{rz-sae}, xmm0, xmm2"); // VADDSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xbd, 0x58, 0xca], "vaddsd xmm1{k5}{z}{rd-sae}, xmm0, xmm2"); // VADDSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x38, 0x58, 0xca], "vaddsd xmm1{rd-sae}, xmm0, xmm2"); // VADDSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x3d, 0x58, 0xca], "vaddsd xmm1{k5}{rd-sae}, xmm0, xmm2"); // VADDSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x58, 0xca], "vaddsd xmm1{k5}{z}, xmm0, xmm2"); // VADDSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x58, 0x0a], "vaddsd xmm1{k5}{z}, xmm0, qword [rdx]"); // VADDSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x58, 0xca], "vaddsd xmm1, xmm0, xmm2"); // VADDSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x58, 0xca], "vaddsd xmm1{k5}, xmm0, xmm2"); // VADDSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x58, 0x0a], "vaddsd xmm1, xmm0, qword [rdx]"); // VADDSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x58, 0x0a], "vaddsd xmm1{k5}, xmm0, qword [rdx]"); // VADDSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xdd, 0x58, 0xca], "vaddsd xmm1{k5}{z}{ru-sae}, xmm0, xmm2"); // VADDSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x58, 0x58, 0xca], "vaddsd xmm1{ru-sae}, xmm0, xmm2"); // VADDSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x5d, 0x58, 0xca], "vaddsd xmm1{k5}{ru-sae}, xmm0, xmm2"); // VADDSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x9d, 0x58, 0xca], "vaddsd xmm1{k5}{z}{rne-sae}, xmm0, xmm2"); // VADDSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x18, 0x58, 0xca], "vaddsd xmm1{rne-sae}, xmm0, xmm2"); // VADDSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x1d, 0x58, 0xca], "vaddsd xmm1{k5}{rne-sae}, xmm0, xmm2"); // VADDSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xfd, 0x59, 0xca], "vmulsd xmm1{k5}{z}{rz-sae}, xmm0, xmm2"); // VMULSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0x59, 0xca], "vmulsd xmm1{rz-sae}, xmm0, xmm2"); // VMULSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x7d, 0x59, 0xca], "vmulsd xmm1{k5}{rz-sae}, xmm0, xmm2"); // VMULSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xbd, 0x59, 0xca], "vmulsd xmm1{k5}{z}{rd-sae}, xmm0, xmm2"); // VMULSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x38, 0x59, 0xca], "vmulsd xmm1{rd-sae}, xmm0, xmm2"); // VMULSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x3d, 0x59, 0xca], "vmulsd xmm1{k5}{rd-sae}, xmm0, xmm2"); // VMULSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x59, 0xca], "vmulsd xmm1{k5}{z}, xmm0, xmm2"); // VMULSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x59, 0x0a], "vmulsd xmm1{k5}{z}, xmm0, qword [rdx]"); // VMULSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x59, 0xca], "vmulsd xmm1, xmm0, xmm2"); // VMULSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x59, 0xca], "vmulsd xmm1{k5}, xmm0, xmm2"); // VMULSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x59, 0x0a], "vmulsd xmm1, xmm0, qword [rdx]"); // VMULSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x59, 0x0a], "vmulsd xmm1{k5}, xmm0, qword [rdx]"); // VMULSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xdd, 0x59, 0xca], "vmulsd xmm1{k5}{z}{ru-sae}, xmm0, xmm2"); // VMULSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x58, 0x59, 0xca], "vmulsd xmm1{ru-sae}, xmm0, xmm2"); // VMULSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x5d, 0x59, 0xca], "vmulsd xmm1{k5}{ru-sae}, xmm0, xmm2"); // VMULSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x9d, 0x59, 0xca], "vmulsd xmm1{k5}{z}{rne-sae}, xmm0, xmm2"); // VMULSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x18, 0x59, 0xca], "vmulsd xmm1{rne-sae}, xmm0, xmm2"); // VMULSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x1d, 0x59, 0xca], "vmulsd xmm1{k5}{rne-sae}, xmm0, xmm2"); // VMULSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xfd, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{z}{rz-sae}, xmm0, xmm2"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0x5a, 0xca], "vcvtsd2ss xmm1{rz-sae}, xmm0, xmm2"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x7d, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{rz-sae}, xmm0, xmm2"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xbd, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{z}{rd-sae}, xmm0, xmm2"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x38, 0x5a, 0xca], "vcvtsd2ss xmm1{rd-sae}, xmm0, xmm2"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x3d, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{rd-sae}, xmm0, xmm2"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{z}, xmm0, xmm2"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x5a, 0x0a], "vcvtsd2ss xmm1{k5}{z}, xmm0, qword [rdx]"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x5a, 0xca], "vcvtsd2ss xmm1, xmm0, xmm2"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}, xmm0, xmm2"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x5a, 0x0a], "vcvtsd2ss xmm1, xmm0, qword [rdx]"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x5a, 0x0a], "vcvtsd2ss xmm1{k5}, xmm0, qword [rdx]"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xdd, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{z}{ru-sae}, xmm0, xmm2"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x58, 0x5a, 0xca], "vcvtsd2ss xmm1{ru-sae}, xmm0, xmm2"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x5d, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{ru-sae}, xmm0, xmm2"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x9d, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{z}{rne-sae}, xmm0, xmm2"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x18, 0x5a, 0xca], "vcvtsd2ss xmm1{rne-sae}, xmm0, xmm2"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x1d, 0x5a, 0xca], "vcvtsd2ss xmm1{k5}{rne-sae}, xmm0, xmm2"); // VCVTSD2SS_XMMf32_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xfd, 0x5c, 0xca], "vsubsd xmm1{k5}{z}{rz-sae}, xmm0, xmm2"); // VSUBSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0x5c, 0xca], "vsubsd xmm1{rz-sae}, xmm0, xmm2"); // VSUBSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x7d, 0x5c, 0xca], "vsubsd xmm1{k5}{rz-sae}, xmm0, xmm2"); // VSUBSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xbd, 0x5c, 0xca], "vsubsd xmm1{k5}{z}{rd-sae}, xmm0, xmm2"); // VSUBSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x38, 0x5c, 0xca], "vsubsd xmm1{rd-sae}, xmm0, xmm2"); // VSUBSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x3d, 0x5c, 0xca], "vsubsd xmm1{k5}{rd-sae}, xmm0, xmm2"); // VSUBSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x5c, 0xca], "vsubsd xmm1{k5}{z}, xmm0, xmm2"); // VSUBSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x5c, 0x0a], "vsubsd xmm1{k5}{z}, xmm0, qword [rdx]"); // VSUBSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x5c, 0xca], "vsubsd xmm1, xmm0, xmm2"); // VSUBSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x5c, 0xca], "vsubsd xmm1{k5}, xmm0, xmm2"); // VSUBSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x5c, 0x0a], "vsubsd xmm1, xmm0, qword [rdx]"); // VSUBSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x5c, 0x0a], "vsubsd xmm1{k5}, xmm0, qword [rdx]"); // VSUBSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xdd, 0x5c, 0xca], "vsubsd xmm1{k5}{z}{ru-sae}, xmm0, xmm2"); // VSUBSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x58, 0x5c, 0xca], "vsubsd xmm1{ru-sae}, xmm0, xmm2"); // VSUBSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x5d, 0x5c, 0xca], "vsubsd xmm1{k5}{ru-sae}, xmm0, xmm2"); // VSUBSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x9d, 0x5c, 0xca], "vsubsd xmm1{k5}{z}{rne-sae}, xmm0, xmm2"); // VSUBSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x18, 0x5c, 0xca], "vsubsd xmm1{rne-sae}, xmm0, xmm2"); // VSUBSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x1d, 0x5c, 0xca], "vsubsd xmm1{k5}{rne-sae}, xmm0, xmm2"); // VSUBSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xfd, 0x5d, 0xca], "vminsd xmm1{k5}{z}{sae}, xmm0, xmm2"); // VMINSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0x5d, 0xca], "vminsd xmm1{sae}, xmm0, xmm2"); // VMINSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x7d, 0x5d, 0xca], "vminsd xmm1{k5}{sae}, xmm0, xmm2"); // VMINSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x5d, 0xca], "vminsd xmm1{k5}{z}, xmm0, xmm2"); // VMINSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x5d, 0x0a], "vminsd xmm1{k5}{z}, xmm0, qword [rdx]"); // VMINSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x5d, 0xca], "vminsd xmm1, xmm0, xmm2"); // VMINSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x5d, 0xca], "vminsd xmm1{k5}, xmm0, xmm2"); // VMINSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x5d, 0x0a], "vminsd xmm1, xmm0, qword [rdx]"); // VMINSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x5d, 0x0a], "vminsd xmm1{k5}, xmm0, qword [rdx]"); // VMINSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xfd, 0x5e, 0xca], "vdivsd xmm1{k5}{z}{rz-sae}, xmm0, xmm2"); // VDIVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0x5e, 0xca], "vdivsd xmm1{rz-sae}, xmm0, xmm2"); // VDIVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x7d, 0x5e, 0xca], "vdivsd xmm1{k5}{rz-sae}, xmm0, xmm2"); // VDIVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xbd, 0x5e, 0xca], "vdivsd xmm1{k5}{z}{rd-sae}, xmm0, xmm2"); // VDIVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x38, 0x5e, 0xca], "vdivsd xmm1{rd-sae}, xmm0, xmm2"); // VDIVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x3d, 0x5e, 0xca], "vdivsd xmm1{k5}{rd-sae}, xmm0, xmm2"); // VDIVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x5e, 0xca], "vdivsd xmm1{k5}{z}, xmm0, xmm2"); // VDIVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x5e, 0x0a], "vdivsd xmm1{k5}{z}, xmm0, qword [rdx]"); // VDIVSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x5e, 0xca], "vdivsd xmm1, xmm0, xmm2"); // VDIVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x5e, 0xca], "vdivsd xmm1{k5}, xmm0, xmm2"); // VDIVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x5e, 0x0a], "vdivsd xmm1, xmm0, qword [rdx]"); // VDIVSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x5e, 0x0a], "vdivsd xmm1{k5}, xmm0, qword [rdx]"); // VDIVSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xdd, 0x5e, 0xca], "vdivsd xmm1{k5}{z}{ru-sae}, xmm0, xmm2"); // VDIVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x58, 0x5e, 0xca], "vdivsd xmm1{ru-sae}, xmm0, xmm2"); // VDIVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x5d, 0x5e, 0xca], "vdivsd xmm1{k5}{ru-sae}, xmm0, xmm2"); // VDIVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x9d, 0x5e, 0xca], "vdivsd xmm1{k5}{z}{rne-sae}, xmm0, xmm2"); // VDIVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x18, 0x5e, 0xca], "vdivsd xmm1{rne-sae}, xmm0, xmm2"); // VDIVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x1d, 0x5e, 0xca], "vdivsd xmm1{k5}{rne-sae}, xmm0, xmm2"); // VDIVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xfd, 0x5f, 0xca], "vmaxsd xmm1{k5}{z}{sae}, xmm0, xmm2"); // VMAXSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0x5f, 0xca], "vmaxsd xmm1{sae}, xmm0, xmm2"); // VMAXSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x7d, 0x5f, 0xca], "vmaxsd xmm1{k5}{sae}, xmm0, xmm2"); // VMAXSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x5f, 0xca], "vmaxsd xmm1{k5}{z}, xmm0, xmm2"); // VMAXSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x5f, 0x0a], "vmaxsd xmm1{k5}{z}, xmm0, qword [rdx]"); // VMAXSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x5f, 0xca], "vmaxsd xmm1, xmm0, xmm2"); // VMAXSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x5f, 0xca], "vmaxsd xmm1{k5}, xmm0, xmm2"); // VMAXSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x5f, 0x0a], "vmaxsd xmm1, xmm0, qword [rdx]"); // VMAXSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x5f, 0x0a], "vmaxsd xmm1{k5}, xmm0, qword [rdx]"); // VMAXSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x6f, 0xca], "vmovdqu16 ymm1{k5}{z}, ymm2"); // VMOVDQU16_YMMu16_MASKmskw_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x6f, 0x0a], "vmovdqu16 ymm1{k5}{z}, ymmword [rdx]"); // VMOVDQU16_YMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x6f, 0xca], "vmovdqu16 ymm1, ymm2"); // VMOVDQU16_YMMu16_MASKmskw_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x6f, 0xca], "vmovdqu16 ymm1{k5}, ymm2"); // VMOVDQU16_YMMu16_MASKmskw_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x6f, 0x0a], "vmovdqu16 ymm1, ymmword [rdx]"); // VMOVDQU16_YMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x6f, 0x0a], "vmovdqu16 ymm1{k5}, ymmword [rdx]"); // VMOVDQU16_YMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0xad, 0x6f, 0xca], "vmovdqu8 ymm1{k5}{z}, ymm2"); // VMOVDQU8_YMMu8_MASKmskw_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0xad, 0x6f, 0x0a], "vmovdqu8 ymm1{k5}{z}, ymmword [rdx]"); // VMOVDQU8_YMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x28, 0x6f, 0xca], "vmovdqu8 ymm1, ymm2"); // VMOVDQU8_YMMu8_MASKmskw_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x2d, 0x6f, 0xca], "vmovdqu8 ymm1{k5}, ymm2"); // VMOVDQU8_YMMu8_MASKmskw_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x28, 0x6f, 0x0a], "vmovdqu8 ymm1, ymmword [rdx]"); // VMOVDQU8_YMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x2d, 0x6f, 0x0a], "vmovdqu8 ymm1{k5}, ymmword [rdx]"); // VMOVDQU8_YMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xcd, 0x6f, 0xca], "vmovdqu16 zmm1{k5}{z}, zmm2"); // VMOVDQU16_ZMMu16_MASKmskw_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xcd, 0x6f, 0x0a], "vmovdqu16 zmm1{k5}{z}, zmmword [rdx]"); // VMOVDQU16_ZMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x48, 0x6f, 0xca], "vmovdqu16 zmm1, zmm2"); // VMOVDQU16_ZMMu16_MASKmskw_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x4d, 0x6f, 0xca], "vmovdqu16 zmm1{k5}, zmm2"); // VMOVDQU16_ZMMu16_MASKmskw_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x48, 0x6f, 0x0a], "vmovdqu16 zmm1, zmmword [rdx]"); // VMOVDQU16_ZMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x4d, 0x6f, 0x0a], "vmovdqu16 zmm1{k5}, zmmword [rdx]"); // VMOVDQU16_ZMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x8d, 0x6f, 0xca], "vmovdqu16 xmm1{k5}{z}, xmm2"); // VMOVDQU16_XMMu16_MASKmskw_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x8d, 0x6f, 0x0a], "vmovdqu16 xmm1{k5}{z}, xmmword [rdx]"); // VMOVDQU16_XMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x08, 0x6f, 0xca], "vmovdqu16 xmm1, xmm2"); // VMOVDQU16_XMMu16_MASKmskw_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x0d, 0x6f, 0xca], "vmovdqu16 xmm1{k5}, xmm2"); // VMOVDQU16_XMMu16_MASKmskw_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x08, 0x6f, 0x0a], "vmovdqu16 xmm1, xmmword [rdx]"); // VMOVDQU16_XMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x0d, 0x6f, 0x0a], "vmovdqu16 xmm1{k5}, xmmword [rdx]"); // VMOVDQU16_XMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0xcd, 0x6f, 0xca], "vmovdqu8 zmm1{k5}{z}, zmm2"); // VMOVDQU8_ZMMu8_MASKmskw_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0xcd, 0x6f, 0x0a], "vmovdqu8 zmm1{k5}{z}, zmmword [rdx]"); // VMOVDQU8_ZMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x48, 0x6f, 0xca], "vmovdqu8 zmm1, zmm2"); // VMOVDQU8_ZMMu8_MASKmskw_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x4d, 0x6f, 0xca], "vmovdqu8 zmm1{k5}, zmm2"); // VMOVDQU8_ZMMu8_MASKmskw_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x48, 0x6f, 0x0a], "vmovdqu8 zmm1, zmmword [rdx]"); // VMOVDQU8_ZMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x4d, 0x6f, 0x0a], "vmovdqu8 zmm1{k5}, zmmword [rdx]"); // VMOVDQU8_ZMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x8d, 0x6f, 0xca], "vmovdqu8 xmm1{k5}{z}, xmm2"); // VMOVDQU8_XMMu8_MASKmskw_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x8d, 0x6f, 0x0a], "vmovdqu8 xmm1{k5}{z}, xmmword [rdx]"); // VMOVDQU8_XMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x08, 0x6f, 0xca], "vmovdqu8 xmm1, xmm2"); // VMOVDQU8_XMMu8_MASKmskw_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x0d, 0x6f, 0xca], "vmovdqu8 xmm1{k5}, xmm2"); // VMOVDQU8_XMMu8_MASKmskw_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x08, 0x6f, 0x0a], "vmovdqu8 xmm1, xmmword [rdx]"); // VMOVDQU8_XMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x0d, 0x6f, 0x0a], "vmovdqu8 xmm1{k5}, xmmword [rdx]"); // VMOVDQU8_XMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x70, 0xca, 0xcc], "vpshuflw ymm1{k5}{z}, ymm2, 0xcc"); // VPSHUFLW_YMMu16_MASKmskw_YMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x70, 0x0a, 0xcc], "vpshuflw ymm1{k5}{z}, ymmword [rdx], 0xcc"); // VPSHUFLW_YMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x70, 0xca, 0xcc], "vpshuflw ymm1, ymm2, 0xcc"); // VPSHUFLW_YMMu16_MASKmskw_YMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x70, 0xca, 0xcc], "vpshuflw ymm1{k5}, ymm2, 0xcc"); // VPSHUFLW_YMMu16_MASKmskw_YMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x70, 0x0a, 0xcc], "vpshuflw ymm1, ymmword [rdx], 0xcc"); // VPSHUFLW_YMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x70, 0x0a, 0xcc], "vpshuflw ymm1{k5}, ymmword [rdx], 0xcc"); // VPSHUFLW_YMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xcd, 0x70, 0xca, 0xcc], "vpshuflw zmm1{k5}{z}, zmm2, 0xcc"); // VPSHUFLW_ZMMu16_MASKmskw_ZMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xcd, 0x70, 0x0a, 0xcc], "vpshuflw zmm1{k5}{z}, zmmword [rdx], 0xcc"); // VPSHUFLW_ZMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x48, 0x70, 0xca, 0xcc], "vpshuflw zmm1, zmm2, 0xcc"); // VPSHUFLW_ZMMu16_MASKmskw_ZMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x4d, 0x70, 0xca, 0xcc], "vpshuflw zmm1{k5}, zmm2, 0xcc"); // VPSHUFLW_ZMMu16_MASKmskw_ZMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x48, 0x70, 0x0a, 0xcc], "vpshuflw zmm1, zmmword [rdx], 0xcc"); // VPSHUFLW_ZMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x4d, 0x70, 0x0a, 0xcc], "vpshuflw zmm1{k5}, zmmword [rdx], 0xcc"); // VPSHUFLW_ZMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x8d, 0x70, 0xca, 0xcc], "vpshuflw xmm1{k5}{z}, xmm2, 0xcc"); // VPSHUFLW_XMMu16_MASKmskw_XMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x8d, 0x70, 0x0a, 0xcc], "vpshuflw xmm1{k5}{z}, xmmword [rdx], 0xcc"); // VPSHUFLW_XMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x08, 0x70, 0xca, 0xcc], "vpshuflw xmm1, xmm2, 0xcc"); // VPSHUFLW_XMMu16_MASKmskw_XMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x0d, 0x70, 0xca, 0xcc], "vpshuflw xmm1{k5}, xmm2, 0xcc"); // VPSHUFLW_XMMu16_MASKmskw_XMMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x08, 0x70, 0x0a, 0xcc], "vpshuflw xmm1, xmmword [rdx], 0xcc"); // VPSHUFLW_XMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x0d, 0x70, 0x0a, 0xcc], "vpshuflw xmm1{k5}, xmmword [rdx], 0xcc"); // VPSHUFLW_XMMu16_MASKmskw_MEMu16_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0x78, 0xca], "vcvttsd2usi rcx{sae}, xmm2"); // VCVTTSD2USI_GPR64u64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x78, 0xca], "vcvttsd2usi rcx, xmm2"); // VCVTTSD2USI_GPR64u64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x78, 0x0a], "vcvttsd2usi rcx, qword [rdx]"); // VCVTTSD2USI_GPR64u64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x78, 0x78, 0xca], "vcvttsd2usi ecx{sae}, xmm2"); // VCVTTSD2USI_GPR32u32_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x28, 0x78, 0xca], "vcvttsd2usi ecx, xmm2"); // VCVTTSD2USI_GPR32u32_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x28, 0x78, 0x0a], "vcvttsd2usi ecx, qword [rdx]"); // VCVTTSD2USI_GPR32u32_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0x79, 0xca], "vcvtsd2usi rcx{rz-sae}, xmm2"); // VCVTSD2USI_GPR64u64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x38, 0x79, 0xca], "vcvtsd2usi rcx{rd-sae}, xmm2"); // VCVTSD2USI_GPR64u64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x79, 0xca], "vcvtsd2usi rcx, xmm2"); // VCVTSD2USI_GPR64u64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x79, 0x0a], "vcvtsd2usi rcx, qword [rdx]"); // VCVTSD2USI_GPR64u64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x78, 0x79, 0xca], "vcvtsd2usi ecx{rz-sae}, xmm2"); // VCVTSD2USI_GPR32u32_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x38, 0x79, 0xca], "vcvtsd2usi ecx{rd-sae}, xmm2"); // VCVTSD2USI_GPR32u32_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x28, 0x79, 0xca], "vcvtsd2usi ecx, xmm2"); // VCVTSD2USI_GPR32u32_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x28, 0x79, 0x0a], "vcvtsd2usi ecx, qword [rdx]"); // VCVTSD2USI_GPR32u32_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x58, 0x79, 0xca], "vcvtsd2usi rcx{ru-sae}, xmm2"); // VCVTSD2USI_GPR64u64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x18, 0x79, 0xca], "vcvtsd2usi rcx{rne-sae}, xmm2"); // VCVTSD2USI_GPR64u64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x58, 0x79, 0xca], "vcvtsd2usi ecx{ru-sae}, xmm2"); // VCVTSD2USI_GPR32u32_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x18, 0x79, 0xca], "vcvtsd2usi ecx{rne-sae}, xmm2"); // VCVTSD2USI_GPR32u32_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xfd, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{z}{rz-sae}, zmm2"); // VCVTUQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0x7a, 0xca], "vcvtuqq2ps ymm1{rz-sae}, zmm2"); // VCVTUQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x7d, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{rz-sae}, zmm2"); // VCVTUQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xbd, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{z}{rd-sae}, zmm2"); // VCVTUQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xbd, 0x7a, 0x0a], "vcvtuqq2ps xmm1{k5}{z}, qword [rdx]{1to4}"); // VCVTUQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x38, 0x7a, 0xca], "vcvtuqq2ps ymm1{rd-sae}, zmm2"); // VCVTUQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x3d, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{rd-sae}, zmm2"); // VCVTUQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x38, 0x7a, 0x0a], "vcvtuqq2ps xmm1, qword [rdx]{1to4}"); // VCVTUQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x3d, 0x7a, 0x0a], "vcvtuqq2ps xmm1{k5}, qword [rdx]{1to4}"); // VCVTUQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x7a, 0xca], "vcvtuqq2ps xmm1{k5}{z}, ymm2"); // VCVTUQQ2PS_XMMf32_MASKmskw_YMMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x7a, 0x0a], "vcvtuqq2ps xmm1{k5}{z}, ymmword [rdx]"); // VCVTUQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x7a, 0xca], "vcvtuqq2ps xmm1, ymm2"); // VCVTUQQ2PS_XMMf32_MASKmskw_YMMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x7a, 0xca], "vcvtuqq2ps xmm1{k5}, ymm2"); // VCVTUQQ2PS_XMMf32_MASKmskw_YMMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x7a, 0x0a], "vcvtuqq2ps xmm1, ymmword [rdx]"); // VCVTUQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x7a, 0x0a], "vcvtuqq2ps xmm1{k5}, ymmword [rdx]"); // VCVTUQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0xfd, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{z}{rz-sae}, zmm2"); // VCVTUDQ2PS_ZMMf32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x78, 0x7a, 0xca], "vcvtudq2ps zmm1{rz-sae}, zmm2"); // VCVTUDQ2PS_ZMMf32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x7d, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{rz-sae}, zmm2"); // VCVTUDQ2PS_ZMMf32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0xbd, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{z}{rd-sae}, zmm2"); // VCVTUDQ2PS_ZMMf32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0xbd, 0x7a, 0x0a], "vcvtudq2ps ymm1{k5}{z}, dword [rdx]{1to8}"); // VCVTUDQ2PS_YMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x38, 0x7a, 0xca], "vcvtudq2ps zmm1{rd-sae}, zmm2"); // VCVTUDQ2PS_ZMMf32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x3d, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{rd-sae}, zmm2"); // VCVTUDQ2PS_ZMMf32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x38, 0x7a, 0x0a], "vcvtudq2ps ymm1, dword [rdx]{1to8}"); // VCVTUDQ2PS_YMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x3d, 0x7a, 0x0a], "vcvtudq2ps ymm1{k5}, dword [rdx]{1to8}"); // VCVTUDQ2PS_YMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0xad, 0x7a, 0xca], "vcvtudq2ps ymm1{k5}{z}, ymm2"); // VCVTUDQ2PS_YMMf32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0xad, 0x7a, 0x0a], "vcvtudq2ps ymm1{k5}{z}, ymmword [rdx]"); // VCVTUDQ2PS_YMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x28, 0x7a, 0xca], "vcvtudq2ps ymm1, ymm2"); // VCVTUDQ2PS_YMMf32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x2d, 0x7a, 0xca], "vcvtudq2ps ymm1{k5}, ymm2"); // VCVTUDQ2PS_YMMf32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x28, 0x7a, 0x0a], "vcvtudq2ps ymm1, ymmword [rdx]"); // VCVTUDQ2PS_YMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x2d, 0x7a, 0x0a], "vcvtudq2ps ymm1{k5}, ymmword [rdx]"); // VCVTUDQ2PS_YMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xdd, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{z}{ru-sae}, zmm2"); // VCVTUQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xdd, 0x7a, 0x0a], "vcvtuqq2ps ymm1{k5}{z}, qword [rdx]{1to8}"); // VCVTUQQ2PS_YMMf32_MASKmskw_MEMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x58, 0x7a, 0xca], "vcvtuqq2ps ymm1{ru-sae}, zmm2"); // VCVTUQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x5d, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{ru-sae}, zmm2"); // VCVTUQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x58, 0x7a, 0x0a], "vcvtuqq2ps ymm1, qword [rdx]{1to8}"); // VCVTUQQ2PS_YMMf32_MASKmskw_MEMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x5d, 0x7a, 0x0a], "vcvtuqq2ps ymm1{k5}, qword [rdx]{1to8}"); // VCVTUQQ2PS_YMMf32_MASKmskw_MEMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x9d, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{z}{rne-sae}, zmm2"); // VCVTUQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x9d, 0x7a, 0x0a], "vcvtuqq2ps xmm1{k5}{z}, qword [rdx]{1to2}"); // VCVTUQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x18, 0x7a, 0xca], "vcvtuqq2ps ymm1{rne-sae}, zmm2"); // VCVTUQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x1d, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{rne-sae}, zmm2"); // VCVTUQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x18, 0x7a, 0x0a], "vcvtuqq2ps xmm1, qword [rdx]{1to2}"); // VCVTUQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x1d, 0x7a, 0x0a], "vcvtuqq2ps xmm1{k5}, qword [rdx]{1to2}"); // VCVTUQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xcd, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}{z}, zmm2"); // VCVTUQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xcd, 0x7a, 0x0a], "vcvtuqq2ps ymm1{k5}{z}, zmmword [rdx]"); // VCVTUQQ2PS_YMMf32_MASKmskw_MEMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x48, 0x7a, 0xca], "vcvtuqq2ps ymm1, zmm2"); // VCVTUQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x4d, 0x7a, 0xca], "vcvtuqq2ps ymm1{k5}, zmm2"); // VCVTUQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x48, 0x7a, 0x0a], "vcvtuqq2ps ymm1, zmmword [rdx]"); // VCVTUQQ2PS_YMMf32_MASKmskw_MEMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x4d, 0x7a, 0x0a], "vcvtuqq2ps ymm1{k5}, zmmword [rdx]"); // VCVTUQQ2PS_YMMf32_MASKmskw_MEMu64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x8d, 0x7a, 0xca], "vcvtuqq2ps xmm1{k5}{z}, xmm2"); // VCVTUQQ2PS_XMMf32_MASKmskw_XMMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x8d, 0x7a, 0x0a], "vcvtuqq2ps xmm1{k5}{z}, xmmword [rdx]"); // VCVTUQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x08, 0x7a, 0xca], "vcvtuqq2ps xmm1, xmm2"); // VCVTUQQ2PS_XMMf32_MASKmskw_XMMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x0d, 0x7a, 0xca], "vcvtuqq2ps xmm1{k5}, xmm2"); // VCVTUQQ2PS_XMMf32_MASKmskw_XMMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x08, 0x7a, 0x0a], "vcvtuqq2ps xmm1, xmmword [rdx]"); // VCVTUQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x0d, 0x7a, 0x0a], "vcvtuqq2ps xmm1{k5}, xmmword [rdx]"); // VCVTUQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0xdd, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{z}{ru-sae}, zmm2"); // VCVTUDQ2PS_ZMMf32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0xdd, 0x7a, 0x0a], "vcvtudq2ps zmm1{k5}{z}, dword [rdx]{1to16}"); // VCVTUDQ2PS_ZMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x58, 0x7a, 0xca], "vcvtudq2ps zmm1{ru-sae}, zmm2"); // VCVTUDQ2PS_ZMMf32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x5d, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{ru-sae}, zmm2"); // VCVTUDQ2PS_ZMMf32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x58, 0x7a, 0x0a], "vcvtudq2ps zmm1, dword [rdx]{1to16}"); // VCVTUDQ2PS_ZMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x5d, 0x7a, 0x0a], "vcvtudq2ps zmm1{k5}, dword [rdx]{1to16}"); // VCVTUDQ2PS_ZMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x9d, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{z}{rne-sae}, zmm2"); // VCVTUDQ2PS_ZMMf32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x9d, 0x7a, 0x0a], "vcvtudq2ps xmm1{k5}{z}, dword [rdx]{1to4}"); // VCVTUDQ2PS_XMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x18, 0x7a, 0xca], "vcvtudq2ps zmm1{rne-sae}, zmm2"); // VCVTUDQ2PS_ZMMf32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x1d, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{rne-sae}, zmm2"); // VCVTUDQ2PS_ZMMf32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x18, 0x7a, 0x0a], "vcvtudq2ps xmm1, dword [rdx]{1to4}"); // VCVTUDQ2PS_XMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x1d, 0x7a, 0x0a], "vcvtudq2ps xmm1{k5}, dword [rdx]{1to4}"); // VCVTUDQ2PS_XMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0xcd, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}{z}, zmm2"); // VCVTUDQ2PS_ZMMf32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0xcd, 0x7a, 0x0a], "vcvtudq2ps zmm1{k5}{z}, zmmword [rdx]"); // VCVTUDQ2PS_ZMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x48, 0x7a, 0xca], "vcvtudq2ps zmm1, zmm2"); // VCVTUDQ2PS_ZMMf32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x4d, 0x7a, 0xca], "vcvtudq2ps zmm1{k5}, zmm2"); // VCVTUDQ2PS_ZMMf32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x48, 0x7a, 0x0a], "vcvtudq2ps zmm1, zmmword [rdx]"); // VCVTUDQ2PS_ZMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x4d, 0x7a, 0x0a], "vcvtudq2ps zmm1{k5}, zmmword [rdx]"); // VCVTUDQ2PS_ZMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x8d, 0x7a, 0xca], "vcvtudq2ps xmm1{k5}{z}, xmm2"); // VCVTUDQ2PS_XMMf32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x8d, 0x7a, 0x0a], "vcvtudq2ps xmm1{k5}{z}, xmmword [rdx]"); // VCVTUDQ2PS_XMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x08, 0x7a, 0xca], "vcvtudq2ps xmm1, xmm2"); // VCVTUDQ2PS_XMMf32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x0d, 0x7a, 0xca], "vcvtudq2ps xmm1{k5}, xmm2"); // VCVTUDQ2PS_XMMf32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x08, 0x7a, 0x0a], "vcvtudq2ps xmm1, xmmword [rdx]"); // VCVTUDQ2PS_XMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x0d, 0x7a, 0x0a], "vcvtudq2ps xmm1{k5}, xmmword [rdx]"); // VCVTUDQ2PS_XMMf32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0x7b, 0xca], "vcvtusi2sd xmm1{rz-sae}, xmm0, rdx"); // VCVTUSI2SD_XMMf64_XMMf64_GPR64u64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x38, 0x7b, 0xca], "vcvtusi2sd xmm1{rd-sae}, xmm0, rdx"); // VCVTUSI2SD_XMMf64_XMMf64_GPR64u64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x7b, 0xca], "vcvtusi2sd xmm1, xmm0, rdx"); // VCVTUSI2SD_XMMf64_XMMf64_GPR64u64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x7b, 0x0a], "vcvtusi2sd xmm1, xmm0, qword [rdx]"); // VCVTUSI2SD_XMMf64_XMMf64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x78, 0x7b, 0xca], "vcvtusi2sd xmm1, xmm0, edx"); // VCVTUSI2SD_XMMf64_XMMf64_GPR32u32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x28, 0x7b, 0x0a], "vcvtusi2sd xmm1, xmm0, dword [rdx]"); // VCVTUSI2SD_XMMf64_XMMf64_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x58, 0x7b, 0xca], "vcvtusi2sd xmm1{ru-sae}, xmm0, rdx"); // VCVTUSI2SD_XMMf64_XMMf64_GPR64u64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x18, 0x7b, 0xca], "vcvtusi2sd xmm1{rne-sae}, xmm0, rdx"); // VCVTUSI2SD_XMMf64_XMMf64_GPR64u64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0x7f, 0xca], "vmovdqu16 ymm2{k5}{z}, ymm1"); // VMOVDQU16_YMMu16_MASKmskw_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x7f, 0xca], "vmovdqu16 ymm2, ymm1"); // VMOVDQU16_YMMu16_MASKmskw_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x7f, 0xca], "vmovdqu16 ymm2{k5}, ymm1"); // VMOVDQU16_YMMu16_MASKmskw_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0x7f, 0x0a], "vmovdqu16 ymmword [rdx], ymm1"); // VMOVDQU16_MEMu16_MASKmskw_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0x7f, 0x0a], "vmovdqu16 ymmword [rdx]{k5}, ymm1"); // VMOVDQU16_MEMu16_MASKmskw_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0xad, 0x7f, 0xca], "vmovdqu8 ymm2{k5}{z}, ymm1"); // VMOVDQU8_YMMu8_MASKmskw_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x28, 0x7f, 0xca], "vmovdqu8 ymm2, ymm1"); // VMOVDQU8_YMMu8_MASKmskw_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x2d, 0x7f, 0xca], "vmovdqu8 ymm2{k5}, ymm1"); // VMOVDQU8_YMMu8_MASKmskw_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x28, 0x7f, 0x0a], "vmovdqu8 ymmword [rdx], ymm1"); // VMOVDQU8_MEMu8_MASKmskw_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x2d, 0x7f, 0x0a], "vmovdqu8 ymmword [rdx]{k5}, ymm1"); // VMOVDQU8_MEMu8_MASKmskw_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xcd, 0x7f, 0xca], "vmovdqu16 zmm2{k5}{z}, zmm1"); // VMOVDQU16_ZMMu16_MASKmskw_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x48, 0x7f, 0xca], "vmovdqu16 zmm2, zmm1"); // VMOVDQU16_ZMMu16_MASKmskw_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x4d, 0x7f, 0xca], "vmovdqu16 zmm2{k5}, zmm1"); // VMOVDQU16_ZMMu16_MASKmskw_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x48, 0x7f, 0x0a], "vmovdqu16 zmmword [rdx], zmm1"); // VMOVDQU16_MEMu16_MASKmskw_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x4d, 0x7f, 0x0a], "vmovdqu16 zmmword [rdx]{k5}, zmm1"); // VMOVDQU16_MEMu16_MASKmskw_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x8d, 0x7f, 0xca], "vmovdqu16 xmm2{k5}{z}, xmm1"); // VMOVDQU16_XMMu16_MASKmskw_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x08, 0x7f, 0xca], "vmovdqu16 xmm2, xmm1"); // VMOVDQU16_XMMu16_MASKmskw_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x0d, 0x7f, 0xca], "vmovdqu16 xmm2{k5}, xmm1"); // VMOVDQU16_XMMu16_MASKmskw_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x08, 0x7f, 0x0a], "vmovdqu16 xmmword [rdx], xmm1"); // VMOVDQU16_MEMu16_MASKmskw_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x0d, 0x7f, 0x0a], "vmovdqu16 xmmword [rdx]{k5}, xmm1"); // VMOVDQU16_MEMu16_MASKmskw_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0xcd, 0x7f, 0xca], "vmovdqu8 zmm2{k5}{z}, zmm1"); // VMOVDQU8_ZMMu8_MASKmskw_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x48, 0x7f, 0xca], "vmovdqu8 zmm2, zmm1"); // VMOVDQU8_ZMMu8_MASKmskw_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x4d, 0x7f, 0xca], "vmovdqu8 zmm2{k5}, zmm1"); // VMOVDQU8_ZMMu8_MASKmskw_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x48, 0x7f, 0x0a], "vmovdqu8 zmmword [rdx], zmm1"); // VMOVDQU8_MEMu8_MASKmskw_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x4d, 0x7f, 0x0a], "vmovdqu8 zmmword [rdx]{k5}, zmm1"); // VMOVDQU8_MEMu8_MASKmskw_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x8d, 0x7f, 0xca], "vmovdqu8 xmm2{k5}{z}, xmm1"); // VMOVDQU8_XMMu8_MASKmskw_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x08, 0x7f, 0xca], "vmovdqu8 xmm2, xmm1"); // VMOVDQU8_XMMu8_MASKmskw_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x0d, 0x7f, 0xca], "vmovdqu8 xmm2{k5}, xmm1"); // VMOVDQU8_XMMu8_MASKmskw_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x08, 0x7f, 0x0a], "vmovdqu8 xmmword [rdx], xmm1"); // VMOVDQU8_MEMu8_MASKmskw_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0x7f, 0x0d, 0x7f, 0x0a], "vmovdqu8 xmmword [rdx]{k5}, xmm1"); // VMOVDQU8_MEMu8_MASKmskw_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0xc2, 0xca, 0xcc], "vcmpsd k1{sae}, xmm0, xmm2, 0xcc"); // VCMPSD_MASKmskw_MASKmskw_XMMf64_XMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x7d, 0xc2, 0xca, 0xcc], "vcmpsd k1{k5}{sae}, xmm0, xmm2, 0xcc"); // VCMPSD_MASKmskw_MASKmskw_XMMf64_XMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0xc2, 0xca, 0xcc], "vcmpsd k1, xmm0, xmm2, 0xcc"); // VCMPSD_MASKmskw_MASKmskw_XMMf64_XMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0xc2, 0xca, 0xcc], "vcmpsd k1{k5}, xmm0, xmm2, 0xcc"); // VCMPSD_MASKmskw_MASKmskw_XMMf64_XMMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0xc2, 0x0a, 0xcc], "vcmpsd k1, xmm0, qword [rdx], 0xcc"); // VCMPSD_MASKmskw_MASKmskw_XMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0xc2, 0x0a, 0xcc], "vcmpsd k1{k5}, xmm0, qword [rdx], 0xcc"); // VCMPSD_MASKmskw_MASKmskw_XMMf64_MEMf64_IMM8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xfd, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{z}{rz-sae}, zmm2"); // VCVTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x78, 0xe6, 0xca], "vcvtpd2dq ymm1{rz-sae}, zmm2"); // VCVTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x7d, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{rz-sae}, zmm2"); // VCVTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xbd, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{z}{rd-sae}, zmm2"); // VCVTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xbd, 0xe6, 0x0a], "vcvtpd2dq xmm1{k5}{z}, qword [rdx]{1to4}"); // VCVTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x38, 0xe6, 0xca], "vcvtpd2dq ymm1{rd-sae}, zmm2"); // VCVTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x3d, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{rd-sae}, zmm2"); // VCVTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x38, 0xe6, 0x0a], "vcvtpd2dq xmm1, qword [rdx]{1to4}"); // VCVTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x3d, 0xe6, 0x0a], "vcvtpd2dq xmm1{k5}, qword [rdx]{1to4}"); // VCVTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0xe6, 0xca], "vcvtpd2dq xmm1{k5}{z}, ymm2"); // VCVTPD2DQ_XMMi32_MASKmskw_YMMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xad, 0xe6, 0x0a], "vcvtpd2dq xmm1{k5}{z}, ymmword [rdx]"); // VCVTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0xe6, 0xca], "vcvtpd2dq xmm1, ymm2"); // VCVTPD2DQ_XMMi32_MASKmskw_YMMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0xe6, 0xca], "vcvtpd2dq xmm1{k5}, ymm2"); // VCVTPD2DQ_XMMi32_MASKmskw_YMMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x28, 0xe6, 0x0a], "vcvtpd2dq xmm1, ymmword [rdx]"); // VCVTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x2d, 0xe6, 0x0a], "vcvtpd2dq xmm1{k5}, ymmword [rdx]"); // VCVTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL256, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xdd, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{z}{ru-sae}, zmm2"); // VCVTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xdd, 0xe6, 0x0a], "vcvtpd2dq ymm1{k5}{z}, qword [rdx]{1to8}"); // VCVTPD2DQ_YMMi32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x58, 0xe6, 0xca], "vcvtpd2dq ymm1{ru-sae}, zmm2"); // VCVTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x5d, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{ru-sae}, zmm2"); // VCVTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x58, 0xe6, 0x0a], "vcvtpd2dq ymm1, qword [rdx]{1to8}"); // VCVTPD2DQ_YMMi32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x5d, 0xe6, 0x0a], "vcvtpd2dq ymm1{k5}, qword [rdx]{1to8}"); // VCVTPD2DQ_YMMi32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x9d, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{z}{rne-sae}, zmm2"); // VCVTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x9d, 0xe6, 0x0a], "vcvtpd2dq xmm1{k5}{z}, qword [rdx]{1to2}"); // VCVTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x18, 0xe6, 0xca], "vcvtpd2dq ymm1{rne-sae}, zmm2"); // VCVTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x1d, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{rne-sae}, zmm2"); // VCVTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x18, 0xe6, 0x0a], "vcvtpd2dq xmm1, qword [rdx]{1to2}"); // VCVTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x1d, 0xe6, 0x0a], "vcvtpd2dq xmm1{k5}, qword [rdx]{1to2}"); // VCVTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xcd, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}{z}, zmm2"); // VCVTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0xcd, 0xe6, 0x0a], "vcvtpd2dq ymm1{k5}{z}, zmmword [rdx]"); // VCVTPD2DQ_YMMi32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x48, 0xe6, 0xca], "vcvtpd2dq ymm1, zmm2"); // VCVTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x4d, 0xe6, 0xca], "vcvtpd2dq ymm1{k5}, zmm2"); // VCVTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x48, 0xe6, 0x0a], "vcvtpd2dq ymm1, zmmword [rdx]"); // VCVTPD2DQ_YMMi32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x4d, 0xe6, 0x0a], "vcvtpd2dq ymm1{k5}, zmmword [rdx]"); // VCVTPD2DQ_YMMi32_MASKmskw_MEMf64_AVX512_VL512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x8d, 0xe6, 0xca], "vcvtpd2dq xmm1{k5}{z}, xmm2"); // VCVTPD2DQ_XMMi32_MASKmskw_XMMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x8d, 0xe6, 0x0a], "vcvtpd2dq xmm1{k5}{z}, xmmword [rdx]"); // VCVTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x08, 0xe6, 0xca], "vcvtpd2dq xmm1, xmm2"); // VCVTPD2DQ_XMMi32_MASKmskw_XMMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x0d, 0xe6, 0xca], "vcvtpd2dq xmm1{k5}, xmm2"); // VCVTPD2DQ_XMMi32_MASKmskw_XMMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x08, 0xe6, 0x0a], "vcvtpd2dq xmm1, xmmword [rdx]"); // VCVTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf1, 0xff, 0x0d, 0xe6, 0x0a], "vcvtpd2dq xmm1{k5}, xmmword [rdx]"); // VCVTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL128, extension: AVX512EVEX
}
#[test]
fn tests_66_0f38() {
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x00, 0xca], "vpshufb ymm1{k5}{z}, ymm0, ymm2"); // VPSHUFB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x00, 0x0a], "vpshufb ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSHUFB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x00, 0xca], "vpshufb ymm1, ymm0, ymm2"); // VPSHUFB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x00, 0xca], "vpshufb ymm1{k5}, ymm0, ymm2"); // VPSHUFB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x00, 0x0a], "vpshufb ymm1, ymm0, ymmword [rdx]"); // VPSHUFB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x00, 0x0a], "vpshufb ymm1{k5}, ymm0, ymmword [rdx]"); // VPSHUFB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x00, 0xca], "vpshufb zmm1{k5}{z}, zmm0, zmm2"); // VPSHUFB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x00, 0x0a], "vpshufb zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSHUFB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x00, 0xca], "vpshufb zmm1, zmm0, zmm2"); // VPSHUFB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x00, 0xca], "vpshufb zmm1{k5}, zmm0, zmm2"); // VPSHUFB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x00, 0x0a], "vpshufb zmm1, zmm0, zmmword [rdx]"); // VPSHUFB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x00, 0x0a], "vpshufb zmm1{k5}, zmm0, zmmword [rdx]"); // VPSHUFB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x00, 0xca], "vpshufb xmm1{k5}{z}, xmm0, xmm2"); // VPSHUFB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x00, 0x0a], "vpshufb xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSHUFB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x00, 0xca], "vpshufb xmm1, xmm0, xmm2"); // VPSHUFB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x00, 0xca], "vpshufb xmm1{k5}, xmm0, xmm2"); // VPSHUFB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x00, 0x0a], "vpshufb xmm1, xmm0, xmmword [rdx]"); // VPSHUFB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x00, 0x0a], "vpshufb xmm1{k5}, xmm0, xmmword [rdx]"); // VPSHUFB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x04, 0xca], "vpmaddubsw ymm1{k5}{z}, ymm0, ymm2"); // VPMADDUBSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x04, 0x0a], "vpmaddubsw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMADDUBSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x04, 0xca], "vpmaddubsw ymm1, ymm0, ymm2"); // VPMADDUBSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x04, 0xca], "vpmaddubsw ymm1{k5}, ymm0, ymm2"); // VPMADDUBSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x04, 0x0a], "vpmaddubsw ymm1, ymm0, ymmword [rdx]"); // VPMADDUBSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x04, 0x0a], "vpmaddubsw ymm1{k5}, ymm0, ymmword [rdx]"); // VPMADDUBSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x04, 0xca], "vpmaddubsw zmm1{k5}{z}, zmm0, zmm2"); // VPMADDUBSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x04, 0x0a], "vpmaddubsw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMADDUBSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x04, 0xca], "vpmaddubsw zmm1, zmm0, zmm2"); // VPMADDUBSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x04, 0xca], "vpmaddubsw zmm1{k5}, zmm0, zmm2"); // VPMADDUBSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x04, 0x0a], "vpmaddubsw zmm1, zmm0, zmmword [rdx]"); // VPMADDUBSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x04, 0x0a], "vpmaddubsw zmm1{k5}, zmm0, zmmword [rdx]"); // VPMADDUBSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x04, 0xca], "vpmaddubsw xmm1{k5}{z}, xmm0, xmm2"); // VPMADDUBSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x04, 0x0a], "vpmaddubsw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMADDUBSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x04, 0xca], "vpmaddubsw xmm1, xmm0, xmm2"); // VPMADDUBSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x04, 0xca], "vpmaddubsw xmm1{k5}, xmm0, xmm2"); // VPMADDUBSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x04, 0x0a], "vpmaddubsw xmm1, xmm0, xmmword [rdx]"); // VPMADDUBSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x04, 0x0a], "vpmaddubsw xmm1{k5}, xmm0, xmmword [rdx]"); // VPMADDUBSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x0b, 0xca], "vpmulhrsw ymm1{k5}{z}, ymm0, ymm2"); // VPMULHRSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x0b, 0x0a], "vpmulhrsw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMULHRSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x0b, 0xca], "vpmulhrsw ymm1, ymm0, ymm2"); // VPMULHRSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x0b, 0xca], "vpmulhrsw ymm1{k5}, ymm0, ymm2"); // VPMULHRSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x0b, 0x0a], "vpmulhrsw ymm1, ymm0, ymmword [rdx]"); // VPMULHRSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x0b, 0x0a], "vpmulhrsw ymm1{k5}, ymm0, ymmword [rdx]"); // VPMULHRSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x0b, 0xca], "vpmulhrsw zmm1{k5}{z}, zmm0, zmm2"); // VPMULHRSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x0b, 0x0a], "vpmulhrsw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMULHRSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x0b, 0xca], "vpmulhrsw zmm1, zmm0, zmm2"); // VPMULHRSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x0b, 0xca], "vpmulhrsw zmm1{k5}, zmm0, zmm2"); // VPMULHRSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x0b, 0x0a], "vpmulhrsw zmm1, zmm0, zmmword [rdx]"); // VPMULHRSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x0b, 0x0a], "vpmulhrsw zmm1{k5}, zmm0, zmmword [rdx]"); // VPMULHRSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x0b, 0xca], "vpmulhrsw xmm1{k5}{z}, xmm0, xmm2"); // VPMULHRSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x0b, 0x0a], "vpmulhrsw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMULHRSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x0b, 0xca], "vpmulhrsw xmm1, xmm0, xmm2"); // VPMULHRSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x0b, 0xca], "vpmulhrsw xmm1{k5}, xmm0, xmm2"); // VPMULHRSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x0b, 0x0a], "vpmulhrsw xmm1, xmm0, xmmword [rdx]"); // VPMULHRSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x0b, 0x0a], "vpmulhrsw xmm1{k5}, xmm0, xmmword [rdx]"); // VPMULHRSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x0c, 0x0a], "vpermilps ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPERMILPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x0c, 0x0a], "vpermilps ymm1, ymm0, dword [rdx]{1to8}"); // VPERMILPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x0c, 0x0a], "vpermilps ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPERMILPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x0c, 0xca], "vpermilps ymm1{k5}{z}, ymm0, ymm2"); // VPERMILPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x0c, 0x0a], "vpermilps ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPERMILPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x0c, 0xca], "vpermilps ymm1, ymm0, ymm2"); // VPERMILPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x0c, 0xca], "vpermilps ymm1{k5}, ymm0, ymm2"); // VPERMILPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x0c, 0x0a], "vpermilps ymm1, ymm0, ymmword [rdx]"); // VPERMILPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x0c, 0x0a], "vpermilps ymm1{k5}, ymm0, ymmword [rdx]"); // VPERMILPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x0c, 0x0a], "vpermilps zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPERMILPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x0c, 0x0a], "vpermilps zmm1, zmm0, dword [rdx]{1to16}"); // VPERMILPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x0c, 0x0a], "vpermilps zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPERMILPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x0c, 0x0a], "vpermilps xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPERMILPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x0c, 0x0a], "vpermilps xmm1, xmm0, dword [rdx]{1to4}"); // VPERMILPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x0c, 0x0a], "vpermilps xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPERMILPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x0c, 0xca], "vpermilps zmm1{k5}{z}, zmm0, zmm2"); // VPERMILPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x0c, 0x0a], "vpermilps zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPERMILPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x0c, 0xca], "vpermilps zmm1, zmm0, zmm2"); // VPERMILPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x0c, 0xca], "vpermilps zmm1{k5}, zmm0, zmm2"); // VPERMILPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x0c, 0x0a], "vpermilps zmm1, zmm0, zmmword [rdx]"); // VPERMILPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x0c, 0x0a], "vpermilps zmm1{k5}, zmm0, zmmword [rdx]"); // VPERMILPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x0c, 0xca], "vpermilps xmm1{k5}{z}, xmm0, xmm2"); // VPERMILPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x0c, 0x0a], "vpermilps xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPERMILPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x0c, 0xca], "vpermilps xmm1, xmm0, xmm2"); // VPERMILPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x0c, 0xca], "vpermilps xmm1{k5}, xmm0, xmm2"); // VPERMILPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x0c, 0x0a], "vpermilps xmm1, xmm0, xmmword [rdx]"); // VPERMILPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x0c, 0x0a], "vpermilps xmm1{k5}, xmm0, xmmword [rdx]"); // VPERMILPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x0d, 0x0a], "vpermilpd ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPERMILPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x0d, 0x0a], "vpermilpd ymm1, ymm0, qword [rdx]{1to4}"); // VPERMILPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x0d, 0x0a], "vpermilpd ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPERMILPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x0d, 0xca], "vpermilpd ymm1{k5}{z}, ymm0, ymm2"); // VPERMILPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x0d, 0x0a], "vpermilpd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPERMILPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x0d, 0xca], "vpermilpd ymm1, ymm0, ymm2"); // VPERMILPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x0d, 0xca], "vpermilpd ymm1{k5}, ymm0, ymm2"); // VPERMILPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x0d, 0x0a], "vpermilpd ymm1, ymm0, ymmword [rdx]"); // VPERMILPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x0d, 0x0a], "vpermilpd ymm1{k5}, ymm0, ymmword [rdx]"); // VPERMILPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x0d, 0x0a], "vpermilpd zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPERMILPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x0d, 0x0a], "vpermilpd zmm1, zmm0, qword [rdx]{1to8}"); // VPERMILPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x0d, 0x0a], "vpermilpd zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPERMILPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x0d, 0x0a], "vpermilpd xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPERMILPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x0d, 0x0a], "vpermilpd xmm1, xmm0, qword [rdx]{1to2}"); // VPERMILPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x0d, 0x0a], "vpermilpd xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPERMILPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x0d, 0xca], "vpermilpd zmm1{k5}{z}, zmm0, zmm2"); // VPERMILPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x0d, 0x0a], "vpermilpd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPERMILPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x0d, 0xca], "vpermilpd zmm1, zmm0, zmm2"); // VPERMILPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x0d, 0xca], "vpermilpd zmm1{k5}, zmm0, zmm2"); // VPERMILPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x0d, 0x0a], "vpermilpd zmm1, zmm0, zmmword [rdx]"); // VPERMILPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x0d, 0x0a], "vpermilpd zmm1{k5}, zmm0, zmmword [rdx]"); // VPERMILPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x0d, 0xca], "vpermilpd xmm1{k5}{z}, xmm0, xmm2"); // VPERMILPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x0d, 0x0a], "vpermilpd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPERMILPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x0d, 0xca], "vpermilpd xmm1, xmm0, xmm2"); // VPERMILPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x0d, 0xca], "vpermilpd xmm1{k5}, xmm0, xmm2"); // VPERMILPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x0d, 0x0a], "vpermilpd xmm1, xmm0, xmmword [rdx]"); // VPERMILPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x0d, 0x0a], "vpermilpd xmm1{k5}, xmm0, xmmword [rdx]"); // VPERMILPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x10, 0xca], "vpsrlvw ymm1{k5}{z}, ymm0, ymm2"); // VPSRLVW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x10, 0x0a], "vpsrlvw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSRLVW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x10, 0xca], "vpsrlvw ymm1, ymm0, ymm2"); // VPSRLVW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x10, 0xca], "vpsrlvw ymm1{k5}, ymm0, ymm2"); // VPSRLVW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x10, 0x0a], "vpsrlvw ymm1, ymm0, ymmword [rdx]"); // VPSRLVW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x10, 0x0a], "vpsrlvw ymm1{k5}, ymm0, ymmword [rdx]"); // VPSRLVW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x10, 0xca], "vpsrlvw zmm1{k5}{z}, zmm0, zmm2"); // VPSRLVW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x10, 0x0a], "vpsrlvw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSRLVW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x10, 0xca], "vpsrlvw zmm1, zmm0, zmm2"); // VPSRLVW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x10, 0xca], "vpsrlvw zmm1{k5}, zmm0, zmm2"); // VPSRLVW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x10, 0x0a], "vpsrlvw zmm1, zmm0, zmmword [rdx]"); // VPSRLVW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x10, 0x0a], "vpsrlvw zmm1{k5}, zmm0, zmmword [rdx]"); // VPSRLVW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x10, 0xca], "vpsrlvw xmm1{k5}{z}, xmm0, xmm2"); // VPSRLVW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x10, 0x0a], "vpsrlvw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSRLVW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x10, 0xca], "vpsrlvw xmm1, xmm0, xmm2"); // VPSRLVW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x10, 0xca], "vpsrlvw xmm1{k5}, xmm0, xmm2"); // VPSRLVW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x10, 0x0a], "vpsrlvw xmm1, xmm0, xmmword [rdx]"); // VPSRLVW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x10, 0x0a], "vpsrlvw xmm1{k5}, xmm0, xmmword [rdx]"); // VPSRLVW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x11, 0xca], "vpsravw ymm1{k5}{z}, ymm0, ymm2"); // VPSRAVW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x11, 0x0a], "vpsravw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSRAVW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x11, 0xca], "vpsravw ymm1, ymm0, ymm2"); // VPSRAVW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x11, 0xca], "vpsravw ymm1{k5}, ymm0, ymm2"); // VPSRAVW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x11, 0x0a], "vpsravw ymm1, ymm0, ymmword [rdx]"); // VPSRAVW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x11, 0x0a], "vpsravw ymm1{k5}, ymm0, ymmword [rdx]"); // VPSRAVW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x11, 0xca], "vpsravw zmm1{k5}{z}, zmm0, zmm2"); // VPSRAVW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x11, 0x0a], "vpsravw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSRAVW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x11, 0xca], "vpsravw zmm1, zmm0, zmm2"); // VPSRAVW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x11, 0xca], "vpsravw zmm1{k5}, zmm0, zmm2"); // VPSRAVW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x11, 0x0a], "vpsravw zmm1, zmm0, zmmword [rdx]"); // VPSRAVW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x11, 0x0a], "vpsravw zmm1{k5}, zmm0, zmmword [rdx]"); // VPSRAVW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x11, 0xca], "vpsravw xmm1{k5}{z}, xmm0, xmm2"); // VPSRAVW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x11, 0x0a], "vpsravw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSRAVW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x11, 0xca], "vpsravw xmm1, xmm0, xmm2"); // VPSRAVW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x11, 0xca], "vpsravw xmm1{k5}, xmm0, xmm2"); // VPSRAVW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x11, 0x0a], "vpsravw xmm1, xmm0, xmmword [rdx]"); // VPSRAVW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x11, 0x0a], "vpsravw xmm1{k5}, xmm0, xmmword [rdx]"); // VPSRAVW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x12, 0xca], "vpsllvw ymm1{k5}{z}, ymm0, ymm2"); // VPSLLVW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x12, 0x0a], "vpsllvw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSLLVW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x12, 0xca], "vpsllvw ymm1, ymm0, ymm2"); // VPSLLVW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x12, 0xca], "vpsllvw ymm1{k5}, ymm0, ymm2"); // VPSLLVW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x12, 0x0a], "vpsllvw ymm1, ymm0, ymmword [rdx]"); // VPSLLVW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x12, 0x0a], "vpsllvw ymm1{k5}, ymm0, ymmword [rdx]"); // VPSLLVW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x12, 0xca], "vpsllvw zmm1{k5}{z}, zmm0, zmm2"); // VPSLLVW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x12, 0x0a], "vpsllvw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSLLVW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x12, 0xca], "vpsllvw zmm1, zmm0, zmm2"); // VPSLLVW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x12, 0xca], "vpsllvw zmm1{k5}, zmm0, zmm2"); // VPSLLVW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x12, 0x0a], "vpsllvw zmm1, zmm0, zmmword [rdx]"); // VPSLLVW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x12, 0x0a], "vpsllvw zmm1{k5}, zmm0, zmmword [rdx]"); // VPSLLVW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x12, 0xca], "vpsllvw xmm1{k5}{z}, xmm0, xmm2"); // VPSLLVW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x12, 0x0a], "vpsllvw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSLLVW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x12, 0xca], "vpsllvw xmm1, xmm0, xmm2"); // VPSLLVW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x12, 0xca], "vpsllvw xmm1{k5}, xmm0, xmm2"); // VPSLLVW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x12, 0x0a], "vpsllvw xmm1, xmm0, xmmword [rdx]"); // VPSLLVW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x12, 0x0a], "vpsllvw xmm1{k5}, xmm0, xmmword [rdx]"); // VPSLLVW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xfd, 0x13, 0xca], "vcvtph2ps zmm1{k5}{z}{sae}, ymm2"); // VCVTPH2PS_ZMMf32_MASKmskw_YMMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x78, 0x13, 0xca], "vcvtph2ps zmm1{sae}, ymm2"); // VCVTPH2PS_ZMMf32_MASKmskw_YMMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x7d, 0x13, 0xca], "vcvtph2ps zmm1{k5}{sae}, ymm2"); // VCVTPH2PS_ZMMf32_MASKmskw_YMMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x13, 0xca], "vcvtph2ps ymm1{k5}{z}, xmm2"); // VCVTPH2PS_YMMf32_MASKmskw_XMMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x13, 0x0a], "vcvtph2ps ymm1{k5}{z}, xmmword [rdx]"); // VCVTPH2PS_YMMf32_MASKmskw_MEMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x13, 0xca], "vcvtph2ps ymm1, xmm2"); // VCVTPH2PS_YMMf32_MASKmskw_XMMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x13, 0xca], "vcvtph2ps ymm1{k5}, xmm2"); // VCVTPH2PS_YMMf32_MASKmskw_XMMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x13, 0x0a], "vcvtph2ps ymm1, xmmword [rdx]"); // VCVTPH2PS_YMMf32_MASKmskw_MEMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x13, 0x0a], "vcvtph2ps ymm1{k5}, xmmword [rdx]"); // VCVTPH2PS_YMMf32_MASKmskw_MEMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x13, 0xca], "vcvtph2ps zmm1{k5}{z}, ymm2"); // VCVTPH2PS_ZMMf32_MASKmskw_YMMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x13, 0x0a], "vcvtph2ps zmm1{k5}{z}, ymmword [rdx]"); // VCVTPH2PS_ZMMf32_MASKmskw_MEMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x13, 0xca], "vcvtph2ps zmm1, ymm2"); // VCVTPH2PS_ZMMf32_MASKmskw_YMMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x13, 0xca], "vcvtph2ps zmm1{k5}, ymm2"); // VCVTPH2PS_ZMMf32_MASKmskw_YMMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x13, 0x0a], "vcvtph2ps zmm1, ymmword [rdx]"); // VCVTPH2PS_ZMMf32_MASKmskw_MEMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x13, 0x0a], "vcvtph2ps zmm1{k5}, ymmword [rdx]"); // VCVTPH2PS_ZMMf32_MASKmskw_MEMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x13, 0xca], "vcvtph2ps xmm1{k5}{z}, xmm2"); // VCVTPH2PS_XMMf32_MASKmskw_XMMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x13, 0x0a], "vcvtph2ps xmm1{k5}{z}, qword [rdx]"); // VCVTPH2PS_XMMf32_MASKmskw_MEMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x13, 0xca], "vcvtph2ps xmm1, xmm2"); // VCVTPH2PS_XMMf32_MASKmskw_XMMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x13, 0xca], "vcvtph2ps xmm1{k5}, xmm2"); // VCVTPH2PS_XMMf32_MASKmskw_XMMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x13, 0x0a], "vcvtph2ps xmm1, qword [rdx]"); // VCVTPH2PS_XMMf32_MASKmskw_MEMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x13, 0x0a], "vcvtph2ps xmm1{k5}, qword [rdx]"); // VCVTPH2PS_XMMf32_MASKmskw_MEMf16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x14, 0x0a], "vprorvq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPRORVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x14, 0x0a], "vprorvq ymm1, ymm0, qword [rdx]{1to4}"); // VPRORVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x14, 0x0a], "vprorvq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPRORVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x14, 0xca], "vprorvq ymm1{k5}{z}, ymm0, ymm2"); // VPRORVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x14, 0x0a], "vprorvq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPRORVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x14, 0xca], "vprorvq ymm1, ymm0, ymm2"); // VPRORVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x14, 0xca], "vprorvq ymm1{k5}, ymm0, ymm2"); // VPRORVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x14, 0x0a], "vprorvq ymm1, ymm0, ymmword [rdx]"); // VPRORVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x14, 0x0a], "vprorvq ymm1{k5}, ymm0, ymmword [rdx]"); // VPRORVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x14, 0x0a], "vprorvd ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPRORVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x14, 0x0a], "vprorvd ymm1, ymm0, dword [rdx]{1to8}"); // VPRORVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x14, 0x0a], "vprorvd ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPRORVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x14, 0xca], "vprorvd ymm1{k5}{z}, ymm0, ymm2"); // VPRORVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x14, 0x0a], "vprorvd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPRORVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x14, 0xca], "vprorvd ymm1, ymm0, ymm2"); // VPRORVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x14, 0xca], "vprorvd ymm1{k5}, ymm0, ymm2"); // VPRORVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x14, 0x0a], "vprorvd ymm1, ymm0, ymmword [rdx]"); // VPRORVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x14, 0x0a], "vprorvd ymm1{k5}, ymm0, ymmword [rdx]"); // VPRORVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x14, 0x0a], "vprorvq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPRORVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x14, 0x0a], "vprorvq zmm1, zmm0, qword [rdx]{1to8}"); // VPRORVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x14, 0x0a], "vprorvq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPRORVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x14, 0x0a], "vprorvq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPRORVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x14, 0x0a], "vprorvq xmm1, xmm0, qword [rdx]{1to2}"); // VPRORVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x14, 0x0a], "vprorvq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPRORVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x14, 0xca], "vprorvq zmm1{k5}{z}, zmm0, zmm2"); // VPRORVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x14, 0x0a], "vprorvq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPRORVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x14, 0xca], "vprorvq zmm1, zmm0, zmm2"); // VPRORVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x14, 0xca], "vprorvq zmm1{k5}, zmm0, zmm2"); // VPRORVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x14, 0x0a], "vprorvq zmm1, zmm0, zmmword [rdx]"); // VPRORVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x14, 0x0a], "vprorvq zmm1{k5}, zmm0, zmmword [rdx]"); // VPRORVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x14, 0xca], "vprorvq xmm1{k5}{z}, xmm0, xmm2"); // VPRORVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x14, 0x0a], "vprorvq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPRORVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x14, 0xca], "vprorvq xmm1, xmm0, xmm2"); // VPRORVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x14, 0xca], "vprorvq xmm1{k5}, xmm0, xmm2"); // VPRORVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x14, 0x0a], "vprorvq xmm1, xmm0, xmmword [rdx]"); // VPRORVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x14, 0x0a], "vprorvq xmm1{k5}, xmm0, xmmword [rdx]"); // VPRORVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x14, 0x0a], "vprorvd zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPRORVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x14, 0x0a], "vprorvd zmm1, zmm0, dword [rdx]{1to16}"); // VPRORVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x14, 0x0a], "vprorvd zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPRORVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x14, 0x0a], "vprorvd xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPRORVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x14, 0x0a], "vprorvd xmm1, xmm0, dword [rdx]{1to4}"); // VPRORVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x14, 0x0a], "vprorvd xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPRORVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x14, 0xca], "vprorvd zmm1{k5}{z}, zmm0, zmm2"); // VPRORVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x14, 0x0a], "vprorvd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPRORVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x14, 0xca], "vprorvd zmm1, zmm0, zmm2"); // VPRORVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x14, 0xca], "vprorvd zmm1{k5}, zmm0, zmm2"); // VPRORVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x14, 0x0a], "vprorvd zmm1, zmm0, zmmword [rdx]"); // VPRORVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x14, 0x0a], "vprorvd zmm1{k5}, zmm0, zmmword [rdx]"); // VPRORVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x14, 0xca], "vprorvd xmm1{k5}{z}, xmm0, xmm2"); // VPRORVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x14, 0x0a], "vprorvd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPRORVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x14, 0xca], "vprorvd xmm1, xmm0, xmm2"); // VPRORVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x14, 0xca], "vprorvd xmm1{k5}, xmm0, xmm2"); // VPRORVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x14, 0x0a], "vprorvd xmm1, xmm0, xmmword [rdx]"); // VPRORVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x14, 0x0a], "vprorvd xmm1{k5}, xmm0, xmmword [rdx]"); // VPRORVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x15, 0x0a], "vprolvq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPROLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x15, 0x0a], "vprolvq ymm1, ymm0, qword [rdx]{1to4}"); // VPROLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x15, 0x0a], "vprolvq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPROLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x15, 0xca], "vprolvq ymm1{k5}{z}, ymm0, ymm2"); // VPROLVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x15, 0x0a], "vprolvq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPROLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x15, 0xca], "vprolvq ymm1, ymm0, ymm2"); // VPROLVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x15, 0xca], "vprolvq ymm1{k5}, ymm0, ymm2"); // VPROLVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x15, 0x0a], "vprolvq ymm1, ymm0, ymmword [rdx]"); // VPROLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x15, 0x0a], "vprolvq ymm1{k5}, ymm0, ymmword [rdx]"); // VPROLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x15, 0x0a], "vprolvd ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPROLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x15, 0x0a], "vprolvd ymm1, ymm0, dword [rdx]{1to8}"); // VPROLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x15, 0x0a], "vprolvd ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPROLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x15, 0xca], "vprolvd ymm1{k5}{z}, ymm0, ymm2"); // VPROLVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x15, 0x0a], "vprolvd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPROLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x15, 0xca], "vprolvd ymm1, ymm0, ymm2"); // VPROLVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x15, 0xca], "vprolvd ymm1{k5}, ymm0, ymm2"); // VPROLVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x15, 0x0a], "vprolvd ymm1, ymm0, ymmword [rdx]"); // VPROLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x15, 0x0a], "vprolvd ymm1{k5}, ymm0, ymmword [rdx]"); // VPROLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x15, 0x0a], "vprolvq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPROLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x15, 0x0a], "vprolvq zmm1, zmm0, qword [rdx]{1to8}"); // VPROLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x15, 0x0a], "vprolvq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPROLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x15, 0x0a], "vprolvq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPROLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x15, 0x0a], "vprolvq xmm1, xmm0, qword [rdx]{1to2}"); // VPROLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x15, 0x0a], "vprolvq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPROLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x15, 0xca], "vprolvq zmm1{k5}{z}, zmm0, zmm2"); // VPROLVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x15, 0x0a], "vprolvq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPROLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x15, 0xca], "vprolvq zmm1, zmm0, zmm2"); // VPROLVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x15, 0xca], "vprolvq zmm1{k5}, zmm0, zmm2"); // VPROLVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x15, 0x0a], "vprolvq zmm1, zmm0, zmmword [rdx]"); // VPROLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x15, 0x0a], "vprolvq zmm1{k5}, zmm0, zmmword [rdx]"); // VPROLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x15, 0xca], "vprolvq xmm1{k5}{z}, xmm0, xmm2"); // VPROLVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x15, 0x0a], "vprolvq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPROLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x15, 0xca], "vprolvq xmm1, xmm0, xmm2"); // VPROLVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x15, 0xca], "vprolvq xmm1{k5}, xmm0, xmm2"); // VPROLVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x15, 0x0a], "vprolvq xmm1, xmm0, xmmword [rdx]"); // VPROLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x15, 0x0a], "vprolvq xmm1{k5}, xmm0, xmmword [rdx]"); // VPROLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x15, 0x0a], "vprolvd zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPROLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x15, 0x0a], "vprolvd zmm1, zmm0, dword [rdx]{1to16}"); // VPROLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x15, 0x0a], "vprolvd zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPROLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x15, 0x0a], "vprolvd xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPROLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x15, 0x0a], "vprolvd xmm1, xmm0, dword [rdx]{1to4}"); // VPROLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x15, 0x0a], "vprolvd xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPROLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x15, 0xca], "vprolvd zmm1{k5}{z}, zmm0, zmm2"); // VPROLVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x15, 0x0a], "vprolvd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPROLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x15, 0xca], "vprolvd zmm1, zmm0, zmm2"); // VPROLVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x15, 0xca], "vprolvd zmm1{k5}, zmm0, zmm2"); // VPROLVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x15, 0x0a], "vprolvd zmm1, zmm0, zmmword [rdx]"); // VPROLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x15, 0x0a], "vprolvd zmm1{k5}, zmm0, zmmword [rdx]"); // VPROLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x15, 0xca], "vprolvd xmm1{k5}{z}, xmm0, xmm2"); // VPROLVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x15, 0x0a], "vprolvd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPROLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x15, 0xca], "vprolvd xmm1, xmm0, xmm2"); // VPROLVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x15, 0xca], "vprolvd xmm1{k5}, xmm0, xmm2"); // VPROLVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x15, 0x0a], "vprolvd xmm1, xmm0, xmmword [rdx]"); // VPROLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x15, 0x0a], "vprolvd xmm1{k5}, xmm0, xmmword [rdx]"); // VPROLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x16, 0x0a], "vpermpd ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPERMPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x16, 0x0a], "vpermpd ymm1, ymm0, qword [rdx]{1to4}"); // VPERMPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x16, 0x0a], "vpermpd ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPERMPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x16, 0xca], "vpermpd ymm1{k5}{z}, ymm0, ymm2"); // VPERMPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x16, 0x0a], "vpermpd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPERMPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x16, 0xca], "vpermpd ymm1, ymm0, ymm2"); // VPERMPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x16, 0xca], "vpermpd ymm1{k5}, ymm0, ymm2"); // VPERMPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x16, 0x0a], "vpermpd ymm1, ymm0, ymmword [rdx]"); // VPERMPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x16, 0x0a], "vpermpd ymm1{k5}, ymm0, ymmword [rdx]"); // VPERMPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x16, 0x0a], "vpermps ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPERMPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x16, 0x0a], "vpermps ymm1, ymm0, dword [rdx]{1to8}"); // VPERMPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x16, 0x0a], "vpermps ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPERMPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x16, 0xca], "vpermps ymm1{k5}{z}, ymm0, ymm2"); // VPERMPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x16, 0x0a], "vpermps ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPERMPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x16, 0xca], "vpermps ymm1, ymm0, ymm2"); // VPERMPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x16, 0xca], "vpermps ymm1{k5}, ymm0, ymm2"); // VPERMPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x16, 0x0a], "vpermps ymm1, ymm0, ymmword [rdx]"); // VPERMPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x16, 0x0a], "vpermps ymm1{k5}, ymm0, ymmword [rdx]"); // VPERMPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x16, 0x0a], "vpermpd zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPERMPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x16, 0x0a], "vpermpd zmm1, zmm0, qword [rdx]{1to8}"); // VPERMPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x16, 0x0a], "vpermpd zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPERMPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x16, 0xca], "vpermpd zmm1{k5}{z}, zmm0, zmm2"); // VPERMPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x16, 0x0a], "vpermpd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPERMPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x16, 0xca], "vpermpd zmm1, zmm0, zmm2"); // VPERMPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x16, 0xca], "vpermpd zmm1{k5}, zmm0, zmm2"); // VPERMPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x16, 0x0a], "vpermpd zmm1, zmm0, zmmword [rdx]"); // VPERMPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x16, 0x0a], "vpermpd zmm1{k5}, zmm0, zmmword [rdx]"); // VPERMPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x16, 0x0a], "vpermps zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPERMPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x16, 0x0a], "vpermps zmm1, zmm0, dword [rdx]{1to16}"); // VPERMPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x16, 0x0a], "vpermps zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPERMPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x16, 0xca], "vpermps zmm1{k5}{z}, zmm0, zmm2"); // VPERMPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x16, 0x0a], "vpermps zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPERMPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x16, 0xca], "vpermps zmm1, zmm0, zmm2"); // VPERMPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x16, 0xca], "vpermps zmm1{k5}, zmm0, zmm2"); // VPERMPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x16, 0x0a], "vpermps zmm1, zmm0, zmmword [rdx]"); // VPERMPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x16, 0x0a], "vpermps zmm1{k5}, zmm0, zmmword [rdx]"); // VPERMPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x18, 0xca], "vbroadcastss ymm1{k5}{z}, xmm2"); // VBROADCASTSS_YMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x18, 0x0a], "vbroadcastss ymm1{k5}{z}, dword [rdx]"); // VBROADCASTSS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x18, 0xca], "vbroadcastss ymm1, xmm2"); // VBROADCASTSS_YMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x18, 0xca], "vbroadcastss ymm1{k5}, xmm2"); // VBROADCASTSS_YMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x18, 0x0a], "vbroadcastss ymm1, dword [rdx]"); // VBROADCASTSS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x18, 0x0a], "vbroadcastss ymm1{k5}, dword [rdx]"); // VBROADCASTSS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x18, 0xca], "vbroadcastss zmm1{k5}{z}, xmm2"); // VBROADCASTSS_ZMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x18, 0x0a], "vbroadcastss zmm1{k5}{z}, dword [rdx]"); // VBROADCASTSS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x18, 0xca], "vbroadcastss zmm1, xmm2"); // VBROADCASTSS_ZMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x18, 0xca], "vbroadcastss zmm1{k5}, xmm2"); // VBROADCASTSS_ZMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x18, 0x0a], "vbroadcastss zmm1, dword [rdx]"); // VBROADCASTSS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x18, 0x0a], "vbroadcastss zmm1{k5}, dword [rdx]"); // VBROADCASTSS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x18, 0xca], "vbroadcastss xmm1{k5}{z}, xmm2"); // VBROADCASTSS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x18, 0x0a], "vbroadcastss xmm1{k5}{z}, dword [rdx]"); // VBROADCASTSS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x18, 0xca], "vbroadcastss xmm1, xmm2"); // VBROADCASTSS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x18, 0xca], "vbroadcastss xmm1{k5}, xmm2"); // VBROADCASTSS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x18, 0x0a], "vbroadcastss xmm1, dword [rdx]"); // VBROADCASTSS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x18, 0x0a], "vbroadcastss xmm1{k5}, dword [rdx]"); // VBROADCASTSS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x19, 0xca], "vbroadcastsd ymm1{k5}{z}, xmm2"); // VBROADCASTSD_YMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x19, 0x0a], "vbroadcastsd ymm1{k5}{z}, qword [rdx]"); // VBROADCASTSD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x19, 0xca], "vbroadcastsd ymm1, xmm2"); // VBROADCASTSD_YMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x19, 0xca], "vbroadcastsd ymm1{k5}, xmm2"); // VBROADCASTSD_YMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x19, 0x0a], "vbroadcastsd ymm1, qword [rdx]"); // VBROADCASTSD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x19, 0x0a], "vbroadcastsd ymm1{k5}, qword [rdx]"); // VBROADCASTSD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x19, 0xca], "vbroadcastf32x2 ymm1{k5}{z}, xmm2"); // VBROADCASTF32X2_YMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x19, 0x0a], "vbroadcastf32x2 ymm1{k5}{z}, qword [rdx]"); // VBROADCASTF32X2_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x19, 0xca], "vbroadcastf32x2 ymm1, xmm2"); // VBROADCASTF32X2_YMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x19, 0xca], "vbroadcastf32x2 ymm1{k5}, xmm2"); // VBROADCASTF32X2_YMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x19, 0x0a], "vbroadcastf32x2 ymm1, qword [rdx]"); // VBROADCASTF32X2_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x19, 0x0a], "vbroadcastf32x2 ymm1{k5}, qword [rdx]"); // VBROADCASTF32X2_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x19, 0xca], "vbroadcastsd zmm1{k5}{z}, xmm2"); // VBROADCASTSD_ZMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x19, 0x0a], "vbroadcastsd zmm1{k5}{z}, qword [rdx]"); // VBROADCASTSD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x19, 0xca], "vbroadcastsd zmm1, xmm2"); // VBROADCASTSD_ZMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x19, 0xca], "vbroadcastsd zmm1{k5}, xmm2"); // VBROADCASTSD_ZMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x19, 0x0a], "vbroadcastsd zmm1, qword [rdx]"); // VBROADCASTSD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x19, 0x0a], "vbroadcastsd zmm1{k5}, qword [rdx]"); // VBROADCASTSD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x19, 0xca], "vbroadcastf32x2 zmm1{k5}{z}, xmm2"); // VBROADCASTF32X2_ZMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x19, 0x0a], "vbroadcastf32x2 zmm1{k5}{z}, qword [rdx]"); // VBROADCASTF32X2_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x19, 0xca], "vbroadcastf32x2 zmm1, xmm2"); // VBROADCASTF32X2_ZMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x19, 0xca], "vbroadcastf32x2 zmm1{k5}, xmm2"); // VBROADCASTF32X2_ZMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x19, 0x0a], "vbroadcastf32x2 zmm1, qword [rdx]"); // VBROADCASTF32X2_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x19, 0x0a], "vbroadcastf32x2 zmm1{k5}, qword [rdx]"); // VBROADCASTF32X2_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1a, 0x0a], "vbroadcastf64x2 ymm1{k5}{z}, xmmword [rdx]"); // VBROADCASTF64X2_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1a, 0x0a], "vbroadcastf64x2 ymm1, xmmword [rdx]"); // VBROADCASTF64X2_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1a, 0x0a], "vbroadcastf64x2 ymm1{k5}, xmmword [rdx]"); // VBROADCASTF64X2_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x1a, 0x0a], "vbroadcastf32x4 ymm1{k5}{z}, xmmword [rdx]"); // VBROADCASTF32X4_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x1a, 0x0a], "vbroadcastf32x4 ymm1, xmmword [rdx]"); // VBROADCASTF32X4_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x1a, 0x0a], "vbroadcastf32x4 ymm1{k5}, xmmword [rdx]"); // VBROADCASTF32X4_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1a, 0x0a], "vbroadcastf64x2 zmm1{k5}{z}, xmmword [rdx]"); // VBROADCASTF64X2_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1a, 0x0a], "vbroadcastf64x2 zmm1, xmmword [rdx]"); // VBROADCASTF64X2_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1a, 0x0a], "vbroadcastf64x2 zmm1{k5}, xmmword [rdx]"); // VBROADCASTF64X2_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x1a, 0x0a], "vbroadcastf32x4 zmm1{k5}{z}, xmmword [rdx]"); // VBROADCASTF32X4_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x1a, 0x0a], "vbroadcastf32x4 zmm1, xmmword [rdx]"); // VBROADCASTF32X4_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x1a, 0x0a], "vbroadcastf32x4 zmm1{k5}, xmmword [rdx]"); // VBROADCASTF32X4_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1b, 0x0a], "vbroadcastf64x4 zmm1{k5}{z}, ymmword [rdx]"); // VBROADCASTF64X4_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1b, 0x0a], "vbroadcastf64x4 zmm1, ymmword [rdx]"); // VBROADCASTF64X4_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1b, 0x0a], "vbroadcastf64x4 zmm1{k5}, ymmword [rdx]"); // VBROADCASTF64X4_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x1b, 0x0a], "vbroadcastf32x8 zmm1{k5}{z}, ymmword [rdx]"); // VBROADCASTF32X8_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x1b, 0x0a], "vbroadcastf32x8 zmm1, ymmword [rdx]"); // VBROADCASTF32X8_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x1b, 0x0a], "vbroadcastf32x8 zmm1{k5}, ymmword [rdx]"); // VBROADCASTF32X8_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1c, 0xca], "vpabsb ymm1{k5}{z}, ymm2"); // VPABSB_YMMi8_MASKmskw_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1c, 0x0a], "vpabsb ymm1{k5}{z}, ymmword [rdx]"); // VPABSB_YMMi8_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1c, 0xca], "vpabsb ymm1, ymm2"); // VPABSB_YMMi8_MASKmskw_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1c, 0xca], "vpabsb ymm1{k5}, ymm2"); // VPABSB_YMMi8_MASKmskw_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1c, 0x0a], "vpabsb ymm1, ymmword [rdx]"); // VPABSB_YMMi8_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1c, 0x0a], "vpabsb ymm1{k5}, ymmword [rdx]"); // VPABSB_YMMi8_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1c, 0xca], "vpabsb zmm1{k5}{z}, zmm2"); // VPABSB_ZMMi8_MASKmskw_ZMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1c, 0x0a], "vpabsb zmm1{k5}{z}, zmmword [rdx]"); // VPABSB_ZMMi8_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1c, 0xca], "vpabsb zmm1, zmm2"); // VPABSB_ZMMi8_MASKmskw_ZMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1c, 0xca], "vpabsb zmm1{k5}, zmm2"); // VPABSB_ZMMi8_MASKmskw_ZMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1c, 0x0a], "vpabsb zmm1, zmmword [rdx]"); // VPABSB_ZMMi8_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1c, 0x0a], "vpabsb zmm1{k5}, zmmword [rdx]"); // VPABSB_ZMMi8_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x1c, 0xca], "vpabsb xmm1{k5}{z}, xmm2"); // VPABSB_XMMi8_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x1c, 0x0a], "vpabsb xmm1{k5}{z}, xmmword [rdx]"); // VPABSB_XMMi8_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x1c, 0xca], "vpabsb xmm1, xmm2"); // VPABSB_XMMi8_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x1c, 0xca], "vpabsb xmm1{k5}, xmm2"); // VPABSB_XMMi8_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x1c, 0x0a], "vpabsb xmm1, xmmword [rdx]"); // VPABSB_XMMi8_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x1c, 0x0a], "vpabsb xmm1{k5}, xmmword [rdx]"); // VPABSB_XMMi8_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1d, 0xca], "vpabsw ymm1{k5}{z}, ymm2"); // VPABSW_YMMi16_MASKmskw_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1d, 0x0a], "vpabsw ymm1{k5}{z}, ymmword [rdx]"); // VPABSW_YMMi16_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1d, 0xca], "vpabsw ymm1, ymm2"); // VPABSW_YMMi16_MASKmskw_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1d, 0xca], "vpabsw ymm1{k5}, ymm2"); // VPABSW_YMMi16_MASKmskw_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1d, 0x0a], "vpabsw ymm1, ymmword [rdx]"); // VPABSW_YMMi16_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1d, 0x0a], "vpabsw ymm1{k5}, ymmword [rdx]"); // VPABSW_YMMi16_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1d, 0xca], "vpabsw zmm1{k5}{z}, zmm2"); // VPABSW_ZMMi16_MASKmskw_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1d, 0x0a], "vpabsw zmm1{k5}{z}, zmmword [rdx]"); // VPABSW_ZMMi16_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1d, 0xca], "vpabsw zmm1, zmm2"); // VPABSW_ZMMi16_MASKmskw_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1d, 0xca], "vpabsw zmm1{k5}, zmm2"); // VPABSW_ZMMi16_MASKmskw_ZMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1d, 0x0a], "vpabsw zmm1, zmmword [rdx]"); // VPABSW_ZMMi16_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1d, 0x0a], "vpabsw zmm1{k5}, zmmword [rdx]"); // VPABSW_ZMMi16_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x1d, 0xca], "vpabsw xmm1{k5}{z}, xmm2"); // VPABSW_XMMi16_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x1d, 0x0a], "vpabsw xmm1{k5}{z}, xmmword [rdx]"); // VPABSW_XMMi16_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x1d, 0xca], "vpabsw xmm1, xmm2"); // VPABSW_XMMi16_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x1d, 0xca], "vpabsw xmm1{k5}, xmm2"); // VPABSW_XMMi16_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x1d, 0x0a], "vpabsw xmm1, xmmword [rdx]"); // VPABSW_XMMi16_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x1d, 0x0a], "vpabsw xmm1{k5}, xmmword [rdx]"); // VPABSW_XMMi16_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x1e, 0x0a], "vpabsd ymm1{k5}{z}, dword [rdx]{1to8}"); // VPABSD_YMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x1e, 0x0a], "vpabsd ymm1, dword [rdx]{1to8}"); // VPABSD_YMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x1e, 0x0a], "vpabsd ymm1{k5}, dword [rdx]{1to8}"); // VPABSD_YMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x1e, 0xca], "vpabsd ymm1{k5}{z}, ymm2"); // VPABSD_YMMi32_MASKmskw_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x1e, 0x0a], "vpabsd ymm1{k5}{z}, ymmword [rdx]"); // VPABSD_YMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x1e, 0xca], "vpabsd ymm1, ymm2"); // VPABSD_YMMi32_MASKmskw_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x1e, 0xca], "vpabsd ymm1{k5}, ymm2"); // VPABSD_YMMi32_MASKmskw_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x1e, 0x0a], "vpabsd ymm1, ymmword [rdx]"); // VPABSD_YMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x1e, 0x0a], "vpabsd ymm1{k5}, ymmword [rdx]"); // VPABSD_YMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x1e, 0x0a], "vpabsd zmm1{k5}{z}, dword [rdx]{1to16}"); // VPABSD_ZMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x1e, 0x0a], "vpabsd zmm1, dword [rdx]{1to16}"); // VPABSD_ZMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x1e, 0x0a], "vpabsd zmm1{k5}, dword [rdx]{1to16}"); // VPABSD_ZMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x1e, 0x0a], "vpabsd xmm1{k5}{z}, dword [rdx]{1to4}"); // VPABSD_XMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x1e, 0x0a], "vpabsd xmm1, dword [rdx]{1to4}"); // VPABSD_XMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x1e, 0x0a], "vpabsd xmm1{k5}, dword [rdx]{1to4}"); // VPABSD_XMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x1e, 0xca], "vpabsd zmm1{k5}{z}, zmm2"); // VPABSD_ZMMi32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x1e, 0x0a], "vpabsd zmm1{k5}{z}, zmmword [rdx]"); // VPABSD_ZMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x1e, 0xca], "vpabsd zmm1, zmm2"); // VPABSD_ZMMi32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x1e, 0xca], "vpabsd zmm1{k5}, zmm2"); // VPABSD_ZMMi32_MASKmskw_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x1e, 0x0a], "vpabsd zmm1, zmmword [rdx]"); // VPABSD_ZMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x1e, 0x0a], "vpabsd zmm1{k5}, zmmword [rdx]"); // VPABSD_ZMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x1e, 0xca], "vpabsd xmm1{k5}{z}, xmm2"); // VPABSD_XMMi32_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x1e, 0x0a], "vpabsd xmm1{k5}{z}, xmmword [rdx]"); // VPABSD_XMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x1e, 0xca], "vpabsd xmm1, xmm2"); // VPABSD_XMMi32_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x1e, 0xca], "vpabsd xmm1{k5}, xmm2"); // VPABSD_XMMi32_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x1e, 0x0a], "vpabsd xmm1, xmmword [rdx]"); // VPABSD_XMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x1e, 0x0a], "vpabsd xmm1{k5}, xmmword [rdx]"); // VPABSD_XMMi32_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x1f, 0x0a], "vpabsq ymm1{k5}{z}, qword [rdx]{1to4}"); // VPABSQ_YMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x1f, 0x0a], "vpabsq ymm1, qword [rdx]{1to4}"); // VPABSQ_YMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x1f, 0x0a], "vpabsq ymm1{k5}, qword [rdx]{1to4}"); // VPABSQ_YMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1f, 0xca], "vpabsq ymm1{k5}{z}, ymm2"); // VPABSQ_YMMi64_MASKmskw_YMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x1f, 0x0a], "vpabsq ymm1{k5}{z}, ymmword [rdx]"); // VPABSQ_YMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1f, 0xca], "vpabsq ymm1, ymm2"); // VPABSQ_YMMi64_MASKmskw_YMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1f, 0xca], "vpabsq ymm1{k5}, ymm2"); // VPABSQ_YMMi64_MASKmskw_YMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x1f, 0x0a], "vpabsq ymm1, ymmword [rdx]"); // VPABSQ_YMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x1f, 0x0a], "vpabsq ymm1{k5}, ymmword [rdx]"); // VPABSQ_YMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x1f, 0x0a], "vpabsq zmm1{k5}{z}, qword [rdx]{1to8}"); // VPABSQ_ZMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x1f, 0x0a], "vpabsq zmm1, qword [rdx]{1to8}"); // VPABSQ_ZMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x1f, 0x0a], "vpabsq zmm1{k5}, qword [rdx]{1to8}"); // VPABSQ_ZMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x1f, 0x0a], "vpabsq xmm1{k5}{z}, qword [rdx]{1to2}"); // VPABSQ_XMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x1f, 0x0a], "vpabsq xmm1, qword [rdx]{1to2}"); // VPABSQ_XMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x1f, 0x0a], "vpabsq xmm1{k5}, qword [rdx]{1to2}"); // VPABSQ_XMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1f, 0xca], "vpabsq zmm1{k5}{z}, zmm2"); // VPABSQ_ZMMi64_MASKmskw_ZMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x1f, 0x0a], "vpabsq zmm1{k5}{z}, zmmword [rdx]"); // VPABSQ_ZMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1f, 0xca], "vpabsq zmm1, zmm2"); // VPABSQ_ZMMi64_MASKmskw_ZMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1f, 0xca], "vpabsq zmm1{k5}, zmm2"); // VPABSQ_ZMMi64_MASKmskw_ZMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x1f, 0x0a], "vpabsq zmm1, zmmword [rdx]"); // VPABSQ_ZMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x1f, 0x0a], "vpabsq zmm1{k5}, zmmword [rdx]"); // VPABSQ_ZMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x1f, 0xca], "vpabsq xmm1{k5}{z}, xmm2"); // VPABSQ_XMMi64_MASKmskw_XMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x1f, 0x0a], "vpabsq xmm1{k5}{z}, xmmword [rdx]"); // VPABSQ_XMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x1f, 0xca], "vpabsq xmm1, xmm2"); // VPABSQ_XMMi64_MASKmskw_XMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x1f, 0xca], "vpabsq xmm1{k5}, xmm2"); // VPABSQ_XMMi64_MASKmskw_XMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x1f, 0x0a], "vpabsq xmm1, xmmword [rdx]"); // VPABSQ_XMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x1f, 0x0a], "vpabsq xmm1{k5}, xmmword [rdx]"); // VPABSQ_XMMi64_MASKmskw_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x20, 0xca], "vpmovsxbw ymm1{k5}{z}, xmm2"); // VPMOVSXBW_YMMi16_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x20, 0x0a], "vpmovsxbw ymm1{k5}{z}, xmmword [rdx]"); // VPMOVSXBW_YMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x20, 0xca], "vpmovsxbw ymm1, xmm2"); // VPMOVSXBW_YMMi16_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x20, 0xca], "vpmovsxbw ymm1{k5}, xmm2"); // VPMOVSXBW_YMMi16_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x20, 0x0a], "vpmovsxbw ymm1, xmmword [rdx]"); // VPMOVSXBW_YMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x20, 0x0a], "vpmovsxbw ymm1{k5}, xmmword [rdx]"); // VPMOVSXBW_YMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x20, 0xca], "vpmovsxbw zmm1{k5}{z}, ymm2"); // VPMOVSXBW_ZMMi16_MASKmskw_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x20, 0x0a], "vpmovsxbw zmm1{k5}{z}, ymmword [rdx]"); // VPMOVSXBW_ZMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x20, 0xca], "vpmovsxbw zmm1, ymm2"); // VPMOVSXBW_ZMMi16_MASKmskw_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x20, 0xca], "vpmovsxbw zmm1{k5}, ymm2"); // VPMOVSXBW_ZMMi16_MASKmskw_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x20, 0x0a], "vpmovsxbw zmm1, ymmword [rdx]"); // VPMOVSXBW_ZMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x20, 0x0a], "vpmovsxbw zmm1{k5}, ymmword [rdx]"); // VPMOVSXBW_ZMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x20, 0xca], "vpmovsxbw xmm1{k5}{z}, xmm2"); // VPMOVSXBW_XMMi16_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x20, 0x0a], "vpmovsxbw xmm1{k5}{z}, qword [rdx]"); // VPMOVSXBW_XMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x20, 0xca], "vpmovsxbw xmm1, xmm2"); // VPMOVSXBW_XMMi16_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x20, 0xca], "vpmovsxbw xmm1{k5}, xmm2"); // VPMOVSXBW_XMMi16_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x20, 0x0a], "vpmovsxbw xmm1, qword [rdx]"); // VPMOVSXBW_XMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x20, 0x0a], "vpmovsxbw xmm1{k5}, qword [rdx]"); // VPMOVSXBW_XMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x21, 0xca], "vpmovsxbd ymm1{k5}{z}, xmm2"); // VPMOVSXBD_YMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x21, 0x0a], "vpmovsxbd ymm1{k5}{z}, qword [rdx]"); // VPMOVSXBD_YMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x21, 0xca], "vpmovsxbd ymm1, xmm2"); // VPMOVSXBD_YMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x21, 0xca], "vpmovsxbd ymm1{k5}, xmm2"); // VPMOVSXBD_YMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x21, 0x0a], "vpmovsxbd ymm1, qword [rdx]"); // VPMOVSXBD_YMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x21, 0x0a], "vpmovsxbd ymm1{k5}, qword [rdx]"); // VPMOVSXBD_YMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x21, 0xca], "vpmovsxbd zmm1{k5}{z}, xmm2"); // VPMOVSXBD_ZMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x21, 0x0a], "vpmovsxbd zmm1{k5}{z}, xmmword [rdx]"); // VPMOVSXBD_ZMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x21, 0xca], "vpmovsxbd zmm1, xmm2"); // VPMOVSXBD_ZMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x21, 0xca], "vpmovsxbd zmm1{k5}, xmm2"); // VPMOVSXBD_ZMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x21, 0x0a], "vpmovsxbd zmm1, xmmword [rdx]"); // VPMOVSXBD_ZMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x21, 0x0a], "vpmovsxbd zmm1{k5}, xmmword [rdx]"); // VPMOVSXBD_ZMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x21, 0xca], "vpmovsxbd xmm1{k5}{z}, xmm2"); // VPMOVSXBD_XMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x21, 0x0a], "vpmovsxbd xmm1{k5}{z}, dword [rdx]"); // VPMOVSXBD_XMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x21, 0xca], "vpmovsxbd xmm1, xmm2"); // VPMOVSXBD_XMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x21, 0xca], "vpmovsxbd xmm1{k5}, xmm2"); // VPMOVSXBD_XMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x21, 0x0a], "vpmovsxbd xmm1, dword [rdx]"); // VPMOVSXBD_XMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x21, 0x0a], "vpmovsxbd xmm1{k5}, dword [rdx]"); // VPMOVSXBD_XMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x22, 0xca], "vpmovsxbq ymm1{k5}{z}, xmm2"); // VPMOVSXBQ_YMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x22, 0x0a], "vpmovsxbq ymm1{k5}{z}, dword [rdx]"); // VPMOVSXBQ_YMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x22, 0xca], "vpmovsxbq ymm1, xmm2"); // VPMOVSXBQ_YMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x22, 0xca], "vpmovsxbq ymm1{k5}, xmm2"); // VPMOVSXBQ_YMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x22, 0x0a], "vpmovsxbq ymm1, dword [rdx]"); // VPMOVSXBQ_YMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x22, 0x0a], "vpmovsxbq ymm1{k5}, dword [rdx]"); // VPMOVSXBQ_YMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x22, 0xca], "vpmovsxbq zmm1{k5}{z}, xmm2"); // VPMOVSXBQ_ZMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x22, 0x0a], "vpmovsxbq zmm1{k5}{z}, qword [rdx]"); // VPMOVSXBQ_ZMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x22, 0xca], "vpmovsxbq zmm1, xmm2"); // VPMOVSXBQ_ZMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x22, 0xca], "vpmovsxbq zmm1{k5}, xmm2"); // VPMOVSXBQ_ZMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x22, 0x0a], "vpmovsxbq zmm1, qword [rdx]"); // VPMOVSXBQ_ZMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x22, 0x0a], "vpmovsxbq zmm1{k5}, qword [rdx]"); // VPMOVSXBQ_ZMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x22, 0xca], "vpmovsxbq xmm1{k5}{z}, xmm2"); // VPMOVSXBQ_XMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x22, 0x0a], "vpmovsxbq xmm1{k5}{z}, word [rdx]"); // VPMOVSXBQ_XMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x22, 0xca], "vpmovsxbq xmm1, xmm2"); // VPMOVSXBQ_XMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x22, 0xca], "vpmovsxbq xmm1{k5}, xmm2"); // VPMOVSXBQ_XMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x22, 0x0a], "vpmovsxbq xmm1, word [rdx]"); // VPMOVSXBQ_XMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x22, 0x0a], "vpmovsxbq xmm1{k5}, word [rdx]"); // VPMOVSXBQ_XMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x23, 0xca], "vpmovsxwd ymm1{k5}{z}, xmm2"); // VPMOVSXWD_YMMi32_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x23, 0x0a], "vpmovsxwd ymm1{k5}{z}, xmmword [rdx]"); // VPMOVSXWD_YMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x23, 0xca], "vpmovsxwd ymm1, xmm2"); // VPMOVSXWD_YMMi32_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x23, 0xca], "vpmovsxwd ymm1{k5}, xmm2"); // VPMOVSXWD_YMMi32_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x23, 0x0a], "vpmovsxwd ymm1, xmmword [rdx]"); // VPMOVSXWD_YMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x23, 0x0a], "vpmovsxwd ymm1{k5}, xmmword [rdx]"); // VPMOVSXWD_YMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x23, 0xca], "vpmovsxwd zmm1{k5}{z}, ymm2"); // VPMOVSXWD_ZMMi32_MASKmskw_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x23, 0x0a], "vpmovsxwd zmm1{k5}{z}, ymmword [rdx]"); // VPMOVSXWD_ZMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x23, 0xca], "vpmovsxwd zmm1, ymm2"); // VPMOVSXWD_ZMMi32_MASKmskw_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x23, 0xca], "vpmovsxwd zmm1{k5}, ymm2"); // VPMOVSXWD_ZMMi32_MASKmskw_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x23, 0x0a], "vpmovsxwd zmm1, ymmword [rdx]"); // VPMOVSXWD_ZMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x23, 0x0a], "vpmovsxwd zmm1{k5}, ymmword [rdx]"); // VPMOVSXWD_ZMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x23, 0xca], "vpmovsxwd xmm1{k5}{z}, xmm2"); // VPMOVSXWD_XMMi32_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x23, 0x0a], "vpmovsxwd xmm1{k5}{z}, qword [rdx]"); // VPMOVSXWD_XMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x23, 0xca], "vpmovsxwd xmm1, xmm2"); // VPMOVSXWD_XMMi32_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x23, 0xca], "vpmovsxwd xmm1{k5}, xmm2"); // VPMOVSXWD_XMMi32_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x23, 0x0a], "vpmovsxwd xmm1, qword [rdx]"); // VPMOVSXWD_XMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x23, 0x0a], "vpmovsxwd xmm1{k5}, qword [rdx]"); // VPMOVSXWD_XMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x24, 0xca], "vpmovsxwq ymm1{k5}{z}, xmm2"); // VPMOVSXWQ_YMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x24, 0x0a], "vpmovsxwq ymm1{k5}{z}, qword [rdx]"); // VPMOVSXWQ_YMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x24, 0xca], "vpmovsxwq ymm1, xmm2"); // VPMOVSXWQ_YMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x24, 0xca], "vpmovsxwq ymm1{k5}, xmm2"); // VPMOVSXWQ_YMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x24, 0x0a], "vpmovsxwq ymm1, qword [rdx]"); // VPMOVSXWQ_YMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x24, 0x0a], "vpmovsxwq ymm1{k5}, qword [rdx]"); // VPMOVSXWQ_YMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x24, 0xca], "vpmovsxwq zmm1{k5}{z}, xmm2"); // VPMOVSXWQ_ZMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x24, 0x0a], "vpmovsxwq zmm1{k5}{z}, xmmword [rdx]"); // VPMOVSXWQ_ZMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x24, 0xca], "vpmovsxwq zmm1, xmm2"); // VPMOVSXWQ_ZMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x24, 0xca], "vpmovsxwq zmm1{k5}, xmm2"); // VPMOVSXWQ_ZMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x24, 0x0a], "vpmovsxwq zmm1, xmmword [rdx]"); // VPMOVSXWQ_ZMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x24, 0x0a], "vpmovsxwq zmm1{k5}, xmmword [rdx]"); // VPMOVSXWQ_ZMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x24, 0xca], "vpmovsxwq xmm1{k5}{z}, xmm2"); // VPMOVSXWQ_XMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x24, 0x0a], "vpmovsxwq xmm1{k5}{z}, dword [rdx]"); // VPMOVSXWQ_XMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x24, 0xca], "vpmovsxwq xmm1, xmm2"); // VPMOVSXWQ_XMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x24, 0xca], "vpmovsxwq xmm1{k5}, xmm2"); // VPMOVSXWQ_XMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x24, 0x0a], "vpmovsxwq xmm1, dword [rdx]"); // VPMOVSXWQ_XMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x24, 0x0a], "vpmovsxwq xmm1{k5}, dword [rdx]"); // VPMOVSXWQ_XMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x25, 0xca], "vpmovsxdq ymm1{k5}{z}, xmm2"); // VPMOVSXDQ_YMMi64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x25, 0x0a], "vpmovsxdq ymm1{k5}{z}, xmmword [rdx]"); // VPMOVSXDQ_YMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x25, 0xca], "vpmovsxdq ymm1, xmm2"); // VPMOVSXDQ_YMMi64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x25, 0xca], "vpmovsxdq ymm1{k5}, xmm2"); // VPMOVSXDQ_YMMi64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x25, 0x0a], "vpmovsxdq ymm1, xmmword [rdx]"); // VPMOVSXDQ_YMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x25, 0x0a], "vpmovsxdq ymm1{k5}, xmmword [rdx]"); // VPMOVSXDQ_YMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x25, 0xca], "vpmovsxdq zmm1{k5}{z}, ymm2"); // VPMOVSXDQ_ZMMi64_MASKmskw_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x25, 0x0a], "vpmovsxdq zmm1{k5}{z}, ymmword [rdx]"); // VPMOVSXDQ_ZMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x25, 0xca], "vpmovsxdq zmm1, ymm2"); // VPMOVSXDQ_ZMMi64_MASKmskw_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x25, 0xca], "vpmovsxdq zmm1{k5}, ymm2"); // VPMOVSXDQ_ZMMi64_MASKmskw_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x25, 0x0a], "vpmovsxdq zmm1, ymmword [rdx]"); // VPMOVSXDQ_ZMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x25, 0x0a], "vpmovsxdq zmm1{k5}, ymmword [rdx]"); // VPMOVSXDQ_ZMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x25, 0xca], "vpmovsxdq xmm1{k5}{z}, xmm2"); // VPMOVSXDQ_XMMi64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x25, 0x0a], "vpmovsxdq xmm1{k5}{z}, qword [rdx]"); // VPMOVSXDQ_XMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x25, 0xca], "vpmovsxdq xmm1, xmm2"); // VPMOVSXDQ_XMMi64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x25, 0xca], "vpmovsxdq xmm1{k5}, xmm2"); // VPMOVSXDQ_XMMi64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x25, 0x0a], "vpmovsxdq xmm1, qword [rdx]"); // VPMOVSXDQ_XMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x25, 0x0a], "vpmovsxdq xmm1{k5}, qword [rdx]"); // VPMOVSXDQ_XMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x26, 0xca], "vptestmw k1, ymm0, ymm2"); // VPTESTMW_MASKmskw_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x26, 0xca], "vptestmw k1{k5}, ymm0, ymm2"); // VPTESTMW_MASKmskw_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x26, 0x0a], "vptestmw k1, ymm0, ymmword [rdx]"); // VPTESTMW_MASKmskw_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x26, 0x0a], "vptestmw k1{k5}, ymm0, ymmword [rdx]"); // VPTESTMW_MASKmskw_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x26, 0xca], "vptestmb k1, ymm0, ymm2"); // VPTESTMB_MASKmskw_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x26, 0xca], "vptestmb k1{k5}, ymm0, ymm2"); // VPTESTMB_MASKmskw_MASKmskw_YMMu8_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x26, 0x0a], "vptestmb k1, ymm0, ymmword [rdx]"); // VPTESTMB_MASKmskw_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x26, 0x0a], "vptestmb k1{k5}, ymm0, ymmword [rdx]"); // VPTESTMB_MASKmskw_MASKmskw_YMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x26, 0xca], "vptestmw k1, zmm0, zmm2"); // VPTESTMW_MASKmskw_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x26, 0xca], "vptestmw k1{k5}, zmm0, zmm2"); // VPTESTMW_MASKmskw_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x26, 0x0a], "vptestmw k1, zmm0, zmmword [rdx]"); // VPTESTMW_MASKmskw_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x26, 0x0a], "vptestmw k1{k5}, zmm0, zmmword [rdx]"); // VPTESTMW_MASKmskw_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x26, 0xca], "vptestmw k1, xmm0, xmm2"); // VPTESTMW_MASKmskw_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x26, 0xca], "vptestmw k1{k5}, xmm0, xmm2"); // VPTESTMW_MASKmskw_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x26, 0x0a], "vptestmw k1, xmm0, xmmword [rdx]"); // VPTESTMW_MASKmskw_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x26, 0x0a], "vptestmw k1{k5}, xmm0, xmmword [rdx]"); // VPTESTMW_MASKmskw_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x26, 0xca], "vptestmb k1, zmm0, zmm2"); // VPTESTMB_MASKmskw_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x26, 0xca], "vptestmb k1{k5}, zmm0, zmm2"); // VPTESTMB_MASKmskw_MASKmskw_ZMMu8_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x26, 0x0a], "vptestmb k1, zmm0, zmmword [rdx]"); // VPTESTMB_MASKmskw_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x26, 0x0a], "vptestmb k1{k5}, zmm0, zmmword [rdx]"); // VPTESTMB_MASKmskw_MASKmskw_ZMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x26, 0xca], "vptestmb k1, xmm0, xmm2"); // VPTESTMB_MASKmskw_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x26, 0xca], "vptestmb k1{k5}, xmm0, xmm2"); // VPTESTMB_MASKmskw_MASKmskw_XMMu8_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x26, 0x0a], "vptestmb k1, xmm0, xmmword [rdx]"); // VPTESTMB_MASKmskw_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x26, 0x0a], "vptestmb k1{k5}, xmm0, xmmword [rdx]"); // VPTESTMB_MASKmskw_MASKmskw_XMMu8_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x27, 0x0a], "vptestmq k1, ymm0, qword [rdx]{1to4}"); // VPTESTMQ_MASKmskw_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x27, 0x0a], "vptestmq k1{k5}, ymm0, qword [rdx]{1to4}"); // VPTESTMQ_MASKmskw_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x27, 0xca], "vptestmq k1, ymm0, ymm2"); // VPTESTMQ_MASKmskw_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x27, 0xca], "vptestmq k1{k5}, ymm0, ymm2"); // VPTESTMQ_MASKmskw_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x27, 0x0a], "vptestmq k1, ymm0, ymmword [rdx]"); // VPTESTMQ_MASKmskw_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x27, 0x0a], "vptestmq k1{k5}, ymm0, ymmword [rdx]"); // VPTESTMQ_MASKmskw_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x27, 0x0a], "vptestmd k1, ymm0, dword [rdx]{1to8}"); // VPTESTMD_MASKmskw_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x27, 0x0a], "vptestmd k1{k5}, ymm0, dword [rdx]{1to8}"); // VPTESTMD_MASKmskw_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x27, 0xca], "vptestmd k1, ymm0, ymm2"); // VPTESTMD_MASKmskw_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x27, 0xca], "vptestmd k1{k5}, ymm0, ymm2"); // VPTESTMD_MASKmskw_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x27, 0x0a], "vptestmd k1, ymm0, ymmword [rdx]"); // VPTESTMD_MASKmskw_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x27, 0x0a], "vptestmd k1{k5}, ymm0, ymmword [rdx]"); // VPTESTMD_MASKmskw_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x27, 0x0a], "vptestmq k1, zmm0, qword [rdx]{1to8}"); // VPTESTMQ_MASKmskw_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x27, 0x0a], "vptestmq k1{k5}, zmm0, qword [rdx]{1to8}"); // VPTESTMQ_MASKmskw_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x27, 0x0a], "vptestmq k1, xmm0, qword [rdx]{1to2}"); // VPTESTMQ_MASKmskw_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x27, 0x0a], "vptestmq k1{k5}, xmm0, qword [rdx]{1to2}"); // VPTESTMQ_MASKmskw_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x27, 0xca], "vptestmq k1, zmm0, zmm2"); // VPTESTMQ_MASKmskw_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x27, 0xca], "vptestmq k1{k5}, zmm0, zmm2"); // VPTESTMQ_MASKmskw_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x27, 0x0a], "vptestmq k1, zmm0, zmmword [rdx]"); // VPTESTMQ_MASKmskw_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x27, 0x0a], "vptestmq k1{k5}, zmm0, zmmword [rdx]"); // VPTESTMQ_MASKmskw_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x27, 0xca], "vptestmq k1, xmm0, xmm2"); // VPTESTMQ_MASKmskw_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x27, 0xca], "vptestmq k1{k5}, xmm0, xmm2"); // VPTESTMQ_MASKmskw_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x27, 0x0a], "vptestmq k1, xmm0, xmmword [rdx]"); // VPTESTMQ_MASKmskw_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x27, 0x0a], "vptestmq k1{k5}, xmm0, xmmword [rdx]"); // VPTESTMQ_MASKmskw_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x27, 0x0a], "vptestmd k1, zmm0, dword [rdx]{1to16}"); // VPTESTMD_MASKmskw_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x27, 0x0a], "vptestmd k1{k5}, zmm0, dword [rdx]{1to16}"); // VPTESTMD_MASKmskw_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x27, 0x0a], "vptestmd k1, xmm0, dword [rdx]{1to4}"); // VPTESTMD_MASKmskw_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x27, 0x0a], "vptestmd k1{k5}, xmm0, dword [rdx]{1to4}"); // VPTESTMD_MASKmskw_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x27, 0xca], "vptestmd k1, zmm0, zmm2"); // VPTESTMD_MASKmskw_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x27, 0xca], "vptestmd k1{k5}, zmm0, zmm2"); // VPTESTMD_MASKmskw_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x27, 0x0a], "vptestmd k1, zmm0, zmmword [rdx]"); // VPTESTMD_MASKmskw_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x27, 0x0a], "vptestmd k1{k5}, zmm0, zmmword [rdx]"); // VPTESTMD_MASKmskw_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x27, 0xca], "vptestmd k1, xmm0, xmm2"); // VPTESTMD_MASKmskw_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x27, 0xca], "vptestmd k1{k5}, xmm0, xmm2"); // VPTESTMD_MASKmskw_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x27, 0x0a], "vptestmd k1, xmm0, xmmword [rdx]"); // VPTESTMD_MASKmskw_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x27, 0x0a], "vptestmd k1{k5}, xmm0, xmmword [rdx]"); // VPTESTMD_MASKmskw_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x28, 0x0a], "vpmuldq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPMULDQ_YMMi64_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x28, 0x0a], "vpmuldq ymm1, ymm0, qword [rdx]{1to4}"); // VPMULDQ_YMMi64_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x28, 0x0a], "vpmuldq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPMULDQ_YMMi64_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x28, 0xca], "vpmuldq ymm1{k5}{z}, ymm0, ymm2"); // VPMULDQ_YMMi64_MASKmskw_YMMi32_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x28, 0x0a], "vpmuldq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMULDQ_YMMi64_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x28, 0xca], "vpmuldq ymm1, ymm0, ymm2"); // VPMULDQ_YMMi64_MASKmskw_YMMi32_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x28, 0xca], "vpmuldq ymm1{k5}, ymm0, ymm2"); // VPMULDQ_YMMi64_MASKmskw_YMMi32_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x28, 0x0a], "vpmuldq ymm1, ymm0, ymmword [rdx]"); // VPMULDQ_YMMi64_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x28, 0x0a], "vpmuldq ymm1{k5}, ymm0, ymmword [rdx]"); // VPMULDQ_YMMi64_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x28, 0x0a], "vpmuldq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPMULDQ_ZMMi64_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x28, 0x0a], "vpmuldq zmm1, zmm0, qword [rdx]{1to8}"); // VPMULDQ_ZMMi64_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x28, 0x0a], "vpmuldq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPMULDQ_ZMMi64_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x28, 0x0a], "vpmuldq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPMULDQ_XMMi64_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x28, 0x0a], "vpmuldq xmm1, xmm0, qword [rdx]{1to2}"); // VPMULDQ_XMMi64_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x28, 0x0a], "vpmuldq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPMULDQ_XMMi64_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x28, 0xca], "vpmuldq zmm1{k5}{z}, zmm0, zmm2"); // VPMULDQ_ZMMi64_MASKmskw_ZMMi32_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x28, 0x0a], "vpmuldq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMULDQ_ZMMi64_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x28, 0xca], "vpmuldq zmm1, zmm0, zmm2"); // VPMULDQ_ZMMi64_MASKmskw_ZMMi32_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x28, 0xca], "vpmuldq zmm1{k5}, zmm0, zmm2"); // VPMULDQ_ZMMi64_MASKmskw_ZMMi32_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x28, 0x0a], "vpmuldq zmm1, zmm0, zmmword [rdx]"); // VPMULDQ_ZMMi64_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x28, 0x0a], "vpmuldq zmm1{k5}, zmm0, zmmword [rdx]"); // VPMULDQ_ZMMi64_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x28, 0xca], "vpmuldq xmm1{k5}{z}, xmm0, xmm2"); // VPMULDQ_XMMi64_MASKmskw_XMMi32_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x28, 0x0a], "vpmuldq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMULDQ_XMMi64_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x28, 0xca], "vpmuldq xmm1, xmm0, xmm2"); // VPMULDQ_XMMi64_MASKmskw_XMMi32_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x28, 0xca], "vpmuldq xmm1{k5}, xmm0, xmm2"); // VPMULDQ_XMMi64_MASKmskw_XMMi32_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x28, 0x0a], "vpmuldq xmm1, xmm0, xmmword [rdx]"); // VPMULDQ_XMMi64_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x28, 0x0a], "vpmuldq xmm1{k5}, xmm0, xmmword [rdx]"); // VPMULDQ_XMMi64_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x29, 0x0a], "vpcmpeqq k1, ymm0, qword [rdx]{1to4}"); // VPCMPEQQ_MASKmskw_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x29, 0x0a], "vpcmpeqq k1{k5}, ymm0, qword [rdx]{1to4}"); // VPCMPEQQ_MASKmskw_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x29, 0xca], "vpcmpeqq k1, ymm0, ymm2"); // VPCMPEQQ_MASKmskw_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x29, 0xca], "vpcmpeqq k1{k5}, ymm0, ymm2"); // VPCMPEQQ_MASKmskw_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x29, 0x0a], "vpcmpeqq k1, ymm0, ymmword [rdx]"); // VPCMPEQQ_MASKmskw_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x29, 0x0a], "vpcmpeqq k1{k5}, ymm0, ymmword [rdx]"); // VPCMPEQQ_MASKmskw_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x29, 0x0a], "vpcmpeqq k1, zmm0, qword [rdx]{1to8}"); // VPCMPEQQ_MASKmskw_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x29, 0x0a], "vpcmpeqq k1{k5}, zmm0, qword [rdx]{1to8}"); // VPCMPEQQ_MASKmskw_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x29, 0x0a], "vpcmpeqq k1, xmm0, qword [rdx]{1to2}"); // VPCMPEQQ_MASKmskw_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x29, 0x0a], "vpcmpeqq k1{k5}, xmm0, qword [rdx]{1to2}"); // VPCMPEQQ_MASKmskw_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x29, 0xca], "vpcmpeqq k1, zmm0, zmm2"); // VPCMPEQQ_MASKmskw_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x29, 0xca], "vpcmpeqq k1{k5}, zmm0, zmm2"); // VPCMPEQQ_MASKmskw_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x29, 0x0a], "vpcmpeqq k1, zmm0, zmmword [rdx]"); // VPCMPEQQ_MASKmskw_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x29, 0x0a], "vpcmpeqq k1{k5}, zmm0, zmmword [rdx]"); // VPCMPEQQ_MASKmskw_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x29, 0xca], "vpcmpeqq k1, xmm0, xmm2"); // VPCMPEQQ_MASKmskw_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x29, 0xca], "vpcmpeqq k1{k5}, xmm0, xmm2"); // VPCMPEQQ_MASKmskw_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x29, 0x0a], "vpcmpeqq k1, xmm0, xmmword [rdx]"); // VPCMPEQQ_MASKmskw_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x29, 0x0a], "vpcmpeqq k1{k5}, xmm0, xmmword [rdx]"); // VPCMPEQQ_MASKmskw_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2a, 0x0a], "vmovntdqa ymm1, ymmword [rdx]"); // VMOVNTDQA_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x2a, 0x0a], "vmovntdqa zmm1, zmmword [rdx]"); // VMOVNTDQA_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x2a, 0x0a], "vmovntdqa xmm1, xmmword [rdx]"); // VMOVNTDQA_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x2b, 0x0a], "vpackusdw ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPACKUSDW_YMMu16_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x2b, 0x0a], "vpackusdw ymm1, ymm0, dword [rdx]{1to8}"); // VPACKUSDW_YMMu16_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x2b, 0x0a], "vpackusdw ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPACKUSDW_YMMu16_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x2b, 0xca], "vpackusdw ymm1{k5}{z}, ymm0, ymm2"); // VPACKUSDW_YMMu16_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x2b, 0x0a], "vpackusdw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPACKUSDW_YMMu16_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2b, 0xca], "vpackusdw ymm1, ymm0, ymm2"); // VPACKUSDW_YMMu16_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x2b, 0xca], "vpackusdw ymm1{k5}, ymm0, ymm2"); // VPACKUSDW_YMMu16_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2b, 0x0a], "vpackusdw ymm1, ymm0, ymmword [rdx]"); // VPACKUSDW_YMMu16_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x2b, 0x0a], "vpackusdw ymm1{k5}, ymm0, ymmword [rdx]"); // VPACKUSDW_YMMu16_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x2b, 0x0a], "vpackusdw zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPACKUSDW_ZMMu16_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x2b, 0x0a], "vpackusdw zmm1, zmm0, dword [rdx]{1to16}"); // VPACKUSDW_ZMMu16_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x2b, 0x0a], "vpackusdw zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPACKUSDW_ZMMu16_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x2b, 0x0a], "vpackusdw xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPACKUSDW_XMMu16_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x2b, 0x0a], "vpackusdw xmm1, xmm0, dword [rdx]{1to4}"); // VPACKUSDW_XMMu16_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x2b, 0x0a], "vpackusdw xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPACKUSDW_XMMu16_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x2b, 0xca], "vpackusdw zmm1{k5}{z}, zmm0, zmm2"); // VPACKUSDW_ZMMu16_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x2b, 0x0a], "vpackusdw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPACKUSDW_ZMMu16_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x2b, 0xca], "vpackusdw zmm1, zmm0, zmm2"); // VPACKUSDW_ZMMu16_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x2b, 0xca], "vpackusdw zmm1{k5}, zmm0, zmm2"); // VPACKUSDW_ZMMu16_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x2b, 0x0a], "vpackusdw zmm1, zmm0, zmmword [rdx]"); // VPACKUSDW_ZMMu16_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x2b, 0x0a], "vpackusdw zmm1{k5}, zmm0, zmmword [rdx]"); // VPACKUSDW_ZMMu16_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x2b, 0xca], "vpackusdw xmm1{k5}{z}, xmm0, xmm2"); // VPACKUSDW_XMMu16_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x2b, 0x0a], "vpackusdw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPACKUSDW_XMMu16_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x2b, 0xca], "vpackusdw xmm1, xmm0, xmm2"); // VPACKUSDW_XMMu16_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x2b, 0xca], "vpackusdw xmm1{k5}, xmm0, xmm2"); // VPACKUSDW_XMMu16_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x2b, 0x0a], "vpackusdw xmm1, xmm0, xmmword [rdx]"); // VPACKUSDW_XMMu16_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x2b, 0x0a], "vpackusdw xmm1{k5}, xmm0, xmmword [rdx]"); // VPACKUSDW_XMMu16_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xfd, 0x2c, 0xca], "vscalefpd zmm1{k5}{z}{rz-sae}, zmm0, zmm2"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x78, 0x2c, 0xca], "vscalefpd zmm1{rz-sae}, zmm0, zmm2"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x7d, 0x2c, 0xca], "vscalefpd zmm1{k5}{rz-sae}, zmm0, zmm2"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x2c, 0xca], "vscalefpd zmm1{k5}{z}{rd-sae}, zmm0, zmm2"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x2c, 0x0a], "vscalefpd ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VSCALEFPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x2c, 0xca], "vscalefpd zmm1{rd-sae}, zmm0, zmm2"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x2c, 0xca], "vscalefpd zmm1{k5}{rd-sae}, zmm0, zmm2"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x2c, 0x0a], "vscalefpd ymm1, ymm0, qword [rdx]{1to4}"); // VSCALEFPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x2c, 0x0a], "vscalefpd ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VSCALEFPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x2c, 0xca], "vscalefpd ymm1{k5}{z}, ymm0, ymm2"); // VSCALEFPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x2c, 0x0a], "vscalefpd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VSCALEFPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x2c, 0xca], "vscalefpd ymm1, ymm0, ymm2"); // VSCALEFPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x2c, 0xca], "vscalefpd ymm1{k5}, ymm0, ymm2"); // VSCALEFPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x2c, 0x0a], "vscalefpd ymm1, ymm0, ymmword [rdx]"); // VSCALEFPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x2c, 0x0a], "vscalefpd ymm1{k5}, ymm0, ymmword [rdx]"); // VSCALEFPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xfd, 0x2c, 0xca], "vscalefps zmm1{k5}{z}{rz-sae}, zmm0, zmm2"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x78, 0x2c, 0xca], "vscalefps zmm1{rz-sae}, zmm0, zmm2"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x7d, 0x2c, 0xca], "vscalefps zmm1{k5}{rz-sae}, zmm0, zmm2"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x2c, 0xca], "vscalefps zmm1{k5}{z}{rd-sae}, zmm0, zmm2"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x2c, 0x0a], "vscalefps ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VSCALEFPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x2c, 0xca], "vscalefps zmm1{rd-sae}, zmm0, zmm2"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x2c, 0xca], "vscalefps zmm1{k5}{rd-sae}, zmm0, zmm2"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x2c, 0x0a], "vscalefps ymm1, ymm0, dword [rdx]{1to8}"); // VSCALEFPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x2c, 0x0a], "vscalefps ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VSCALEFPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x2c, 0xca], "vscalefps ymm1{k5}{z}, ymm0, ymm2"); // VSCALEFPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x2c, 0x0a], "vscalefps ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VSCALEFPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2c, 0xca], "vscalefps ymm1, ymm0, ymm2"); // VSCALEFPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x2c, 0xca], "vscalefps ymm1{k5}, ymm0, ymm2"); // VSCALEFPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2c, 0x0a], "vscalefps ymm1, ymm0, ymmword [rdx]"); // VSCALEFPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x2c, 0x0a], "vscalefps ymm1{k5}, ymm0, ymmword [rdx]"); // VSCALEFPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x2c, 0xca], "vscalefpd zmm1{k5}{z}{ru-sae}, zmm0, zmm2"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x2c, 0x0a], "vscalefpd zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x2c, 0xca], "vscalefpd zmm1{ru-sae}, zmm0, zmm2"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x2c, 0xca], "vscalefpd zmm1{k5}{ru-sae}, zmm0, zmm2"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x2c, 0x0a], "vscalefpd zmm1, zmm0, qword [rdx]{1to8}"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x2c, 0x0a], "vscalefpd zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x2c, 0xca], "vscalefpd zmm1{k5}{z}{rne-sae}, zmm0, zmm2"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x2c, 0x0a], "vscalefpd xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VSCALEFPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x2c, 0xca], "vscalefpd zmm1{rne-sae}, zmm0, zmm2"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x2c, 0xca], "vscalefpd zmm1{k5}{rne-sae}, zmm0, zmm2"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x2c, 0x0a], "vscalefpd xmm1, xmm0, qword [rdx]{1to2}"); // VSCALEFPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x2c, 0x0a], "vscalefpd xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VSCALEFPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x2c, 0xca], "vscalefpd zmm1{k5}{z}, zmm0, zmm2"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x2c, 0x0a], "vscalefpd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x2c, 0xca], "vscalefpd zmm1, zmm0, zmm2"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x2c, 0xca], "vscalefpd zmm1{k5}, zmm0, zmm2"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x2c, 0x0a], "vscalefpd zmm1, zmm0, zmmword [rdx]"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x2c, 0x0a], "vscalefpd zmm1{k5}, zmm0, zmmword [rdx]"); // VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x2c, 0xca], "vscalefpd xmm1{k5}{z}, xmm0, xmm2"); // VSCALEFPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x2c, 0x0a], "vscalefpd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VSCALEFPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x2c, 0xca], "vscalefpd xmm1, xmm0, xmm2"); // VSCALEFPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x2c, 0xca], "vscalefpd xmm1{k5}, xmm0, xmm2"); // VSCALEFPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x2c, 0x0a], "vscalefpd xmm1, xmm0, xmmword [rdx]"); // VSCALEFPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x2c, 0x0a], "vscalefpd xmm1{k5}, xmm0, xmmword [rdx]"); // VSCALEFPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x2c, 0xca], "vscalefps zmm1{k5}{z}{ru-sae}, zmm0, zmm2"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x2c, 0x0a], "vscalefps zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x2c, 0xca], "vscalefps zmm1{ru-sae}, zmm0, zmm2"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x2c, 0xca], "vscalefps zmm1{k5}{ru-sae}, zmm0, zmm2"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x2c, 0x0a], "vscalefps zmm1, zmm0, dword [rdx]{1to16}"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x2c, 0x0a], "vscalefps zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x2c, 0xca], "vscalefps zmm1{k5}{z}{rne-sae}, zmm0, zmm2"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x2c, 0x0a], "vscalefps xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VSCALEFPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x2c, 0xca], "vscalefps zmm1{rne-sae}, zmm0, zmm2"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x2c, 0xca], "vscalefps zmm1{k5}{rne-sae}, zmm0, zmm2"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x2c, 0x0a], "vscalefps xmm1, xmm0, dword [rdx]{1to4}"); // VSCALEFPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x2c, 0x0a], "vscalefps xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VSCALEFPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x2c, 0xca], "vscalefps zmm1{k5}{z}, zmm0, zmm2"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x2c, 0x0a], "vscalefps zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x2c, 0xca], "vscalefps zmm1, zmm0, zmm2"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x2c, 0xca], "vscalefps zmm1{k5}, zmm0, zmm2"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x2c, 0x0a], "vscalefps zmm1, zmm0, zmmword [rdx]"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x2c, 0x0a], "vscalefps zmm1{k5}, zmm0, zmmword [rdx]"); // VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x2c, 0xca], "vscalefps xmm1{k5}{z}, xmm0, xmm2"); // VSCALEFPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x2c, 0x0a], "vscalefps xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VSCALEFPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x2c, 0xca], "vscalefps xmm1, xmm0, xmm2"); // VSCALEFPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x2c, 0xca], "vscalefps xmm1{k5}, xmm0, xmm2"); // VSCALEFPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x2c, 0x0a], "vscalefps xmm1, xmm0, xmmword [rdx]"); // VSCALEFPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x2c, 0x0a], "vscalefps xmm1{k5}, xmm0, xmmword [rdx]"); // VSCALEFPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xfd, 0x2d, 0xca], "vscalefsd xmm1{k5}{z}{rz-sae}, xmm0, xmm2"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x78, 0x2d, 0xca], "vscalefsd xmm1{rz-sae}, xmm0, xmm2"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x7d, 0x2d, 0xca], "vscalefsd xmm1{k5}{rz-sae}, xmm0, xmm2"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x2d, 0xca], "vscalefsd xmm1{k5}{z}{rd-sae}, xmm0, xmm2"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x2d, 0xca], "vscalefsd xmm1{rd-sae}, xmm0, xmm2"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x2d, 0xca], "vscalefsd xmm1{k5}{rd-sae}, xmm0, xmm2"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x2d, 0xca], "vscalefsd xmm1{k5}{z}, xmm0, xmm2"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x2d, 0x0a], "vscalefsd xmm1{k5}{z}, xmm0, qword [rdx]"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x2d, 0xca], "vscalefsd xmm1, xmm0, xmm2"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x2d, 0xca], "vscalefsd xmm1{k5}, xmm0, xmm2"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x2d, 0x0a], "vscalefsd xmm1, xmm0, qword [rdx]"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x2d, 0x0a], "vscalefsd xmm1{k5}, xmm0, qword [rdx]"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xfd, 0x2d, 0xca], "vscalefss xmm1{k5}{z}{rz-sae}, xmm0, xmm2"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x78, 0x2d, 0xca], "vscalefss xmm1{rz-sae}, xmm0, xmm2"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x7d, 0x2d, 0xca], "vscalefss xmm1{k5}{rz-sae}, xmm0, xmm2"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x2d, 0xca], "vscalefss xmm1{k5}{z}{rd-sae}, xmm0, xmm2"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x2d, 0xca], "vscalefss xmm1{rd-sae}, xmm0, xmm2"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x2d, 0xca], "vscalefss xmm1{k5}{rd-sae}, xmm0, xmm2"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x2d, 0xca], "vscalefss xmm1{k5}{z}, xmm0, xmm2"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x2d, 0x0a], "vscalefss xmm1{k5}{z}, xmm0, dword [rdx]"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2d, 0xca], "vscalefss xmm1, xmm0, xmm2"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x2d, 0xca], "vscalefss xmm1{k5}, xmm0, xmm2"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x2d, 0x0a], "vscalefss xmm1, xmm0, dword [rdx]"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x2d, 0x0a], "vscalefss xmm1{k5}, xmm0, dword [rdx]"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x2d, 0xca], "vscalefsd xmm1{k5}{z}{ru-sae}, xmm0, xmm2"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x2d, 0xca], "vscalefsd xmm1{ru-sae}, xmm0, xmm2"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x2d, 0xca], "vscalefsd xmm1{k5}{ru-sae}, xmm0, xmm2"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x2d, 0xca], "vscalefsd xmm1{k5}{z}{rne-sae}, xmm0, xmm2"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x2d, 0xca], "vscalefsd xmm1{rne-sae}, xmm0, xmm2"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x2d, 0xca], "vscalefsd xmm1{k5}{rne-sae}, xmm0, xmm2"); // VSCALEFSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x2d, 0xca], "vscalefss xmm1{k5}{z}{ru-sae}, xmm0, xmm2"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x2d, 0xca], "vscalefss xmm1{ru-sae}, xmm0, xmm2"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x2d, 0xca], "vscalefss xmm1{k5}{ru-sae}, xmm0, xmm2"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x2d, 0xca], "vscalefss xmm1{k5}{z}{rne-sae}, xmm0, xmm2"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x2d, 0xca], "vscalefss xmm1{rne-sae}, xmm0, xmm2"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x2d, 0xca], "vscalefss xmm1{k5}{rne-sae}, xmm0, xmm2"); // VSCALEFSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x30, 0xca], "vpmovzxbw ymm1{k5}{z}, xmm2"); // VPMOVZXBW_YMMi16_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x30, 0x0a], "vpmovzxbw ymm1{k5}{z}, xmmword [rdx]"); // VPMOVZXBW_YMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x30, 0xca], "vpmovzxbw ymm1, xmm2"); // VPMOVZXBW_YMMi16_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x30, 0xca], "vpmovzxbw ymm1{k5}, xmm2"); // VPMOVZXBW_YMMi16_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x30, 0x0a], "vpmovzxbw ymm1, xmmword [rdx]"); // VPMOVZXBW_YMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x30, 0x0a], "vpmovzxbw ymm1{k5}, xmmword [rdx]"); // VPMOVZXBW_YMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x30, 0xca], "vpmovzxbw zmm1{k5}{z}, ymm2"); // VPMOVZXBW_ZMMi16_MASKmskw_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x30, 0x0a], "vpmovzxbw zmm1{k5}{z}, ymmword [rdx]"); // VPMOVZXBW_ZMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x30, 0xca], "vpmovzxbw zmm1, ymm2"); // VPMOVZXBW_ZMMi16_MASKmskw_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x30, 0xca], "vpmovzxbw zmm1{k5}, ymm2"); // VPMOVZXBW_ZMMi16_MASKmskw_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x30, 0x0a], "vpmovzxbw zmm1, ymmword [rdx]"); // VPMOVZXBW_ZMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x30, 0x0a], "vpmovzxbw zmm1{k5}, ymmword [rdx]"); // VPMOVZXBW_ZMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x30, 0xca], "vpmovzxbw xmm1{k5}{z}, xmm2"); // VPMOVZXBW_XMMi16_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x30, 0x0a], "vpmovzxbw xmm1{k5}{z}, qword [rdx]"); // VPMOVZXBW_XMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x30, 0xca], "vpmovzxbw xmm1, xmm2"); // VPMOVZXBW_XMMi16_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x30, 0xca], "vpmovzxbw xmm1{k5}, xmm2"); // VPMOVZXBW_XMMi16_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x30, 0x0a], "vpmovzxbw xmm1, qword [rdx]"); // VPMOVZXBW_XMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x30, 0x0a], "vpmovzxbw xmm1{k5}, qword [rdx]"); // VPMOVZXBW_XMMi16_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x31, 0xca], "vpmovzxbd ymm1{k5}{z}, xmm2"); // VPMOVZXBD_YMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x31, 0x0a], "vpmovzxbd ymm1{k5}{z}, qword [rdx]"); // VPMOVZXBD_YMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x31, 0xca], "vpmovzxbd ymm1, xmm2"); // VPMOVZXBD_YMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x31, 0xca], "vpmovzxbd ymm1{k5}, xmm2"); // VPMOVZXBD_YMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x31, 0x0a], "vpmovzxbd ymm1, qword [rdx]"); // VPMOVZXBD_YMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x31, 0x0a], "vpmovzxbd ymm1{k5}, qword [rdx]"); // VPMOVZXBD_YMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x31, 0xca], "vpmovzxbd zmm1{k5}{z}, xmm2"); // VPMOVZXBD_ZMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x31, 0x0a], "vpmovzxbd zmm1{k5}{z}, xmmword [rdx]"); // VPMOVZXBD_ZMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x31, 0xca], "vpmovzxbd zmm1, xmm2"); // VPMOVZXBD_ZMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x31, 0xca], "vpmovzxbd zmm1{k5}, xmm2"); // VPMOVZXBD_ZMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x31, 0x0a], "vpmovzxbd zmm1, xmmword [rdx]"); // VPMOVZXBD_ZMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x31, 0x0a], "vpmovzxbd zmm1{k5}, xmmword [rdx]"); // VPMOVZXBD_ZMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x31, 0xca], "vpmovzxbd xmm1{k5}{z}, xmm2"); // VPMOVZXBD_XMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x31, 0x0a], "vpmovzxbd xmm1{k5}{z}, dword [rdx]"); // VPMOVZXBD_XMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x31, 0xca], "vpmovzxbd xmm1, xmm2"); // VPMOVZXBD_XMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x31, 0xca], "vpmovzxbd xmm1{k5}, xmm2"); // VPMOVZXBD_XMMi32_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x31, 0x0a], "vpmovzxbd xmm1, dword [rdx]"); // VPMOVZXBD_XMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x31, 0x0a], "vpmovzxbd xmm1{k5}, dword [rdx]"); // VPMOVZXBD_XMMi32_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x32, 0xca], "vpmovzxbq ymm1{k5}{z}, xmm2"); // VPMOVZXBQ_YMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x32, 0x0a], "vpmovzxbq ymm1{k5}{z}, dword [rdx]"); // VPMOVZXBQ_YMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x32, 0xca], "vpmovzxbq ymm1, xmm2"); // VPMOVZXBQ_YMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x32, 0xca], "vpmovzxbq ymm1{k5}, xmm2"); // VPMOVZXBQ_YMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x32, 0x0a], "vpmovzxbq ymm1, dword [rdx]"); // VPMOVZXBQ_YMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x32, 0x0a], "vpmovzxbq ymm1{k5}, dword [rdx]"); // VPMOVZXBQ_YMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x32, 0xca], "vpmovzxbq zmm1{k5}{z}, xmm2"); // VPMOVZXBQ_ZMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x32, 0x0a], "vpmovzxbq zmm1{k5}{z}, qword [rdx]"); // VPMOVZXBQ_ZMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x32, 0xca], "vpmovzxbq zmm1, xmm2"); // VPMOVZXBQ_ZMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x32, 0xca], "vpmovzxbq zmm1{k5}, xmm2"); // VPMOVZXBQ_ZMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x32, 0x0a], "vpmovzxbq zmm1, qword [rdx]"); // VPMOVZXBQ_ZMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x32, 0x0a], "vpmovzxbq zmm1{k5}, qword [rdx]"); // VPMOVZXBQ_ZMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x32, 0xca], "vpmovzxbq xmm1{k5}{z}, xmm2"); // VPMOVZXBQ_XMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x32, 0x0a], "vpmovzxbq xmm1{k5}{z}, word [rdx]"); // VPMOVZXBQ_XMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x32, 0xca], "vpmovzxbq xmm1, xmm2"); // VPMOVZXBQ_XMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x32, 0xca], "vpmovzxbq xmm1{k5}, xmm2"); // VPMOVZXBQ_XMMi64_MASKmskw_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x32, 0x0a], "vpmovzxbq xmm1, word [rdx]"); // VPMOVZXBQ_XMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x32, 0x0a], "vpmovzxbq xmm1{k5}, word [rdx]"); // VPMOVZXBQ_XMMi64_MASKmskw_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x33, 0xca], "vpmovzxwd ymm1{k5}{z}, xmm2"); // VPMOVZXWD_YMMi32_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x33, 0x0a], "vpmovzxwd ymm1{k5}{z}, xmmword [rdx]"); // VPMOVZXWD_YMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x33, 0xca], "vpmovzxwd ymm1, xmm2"); // VPMOVZXWD_YMMi32_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x33, 0xca], "vpmovzxwd ymm1{k5}, xmm2"); // VPMOVZXWD_YMMi32_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x33, 0x0a], "vpmovzxwd ymm1, xmmword [rdx]"); // VPMOVZXWD_YMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x33, 0x0a], "vpmovzxwd ymm1{k5}, xmmword [rdx]"); // VPMOVZXWD_YMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x33, 0xca], "vpmovzxwd zmm1{k5}{z}, ymm2"); // VPMOVZXWD_ZMMi32_MASKmskw_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x33, 0x0a], "vpmovzxwd zmm1{k5}{z}, ymmword [rdx]"); // VPMOVZXWD_ZMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x33, 0xca], "vpmovzxwd zmm1, ymm2"); // VPMOVZXWD_ZMMi32_MASKmskw_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x33, 0xca], "vpmovzxwd zmm1{k5}, ymm2"); // VPMOVZXWD_ZMMi32_MASKmskw_YMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x33, 0x0a], "vpmovzxwd zmm1, ymmword [rdx]"); // VPMOVZXWD_ZMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x33, 0x0a], "vpmovzxwd zmm1{k5}, ymmword [rdx]"); // VPMOVZXWD_ZMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x33, 0xca], "vpmovzxwd xmm1{k5}{z}, xmm2"); // VPMOVZXWD_XMMi32_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x33, 0x0a], "vpmovzxwd xmm1{k5}{z}, qword [rdx]"); // VPMOVZXWD_XMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x33, 0xca], "vpmovzxwd xmm1, xmm2"); // VPMOVZXWD_XMMi32_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x33, 0xca], "vpmovzxwd xmm1{k5}, xmm2"); // VPMOVZXWD_XMMi32_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x33, 0x0a], "vpmovzxwd xmm1, qword [rdx]"); // VPMOVZXWD_XMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x33, 0x0a], "vpmovzxwd xmm1{k5}, qword [rdx]"); // VPMOVZXWD_XMMi32_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x34, 0xca], "vpmovzxwq ymm1{k5}{z}, xmm2"); // VPMOVZXWQ_YMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x34, 0x0a], "vpmovzxwq ymm1{k5}{z}, qword [rdx]"); // VPMOVZXWQ_YMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x34, 0xca], "vpmovzxwq ymm1, xmm2"); // VPMOVZXWQ_YMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x34, 0xca], "vpmovzxwq ymm1{k5}, xmm2"); // VPMOVZXWQ_YMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x34, 0x0a], "vpmovzxwq ymm1, qword [rdx]"); // VPMOVZXWQ_YMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x34, 0x0a], "vpmovzxwq ymm1{k5}, qword [rdx]"); // VPMOVZXWQ_YMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x34, 0xca], "vpmovzxwq zmm1{k5}{z}, xmm2"); // VPMOVZXWQ_ZMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x34, 0x0a], "vpmovzxwq zmm1{k5}{z}, xmmword [rdx]"); // VPMOVZXWQ_ZMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x34, 0xca], "vpmovzxwq zmm1, xmm2"); // VPMOVZXWQ_ZMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x34, 0xca], "vpmovzxwq zmm1{k5}, xmm2"); // VPMOVZXWQ_ZMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x34, 0x0a], "vpmovzxwq zmm1, xmmword [rdx]"); // VPMOVZXWQ_ZMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x34, 0x0a], "vpmovzxwq zmm1{k5}, xmmword [rdx]"); // VPMOVZXWQ_ZMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x34, 0xca], "vpmovzxwq xmm1{k5}{z}, xmm2"); // VPMOVZXWQ_XMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x34, 0x0a], "vpmovzxwq xmm1{k5}{z}, dword [rdx]"); // VPMOVZXWQ_XMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x34, 0xca], "vpmovzxwq xmm1, xmm2"); // VPMOVZXWQ_XMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x34, 0xca], "vpmovzxwq xmm1{k5}, xmm2"); // VPMOVZXWQ_XMMi64_MASKmskw_XMMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x34, 0x0a], "vpmovzxwq xmm1, dword [rdx]"); // VPMOVZXWQ_XMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x34, 0x0a], "vpmovzxwq xmm1{k5}, dword [rdx]"); // VPMOVZXWQ_XMMi64_MASKmskw_MEMi16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x35, 0xca], "vpmovzxdq ymm1{k5}{z}, xmm2"); // VPMOVZXDQ_YMMi64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x35, 0x0a], "vpmovzxdq ymm1{k5}{z}, xmmword [rdx]"); // VPMOVZXDQ_YMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x35, 0xca], "vpmovzxdq ymm1, xmm2"); // VPMOVZXDQ_YMMi64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x35, 0xca], "vpmovzxdq ymm1{k5}, xmm2"); // VPMOVZXDQ_YMMi64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x35, 0x0a], "vpmovzxdq ymm1, xmmword [rdx]"); // VPMOVZXDQ_YMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x35, 0x0a], "vpmovzxdq ymm1{k5}, xmmword [rdx]"); // VPMOVZXDQ_YMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x35, 0xca], "vpmovzxdq zmm1{k5}{z}, ymm2"); // VPMOVZXDQ_ZMMi64_MASKmskw_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x35, 0x0a], "vpmovzxdq zmm1{k5}{z}, ymmword [rdx]"); // VPMOVZXDQ_ZMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x35, 0xca], "vpmovzxdq zmm1, ymm2"); // VPMOVZXDQ_ZMMi64_MASKmskw_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x35, 0xca], "vpmovzxdq zmm1{k5}, ymm2"); // VPMOVZXDQ_ZMMi64_MASKmskw_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x35, 0x0a], "vpmovzxdq zmm1, ymmword [rdx]"); // VPMOVZXDQ_ZMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x35, 0x0a], "vpmovzxdq zmm1{k5}, ymmword [rdx]"); // VPMOVZXDQ_ZMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x35, 0xca], "vpmovzxdq xmm1{k5}{z}, xmm2"); // VPMOVZXDQ_XMMi64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x35, 0x0a], "vpmovzxdq xmm1{k5}{z}, qword [rdx]"); // VPMOVZXDQ_XMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x35, 0xca], "vpmovzxdq xmm1, xmm2"); // VPMOVZXDQ_XMMi64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x35, 0xca], "vpmovzxdq xmm1{k5}, xmm2"); // VPMOVZXDQ_XMMi64_MASKmskw_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x35, 0x0a], "vpmovzxdq xmm1, qword [rdx]"); // VPMOVZXDQ_XMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x35, 0x0a], "vpmovzxdq xmm1{k5}, qword [rdx]"); // VPMOVZXDQ_XMMi64_MASKmskw_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x36, 0x0a], "vpermq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPERMQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x36, 0x0a], "vpermq ymm1, ymm0, qword [rdx]{1to4}"); // VPERMQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x36, 0x0a], "vpermq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPERMQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x36, 0xca], "vpermq ymm1{k5}{z}, ymm0, ymm2"); // VPERMQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x36, 0x0a], "vpermq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPERMQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x36, 0xca], "vpermq ymm1, ymm0, ymm2"); // VPERMQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x36, 0xca], "vpermq ymm1{k5}, ymm0, ymm2"); // VPERMQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x36, 0x0a], "vpermq ymm1, ymm0, ymmword [rdx]"); // VPERMQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x36, 0x0a], "vpermq ymm1{k5}, ymm0, ymmword [rdx]"); // VPERMQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x36, 0x0a], "vpermd ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPERMD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x36, 0x0a], "vpermd ymm1, ymm0, dword [rdx]{1to8}"); // VPERMD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x36, 0x0a], "vpermd ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPERMD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x36, 0xca], "vpermd ymm1{k5}{z}, ymm0, ymm2"); // VPERMD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x36, 0x0a], "vpermd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPERMD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x36, 0xca], "vpermd ymm1, ymm0, ymm2"); // VPERMD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x36, 0xca], "vpermd ymm1{k5}, ymm0, ymm2"); // VPERMD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x36, 0x0a], "vpermd ymm1, ymm0, ymmword [rdx]"); // VPERMD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x36, 0x0a], "vpermd ymm1{k5}, ymm0, ymmword [rdx]"); // VPERMD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x36, 0x0a], "vpermq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPERMQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x36, 0x0a], "vpermq zmm1, zmm0, qword [rdx]{1to8}"); // VPERMQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x36, 0x0a], "vpermq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPERMQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x36, 0xca], "vpermq zmm1{k5}{z}, zmm0, zmm2"); // VPERMQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x36, 0x0a], "vpermq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPERMQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x36, 0xca], "vpermq zmm1, zmm0, zmm2"); // VPERMQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x36, 0xca], "vpermq zmm1{k5}, zmm0, zmm2"); // VPERMQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x36, 0x0a], "vpermq zmm1, zmm0, zmmword [rdx]"); // VPERMQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x36, 0x0a], "vpermq zmm1{k5}, zmm0, zmmword [rdx]"); // VPERMQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x36, 0x0a], "vpermd zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPERMD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x36, 0x0a], "vpermd zmm1, zmm0, dword [rdx]{1to16}"); // VPERMD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x36, 0x0a], "vpermd zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPERMD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x36, 0xca], "vpermd zmm1{k5}{z}, zmm0, zmm2"); // VPERMD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x36, 0x0a], "vpermd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPERMD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x36, 0xca], "vpermd zmm1, zmm0, zmm2"); // VPERMD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x36, 0xca], "vpermd zmm1{k5}, zmm0, zmm2"); // VPERMD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x36, 0x0a], "vpermd zmm1, zmm0, zmmword [rdx]"); // VPERMD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x36, 0x0a], "vpermd zmm1{k5}, zmm0, zmmword [rdx]"); // VPERMD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x37, 0x0a], "vpcmpgtq k1, ymm0, qword [rdx]{1to4}"); // VPCMPGTQ_MASKmskw_MASKmskw_YMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x37, 0x0a], "vpcmpgtq k1{k5}, ymm0, qword [rdx]{1to4}"); // VPCMPGTQ_MASKmskw_MASKmskw_YMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x37, 0xca], "vpcmpgtq k1, ymm0, ymm2"); // VPCMPGTQ_MASKmskw_MASKmskw_YMMi64_YMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x37, 0xca], "vpcmpgtq k1{k5}, ymm0, ymm2"); // VPCMPGTQ_MASKmskw_MASKmskw_YMMi64_YMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x37, 0x0a], "vpcmpgtq k1, ymm0, ymmword [rdx]"); // VPCMPGTQ_MASKmskw_MASKmskw_YMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x37, 0x0a], "vpcmpgtq k1{k5}, ymm0, ymmword [rdx]"); // VPCMPGTQ_MASKmskw_MASKmskw_YMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x37, 0x0a], "vpcmpgtq k1, zmm0, qword [rdx]{1to8}"); // VPCMPGTQ_MASKmskw_MASKmskw_ZMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x37, 0x0a], "vpcmpgtq k1{k5}, zmm0, qword [rdx]{1to8}"); // VPCMPGTQ_MASKmskw_MASKmskw_ZMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x37, 0x0a], "vpcmpgtq k1, xmm0, qword [rdx]{1to2}"); // VPCMPGTQ_MASKmskw_MASKmskw_XMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x37, 0x0a], "vpcmpgtq k1{k5}, xmm0, qword [rdx]{1to2}"); // VPCMPGTQ_MASKmskw_MASKmskw_XMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x37, 0xca], "vpcmpgtq k1, zmm0, zmm2"); // VPCMPGTQ_MASKmskw_MASKmskw_ZMMi64_ZMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x37, 0xca], "vpcmpgtq k1{k5}, zmm0, zmm2"); // VPCMPGTQ_MASKmskw_MASKmskw_ZMMi64_ZMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x37, 0x0a], "vpcmpgtq k1, zmm0, zmmword [rdx]"); // VPCMPGTQ_MASKmskw_MASKmskw_ZMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x37, 0x0a], "vpcmpgtq k1{k5}, zmm0, zmmword [rdx]"); // VPCMPGTQ_MASKmskw_MASKmskw_ZMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x37, 0xca], "vpcmpgtq k1, xmm0, xmm2"); // VPCMPGTQ_MASKmskw_MASKmskw_XMMi64_XMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x37, 0xca], "vpcmpgtq k1{k5}, xmm0, xmm2"); // VPCMPGTQ_MASKmskw_MASKmskw_XMMi64_XMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x37, 0x0a], "vpcmpgtq k1, xmm0, xmmword [rdx]"); // VPCMPGTQ_MASKmskw_MASKmskw_XMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x37, 0x0a], "vpcmpgtq k1{k5}, xmm0, xmmword [rdx]"); // VPCMPGTQ_MASKmskw_MASKmskw_XMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x38, 0xca], "vpminsb ymm1{k5}{z}, ymm0, ymm2"); // VPMINSB_YMMi8_MASKmskw_YMMi8_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x38, 0x0a], "vpminsb ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMINSB_YMMi8_MASKmskw_YMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x38, 0xca], "vpminsb ymm1, ymm0, ymm2"); // VPMINSB_YMMi8_MASKmskw_YMMi8_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x38, 0xca], "vpminsb ymm1{k5}, ymm0, ymm2"); // VPMINSB_YMMi8_MASKmskw_YMMi8_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x38, 0x0a], "vpminsb ymm1, ymm0, ymmword [rdx]"); // VPMINSB_YMMi8_MASKmskw_YMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x38, 0x0a], "vpminsb ymm1{k5}, ymm0, ymmword [rdx]"); // VPMINSB_YMMi8_MASKmskw_YMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x38, 0xca], "vpminsb zmm1{k5}{z}, zmm0, zmm2"); // VPMINSB_ZMMi8_MASKmskw_ZMMi8_ZMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x38, 0x0a], "vpminsb zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMINSB_ZMMi8_MASKmskw_ZMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x38, 0xca], "vpminsb zmm1, zmm0, zmm2"); // VPMINSB_ZMMi8_MASKmskw_ZMMi8_ZMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x38, 0xca], "vpminsb zmm1{k5}, zmm0, zmm2"); // VPMINSB_ZMMi8_MASKmskw_ZMMi8_ZMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x38, 0x0a], "vpminsb zmm1, zmm0, zmmword [rdx]"); // VPMINSB_ZMMi8_MASKmskw_ZMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x38, 0x0a], "vpminsb zmm1{k5}, zmm0, zmmword [rdx]"); // VPMINSB_ZMMi8_MASKmskw_ZMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x38, 0xca], "vpminsb xmm1{k5}{z}, xmm0, xmm2"); // VPMINSB_XMMi8_MASKmskw_XMMi8_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x38, 0x0a], "vpminsb xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMINSB_XMMi8_MASKmskw_XMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x38, 0xca], "vpminsb xmm1, xmm0, xmm2"); // VPMINSB_XMMi8_MASKmskw_XMMi8_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x38, 0xca], "vpminsb xmm1{k5}, xmm0, xmm2"); // VPMINSB_XMMi8_MASKmskw_XMMi8_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x38, 0x0a], "vpminsb xmm1, xmm0, xmmword [rdx]"); // VPMINSB_XMMi8_MASKmskw_XMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x38, 0x0a], "vpminsb xmm1{k5}, xmm0, xmmword [rdx]"); // VPMINSB_XMMi8_MASKmskw_XMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x39, 0x0a], "vpminsq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPMINSQ_YMMi64_MASKmskw_YMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x39, 0x0a], "vpminsq ymm1, ymm0, qword [rdx]{1to4}"); // VPMINSQ_YMMi64_MASKmskw_YMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x39, 0x0a], "vpminsq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPMINSQ_YMMi64_MASKmskw_YMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x39, 0xca], "vpminsq ymm1{k5}{z}, ymm0, ymm2"); // VPMINSQ_YMMi64_MASKmskw_YMMi64_YMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x39, 0x0a], "vpminsq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMINSQ_YMMi64_MASKmskw_YMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x39, 0xca], "vpminsq ymm1, ymm0, ymm2"); // VPMINSQ_YMMi64_MASKmskw_YMMi64_YMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x39, 0xca], "vpminsq ymm1{k5}, ymm0, ymm2"); // VPMINSQ_YMMi64_MASKmskw_YMMi64_YMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x39, 0x0a], "vpminsq ymm1, ymm0, ymmword [rdx]"); // VPMINSQ_YMMi64_MASKmskw_YMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x39, 0x0a], "vpminsq ymm1{k5}, ymm0, ymmword [rdx]"); // VPMINSQ_YMMi64_MASKmskw_YMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x39, 0x0a], "vpminsd ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPMINSD_YMMi32_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x39, 0x0a], "vpminsd ymm1, ymm0, dword [rdx]{1to8}"); // VPMINSD_YMMi32_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x39, 0x0a], "vpminsd ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPMINSD_YMMi32_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x39, 0xca], "vpminsd ymm1{k5}{z}, ymm0, ymm2"); // VPMINSD_YMMi32_MASKmskw_YMMi32_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x39, 0x0a], "vpminsd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMINSD_YMMi32_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x39, 0xca], "vpminsd ymm1, ymm0, ymm2"); // VPMINSD_YMMi32_MASKmskw_YMMi32_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x39, 0xca], "vpminsd ymm1{k5}, ymm0, ymm2"); // VPMINSD_YMMi32_MASKmskw_YMMi32_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x39, 0x0a], "vpminsd ymm1, ymm0, ymmword [rdx]"); // VPMINSD_YMMi32_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x39, 0x0a], "vpminsd ymm1{k5}, ymm0, ymmword [rdx]"); // VPMINSD_YMMi32_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x39, 0x0a], "vpminsq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPMINSQ_ZMMi64_MASKmskw_ZMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x39, 0x0a], "vpminsq zmm1, zmm0, qword [rdx]{1to8}"); // VPMINSQ_ZMMi64_MASKmskw_ZMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x39, 0x0a], "vpminsq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPMINSQ_ZMMi64_MASKmskw_ZMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x39, 0x0a], "vpminsq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPMINSQ_XMMi64_MASKmskw_XMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x39, 0x0a], "vpminsq xmm1, xmm0, qword [rdx]{1to2}"); // VPMINSQ_XMMi64_MASKmskw_XMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x39, 0x0a], "vpminsq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPMINSQ_XMMi64_MASKmskw_XMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x39, 0xca], "vpminsq zmm1{k5}{z}, zmm0, zmm2"); // VPMINSQ_ZMMi64_MASKmskw_ZMMi64_ZMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x39, 0x0a], "vpminsq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMINSQ_ZMMi64_MASKmskw_ZMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x39, 0xca], "vpminsq zmm1, zmm0, zmm2"); // VPMINSQ_ZMMi64_MASKmskw_ZMMi64_ZMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x39, 0xca], "vpminsq zmm1{k5}, zmm0, zmm2"); // VPMINSQ_ZMMi64_MASKmskw_ZMMi64_ZMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x39, 0x0a], "vpminsq zmm1, zmm0, zmmword [rdx]"); // VPMINSQ_ZMMi64_MASKmskw_ZMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x39, 0x0a], "vpminsq zmm1{k5}, zmm0, zmmword [rdx]"); // VPMINSQ_ZMMi64_MASKmskw_ZMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x39, 0xca], "vpminsq xmm1{k5}{z}, xmm0, xmm2"); // VPMINSQ_XMMi64_MASKmskw_XMMi64_XMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x39, 0x0a], "vpminsq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMINSQ_XMMi64_MASKmskw_XMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x39, 0xca], "vpminsq xmm1, xmm0, xmm2"); // VPMINSQ_XMMi64_MASKmskw_XMMi64_XMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x39, 0xca], "vpminsq xmm1{k5}, xmm0, xmm2"); // VPMINSQ_XMMi64_MASKmskw_XMMi64_XMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x39, 0x0a], "vpminsq xmm1, xmm0, xmmword [rdx]"); // VPMINSQ_XMMi64_MASKmskw_XMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x39, 0x0a], "vpminsq xmm1{k5}, xmm0, xmmword [rdx]"); // VPMINSQ_XMMi64_MASKmskw_XMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x39, 0x0a], "vpminsd zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPMINSD_ZMMi32_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x39, 0x0a], "vpminsd zmm1, zmm0, dword [rdx]{1to16}"); // VPMINSD_ZMMi32_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x39, 0x0a], "vpminsd zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPMINSD_ZMMi32_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x39, 0x0a], "vpminsd xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPMINSD_XMMi32_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x39, 0x0a], "vpminsd xmm1, xmm0, dword [rdx]{1to4}"); // VPMINSD_XMMi32_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x39, 0x0a], "vpminsd xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPMINSD_XMMi32_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x39, 0xca], "vpminsd zmm1{k5}{z}, zmm0, zmm2"); // VPMINSD_ZMMi32_MASKmskw_ZMMi32_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x39, 0x0a], "vpminsd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMINSD_ZMMi32_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x39, 0xca], "vpminsd zmm1, zmm0, zmm2"); // VPMINSD_ZMMi32_MASKmskw_ZMMi32_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x39, 0xca], "vpminsd zmm1{k5}, zmm0, zmm2"); // VPMINSD_ZMMi32_MASKmskw_ZMMi32_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x39, 0x0a], "vpminsd zmm1, zmm0, zmmword [rdx]"); // VPMINSD_ZMMi32_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x39, 0x0a], "vpminsd zmm1{k5}, zmm0, zmmword [rdx]"); // VPMINSD_ZMMi32_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x39, 0xca], "vpminsd xmm1{k5}{z}, xmm0, xmm2"); // VPMINSD_XMMi32_MASKmskw_XMMi32_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x39, 0x0a], "vpminsd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMINSD_XMMi32_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x39, 0xca], "vpminsd xmm1, xmm0, xmm2"); // VPMINSD_XMMi32_MASKmskw_XMMi32_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x39, 0xca], "vpminsd xmm1{k5}, xmm0, xmm2"); // VPMINSD_XMMi32_MASKmskw_XMMi32_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x39, 0x0a], "vpminsd xmm1, xmm0, xmmword [rdx]"); // VPMINSD_XMMi32_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x39, 0x0a], "vpminsd xmm1{k5}, xmm0, xmmword [rdx]"); // VPMINSD_XMMi32_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3a, 0xca], "vpminuw ymm1{k5}{z}, ymm0, ymm2"); // VPMINUW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3a, 0x0a], "vpminuw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMINUW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3a, 0xca], "vpminuw ymm1, ymm0, ymm2"); // VPMINUW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3a, 0xca], "vpminuw ymm1{k5}, ymm0, ymm2"); // VPMINUW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3a, 0x0a], "vpminuw ymm1, ymm0, ymmword [rdx]"); // VPMINUW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3a, 0x0a], "vpminuw ymm1{k5}, ymm0, ymmword [rdx]"); // VPMINUW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3a, 0xca], "vpminuw zmm1{k5}{z}, zmm0, zmm2"); // VPMINUW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3a, 0x0a], "vpminuw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMINUW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3a, 0xca], "vpminuw zmm1, zmm0, zmm2"); // VPMINUW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3a, 0xca], "vpminuw zmm1{k5}, zmm0, zmm2"); // VPMINUW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3a, 0x0a], "vpminuw zmm1, zmm0, zmmword [rdx]"); // VPMINUW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3a, 0x0a], "vpminuw zmm1{k5}, zmm0, zmmword [rdx]"); // VPMINUW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3a, 0xca], "vpminuw xmm1{k5}{z}, xmm0, xmm2"); // VPMINUW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3a, 0x0a], "vpminuw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMINUW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3a, 0xca], "vpminuw xmm1, xmm0, xmm2"); // VPMINUW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3a, 0xca], "vpminuw xmm1{k5}, xmm0, xmm2"); // VPMINUW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3a, 0x0a], "vpminuw xmm1, xmm0, xmmword [rdx]"); // VPMINUW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3a, 0x0a], "vpminuw xmm1{k5}, xmm0, xmmword [rdx]"); // VPMINUW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x3b, 0x0a], "vpminuq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPMINUQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x3b, 0x0a], "vpminuq ymm1, ymm0, qword [rdx]{1to4}"); // VPMINUQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x3b, 0x0a], "vpminuq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPMINUQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3b, 0xca], "vpminuq ymm1{k5}{z}, ymm0, ymm2"); // VPMINUQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3b, 0x0a], "vpminuq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMINUQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3b, 0xca], "vpminuq ymm1, ymm0, ymm2"); // VPMINUQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3b, 0xca], "vpminuq ymm1{k5}, ymm0, ymm2"); // VPMINUQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3b, 0x0a], "vpminuq ymm1, ymm0, ymmword [rdx]"); // VPMINUQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3b, 0x0a], "vpminuq ymm1{k5}, ymm0, ymmword [rdx]"); // VPMINUQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x3b, 0x0a], "vpminud ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPMINUD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x3b, 0x0a], "vpminud ymm1, ymm0, dword [rdx]{1to8}"); // VPMINUD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x3b, 0x0a], "vpminud ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPMINUD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x3b, 0xca], "vpminud ymm1{k5}{z}, ymm0, ymm2"); // VPMINUD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x3b, 0x0a], "vpminud ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMINUD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x3b, 0xca], "vpminud ymm1, ymm0, ymm2"); // VPMINUD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x3b, 0xca], "vpminud ymm1{k5}, ymm0, ymm2"); // VPMINUD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x3b, 0x0a], "vpminud ymm1, ymm0, ymmword [rdx]"); // VPMINUD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x3b, 0x0a], "vpminud ymm1{k5}, ymm0, ymmword [rdx]"); // VPMINUD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x3b, 0x0a], "vpminuq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPMINUQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x3b, 0x0a], "vpminuq zmm1, zmm0, qword [rdx]{1to8}"); // VPMINUQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x3b, 0x0a], "vpminuq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPMINUQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x3b, 0x0a], "vpminuq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPMINUQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x3b, 0x0a], "vpminuq xmm1, xmm0, qword [rdx]{1to2}"); // VPMINUQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x3b, 0x0a], "vpminuq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPMINUQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3b, 0xca], "vpminuq zmm1{k5}{z}, zmm0, zmm2"); // VPMINUQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3b, 0x0a], "vpminuq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMINUQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3b, 0xca], "vpminuq zmm1, zmm0, zmm2"); // VPMINUQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3b, 0xca], "vpminuq zmm1{k5}, zmm0, zmm2"); // VPMINUQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3b, 0x0a], "vpminuq zmm1, zmm0, zmmword [rdx]"); // VPMINUQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3b, 0x0a], "vpminuq zmm1{k5}, zmm0, zmmword [rdx]"); // VPMINUQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3b, 0xca], "vpminuq xmm1{k5}{z}, xmm0, xmm2"); // VPMINUQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3b, 0x0a], "vpminuq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMINUQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3b, 0xca], "vpminuq xmm1, xmm0, xmm2"); // VPMINUQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3b, 0xca], "vpminuq xmm1{k5}, xmm0, xmm2"); // VPMINUQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3b, 0x0a], "vpminuq xmm1, xmm0, xmmword [rdx]"); // VPMINUQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3b, 0x0a], "vpminuq xmm1{k5}, xmm0, xmmword [rdx]"); // VPMINUQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x3b, 0x0a], "vpminud zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPMINUD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x3b, 0x0a], "vpminud zmm1, zmm0, dword [rdx]{1to16}"); // VPMINUD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x3b, 0x0a], "vpminud zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPMINUD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x3b, 0x0a], "vpminud xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPMINUD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x3b, 0x0a], "vpminud xmm1, xmm0, dword [rdx]{1to4}"); // VPMINUD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x3b, 0x0a], "vpminud xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPMINUD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x3b, 0xca], "vpminud zmm1{k5}{z}, zmm0, zmm2"); // VPMINUD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x3b, 0x0a], "vpminud zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMINUD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x3b, 0xca], "vpminud zmm1, zmm0, zmm2"); // VPMINUD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x3b, 0xca], "vpminud zmm1{k5}, zmm0, zmm2"); // VPMINUD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x3b, 0x0a], "vpminud zmm1, zmm0, zmmword [rdx]"); // VPMINUD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x3b, 0x0a], "vpminud zmm1{k5}, zmm0, zmmword [rdx]"); // VPMINUD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x3b, 0xca], "vpminud xmm1{k5}{z}, xmm0, xmm2"); // VPMINUD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x3b, 0x0a], "vpminud xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMINUD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x3b, 0xca], "vpminud xmm1, xmm0, xmm2"); // VPMINUD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x3b, 0xca], "vpminud xmm1{k5}, xmm0, xmm2"); // VPMINUD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x3b, 0x0a], "vpminud xmm1, xmm0, xmmword [rdx]"); // VPMINUD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x3b, 0x0a], "vpminud xmm1{k5}, xmm0, xmmword [rdx]"); // VPMINUD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3c, 0xca], "vpmaxsb ymm1{k5}{z}, ymm0, ymm2"); // VPMAXSB_YMMi8_MASKmskw_YMMi8_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3c, 0x0a], "vpmaxsb ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMAXSB_YMMi8_MASKmskw_YMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3c, 0xca], "vpmaxsb ymm1, ymm0, ymm2"); // VPMAXSB_YMMi8_MASKmskw_YMMi8_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3c, 0xca], "vpmaxsb ymm1{k5}, ymm0, ymm2"); // VPMAXSB_YMMi8_MASKmskw_YMMi8_YMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3c, 0x0a], "vpmaxsb ymm1, ymm0, ymmword [rdx]"); // VPMAXSB_YMMi8_MASKmskw_YMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3c, 0x0a], "vpmaxsb ymm1{k5}, ymm0, ymmword [rdx]"); // VPMAXSB_YMMi8_MASKmskw_YMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3c, 0xca], "vpmaxsb zmm1{k5}{z}, zmm0, zmm2"); // VPMAXSB_ZMMi8_MASKmskw_ZMMi8_ZMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3c, 0x0a], "vpmaxsb zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMAXSB_ZMMi8_MASKmskw_ZMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3c, 0xca], "vpmaxsb zmm1, zmm0, zmm2"); // VPMAXSB_ZMMi8_MASKmskw_ZMMi8_ZMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3c, 0xca], "vpmaxsb zmm1{k5}, zmm0, zmm2"); // VPMAXSB_ZMMi8_MASKmskw_ZMMi8_ZMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3c, 0x0a], "vpmaxsb zmm1, zmm0, zmmword [rdx]"); // VPMAXSB_ZMMi8_MASKmskw_ZMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3c, 0x0a], "vpmaxsb zmm1{k5}, zmm0, zmmword [rdx]"); // VPMAXSB_ZMMi8_MASKmskw_ZMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3c, 0xca], "vpmaxsb xmm1{k5}{z}, xmm0, xmm2"); // VPMAXSB_XMMi8_MASKmskw_XMMi8_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3c, 0x0a], "vpmaxsb xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMAXSB_XMMi8_MASKmskw_XMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3c, 0xca], "vpmaxsb xmm1, xmm0, xmm2"); // VPMAXSB_XMMi8_MASKmskw_XMMi8_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3c, 0xca], "vpmaxsb xmm1{k5}, xmm0, xmm2"); // VPMAXSB_XMMi8_MASKmskw_XMMi8_XMMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3c, 0x0a], "vpmaxsb xmm1, xmm0, xmmword [rdx]"); // VPMAXSB_XMMi8_MASKmskw_XMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3c, 0x0a], "vpmaxsb xmm1{k5}, xmm0, xmmword [rdx]"); // VPMAXSB_XMMi8_MASKmskw_XMMi8_MEMi8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x3d, 0x0a], "vpmaxsq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPMAXSQ_YMMi64_MASKmskw_YMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x3d, 0x0a], "vpmaxsq ymm1, ymm0, qword [rdx]{1to4}"); // VPMAXSQ_YMMi64_MASKmskw_YMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x3d, 0x0a], "vpmaxsq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPMAXSQ_YMMi64_MASKmskw_YMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3d, 0xca], "vpmaxsq ymm1{k5}{z}, ymm0, ymm2"); // VPMAXSQ_YMMi64_MASKmskw_YMMi64_YMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3d, 0x0a], "vpmaxsq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMAXSQ_YMMi64_MASKmskw_YMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3d, 0xca], "vpmaxsq ymm1, ymm0, ymm2"); // VPMAXSQ_YMMi64_MASKmskw_YMMi64_YMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3d, 0xca], "vpmaxsq ymm1{k5}, ymm0, ymm2"); // VPMAXSQ_YMMi64_MASKmskw_YMMi64_YMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3d, 0x0a], "vpmaxsq ymm1, ymm0, ymmword [rdx]"); // VPMAXSQ_YMMi64_MASKmskw_YMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3d, 0x0a], "vpmaxsq ymm1{k5}, ymm0, ymmword [rdx]"); // VPMAXSQ_YMMi64_MASKmskw_YMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x3d, 0x0a], "vpmaxsd ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPMAXSD_YMMi32_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x3d, 0x0a], "vpmaxsd ymm1, ymm0, dword [rdx]{1to8}"); // VPMAXSD_YMMi32_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x3d, 0x0a], "vpmaxsd ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPMAXSD_YMMi32_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x3d, 0xca], "vpmaxsd ymm1{k5}{z}, ymm0, ymm2"); // VPMAXSD_YMMi32_MASKmskw_YMMi32_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x3d, 0x0a], "vpmaxsd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMAXSD_YMMi32_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x3d, 0xca], "vpmaxsd ymm1, ymm0, ymm2"); // VPMAXSD_YMMi32_MASKmskw_YMMi32_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x3d, 0xca], "vpmaxsd ymm1{k5}, ymm0, ymm2"); // VPMAXSD_YMMi32_MASKmskw_YMMi32_YMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x3d, 0x0a], "vpmaxsd ymm1, ymm0, ymmword [rdx]"); // VPMAXSD_YMMi32_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x3d, 0x0a], "vpmaxsd ymm1{k5}, ymm0, ymmword [rdx]"); // VPMAXSD_YMMi32_MASKmskw_YMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x3d, 0x0a], "vpmaxsq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPMAXSQ_ZMMi64_MASKmskw_ZMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x3d, 0x0a], "vpmaxsq zmm1, zmm0, qword [rdx]{1to8}"); // VPMAXSQ_ZMMi64_MASKmskw_ZMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x3d, 0x0a], "vpmaxsq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPMAXSQ_ZMMi64_MASKmskw_ZMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x3d, 0x0a], "vpmaxsq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPMAXSQ_XMMi64_MASKmskw_XMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x3d, 0x0a], "vpmaxsq xmm1, xmm0, qword [rdx]{1to2}"); // VPMAXSQ_XMMi64_MASKmskw_XMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x3d, 0x0a], "vpmaxsq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPMAXSQ_XMMi64_MASKmskw_XMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3d, 0xca], "vpmaxsq zmm1{k5}{z}, zmm0, zmm2"); // VPMAXSQ_ZMMi64_MASKmskw_ZMMi64_ZMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3d, 0x0a], "vpmaxsq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMAXSQ_ZMMi64_MASKmskw_ZMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3d, 0xca], "vpmaxsq zmm1, zmm0, zmm2"); // VPMAXSQ_ZMMi64_MASKmskw_ZMMi64_ZMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3d, 0xca], "vpmaxsq zmm1{k5}, zmm0, zmm2"); // VPMAXSQ_ZMMi64_MASKmskw_ZMMi64_ZMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3d, 0x0a], "vpmaxsq zmm1, zmm0, zmmword [rdx]"); // VPMAXSQ_ZMMi64_MASKmskw_ZMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3d, 0x0a], "vpmaxsq zmm1{k5}, zmm0, zmmword [rdx]"); // VPMAXSQ_ZMMi64_MASKmskw_ZMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3d, 0xca], "vpmaxsq xmm1{k5}{z}, xmm0, xmm2"); // VPMAXSQ_XMMi64_MASKmskw_XMMi64_XMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3d, 0x0a], "vpmaxsq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMAXSQ_XMMi64_MASKmskw_XMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3d, 0xca], "vpmaxsq xmm1, xmm0, xmm2"); // VPMAXSQ_XMMi64_MASKmskw_XMMi64_XMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3d, 0xca], "vpmaxsq xmm1{k5}, xmm0, xmm2"); // VPMAXSQ_XMMi64_MASKmskw_XMMi64_XMMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3d, 0x0a], "vpmaxsq xmm1, xmm0, xmmword [rdx]"); // VPMAXSQ_XMMi64_MASKmskw_XMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3d, 0x0a], "vpmaxsq xmm1{k5}, xmm0, xmmword [rdx]"); // VPMAXSQ_XMMi64_MASKmskw_XMMi64_MEMi64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x3d, 0x0a], "vpmaxsd zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPMAXSD_ZMMi32_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x3d, 0x0a], "vpmaxsd zmm1, zmm0, dword [rdx]{1to16}"); // VPMAXSD_ZMMi32_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x3d, 0x0a], "vpmaxsd zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPMAXSD_ZMMi32_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x3d, 0x0a], "vpmaxsd xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPMAXSD_XMMi32_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x3d, 0x0a], "vpmaxsd xmm1, xmm0, dword [rdx]{1to4}"); // VPMAXSD_XMMi32_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x3d, 0x0a], "vpmaxsd xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPMAXSD_XMMi32_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x3d, 0xca], "vpmaxsd zmm1{k5}{z}, zmm0, zmm2"); // VPMAXSD_ZMMi32_MASKmskw_ZMMi32_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x3d, 0x0a], "vpmaxsd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMAXSD_ZMMi32_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x3d, 0xca], "vpmaxsd zmm1, zmm0, zmm2"); // VPMAXSD_ZMMi32_MASKmskw_ZMMi32_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x3d, 0xca], "vpmaxsd zmm1{k5}, zmm0, zmm2"); // VPMAXSD_ZMMi32_MASKmskw_ZMMi32_ZMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x3d, 0x0a], "vpmaxsd zmm1, zmm0, zmmword [rdx]"); // VPMAXSD_ZMMi32_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x3d, 0x0a], "vpmaxsd zmm1{k5}, zmm0, zmmword [rdx]"); // VPMAXSD_ZMMi32_MASKmskw_ZMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x3d, 0xca], "vpmaxsd xmm1{k5}{z}, xmm0, xmm2"); // VPMAXSD_XMMi32_MASKmskw_XMMi32_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x3d, 0x0a], "vpmaxsd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMAXSD_XMMi32_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x3d, 0xca], "vpmaxsd xmm1, xmm0, xmm2"); // VPMAXSD_XMMi32_MASKmskw_XMMi32_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x3d, 0xca], "vpmaxsd xmm1{k5}, xmm0, xmm2"); // VPMAXSD_XMMi32_MASKmskw_XMMi32_XMMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x3d, 0x0a], "vpmaxsd xmm1, xmm0, xmmword [rdx]"); // VPMAXSD_XMMi32_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x3d, 0x0a], "vpmaxsd xmm1{k5}, xmm0, xmmword [rdx]"); // VPMAXSD_XMMi32_MASKmskw_XMMi32_MEMi32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3e, 0xca], "vpmaxuw ymm1{k5}{z}, ymm0, ymm2"); // VPMAXUW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3e, 0x0a], "vpmaxuw ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMAXUW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3e, 0xca], "vpmaxuw ymm1, ymm0, ymm2"); // VPMAXUW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3e, 0xca], "vpmaxuw ymm1{k5}, ymm0, ymm2"); // VPMAXUW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3e, 0x0a], "vpmaxuw ymm1, ymm0, ymmword [rdx]"); // VPMAXUW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3e, 0x0a], "vpmaxuw ymm1{k5}, ymm0, ymmword [rdx]"); // VPMAXUW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3e, 0xca], "vpmaxuw zmm1{k5}{z}, zmm0, zmm2"); // VPMAXUW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3e, 0x0a], "vpmaxuw zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMAXUW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3e, 0xca], "vpmaxuw zmm1, zmm0, zmm2"); // VPMAXUW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3e, 0xca], "vpmaxuw zmm1{k5}, zmm0, zmm2"); // VPMAXUW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3e, 0x0a], "vpmaxuw zmm1, zmm0, zmmword [rdx]"); // VPMAXUW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3e, 0x0a], "vpmaxuw zmm1{k5}, zmm0, zmmword [rdx]"); // VPMAXUW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3e, 0xca], "vpmaxuw xmm1{k5}{z}, xmm0, xmm2"); // VPMAXUW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3e, 0x0a], "vpmaxuw xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMAXUW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3e, 0xca], "vpmaxuw xmm1, xmm0, xmm2"); // VPMAXUW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3e, 0xca], "vpmaxuw xmm1{k5}, xmm0, xmm2"); // VPMAXUW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3e, 0x0a], "vpmaxuw xmm1, xmm0, xmmword [rdx]"); // VPMAXUW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3e, 0x0a], "vpmaxuw xmm1{k5}, xmm0, xmmword [rdx]"); // VPMAXUW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x3f, 0x0a], "vpmaxuq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPMAXUQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x3f, 0x0a], "vpmaxuq ymm1, ymm0, qword [rdx]{1to4}"); // VPMAXUQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x3f, 0x0a], "vpmaxuq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPMAXUQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3f, 0xca], "vpmaxuq ymm1{k5}{z}, ymm0, ymm2"); // VPMAXUQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x3f, 0x0a], "vpmaxuq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMAXUQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3f, 0xca], "vpmaxuq ymm1, ymm0, ymm2"); // VPMAXUQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3f, 0xca], "vpmaxuq ymm1{k5}, ymm0, ymm2"); // VPMAXUQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x3f, 0x0a], "vpmaxuq ymm1, ymm0, ymmword [rdx]"); // VPMAXUQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x3f, 0x0a], "vpmaxuq ymm1{k5}, ymm0, ymmword [rdx]"); // VPMAXUQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x3f, 0x0a], "vpmaxud ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPMAXUD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x3f, 0x0a], "vpmaxud ymm1, ymm0, dword [rdx]{1to8}"); // VPMAXUD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x3f, 0x0a], "vpmaxud ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPMAXUD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x3f, 0xca], "vpmaxud ymm1{k5}{z}, ymm0, ymm2"); // VPMAXUD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x3f, 0x0a], "vpmaxud ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMAXUD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x3f, 0xca], "vpmaxud ymm1, ymm0, ymm2"); // VPMAXUD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x3f, 0xca], "vpmaxud ymm1{k5}, ymm0, ymm2"); // VPMAXUD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x3f, 0x0a], "vpmaxud ymm1, ymm0, ymmword [rdx]"); // VPMAXUD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x3f, 0x0a], "vpmaxud ymm1{k5}, ymm0, ymmword [rdx]"); // VPMAXUD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x3f, 0x0a], "vpmaxuq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPMAXUQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x3f, 0x0a], "vpmaxuq zmm1, zmm0, qword [rdx]{1to8}"); // VPMAXUQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x3f, 0x0a], "vpmaxuq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPMAXUQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x3f, 0x0a], "vpmaxuq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPMAXUQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x3f, 0x0a], "vpmaxuq xmm1, xmm0, qword [rdx]{1to2}"); // VPMAXUQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x3f, 0x0a], "vpmaxuq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPMAXUQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3f, 0xca], "vpmaxuq zmm1{k5}{z}, zmm0, zmm2"); // VPMAXUQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x3f, 0x0a], "vpmaxuq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMAXUQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3f, 0xca], "vpmaxuq zmm1, zmm0, zmm2"); // VPMAXUQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3f, 0xca], "vpmaxuq zmm1{k5}, zmm0, zmm2"); // VPMAXUQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x3f, 0x0a], "vpmaxuq zmm1, zmm0, zmmword [rdx]"); // VPMAXUQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x3f, 0x0a], "vpmaxuq zmm1{k5}, zmm0, zmmword [rdx]"); // VPMAXUQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3f, 0xca], "vpmaxuq xmm1{k5}{z}, xmm0, xmm2"); // VPMAXUQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x3f, 0x0a], "vpmaxuq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMAXUQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3f, 0xca], "vpmaxuq xmm1, xmm0, xmm2"); // VPMAXUQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3f, 0xca], "vpmaxuq xmm1{k5}, xmm0, xmm2"); // VPMAXUQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x3f, 0x0a], "vpmaxuq xmm1, xmm0, xmmword [rdx]"); // VPMAXUQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x3f, 0x0a], "vpmaxuq xmm1{k5}, xmm0, xmmword [rdx]"); // VPMAXUQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x3f, 0x0a], "vpmaxud zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPMAXUD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x3f, 0x0a], "vpmaxud zmm1, zmm0, dword [rdx]{1to16}"); // VPMAXUD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x3f, 0x0a], "vpmaxud zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPMAXUD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x3f, 0x0a], "vpmaxud xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPMAXUD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x3f, 0x0a], "vpmaxud xmm1, xmm0, dword [rdx]{1to4}"); // VPMAXUD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x3f, 0x0a], "vpmaxud xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPMAXUD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x3f, 0xca], "vpmaxud zmm1{k5}{z}, zmm0, zmm2"); // VPMAXUD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x3f, 0x0a], "vpmaxud zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMAXUD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x3f, 0xca], "vpmaxud zmm1, zmm0, zmm2"); // VPMAXUD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x3f, 0xca], "vpmaxud zmm1{k5}, zmm0, zmm2"); // VPMAXUD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x3f, 0x0a], "vpmaxud zmm1, zmm0, zmmword [rdx]"); // VPMAXUD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x3f, 0x0a], "vpmaxud zmm1{k5}, zmm0, zmmword [rdx]"); // VPMAXUD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x3f, 0xca], "vpmaxud xmm1{k5}{z}, xmm0, xmm2"); // VPMAXUD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x3f, 0x0a], "vpmaxud xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMAXUD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x3f, 0xca], "vpmaxud xmm1, xmm0, xmm2"); // VPMAXUD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x3f, 0xca], "vpmaxud xmm1{k5}, xmm0, xmm2"); // VPMAXUD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x3f, 0x0a], "vpmaxud xmm1, xmm0, xmmword [rdx]"); // VPMAXUD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x3f, 0x0a], "vpmaxud xmm1{k5}, xmm0, xmmword [rdx]"); // VPMAXUD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x40, 0x0a], "vpmullq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPMULLQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x40, 0x0a], "vpmullq ymm1, ymm0, qword [rdx]{1to4}"); // VPMULLQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x40, 0x0a], "vpmullq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPMULLQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x40, 0xca], "vpmullq ymm1{k5}{z}, ymm0, ymm2"); // VPMULLQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x40, 0x0a], "vpmullq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMULLQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x40, 0xca], "vpmullq ymm1, ymm0, ymm2"); // VPMULLQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x40, 0xca], "vpmullq ymm1{k5}, ymm0, ymm2"); // VPMULLQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x40, 0x0a], "vpmullq ymm1, ymm0, ymmword [rdx]"); // VPMULLQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x40, 0x0a], "vpmullq ymm1{k5}, ymm0, ymmword [rdx]"); // VPMULLQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x40, 0x0a], "vpmulld ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPMULLD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x40, 0x0a], "vpmulld ymm1, ymm0, dword [rdx]{1to8}"); // VPMULLD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x40, 0x0a], "vpmulld ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPMULLD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x40, 0xca], "vpmulld ymm1{k5}{z}, ymm0, ymm2"); // VPMULLD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x40, 0x0a], "vpmulld ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPMULLD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x40, 0xca], "vpmulld ymm1, ymm0, ymm2"); // VPMULLD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x40, 0xca], "vpmulld ymm1{k5}, ymm0, ymm2"); // VPMULLD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x40, 0x0a], "vpmulld ymm1, ymm0, ymmword [rdx]"); // VPMULLD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x40, 0x0a], "vpmulld ymm1{k5}, ymm0, ymmword [rdx]"); // VPMULLD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x40, 0x0a], "vpmullq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPMULLQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x40, 0x0a], "vpmullq zmm1, zmm0, qword [rdx]{1to8}"); // VPMULLQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x40, 0x0a], "vpmullq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPMULLQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x40, 0x0a], "vpmullq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPMULLQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x40, 0x0a], "vpmullq xmm1, xmm0, qword [rdx]{1to2}"); // VPMULLQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x40, 0x0a], "vpmullq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPMULLQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x40, 0xca], "vpmullq zmm1{k5}{z}, zmm0, zmm2"); // VPMULLQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x40, 0x0a], "vpmullq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMULLQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x40, 0xca], "vpmullq zmm1, zmm0, zmm2"); // VPMULLQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x40, 0xca], "vpmullq zmm1{k5}, zmm0, zmm2"); // VPMULLQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x40, 0x0a], "vpmullq zmm1, zmm0, zmmword [rdx]"); // VPMULLQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x40, 0x0a], "vpmullq zmm1{k5}, zmm0, zmmword [rdx]"); // VPMULLQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x40, 0xca], "vpmullq xmm1{k5}{z}, xmm0, xmm2"); // VPMULLQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x40, 0x0a], "vpmullq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMULLQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x40, 0xca], "vpmullq xmm1, xmm0, xmm2"); // VPMULLQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x40, 0xca], "vpmullq xmm1{k5}, xmm0, xmm2"); // VPMULLQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x40, 0x0a], "vpmullq xmm1, xmm0, xmmword [rdx]"); // VPMULLQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x40, 0x0a], "vpmullq xmm1{k5}, xmm0, xmmword [rdx]"); // VPMULLQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x40, 0x0a], "vpmulld zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPMULLD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x40, 0x0a], "vpmulld zmm1, zmm0, dword [rdx]{1to16}"); // VPMULLD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x40, 0x0a], "vpmulld zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPMULLD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x40, 0x0a], "vpmulld xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPMULLD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x40, 0x0a], "vpmulld xmm1, xmm0, dword [rdx]{1to4}"); // VPMULLD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x40, 0x0a], "vpmulld xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPMULLD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x40, 0xca], "vpmulld zmm1{k5}{z}, zmm0, zmm2"); // VPMULLD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x40, 0x0a], "vpmulld zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPMULLD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x40, 0xca], "vpmulld zmm1, zmm0, zmm2"); // VPMULLD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x40, 0xca], "vpmulld zmm1{k5}, zmm0, zmm2"); // VPMULLD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x40, 0x0a], "vpmulld zmm1, zmm0, zmmword [rdx]"); // VPMULLD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x40, 0x0a], "vpmulld zmm1{k5}, zmm0, zmmword [rdx]"); // VPMULLD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x40, 0xca], "vpmulld xmm1{k5}{z}, xmm0, xmm2"); // VPMULLD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x40, 0x0a], "vpmulld xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPMULLD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x40, 0xca], "vpmulld xmm1, xmm0, xmm2"); // VPMULLD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x40, 0xca], "vpmulld xmm1{k5}, xmm0, xmm2"); // VPMULLD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x40, 0x0a], "vpmulld xmm1, xmm0, xmmword [rdx]"); // VPMULLD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x40, 0x0a], "vpmulld xmm1{k5}, xmm0, xmmword [rdx]"); // VPMULLD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xfd, 0x42, 0xca], "vgetexppd zmm1{k5}{z}{sae}, zmm2"); // VGETEXPPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x78, 0x42, 0xca], "vgetexppd zmm1{sae}, zmm2"); // VGETEXPPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x7d, 0x42, 0xca], "vgetexppd zmm1{k5}{sae}, zmm2"); // VGETEXPPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x42, 0x0a], "vgetexppd ymm1{k5}{z}, qword [rdx]{1to4}"); // VGETEXPPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x42, 0x0a], "vgetexppd ymm1, qword [rdx]{1to4}"); // VGETEXPPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x42, 0x0a], "vgetexppd ymm1{k5}, qword [rdx]{1to4}"); // VGETEXPPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x42, 0xca], "vgetexppd ymm1{k5}{z}, ymm2"); // VGETEXPPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x42, 0x0a], "vgetexppd ymm1{k5}{z}, ymmword [rdx]"); // VGETEXPPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x42, 0xca], "vgetexppd ymm1, ymm2"); // VGETEXPPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x42, 0xca], "vgetexppd ymm1{k5}, ymm2"); // VGETEXPPD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x42, 0x0a], "vgetexppd ymm1, ymmword [rdx]"); // VGETEXPPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x42, 0x0a], "vgetexppd ymm1{k5}, ymmword [rdx]"); // VGETEXPPD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xfd, 0x42, 0xca], "vgetexpps zmm1{k5}{z}{sae}, zmm2"); // VGETEXPPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x78, 0x42, 0xca], "vgetexpps zmm1{sae}, zmm2"); // VGETEXPPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x7d, 0x42, 0xca], "vgetexpps zmm1{k5}{sae}, zmm2"); // VGETEXPPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x42, 0x0a], "vgetexpps ymm1{k5}{z}, dword [rdx]{1to8}"); // VGETEXPPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x42, 0x0a], "vgetexpps ymm1, dword [rdx]{1to8}"); // VGETEXPPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x42, 0x0a], "vgetexpps ymm1{k5}, dword [rdx]{1to8}"); // VGETEXPPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x42, 0xca], "vgetexpps ymm1{k5}{z}, ymm2"); // VGETEXPPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x42, 0x0a], "vgetexpps ymm1{k5}{z}, ymmword [rdx]"); // VGETEXPPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x42, 0xca], "vgetexpps ymm1, ymm2"); // VGETEXPPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x42, 0xca], "vgetexpps ymm1{k5}, ymm2"); // VGETEXPPS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x42, 0x0a], "vgetexpps ymm1, ymmword [rdx]"); // VGETEXPPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x42, 0x0a], "vgetexpps ymm1{k5}, ymmword [rdx]"); // VGETEXPPS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x42, 0x0a], "vgetexppd zmm1{k5}{z}, qword [rdx]{1to8}"); // VGETEXPPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x42, 0x0a], "vgetexppd zmm1, qword [rdx]{1to8}"); // VGETEXPPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x42, 0x0a], "vgetexppd zmm1{k5}, qword [rdx]{1to8}"); // VGETEXPPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x42, 0x0a], "vgetexppd xmm1{k5}{z}, qword [rdx]{1to2}"); // VGETEXPPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x42, 0x0a], "vgetexppd xmm1, qword [rdx]{1to2}"); // VGETEXPPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x42, 0x0a], "vgetexppd xmm1{k5}, qword [rdx]{1to2}"); // VGETEXPPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x42, 0xca], "vgetexppd zmm1{k5}{z}, zmm2"); // VGETEXPPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x42, 0x0a], "vgetexppd zmm1{k5}{z}, zmmword [rdx]"); // VGETEXPPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x42, 0xca], "vgetexppd zmm1, zmm2"); // VGETEXPPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x42, 0xca], "vgetexppd zmm1{k5}, zmm2"); // VGETEXPPD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x42, 0x0a], "vgetexppd zmm1, zmmword [rdx]"); // VGETEXPPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x42, 0x0a], "vgetexppd zmm1{k5}, zmmword [rdx]"); // VGETEXPPD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x42, 0xca], "vgetexppd xmm1{k5}{z}, xmm2"); // VGETEXPPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x42, 0x0a], "vgetexppd xmm1{k5}{z}, xmmword [rdx]"); // VGETEXPPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x42, 0xca], "vgetexppd xmm1, xmm2"); // VGETEXPPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x42, 0xca], "vgetexppd xmm1{k5}, xmm2"); // VGETEXPPD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x42, 0x0a], "vgetexppd xmm1, xmmword [rdx]"); // VGETEXPPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x42, 0x0a], "vgetexppd xmm1{k5}, xmmword [rdx]"); // VGETEXPPD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x42, 0x0a], "vgetexpps zmm1{k5}{z}, dword [rdx]{1to16}"); // VGETEXPPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x42, 0x0a], "vgetexpps zmm1, dword [rdx]{1to16}"); // VGETEXPPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x42, 0x0a], "vgetexpps zmm1{k5}, dword [rdx]{1to16}"); // VGETEXPPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x42, 0x0a], "vgetexpps xmm1{k5}{z}, dword [rdx]{1to4}"); // VGETEXPPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x42, 0x0a], "vgetexpps xmm1, dword [rdx]{1to4}"); // VGETEXPPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x42, 0x0a], "vgetexpps xmm1{k5}, dword [rdx]{1to4}"); // VGETEXPPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x42, 0xca], "vgetexpps zmm1{k5}{z}, zmm2"); // VGETEXPPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x42, 0x0a], "vgetexpps zmm1{k5}{z}, zmmword [rdx]"); // VGETEXPPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x42, 0xca], "vgetexpps zmm1, zmm2"); // VGETEXPPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x42, 0xca], "vgetexpps zmm1{k5}, zmm2"); // VGETEXPPS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x42, 0x0a], "vgetexpps zmm1, zmmword [rdx]"); // VGETEXPPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x42, 0x0a], "vgetexpps zmm1{k5}, zmmword [rdx]"); // VGETEXPPS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x42, 0xca], "vgetexpps xmm1{k5}{z}, xmm2"); // VGETEXPPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x42, 0x0a], "vgetexpps xmm1{k5}{z}, xmmword [rdx]"); // VGETEXPPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x42, 0xca], "vgetexpps xmm1, xmm2"); // VGETEXPPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x42, 0xca], "vgetexpps xmm1{k5}, xmm2"); // VGETEXPPS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x42, 0x0a], "vgetexpps xmm1, xmmword [rdx]"); // VGETEXPPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x42, 0x0a], "vgetexpps xmm1{k5}, xmmword [rdx]"); // VGETEXPPS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xfd, 0x43, 0xca], "vgetexpsd xmm1{k5}{z}{sae}, xmm0, xmm2"); // VGETEXPSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x78, 0x43, 0xca], "vgetexpsd xmm1{sae}, xmm0, xmm2"); // VGETEXPSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x7d, 0x43, 0xca], "vgetexpsd xmm1{k5}{sae}, xmm0, xmm2"); // VGETEXPSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x43, 0xca], "vgetexpsd xmm1{k5}{z}, xmm0, xmm2"); // VGETEXPSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x43, 0x0a], "vgetexpsd xmm1{k5}{z}, xmm0, qword [rdx]"); // VGETEXPSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x43, 0xca], "vgetexpsd xmm1, xmm0, xmm2"); // VGETEXPSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x43, 0xca], "vgetexpsd xmm1{k5}, xmm0, xmm2"); // VGETEXPSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x43, 0x0a], "vgetexpsd xmm1, xmm0, qword [rdx]"); // VGETEXPSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x43, 0x0a], "vgetexpsd xmm1{k5}, xmm0, qword [rdx]"); // VGETEXPSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xfd, 0x43, 0xca], "vgetexpss xmm1{k5}{z}{sae}, xmm0, xmm2"); // VGETEXPSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x78, 0x43, 0xca], "vgetexpss xmm1{sae}, xmm0, xmm2"); // VGETEXPSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x7d, 0x43, 0xca], "vgetexpss xmm1{k5}{sae}, xmm0, xmm2"); // VGETEXPSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x43, 0xca], "vgetexpss xmm1{k5}{z}, xmm0, xmm2"); // VGETEXPSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x43, 0x0a], "vgetexpss xmm1{k5}{z}, xmm0, dword [rdx]"); // VGETEXPSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x43, 0xca], "vgetexpss xmm1, xmm0, xmm2"); // VGETEXPSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x43, 0xca], "vgetexpss xmm1{k5}, xmm0, xmm2"); // VGETEXPSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x43, 0x0a], "vgetexpss xmm1, xmm0, dword [rdx]"); // VGETEXPSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x43, 0x0a], "vgetexpss xmm1{k5}, xmm0, dword [rdx]"); // VGETEXPSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x44, 0x0a], "vplzcntq ymm1{k5}{z}, qword [rdx]{1to4}"); // VPLZCNTQ_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x44, 0x0a], "vplzcntq ymm1, qword [rdx]{1to4}"); // VPLZCNTQ_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x44, 0x0a], "vplzcntq ymm1{k5}, qword [rdx]{1to4}"); // VPLZCNTQ_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x44, 0xca], "vplzcntq ymm1{k5}{z}, ymm2"); // VPLZCNTQ_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x44, 0x0a], "vplzcntq ymm1{k5}{z}, ymmword [rdx]"); // VPLZCNTQ_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x44, 0xca], "vplzcntq ymm1, ymm2"); // VPLZCNTQ_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x44, 0xca], "vplzcntq ymm1{k5}, ymm2"); // VPLZCNTQ_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x44, 0x0a], "vplzcntq ymm1, ymmword [rdx]"); // VPLZCNTQ_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x44, 0x0a], "vplzcntq ymm1{k5}, ymmword [rdx]"); // VPLZCNTQ_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x44, 0x0a], "vplzcntd ymm1{k5}{z}, dword [rdx]{1to8}"); // VPLZCNTD_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x44, 0x0a], "vplzcntd ymm1, dword [rdx]{1to8}"); // VPLZCNTD_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x44, 0x0a], "vplzcntd ymm1{k5}, dword [rdx]{1to8}"); // VPLZCNTD_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x44, 0xca], "vplzcntd ymm1{k5}{z}, ymm2"); // VPLZCNTD_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x44, 0x0a], "vplzcntd ymm1{k5}{z}, ymmword [rdx]"); // VPLZCNTD_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x44, 0xca], "vplzcntd ymm1, ymm2"); // VPLZCNTD_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x44, 0xca], "vplzcntd ymm1{k5}, ymm2"); // VPLZCNTD_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x44, 0x0a], "vplzcntd ymm1, ymmword [rdx]"); // VPLZCNTD_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x44, 0x0a], "vplzcntd ymm1{k5}, ymmword [rdx]"); // VPLZCNTD_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x44, 0x0a], "vplzcntq zmm1{k5}{z}, qword [rdx]{1to8}"); // VPLZCNTQ_ZMMu64_MASKmskw_MEMu64_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x44, 0x0a], "vplzcntq zmm1, qword [rdx]{1to8}"); // VPLZCNTQ_ZMMu64_MASKmskw_MEMu64_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x44, 0x0a], "vplzcntq zmm1{k5}, qword [rdx]{1to8}"); // VPLZCNTQ_ZMMu64_MASKmskw_MEMu64_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x44, 0x0a], "vplzcntq xmm1{k5}{z}, qword [rdx]{1to2}"); // VPLZCNTQ_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x44, 0x0a], "vplzcntq xmm1, qword [rdx]{1to2}"); // VPLZCNTQ_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x44, 0x0a], "vplzcntq xmm1{k5}, qword [rdx]{1to2}"); // VPLZCNTQ_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x44, 0xca], "vplzcntq zmm1{k5}{z}, zmm2"); // VPLZCNTQ_ZMMu64_MASKmskw_ZMMu64_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x44, 0x0a], "vplzcntq zmm1{k5}{z}, zmmword [rdx]"); // VPLZCNTQ_ZMMu64_MASKmskw_MEMu64_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x44, 0xca], "vplzcntq zmm1, zmm2"); // VPLZCNTQ_ZMMu64_MASKmskw_ZMMu64_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x44, 0xca], "vplzcntq zmm1{k5}, zmm2"); // VPLZCNTQ_ZMMu64_MASKmskw_ZMMu64_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x44, 0x0a], "vplzcntq zmm1, zmmword [rdx]"); // VPLZCNTQ_ZMMu64_MASKmskw_MEMu64_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x44, 0x0a], "vplzcntq zmm1{k5}, zmmword [rdx]"); // VPLZCNTQ_ZMMu64_MASKmskw_MEMu64_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x44, 0xca], "vplzcntq xmm1{k5}{z}, xmm2"); // VPLZCNTQ_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x44, 0x0a], "vplzcntq xmm1{k5}{z}, xmmword [rdx]"); // VPLZCNTQ_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x44, 0xca], "vplzcntq xmm1, xmm2"); // VPLZCNTQ_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x44, 0xca], "vplzcntq xmm1{k5}, xmm2"); // VPLZCNTQ_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x44, 0x0a], "vplzcntq xmm1, xmmword [rdx]"); // VPLZCNTQ_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x44, 0x0a], "vplzcntq xmm1{k5}, xmmword [rdx]"); // VPLZCNTQ_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x44, 0x0a], "vplzcntd zmm1{k5}{z}, dword [rdx]{1to16}"); // VPLZCNTD_ZMMu32_MASKmskw_MEMu32_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x44, 0x0a], "vplzcntd zmm1, dword [rdx]{1to16}"); // VPLZCNTD_ZMMu32_MASKmskw_MEMu32_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x44, 0x0a], "vplzcntd zmm1{k5}, dword [rdx]{1to16}"); // VPLZCNTD_ZMMu32_MASKmskw_MEMu32_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x44, 0x0a], "vplzcntd xmm1{k5}{z}, dword [rdx]{1to4}"); // VPLZCNTD_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x44, 0x0a], "vplzcntd xmm1, dword [rdx]{1to4}"); // VPLZCNTD_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x44, 0x0a], "vplzcntd xmm1{k5}, dword [rdx]{1to4}"); // VPLZCNTD_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x44, 0xca], "vplzcntd zmm1{k5}{z}, zmm2"); // VPLZCNTD_ZMMu32_MASKmskw_ZMMu32_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x44, 0x0a], "vplzcntd zmm1{k5}{z}, zmmword [rdx]"); // VPLZCNTD_ZMMu32_MASKmskw_MEMu32_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x44, 0xca], "vplzcntd zmm1, zmm2"); // VPLZCNTD_ZMMu32_MASKmskw_ZMMu32_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x44, 0xca], "vplzcntd zmm1{k5}, zmm2"); // VPLZCNTD_ZMMu32_MASKmskw_ZMMu32_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x44, 0x0a], "vplzcntd zmm1, zmmword [rdx]"); // VPLZCNTD_ZMMu32_MASKmskw_MEMu32_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x44, 0x0a], "vplzcntd zmm1{k5}, zmmword [rdx]"); // VPLZCNTD_ZMMu32_MASKmskw_MEMu32_AVX512CD, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x44, 0xca], "vplzcntd xmm1{k5}{z}, xmm2"); // VPLZCNTD_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x44, 0x0a], "vplzcntd xmm1{k5}{z}, xmmword [rdx]"); // VPLZCNTD_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x44, 0xca], "vplzcntd xmm1, xmm2"); // VPLZCNTD_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x44, 0xca], "vplzcntd xmm1{k5}, xmm2"); // VPLZCNTD_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x44, 0x0a], "vplzcntd xmm1, xmmword [rdx]"); // VPLZCNTD_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x44, 0x0a], "vplzcntd xmm1{k5}, xmmword [rdx]"); // VPLZCNTD_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x45, 0x0a], "vpsrlvq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPSRLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x45, 0x0a], "vpsrlvq ymm1, ymm0, qword [rdx]{1to4}"); // VPSRLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x45, 0x0a], "vpsrlvq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPSRLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x45, 0xca], "vpsrlvq ymm1{k5}{z}, ymm0, ymm2"); // VPSRLVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x45, 0x0a], "vpsrlvq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSRLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x45, 0xca], "vpsrlvq ymm1, ymm0, ymm2"); // VPSRLVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x45, 0xca], "vpsrlvq ymm1{k5}, ymm0, ymm2"); // VPSRLVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x45, 0x0a], "vpsrlvq ymm1, ymm0, ymmword [rdx]"); // VPSRLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x45, 0x0a], "vpsrlvq ymm1{k5}, ymm0, ymmword [rdx]"); // VPSRLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x45, 0x0a], "vpsrlvd ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPSRLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x45, 0x0a], "vpsrlvd ymm1, ymm0, dword [rdx]{1to8}"); // VPSRLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x45, 0x0a], "vpsrlvd ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPSRLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x45, 0xca], "vpsrlvd ymm1{k5}{z}, ymm0, ymm2"); // VPSRLVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x45, 0x0a], "vpsrlvd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSRLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x45, 0xca], "vpsrlvd ymm1, ymm0, ymm2"); // VPSRLVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x45, 0xca], "vpsrlvd ymm1{k5}, ymm0, ymm2"); // VPSRLVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x45, 0x0a], "vpsrlvd ymm1, ymm0, ymmword [rdx]"); // VPSRLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x45, 0x0a], "vpsrlvd ymm1{k5}, ymm0, ymmword [rdx]"); // VPSRLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x45, 0x0a], "vpsrlvq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPSRLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x45, 0x0a], "vpsrlvq zmm1, zmm0, qword [rdx]{1to8}"); // VPSRLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x45, 0x0a], "vpsrlvq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPSRLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x45, 0x0a], "vpsrlvq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPSRLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x45, 0x0a], "vpsrlvq xmm1, xmm0, qword [rdx]{1to2}"); // VPSRLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x45, 0x0a], "vpsrlvq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPSRLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x45, 0xca], "vpsrlvq zmm1{k5}{z}, zmm0, zmm2"); // VPSRLVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x45, 0x0a], "vpsrlvq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSRLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x45, 0xca], "vpsrlvq zmm1, zmm0, zmm2"); // VPSRLVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x45, 0xca], "vpsrlvq zmm1{k5}, zmm0, zmm2"); // VPSRLVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x45, 0x0a], "vpsrlvq zmm1, zmm0, zmmword [rdx]"); // VPSRLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x45, 0x0a], "vpsrlvq zmm1{k5}, zmm0, zmmword [rdx]"); // VPSRLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x45, 0xca], "vpsrlvq xmm1{k5}{z}, xmm0, xmm2"); // VPSRLVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x45, 0x0a], "vpsrlvq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSRLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x45, 0xca], "vpsrlvq xmm1, xmm0, xmm2"); // VPSRLVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x45, 0xca], "vpsrlvq xmm1{k5}, xmm0, xmm2"); // VPSRLVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x45, 0x0a], "vpsrlvq xmm1, xmm0, xmmword [rdx]"); // VPSRLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x45, 0x0a], "vpsrlvq xmm1{k5}, xmm0, xmmword [rdx]"); // VPSRLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x45, 0x0a], "vpsrlvd zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPSRLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x45, 0x0a], "vpsrlvd zmm1, zmm0, dword [rdx]{1to16}"); // VPSRLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x45, 0x0a], "vpsrlvd zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPSRLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x45, 0x0a], "vpsrlvd xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPSRLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x45, 0x0a], "vpsrlvd xmm1, xmm0, dword [rdx]{1to4}"); // VPSRLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x45, 0x0a], "vpsrlvd xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPSRLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x45, 0xca], "vpsrlvd zmm1{k5}{z}, zmm0, zmm2"); // VPSRLVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x45, 0x0a], "vpsrlvd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSRLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x45, 0xca], "vpsrlvd zmm1, zmm0, zmm2"); // VPSRLVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x45, 0xca], "vpsrlvd zmm1{k5}, zmm0, zmm2"); // VPSRLVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x45, 0x0a], "vpsrlvd zmm1, zmm0, zmmword [rdx]"); // VPSRLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x45, 0x0a], "vpsrlvd zmm1{k5}, zmm0, zmmword [rdx]"); // VPSRLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x45, 0xca], "vpsrlvd xmm1{k5}{z}, xmm0, xmm2"); // VPSRLVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x45, 0x0a], "vpsrlvd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSRLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x45, 0xca], "vpsrlvd xmm1, xmm0, xmm2"); // VPSRLVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x45, 0xca], "vpsrlvd xmm1{k5}, xmm0, xmm2"); // VPSRLVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x45, 0x0a], "vpsrlvd xmm1, xmm0, xmmword [rdx]"); // VPSRLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x45, 0x0a], "vpsrlvd xmm1{k5}, xmm0, xmmword [rdx]"); // VPSRLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x46, 0x0a], "vpsravq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPSRAVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x46, 0x0a], "vpsravq ymm1, ymm0, qword [rdx]{1to4}"); // VPSRAVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x46, 0x0a], "vpsravq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPSRAVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x46, 0xca], "vpsravq ymm1{k5}{z}, ymm0, ymm2"); // VPSRAVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x46, 0x0a], "vpsravq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSRAVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x46, 0xca], "vpsravq ymm1, ymm0, ymm2"); // VPSRAVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x46, 0xca], "vpsravq ymm1{k5}, ymm0, ymm2"); // VPSRAVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x46, 0x0a], "vpsravq ymm1, ymm0, ymmword [rdx]"); // VPSRAVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x46, 0x0a], "vpsravq ymm1{k5}, ymm0, ymmword [rdx]"); // VPSRAVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x46, 0x0a], "vpsravd ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPSRAVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x46, 0x0a], "vpsravd ymm1, ymm0, dword [rdx]{1to8}"); // VPSRAVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x46, 0x0a], "vpsravd ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPSRAVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x46, 0xca], "vpsravd ymm1{k5}{z}, ymm0, ymm2"); // VPSRAVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x46, 0x0a], "vpsravd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSRAVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x46, 0xca], "vpsravd ymm1, ymm0, ymm2"); // VPSRAVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x46, 0xca], "vpsravd ymm1{k5}, ymm0, ymm2"); // VPSRAVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x46, 0x0a], "vpsravd ymm1, ymm0, ymmword [rdx]"); // VPSRAVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x46, 0x0a], "vpsravd ymm1{k5}, ymm0, ymmword [rdx]"); // VPSRAVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x46, 0x0a], "vpsravq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPSRAVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x46, 0x0a], "vpsravq zmm1, zmm0, qword [rdx]{1to8}"); // VPSRAVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x46, 0x0a], "vpsravq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPSRAVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x46, 0x0a], "vpsravq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPSRAVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x46, 0x0a], "vpsravq xmm1, xmm0, qword [rdx]{1to2}"); // VPSRAVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x46, 0x0a], "vpsravq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPSRAVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x46, 0xca], "vpsravq zmm1{k5}{z}, zmm0, zmm2"); // VPSRAVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x46, 0x0a], "vpsravq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSRAVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x46, 0xca], "vpsravq zmm1, zmm0, zmm2"); // VPSRAVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x46, 0xca], "vpsravq zmm1{k5}, zmm0, zmm2"); // VPSRAVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x46, 0x0a], "vpsravq zmm1, zmm0, zmmword [rdx]"); // VPSRAVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x46, 0x0a], "vpsravq zmm1{k5}, zmm0, zmmword [rdx]"); // VPSRAVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x46, 0xca], "vpsravq xmm1{k5}{z}, xmm0, xmm2"); // VPSRAVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x46, 0x0a], "vpsravq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSRAVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x46, 0xca], "vpsravq xmm1, xmm0, xmm2"); // VPSRAVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x46, 0xca], "vpsravq xmm1{k5}, xmm0, xmm2"); // VPSRAVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x46, 0x0a], "vpsravq xmm1, xmm0, xmmword [rdx]"); // VPSRAVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x46, 0x0a], "vpsravq xmm1{k5}, xmm0, xmmword [rdx]"); // VPSRAVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x46, 0x0a], "vpsravd zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPSRAVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x46, 0x0a], "vpsravd zmm1, zmm0, dword [rdx]{1to16}"); // VPSRAVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x46, 0x0a], "vpsravd zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPSRAVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x46, 0x0a], "vpsravd xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPSRAVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x46, 0x0a], "vpsravd xmm1, xmm0, dword [rdx]{1to4}"); // VPSRAVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x46, 0x0a], "vpsravd xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPSRAVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x46, 0xca], "vpsravd zmm1{k5}{z}, zmm0, zmm2"); // VPSRAVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x46, 0x0a], "vpsravd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSRAVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x46, 0xca], "vpsravd zmm1, zmm0, zmm2"); // VPSRAVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x46, 0xca], "vpsravd zmm1{k5}, zmm0, zmm2"); // VPSRAVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x46, 0x0a], "vpsravd zmm1, zmm0, zmmword [rdx]"); // VPSRAVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x46, 0x0a], "vpsravd zmm1{k5}, zmm0, zmmword [rdx]"); // VPSRAVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x46, 0xca], "vpsravd xmm1{k5}{z}, xmm0, xmm2"); // VPSRAVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x46, 0x0a], "vpsravd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSRAVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x46, 0xca], "vpsravd xmm1, xmm0, xmm2"); // VPSRAVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x46, 0xca], "vpsravd xmm1{k5}, xmm0, xmm2"); // VPSRAVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x46, 0x0a], "vpsravd xmm1, xmm0, xmmword [rdx]"); // VPSRAVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x46, 0x0a], "vpsravd xmm1{k5}, xmm0, xmmword [rdx]"); // VPSRAVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x47, 0x0a], "vpsllvq ymm1{k5}{z}, ymm0, qword [rdx]{1to4}"); // VPSLLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x47, 0x0a], "vpsllvq ymm1, ymm0, qword [rdx]{1to4}"); // VPSLLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x47, 0x0a], "vpsllvq ymm1{k5}, ymm0, qword [rdx]{1to4}"); // VPSLLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x47, 0xca], "vpsllvq ymm1{k5}{z}, ymm0, ymm2"); // VPSLLVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x47, 0x0a], "vpsllvq ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSLLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x47, 0xca], "vpsllvq ymm1, ymm0, ymm2"); // VPSLLVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x47, 0xca], "vpsllvq ymm1{k5}, ymm0, ymm2"); // VPSLLVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x47, 0x0a], "vpsllvq ymm1, ymm0, ymmword [rdx]"); // VPSLLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x47, 0x0a], "vpsllvq ymm1{k5}, ymm0, ymmword [rdx]"); // VPSLLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x47, 0x0a], "vpsllvd ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPSLLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x47, 0x0a], "vpsllvd ymm1, ymm0, dword [rdx]{1to8}"); // VPSLLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x47, 0x0a], "vpsllvd ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPSLLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x47, 0xca], "vpsllvd ymm1{k5}{z}, ymm0, ymm2"); // VPSLLVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x47, 0x0a], "vpsllvd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPSLLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x47, 0xca], "vpsllvd ymm1, ymm0, ymm2"); // VPSLLVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x47, 0xca], "vpsllvd ymm1{k5}, ymm0, ymm2"); // VPSLLVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x47, 0x0a], "vpsllvd ymm1, ymm0, ymmword [rdx]"); // VPSLLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x47, 0x0a], "vpsllvd ymm1{k5}, ymm0, ymmword [rdx]"); // VPSLLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x47, 0x0a], "vpsllvq zmm1{k5}{z}, zmm0, qword [rdx]{1to8}"); // VPSLLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x47, 0x0a], "vpsllvq zmm1, zmm0, qword [rdx]{1to8}"); // VPSLLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x47, 0x0a], "vpsllvq zmm1{k5}, zmm0, qword [rdx]{1to8}"); // VPSLLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x47, 0x0a], "vpsllvq xmm1{k5}{z}, xmm0, qword [rdx]{1to2}"); // VPSLLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x47, 0x0a], "vpsllvq xmm1, xmm0, qword [rdx]{1to2}"); // VPSLLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x47, 0x0a], "vpsllvq xmm1{k5}, xmm0, qword [rdx]{1to2}"); // VPSLLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x47, 0xca], "vpsllvq zmm1{k5}{z}, zmm0, zmm2"); // VPSLLVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x47, 0x0a], "vpsllvq zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSLLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x47, 0xca], "vpsllvq zmm1, zmm0, zmm2"); // VPSLLVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x47, 0xca], "vpsllvq zmm1{k5}, zmm0, zmm2"); // VPSLLVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x47, 0x0a], "vpsllvq zmm1, zmm0, zmmword [rdx]"); // VPSLLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x47, 0x0a], "vpsllvq zmm1{k5}, zmm0, zmmword [rdx]"); // VPSLLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x47, 0xca], "vpsllvq xmm1{k5}{z}, xmm0, xmm2"); // VPSLLVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x47, 0x0a], "vpsllvq xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSLLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x47, 0xca], "vpsllvq xmm1, xmm0, xmm2"); // VPSLLVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x47, 0xca], "vpsllvq xmm1{k5}, xmm0, xmm2"); // VPSLLVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x47, 0x0a], "vpsllvq xmm1, xmm0, xmmword [rdx]"); // VPSLLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x47, 0x0a], "vpsllvq xmm1{k5}, xmm0, xmmword [rdx]"); // VPSLLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x47, 0x0a], "vpsllvd zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPSLLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x47, 0x0a], "vpsllvd zmm1, zmm0, dword [rdx]{1to16}"); // VPSLLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x47, 0x0a], "vpsllvd zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPSLLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x47, 0x0a], "vpsllvd xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPSLLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x47, 0x0a], "vpsllvd xmm1, xmm0, dword [rdx]{1to4}"); // VPSLLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x47, 0x0a], "vpsllvd xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPSLLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x47, 0xca], "vpsllvd zmm1{k5}{z}, zmm0, zmm2"); // VPSLLVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x47, 0x0a], "vpsllvd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPSLLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x47, 0xca], "vpsllvd zmm1, zmm0, zmm2"); // VPSLLVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x47, 0xca], "vpsllvd zmm1{k5}, zmm0, zmm2"); // VPSLLVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x47, 0x0a], "vpsllvd zmm1, zmm0, zmmword [rdx]"); // VPSLLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x47, 0x0a], "vpsllvd zmm1{k5}, zmm0, zmmword [rdx]"); // VPSLLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x47, 0xca], "vpsllvd xmm1{k5}{z}, xmm0, xmm2"); // VPSLLVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x47, 0x0a], "vpsllvd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPSLLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x47, 0xca], "vpsllvd xmm1, xmm0, xmm2"); // VPSLLVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x47, 0xca], "vpsllvd xmm1{k5}, xmm0, xmm2"); // VPSLLVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x47, 0x0a], "vpsllvd xmm1, xmm0, xmmword [rdx]"); // VPSLLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x47, 0x0a], "vpsllvd xmm1{k5}, xmm0, xmmword [rdx]"); // VPSLLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x4c, 0x0a], "vrcp14pd ymm1{k5}{z}, qword [rdx]{1to4}"); // VRCP14PD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x4c, 0x0a], "vrcp14pd ymm1, qword [rdx]{1to4}"); // VRCP14PD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x4c, 0x0a], "vrcp14pd ymm1{k5}, qword [rdx]{1to4}"); // VRCP14PD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4c, 0xca], "vrcp14pd ymm1{k5}{z}, ymm2"); // VRCP14PD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4c, 0x0a], "vrcp14pd ymm1{k5}{z}, ymmword [rdx]"); // VRCP14PD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4c, 0xca], "vrcp14pd ymm1, ymm2"); // VRCP14PD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4c, 0xca], "vrcp14pd ymm1{k5}, ymm2"); // VRCP14PD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4c, 0x0a], "vrcp14pd ymm1, ymmword [rdx]"); // VRCP14PD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4c, 0x0a], "vrcp14pd ymm1{k5}, ymmword [rdx]"); // VRCP14PD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x4c, 0x0a], "vrcp14ps ymm1{k5}{z}, dword [rdx]{1to8}"); // VRCP14PS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x4c, 0x0a], "vrcp14ps ymm1, dword [rdx]{1to8}"); // VRCP14PS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x4c, 0x0a], "vrcp14ps ymm1{k5}, dword [rdx]{1to8}"); // VRCP14PS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4c, 0xca], "vrcp14ps ymm1{k5}{z}, ymm2"); // VRCP14PS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4c, 0x0a], "vrcp14ps ymm1{k5}{z}, ymmword [rdx]"); // VRCP14PS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4c, 0xca], "vrcp14ps ymm1, ymm2"); // VRCP14PS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4c, 0xca], "vrcp14ps ymm1{k5}, ymm2"); // VRCP14PS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4c, 0x0a], "vrcp14ps ymm1, ymmword [rdx]"); // VRCP14PS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4c, 0x0a], "vrcp14ps ymm1{k5}, ymmword [rdx]"); // VRCP14PS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x4c, 0x0a], "vrcp14pd zmm1{k5}{z}, qword [rdx]{1to8}"); // VRCP14PD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x4c, 0x0a], "vrcp14pd zmm1, qword [rdx]{1to8}"); // VRCP14PD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x4c, 0x0a], "vrcp14pd zmm1{k5}, qword [rdx]{1to8}"); // VRCP14PD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x4c, 0x0a], "vrcp14pd xmm1{k5}{z}, qword [rdx]{1to2}"); // VRCP14PD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x4c, 0x0a], "vrcp14pd xmm1, qword [rdx]{1to2}"); // VRCP14PD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x4c, 0x0a], "vrcp14pd xmm1{k5}, qword [rdx]{1to2}"); // VRCP14PD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x4c, 0xca], "vrcp14pd zmm1{k5}{z}, zmm2"); // VRCP14PD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x4c, 0x0a], "vrcp14pd zmm1{k5}{z}, zmmword [rdx]"); // VRCP14PD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x4c, 0xca], "vrcp14pd zmm1, zmm2"); // VRCP14PD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x4c, 0xca], "vrcp14pd zmm1{k5}, zmm2"); // VRCP14PD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x4c, 0x0a], "vrcp14pd zmm1, zmmword [rdx]"); // VRCP14PD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x4c, 0x0a], "vrcp14pd zmm1{k5}, zmmword [rdx]"); // VRCP14PD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x4c, 0xca], "vrcp14pd xmm1{k5}{z}, xmm2"); // VRCP14PD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x4c, 0x0a], "vrcp14pd xmm1{k5}{z}, xmmword [rdx]"); // VRCP14PD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x4c, 0xca], "vrcp14pd xmm1, xmm2"); // VRCP14PD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x4c, 0xca], "vrcp14pd xmm1{k5}, xmm2"); // VRCP14PD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x4c, 0x0a], "vrcp14pd xmm1, xmmword [rdx]"); // VRCP14PD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x4c, 0x0a], "vrcp14pd xmm1{k5}, xmmword [rdx]"); // VRCP14PD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x4c, 0x0a], "vrcp14ps zmm1{k5}{z}, dword [rdx]{1to16}"); // VRCP14PS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x4c, 0x0a], "vrcp14ps zmm1, dword [rdx]{1to16}"); // VRCP14PS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x4c, 0x0a], "vrcp14ps zmm1{k5}, dword [rdx]{1to16}"); // VRCP14PS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x4c, 0x0a], "vrcp14ps xmm1{k5}{z}, dword [rdx]{1to4}"); // VRCP14PS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x4c, 0x0a], "vrcp14ps xmm1, dword [rdx]{1to4}"); // VRCP14PS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x4c, 0x0a], "vrcp14ps xmm1{k5}, dword [rdx]{1to4}"); // VRCP14PS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x4c, 0xca], "vrcp14ps zmm1{k5}{z}, zmm2"); // VRCP14PS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x4c, 0x0a], "vrcp14ps zmm1{k5}{z}, zmmword [rdx]"); // VRCP14PS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x4c, 0xca], "vrcp14ps zmm1, zmm2"); // VRCP14PS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x4c, 0xca], "vrcp14ps zmm1{k5}, zmm2"); // VRCP14PS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x4c, 0x0a], "vrcp14ps zmm1, zmmword [rdx]"); // VRCP14PS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x4c, 0x0a], "vrcp14ps zmm1{k5}, zmmword [rdx]"); // VRCP14PS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x4c, 0xca], "vrcp14ps xmm1{k5}{z}, xmm2"); // VRCP14PS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x4c, 0x0a], "vrcp14ps xmm1{k5}{z}, xmmword [rdx]"); // VRCP14PS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x4c, 0xca], "vrcp14ps xmm1, xmm2"); // VRCP14PS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x4c, 0xca], "vrcp14ps xmm1{k5}, xmm2"); // VRCP14PS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x4c, 0x0a], "vrcp14ps xmm1, xmmword [rdx]"); // VRCP14PS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x4c, 0x0a], "vrcp14ps xmm1{k5}, xmmword [rdx]"); // VRCP14PS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4d, 0xca], "vrcp14sd xmm1{k5}{z}, xmm0, xmm2"); // VRCP14SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4d, 0x0a], "vrcp14sd xmm1{k5}{z}, xmm0, qword [rdx]"); // VRCP14SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4d, 0xca], "vrcp14sd xmm1, xmm0, xmm2"); // VRCP14SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4d, 0xca], "vrcp14sd xmm1{k5}, xmm0, xmm2"); // VRCP14SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4d, 0x0a], "vrcp14sd xmm1, xmm0, qword [rdx]"); // VRCP14SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4d, 0x0a], "vrcp14sd xmm1{k5}, xmm0, qword [rdx]"); // VRCP14SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4d, 0xca], "vrcp14ss xmm1{k5}{z}, xmm0, xmm2"); // VRCP14SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4d, 0x0a], "vrcp14ss xmm1{k5}{z}, xmm0, dword [rdx]"); // VRCP14SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4d, 0xca], "vrcp14ss xmm1, xmm0, xmm2"); // VRCP14SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4d, 0xca], "vrcp14ss xmm1{k5}, xmm0, xmm2"); // VRCP14SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4d, 0x0a], "vrcp14ss xmm1, xmm0, dword [rdx]"); // VRCP14SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4d, 0x0a], "vrcp14ss xmm1{k5}, xmm0, dword [rdx]"); // VRCP14SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x4e, 0x0a], "vrsqrt14pd ymm1{k5}{z}, qword [rdx]{1to4}"); // VRSQRT14PD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x4e, 0x0a], "vrsqrt14pd ymm1, qword [rdx]{1to4}"); // VRSQRT14PD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x4e, 0x0a], "vrsqrt14pd ymm1{k5}, qword [rdx]{1to4}"); // VRSQRT14PD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4e, 0xca], "vrsqrt14pd ymm1{k5}{z}, ymm2"); // VRSQRT14PD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4e, 0x0a], "vrsqrt14pd ymm1{k5}{z}, ymmword [rdx]"); // VRSQRT14PD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4e, 0xca], "vrsqrt14pd ymm1, ymm2"); // VRSQRT14PD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4e, 0xca], "vrsqrt14pd ymm1{k5}, ymm2"); // VRSQRT14PD_YMMf64_MASKmskw_YMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4e, 0x0a], "vrsqrt14pd ymm1, ymmword [rdx]"); // VRSQRT14PD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4e, 0x0a], "vrsqrt14pd ymm1{k5}, ymmword [rdx]"); // VRSQRT14PD_YMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x4e, 0x0a], "vrsqrt14ps ymm1{k5}{z}, dword [rdx]{1to8}"); // VRSQRT14PS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x4e, 0x0a], "vrsqrt14ps ymm1, dword [rdx]{1to8}"); // VRSQRT14PS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x4e, 0x0a], "vrsqrt14ps ymm1{k5}, dword [rdx]{1to8}"); // VRSQRT14PS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4e, 0xca], "vrsqrt14ps ymm1{k5}{z}, ymm2"); // VRSQRT14PS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4e, 0x0a], "vrsqrt14ps ymm1{k5}{z}, ymmword [rdx]"); // VRSQRT14PS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4e, 0xca], "vrsqrt14ps ymm1, ymm2"); // VRSQRT14PS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4e, 0xca], "vrsqrt14ps ymm1{k5}, ymm2"); // VRSQRT14PS_YMMf32_MASKmskw_YMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4e, 0x0a], "vrsqrt14ps ymm1, ymmword [rdx]"); // VRSQRT14PS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4e, 0x0a], "vrsqrt14ps ymm1{k5}, ymmword [rdx]"); // VRSQRT14PS_YMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x4e, 0x0a], "vrsqrt14pd zmm1{k5}{z}, qword [rdx]{1to8}"); // VRSQRT14PD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x4e, 0x0a], "vrsqrt14pd zmm1, qword [rdx]{1to8}"); // VRSQRT14PD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x4e, 0x0a], "vrsqrt14pd zmm1{k5}, qword [rdx]{1to8}"); // VRSQRT14PD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x4e, 0x0a], "vrsqrt14pd xmm1{k5}{z}, qword [rdx]{1to2}"); // VRSQRT14PD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x4e, 0x0a], "vrsqrt14pd xmm1, qword [rdx]{1to2}"); // VRSQRT14PD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x4e, 0x0a], "vrsqrt14pd xmm1{k5}, qword [rdx]{1to2}"); // VRSQRT14PD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x4e, 0xca], "vrsqrt14pd zmm1{k5}{z}, zmm2"); // VRSQRT14PD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x4e, 0x0a], "vrsqrt14pd zmm1{k5}{z}, zmmword [rdx]"); // VRSQRT14PD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x4e, 0xca], "vrsqrt14pd zmm1, zmm2"); // VRSQRT14PD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x4e, 0xca], "vrsqrt14pd zmm1{k5}, zmm2"); // VRSQRT14PD_ZMMf64_MASKmskw_ZMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x4e, 0x0a], "vrsqrt14pd zmm1, zmmword [rdx]"); // VRSQRT14PD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x4e, 0x0a], "vrsqrt14pd zmm1{k5}, zmmword [rdx]"); // VRSQRT14PD_ZMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x4e, 0xca], "vrsqrt14pd xmm1{k5}{z}, xmm2"); // VRSQRT14PD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x4e, 0x0a], "vrsqrt14pd xmm1{k5}{z}, xmmword [rdx]"); // VRSQRT14PD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x4e, 0xca], "vrsqrt14pd xmm1, xmm2"); // VRSQRT14PD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x4e, 0xca], "vrsqrt14pd xmm1{k5}, xmm2"); // VRSQRT14PD_XMMf64_MASKmskw_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x4e, 0x0a], "vrsqrt14pd xmm1, xmmword [rdx]"); // VRSQRT14PD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x4e, 0x0a], "vrsqrt14pd xmm1{k5}, xmmword [rdx]"); // VRSQRT14PD_XMMf64_MASKmskw_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x4e, 0x0a], "vrsqrt14ps zmm1{k5}{z}, dword [rdx]{1to16}"); // VRSQRT14PS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x4e, 0x0a], "vrsqrt14ps zmm1, dword [rdx]{1to16}"); // VRSQRT14PS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x4e, 0x0a], "vrsqrt14ps zmm1{k5}, dword [rdx]{1to16}"); // VRSQRT14PS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x4e, 0x0a], "vrsqrt14ps xmm1{k5}{z}, dword [rdx]{1to4}"); // VRSQRT14PS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x4e, 0x0a], "vrsqrt14ps xmm1, dword [rdx]{1to4}"); // VRSQRT14PS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x4e, 0x0a], "vrsqrt14ps xmm1{k5}, dword [rdx]{1to4}"); // VRSQRT14PS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x4e, 0xca], "vrsqrt14ps zmm1{k5}{z}, zmm2"); // VRSQRT14PS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x4e, 0x0a], "vrsqrt14ps zmm1{k5}{z}, zmmword [rdx]"); // VRSQRT14PS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x4e, 0xca], "vrsqrt14ps zmm1, zmm2"); // VRSQRT14PS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x4e, 0xca], "vrsqrt14ps zmm1{k5}, zmm2"); // VRSQRT14PS_ZMMf32_MASKmskw_ZMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x4e, 0x0a], "vrsqrt14ps zmm1, zmmword [rdx]"); // VRSQRT14PS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x4e, 0x0a], "vrsqrt14ps zmm1{k5}, zmmword [rdx]"); // VRSQRT14PS_ZMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x4e, 0xca], "vrsqrt14ps xmm1{k5}{z}, xmm2"); // VRSQRT14PS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x4e, 0x0a], "vrsqrt14ps xmm1{k5}{z}, xmmword [rdx]"); // VRSQRT14PS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x4e, 0xca], "vrsqrt14ps xmm1, xmm2"); // VRSQRT14PS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x4e, 0xca], "vrsqrt14ps xmm1{k5}, xmm2"); // VRSQRT14PS_XMMf32_MASKmskw_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x4e, 0x0a], "vrsqrt14ps xmm1, xmmword [rdx]"); // VRSQRT14PS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x4e, 0x0a], "vrsqrt14ps xmm1{k5}, xmmword [rdx]"); // VRSQRT14PS_XMMf32_MASKmskw_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4f, 0xca], "vrsqrt14sd xmm1{k5}{z}, xmm0, xmm2"); // VRSQRT14SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x4f, 0x0a], "vrsqrt14sd xmm1{k5}{z}, xmm0, qword [rdx]"); // VRSQRT14SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4f, 0xca], "vrsqrt14sd xmm1, xmm0, xmm2"); // VRSQRT14SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4f, 0xca], "vrsqrt14sd xmm1{k5}, xmm0, xmm2"); // VRSQRT14SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x4f, 0x0a], "vrsqrt14sd xmm1, xmm0, qword [rdx]"); // VRSQRT14SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x4f, 0x0a], "vrsqrt14sd xmm1{k5}, xmm0, qword [rdx]"); // VRSQRT14SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4f, 0xca], "vrsqrt14ss xmm1{k5}{z}, xmm0, xmm2"); // VRSQRT14SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x4f, 0x0a], "vrsqrt14ss xmm1{k5}{z}, xmm0, dword [rdx]"); // VRSQRT14SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4f, 0xca], "vrsqrt14ss xmm1, xmm0, xmm2"); // VRSQRT14SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4f, 0xca], "vrsqrt14ss xmm1{k5}, xmm0, xmm2"); // VRSQRT14SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x4f, 0x0a], "vrsqrt14ss xmm1, xmm0, dword [rdx]"); // VRSQRT14SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x4f, 0x0a], "vrsqrt14ss xmm1{k5}, xmm0, dword [rdx]"); // VRSQRT14SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x50, 0x0a], "vpdpbusd ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPDPBUSD_YMMi32_MASKmskw_YMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x50, 0x0a], "vpdpbusd ymm1, ymm0, dword [rdx]{1to8}"); // VPDPBUSD_YMMi32_MASKmskw_YMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x50, 0x0a], "vpdpbusd ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPDPBUSD_YMMi32_MASKmskw_YMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x50, 0xca], "vpdpbusd ymm1{k5}{z}, ymm0, ymm2"); // VPDPBUSD_YMMi32_MASKmskw_YMMu8_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x50, 0x0a], "vpdpbusd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPDPBUSD_YMMi32_MASKmskw_YMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x50, 0xca], "vpdpbusd ymm1, ymm0, ymm2"); // VPDPBUSD_YMMi32_MASKmskw_YMMu8_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x50, 0xca], "vpdpbusd ymm1{k5}, ymm0, ymm2"); // VPDPBUSD_YMMi32_MASKmskw_YMMu8_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x50, 0x0a], "vpdpbusd ymm1, ymm0, ymmword [rdx]"); // VPDPBUSD_YMMi32_MASKmskw_YMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x50, 0x0a], "vpdpbusd ymm1{k5}, ymm0, ymmword [rdx]"); // VPDPBUSD_YMMi32_MASKmskw_YMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x50, 0x0a], "vpdpbusd zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPDPBUSD_ZMMi32_MASKmskw_ZMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x50, 0x0a], "vpdpbusd zmm1, zmm0, dword [rdx]{1to16}"); // VPDPBUSD_ZMMi32_MASKmskw_ZMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x50, 0x0a], "vpdpbusd zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPDPBUSD_ZMMi32_MASKmskw_ZMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x50, 0x0a], "vpdpbusd xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPDPBUSD_XMMi32_MASKmskw_XMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x50, 0x0a], "vpdpbusd xmm1, xmm0, dword [rdx]{1to4}"); // VPDPBUSD_XMMi32_MASKmskw_XMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x50, 0x0a], "vpdpbusd xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPDPBUSD_XMMi32_MASKmskw_XMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x50, 0xca], "vpdpbusd zmm1{k5}{z}, zmm0, zmm2"); // VPDPBUSD_ZMMi32_MASKmskw_ZMMu8_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x50, 0x0a], "vpdpbusd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPDPBUSD_ZMMi32_MASKmskw_ZMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x50, 0xca], "vpdpbusd zmm1, zmm0, zmm2"); // VPDPBUSD_ZMMi32_MASKmskw_ZMMu8_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x50, 0xca], "vpdpbusd zmm1{k5}, zmm0, zmm2"); // VPDPBUSD_ZMMi32_MASKmskw_ZMMu8_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x50, 0x0a], "vpdpbusd zmm1, zmm0, zmmword [rdx]"); // VPDPBUSD_ZMMi32_MASKmskw_ZMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x50, 0x0a], "vpdpbusd zmm1{k5}, zmm0, zmmword [rdx]"); // VPDPBUSD_ZMMi32_MASKmskw_ZMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x50, 0xca], "vpdpbusd xmm1{k5}{z}, xmm0, xmm2"); // VPDPBUSD_XMMi32_MASKmskw_XMMu8_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x50, 0x0a], "vpdpbusd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPDPBUSD_XMMi32_MASKmskw_XMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x50, 0xca], "vpdpbusd xmm1, xmm0, xmm2"); // VPDPBUSD_XMMi32_MASKmskw_XMMu8_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x50, 0xca], "vpdpbusd xmm1{k5}, xmm0, xmm2"); // VPDPBUSD_XMMi32_MASKmskw_XMMu8_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x50, 0x0a], "vpdpbusd xmm1, xmm0, xmmword [rdx]"); // VPDPBUSD_XMMi32_MASKmskw_XMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x50, 0x0a], "vpdpbusd xmm1{k5}, xmm0, xmmword [rdx]"); // VPDPBUSD_XMMi32_MASKmskw_XMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x51, 0x0a], "vpdpbusds ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPDPBUSDS_YMMi32_MASKmskw_YMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x51, 0x0a], "vpdpbusds ymm1, ymm0, dword [rdx]{1to8}"); // VPDPBUSDS_YMMi32_MASKmskw_YMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x51, 0x0a], "vpdpbusds ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPDPBUSDS_YMMi32_MASKmskw_YMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x51, 0xca], "vpdpbusds ymm1{k5}{z}, ymm0, ymm2"); // VPDPBUSDS_YMMi32_MASKmskw_YMMu8_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x51, 0x0a], "vpdpbusds ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPDPBUSDS_YMMi32_MASKmskw_YMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x51, 0xca], "vpdpbusds ymm1, ymm0, ymm2"); // VPDPBUSDS_YMMi32_MASKmskw_YMMu8_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x51, 0xca], "vpdpbusds ymm1{k5}, ymm0, ymm2"); // VPDPBUSDS_YMMi32_MASKmskw_YMMu8_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x51, 0x0a], "vpdpbusds ymm1, ymm0, ymmword [rdx]"); // VPDPBUSDS_YMMi32_MASKmskw_YMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x51, 0x0a], "vpdpbusds ymm1{k5}, ymm0, ymmword [rdx]"); // VPDPBUSDS_YMMi32_MASKmskw_YMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x51, 0x0a], "vpdpbusds zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPDPBUSDS_ZMMi32_MASKmskw_ZMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x51, 0x0a], "vpdpbusds zmm1, zmm0, dword [rdx]{1to16}"); // VPDPBUSDS_ZMMi32_MASKmskw_ZMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x51, 0x0a], "vpdpbusds zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPDPBUSDS_ZMMi32_MASKmskw_ZMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x51, 0x0a], "vpdpbusds xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPDPBUSDS_XMMi32_MASKmskw_XMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x51, 0x0a], "vpdpbusds xmm1, xmm0, dword [rdx]{1to4}"); // VPDPBUSDS_XMMi32_MASKmskw_XMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x51, 0x0a], "vpdpbusds xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPDPBUSDS_XMMi32_MASKmskw_XMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x51, 0xca], "vpdpbusds zmm1{k5}{z}, zmm0, zmm2"); // VPDPBUSDS_ZMMi32_MASKmskw_ZMMu8_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x51, 0x0a], "vpdpbusds zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPDPBUSDS_ZMMi32_MASKmskw_ZMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x51, 0xca], "vpdpbusds zmm1, zmm0, zmm2"); // VPDPBUSDS_ZMMi32_MASKmskw_ZMMu8_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x51, 0xca], "vpdpbusds zmm1{k5}, zmm0, zmm2"); // VPDPBUSDS_ZMMi32_MASKmskw_ZMMu8_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x51, 0x0a], "vpdpbusds zmm1, zmm0, zmmword [rdx]"); // VPDPBUSDS_ZMMi32_MASKmskw_ZMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x51, 0x0a], "vpdpbusds zmm1{k5}, zmm0, zmmword [rdx]"); // VPDPBUSDS_ZMMi32_MASKmskw_ZMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x51, 0xca], "vpdpbusds xmm1{k5}{z}, xmm0, xmm2"); // VPDPBUSDS_XMMi32_MASKmskw_XMMu8_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x51, 0x0a], "vpdpbusds xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPDPBUSDS_XMMi32_MASKmskw_XMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x51, 0xca], "vpdpbusds xmm1, xmm0, xmm2"); // VPDPBUSDS_XMMi32_MASKmskw_XMMu8_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x51, 0xca], "vpdpbusds xmm1{k5}, xmm0, xmm2"); // VPDPBUSDS_XMMi32_MASKmskw_XMMu8_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x51, 0x0a], "vpdpbusds xmm1, xmm0, xmmword [rdx]"); // VPDPBUSDS_XMMi32_MASKmskw_XMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x51, 0x0a], "vpdpbusds xmm1{k5}, xmm0, xmmword [rdx]"); // VPDPBUSDS_XMMi32_MASKmskw_XMMu8_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x52, 0x0a], "vpdpwssd ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPDPWSSD_YMMi32_MASKmskw_YMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x52, 0x0a], "vpdpwssd ymm1, ymm0, dword [rdx]{1to8}"); // VPDPWSSD_YMMi32_MASKmskw_YMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x52, 0x0a], "vpdpwssd ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPDPWSSD_YMMi32_MASKmskw_YMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x52, 0xca], "vpdpwssd ymm1{k5}{z}, ymm0, ymm2"); // VPDPWSSD_YMMi32_MASKmskw_YMMi16_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x52, 0x0a], "vpdpwssd ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPDPWSSD_YMMi32_MASKmskw_YMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x52, 0xca], "vpdpwssd ymm1, ymm0, ymm2"); // VPDPWSSD_YMMi32_MASKmskw_YMMi16_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x52, 0xca], "vpdpwssd ymm1{k5}, ymm0, ymm2"); // VPDPWSSD_YMMi32_MASKmskw_YMMi16_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x52, 0x0a], "vpdpwssd ymm1, ymm0, ymmword [rdx]"); // VPDPWSSD_YMMi32_MASKmskw_YMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x52, 0x0a], "vpdpwssd ymm1{k5}, ymm0, ymmword [rdx]"); // VPDPWSSD_YMMi32_MASKmskw_YMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x52, 0x0a], "vpdpwssd zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPDPWSSD_ZMMi32_MASKmskw_ZMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x52, 0x0a], "vpdpwssd zmm1, zmm0, dword [rdx]{1to16}"); // VPDPWSSD_ZMMi32_MASKmskw_ZMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x52, 0x0a], "vpdpwssd zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPDPWSSD_ZMMi32_MASKmskw_ZMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x52, 0x0a], "vpdpwssd xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPDPWSSD_XMMi32_MASKmskw_XMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x52, 0x0a], "vpdpwssd xmm1, xmm0, dword [rdx]{1to4}"); // VPDPWSSD_XMMi32_MASKmskw_XMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x52, 0x0a], "vpdpwssd xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPDPWSSD_XMMi32_MASKmskw_XMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x52, 0xca], "vpdpwssd zmm1{k5}{z}, zmm0, zmm2"); // VPDPWSSD_ZMMi32_MASKmskw_ZMMi16_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x52, 0x0a], "vpdpwssd zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPDPWSSD_ZMMi32_MASKmskw_ZMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x52, 0xca], "vpdpwssd zmm1, zmm0, zmm2"); // VPDPWSSD_ZMMi32_MASKmskw_ZMMi16_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x52, 0xca], "vpdpwssd zmm1{k5}, zmm0, zmm2"); // VPDPWSSD_ZMMi32_MASKmskw_ZMMi16_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x52, 0x0a], "vpdpwssd zmm1, zmm0, zmmword [rdx]"); // VPDPWSSD_ZMMi32_MASKmskw_ZMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x52, 0x0a], "vpdpwssd zmm1{k5}, zmm0, zmmword [rdx]"); // VPDPWSSD_ZMMi32_MASKmskw_ZMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x52, 0xca], "vpdpwssd xmm1{k5}{z}, xmm0, xmm2"); // VPDPWSSD_XMMi32_MASKmskw_XMMi16_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x52, 0x0a], "vpdpwssd xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPDPWSSD_XMMi32_MASKmskw_XMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x52, 0xca], "vpdpwssd xmm1, xmm0, xmm2"); // VPDPWSSD_XMMi32_MASKmskw_XMMi16_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x52, 0xca], "vpdpwssd xmm1{k5}, xmm0, xmm2"); // VPDPWSSD_XMMi32_MASKmskw_XMMi16_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x52, 0x0a], "vpdpwssd xmm1, xmm0, xmmword [rdx]"); // VPDPWSSD_XMMi32_MASKmskw_XMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x52, 0x0a], "vpdpwssd xmm1{k5}, xmm0, xmmword [rdx]"); // VPDPWSSD_XMMi32_MASKmskw_XMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x53, 0x0a], "vpdpwssds ymm1{k5}{z}, ymm0, dword [rdx]{1to8}"); // VPDPWSSDS_YMMi32_MASKmskw_YMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x53, 0x0a], "vpdpwssds ymm1, ymm0, dword [rdx]{1to8}"); // VPDPWSSDS_YMMi32_MASKmskw_YMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x53, 0x0a], "vpdpwssds ymm1{k5}, ymm0, dword [rdx]{1to8}"); // VPDPWSSDS_YMMi32_MASKmskw_YMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x53, 0xca], "vpdpwssds ymm1{k5}{z}, ymm0, ymm2"); // VPDPWSSDS_YMMi32_MASKmskw_YMMi16_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x53, 0x0a], "vpdpwssds ymm1{k5}{z}, ymm0, ymmword [rdx]"); // VPDPWSSDS_YMMi32_MASKmskw_YMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x53, 0xca], "vpdpwssds ymm1, ymm0, ymm2"); // VPDPWSSDS_YMMi32_MASKmskw_YMMi16_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x53, 0xca], "vpdpwssds ymm1{k5}, ymm0, ymm2"); // VPDPWSSDS_YMMi32_MASKmskw_YMMi16_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x53, 0x0a], "vpdpwssds ymm1, ymm0, ymmword [rdx]"); // VPDPWSSDS_YMMi32_MASKmskw_YMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x53, 0x0a], "vpdpwssds ymm1{k5}, ymm0, ymmword [rdx]"); // VPDPWSSDS_YMMi32_MASKmskw_YMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x53, 0x0a], "vpdpwssds zmm1{k5}{z}, zmm0, dword [rdx]{1to16}"); // VPDPWSSDS_ZMMi32_MASKmskw_ZMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x53, 0x0a], "vpdpwssds zmm1, zmm0, dword [rdx]{1to16}"); // VPDPWSSDS_ZMMi32_MASKmskw_ZMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x53, 0x0a], "vpdpwssds zmm1{k5}, zmm0, dword [rdx]{1to16}"); // VPDPWSSDS_ZMMi32_MASKmskw_ZMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x53, 0x0a], "vpdpwssds xmm1{k5}{z}, xmm0, dword [rdx]{1to4}"); // VPDPWSSDS_XMMi32_MASKmskw_XMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x53, 0x0a], "vpdpwssds xmm1, xmm0, dword [rdx]{1to4}"); // VPDPWSSDS_XMMi32_MASKmskw_XMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x53, 0x0a], "vpdpwssds xmm1{k5}, xmm0, dword [rdx]{1to4}"); // VPDPWSSDS_XMMi32_MASKmskw_XMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x53, 0xca], "vpdpwssds zmm1{k5}{z}, zmm0, zmm2"); // VPDPWSSDS_ZMMi32_MASKmskw_ZMMi16_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x53, 0x0a], "vpdpwssds zmm1{k5}{z}, zmm0, zmmword [rdx]"); // VPDPWSSDS_ZMMi32_MASKmskw_ZMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x53, 0xca], "vpdpwssds zmm1, zmm0, zmm2"); // VPDPWSSDS_ZMMi32_MASKmskw_ZMMi16_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x53, 0xca], "vpdpwssds zmm1{k5}, zmm0, zmm2"); // VPDPWSSDS_ZMMi32_MASKmskw_ZMMi16_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x53, 0x0a], "vpdpwssds zmm1, zmm0, zmmword [rdx]"); // VPDPWSSDS_ZMMi32_MASKmskw_ZMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x53, 0x0a], "vpdpwssds zmm1{k5}, zmm0, zmmword [rdx]"); // VPDPWSSDS_ZMMi32_MASKmskw_ZMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x53, 0xca], "vpdpwssds xmm1{k5}{z}, xmm0, xmm2"); // VPDPWSSDS_XMMi32_MASKmskw_XMMi16_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x53, 0x0a], "vpdpwssds xmm1{k5}{z}, xmm0, xmmword [rdx]"); // VPDPWSSDS_XMMi32_MASKmskw_XMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x53, 0xca], "vpdpwssds xmm1, xmm0, xmm2"); // VPDPWSSDS_XMMi32_MASKmskw_XMMi16_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x53, 0xca], "vpdpwssds xmm1{k5}, xmm0, xmm2"); // VPDPWSSDS_XMMi32_MASKmskw_XMMi16_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x53, 0x0a], "vpdpwssds xmm1, xmm0, xmmword [rdx]"); // VPDPWSSDS_XMMi32_MASKmskw_XMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x53, 0x0a], "vpdpwssds xmm1{k5}, xmm0, xmmword [rdx]"); // VPDPWSSDS_XMMi32_MASKmskw_XMMi16_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x54, 0xca], "vpopcntw ymm1{k5}{z}, ymm2"); // VPOPCNTW_YMMu16_MASKmskw_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x54, 0x0a], "vpopcntw ymm1{k5}{z}, ymmword [rdx]"); // VPOPCNTW_YMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x54, 0xca], "vpopcntw ymm1, ymm2"); // VPOPCNTW_YMMu16_MASKmskw_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x54, 0xca], "vpopcntw ymm1{k5}, ymm2"); // VPOPCNTW_YMMu16_MASKmskw_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x54, 0x0a], "vpopcntw ymm1, ymmword [rdx]"); // VPOPCNTW_YMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x54, 0x0a], "vpopcntw ymm1{k5}, ymmword [rdx]"); // VPOPCNTW_YMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x54, 0xca], "vpopcntb ymm1{k5}{z}, ymm2"); // VPOPCNTB_YMMu8_MASKmskw_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x54, 0x0a], "vpopcntb ymm1{k5}{z}, ymmword [rdx]"); // VPOPCNTB_YMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x54, 0xca], "vpopcntb ymm1, ymm2"); // VPOPCNTB_YMMu8_MASKmskw_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x54, 0xca], "vpopcntb ymm1{k5}, ymm2"); // VPOPCNTB_YMMu8_MASKmskw_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x54, 0x0a], "vpopcntb ymm1, ymmword [rdx]"); // VPOPCNTB_YMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x54, 0x0a], "vpopcntb ymm1{k5}, ymmword [rdx]"); // VPOPCNTB_YMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x54, 0xca], "vpopcntw zmm1{k5}{z}, zmm2"); // VPOPCNTW_ZMMu16_MASKmskw_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x54, 0x0a], "vpopcntw zmm1{k5}{z}, zmmword [rdx]"); // VPOPCNTW_ZMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x54, 0xca], "vpopcntw zmm1, zmm2"); // VPOPCNTW_ZMMu16_MASKmskw_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x54, 0xca], "vpopcntw zmm1{k5}, zmm2"); // VPOPCNTW_ZMMu16_MASKmskw_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x54, 0x0a], "vpopcntw zmm1, zmmword [rdx]"); // VPOPCNTW_ZMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x54, 0x0a], "vpopcntw zmm1{k5}, zmmword [rdx]"); // VPOPCNTW_ZMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x54, 0xca], "vpopcntw xmm1{k5}{z}, xmm2"); // VPOPCNTW_XMMu16_MASKmskw_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x54, 0x0a], "vpopcntw xmm1{k5}{z}, xmmword [rdx]"); // VPOPCNTW_XMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x54, 0xca], "vpopcntw xmm1, xmm2"); // VPOPCNTW_XMMu16_MASKmskw_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x54, 0xca], "vpopcntw xmm1{k5}, xmm2"); // VPOPCNTW_XMMu16_MASKmskw_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x54, 0x0a], "vpopcntw xmm1, xmmword [rdx]"); // VPOPCNTW_XMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x54, 0x0a], "vpopcntw xmm1{k5}, xmmword [rdx]"); // VPOPCNTW_XMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x54, 0xca], "vpopcntb zmm1{k5}{z}, zmm2"); // VPOPCNTB_ZMMu8_MASKmskw_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x54, 0x0a], "vpopcntb zmm1{k5}{z}, zmmword [rdx]"); // VPOPCNTB_ZMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x54, 0xca], "vpopcntb zmm1, zmm2"); // VPOPCNTB_ZMMu8_MASKmskw_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x54, 0xca], "vpopcntb zmm1{k5}, zmm2"); // VPOPCNTB_ZMMu8_MASKmskw_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x54, 0x0a], "vpopcntb zmm1, zmmword [rdx]"); // VPOPCNTB_ZMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x54, 0x0a], "vpopcntb zmm1{k5}, zmmword [rdx]"); // VPOPCNTB_ZMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x54, 0xca], "vpopcntb xmm1{k5}{z}, xmm2"); // VPOPCNTB_XMMu8_MASKmskw_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x54, 0x0a], "vpopcntb xmm1{k5}{z}, xmmword [rdx]"); // VPOPCNTB_XMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x54, 0xca], "vpopcntb xmm1, xmm2"); // VPOPCNTB_XMMu8_MASKmskw_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x54, 0xca], "vpopcntb xmm1{k5}, xmm2"); // VPOPCNTB_XMMu8_MASKmskw_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x54, 0x0a], "vpopcntb xmm1, xmmword [rdx]"); // VPOPCNTB_XMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x54, 0x0a], "vpopcntb xmm1{k5}, xmmword [rdx]"); // VPOPCNTB_XMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xbd, 0x55, 0x0a], "vpopcntq ymm1{k5}{z}, qword [rdx]{1to4}"); // VPOPCNTQ_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x38, 0x55, 0x0a], "vpopcntq ymm1, qword [rdx]{1to4}"); // VPOPCNTQ_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x3d, 0x55, 0x0a], "vpopcntq ymm1{k5}, qword [rdx]{1to4}"); // VPOPCNTQ_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x55, 0xca], "vpopcntq ymm1{k5}{z}, ymm2"); // VPOPCNTQ_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x55, 0x0a], "vpopcntq ymm1{k5}{z}, ymmword [rdx]"); // VPOPCNTQ_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x55, 0xca], "vpopcntq ymm1, ymm2"); // VPOPCNTQ_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x55, 0xca], "vpopcntq ymm1{k5}, ymm2"); // VPOPCNTQ_YMMu64_MASKmskw_YMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x55, 0x0a], "vpopcntq ymm1, ymmword [rdx]"); // VPOPCNTQ_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x55, 0x0a], "vpopcntq ymm1{k5}, ymmword [rdx]"); // VPOPCNTQ_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xbd, 0x55, 0x0a], "vpopcntd ymm1{k5}{z}, dword [rdx]{1to8}"); // VPOPCNTD_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x38, 0x55, 0x0a], "vpopcntd ymm1, dword [rdx]{1to8}"); // VPOPCNTD_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x3d, 0x55, 0x0a], "vpopcntd ymm1{k5}, dword [rdx]{1to8}"); // VPOPCNTD_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x55, 0xca], "vpopcntd ymm1{k5}{z}, ymm2"); // VPOPCNTD_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x55, 0x0a], "vpopcntd ymm1{k5}{z}, ymmword [rdx]"); // VPOPCNTD_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x55, 0xca], "vpopcntd ymm1, ymm2"); // VPOPCNTD_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x55, 0xca], "vpopcntd ymm1{k5}, ymm2"); // VPOPCNTD_YMMu32_MASKmskw_YMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x55, 0x0a], "vpopcntd ymm1, ymmword [rdx]"); // VPOPCNTD_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x55, 0x0a], "vpopcntd ymm1{k5}, ymmword [rdx]"); // VPOPCNTD_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xdd, 0x55, 0x0a], "vpopcntq zmm1{k5}{z}, qword [rdx]{1to8}"); // VPOPCNTQ_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x58, 0x55, 0x0a], "vpopcntq zmm1, qword [rdx]{1to8}"); // VPOPCNTQ_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x5d, 0x55, 0x0a], "vpopcntq zmm1{k5}, qword [rdx]{1to8}"); // VPOPCNTQ_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x9d, 0x55, 0x0a], "vpopcntq xmm1{k5}{z}, qword [rdx]{1to2}"); // VPOPCNTQ_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x18, 0x55, 0x0a], "vpopcntq xmm1, qword [rdx]{1to2}"); // VPOPCNTQ_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x1d, 0x55, 0x0a], "vpopcntq xmm1{k5}, qword [rdx]{1to2}"); // VPOPCNTQ_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x55, 0xca], "vpopcntq zmm1{k5}{z}, zmm2"); // VPOPCNTQ_ZMMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x55, 0x0a], "vpopcntq zmm1{k5}{z}, zmmword [rdx]"); // VPOPCNTQ_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x55, 0xca], "vpopcntq zmm1, zmm2"); // VPOPCNTQ_ZMMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x55, 0xca], "vpopcntq zmm1{k5}, zmm2"); // VPOPCNTQ_ZMMu64_MASKmskw_ZMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x55, 0x0a], "vpopcntq zmm1, zmmword [rdx]"); // VPOPCNTQ_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x55, 0x0a], "vpopcntq zmm1{k5}, zmmword [rdx]"); // VPOPCNTQ_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x55, 0xca], "vpopcntq xmm1{k5}{z}, xmm2"); // VPOPCNTQ_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x55, 0x0a], "vpopcntq xmm1{k5}{z}, xmmword [rdx]"); // VPOPCNTQ_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x55, 0xca], "vpopcntq xmm1, xmm2"); // VPOPCNTQ_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x55, 0xca], "vpopcntq xmm1{k5}, xmm2"); // VPOPCNTQ_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x55, 0x0a], "vpopcntq xmm1, xmmword [rdx]"); // VPOPCNTQ_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x55, 0x0a], "vpopcntq xmm1{k5}, xmmword [rdx]"); // VPOPCNTQ_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xdd, 0x55, 0x0a], "vpopcntd zmm1{k5}{z}, dword [rdx]{1to16}"); // VPOPCNTD_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x58, 0x55, 0x0a], "vpopcntd zmm1, dword [rdx]{1to16}"); // VPOPCNTD_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x5d, 0x55, 0x0a], "vpopcntd zmm1{k5}, dword [rdx]{1to16}"); // VPOPCNTD_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x9d, 0x55, 0x0a], "vpopcntd xmm1{k5}{z}, dword [rdx]{1to4}"); // VPOPCNTD_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x18, 0x55, 0x0a], "vpopcntd xmm1, dword [rdx]{1to4}"); // VPOPCNTD_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x1d, 0x55, 0x0a], "vpopcntd xmm1{k5}, dword [rdx]{1to4}"); // VPOPCNTD_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x55, 0xca], "vpopcntd zmm1{k5}{z}, zmm2"); // VPOPCNTD_ZMMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x55, 0x0a], "vpopcntd zmm1{k5}{z}, zmmword [rdx]"); // VPOPCNTD_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x55, 0xca], "vpopcntd zmm1, zmm2"); // VPOPCNTD_ZMMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x55, 0xca], "vpopcntd zmm1{k5}, zmm2"); // VPOPCNTD_ZMMu32_MASKmskw_ZMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x55, 0x0a], "vpopcntd zmm1, zmmword [rdx]"); // VPOPCNTD_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x55, 0x0a], "vpopcntd zmm1{k5}, zmmword [rdx]"); // VPOPCNTD_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x55, 0xca], "vpopcntd xmm1{k5}{z}, xmm2"); // VPOPCNTD_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x55, 0x0a], "vpopcntd xmm1{k5}{z}, xmmword [rdx]"); // VPOPCNTD_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x55, 0xca], "vpopcntd xmm1, xmm2"); // VPOPCNTD_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x55, 0xca], "vpopcntd xmm1{k5}, xmm2"); // VPOPCNTD_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x55, 0x0a], "vpopcntd xmm1, xmmword [rdx]"); // VPOPCNTD_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x55, 0x0a], "vpopcntd xmm1{k5}, xmmword [rdx]"); // VPOPCNTD_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x58, 0xca], "vpbroadcastd ymm1{k5}{z}, xmm2"); // VPBROADCASTD_YMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x58, 0x0a], "vpbroadcastd ymm1{k5}{z}, dword [rdx]"); // VPBROADCASTD_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x58, 0xca], "vpbroadcastd ymm1, xmm2"); // VPBROADCASTD_YMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x58, 0xca], "vpbroadcastd ymm1{k5}, xmm2"); // VPBROADCASTD_YMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x58, 0x0a], "vpbroadcastd ymm1, dword [rdx]"); // VPBROADCASTD_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x58, 0x0a], "vpbroadcastd ymm1{k5}, dword [rdx]"); // VPBROADCASTD_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x58, 0xca], "vpbroadcastd zmm1{k5}{z}, xmm2"); // VPBROADCASTD_ZMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x58, 0x0a], "vpbroadcastd zmm1{k5}{z}, dword [rdx]"); // VPBROADCASTD_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x58, 0xca], "vpbroadcastd zmm1, xmm2"); // VPBROADCASTD_ZMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x58, 0xca], "vpbroadcastd zmm1{k5}, xmm2"); // VPBROADCASTD_ZMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x58, 0x0a], "vpbroadcastd zmm1, dword [rdx]"); // VPBROADCASTD_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x58, 0x0a], "vpbroadcastd zmm1{k5}, dword [rdx]"); // VPBROADCASTD_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x58, 0xca], "vpbroadcastd xmm1{k5}{z}, xmm2"); // VPBROADCASTD_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x58, 0x0a], "vpbroadcastd xmm1{k5}{z}, dword [rdx]"); // VPBROADCASTD_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x58, 0xca], "vpbroadcastd xmm1, xmm2"); // VPBROADCASTD_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x58, 0xca], "vpbroadcastd xmm1{k5}, xmm2"); // VPBROADCASTD_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x58, 0x0a], "vpbroadcastd xmm1, dword [rdx]"); // VPBROADCASTD_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x58, 0x0a], "vpbroadcastd xmm1{k5}, dword [rdx]"); // VPBROADCASTD_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x59, 0xca], "vpbroadcastq ymm1{k5}{z}, xmm2"); // VPBROADCASTQ_YMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x59, 0x0a], "vpbroadcastq ymm1{k5}{z}, qword [rdx]"); // VPBROADCASTQ_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x59, 0xca], "vpbroadcastq ymm1, xmm2"); // VPBROADCASTQ_YMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x59, 0xca], "vpbroadcastq ymm1{k5}, xmm2"); // VPBROADCASTQ_YMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x59, 0x0a], "vpbroadcastq ymm1, qword [rdx]"); // VPBROADCASTQ_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x59, 0x0a], "vpbroadcastq ymm1{k5}, qword [rdx]"); // VPBROADCASTQ_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x59, 0xca], "vbroadcasti32x2 ymm1{k5}{z}, xmm2"); // VBROADCASTI32X2_YMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x59, 0x0a], "vbroadcasti32x2 ymm1{k5}{z}, qword [rdx]"); // VBROADCASTI32X2_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x59, 0xca], "vbroadcasti32x2 ymm1, xmm2"); // VBROADCASTI32X2_YMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x59, 0xca], "vbroadcasti32x2 ymm1{k5}, xmm2"); // VBROADCASTI32X2_YMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x59, 0x0a], "vbroadcasti32x2 ymm1, qword [rdx]"); // VBROADCASTI32X2_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x59, 0x0a], "vbroadcasti32x2 ymm1{k5}, qword [rdx]"); // VBROADCASTI32X2_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x59, 0xca], "vpbroadcastq zmm1{k5}{z}, xmm2"); // VPBROADCASTQ_ZMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x59, 0x0a], "vpbroadcastq zmm1{k5}{z}, qword [rdx]"); // VPBROADCASTQ_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x59, 0xca], "vpbroadcastq zmm1, xmm2"); // VPBROADCASTQ_ZMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x59, 0xca], "vpbroadcastq zmm1{k5}, xmm2"); // VPBROADCASTQ_ZMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x59, 0x0a], "vpbroadcastq zmm1, qword [rdx]"); // VPBROADCASTQ_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x59, 0x0a], "vpbroadcastq zmm1{k5}, qword [rdx]"); // VPBROADCASTQ_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x59, 0xca], "vpbroadcastq xmm1{k5}{z}, xmm2"); // VPBROADCASTQ_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x59, 0x0a], "vpbroadcastq xmm1{k5}{z}, qword [rdx]"); // VPBROADCASTQ_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x59, 0xca], "vpbroadcastq xmm1, xmm2"); // VPBROADCASTQ_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x59, 0xca], "vpbroadcastq xmm1{k5}, xmm2"); // VPBROADCASTQ_XMMu64_MASKmskw_XMMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x59, 0x0a], "vpbroadcastq xmm1, qword [rdx]"); // VPBROADCASTQ_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x59, 0x0a], "vpbroadcastq xmm1{k5}, qword [rdx]"); // VPBROADCASTQ_XMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x59, 0xca], "vbroadcasti32x2 zmm1{k5}{z}, xmm2"); // VBROADCASTI32X2_ZMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x59, 0x0a], "vbroadcasti32x2 zmm1{k5}{z}, qword [rdx]"); // VBROADCASTI32X2_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x59, 0xca], "vbroadcasti32x2 zmm1, xmm2"); // VBROADCASTI32X2_ZMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x59, 0xca], "vbroadcasti32x2 zmm1{k5}, xmm2"); // VBROADCASTI32X2_ZMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x59, 0x0a], "vbroadcasti32x2 zmm1, qword [rdx]"); // VBROADCASTI32X2_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x59, 0x0a], "vbroadcasti32x2 zmm1{k5}, qword [rdx]"); // VBROADCASTI32X2_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x59, 0xca], "vbroadcasti32x2 xmm1{k5}{z}, xmm2"); // VBROADCASTI32X2_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x59, 0x0a], "vbroadcasti32x2 xmm1{k5}{z}, qword [rdx]"); // VBROADCASTI32X2_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x59, 0xca], "vbroadcasti32x2 xmm1, xmm2"); // VBROADCASTI32X2_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x59, 0xca], "vbroadcasti32x2 xmm1{k5}, xmm2"); // VBROADCASTI32X2_XMMu32_MASKmskw_XMMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x59, 0x0a], "vbroadcasti32x2 xmm1, qword [rdx]"); // VBROADCASTI32X2_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x59, 0x0a], "vbroadcasti32x2 xmm1{k5}, qword [rdx]"); // VBROADCASTI32X2_XMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x5a, 0x0a], "vbroadcasti64x2 ymm1{k5}{z}, xmmword [rdx]"); // VBROADCASTI64X2_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x5a, 0x0a], "vbroadcasti64x2 ymm1, xmmword [rdx]"); // VBROADCASTI64X2_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x5a, 0x0a], "vbroadcasti64x2 ymm1{k5}, xmmword [rdx]"); // VBROADCASTI64X2_YMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x5a, 0x0a], "vbroadcasti32x4 ymm1{k5}{z}, xmmword [rdx]"); // VBROADCASTI32X4_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x5a, 0x0a], "vbroadcasti32x4 ymm1, xmmword [rdx]"); // VBROADCASTI32X4_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x5a, 0x0a], "vbroadcasti32x4 ymm1{k5}, xmmword [rdx]"); // VBROADCASTI32X4_YMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x5a, 0x0a], "vbroadcasti64x2 zmm1{k5}{z}, xmmword [rdx]"); // VBROADCASTI64X2_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x5a, 0x0a], "vbroadcasti64x2 zmm1, xmmword [rdx]"); // VBROADCASTI64X2_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x5a, 0x0a], "vbroadcasti64x2 zmm1{k5}, xmmword [rdx]"); // VBROADCASTI64X2_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x5a, 0x0a], "vbroadcasti32x4 zmm1{k5}{z}, xmmword [rdx]"); // VBROADCASTI32X4_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x5a, 0x0a], "vbroadcasti32x4 zmm1, xmmword [rdx]"); // VBROADCASTI32X4_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x5a, 0x0a], "vbroadcasti32x4 zmm1{k5}, xmmword [rdx]"); // VBROADCASTI32X4_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x5b, 0x0a], "vbroadcasti64x4 zmm1{k5}{z}, ymmword [rdx]"); // VBROADCASTI64X4_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x5b, 0x0a], "vbroadcasti64x4 zmm1, ymmword [rdx]"); // VBROADCASTI64X4_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x5b, 0x0a], "vbroadcasti64x4 zmm1{k5}, ymmword [rdx]"); // VBROADCASTI64X4_ZMMu64_MASKmskw_MEMu64_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x5b, 0x0a], "vbroadcasti32x8 zmm1{k5}{z}, ymmword [rdx]"); // VBROADCASTI32X8_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x5b, 0x0a], "vbroadcasti32x8 zmm1, ymmword [rdx]"); // VBROADCASTI32X8_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x5b, 0x0a], "vbroadcasti32x8 zmm1{k5}, ymmword [rdx]"); // VBROADCASTI32X8_ZMMu32_MASKmskw_MEMu32_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x62, 0xca], "vpexpandw ymm1{k5}{z}, ymm2"); // VPEXPANDW_YMMu16_MASKmskw_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xad, 0x62, 0x0a], "vpexpandw ymm1{k5}{z}, ymmword [rdx]"); // VPEXPANDW_YMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x62, 0xca], "vpexpandw ymm1, ymm2"); // VPEXPANDW_YMMu16_MASKmskw_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x62, 0xca], "vpexpandw ymm1{k5}, ymm2"); // VPEXPANDW_YMMu16_MASKmskw_YMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x28, 0x62, 0x0a], "vpexpandw ymm1, ymmword [rdx]"); // VPEXPANDW_YMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x2d, 0x62, 0x0a], "vpexpandw ymm1{k5}, ymmword [rdx]"); // VPEXPANDW_YMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x62, 0xca], "vpexpandb ymm1{k5}{z}, ymm2"); // VPEXPANDB_YMMu8_MASKmskw_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xad, 0x62, 0x0a], "vpexpandb ymm1{k5}{z}, ymmword [rdx]"); // VPEXPANDB_YMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x62, 0xca], "vpexpandb ymm1, ymm2"); // VPEXPANDB_YMMu8_MASKmskw_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x62, 0xca], "vpexpandb ymm1{k5}, ymm2"); // VPEXPANDB_YMMu8_MASKmskw_YMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x28, 0x62, 0x0a], "vpexpandb ymm1, ymmword [rdx]"); // VPEXPANDB_YMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x2d, 0x62, 0x0a], "vpexpandb ymm1{k5}, ymmword [rdx]"); // VPEXPANDB_YMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x62, 0xca], "vpexpandw zmm1{k5}{z}, zmm2"); // VPEXPANDW_ZMMu16_MASKmskw_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0xcd, 0x62, 0x0a], "vpexpandw zmm1{k5}{z}, zmmword [rdx]"); // VPEXPANDW_ZMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x62, 0xca], "vpexpandw zmm1, zmm2"); // VPEXPANDW_ZMMu16_MASKmskw_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x62, 0xca], "vpexpandw zmm1{k5}, zmm2"); // VPEXPANDW_ZMMu16_MASKmskw_ZMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x48, 0x62, 0x0a], "vpexpandw zmm1, zmmword [rdx]"); // VPEXPANDW_ZMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x4d, 0x62, 0x0a], "vpexpandw zmm1{k5}, zmmword [rdx]"); // VPEXPANDW_ZMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x62, 0xca], "vpexpandw xmm1{k5}{z}, xmm2"); // VPEXPANDW_XMMu16_MASKmskw_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x8d, 0x62, 0x0a], "vpexpandw xmm1{k5}{z}, xmmword [rdx]"); // VPEXPANDW_XMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x62, 0xca], "vpexpandw xmm1, xmm2"); // VPEXPANDW_XMMu16_MASKmskw_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x62, 0xca], "vpexpandw xmm1{k5}, xmm2"); // VPEXPANDW_XMMu16_MASKmskw_XMMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x08, 0x62, 0x0a], "vpexpandw xmm1, xmmword [rdx]"); // VPEXPANDW_XMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0xfd, 0x0d, 0x62, 0x0a], "vpexpandw xmm1{k5}, xmmword [rdx]"); // VPEXPANDW_XMMu16_MASKmskw_MEMu16_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x62, 0xca], "vpexpandb zmm1{k5}{z}, zmm2"); // VPEXPANDB_ZMMu8_MASKmskw_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0xcd, 0x62, 0x0a], "vpexpandb zmm1{k5}{z}, zmmword [rdx]"); // VPEXPANDB_ZMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x62, 0xca], "vpexpandb zmm1, zmm2"); // VPEXPANDB_ZMMu8_MASKmskw_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x62, 0xca], "vpexpandb zmm1{k5}, zmm2"); // VPEXPANDB_ZMMu8_MASKmskw_ZMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x48, 0x62, 0x0a], "vpexpandb zmm1, zmmword [rdx]"); // VPEXPANDB_ZMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x4d, 0x62, 0x0a], "vpexpandb zmm1{k5}, zmmword [rdx]"); // VPEXPANDB_ZMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x62, 0xca], "vpexpandb xmm1{k5}{z}, xmm2"); // VPEXPANDB_XMMu8_MASKmskw_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x8d, 0x62, 0x0a], "vpexpandb xmm1{k5}{z}, xmmword [rdx]"); // VPEXPANDB_XMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x62, 0xca], "vpexpandb xmm1, xmm2"); // VPEXPANDB_XMMu8_MASKmskw_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x62, 0xca], "vpexpandb xmm1{k5}, xmm2"); // VPEXPANDB_XMMu8_MASKmskw_XMMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x08, 0x62, 0x0a], "vpexpandb xmm1, xmmword [rdx]"); // VPEXPANDB_XMMu8_MASKmskw_MEMu8_AVX512, extension: AVX512EVEX
    test_avx_full(&[0x62, 0xf2, 0x7d, 0x0d, 0x62, 0x0a], "vpexpandb xmm1{k5}, xmmword [rdx]"); // VPE