//! SuperH disassembler with optional [yaxpeax](//git.iximeow.net/yaxpeax-arch/about) support.
//!
//! ## Examples
//!
//! To decode a byte stream with yaxpeax:
//! ```
//! # #[cfg(feature = "yaxpeax")]
//! # extern crate yaxpeax_superh;
//! # #[cfg(feature = "yaxpeax")]
//! # extern crate yaxpeax_arch;
//! # fn main() {
//! # #[cfg(feature = "yaxpeax")]
//! # {
//! # use yaxpeax_superh::{SuperHInstructionDecodeError, SuperHFloatRegister, SuperHInstruction, SuperHDecoder};
//! # use yaxpeax_arch::{Decoder, U8Reader};
//! assert_eq!(SuperHDecoder::SH4.decode(&mut U8Reader::new(&[0b0101_0101, 0b1100_1010])),
//!            Ok(SuperHInstruction::XorImm(0x55)));
//! assert_eq!(SuperHDecoder::SH4.decode(&mut U8Reader::new(&[0b0101_1100, 0b1111_1010])),
//!            Ok(SuperHInstruction::FMovS(SuperHFloatRegister::Fr5, SuperHFloatRegister::Fr10)));
//! assert_eq!(SuperHDecoder::SH4.decode(&mut U8Reader::new(&[0b0011_1000, 0b0000_0000])),
//!            Ok(SuperHInstruction::Ldtlb));
//! assert_eq!(SuperHDecoder::SH4.decode(&mut U8Reader::new(&[0b0000_0001, 0b0000_0000])),
//!            Err(SuperHInstructionDecodeError::NotFound));
//! # }
//! # }
//! ```
//!
//! The same thing on a MMU-less processor booted in big-endian with `FPSCR.SZ` set:
//! ```
//! # #[cfg(feature = "yaxpeax")]
//! # extern crate yaxpeax_superh;
//! # #[cfg(feature = "yaxpeax")]
//! # extern crate yaxpeax_arch;
//! # fn main() {
//! # #[cfg(feature = "yaxpeax")]
//! # {
//! # use self::yaxpeax_superh::{SuperHInstructionDecodeError, SuperHExtendedDoubleRegister, SuperHDoubleRegister,
//! #                            SuperHInstruction, SuperHFeature, SuperHDecoder};
//! # use self::yaxpeax_arch::{Decoder, U8Reader};
//! let cpu = SuperHDecoder {
//!     little_endian: false,
//!     fpscr_sz: true,
//!     features: &[SuperHFeature::FPU],
//!     ..SuperHDecoder::SH4
//! };
//!
//! assert_eq!(cpu.decode(&mut U8Reader::new(&[0b1100_1010, 0b0101_0101])),
//!            Ok(SuperHInstruction::XorImm(0x55)));
//! assert_eq!(cpu.decode(&mut U8Reader::new(&[0b1111_1010, 0b0101_1100])),
//!            Ok(SuperHInstruction::FMovFromX(SuperHExtendedDoubleRegister::Xd4,
//!                                            SuperHDoubleRegister::Dr10)));
//! assert_eq!(cpu.decode(&mut U8Reader::new(&[0b0000_0000, 0b0011_1000])),
//!            Err(SuperHInstructionDecodeError::NotFound));
//! assert_eq!(cpu.decode(&mut U8Reader::new(&[0b0000_0000, 0b0000_0001])),
//!            Err(SuperHInstructionDecodeError::NotFound));
//! # }
//! # }
//! ```
//!
//! Simple instruction parsing is also available without yaxpeax:
//! ```
//! # use yaxpeax_superh::{SuperHExtendedDoubleRegister, SuperHDoubleRegister, SuperHInstruction};
//! assert_eq!(SuperHInstruction::parse(0b1100_1010_0101_0101, true),
//!            Some(SuperHInstruction::XorImm(0x55)));
//! assert_eq!(SuperHInstruction::parse(0b1111_1010_0101_1100, true),
//!            Some(SuperHInstruction::FMovFromX(SuperHExtendedDoubleRegister::Xd4,
//!                                              SuperHDoubleRegister::Dr10)));
//! assert_eq!(SuperHInstruction::parse(0b0000_0000_0011_1000, true),
//!            Some(SuperHInstruction::Ldtlb));
//! assert_eq!(SuperHInstruction::parse(0b0000_0000_0000_0001, true),
//!            None);
//! ```
//!
//! ## Normative documents
//!
//! [SH-2](https://antime.kapsi.fi/sega/files/h12p0.pdf), and by extension SH-1, since SH-2-only instructions are marked
//! accordingly there.
//!
//! [SH-4](http://www.st.com/st-web-ui/static/active/en/resource/technical/document/user_manual/CD00147165.pdf)
//! (a.k.a. "cd00147165-sh-4-32-bit-cpu-core-architecture-stmicroelectronics.pdf")
//!
//! [J-core](https://j-core.org/), specifically the bit about `SHAD`/`SHAL` SH3 barrel shift ops and
//! [`CAS.L`](https://github.com/j-core/jcore-cpu/blob/00f398e/decode/decode_table_simple.vhd#L1919).
//!
//! SH-5, for which [no hardware has shipped](https://en.wikipedia.org/wiki/SuperH#SH-5),
//! was [dropped](https://gcc.gnu.org/gcc-7/changes.html) in GCC 7,
//! the link from the [press release](http://www.bsdnewsletter.com/2002/10/News47.html) is dead,
//! and the [documentation](http://lars.nocrew.org/computers/processors/SuperH/) kinda blows;
//! no attempt is made to support it.
//!
//! According to [this J-core post](https://lists.j-core.org/pipermail/j-core/2017-May/000595.html), SH-DSP is incompatible
//! with newer revisions; no attempt is made to support it.
//!
//! Uncertainty so as to ISA revisions was resolved according to [this page](http://www.shared-ptr.com/sh_insns.html).
//!
//! ## Special thanks
//!
//! To all who support further development on [Patreon](https://patreon.com/nabijaczleweli), in particular:
//!
//!   * ThePhD
//!   * Embark Studios


#![no_std]


#[cfg(feature = "yaxpeax")]
extern crate yaxpeax_arch;
#[cfg(feature = "std")]
extern crate std;

use core::fmt;
use core::convert::TryFrom;
#[cfg(feature = "yaxpeax")]
use core::marker::PhantomData;
#[cfg(feature = "yaxpeax")]
use yaxpeax_arch::{AddressDiff, Arch, DecodeError};


#[derive(Debug, Copy, Clone, Hash, PartialEq, Eq, PartialOrd, Ord)]
pub enum SuperHInstruction {
    // http://www.st.com/st-web-ui/static/active/en/resource/technical/document/user_manual/CD00147165.pdf Table 39 "Fixed-point transfer instructions"
    // pp. 164-166
    // https://antime.kapsi.fi/sega/files/h12p0.pdf Table 5.3 "Data Transfer Instructions" pp. 40(44)-41(45)
    /// MOV #imm,Rn (`imm` → sign extension → `Rn`) `1110nnnniiiiiiii`
    MovImm(u8, SuperHRegister),
    /// MOV.W @(disp,PC),Rn (`(disp * 2 + PC + 4)` → sign extension → `Rn`) `1001nnnndddddddd`
    MovImmW(Displacement8, SuperHRegister),
    /// MOV.L @(disp,PC),Rn (`(disp * 2 + PC & 0xFFFF_FFFC + 4)` → `Rn`) `1101nnnndddddddd`
    MovImmL(Displacement8, SuperHRegister),

    /// MOV Rm,Rn (`Rm` → `Rn`) `0110nnnnmmmm0011`
    MovReg(SuperHRegister, SuperHRegister),

    /// MOV.B Rm,@Rn (`Rm` → sign extension → `(Rn)`) `0010nnnnmmmm0000`
    MovToAtRegB(SuperHRegister, SuperHRegister),
    /// MOV.W Rm,@Rn (`Rm` → sign extension → `(Rn)`) `0010nnnnmmmm0001`
    MovToAtRegW(SuperHRegister, SuperHRegister),
    /// MOV.L Rm,@Rn (`Rm` → `(Rn)`) `0010nnnnmmmm0010`
    MovToAtRegL(SuperHRegister, SuperHRegister),

    /// MOV.B @Rm,Rn (`(Rm)` → sign extension → `Rn`) `0110nnnnmmmm0000`
    MovFromAtRegB(SuperHRegister, SuperHRegister),
    /// MOV.W @Rm,Rn (`(Rm)` → sign extension → `Rn`) `0110nnnnmmmm0001`
    MovFromAtRegW(SuperHRegister, SuperHRegister),
    /// MOV.L @Rm,Rn (`(Rm)` → `Rn`) `0110nnnnmmmm0010`
    MovFromAtRegL(SuperHRegister, SuperHRegister),

    /// MOV.B Rm,@-Rn (`Rn - 1` → `Rn`, `Rm` → `(Rn)`) `0010nnnnmmmm0100`
    MovToAtRegPreDecrementB(SuperHRegister, SuperHRegister),
    /// MOV.W Rm,@-Rn (`Rn - 2` → `Rn`, `Rm` → `(Rn)`) `0010nnnnmmmm0101`
    MovToAtRegPreDecrementW(SuperHRegister, SuperHRegister),
    /// MOV.L Rm,@-Rn (`Rn - 4` → `Rn`, `Rm` → `(Rn)`) `0010nnnnmmmm0110`
    MovToAtRegPreDecrementL(SuperHRegister, SuperHRegister),

    /// MOV.B @Rm+,Rn (`(Rm)` → sign extension → `Rn`, `Rm + 1` → `Rm`) `0110nnnnmmmm0100`
    MovFromAtRegPostIncrementB(SuperHRegister, SuperHRegister),
    /// MOV.W @Rm+,Rn (`(Rm)` → sign extension → `Rn`, `Rm + 2` → `Rm`) `0110nnnnmmmm0101`
    MovFromAtRegPostIncrementW(SuperHRegister, SuperHRegister),
    /// MOV.L @Rm+,Rn (`(Rm)` → sign extension → `Rn`, `Rm + 4` → `Rm`) `0110nnnnmmmm0110`
    MovFromAtRegPostIncrementL(SuperHRegister, SuperHRegister),

    // ^ page 164; page 165 v
    /// MOV.B R0,@(disp,Rn) (`R0` → `(disp + Rn)`) `10000000nnnndddd`
    MovToAtDisplacedRegisterB(Displacement4, SuperHRegister),
    /// MOV.W R0,@(disp,Rn) (`R0` → `(disp * 2 + Rn)`) `10000001nnnndddd`
    MovToAtDisplacedRegisterW(Displacement4, SuperHRegister),
    /// MOV.L Rm,@(disp,Rn) (`Rm` → `(disp * 4 + Rn)`) `0001nnnnmmmmdddd`
    MovToAtDisplacedRegisterL(SuperHRegister, Displacement4, SuperHRegister),

    /// MOV.B @(disp,Rm),R0 (`(disp + Rm)` → sign extension → `R0`) `10000100mmmmdddd`
    MovFromAtDisplacedRegisterB(Displacement4, SuperHRegister),
    /// MOV.W @(disp,Rm),R0 (`(disp * 2 + Rm)` → sign extension → `R0`) `10000101mmmmdddd`
    MovFromAtDisplacedRegisterW(Displacement4, SuperHRegister),
    /// MOV.L @(disp,Rm),Rn (`(disp * 4 + Rm)` → `Rn`) `0101nnnnmmmmdddd`
    MovFromAtDisplacedRegisterL(Displacement4, SuperHRegister, SuperHRegister),

    /// MOV.B Rm,@(R0,Rn) (`Rm` → `(R0 + Rn)`) `0000nnnnmmmm0100`
    MovToAtIndexedRegisterB(SuperHRegister, SuperHRegister),
    /// MOV.W Rm,@(R0,Rn) (`Rm` → `(R0 + Rn)`) `0000nnnnmmmm0101`
    MovToAtIndexedRegisterW(SuperHRegister, SuperHRegister),
    // ^ page 40(44); page 41(45) v
    /// MOV.L Rm,@(R0,Rn) (`Rm` → `(R0 + Rn)`) `0000nnnnmmmm0110`
    MovToAtIndexedRegisterL(SuperHRegister, SuperHRegister),

    /// MOV.B @(R0,Rm),Rn (`(R0 + Rm)` → sign extension → `Rn`) `0000nnnnmmmm1100`
    MovFromAtIndexedRegisterB(SuperHRegister, SuperHRegister),
    /// MOV.W @(R0,Rm),Rn (`(R0 + Rm)` → sign extension → `Rn`) `0000nnnnmmmm1101`
    MovFromAtIndexedRegisterW(SuperHRegister, SuperHRegister),
    /// MOV.L @(R0,Rm),Rn (`(R0 + Rm)` → `Rn`) `0000nnnnmmmm1110`
    MovFromAtIndexedRegisterL(SuperHRegister, SuperHRegister),

    /// MOV.B R0,@(disp,GBR) (`R0` → `(disp + GBR)`) `11000000dddddddd`
    MovToAtDisplacedGbrB(Displacement8),
    /// MOV.W R0,@(disp,GBR) (`R0` → `(disp * 2 + GBR)`) `11000001dddddddd`
    MovToAtDisplacedGbrW(Displacement8),
    /// MOV.L R0,@(disp,GBR) (`R0` → `(disp * 4 + GBR)`) `11000010dddddddd`
    MovToAtDisplacedGbrL(Displacement8),

    /// MOV.B @(disp,GBR),R0 (`(disp + GBR)` → sign extension →  `R0`) `11000100dddddddd`
    MovFromAtDisplacedGbrB(Displacement8),
    /// MOV.W @(disp,GBR),R0 (`(disp * 2 + GBR)` → sign extension →  `R0`) `11000101dddddddd`
    MovFromAtDisplacedGbrW(Displacement8),
    /// MOV.L @(disp,GBR),R0 (`(disp * 4 + GBR)` →  `R0`) `11000110dddddddd`
    MovFromAtDisplacedGbrL(Displacement8),

    /// MOVA @(disp,PC),R0 (`disp * 4 + PC & 0xFFFF_FFFC + 4` → `R0`) `11000111dddddddd`
    Mova(Displacement8),

    /// MOVT Rn (`T` → `Rn`) `0000nnnn00101001`
    Movt(SuperHRegister),

    // ^ page 165; page 166 v
    /// SWAP.B Rm,Rn (`Rm` → swap lower 2 bytes → `REG`) `0110nnnnmmmm1000`
    SwapB(SuperHRegister, SuperHRegister),
    /// SWAP.W Rm,Rn (`Rm` → swap upper/lower words → `REG`) `0110nnnnmmmm1001`
    SwapW(SuperHRegister, SuperHRegister),

    /// XTRCT Rm,Rn (`Rm:Rn` middle 32 bits → Rn) `0010nnnnmmmm1101`
    Xtrct(SuperHRegister, SuperHRegister),

    // http://www.st.com/st-web-ui/static/active/en/resource/technical/document/user_manual/CD00147165.pdf Table 40 "Arithmetic operation instructions"
    // pp. 166-168
    // ^ page 41(45); page 42(46) v
    // https://antime.kapsi.fi/sega/files/h12p0.pdf Table 5.4 "Arithmetic Instructions" pp. 42(46)-44(48)
    /// ADD Rm,Rn (`Rn + Rm` → `Rn`) `0011nnnnmmmm1100`
    AddReg(SuperHRegister, SuperHRegister),
    /// ADD #imm,Rn (`Rn + imm` → `Rn`) `0111nnnniiiiiiii`
    AddImm(u8, SuperHRegister),
    /// ADDC Rm,Rn (`Rn + Rm + T` → `Rn`, carry → `T`) `0011nnnnmmmm1110`
    /// T = Carry
    AddC(SuperHRegister, SuperHRegister),
    /// ADDV Rm,Rn (`Rn + Rm` → `Rn`, overflow → `T`) `0011nnnnmmmm1111`
    /// T = Overflow
    AddV(SuperHRegister, SuperHRegister),

    /// CMP/EQ #imm,R0 (When `R0` = `imm`, 1 → `T`; Otherwise, 0 → `T`) `10001000iiiiiiii`
    /// T = Comparison result
    CmpEqImm(u8),
    /// CMP/EQ Rm,Rn (When `Rn` = `Rm`, 1 → `T`; Otherwise, 0 → `T`) `0011nnnnmmmm0000`
    /// T = Comparison result
    CmpEqReg(SuperHRegister, SuperHRegister),
    /// CMP/HS Rm,Rn (When `Rn` ≥ `Rm` (unsigned), 1 → `T`; Otherwise, 0 → `T`) `0011nnnnmmmm0010`
    /// T = Comparison result
    CmpHs(SuperHRegister, SuperHRegister),
    /// CMP/GE Rm,Rn (When `Rn` ≥ `Rm` (signed), 1 → `T`; Otherwise, 0 → `T`) `0011nnnnmmmm0011`
    /// T = Comparison result
    CmpGe(SuperHRegister, SuperHRegister),
    /// CMP/HI Rm,Rn (When `Rn` > `Rm` (unsigned), 1 → `T`; Otherwise, 0 → `T`) `0011nnnnmmmm0110`
    /// T = Comparison result
    CmpHi(SuperHRegister, SuperHRegister),
    /// CMP/GT Rm,Rn (When `Rn` > `Rm` (signed), 1 → `T`; Otherwise, 0 → `T`) `0011nnnnmmmm0111`
    /// T = Comparison result
    CmpGt(SuperHRegister, SuperHRegister),

    // ^ page 166; page 167 v
    /// CMP/PZ Rn (When `Rn` ≥ 0, 1 → `T`; Otherwise, 0 → `T`) `0100nnnn00010001`
    /// T = Comparison result
    CmpPz(SuperHRegister),
    /// CMP/PL Rn (When `Rn` > 0, 1 → `T`; Otherwise, 0 → `T`) `0100nnnn00010101`
    /// T = Comparison result
    CmpPl(SuperHRegister),
    /// CMP/STR Rm,Rn (When any bytes are equal, 1 → `T`; Otherwise, 0 → `T`) `0010nnnnmmmm1100`
    /// T = Comparison result
    CmpStr(SuperHRegister, SuperHRegister),

    /// DIV1 Rm,Rn (1-step division (`Rn ÷ Rm`)) `0011nnnnmmmm0100`
    /// T = Calculation result
    Div1(SuperHRegister, SuperHRegister),
    /// DIV0S Rm,Rn (MSB of `Rn` → `Q`, MSB of `Rm` → `M`, `M ^ Q` → `T`) `0010nnnnmmmm0111`
    /// T = Calculation result
    Div0S(SuperHRegister, SuperHRegister),
    /// DIV0U (0 → `M`/`Q`/`T`) `0000000000011001`
    /// T = 0
    Div0U,

    // ^ page 42(46); page 43(47) v
    /// DMULS.L Rm,Rn (Signed, `Rn * Rm` → `MAC`, 32 * 32 → 64 bits) `0011nnnnmmmm1101`
    ///
    /// Level: Sh2
    DmulsL(SuperHRegister, SuperHRegister),
    /// DMULU.L Rm,Rn (Unsigned, `Rn * Rm` → `MAC`, 32 * 32 → 64 bits) `0011nnnnmmmm0101`
    ///
    /// Level: Sh2
    DmuluL(SuperHRegister, SuperHRegister),

    /// DT Rn (`Rn - 1` → `Rn`; When `Rn` = 0, 1 → T; Otherwise `Rn` ≠ 0, 0 → `T`) `0100nnnn00010000`
    /// T = Comparison result
    ///
    /// Level: Sh2
    Dt(SuperHRegister),

    /// EXTS.B Rm,Rn (`Rm` sign-extended from byte → `Rn`) `0110nnnnmmmm1110`
    ExtsB(SuperHRegister, SuperHRegister),
    /// EXTS.W Rm,Rn (`Rm` sign-extended from word → `Rn`) `0110nnnnmmmm1111`
    ExtsW(SuperHRegister, SuperHRegister),

    /// EXTU.B Rm,Rn (`Rm` zero-extended from byte → `Rn`) `0110nnnnmmmm1100`
    ExtuB(SuperHRegister, SuperHRegister),
    /// EXTU.W Rm,Rn (`Rm` zero-extended from word → `Rn`) `0110nnnnmmmm1101`
    ExtuW(SuperHRegister, SuperHRegister),

    /// MAC.L @Rm+,@Rn+ (Signed, `(Rn) * (Rm) + MAC` → `MAC`, `Rn + 4` → `Rn`, `Rm + 4` → Rm`; 32 * 32 + 64 → 64 bits)
    /// `0000nnnnmmmm1111`
    ///
    /// Level: Sh2
    MacL(SuperHRegister, SuperHRegister),
    // ^ page 167; page 168 v
    /// MAC.L @Rm+,@Rn+ (Signed, `(Rn) * (Rm) + MAC` → `MAC`, `Rn + 2` → `Rn`, `Rm + 2` → Rm`; 16 * 16 + 64 → 64 bits)
    /// `0100nnnnmmmm1111`
    MacW(SuperHRegister, SuperHRegister),

    /// MAC.L Rm,Rn (`Rn * Rm` → `MACL`, 32 * 32→ 32 bits) `0000nnnnmmmm0111`
    ///
    /// Level: Sh2
    MulL(SuperHRegister, SuperHRegister),

    /// MULS.W Rm,Rn (Signed, `Rn * Rm` → `MACL`, 16 * 16 → 32 bits) `0010nnnnmmmm1111`
    MulsW(SuperHRegister, SuperHRegister),
    // ^ page 43(47); page 44(48) v
    /// MULU.W Rm,Rn (Unsigned, `Rn * Rm` → `MACL`, 16 * 16 → 32 bits) `0010nnnnmmmm1110`
    MuluW(SuperHRegister, SuperHRegister),

    /// NEG Rm,Rn (`0 - Rm` → Rn) `0110nnnnmmmm1011`
    Neg(SuperHRegister, SuperHRegister),
    /// NEGC Rm,Rn (`0 - Rm - T` → `Rn`, borrow → `T`) `0110nnnnmmmm1010`
    /// T = Borrow
    Negc(SuperHRegister, SuperHRegister),

    /// SUB Rm,Rn (`Rn - Rm` → Rn) `0011nnnnmmmm1000`
    Sub(SuperHRegister, SuperHRegister),
    /// SUBC Rm,Rn (`Rn - Rm - T` → `Rn`, borrow → `T`) `0011nnnnmmmm1010`
    /// T = Borrow
    Subc(SuperHRegister, SuperHRegister),
    /// SUBV Rm,Rn (`Rn - Rm` → `Rn`, underflow → `T`) `0011nnnnmmmm1011`
    /// T = Underflow
    Subv(SuperHRegister, SuperHRegister),

    // ^ page 168; page 169 v
    // http://www.st.com/st-web-ui/static/active/en/resource/technical/document/user_manual/CD00147165.pdf Table 41 "Logic operation instructions" p. 169
    // https://antime.kapsi.fi/sega/files/h12p0.pdf Table 5.5 "Logic Operation Instructions" pp. 44(48)-45(49)
    /// AND Rm,Rn (`Rn & Rm` → `Rn`) `0010nnnnmmmm1001`
    AndReg(SuperHRegister, SuperHRegister),
    /// AND #imm,R0 (`R0 & imm` → `R0`) `11001001iiiiiiii`
    AndImm(u8),
    /// AND.B #imm,@(R0,GBR) (`(R0 + GBR) & imm` → `(R0 + GBR)`) `11001101iiiiiiii`
    AndB(u8),

    /// NOT Rm,Rn (`~Rm` → `Rn`) `0110nnnnmmmm0111`
    Not(SuperHRegister, SuperHRegister),

    /// OR Rm,Rn (`Rn | Rm` → `Rn`) `0010nnnnmmmm1011`
    OrReg(SuperHRegister, SuperHRegister),
    /// OR #imm,R0 (`R0 | imm` → `R0`) `11001011iiiiiiii`
    OrImm(u8),
    /// OR.B #imm,@(R0,GBR) (`(R0 + GBR) | imm` → `(R0 + GBR)`) `11001111iiiiiiii`
    OrB(u8),

    /// TAS.B @Rn (When `(Rn)` = 0, 1 → `T`; Otherwise, 0 → `T`; In both cases, 1 → MSB of `(Rn)`) `0100nnnn00011011`
    /// T = Test result
    TasB(SuperHRegister),

    /// TST Rm,Rn (`Rn & Rm`; when result = 0, 1 → `T`; Otherwise, 0 → `T`) `0010nnnnmmmm1000`
    /// T = Test result
    TstReg(SuperHRegister, SuperHRegister),
    /// TST #imm,R0 (`R0 & imm`; when result = 0, 1 → `T`; Otherwise, 0 → `T`) `11001000iiiiiiii`
    /// T = Test result
    TstImm(u8),
    // ^ page 44(48); page 45(49) v
    /// TST.B #imm,@(R0,GBR) (`(R0 + GBR) & imm`, When result = 0, 1 → `T`; Otherwise, 0 → `T`) `11001100iiiiiiii`
    /// T = Test result
    TstB(u8),

    /// XOR Rm,Rn (`Rn ^ Rm → Rn`) `0010nnnnmmmm1010`
    /// T = Test result
    XorReg(SuperHRegister, SuperHRegister),
    /// XOR #imm,R0 (`R0 ^ imm → R0`) `11001010iiiiiiii`
    /// T = Test result
    XorImm(u8),
    /// XOR.B #imm,@(R0,GBR) (`(R0 + GBR) ^ imm → R0`) `11001110iiiiiiii`
    /// T = Test result
    XorB(u8),

    // ^ page 169; page 170 v
    // http://www.st.com/st-web-ui/static/active/en/resource/technical/document/user_manual/CD00147165.pdf Table 42 "Shift instructions" p. 170
    // https://antime.kapsi.fi/sega/files/h12p0.pdf Table 5.5 "Shift Instructions" p. 45(49)
    /// ROTL Rn (`T` ← `Rn` ← `MSB`) `0100nnnn00000100`
    /// T = MSB
    RotL(SuperHRegister),
    /// ROTR Rn (`LSB` → `Rn` → `T`) `0100nnnn00000101`
    /// T = LSB
    RotR(SuperHRegister),
    /// ROTCL Rn (`T` ← `Rn` ← `T`) `0100nnnn00100100`
    /// T = MSB
    RotcL(SuperHRegister),
    /// ROTCR Rn (`T` → `Rn` → `T`) `0100nnnn00100101`
    /// T = LSB
    RotcR(SuperHRegister),

    /// SHAD Rm,Rn (When `Rn` ≥ 0, `Rn << Rm` → `Rn`, When `Rn` < 0, `Rn >> Rm` → [`MSB` → `Rn`]) `0100nnnnmmmm1100`
    ///
    /// Level: J2
    ShaD(SuperHRegister, SuperHRegister),
    /// SHAL Rn (`T` ← `Rn` ← 0) `0100nnnn00100000`
    /// T = MSB
    ShaL(SuperHRegister),
    /// SHAR Rn (MSB → `Rn` → `T`) `0100nnnn00100001`
    /// T = LSB
    ShaR(SuperHRegister),

    /// SHLD Rm,Rn (When `Rn` ≥ 0, `Rn << Rm` → `Rn`, When `Rn` < 0, `Rn >> Rm` → [`0` → `Rn`]) `0100nnnnmmmm1101`
    ///
    /// Level: J2
    ShlD(SuperHRegister, SuperHRegister),
    /// SHLL Rn (`T` ← `Rn` ← 0) `0100nnnn00000000`
    /// T = MSB
    ShlL(SuperHRegister),
    /// SHLR Rn (0 → `Rn` → `T`) `0100nnnn00000001`
    /// T = LSB
    ShlR(SuperHRegister),

    /// SHLL2 Rn (`Rn << 2` → `Rn`) `0100nnnn00001000`
    ShlL2(SuperHRegister),
    /// SHLR2 Rn (`Rn >> 2` → `Rn`) `0100nnnn00001001`
    ShlR2(SuperHRegister),
    /// SHLL8 Rn (`Rn << 8` → `Rn`) `0100nnnn00011000`
    ShlL8(SuperHRegister),
    /// SHLR8 Rn (`Rn >> 8` → `Rn`) `0100nnnn00011001`
    ShlR8(SuperHRegister),
    /// SHLL16 Rn (`Rn << 16` → `Rn`) `0100nnnn00101000`
    ShlL16(SuperHRegister),
    /// SHLR16 Rn (`Rn >> 16` → `Rn`) `0100nnnn00101001`
    ShlR16(SuperHRegister),

    // ^ page 170; page 171 v
    // http://www.st.com/st-web-ui/static/active/en/resource/technical/document/user_manual/CD00147165.pdf Table 43 "Branch instructions" p. 171
    // ^ page 45(49); page 46(50) v
    // https://antime.kapsi.fi/sega/files/h12p0.pdf Table 5.5 "Branch Instructions" p. 46(50)
    /// BF label (When `T` = 0, `disp * 2 + PC + 4` → `PC`, When `T` = 1, nop) `10001011dddddddd`
    Bf(Displacement8),
    /// BF/S label (Delayed branch; When `T` = 0, `disp * 2 + PC + 4` → `PC`, When `T` = 1, nop) `10001111dddddddd`
    ///
    /// Level: Sh2
    BfS(Displacement8),

    /// BT label (When `T` = 1, `disp * 2 + PC + 4` → `PC`, When `T` = 0, nop) `10001001dddddddd`
    Bt(Displacement8),
    /// BT/S label (Delayed branch; When `T` = 1, `disp * 2 + PC + 4` → `PC`, When `T` = 0, nop) `10001101dddddddd`
    ///
    /// Level: Sh2
    BtS(Displacement8),

    /// BRA label (Delayed branch; `disp * 2 + PC + 4` → `PC`) `1010dddddddddddd`
    Bra(Displacement12),
    /// BRA Rn (Delayed branch; `Rn + PC + 4` → `PC`) `0000nnnn00100011`
    ///
    /// Level: Sh2
    BraF(SuperHRegister),

    /// BSR label (Delayed branch; `PC + 4` → `PR`, `disp * 2 + PC + 4` → `PC`) `1011dddddddddddd`
    Bsr(Displacement12),
    /// BSR Rn (Delayed branch; `PC + 4` → `PR`, `Rn + PC + 4` → `PC`) `0000nnnn00000011`
    ///
    /// Level: Sh2
    BsrF(SuperHRegister),

    /// JMP @Rn (Delayed branch; `Rn` → `PC`) `0100nnnn00101011`
    Jmp(SuperHRegister),
    /// JSR @Rn (Delayed branch; `PC + 4` → `PR`, `Rn` → `PC`) `0100nnnn00001011`
    Jsr(SuperHRegister),
    /// RTS (Delayed branch; `PR` → `PC`) `0000000000001011`
    Rts,

    // ^ page 171; page 172 v
    // http://www.st.com/st-web-ui/static/active/en/resource/technical/document/user_manual/CD00147165.pdf Table 44 "System control instructions" pp. 172-174
    // ^ page 46(50); page 47(51) v
    // https://antime.kapsi.fi/sega/files/h12p0.pdf Table 5.5 "System Control Instructions" pp. 47(51)-48(52)
    /// CLRMAC (0 → `MACH`, `MACL`) `0000000000101000`
    ClrMac,
    /// CLRS (0 → `S`) `0000000001001000`
    ///
    /// Level: Sh3
    ClrS,
    /// CLRT (0 → `T`) `0000000000001000`
    /// T = 0
    ClrT,

    /// LDC Rm,SR (`Rm` → `SR`) `0100mmmm00001110`
    /// T = LSB
    ///
    /// Privileged
    LdcSr(SuperHRegister),
    /// LDC Rm,GBR (`Rm` → `GBR`) `0100mmmm00011110`
    LdcGbr(SuperHRegister),
    /// LDC Rm,VBR (`Rm` → `VBR`) `0100mmmm00101110`
    ///
    /// Privileged
    LdcVbr(SuperHRegister),
    /// LDC Rm,SSR (`Rm` → `SSR`) `0100mmmm00111110`
    ///
    /// Privileged
    /// Level: Sh3
    LdcSsr(SuperHRegister),
    /// LDC Rm,SPC (`Rm` → `SPC`) `0100mmmm01001110`
    ///
    /// Privileged
    /// Level: Sh3
    LdcSpc(SuperHRegister),
    /// LDC Rm,DBR (`Rm` → `DBR`) `0100mmmm11111010`
    ///
    /// Privileged
    /// Level: Sh4
    LdcDbr(SuperHRegister),
    /// LDC Rm,Rn_BANK (`Rm` → `Rn_BANK` (`n` = 0 to 7)) `0100mmmm1nnn1110`
    ///
    /// Privileged
    /// Level: Sh3
    LdcRnBank(SuperHRegister, SuperHRegisterBank),

    /// LDC.L @Rm+,SR (`(Rm)` → `SR`, `Rm + 4` → `Rm`) `0100mmmm00000111`
    /// T = LSB
    ///
    /// Privileged
    LdcLSr(SuperHRegister),
    /// LDC.L @Rm+,GBR (`(Rm)` → `GBR`, `Rm + 4` → `Rm`) `0100mmmm00010111`
    LdcLGbr(SuperHRegister),
    /// LDC.L @Rm+,VBR (`(Rm)` → `VBR`, `Rm + 4` → `Rm`) `0100mmmm00100111`
    ///
    /// Privileged
    LdcLVbr(SuperHRegister),
    /// LDC.L @Rm+,SSR (`(Rm)` → `SSR`, `Rm + 4` → `Rm`) `0100mmmm00110111`
    ///
    /// Privileged
    /// Level: Sh3
    LdcLSsr(SuperHRegister),
    /// LDC.L @Rm+,SPC (`(Rm)` → `SPC`, `Rm + 4` → `Rm`) `0100mmmm01000111`
    ///
    /// Privileged
    /// Level: Sh3
    LdcLSpc(SuperHRegister),
    /// LDC.L @Rm+,DBR (`(Rm)` → `DBR`, `Rm + 4` → `Rm`) `0100mmmm11110110`
    ///
    /// Privileged
    /// Level: Sh4
    LdcLDbr(SuperHRegister),
    /// LDC.L @Rm+,Rn_BANK (`(Rm)` → `Rn_BANK` (`n` = 0 to 7), `Rm + 4` → `Rm`) `0100mmmm1nnn0111`
    ///
    /// Privileged
    /// Level: Sh3
    LdcLRnBank(SuperHRegister, SuperHRegisterBank),

    /// LDS Rm,MACH (`Rm` → `MACH`) `0100mmmm00001010`
    LdsMach(SuperHRegister),
    /// LDS Rm,MACL (`Rm` → `MACL`) `0100mmmm00011010`
    LdsMacl(SuperHRegister),
    /// LDS Rm,PR (`Rm` → `PR`) `0100mmmm00101010`
    LdsPr(SuperHRegister),

    /// LDS.L @Rm+,MACH (`(Rm)` → `MACH`, `Rm + 4` → `Rm`) `0100mmmm00000110`
    LdsLMach(SuperHRegister),
    /// LDS.L @Rm+,MACL (`(Rm)` → `MACL`, `Rm + 4` → `Rm`) `0100mmmm00010110`
    LdsLMacl(SuperHRegister),
    /// LDS.L @Rm+,PR (`(Rm)` → `PR`, `Rm + 4` → `Rm`) `0100mmmm00100110`
    LdsLPr(SuperHRegister),

    // ^ page 172; page 173 v
    /// LDTLB (`PTEH/PTEL` → `TLB`) `0000000000111000`
    ///
    /// Privileged
    /// Features: MMU
    /// Level: Sh3
    Ldtlb,

    /// MOVCA.L R0,@Rn (`R0` → `(Rn)` (without fetchingcache block)) `0000nnnn11000011`
    ///
    /// Level: Sh4
    MovcaL(SuperHRegister),

    /// NOP (No operation) `0000000000001001`
    Nop,

    /// OCBI @Rn (Invalidates operand cache block) `0000nnnn10010011`
    ///
    /// Level: Sh4
    OcbI(SuperHRegister),
    /// OCBP @Rn (Writes back and invalidates operand cache block) `0000nnnn10100011`
    ///
    /// Level: Sh4
    OcbP(SuperHRegister),
    /// OCBWB @Rn (Writes back operand cache block) `0000nnnn10110011`
    ///
    /// Level: Sh4
    OcbWb(SuperHRegister),

    /// PREF @Rn (`(Rn)` → operand cache) `0000nnnn10000011`
    ///
    /// Level: Sh3
    Pref(SuperHRegister),

    /// RTE (Delayed branch; `SSR/SPC` → `SR/PC`) `0000000000101011`
    ///
    /// Privileged
    Rte,

    /// SETS (1 → `S`) `0000000001011000`
    ///
    /// Level: Sh3
    SetS,
    /// SETT (1 → `T`) `0000000000011000`
    /// T = 0
    SetT,

    /// SLEEP (Sleep or standby) `0000000000011011`
    ///
    /// Privileged
    Sleep,

    /// STC SR,Rn (`SR` → `Rn`) `0000nnnn00000010`
    ///
    /// Privileged
    StcSr(SuperHRegister),
    /// STC GBR,Rn (`GBR` → `Rn`) `0000nnnn00010010`
    StcGbr(SuperHRegister),
    /// STC VBR,Rn (`VBR` → `Rn`) `0000nnnn00100010`
    ///
    /// Privileged
    StcVbr(SuperHRegister),
    /// STC SSR,Rn (`SSR` → `Rn`) `0000nnnn00110010`
    ///
    /// Privileged
    /// Level: Sh3
    StcSsr(SuperHRegister),
    /// STC SPC,Rn (`SPC` → `Rn`) `0000nnnn01000010`
    ///
    /// Privileged
    /// Level: Sh3
    StcSpc(SuperHRegister),
    /// STC SGR,Rn (`SGR` → `Rn`) `0000nnnn00111010`
    ///
    /// Privileged
    /// Level: Sh4
    StcSgr(SuperHRegister),
    /// STC DBR,Rn (`DBR` → `Rn`) `0000nnnn11111010`
    ///
    /// Privileged
    /// Level: Sh4
    StcDbr(SuperHRegister),
    /// STC Rm_BANK,Rn (`Rm_BANK` → `Rn` (`m` = 0 to 7)) `0000nnnn1mmm0010`
    ///
    /// Privileged
    /// Level: Sh3
    StcRmBank(SuperHRegisterBank, SuperHRegister),

    /// STC.L SR,@-Rn (`Rn - 4` → `Rn`, `SR` → `(Rn)`) `0100nnnn00000011`
    ///
    /// Privileged
    StcLSr(SuperHRegister),
    /// STC.L GBR,@-Rn (`Rn - 4` → `Rn`, `GBR` → `(Rn)`) `0100nnnn00010011`
    StcLGbr(SuperHRegister),
    /// STC.L VBR,@-Rn (`Rn - 4` → `Rn`, `VBR` → `(Rn)`) `0100nnnn00100011`
    ///
    /// Privileged
    StcLVbr(SuperHRegister),
    // ^ page 173; page 174 v
    /// STC.L SSR,@-Rn (`Rn - 4` → `Rn`, `SSR` → `(Rn)`) `0100nnnn00110011`
    ///
    /// Privileged
    /// Level: Sh3
    StcLSsr(SuperHRegister),
    /// STC.L SPC,@-Rn (`Rn - 4` → `Rn`, `SPC` → `(Rn)`) `0100nnnn01000011`
    ///
    /// Privileged
    /// Level: Sh3
    StcLSpc(SuperHRegister),
    /// STC.L SGR,@-Rn (`Rn - 4` → `Rn`, `SGR` → `(Rn)`) `0100nnnn00110010`
    ///
    /// Privileged
    /// Level: Sh4
    StcLSgr(SuperHRegister),
    /// STC.L DBR,@-Rn (`Rn - 4` → `Rn`, `DBR` → `(Rn)`) `0100nnnn11110010`
    ///
    /// Privileged
    /// Level: Sh4
    StcLDbr(SuperHRegister),
    /// STC.L Rm_BANK,@-Rn (`Rn - 4` → `Rn`, `Rm_BANK` → `(Rn)` (`m` = 0 to 7)) `0100nnnn1mmm0011`
    ///
    /// Privileged
    /// Level: Sh3
    StcLRmBank(SuperHRegisterBank, SuperHRegister),

    /// STS MACH,Rn (`MACH` → `Rn`) `0000nnnn00001010`
    StsMach(SuperHRegister),
    /// STS MACL,Rn (`MACL` → `Rn`) `0000nnnn00011010`
    StsMacl(SuperHRegister),
    /// STS PR,Rn (`PR` → `Rn`) `0000nnnn00101010`
    StsPr(SuperHRegister),

    // ^ page 47(51); page 48(52) v
    /// STS.L MACH,@-Rn (`Rn - 4` → `Rn`, `MACH` → `(Rn)`) `0100nnnn00000010`
    StsLMach(SuperHRegister),
    /// STS.L MACL,@-Rn (`Rn - 4` → `Rn`, `MACL` → `(Rn)`) `0100nnnn00010010`
    StsLMacl(SuperHRegister),
    /// STS.L PR,@-Rn (`Rn - 4` → `Rn`, `PR` → `(Rn)`) `0100nnnn00100010`
    StsLPr(SuperHRegister),

    /// TRAPA #imm (`PC + 2` → `SPC`, `SR` → `SSR`, `#imm << 2` → `TRA`, `0x160` → `EXPEVT`, `VBR + 0x0100` → `PC`)
    /// `11000011iiiiiiii`
    Trapa(u8),

    // ^ page 174; page 175 v
    // http://www.st.com/st-web-ui/static/active/en/resource/technical/document/user_manual/CD00147165.pdf
    // Table 45 "Floating-point single-precision instructions" pp. 175-176
    /// FLDI0 FRn (`0x00000000` → `FRn`) `1111nnnn10001101`
    ///
    /// Features: FPU
    /// Level: Sh4
    Fldi0(SuperHFloatRegister),
    /// FLDI1 FRn (`0x3F800000` → `FRn`) `1111nnnn10011101`
    ///
    /// Features: FPU
    /// Level: Sh4
    Fldi1(SuperHFloatRegister),

    /// FMOV FRm,FRn (`FRm` → `FRn`) `1111nnnnmmmm1100`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovS(SuperHFloatRegister, SuperHFloatRegister),

    /// FMOV.S @Rm,FRn (`(Rm)` → `FRn`) `1111nnnnmmmm1000`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovAtRegS(SuperHRegister, SuperHFloatRegister),
    /// FMOV.S @(R0,Rm),FRn (`(R0 + Rm)` → `FRn`) `1111nnnnmmmm0110`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovAtIndexedRegisterS(SuperHRegister, SuperHFloatRegister),
    /// FMOV.S @Rm+,FRn (`(Rm)` → `FRn`, `Rm + 4` → `Rm`) `1111nnnnmmmm1001`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovAtPostIncrementRegS(SuperHRegister, SuperHFloatRegister),

    /// FMOV.S FRm,@Rn (`FRm` → `(Rn)`) `1111nnnnmmmm1010`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovToAtRegS(SuperHFloatRegister, SuperHRegister),
    /// FMOV.S FRm,@-Rn (`Rn - 4` → `Rn`, `FRm` → `(Rn)`) `1111nnnnmmmm1011`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovToAtPreDecrementRegS(SuperHFloatRegister, SuperHRegister),
    /// FMOV.S FRm,@(R0,Rn) (`FRm` → `(R0 + Rn)`) `1111nnnnmmmm0111`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovToAtIndexedRegisterS(SuperHFloatRegister, SuperHRegister),

    /// FMOV DRm,DRn (`DRm` → `DRn`) `1111nnn0mmm01100`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMov(SuperHDoubleRegister, SuperHDoubleRegister),

    /// FMOV @Rm,DRn (`(Rm)` → `DRn`) `1111nnn0mmmm1000`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovAtReg(SuperHRegister, SuperHDoubleRegister),
    /// FMOV @(R0,Rm),DRn (`(R0 + Rm)` → `DRn`) `1111nnn0mmmm0110`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovAtIndexedRegister(SuperHRegister, SuperHDoubleRegister),
    /// FMOV @Rm+,DRn (`(Rm)` → `DRn`, `Rm + 8` → `Rm`) `1111nnn0mmmm1001`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovAtPostIncrementReg(SuperHRegister, SuperHDoubleRegister),

    /// FMOV DRm,@Rn (`DRm` → `(Rn)`) `1111nnnnmmm01010`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovToAtReg(SuperHDoubleRegister, SuperHRegister),
    /// FMOV DRm,@-Rn (`Rn - 8` → `Rn`, `DRm` → `(Rn)`) `1111nnnnmmm01011`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovToAtPreDecrementReg(SuperHDoubleRegister, SuperHRegister),
    /// FMOV DRm,@(R0,Rn) (`DRm` → `(R0 + Rn)`) `1111nnnnmmm00111`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovToAtIndexedRegister(SuperHDoubleRegister, SuperHRegister),

    /// FLDS FRm,FPUL (`FRm` → `FPUL`) `1111mmmm00011101`
    ///
    /// Features: FPU
    /// Level: Sh4
    Flds(SuperHFloatRegister),
    /// FSTS FPUL,FRn (`FPUL` → `FRn`) `1111nnnn00001101`
    ///
    /// Features: FPU
    /// Level: Sh4
    Fsts(SuperHFloatRegister),

    /// FABS FRn (`FRn & 0x7FFF_FFFF` → `FRn`) `1111nnnn01011101`
    ///
    /// Features: FPU
    /// Level: Sh4
    FabsS(SuperHFloatRegister),

    /// FADD FRm,FRn (`FRn + FRm` → `FRn`) `1111nnnnmmmm0000`
    ///
    /// Features: FPU
    /// Level: Sh4
    FaddS(SuperHFloatRegister, SuperHFloatRegister),

    /// FCMP/EQ FRm,FRn (When `FRn` = `FRm`, 1 → `T`; Otherwise, 0 → `T`) `1111nnnnmmmm0100`
    /// T = Comparison result
    ///
    /// Features: FPU
    /// Level: Sh4
    FcmpEqS(SuperHFloatRegister, SuperHFloatRegister),
    // ^ page 175; page 176 v
    /// FCMP/GT FRm,FRn (When `FRn` > `FRm`, 1 → `T`; Otherwise, 0 → `T`) `1111nnnnmmmm0101`
    /// T = Comparison result
    ///
    /// Features: FPU
    /// Level: Sh4
    FcmpGtS(SuperHFloatRegister, SuperHFloatRegister),

    /// FDIV FRm,FRn (`FRn / FRm` → `FRn`) `1111nnnnmmmm0011`
    ///
    /// Features: FPU
    /// Level: Sh4
    FdivS(SuperHFloatRegister, SuperHFloatRegister),

    /// FLOAT FPUL,FRn (`(float)FPUL` → `FRn`) `1111nnnn00101101`
    ///
    /// Features: FPU
    /// Level: Sh4
    FloatS(SuperHFloatRegister),

    /// FMAC FR0,FRm,FRn (`FR0 * FRm + FRn` → `FRn`) `1111nnnnmmmm1110`
    ///
    /// Features: FPU
    /// Level: Sh4
    Fmac(SuperHFloatRegister, SuperHFloatRegister),
    /// FMUL FRm,FRn (`FRn * FRm` → `FRn`) `1111nnnnmmmm0010`
    ///
    /// Features: FPU
    /// Level: Sh4
    FmulS(SuperHFloatRegister, SuperHFloatRegister),

    /// FNEG FRn (`FRn ^ 0x8000_0000` → `FRn`) `1111nnnn01001101`
    ///
    /// Features: FPU
    /// Level: Sh4
    FnegS(SuperHFloatRegister),

    /// FSQRT FRn (`√FRn` → `FRn`) `1111nnnn01101101`
    ///
    /// Features: FPU
    /// Level: Sh4
    FsqrtS(SuperHFloatRegister),

    /// FSUB FRm,FRn (`FRn - FRm` → `FRn`) `1111nnnnmmmm0001`
    ///
    /// Features: FPU
    /// Level: Sh4
    FsubS(SuperHFloatRegister, SuperHFloatRegister),

    /// FRTC FPUL,FRm (`(long)FRm` → `FPUL`) `1111mmmm00111101`
    ///
    /// Features: FPU
    /// Level: Sh4
    FrtcS(SuperHFloatRegister),

    // http://www.st.com/st-web-ui/static/active/en/resource/technical/document/user_manual/CD00147165.pdf
    // Table 46 "Floating-point double-precision instructions" pp. 176-177
    /// FABS DRn (`DRn & 0x7FFF_FFFF_FFFF_FFFF` → `DRn`) `1111nnn001011101`
    ///
    /// Features: FPU
    /// Level: Sh4
    Fabs(SuperHDoubleRegister),

    /// FADD DRm,DRn (`DRn + DRm` → `DRn`) `1111nnn0mmm00000`
    ///
    /// Features: FPU
    /// Level: Sh4
    Fadd(SuperHDoubleRegister, SuperHDoubleRegister),

    /// FCMP/EQ DRm,DRn (When `DRn` = `DRm`, 1 → `T`; Otherwise, 0 → `T`) `1111nnn0mmm00100`
    /// T = Comparison result
    ///
    /// Features: FPU
    /// Level: Sh4
    FcmpEq(SuperHDoubleRegister, SuperHDoubleRegister),
    /// FCMP/GT DRm,DRn (When `DRn` > `DRm`, 1 → `T`; Otherwise, 0 → `T`) `1111nnn0mmm00101`
    /// T = Comparison result
    ///
    /// Features: FPU
    /// Level: Sh4
    FcmpGt(SuperHDoubleRegister, SuperHDoubleRegister),

    /// FDIV DRm,DRn (`DRn / DRm` → `DRn`) `1111nnn0mmm00011`
    ///
    /// Features: FPU
    /// Level: Sh4
    Fdiv(SuperHDoubleRegister, SuperHDoubleRegister),

    /// FCNVDS DRm,FPUL (`double_to_float[DRm]` → `FPUL`) `1111mmm010111101`
    ///
    /// Features: FPU
    /// Level: Sh4
    FcnvDs(SuperHDoubleRegister),
    /// FCNVSD FPUL,DRn (`float_to_double[FPUL]` → `DRn`) `1111nnn010101101`
    ///
    /// Features: FPU
    /// Level: Sh4
    FcnvSd(SuperHDoubleRegister),

    // ^ page 176; page 177 v
    /// FLOAT FPUL,DRn (`(float)FPUL` → `DRn`) `1111nnn000101101`
    ///
    /// Features: FPU
    /// Level: Sh4
    Float(SuperHDoubleRegister),

    /// FMUL DRm,DRn (`DRn * DRm` → `DRn`) `1111nnn0mmm00010`
    ///
    /// Features: FPU
    /// Level: Sh4
    Fmul(SuperHDoubleRegister, SuperHDoubleRegister),

    /// FNEG DRn (`DRn ^ 0x8000_0000_0000_0000` → `DRn`) `1111nnn001001101`
    ///
    /// Features: FPU
    /// Level: Sh4
    Fneg(SuperHDoubleRegister),

    /// FSQRT DRn (`√DRn` → `DRn`) `1111nnn001101101`
    ///
    /// Features: FPU
    /// Level: Sh4
    Fsqrt(SuperHDoubleRegister),

    /// FSUB DRm,DRn (`DRn - DRm` → `DRn`) `1111nnn0mmm00001`
    ///
    /// Features: FPU
    /// Level: Sh4
    Fsub(SuperHDoubleRegister, SuperHDoubleRegister),

    /// FRTC FPUL,DRm (`(long)DRm` → `FPUL`) `1111mmm000111101`
    ///
    /// Features: FPU
    /// Level: Sh4
    Frtc(SuperHDoubleRegister),

    // http://www.st.com/st-web-ui/static/active/en/resource/technical/document/user_manual/CD00147165.pdf
    // Table 47 "Floating-point control instructions" p. 177
    /// LDS Rm,FPSCR (`Rm` → `FPSCR`) `0100mmmm01101010`
    ///
    /// Features: FPU
    /// Level: Sh4
    LdsFpscr(SuperHRegister),
    /// LDS Rm,FPUL (`Rm` → `FPUL`) `0100mmmm01011010`
    ///
    /// Features: FPU
    /// Level: Sh4
    LdsFpul(SuperHRegister),

    /// LDS @Rm+,FPSCR (`Rm` → `FPSCR`, `Rm + 4` → `Rm`) `0100mmmm01100110`
    ///
    /// Features: FPU
    /// Level: Sh4
    LdsFpscrL(SuperHRegister),
    /// LDS @Rm+,FPUL (`Rm` → `FPUL`, `Rm + 4` → `Rm`) `0100mmmm01010110`
    ///
    /// Features: FPU
    /// Level: Sh4
    LdsFpulL(SuperHRegister),

    /// STS FPSCR,Rn (`FPSCR` → `Rn`) `0000nnnn01101010`
    ///
    /// Features: FPU
    /// Level: Sh4
    StsFpscr(SuperHRegister),
    /// STS FPUL,Rn (`FPUL` → `Rn`) `0000nnnn01011010`
    ///
    /// Features: FPU
    /// Level: Sh4
    StsFpul(SuperHRegister),

    /// STS FPSCR,@-Rn (`Rn - 4` → `Rn`, `FPSCR` → `Rn`) `0100nnnn01100010`
    ///
    /// Features: FPU
    /// Level: Sh4
    StsFpscrL(SuperHRegister),
    /// STS FPUL,@-Rn (`Rn - 4` → `Rn`, `FPUL` → `Rn`) `0100nnnn01010010`
    ///
    /// Features: FPU
    /// Level: Sh4
    StsFpulL(SuperHRegister),

    // ^ page 177; page 178 v
    // http://www.st.com/st-web-ui/static/active/en/resource/technical/document/user_manual/CD00147165.pdf
    // Table 48 "Floating-point graphics acceleration instructions" p. 178
    /// FMOV DRm,XDn (`DRm` → `XDn`) `1111nnn1mmm01100`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovToX(SuperHDoubleRegister, SuperHExtendedDoubleRegister),
    /// FMOV XDm,DRn (`XDm` → `DRn`) `1111nnn0mmm11100`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovFromX(SuperHExtendedDoubleRegister, SuperHDoubleRegister),
    /// FMOV XDm,XDn (`XDm` → `XDn`) `1111nnn1mmm11100`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovX(SuperHExtendedDoubleRegister, SuperHExtendedDoubleRegister),

    /// FMOV @Rm,XDn (`(Rm)` → `XDn`) `1111nnn1mmmm1000`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovAtRegX(SuperHRegister, SuperHExtendedDoubleRegister),
    /// FMOV @Rm+,XDn (`(Rm)` → `XDn`, `Rm + 8` → `Rm`) `1111nnn1mmmm1001`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovAtPostIncrementRegX(SuperHRegister, SuperHExtendedDoubleRegister),
    /// FMOV @(R0,Rm),XDn (`(R0 + Rm)` → `XDn`) `1111nnn1mmmm0110`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovAtIndexedRegisterX(SuperHRegister, SuperHExtendedDoubleRegister),

    /// FMOV XDm,@Rn (`XDm` → `(Rn)`) `1111nnnnmmm11010`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovToAtRegX(SuperHExtendedDoubleRegister, SuperHRegister),
    /// FMOV XDm,@-Rn (`Rn - 8` → `Rn`, `XDm` → `(Rn)`) `1111nnnnmmm11011`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovToAtPreDecrementRegX(SuperHExtendedDoubleRegister, SuperHRegister),
    /// FMOV XDm,@(R0,Rn) (`XDm` → `(R0 + Rn)`) `1111nnnnmmm10111`
    ///
    /// Features: FPU
    /// Level: Sh4
    FMovToAtIndexedRegisterX(SuperHExtendedDoubleRegister, SuperHRegister),

    /// FIPR FVm,FVn (`inner_product[FVm, FVn]` → `FR[n + 3]`) `1111nnmm11101101`
    ///
    /// Features: FPU
    /// Level: Sh4
    Fipr(SuperHVectorFloatRegister, SuperHVectorFloatRegister),
    /// XMRTX FVn (`transform_vector[XMTRX, FVn]` → `FVn`) `1111nn0111111101`
    ///
    /// Features: FPU
    /// Level: Sh4
    Xmrtx(SuperHVectorFloatRegister),

    /// FRCHG (`~FPSCR.FR` → `SPFCR.FR`) `1111101111111101`
    ///
    /// Features: FPU
    /// Level: Sh4
    FrChg,
    /// FSCHG (`~FPSCR.SZ` → `SPFCR.SZ`) `1111001111111101`
    ///
    /// Features: FPU
    /// Level: Sh4
    FsChg,

    /// CAS.L Rm,Rn,@R0 (When a byte in Rn equals a byte in Rm, 1 → `T`) `0010nnnnmmmm0011`
    ///
    /// <https://github.com/j-core/jcore-cpu/blob/00f398e/decode/decode_table_simple.vhd#L1919>
    ///
    /// Features: J2
    /// Level: J2
    CasL(SuperHRegister, SuperHRegister),
}

macro_rules! eas {
    ($from:expr, $to:ident) => {{
        $to::try_from($from).map_err(|()| unreachable!()).unwrap()
    }}
}

impl SuperHInstruction {
    /// Parse a single instruction with the specified architectural state
    ///
    /// This parses among *all* valid instructions; feature-gating must be performed separately
    pub fn parse(ins: u16, fpscr_sz: bool) -> Option<SuperHInstruction> {
        let n3 = ((ins >> 3 * 4) & 0b1111) as u8;
        let n2 = ((ins >> 2 * 4) & 0b1111) as u8;
        let n1 = ((ins >> 1 * 4) & 0b1111) as u8;
        let n0 = ((ins >> 0 * 4) & 0b1111) as u8;
        Some(match (n3, n2, n1, n0) {
            (0b1110, nnnn, _iiii1, _iiii0) => SuperHInstruction::MovImm((ins & 0xFF) as u8, eas!(nnnn, SuperHRegister)),  // MOV #imm,Rn
            (0b1001, nnnn, _dddd1, _dddd0) => SuperHInstruction::MovImmW(Displacement8((ins & 0xFF) as u8), eas!(nnnn, SuperHRegister)),  // MOV.W @(disp,PC),Rn
            (0b1101, nnnn, _dddd1, _dddd0) => SuperHInstruction::MovImmL(Displacement8((ins & 0xFF) as u8), eas!(nnnn, SuperHRegister)),  // MOV.L @(disp,PC),Rn

            (0b0110, nnnn, mmmm, 0b0011) => SuperHInstruction::MovReg(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // MOV Rm,Rn

            (0b0010, nnnn, mmmm, 0b0000) => SuperHInstruction::MovToAtRegB(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // MOV.B Rm,@Rn
            (0b0010, nnnn, mmmm, 0b0001) => SuperHInstruction::MovToAtRegW(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // MOV.W Rm,@Rn
            (0b0010, nnnn, mmmm, 0b0010) => SuperHInstruction::MovToAtRegL(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // MOV.L Rm,@Rn

            (0b0110, nnnn, mmmm, 0b0000) => SuperHInstruction::MovFromAtRegB(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // MOV.B @Rm,Rn
            (0b0110, nnnn, mmmm, 0b0001) => SuperHInstruction::MovFromAtRegW(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // MOV.W @Rm,Rn
            (0b0110, nnnn, mmmm, 0b0010) => SuperHInstruction::MovFromAtRegL(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // MOV.L @Rm,Rn

            // MOV.B Rm,@-Rn
            (0b0010, nnnn, mmmm, 0b0100) => SuperHInstruction::MovToAtRegPreDecrementB(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),
            // MOV.W Rm,@-Rn
            (0b0010, nnnn, mmmm, 0b0101) => SuperHInstruction::MovToAtRegPreDecrementW(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),
            // MOV.L Rm,@-Rn
            (0b0010, nnnn, mmmm, 0b0110) => SuperHInstruction::MovToAtRegPreDecrementL(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),

            // MOV.B @Rm+,Rn
            (0b0110, nnnn, mmmm, 0b0100) => SuperHInstruction::MovFromAtRegPostIncrementB(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),
            // MOV.W @Rm+,Rn
            (0b0110, nnnn, mmmm, 0b0101) => SuperHInstruction::MovFromAtRegPostIncrementW(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),
            // MOV.L @Rm+,Rn
            (0b0110, nnnn, mmmm, 0b0110) => SuperHInstruction::MovFromAtRegPostIncrementL(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),

            // MOV.B R0,@(disp,Rn)
            (0b1000, 0b0000, nnnn, dddd) => SuperHInstruction::MovToAtDisplacedRegisterB(Displacement4(dddd as u8), eas!(nnnn, SuperHRegister)),
            // MOV.W R0,@(disp,Rn)
            (0b1000, 0b0001, nnnn, dddd) => SuperHInstruction::MovToAtDisplacedRegisterW(Displacement4(dddd as u8), eas!(nnnn, SuperHRegister)),
            // MOV.L Rm,@(disp,Rn)
            (0b0001, nnnn, mmmm, dddd) => {
                SuperHInstruction::MovToAtDisplacedRegisterL(eas!(mmmm, SuperHRegister), Displacement4(dddd as u8), eas!(nnnn, SuperHRegister))
            }

            // MOV.B @(disp,Rm),R0
            (0b1000, 0b0100, mmmm, dddd) => SuperHInstruction::MovFromAtDisplacedRegisterB(Displacement4(dddd as u8), eas!(mmmm, SuperHRegister)),
            // MOV.W @(disp,Rm),R0
            (0b1000, 0b0101, mmmm, dddd) => SuperHInstruction::MovFromAtDisplacedRegisterW(Displacement4(dddd as u8), eas!(mmmm, SuperHRegister)),
            // MOV.L @(disp,Rm),Rn
            (0b0101, nnnn, mmmm, dddd) => {
                SuperHInstruction::MovFromAtDisplacedRegisterL(Displacement4(dddd), eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister))
            }

            // MOV.B Rm,@(R0,Rn)
            (0b0000, nnnn, mmmm, 0b0100) => SuperHInstruction::MovToAtIndexedRegisterB(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),
            // MOV.W Rm,@(R0,Rn)
            (0b0000, nnnn, mmmm, 0b0101) => SuperHInstruction::MovToAtIndexedRegisterW(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),
            // MOV.L Rm,@(R0,Rn)
            (0b0000, nnnn, mmmm, 0b0110) => SuperHInstruction::MovToAtIndexedRegisterL(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),

            // MOV.B @(R0,Rm),Rn
            (0b0000, nnnn, mmmm, 0b1100) => SuperHInstruction::MovFromAtIndexedRegisterB(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),
            // MOV.W @(R0,Rm),Rn
            (0b0000, nnnn, mmmm, 0b1101) => SuperHInstruction::MovFromAtIndexedRegisterW(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),
            // MOV.L @(R0,Rm),Rn
            (0b0000, nnnn, mmmm, 0b1110) => SuperHInstruction::MovFromAtIndexedRegisterL(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),

            (0b1100, 0b0000, _dddd1, _dddd0) => SuperHInstruction::MovToAtDisplacedGbrB(Displacement8((ins & 0xFF) as u8)),  // MOV.B R0,@(disp,GBR)
            (0b1100, 0b0001, _dddd1, _dddd0) => SuperHInstruction::MovToAtDisplacedGbrW(Displacement8((ins & 0xFF) as u8)),  // MOV.W R0,@(disp,GBR)
            (0b1100, 0b0010, _dddd1, _dddd0) => SuperHInstruction::MovToAtDisplacedGbrL(Displacement8((ins & 0xFF) as u8)),  // MOV.L R0,@(disp,GBR)

            (0b1100, 0b0100, _dddd1, _dddd0) => SuperHInstruction::MovFromAtDisplacedGbrB(Displacement8((ins & 0xFF) as u8)),  // MOV.B @(disp,GBR),R0
            (0b1100, 0b0101, _dddd1, _dddd0) => SuperHInstruction::MovFromAtDisplacedGbrW(Displacement8((ins & 0xFF) as u8)),  // MOV.W @(disp,GBR),R0
            (0b1100, 0b0110, _dddd1, _dddd0) => SuperHInstruction::MovFromAtDisplacedGbrL(Displacement8((ins & 0xFF) as u8)),  // MOV.L @(disp,GBR),R0

            (0b1100, 0b0111, _dddd1, _dddd0) => SuperHInstruction::Mova(Displacement8((ins & 0xFF) as u8)),  // MOVA @(disp,PC),R0

            (0b0000, nnnn, 0b0010, 0b1001) => SuperHInstruction::Movt(eas!(nnnn, SuperHRegister)),  // MOVT Rn

            (0b0110, nnnn, mmmm, 0b1000) => SuperHInstruction::SwapB(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // SWAP.B Rm,Rn
            (0b0110, nnnn, mmmm, 0b1001) => SuperHInstruction::SwapW(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // SWAP.W Rm,Rn

            (0b0010, nnnn, mmmm, 0b1101) => SuperHInstruction::Xtrct(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // XTRCT Rm,Rn

            (0b0011, nnnn, mmmm, 0b1100) => SuperHInstruction::AddReg(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // ADD Rm,Rn
            (0b0111, nnnn, _iiii1, _iiii0) => SuperHInstruction::AddImm((ins & 0xFF) as u8, eas!(nnnn, SuperHRegister)),  // ADD #imm,Rn
            (0b0011, nnnn, mmmm, 0b1110) => SuperHInstruction::AddC(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // ADDC Rm,Rn
            (0b0011, nnnn, mmmm, 0b1111) => SuperHInstruction::AddV(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // ADDV Rm,Rn

            (0b1000, 0b1000, _iiii1, _iiii0) => SuperHInstruction::CmpEqImm((ins & 0xFF) as u8),  // CMP/EQ #imm,R0
            (0b0011, nnnn, mmmm, 0b0000) => SuperHInstruction::CmpEqReg(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // CMP/EQ Rm,Rn
            (0b0011, nnnn, mmmm, 0b0010) => SuperHInstruction::CmpHs(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // CMP/HS Rm,Rn
            (0b0011, nnnn, mmmm, 0b0011) => SuperHInstruction::CmpGe(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // CMP/GE Rm,Rn
            (0b0011, nnnn, mmmm, 0b0110) => SuperHInstruction::CmpHi(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // CMP/HI Rm,Rn
            (0b0011, nnnn, mmmm, 0b0111) => SuperHInstruction::CmpGt(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // CMP/GT Rm,Rn

            (0b0100, nnnn, 0b0001, 0b0001) => SuperHInstruction::CmpPz(eas!(nnnn, SuperHRegister)),  // CMP/PZ Rn
            (0b0100, nnnn, 0b0001, 0b0101) => SuperHInstruction::CmpPl(eas!(nnnn, SuperHRegister)),  // CMP/PL Rn
            (0b0010, nnnn, mmmm, 0b1100) => SuperHInstruction::CmpStr(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // CMP/STR Rm,Rn

            (0b0011, nnnn, mmmm, 0b0100) => SuperHInstruction::Div1(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // DIV1 Rm,Rn
            (0b0010, nnnn, mmmm, 0b0111) => SuperHInstruction::Div0S(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // DIV0S Rm,Rn
            (0b0000, 0b0000, 0b0001, 0b1001) => SuperHInstruction::Div0U,  // DIV0U (0 → `M`/`Q`/`T`)

            (0b0011, nnnn, mmmm, 0b1101) => SuperHInstruction::DmulsL(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // DMULS.L Rm,Rn
            (0b0011, nnnn, mmmm, 0b0101) => SuperHInstruction::DmuluL(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // DMULU.L Rm,Rn

            (0b0100, nnnn, 0b0001, 0b0000) => SuperHInstruction::Dt(eas!(nnnn, SuperHRegister)),  // DT Rn

            (0b0110, nnnn, mmmm, 0b1110) => SuperHInstruction::ExtsB(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // EXTS.B Rm,Rn
            (0b0110, nnnn, mmmm, 0b1111) => SuperHInstruction::ExtsW(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // EXTS.W Rm,Rn

            (0b0110, nnnn, mmmm, 0b1100) => SuperHInstruction::ExtuB(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // EXTU.B Rm,Rn
            (0b0110, nnnn, mmmm, 0b1101) => SuperHInstruction::ExtuW(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // EXTU.W Rm,Rn

            (0b0000, nnnn, mmmm, 0b1111) => SuperHInstruction::MacL(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // MAC.L @Rm+,@Rn+
            (0b0100, nnnn, mmmm, 0b1111) => SuperHInstruction::MacW(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // MAC.L @Rm+,@Rn+

            (0b0000, nnnn, mmmm, 0b0111) => SuperHInstruction::MulL(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // MAC.L Rm,Rn

            (0b0010, nnnn, mmmm, 0b1111) => SuperHInstruction::MulsW(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // MULS.W Rm,Rn
            (0b0010, nnnn, mmmm, 0b1110) => SuperHInstruction::MuluW(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // MULU.W Rm,Rn

            (0b0110, nnnn, mmmm, 0b1011) => SuperHInstruction::Neg(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // NEG Rm,Rn
            (0b0110, nnnn, mmmm, 0b1010) => SuperHInstruction::Negc(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // NEGC Rm,Rn

            (0b0011, nnnn, mmmm, 0b1000) => SuperHInstruction::Sub(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // SUB Rm,Rn
            (0b0011, nnnn, mmmm, 0b1010) => SuperHInstruction::Subc(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // SUBC Rm,Rn
            (0b0011, nnnn, mmmm, 0b1011) => SuperHInstruction::Subv(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // SUBV Rm,Rn

            (0b0010, nnnn, mmmm, 0b1001) => SuperHInstruction::AndReg(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // AND Rm,Rn
            (0b1100, 0b1001, _iiii1, _iiii0) => SuperHInstruction::AndImm((ins & 0xFF) as u8),  // AND #imm,R0
            (0b1100, 0b1101, _iiii1, _iiii0) => SuperHInstruction::AndB((ins & 0xFF) as u8),  // AND.B #imm,@(R0,GBR)

            (0b0110, nnnn, mmmm, 0b0111) => SuperHInstruction::Not(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // NOT Rm,Rn

            (0b0010, nnnn, mmmm, 0b1011) => SuperHInstruction::OrReg(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // OR Rm,Rn
            (0b1100, 0b1011, _iiii1, _iiii0) => SuperHInstruction::OrImm((ins & 0xFF) as u8),  // OR #imm,R0
            (0b1100, 0b1111, _iiii1, _iiii0) => SuperHInstruction::OrB((ins & 0xFF) as u8),  // OR.B #imm,@(R0,GBR)

            (0b0100, nnnn, 0b0001, 0b1011) => SuperHInstruction::TasB(eas!(nnnn, SuperHRegister)),  // TAS.B @Rn

            (0b0010, nnnn, mmmm, 0b1000) => SuperHInstruction::TstReg(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // TST Rm,Rn
            (0b1100, 0b1000, _iiii1, _iiii0) => SuperHInstruction::TstImm((ins & 0xFF) as u8),  // TST #imm,R0
            (0b1100, 0b1100, _iiii1, _iiii0) => SuperHInstruction::TstB((ins & 0xFF) as u8),  // TST.B #imm,@(R0,GBR)

            (0b0010, nnnn, mmmm, 0b1010) => SuperHInstruction::XorReg(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // XOR Rm,Rn
            (0b1100, 0b1010, _iiii1, _iiii0) => SuperHInstruction::XorImm((ins & 0xFF) as u8),  // XOR #imm,R0
            (0b1100, 0b1110, _iiii1, _iiii0) => SuperHInstruction::XorB((ins & 0xFF) as u8),  // XOR.B #imm,@(R0,GBR)

            (0b0100, nnnn, 0b0000, 0b0100) => SuperHInstruction::RotL(eas!(nnnn, SuperHRegister)),  // ROTL Rn
            (0b0100, nnnn, 0b0000, 0b0101) => SuperHInstruction::RotR(eas!(nnnn, SuperHRegister)),  // ROTR Rn
            (0b0100, nnnn, 0b0010, 0b0100) => SuperHInstruction::RotcL(eas!(nnnn, SuperHRegister)),  // ROTCL Rn
            (0b0100, nnnn, 0b0010, 0b0101) => SuperHInstruction::RotcR(eas!(nnnn, SuperHRegister)),  // ROTCR Rn

            (0b0100, nnnn, mmmm, 0b1100) => SuperHInstruction::ShaD(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // SHAD Rm,Rn
            (0b0100, nnnn, 0b0010, 0b0000) => SuperHInstruction::ShaL(eas!(nnnn, SuperHRegister)),  // SHAL Rn
            (0b0100, nnnn, 0b0010, 0b0001) => SuperHInstruction::ShaR(eas!(nnnn, SuperHRegister)),  // SHAR Rn (MSB → `Rn` → `T`)

            (0b0100, nnnn, mmmm, 0b1101) => SuperHInstruction::ShlD(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // SHLD Rm,Rn
            (0b0100, nnnn, 0b0000, 0b0000) => SuperHInstruction::ShlL(eas!(nnnn, SuperHRegister)),  // SHLL Rn
            (0b0100, nnnn, 0b0000, 0b0001) => SuperHInstruction::ShlR(eas!(nnnn, SuperHRegister)),  // SHLR Rn (0 → `Rn` → `T`)

            (0b0100, nnnn, 0b0000, 0b1000) => SuperHInstruction::ShlL2(eas!(nnnn, SuperHRegister)),  // SHLL2 Rn
            (0b0100, nnnn, 0b0000, 0b1001) => SuperHInstruction::ShlR2(eas!(nnnn, SuperHRegister)),  // SHLR2 Rn
            (0b0100, nnnn, 0b0001, 0b1000) => SuperHInstruction::ShlL8(eas!(nnnn, SuperHRegister)),  // SHLL8 Rn
            (0b0100, nnnn, 0b0001, 0b1001) => SuperHInstruction::ShlR8(eas!(nnnn, SuperHRegister)),  // SHLR8 Rn
            (0b0100, nnnn, 0b0010, 0b1000) => SuperHInstruction::ShlL16(eas!(nnnn, SuperHRegister)),  // SHLL16 Rn
            (0b0100, nnnn, 0b0010, 0b1001) => SuperHInstruction::ShlR16(eas!(nnnn, SuperHRegister)),  // SHLR16 Rn

            (0b1000, 0b1011, _dddd1, _dddd0) => SuperHInstruction::Bf(Displacement8((ins & 0xFF) as u8)),  // BF label
            (0b1000, 0b1111, _dddd1, _dddd0) => SuperHInstruction::BfS(Displacement8((ins & 0xFF) as u8)),  // BF/S label

            (0b1000, 0b1001, _dddd1, _dddd0) => SuperHInstruction::Bt(Displacement8((ins & 0xFF) as u8)),  // BT label
            (0b1000, 0b1101, _dddd1, _dddd0) => SuperHInstruction::BtS(Displacement8((ins & 0xFF) as u8)),  // BT/S label

            (0b1010, _dddd2, _dddd1, _dddd0) => SuperHInstruction::Bra(Displacement12(ins & 0x0FFF)),  // BRA label
            (0b0000, nnnn, 0b0010, 0b0011) => SuperHInstruction::BraF(eas!(nnnn, SuperHRegister)),  // BRA Rn

            (0b1011, _dddd2, _dddd1, _dddd0) => SuperHInstruction::Bsr(Displacement12(ins & 0x0FFF)),  // BSR label
            (0b0000, nnnn, 0b0000, 0b0011) => SuperHInstruction::BsrF(eas!(nnnn, SuperHRegister)),  // BSR Rn

            (0b0100, nnnn, 0b0010, 0b1011) => SuperHInstruction::Jmp(eas!(nnnn, SuperHRegister)),  // JMP @Rn
            (0b0100, nnnn, 0b0000, 0b1011) => SuperHInstruction::Jsr(eas!(nnnn, SuperHRegister)),  // JSR @Rn
            (0b0000, 0b0000, 0b0000, 0b1011) => SuperHInstruction::Rts,  // RTS

            (0b0000, 0b0000, 0b0010, 0b1000) => SuperHInstruction::ClrMac,  // CLRMAC (0 → `MACH`, `MACL`)
            (0b0000, 0b0000, 0b0100, 0b1000) => SuperHInstruction::ClrS,  // CLRS (0 → `S`)
            (0b0000, 0b0000, 0b0000, 0b1000) => SuperHInstruction::ClrT,  // CLRT (0 → `T`)

            (0b0100, mmmm, 0b0000, 0b1110) => SuperHInstruction::LdcSr(eas!(mmmm, SuperHRegister)),  // LDC Rm,SR
            (0b0100, mmmm, 0b0001, 0b1110) => SuperHInstruction::LdcGbr(eas!(mmmm, SuperHRegister)),  // LDC Rm,GBR
            (0b0100, mmmm, 0b0010, 0b1110) => SuperHInstruction::LdcVbr(eas!(mmmm, SuperHRegister)),  // LDC Rm,VBR
            (0b0100, mmmm, 0b0011, 0b1110) => SuperHInstruction::LdcSsr(eas!(mmmm, SuperHRegister)),  // LDC Rm,SSR
            (0b0100, mmmm, 0b0100, 0b1110) => SuperHInstruction::LdcSpc(eas!(mmmm, SuperHRegister)),  // LDC Rm,SPC
            (0b0100, mmmm, 0b1111, 0b1010) => SuperHInstruction::LdcDbr(eas!(mmmm, SuperHRegister)),  // LDC Rm,DBR
            // LDC Rm,Rn_BANK
            (0b0100, mmmm, v1nnn, 0b1110) if (v1nnn & 0b1000) == 0b1000 => {
                SuperHInstruction::LdcRnBank(eas!(mmmm, SuperHRegister), eas!(v1nnn & 0b0111, SuperHRegisterBank))
            }

            (0b0100, mmmm, 0b0000, 0b0111) => SuperHInstruction::LdcLSr(eas!(mmmm, SuperHRegister)),  // LDC.L @Rm+,SR
            (0b0100, mmmm, 0b0001, 0b0111) => SuperHInstruction::LdcLGbr(eas!(mmmm, SuperHRegister)),  // LDC.L @Rm+,GBR
            (0b0100, mmmm, 0b0010, 0b0111) => SuperHInstruction::LdcLVbr(eas!(mmmm, SuperHRegister)),  // LDC.L @Rm+,VBR
            (0b0100, mmmm, 0b0011, 0b0111) => SuperHInstruction::LdcLSsr(eas!(mmmm, SuperHRegister)),  // LDC.L @Rm+,SSR
            (0b0100, mmmm, 0b0100, 0b0111) => SuperHInstruction::LdcLSpc(eas!(mmmm, SuperHRegister)),  // LDC.L @Rm+,SPC
            (0b0100, mmmm, 0b1111, 0b0110) => SuperHInstruction::LdcLDbr(eas!(mmmm, SuperHRegister)),  // LDC.L @Rm+,DBR
            // LDC.L @Rm+,Rn_BANK
            (0b0100, mmmm, v1nnn, 0b0111) if (v1nnn & 0b1000) == 0b1000 => {
                SuperHInstruction::LdcLRnBank(eas!(mmmm, SuperHRegister), eas!(v1nnn & 0b0111, SuperHRegisterBank))
            }

            (0b0100, mmmm, 0b0000, 0b1010) => SuperHInstruction::LdsMach(eas!(mmmm, SuperHRegister)),  // LDS Rm,MACH
            (0b0100, mmmm, 0b0001, 0b1010) => SuperHInstruction::LdsMacl(eas!(mmmm, SuperHRegister)),  // LDS Rm,MACL
            (0b0100, mmmm, 0b0010, 0b1010) => SuperHInstruction::LdsPr(eas!(mmmm, SuperHRegister)),  // LDS Rm,PR

            (0b0100, mmmm, 0b0000, 0b0110) => SuperHInstruction::LdsLMach(eas!(mmmm, SuperHRegister)),  // LDS.L @Rm+,MACH
            (0b0100, mmmm, 0b0001, 0b0110) => SuperHInstruction::LdsLMacl(eas!(mmmm, SuperHRegister)),  // LDS.L @Rm+,MACL
            (0b0100, mmmm, 0b0010, 0b0110) => SuperHInstruction::LdsLPr(eas!(mmmm, SuperHRegister)),  // LDS.L @Rm+,PR

            (0b0000, 0b0000, 0b0011, 0b1000) => SuperHInstruction::Ldtlb,  // LDTLB

            (0b0000, nnnn, 0b1100, 0b0011) => SuperHInstruction::MovcaL(eas!(nnnn, SuperHRegister)),  // MOVCA.L R0,@Rn

            (0b0000, 0b0000, 0b0000, 0b1001) => SuperHInstruction::Nop,  // NOP (No operation)

            (0b0000, nnnn, 0b1001, 0b0011) => SuperHInstruction::OcbI(eas!(nnnn, SuperHRegister)),  // OCBI @Rn
            (0b0000, nnnn, 0b1010, 0b0011) => SuperHInstruction::OcbP(eas!(nnnn, SuperHRegister)),  // OCBP @Rn
            (0b0000, nnnn, 0b1011, 0b0011) => SuperHInstruction::OcbWb(eas!(nnnn, SuperHRegister)),  // OCBWB @Rn

            (0b0000, nnnn, 0b1000, 0b0011) => SuperHInstruction::Pref(eas!(nnnn, SuperHRegister)),  // PREF @Rn

            (0b0000, 0b0000, 0b0010, 0b1011) => SuperHInstruction::Rte,  // RTE

            (0b0000, 0b0000, 0b0101, 0b1000) => SuperHInstruction::SetS,  // SETS (1 → `S`)
            (0b0000, 0b0000, 0b0001, 0b1000) => SuperHInstruction::SetT,  // SETT (1 → `T`)

            (0b0000, 0b0000, 0b0001, 0b1011) => SuperHInstruction::Sleep,  // SLEEP (Sleep or standby)

            (0b0000, nnnn, 0b0000, 0b0010) => SuperHInstruction::StcSr(eas!(nnnn, SuperHRegister)),  // STC SR,Rn
            (0b0000, nnnn, 0b0001, 0b0010) => SuperHInstruction::StcGbr(eas!(nnnn, SuperHRegister)),  // STC GBR,Rn
            (0b0000, nnnn, 0b0010, 0b0010) => SuperHInstruction::StcVbr(eas!(nnnn, SuperHRegister)),  // STC VBR,Rn
            (0b0000, nnnn, 0b0011, 0b0010) => SuperHInstruction::StcSsr(eas!(nnnn, SuperHRegister)),  // STC SSR,Rn
            (0b0000, nnnn, 0b0100, 0b0010) => SuperHInstruction::StcSpc(eas!(nnnn, SuperHRegister)),  // STC SPC,Rn
            (0b0000, nnnn, 0b0011, 0b1010) => SuperHInstruction::StcSgr(eas!(nnnn, SuperHRegister)),  // STC SGR,Rn
            (0b0000, nnnn, 0b1111, 0b1010) => SuperHInstruction::StcDbr(eas!(nnnn, SuperHRegister)),  // STC DBR,Rn
            // STC Rm_BANK,Rn
            (0b0000, nnnn, v1mmm, 0b0010) if (v1mmm & 0b1000) == 0b1000 => {
                SuperHInstruction::StcRmBank(eas!(v1mmm & 0b0111, SuperHRegisterBank), eas!(nnnn, SuperHRegister))
            }

            (0b0100, nnnn, 0b0000, 0b0011) => SuperHInstruction::StcLSr(eas!(nnnn, SuperHRegister)),  // STC.L SR,@-Rn
            (0b0100, nnnn, 0b0001, 0b0011) => SuperHInstruction::StcLGbr(eas!(nnnn, SuperHRegister)),  // STC.L GBR,@-Rn
            (0b0100, nnnn, 0b0010, 0b0011) => SuperHInstruction::StcLVbr(eas!(nnnn, SuperHRegister)),  // STC.L VBR,@-Rn
            (0b0100, nnnn, 0b0011, 0b0011) => SuperHInstruction::StcLSsr(eas!(nnnn, SuperHRegister)),  // STC.L SSR,@-Rn
            (0b0100, nnnn, 0b0100, 0b0011) => SuperHInstruction::StcLSpc(eas!(nnnn, SuperHRegister)),  // STC.L SPC,@-Rn
            (0b0100, nnnn, 0b0011, 0b0010) => SuperHInstruction::StcLSgr(eas!(nnnn, SuperHRegister)),  // STC.L SGR,@-Rn
            (0b0100, nnnn, 0b1111, 0b0010) => SuperHInstruction::StcLDbr(eas!(nnnn, SuperHRegister)),  // STC.L DBR,@-Rn
            // STC.L Rm_BANK,@-Rn
            (0b0100, nnnn, v1mmm, 0b0011) if (v1mmm & 0b1000) == 0b1000 => {
                SuperHInstruction::StcLRmBank(eas!(v1mmm & 0b0111, SuperHRegisterBank), eas!(nnnn, SuperHRegister))
            }

            (0b0000, nnnn, 0b0000, 0b1010) => SuperHInstruction::StsMach(eas!(nnnn, SuperHRegister)),  // STS MACH,Rn
            (0b0000, nnnn, 0b0001, 0b1010) => SuperHInstruction::StsMacl(eas!(nnnn, SuperHRegister)),  // STS MACL,Rn
            (0b0000, nnnn, 0b0010, 0b1010) => SuperHInstruction::StsPr(eas!(nnnn, SuperHRegister)),  // STS PR,Rn

            (0b0100, nnnn, 0b0000, 0b0010) => SuperHInstruction::StsLMach(eas!(nnnn, SuperHRegister)),  // STS.L MACH,@-Rn
            (0b0100, nnnn, 0b0001, 0b0010) => SuperHInstruction::StsLMacl(eas!(nnnn, SuperHRegister)),  // STS.L MACL,@-Rn
            (0b0100, nnnn, 0b0010, 0b0010) => SuperHInstruction::StsLPr(eas!(nnnn, SuperHRegister)),  // STS.L PR,@-Rn

            (0b1100, 0b0011, _iiii1, _iiii0) => SuperHInstruction::Trapa((ins & 0xFF) as u8),  // TRAPA #imm

            (0b1111, nnnn, 0b1000, 0b1101) => SuperHInstruction::Fldi0(eas!(nnnn, SuperHFloatRegister)),  // FLDI0 FRn
            (0b1111, nnnn, 0b1001, 0b1101) => SuperHInstruction::Fldi1(eas!(nnnn, SuperHFloatRegister)),  // FLDI1 FRn

            (0b1111, nnnn, mmmm, 0b1100) => {
                if !fpscr_sz {
                    SuperHInstruction::FMovS(eas!(mmmm, SuperHFloatRegister), eas!(nnnn, SuperHFloatRegister)) // FMOV FRm,FRn
                } else {
                    let nnn = nnnn >> 1;
                    let mmm = mmmm >> 1;
                    match ((nnnn & 0b0001), (mmmm & 0b0001)) {
                        (0b0000, 0b0000) => SuperHInstruction::FMov(eas!(mmm, SuperHDoubleRegister), eas!(nnn, SuperHDoubleRegister)),  // FMOV DRm,DRn

                        // FMOV DRm,XDn
                        (0b0001, 0b0000) => SuperHInstruction::FMovToX(eas!(mmm, SuperHDoubleRegister), eas!(nnn, SuperHExtendedDoubleRegister)),
                        // FMOV XDm,DRn
                        (0b0000, 0b0001) => SuperHInstruction::FMovFromX(eas!(mmm, SuperHExtendedDoubleRegister), eas!(nnn, SuperHDoubleRegister)),
                        // FMOV XDm,XDn
                        (0b0001, 0b0001) => SuperHInstruction::FMovX(eas!(mmm, SuperHExtendedDoubleRegister), eas!(nnn, SuperHExtendedDoubleRegister)),

                        _ => unreachable!(),
                    }
                }
            }

            (0b1111, nnnn, mmmm, 0b1000) => {
                if !fpscr_sz {
                    SuperHInstruction::FMovAtRegS(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHFloatRegister)) // FMOV.S @Rm,FRn
                } else {
                    let nnn = nnnn >> 1;
                    match nnnn & 0b0001 {
                        0b0000 => SuperHInstruction::FMovAtReg(eas!(mmmm, SuperHRegister), eas!(nnn, SuperHDoubleRegister)),  // FMOV @Rm,DRn

                        0b0001 => SuperHInstruction::FMovAtRegX(eas!(mmmm, SuperHRegister), eas!(nnn, SuperHExtendedDoubleRegister)),  // FMOV @Rm,XDn

                        _ => unreachable!(),
                    }
                }
            }
            (0b1111, nnnn, mmmm, 0b0110) => {
                if !fpscr_sz {
                    SuperHInstruction::FMovAtIndexedRegisterS(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHFloatRegister)) // FMOV.S @(R0,Rm),FRn
                } else {
                    let nnn = nnnn >> 1;
                    match nnnn & 0b0001 {
                        0b0000 => SuperHInstruction::FMovAtIndexedRegister(eas!(mmmm, SuperHRegister), eas!(nnn, SuperHDoubleRegister)),  // FMOV @(R0,Rm),DRn

                        // FMOV @(R0,Rm),XDn
                        0b0001 => SuperHInstruction::FMovAtIndexedRegisterX(eas!(mmmm, SuperHRegister), eas!(nnn, SuperHExtendedDoubleRegister)),

                        _ => unreachable!(),
                    }
                }
            }
            (0b1111, nnnn, mmmm, 0b1001) => {
                if !fpscr_sz {
                    SuperHInstruction::FMovAtPostIncrementRegS(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHFloatRegister)) // FMOV.S @Rm+,FRn
                } else {
                    let nnn = nnnn >> 1;
                    match nnnn & 0b0001 {
                        0b0000 => SuperHInstruction::FMovAtPostIncrementReg(eas!(mmmm, SuperHRegister), eas!(nnn, SuperHDoubleRegister)),  // FMOV @Rm+,DRn

                        // FMOV @Rm+,XDn
                        0b0001 => SuperHInstruction::FMovAtPostIncrementRegX(eas!(mmmm, SuperHRegister), eas!(nnn, SuperHExtendedDoubleRegister)),

                        _ => unreachable!(),
                    }
                }
            }
            (0b1111, nnnn, mmmm, 0b1010) => {
                if !fpscr_sz {
                    SuperHInstruction::FMovToAtRegS(eas!(mmmm, SuperHFloatRegister), eas!(nnnn, SuperHRegister)) // FMOV.S FRm,@Rn
                } else {
                    let mmm = mmmm >> 1;
                    match mmmm & 0b0001 {
                        0b0000 => SuperHInstruction::FMovToAtReg(eas!(mmm, SuperHDoubleRegister), eas!(nnnn, SuperHRegister)),  // FMOV DRm,@Rn

                        0b0001 => SuperHInstruction::FMovToAtRegX(eas!(mmm, SuperHExtendedDoubleRegister), eas!(nnnn, SuperHRegister)),  // FMOV XDm,@Rn

                        _ => unreachable!(),
                    }
                }
            }
            (0b1111, nnnn, mmmm, 0b1011) => {
                if !fpscr_sz {
                    SuperHInstruction::FMovToAtPreDecrementRegS(eas!(mmmm, SuperHFloatRegister), eas!(nnnn, SuperHRegister)) // FMOV.S FRm,@-Rn
                } else {
                    let mmm = mmmm >> 1;
                    match mmmm & 0b0001 {
                        0b0000 => SuperHInstruction::FMovToAtPreDecrementReg(eas!(mmm, SuperHDoubleRegister), eas!(nnnn, SuperHRegister)),  // FMOV DRm,@-Rn

                        // FMOV XDm,@-Rn
                        0b0001 => SuperHInstruction::FMovToAtPreDecrementRegX(eas!(mmm, SuperHExtendedDoubleRegister), eas!(nnnn, SuperHRegister)),

                        _ => unreachable!(),
                    }
                }
            }
            (0b1111, nnnn, mmmm, 0b0111) => {
                if !fpscr_sz {
                    SuperHInstruction::FMovToAtIndexedRegisterS(eas!(mmmm, SuperHFloatRegister), eas!(nnnn, SuperHRegister)) // FMOV.S FRm,@(R0,Rn)
                } else {
                    let mmm = mmmm >> 1;
                    match mmmm & 0b0001 {
                        0b0000 => SuperHInstruction::FMovToAtIndexedRegister(eas!(mmm, SuperHDoubleRegister), eas!(nnnn, SuperHRegister)),  // FMOV DRm,@(R0,Rn)

                        // FMOV XDm,@(R0,Rn)
                        0b0001 => SuperHInstruction::FMovToAtIndexedRegisterX(eas!(mmm, SuperHExtendedDoubleRegister), eas!(nnnn, SuperHRegister)),

                        _ => unreachable!(),
                    }
                }
            }

            (0b1111, mmmm, 0b0001, 0b1101) => SuperHInstruction::Flds(eas!(mmmm, SuperHFloatRegister)),  // FLDS FRm,FPUL
            (0b1111, nnnn, 0b0000, 0b1101) => SuperHInstruction::Fsts(eas!(nnnn, SuperHFloatRegister)),  // FSTS FPUL,FRn

            (0b1111, nnnn, 0b0101, 0b1101) => {
                let nnn = nnnn >> 1;
                match (fpscr_sz, nnnn & 0b0001) {
                    (true, 0b0000) => SuperHInstruction::Fabs(eas!(nnn, SuperHDoubleRegister)),  // FABS DRn
                    _ => SuperHInstruction::FabsS(eas!(nnnn, SuperHFloatRegister)),  // FABS FRn
                }
            }

            (0b1111, nnnn, mmmm, 0b0000) => {
                let nnn = nnnn >> 1;
                let mmm = mmmm >> 1;
                match (fpscr_sz, nnnn & 0b0001, mmmm & 0b0001) {
                    (true, 0b0000, 0b0000) => SuperHInstruction::Fadd(eas!(mmm, SuperHDoubleRegister), eas!(nnn, SuperHDoubleRegister)),  // FADD DRm,DRn
                    _ => SuperHInstruction::FaddS(eas!(mmmm, SuperHFloatRegister), eas!(nnnn, SuperHFloatRegister)),  // FADD FRm,FRn
                }
            }

            (0b1111, nnnn, mmmm, 0b0100) => {
                let nnn = nnnn >> 1;
                let mmm = mmmm >> 1;
                match (fpscr_sz, nnnn & 0b0001, mmmm & 0b0001) {
                    (true, 0b0000, 0b0000) => SuperHInstruction::FcmpEq(eas!(mmm, SuperHDoubleRegister), eas!(nnn, SuperHDoubleRegister)),  // FCMP/EQ DRm,DRn
                    _ => SuperHInstruction::FcmpEqS(eas!(mmmm, SuperHFloatRegister), eas!(nnnn, SuperHFloatRegister)),  // FCMP/EQ FRm,FRn
                }
            }
            (0b1111, nnnn, mmmm, 0b0101) => {
                let nnn = nnnn >> 1;
                let mmm = mmmm >> 1;
                match (fpscr_sz, nnnn & 0b0001, mmmm & 0b0001) {
                    (true, 0b0000, 0b0000) => SuperHInstruction::FcmpGt(eas!(mmm, SuperHDoubleRegister), eas!(nnn, SuperHDoubleRegister)),  // FCMP/GT DRm,DRn
                    _ => SuperHInstruction::FcmpGtS(eas!(mmmm, SuperHFloatRegister), eas!(nnnn, SuperHFloatRegister)),  // FCMP/GT FRm,FRn
                }
            }

            (0b1111, nnnn, mmmm, 0b0011) => {
                let nnn = nnnn >> 1;
                let mmm = mmmm >> 1;
                match (fpscr_sz, nnnn & 0b0001, mmmm & 0b0001) {
                    (true, 0b0000, 0b0000) => SuperHInstruction::Fdiv(eas!(mmm, SuperHDoubleRegister), eas!(nnn, SuperHDoubleRegister)),  // FDIV DRm,DRn
                    _ => SuperHInstruction::FdivS(eas!(mmmm, SuperHFloatRegister), eas!(nnnn, SuperHFloatRegister)),  // FDIV FRm,FRn
                }
            }

            // FCNVDS DRm,FPUL
            (0b1111, vmmm0, 0b1011, 0b1101) if (vmmm0 & 0b0001) == 0b0000 => SuperHInstruction::FcnvDs(eas!(vmmm0 >> 1, SuperHDoubleRegister)),
            // FCNVSD FPUL,DRn
            (0b1111, vnnn0, 0b1010, 0b1101) if (vnnn0 & 0b0001) == 0b0000 => SuperHInstruction::FcnvSd(eas!(vnnn0 >> 1, SuperHDoubleRegister)),

            (0b1111, nnnn, 0b0010, 0b1101) => {
                let nnn = nnnn >> 1;
                match (fpscr_sz, nnnn & 0b0001) {
                    (true, 0b0000) => SuperHInstruction::Float(eas!(nnn, SuperHDoubleRegister)),  // FLOAT FPUL,DRn
                    _ => SuperHInstruction::FloatS(eas!(nnnn, SuperHFloatRegister)),  // FLOAT FPUL,FRn
                }
            }

            (0b1111, nnnn, mmmm, 0b1110) => SuperHInstruction::Fmac(eas!(mmmm, SuperHFloatRegister), eas!(nnnn, SuperHFloatRegister)),  // FMAC FR0,FRm,FRn
            (0b1111, nnnn, mmmm, 0b0010) => {
                let nnn = nnnn >> 1;
                let mmm = mmmm >> 1;
                match (fpscr_sz, nnnn & 0b0001, mmmm & 0b0001) {
                    (true, 0b0000, 0b0000) => SuperHInstruction::Fmul(eas!(mmm, SuperHDoubleRegister), eas!(nnn, SuperHDoubleRegister)),  // FMUL DRm,DRn
                    _ => SuperHInstruction::FmulS(eas!(mmmm, SuperHFloatRegister), eas!(nnnn, SuperHFloatRegister)),  // FMUL FRm,FRn
                }
            }

            (0b1111, nnnn, 0b0100, 0b1101) => {
                let nnn = nnnn >> 1;
                match (fpscr_sz, nnnn & 0b0001) {
                    (true, 0b0000) => SuperHInstruction::Fneg(eas!(nnn, SuperHDoubleRegister)),  // FNEG DRn
                    _ => SuperHInstruction::FnegS(eas!(nnnn, SuperHFloatRegister)),  // FNEG FRn
                }
            }

            (0b1111, nnnn, 0b0110, 0b1101) => {
                let nnn = nnnn >> 1;
                match (fpscr_sz, nnnn & 0b0001) {
                    (true, 0b0000) => SuperHInstruction::Fsqrt(eas!(nnn, SuperHDoubleRegister)),  // FSQRT DRn
                    _ => SuperHInstruction::FsqrtS(eas!(nnnn, SuperHFloatRegister)),  // FSQRT FRn
                }
            }

            (0b1111, nnnn, mmmm, 0b0001) => {
                let nnn = nnnn >> 1;
                let mmm = mmmm >> 1;
                match (fpscr_sz, nnnn & 0b0001, mmmm & 0b0001) {
                    (true, 0b0000, 0b0000) => SuperHInstruction::Fsub(eas!(mmm, SuperHDoubleRegister), eas!(nnn, SuperHDoubleRegister)),  // FSUB DRm,DRn
                    _ => SuperHInstruction::FsubS(eas!(mmmm, SuperHFloatRegister), eas!(nnnn, SuperHFloatRegister)),  // FSUB FRm,FRn
                }
            }

            (0b1111, mmmm, 0b0011, 0b1101) => {
                let mmm = mmmm >> 1;
                match (fpscr_sz, mmmm & 0b0001) {
                    (true, 0b0000) => SuperHInstruction::Frtc(eas!(mmm, SuperHDoubleRegister)),  // FRTC FPUL,DRm
                    _ => SuperHInstruction::FrtcS(eas!(mmmm, SuperHFloatRegister)),  // FRTC FPUL,FRm
                }
            }

            (0b0100, mmmm, 0b0110, 0b1010) => SuperHInstruction::LdsFpscr(eas!(mmmm, SuperHRegister)),  // LDS Rm,FPSCR
            (0b0100, mmmm, 0b0101, 0b1010) => SuperHInstruction::LdsFpul(eas!(mmmm, SuperHRegister)),  // LDS Rm,FPUL

            (0b0100, mmmm, 0b0110, 0b0110) => SuperHInstruction::LdsFpscrL(eas!(mmmm, SuperHRegister)),  // LDS @Rm+,FPSCR
            (0b0100, mmmm, 0b0101, 0b0110) => SuperHInstruction::LdsFpulL(eas!(mmmm, SuperHRegister)),  // LDS @Rm+,FPUL

            (0b0000, nnnn, 0b0110, 0b1010) => SuperHInstruction::StsFpscr(eas!(nnnn, SuperHRegister)),  // STS FPSCR,Rn
            (0b0000, nnnn, 0b0101, 0b1010) => SuperHInstruction::StsFpul(eas!(nnnn, SuperHRegister)),  // STS FPUL,Rn

            (0b0100, nnnn, 0b0110, 0b0010) => SuperHInstruction::StsFpscrL(eas!(nnnn, SuperHRegister)),  // STS FPSCR,@-Rn
            (0b0100, nnnn, 0b0101, 0b0010) => SuperHInstruction::StsFpulL(eas!(nnnn, SuperHRegister)),  // STS FPUL,@-Rn

            // FIPR FVm,FVn
            (0b1111, nnmm, 0b1110, 0b1101) => {
                SuperHInstruction::Fipr(eas!(nnmm & 0b0011, SuperHVectorFloatRegister), eas!(nnmm >> 2, SuperHVectorFloatRegister))
            }
            // XMRTX FVn
            (0b1111, vnn01, 0b1111, 0b1101) if (vnn01 & 0b0011) == 0b0001 => SuperHInstruction::Xmrtx(eas!(vnn01 >> 2, SuperHVectorFloatRegister)),

            (0b1111, 0b1011, 0b1111, 0b1101) => SuperHInstruction::FrChg,  // FRCHG
            (0b1111, 0b0011, 0b1111, 0b1101) => SuperHInstruction::FsChg,  // FSCHG

            (0b0010, nnnn, mmmm, 0b0011) => SuperHInstruction::CasL(eas!(mmmm, SuperHRegister), eas!(nnnn, SuperHRegister)),  // CAS.L Rm,Rn,@R0

            _ => return None,
        })
    }
}

impl Default for SuperHInstruction {
    fn default() -> SuperHInstruction {
        SuperHInstruction::Nop
    }
}

include!(concat!(env!("OUT_DIR"), "/instruction_reassemble.rs"));
include!(concat!(env!("OUT_DIR"), "/instruction_display.rs"));
include!(concat!(env!("OUT_DIR"), "/instruction_feature.rs"));
include!(concat!(env!("OUT_DIR"), "/instruction_level.rs"));
include!(concat!(env!("OUT_DIR"), "/instruction_privilege.rs"));

#[derive(Default, Debug, Copy, Clone, Hash, PartialEq, Eq, PartialOrd, Ord)]
#[repr(transparent)]
pub struct Displacement12(pub u16);

#[derive(Default, Debug, Copy, Clone, Hash, PartialEq, Eq, PartialOrd, Ord)]
#[repr(transparent)]
pub struct Displacement8(pub u8);

#[derive(Default, Debug, Copy, Clone, Hash, PartialEq, Eq, PartialOrd, Ord)]
#[repr(transparent)]
pub struct Displacement4(pub u8);

#[derive(Debug, Copy, Clone, Hash, PartialEq, Eq, PartialOrd, Ord)]
pub enum SuperHFeature {
    FPU,
    MMU,
    /// [`CAS.L`](https://github.com/j-core/jcore-cpu/blob/00f398e/decode/decode_table_simple.vhd#L1919)
    J2,
}

/// The revision of the ISA
///
///
#[derive(Debug, Copy, Clone, Hash, PartialEq, Eq, PartialOrd, Ord)]
pub enum SuperHLevel {
    Sh,
    Sh2,
    /// SHAD and SHLD borrowed from SH-3 into an otherwise SH-2 CPU
    J2,
    Sh3,
    Sh4,
}

// http://www.st.com/st-web-ui/static/active/en/resource/technical/document/user_manual/CD00147165.pdf
// Section 2 "Programming model" p. 21
/// R*
#[derive(Debug, Copy, Clone, Hash, PartialEq, Eq, PartialOrd, Ord)]
pub enum SuperHRegister {
    R0 = 0,
    R1 = 1,
    R2 = 2,
    R3 = 3,
    R4 = 4,
    R5 = 5,
    R6 = 6,
    R7 = 7,
    R8 = 8,
    R9 = 9,
    R10 = 10,
    R11 = 11,
    R12 = 12,
    R13 = 13,
    R14 = 14,
    R15 = 15,
}

/// R*_BANK
#[derive(Debug, Copy, Clone, Hash, PartialEq, Eq, PartialOrd, Ord)]
pub enum SuperHRegisterBank {
    R0Bank = 0,
    R1Bank = 1,
    R2Bank = 2,
    R3Bank = 3,
    R4Bank = 4,
    R5Bank = 5,
    R6Bank = 6,
    R7Bank = 7,
}

// http://www.st.com/st-web-ui/static/active/en/resource/technical/document/user_manual/CD00147165.pdf
// Section 2.4 "Programming model" p. 34
/// FR*
#[derive(Debug, Copy, Clone, Hash, PartialEq, Eq, PartialOrd, Ord)]
pub enum SuperHFloatRegister {
    Fr0 = 0,
    Fr1 = 1,
    Fr2 = 2,
    Fr3 = 3,
    Fr4 = 4,
    Fr5 = 5,
    Fr6 = 6,
    Fr7 = 7,
    Fr8 = 8,
    Fr9 = 9,
    Fr10 = 10,
    Fr11 = 11,
    Fr12 = 12,
    Fr13 = 13,
    Fr14 = 14,
    Fr15 = 15,
}

/// XF*
#[derive(Debug, Copy, Clone, Hash, PartialEq, Eq, PartialOrd, Ord)]
pub enum SuperHExtendedFloatRegister {
    Xf0 = 0,
    Xf1 = 1,
    Xf2 = 2,
    Xf3 = 3,
    Xf4 = 4,
    Xf5 = 5,
    Xf6 = 6,
    Xf7 = 7,
    Xf8 = 8,
    Xf9 = 9,
    Xf10 = 10,
    Xf11 = 11,
    Xf12 = 12,
    Xf13 = 13,
    Xf14 = 14,
    Xf15 = 15,
}

/// FV* 4x
#[derive(Debug, Copy, Clone, Hash, PartialEq, Eq, PartialOrd, Ord)]
pub enum SuperHVectorFloatRegister {
    Fv0 = 0,
    Fv4 = 1,
    Fv8 = 2,
    Fv12 = 3,
}

/// DR* 2x
#[derive(Debug, Copy, Clone, Hash, PartialEq, Eq, PartialOrd, Ord)]
pub enum SuperHDoubleRegister {
    Dr0 = 0,
    Dr2 = 1,
    Dr4 = 2,
    Dr6 = 3,
    Dr8 = 4,
    Dr10 = 5,
    Dr12 = 6,
    Dr14 = 7,
}

/// XD* 2x
#[derive(Debug, Copy, Clone, Hash, PartialEq, Eq, PartialOrd, Ord)]
pub enum SuperHExtendedDoubleRegister {
    Xd0 = 0,
    Xd2 = 1,
    Xd4 = 2,
    Xd6 = 3,
    Xd8 = 4,
    Xd10 = 5,
    Xd12 = 6,
    Xd14 = 7,
}

include!(concat!(env!("OUT_DIR"), "/argument_display.rs"));
include!(concat!(env!("OUT_DIR"), "/argument_try_from.rs"));


#[cfg(feature = "yaxpeax")]
impl yaxpeax_arch::Instruction for SuperHInstruction {
    fn well_defined(&self) -> bool {
        true
    }
}

#[cfg(feature = "yaxpeax")]
impl yaxpeax_arch::LengthedInstruction for SuperHInstruction {
    type Unit = AddressDiff<u32>;

    fn len(&self) -> Self::Unit {
        AddressDiff::from_const(2)
    }

    fn min_size() -> Self::Unit {
        AddressDiff::from_const(2)
    }
}


#[cfg(feature = "yaxpeax")]
#[derive(Debug, Copy, Clone, Hash, PartialEq, Eq, PartialOrd, Ord)]
pub enum SuperHInstructionDecodeError {
    Exhausted,
    NotFound,
}

#[cfg(feature = "yaxpeax")]
impl fmt::Display for SuperHInstructionDecodeError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        f.write_str(self.description())
    }
}

#[cfg(all(feature = "yaxpeax", feature = "std"))]
impl std::error::Error for SuperHInstructionDecodeError {}

#[cfg(feature = "yaxpeax")]
impl DecodeError for SuperHInstructionDecodeError {
    fn data_exhausted(&self) -> bool {
        *self == SuperHInstructionDecodeError::Exhausted
    }

    fn bad_opcode(&self) -> bool {
        *self == SuperHInstructionDecodeError::NotFound
    }

    fn bad_operand(&self) -> bool {
        false
    }

    fn description(&self) -> &'static str {
        match self {
            SuperHInstructionDecodeError::Exhausted => "exhausted input",
            SuperHInstructionDecodeError::NotFound => "invalid opcode",
        }
    }
}

#[cfg(feature = "yaxpeax")]
impl From<yaxpeax_arch::ReadError> for SuperHInstructionDecodeError {
    fn from(_e: yaxpeax_arch::ReadError) -> SuperHInstructionDecodeError {
        SuperHInstructionDecodeError::Exhausted
    }
}

#[cfg(feature = "yaxpeax")]
#[derive(Debug, Copy, Clone, Hash, PartialEq, Eq, PartialOrd, Ord)]
pub struct SuperHDecoder<'f> {
    pub little_endian: bool,

    /// When `FPSCR.SZ` = 1, the SH-4 CPU core can perform data transfer by means of pair single-precision data transfer
    /// instructions.
    ///
    /// <http://www.st.com/st-web-ui/static/active/en/resource/technical/document/user_manual/CD00147165.pdf>
    /// Section 6.5.2 "Pair single-precision data transfer" p. 154
    pub fpscr_sz: bool,

    pub level: SuperHLevel,
    pub features: &'f [SuperHFeature],
}

#[cfg(feature = "yaxpeax")]
impl SuperHDecoder<'_> {
    /// SuperH CPU in little-endian mode
    pub const SH1: SuperHDecoder<'static> = SuperHDecoder {
        little_endian: true,
        fpscr_sz: false,
        level: SuperHLevel::Sh,
        features: &[],
    };

    /// SH-2 CPU in little-endian mode
    pub const SH2: SuperHDecoder<'static> = SuperHDecoder {
        little_endian: true,
        fpscr_sz: false,
        level: SuperHLevel::Sh2,
        features: &[],
    };

    /// J-core J2 CPU in little-endian mode
    pub const J2: SuperHDecoder<'static> = SuperHDecoder {
        little_endian: true,
        fpscr_sz: false,
        level: SuperHLevel::J2,
        features: &[SuperHFeature::J2],
    };

    /// SH-3 CPU with MMU in little-endian mode
    pub const SH3: SuperHDecoder<'static> = SuperHDecoder {
        little_endian: true,
        fpscr_sz: false,
        level: SuperHLevel::Sh3,
        features: &[SuperHFeature::MMU],
    };

    /// SH-4 CPU with MMU and FPU (double-precision instructions off) in little-endian mode
    pub const SH4: SuperHDecoder<'static> = SuperHDecoder {
        little_endian: true,
        fpscr_sz: false,
        level: SuperHLevel::Sh4,
        features: &[SuperHFeature::MMU, SuperHFeature::FPU],
    };
}

#[cfg(feature = "yaxpeax")]
impl<'f> yaxpeax_arch::Decoder<SuperH<'_>> for SuperHDecoder<'_> {
    fn decode_into<T: yaxpeax_arch::Reader<<SuperH<'f> as Arch>::Address, <SuperH<'f> as Arch>::Word>>(&self, inst: &mut SuperHInstruction, words: &mut T)
                                                                                                       -> Result<(), <SuperH<'_> as Arch>::DecodeError> {
        *inst = self.decode(words)?;
        Ok(())
    }

    fn decode<T: yaxpeax_arch::Reader<<SuperH<'f> as Arch>::Address, <SuperH<'f> as Arch>::Word>>(
        &self, words: &mut T)
        -> Result<SuperHInstruction, <SuperH<'_> as Arch>::DecodeError> {
        // mov #0x69,r4 (1110nnnniiiiiiii) encodes as 69 e4 in LE and e4 69 in BE
        let ins = ((words.next()? as u16) << (if self.little_endian { 0 } else { 8 })) | ((words.next()? as u16) << (if self.little_endian { 8 } else { 0 }));

        let ret = SuperHInstruction::parse(ins, self.fpscr_sz).ok_or(SuperHInstructionDecodeError::NotFound)?;

        if ret.level() > self.level {
            return Err(SuperHInstructionDecodeError::NotFound);
        }

        if !ret.features().iter().all(|rf| self.features.contains(rf)) {
            return Err(SuperHInstructionDecodeError::NotFound);
        }

        Ok(ret)
    }
}

#[cfg(feature = "yaxpeax")]
impl<'f> Default for SuperHDecoder<'f> {
    fn default() -> SuperHDecoder<'f> {
        SuperHDecoder {
            little_endian: true,
            fpscr_sz: false,
            level: SuperHLevel::Sh,
            features: &[],
        }
    }
}


#[cfg(feature = "yaxpeax")]
pub struct SuperH<'f>(PhantomData<&'f ()>);
#[cfg(feature = "yaxpeax")]
impl<'f> Arch for SuperH<'f> {
    type Address = u32;
    type Word = u8;
    type Instruction = SuperHInstruction;
    type DecodeError = SuperHInstructionDecodeError;
    type Decoder = SuperHDecoder<'f>;
    type Operand = ();
}
