use yaxpeax_ia64::InstDecoder;
use yaxpeax_arch::{Decoder, U8Reader};

// from elf64-ia64-vms.c
// 0x0b, 0x78, 0x00, 0x02, 0x00, 0x24, 0x00, 0x41, 0x3c, 0x70, 0x27, 0xc0, 0x01, 0x08, 0x00, 0x84
// [MMI] addl r15=0,r1;;
//       ld8.acq r16=[r15],8
//       mov r14=r1;;
// 0x11, 0x08, 0x00, 0x1e, 0x18, 0x10, 0x60, 0x80, 0x04, 0x80, 0x03, 0x00, 0x60, 0x00, 0x80, 0x00
// [MIB] ld8 r1=[r15]
//       mov b6=r16
//       br.few b6;;
// 0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0
// [MLX] nop.m 0
//       brl.sptk.few tgt;;

// from ia64 bash_4.2+dfsg-0.1+deb7u3_ia64:
// 0410 1c00 8045 024c 8009 0060 f013 1a60
// 0510 4118 0021 0000 0000 6020 0023 c86f
// 0908 2144 1814 a000 4444 0820 0100 c000
// 0100 0c50 2a04 1048 040a 40c0 0461 0084
// 0158 80fb f027 0082 f5e5 4f60 04ed c79f
// 0918 0016 1810 0002 8030 2080 04e9 b79f
// 0818 0146 1810 4002 9030 2000 0000 0400
// 1000 2806 9811 5002 2000 4200 50a5 ff58

#[test]
fn test_invalid_instruction() {
    let decoder = InstDecoder::default();

    let expected = "[MII] (p07) mov r16=r0; (p09) mov r14=r0;; purple;;";
    let data = [0xe3, 0x80, 0x00, 0x00, 0x00, 0x61, 0xe2, 0x00, 0x00, 0x00, 0x42, 0xc0, 0xe1, 0x80, 0x30, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    let expected = "[MII] purple.nt3; break.i 0x0; break.i 0x0";
    let data = [0x00, 0x00, 0x00, 0x00, 0x0E, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
}
#[test]
fn test_ad_hoc() {
    let decoder = InstDecoder::default();

    let expected = "[MII] nop.m 0x0; mov r2=ip;; addl r2=0x40,r2;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00, 0x00, 0x60, 0x00, 0x40, 0x00, 0x14, 0x00, 0x90];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    let expected = "[MMI] srlz.i;; mov r9=0x1; nop.i 0x0;;";
    let data = [0x0b, 0x00, 0x00, 0x00, 0x31, 0x00, 0x90, 0x08, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    let expected = "[MMI] mov pkr[r0]=r0;; mov rr[r0]=r5; mov r1=r0;;";
    let data = [0x0b, 0x00, 0x00, 0x00, 0x03, 0x04, 0x00, 0x28, 0x00, 0x00, 0x08, 0x20, 0x00, 0x00, 0x00, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
}
#[test]
fn test_shr_shl_dep_ext() {
    // encoding of immediates for dep/ext and their pseudo-ops (shl/shr) is weird.
    // `pos` is recorded as `63 - the_actual_value`? `len` is encoded as the actual length minus
    // one?
    //
    // there were enough quirks this warrants collecting them as a standalone test.

    let decoder = InstDecoder::default();

    // dep
    let expected = "[MMI] ld1 r17=[r17];; nop.m 0x0; dep r14=r18,r14,0x0,0x8";
    let data = [0x0a, 0x88, 0x00, 0x22, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0xc0, 0x21, 0x71, 0xdc, 0x4f];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; dep r14=r18,r14,0x8,0x8;; dep r14=r17,r14,0x10,0x8;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x90, 0x38, 0xee, 0x26, 0xc0, 0x11, 0x71, 0xdc, 0x4b];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; dep r14=r16,r14,0x18,0x8; nop.i 0x0;;";
    let data = [0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x80, 0x38, 0xee, 0x24, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; dep r14=r18,r14,0x0,0x8; nop.i 0x0";
    let data = [0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x90, 0x38, 0xee, 0x27, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; dep r14=r18,r14,0x8,0x8;; dep r14=r17,r14,0x10,0x8;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x90, 0x38, 0xee, 0x26, 0xc0, 0x11, 0x71, 0xdc, 0x4b];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; dep r14=r16,r14,0x18,0x8; nop.i 0x0;;";
    let data = [0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x80, 0x38, 0xee, 0x24, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    // extr
    let expected = "[MIB] nop.m 0x0; extr r36=r8,0x1f,0x1; nop.b 0x0";
    let data = [0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x40, 0xfa, 0x21, 0x00, 0x29, 0x00, 0x00, 0x00, 0x00, 0x20];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; extr r36=r36,0x6,0x1a;; shladd r36=r36,0x3,r14;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x40, 0x6a, 0x90, 0x32, 0x29, 0x80, 0x44, 0x72, 0x48, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; extr r17=r8,0x1f,0x1; nop.b 0x0";
    let data = [0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0xf9, 0x21, 0x00, 0x29, 0x00, 0x00, 0x00, 0x00, 0x20];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    // shr
    let expected = "[MII] nop.m 0x0; shr.u r36=r36,0x3a;; add r36=r36,r8;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x40, 0xa2, 0x93, 0x0a, 0x29, 0x80, 0x44, 0x42, 0x00, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] addl r14=0x3dd4,r1; nop.m 0x0; shr.u r17=r17,0x3a";
    let data = [0x08, 0x70, 0x50, 0x03, 0x7b, 0x24, 0x00, 0x00, 0x00, 0x02, 0x00, 0x20, 0x42, 0x8f, 0x14, 0x52];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] setf.sig f6=r16; shr r19=r16,0x3f; sub r17=r8,r17;;";
    let data = [0x01, 0x30, 0x40, 0x00, 0xe1, 0x18, 0x30, 0xf9, 0x43, 0x00, 0x29, 0x20, 0x82, 0x88, 0x14, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; shr r14=r14,0x12;; sub r14=r14,r19;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x28, 0x39, 0x5a, 0x29, 0xc0, 0xe1, 0x98, 0x14, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] (p07) adds r14=-0x1,r49; (p07) and r49=0x7,r49;; (p07) shr.u r14=r14,0x3;;";
    let data = [0xe3, 0x70, 0xfc, 0x63, 0x3f, 0xe3, 0x11, 0x3b, 0xc4, 0x58, 0xc0, 0xc3, 0x61, 0x70, 0xf0, 0x52];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    // shl
    let expected = "[MII] nop.m 0x0; shl r17=r14,0x5; sxt4 r19=r14;;";
    let data = [0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x71, 0xe8, 0xf4, 0x29, 0x60, 0x02, 0x70, 0x58, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; shl r19=r17,0x6;; sub r17=r19,r17;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x30, 0x89, 0xe4, 0xf2, 0x29, 0x20, 0x32, 0x89, 0x14, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] shladd r14=r17,0x3,r14;; nop.m 0x0; shl r14=r14,0x6;;";
    let data = [0x0b, 0x70, 0x44, 0x1c, 0x12, 0x20, 0x00, 0x00, 0x00, 0x02, 0x00, 0xc0, 0xe1, 0xc8, 0xe5, 0x53];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    // shladd is special and not complicated like the above
    let expected = "[MII] nop.m 0x0; extr r36=r36,0x6,0x1a;; shladd r36=r36,0x3,r14;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x40, 0x6a, 0x90, 0x32, 0x29, 0x80, 0x44, 0x72, 0x48, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
}

#[test]
fn test_addl_imm() {
    let decoder = InstDecoder::default();

    let expected = "[MMI] adds r39=-0x1,r36; addl r14=0x8f40,r1; adds r36=-0x3,r36;;";
    let data = [0x09, 0x38, 0xfd, 0x49, 0x3f, 0x23, 0xe0, 0x00, 0x06, 0x3c, 0x4a, 0x80, 0xd4, 0x27, 0xfd, 0x8c];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
}

#[test]
fn test_mlx_bundle() {
    let decoder = InstDecoder::default();

    let expected = "[MLX] alloc r34=ar.pfs,5,5,0; movl r32=0xfffffffffffff5f8;;";
    let data = [0x05, 0x10, 0x15, 0x0a, 0x80, 0xc5, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x00, 0x84, 0xf7, 0xaf, 0x6f];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    let expected = "[MLX] ld4 r16=[r14]; movl r14=0x431bde82d7b634db";
    let data = [0x04, 0x80, 0x00, 0x1c, 0x10, 0x90, 0xd7, 0x82, 0xde, 0x1b, 0x43, 0xc0, 0xb1, 0xd5, 0xa6, 0x61];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
}

#[test]
fn test_br_cosmetics() {
    let decoder = InstDecoder::default();

    let data = [0x19, 0x50, 0x40, 0x19, 0x3f, 0x23, 0x80, 0x00, 0x00, 0x00, 0x48, 0x80, 0x00, 0x00, 0x84, 0x02];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    let stringy = format!("{}", inst);
    assert_eq!(stringy, "[MMB] adds r10=-0x30,r12; mov r8=0x0; br.ret.dptk.few b0;;");
    let expected = "[MMB] cmp.eq p6,p7=0x0,r32; adds r36=0x1,r32; (p06) br.cond.dpnt.few $+0x5d0;;";
    let data = [0x19, 0x30, 0x00, 0x40, 0x07, 0x39, 0x40, 0x0a, 0x80, 0x00, 0x42, 0x03, 0xd0, 0x05, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] alloc r2=ar.pfs,3,0,0; nop.i 0x0; br.many $-0x2a0;;";
    let data = [0x11, 0x10, 0x0c, 0x00, 0x80, 0x05, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x68, 0xfd, 0xff, 0x48];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r41=r32; nop.i 0x0; br.call.sptk.many b0=$-0x1745d0;;";
    let data = [0x11, 0x48, 0x01, 0x40, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x38, 0xba, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] nop.m 0x0; adds r12=0x10,r12; br.ret.sptk.many b0;;";
    let data = [0x19, 0x00, 0x00, 0x00, 0x01, 0x00, 0xc0, 0x80, 0x30, 0x00, 0x42, 0x80, 0x08, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    // TODO: need tests for `B6` and `B7` operand encodings, for `brp` and `brp.ret`.
    // TODO: need tests for `X3` and `X4` operand encodings, for `brl` and `brl.call`.
}

// from ia64 bash_4.2+dfsg-0.1+deb7u3_ia64 via binutils-ia64-linux-gnu=2.30-21ubuntu1~18.04.4
// bfd reports immediates in decimal in many places. all immediates are corrected to hex, as
// yaxpeax-ia64 only reports immediates as hexadecimal.
#[test]
fn test_bash_tilde_expand_word() {
    // 400000000019b100 <tilde_expand_word@@Base>:
    let decoder = InstDecoder::default();

    let expected = "[MMI] alloc r39=ar.pfs,12,9,0; adds r12=-0x10,r12; mov r38=b0";
    let data = [0x08, 0x38, 0x31, 0x12, 0x80, 0x05, 0xc0, 0x80, 0x33, 0x7e, 0x46, 0xc0, 0x04, 0x00, 0xc4, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] cmp.eq p6,p7=0x0,r32; adds r36=0x1,r32; (p06) br.cond.dpnt.few $+0x5d0;;";
    let data = [0x19, 0x30, 0x00, 0x40, 0x07, 0x39, 0x40, 0x0a, 0x80, 0x00, 0x42, 0x03, 0xd0, 0x05, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld1 r14=[r32]; mov r40=r1; mov r33=r32;;";
    let data = [0x09, 0x70, 0x00, 0x40, 0x00, 0x10, 0x80, 0x02, 0x04, 0x00, 0x42, 0x20, 0x04, 0x00, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; sxt1 r14=r14; nop.i 0x0;;";
    let data = [0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x00, 0x38, 0x28, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] cmp4.eq p6,p7=0x7e,r14; nop.i 0x0; (p07) br.cond.dpnt.few $+0x2d0;;";
    let data = [0x11, 0x30, 0xf8, 0x1d, 0x87, 0x39, 0x00, 0x00, 0x00, 0x02, 0x80, 0x03, 0xd0, 0x02, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; ld1 r14=[r36]; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x00, 0x90, 0x00, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; sxt1 r14=r14;; cmp4.eq p7,p6=0x2f,r14;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x00, 0x38, 0x28, 0x00, 0xe0, 0xf0, 0x72, 0x18, 0xe6];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; cmp4.eq.or.andcm p7,p6=0x0,r14; (p06) br.cond.dptk.few $+0x80";
    let data = [0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x70, 0x00, 0x38, 0x8c, 0x73, 0x03, 0x80, 0x00, 0x00, 0x42];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] addl r41=-0x628,r1; nop.m 0x0; mov r34=0x1;;";
    let data = [0x09, 0x48, 0x61, 0xfb, 0xf3, 0x27, 0x00, 0x00, 0x00, 0x02, 0x00, 0x40, 0x14, 0x00, 0x00, 0x90];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] ld8 r41=[r41]; nop.i 0x0; br.call.sptk.many b0=$-0x12ad10;;";
    let data = [0x11, 0x48, 0x01, 0x52, 0x18, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xf8, 0x52, 0xed, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] cmp.eq p7,p6=0x0,r8; nop.m 0x0; mov r1=r40";
    let data = [0x08, 0x38, 0x00, 0x10, 0x06, 0x39, 0x00, 0x00, 0x00, 0x02, 0x00, 0x20, 0x00, 0x40, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] mov r32=r8; nop.m 0x0; (p07) br.cond.dpnt.few $+0x380;;";
    let data = [0x19, 0x00, 0x01, 0x10, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x80, 0x03, 0x80, 0x03, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; mov.i ar.pfs=r39; nop.i 0x0;;";
    let data = [0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x38, 0x01, 0x55, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; mov b0=r38; adds r12=0x10,r12;;";
    let data = [0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x30, 0x05, 0x80, 0x03, 0x80, 0x01, 0x61, 0x00, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] alloc r2=ar.pfs,3,0,0; nop.i 0x0; br.many $-0x2a0;;";
    let data = [0x11, 0x10, 0x0c, 0x00, 0x80, 0x05, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x68, 0xfd, 0xff, 0x48];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r41=r32; nop.i 0x0; br.call.sptk.many b0=$-0x1745d0;;";
    let data = [0x11, 0x48, 0x01, 0x40, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x38, 0xba, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r1=r40; mov r41=r8; br.call.sptk.many b0=$-0xa0700;;";
    let data = [0x11, 0x08, 0x00, 0x50, 0x00, 0x21, 0x90, 0x02, 0x20, 0x00, 0x42, 0x00, 0x08, 0xf9, 0xf5, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r35=r8; ld1 r15=[r36]; mov r1=r40;;";
    let data = [0x09, 0x18, 0x01, 0x10, 0x00, 0x21, 0xf0, 0x00, 0x90, 0x00, 0x20, 0x20, 0x00, 0x40, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; sxt1 r15=r15;; cmp4.eq p7,p6=0x0,r15";
    let data = [0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0xf0, 0x00, 0x3c, 0x28, 0x00, 0xe0, 0x00, 0x78, 0x18, 0xe6];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] cmp4.eq p9,p8=0x2f,r15;; (p06) mov r16=0x1; (p08) mov r14=0x1;;";
    let data = [0x0b, 0x48, 0xbc, 0x1e, 0x88, 0xb9, 0x01, 0x09, 0x00, 0x00, 0x48, 0xc4, 0x11, 0x00, 0x00, 0x90];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] (p07) mov r16=r0; (p09) mov r14=r0;; and r14=r14,r16;;";
    let data = [0xe3, 0x80, 0x00, 0x00, 0x00, 0x61, 0xe2, 0x00, 0x00, 0x00, 0x42, 0xc0, 0xe1, 0x80, 0x30, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] cmp4.eq p6,p7=0x0,r14; nop.i 0x0; (p06) br.cond.dpnt.few $+0x320;;";
    let data = [0x11, 0x30, 0x00, 0x1c, 0x87, 0x39, 0x00, 0x00, 0x00, 0x02, 0x00, 0x03, 0x20, 0x03, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] (p07) mov r16=0x1; (p07) adds r17=0x2,r32; (p07) mov r14=0x2;;";
    let data = [0xe9, 0x80, 0x04, 0x00, 0x00, 0xe4, 0x11, 0x11, 0x80, 0x00, 0xc2, 0xc3, 0x21, 0x00, 0x00, 0x90];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] add r18=r35,r16; nop.m 0x0; adds r19=0x1,r14";
    let data = [0x08, 0x90, 0x8c, 0x20, 0x00, 0x20, 0x00, 0x00, 0x00, 0x02, 0x00, 0x60, 0x12, 0x70, 0x00, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] mov r16=r14; mov r36=r14;; adds r18=-0x1,r18;;";
    let data = [0x03, 0x80, 0x00, 0x1c, 0x00, 0x21, 0x40, 0x02, 0x38, 0x00, 0x42, 0x40, 0xf2, 0x97, 0xfc, 0x8c];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] st1 [r18]=r15;; ld1 r15=[r17],0x1; nop.i 0x0;;";
    let data = [0x0b, 0x00, 0x3c, 0x24, 0x80, 0x11, 0xf0, 0x08, 0x44, 0x00, 0x28, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; sxt1 r15=r15;; cmp4.eq p7,p6=0x0,r15;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0xf0, 0x00, 0x3c, 0x28, 0x00, 0xe0, 0x00, 0x78, 0x18, 0xe6];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; cmp4.eq.or.andcm p7,p6=0x2f,r15; (p07) br.cond.dpnt.few $+0x60;;";
    let data = [0x11, 0x00, 0x00, 0x00, 0x01, 0x00, 0x70, 0x78, 0x3d, 0x8c, 0xf3, 0x03, 0x60, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] add r18=r35,r16; mov r14=r19;; adds r18=-0x1,r18";
    let data = [0x02, 0x90, 0x8c, 0x20, 0x00, 0x20, 0xe0, 0x00, 0x4c, 0x00, 0x42, 0x40, 0xf2, 0x97, 0xfc, 0x8c];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r36=r14; mov r16=r14; adds r19=0x1,r14;;";
    let data = [0x09, 0x20, 0x01, 0x1c, 0x00, 0x21, 0x00, 0x01, 0x38, 0x00, 0x42, 0x60, 0x12, 0x70, 0x00, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] st1 [r18]=r15;; ld1 r15=[r17],0x1; nop.i 0x0;;";
    let data = [0x0b, 0x00, 0x3c, 0x24, 0x80, 0x11, 0xf0, 0x08, 0x44, 0x00, 0x28, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; sxt1 r15=r15;; cmp4.eq p7,p6=0x0,r15;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0xf0, 0x00, 0x3c, 0x28, 0x00, 0xe0, 0x00, 0x78, 0x18, 0xe6];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; cmp4.eq.or.andcm p7,p6=0x2f,r15; (p06) br.cond.dptk.few $-0x40";
    let data = [0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x70, 0x78, 0x3d, 0x8c, 0x73, 0x03, 0xc0, 0xff, 0xff, 0x4a];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] adds r14=-0x1,r14; nop.m 0x0; nop.i 0x0;;";
    let data = [0x09, 0x70, 0xfc, 0x1d, 0x3f, 0x23, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] add r14=r35,r14; nop.m 0x0; mov r41=r35;;";
    let data = [0x09, 0x70, 0x8c, 0x1c, 0x00, 0x20, 0x00, 0x00, 0x00, 0x02, 0x00, 0x20, 0x05, 0x18, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] st1 [r14]=r0; nop.m 0x0; addl r14=0x4ce8,r1;;";
    let data = [0x09, 0x00, 0x00, 0x1c, 0x80, 0x11, 0x00, 0x00, 0x00, 0x02, 0x00, 0xc0, 0x81, 0x0e, 0x64, 0x92];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; ld8 r8=[r14]; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0x80, 0x00, 0x38, 0x30, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] cmp.eq p6,p7=0x0,r8; nop.i 0x0; (p06) br.cond.dpnt.few $+0x120;;";
    let data = [0x11, 0x30, 0x00, 0x10, 0x07, 0x39, 0x00, 0x00, 0x00, 0x02, 0x00, 0x03, 0x20, 0x01, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld8 r14=[r8],0x8;; nop.m 0x0; mov b6=r14";
    let data = [0x0a, 0x70, 0x20, 0x10, 0x18, 0x14, 0x00, 0x00, 0x00, 0x02, 0x00, 0xc0, 0xe0, 0x08, 0x00, 0x07];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] ld8 r1=[r8]; nop.m 0x0; br.call.sptk.many b0=b6;;";
    let data = [0x19, 0x08, 0x00, 0x10, 0x18, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x68, 0x00, 0x80, 0x10];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] cmp.eq p6,p7=0x0,r8; mov r1=r40; mov r42=r33";
    let data = [0x08, 0x30, 0x00, 0x10, 0x07, 0x39, 0x10, 0x00, 0xa0, 0x00, 0x42, 0x40, 0x05, 0x08, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] mov r43=r36; mov r37=r8; (p06) br.cond.dpnt.few $+0xe0;;";
    let data = [0x19, 0x58, 0x01, 0x48, 0x00, 0x21, 0x50, 0x02, 0x20, 0x00, 0x42, 0x03, 0xe0, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r41=r8; nop.i 0x0; br.call.sptk.many b0=$-0x460;;";
    let data = [0x11, 0x48, 0x01, 0x10, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xa8, 0xfb, 0xff, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] adds r14=0x10,r12; mov r1=r40; mov r41=r35;;";
    let data = [0x09, 0x70, 0x40, 0x18, 0x00, 0x21, 0x10, 0x00, 0xa0, 0x00, 0x42, 0x20, 0x05, 0x18, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] st8 [r14]=r8; nop.i 0x0; br.call.sptk.many b0=$-0xa0600;;";
    let data = [0x11, 0x00, 0x20, 0x1c, 0x98, 0x11, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x08, 0xfa, 0xf5, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r1=r40; mov r41=r37; br.call.sptk.many b0=$-0xa0610;;";
    let data = [0x11, 0x08, 0x00, 0x50, 0x00, 0x21, 0x90, 0x02, 0x94, 0x00, 0x42, 0x00, 0xf8, 0xf9, 0xf5, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] adds r14=0x10,r12; nop.m 0x0; mov r1=r40;;";
    let data = [0x09, 0x70, 0x40, 0x18, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x20, 0x00, 0x40, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] ld8 r8=[r14]; mov.i ar.pfs=r39;; mov b0=r38";
    let data = [0x02, 0x40, 0x00, 0x1c, 0x18, 0x10, 0x00, 0x38, 0x01, 0x55, 0x00, 0x00, 0x60, 0x0a, 0x00, 0x07];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] nop.m 0x0; adds r12=0x10,r12; br.ret.sptk.many b0;;";
    let data = [0x19, 0x00, 0x00, 0x00, 0x01, 0x00, 0xc0, 0x80, 0x30, 0x00, 0x42, 0x80, 0x08, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r41=r32; nop.i 0x0; br.call.sptk.many b0=$-0x1747f0;;";
    let data = [0x11, 0x48, 0x01, 0x40, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x18, 0xb8, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r1=r40; adds r41=0x1,r8; br.call.sptk.many b0=$-0xa0920;;";
    let data = [0x11, 0x08, 0x00, 0x50, 0x00, 0x21, 0x90, 0x0a, 0x20, 0x00, 0x42, 0x00, 0xe8, 0xf6, 0xf5, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r1=r40; nop.m 0x0; mov r41=r8";
    let data = [0x08, 0x08, 0x00, 0x50, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x20, 0x05, 0x40, 0x00, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] mov r42=r32; nop.m 0x0; br.call.sptk.many b0=$-0x174c60;;";
    let data = [0x19, 0x50, 0x01, 0x40, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xa8, 0xb3, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] mov r1=r40; mov.i ar.pfs=r39;; mov b0=r38";
    let data = [0x02, 0x08, 0x00, 0x50, 0x00, 0x21, 0x00, 0x38, 0x01, 0x55, 0x00, 0x00, 0x60, 0x0a, 0x00, 0x07];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] nop.m 0x0; adds r12=0x10,r12; br.ret.sptk.many b0;;";
    let data = [0x19, 0x00, 0x00, 0x00, 0x01, 0x00, 0xc0, 0x80, 0x30, 0x00, 0x42, 0x80, 0x08, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r41=r35; nop.i 0x0; br.call.sptk.many b0=$-0x175610;;";
    let data = [0x11, 0x48, 0x01, 0x46, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xf8, 0xa9, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] adds r14=0x20,r8; cmp.eq p7,p6=0x0,r8; mov r1=r40";
    let data = [0x08, 0x70, 0x80, 0x10, 0x00, 0x21, 0x70, 0x00, 0x20, 0x0c, 0x72, 0x20, 0x00, 0x40, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] mov r42=r33; mov r43=r36; (p07) br.cond.dpnt.few $+0xf0;;";
    let data = [0x19, 0x50, 0x01, 0x42, 0x00, 0x21, 0xb0, 0x02, 0x90, 0x00, 0xc2, 0x03, 0xf0, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MBB] ld8 r41=[r14]; br.call.sptk.many b0=$-0x560; nop.b 0x0;;";
    let data = [0x13, 0x48, 0x01, 0x1c, 0x18, 0x10, 0x00, 0x54, 0xfd, 0x7f, 0x2c, 0x00, 0x00, 0x00, 0x00, 0x20];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r1=r40; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x08, 0x00, 0x50, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] adds r14=0x10,r12; nop.m 0x0; mov r41=r35;;";
    let data = [0x09, 0x70, 0x40, 0x18, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x20, 0x05, 0x18, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] st8 [r14]=r8; nop.i 0x0; br.call.sptk.many b0=$-0xa0710;;";
    let data = [0x11, 0x00, 0x20, 0x1c, 0x98, 0x11, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xf8, 0xf8, 0xf5, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r1=r40; nop.i 0x0; br.call.sptk.many b0=$-0x1743a0;;";
    let data = [0x11, 0x08, 0x00, 0x50, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x68, 0xbc, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] adds r14=0x10,r12; nop.m 0x0; mov r1=r40;;";
    let data = [0x09, 0x70, 0x40, 0x18, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x20, 0x00, 0x40, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld8 r8=[r14]; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x40, 0x00, 0x1c, 0x18, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; mov.i ar.pfs=r39;; mov b0=r38";
    let data = [0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x38, 0x01, 0x55, 0x00, 0x00, 0x60, 0x0a, 0x00, 0x07];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] nop.m 0x0; adds r12=0x10,r12; br.ret.sptk.many b0;;";
    let data = [0x19, 0x00, 0x00, 0x00, 0x01, 0x00, 0xc0, 0x80, 0x30, 0x00, 0x42, 0x80, 0x08, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r34=0x1; nop.i 0x0; br.call.sptk.many b0=$-0x12d7b0;;";
    let data = [0x11, 0x10, 0x05, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x58, 0x28, 0xed, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r32=r8; mov r1=r40; mov.i ar.pfs=r39;;";
    let data = [0x09, 0x00, 0x01, 0x10, 0x00, 0x21, 0x10, 0x00, 0xa0, 0x00, 0x42, 0x00, 0x70, 0x02, 0xaa, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; mov b0=r38; adds r12=0x10,r12;;";
    let data = [0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x30, 0x05, 0x80, 0x03, 0x80, 0x01, 0x61, 0x00, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] alloc r2=ar.pfs,3,0,0; nop.i 0x0; br.many $-0x620";
    let data = [0x10, 0x10, 0x0c, 0x00, 0x80, 0x05, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xe8, 0xf9, 0xff, 0x48];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r14=r0; mov r36=0x1; br.few $-0x250;;";
    let data = [0x11, 0x70, 0x00, 0x00, 0x00, 0x21, 0x40, 0x0a, 0x00, 0x00, 0x48, 0x00, 0xb0, 0xfd, 0xff, 0x48];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] addl r14=0x4ce0,r1; nop.m 0x0; mov r41=r35;;";
    let data = [0x09, 0x70, 0x80, 0x03, 0x99, 0x24, 0x00, 0x00, 0x00, 0x02, 0x00, 0x20, 0x05, 0x18, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; ld8 r8=[r14]; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0x80, 0x00, 0x38, 0x30, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] cmp.eq p6,p7=0x0,r8; nop.i 0x0; (p06) br.cond.dpnt.few $+0xb0;;";
    let data = [0x11, 0x30, 0x00, 0x10, 0x07, 0x39, 0x00, 0x00, 0x00, 0x02, 0x00, 0x03, 0xb0, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld8 r14=[r8],0x8;; nop.m 0x0; mov b6=r14";
    let data = [0x0a, 0x70, 0x20, 0x10, 0x18, 0x14, 0x00, 0x00, 0x00, 0x02, 0x00, 0xc0, 0xe0, 0x08, 0x00, 0x07];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] ld8 r1=[r8]; nop.m 0x0; br.call.sptk.many b0=b6;;";
    let data = [0x19, 0x08, 0x00, 0x10, 0x18, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x68, 0x00, 0x80, 0x10];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] cmp.eq p6,p7=0x0,r8; mov r1=r40; mov r37=r8";
    let data = [0x08, 0x30, 0x00, 0x10, 0x07, 0x39, 0x10, 0x00, 0xa0, 0x00, 0x42, 0xa0, 0x04, 0x40, 0x00, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] mov r43=r36; mov r42=r33; (p06) br.cond.dpnt.few $+0x70;;";
    let data = [0x19, 0x58, 0x01, 0x48, 0x00, 0x21, 0xa0, 0x02, 0x84, 0x00, 0x42, 0x03, 0x70, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r41=r8; nop.i 0x0; br.call.sptk.many b0=$-0x6b0;;";
    let data = [0x11, 0x48, 0x01, 0x10, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x58, 0xf9, 0xff, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] adds r14=0x10,r12; mov r1=r40; mov r41=r37;;";
    let data = [0x09, 0x70, 0x40, 0x18, 0x00, 0x21, 0x10, 0x00, 0xa0, 0x00, 0x42, 0x20, 0x05, 0x28, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] st8 [r14]=r8; nop.i 0x0; br.call.sptk.many b0=$-0xa0850;;";
    let data = [0x11, 0x00, 0x20, 0x1c, 0x98, 0x11, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xb8, 0xf7, 0xf5, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] adds r14=0x10,r12; nop.m 0x0; mov r1=r40;;";
    let data = [0x09, 0x70, 0x40, 0x18, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x20, 0x00, 0x40, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; ld8 r8=[r14]; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0x80, 0x00, 0x38, 0x30, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; cmp.eq p7,p6=0x0,r8; (p06) br.cond.sptk.few $-0x180";
    let data = [0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x70, 0x00, 0x20, 0x0c, 0x72, 0x03, 0x80, 0xfe, 0xff, 0x48];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r41=r33; nop.i 0x0; br.call.sptk.many b0=$-0x174a30;;";
    let data = [0x11, 0x48, 0x01, 0x42, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xd8, 0xb5, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r1=r40; adds r41=0x1,r8; br.call.sptk.many b0=$-0xa0b60;;";
    let data = [0x11, 0x08, 0x00, 0x50, 0x00, 0x21, 0x90, 0x0a, 0x20, 0x00, 0x42, 0x00, 0xa8, 0xf4, 0xf5, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r42=r33; nop.m 0x0; mov r1=r40";
    let data = [0x08, 0x50, 0x01, 0x42, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x20, 0x00, 0x40, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] mov r41=r8; nop.m 0x0; br.call.sptk.many b0=$-0x174ea0;;";
    let data = [0x19, 0x48, 0x01, 0x10, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x68, 0xb1, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] adds r14=0x10,r12; mov r1=r40; mov r41=r35;;";
    let data = [0x09, 0x70, 0x40, 0x18, 0x00, 0x21, 0x10, 0x00, 0xa0, 0x00, 0x42, 0x20, 0x05, 0x18, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] st8 [r14]=r8; nop.i 0x0; br.call.sptk.many b0=$-0xa08e0;;";
    let data = [0x11, 0x00, 0x20, 0x1c, 0x98, 0x11, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x28, 0xf7, 0xf5, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r1=r40; nop.i 0x0; br.call.sptk.many b0=$-0x174570;;";
    let data = [0x11, 0x08, 0x00, 0x50, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x98, 0xba, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] adds r14=0x10,r12; nop.m 0x0; mov r1=r40;;";
    let data = [0x09, 0x70, 0x40, 0x18, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x20, 0x00, 0x40, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] ld8 r8=[r14]; nop.i 0x0; br.few $-0x1c0;;";
    let data = [0x11, 0x40, 0x00, 0x1c, 0x18, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x40, 0xfe, 0xff, 0x48];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] mov r8=r0; mov.i ar.pfs=r39;; mov b0=r38";
    let data = [0x02, 0x40, 0x00, 0x00, 0x00, 0x21, 0x00, 0x38, 0x01, 0x55, 0x00, 0x00, 0x60, 0x0a, 0x00, 0x07];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] nop.m 0x0; adds r12=0x10,r12; br.ret.sptk.many b0;;";
    let data = [0x19, 0x00, 0x00, 0x00, 0x01, 0x00, 0xc0, 0x80, 0x30, 0x00, 0x42, 0x80, 0x08, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
}

// from ia64 bash_4.2+dfsg-0.1+deb7u3_ia64 via binutils-ia64-linux-gnu=2.30-21ubuntu1~18.04.4
// bfd reports immediates in decimal in many places. all immediates are corrected to hex, as
// yaxpeax-ia64 only reports immediates as hexadecimal.
#[test]
fn test_bash_rl_insert_close() {
    // 400000000019bdc0 <rl_insert_close@@Base>:
    let decoder = InstDecoder::default();

    let expected = "[MMB] alloc r44=ar.pfs,20,15,0; adds r12=-0x90,r12; nop.b 0x0";
    let data = [0x18, 0x60, 0x51, 0x1e, 0x80, 0x05, 0xc0, 0x80, 0x33, 0x7c, 0x46, 0x00, 0x00, 0x00, 0x00, 0x20];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] addl r14=0x47d4,r1; mov.i r46=ar.lc; mov r45=r1;;";
    let data = [0x01, 0x70, 0x50, 0x03, 0x8f, 0x24, 0xe0, 0x02, 0x04, 0x65, 0x00, 0xa0, 0x05, 0x08, 0x00, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; nop.m 0x0; mov r43=b0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x60, 0x05, 0x00, 0xc4, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; ld4 r14=[r14]; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x00, 0x38, 0x20, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] cmp4.eq p7,p6=0x0,r14; addl r14=0x3dd0,r1; (p06) br.cond.dpnt.few $+0x30;;";
    let data = [0x11, 0x38, 0x00, 0x1c, 0x86, 0x39, 0xe0, 0x80, 0x06, 0xf6, 0x48, 0x03, 0x30, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; ld4 r14=[r14]; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x00, 0x38, 0x20, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; cmp4.eq p7,p6=0x0,r14; (p06) br.cond.dpnt.few $+0x50";
    let data = [0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x70, 0x00, 0x38, 0x0c, 0x73, 0x03, 0x50, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r47=r32; mov r48=r33; br.call.sptk.many b0=$-0x10ab0;;";
    let data = [0x11, 0x78, 0x01, 0x40, 0x00, 0x21, 0x00, 0x03, 0x84, 0x00, 0x42, 0x00, 0x58, 0xf5, 0xfe, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r8=r0; mov r1=r45; mov.i ar.pfs=r44;;";
    let data = [0x09, 0x40, 0x00, 0x00, 0x00, 0x21, 0x10, 0x00, 0xb4, 0x00, 0x42, 0x00, 0xc0, 0x02, 0xaa, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; mov.i ar.lc=r46;; mov b0=r43";
    let data = [0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x70, 0x05, 0x55, 0x00, 0x00, 0xb0, 0x0a, 0x00, 0x07];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] nop.m 0x0; adds r12=0x90,r12; br.ret.sptk.many b0;;";
    let data = [0x19, 0x00, 0x00, 0x00, 0x01, 0x00, 0xc0, 0x80, 0x30, 0x02, 0x42, 0x80, 0x08, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] addl r41=0x3d48,r1; addl r34=0x4ef0,r1; nop.i 0x0";
    let data = [0x08, 0x48, 0x21, 0x03, 0x7a, 0x24, 0x20, 0x82, 0x07, 0x3a, 0x49, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] mov r47=0x1; mov r48=r33; br.call.sptk.many b0=$-0x10b00;;";
    let data = [0x19, 0x78, 0x05, 0x00, 0x00, 0x24, 0x00, 0x03, 0x84, 0x00, 0x42, 0x00, 0x08, 0xf5, 0xfe, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; nop.m 0x0; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld8 r14=[r41];; ld8 r15=[r14],0x8; nop.i 0x0;;";
    let data = [0x0b, 0x70, 0x00, 0x52, 0x18, 0x10, 0xf0, 0x40, 0x38, 0x30, 0x28, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] ld8 r1=[r14]; mov b6=r15; br.call.sptk.many b0=b6;;";
    let data = [0x11, 0x08, 0x00, 0x1c, 0x18, 0x10, 0x60, 0x78, 0x04, 0x80, 0x03, 0x00, 0x68, 0x00, 0x80, 0x10];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld4 r37=[r34]; mov r1=r45; cmp4.eq p6,p7=0x5d,r33;;";
    let data = [0x09, 0x28, 0x01, 0x44, 0x10, 0x10, 0x10, 0x00, 0xb4, 0x00, 0x42, 0xc0, 0xd0, 0x0d, 0x1d, 0xe6];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] addl r14=0x4760,r1;; nop.m 0x0; nop.i 0x0;;";
    let data = [0x0b, 0x70, 0x80, 0x03, 0x8e, 0x24, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] ld8 r14=[r14]; nop.i 0x0; (p06) br.cond.dpnt.few $+0x480;;";
    let data = [0x11, 0x70, 0x00, 0x1c, 0x18, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x03, 0x80, 0x04, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] cmp4.eq p6,p7=0x7d,r33; nop.i 0x0; (p06) br.cond.dpnt.few $+0x60;;";
    let data = [0x11, 0x30, 0xf4, 0x43, 0x87, 0x39, 0x00, 0x00, 0x00, 0x02, 0x00, 0x03, 0x60, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; cmp4.eq p6,p7=0x29,r33; (p06) br.cond.dpnt.few $+0x450";
    let data = [0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x60, 0x48, 0x85, 0x0e, 0x73, 0x03, 0x50, 0x04, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r8=-0x1; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x40, 0xfc, 0xf9, 0xff, 0x27, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; mov.i ar.pfs=r44; nop.i 0x0;;";
    let data = [0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x60, 0x01, 0x55, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; mov.i ar.lc=r46;; mov b0=r43";
    let data = [0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x70, 0x05, 0x55, 0x00, 0x00, 0xb0, 0x0a, 0x00, 0x07];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] nop.m 0x0; adds r12=0x90,r12; br.ret.sptk.many b0";
    let data = [0x18, 0x00, 0x00, 0x00, 0x01, 0x00, 0xc0, 0x80, 0x30, 0x02, 0x42, 0x80, 0x08, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r42=0x7b; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x50, 0xed, 0x01, 0x00, 0x24, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] adds r37=-0x2,r37; nop.m 0x0; addl r40=0x3cc0,r1";
    let data = [0x08, 0x28, 0xf9, 0x4b, 0x3f, 0x23, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x05, 0x0c, 0xe4, 0x91];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r39=0x1; nop.m 0x0; mov r36=r0;;";
    let data = [0x09, 0x38, 0x05, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x02, 0x00, 0x80, 0x04, 0x00, 0x00, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; sxt4 r35=r37; addp4 r15=r37,r0";
    let data = [0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x30, 0x02, 0x94, 0x2c, 0x00, 0xe0, 0x51, 0x02, 0x20, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] nop.m 0x0; cmp4.lt p6,p7=r37,r0; (p06) br.cond.dpnt.few $-0x80;;";
    let data = [0x19, 0x00, 0x00, 0x00, 0x01, 0x00, 0x60, 0x28, 0x01, 0x0e, 0x61, 0x03, 0x80, 0xff, 0xff, 0x4b];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; add r35=r14,r35; mov.i ar.lc=r15;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0x30, 0x72, 0x8c, 0x00, 0x40, 0x00, 0xf0, 0x08, 0xaa, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; cmp4.eq p6,p7=0x0,r36; (p06) br.cond.dptk.few $+0x40;;";
    let data = [0x11, 0x00, 0x00, 0x00, 0x01, 0x00, 0x60, 0x00, 0x90, 0x0e, 0x73, 0x03, 0x40, 0x00, 0x00, 0x42];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld1 r14=[r35];; nop.m 0x0; sxt1 r14=r14;;";
    let data = [0x0b, 0x70, 0x00, 0x46, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0xc0, 0x01, 0x70, 0x50, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] cmp4.eq p6,p7=r14,r36;; (p06) mov r36=r0; nop.i 0x0;;";
    let data = [0x0b, 0x30, 0x38, 0x48, 0x87, 0xb8, 0x41, 0x02, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; nop.i 0x0; (p06) br.cond.dpnt.few $+0x80";
    let data = [0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x03, 0x80, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; ld8 r47=[r40]; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0xf0, 0x02, 0xa0, 0x30, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] cmp.eq p6,p7=0x0,r47; nop.i 0x0; (p06) br.cond.dpnt.few $+0x90;;";
    let data = [0x11, 0x30, 0x00, 0x5e, 0x07, 0x39, 0x00, 0x00, 0x00, 0x02, 0x00, 0x03, 0x90, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld1 r38=[r35];; nop.m 0x0; sxt1 r38=r38;;";
    let data = [0x0b, 0x30, 0x01, 0x46, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0xc0, 0x04, 0x30, 0x51, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r48=r38; nop.i 0x0; br.call.sptk.many b0=$-0x175430;;";
    let data = [0x11, 0x80, 0x01, 0x4c, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xd8, 0xab, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; cmp.eq p7,p6=0x0,r8; mov r1=r45";
    let data = [0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x70, 0x00, 0x20, 0x0c, 0x72, 0x20, 0x00, 0x68, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] nop.m 0x0; nop.m 0x0; (p07) br.cond.dpnt.few $+0x50;;";
    let data = [0x19, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x80, 0x03, 0x50, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] (p06) mov r36=r38; nop.m 0x0; nop.i 0x0";
    let data = [0xc8, 0x20, 0x01, 0x4c, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; cmp4.eq p6,p7=0x0,r39; (p06) br.cond.dpnt.few $+0x90";
    let data = [0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x60, 0x00, 0x9c, 0x0e, 0x73, 0x03, 0x90, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] adds r37=-0x1,r37; adds r35=-0x1,r35; br.cloop.sptk.few $-0xc0";
    let data = [0x10, 0x28, 0xfd, 0x4b, 0x3f, 0x23, 0x30, 0xfa, 0x8f, 0x7e, 0x46, 0xa0, 0x40, 0xff, 0xff, 0x48];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; mov r8=-0x1; br.few $-0x170;;";
    let data = [0x11, 0x00, 0x00, 0x00, 0x01, 0x00, 0x80, 0xf8, 0xf3, 0xff, 0x4f, 0x00, 0x90, 0xfe, 0xff, 0x48];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; cmp4.eq p7,p6=0x0,r36; (p06) br.cond.dptk.few $-0x30";
    let data = [0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x70, 0x00, 0x90, 0x0c, 0x73, 0x03, 0xd0, 0xff, 0xff, 0x4a];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld1 r14=[r35];; nop.m 0x0; sxt1 r14=r14;;";
    let data = [0x0b, 0x70, 0x00, 0x46, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0xc0, 0x01, 0x70, 0x50, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] cmp4.eq p7,p6=r14,r33;; (p07) adds r39=0x1,r39; nop.i 0x0;;";
    let data = [0x0b, 0x38, 0x38, 0x42, 0x86, 0xf8, 0x71, 0x0a, 0x9c, 0x00, 0x42, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; nop.i 0x0; (p07) br.cond.dpnt.few $-0x60;;";
    let data = [0x11, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x80, 0x03, 0xa0, 0xff, 0xff, 0x4b];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] cmp4.eq p7,p6=r42,r14;; (p07) adds r39=-0x1,r39; nop.i 0x0;;";
    let data = [0x0b, 0x38, 0xa8, 0x1c, 0x86, 0xf8, 0x71, 0xfa, 0x9f, 0x7e, 0x46, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; cmp4.eq p6,p7=0x0,r39; (p07) br.cond.dptk.few $-0x70;;";
    let data = [0x11, 0x00, 0x00, 0x00, 0x01, 0x00, 0x60, 0x00, 0x9c, 0x0e, 0xf3, 0x03, 0x90, 0xff, 0xff, 0x4a];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; adds r14=0x20,r12; mov.i ar.lc=0xf;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x00, 0x31, 0x00, 0x42, 0x00, 0xf0, 0x08, 0x2a, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] st8 [r14]=r0,0x8; nop.i 0x0; br.cloop.sptk.few $+0x0";
    let data = [0x10, 0x40, 0x00, 0x1c, 0x98, 0x15, 0x00, 0x00, 0x00, 0x02, 0x00, 0xa0, 0x00, 0x00, 0x00, 0x40];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] addl r35=0x47c0,r1;; nop.m 0x0; nop.i 0x0;;";
    let data = [0x0b, 0x18, 0x01, 0x03, 0x8f, 0x24, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] ld8 r47=[r35]; nop.i 0x0; br.call.sptk.many b0=$-0x175100;;";
    let data = [0x11, 0x78, 0x01, 0x46, 0x18, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x08, 0xaf, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; extr r36=r8,0x1f,0x1; nop.b 0x0";
    let data = [0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x40, 0xfa, 0x21, 0x00, 0x29, 0x00, 0x00, 0x00, 0x00, 0x20];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] adds r14=0x20,r12; ld8 r47=[r35]; mov r1=r45;;";
    let data = [0x09, 0x70, 0x80, 0x18, 0x00, 0x21, 0xf0, 0x02, 0x8c, 0x30, 0x20, 0x20, 0x00, 0x68, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; shr.u r36=r36,0x3a;; add r36=r36,r8;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x40, 0xa2, 0x93, 0x0a, 0x29, 0x80, 0x44, 0x42, 0x00, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; extr r36=r36,0x6,0x1a;; shladd r36=r36,0x3,r14;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x40, 0x6a, 0x90, 0x32, 0x29, 0x80, 0x44, 0x72, 0x48, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] ld8 r38=[r36]; nop.i 0x0; br.call.sptk.many b0=$-0x175150;;";
    let data = [0x11, 0x30, 0x01, 0x48, 0x18, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xb8, 0xae, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; extr r17=r8,0x1f,0x1; nop.b 0x0";
    let data = [0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0xf9, 0x21, 0x00, 0x29, 0x00, 0x00, 0x00, 0x00, 0x20];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r1=r45; mov r20=0x1; adds r15=0x10,r12;;";
    let data = [0x09, 0x08, 0x00, 0x5a, 0x00, 0x21, 0x40, 0x09, 0x00, 0x00, 0x48, 0xe0, 0x01, 0x61, 0x00, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] addl r14=0x3dd4,r1; nop.m 0x0; shr.u r17=r17,0x3a";
    let data = [0x08, 0x70, 0x50, 0x03, 0x7b, 0x24, 0x00, 0x00, 0x00, 0x02, 0x00, 0x20, 0x42, 0x8f, 0x14, 0x52];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld8 r18=[r41]; ld4 r35=[r34]; nop.i 0x0;;";
    let data = [0x09, 0x90, 0x00, 0x52, 0x18, 0x10, 0x30, 0x02, 0x88, 0x20, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MLX] ld4 r16=[r14]; movl r14=0x431bde82d7b634db";
    let data = [0x04, 0x80, 0x00, 0x1c, 0x10, 0x90, 0xd7, 0x82, 0xde, 0x1b, 0x43, 0xc0, 0xb1, 0xd5, 0xa6, 0x61];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] add r8=r8,r17; st4 [r34]=r37; nop.i 0x0;;";
    let data = [0x09, 0x40, 0x20, 0x22, 0x00, 0x20, 0x00, 0x28, 0x89, 0x20, 0x23, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; sxt4 r16=r16; nop.b 0x0";
    let data = [0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x40, 0x2c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] setf.sig f7=r14; nop.m 0x0; and r8=0x3f,r8;;";
    let data = [0x09, 0x38, 0x38, 0x00, 0xe1, 0x18, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xf1, 0x43, 0xb0, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] setf.sig f6=r16; shr r19=r16,0x3f; sub r17=r8,r17;;";
    let data = [0x01, 0x30, 0x40, 0x00, 0xe1, 0x18, 0x30, 0xf9, 0x43, 0x00, 0x29, 0x20, 0x82, 0x88, 0x14, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; sxt4 r17=r17;; shl r20=r20,r17;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x01, 0x44, 0x2c, 0x00, 0x80, 0x42, 0x89, 0x20, 0x79];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; xmpy.h f6=f6,f7; or r38=r38,r20;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x60, 0x00, 0x18, 0x0e, 0x77, 0xc0, 0x64, 0xa2, 0x38, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] st8 [r36]=r38;; getf.sig r14=f6; nop.i 0x0;;";
    let data = [0x0b, 0x00, 0x98, 0x48, 0x98, 0x11, 0xe0, 0x30, 0x00, 0xc2, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; shr r14=r14,0x12;; sub r14=r14,r19;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x28, 0x39, 0x5a, 0x29, 0xc0, 0xe1, 0x98, 0x14, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; shl r17=r14,0x5; sxt4 r19=r14;;";
    let data = [0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x71, 0xe8, 0xf4, 0x29, 0x60, 0x02, 0x70, 0x58, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] sub r17=r17,r14; st8 [r15]=r19,0x8; nop.i 0x0;;";
    let data = [0x09, 0x88, 0x44, 0x1c, 0x05, 0x20, 0x80, 0x98, 0x3c, 0x30, 0x2b, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; shl r19=r17,0x6;; sub r17=r19,r17;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x30, 0x89, 0xe4, 0xf2, 0x29, 0x20, 0x32, 0x89, 0x14, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] shladd r14=r17,0x3,r14;; nop.m 0x0; shl r14=r14,0x6;;";
    let data = [0x0b, 0x70, 0x44, 0x1c, 0x12, 0x20, 0x00, 0x00, 0x00, 0x02, 0x00, 0xc0, 0xe1, 0xc8, 0xe5, 0x53];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] sub r16=r16,r14;; nop.m 0x0; sxt4 r16=r16;;";
    let data = [0x0b, 0x80, 0x40, 0x1c, 0x05, 0x20, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x80, 0x58, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] st8 [r15]=r16;; ld8 r14=[r18],0x8; nop.i 0x0;;";
    let data = [0x0b, 0x00, 0x40, 0x1e, 0x98, 0x11, 0xe0, 0x40, 0x48, 0x30, 0x28, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] ld8 r1=[r18]; mov b6=r14; br.call.sptk.many b0=b6;;";
    let data = [0x11, 0x08, 0x00, 0x24, 0x18, 0x10, 0x60, 0x70, 0x04, 0x80, 0x03, 0x00, 0x68, 0x00, 0x80, 0x10];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r1=r45; mov r47=0x1; adds r48=0x20,r12";
    let data = [0x08, 0x08, 0x00, 0x5a, 0x00, 0x21, 0xf0, 0x0a, 0x00, 0x00, 0x48, 0x00, 0x06, 0x62, 0x00, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r49=r0; mov r50=r0; adds r51=0x10,r12;;";
    let data = [0x09, 0x88, 0x01, 0x00, 0x00, 0x21, 0x20, 0x03, 0x00, 0x00, 0x42, 0x60, 0x06, 0x61, 0x00, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; nop.i 0x0; br.call.sptk.many b0=$-0x175fa0;;";
    let data = [0x11, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x68, 0xa0, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] st4 [r34]=r35; mov r8=r0; nop.b 0x0";
    let data = [0x18, 0x00, 0x8c, 0x44, 0x90, 0x11, 0x80, 0x00, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x00, 0x20];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] mov r1=r45; mov.i ar.pfs=r44;; mov.i ar.lc=r46;;";
    let data = [0x03, 0x08, 0x00, 0x5a, 0x00, 0x21, 0x00, 0x60, 0x01, 0x55, 0x00, 0x00, 0xe0, 0x0a, 0xaa, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; mov b0=r43; nop.i 0x0";
    let data = [0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x58, 0x05, 0x80, 0x03, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] adds r12=0x90,r12; nop.m 0x0; br.ret.sptk.many b0";
    let data = [0x18, 0x60, 0x40, 0x18, 0x01, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x80, 0x08, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; mov r42=0x28; br.few $-0x3f0;;";
    let data = [0x11, 0x00, 0x00, 0x00, 0x01, 0x00, 0xa0, 0x42, 0x01, 0x00, 0x48, 0x00, 0x10, 0xfc, 0xff, 0x48];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; mov r42=0x5b; br.few $-0x400;;";
    let data = [0x11, 0x00, 0x00, 0x00, 0x01, 0x00, 0xa0, 0xda, 0x02, 0x00, 0x48, 0x00, 0x00, 0xfc, 0xff, 0x48];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
}

// from ia64 bash_4.2+dfsg-0.1+deb7u3_ia64 via binutils-ia64-linux-gnu=2.30-21ubuntu1~18.04.4
// bfd reports immediates in decimal in many places. all immediates are corrected to hex, as
// yaxpeax-ia64 only reports immediates as hexadecimal.
#[test]
fn test_bash_rl_set_paren_blink_timeout() {
    // 400000000019c500 <rl_set_paren_blink_timeout@@Base>:
    let decoder = InstDecoder::default();

    let expected = "[MII] addl r14=0x3dd4,r1; cmp4.lt p7,p6=0x0,r32;; nop.i 0x0";
    let data = [0x02, 0x70, 0x50, 0x03, 0x7b, 0x24, 0x70, 0x00, 0x80, 0x0c, 0x63, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] ld4 r8=[r14]; (p07) st4 [r14]=r32; br.ret.sptk.many b0;;";
    let data = [0x19, 0x40, 0x00, 0x1c, 0x10, 0xd0, 0x01, 0x00, 0x39, 0x20, 0x23, 0x80, 0x08, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x6; nop.i 0x7; nop.i 0x8";
    let data = [0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x70, 0x00, 0x00, 0x02, 0x00, 0x00, 0x01, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x6; nop.i 0x7; nop.i 0x8";
    let data = [0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x70, 0x00, 0x00, 0x02, 0x00, 0x00, 0x01, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] alloc r39=ar.pfs,14,9,0; addl r35=-0x9d0,r1; nop.b 0x0";
    let data = [0x18, 0x38, 0x39, 0x12, 0x80, 0x05, 0x30, 0x82, 0xf5, 0xd9, 0x4f, 0x00, 0x00, 0x00, 0x00, 0x20];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] addl r42=0x3f0,r1; mov r38=b0; mov r40=r1;;";
    let data = [0x01, 0x50, 0xc1, 0x03, 0x07, 0x24, 0x60, 0x02, 0x00, 0x62, 0x00, 0x00, 0x05, 0x08, 0x00, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld8 r35=[r35]; nop.m 0x0; mov r41=r0";
    let data = [0x08, 0x18, 0x01, 0x46, 0x18, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x20, 0x05, 0x00, 0x00, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r43=0x5; ld8 r42=[r42]; nop.i 0x0;;";
    let data = [0x09, 0x58, 0x15, 0x00, 0x00, 0x24, 0xa0, 0x02, 0xa8, 0x30, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] ld8 r36=[r35]; nop.i 0x0; br.call.sptk.many b0=$-0x1761c0;;";
    let data = [0x11, 0x20, 0x01, 0x46, 0x18, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x48, 0x9e, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r37=r8; cmp.eq p7,p6=0x0,r33; mov r1=r40";
    let data = [0x08, 0x28, 0x01, 0x10, 0x00, 0x21, 0x70, 0x00, 0x84, 0x0c, 0x72, 0x20, 0x00, 0x40, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] mov r44=r33; mov r45=r34; (p07) br.cond.dpnt.few $+0x90;;";
    let data = [0x19, 0x60, 0x01, 0x42, 0x00, 0x21, 0xd0, 0x02, 0x88, 0x00, 0xc2, 0x03, 0x90, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r43=r37; nop.m 0x0; mov r41=r36";
    let data = [0x08, 0x58, 0x01, 0x4a, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x20, 0x05, 0x20, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] mov r42=0x1; nop.m 0x0; br.call.sptk.many b0=$-0x175180;;";
    let data = [0x19, 0x50, 0x05, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x88, 0xae, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; mov r1=r40; nop.i 0x0";
    let data = [0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x00, 0xa0, 0x00, 0x42, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] ld8 r41=[r35]; nop.m 0x0; br.call.sptk.many b0=$-0x176500;;";
    let data = [0x19, 0x48, 0x01, 0x46, 0x18, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x08, 0x9b, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r1=r40; mov r41=r32; nop.i 0x0";
    let data = [0x08, 0x08, 0x00, 0x50, 0x00, 0x21, 0x90, 0x02, 0x80, 0x00, 0x42, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] mov r42=r33; mov r43=r34; br.call.sptk.many b0=$-0x11e900;;";
    let data = [0x19, 0x50, 0x01, 0x42, 0x00, 0x21, 0xb0, 0x02, 0x88, 0x00, 0x42, 0x00, 0x08, 0x17, 0xee, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r1=r40; nop.m 0x0; mov.i ar.pfs=r39;;";
    let data = [0x09, 0x08, 0x00, 0x50, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x70, 0x02, 0xaa, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; mov b0=r38; br.ret.sptk.many b0";
    let data = [0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x30, 0x05, 0x80, 0x03, 0x80, 0x08, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] addl r42=0x3f8,r1; mov r41=r0; mov r43=0x5;;";
    let data = [0x09, 0x50, 0xe1, 0x03, 0x07, 0x24, 0x90, 0x02, 0x00, 0x00, 0x42, 0x60, 0x55, 0x00, 0x00, 0x90];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] ld8 r42=[r42]; nop.i 0x0; br.call.sptk.many b0=$-0x176280;;";
    let data = [0x11, 0x50, 0x01, 0x54, 0x18, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x88, 0x9d, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r44=r8; mov r43=r37; mov r45=r34";
    let data = [0x08, 0x60, 0x01, 0x10, 0x00, 0x21, 0xb0, 0x02, 0x94, 0x00, 0x42, 0xa0, 0x05, 0x10, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r1=r40; mov r41=r36; mov r42=0x1;;";
    let data = [0x09, 0x08, 0x00, 0x50, 0x00, 0x21, 0x90, 0x02, 0x90, 0x00, 0x42, 0x40, 0x15, 0x00, 0x00, 0x90];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; nop.i 0x0; br.call.sptk.many b0=$-0x175230;;";
    let data = [0x11, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xd8, 0xad, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; mov r1=r40; nop.i 0x0";
    let data = [0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x00, 0xa0, 0x00, 0x42, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] ld8 r41=[r35]; nop.m 0x0; br.call.sptk.many b0=$-0x1765b0;;";
    let data = [0x19, 0x48, 0x01, 0x46, 0x18, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x58, 0x9a, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r1=r40; mov r41=r32; nop.i 0x0";
    let data = [0x08, 0x08, 0x00, 0x50, 0x00, 0x21, 0x90, 0x02, 0x80, 0x00, 0x42, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] mov r42=r33; mov r43=r34; br.call.sptk.many b0=$-0x11e9b0;;";
    let data = [0x19, 0x50, 0x01, 0x42, 0x00, 0x21, 0xb0, 0x02, 0x88, 0x00, 0x42, 0x00, 0x58, 0x16, 0xee, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r1=r40; nop.m 0x0; mov.i ar.pfs=r39;;";
    let data = [0x09, 0x08, 0x00, 0x50, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x70, 0x02, 0xaa, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; mov b0=r38; br.ret.sptk.many b0;;";
    let data = [0x11, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x30, 0x05, 0x80, 0x03, 0x80, 0x08, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] alloc r44=ar.pfs,18,15,0; adds r35=-0x8,r32; mov r43=b0";
    let data = [0x08, 0x60, 0x49, 0x1e, 0x80, 0x05, 0x30, 0xc2, 0x83, 0x7e, 0x46, 0x60, 0x05, 0x00, 0xc4, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] cmp.eq p6,p7=0x0,r32; mov r45=r1;; mov.i r46=ar.lc";
    let data = [0x02, 0x30, 0x00, 0x40, 0x07, 0x39, 0xd0, 0x02, 0x04, 0x00, 0x42, 0xc0, 0x05, 0x08, 0xca, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] mov r47=r0; mov r49=0x5; (p06) br.cond.dpnt.few $+0x4a0;;";
    let data = [0x19, 0x78, 0x01, 0x00, 0x00, 0x21, 0x10, 0x2b, 0x00, 0x00, 0x48, 0x03, 0xa0, 0x04, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld1 r14=[r35];; nop.m 0x0; sxt1 r14=r14;;";
    let data = [0x0b, 0x70, 0x00, 0x46, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0xc0, 0x01, 0x70, 0x50, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] cmp4.eq p6,p7=-0x2a,r14;; (p06) adds r14=-0x4,r32; (p07) mov r37=r32;;";
    let data = [0x0b, 0x30, 0x58, 0x1d, 0x87, 0xbb, 0xe1, 0xe0, 0x83, 0x7e, 0xc6, 0xa3, 0x04, 0x00, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] (p06) ld4 r37=[r14];; (p06) sub r37=r32,r37; nop.i 0x0;;";
    let data = [0xcb, 0x28, 0x01, 0x1c, 0x10, 0x90, 0x51, 0x02, 0x95, 0x0a, 0x40, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] (p06) adds r35=-0x8,r37;; (p06) ld1 r14=[r35]; nop.i 0x0;;";
    let data = [0xcb, 0x18, 0xe1, 0x4b, 0x3f, 0xa3, 0xe1, 0x00, 0x8c, 0x00, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; (p06) sxt1 r14=r14; nop.i 0x0;;";
    let data = [0x01, 0x00, 0x00, 0x00, 0x01, 0x80, 0xe1, 0x00, 0x38, 0x28, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] cmp4.eq p6,p7=-0x9,r14; nop.i 0x0; (p06) br.cond.dpnt.few $+0x70;;";
    let data = [0x11, 0x30, 0xdc, 0x1d, 0x87, 0x3b, 0x00, 0x00, 0x00, 0x02, 0x00, 0x03, 0x70, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] cmp4.eq p7,p6=0x54,r14;; (p07) addl r48=0x400,r1; nop.i 0x0;;";
    let data = [0x0b, 0x38, 0x50, 0x1d, 0x86, 0xf9, 0x01, 0x03, 0x04, 0x10, 0x48, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] (p06) addl r48=0x408,r1;; (p07) ld8 r48=[r48]; nop.i 0x0;;";
    let data = [0xcb, 0x80, 0x21, 0x02, 0x08, 0xe4, 0x01, 0x03, 0xc0, 0x30, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] (p06) ld8 r48=[r48]; nop.i 0x0; br.call.sptk.many b0=$-0x1763f0;;";
    let data = [0xd1, 0x80, 0x01, 0x60, 0x18, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x18, 0x9c, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r47=r8; mov r1=r45; nop.i 0x0";
    let data = [0x08, 0x78, 0x01, 0x10, 0x00, 0x21, 0x10, 0x00, 0xb4, 0x00, 0x42, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMB] mov r48=r33; mov r49=r34; br.call.sptk.many b0=$-0x290;;";
    let data = [0x19, 0x80, 0x01, 0x42, 0x00, 0x21, 0x10, 0x03, 0x88, 0x00, 0x42, 0x00, 0x78, 0xfd, 0xff, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r1=r45; nop.m 0x0; nop.i 0x0;;";
    let data = [0x09, 0x08, 0x00, 0x5a, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] adds r14=0x2,r35; mov r15=0x5555; nop.i 0x0";
    let data = [0x08, 0x70, 0x08, 0x46, 0x00, 0x21, 0xf0, 0xa8, 0x02, 0x54, 0x49, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] addl r47=0x410,r1; mov r48=r33; mov r49=r34;;";
    let data = [0x09, 0x78, 0x41, 0x02, 0x08, 0x24, 0x00, 0x03, 0x84, 0x00, 0x42, 0x20, 0x06, 0x10, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld2 r14=[r14]; ld8 r47=[r47]; nop.i 0x0;;";
    let data = [0x09, 0x70, 0x00, 0x1c, 0x08, 0x10, 0xf0, 0x02, 0xbc, 0x30, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] cmp4.eq p6,p7=r14,r15; nop.i 0x0; (p06) br.cond.dpnt.few $+0x30;;";
    let data = [0x11, 0x30, 0x38, 0x1e, 0x87, 0x38, 0x00, 0x00, 0x00, 0x02, 0x00, 0x03, 0x30, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; nop.i 0x0; br.call.sptk.many b0=$-0x2f0;;";
    let data = [0x11, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x18, 0xfd, 0xff, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r1=r45; nop.m 0x0; nop.i 0x0;;";
    let data = [0x09, 0x08, 0x00, 0x5a, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] adds r42=0x1,r35; adds r39=0x4,r35; addl r41=0x480,r1;;";
    let data = [0x09, 0x50, 0x05, 0x46, 0x00, 0x21, 0x70, 0x22, 0x8c, 0x00, 0x42, 0x20, 0x05, 0x08, 0x24, 0x90];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld1 r36=[r42]; ld4 r15=[r39]; nop.i 0x0;;";
    let data = [0x09, 0x20, 0x01, 0x54, 0x00, 0x10, 0xf0, 0x00, 0x9c, 0x20, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; sxt1 r36=r36; nop.i 0x0";
    let data = [0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x40, 0x02, 0x90, 0x28, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] ld8 r41=[r41]; adds r14=0x13,r15;; sxt4 r38=r36";
    let data = [0x02, 0x48, 0x01, 0x52, 0x18, 0x10, 0xe0, 0x98, 0x3c, 0x00, 0x42, 0xc0, 0x04, 0x20, 0x59, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] and r14=-0x8,r14;; shladd r16=r38,0x3,r41; addp4 r14=r14,r0;;";
    let data = [0x0b, 0x70, 0xe0, 0x1d, 0x2c, 0x22, 0x00, 0x31, 0xa5, 0x24, 0x40, 0xc0, 0xe1, 0x00, 0x20, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld8 r40=[r16]; nop.m 0x0; addp4 r16=r15,r0;;";
    let data = [0x09, 0x40, 0x01, 0x20, 0x18, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xf2, 0x00, 0x20, 0x80];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] cmp.ltu p7,p6=r40,r14; mov r14=r0; (p07) br.cond.dpnt.few $+0x330;;";
    let data = [0x11, 0x38, 0xa0, 0x1c, 0x06, 0x34, 0xe0, 0x00, 0x00, 0x00, 0xc2, 0x03, 0x30, 0x03, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] add r37=r37,r16;; mov r16=r37; adds r17=0x2,r37";
    let data = [0x0a, 0x28, 0x95, 0x20, 0x00, 0x20, 0x00, 0x01, 0x94, 0x00, 0x42, 0x20, 0x22, 0x28, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] adds r37=0x3,r37;; ld1 r18=[r16],0x1; nop.i 0x0";
    let data = [0x0a, 0x28, 0x0d, 0x4a, 0x00, 0x21, 0x20, 0x09, 0x40, 0x00, 0x28, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld1 r17=[r17];; nop.m 0x0; dep r14=r18,r14,0x0,0x8";
    let data = [0x0a, 0x88, 0x00, 0x22, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0xc0, 0x21, 0x71, 0xdc, 0x4f];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld1 r18=[r16]; ld1 r16=[r37]; nop.i 0x0;;";
    let data = [0x09, 0x90, 0x00, 0x20, 0x00, 0x10, 0x00, 0x01, 0x94, 0x00, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; dep r14=r18,r14,0x8,0x8;; dep r14=r17,r14,0x10,0x8;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x90, 0x38, 0xee, 0x26, 0xc0, 0x11, 0x71, 0xdc, 0x4b];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; dep r14=r16,r14,0x18,0x8; nop.i 0x0;;";
    let data = [0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x80, 0x38, 0xee, 0x24, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; cmp4.eq p6,p7=r15,r14; (p06) br.cond.dpnt.few $+0x60";
    let data = [0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x60, 0x78, 0x38, 0x0e, 0x71, 0x03, 0x60, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] addl r48=0x420,r1; mov r47=r0; mov r49=0x5;;";
    let data = [0x09, 0x80, 0x81, 0x02, 0x08, 0x24, 0xf0, 0x02, 0x00, 0x00, 0x42, 0x20, 0x56, 0x00, 0x00, 0x90];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] ld8 r48=[r48]; nop.i 0x0; br.call.sptk.many b0=$-0x176580;;";
    let data = [0x11, 0x80, 0x01, 0x60, 0x18, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x88, 0x9a, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r47=r8; mov r1=r45; nop.i 0x0";
    let data = [0x08, 0x78, 0x01, 0x10, 0x00, 0x21, 0x10, 0x00, 0xb4, 0x00, 0x42, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r1=r45; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x08, 0x00, 0x5a, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; addl r37=0x8f40,r1; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0x50, 0x02, 0x06, 0x3c, 0x4a, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; ld4 r49=[r39]; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x03, 0x9c, 0x20, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] addp4 r49=r49,r0;; cmp.ltu p7,p6=0x7,r49; nop.i 0x0;;";
    let data = [0x0b, 0x88, 0xc5, 0x00, 0x08, 0x20, 0x70, 0x38, 0xc4, 0x0c, 0x6a, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] (p07) adds r14=-0x1,r49; (p07) and r49=0x7,r49;; (p07) shr.u r14=r14,0x3;;";
    let data = [0xe3, 0x70, 0xfc, 0x63, 0x3f, 0xe3, 0x11, 0x3b, 0xc4, 0x58, 0xc0, 0xc3, 0x61, 0x70, 0xf0, 0x52];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] (p06) mov r14=r0; nop.m 0x0; cmp.ltu p6,p7=0x7,r49;;";
    let data = [0xc9, 0x70, 0x00, 0x00, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0xc0, 0x70, 0x88, 0x1d, 0xd4];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; mov r15=-0x31; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0xf0, 0x78, 0xf2, 0xff, 0x4f, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] st1 [r32]=r15,0x1; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x08, 0x3c, 0x40, 0x80, 0x15, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; mov r15=-0x31; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0xf0, 0x78, 0xf2, 0xff, 0x4f, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] st1 [r32]=r15,0x1; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x08, 0x3c, 0x40, 0x80, 0x15, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; mov r15=-0x31; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0xf0, 0x78, 0xf2, 0xff, 0x4f, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] st1 [r32]=r15,0x1; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x08, 0x3c, 0x40, 0x80, 0x15, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; mov r15=-0x31; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0xf0, 0x78, 0xf2, 0xff, 0x4f, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] st1 [r32]=r15,0x1; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x08, 0x3c, 0x40, 0x80, 0x15, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; mov r15=-0x31; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0xf0, 0x78, 0xf2, 0xff, 0x4f, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] st1 [r32]=r15,0x1; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x08, 0x3c, 0x40, 0x80, 0x15, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; mov r15=-0x31; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0xf0, 0x78, 0xf2, 0xff, 0x4f, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] st1 [r32]=r15,0x1; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x08, 0x3c, 0x40, 0x80, 0x15, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; mov r15=-0x31; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0xf0, 0x78, 0xf2, 0xff, 0x4f, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] st1 [r32]=r15,0x1; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x08, 0x3c, 0x40, 0x80, 0x15, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] cmp.eq p6,p7=0x0,r14; nop.m 0x0; mov r15=-0x31;;";
    let data = [0x09, 0x30, 0x00, 0x1c, 0x07, 0x39, 0x00, 0x00, 0x00, 0x02, 0x00, 0xe0, 0xf1, 0xe4, 0xff, 0x9f];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; st1 [r32]=r15; (p07) adds r14=-0x1,r14";
    let data = [0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x78, 0x80, 0x00, 0xa3, 0xc3, 0xf1, 0x77, 0xfc, 0x8c];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] add r15=r37,r38;; ld1 r14=[r15]; nop.i 0x0;;";
    let data = [0x0b, 0x78, 0x94, 0x4c, 0x00, 0x20, 0xe0, 0x00, 0x3c, 0x00, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; sxt1 r14=r14; nop.i 0x0;;";
    let data = [0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x00, 0x38, 0x28, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] st1 [r15]=r0; addl r15=0x8f60,r1; mov r14=r35;;";
    let data = [0x09, 0x00, 0x00, 0x1e, 0x80, 0x11, 0xf0, 0x00, 0x07, 0x3c, 0x4a, 0xc0, 0x01, 0x18, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0;; shladd r38=r38,0x3,r15; mov r15=0x54;;";
    let data = [0x0b, 0x00, 0x00, 0x00, 0x01, 0x00, 0x60, 0x32, 0x3d, 0x24, 0x40, 0xe0, 0x41, 0x05, 0x00, 0x90];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] st1 [r14]=r15,0x8; ld8 r15=[r38]; nop.i 0x0";
    let data = [0x08, 0x40, 0x3c, 0x1c, 0x80, 0x15, 0xf0, 0x00, 0x98, 0x30, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] st8 [r38]=r35;; st8 [r14]=r15; nop.i 0x0";
    let data = [0x0a, 0x00, 0x8c, 0x4c, 0x98, 0x11, 0x00, 0x78, 0x38, 0x30, 0x23, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; mov.i ar.pfs=r44;; mov.i ar.lc=r46;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x60, 0x01, 0x55, 0x00, 0x00, 0xe0, 0x0a, 0xaa, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] addl r48=0x418,r1; mov r47=r0; mov r49=0x5;;";
    let data = [0x09, 0x80, 0x61, 0x02, 0x08, 0x24, 0xf0, 0x02, 0x00, 0x00, 0x42, 0x20, 0x56, 0x00, 0x00, 0x90];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r47=r8; mov r1=r45; nop.i 0x0";
    let data = [0x08, 0x78, 0x01, 0x10, 0x00, 0x21, 0x10, 0x00, 0xb4, 0x00, 0x42, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld4 r15=[r39]; mov r14=r0; mov r1=r45;;";
    let data = [0x09, 0x78, 0x00, 0x4e, 0x10, 0x10, 0xe0, 0x00, 0x00, 0x00, 0x42, 0x20, 0x00, 0x68, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] addp4 r16=r15,r0;; add r37=r37,r16; nop.i 0x0;;";
    let data = [0x0b, 0x80, 0x3c, 0x00, 0x08, 0x20, 0x50, 0x2a, 0x41, 0x00, 0x40, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r16=r37; adds r17=0x2,r37; adds r37=0x3,r37;;";
    let data = [0x09, 0x80, 0x00, 0x4a, 0x00, 0x21, 0x10, 0x11, 0x94, 0x00, 0x42, 0xa0, 0x34, 0x28, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld1 r18=[r16],0x1; ld1 r17=[r17]; nop.i 0x0;;";
    let data = [0x09, 0x90, 0x04, 0x20, 0x00, 0x14, 0x10, 0x01, 0x44, 0x00, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; dep r14=r18,r14,0x0,0x8; nop.i 0x0";
    let data = [0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x90, 0x38, 0xee, 0x27, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld1 r18=[r16]; ld1 r16=[r37]; nop.i 0x0;;";
    let data = [0x09, 0x90, 0x00, 0x20, 0x00, 0x10, 0x00, 0x01, 0x94, 0x00, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; dep r14=r18,r14,0x8,0x8;; dep r14=r17,r14,0x10,0x8;;";
    let data = [0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x90, 0x38, 0xee, 0x26, 0xc0, 0x11, 0x71, 0xdc, 0x4b];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; dep r14=r16,r14,0x18,0x8; nop.i 0x0;;";
    let data = [0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x80, 0x38, 0xee, 0x24, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r32=r33; nop.m 0x0; mov b0=r46";
    let data = [0x08, 0x00, 0x01, 0x42, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xe0, 0x0a, 0x00, 0x07];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] mov r33=r34; mov r34=r35;; mov.i ar.pfs=r47;;";
    let data = [0x03, 0x08, 0x01, 0x44, 0x00, 0x21, 0x20, 0x02, 0x8c, 0x00, 0x42, 0x00, 0xf0, 0x02, 0xaa, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] alloc r2=ar.pfs,4,0,0; nop.i 0x0; br.many $-0x1990;;";
    let data = [0x11, 0x10, 0x10, 0x00, 0x80, 0x05, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x78, 0xe6, 0xff, 0x48];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; break.f 0x0; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; nop.m 0x0; nop.i 0x0";
    let data = [0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
}

// from ia64 bash_4.2+dfsg-0.1+deb7u3_ia64 via binutils-ia64-linux-gnu=2.30-21ubuntu1~18.04.4
// bfd reports immediates in decimal in many places. all immediates are corrected to hex, as
// yaxpeax-ia64 only reports immediates as hexadecimal.
#[test]
fn test_bash_libc_csu_init() {
    // 400000000019f500 <__libc_csu_init@@Base>:
    let decoder = InstDecoder::default();

    let expected = "[MII] alloc r39=ar.pfs,12,9,0; mov r38=b0; mov r40=r1;;";
    let data = [0x01, 0x38, 0x31, 0x12, 0x80, 0x05, 0x60, 0x02, 0x00, 0x62, 0x00, 0x00, 0x05, 0x08, 0x00, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r35=r0; nop.i 0x0; br.call.sptk.many b0=$-0x17a3f0;;";
    let data = [0x11, 0x18, 0x01, 0x00, 0x00, 0x21, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x18, 0x5c, 0xe8, 0x58];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r1=r40;; addl r36=-0xa330,r1; addl r37=-0xa318,r1;;";
    let data = [0x0b, 0x08, 0x00, 0x50, 0x00, 0x21, 0x40, 0x82, 0xf6, 0x73, 0x4d, 0xa0, 0x84, 0xee, 0xe7, 0x9a];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; nop.m 0x0; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] sub r37=r37,r36;; nop.m 0x0; shr r37=r37,0x3;;";
    let data = [0x0b, 0x28, 0x95, 0x48, 0x05, 0x20, 0x00, 0x00, 0x00, 0x02, 0x00, 0xa0, 0x74, 0x28, 0xf1, 0x52];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; cmp.eq p6,p7=0x0,r37; (p06) br.cond.dpnt.few $+0x60;;";
    let data = [0x11, 0x00, 0x00, 0x00, 0x01, 0x00, 0x60, 0x00, 0x94, 0x0e, 0x72, 0x03, 0x60, 0x00, 0x00, 0x43];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; ld8 r14=[r36],0x8; mov r41=r32";
    let data = [0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe0, 0x40, 0x90, 0x30, 0x28, 0x20, 0x05, 0x00, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] mov r42=r33; mov r43=r34; adds r35=0x1,r35;;";
    let data = [0x09, 0x50, 0x01, 0x42, 0x00, 0x21, 0xb0, 0x02, 0x88, 0x00, 0x42, 0x60, 0x14, 0x18, 0x01, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] nop.m 0x0; ld8 r15=[r14],0x8; nop.i 0x0;;";
    let data = [0x09, 0x00, 0x00, 0x00, 0x01, 0x00, 0xf0, 0x40, 0x38, 0x30, 0x28, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] ld8 r1=[r14]; mov b6=r15; br.call.sptk.many b0=b6;;";
    let data = [0x11, 0x08, 0x00, 0x1c, 0x18, 0x10, 0x60, 0x78, 0x04, 0x80, 0x03, 0x00, 0x68, 0x00, 0x80, 0x10];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] mov r1=r40; cmp.ltu p6,p7=r35,r37; (p06) br.cond.dptk.few $-0x40";
    let data = [0x10, 0x08, 0x00, 0x50, 0x00, 0x21, 0x60, 0x18, 0x95, 0x0e, 0x68, 0x03, 0xc0, 0xff, 0xff, 0x4a];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x0; mov.i ar.pfs=r39; nop.i 0x0;;";
    let data = [0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x38, 0x01, 0x55, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; mov b0=r38; br.ret.sptk.many b0;;";
    let data = [0x11, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x30, 0x05, 0x80, 0x03, 0x80, 0x08, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x6; nop.i 0x7; nop.i 0x8";
    let data = [0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x70, 0x00, 0x00, 0x02, 0x00, 0x00, 0x01, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MLX] alloc r34=ar.pfs,5,5,0; movl r32=0xfffffffffffff5f8;;";
    let data = [0x05, 0x10, 0x15, 0x0a, 0x80, 0xc5, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x00, 0x84, 0xf7, 0xaf, 0x6f];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] add r32=r32,r1;; ld8 r35=[r32],-0x8; mov r33=b0;;";
    let data = [0x0b, 0x00, 0x81, 0x02, 0x00, 0x20, 0x30, 0xc2, 0x83, 0x32, 0x2c, 0x20, 0x04, 0x00, 0xc4, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] cmp.eq p6,p0=-0x1,r35; mov r36=r1; (p06) br.cond.spnt.few $+0x40";
    let data = [0x10, 0x30, 0xfc, 0x47, 0x00, 0x3b, 0x40, 0x02, 0x04, 0x00, 0x42, 0x03, 0x40, 0x00, 0x00, 0x41];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] ld8 r15=[r35],0x8;; ld8 r1=[r35]; mov b6=r15";
    let data = [0x0a, 0x78, 0x20, 0x46, 0x18, 0x14, 0x10, 0x00, 0x8c, 0x30, 0x20, 0xc0, 0xf0, 0x08, 0x00, 0x07];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] ld8 r35=[r32],-0x8; nop.i 0x0; br.call.sptk.many b0=b6;;";
    let data = [0x11, 0x18, 0xe1, 0x41, 0x19, 0x16, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x68, 0x00, 0x80, 0x10];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] cmp.eq p0,p6=-0x1,r35; nop.i 0x0; (p06) br.cond.sptk.few $-0x20";
    let data = [0x10, 0x00, 0xfc, 0x47, 0x06, 0x3b, 0x00, 0x00, 0x00, 0x02, 0x00, 0x03, 0xe0, 0xff, 0xff, 0x48];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] mov r1=r35; mov b0=r33; mov.i ar.pfs=r34";
    let data = [0x00, 0x08, 0x00, 0x46, 0x00, 0x21, 0x00, 0x08, 0x05, 0x80, 0x03, 0x00, 0x20, 0x02, 0xaa, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] nop.m 0x0; nop.i 0x0; br.ret.sptk.many b0;;";
    let data = [0x11, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x80, 0x08, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
}

// from ia64 bash_4.2+dfsg-0.1+deb7u3_ia64 via binutils-ia64-linux-gnu=2.30-21ubuntu1~18.04.4
// bfd reports immediates in decimal in many places. all immediates are corrected to hex, as
// yaxpeax-ia64 only reports immediates as hexadecimal.
#[test]
fn test_bash_cfree() {
    // 400000000019f080 <cfree@@Base>:
    let decoder = InstDecoder::default();

    let expected = "[MMI] alloc r16=ar.pfs,3,3,0;; mov r33=r0; mov r34=r0;;";
    let data = [0x0b, 0x80, 0x0c, 0x06, 0x80, 0x05, 0x10, 0x02, 0x00, 0x00, 0x42, 0x40, 0x04, 0x00, 0x00, 0x84];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] alloc r2=ar.pfs,3,0,0; nop.i 0x0; br.many $-0x2990;;";
    let data = [0x11, 0x10, 0x0c, 0x00, 0x80, 0x05, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x78, 0xd6, 0xff, 0x48];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; break.f 0x0; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] setf.sig f8=r32; setf.sig f9=r33; cmp.eq.unc p7,p0=0x0,r33;;";
    let data = [0x09, 0x40, 0x80, 0x00, 0xe1, 0x18, 0x90, 0x08, 0x01, 0xc2, 0x31, 0xe0, 0x08, 0x08, 0x01, 0xe4];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; fcvt.xf f8=f8; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x80, 0x40, 0x00, 0x38, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; fcvt.xf f9=f9; (p07) break.i 0x1;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x90, 0x48, 0x00, 0x38, 0x80, 0x23, 0x00, 0x00, 0x00, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; frcpa.s1 f10,p6=f8,f9; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0xa0, 0x40, 0x24, 0x8c, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fnma.s1 f11=f9,f10,f1; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x80, 0xb1, 0x08, 0x24, 0x14, 0x61, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fmpy.s1 f12=f8,f10; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xc1, 0x00, 0x20, 0x14, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fmpy.s1 f13=f11,f11; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x80, 0xd1, 0x00, 0x2c, 0x16, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fma.s1 f12=f11,f12,f12; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xc1, 0x60, 0x2c, 0x18, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fma.s1 f10=f11,f10,f10; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x80, 0xa1, 0x50, 0x2c, 0x14, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fma.s1 f11=f13,f12,f12; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xb1, 0x60, 0x34, 0x18, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fma.s1 f10=f13,f10,f10; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x80, 0xa1, 0x50, 0x34, 0x14, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fnma.s1 f12=f9,f11,f8; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xc1, 0x40, 0x24, 0x16, 0x61, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fma.s1 f10=f12,f10,f11; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xa1, 0x58, 0x30, 0x14, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; fcvt.fx.trunc.s1 f10=f10; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0xa0, 0x50, 0x00, 0x34, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] getf.sig r8=f10; nop.i 0x0; br.ret.sptk.few b0;;";
    let data = [0x11, 0x40, 0x28, 0x00, 0xe1, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x80, 0x00, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] setf.sig f14=r32; setf.sig f9=r33; cmp.eq.unc p7,p0=0x0,r33;;";
    let data = [0x09, 0x70, 0x80, 0x00, 0xe1, 0x18, 0x90, 0x08, 0x01, 0xc2, 0x31, 0xe0, 0x08, 0x08, 0x01, 0xe4];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; fcvt.xf f8=f14; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x80, 0x70, 0x00, 0x38, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; fcvt.xf f9=f9; (p07) break.i 0x1;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x90, 0x48, 0x00, 0x38, 0x80, 0x23, 0x00, 0x00, 0x00, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; frcpa.s1 f10,p6=f8,f9; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0xa0, 0x40, 0x24, 0x8c, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fmpy.s1 f12=f8,f10; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x80, 0xc1, 0x00, 0x20, 0x14, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fnma.s1 f11=f9,f10,f1; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xb1, 0x08, 0x24, 0x14, 0x61, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fma.s1 f12=f11,f12,f12; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x80, 0xc1, 0x60, 0x2c, 0x18, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fmpy.s1 f13=f11,f11; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xd1, 0x00, 0x2c, 0x16, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fma.s1 f10=f11,f10,f10; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x80, 0xa1, 0x50, 0x2c, 0x14, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fma.s1 f11=f13,f12,f12; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xb1, 0x60, 0x34, 0x18, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] sub r33=r0,r33; (p06) fma.s1 f10=f13,f10,f10; nop.i 0x0";
    let data = [0x0c, 0x08, 0x01, 0x42, 0x05, 0xa0, 0xa1, 0x50, 0x34, 0x14, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fnma.s1 f12=f9,f11,f8; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xc1, 0x40, 0x24, 0x16, 0x61, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] setf.sig f9=r33; (p06) fma.s1 f10=f12,f10,f11; nop.i 0x0;;";
    let data = [0x0d, 0x48, 0x84, 0x00, 0xe1, 0x98, 0xa1, 0x58, 0x30, 0x14, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; fcvt.fx.trunc.s1 f10=f10; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0xa0, 0x50, 0x00, 0x34, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; xma.l f10=f10,f9,f14; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0xa0, 0x70, 0x28, 0x12, 0x74, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] getf.sig r8=f10; nop.i 0x0; br.ret.sptk.few b0;;";
    let data = [0x11, 0x40, 0x28, 0x00, 0xe1, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x80, 0x00, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] setf.sig f8=r32; setf.sig f9=r33; cmp.eq.unc p7,p0=0x0,r33;;";
    let data = [0x09, 0x40, 0x80, 0x00, 0xe1, 0x18, 0x90, 0x08, 0x01, 0xc2, 0x31, 0xe0, 0x08, 0x08, 0x01, 0xe4];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; fnorm.s1 f8=f8; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x80, 0x00, 0x20, 0x02, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; fnorm.s1 f9=f9; (p07) break.i 0x1;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x90, 0x00, 0x24, 0x02, 0xc1, 0x23, 0x00, 0x00, 0x00, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; frcpa.s1 f10,p6=f8,f9; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0xa0, 0x40, 0x24, 0x8c, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fnma.s1 f11=f9,f10,f1; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x80, 0xb1, 0x08, 0x24, 0x14, 0x61, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fmpy.s1 f12=f8,f10; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xc1, 0x00, 0x20, 0x14, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fmpy.s1 f13=f11,f11; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x80, 0xd1, 0x00, 0x2c, 0x16, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fma.s1 f12=f11,f12,f12; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xc1, 0x60, 0x2c, 0x18, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fma.s1 f10=f11,f10,f10; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x80, 0xa1, 0x50, 0x2c, 0x14, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fma.s1 f11=f13,f12,f12; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xb1, 0x60, 0x34, 0x18, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fma.s1 f10=f13,f10,f10; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x80, 0xa1, 0x50, 0x34, 0x14, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fnma.s1 f12=f9,f11,f8; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xc1, 0x40, 0x24, 0x16, 0x61, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fma.s1 f10=f12,f10,f11; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xa1, 0x58, 0x30, 0x14, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; fcvt.fxu.trunc.s1 f10=f10; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0xa0, 0x50, 0x00, 0x36, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] getf.sig r8=f10; nop.i 0x0; br.ret.sptk.few b0;;";
    let data = [0x11, 0x40, 0x28, 0x00, 0xe1, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x80, 0x00, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MMI] setf.sig f14=r32; setf.sig f9=r33; cmp.eq.unc p7,p0=0x0,r33;;";
    let data = [0x09, 0x70, 0x80, 0x00, 0xe1, 0x18, 0x90, 0x08, 0x01, 0xc2, 0x31, 0xe0, 0x08, 0x08, 0x01, 0xe4];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; fnorm.s1 f8=f14; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x80, 0x00, 0x38, 0x02, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; fnorm.s1 f9=f9; (p07) break.i 0x1;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x90, 0x00, 0x24, 0x02, 0xc1, 0x23, 0x00, 0x00, 0x00, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; frcpa.s1 f10,p6=f8,f9; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0xa0, 0x40, 0x24, 0x8c, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fmpy.s1 f12=f8,f10; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x80, 0xc1, 0x00, 0x20, 0x14, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fnma.s1 f11=f9,f10,f1; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xb1, 0x08, 0x24, 0x14, 0x61, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fma.s1 f12=f11,f12,f12; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x80, 0xc1, 0x60, 0x2c, 0x18, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fmpy.s1 f13=f11,f11; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xd1, 0x00, 0x2c, 0x16, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fma.s1 f10=f11,f10,f10; nop.i 0x0";
    let data = [0x0c, 0x00, 0x00, 0x00, 0x01, 0x80, 0xa1, 0x50, 0x2c, 0x14, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fma.s1 f11=f13,f12,f12; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xb1, 0x60, 0x34, 0x18, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] sub r33=r0,r33; (p06) fma.s1 f10=f13,f10,f10; nop.i 0x0";
    let data = [0x0c, 0x08, 0x01, 0x42, 0x05, 0xa0, 0xa1, 0x50, 0x34, 0x14, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; (p06) fnma.s1 f12=f9,f11,f8; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x80, 0xc1, 0x40, 0x24, 0x16, 0x61, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] setf.sig f9=r33; (p06) fma.s1 f10=f12,f10,f11; nop.i 0x0;;";
    let data = [0x0d, 0x48, 0x84, 0x00, 0xe1, 0x98, 0xa1, 0x58, 0x30, 0x14, 0x41, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; fcvt.fxu.trunc.s1 f10=f10; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0xa0, 0x50, 0x00, 0x36, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MFI] nop.m 0x0; xma.l f10=f10,f9,f14; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0xa0, 0x70, 0x28, 0x12, 0x74, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MIB] getf.sig r8=f10; nop.i 0x0; br.ret.sptk.few b0;;";
    let data = [0x11, 0x40, 0x28, 0x00, 0xe1, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x80, 0x00, 0x00, 0x84, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x6; nop.i 0x7; nop.i 0x8";
    let data = [0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x70, 0x00, 0x00, 0x02, 0x00, 0x00, 0x01, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x6; nop.i 0x7; nop.i 0x8";
    let data = [0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x70, 0x00, 0x00, 0x02, 0x00, 0x00, 0x01, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
    let expected = "[MII] nop.m 0x6; nop.i 0x7; nop.i 0x8";
    let data = [0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x70, 0x00, 0x00, 0x02, 0x00, 0x00, 0x01, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
}

// from 460gx system firmware
#[test]
fn test_msr() {
    let decoder = InstDecoder::default();

    let expected = "[MMI] mov r44=0x40; mov r31=msr[r37]; nop.i 0x0;;";
    let data = [0x09, 0x60, 0x01, 0x01, 0x00, 0x24, 0xf0, 0x01, 0x94, 0x2c, 0x08, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    let expected = "[MFI] mov msr[r37]=r9; nop.f 0x0; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x24, 0x4a, 0x06, 0x04, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    let expected = "[MFI] mov msr[r2]=r9; nop.f 0x0; nop.i 0x0;;";
    let data = [0x0d, 0x00, 0x24, 0x04, 0x06, 0x04, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    let expected = "[MMI] mov r20=msr[r3]; mov r21=msr[r21]; nop.i 0x0";
    let data = [0x08, 0xa0, 0x00, 0x06, 0x16, 0x04, 0x50, 0x01, 0x54, 0x2c, 0x08, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    let expected = "[MMI] mov r50=0x60e; mov r54=msr[r48]; nop.i 0x0";
    let data = [0x08, 0x90, 0x39, 0x00, 0x0c, 0x24, 0x60, 0x03, 0xc0, 0x2c, 0x08, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    let expected = "[MII] (p04) mov msr[r3]=r38; (p08) mov r16=0x405; (p09) mov r17=0x406;;";
    let data = [0x81, 0x00, 0x98, 0x06, 0x06, 0x04, 0x02, 0x29, 0x00, 0x10, 0xc8, 0x24, 0x62, 0x00, 0x20, 0x90];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
}

// from 460gx system firmware
#[test]
fn test_indirection() {
    let decoder = InstDecoder::default();

    let expected = "[MMI] mov r2=0x3;; mov r2=cpuid[r2]; nop.i 0x0;;";
    let data = [0x0B, 0x10, 0x0C, 0x00, 0x00, 0x24, 0x20, 0x00, 0x08, 0x2E, 0x08, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    let expected = "[MMI] mov r20=0x3;; mov r20=cpuid[r20]; nop.i 0x0;;";
    let data = [0x0B, 0xA0, 0x0C, 0x00, 0x00, 0x24, 0x40, 0x01, 0x50, 0x2E, 0x08, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    let expected = "[MMI] mov r41=dbr[r0]; mov r42=dbr[r37]; nop.i 0x0";
    let data = [0x08, 0x48, 0x01, 0x00, 0x11, 0x04, 0xA0, 0x02, 0x94, 0x22, 0x08, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    let expected = "[MMI] mov dbr[r37]=r42;; mov dbr[r38]=r43; nop.i 0x0;;";
    let data = [0x0B, 0x00, 0xA8, 0x4A, 0x01, 0x04, 0x00, 0x58, 0x99, 0x02, 0x08, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);

    let expected = "[MMI] mov r44=pmc[r0]; mov pmc[r0]=r2; nop.i 0x0;;";
    let data = [0x09, 0x60, 0x01, 0x00, 0x14, 0x04, 0x00, 0x10, 0x00, 0x08, 0x08, 0x00, 0x00, 0x00, 0x04, 0x00];
    let mut reader = U8Reader::new(&data[..]);
    let inst = decoder.decode(&mut reader).unwrap();
    assert_eq!(format!("{}", inst), expected);
}
