// Copyright 2021, Nobuto Kaitoh
//
// This file is part of yatotp.
//
// Yatotp is free software: you can redistribute it and/or
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Yatotp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with yatotp.  If not, see <https://www.gnu.org/licenses/>.

extern crate yatotp;

use anyhow::Result;
use structopt::StructOpt;
use yatotp::*;

#[derive(StructOpt)]
#[structopt(about = "Yet Another TOTP Client.")]
struct Args {
    #[structopt(subcommand)]
    command: Command,
    #[structopt(short = "i", long = "database", parse(from_os_str))]
    database: std::path::PathBuf,
}

#[derive(StructOpt)]
enum Command {
    /// Add new entry to database.
    Add {
        #[structopt(short = "e", long, help = "Treat key as base32 encoded.")]
        base32_encode: bool,
    },
    /// Remove specified entry from database.
    Remove {
        #[structopt(help = "Name of entry.")]
        name: String,
    },
    /// Show TOTP value of specified entry.
    Show {
        #[structopt(help = "Name of entry.")]
        name: String,
    },
    /// Print list of TOTP entries.
    List,
}

fn main() -> Result<()> {
    let args = Args::from_args();
    match args.command {
        Command::Add { base32_encode } => cli::add(&args.database, base32_encode)?,
        Command::Remove { name } => cli::remove(&args.database, &name)?,
        Command::Show { name } => cli::show(&args.database, &name)?,
        Command::List => cli::list(&args.database)?,
    }
    Ok(())
    // totp.insert(
    //     "test".to_string(),
    //     otp::TotpClient::new_from_base32key(
    //         "JBSWY3DPEHPK3PXP".to_string(),
    //         30,
    //         0,
    //         6,
    //         otp::HashType::Sha1,
    //     )
    //     .unwrap(),
    // );
    // println!("{}", totp["test"].totp(&Utc::now()));
    // let save_path = path::Path::new("./clients.json");
    // database::save_database(&totp, &save_path).unwrap();
}
